/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/. */

// Test from official test vectors in
// https://github.com/usnistgov/ACVP-Server/tree/v1.1.0.35/gen-val/json-files/ML-KEM-keyGen-FIPS203
// Note that the ciphertext values are replaced with their SHA3-256 hash values.

#include <vector>

#include "kyber.h"

struct MlKem768Encap {
  KyberParams params;
  uint8_t entropy[KYBER_ENC_COIN_BYTES];
  uint8_t publicKey[KYBER768_PUBLIC_KEY_BYTES];
  bool expectedResult;
  uint8_t ciphertextDigest[SHA3_256_LENGTH];
  uint8_t secret[KYBER_SHARED_SECRET_BYTES];
};

std::vector<MlKem768Encap> MlKem768EncapTests = {
    {params_ml_kem768_test_mode,
     {0x2C, 0xE7, 0x4A, 0xD2, 0x91, 0x13, 0x35, 0x18, 0xFE, 0x60, 0xC7,
      0xDF, 0x5D, 0x25, 0x1B, 0x9D, 0x82, 0xAD, 0xD4, 0x84, 0x62, 0xFF,
      0x50, 0x5C, 0x6E, 0x54, 0x7E, 0x94, 0x9E, 0x6B, 0x6B, 0xF7},
     {0x89, 0xD2, 0xCB, 0x65, 0xF9, 0x4D, 0xCB, 0xFC, 0x89, 0x0E, 0xFC, 0x7D,
      0x0E, 0x5A, 0x7A, 0x38, 0x34, 0x4D, 0x16, 0x41, 0xA3, 0xD0, 0xB0, 0x24,
      0xD5, 0x07, 0x97, 0xA5, 0xF2, 0x3C, 0x3A, 0x18, 0xB3, 0x10, 0x1A, 0x12,
      0x69, 0x06, 0x9F, 0x43, 0xA8, 0x42, 0xBA, 0xCC, 0x09, 0x8A, 0x88, 0x21,
      0x27, 0x1C, 0x67, 0x3D, 0xB1, 0xBE, 0xB3, 0x30, 0x34, 0xE4, 0xD7, 0x77,
      0x4D, 0x16, 0x63, 0x5C, 0x7C, 0x2C, 0x3C, 0x27, 0x63, 0x45, 0x35, 0x38,
      0xBC, 0x16, 0x32, 0xE1, 0x85, 0x15, 0x91, 0xA5, 0x16, 0x42, 0x97, 0x4E,
      0x59, 0x28, 0xAB, 0xB8, 0xE5, 0x5F, 0xE5, 0x56, 0x12, 0xF9, 0xB1, 0x41,
      0xAF, 0xF0, 0x15, 0x54, 0x53, 0x94, 0xB2, 0x09, 0x2E, 0x59, 0x09, 0x70,
      0xEC, 0x29, 0xA7, 0xB7, 0xE7, 0xAA, 0x1F, 0xB4, 0x49, 0x3B, 0xF7, 0xCB,
      0x73, 0x19, 0x06, 0xC2, 0xA5, 0xCB, 0x49, 0xE6, 0x61, 0x48, 0x59, 0x06,
      0x4E, 0x19, 0xB8, 0xFA, 0x26, 0xAF, 0x51, 0xC4, 0x4B, 0x5E, 0x75, 0x35,
      0xBF, 0xDA, 0xC0, 0x72, 0xB6, 0x46, 0xD3, 0xEA, 0x49, 0x0D, 0x27, 0x7F,
      0x0D, 0x97, 0xCE, 0xD4, 0x73, 0x95, 0xFE, 0xD9, 0x1E, 0x8F, 0x2B, 0xCE,
      0x0E, 0x3C, 0xA1, 0x22, 0xC2, 0x02, 0x5F, 0x74, 0x06, 0x7A, 0xB9, 0x28,
      0xA8, 0x22, 0xB3, 0x56, 0x53, 0xA7, 0x4F, 0x06, 0x75, 0x76, 0x29, 0xAF,
      0xB1, 0xA1, 0xCA, 0xF2, 0x37, 0x10, 0x0E, 0xA9, 0x35, 0xE7, 0x93, 0xC8,
      0xF5, 0x8A, 0x71, 0xB3, 0xD6, 0xAE, 0x2C, 0x86, 0x58, 0xB1, 0x01, 0x50,
      0xD4, 0xA3, 0x8F, 0x57, 0x2A, 0x0D, 0x49, 0xD2, 0x8A, 0xE8, 0x94, 0x51,
      0xD3, 0x38, 0x32, 0x6F, 0xDB, 0x3B, 0x43, 0x50, 0x03, 0x6C, 0x10, 0x81,
      0x11, 0x77, 0x40, 0xED, 0xB8, 0x6B, 0x12, 0x08, 0x1C, 0x5C, 0x12, 0x23,
      0xDB, 0xB5, 0x66, 0x0D, 0x5B, 0x3C, 0xB3, 0x78, 0x7D, 0x48, 0x18, 0x49,
      0x30, 0x4C, 0x68, 0xBE, 0x87, 0x54, 0x66, 0xF1, 0x4E, 0xE5, 0x49, 0x5C,
      0x2B, 0xD7, 0x95, 0xAE, 0x41, 0x2D, 0x09, 0x00, 0x2D, 0x65, 0xB8, 0x71,
      0x9B, 0x90, 0xCB, 0xA3, 0x60, 0x3A, 0xC4, 0x95, 0x8E, 0xA0, 0x3C, 0xC1,
      0x38, 0xC8, 0x6F, 0x78, 0x51, 0x59, 0x31, 0x25, 0x33, 0x47, 0x01, 0xB6,
      0x77, 0xF8, 0x2F, 0x49, 0x52, 0xA4, 0xC9, 0x3B, 0x5B, 0x4C, 0x13, 0x4B,
      0xB4, 0x2A, 0x85, 0x7F, 0xD1, 0x5C, 0x65, 0x08, 0x64, 0xA6, 0xAA, 0x94,
      0xEB, 0x69, 0x1C, 0x0B, 0x69, 0x1B, 0xE4, 0x68, 0x4C, 0x1F, 0x5B, 0x74,
      0x90, 0x46, 0x7F, 0xC0, 0x1B, 0x1D, 0x1F, 0xDA, 0x4D, 0xDA, 0x35, 0xC4,
      0xEC, 0xC2, 0x31, 0xBC, 0x73, 0xA6, 0xFE, 0xF4, 0x2C, 0x99, 0xD3, 0x4E,
      0xB8, 0x2A, 0x4D, 0x01, 0x49, 0x87, 0xB3, 0xE3, 0x86, 0x91, 0x0C, 0x62,
      0x67, 0x9A, 0x11, 0x8F, 0x3C, 0x5B, 0xD9, 0xF4, 0x67, 0xE4, 0x16, 0x20,
      0x42, 0x42, 0x43, 0x57, 0xDB, 0x92, 0xEF, 0x48, 0x4A, 0x4A, 0x17, 0x98,
      0xC1, 0x25, 0x7E, 0x87, 0x0A, 0x30, 0xCB, 0x20, 0xAA, 0xA0, 0x33, 0x5D,
      0x83, 0x31, 0x4F, 0xE0, 0xAA, 0x7E, 0x63, 0xA8, 0x62, 0x64, 0x80, 0x41,
      0xA7, 0x2A, 0x63, 0x21, 0x52, 0x32, 0x20, 0xB1, 0xAC, 0xE9, 0xBB, 0x70,
      0x1B, 0x21, 0xAC, 0x12, 0x53, 0xCB, 0x81, 0x2C, 0x15, 0x57, 0x5A, 0x90,
      0x85, 0xEA, 0xBE, 0xAD, 0xE7, 0x3A, 0x4A, 0xE7, 0x6E, 0x6A, 0x7B, 0x15,
      0x8A, 0x20, 0x58, 0x6D, 0x78, 0xA5, 0xAC, 0x62, 0x0A, 0x5C, 0x9A, 0xBC,
      0xC9, 0xC0, 0x43, 0x35, 0x0A, 0x73, 0x65, 0x6B, 0x0A, 0xBE, 0x82, 0x2D,
      0xA5, 0xE0, 0xBA, 0x76, 0x04, 0x5F, 0xAD, 0x75, 0x40, 0x1D, 0x7A, 0x3B,
      0x70, 0x37, 0x91, 0xB7, 0xE9, 0x92, 0x61, 0x71, 0x0F, 0x86, 0xB7, 0x24,
      0x21, 0xD2, 0x40, 0xA3, 0x47, 0x63, 0x83, 0x77, 0x20, 0x5A, 0x15, 0x2C,
      0x79, 0x41, 0x30, 0xA4, 0xE0, 0x47, 0x74, 0x2B, 0x88, 0x83, 0x03, 0xBD,
      0xDC, 0x30, 0x91, 0x16, 0x76, 0x4D, 0xE7, 0x42, 0x4C, 0xEB, 0xEA, 0x6D,
      0xB6, 0x53, 0x48, 0xAC, 0x53, 0x7E, 0x01, 0xA9, 0xCC, 0x56, 0xEA, 0x66,
      0x7D, 0x5A, 0xA8, 0x7A, 0xC9, 0xAA, 0xA4, 0x31, 0x7D, 0x26, 0x2C, 0x10,
      0x14, 0x30, 0x50, 0xB8, 0xD0, 0x7A, 0x72, 0x8C, 0xA6, 0x33, 0xC1, 0x3E,
      0x46, 0x8A, 0xBC, 0xEA, 0xD3, 0x72, 0xC7, 0x7B, 0x8E, 0xCF, 0x3B, 0x98,
      0x6B, 0x98, 0xC1, 0xE5, 0x58, 0x60, 0xB2, 0xB4, 0x21, 0x67, 0x66, 0xAD,
      0x87, 0x4C, 0x35, 0xED, 0x72, 0x05, 0x06, 0x87, 0x39, 0x23, 0x02, 0x20,
      0xB5, 0xA2, 0x31, 0x7D, 0x10, 0x2C, 0x59, 0x83, 0x56, 0xF1, 0x68, 0xAC,
      0xBE, 0x80, 0x60, 0x8D, 0xE4, 0xC9, 0xA7, 0x10, 0xB8, 0xDD, 0x07, 0x07,
      0x8C, 0xD7, 0xC6, 0x71, 0x05, 0x8A, 0xF1, 0xB0, 0xB8, 0x30, 0x4A, 0x31,
      0x4F, 0x7B, 0x29, 0xBE, 0x78, 0xA9, 0x33, 0xC7, 0xB9, 0x29, 0x44, 0x24,
      0x95, 0x4A, 0x1B, 0xF8, 0xBC, 0x74, 0x5D, 0xE8, 0x61, 0x98, 0x65, 0x9E,
      0x0E, 0x12, 0x25, 0xA9, 0x10, 0x72, 0x60, 0x74, 0x96, 0x9C, 0x39, 0xA9,
      0x7C, 0x19, 0x24, 0x06, 0x01, 0xA4, 0x6E, 0x01, 0x3D, 0xCD, 0xCB, 0x67,
      0x7A, 0x8C, 0xBD, 0x2C, 0x95, 0xA4, 0x06, 0x29, 0xC2, 0x56, 0xF2, 0x4A,
      0x32, 0x89, 0x51, 0xDF, 0x57, 0x50, 0x2A, 0xB3, 0x07, 0x72, 0xCC, 0x7E,
      0x5B, 0x85, 0x00, 0x27, 0xC8, 0x55, 0x17, 0x81, 0xCE, 0x49, 0x85, 0xBD,
      0xAC, 0xF6, 0xB8, 0x65, 0xC1, 0x04, 0xE8, 0xA4, 0xBC, 0x65, 0xC4, 0x16,
      0x94, 0xD4, 0x56, 0xB7, 0x16, 0x9E, 0x45, 0xAB, 0x3D, 0x7A, 0xCA, 0xBE,
      0xAF, 0xE2, 0x3A, 0xD6, 0xA7, 0xB9, 0x4D, 0x19, 0x79, 0xA2, 0xF4, 0xC1,
      0xCA, 0xE7, 0xCD, 0x77, 0xD6, 0x81, 0xD2, 0x90, 0xB5, 0xD8, 0xE4, 0x51,
      0xBF, 0xDC, 0xCC, 0xF5, 0x31, 0x0B, 0x9D, 0x12, 0xA8, 0x8E, 0xC2, 0x9B,
      0x10, 0x25, 0x5D, 0x5E, 0x17, 0xA1, 0x92, 0x67, 0x0A, 0xA9, 0x73, 0x1C,
      0x5C, 0xA6, 0x7E, 0xC7, 0x84, 0xC5, 0x02, 0x78, 0x1B, 0xE8, 0x52, 0x7D,
      0x6F, 0xC0, 0x03, 0xC6, 0x70, 0x1B, 0x36, 0x32, 0x28, 0x4B, 0x40, 0x30,
      0x7A, 0x52, 0x7C, 0x76, 0x20, 0x37, 0x7F, 0xEB, 0x0B, 0x73, 0xF7, 0x22,
      0xC9, 0xE3, 0xCD, 0x4D, 0xEC, 0x64, 0x87, 0x6B, 0x93, 0xAB, 0x5B, 0x7C,
      0xFC, 0x4A, 0x65, 0x7F, 0x85, 0x2B, 0x65, 0x92, 0x82, 0x86, 0x43, 0x84,
      0xF4, 0x42, 0xB2, 0x2E, 0x8A, 0x21, 0x10, 0x93, 0x87, 0xB8, 0xB4, 0x75,
      0x85, 0xFC, 0x68, 0x0D, 0x0B, 0xA4, 0x5C, 0x7A, 0x8B, 0x1D, 0x72, 0x74,
      0xBD, 0xA5, 0x78, 0x45, 0xD1, 0x00, 0xD0, 0xF4, 0x2A, 0x3B, 0x74, 0x62,
      0x87, 0x73, 0x35, 0x1F, 0xD7, 0xAC, 0x30, 0x5B, 0x24, 0x97, 0x63, 0x9B,
      0xE9, 0x0B, 0x3F, 0x4F, 0x71, 0xA6, 0xAA, 0x35, 0x61, 0xEE, 0xCC, 0x6A,
      0x69, 0x1B, 0xB5, 0xCB, 0x39, 0x14, 0xD8, 0x63, 0x4C, 0xA1, 0xE1, 0xAF,
      0x54, 0x3C, 0x04, 0x9A, 0x8C, 0x6E, 0x86, 0x8C, 0x51, 0xF0, 0x42, 0x3B,
      0xD2, 0xD5, 0xAE, 0x09, 0xB7, 0x9E, 0x57, 0xC2, 0x7F, 0x3F, 0xE3, 0xAE,
      0x2B, 0x26, 0xA4, 0x41, 0xBA, 0xBF, 0xC6, 0x71, 0x8C, 0xE8, 0xC0, 0x5B,
      0x4F, 0xE7, 0x93, 0xB9, 0x10, 0xB8, 0xFB, 0xCB, 0xBE, 0x7F, 0x10, 0x13,
      0x24, 0x2B, 0x40, 0xE0, 0x51, 0x4D, 0x0B, 0xDC, 0x5C, 0x88, 0xBA, 0xC5,
      0x94, 0xC7, 0x94, 0xCE, 0x51, 0x22, 0xFB, 0xF3, 0x48, 0x96, 0x81, 0x91,
      0x47, 0xB9, 0x28, 0x38, 0x15, 0x87, 0x96, 0x3B, 0x0B, 0x90, 0x03, 0x4A,
      0xA0, 0x7A, 0x10, 0xBE, 0x17, 0x6E, 0x01, 0xC8, 0x0A, 0xD6, 0xA4, 0xB7,
      0x1B, 0x10, 0xAF, 0x42, 0x41, 0x40, 0x0A, 0x2A, 0x4C, 0xBB, 0xC0, 0x59,
      0x61, 0xA1, 0x5E, 0xC1, 0x47, 0x4E, 0xD5, 0x1A, 0x3C, 0xC6, 0xD3, 0x58,
      0x00, 0x67, 0x9A, 0x46, 0x28, 0x09, 0xCA, 0xA3, 0xAB, 0x4F, 0x70, 0x94,
      0xCD, 0x66, 0x10, 0xB4, 0xA7, 0x00, 0xCB, 0xA9, 0x39, 0xE7, 0xEA, 0xC9,
      0x3E, 0x38, 0xC9, 0x97, 0x55, 0x90, 0x87, 0x27, 0x61, 0x9E, 0xD7, 0x6A,
      0x34, 0xE5, 0x3C, 0x4F, 0xA2, 0x5B, 0xFC, 0x97, 0x00, 0x82, 0x06, 0x69,
      0x7D, 0xD1, 0x45, 0xE5, 0xB9, 0x18, 0x8E, 0x5B, 0x01, 0x4E, 0x94, 0x16,
      0x81, 0xE1, 0x5F, 0xE3, 0xE1, 0x32, 0xB8, 0xA3, 0x90, 0x34, 0x74, 0x14,
      0x8B, 0xA2, 0x8B, 0x98, 0x71, 0x11, 0xC9, 0xBC, 0xB3, 0x98, 0x9B, 0xBB,
      0xC6, 0x71, 0xC5, 0x81, 0xB4, 0x4A, 0x49, 0x28, 0x45, 0xF2, 0x88, 0xE6,
      0x21, 0x96, 0xE4, 0x71, 0xFE, 0xD3, 0xC3, 0x9C, 0x1B, 0xBD, 0xDB, 0x08,
      0x37, 0xD0, 0xD4, 0x70, 0x6B, 0x09, 0x22, 0xC4},
     true,
     {0x6a, 0x09, 0x40, 0xcb, 0x38, 0xcb, 0xbd, 0xf2, 0xdf, 0xa5, 0x3d,
      0x1b, 0x51, 0x0c, 0xd8, 0x76, 0xb2, 0x85, 0x4e, 0x12, 0xa3, 0x54,
      0xed, 0x3e, 0x9e, 0x0b, 0x07, 0x99, 0x9c, 0x7c, 0xf2, 0xfe},
     {0x26, 0x96, 0xD2, 0x8E, 0x9C, 0x61, 0xC2, 0xA0, 0x1C, 0xE9, 0xB1,
      0x60, 0x8D, 0xCB, 0x9D, 0x29, 0x27, 0x85, 0xA0, 0xCD, 0x58, 0xEF,
      0xB7, 0xFE, 0x13, 0xB1, 0xDE, 0x95, 0xF0, 0xDB, 0x55, 0xB3}},

    {params_ml_kem768_test_mode,
     {0x76, 0xD0, 0x4F, 0x48, 0x1E, 0x68, 0xB2, 0xF9, 0x01, 0xEC, 0xAB,
      0x58, 0xB6, 0x36, 0x9A, 0x2C, 0xC3, 0x1A, 0x9D, 0xCC, 0xED, 0x82,
      0xA1, 0xBB, 0xD4, 0x26, 0xBE, 0x0A, 0xEE, 0x26, 0x6A, 0xEE},
     {0xF5, 0x84, 0x1D, 0x6A, 0xEA, 0x68, 0x3F, 0xDB, 0xA1, 0x63, 0x08, 0xBD,
      0xAB, 0x82, 0x8D, 0xDD, 0xD7, 0x73, 0x5B, 0x8B, 0x7A, 0x0D, 0xAC, 0x6A,
      0x57, 0xEB, 0x51, 0x34, 0xB9, 0x1D, 0x8D, 0x6C, 0xBD, 0x98, 0x95, 0x80,
      0x41, 0x11, 0x44, 0xE1, 0xFB, 0x5A, 0x6A, 0x55, 0x9A, 0x70, 0x56, 0x37,
      0x62, 0x10, 0xA8, 0x28, 0x47, 0x42, 0xD2, 0x2A, 0x58, 0x81, 0xC5, 0x21,
      0x4C, 0x90, 0x02, 0x3F, 0xC9, 0x10, 0xD5, 0xD0, 0x2A, 0x86, 0x90, 0x87,
      0x55, 0x79, 0x00, 0x27, 0x3B, 0xB8, 0x75, 0x42, 0x0B, 0x57, 0x17, 0xCD,
      0x0B, 0x23, 0x06, 0x4A, 0xA8, 0x20, 0xCD, 0xF3, 0x72, 0xF3, 0xE4, 0x77,
      0x8D, 0x70, 0xAE, 0xB5, 0xD0, 0x2B, 0x61, 0x82, 0xC4, 0xD3, 0x71, 0x10,
      0xD7, 0x82, 0xB6, 0xE8, 0x03, 0x03, 0x33, 0x26, 0x97, 0xB4, 0xC6, 0x10,
      0xA3, 0x84, 0xA0, 0xC6, 0x32, 0xC0, 0xD9, 0x48, 0x4A, 0x1D, 0x3B, 0x5E,
      0xA9, 0x21, 0x52, 0x5B, 0xEC, 0x57, 0x55, 0xC8, 0x39, 0xDF, 0x94, 0x2F,
      0x24, 0xA0, 0x27, 0xDB, 0x50, 0xB2, 0xD7, 0x60, 0x06, 0x6D, 0x10, 0xA1,
      0x17, 0xBC, 0x9A, 0x1B, 0x65, 0xC4, 0x48, 0xCB, 0x9A, 0xCF, 0x3B, 0x4F,
      0x64, 0x43, 0x16, 0xE8, 0x94, 0x1C, 0x44, 0x98, 0x03, 0xF6, 0x85, 0x1A,
      0x74, 0xD8, 0x32, 0xA7, 0x39, 0xB2, 0xC0, 0xEA, 0x92, 0x58, 0xC7, 0x25,
      0x8E, 0x98, 0xBD, 0x3E, 0x83, 0x3D, 0x87, 0x9A, 0x68, 0x45, 0xEC, 0x4E,
      0xCC, 0x44, 0xB6, 0xFA, 0x69, 0x93, 0x88, 0x13, 0x5F, 0x5E, 0x48, 0x30,
      0xF2, 0x62, 0x5E, 0x9F, 0xA5, 0xCC, 0x98, 0x2C, 0x57, 0x8B, 0x25, 0x93,
      0xD3, 0x50, 0xB0, 0x62, 0x88, 0xA8, 0x54, 0xD3, 0x34, 0x9C, 0x24, 0x58,
      0x6D, 0x3A, 0xA2, 0xE6, 0x87, 0x26, 0xA8, 0x73, 0xB1, 0xE5, 0xAA, 0xA3,
      0xB2, 0x26, 0x71, 0xD8, 0xC6, 0x9A, 0xEB, 0x18, 0x07, 0x18, 0xCB, 0x45,
      0x6B, 0x94, 0x2E, 0x4B, 0x66, 0x78, 0xE6, 0x20, 0xA0, 0x0B, 0xCA, 0x31,
      0x0C, 0x72, 0x2D, 0xDD, 0x49, 0x9E, 0xAD, 0x9C, 0x6B, 0x66, 0x66, 0x6A,
      0x3D, 0xE3, 0x9A, 0x45, 0xD7, 0xAF, 0x0B, 0xBB, 0x7A, 0xB6, 0xA0, 0xBE,
      0xAF, 0x8B, 0xBC, 0xBB, 0xA1, 0x7B, 0x1D, 0x09, 0x7A, 0xBB, 0x09, 0xA7,
      0x0E, 0x41, 0x03, 0x52, 0xD2, 0x08, 0x44, 0x23, 0xAC, 0x53, 0xEC, 0xBB,
      0x4C, 0x19, 0x60, 0x21, 0xF0, 0x1E, 0x66, 0x2A, 0x60, 0xC6, 0x8B, 0x3B,
      0xF4, 0x8A, 0x5F, 0x08, 0x64, 0xA2, 0x55, 0x77, 0x91, 0x2F, 0x52, 0x62,
      0x0C, 0xE6, 0x34, 0x7B, 0xD2, 0x7F, 0xF6, 0x8A, 0x17, 0xD4, 0xB9, 0x2C,
      0xD7, 0xD0, 0x1B, 0x89, 0xE3, 0x48, 0x7A, 0x5B, 0xC2, 0x85, 0x97, 0x81,
      0xF3, 0xEB, 0xB8, 0xB5, 0xB4, 0xC2, 0xD6, 0x82, 0x63, 0x6C, 0x48, 0x6A,
      0x00, 0x0A, 0x57, 0x6A, 0x4B, 0x63, 0xAF, 0xFC, 0x05, 0x08, 0x2B, 0x5A,
      0xBE, 0x3C, 0xC0, 0xB3, 0x7B, 0x1E, 0x58, 0x6C, 0x21, 0x07, 0xD9, 0x71,
      0x57, 0xE3, 0x25, 0xA0, 0x67, 0xBB, 0x86, 0x45, 0x34, 0x14, 0xA1, 0x55,
      0x94, 0xA5, 0x10, 0xDC, 0xFB, 0x2F, 0xE1, 0xA0, 0x07, 0x44, 0x83, 0x12,
      0x0F, 0xB8, 0x34, 0x40, 0xDB, 0x1B, 0x8C, 0x3B, 0x41, 0xE3, 0x63, 0x64,
      0xF9, 0x20, 0x56, 0x08, 0x3C, 0xB9, 0xCF, 0x91, 0xB3, 0x9F, 0x28, 0xCF,
      0x00, 0xF6, 0xAD, 0x09, 0x8A, 0xA1, 0x0F, 0xDB, 0x4B, 0x4D, 0x9B, 0x64,
      0xED, 0x13, 0x38, 0xE0, 0xD5, 0xB7, 0xA5, 0x16, 0x9C, 0x3D, 0x8C, 0x01,
      0x84, 0xB1, 0x99, 0x66, 0xE5, 0x42, 0x72, 0xF7, 0x65, 0xC0, 0x33, 0x7B,
      0xBD, 0x30, 0x7F, 0x8C, 0x97, 0x36, 0x9A, 0x7A, 0x87, 0xDA, 0x44, 0xA5,
      0xBF, 0x46, 0x8D, 0xB8, 0xA9, 0xAA, 0x5E, 0xA5, 0x98, 0xF8, 0x85, 0xAB,
      0x50, 0x17, 0x4B, 0x0F, 0x90, 0x25, 0xA4, 0xEB, 0x53, 0xD2, 0x32, 0x3D,
      0x20, 0x2A, 0x05, 0x26, 0x53, 0x31, 0xFD, 0x83, 0x6D, 0xF8, 0xE0, 0x2B,
      0x45, 0x95, 0x45, 0x85, 0x51, 0xAB, 0xED, 0x8A, 0x38, 0x75, 0xB8, 0x3B,
      0xF9, 0x76, 0x94, 0x23, 0x72, 0xCB, 0x37, 0x29, 0x6C, 0x81, 0x3A, 0xCD,
      0x2C, 0x27, 0xB4, 0x1A, 0x55, 0x14, 0xB6, 0x6A, 0xB2, 0x57, 0x59, 0x00,
      0x9D, 0xB3, 0x8A, 0x9D, 0x04, 0x73, 0xD5, 0xB7, 0xA9, 0xA7, 0xD6, 0x79,
      0x5F, 0x11, 0x88, 0xA0, 0x79, 0xB1, 0x79, 0x2A, 0x01, 0x14, 0x13, 0x47,
      0xAF, 0x21, 0x94, 0xCA, 0x68, 0x10, 0x55, 0xD3, 0x6E, 0x95, 0x4C, 0x02,
      0xD6, 0x93, 0x5B, 0xBA, 0x7C, 0x2E, 0xF7, 0xF4, 0xB5, 0xE4, 0x7C, 0x8B,
      0x0A, 0x00, 0x69, 0xF2, 0x95, 0x75, 0xE8, 0x63, 0x96, 0x7C, 0xE4, 0xC5,
      0x31, 0x05, 0x23, 0x04, 0x72, 0x17, 0x2F, 0xB7, 0x9E, 0x69, 0x08, 0x9D,
      0x5A, 0x7B, 0xCA, 0xA9, 0x57, 0x84, 0xBF, 0xA2, 0x79, 0xEF, 0xE6, 0x7D,
      0xA1, 0x45, 0x30, 0x8B, 0xAA, 0xA1, 0xA5, 0xA3, 0x03, 0x75, 0x79, 0x46,
      0xC2, 0x86, 0x6B, 0x48, 0x41, 0x66, 0x0A, 0x99, 0xC1, 0x96, 0x8B, 0x8F,
      0x7D, 0xE7, 0x99, 0xAB, 0xD7, 0x18, 0x06, 0xEB, 0x9F, 0x09, 0x13, 0x97,
      0xC1, 0xCC, 0x41, 0x71, 0x15, 0x2A, 0x6A, 0xFC, 0x36, 0xBD, 0x73, 0x3F,
      0xC6, 0xC5, 0x35, 0x45, 0x36, 0x1A, 0xB6, 0x25, 0x8C, 0xB4, 0x5C, 0x9F,
      0x13, 0x31, 0xBA, 0xEA, 0x85, 0xBE, 0x45, 0x58, 0x93, 0x59, 0x84, 0xC0,
      0x81, 0xF7, 0x3E, 0x4B, 0x37, 0x7E, 0x02, 0x51, 0xCA, 0x7C, 0x39, 0x6B,
      0xBB, 0xB8, 0x1D, 0x27, 0x1B, 0xB9, 0xF0, 0x58, 0x9E, 0x1B, 0xE3, 0x21,
      0x8B, 0x0B, 0x58, 0x40, 0x37, 0x22, 0x53, 0xAA, 0x80, 0xA5, 0xDB, 0x79,
      0xE1, 0x11, 0x99, 0xC0, 0x83, 0x2B, 0x24, 0x33, 0x88, 0x0B, 0x68, 0xBD,
      0x84, 0xFC, 0x02, 0xAA, 0x3C, 0xBB, 0xEC, 0x20, 0x5E, 0xBB, 0xC7, 0xB0,
      0x50, 0x96, 0x7B, 0x4D, 0xFB, 0x11, 0xE2, 0xFA, 0x63, 0xBC, 0xF6, 0xB7,
      0x65, 0x6A, 0x80, 0x28, 0xAB, 0x60, 0x7C, 0xB0, 0x84, 0xC2, 0x17, 0x47,
      0xED, 0x57, 0x3A, 0x05, 0x51, 0x66, 0xF8, 0x22, 0x15, 0xD7, 0x20, 0x1D,
      0x5D, 0x43, 0x9A, 0x19, 0xF5, 0x84, 0xF4, 0x70, 0xB4, 0x27, 0x29, 0x62,
      0xC1, 0x37, 0xB3, 0x85, 0x45, 0x30, 0x95, 0x47, 0xCE, 0xC2, 0x5B, 0x09,
      0xC9, 0x64, 0x59, 0xAB, 0x7B, 0x4D, 0xA6, 0x9C, 0x8D, 0x7B, 0x92, 0x77,
      0xBB, 0xC4, 0xB5, 0x56, 0x88, 0x13, 0xDA, 0x90, 0x41, 0x41, 0xA0, 0x11,
      0xD9, 0xB4, 0x5A, 0xC1, 0xF1, 0x81, 0x27, 0x31, 0x49, 0xF3, 0xC4, 0x6F,
      0x45, 0xCA, 0x97, 0x35, 0x22, 0x1B, 0x97, 0xCB, 0x52, 0x8E, 0x8A, 0xB5,
      0x9C, 0x57, 0x11, 0xA5, 0x7C, 0x60, 0x3F, 0x7A, 0x91, 0x80, 0x32, 0x54,
      0xE8, 0xCC, 0x4A, 0x37, 0xD8, 0x4D, 0x1F, 0x65, 0x35, 0xE5, 0xA7, 0x91,
      0xA5, 0x01, 0x45, 0xE1, 0xE0, 0x73, 0x43, 0x08, 0x10, 0xB3, 0xAB, 0x79,
      0xDF, 0x40, 0x53, 0x53, 0x8C, 0x7D, 0xB4, 0x82, 0x6A, 0x1B, 0x42, 0x8A,
      0x84, 0x55, 0x3B, 0xB8, 0x81, 0xA2, 0x35, 0x07, 0x38, 0x52, 0x71, 0xB3,
      0x2F, 0x85, 0x47, 0x06, 0xBB, 0x2D, 0x3E, 0x88, 0x4E, 0x7B, 0x39, 0x19,
      0x85, 0xB3, 0x9B, 0x7B, 0xA3, 0x73, 0x07, 0x14, 0x55, 0x18, 0x7B, 0x3D,
      0xD7, 0xDA, 0x75, 0xF6, 0x98, 0x8B, 0xBD, 0x6B, 0xC3, 0x9E, 0xF2, 0x80,
      0x8C, 0x24, 0x5A, 0xEC, 0x9C, 0x02, 0x4C, 0xA1, 0x65, 0x46, 0xA1, 0x6F,
      0x63, 0x83, 0x1A, 0x7B, 0x67, 0x97, 0x95, 0x1A, 0x40, 0x89, 0x4A, 0x5E,
      0x38, 0x42, 0x2F, 0x30, 0xB8, 0x7E, 0x70, 0x35, 0x5C, 0xCB, 0xE9, 0x60,
      0xB2, 0x16, 0x59, 0x2D, 0x00, 0x73, 0xF1, 0x24, 0x0C, 0x21, 0xBB, 0x10,
      0x9A, 0xE7, 0x6C, 0x9D, 0xE5, 0xB7, 0x83, 0x5B, 0xC0, 0x8A, 0xC6, 0x60,
      0x1C, 0x31, 0x4A, 0x82, 0x23, 0x2F, 0xA6, 0xF6, 0x89, 0x6B, 0xD7, 0x83,
      0x4F, 0x02, 0x54, 0xBF, 0x11, 0x26, 0x02, 0x02, 0x28, 0x44, 0xF0, 0xCB,
      0xA9, 0xFC, 0x3D, 0x2E, 0x3A, 0x58, 0xED, 0xD5, 0x6D, 0xDC, 0x49, 0x8A,
      0xDC, 0x9A, 0x03, 0xFC, 0xB4, 0x3C, 0xA1, 0x38, 0x64, 0x0F, 0x85, 0x39,
      0x7F, 0xD5, 0x73, 0x1F, 0x53, 0x7D, 0x6B, 0xDC, 0x3A, 0xC7, 0x65, 0x63,
      0xD6, 0x51, 0x6F, 0x1C, 0xF2, 0x4F, 0x84, 0xB7, 0xC9, 0x57, 0x63, 0x5D,
      0xEF, 0xBB, 0xB7, 0x00, 0x71, 0x62, 0x1C, 0x8B, 0x25, 0x85, 0x38, 0x0A,
      0x63, 0x66, 0x0E, 0xF2, 0xCB, 0x6C, 0xA5, 0x91, 0x0B, 0xAD, 0x42, 0xA1,
      0xB6, 0x21, 0xCA, 0xB8, 0xC2, 0x67, 0x80, 0xD4, 0x25, 0x1D, 0xFD, 0x1C,
      0x63, 0x70, 0xEF, 0x12, 0x19, 0x3C, 0x3C, 0xEF, 0x02, 0x23, 0x18, 0x7A,
      0x45, 0x57, 0xBC, 0x08, 0xF4, 0xAD, 0xD3, 0x82},
     true,
     {0xF2, 0xD0, 0x45, 0x74, 0xE0, 0x7B, 0xEF, 0xD3, 0x6F, 0x26, 0x7E,
      0x9E, 0x17, 0x31, 0x25, 0xC2, 0x46, 0x23, 0x8A, 0xC0, 0x2F, 0xC0,
      0x91, 0x1F, 0xE6, 0xD4, 0x65, 0xDC, 0xDE, 0x74, 0xE5, 0x4D},
     {0x44, 0x26, 0x36, 0x24, 0x05, 0x2C, 0x18, 0xE3, 0xAA, 0x23, 0x31,
      0x06, 0x97, 0x41, 0x44, 0x99, 0xF1, 0xC0, 0xEA, 0xE4, 0x5A, 0x10,
      0x60, 0xD8, 0x4E, 0xEB, 0x65, 0xFC, 0xDB, 0xCB, 0x57, 0x33}},

    {params_ml_kem768_test_mode,
     {0xFD, 0x3C, 0x91, 0x29, 0x4D, 0x8C, 0x97, 0x49, 0x30, 0xB4, 0xB6,
      0x13, 0x5A, 0xB6, 0x47, 0xD4, 0xA7, 0x88, 0x5C, 0x83, 0xFC, 0xDC,
      0xB3, 0x0C, 0xBD, 0x38, 0x33, 0x2E, 0x14, 0x09, 0x44, 0x91},
     {0x92, 0xD1, 0xA8, 0x17, 0x51, 0xC4, 0x0C, 0x60, 0x68, 0x85, 0xC7, 0x37,
      0xEF, 0xD2, 0xB5, 0x99, 0x41, 0x33, 0x11, 0xEA, 0xAC, 0x70, 0x79, 0x39,
      0xB3, 0x75, 0x00, 0x69, 0x91, 0x31, 0xA4, 0x45, 0x35, 0xF2, 0x1C, 0x5A,
      0xE5, 0x96, 0x74, 0x1F, 0x76, 0x68, 0x52, 0x51, 0x08, 0xB4, 0xB7, 0xAF,
      0xBA, 0x81, 0x4F, 0xAC, 0x8A, 0xB0, 0x06, 0x3B, 0x6A, 0x90, 0x60, 0xCE,
      0xD9, 0x36, 0xCC, 0x6D, 0xA2, 0xCE, 0x41, 0x31, 0x69, 0x5A, 0x89, 0xC3,
      0x5F, 0x2B, 0xA2, 0xF3, 0x9A, 0x27, 0xD3, 0x92, 0x57, 0x75, 0xFA, 0x9F,
      0x43, 0x48, 0x6E, 0x4C, 0x95, 0xC1, 0x65, 0xA6, 0x66, 0xFC, 0x33, 0x05,
      0xAF, 0x30, 0xB4, 0x19, 0x61, 0x1D, 0x29, 0x17, 0x75, 0xE0, 0xF0, 0x8F,
      0x34, 0xA6, 0x5E, 0xFA, 0x14, 0x6E, 0x46, 0xD2, 0x07, 0x53, 0x3B, 0x90,
      0x8F, 0x74, 0x4B, 0xD2, 0x46, 0xA9, 0x4A, 0x4A, 0x35, 0x13, 0x77, 0x31,
      0xD0, 0x2A, 0xC4, 0x3E, 0x77, 0x9E, 0x26, 0x2A, 0x66, 0xF6, 0x68, 0x78,
      0x4B, 0x30, 0xB2, 0x31, 0xD8, 0x3E, 0x43, 0x69, 0x40, 0x02, 0x48, 0xAF,
      0x3E, 0xE2, 0x84, 0x32, 0x82, 0x1F, 0x07, 0xB5, 0x02, 0x07, 0x25, 0xC8,
      0xD7, 0x69, 0xB3, 0x05, 0xB3, 0xAF, 0xA6, 0x85, 0xA4, 0x2E, 0x28, 0xC4,
      0xF0, 0xE3, 0x5B, 0xF4, 0x07, 0x54, 0x93, 0x61, 0xA6, 0x7D, 0x7B, 0x66,
      0x99, 0xCA, 0x0F, 0x29, 0x3C, 0xCB, 0x77, 0x60, 0x19, 0x58, 0x57, 0x59,
      0x50, 0x27, 0x92, 0xF8, 0xD7, 0x6A, 0x36, 0x98, 0x87, 0x2F, 0x81, 0x7A,
      0x0C, 0x62, 0x10, 0x84, 0xE5, 0x36, 0x95, 0x70, 0x17, 0x95, 0xAB, 0xBE,
      0x16, 0xC4, 0x66, 0x01, 0x7B, 0xCC, 0x02, 0xB5, 0x18, 0xEA, 0x38, 0x71,
      0x03, 0xC5, 0x9D, 0x17, 0x12, 0x7B, 0x84, 0x43, 0x50, 0xAE, 0x42, 0x89,
      0x29, 0x81, 0x05, 0x59, 0xA0, 0x8B, 0xC9, 0x1C, 0x2A, 0x29, 0xDA, 0xC3,
      0xD6, 0xC1, 0x4D, 0xA0, 0x97, 0x9D, 0xCB, 0x42, 0x10, 0x14, 0x2C, 0x6C,
      0xD5, 0xB7, 0xCF, 0x18, 0xCB, 0x77, 0xE2, 0xE1, 0x30, 0x29, 0xC3, 0xC2,
      0x3D, 0x20, 0x89, 0xC2, 0x95, 0x41, 0x15, 0x60, 0x02, 0x4A, 0xC2, 0xAF,
      0x25, 0xB9, 0x4F, 0xBC, 0x14, 0x79, 0x66, 0x52, 0xCF, 0xD8, 0xA5, 0x24,
      0xB6, 0xAC, 0xB8, 0xD9, 0xA2, 0x62, 0xB7, 0xC2, 0x6A, 0x27, 0x9B, 0xBA,
      0x7D, 0x49, 0x95, 0xA9, 0x2A, 0x55, 0x00, 0xE0, 0x81, 0x86, 0x42, 0x00,
      0xBF, 0xB5, 0x1D, 0x46, 0x68, 0x6A, 0xE1, 0x41, 0x30, 0xE3, 0xC5, 0xA7,
      0x28, 0xFB, 0xB7, 0x69, 0x44, 0xBA, 0x65, 0x87, 0x18, 0xFC, 0x04, 0x1D,
      0xFD, 0x3A, 0x24, 0x80, 0xB9, 0xB6, 0x65, 0x8A, 0x9D, 0x59, 0x5B, 0xC4,
      0xCB, 0xDC, 0x10, 0x5B, 0xE0, 0x19, 0xE1, 0x28, 0x90, 0x99, 0x78, 0x24,
      0x0E, 0xA2, 0x9D, 0xA7, 0xC6, 0x66, 0x64, 0xE1, 0x71, 0x83, 0xE0, 0xB4,
      0x49, 0x69, 0xB2, 0x84, 0xDB, 0x06, 0xD4, 0x31, 0x17, 0x51, 0xDA, 0x4E,
      0xCC, 0x6C, 0xC7, 0x5C, 0x06, 0x39, 0x5D, 0x5B, 0x95, 0x37, 0x07, 0x8D,
      0x24, 0xE2, 0x09, 0x1A, 0xA4, 0x5A, 0x92, 0xD1, 0x83, 0x78, 0x41, 0x5F,
      0x11, 0x83, 0xC6, 0xB4, 0xE7, 0x54, 0x6A, 0x1A, 0x17, 0x92, 0xCC, 0x07,
      0x38, 0x41, 0x06, 0xA5, 0xD5, 0xC8, 0xB1, 0xA3, 0x69, 0xD3, 0xD6, 0xA8,
      0xC8, 0x3B, 0x92, 0x7B, 0x72, 0xC1, 0xFD, 0xC7, 0xCE, 0x27, 0x44, 0x9A,
      0x52, 0x28, 0xC8, 0x5B, 0xB6, 0xB0, 0xCF, 0xA8, 0x59, 0x54, 0xC0, 0xCE,
      0x5A, 0x5B, 0xB9, 0x47, 0xF6, 0x8C, 0x81, 0x07, 0xC1, 0xFF, 0x3B, 0x7D,
      0x3D, 0x49, 0x00, 0xFE, 0xE5, 0x92, 0x06, 0xB4, 0xCC, 0xAC, 0x5A, 0x1B,
      0x4E, 0x65, 0x46, 0x56, 0x09, 0x69, 0x2F, 0x76, 0x22, 0x7E, 0xEC, 0x07,
      0x21, 0xA5, 0x9B, 0x92, 0x26, 0x2D, 0xB0, 0xF7, 0x35, 0xE3, 0x91, 0x34,
      0x3D, 0xC5, 0x83, 0x6B, 0xBA, 0x77, 0x9D, 0x6A, 0x55, 0x8F, 0x8B, 0xC0,
      0x00, 0x13, 0x88, 0xE8, 0x36, 0x3E, 0x3C, 0xB6, 0x3C, 0xE4, 0x9C, 0x4C,
      0x76, 0x69, 0xC8, 0x2B, 0x2B, 0x65, 0x0B, 0x46, 0x11, 0xD0, 0x94, 0x70,
      0x75, 0x71, 0x06, 0x5B, 0x94, 0x3F, 0x21, 0x08, 0xBC, 0xA3, 0x37, 0x47,
      0x36, 0x7A, 0xB9, 0x53, 0xD9, 0x42, 0x3A, 0xFC, 0x56, 0x09, 0x59, 0x1B,
      0xF4, 0x9B, 0x8A, 0x99, 0x65, 0x0E, 0x4D, 0x80, 0x10, 0x61, 0x7C, 0xC5,
      0x86, 0x45, 0x08, 0x0D, 0xC0, 0xA1, 0x41, 0xC3, 0x4D, 0xE1, 0xD6, 0x9E,
      0x59, 0x32, 0x03, 0x2E, 0x7B, 0x1B, 0xAB, 0x0C, 0xB2, 0xA8, 0xBA, 0xC3,
      0x50, 0x6B, 0x7D, 0x5E, 0x71, 0x3D, 0xA7, 0x9C, 0xA4, 0xE1, 0x77, 0xA6,
      0xCB, 0x27, 0xA5, 0x45, 0xC9, 0xA8, 0x0B, 0x3A, 0x48, 0x99, 0x41, 0xA4,
      0x7A, 0xF8, 0x4F, 0x59, 0xF2, 0x92, 0xE3, 0x14, 0x30, 0x2A, 0xCB, 0x8E,
      0xF0, 0x00, 0x6F, 0x50, 0xA5, 0x39, 0xE3, 0x19, 0x95, 0x1F, 0x6C, 0xCE,
      0xE9, 0xF4, 0x78, 0x77, 0x3A, 0x8B, 0x0A, 0xE7, 0x3C, 0x14, 0xB7, 0x29,
      0xEF, 0x4C, 0x0B, 0x89, 0xA9, 0x9B, 0x87, 0xF4, 0xC9, 0xB8, 0xBA, 0xC7,
      0x35, 0xD3, 0x1B, 0xB8, 0x33, 0x34, 0x2B, 0xD5, 0x01, 0xEF, 0x45, 0x8F,
      0x95, 0x54, 0x96, 0x13, 0x8A, 0x6D, 0x07, 0xD1, 0x77, 0x7A, 0x94, 0x89,
      0xA2, 0x4C, 0x74, 0xA5, 0x79, 0x9A, 0x70, 0xC9, 0x42, 0xFC, 0x83, 0x9D,
      0x20, 0xA8, 0xC2, 0x28, 0xF7, 0x45, 0x3B, 0xC2, 0x9C, 0x02, 0xBB, 0xA3,
      0xB0, 0x82, 0x78, 0x01, 0x14, 0x3F, 0x67, 0x69, 0x1E, 0xC3, 0x48, 0x1F,
      0x96, 0x09, 0xBF, 0x79, 0xD9, 0xA8, 0xB7, 0xA1, 0xA7, 0xA6, 0x10, 0xB0,
      0x58, 0x56, 0xB5, 0xFC, 0x8C, 0x35, 0x21, 0x96, 0x8E, 0xD9, 0x69, 0x5A,
      0x00, 0xD7, 0x1F, 0xE8, 0xC3, 0x90, 0xC6, 0x0A, 0x59, 0xD6, 0x73, 0x4C,
      0x60, 0x8B, 0x7A, 0xC0, 0xB4, 0x64, 0x3F, 0x7B, 0xA1, 0xDF, 0xD0, 0x5B,
      0x5B, 0xD8, 0x53, 0xC9, 0x43, 0x22, 0x69, 0xA9, 0x55, 0x5E, 0x39, 0x12,
      0xE9, 0xB2, 0x63, 0xC7, 0xB9, 0x39, 0x38, 0x4A, 0x17, 0x94, 0xA5, 0x0F,
      0x86, 0x88, 0x29, 0x68, 0x69, 0xAA, 0xDB, 0x4B, 0x85, 0x30, 0x91, 0xA2,
      0x91, 0xE4, 0x2F, 0x48, 0x5A, 0x6F, 0x93, 0x54, 0x7E, 0x03, 0xBC, 0x1B,
      0x57, 0xA6, 0x03, 0xC8, 0x1B, 0x78, 0x97, 0x19, 0x8D, 0xC5, 0x92, 0x52,
      0xF9, 0x80, 0x5A, 0x62, 0x66, 0x43, 0x5E, 0xB2, 0xA2, 0x6B, 0x63, 0x00,
      0xD2, 0x26, 0x67, 0xA8, 0x78, 0xC3, 0x40, 0x18, 0x00, 0xE6, 0x61, 0x2C,
      0x02, 0x6C, 0x4F, 0x0F, 0xF9, 0x9C, 0x88, 0x95, 0x31, 0xD6, 0x37, 0x03,
      0x61, 0x26, 0x22, 0x7B, 0x67, 0x4B, 0x95, 0xA3, 0x8A, 0x2A, 0x93, 0x49,
      0x7F, 0xF8, 0x3C, 0x8D, 0x31, 0x43, 0xA5, 0x39, 0x8B, 0xE9, 0xC5, 0x99,
      0x09, 0x80, 0x0B, 0x02, 0xC6, 0x77, 0xB2, 0x7A, 0x42, 0x62, 0x1C, 0x19,
      0x0D, 0x86, 0x5A, 0xFA, 0xC0, 0x55, 0x13, 0xF7, 0x27, 0x58, 0xB4, 0x94,
      0x58, 0x54, 0x35, 0xF2, 0x35, 0x7B, 0x97, 0x34, 0x2D, 0x95, 0x1A, 0x2A,
      0xB2, 0x3A, 0x1C, 0xF8, 0xA2, 0x29, 0xBE, 0x90, 0x94, 0x87, 0xBB, 0x2B,
      0x8F, 0x52, 0x1B, 0x09, 0xE0, 0xC4, 0x84, 0x96, 0x32, 0xBF, 0xCC, 0x82,
      0x1C, 0xE3, 0x00, 0x25, 0xB8, 0x37, 0xA4, 0x55, 0xB2, 0xD7, 0xD5, 0x8E,
      0xE4, 0xB0, 0xAA, 0xE1, 0xA2, 0x5F, 0x8A, 0x56, 0x93, 0xF6, 0x2B, 0x1A,
      0xB7, 0x7C, 0x22, 0x98, 0x90, 0x89, 0x92, 0x64, 0xBF, 0x63, 0x18, 0x9A,
      0xBB, 0xCC, 0x80, 0xAD, 0x1B, 0x8A, 0xDF, 0xDB, 0x21, 0xB0, 0xC2, 0x48,
      0x13, 0x42, 0xA1, 0x37, 0xFC, 0xAE, 0x8A, 0x64, 0xB1, 0xE2, 0x1C, 0x80,
      0x5B, 0x18, 0x7A, 0xB7, 0xC1, 0xB6, 0x37, 0xD5, 0x7F, 0xCD, 0x88, 0x11,
      0xE4, 0x9C, 0x1D, 0x2A, 0x06, 0x58, 0x48, 0xA7, 0x69, 0xB7, 0xF0, 0x2D,
      0x99, 0xE4, 0x0F, 0x4B, 0xE3, 0x78, 0x3D, 0xE3, 0xAE, 0x4F, 0xE9, 0x7E,
      0x23, 0xCA, 0x71, 0x6A, 0xFC, 0x08, 0x14, 0xC9, 0x35, 0x29, 0x36, 0x41,
      0xD7, 0xC4, 0x0E, 0xA1, 0x08, 0x8E, 0xE8, 0x9C, 0x2A, 0x43, 0x50, 0x52,
      0x37, 0xA5, 0x93, 0x56, 0x5A, 0x05, 0x06, 0x50, 0x81, 0xF6, 0x18, 0x1F,
      0x35, 0xC5, 0x53, 0x38, 0xC4, 0x27, 0xCA, 0x62, 0x87, 0x27, 0xDA, 0xAF,
      0x8F, 0x5B, 0x53, 0x22, 0xE3, 0x44, 0x88, 0x90, 0x49, 0x49, 0xE4, 0x5C,
      0x61, 0xBB, 0x91, 0x55, 0x25, 0x67, 0x6E, 0xD2, 0x65, 0x9E, 0xFC, 0x97,
      0xC6, 0xA5, 0x33, 0x76, 0x47, 0x8B, 0x62, 0x9F, 0xB3, 0x2D, 0x49, 0x04,
      0x74, 0x12, 0xA4, 0x9E, 0x98, 0xF1, 0x86, 0x56, 0x4A, 0x36, 0xEE, 0xF1,
      0xCA, 0x49, 0x20, 0xC9, 0x12, 0xB1, 0x21, 0x1B},
     true,
     {0x8A, 0xDB, 0xFA, 0xC1, 0xF7, 0xE6, 0xCF, 0x80, 0x3D, 0xDB, 0x3D,
      0xF0, 0x96, 0xCA, 0x36, 0x0E, 0x3B, 0xED, 0xBC, 0xE6, 0xF4, 0x7F,
      0x14, 0x49, 0x34, 0x05, 0x02, 0x0A, 0x61, 0xCC, 0x95, 0xAF},
     {0x69, 0xB8, 0xF0, 0x91, 0xA4, 0x50, 0x89, 0x0C, 0x0D, 0xCC, 0xE0,
      0x12, 0x0E, 0x9B, 0xAB, 0x05, 0x05, 0x4C, 0x77, 0x85, 0xA7, 0x97,
      0xC9, 0x3B, 0x6F, 0xA3, 0x9F, 0xF5, 0xE0, 0xBC, 0x5A, 0x70}},

    {params_ml_kem768_test_mode,
     {0x7D, 0xB1, 0x8C, 0xA3, 0x5A, 0x53, 0xAB, 0x3A, 0x65, 0xE4, 0xC1,
      0x7F, 0xA0, 0x96, 0xDD, 0xEC, 0xB1, 0x9F, 0xC7, 0x74, 0x7E, 0x65,
      0x7B, 0x49, 0xD1, 0xC1, 0x71, 0x0D, 0xBD, 0x1D, 0x19, 0x7B},
     {0xCB, 0x24, 0x68, 0xA0, 0x18, 0x55, 0x67, 0xF8, 0xA6, 0x0A, 0xDB, 0x33,
      0xCA, 0x52, 0x39, 0xC1, 0x1C, 0x4A, 0x3E, 0x0C, 0x03, 0x1D, 0x38, 0x5D,
      0xCF, 0xA2, 0x8C, 0x3A, 0xE2, 0xA9, 0xF7, 0x19, 0x04, 0xBF, 0x37, 0x9C,
      0xB9, 0xE0, 0xBE, 0xDE, 0xAC, 0x82, 0xB2, 0xA5, 0x37, 0x35, 0x7A, 0x9C,
      0x3A, 0xD3, 0x33, 0x62, 0x60, 0x2A, 0x1C, 0xF6, 0x45, 0x8A, 0x74, 0x5A,
      0xBE, 0xD9, 0x23, 0x3A, 0x09, 0x2B, 0x96, 0x2B, 0xBA, 0x2B, 0x0A, 0x66,
      0xDD, 0x4A, 0x85, 0xDA, 0xE1, 0x1B, 0x0C, 0x28, 0x23, 0x0A, 0xA4, 0x4C,
      0x40, 0xF2, 0xB6, 0x86, 0x87, 0xB7, 0xD5, 0x48, 0x33, 0x06, 0x2C, 0xBB,
      0x5B, 0x02, 0x33, 0xE2, 0x54, 0x96, 0xD1, 0xC8, 0x42, 0x04, 0xB2, 0xBA,
      0xB0, 0x60, 0x50, 0xC0, 0x0C, 0x30, 0x8E, 0xF5, 0x3D, 0xD8, 0x34, 0x51,
      0x43, 0xBA, 0x81, 0x0A, 0xAC, 0x47, 0x7C, 0x81, 0x19, 0xB1, 0xC5, 0x95,
      0xD9, 0x64, 0xB1, 0x3F, 0x83, 0x78, 0x49, 0xB5, 0x8D, 0x8E, 0x1B, 0xCE,
      0x56, 0x43, 0x7F, 0x2E, 0xCA, 0x84, 0xB8, 0x6C, 0x91, 0x17, 0x36, 0x70,
      0xE0, 0x99, 0x5D, 0x97, 0x69, 0x64, 0x2B, 0x0A, 0xB0, 0xBB, 0x07, 0x13,
      0xD3, 0x13, 0xAE, 0xB7, 0xC4, 0x1C, 0x9B, 0x2C, 0xC4, 0x11, 0xB3, 0xB6,
      0x2B, 0x11, 0x03, 0x08, 0xD9, 0x4F, 0x0D, 0xA1, 0xBB, 0x69, 0xD4, 0x06,
      0xDF, 0xE7, 0x28, 0x65, 0x19, 0x69, 0x25, 0x02, 0xBC, 0x83, 0xE1, 0x5B,
      0xD5, 0x04, 0x94, 0xC1, 0x04, 0x79, 0x80, 0xAC, 0x60, 0x43, 0xB1, 0x8D,
      0x7C, 0xB7, 0x2E, 0xEC, 0xB0, 0x0E, 0xED, 0xA5, 0x15, 0xC9, 0x7C, 0x0E,
      0xD0, 0x8B, 0x5D, 0x4B, 0xB0, 0x01, 0xBB, 0xF0, 0x8D, 0x98, 0x21, 0xB9,
      0xA7, 0x5C, 0x02, 0x66, 0x6C, 0x35, 0x7F, 0x00, 0x27, 0x82, 0x79, 0x34,
      0x84, 0x62, 0x75, 0x9A, 0x60, 0x23, 0x59, 0xF5, 0xA9, 0x53, 0x24, 0x7C,
      0x19, 0x28, 0x17, 0x2A, 0x01, 0x3E, 0x3C, 0x53, 0xB7, 0xE9, 0x5B, 0x81,
      0xA6, 0x4A, 0x6D, 0xB3, 0xAA, 0x86, 0xCD, 0x76, 0x70, 0xCF, 0x7C, 0x38,
      0xAF, 0x35, 0x7A, 0x31, 0x7B, 0x71, 0x67, 0x1B, 0x95, 0x0E, 0x2B, 0xB8,
      0x5E, 0xAA, 0x61, 0x38, 0xA5, 0xAD, 0x93, 0xA1, 0x64, 0x06, 0x18, 0xBD,
      0x98, 0x09, 0x2D, 0x85, 0xD3, 0x60, 0x15, 0xC0, 0xB0, 0xDF, 0xF6, 0x05,
      0x9F, 0x3B, 0x09, 0xAE, 0x50, 0x07, 0xF8, 0x9A, 0xB5, 0x51, 0x23, 0x0D,
      0x8B, 0x9A, 0x2B, 0xCA, 0x59, 0x23, 0x84, 0x05, 0x37, 0x2B, 0xAB, 0x7F,
      0x4D, 0x80, 0x69, 0xFF, 0x74, 0x57, 0xEF, 0x37, 0x20, 0xA1, 0x71, 0xC3,
      0xA6, 0x31, 0x2C, 0xC8, 0xD6, 0x91, 0x91, 0xCA, 0x90, 0x9A, 0x13, 0x94,
      0x6C, 0xF8, 0x57, 0x7A, 0x96, 0x08, 0x68, 0x93, 0xC0, 0xC5, 0x9B, 0x02,
      0x6A, 0x21, 0x83, 0x55, 0x43, 0xB8, 0x58, 0x3A, 0x05, 0xF5, 0x69, 0x97,
      0x7E, 0x07, 0x22, 0x99, 0xBA, 0x58, 0x0C, 0x17, 0x9E, 0xA2, 0x97, 0x68,
      0x10, 0xAA, 0xAA, 0x93, 0xF5, 0x7D, 0xE2, 0x23, 0xB9, 0x7C, 0xBB, 0x4C,
      0xBE, 0x55, 0x9E, 0x18, 0x13, 0x93, 0x72, 0xEB, 0x1A, 0x20, 0x95, 0xB8,
      0xB5, 0x09, 0xAE, 0x2C, 0xEA, 0x5B, 0x49, 0x44, 0x33, 0x5E, 0xD3, 0x85,
      0xB2, 0x3B, 0x5E, 0xB7, 0xD0, 0x84, 0x7A, 0x55, 0x7A, 0xAA, 0xE8, 0x96,
      0x11, 0x93, 0x74, 0x95, 0xC8, 0x25, 0xB7, 0x23, 0x62, 0x80, 0xAB, 0xB6,
      0xD4, 0x09, 0xCD, 0x51, 0x3A, 0x03, 0xC9, 0x11, 0x39, 0xA2, 0x51, 0xB3,
      0x27, 0x8C, 0x92, 0x9D, 0xB2, 0x3B, 0xD8, 0x15, 0xBF, 0x68, 0x00, 0x7D,
      0x50, 0xB8, 0x35, 0x6C, 0x03, 0xBE, 0x88, 0xF8, 0x6E, 0x41, 0x0A, 0xBF,
      0xF6, 0xCC, 0xB3, 0xB1, 0x1B, 0xC5, 0xD4, 0x1C, 0x78, 0x39, 0xD4, 0x8F,
      0x36, 0x5A, 0xB4, 0xE9, 0xE2, 0x9F, 0x0B, 0x00, 0x94, 0xA4, 0xC4, 0xAB,
      0xBD, 0x67, 0x61, 0x52, 0x78, 0x50, 0x28, 0x64, 0x32, 0x53, 0x02, 0x41,
      0x96, 0x81, 0x22, 0xD2, 0xBC, 0x87, 0x7D, 0xF5, 0x78, 0x1D, 0x17, 0xAE,
      0x12, 0x09, 0x5C, 0x17, 0x7B, 0x69, 0xBE, 0x3B, 0x98, 0x9E, 0x81, 0xCB,
      0x53, 0x9A, 0x2F, 0xF2, 0xE2, 0x8D, 0x62, 0x33, 0x03, 0xFE, 0xD5, 0x75,
      0x0C, 0x58, 0xB9, 0xCE, 0x05, 0x1A, 0x6B, 0x81, 0x38, 0x99, 0xBB, 0x3C,
      0x74, 0xD8, 0x2B, 0x2F, 0x12, 0x7E, 0x51, 0x03, 0x0B, 0xCE, 0xB3, 0x96,
      0xEF, 0x54, 0x7E, 0xD3, 0x7A, 0xCA, 0xF1, 0xD4, 0x6A, 0x77, 0xF7, 0xB2,
      0xB2, 0x4B, 0x48, 0xAD, 0x39, 0x9B, 0x53, 0xFA, 0xA6, 0x9C, 0x07, 0x9F,
      0xFE, 0xFC, 0xB9, 0xF8, 0x36, 0x7F, 0x7C, 0x51, 0x3D, 0x14, 0x2C, 0x83,
      0x92, 0xA8, 0xBE, 0x60, 0x89, 0xCC, 0x23, 0x01, 0x68, 0x54, 0x60, 0xBC,
      0xCB, 0x19, 0xBF, 0xD6, 0xAC, 0x82, 0x1D, 0xD8, 0x40, 0xA3, 0xDA, 0x30,
      0xC3, 0x20, 0x86, 0x68, 0xC7, 0x10, 0x3E, 0xAB, 0x78, 0xC6, 0x52, 0x0C,
      0x12, 0x91, 0x23, 0x9D, 0xF8, 0x21, 0x7C, 0x25, 0x45, 0x0B, 0x70, 0x30,
      0x20, 0x20, 0xA4, 0xBF, 0xF3, 0x84, 0xF0, 0x61, 0x91, 0x42, 0xA0, 0x4C,
      0x76, 0x9C, 0x9B, 0x18, 0xD2, 0x7E, 0x42, 0xFA, 0x30, 0xBC, 0xE6, 0x0A,
      0x1E, 0xC2, 0xAF, 0xB6, 0x18, 0x75, 0x2A, 0x39, 0x17, 0xEE, 0xDC, 0x37,
      0xBD, 0x15, 0xC4, 0x4D, 0x0A, 0xBA, 0x1F, 0xA3, 0xA4, 0x0C, 0x23, 0xAA,
      0x14, 0xC9, 0x8E, 0x01, 0x6A, 0x59, 0x2B, 0x45, 0x9B, 0xF4, 0xC1, 0x3C,
      0x07, 0x35, 0x4A, 0x1D, 0xE0, 0x53, 0x94, 0x45, 0x88, 0x2E, 0x21, 0xB9,
      0x7B, 0x17, 0x67, 0x01, 0x58, 0x10, 0x32, 0x50, 0x86, 0xAE, 0x71, 0xCD,
      0xC0, 0xB5, 0xC0, 0xD7, 0x28, 0x7B, 0xBD, 0x99, 0xB3, 0x81, 0x68, 0x0B,
      0xD6, 0x55, 0x9B, 0x04, 0xFB, 0x84, 0x83, 0x5C, 0x44, 0x19, 0x31, 0x6F,
      0xCD, 0x22, 0x3B, 0x84, 0xD0, 0x38, 0x16, 0xD8, 0x6C, 0x73, 0x83, 0x34,
      0xFD, 0xC8, 0x94, 0xBC, 0x81, 0xB1, 0xE2, 0x68, 0x13, 0xA2, 0x15, 0x9D,
      0x42, 0x6A, 0xB1, 0xFB, 0x70, 0xA8, 0x8F, 0xFC, 0x17, 0x82, 0xE6, 0x49,
      0xBD, 0x86, 0x0D, 0x14, 0x8B, 0x33, 0xF7, 0x60, 0x70, 0x83, 0xA5, 0x92,
      0x8F, 0x1C, 0x83, 0x5F, 0x88, 0x0D, 0xC9, 0xE3, 0x23, 0x5E, 0xA5, 0x1E,
      0x78, 0xA4, 0x03, 0xF7, 0xBA, 0x72, 0x6D, 0x17, 0x95, 0x1C, 0xB7, 0xAE,
      0x76, 0x30, 0xAA, 0x73, 0x94, 0xA8, 0x0D, 0x95, 0xA1, 0xED, 0xB1, 0x05,
      0x9F, 0x14, 0x0F, 0x3E, 0xBC, 0x8D, 0x1C, 0x9B, 0xA9, 0xF4, 0x3B, 0x75,
      0xED, 0x90, 0x53, 0x8C, 0xB2, 0x19, 0xF8, 0xE4, 0xA2, 0xB2, 0x02, 0x08,
      0x4E, 0x04, 0x94, 0x73, 0xD5, 0x7D, 0x98, 0x7C, 0xAA, 0xD2, 0xB5, 0x1D,
      0xB0, 0x09, 0xCF, 0x8A, 0x53, 0x8F, 0x82, 0x05, 0xB0, 0xB4, 0x04, 0x9F,
      0x41, 0xEC, 0xB3, 0x44, 0xF8, 0x4C, 0x53, 0x8C, 0xA4, 0x9F, 0x5A, 0x8F,
      0xF3, 0x09, 0xD0, 0x36, 0xC6, 0xC5, 0x20, 0x93, 0x2F, 0x08, 0x07, 0x2F,
      0x56, 0x78, 0xCA, 0x68, 0x56, 0x8A, 0x41, 0x33, 0x0D, 0x9B, 0xEB, 0x34,
      0xBD, 0x30, 0x8F, 0x96, 0x32, 0x24, 0xF8, 0x0B, 0x5D, 0xB9, 0xB1, 0x0E,
      0x54, 0xD1, 0x46, 0xCF, 0xAB, 0x5A, 0xEF, 0x84, 0xA7, 0xF9, 0xC6, 0x2B,
      0x7B, 0xC2, 0x4A, 0x26, 0xA5, 0x78, 0xFC, 0xC5, 0x1E, 0x36, 0x73, 0x8B,
      0xDC, 0xC8, 0xAD, 0x24, 0x41, 0x01, 0x55, 0xAA, 0xB7, 0x4D, 0x86, 0x91,
      0xC1, 0xE6, 0x99, 0xE3, 0x72, 0x24, 0x81, 0xBA, 0x07, 0x4E, 0xEC, 0x75,
      0xDA, 0xFB, 0x72, 0x38, 0x63, 0xC3, 0x08, 0xF6, 0x0E, 0x10, 0xD1, 0x95,
      0x61, 0x7B, 0xCD, 0xB8, 0x77, 0x15, 0x3D, 0x9C, 0xAA, 0x77, 0x56, 0x47,
      0xA8, 0x3B, 0x1D, 0xE1, 0x13, 0x01, 0x9C, 0xD6, 0xC3, 0xA7, 0xCA, 0x38,
      0x55, 0x40, 0x23, 0xA6, 0xBC, 0x7E, 0xEC, 0x59, 0x4F, 0xA6, 0xC1, 0x4D,
      0xA8, 0x35, 0x37, 0x37, 0xF0, 0xA5, 0x49, 0x71, 0x5E, 0x39, 0xC4, 0x9F,
      0xD9, 0x62, 0x5F, 0x18, 0x7C, 0x3B, 0xE1, 0x60, 0x2B, 0x6B, 0xA1, 0x78,
      0xD1, 0x78, 0x4B, 0x52, 0x69, 0x0B, 0x1E, 0x13, 0x80, 0x84, 0x72, 0x03,
      0xC1, 0x36, 0x24, 0x41, 0x8C, 0x1C, 0x1D, 0xAA, 0x0B, 0x23, 0x1A, 0xC0,
      0xFB, 0x39, 0x29, 0x38, 0x43, 0xCC, 0x3D, 0x6B, 0x48, 0xC3, 0x2B, 0x15,
      0x09, 0x87, 0x48, 0xDB, 0x0B, 0x36, 0x72, 0x37, 0x74, 0x07, 0xEB, 0x74,
      0x41, 0xB8, 0x23, 0x71, 0xB5, 0x6E, 0xB3, 0xE9, 0x0C, 0x98, 0x3A, 0x89,
      0x5A, 0xF8, 0x5D, 0x57, 0xE7, 0x6C, 0x53, 0x08, 0x8D, 0x94, 0x48, 0x40,
      0xCC, 0x30, 0x98, 0x53, 0x81, 0x42, 0x66, 0xD6, 0x6D, 0xCE, 0x88, 0x91,
      0x50, 0x49, 0x57, 0x9C, 0xC4, 0x5C, 0xD6, 0x02},
     true,
     {0xCA, 0xA0, 0xA9, 0x34, 0xD1, 0xDE, 0x75, 0xDF, 0x05, 0xE4, 0x5F,
      0xCD, 0x4E, 0x39, 0xC9, 0xF1, 0x49, 0xA9, 0xA5, 0xE4, 0x61, 0x7D,
      0x0A, 0xFF, 0x6D, 0x13, 0x25, 0x85, 0xE4, 0x26, 0xFE, 0x5D},
     {0xC2, 0x1C, 0x8C, 0x4B, 0x59, 0x90, 0x6D, 0x0C, 0x4A, 0xDB, 0x1F,
      0x3C, 0xAF, 0x47, 0xF9, 0xEB, 0x32, 0x6B, 0x8A, 0x62, 0xB3, 0x39,
      0x24, 0x07, 0x21, 0x1D, 0x50, 0x2F, 0x40, 0xC7, 0xE0, 0x7A}},

    {params_ml_kem768_test_mode,
     {0x87, 0x6B, 0x17, 0x26, 0x3B, 0x40, 0x91, 0x71, 0xB7, 0x46, 0xC6,
      0x93, 0x6E, 0xC6, 0x5F, 0xC9, 0x41, 0x37, 0xF9, 0x58, 0xDC, 0x97,
      0x4B, 0xF9, 0x81, 0x10, 0xA1, 0xD0, 0x7F, 0x6D, 0x95, 0xF9},
     {0x0F, 0x61, 0x3B, 0x04, 0x12, 0x8F, 0x82, 0xA7, 0x38, 0x67, 0xD9, 0x18,
      0x58, 0x91, 0xC2, 0x9D, 0x6C, 0x3E, 0x13, 0x81, 0x84, 0x3B, 0xD5, 0x02,
      0xD8, 0x60, 0x99, 0xA7, 0x40, 0xBA, 0xD5, 0xBA, 0xC6, 0x8C, 0x59, 0x05,
      0x10, 0xCA, 0x3F, 0x6F, 0x2B, 0x54, 0x63, 0xB2, 0x64, 0xBE, 0xD3, 0x49,
      0x52, 0xF1, 0x0C, 0x78, 0x4A, 0x92, 0xAB, 0x76, 0x96, 0x26, 0x84, 0x10,
      0xF1, 0xF2, 0x8C, 0x06, 0xDA, 0x7A, 0x18, 0x41, 0x6A, 0x7B, 0x1B, 0x5F,
      0xD2, 0x33, 0x93, 0xC3, 0x35, 0x92, 0x24, 0x8C, 0x9A, 0x8B, 0x39, 0x56,
      0xA9, 0x99, 0x48, 0x3E, 0x00, 0x0F, 0x2A, 0x2C, 0x6F, 0x79, 0x60, 0x52,
      0xFB, 0xB2, 0x2F, 0x7F, 0x18, 0x2A, 0x19, 0x16, 0x02, 0xFD, 0x93, 0xAC,
      0x06, 0x63, 0x55, 0xB7, 0x1B, 0x7E, 0x6B, 0xE3, 0x6E, 0x53, 0x1B, 0x1D,
      0x34, 0xF0, 0x38, 0x2E, 0x34, 0xA4, 0xCF, 0x62, 0x3B, 0x7B, 0x11, 0x27,
      0x51, 0x9A, 0x7B, 0xC4, 0xEA, 0x3F, 0xB0, 0xD1, 0xC9, 0x16, 0x26, 0xA4,
      0x17, 0xB6, 0x12, 0x9D, 0xEC, 0xAF, 0xF2, 0x86, 0x52, 0x73, 0xF7, 0x59,
      0xB4, 0xDA, 0x1A, 0x95, 0xF7, 0x9F, 0xAD, 0x0C, 0xAB, 0x09, 0xFB, 0x61,
      0xD3, 0x4B, 0x9A, 0xD7, 0x8B, 0x80, 0x46, 0xF5, 0x60, 0x1F, 0xB5, 0x3D,
      0x28, 0x95, 0x1A, 0xB7, 0x38, 0x42, 0xB8, 0x92, 0x1B, 0x2F, 0xF1, 0x04,
      0x17, 0xDB, 0x4F, 0x99, 0x64, 0xA4, 0x1E, 0xE8, 0x20, 0xFB, 0xA8, 0x3D,
      0x61, 0xD3, 0x0D, 0x0E, 0xF2, 0xC2, 0xCB, 0xCB, 0x1F, 0x6C, 0xCB, 0x9E,
      0x77, 0x52, 0x3D, 0xCC, 0xBC, 0x37, 0xAD, 0xB8, 0xB9, 0xE3, 0x1A, 0x0F,
      0x0E, 0x1C, 0x41, 0x92, 0x91, 0x10, 0x60, 0xE8, 0x67, 0x71, 0x40, 0x69,
      0x0E, 0xC6, 0x71, 0xA5, 0x44, 0x5F, 0x42, 0xFB, 0x1A, 0x68, 0xDB, 0x4D,
      0x16, 0x78, 0xBF, 0x9B, 0x60, 0x12, 0x9D, 0x98, 0xA8, 0x59, 0x83, 0x75,
      0x99, 0xAB, 0xA0, 0xDF, 0x46, 0x5D, 0xAE, 0x76, 0x97, 0x29, 0x46, 0xC7,
      0x3E, 0x83, 0x43, 0xA1, 0x9A, 0x3C, 0x03, 0x65, 0x78, 0x06, 0x57, 0x4F,
      0x59, 0xD2, 0x61, 0x11, 0x63, 0x33, 0x4F, 0xDB, 0x0E, 0xE8, 0xB0, 0xC1,
      0x3C, 0x67, 0x9C, 0x6D, 0x17, 0x5C, 0x22, 0x80, 0x7C, 0x86, 0xF0, 0xC1,
      0x99, 0xC8, 0x9C, 0xC4, 0x3C, 0x0D, 0xBE, 0x65, 0x87, 0xF0, 0xA3, 0x61,
      0x99, 0x14, 0x3E, 0xAE, 0x30, 0x11, 0x6B, 0x3B, 0x0D, 0x49, 0x83, 0x9A,
      0xA1, 0x8C, 0xA4, 0x9E, 0x29, 0x92, 0x74, 0x0B, 0x5D, 0xBF, 0x1C, 0x91,
      0xAD, 0xA3, 0x52, 0xD3, 0x9A, 0xB7, 0xD0, 0xC2, 0x3F, 0xCC, 0xFC, 0x41,
      0xE7, 0x83, 0xCA, 0x0A, 0x33, 0x3F, 0xFE, 0x00, 0x07, 0x4E, 0x72, 0xBC,
      0x83, 0x46, 0x69, 0x93, 0x16, 0x30, 0x89, 0x8B, 0x71, 0x8C, 0xD5, 0x30,
      0x4E, 0x25, 0x30, 0x71, 0xE7, 0x30, 0xF1, 0xEB, 0x06, 0x7F, 0x94, 0x86,
      0x1D, 0xD9, 0x8F, 0xD9, 0xA2, 0x62, 0xFA, 0xC9, 0x19, 0xFE, 0x87, 0x0E,
      0x3D, 0x21, 0xBC, 0xFB, 0xF6, 0x71, 0x80, 0xC5, 0x7A, 0x5D, 0x27, 0x97,
      0xF6, 0xC7, 0xB9, 0x6F, 0x54, 0x4C, 0xDF, 0x92, 0xC3, 0xFA, 0x8B, 0x49,
      0xEB, 0x36, 0x6C, 0x02, 0x88, 0x51, 0x40, 0xD1, 0x28, 0xB8, 0x82, 0x25,
      0x75, 0x07, 0x33, 0xE6, 0x10, 0x58, 0x68, 0xAC, 0x48, 0xA4, 0x68, 0x00,
      0x2F, 0x35, 0xC3, 0x4C, 0x38, 0xAD, 0x70, 0xA7, 0xBE, 0xC9, 0xC3, 0x77,
      0x13, 0xB9, 0x8D, 0x9E, 0xA8, 0xA7, 0x16, 0xCB, 0xC8, 0x5D, 0x0C, 0x3E,
      0xF5, 0xB4, 0xBF, 0x8F, 0x20, 0xBC, 0x6B, 0xBC, 0x61, 0xCE, 0x81, 0x50,
      0xB4, 0xF8, 0x42, 0xCE, 0xE0, 0xA4, 0x0E, 0x0A, 0x7F, 0xBA, 0xDB, 0x76,
      0xBE, 0x9B, 0x5C, 0xF2, 0xB3, 0x97, 0x49, 0xF5, 0x1B, 0xEE, 0xD8, 0x40,
      0x08, 0x22, 0x04, 0x4E, 0x2C, 0xB0, 0x1C, 0x02, 0x1B, 0x0D, 0x9B, 0x7F,
      0xE6, 0x7A, 0x8A, 0xFF, 0x9B, 0x22, 0x7C, 0x12, 0x0F, 0x64, 0x3B, 0x85,
      0x15, 0x2B, 0x40, 0xEC, 0x4A, 0x17, 0x34, 0xE2, 0xA1, 0x41, 0x15, 0x9A,
      0x5A, 0x96, 0xC8, 0xB7, 0x41, 0x15, 0xA0, 0xC9, 0x2E, 0x6B, 0x91, 0x34,
      0x64, 0x99, 0x4E, 0x98, 0x0A, 0x8A, 0x30, 0x41, 0x94, 0xA5, 0x95, 0xBA,
      0xB8, 0x92, 0x80, 0x1C, 0x15, 0xCB, 0xC5, 0x03, 0x3F, 0xFA, 0xE7, 0x0F,
      0x36, 0x8A, 0x62, 0xC6, 0x5A, 0x59, 0x87, 0x73, 0xC1, 0x99, 0xE2, 0x0E,
      0xC0, 0x7B, 0x73, 0xB1, 0xAC, 0xAA, 0xF4, 0x8C, 0xC9, 0x63, 0xA6, 0x98,
      0x6D, 0x9A, 0x3D, 0x2D, 0x23, 0x63, 0x52, 0xF6, 0x60, 0x65, 0x02, 0x57,
      0x51, 0xB9, 0x88, 0x4D, 0x7C, 0xBE, 0x3A, 0x55, 0x38, 0xA9, 0xB9, 0x03,
      0x72, 0xD4, 0xC2, 0x27, 0x97, 0x65, 0x9B, 0x5C, 0xA8, 0xD3, 0x57, 0xCF,
      0x00, 0xB2, 0x7C, 0x57, 0x14, 0x34, 0x43, 0x74, 0xA5, 0x13, 0x1B, 0x26,
      0x1D, 0x94, 0x8D, 0xE2, 0x23, 0x90, 0xD0, 0x52, 0x1F, 0x50, 0x55, 0x3E,
      0x78, 0x22, 0x33, 0x88, 0x90, 0x3C, 0xCF, 0x24, 0x1F, 0xD6, 0x08, 0xCB,
      0x4F, 0xC1, 0xAC, 0x7C, 0x44, 0xA2, 0xD3, 0x14, 0x1F, 0x0B, 0x99, 0x8B,
      0x99, 0xE7, 0x04, 0xAB, 0xEA, 0x1E, 0x63, 0x4C, 0x1F, 0x44, 0x23, 0x1D,
      0x34, 0x72, 0x54, 0xDC, 0x2C, 0xCD, 0xDC, 0x11, 0x4A, 0xC4, 0x48, 0x8A,
      0x54, 0x9B, 0xC4, 0x69, 0x64, 0x00, 0xF0, 0xBA, 0xBA, 0x18, 0x19, 0x8F,
      0x46, 0xE8, 0xCE, 0x43, 0xC9, 0x35, 0x81, 0xFB, 0x66, 0x1C, 0x74, 0x5A,
      0xB5, 0xA5, 0x52, 0x0D, 0x44, 0x50, 0x98, 0x25, 0x69, 0xBA, 0x1B, 0x19,
      0x2F, 0xEA, 0xCF, 0x37, 0xEA, 0x8E, 0xC8, 0xB8, 0xB2, 0x67, 0x27, 0x1A,
      0x04, 0x20, 0x22, 0xDF, 0xA1, 0x77, 0xA7, 0xC1, 0xCC, 0x1C, 0xF6, 0x76,
      0x54, 0x92, 0xAE, 0xC5, 0x97, 0x29, 0xD6, 0xF6, 0x16, 0x90, 0xF2, 0x7B,
      0x5D, 0x01, 0x0F, 0x22, 0xC3, 0x0E, 0x44, 0xD1, 0x58, 0x64, 0x39, 0x5B,
      0xF0, 0x17, 0x71, 0x15, 0x8B, 0xCC, 0x5F, 0x8B, 0x48, 0xF6, 0x56, 0xCE,
      0x5D, 0x90, 0x49, 0xD6, 0xF1, 0xC1, 0x1C, 0xDA, 0x96, 0xF8, 0xC7, 0x24,
      0xE0, 0x46, 0x0B, 0xDA, 0xAA, 0x1D, 0xD9, 0xCC, 0xAF, 0x0C, 0x46, 0x16,
      0x2B, 0x94, 0x34, 0x61, 0xB7, 0x29, 0xAF, 0x35, 0x43, 0xBC, 0x9A, 0x8C,
      0xE3, 0xB3, 0x49, 0x71, 0xB3, 0x9D, 0xD2, 0x5A, 0x60, 0x51, 0xD4, 0x1A,
      0x24, 0x8B, 0xBC, 0xD5, 0x55, 0x5D, 0x7E, 0x20, 0x28, 0xA5, 0xB4, 0x53,
      0xFC, 0x40, 0x62, 0xDA, 0xB1, 0x33, 0x7B, 0x13, 0x40, 0x09, 0xC7, 0x1B,
      0xDD, 0xD7, 0x05, 0xC2, 0xE4, 0x4C, 0x64, 0xE1, 0x5C, 0x18, 0xF2, 0x63,
      0x61, 0xD8, 0xA7, 0xA2, 0xF7, 0xA1, 0x3D, 0x3C, 0x31, 0xCB, 0x31, 0x47,
      0xED, 0x77, 0x70, 0x24, 0xB8, 0xA9, 0x0B, 0x55, 0x88, 0x12, 0xB4, 0x7A,
      0xC8, 0x61, 0xBE, 0x5B, 0x89, 0x4B, 0x98, 0x92, 0x00, 0xBE, 0xDB, 0x24,
      0x2E, 0x55, 0x98, 0x01, 0xC1, 0x64, 0x00, 0x3C, 0x6E, 0xFA, 0x61, 0xAC,
      0x34, 0x64, 0x99, 0x93, 0xF7, 0x82, 0xCC, 0xA1, 0x50, 0xDE, 0xE1, 0x44,
      0x38, 0x92, 0xBA, 0x3D, 0xB8, 0x7B, 0x09, 0x4C, 0x49, 0x1C, 0x79, 0x8B,
      0x5F, 0xA1, 0xAA, 0x30, 0x49, 0x1B, 0x7C, 0x7C, 0xC3, 0x76, 0x94, 0x76,
      0xD2, 0x8A, 0xA1, 0x7E, 0x75, 0x47, 0x73, 0x39, 0x03, 0x16, 0x64, 0x7C,
      0xF3, 0x30, 0x76, 0xCC, 0x6C, 0xA8, 0x4C, 0x80, 0x4B, 0x26, 0xE7, 0x52,
      0x95, 0xD2, 0xA4, 0x75, 0x4B, 0x50, 0x5A, 0x38, 0x31, 0xDC, 0x79, 0x3A,
      0xBD, 0x87, 0x4C, 0x6C, 0x14, 0x51, 0x91, 0x1B, 0x97, 0xAD, 0x39, 0x82,
      0x5E, 0xA4, 0x2D, 0x65, 0xDA, 0x63, 0xC4, 0x2C, 0xCE, 0x1E, 0xFB, 0x13,
      0x97, 0x16, 0x7F, 0x1E, 0x89, 0x33, 0x39, 0x9C, 0x2E, 0x28, 0x8A, 0xA5,
      0x7E, 0x00, 0x04, 0x06, 0xA4, 0xC6, 0x6B, 0x5A, 0x35, 0x7A, 0xC5, 0x90,
      0x36, 0xF3, 0x7A, 0x52, 0x51, 0x67, 0x01, 0xB5, 0x29, 0x34, 0xC6, 0xA1,
      0x10, 0x95, 0x93, 0x80, 0x9C, 0x0F, 0x72, 0x17, 0x0A, 0x0E, 0xC9, 0xAD,
      0x22, 0xEA, 0x6D, 0xA6, 0x4B, 0x00, 0x28, 0x68, 0x0B, 0x40, 0x59, 0x24,
      0x28, 0x72, 0x80, 0xA8, 0xC4, 0x9B, 0x2D, 0x51, 0x6D, 0xC9, 0xD6, 0xB9,
      0x3D, 0x42, 0xA9, 0xB5, 0x44, 0xC6, 0x83, 0x3C, 0x03, 0x40, 0x86, 0x59,
      0x05, 0xEB, 0x00, 0xF4, 0x66, 0x1B, 0x20, 0x06, 0x86, 0xA0, 0xA4, 0x7F,
      0xE2, 0x80, 0x93, 0x7B, 0xB0, 0x0F, 0x80, 0x22, 0xB8, 0xF0, 0xE6, 0x4A,
      0xC2, 0x51, 0xBB, 0x62, 0xD0, 0x9F, 0xBA, 0xB3, 0xE7, 0xC7, 0x9C, 0xCD,
      0x45, 0x0E, 0xEC, 0xA9, 0x41, 0x20, 0xB0, 0x5A, 0x0B, 0x07, 0x15, 0x88,
      0xE2, 0x15, 0x0E, 0xDA, 0x6B, 0x14, 0x15, 0x0F},
     true,
     {0x72, 0x2F, 0xE8, 0x73, 0x15, 0xF9, 0x77, 0x31, 0xC3, 0xF4, 0xD1,
      0x7F, 0x0F, 0xCF, 0x52, 0xC7, 0x0D, 0x40, 0xC7, 0x09, 0x58, 0x3F,
      0xC8, 0xA8, 0x51, 0x51, 0x4F, 0x2B, 0xD3, 0x5B, 0x3D, 0x52},
     {0x72, 0x65, 0x69, 0x61, 0x82, 0x16, 0x92, 0x79, 0xEF, 0x65, 0x77,
      0x9A, 0x02, 0x1A, 0xC0, 0xA0, 0xE0, 0xE7, 0xE4, 0xCF, 0xD3, 0x7C,
      0x85, 0x46, 0xD4, 0xDC, 0xB1, 0xBF, 0x08, 0x57, 0x2A, 0xA3}},

    {params_ml_kem768_test_mode,
     {0xE0, 0xAA, 0xD4, 0x6F, 0xDD, 0xE0, 0xB8, 0xE6, 0x43, 0x61, 0xC3,
      0x23, 0x32, 0x63, 0xD8, 0xA7, 0x51, 0xF5, 0x58, 0x3D, 0xBE, 0x91,
      0xAA, 0xA6, 0xE6, 0x9E, 0x63, 0x18, 0xFC, 0x7A, 0x8E, 0xE0},
     {0xC9, 0xCB, 0x9F, 0xD0, 0x40, 0x57, 0xEB, 0x96, 0x00, 0x64, 0x55, 0xC0,
      0x62, 0xE3, 0xC0, 0x72, 0x23, 0x46, 0xAD, 0xB3, 0x66, 0xDA, 0x0A, 0xB9,
      0x80, 0xC7, 0x82, 0xC4, 0x17, 0xB3, 0x60, 0xEB, 0x1C, 0x1F, 0x67, 0x62,
      0xEB, 0xF9, 0x67, 0xD7, 0x13, 0xA0, 0xD9, 0x3A, 0x28, 0xCF, 0x92, 0x06,
      0xE9, 0x54, 0x51, 0xE9, 0x13, 0x73, 0x80, 0x50, 0x47, 0xD8, 0xA1, 0x4F,
      0xFD, 0x20, 0x41, 0xB4, 0x46, 0x8B, 0x26, 0xC7, 0x9B, 0x69, 0x7A, 0x14,
      0xEA, 0x75, 0xA3, 0x38, 0x76, 0xBB, 0x86, 0x50, 0x96, 0xC0, 0x28, 0x9C,
      0x1A, 0xC4, 0xB9, 0x1A, 0x43, 0x99, 0x82, 0x13, 0x49, 0xDC, 0x66, 0x49,
      0x6D, 0xF0, 0x2B, 0x15, 0xCA, 0x43, 0x3F, 0xD9, 0x7D, 0x96, 0xF4, 0x6F,
      0x72, 0xE0, 0xB2, 0x3E, 0xB5, 0x61, 0xE8, 0x09, 0x60, 0x1C, 0x05, 0x3A,
      0x35, 0xF4, 0x17, 0x1A, 0x96, 0x3E, 0xC3, 0xF5, 0x42, 0xB4, 0x23, 0xBA,
      0xFC, 0x56, 0x13, 0x4B, 0x7C, 0x0C, 0x59, 0x27, 0xC7, 0x46, 0xE6, 0xC8,
      0x05, 0x5B, 0x3B, 0x70, 0xB3, 0x1C, 0xAD, 0x6A, 0x16, 0x8D, 0xD7, 0x8F,
      0x63, 0xC6, 0x4F, 0xE3, 0x04, 0x42, 0x80, 0x29, 0x7C, 0x66, 0x30, 0x56,
      0x2C, 0x48, 0xA8, 0x22, 0xB5, 0x70, 0xE3, 0xFA, 0x8A, 0x76, 0x99, 0x5B,
      0xEF, 0xE6, 0x77, 0x34, 0x27, 0x43, 0x37, 0xF1, 0x4F, 0xE0, 0x0C, 0x72,
      0x3A, 0xF5, 0x5D, 0x59, 0x6B, 0xBD, 0x0B, 0x2C, 0x04, 0xE5, 0xAC, 0x6D,
      0x52, 0xA0, 0xAE, 0xDB, 0x04, 0xB2, 0xBF, 0x09, 0xB1, 0xF9, 0x73, 0x6E,
      0x94, 0x56, 0xC4, 0x0D, 0x59, 0x76, 0xB1, 0xEC, 0x2C, 0xA2, 0x1C, 0x28,
      0xAC, 0x76, 0x1E, 0x39, 0xC5, 0x83, 0xCD, 0xF2, 0x56, 0xAA, 0x32, 0x62,
      0xC3, 0x26, 0x42, 0x50, 0xB1, 0xC1, 0x9B, 0x00, 0x84, 0x9F, 0xA8, 0x3B,
      0xCB, 0x66, 0x14, 0xDA, 0x0B, 0xB1, 0x97, 0x52, 0xAE, 0x9B, 0xCB, 0x7B,
      0xD1, 0x6A, 0x71, 0x01, 0x70, 0x66, 0xF7, 0x38, 0x04, 0xAE, 0xB3, 0xC7,
      0xF9, 0x21, 0x3B, 0xE4, 0x63, 0x44, 0x01, 0xA7, 0xAA, 0xE7, 0xBC, 0x56,
      0xE6, 0x03, 0x51, 0x6F, 0xA2, 0x83, 0x9A, 0x79, 0x1E, 0xA8, 0x9C, 0x58,
      0x05, 0x20, 0x30, 0xDB, 0x4A, 0xA2, 0x88, 0x77, 0x37, 0xD9, 0xBC, 0x9A,
      0xD2, 0x1A, 0xBB, 0x94, 0x24, 0x65, 0x46, 0x89, 0x27, 0x85, 0xCB, 0xBB,
      0xFB, 0xD4, 0x51, 0x02, 0xC2, 0x4D, 0x2F, 0x23, 0x70, 0xBE, 0x91, 0x92,
      0x3E, 0x73, 0x28, 0x92, 0xE5, 0x99, 0xD3, 0x15, 0x45, 0x02, 0xE8, 0x78,
      0xCC, 0xB9, 0x90, 0xCE, 0x83, 0x83, 0xD2, 0xF5, 0xAB, 0xF0, 0x21, 0xBA,
      0x42, 0x40, 0x08, 0xD0, 0x76, 0x85, 0xB8, 0xB5, 0x4C, 0x6A, 0x65, 0x50,
      0xD9, 0xCC, 0xBD, 0x93, 0x06, 0x6A, 0x5A, 0x65, 0x1A, 0x9B, 0x27, 0x10,
      0xD2, 0x5B, 0x38, 0x2E, 0x72, 0xCC, 0x57, 0x09, 0x69, 0x00, 0xE1, 0x6B,
      0x7B, 0x98, 0x68, 0xA0, 0x69, 0x09, 0xF4, 0x41, 0x10, 0x48, 0x60, 0xBF,
      0x03, 0x13, 0x43, 0x16, 0x38, 0x2F, 0x64, 0xD4, 0xCD, 0xDB, 0x59, 0x6C,
      0x57, 0x86, 0x6E, 0x07, 0x76, 0xC0, 0x4F, 0x8C, 0x0F, 0x67, 0xF2, 0x28,
      0x6B, 0x72, 0x42, 0x47, 0x70, 0x59, 0xEB, 0xE0, 0x22, 0xBD, 0xA2, 0x00,
      0xB6, 0x27, 0x1A, 0xF8, 0x75, 0xA8, 0x29, 0xFC, 0x36, 0x8E, 0x15, 0x74,
      0xF4, 0xF3, 0xC5, 0x6B, 0xB3, 0x9D, 0x7F, 0xF6, 0x11, 0x31, 0xF0, 0x14,
      0xFC, 0x59, 0x87, 0x0F, 0xE4, 0x54, 0x66, 0xD9, 0x68, 0xC1, 0x09, 0x33,
      0x9F, 0x0C, 0xCC, 0x67, 0xF6, 0xA7, 0x09, 0x23, 0x72, 0xCB, 0x97, 0x7C,
      0x75, 0x99, 0x8E, 0xBE, 0xCB, 0x68, 0xA4, 0x60, 0x59, 0xD7, 0x14, 0x6A,
      0x2E, 0xEA, 0xAD, 0xDB, 0x25, 0x8F, 0xCD, 0xD4, 0x04, 0xD6, 0xE9, 0x5E,
      0x54, 0x00, 0xC8, 0x43, 0x25, 0x35, 0xEF, 0xD6, 0x35, 0xAE, 0xE5, 0x23,
      0x5D, 0x3B, 0x8A, 0xBA, 0x41, 0x53, 0xB4, 0x6C, 0x7D, 0x23, 0x4A, 0xA9,
      0x4D, 0xDB, 0x66, 0xB2, 0x29, 0x45, 0x6C, 0x41, 0xB2, 0xFB, 0xF9, 0x3F,
      0x39, 0xAA, 0x04, 0x8C, 0x15, 0x8E, 0x50, 0xC3, 0x12, 0xFD, 0x62, 0x3D,
      0xC1, 0xE8, 0x18, 0xA4, 0x94, 0x98, 0x0A, 0x34, 0xC4, 0x15, 0x68, 0x94,
      0x2C, 0x30, 0x37, 0x3E, 0x43, 0x0D, 0xD5, 0x25, 0x0D, 0x0C, 0xC2, 0x7D,
      0x4A, 0xAB, 0x2A, 0xAF, 0xB1, 0x3C, 0xD7, 0x19, 0xAB, 0xC7, 0xF4, 0x66,
      0xB7, 0x02, 0xA9, 0x83, 0x31, 0x8B, 0x8D, 0x0C, 0x30, 0x40, 0xDC, 0x56,
      0xB0, 0x96, 0x02, 0x91, 0xF3, 0x0E, 0xA7, 0x59, 0x86, 0x91, 0x7A, 0x17,
      0xBF, 0xB7, 0x6B, 0xC4, 0xB4, 0xAA, 0x8E, 0xA3, 0x96, 0xF1, 0x27, 0xA4,
      0x1F, 0x73, 0x21, 0x9D, 0xD0, 0x6F, 0x27, 0x12, 0x6A, 0x5F, 0x06, 0x51,
      0x9E, 0x32, 0x8B, 0x5C, 0xEA, 0x9D, 0x67, 0xAC, 0x12, 0x2B, 0x09, 0x2F,
      0xC3, 0x61, 0x3D, 0x99, 0xF4, 0x03, 0xDB, 0x86, 0xCC, 0xDD, 0x76, 0x0B,
      0xF8, 0x39, 0x8C, 0x1A, 0xD6, 0xBE, 0x4B, 0x54, 0x74, 0xC3, 0x48, 0x64,
      0xF2, 0xB1, 0x0D, 0x97, 0xE2, 0x65, 0xFE, 0xDB, 0x46, 0x4E, 0x01, 0xCB,
      0x3A, 0x42, 0x6A, 0x3B, 0xD8, 0xB3, 0x45, 0x72, 0x67, 0x31, 0x7B, 0x76,
      0x49, 0xD6, 0x2D, 0x48, 0x47, 0xC3, 0xAA, 0xDB, 0x70, 0x36, 0x39, 0x1C,
      0xE0, 0x68, 0x75, 0x8B, 0xF1, 0x84, 0x7B, 0x96, 0xC5, 0xF3, 0xA6, 0x9B,
      0xD3, 0x6A, 0x94, 0x68, 0xF9, 0xCA, 0xB2, 0x74, 0x76, 0x45, 0x6B, 0x4E,
      0xB5, 0x10, 0xBB, 0x06, 0x3C, 0x46, 0xD5, 0x29, 0x75, 0x8C, 0x66, 0x80,
      0xB4, 0x58, 0x8C, 0xE4, 0xF5, 0xCD, 0x9D, 0x22, 0x5B, 0xF7, 0x00, 0xB2,
      0x7F, 0x53, 0xC1, 0x3D, 0x49, 0x80, 0x8D, 0x56, 0x1F, 0x04, 0x13, 0x31,
      0x5C, 0x01, 0x98, 0x9B, 0x15, 0x82, 0x75, 0x5C, 0xC4, 0xCC, 0x76, 0x53,
      0x91, 0xAA, 0x68, 0xA0, 0x61, 0x7F, 0x39, 0xB8, 0x43, 0x52, 0x60, 0x81,
      0x65, 0x85, 0x32, 0xC2, 0xC0, 0x02, 0x5C, 0xF3, 0x78, 0xB3, 0x14, 0x11,
      0xE8, 0x67, 0xC9, 0x78, 0x10, 0x94, 0x54, 0x81, 0x89, 0x16, 0xF8, 0x05,
      0x23, 0x36, 0xCF, 0xC9, 0x78, 0x8F, 0x87, 0xE5, 0x32, 0x36, 0xDA, 0xB5,
      0x41, 0x12, 0xB8, 0x76, 0x04, 0x2F, 0xE9, 0x05, 0x4D, 0x9C, 0x07, 0xB8,
      0x0F, 0x7B, 0x43, 0xAA, 0x90, 0x0B, 0xD2, 0x58, 0x0F, 0x38, 0x60, 0x58,
      0xCF, 0xF5, 0x62, 0x4A, 0x9A, 0x06, 0x24, 0x74, 0x99, 0x0C, 0x2C, 0x12,
      0x6E, 0x22, 0xBF, 0xB3, 0x00, 0xC4, 0x52, 0x30, 0x6B, 0x85, 0x92, 0x01,
      0xB3, 0xC6, 0x7B, 0x73, 0x56, 0xCC, 0x7D, 0xCC, 0x36, 0xCA, 0xD7, 0xA7,
      0xBF, 0xC4, 0x4D, 0xA0, 0x59, 0x1D, 0x8C, 0xC4, 0x5B, 0xC7, 0x76, 0x5E,
      0x40, 0x5A, 0x2F, 0x23, 0x73, 0x99, 0x6A, 0x72, 0x43, 0x3F, 0x33, 0x2C,
      0xF7, 0x7A, 0x5C, 0x9F, 0x31, 0x46, 0xDE, 0x12, 0x14, 0x47, 0x26, 0x4A,
      0x84, 0x33, 0x9A, 0x65, 0x29, 0x60, 0x10, 0x90, 0x44, 0xB1, 0xFC, 0x70,
      0x76, 0x80, 0xAF, 0x70, 0xE4, 0x57, 0xB4, 0x12, 0x59, 0xA1, 0xB8, 0x4C,
      0x37, 0x89, 0x2C, 0x10, 0x79, 0xB8, 0xF0, 0x82, 0x04, 0xF2, 0x18, 0xCE,
      0x85, 0x79, 0x67, 0x01, 0xB0, 0x5B, 0xEE, 0x59, 0x61, 0xF5, 0x55, 0x84,
      0xFD, 0xD3, 0xBB, 0xAF, 0xF2, 0x5F, 0x53, 0xF7, 0x8D, 0x0B, 0x48, 0x26,
      0x6E, 0xFC, 0x49, 0x91, 0xE0, 0x90, 0xEE, 0xB2, 0x87, 0x99, 0xC0, 0x02,
      0xCF, 0xE9, 0x02, 0xD1, 0x22, 0x2E, 0xDB, 0x8A, 0x02, 0x68, 0x13, 0x29,
      0x5E, 0x60, 0x49, 0x21, 0x26, 0x9D, 0x0E, 0x0A, 0x1F, 0x4C, 0xC7, 0x0F,
      0x4E, 0x52, 0x8A, 0x09, 0x2B, 0x4E, 0x15, 0x30, 0x0E, 0xB6, 0x78, 0x08,
      0xED, 0x20, 0x99, 0x5E, 0xF3, 0xA8, 0x0C, 0xB9, 0x99, 0x55, 0x34, 0x56,
      0xDE, 0xFA, 0x47, 0x69, 0x79, 0x1F, 0xE2, 0x4C, 0x93, 0x4D, 0xDB, 0x49,
      0x0A, 0xF6, 0x1F, 0xE1, 0xFC, 0x00, 0x27, 0xFB, 0x10, 0xAD, 0x26, 0x0A,
      0x7C, 0x33, 0xBC, 0xFB, 0xF2, 0x12, 0xD3, 0xB5, 0x05, 0x68, 0x84, 0x4F,
      0xB0, 0x22, 0xB7, 0xDA, 0xC9, 0x50, 0x9B, 0x30, 0x8A, 0x00, 0x40, 0xC9,
      0xB5, 0x48, 0x2D, 0x89, 0xEB, 0x41, 0xA6, 0x55, 0xB7, 0xC1, 0x93, 0xC0,
      0x1B, 0xF9, 0x6E, 0x5C, 0xF8, 0xA0, 0x8B, 0x4C, 0x1C, 0x34, 0x43, 0x36,
      0xFF, 0x53, 0xCE, 0x9F, 0x79, 0x00, 0x9F, 0xAA, 0x3A, 0x39, 0x21, 0x80,
      0x7D, 0x9B, 0x4C, 0x25, 0x73, 0x9C, 0x38, 0x56, 0x85, 0x84, 0x36, 0x7F,
      0x5D, 0x88, 0x2E, 0x4A, 0xFD, 0x33, 0x69, 0x7E, 0xB2, 0x2A, 0xD0, 0x3D,
      0x36, 0x9E, 0x37, 0xC0, 0xFE, 0x3B, 0x98, 0x10, 0x47, 0xBE, 0xD5, 0x5E,
      0x0B, 0xC0, 0x99, 0x99, 0x76, 0xE4, 0xA3, 0x6C},
     true,
     {0x22, 0x0F, 0x63, 0xD2, 0xC1, 0x40, 0xEF, 0x64, 0x35, 0xF1, 0x85,
      0xD9, 0xCC, 0xB0, 0x17, 0x64, 0xCD, 0x2E, 0x5A, 0xA5, 0xA2, 0xD9,
      0xF5, 0x72, 0x4D, 0xCA, 0xEB, 0xB5, 0x65, 0xE9, 0x89, 0x6E},
     {0x9C, 0x6E, 0xF5, 0x0D, 0xAE, 0x26, 0x88, 0x7F, 0x7F, 0xE5, 0xB0,
      0x17, 0x3C, 0x05, 0x5E, 0x88, 0xDC, 0x2F, 0xE0, 0x93, 0x84, 0x89,
      0x0E, 0x11, 0x77, 0x7F, 0x74, 0x2B, 0x99, 0xAD, 0x7C, 0x6C}},

    {params_ml_kem768_test_mode,
     {0x90, 0x34, 0x7D, 0x47, 0x8D, 0x5D, 0x96, 0x4D, 0x66, 0xA5, 0x4B,
      0xE9, 0x30, 0xFD, 0x9F, 0x7F, 0xD3, 0xC2, 0xAE, 0x14, 0x92, 0xDA,
      0xC3, 0x5A, 0x6C, 0xBD, 0xD0, 0x26, 0x16, 0xBC, 0xE1, 0x4A},
     {0xB7, 0xF5, 0x22, 0x43, 0x8A, 0x05, 0x31, 0x0B, 0x12, 0x92, 0x1A, 0x8A,
      0xBE, 0x79, 0xB4, 0x88, 0x7C, 0xB5, 0x48, 0x31, 0x7B, 0x23, 0xA2, 0x1A,
      0x42, 0x13, 0xA0, 0xF9, 0x40, 0xBC, 0x32, 0x68, 0x46, 0x43, 0x43, 0xC8,
      0x95, 0x40, 0xB8, 0x31, 0x55, 0xC9, 0x80, 0x69, 0x8C, 0x65, 0x72, 0x51,
      0xCE, 0x30, 0xC7, 0x50, 0x73, 0xC3, 0xD3, 0xC7, 0x3A, 0xD9, 0x90, 0x2E,
      0xD1, 0x0C, 0x32, 0xCB, 0x10, 0x1B, 0x05, 0x37, 0x82, 0x7F, 0x33, 0x8A,
      0x94, 0xD9, 0x61, 0x51, 0x54, 0x37, 0x01, 0x65, 0xB9, 0xD2, 0x27, 0x07,
      0x69, 0x3C, 0x5E, 0x06, 0x57, 0xBF, 0x89, 0x0A, 0x1C, 0x5E, 0x82, 0x08,
      0x99, 0x08, 0x76, 0x83, 0xD1, 0x76, 0x98, 0x61, 0x62, 0x15, 0x6A, 0xBA,
      0x10, 0x9C, 0xAD, 0xB1, 0x72, 0x9A, 0x0B, 0x0A, 0x05, 0x7C, 0x80, 0x07,
      0xE4, 0xA6, 0xAE, 0x7E, 0x27, 0x2C, 0x86, 0x01, 0x90, 0xED, 0x4C, 0xBC,
      0x12, 0x60, 0x82, 0xC1, 0x99, 0x2B, 0x9F, 0x54, 0x1F, 0xF6, 0x8A, 0x41,
      0x92, 0x26, 0xA6, 0x66, 0xF6, 0xA9, 0x43, 0x9C, 0xC8, 0x2B, 0x34, 0x24,
      0xC3, 0x51, 0x5E, 0xD1, 0x82, 0x1D, 0xA5, 0x01, 0x44, 0x9B, 0x84, 0xB1,
      0xD6, 0xC1, 0x69, 0xB9, 0xF3, 0x36, 0xAF, 0xC9, 0x10, 0x9F, 0x4B, 0x0A,
      0xA6, 0xA0, 0x69, 0xD5, 0xC4, 0x60, 0x95, 0xB4, 0x0E, 0xC7, 0xC8, 0x86,
      0xC1, 0xB2, 0xBA, 0x06, 0x0C, 0xC8, 0xC0, 0xA1, 0xB2, 0xFB, 0xB5, 0x3F,
      0xC1, 0x02, 0x70, 0x1F, 0xF3, 0x3B, 0xE4, 0x2C, 0x92, 0xFC, 0x82, 0x87,
      0xEB, 0x31, 0x1C, 0xBE, 0x4B, 0xB5, 0xD5, 0x23, 0x42, 0x08, 0x50, 0xB9,
      0xD9, 0x62, 0x15, 0x8D, 0x14, 0x18, 0x8C, 0x4A, 0x79, 0x4E, 0x4C, 0x0A,
      0x8A, 0x06, 0x57, 0x57, 0xEA, 0x0C, 0x9A, 0x65, 0x42, 0x9F, 0x11, 0x4A,
      0x98, 0x3B, 0x4D, 0x55, 0x74, 0x3F, 0xBE, 0xE2, 0x67, 0xAC, 0xA7, 0x6C,
      0x14, 0xB1, 0x49, 0x2E, 0x2C, 0xA8, 0x39, 0xAA, 0x5B, 0x47, 0x62, 0x86,
      0x8F, 0xF2, 0x12, 0x3A, 0x30, 0xB7, 0x6D, 0x85, 0x8B, 0x0E, 0xE4, 0xCE,
      0xB0, 0xB9, 0xAB, 0xD3, 0xA2, 0x71, 0x1B, 0xA5, 0x16, 0x7B, 0x73, 0x22,
      0x6C, 0x3B, 0x96, 0xC1, 0x84, 0xA1, 0x68, 0x1A, 0x56, 0x2F, 0x07, 0x36,
      0x5D, 0x68, 0x6E, 0x17, 0xF1, 0x0E, 0xB7, 0x55, 0x45, 0xF4, 0xE0, 0xC5,
      0x2C, 0xA6, 0xBF, 0x41, 0x62, 0x7B, 0x8D, 0x6A, 0x19, 0x78, 0x7A, 0x36,
      0x76, 0x15, 0x9E, 0xF3, 0x34, 0xA3, 0xF4, 0xC5, 0x38, 0xE5, 0x04, 0x84,
      0xF9, 0x7C, 0xBA, 0xA3, 0x34, 0xC5, 0xB8, 0x00, 0xA3, 0x4C, 0x64, 0x2D,
      0x09, 0x75, 0x49, 0xD8, 0xAA, 0xA6, 0xE6, 0xE1, 0x62, 0x61, 0xE0, 0x38,
      0xCD, 0xDB, 0x81, 0x6C, 0xE4, 0x6B, 0xA2, 0xB8, 0xB8, 0x21, 0x44, 0x52,
      0x09, 0xE5, 0xAB, 0x1C, 0xF7, 0x1B, 0x00, 0x89, 0x65, 0xFB, 0xF3, 0xA2,
      0x67, 0x3B, 0x50, 0x74, 0x94, 0x3D, 0x21, 0x02, 0x09, 0x2D, 0xF6, 0x03,
      0xF2, 0x98, 0x9F, 0x32, 0x08, 0x5B, 0x96, 0xCC, 0xB2, 0xBA, 0x24, 0x5A,
      0x7A, 0xAA, 0xC0, 0x6C, 0x76, 0x36, 0x40, 0xD1, 0x17, 0x30, 0xE8, 0x28,
      0xA0, 0xBA, 0x31, 0x29, 0xA4, 0x3F, 0x6F, 0x08, 0x25, 0xA6, 0x82, 0x88,
      0x6D, 0x8C, 0xCD, 0x9E, 0x84, 0xB9, 0x39, 0x99, 0x47, 0x4B, 0x63, 0xCC,
      0x8D, 0x27, 0x29, 0x11, 0xD7, 0x27, 0x29, 0xE7, 0x9E, 0x2A, 0x49, 0x4A,
      0xF5, 0xEA, 0x20, 0xC1, 0xCA, 0x84, 0x34, 0x32, 0x31, 0x64, 0x70, 0x90,
      0x90, 0x38, 0x26, 0x44, 0xB1, 0x1F, 0x0F, 0x6C, 0x80, 0x6A, 0x50, 0x44,
      0x09, 0x8B, 0xB9, 0x36, 0xF5, 0xCE, 0xCB, 0x28, 0xA8, 0x1A, 0xB5, 0x16,
      0xA1, 0x85, 0x7B, 0x65, 0x1B, 0x93, 0xCB, 0xAB, 0xB6, 0xE3, 0xD7, 0x6B,
      0x7F, 0x79, 0xAA, 0xF6, 0xB5, 0x03, 0x76, 0x2A, 0x10, 0xED, 0x84, 0x99,
      0x34, 0x77, 0x71, 0xF2, 0x21, 0xC2, 0xA9, 0x6A, 0x90, 0x04, 0x9C, 0x73,
      0xE6, 0xE1, 0x13, 0x2A, 0xF5, 0x3A, 0x88, 0xD3, 0x8E, 0xA8, 0x17, 0xCE,
      0xFA, 0x29, 0x84, 0x8F, 0xCC, 0x0A, 0xAD, 0x12, 0x5C, 0xA1, 0xC8, 0x29,
      0xF0, 0x38, 0x96, 0xCB, 0xA6, 0x35, 0xDA, 0xA3, 0x3A, 0xCE, 0x7C, 0x7E,
      0xBD, 0x18, 0xCD, 0x7A, 0xE0, 0x3A, 0x0F, 0x9A, 0x36, 0x59, 0x09, 0x0F,
      0x5F, 0x99, 0x76, 0xD6, 0x00, 0xB4, 0x51, 0xBB, 0x6C, 0x1F, 0x32, 0x81,
      0xD6, 0xC2, 0x88, 0x45, 0x22, 0x61, 0x8A, 0xE2, 0x0C, 0xD5, 0x83, 0xAB,
      0x9B, 0x32, 0x34, 0x4B, 0x6A, 0x2C, 0x29, 0x4B, 0x06, 0x84, 0xE6, 0xBA,
      0xB6, 0x86, 0xA3, 0x94, 0x31, 0x29, 0x2F, 0x70, 0x74, 0x8A, 0xD0, 0x2A,
      0xAC, 0xB5, 0x7C, 0xC3, 0x15, 0x3C, 0xA6, 0x86, 0x2D, 0x2B, 0x15, 0x10,
      0x8E, 0xE7, 0x1F, 0xD5, 0x1C, 0x2B, 0x7F, 0x81, 0xAB, 0xCA, 0x93, 0x67,
      0xD4, 0x61, 0x99, 0x64, 0xEC, 0x7C, 0x63, 0xEC, 0x65, 0x3F, 0xD1, 0x37,
      0x4E, 0xE3, 0xA9, 0xBF, 0x22, 0xC4, 0x81, 0x92, 0x64, 0xC2, 0xA7, 0x50,
      0xDE, 0xC1, 0x93, 0xD5, 0xCC, 0x83, 0x64, 0x30, 0xB1, 0x8C, 0x53, 0x5D,
      0x3E, 0x57, 0x3F, 0x45, 0x74, 0x10, 0x4D, 0x47, 0x1D, 0x97, 0x23, 0x61,
      0xA6, 0x95, 0x5E, 0x2E, 0x10, 0xAF, 0x34, 0xD9, 0x5B, 0x4A, 0x12, 0x32,
      0x18, 0x66, 0x76, 0x85, 0xB1, 0xAC, 0x6D, 0x69, 0x60, 0x53, 0x96, 0x6F,
      0x3C, 0x56, 0x29, 0x7B, 0x84, 0x46, 0x57, 0xC8, 0x54, 0x02, 0xA3, 0x57,
      0x64, 0x9A, 0x82, 0xE4, 0x57, 0x9B, 0x66, 0x7A, 0x36, 0x73, 0x4B, 0x66,
      0x55, 0x83, 0xAB, 0x02, 0x66, 0x99, 0xD2, 0xF6, 0xB8, 0xF4, 0xF1, 0xBB,
      0x25, 0x18, 0x68, 0x58, 0x54, 0x3B, 0x61, 0x26, 0x0B, 0xD3, 0xCC, 0xBB,
      0xA2, 0x64, 0xA3, 0x01, 0x35, 0x08, 0x8C, 0x04, 0x22, 0xAD, 0xB4, 0x50,
      0x8A, 0x60, 0x48, 0x5A, 0xAB, 0x3C, 0x63, 0xEC, 0xC9, 0xAF, 0x98, 0x1A,
      0x1C, 0x59, 0x13, 0x95, 0x0C, 0xB8, 0x94, 0x9B, 0x51, 0x79, 0xE4, 0x8C,
      0x72, 0x40, 0xCD, 0x6C, 0x85, 0x8E, 0x11, 0x44, 0xC7, 0xEE, 0x92, 0xA8,
      0x85, 0x00, 0x82, 0x09, 0xA0, 0x55, 0xC1, 0xC3, 0x79, 0x5F, 0xA3, 0xC4,
      0x9A, 0x98, 0x08, 0x2C, 0xA0, 0x77, 0xCE, 0x4A, 0x75, 0xB0, 0x02, 0x11,
      0x08, 0x45, 0x4D, 0x3B, 0x0C, 0x9F, 0x01, 0xC0, 0x0C, 0xB2, 0x91, 0x66,
      0x28, 0x1A, 0x24, 0xBC, 0xA7, 0x7A, 0xBB, 0x47, 0x1A, 0x03, 0x6C, 0x9D,
      0xB1, 0xB8, 0x94, 0x6B, 0x93, 0x0B, 0xBE, 0x86, 0x15, 0xCB, 0xE4, 0xAE,
      0xF6, 0x60, 0x8D, 0xEC, 0x38, 0x9B, 0xC7, 0x5C, 0x44, 0x37, 0x53, 0xC5,
      0x6F, 0x5A, 0x62, 0xFB, 0x6C, 0x53, 0x2B, 0xD1, 0x3D, 0x89, 0x50, 0x6C,
      0x4C, 0x59, 0x85, 0x30, 0x7A, 0x53, 0xB2, 0x40, 0x3B, 0x42, 0x26, 0x5B,
      0x35, 0xB1, 0xCF, 0xFD, 0x6B, 0x2C, 0xB4, 0x43, 0xA3, 0x59, 0xD7, 0x55,
      0x87, 0xB2, 0x94, 0x7C, 0x70, 0x76, 0xEB, 0x2A, 0x2B, 0x5B, 0xFC, 0xA8,
      0xB1, 0xCB, 0xA8, 0x65, 0x67, 0x4C, 0x1A, 0x2C, 0x1A, 0x5E, 0xB4, 0x91,
      0x7B, 0xFB, 0x97, 0x07, 0x59, 0x04, 0x34, 0xAC, 0xAD, 0xAD, 0xC9, 0x2C,
      0xC7, 0x73, 0x99, 0x33, 0x32, 0x39, 0x24, 0xB1, 0x7F, 0xB5, 0x16, 0x97,
      0x47, 0x14, 0xA4, 0x2E, 0x38, 0xBF, 0xB7, 0xF2, 0x0E, 0x75, 0x2A, 0xC6,
      0x46, 0x45, 0x03, 0x17, 0x0C, 0x86, 0x0F, 0xD9, 0x1A, 0xBF, 0xAA, 0x4C,
      0x6B, 0x70, 0xCC, 0xBB, 0x8A, 0x2F, 0xE3, 0xEA, 0x91, 0xC9, 0x4B, 0x26,
      0xD5, 0xC6, 0x4C, 0xDE, 0xE5, 0x7F, 0x93, 0x6C, 0x1A, 0xC1, 0x07, 0x04,
      0xD4, 0x73, 0xAE, 0x10, 0x47, 0x9D, 0x79, 0x8B, 0xCD, 0x60, 0x26, 0xBA,
      0xCF, 0x4C, 0x59, 0x9D, 0xC1, 0x04, 0xD7, 0x04, 0x39, 0xB0, 0xE2, 0xA3,
      0x2C, 0x91, 0xA5, 0xD5, 0x80, 0x10, 0xAB, 0xA9, 0x98, 0x4A, 0x34, 0x68,
      0x9E, 0x36, 0x0C, 0x7B, 0x29, 0x64, 0x86, 0xA4, 0xC6, 0x81, 0x40, 0x5E,
      0xC3, 0xFB, 0x92, 0xB4, 0x42, 0x10, 0xD5, 0xD3, 0xCF, 0x52, 0x53, 0x40,
      0x95, 0x85, 0x9E, 0x42, 0xBC, 0x9B, 0xCA, 0x7B, 0x38, 0x64, 0xD4, 0x3C,
      0xB9, 0xF4, 0x6E, 0x68, 0x5C, 0x0B, 0xEE, 0x88, 0x53, 0xFB, 0xD3, 0x12,
      0x85, 0xDC, 0x2E, 0xDF, 0x12, 0x96, 0x35, 0xC7, 0x04, 0x37, 0xB6, 0x12,
      0x95, 0x5B, 0x7E, 0x56, 0xA1, 0x28, 0xDE, 0xA3, 0x21, 0x29, 0x86, 0x62,
      0xF4, 0xF9, 0x6A, 0x8E, 0x1A, 0xBE, 0x81, 0xB5, 0x4D, 0xA4, 0x29, 0x2A,
      0xFD, 0x5F, 0xE4, 0x5E, 0x31, 0xE1, 0x6B, 0x17, 0x91, 0x9D, 0x9E, 0xBF,
      0x8E, 0x87, 0xB3, 0x8D, 0x48, 0x05, 0x3A, 0xA9, 0xF6, 0xF4, 0x1C, 0x5B,
      0x55, 0xAB, 0x86, 0xC4, 0xE0, 0xBE, 0xE5, 0x58},
     true,
     {0x42, 0xAF, 0x32, 0x4B, 0x61, 0x5E, 0x60, 0x65, 0x85, 0x3D, 0x01,
      0xF3, 0x04, 0x98, 0x6A, 0xFF, 0x05, 0x04, 0x82, 0xA6, 0x3E, 0x22,
      0xD9, 0xD2, 0xF3, 0x51, 0xD7, 0x8D, 0xFA, 0xA8, 0xB5, 0x9A},
     {0x05, 0xBD, 0x5B, 0x91, 0xC2, 0xF6, 0x34, 0xE5, 0xB8, 0xBC, 0x59,
      0x69, 0x7D, 0x18, 0x0C, 0xF1, 0xB3, 0x6A, 0x24, 0x4C, 0x6E, 0xDF,
      0xEF, 0xE7, 0x45, 0x83, 0x08, 0xB5, 0x85, 0x4C, 0x77, 0xFB}},

    {params_ml_kem768_test_mode,
     {0x11, 0x9B, 0xC3, 0x6B, 0x5F, 0x85, 0x6C, 0x0A, 0x2F, 0x13, 0x6B,
      0x3E, 0xE4, 0x20, 0x41, 0xB8, 0x17, 0x12, 0x5A, 0x60, 0x0E, 0x82,
      0x9F, 0xF6, 0xB4, 0xB4, 0x02, 0x13, 0x1A, 0x26, 0xAB, 0xF1},
     {0xAF, 0x98, 0x33, 0x8A, 0x68, 0x2D, 0x43, 0x1C, 0xA0, 0xE1, 0x77, 0x75,
      0xEB, 0x17, 0x0E, 0x37, 0x42, 0xAB, 0xEA, 0x30, 0x0D, 0x6A, 0x46, 0xC5,
      0x67, 0xC3, 0x64, 0xDE, 0x89, 0x39, 0x83, 0x16, 0x95, 0xC5, 0x9B, 0xB7,
      0x68, 0x67, 0x29, 0xC9, 0x00, 0x1E, 0x25, 0xA8, 0x5F, 0xE9, 0x26, 0xCC,
      0x6E, 0x58, 0x4E, 0x2B, 0xC8, 0x6D, 0x3B, 0x25, 0xBC, 0x9D, 0x6A, 0xBB,
      0x97, 0xEB, 0x7F, 0x15, 0xAC, 0x23, 0x65, 0x6B, 0x31, 0x85, 0xCB, 0xAF,
      0xE0, 0xC3, 0x9F, 0xA0, 0x78, 0x9D, 0xF0, 0x67, 0x8F, 0xBF, 0x5A, 0x43,
      0xE6, 0xE0, 0xC5, 0x3E, 0xC3, 0x80, 0x76, 0x57, 0x2D, 0x9D, 0x84, 0xB1,
      0xAB, 0xE7, 0x42, 0xE2, 0xF6, 0xC0, 0xC8, 0xCB, 0x08, 0xCD, 0xCA, 0xCD,
      0x23, 0xB7, 0x1D, 0x57, 0xD0, 0x67, 0x08, 0xF3, 0x2D, 0x50, 0x87, 0x0D,
      0x9D, 0x63, 0x6D, 0xF1, 0xDC, 0x01, 0xA8, 0x37, 0x8A, 0x21, 0x1A, 0x13,
      0x4B, 0xB2, 0x55, 0xDD, 0xDC, 0x0B, 0x62, 0xC7, 0x58, 0x12, 0xAB, 0x16,
      0x77, 0xC5, 0x0D, 0xF5, 0x6B, 0x1F, 0xF6, 0x20, 0x24, 0xFD, 0x72, 0x2D,
      0x3E, 0x73, 0x2F, 0x56, 0xA2, 0xC6, 0xEA, 0x10, 0xCC, 0x31, 0xF2, 0x80,
      0xAB, 0xC8, 0x34, 0x77, 0x88, 0xCB, 0x29, 0x1A, 0xC5, 0xA1, 0x82, 0x05,
      0x25, 0xB9, 0xA3, 0x3A, 0x70, 0x89, 0xDD, 0x68, 0x99, 0x62, 0xA0, 0x46,
      0xB6, 0x52, 0xAD, 0x18, 0x26, 0x39, 0x27, 0x82, 0x79, 0xEB, 0x88, 0x41,
      0x63, 0xE2, 0xB1, 0x15, 0xA2, 0x9A, 0x38, 0x99, 0xCB, 0x0E, 0xDB, 0x45,
      0x14, 0xA0, 0x83, 0x6B, 0xFB, 0x8A, 0x51, 0xD8, 0x34, 0xC1, 0x93, 0x9B,
      0x8D, 0xC1, 0x08, 0xB6, 0x13, 0x8F, 0xB8, 0x8B, 0x91, 0x99, 0xDC, 0xA5,
      0xF7, 0xF6, 0x4A, 0xFF, 0x36, 0xB9, 0x29, 0x66, 0x13, 0xF8, 0x91, 0x26,
      0x57, 0x78, 0xC7, 0x96, 0x3C, 0x3E, 0x70, 0x2B, 0x81, 0xC5, 0x48, 0x34,
      0x46, 0x9A, 0xC8, 0xB5, 0x99, 0x20, 0xBE, 0xE7, 0x87, 0x8F, 0x01, 0x05,
      0x2B, 0x77, 0xB0, 0xF5, 0x4B, 0x6D, 0xC6, 0x1A, 0xA3, 0xDE, 0x69, 0x5D,
      0x20, 0x78, 0x6F, 0x7D, 0x30, 0x9C, 0x3B, 0x16, 0xA8, 0xD2, 0xC9, 0x0A,
      0x92, 0x1C, 0xC3, 0x17, 0xE9, 0x1C, 0x01, 0x5A, 0xC8, 0x0D, 0xB1, 0x06,
      0xD4, 0x81, 0x0E, 0xBB, 0xF2, 0xC5, 0xA1, 0x53, 0x05, 0x06, 0xAB, 0x1E,
      0x28, 0xB7, 0xA3, 0x2B, 0xF6, 0x7A, 0x69, 0x81, 0x18, 0x5F, 0x98, 0xB4,
      0x4C, 0xAB, 0xCF, 0x67, 0x06, 0xB1, 0x34, 0xB5, 0x53, 0x7D, 0xC6, 0x97,
      0xD1, 0x6A, 0xC0, 0x03, 0x80, 0x81, 0x19, 0xB0, 0xBF, 0x94, 0xC8, 0x4B,
      0xF5, 0x69, 0x42, 0x2C, 0x8B, 0xCF, 0xF8, 0x34, 0x23, 0x7B, 0x90, 0x2D,
      0x83, 0xBC, 0x4C, 0x16, 0xF5, 0xCD, 0xFB, 0xBA, 0x5C, 0xDA, 0xF0, 0xA6,
      0x9F, 0xC8, 0x7D, 0xDC, 0x88, 0x5D, 0xD3, 0xAA, 0xA8, 0x52, 0x34, 0x2C,
      0x1E, 0xB8, 0x17, 0x97, 0x56, 0x08, 0x7C, 0x67, 0x8E, 0xAE, 0x28, 0x78,
      0xA7, 0x77, 0x65, 0x5F, 0xC1, 0x97, 0x19, 0x59, 0x3F, 0xF6, 0x00, 0x78,
      0x9A, 0xCC, 0x63, 0x22, 0x01, 0x87, 0x91, 0x87, 0x4D, 0x33, 0xD9, 0x28,
      0x4E, 0xA5, 0x12, 0xAF, 0x12, 0x31, 0xD4, 0xCC, 0x87, 0xF1, 0xBA, 0x6F,
      0x62, 0x93, 0xA6, 0x53, 0x03, 0x6E, 0x59, 0x0A, 0x89, 0x2F, 0x10, 0x1C,
      0xC5, 0x18, 0x11, 0x0C, 0x98, 0x4A, 0xC5, 0x5C, 0x29, 0x31, 0xC9, 0x76,
      0x82, 0x82, 0x83, 0xF0, 0x26, 0x66, 0x09, 0xA4, 0x5A, 0x8C, 0x7C, 0xBF,
      0x8C, 0x23, 0xCA, 0x02, 0x79, 0xB1, 0x33, 0xA3, 0x8F, 0x9D, 0xC7, 0x97,
      0xD5, 0xE5, 0x80, 0x11, 0x09, 0x6D, 0x45, 0x44, 0x1F, 0x11, 0xDC, 0x59,
      0xAB, 0x5B, 0x66, 0x84, 0x6A, 0x87, 0x05, 0x93, 0x14, 0xFD, 0xC6, 0x4E,
      0x8C, 0xF5, 0x00, 0x40, 0xB5, 0x74, 0x32, 0xA1, 0x8F, 0x46, 0x07, 0x8A,
      0xF5, 0xA6, 0x13, 0x2B, 0x00, 0x68, 0x56, 0x60, 0x4C, 0xE0, 0x09, 0x12,
      0x8F, 0xC5, 0x44, 0x5B, 0xCB, 0x68, 0x91, 0xE9, 0x1A, 0x86, 0x77, 0x06,
      0x0F, 0x39, 0xA2, 0xCA, 0xAA, 0x18, 0x3F, 0xB8, 0xA0, 0xF6, 0x75, 0x10,
      0x51, 0xAB, 0x85, 0x47, 0x4A, 0x53, 0x9A, 0x41, 0x83, 0xA4, 0x48, 0x64,
      0x71, 0x41, 0x8F, 0xDB, 0xEB, 0xCF, 0x40, 0x05, 0x5A, 0xC4, 0x60, 0x75,
      0xC3, 0xB2, 0x19, 0x02, 0x48, 0xA8, 0x20, 0x24, 0x31, 0xDB, 0x82, 0x8B,
      0x82, 0xE1, 0x32, 0x0D, 0xBA, 0x47, 0xA2, 0x3A, 0x94, 0xDE, 0x5C, 0xC3,
      0x78, 0xB4, 0x8F, 0xF4, 0xB6, 0x33, 0xE2, 0xD6, 0x66, 0xB5, 0x61, 0x82,
      0x7F, 0x80, 0x13, 0x42, 0x99, 0x56, 0xD3, 0xF5, 0x94, 0x7F, 0xBB, 0x84,
      0x8B, 0x3A, 0x51, 0x14, 0x52, 0x87, 0x0D, 0x00, 0xBE, 0x30, 0x61, 0x0D,
      0x0A, 0xCC, 0x41, 0x8F, 0x85, 0x36, 0x68, 0x6A, 0xBF, 0x66, 0x85, 0x17,
      0x07, 0xD8, 0x9F, 0xCB, 0xF6, 0x59, 0x59, 0xBA, 0x30, 0x62, 0xF6, 0xA1,
      0x6E, 0x26, 0x84, 0x88, 0xD0, 0x4B, 0xE4, 0xE3, 0x70, 0xC4, 0x24, 0x79,
      0x47, 0xFA, 0x82, 0x3B, 0x27, 0x13, 0x3A, 0xDA, 0x76, 0xDB, 0x58, 0xCF,
      0x3E, 0xD1, 0x10, 0x9C, 0xE4, 0x33, 0xCA, 0x00, 0x42, 0xCD, 0xB8, 0x09,
      0xAA, 0x1B, 0x5D, 0x5E, 0x9C, 0x42, 0x58, 0xC6, 0x06, 0x7B, 0x00, 0x04,
      0xA9, 0x05, 0x78, 0x64, 0x73, 0xA0, 0xCA, 0x4B, 0x3E, 0x9D, 0x90, 0x44,
      0x38, 0x91, 0xA1, 0x48, 0x40, 0x7A, 0x8D, 0x89, 0x12, 0x13, 0x79, 0xA1,
      0xF3, 0x7C, 0x70, 0x77, 0x05, 0x86, 0x36, 0x5A, 0x74, 0xFF, 0xE5, 0xC6,
      0xFF, 0x17, 0x07, 0x91, 0x68, 0x1E, 0x5C, 0xE1, 0x0B, 0xB8, 0x8B, 0xC0,
      0xBC, 0xCA, 0x82, 0xCF, 0xC8, 0x14, 0x24, 0xAA, 0x5B, 0xDA, 0x07, 0x2D,
      0x1B, 0xB5, 0xBC, 0x62, 0xF6, 0x68, 0x7A, 0x05, 0xA9, 0x49, 0xC2, 0xB0,
      0x40, 0x05, 0x28, 0x4F, 0xBA, 0x8E, 0xD0, 0x64, 0x6A, 0x39, 0x46, 0x58,
      0xA5, 0x86, 0x83, 0x04, 0xC4, 0x2C, 0x05, 0x60, 0x5E, 0x92, 0x92, 0x80,
      0x16, 0x30, 0x12, 0x9C, 0x76, 0xA1, 0xFC, 0x08, 0x3D, 0xC3, 0x69, 0x6D,
      0xC4, 0xE9, 0x04, 0xBB, 0x19, 0x16, 0xBA, 0x22, 0x87, 0xDC, 0x37, 0x23,
      0x2A, 0xE2, 0xC9, 0x62, 0x10, 0x8F, 0x24, 0x5C, 0x3A, 0x0B, 0x70, 0x2F,
      0x53, 0x27, 0x1F, 0x6B, 0xB0, 0x03, 0x63, 0xD8, 0x86, 0x90, 0xA7, 0xBF,
      0x37, 0xB3, 0x45, 0xAC, 0xEB, 0x54, 0x26, 0xF3, 0x4B, 0xCD, 0x76, 0x70,
      0xAC, 0xF1, 0x59, 0x55, 0x40, 0x1A, 0x20, 0xF9, 0x38, 0xA4, 0x40, 0x7E,
      0x00, 0xD8, 0x00, 0x8C, 0x14, 0x7D, 0xA0, 0x0C, 0x08, 0x0A, 0x22, 0x6F,
      0x71, 0xF1, 0x5D, 0xD8, 0xDA, 0x20, 0xD0, 0xC6, 0xAF, 0xEE, 0x43, 0x1B,
      0x6B, 0x27, 0x33, 0x51, 0x3B, 0xC8, 0x77, 0xF5, 0x54, 0x57, 0x53, 0x30,
      0x5E, 0x8C, 0x03, 0xFA, 0xEA, 0x93, 0x5C, 0x18, 0x3B, 0xFC, 0x1C, 0x40,
      0x56, 0x1B, 0x59, 0xFE, 0x80, 0x95, 0x6C, 0xE2, 0xC5, 0xBD, 0x3C, 0x28,
      0xF2, 0x24, 0x82, 0x49, 0xE2, 0xA0, 0x52, 0x78, 0x91, 0xB3, 0xB6, 0x4B,
      0xF0, 0xEB, 0x89, 0x93, 0x8A, 0x42, 0x00, 0x8C, 0x32, 0xD6, 0x68, 0x05,
      0x4E, 0x50, 0x88, 0x71, 0x01, 0x2F, 0xE8, 0x65, 0x34, 0x73, 0xFB, 0x70,
      0xCA, 0xFA, 0x0A, 0x8F, 0x1A, 0x50, 0x99, 0x48, 0xA2, 0xA9, 0x67, 0xCE,
      0xCF, 0xC7, 0xB5, 0xDA, 0xE3, 0x4E, 0x16, 0x60, 0x86, 0x43, 0x3C, 0x59,
      0x77, 0x20, 0x8C, 0x45, 0xD9, 0x7B, 0x24, 0x23, 0x8E, 0x87, 0x5A, 0x79,
      0x0C, 0x27, 0x79, 0x97, 0x4B, 0x47, 0x21, 0xF5, 0x73, 0xC0, 0x3C, 0x4D,
      0x20, 0x77, 0x7F, 0x1C, 0x45, 0x89, 0x22, 0x77, 0x65, 0xE8, 0xE7, 0xAF,
      0xD0, 0xD5, 0x9E, 0xDD, 0xDC, 0x72, 0x2C, 0xC6, 0x23, 0x03, 0x56, 0xB4,
      0x77, 0xC4, 0x90, 0x23, 0x4C, 0xAF, 0x85, 0x88, 0x93, 0xF0, 0xE4, 0x46,
      0x72, 0x7C, 0xC8, 0x8E, 0x14, 0x11, 0x24, 0x23, 0x44, 0xDA, 0xC6, 0xAE,
      0x6C, 0x2C, 0xC1, 0xD2, 0xB1, 0xAF, 0x2B, 0x8C, 0x32, 0xBC, 0xFB, 0x25,
      0x30, 0x13, 0x41, 0x1F, 0x18, 0x69, 0x3A, 0xCA, 0xB9, 0xA7, 0xA8, 0x6C,
      0xA5, 0x59, 0x09, 0x64, 0xD3, 0x9A, 0x8A, 0x50, 0x76, 0x87, 0x18, 0xBD,
      0x94, 0x85, 0x66, 0xA2, 0x82, 0x22, 0x06, 0x22, 0x60, 0x20, 0x16, 0x59,
      0x65, 0xF7, 0xB6, 0x88, 0x71, 0xAA, 0xFD, 0x34, 0x74, 0xFA, 0x30, 0x6A,
      0x2D, 0xC3, 0x1A, 0x98, 0xC6, 0x0F, 0xD2, 0xE5, 0xAA, 0xA8, 0xA0, 0xB7,
      0x2B, 0xDD, 0x2F, 0x70, 0xD6, 0xD5, 0xDE, 0xDE, 0x7D, 0x67, 0x97, 0x58,
      0xD8, 0xA3, 0x25, 0xB6, 0xCF, 0x11, 0xE7, 0x92, 0x29, 0x02, 0xAC, 0xD9,
      0x2A, 0x3A, 0x8C, 0xB4, 0x38, 0x63, 0xCE, 0x98},
     true,
     {0x74, 0xF1, 0x56, 0xB3, 0xA9, 0x16, 0x99, 0x35, 0x4E, 0xE2, 0xCC,
      0xB0, 0xAE, 0x0F, 0xDF, 0x55, 0x32, 0x21, 0x36, 0x86, 0xBE, 0xB2,
      0xD4, 0x06, 0xE5, 0x29, 0xB8, 0x5D, 0x05, 0x37, 0x75, 0x05},
     {0x4E, 0xF3, 0x3F, 0x2E, 0x08, 0xDB, 0x26, 0xB1, 0x19, 0x79, 0xF9,
      0x5F, 0xF6, 0xC6, 0x24, 0xB4, 0x16, 0x8C, 0xE9, 0x05, 0x5F, 0xD3,
      0x13, 0x90, 0xED, 0xFA, 0xAD, 0x5E, 0x2D, 0xAB, 0xA6, 0xA8}},

    {params_ml_kem768_test_mode,
     {0x69, 0x7C, 0xC7, 0x44, 0x5A, 0xE2, 0xC9, 0xEC, 0xCA, 0x25, 0x69,
      0xB7, 0x87, 0x1F, 0x0B, 0xBB, 0x36, 0x4E, 0x63, 0xE4, 0xB7, 0x82,
      0xF7, 0x34, 0xFA, 0xFE, 0xD4, 0xFE, 0x33, 0xE4, 0xAF, 0x14},
     {0x46, 0x3B, 0x55, 0x31, 0x02, 0x89, 0x8C, 0xA2, 0x97, 0xE0, 0xC2, 0x05,
      0xF3, 0xC9, 0x58, 0x22, 0x73, 0xAC, 0xFB, 0xDB, 0x13, 0xBE, 0xC5, 0x33,
      0x41, 0xBB, 0xB6, 0x72, 0x4C, 0x77, 0x4C, 0x74, 0x1A, 0xBB, 0x16, 0xB2,
      0xF9, 0x92, 0xC8, 0x78, 0x11, 0x4F, 0xAF, 0x67, 0xC2, 0x48, 0x60, 0x3D,
      0x8A, 0xD5, 0x84, 0x2E, 0x72, 0x30, 0xE5, 0x9B, 0x75, 0x37, 0x11, 0x9B,
      0x20, 0x4A, 0xA8, 0xB2, 0xE0, 0x83, 0x0C, 0xA3, 0x51, 0x18, 0x86, 0xAC,
      0x52, 0x80, 0x24, 0x2A, 0x9A, 0x81, 0x7D, 0x9A, 0x8C, 0x7E, 0xC8, 0xB9,
      0x5D, 0xE2, 0x32, 0x54, 0x17, 0x4D, 0x1F, 0x50, 0x71, 0x29, 0xBA, 0xA4,
      0x9C, 0xAC, 0x9C, 0x68, 0x00, 0xCD, 0x10, 0x39, 0xBA, 0x2F, 0x66, 0x25,
      0x58, 0x4B, 0x8F, 0x3A, 0xB2, 0x76, 0x57, 0xE6, 0xB7, 0xBA, 0x2A, 0x58,
      0x08, 0x05, 0x91, 0xD5, 0x72, 0xC7, 0xE0, 0xB5, 0xAB, 0xCB, 0xA3, 0xBA,
      0xBF, 0x05, 0x25, 0x73, 0x6A, 0xCA, 0x42, 0x7A, 0x1F, 0x13, 0x24, 0x54,
      0x26, 0xC0, 0x8D, 0xCA, 0xC5, 0x75, 0x28, 0x91, 0x16, 0x9B, 0x1A, 0x83,
      0x73, 0xB4, 0xBE, 0xA8, 0xA4, 0x9D, 0xAC, 0x3B, 0x81, 0x63, 0xE7, 0x8E,
      0x74, 0x19, 0x5E, 0x8A, 0xB5, 0x40, 0x07, 0x53, 0x41, 0x33, 0xC0, 0x2A,
      0xD1, 0x93, 0x0D, 0xD3, 0xCB, 0x06, 0x6E, 0x71, 0x14, 0xDB, 0xF4, 0xCC,
      0x95, 0x45, 0xC3, 0x61, 0x6B, 0x18, 0x95, 0x46, 0x9A, 0x1E, 0x08, 0xA5,
      0x72, 0x73, 0xB2, 0x35, 0x87, 0x4A, 0xEE, 0xB3, 0x2E, 0xA3, 0x43, 0x3D,
      0xBE, 0x87, 0x32, 0xB8, 0x71, 0xB4, 0xD0, 0xE3, 0x19, 0x2A, 0xDC, 0x30,
      0xAC, 0x00, 0x2E, 0x4B, 0x82, 0x79, 0x8E, 0x0C, 0x9A, 0xF0, 0x04, 0x97,
      0x3F, 0x57, 0x49, 0x45, 0x66, 0x05, 0xB8, 0x97, 0xC5, 0xAA, 0x99, 0x38,
      0xC3, 0x83, 0x43, 0xB3, 0x69, 0x01, 0x60, 0x61, 0x5B, 0x34, 0x06, 0xAF,
      0xC6, 0xFA, 0x20, 0xD0, 0x4A, 0xA0, 0x21, 0x4B, 0x9C, 0x22, 0x4A, 0xC7,
      0xE6, 0x4A, 0x6E, 0x85, 0xF3, 0x8A, 0xAC, 0x01, 0x63, 0x2A, 0x50, 0x1A,
      0x6C, 0xA1, 0xA4, 0xD3, 0xE0, 0xAB, 0xAE, 0x38, 0x4A, 0xA2, 0xA5, 0x09,
      0x11, 0xE8, 0xBF, 0xE9, 0x45, 0x79, 0x2C, 0x92, 0x83, 0xF9, 0xA8, 0x98,
      0xA1, 0xCB, 0x2E, 0x29, 0x34, 0x55, 0x0E, 0x3C, 0xC5, 0x2E, 0x9B, 0xB4,
      0xC2, 0xEC, 0x12, 0x5F, 0x64, 0xC2, 0x8B, 0x0B, 0x12, 0xE0, 0x80, 0x5A,
      0x03, 0x18, 0x24, 0xF4, 0x23, 0xB7, 0xCF, 0xD9, 0xA1, 0x10, 0x86, 0x52,
      0x3F, 0x13, 0x05, 0xE4, 0xC1, 0x30, 0xC1, 0x71, 0x61, 0xAF, 0xF3, 0x66,
      0x47, 0x28, 0x75, 0x64, 0x5B, 0xC8, 0xA1, 0xB1, 0x4D, 0xBB, 0x10, 0x44,
      0x94, 0xC7, 0x46, 0xF8, 0x4B, 0x45, 0x99, 0xC0, 0x7F, 0xF4, 0xE7, 0x4A,
      0x3A, 0x14, 0x08, 0xCC, 0x98, 0x3C, 0xC2, 0xC6, 0xCC, 0x34, 0xBA, 0x04,
      0x33, 0x36, 0x59, 0xC1, 0xF9, 0x60, 0xE0, 0x6B, 0x73, 0x25, 0x7C, 0x5E,
      0xAD, 0x5B, 0x34, 0x06, 0xD7, 0x0C, 0x3D, 0xCA, 0x05, 0xB9, 0xEB, 0x1E,
      0x0E, 0xAA, 0x89, 0xEF, 0xF0, 0x46, 0xB5, 0x84, 0x71, 0xFB, 0x72, 0x68,
      0xBD, 0xB6, 0x40, 0x21, 0x74, 0xBF, 0x03, 0xF9, 0x2D, 0x4D, 0x22, 0xC7,
      0xF2, 0x04, 0xA3, 0x5A, 0xC6, 0x55, 0x14, 0x55, 0x60, 0x3D, 0x83, 0x82,
      0xCA, 0x3A, 0xC7, 0x26, 0xC1, 0xBA, 0x3E, 0xCB, 0x5D, 0x42, 0x44, 0xBC,
      0x62, 0xF1, 0x09, 0x89, 0xB4, 0xB9, 0x85, 0x1C, 0x43, 0x46, 0xDB, 0xCD,
      0xB9, 0x59, 0x51, 0x6C, 0xC2, 0x60, 0xCE, 0x25, 0x99, 0xEB, 0x12, 0x32,
      0xCF, 0xD6, 0x5F, 0xDF, 0x39, 0xAE, 0x7E, 0xB0, 0x76, 0x3A, 0xE4, 0x7A,
      0x35, 0x93, 0x67, 0xA7, 0x72, 0x0D, 0x02, 0xB0, 0x7A, 0x5A, 0x63, 0x4D,
      0xBD, 0xF7, 0x4F, 0x4A, 0xD5, 0x89, 0xEE, 0x26, 0x91, 0xBB, 0x48, 0x62,
      0xD0, 0x6B, 0x68, 0x36, 0xF7, 0x0B, 0xA8, 0xC5, 0x82, 0xF3, 0xA7, 0x99,
      0x6B, 0xD7, 0xBE, 0xCB, 0x99, 0x8C, 0xB1, 0xAB, 0x15, 0x64, 0xC8, 0x12,
      0xAD, 0x04, 0x92, 0x0B, 0x91, 0x21, 0x13, 0x98, 0x17, 0xE8, 0xF0, 0x61,
      0x72, 0xB2, 0x17, 0x48, 0x1C, 0x37, 0x9E, 0x2C, 0x36, 0x8C, 0x0A, 0x12,
      0x19, 0x41, 0x95, 0xA0, 0x31, 0x77, 0xA8, 0x22, 0xC7, 0x6B, 0x0C, 0x7C,
      0x79, 0xAB, 0x70, 0x0F, 0xD5, 0xC2, 0x4E, 0xFB, 0x24, 0xBF, 0x29, 0x13,
      0x9F, 0x49, 0x30, 0xE6, 0x51, 0x5E, 0x7A, 0x18, 0x80, 0xD9, 0x71, 0x64,
      0x4A, 0xCA, 0x12, 0xE4, 0x27, 0x8A, 0xD2, 0x2C, 0xB9, 0x63, 0x70, 0xAE,
      0x09, 0x61, 0x4D, 0x96, 0xCC, 0x03, 0x28, 0x26, 0x5F, 0x27, 0xE2, 0xB2,
      0xD6, 0x04, 0x65, 0xC8, 0xA3, 0x70, 0x73, 0x66, 0x32, 0x25, 0x7B, 0x19,
      0x1B, 0x72, 0x18, 0x5D, 0xA1, 0xA2, 0xE6, 0xFA, 0xA6, 0x75, 0x95, 0x80,
      0xC6, 0xE5, 0x1E, 0x62, 0x90, 0xB7, 0x69, 0x44, 0xC2, 0xA8, 0x74, 0x36,
      0x10, 0x42, 0x4D, 0xCA, 0xB8, 0x9C, 0xC9, 0xB7, 0x9A, 0xFE, 0xC5, 0x12,
      0xDF, 0x9C, 0xC5, 0xB4, 0x80, 0x7B, 0x0C, 0x93, 0xA8, 0x89, 0xF8, 0x72,
      0xD1, 0xD8, 0x95, 0x0C, 0xB4, 0xB3, 0x71, 0x33, 0x57, 0x18, 0x03, 0x57,
      0x09, 0xF3, 0x67, 0x10, 0x64, 0x71, 0x57, 0x14, 0x1E, 0x6F, 0x59, 0x6F,
      0xF6, 0x77, 0xB1, 0x32, 0x35, 0x2D, 0x3C, 0x35, 0x38, 0x99, 0x03, 0x37,
      0x59, 0x99, 0xB2, 0xA0, 0xD4, 0x60, 0xCF, 0x16, 0xA5, 0x81, 0xC3, 0xCE,
      0x5A, 0x05, 0x7D, 0xF4, 0x20, 0x51, 0xA7, 0xBA, 0x16, 0x7D, 0x66, 0x47,
      0xBD, 0x68, 0x32, 0x90, 0xAB, 0xBE, 0xB3, 0xE6, 0x72, 0x77, 0xC0, 0x55,
      0xC5, 0xE7, 0x06, 0x74, 0x55, 0x5B, 0x98, 0xD3, 0x65, 0x44, 0x2B, 0xB8,
      0xD2, 0x85, 0x9F, 0xE4, 0x7C, 0x13, 0x09, 0x17, 0x85, 0x37, 0xE7, 0x8D,
      0xA7, 0x9B, 0x70, 0x16, 0x89, 0x0C, 0x94, 0x86, 0x43, 0x28, 0xCA, 0xCF,
      0x32, 0x77, 0x46, 0xD0, 0xE3, 0x2D, 0x65, 0xD9, 0xC7, 0xC0, 0x57, 0x4F,
      0xA5, 0xBA, 0x9E, 0x1F, 0xBB, 0xA7, 0x2D, 0x99, 0x0C, 0x62, 0xE1, 0xAD,
      0x74, 0x75, 0x9B, 0xAD, 0x3B, 0x7F, 0x97, 0x59, 0xB3, 0x34, 0xB1, 0x0E,
      0xC8, 0xA1, 0xAC, 0x3D, 0x84, 0x53, 0x78, 0x3A, 0x62, 0x1C, 0x89, 0x8C,
      0x99, 0x94, 0x51, 0x95, 0x92, 0x84, 0x45, 0xD2, 0x1B, 0x6F, 0x40, 0x21,
      0x57, 0x24, 0x91, 0x9C, 0x39, 0x59, 0xD7, 0x77, 0x31, 0xC2, 0xEC, 0xB1,
      0x71, 0x48, 0x9A, 0xF6, 0xF5, 0x45, 0xC7, 0x05, 0x5D, 0x55, 0x77, 0x8C,
      0xAD, 0xD2, 0x50, 0xDF, 0x49, 0x91, 0x3A, 0x77, 0x69, 0xD1, 0x47, 0x74,
      0xD8, 0x79, 0x40, 0xD9, 0xC0, 0xC5, 0x39, 0x69, 0xBB, 0x2A, 0xA1, 0x51,
      0xE0, 0x63, 0xCF, 0x78, 0xCC, 0x54, 0x1E, 0xF6, 0x43, 0xAD, 0x7C, 0x91,
      0x6B, 0x96, 0x19, 0x83, 0x98, 0x3F, 0x88, 0x6C, 0x7D, 0x7B, 0x6A, 0x01,
      0x63, 0x91, 0xC6, 0xB9, 0xC6, 0x4E, 0x95, 0xA0, 0x3F, 0xD1, 0xF6, 0x5B,
      0x6C, 0x92, 0x4B, 0x23, 0x53, 0x67, 0x8D, 0x30, 0x79, 0x67, 0x6C, 0x25,
      0xE2, 0x36, 0x74, 0x59, 0x55, 0xCE, 0x5A, 0x95, 0x00, 0xBD, 0xE9, 0x6A,
      0x12, 0x87, 0xB6, 0xA7, 0x95, 0xB3, 0x38, 0x2B, 0x43, 0x8E, 0x64, 0xC5,
      0x88, 0x87, 0x8C, 0xE4, 0x47, 0x53, 0x0D, 0x17, 0x4B, 0x53, 0x64, 0x14,
      0x16, 0xE3, 0x28, 0xE4, 0x28, 0xBB, 0xC3, 0x47, 0xAA, 0xD3, 0x23, 0x42,
      0xC0, 0xF1, 0x7D, 0x1E, 0xA1, 0x5F, 0xBE, 0x14, 0x7E, 0x3D, 0x00, 0x09,
      0xC1, 0xCA, 0x0C, 0x0E, 0x5A, 0x43, 0x7F, 0x17, 0x08, 0xA6, 0x02, 0x02,
      0x6F, 0x02, 0x52, 0xBB, 0xEA, 0x8B, 0x7C, 0x70, 0x44, 0x5A, 0x45, 0x66,
      0x78, 0xE0, 0x77, 0x79, 0x94, 0xC0, 0x21, 0xF6, 0x1A, 0x8C, 0x68, 0x9D,
      0xCD, 0xA4, 0x0B, 0xB1, 0x96, 0x51, 0x2C, 0x8C, 0x53, 0xFC, 0x3B, 0x0E,
      0xC2, 0x6B, 0x3E, 0x24, 0x74, 0x41, 0x9A, 0xD7, 0x9E, 0xBF, 0xA5, 0x80,
      0x87, 0xAC, 0x89, 0x87, 0xBC, 0x15, 0xC8, 0x02, 0x02, 0x8A, 0xDB, 0xCF,
      0xB3, 0x77, 0x22, 0xB0, 0x99, 0x9F, 0x6E, 0xF7, 0x16, 0x29, 0xAC, 0x29,
      0x66, 0xFA, 0xB9, 0x5A, 0x5A, 0x2D, 0x3F, 0x74, 0x30, 0x0F, 0xE6, 0x68,
      0xF9, 0xD9, 0x12, 0xF8, 0x81, 0x55, 0x61, 0xC5, 0x3E, 0x1B, 0xD2, 0x44,
      0x55, 0xE5, 0x8F, 0xF3, 0xF7, 0xBF, 0xBC, 0x22, 0x07, 0xD7, 0x96, 0x6B,
      0x14, 0x14, 0xCD, 0x0D, 0x69, 0x5C, 0x5B, 0xAB, 0xA9, 0x36, 0x18, 0xA8,
      0x9F, 0x32, 0xCF, 0x29, 0xB3, 0x3F, 0xE9, 0x7E, 0xE9, 0x61, 0xF5, 0xDF,
      0x14, 0xFD, 0xCC, 0xD0, 0xE8, 0x18, 0x78, 0xF6, 0xC7, 0x6D, 0x56, 0x51,
      0x73, 0x0F, 0x64, 0x56, 0xDB, 0x09, 0x38, 0xBF},
     true,
     {0x76, 0xBE, 0x9F, 0xB9, 0xC5, 0xB0, 0x4D, 0x39, 0xDE, 0x78, 0x7F,
      0x15, 0xE1, 0xAA, 0xC0, 0x21, 0xBC, 0xF2, 0x40, 0xE6, 0x5D, 0x0E,
      0x07, 0x03, 0x34, 0x1F, 0x08, 0x98, 0xC6, 0xE6, 0x75, 0xD4},
     {0xA2, 0xF6, 0x46, 0xAC, 0x5A, 0x87, 0x35, 0x5F, 0xBF, 0xE9, 0xA3,
      0x7E, 0x58, 0xF4, 0x05, 0x42, 0x02, 0x21, 0xE5, 0x23, 0x84, 0x4C,
      0x9D, 0x00, 0xAB, 0x08, 0x9E, 0xFA, 0x0F, 0xAB, 0xF2, 0x80}},

    {params_ml_kem768_test_mode,
     {0x52, 0xCE, 0xBD, 0xEC, 0xF0, 0x65, 0x79, 0xF4, 0xA9, 0x35, 0x1F,
      0x77, 0xCA, 0x95, 0xB5, 0xCE, 0xDD, 0x03, 0x4D, 0x81, 0x2F, 0x3F,
      0xB7, 0xFB, 0x50, 0x32, 0x0C, 0xA8, 0x0E, 0x41, 0x18, 0xD5},
     {0x03, 0x87, 0xB2, 0xD6, 0x69, 0x85, 0x0A, 0xD9, 0x37, 0x9C, 0xA7, 0x0B,
      0x3E, 0xF1, 0xBB, 0xFD, 0x48, 0x72, 0x14, 0xF0, 0x86, 0x16, 0x82, 0x47,
      0x87, 0xC4, 0x50, 0x6C, 0x83, 0x74, 0x0B, 0xE7, 0x6C, 0xBE, 0x45, 0x2C,
      0xF9, 0x93, 0x69, 0xCA, 0x80, 0x67, 0x4D, 0x3B, 0x22, 0xD3, 0xE0, 0x4F,
      0xF9, 0x50, 0x93, 0xBD, 0x00, 0x36, 0x9B, 0xDC, 0x5C, 0x81, 0x26, 0x13,
      0x3C, 0x5A, 0x15, 0xB8, 0xA9, 0x9A, 0x91, 0x21, 0x32, 0xB1, 0x9C, 0x27,
      0x54, 0x89, 0xB7, 0x91, 0x48, 0x58, 0xE8, 0x06, 0xCF, 0x4C, 0xB1, 0x0D,
      0x3F, 0xF3, 0x3E, 0x01, 0x8C, 0xA8, 0x77, 0x6A, 0x88, 0xFA, 0x30, 0xC8,
      0x78, 0x21, 0x79, 0xD2, 0x89, 0x13, 0x2E, 0x0C, 0xA4, 0xA1, 0xBB, 0x19,
      0x46, 0x35, 0x71, 0xBE, 0x19, 0x77, 0x10, 0x9C, 0x2B, 0xBE, 0x4C, 0x94,
      0x33, 0xD6, 0x98, 0x6A, 0xF1, 0x2E, 0x70, 0x79, 0xC4, 0xB5, 0x78, 0x33,
      0x90, 0xC9, 0x5C, 0x84, 0x68, 0x54, 0x5F, 0x48, 0xA5, 0xD6, 0x69, 0xCC,
      0x05, 0x67, 0xCD, 0xBA, 0xC5, 0x44, 0x20, 0xDA, 0xB1, 0xDA, 0x63, 0x45,
      0x6C, 0x69, 0x26, 0x78, 0x10, 0x5E, 0x08, 0xA8, 0x22, 0xD0, 0xF9, 0x95,
      0x84, 0xD7, 0xA8, 0xBC, 0x76, 0xCD, 0x90, 0x38, 0xC2, 0x76, 0x86, 0x63,
      0xFA, 0x11, 0xA8, 0xD8, 0xE7, 0xB8, 0x6B, 0xA5, 0x2D, 0xD2, 0x1B, 0x32,
      0x3E, 0xC2, 0x2A, 0xC2, 0xE7, 0xB9, 0x21, 0x21, 0x62, 0x82, 0x23, 0x30,
      0x39, 0x54, 0x68, 0x03, 0xD5, 0x09, 0xFF, 0xB1, 0x2B, 0xA2, 0xFB, 0x7D,
      0x3F, 0xD3, 0x8F, 0x4B, 0x06, 0xB8, 0x48, 0x23, 0x3B, 0xC9, 0x0C, 0x2E,
      0xD6, 0x8A, 0xC4, 0x1F, 0xF2, 0x4C, 0x64, 0x25, 0xC6, 0xC9, 0x32, 0x1A,
      0x7B, 0x69, 0x7B, 0xA5, 0x58, 0x8B, 0x43, 0x4B, 0x1B, 0x5A, 0x49, 0xCB,
      0xB9, 0xFC, 0x47, 0x9B, 0xE6, 0x82, 0x57, 0x49, 0x8A, 0x05, 0x86, 0x5A,
      0x7A, 0x02, 0x17, 0xB8, 0x31, 0x9E, 0x7B, 0xB8, 0xB5, 0x5C, 0x15, 0x96,
      0x73, 0x19, 0xCC, 0x51, 0x01, 0x47, 0xBE, 0x9B, 0xA5, 0x51, 0x1A, 0x96,
      0x7C, 0xAA, 0x21, 0x02, 0xC6, 0xA4, 0x12, 0x98, 0xC0, 0xF7, 0x62, 0x0E,
      0x62, 0x66, 0x93, 0xE9, 0x76, 0x31, 0x8E, 0x6A, 0x76, 0x35, 0x53, 0x6B,
      0x2C, 0xAC, 0x9D, 0xDD, 0xFA, 0x58, 0xE2, 0x38, 0x24, 0x01, 0xAC, 0x77,
      0x8D, 0x14, 0x57, 0x74, 0x49, 0x4D, 0x2F, 0x47, 0x6F, 0x43, 0xE0, 0xC3,
      0xAE, 0xE0, 0x04, 0xD3, 0x60, 0xC4, 0xA6, 0x55, 0x71, 0xBA, 0x02, 0xC7,
      0xDA, 0x13, 0x00, 0x26, 0x61, 0xB4, 0x83, 0x8B, 0x22, 0xB9, 0x37, 0x8F,
      0x34, 0x57, 0x9F, 0x46, 0x65, 0x3A, 0x30, 0xA8, 0x8B, 0xB5, 0x01, 0xB3,
      0x25, 0x95, 0x6B, 0x0B, 0x5A, 0x26, 0x11, 0x74, 0x36, 0x0F, 0x12, 0x88,
      0x50, 0x90, 0x8B, 0x5A, 0x3C, 0xCD, 0xB5, 0xB6, 0x11, 0xCA, 0xA2, 0x67,
      0x73, 0x36, 0x8B, 0x13, 0x33, 0x0C, 0x48, 0xC6, 0x66, 0x7B, 0xB3, 0x51,
      0xA2, 0xAA, 0xAC, 0xEC, 0x4B, 0x2A, 0x2C, 0x70, 0x60, 0x15, 0x13, 0xCC,
      0x3A, 0x18, 0xB5, 0xB1, 0xDC, 0x42, 0x1C, 0x5A, 0x05, 0x6D, 0x16, 0x45,
      0xFE, 0x93, 0x12, 0x50, 0x7A, 0xA1, 0xD5, 0x29, 0x29, 0xAC, 0x28, 0x9F,
      0x6D, 0x75, 0x26, 0x9B, 0xA4, 0x20, 0x61, 0xD5, 0x56, 0x24, 0x3B, 0x34,
      0xC6, 0x38, 0xA7, 0x2D, 0xEA, 0xB9, 0x39, 0xB4, 0x96, 0x43, 0x59, 0x51,
      0xC5, 0x3B, 0x22, 0x09, 0x59, 0x06, 0xC6, 0x8B, 0x12, 0x99, 0xA2, 0x88,
      0xD5, 0xF5, 0xCC, 0xB8, 0x76, 0x2E, 0x76, 0x47, 0x7C, 0x5A, 0x06, 0x11,
      0xEB, 0x46, 0x8A, 0xDA, 0x90, 0x73, 0xC7, 0x9B, 0x09, 0xE8, 0x06, 0x21,
      0xCA, 0x9C, 0x4D, 0x69, 0xA7, 0x14, 0x6A, 0x81, 0x21, 0x33, 0xB9, 0x63,
      0x7F, 0x07, 0x45, 0x86, 0xA5, 0xBF, 0xF4, 0xAA, 0xB6, 0xCD, 0x01, 0x40,
      0xB4, 0x3A, 0x1B, 0x6E, 0x2A, 0xBE, 0xE7, 0xAC, 0x50, 0x48, 0x9B, 0x9E,
      0x52, 0x70, 0xA9, 0x35, 0x41, 0xCE, 0x03, 0xB7, 0x2C, 0x00, 0xA2, 0x6E,
      0x27, 0x4A, 0x7C, 0xD3, 0x79, 0x33, 0x34, 0xCC, 0x04, 0xE9, 0x28, 0x5F,
      0xDA, 0x67, 0x3E, 0x02, 0x41, 0xBB, 0x9C, 0x9A, 0x34, 0xAB, 0x79, 0x35,
      0x50, 0x87, 0x39, 0xE5, 0x2A, 0x29, 0xF2, 0x2C, 0xBB, 0x96, 0x64, 0x25,
      0x1C, 0x60, 0x68, 0xFD, 0x38, 0x42, 0xC8, 0x0B, 0x17, 0xA0, 0xC1, 0x77,
      0x2F, 0x28, 0x55, 0x25, 0xB2, 0x2C, 0x65, 0xAC, 0x1D, 0xCD, 0x52, 0x02,
      0x17, 0xDB, 0x57, 0x15, 0x38, 0x1C, 0x9A, 0x66, 0x3D, 0xB0, 0xE5, 0x12,
      0xB9, 0xF9, 0x71, 0x15, 0x37, 0xBE, 0xF2, 0x35, 0x31, 0x3D, 0xC4, 0x6B,
      0x75, 0x88, 0x39, 0x32, 0x13, 0x41, 0xE9, 0x53, 0x48, 0xA7, 0x59, 0x72,
      0xA7, 0xFB, 0x1D, 0xBF, 0xB1, 0x83, 0x91, 0x0A, 0x44, 0x2E, 0x98, 0x56,
      0x28, 0x84, 0x3B, 0x93, 0x50, 0x55, 0x34, 0xC6, 0x3C, 0x8D, 0x8A, 0x53,
      0x58, 0xB6, 0x1D, 0x73, 0x53, 0xC2, 0xD1, 0x74, 0xAA, 0x14, 0x16, 0x67,
      0x71, 0x72, 0xAB, 0xC6, 0x67, 0x25, 0xE7, 0x31, 0x7E, 0xCC, 0x83, 0xBF,
      0xF2, 0x57, 0x81, 0x97, 0x90, 0xC9, 0xF9, 0xC5, 0x06, 0xDB, 0xFC, 0x64,
      0x7D, 0xB2, 0x7B, 0xD9, 0x74, 0x76, 0x73, 0xF6, 0x66, 0xD1, 0x23, 0x98,
      0x45, 0x74, 0x4B, 0xE7, 0x42, 0xBB, 0x27, 0x06, 0x7B, 0xEC, 0x53, 0x43,
      0x7D, 0x48, 0xA5, 0x28, 0xF3, 0x22, 0x1D, 0x60, 0xC3, 0xE1, 0xEB, 0x00,
      0xD8, 0x70, 0x45, 0xD0, 0x50, 0xBE, 0xBC, 0xC2, 0xB4, 0x33, 0xF2, 0xBB,
      0x29, 0x62, 0xB0, 0xE2, 0xE7, 0x0F, 0xF0, 0xCC, 0x52, 0xA5, 0xBA, 0x74,
      0x12, 0xE3, 0x91, 0x94, 0xAA, 0x22, 0x68, 0x72, 0x18, 0x41, 0x08, 0xC4,
      0xFB, 0xC4, 0x59, 0x09, 0x0C, 0x59, 0xE4, 0xA2, 0x7F, 0xA4, 0x82, 0x3E,
      0xD8, 0xB3, 0xCD, 0x3C, 0x59, 0xC2, 0xCC, 0x6A, 0x81, 0xD3, 0x01, 0x7A,
      0x08, 0x23, 0x42, 0x06, 0xC3, 0x85, 0x4E, 0xB3, 0xB7, 0x5B, 0xA4, 0x41,
      0xAB, 0x8C, 0x32, 0x1C, 0x55, 0xCC, 0x85, 0x74, 0xA0, 0x4D, 0xF4, 0x2E,
      0x20, 0x11, 0x80, 0x84, 0x69, 0x82, 0x33, 0x3A, 0x92, 0xD0, 0x57, 0xC6,
      0x11, 0x40, 0x87, 0x4A, 0x9A, 0x80, 0xA4, 0x60, 0x37, 0xDC, 0x2A, 0x3C,
      0xEC, 0xD5, 0x34, 0xE9, 0xEC, 0x39, 0x94, 0x3B, 0xB3, 0x00, 0xB1, 0x77,
      0x25, 0x10, 0xB7, 0xE8, 0x18, 0x6C, 0x2D, 0x5B, 0x65, 0x55, 0x58, 0x4F,
      0x54, 0xE3, 0x9A, 0x2A, 0xB6, 0x47, 0x38, 0x06, 0xB9, 0xC9, 0xE7, 0x1A,
      0xCD, 0xC8, 0xC1, 0x5E, 0x43, 0xA8, 0x85, 0xA4, 0xC9, 0xB4, 0x27, 0x63,
      0x76, 0x6C, 0x49, 0xDD, 0xDB, 0x89, 0x0C, 0x8C, 0x7B, 0x22, 0x15, 0x9D,
      0xC9, 0x18, 0x65, 0xD0, 0xE7, 0x9A, 0xB1, 0x03, 0x72, 0xA9, 0xF4, 0x59,
      0x63, 0xC5, 0x9D, 0x8C, 0x18, 0x57, 0xCB, 0x65, 0x4B, 0x2B, 0xDA, 0xC0,
      0xAF, 0x0A, 0x24, 0xA6, 0x63, 0xC7, 0x1B, 0x74, 0x4D, 0xAD, 0x1A, 0xB6,
      0x8A, 0x90, 0x46, 0x2B, 0xD7, 0x9F, 0x6B, 0xA5, 0x53, 0x76, 0x12, 0x55,
      0x62, 0xA5, 0xB1, 0x32, 0x6A, 0xAB, 0xA7, 0xC1, 0x9E, 0x59, 0x3B, 0x7D,
      0x0C, 0xC1, 0xC8, 0x71, 0x1B, 0x8C, 0xFE, 0x35, 0x47, 0xA0, 0x0A, 0xCC,
      0xE1, 0x64, 0x6E, 0x76, 0x6B, 0xC0, 0x37, 0x67, 0xCF, 0x5B, 0xFB, 0xAD,
      0x8F, 0x08, 0x59, 0x0E, 0x21, 0x65, 0xC7, 0x70, 0x65, 0x44, 0xDB, 0x5A,
      0x94, 0x29, 0x25, 0x07, 0x0A, 0x87, 0x8D, 0x13, 0x41, 0x8D, 0x31, 0xCE,
      0xD3, 0x50, 0x92, 0xF1, 0x9C, 0x19, 0x7D, 0xC2, 0xA5, 0x17, 0x96, 0x7E,
      0x41, 0x89, 0x87, 0x96, 0xA5, 0x1A, 0xF8, 0x91, 0x1A, 0x18, 0x57, 0x24,
      0x4C, 0x08, 0x50, 0x45, 0x67, 0x78, 0x65, 0xB1, 0x25, 0x57, 0x3C, 0xA5,
      0xE2, 0x42, 0x31, 0x25, 0x12, 0x43, 0x68, 0x2C, 0x6D, 0xCE, 0xE5, 0x24,
      0x08, 0x85, 0x72, 0xD2, 0x01, 0x5D, 0x9A, 0x25, 0x69, 0x1F, 0x25, 0x32,
      0xE7, 0x59, 0x07, 0xC1, 0x07, 0x32, 0x37, 0x31, 0x59, 0x1B, 0xB0, 0x61,
      0xD5, 0x25, 0x75, 0x75, 0x40, 0x33, 0x8B, 0xE1, 0x99, 0x13, 0x32, 0x3D,
      0x08, 0xA4, 0x08, 0x22, 0x39, 0x36, 0x3C, 0xE4, 0x74, 0x1C, 0x92, 0xAA,
      0x29, 0xF8, 0xCA, 0xF1, 0x46, 0xAF, 0x1D, 0x27, 0x7A, 0x76, 0xBB, 0x22,
      0xCD, 0x54, 0x66, 0x39, 0x14, 0x64, 0x75, 0x32, 0xAE, 0xFD, 0xA0, 0x24,
      0x06, 0x09, 0x9A, 0x75, 0xA6, 0x3F, 0x7F, 0x2C, 0xA5, 0xBE, 0xFC, 0x74,
      0xA6, 0x72, 0x48, 0x96, 0xCA, 0xB8, 0x4D, 0x12, 0x37, 0x67, 0x44, 0xCC,
      0xB1, 0xC6, 0xEC, 0xB1, 0xDC, 0xAB, 0xFD, 0x20, 0xAA, 0xEB, 0x88, 0xBD,
      0xBD, 0x04, 0xAA, 0x5A, 0x7E, 0x2C, 0x86, 0x7B},
     true,
     {0x32, 0xE0, 0x4E, 0x1E, 0x3A, 0x2C, 0x36, 0x20, 0x0E, 0xB8, 0xB0,
      0xF4, 0x05, 0x76, 0x46, 0xA3, 0x47, 0x07, 0x37, 0x70, 0x80, 0xB6,
      0x31, 0x25, 0x0A, 0xBD, 0x3D, 0x0D, 0x6D, 0xC9, 0x99, 0x62},
     {0xDE, 0xC4, 0x78, 0x07, 0x93, 0xA6, 0x1D, 0xC6, 0x22, 0x21, 0x67,
      0x54, 0x7E, 0x25, 0x1B, 0xEC, 0x41, 0x9B, 0x28, 0x28, 0x83, 0xB1,
      0x8F, 0x9B, 0xD0, 0x6E, 0x05, 0x3D, 0xB2, 0x58, 0xC1, 0x74}},

    {params_ml_kem768_test_mode,
     {0x16, 0x18, 0x89, 0xF2, 0xE9, 0x2B, 0x1B, 0xB2, 0x8A, 0x25, 0x7B,
      0x45, 0xD1, 0x79, 0xFB, 0x76, 0x84, 0x7B, 0x66, 0x4E, 0x6D, 0x7B,
      0x5F, 0xD9, 0x69, 0x82, 0x04, 0xA4, 0x26, 0xEE, 0x96, 0xEC},
     {0x82, 0xDC, 0xBC, 0x98, 0x65, 0x0A, 0x04, 0x86, 0x1D, 0xDF, 0x15, 0x38,
      0x0C, 0x86, 0x44, 0xC6, 0xB9, 0x3F, 0x19, 0x7A, 0x5B, 0x10, 0x70, 0x2C,
      0xB9, 0x44, 0x43, 0x9B, 0xF7, 0xAA, 0xFF, 0x09, 0x0C, 0x49, 0xE8, 0xCA,
      0x58, 0xDC, 0x50, 0x7E, 0x56, 0x43, 0xC1, 0x7A, 0x4D, 0x29, 0x12, 0xBA,
      0xCF, 0xB7, 0x64, 0x54, 0xD4, 0x5F, 0xD6, 0xEA, 0xCD, 0x19, 0x19, 0x10,
      0xB4, 0x72, 0x46, 0x3C, 0x49, 0xB7, 0x66, 0x84, 0x77, 0x7B, 0xFA, 0x71,
      0xBC, 0x18, 0x97, 0x36, 0x77, 0x25, 0x6F, 0x64, 0x9F, 0xA6, 0x04, 0x1A,
      0x61, 0x58, 0x04, 0x6F, 0x75, 0x26, 0x8E, 0x7C, 0xB7, 0x2E, 0x8A, 0x97,
      0x4E, 0xB2, 0xCC, 0x7D, 0xF1, 0xB8, 0xAB, 0x45, 0xB0, 0xC6, 0x51, 0xBF,
      0x3D, 0x99, 0x21, 0x1C, 0x07, 0x1D, 0x55, 0xB9, 0x44, 0x3E, 0x6A, 0x4A,
      0x65, 0xB9, 0x76, 0x96, 0x23, 0x00, 0xE5, 0xF7, 0x32, 0x5A, 0x22, 0x8A,
      0x61, 0x72, 0x7D, 0x87, 0x33, 0xC0, 0xB3, 0x01, 0x2C, 0xC5, 0x1C, 0x23,
      0x32, 0xFC, 0x5B, 0xED, 0xB0, 0x59, 0x62, 0xB7, 0x71, 0xA2, 0x32, 0xB5,
      0x5B, 0x4B, 0xCB, 0x41, 0xAC, 0x4D, 0x85, 0xFC, 0x6B, 0xE5, 0x38, 0x0E,
      0xCD, 0x32, 0x59, 0xCD, 0xE1, 0xB8, 0x09, 0xD8, 0xB6, 0xE6, 0x79, 0x78,
      0x21, 0x37, 0x98, 0xA8, 0x28, 0x54, 0x0A, 0xDB, 0x86, 0x12, 0x4B, 0x6F,
      0x13, 0x7C, 0xBE, 0xDC, 0x4B, 0x64, 0xE7, 0xCC, 0xAC, 0xF7, 0x80, 0x31,
      0x21, 0x1A, 0x30, 0x0A, 0x79, 0xC5, 0x95, 0x08, 0xA6, 0x0A, 0x73, 0xAB,
      0xF9, 0x35, 0x80, 0xAA, 0x64, 0xA7, 0xE6, 0x61, 0xC2, 0x82, 0xF5, 0xC0,
      0x0A, 0xD8, 0x78, 0x4B, 0x00, 0x2F, 0x5A, 0xC7, 0x9E, 0xC4, 0xAC, 0x85,
      0xE0, 0x29, 0x0F, 0xA8, 0x61, 0xB9, 0xD8, 0xB3, 0x0D, 0x9E, 0x56, 0xCB,
      0x93, 0x4C, 0x6F, 0x4C, 0xD5, 0xC0, 0x33, 0xC7, 0xAF, 0x7F, 0xB5, 0x5F,
      0x20, 0xC5, 0xA1, 0x15, 0x74, 0xB3, 0xA4, 0x60, 0xC7, 0xA3, 0xD7, 0x33,
      0x76, 0xE6, 0xCA, 0x77, 0xAB, 0x91, 0x1C, 0xA1, 0x18, 0xC9, 0x63, 0xB9,
      0x7A, 0xB6, 0x75, 0xAC, 0xAA, 0xC6, 0xBB, 0x73, 0xB0, 0x08, 0x0B, 0xB5,
      0xE0, 0x26, 0x1F, 0x79, 0xE4, 0x0C, 0x0F, 0xCA, 0x91, 0x29, 0xA9, 0xC1,
      0x18, 0xEA, 0x70, 0x5B, 0x0B, 0xC7, 0xA2, 0xB4, 0x5F, 0xF0, 0x6C, 0xC1,
      0x52, 0x14, 0xA8, 0x07, 0xD3, 0x52, 0x10, 0x26, 0xBF, 0xF0, 0x08, 0x02,
      0x4E, 0x95, 0x1A, 0xA0, 0x75, 0x2B, 0xFE, 0xC0, 0x22, 0xCC, 0xF9, 0x2A,
      0x51, 0x9B, 0x8D, 0x1C, 0xA9, 0x10, 0x80, 0x9B, 0x63, 0xCF, 0xA5, 0x7C,
      0x34, 0x14, 0x90, 0xBC, 0x67, 0x32, 0x43, 0xB7, 0xCC, 0x3E, 0x2A, 0x6A,
      0xEA, 0x2B, 0xAD, 0x2B, 0x88, 0x5E, 0x39, 0x73, 0x8E, 0x39, 0x73, 0x01,
      0x1A, 0x67, 0xB2, 0x66, 0xC4, 0x38, 0xF2, 0xC2, 0xC1, 0xF6, 0x3B, 0x25,
      0x15, 0x58, 0x94, 0x7C, 0x00, 0x74, 0x19, 0x39, 0x27, 0x80, 0x55, 0xAC,
      0xB8, 0x56, 0x9F, 0x91, 0x11, 0x0D, 0x89, 0x5B, 0x97, 0xDC, 0x07, 0xBA,
      0xCF, 0x13, 0x89, 0xEE, 0xD1, 0x27, 0x10, 0x51, 0xCB, 0xF6, 0xA7, 0xB6,
      0x33, 0xA4, 0x78, 0x48, 0xFB, 0x57, 0xBE, 0xF4, 0x85, 0x99, 0x3C, 0x54,
      0x62, 0xE3, 0xB5, 0x54, 0x8B, 0x96, 0xCD, 0x16, 0x1D, 0x58, 0x51, 0x07,
      0x94, 0xCA, 0x60, 0x95, 0x1A, 0x2A, 0x04, 0x6C, 0x87, 0x43, 0x80, 0x76,
      0x19, 0x86, 0x1D, 0x37, 0xA4, 0x20, 0xDB, 0xE9, 0x46, 0x33, 0x87, 0x71,
      0x34, 0xA9, 0x21, 0x80, 0xC0, 0x5B, 0x14, 0xE1, 0x06, 0x15, 0x78, 0x3C,
      0x7B, 0x32, 0xB1, 0x35, 0x42, 0x8E, 0x59, 0x31, 0x1B, 0xAA, 0xA3, 0x2F,
      0xE3, 0x7A, 0x7C, 0x35, 0xC4, 0x39, 0x0D, 0x11, 0x85, 0xF2, 0x93, 0x1B,
      0xAD, 0x53, 0x9F, 0x08, 0xA0, 0x21, 0x98, 0xD6, 0x45, 0x0D, 0xC5, 0x5B,
      0xFE, 0xF6, 0x25, 0xE3, 0xE2, 0x09, 0xEE, 0xF2, 0xA7, 0xF3, 0x80, 0x74,
      0xDC, 0x68, 0x11, 0x26, 0xCB, 0xBE, 0xE6, 0xAA, 0x3C, 0x97, 0x4B, 0xA8,
      0xD7, 0xAC, 0xCB, 0xA7, 0x19, 0x29, 0x2C, 0xFA, 0x33, 0xE4, 0xD5, 0x09,
      0x4F, 0x52, 0x7B, 0xE9, 0x4A, 0xA2, 0x23, 0x1A, 0x76, 0x2C, 0xC0, 0x97,
      0x0C, 0x7C, 0x81, 0x79, 0x34, 0x71, 0x29, 0x5B, 0x9B, 0x2A, 0x58, 0x81,
      0x09, 0x52, 0xA6, 0x3B, 0x65, 0x89, 0xB2, 0x99, 0x53, 0xC9, 0x42, 0xCE,
      0x49, 0xC8, 0x7F, 0xB7, 0xB4, 0x9B, 0x40, 0x02, 0x10, 0xFF, 0x64, 0x5C,
      0x88, 0x17, 0x9F, 0xD7, 0x6B, 0xA4, 0x08, 0xDA, 0x81, 0xC7, 0x46, 0x4E,
      0x76, 0xFC, 0xC3, 0x32, 0x95, 0x89, 0xAA, 0x99, 0x85, 0x55, 0x64, 0x32,
      0x9D, 0x89, 0x14, 0xE3, 0x4A, 0x7B, 0x0D, 0x02, 0x68, 0x5A, 0x92, 0x81,
      0x7C, 0x28, 0xB2, 0xB3, 0xF8, 0xA7, 0xA7, 0x61, 0x5B, 0xBD, 0x56, 0x9F,
      0x2D, 0xAC, 0x20, 0xE1, 0x61, 0x07, 0x7E, 0xF4, 0x6C, 0x6F, 0xB9, 0x26,
      0xAC, 0xF3, 0xC9, 0x4D, 0xCC, 0x9A, 0x4B, 0x0A, 0x27, 0x64, 0x13, 0x70,
      0xB4, 0x02, 0x7A, 0xB0, 0x82, 0x89, 0x6E, 0x18, 0x8B, 0xC5, 0xCA, 0x32,
      0xBD, 0xE9, 0x11, 0xC5, 0xD8, 0x65, 0xC8, 0x7A, 0x22, 0x20, 0x7A, 0xC3,
      0x5B, 0x72, 0xB0, 0xB8, 0xFB, 0x34, 0x2E, 0x3A, 0x53, 0xBF, 0x73, 0x57,
      0x96, 0xF3, 0xC8, 0x93, 0x04, 0x17, 0xCA, 0x84, 0xCE, 0xBD, 0xA4, 0x8F,
      0xC1, 0xE3, 0x4A, 0xF9, 0x61, 0x0F, 0x88, 0xC1, 0x85, 0x23, 0xA4, 0x03,
      0x7E, 0x35, 0x40, 0x75, 0x71, 0x62, 0x87, 0x5C, 0xB5, 0xF8, 0x2B, 0x17,
      0x18, 0xB3, 0x2E, 0xE7, 0x0A, 0x39, 0xE5, 0xB2, 0x16, 0x22, 0x0A, 0xB3,
      0x14, 0x8C, 0xB1, 0xCC, 0xD1, 0x18, 0xCE, 0x21, 0x3F, 0x2D, 0x56, 0x80,
      0xEB, 0xF8, 0x65, 0xAB, 0x01, 0x86, 0x31, 0xC8, 0x8F, 0xDB, 0x6C, 0x76,
      0x3C, 0x08, 0x2F, 0x59, 0xC6, 0xCD, 0x95, 0x15, 0x65, 0x95, 0xE2, 0x2F,
      0xB0, 0xDA, 0xAC, 0xB5, 0x63, 0x41, 0xB7, 0x64, 0x50, 0x27, 0x84, 0x58,
      0x13, 0xA5, 0x9D, 0xF2, 0x03, 0x10, 0x4D, 0x9C, 0x8B, 0x1F, 0x64, 0x7B,
      0xD3, 0x53, 0x79, 0x37, 0xF5, 0x90, 0x30, 0x97, 0x94, 0x63, 0xBB, 0x63,
      0xF7, 0xC9, 0xB6, 0x9E, 0x57, 0xCA, 0x17, 0x49, 0x1F, 0x36, 0x18, 0xCD,
      0xD1, 0x62, 0xA9, 0xFE, 0x1C, 0x3F, 0x9B, 0x32, 0x6F, 0x56, 0x7C, 0x26,
      0xD1, 0xAC, 0x60, 0x50, 0x68, 0x5F, 0xC6, 0x8A, 0x93, 0xCF, 0x59, 0x17,
      0x8B, 0xB8, 0x7E, 0xC8, 0xD8, 0xB7, 0x03, 0x51, 0x9E, 0x60, 0xA2, 0x27,
      0x02, 0xBB, 0x5F, 0xBC, 0xF3, 0xBC, 0xD6, 0xD4, 0x96, 0xB7, 0x9B, 0x65,
      0x41, 0x5C, 0x17, 0x5C, 0x55, 0x42, 0x98, 0x57, 0x2A, 0xAE, 0x35, 0x43,
      0x7B, 0xF8, 0x99, 0xAD, 0xAA, 0x5C, 0x4F, 0x51, 0x3E, 0x9F, 0x22, 0x05,
      0x53, 0x09, 0x3B, 0xEB, 0xF1, 0x76, 0x97, 0x43, 0xC6, 0xAB, 0x0B, 0x9F,
      0xB6, 0x47, 0x03, 0xC6, 0x30, 0x5C, 0xC6, 0x4B, 0x84, 0xCD, 0x76, 0xB7,
      0xC8, 0xE6, 0x14, 0x29, 0x78, 0x1A, 0x82, 0x29, 0x90, 0xB6, 0x98, 0x85,
      0x2A, 0x3B, 0x58, 0x3A, 0xF4, 0x8C, 0x3B, 0x22, 0x09, 0x2C, 0x51, 0x3A,
      0x22, 0x31, 0x10, 0x58, 0x99, 0x41, 0xD5, 0x30, 0xC8, 0xB4, 0xCA, 0x25,
      0xCB, 0x43, 0x61, 0xF2, 0x00, 0x1B, 0x16, 0x6C, 0x85, 0xCF, 0x91, 0x0F,
      0x00, 0x6C, 0x5B, 0x0B, 0x71, 0xA1, 0x09, 0xAC, 0x59, 0x86, 0xF8, 0x26,
      0x4B, 0xA2, 0x65, 0x63, 0xE6, 0x9D, 0x12, 0x26, 0x44, 0xB6, 0x34, 0x88,
      0x18, 0x14, 0x5C, 0xC0, 0x26, 0xB7, 0x83, 0xD9, 0x17, 0xE7, 0x4C, 0x4F,
      0x46, 0xE0, 0xC9, 0xBF, 0x86, 0x25, 0x0F, 0x52, 0x1C, 0x21, 0xB6, 0x4D,
      0x50, 0xE9, 0x10, 0xD1, 0xF1, 0xCA, 0xEF, 0x4B, 0x05, 0xF5, 0xA5, 0x04,
      0x91, 0xE9, 0xCA, 0xC2, 0xF3, 0xB5, 0xCE, 0x36, 0x40, 0x7D, 0x99, 0x37,
      0xD0, 0xE2, 0x8E, 0x47, 0xB5, 0x2A, 0xC4, 0x60, 0x10, 0x3F, 0x27, 0x26,
      0x42, 0x09, 0x3C, 0xC9, 0xEB, 0x18, 0xBA, 0xD4, 0x4A, 0xD3, 0x6B, 0x8E,
      0xEC, 0x36, 0x7F, 0x15, 0x51, 0x59, 0xAD, 0xBA, 0x02, 0x51, 0xF1, 0x95,
      0xE5, 0x20, 0x32, 0xF1, 0xB0, 0x51, 0x4B, 0xE8, 0x3F, 0x8D, 0xEB, 0xCB,
      0xBF, 0xDB, 0x57, 0x55, 0x26, 0x7E, 0x76, 0xB0, 0x28, 0xBD, 0x07, 0xCB,
      0x83, 0x27, 0xCD, 0x43, 0x1A, 0x7B, 0x73, 0xD2, 0x89, 0x24, 0x72, 0x10,
      0xEC, 0x90, 0x5A, 0x05, 0x29, 0x23, 0x4B, 0x2C, 0x62, 0xC7, 0xA6, 0x63,
      0x38, 0xC1, 0xD3, 0x81, 0xC8, 0x84, 0x66, 0xB4, 0x83, 0x22, 0x04, 0xB1,
      0xB0, 0x5C, 0xD1, 0xBF, 0x8E, 0x0A, 0x46, 0x93, 0xD9, 0x41, 0xA1, 0x78,
      0xF6, 0x2E, 0x9E, 0x09, 0xB7, 0x4C, 0xAD, 0x5B},
     true,
     {0xA0, 0xF6, 0x56, 0x9E, 0xEB, 0x62, 0xA3, 0x2C, 0xF3, 0x76, 0x1C,
      0x55, 0x30, 0x97, 0x0B, 0x19, 0xBC, 0x0F, 0x58, 0xE7, 0xCB, 0x41,
      0xB2, 0xFC, 0x9C, 0xE2, 0x17, 0xBE, 0xF4, 0xF8, 0x7D, 0x6C},
     {0x9D, 0xC0, 0xB2, 0xED, 0x91, 0xCF, 0x46, 0x09, 0xFF, 0xB8, 0xF7,
      0x24, 0x0D, 0x6C, 0xD3, 0xF6, 0x5D, 0x45, 0x10, 0x5A, 0x35, 0x77,
      0x0A, 0x10, 0x5B, 0x91, 0x0B, 0xD9, 0xCC, 0x91, 0x1C, 0xD1}},

    {params_ml_kem768_test_mode,
     {0x33, 0x49, 0x55, 0x7D, 0xA7, 0x0F, 0xF6, 0x98, 0x86, 0xED, 0x03,
      0x2A, 0x91, 0xD8, 0xFC, 0x23, 0xBE, 0x9E, 0x52, 0x45, 0x40, 0x66,
      0x70, 0x67, 0x9A, 0x6E, 0x92, 0xAE, 0xD8, 0x70, 0xD3, 0x69},
     {0xFC, 0x96, 0x4F, 0xC0, 0x82, 0x0E, 0x5D, 0xE7, 0xA7, 0x3B, 0xC5, 0x07,
      0x46, 0x9B, 0x01, 0x3F, 0x2C, 0x81, 0xA2, 0x25, 0xC8, 0xC0, 0x67, 0xC4,
      0xA9, 0x35, 0x14, 0x67, 0x84, 0x7D, 0xC4, 0xD3, 0x8F, 0xF3, 0x23, 0x7A,
      0x9A, 0x38, 0xA8, 0xF7, 0xC2, 0x73, 0xB0, 0x8B, 0x26, 0x0C, 0x5A, 0x9B,
      0x20, 0xF8, 0x49, 0x3A, 0x66, 0x84, 0x59, 0xE0, 0xBC, 0xAC, 0x48, 0x0E,
      0x0B, 0xB5, 0x0B, 0xC6, 0x29, 0x41, 0x72, 0x02, 0x83, 0xF7, 0x58, 0xB7,
      0xAB, 0x13, 0xB4, 0x44, 0xE8, 0xA1, 0xF8, 0x36, 0x6F, 0xB2, 0x47, 0xC3,
      0xCC, 0x33, 0x9D, 0x41, 0x36, 0x71, 0xAC, 0xDA, 0x63, 0x54, 0x22, 0x0B,
      0x9C, 0xA0, 0x51, 0x58, 0x81, 0xA6, 0x88, 0x06, 0x3F, 0xA8, 0x74, 0x87,
      0x35, 0x6C, 0x84, 0x23, 0x24, 0xB3, 0x5B, 0x06, 0x4B, 0x58, 0x47, 0xA7,
      0x55, 0x09, 0xAF, 0xF0, 0x86, 0x08, 0x6C, 0xAB, 0x80, 0xDE, 0x6C, 0x72,
      0x18, 0xD9, 0x58, 0x77, 0x31, 0x3B, 0x06, 0x63, 0x03, 0x1E, 0x86, 0x76,
      0x10, 0x10, 0x7F, 0x14, 0x91, 0xA9, 0xF0, 0x69, 0xA2, 0xDA, 0xA4, 0xA8,
      0x8F, 0x26, 0xCD, 0x6C, 0xB9, 0x7B, 0x43, 0xB5, 0x1F, 0xB7, 0x15, 0x6A,
      0x47, 0x55, 0x04, 0x34, 0x42, 0x10, 0x24, 0xCC, 0x36, 0x05, 0x00, 0xAA,
      0x75, 0x95, 0xBB, 0xBA, 0x99, 0x11, 0x60, 0x57, 0xA6, 0x4A, 0x98, 0xC7,
      0x6D, 0x61, 0x08, 0x2A, 0xBC, 0x47, 0xBB, 0xC2, 0x61, 0x4D, 0x2B, 0x23,
      0xEE, 0xF8, 0xA6, 0x8D, 0x10, 0x65, 0xAB, 0xBC, 0xC0, 0x57, 0xB2, 0x71,
      0x7D, 0xBA, 0x5D, 0x6B, 0x82, 0xBE, 0xC6, 0x68, 0x35, 0x51, 0xF0, 0x82,
      0xC8, 0x5C, 0x3C, 0x43, 0x05, 0x80, 0xD7, 0x12, 0x5D, 0x1E, 0xBB, 0x1E,
      0xA4, 0x10, 0x74, 0x8D, 0x68, 0x35, 0xC5, 0xBA, 0x08, 0x0D, 0x11, 0x40,
      0xF9, 0x57, 0x4A, 0x44, 0x60, 0x75, 0x2C, 0x52, 0x78, 0xE5, 0xF2, 0x4D,
      0xA8, 0x29, 0x23, 0x5D, 0xEA, 0xCB, 0xFC, 0x36, 0x63, 0xFF, 0x71, 0x42,
      0x1F, 0x3B, 0x4C, 0x3F, 0x07, 0x37, 0x64, 0xD1, 0x1C, 0x39, 0xC9, 0xAB,
      0x6B, 0x6B, 0x2F, 0x1F, 0x24, 0xA7, 0xCC, 0x84, 0xAA, 0x73, 0xA9, 0x76,
      0x72, 0x1A, 0xA7, 0xBB, 0xEC, 0x51, 0x5D, 0x0A, 0x06, 0x13, 0xD0, 0x6F,
      0x80, 0xFA, 0x46, 0x1A, 0x03, 0x2A, 0x27, 0x49, 0x11, 0xB8, 0x2C, 0xCB,
      0x9B, 0x9A, 0x4A, 0xC8, 0x6C, 0x35, 0x55, 0xD3, 0x7A, 0x26, 0x7C, 0x54,
      0x13, 0x7B, 0x14, 0x1A, 0x04, 0x3E, 0x9C, 0x86, 0x1C, 0x58, 0xD6, 0xA8,
      0x8F, 0x5A, 0x47, 0xB8, 0x95, 0xAD, 0xDC, 0xA7, 0x36, 0xD1, 0x1C, 0xB2,
      0x27, 0xD1, 0x90, 0x48, 0x1B, 0x4F, 0xDF, 0x68, 0x09, 0x43, 0x53, 0x94,
      0x37, 0xB0, 0x4B, 0xA2, 0x02, 0xB2, 0x1B, 0x19, 0xC0, 0x6B, 0xC8, 0x03,
      0x7B, 0x93, 0x68, 0x1E, 0x21, 0x21, 0xE7, 0xA4, 0x9D, 0xB8, 0x27, 0x14,
      0x22, 0x83, 0x4C, 0xA4, 0x25, 0x3F, 0x5B, 0x96, 0xAB, 0x7E, 0x72, 0x3F,
      0xDC, 0x59, 0x9B, 0x8B, 0xF0, 0x56, 0x7C, 0x17, 0x6B, 0xC7, 0xC7, 0x54,
      0xB6, 0x40, 0x26, 0x42, 0x92, 0x1A, 0xDB, 0x8C, 0x30, 0xCB, 0x83, 0x29,
      0xA1, 0xE3, 0xCC, 0xFF, 0xC8, 0xA5, 0x01, 0xA0, 0x77, 0xE3, 0x44, 0x95,
      0xD9, 0x33, 0x3B, 0x89, 0x7C, 0x41, 0xF9, 0xE2, 0xA1, 0x11, 0x51, 0x90,
      0x2A, 0x9B, 0x19, 0xAE, 0xE1, 0x6B, 0xD4, 0x03, 0x8D, 0xCF, 0xA6, 0x11,
      0x81, 0x92, 0xCC, 0x3B, 0x09, 0x52, 0x3B, 0xD8, 0x9C, 0x0D, 0xD1, 0x58,
      0xB8, 0x72, 0x76, 0xA6, 0x31, 0x6D, 0xA1, 0x18, 0x78, 0x55, 0x51, 0xB9,
      0xE1, 0x07, 0x8C, 0x22, 0x94, 0x9D, 0x6D, 0x53, 0x02, 0xEE, 0x59, 0x45,
      0xF8, 0x7C, 0x5A, 0x8E, 0xBB, 0x62, 0x20, 0xB6, 0x73, 0xB8, 0xB7, 0xAE,
      0x64, 0x99, 0xC9, 0x93, 0x63, 0x3E, 0x84, 0x4B, 0xA8, 0xA1, 0x4C, 0xCB,
      0xDF, 0xA9, 0x31, 0xD1, 0x14, 0x5B, 0xBC, 0x26, 0x0E, 0xC9, 0x33, 0xCA,
      0x2B, 0x8B, 0x86, 0x1F, 0x50, 0x5A, 0x96, 0x91, 0xCE, 0xB2, 0xB5, 0x71,
      0x6C, 0xDB, 0x1E, 0x07, 0xDC, 0x6E, 0xCA, 0x06, 0x93, 0xCE, 0x0B, 0x8B,
      0xBE, 0xAC, 0xB5, 0x9F, 0x70, 0x58, 0x14, 0xF2, 0xC4, 0x6B, 0x33, 0x6C,
      0x65, 0x24, 0x0B, 0xA7, 0x79, 0x1B, 0x10, 0x22, 0x6D, 0x38, 0x92, 0xAC,
      0x60, 0xD1, 0x1F, 0x67, 0xF2, 0x92, 0x83, 0xEC, 0xBC, 0xB4, 0x59, 0xCA,
      0xDF, 0x68, 0x67, 0xF4, 0x18, 0x47, 0xA9, 0x9A, 0x2E, 0x04, 0x69, 0xA4,
      0x17, 0x07, 0x8A, 0xD8, 0x0B, 0x1B, 0xB3, 0x55, 0x49, 0xFE, 0xA2, 0x16,
      0x4B, 0x8B, 0x55, 0x7B, 0x56, 0x10, 0xC6, 0x4C, 0x81, 0x0E, 0x21, 0x1D,
      0xB6, 0x97, 0xAD, 0xC4, 0x30, 0x34, 0xDE, 0x45, 0xBF, 0x40, 0x77, 0x29,
      0xB9, 0xF8, 0xCC, 0xBE, 0x33, 0x79, 0xD4, 0x71, 0x36, 0x7D, 0xE8, 0x81,
      0xE6, 0xEC, 0x74, 0x1A, 0x41, 0x30, 0xC7, 0x8A, 0x90, 0xFF, 0x24, 0xC7,
      0xC3, 0x89, 0x91, 0x29, 0x03, 0x88, 0x51, 0x06, 0x9B, 0xC6, 0x8B, 0x9E,
      0xB5, 0x33, 0x35, 0x4A, 0xAB, 0x15, 0x5F, 0x82, 0x9F, 0x6A, 0xEC, 0x9E,
      0xC9, 0xD8, 0x2B, 0x20, 0xB4, 0x92, 0x7E, 0xEC, 0xC2, 0x43, 0xF0, 0x14,
      0xEC, 0x84, 0x6D, 0x59, 0xD5, 0xC8, 0xF3, 0x51, 0x7F, 0x96, 0x87, 0x35,
      0x11, 0x96, 0x74, 0x08, 0x12, 0x8A, 0x17, 0x05, 0xB8, 0x34, 0xF8, 0x1C,
      0x4F, 0x11, 0x69, 0xEA, 0x50, 0x0A, 0x66, 0xEB, 0x21, 0x68, 0x2A, 0x09,
      0xCE, 0x44, 0xCE, 0x4C, 0x95, 0x41, 0xFD, 0x50, 0x6A, 0x6A, 0xF1, 0x2C,
      0xA3, 0x63, 0x47, 0x33, 0xE4, 0x66, 0x16, 0x81, 0x87, 0x17, 0x17, 0xA5,
      0x60, 0x68, 0x57, 0x94, 0x68, 0x5D, 0xA1, 0x21, 0x56, 0x56, 0x71, 0xB5,
      0x0F, 0xE8, 0x66, 0xB0, 0xA5, 0x65, 0xB0, 0xA9, 0x72, 0x8A, 0x97, 0xC6,
      0xAE, 0xC4, 0x44, 0x2C, 0x49, 0x8C, 0x54, 0xD7, 0x71, 0x84, 0xC4, 0x4D,
      0x91, 0x90, 0x83, 0x92, 0x12, 0x04, 0x33, 0x4C, 0xB0, 0x58, 0xFC, 0x5E,
      0xA4, 0x76, 0x5C, 0x04, 0x78, 0x10, 0x47, 0x70, 0xB7, 0x43, 0xA5, 0x09,
      0x3E, 0x84, 0x2C, 0x01, 0x37, 0x69, 0xF1, 0x6B, 0x02, 0xA4, 0x60, 0x83,
      0x87, 0x2C, 0x58, 0xD3, 0xD6, 0x78, 0xDC, 0xE8, 0x7A, 0x37, 0x99, 0x38,
      0x1D, 0xF1, 0x17, 0xD0, 0xF3, 0x57, 0x99, 0x8C, 0x85, 0x36, 0x06, 0x69,
      0x2E, 0xCC, 0x1F, 0x08, 0x7C, 0x08, 0x83, 0xCA, 0x57, 0x5D, 0x91, 0x59,
      0xBE, 0x88, 0x20, 0x45, 0xE6, 0x0D, 0xDB, 0xA3, 0x8E, 0xD3, 0xBC, 0x86,
      0x60, 0x12, 0x1F, 0xB8, 0xEB, 0x2D, 0x60, 0x49, 0xA5, 0x84, 0x3A, 0x2D,
      0x2A, 0x53, 0xBD, 0x52, 0x98, 0x81, 0x79, 0x94, 0x87, 0xB3, 0x04, 0x38,
      0xB4, 0xA5, 0x85, 0xB2, 0xE7, 0x77, 0x68, 0xE8, 0x7D, 0x06, 0x0C, 0x0D,
      0x67, 0x62, 0x83, 0x43, 0x1A, 0x55, 0xBA, 0x9C, 0x12, 0x9F, 0x03, 0x67,
      0x51, 0x83, 0x52, 0x99, 0xD7, 0x8C, 0xD2, 0xD8, 0x57, 0x48, 0xC2, 0x46,
      0xF8, 0x91, 0x59, 0x27, 0xB1, 0xC9, 0xA1, 0x19, 0x4B, 0xAC, 0x1A, 0x10,
      0x3B, 0x5A, 0x1B, 0xB6, 0x93, 0x70, 0xC6, 0x85, 0xB1, 0x5A, 0xA2, 0x64,
      0xC4, 0x36, 0x68, 0x3A, 0xBA, 0x9F, 0x4A, 0x03, 0xC4, 0x9D, 0xAC, 0xC5,
      0xA1, 0xD3, 0xCD, 0x5C, 0x13, 0x35, 0x99, 0xEB, 0x16, 0x00, 0x68, 0x50,
      0xBA, 0xC4, 0x10, 0x82, 0x39, 0x14, 0x4C, 0xE6, 0x82, 0xB7, 0x30, 0x1E,
      0xB2, 0x20, 0xA3, 0x63, 0x03, 0x2F, 0xB6, 0x08, 0x13, 0x77, 0x86, 0x89,
      0x83, 0x75, 0x36, 0x6B, 0x00, 0x27, 0x19, 0x58, 0x0E, 0xC7, 0xF7, 0x73,
      0x3B, 0xB2, 0xC3, 0xDE, 0x27, 0x97, 0x4F, 0xEB, 0x56, 0xC7, 0x2A, 0xA6,
      0x12, 0x52, 0xA2, 0xD6, 0x3C, 0xB6, 0x3D, 0x38, 0x0F, 0x1F, 0xE3, 0x14,
      0x24, 0x39, 0xAD, 0x11, 0x51, 0xB5, 0x43, 0x59, 0x1A, 0xB0, 0x91, 0x5B,
      0x51, 0x4B, 0x3E, 0x97, 0xEB, 0xA1, 0x7C, 0xCA, 0xA7, 0x3F, 0xF0, 0xA3,
      0x45, 0x93, 0x93, 0xBA, 0xE2, 0x0B, 0x1D, 0x8A, 0x89, 0x61, 0xC0, 0x61,
      0x5B, 0x18, 0x0D, 0xEC, 0xE6, 0x02, 0x33, 0x5A, 0x11, 0xF7, 0x61, 0x7E,
      0xBC, 0xE0, 0xA3, 0xA0, 0xB3, 0xAD, 0x1A, 0x78, 0xA3, 0xC8, 0xC6, 0x82,
      0x97, 0x63, 0x88, 0x85, 0x12, 0x85, 0xD8, 0x3B, 0xA9, 0xE3, 0x58, 0xAF,
      0x22, 0xA1, 0x40, 0x39, 0xCA, 0x01, 0x5F, 0xE1, 0xB4, 0x2F, 0x8A, 0x3E,
      0xB4, 0x54, 0x57, 0x85, 0x04, 0x15, 0x38, 0x17, 0x45, 0x50, 0x45, 0x16,
      0x6F, 0x24, 0xD6, 0xDF, 0x00, 0x71, 0xE8, 0x84, 0xAF, 0x76, 0xEC, 0xBB,
      0xFA, 0x43, 0x0F, 0xC3, 0x1D, 0x1F, 0x77, 0x40, 0x5F, 0x4B, 0x40, 0x4B,
      0x53, 0x87, 0x25, 0xF5, 0x61, 0x88, 0x4E, 0xDA},
     true,
     {0xBA, 0xE4, 0x04, 0xE6, 0x3A, 0xA4, 0x9C, 0x1E, 0x8D, 0x26, 0x12,
      0xE7, 0x3E, 0x54, 0xC5, 0x52, 0xB3, 0x9D, 0xEB, 0x5F, 0x30, 0x02,
      0x42, 0x08, 0x79, 0x81, 0x95, 0x88, 0x2E, 0x0F, 0x1B, 0x2C},
     {0xD8, 0xD2, 0x40, 0x17, 0x60, 0x9D, 0x9A, 0xBA, 0x14, 0x14, 0xD1,
      0x8A, 0xD4, 0xAC, 0x9E, 0x14, 0xA0, 0x95, 0x4A, 0xC1, 0xA8, 0x0A,
      0xE9, 0xF2, 0x95, 0x27, 0x35, 0x18, 0x98, 0xF6, 0x14, 0x83}},

    {params_ml_kem768_test_mode,
     {0x6F, 0x16, 0x94, 0x58, 0x9D, 0xFF, 0xCE, 0x02, 0x2D, 0xC4, 0xDF,
      0x18, 0x52, 0xFA, 0x49, 0xA4, 0x1C, 0x6E, 0x8A, 0xB9, 0xF7, 0x88,
      0x7E, 0x70, 0xDD, 0xAE, 0xF4, 0x23, 0x2B, 0x04, 0x5D, 0xFE},
     {0xEC, 0xEC, 0x37, 0x75, 0x23, 0x15, 0x0E, 0x39, 0xB8, 0xE5, 0xA4, 0xB8,
      0x5F, 0x82, 0x37, 0xA6, 0x85, 0x63, 0x0E, 0xA7, 0xA3, 0x44, 0x3A, 0x92,
      0x49, 0xD4, 0xB9, 0x0E, 0x1B, 0xA3, 0x43, 0x8C, 0xB0, 0x95, 0xAB, 0x55,
      0x55, 0x70, 0x6E, 0xAC, 0xB3, 0xA6, 0x1D, 0x51, 0x43, 0x02, 0x98, 0x56,
      0xC7, 0xE8, 0x0B, 0x0D, 0x91, 0x56, 0xBD, 0xE6, 0x3A, 0xB4, 0x63, 0x26,
      0x05, 0xF6, 0x00, 0x94, 0x50, 0xB1, 0xCF, 0xB6, 0xB8, 0x1A, 0x58, 0xA4,
      0x36, 0x87, 0xC4, 0xDB, 0x56, 0xBF, 0x08, 0x7A, 0x75, 0xDF, 0x04, 0x72,
      0x37, 0xE9, 0xA9, 0xB7, 0x94, 0x08, 0xA8, 0x36, 0x9C, 0xA3, 0xAA, 0x17,
      0x31, 0xA7, 0x19, 0x0F, 0xF3, 0xC1, 0x1A, 0x76, 0x99, 0xE3, 0xF8, 0x3A,
      0x1F, 0x65, 0x5B, 0x3E, 0x2C, 0x25, 0x2D, 0x73, 0x2B, 0xEE, 0xD0, 0x70,
      0x3A, 0x64, 0x2B, 0xA3, 0x50, 0x08, 0x59, 0xAC, 0x70, 0xC5, 0x1C, 0x3D,
      0xE8, 0x86, 0xC2, 0x6B, 0xAA, 0x47, 0xEB, 0x22, 0x10, 0x74, 0xF9, 0x7F,
      0xB7, 0x49, 0x57, 0xF9, 0xC6, 0x1A, 0x02, 0x40, 0x5D, 0x2E, 0x2B, 0x95,
      0xFE, 0x19, 0x8C, 0x2A, 0xA2, 0x4A, 0xB3, 0xD6, 0x49, 0xA3, 0xB6, 0x80,
      0x3C, 0x36, 0x36, 0xC0, 0x3B, 0x3E, 0x19, 0x9A, 0xC0, 0x7D, 0x10, 0x39,
      0x60, 0x57, 0x77, 0x38, 0xCC, 0x83, 0x3C, 0xD4, 0x2C, 0xFF, 0x97, 0xC5,
      0x5F, 0xB4, 0xBC, 0x57, 0x7B, 0x70, 0xD1, 0xA8, 0x64, 0x2A, 0xBB, 0x77,
      0xA1, 0x53, 0x0E, 0xD0, 0xF1, 0x40, 0x5B, 0x22, 0x45, 0x34, 0x72, 0x56,
      0xC0, 0xD4, 0x70, 0x05, 0xA6, 0x34, 0xC9, 0x23, 0x74, 0x20, 0x57, 0x41,
      0x30, 0xF6, 0x07, 0xBE, 0x82, 0xCD, 0x00, 0xD8, 0x1E, 0x15, 0x56, 0x2A,
      0xB8, 0x46, 0x1F, 0x68, 0x11, 0x59, 0x73, 0x7A, 0xBD, 0xA7, 0x44, 0x1F,
      0x78, 0x71, 0xAE, 0x7B, 0x28, 0x40, 0x1A, 0x6B, 0xAB, 0x14, 0x41, 0x2A,
      0x5B, 0xE5, 0xA4, 0xBD, 0xA2, 0x2A, 0x46, 0x42, 0x5B, 0x8B, 0x51, 0x96,
      0x4A, 0x67, 0x32, 0x78, 0xB8, 0x73, 0xC4, 0xF6, 0x1D, 0xD8, 0xCC, 0x5A,
      0x7D, 0xD3, 0xB8, 0x9E, 0x11, 0x65, 0x2C, 0x67, 0x7E, 0x05, 0x26, 0x43,
      0xA3, 0x21, 0x39, 0xAB, 0x97, 0x1B, 0x5B, 0x00, 0x61, 0xD6, 0x00, 0x44,
      0x04, 0x95, 0xB9, 0x6A, 0xA3, 0x11, 0x37, 0x25, 0x64, 0x0F, 0x6A, 0x91,
      0xDA, 0x0B, 0x4B, 0x65, 0x16, 0x30, 0xEE, 0xD4, 0x46, 0xBE, 0x53, 0xC6,
      0xC6, 0x36, 0x5B, 0xDD, 0x2B, 0xA0, 0xF7, 0x57, 0xCE, 0xFE, 0x05, 0xC1,
      0x23, 0x1B, 0x66, 0x64, 0x04, 0x19, 0x99, 0x37, 0x73, 0x6E, 0x6A, 0x5F,
      0xB3, 0x46, 0x6B, 0xC4, 0x59, 0x07, 0x67, 0xC2, 0xBF, 0xB2, 0x04, 0x1D,
      0xCF, 0x25, 0x48, 0x86, 0x46, 0x62, 0x88, 0xBB, 0x45, 0xF8, 0xE9, 0x1B,
      0xBA, 0x87, 0x6B, 0x9C, 0x09, 0x3B, 0xE2, 0x61, 0x6B, 0xB3, 0xD2, 0xB5,
      0x16, 0x52, 0x64, 0x95, 0xE7, 0x4B, 0xD1, 0x9B, 0x1E, 0x9C, 0x36, 0x54,
      0x36, 0x97, 0xAC, 0xDA, 0x98, 0x16, 0x87, 0x11, 0x1F, 0xD0, 0x5A, 0x1B,
      0x9F, 0x22, 0x55, 0xE8, 0x80, 0x91, 0x52, 0x13, 0x8B, 0x98, 0x37, 0x90,
      0xEF, 0xE8, 0x36, 0xDD, 0x7A, 0x68, 0xA8, 0xA1, 0x50, 0x51, 0x36, 0x99,
      0x96, 0x90, 0x15, 0xBD, 0x65, 0xB0, 0x5D, 0x36, 0x4F, 0xBC, 0x85, 0x1E,
      0x11, 0xB5, 0x06, 0xC6, 0x07, 0xBE, 0x0C, 0xD1, 0x16, 0x91, 0x0C, 0x94,
      0xA3, 0xE3, 0x25, 0x8A, 0xD2, 0x63, 0x68, 0x34, 0x62, 0xF5, 0x82, 0x6B,
      0x85, 0xB4, 0xBB, 0x1D, 0xD2, 0x91, 0x7B, 0x40, 0x34, 0xAD, 0x25, 0x21,
      0x0A, 0xC2, 0x6F, 0x23, 0x55, 0xB0, 0x56, 0x81, 0x01, 0x5C, 0x9A, 0x2B,
      0xF6, 0x22, 0xBD, 0x28, 0x43, 0x1C, 0xF2, 0x86, 0x8E, 0xD2, 0x13, 0x12,
      0x4F, 0x12, 0x4C, 0x7C, 0x67, 0x97, 0xC3, 0x49, 0x52, 0x87, 0x0C, 0xCE,
      0xA4, 0x97, 0x0A, 0xF9, 0x09, 0xBE, 0x10, 0xA5, 0x7B, 0xC7, 0xAC, 0x37,
      0xA5, 0x36, 0xB7, 0xDE, 0x2A, 0x8B, 0x0C, 0x44, 0x59, 0xA6, 0x22, 0x3B,
      0x4E, 0xB7, 0x81, 0xC2, 0x46, 0x2E, 0xD3, 0xAA, 0x13, 0x50, 0x15, 0x75,
      0x4B, 0xE5, 0x6F, 0x2B, 0x50, 0x1A, 0xFE, 0x68, 0x02, 0x2C, 0x81, 0xC1,
      0x6D, 0x40, 0x72, 0x6B, 0x8C, 0x77, 0xB0, 0x84, 0x52, 0x09, 0xFC, 0x4B,
      0x77, 0xB6, 0x4F, 0xF0, 0x86, 0x09, 0xC0, 0x56, 0x73, 0x90, 0xF4, 0xC9,
      0xA8, 0x18, 0x7A, 0x15, 0xC7, 0x19, 0xFC, 0x5C, 0x9C, 0x01, 0x43, 0x3C,
      0xA7, 0x78, 0x09, 0xD3, 0xF9, 0x19, 0x5D, 0x8B, 0xAA, 0x94, 0xB2, 0x4C,
      0xFD, 0x90, 0x9D, 0x63, 0x74, 0xB8, 0xDC, 0xD2, 0x35, 0x5D, 0x17, 0x23,
      0x40, 0x0A, 0xB5, 0x52, 0xF2, 0x64, 0xC9, 0x38, 0xB4, 0xE1, 0xE8, 0x03,
      0x3B, 0xE2, 0xA0, 0xB6, 0xFB, 0xA0, 0x37, 0x29, 0xB3, 0x8F, 0xA3, 0x9F,
      0x28, 0x5A, 0xC0, 0x1A, 0x13, 0x6C, 0x07, 0xD1, 0x0D, 0xEF, 0x39, 0x0B,
      0xEA, 0x56, 0x88, 0xC4, 0x1B, 0x15, 0x5D, 0x81, 0x17, 0x46, 0x62, 0xAE,
      0x48, 0x76, 0x6D, 0x17, 0x91, 0xC6, 0x0F, 0x16, 0x06, 0xBD, 0xF9, 0x17,
      0x7D, 0x99, 0x5D, 0x91, 0x67, 0xB7, 0x55, 0x90, 0x8A, 0x68, 0xE7, 0xA7,
      0xCE, 0xE4, 0xAF, 0x53, 0xA4, 0x2E, 0xDB, 0x48, 0x6C, 0x8D, 0x11, 0x9F,
      0xED, 0x00, 0xBF, 0xA6, 0x43, 0xAB, 0xEB, 0x12, 0x66, 0x6E, 0x2C, 0xC2,
      0x29, 0x5B, 0x83, 0x36, 0x12, 0x2E, 0x7D, 0x96, 0x01, 0xF4, 0xDA, 0x40,
      0x5B, 0x76, 0x66, 0xEA, 0x27, 0x6A, 0x17, 0x20, 0x26, 0x26, 0x4C, 0x23,
      0x86, 0x36, 0x65, 0xE2, 0xC2, 0x1F, 0xF7, 0x2A, 0x6C, 0xE4, 0x18, 0x60,
      0x16, 0xD8, 0x8F, 0x16, 0x7A, 0x35, 0x88, 0x89, 0xAC, 0x1F, 0x8B, 0x02,
      0x8C, 0xB4, 0x62, 0x2C, 0xF0, 0x74, 0x74, 0x18, 0x1B, 0xBA, 0x6B, 0xC5,
      0xD3, 0x31, 0x08, 0x8B, 0x88, 0xA1, 0xC1, 0x85, 0xB2, 0x8B, 0x90, 0xCB,
      0xDD, 0xD5, 0x4D, 0x88, 0xD4, 0xA9, 0x11, 0xDB, 0x63, 0x8E, 0xF2, 0x36,
      0xD6, 0xD0, 0x6E, 0xE9, 0x68, 0xB8, 0xE5, 0xFC, 0x11, 0xF2, 0xD9, 0xAE,
      0xF1, 0x4A, 0x32, 0x48, 0xEC, 0x92, 0x1F, 0xF0, 0xAC, 0xA4, 0xBB, 0x13,
      0x69, 0xF7, 0xB5, 0xB3, 0x99, 0x1E, 0xD6, 0xB1, 0x6C, 0x3A, 0x91, 0x8F,
      0xD9, 0x39, 0x7B, 0xA4, 0xE4, 0x54, 0xFD, 0x20, 0x95, 0x67, 0xD9, 0x7C,
      0x8B, 0xFB, 0x47, 0x55, 0x8B, 0x6A, 0xC7, 0x72, 0x6D, 0x6C, 0x83, 0x76,
      0x8E, 0x0A, 0x64, 0xB0, 0x34, 0xC0, 0xEB, 0x36, 0xC8, 0xBF, 0xB0, 0xBB,
      0x86, 0xC5, 0x2B, 0x51, 0xFB, 0x7D, 0xE5, 0x03, 0xC3, 0x90, 0xB9, 0x7B,
      0x00, 0xA4, 0x53, 0x42, 0x06, 0xB2, 0x67, 0xC4, 0x23, 0x33, 0xB4, 0xC0,
      0x5B, 0xC9, 0x0C, 0x7F, 0xD2, 0x56, 0x09, 0x54, 0x8B, 0x2E, 0x70, 0x87,
      0xE7, 0x92, 0xBA, 0x21, 0xE3, 0x60, 0x03, 0x8A, 0x62, 0x18, 0x87, 0x2F,
      0x94, 0x03, 0x5E, 0x5F, 0x13, 0x14, 0x9C, 0xB4, 0x40, 0xB3, 0x02, 0xA5,
      0x91, 0x81, 0x7E, 0xB0, 0xE2, 0x29, 0xAF, 0xF0, 0x3C, 0xE9, 0x00, 0x55,
      0x35, 0x97, 0x8F, 0x93, 0x06, 0x22, 0xDD, 0x42, 0x27, 0x22, 0xC0, 0x59,
      0xE0, 0xBB, 0x85, 0x51, 0x43, 0x05, 0x3F, 0x34, 0x8B, 0x23, 0xD5, 0xAB,
      0xBD, 0x49, 0xB7, 0xDE, 0x94, 0x78, 0xEB, 0xB6, 0x33, 0x80, 0xE7, 0x5F,
      0x90, 0x54, 0x92, 0x2A, 0xB6, 0x5F, 0x4E, 0x89, 0x7D, 0x02, 0xF9, 0xAF,
      0x64, 0x1C, 0xC6, 0xA3, 0x49, 0x57, 0xB9, 0x1A, 0x79, 0x69, 0xA1, 0x27,
      0x0E, 0x42, 0x64, 0x45, 0x10, 0x53, 0x26, 0x71, 0x37, 0xEC, 0xAB, 0x8C,
      0x8C, 0x8B, 0x9B, 0x76, 0x35, 0x4A, 0x63, 0xE1, 0xAD, 0xFD, 0xCA, 0x08,
      0x7E, 0x54, 0xA6, 0x7F, 0x8A, 0x2D, 0xE4, 0xAC, 0x6A, 0x63, 0x63, 0x48,
      0x01, 0x7C, 0x62, 0x03, 0xB4, 0xA2, 0x73, 0x22, 0x16, 0x7C, 0x2A, 0xCC,
      0x85, 0xF6, 0x33, 0x35, 0xE8, 0xAE, 0x4D, 0x19, 0x09, 0xDD, 0x1C, 0x3D,
      0xD2, 0x28, 0xC0, 0xE3, 0xB8, 0x44, 0x70, 0x44, 0x6E, 0xDD, 0xE5, 0x72,
      0xF7, 0x92, 0x4D, 0xD4, 0x09, 0x76, 0x4F, 0xD9, 0x8F, 0x43, 0xA2, 0x0E,
      0x5A, 0xC1, 0x5A, 0xE9, 0x20, 0x2B, 0xC1, 0xAB, 0x58, 0xBA, 0xDB, 0x3D,
      0x06, 0xB6, 0xB7, 0x7E, 0x81, 0xAD, 0xF7, 0x63, 0x83, 0x63, 0x98, 0xBB,
      0xC4, 0x54, 0x33, 0x01, 0x5B, 0x65, 0x23, 0x65, 0x27, 0x51, 0xB8, 0x29,
      0x2A, 0xC5, 0x12, 0x30, 0x44, 0x0C, 0x11, 0x2C, 0x63, 0x17, 0xA6, 0x6F,
      0x24, 0xC4, 0xBF, 0x92, 0x7C, 0x7E, 0xB8, 0xC1, 0x86, 0xC9, 0xDA, 0x10,
      0xE1, 0xBC, 0x25, 0xA4, 0xDF, 0x1C, 0xCA, 0x9B, 0x6C, 0x34, 0x07, 0x95,
      0x59, 0x72, 0x44, 0x8D, 0xEB, 0xDB, 0x28, 0x4B},
     true,
     {0xC5, 0x4B, 0x2F, 0xD2, 0x04, 0x41, 0xF5, 0x07, 0xA3, 0xE8, 0x09,
      0x2F, 0x2A, 0x0C, 0x22, 0x74, 0x86, 0xBE, 0xA6, 0xDA, 0x18, 0xF1,
      0xD4, 0x5E, 0x35, 0xDE, 0xFA, 0x5C, 0x72, 0x78, 0x82, 0xCF},
     {0x33, 0x0D, 0x1A, 0x2D, 0x0E, 0x0B, 0x5D, 0xD7, 0xF7, 0x59, 0xC2,
      0x9A, 0x22, 0xD9, 0x1A, 0x09, 0xBE, 0xF1, 0x7F, 0x8C, 0x55, 0x66,
      0xA0, 0xD3, 0xF3, 0x0E, 0x38, 0x17, 0xCF, 0xB7, 0xCB, 0xFF}},

    {params_ml_kem768_test_mode,
     {0xD8, 0xEF, 0x97, 0x42, 0x11, 0x96, 0xB1, 0xA9, 0x14, 0x48, 0xB2,
      0xBA, 0x7E, 0x2B, 0x4D, 0x4B, 0x03, 0x5B, 0x91, 0xDD, 0x85, 0xAE,
      0x4E, 0x57, 0xE8, 0xFE, 0x3F, 0x0B, 0x0D, 0x52, 0x4A, 0xE7},
     {0x48, 0xE1, 0x1D, 0x1B, 0x05, 0x72, 0xF2, 0x89, 0x2B, 0x8F, 0x81, 0xA8,
      0xBE, 0x33, 0x0B, 0x16, 0xC7, 0x35, 0x3B, 0x5C, 0x31, 0x9A, 0x34, 0x87,
      0x4F, 0x95, 0x94, 0x09, 0xE6, 0xA4, 0x1A, 0x71, 0x58, 0x5B, 0xF3, 0x6A,
      0xEB, 0x65, 0x18, 0x69, 0xAA, 0x87, 0xD4, 0x00, 0xC6, 0xAF, 0x32, 0xB7,
      0xF8, 0x27, 0x0B, 0xB9, 0xD5, 0xC5, 0x3B, 0xC3, 0xB6, 0x00, 0xF4, 0x93,
      0x90, 0x4A, 0xB0, 0x00, 0x14, 0x9F, 0xAF, 0xA1, 0xB0, 0x90, 0xB5, 0x44,
      0xEA, 0x01, 0x01, 0xEE, 0x94, 0x0E, 0xFE, 0x87, 0x8D, 0xD6, 0x47, 0x39,
      0xCB, 0xD7, 0x8D, 0x31, 0x7A, 0x09, 0xA4, 0x57, 0x75, 0xB2, 0xAB, 0x9E,
      0x03, 0x83, 0xC7, 0x5D, 0x04, 0x90, 0x83, 0xBA, 0x64, 0x01, 0x96, 0x58,
      0xB4, 0xF4, 0x45, 0x73, 0xC1, 0xB7, 0xC0, 0xB3, 0xA5, 0x94, 0x95, 0x60,
      0x6D, 0x91, 0x62, 0x43, 0x8B, 0x1A, 0x0A, 0x00, 0x0C, 0x3A, 0xA5, 0x83,
      0x4A, 0xD5, 0xBB, 0xD1, 0x18, 0x93, 0x59, 0x80, 0x52, 0xE5, 0xB5, 0x43,
      0xE2, 0x09, 0x21, 0x5E, 0x41, 0xB8, 0xD4, 0x9C, 0x41, 0x07, 0xF5, 0x65,
      0x8F, 0x76, 0x85, 0xCE, 0x53, 0x45, 0x9F, 0x21, 0x45, 0x2A, 0xC6, 0x6F,
      0x0D, 0x23, 0x98, 0xB6, 0x97, 0x1A, 0x5A, 0x23, 0xA0, 0x56, 0x44, 0x49,
      0x47, 0x22, 0x0E, 0x95, 0x45, 0x4A, 0x40, 0x4C, 0x72, 0x13, 0x68, 0x6E,
      0x58, 0xA4, 0x2E, 0x34, 0x6C, 0xC0, 0xC4, 0x47, 0x59, 0x50, 0x4B, 0x2F,
      0xA0, 0x71, 0x98, 0xD7, 0x42, 0xB1, 0xBE, 0x01, 0xA6, 0x6C, 0x2B, 0xC6,
      0x44, 0x98, 0xCB, 0xCF, 0x64, 0x82, 0xDA, 0xC9, 0x9D, 0xE7, 0xDA, 0x70,
      0x3E, 0xC1, 0x44, 0x1D, 0x08, 0xC4, 0xA7, 0x01, 0xBF, 0x84, 0xFC, 0x95,
      0x6B, 0x11, 0x41, 0x1A, 0xD6, 0x10, 0x14, 0x6C, 0x59, 0xDA, 0x64, 0x54,
      0xA8, 0xB3, 0xB7, 0x42, 0xD9, 0x1D, 0xE3, 0xC0, 0x9A, 0x53, 0x17, 0xA9,
      0xDA, 0xDC, 0x23, 0xED, 0x0A, 0x92, 0x2A, 0xF3, 0x39, 0xC0, 0x90, 0x06,
      0x0B, 0x86, 0x9B, 0x55, 0xFB, 0x3A, 0xA1, 0xC6, 0x03, 0xC4, 0x08, 0x68,
      0x33, 0x76, 0x87, 0xEB, 0xB2, 0x7E, 0x37, 0xB8, 0x04, 0xEA, 0x83, 0x4D,
      0x4D, 0x00, 0x84, 0xB2, 0x13, 0x07, 0xDB, 0x0A, 0x64, 0xF6, 0x9B, 0x7B,
      0x49, 0xB5, 0xB0, 0xD2, 0x37, 0xAC, 0x42, 0x8A, 0xBD, 0xFA, 0xCC, 0x01,
      0xD1, 0x01, 0x2A, 0x04, 0x00, 0x8C, 0xFD, 0xD3, 0x36, 0xA0, 0x6B, 0x97,
      0x28, 0xA0, 0x36, 0xB6, 0xC6, 0x04, 0x30, 0xA8, 0xCE, 0x88, 0x7C, 0x4B,
      0x00, 0x6A, 0xA7, 0xE7, 0xC5, 0x48, 0x34, 0xE2, 0x93, 0x4F, 0x84, 0x00,
      0xD0, 0x30, 0xAA, 0x63, 0xE6, 0x1F, 0xEE, 0x96, 0xC2, 0x2F, 0xD4, 0x96,
      0x85, 0xA7, 0xC3, 0xDC, 0x22, 0x87, 0xEC, 0x29, 0x64, 0xCA, 0xC7, 0x8B,
      0x40, 0x34, 0x02, 0x00, 0xC3, 0xAA, 0x6C, 0x8A, 0x67, 0xBD, 0xD3, 0x0F,
      0xB7, 0x60, 0x5F, 0x02, 0x05, 0xCC, 0x0C, 0xA0, 0x43, 0xA4, 0x57, 0x42,
      0x35, 0xB3, 0x42, 0xFC, 0xE5, 0x11, 0x95, 0x87, 0x3F, 0x7F, 0xD9, 0x90,
      0x77, 0xCB, 0x31, 0x22, 0x9B, 0x91, 0xD5, 0x92, 0x3B, 0x67, 0xD3, 0x91,
      0xB4, 0x57, 0x19, 0x81, 0x25, 0xB7, 0xF1, 0x98, 0xBC, 0x7C, 0xE5, 0x8F,
      0xD6, 0xB9, 0x6B, 0xC3, 0x5C, 0xC3, 0xD0, 0x28, 0x19, 0x57, 0xB4, 0x11,
      0x0A, 0x01, 0x12, 0x62, 0xB1, 0xC4, 0xCC, 0x04, 0x45, 0x72, 0x2C, 0xB5,
      0xE8, 0xC1, 0x02, 0x40, 0x85, 0x93, 0xCE, 0x64, 0x45, 0xC4, 0x58, 0x4E,
      0x37, 0x3A, 0xC8, 0x90, 0x18, 0x79, 0x86, 0xA4, 0x94, 0x14, 0x64, 0x3C,
      0xCB, 0x28, 0x61, 0x74, 0x24, 0x66, 0x93, 0x9C, 0x34, 0x7C, 0xF0, 0x21,
      0xD1, 0xC1, 0x9D, 0x36, 0x39, 0x8A, 0x82, 0xC6, 0x4D, 0x1F, 0xBA, 0x4C,
      0x44, 0xB5, 0x6C, 0xF3, 0x84, 0xC0, 0x4A, 0x19, 0x44, 0x44, 0x92, 0x58,
      0x70, 0x0A, 0x66, 0xB1, 0xE2, 0xC9, 0x18, 0x5A, 0x5E, 0x8B, 0x17, 0x77,
      0x25, 0x38, 0xB3, 0x3B, 0x03, 0x8B, 0x87, 0x73, 0x53, 0xDC, 0x25, 0x19,
      0x52, 0xDB, 0xB8, 0x99, 0xF0, 0x4A, 0x6C, 0x67, 0x72, 0x47, 0xD4, 0xCF,
      0x4E, 0x4A, 0x5D, 0x2B, 0x22, 0x31, 0x3A, 0x33, 0xA6, 0xD4, 0x92, 0x37,
      0xE7, 0xC0, 0x4C, 0x61, 0xB2, 0xB0, 0x02, 0x65, 0x31, 0x58, 0x3C, 0x6C,
      0xF6, 0x33, 0x2C, 0x4C, 0x78, 0x53, 0xB7, 0xA8, 0xC6, 0xA4, 0xA5, 0x78,
      0x3D, 0x1B, 0x79, 0xFC, 0x57, 0x25, 0xF9, 0xC7, 0x15, 0x76, 0x00, 0x74,
      0x7C, 0x02, 0x03, 0xE3, 0x13, 0xCE, 0x91, 0x97, 0x6C, 0xE4, 0x4C, 0xC4,
      0xE0, 0xD6, 0x7D, 0xFA, 0x83, 0x04, 0x48, 0x99, 0x93, 0x57, 0xF0, 0x76,
      0xF2, 0xD4, 0x1E, 0xE6, 0xFB, 0xA6, 0x8C, 0x53, 0xC9, 0xAE, 0x5C, 0x02,
      0x5D, 0xF8, 0x67, 0xE3, 0x06, 0x0F, 0x3C, 0xF1, 0x64, 0x21, 0x81, 0x42,
      0xD6, 0xC2, 0xB5, 0xB5, 0x26, 0xBC, 0xE6, 0x78, 0x9D, 0xBC, 0xBC, 0x57,
      0x65, 0x84, 0x65, 0x65, 0x57, 0x5A, 0xB1, 0x54, 0xC5, 0x45, 0xFC, 0x3A,
      0x81, 0xAB, 0x8F, 0xA1, 0x35, 0x3B, 0x70, 0x3B, 0x72, 0xB0, 0x11, 0xA0,
      0x40, 0x55, 0x88, 0xFC, 0x0B, 0x8C, 0x20, 0xD1, 0x3A, 0xCE, 0x9A, 0x93,
      0xF6, 0xB2, 0x8E, 0xA1, 0xC0, 0x5C, 0xC6, 0x86, 0xCB, 0x0A, 0xE7, 0x22,
      0x66, 0x2A, 0x9E, 0xD9, 0x02, 0xAE, 0x87, 0xD4, 0x8B, 0x8A, 0x96, 0x67,
      0xCB, 0xC0, 0xBD, 0x75, 0x74, 0x2C, 0xEF, 0x31, 0x3F, 0x16, 0x49, 0xC4,
      0x2C, 0x76, 0x07, 0x96, 0xC9, 0x79, 0x36, 0x97, 0x36, 0xCF, 0xF0, 0x55,
      0x16, 0xA7, 0x2B, 0xA3, 0x68, 0x49, 0xD2, 0xE2, 0x13, 0xAA, 0x49, 0x7A,
      0x6E, 0x77, 0x74, 0xEB, 0x9A, 0x89, 0x6A, 0xF0, 0xA1, 0x33, 0x87, 0xAD,
      0xE8, 0xF5, 0x94, 0xDE, 0xB4, 0x82, 0xD9, 0xFB, 0x9E, 0x4C, 0x74, 0x03,
      0x60, 0xB5, 0x08, 0x9C, 0xD1, 0x01, 0xE5, 0x45, 0x0C, 0x83, 0x7C, 0x82,
      0x14, 0x24, 0x72, 0x43, 0x41, 0x7C, 0x05, 0xAC, 0xB9, 0xD3, 0x19, 0x80,
      0x9D, 0x33, 0x16, 0x8E, 0x70, 0x7B, 0xD1, 0x16, 0x87, 0x09, 0xF7, 0x82,
      0x9B, 0x58, 0x40, 0x53, 0xB5, 0x54, 0xD9, 0x55, 0x57, 0xDC, 0x03, 0x26,
      0x83, 0xD8, 0x85, 0xAB, 0x40, 0x6E, 0x20, 0x5B, 0x6F, 0x49, 0x5C, 0x23,
      0xEB, 0x67, 0x3E, 0x05, 0x65, 0x1E, 0xD9, 0xDC, 0xB8, 0x7C, 0x49, 0x67,
      0x98, 0x7B, 0x7B, 0x78, 0xEB, 0x34, 0x9F, 0x4A, 0xAB, 0xD3, 0x5C, 0x72,
      0x42, 0x07, 0x9F, 0xB3, 0x06, 0x07, 0xED, 0x82, 0xA0, 0x50, 0x31, 0x73,
      0xA1, 0xB7, 0x85, 0xCC, 0x36, 0xA2, 0x18, 0x18, 0x92, 0x11, 0x40, 0x03,
      0xE2, 0xD3, 0x3D, 0x17, 0xAA, 0x3B, 0x0A, 0x5C, 0xA4, 0x79, 0x74, 0xB8,
      0xF1, 0xC9, 0x3B, 0x35, 0x10, 0x7D, 0xA3, 0x90, 0x0A, 0x87, 0x79, 0x24,
      0x33, 0xF1, 0x54, 0xC6, 0x43, 0xB3, 0x45, 0x19, 0x94, 0x1C, 0xC6, 0x07,
      0x10, 0xF0, 0x7A, 0x71, 0xE4, 0xAE, 0x09, 0xA7, 0x53, 0x9E, 0x72, 0xB9,
      0x23, 0x5A, 0x07, 0xD5, 0xDB, 0x53, 0xC0, 0x48, 0x49, 0x3B, 0x4A, 0xC8,
      0xB9, 0xF0, 0xA3, 0x69, 0x68, 0x92, 0xA0, 0x16, 0x4D, 0xCE, 0x38, 0xB7,
      0xF8, 0xE7, 0x11, 0x50, 0x6A, 0x92, 0x20, 0xA5, 0xC6, 0xC6, 0x1A, 0xC2,
      0xEF, 0x39, 0x42, 0xD3, 0x98, 0x49, 0x07, 0xE5, 0x9D, 0xC7, 0x03, 0xAD,
      0xBD, 0x93, 0xB6, 0xB5, 0x09, 0xA1, 0x56, 0xCC, 0x34, 0xB2, 0x26, 0x18,
      0x87, 0x78, 0xBD, 0x19, 0x14, 0x02, 0xDC, 0x94, 0x12, 0x4C, 0xC7, 0x98,
      0x11, 0x51, 0x93, 0x4F, 0x68, 0x73, 0x3D, 0x13, 0x15, 0x40, 0x05, 0x64,
      0x66, 0x43, 0x7C, 0xE2, 0xB3, 0x1C, 0x13, 0x20, 0x36, 0x2A, 0xE5, 0x24,
      0x7C, 0x35, 0xCD, 0xA6, 0xE1, 0x3E, 0x5D, 0x9B, 0xB8, 0x9D, 0x0C, 0x8D,
      0xCF, 0xB1, 0x6B, 0xE8, 0xBC, 0x35, 0x5A, 0x0C, 0x0C, 0x54, 0xE7, 0xAB,
      0xAC, 0x12, 0x62, 0x1A, 0xDB, 0x1E, 0xAF, 0x82, 0xB3, 0xD1, 0xE9, 0x15,
      0x31, 0x5A, 0x3B, 0x7F, 0x3C, 0x98, 0xF6, 0xCC, 0x11, 0x7B, 0xC3, 0xC8,
      0x50, 0x25, 0x80, 0x59, 0x73, 0x71, 0x71, 0x47, 0x8E, 0x3A, 0xAA, 0x7F,
      0xE5, 0xD7, 0xB5, 0xCD, 0x21, 0x8C, 0xB4, 0x28, 0x15, 0x3C, 0x07, 0xAF,
      0x80, 0x66, 0x7C, 0x09, 0x6C, 0x96, 0x65, 0x6C, 0x85, 0xB8, 0xB0, 0xBD,
      0x1F, 0x60, 0x6A, 0x7C, 0xEA, 0x96, 0x37, 0x17, 0xA1, 0x8A, 0xE6, 0x42,
      0xE9, 0xE8, 0xA8, 0x53, 0xD9, 0x38, 0xC6, 0x66, 0x51, 0xEE, 0x31, 0x03,
      0x4C, 0xAE, 0xC7, 0xDA, 0x60, 0x97, 0xA3, 0xC3, 0x5B, 0xA4, 0x20, 0x02,
      0x23, 0x24, 0xEC, 0x00, 0xCF, 0x53, 0xB5, 0x3E, 0x9E, 0xBA, 0xDC, 0x6F,
      0xEB, 0x57, 0xC9, 0xB5, 0xBF, 0x5F, 0x53, 0xDA},
     true,
     {0x2A, 0xA1, 0x08, 0x46, 0x5D, 0xF4, 0x1C, 0x30, 0x84, 0x79, 0x69,
      0x46, 0xB6, 0x38, 0x29, 0x30, 0xDF, 0x7A, 0xBE, 0x3B, 0x37, 0x48,
      0x73, 0x08, 0xD5, 0xAD, 0xB2, 0x1A, 0x96, 0xF2, 0x0F, 0xCA},
     {0x44, 0xAB, 0x39, 0x6F, 0x38, 0x94, 0x2B, 0xB6, 0x9C, 0x09, 0xB2,
      0x60, 0x26, 0x29, 0xB5, 0x3B, 0x82, 0x0F, 0xCA, 0x6D, 0x3D, 0x10,
      0x43, 0xB2, 0x4A, 0xFB, 0x31, 0x84, 0xAE, 0x9B, 0x55, 0x65}},

    {params_ml_kem768_test_mode,
     {0x13, 0x2E, 0x7C, 0xDA, 0xB9, 0xCD, 0x51, 0x99, 0xFF, 0x09, 0x37,
      0xC2, 0x66, 0xD5, 0x0B, 0xC5, 0x0B, 0xE7, 0x64, 0xAD, 0x02, 0x7D,
      0xE4, 0x5C, 0x85, 0x8E, 0x3C, 0x2F, 0x79, 0xB7, 0xF0, 0x7A},
     {0xAE, 0x74, 0x2E, 0x0C, 0x47, 0x61, 0xE7, 0x73, 0x1F, 0x98, 0xA9, 0x6F,
      0x57, 0x37, 0x4E, 0xB4, 0xE3, 0x21, 0xE1, 0xD3, 0x3E, 0x31, 0x33, 0xC0,
      0x31, 0x35, 0x5A, 0xFA, 0x97, 0x4E, 0x82, 0xF5, 0x0E, 0x7E, 0xB3, 0x60,
      0x84, 0x47, 0x1D, 0x94, 0x49, 0x01, 0x50, 0x18, 0x8B, 0x34, 0xA1, 0x13,
      0xE3, 0x00, 0xB2, 0xCB, 0x65, 0xBB, 0x44, 0xE6, 0x94, 0x01, 0x24, 0xBF,
      0xE3, 0xE2, 0x1F, 0xBB, 0x4B, 0x01, 0xA7, 0x64, 0x08, 0x17, 0x56, 0x79,
      0x49, 0x28, 0x69, 0x22, 0xA2, 0x2A, 0x8E, 0xF6, 0x54, 0xD8, 0x46, 0x5C,
      0xA3, 0x97, 0x22, 0xAE, 0x9A, 0x4D, 0x00, 0x54, 0x90, 0x6F, 0x52, 0xC7,
      0x94, 0x28, 0x71, 0x69, 0x71, 0xAB, 0xA0, 0x70, 0x3C, 0x2A, 0x3C, 0x49,
      0x7D, 0xF7, 0x3D, 0x0D, 0x0C, 0x8A, 0xF3, 0xBC, 0x36, 0xD1, 0x2C, 0x80,
      0xAC, 0x27, 0x01, 0xD8, 0xC4, 0x68, 0x9E, 0x05, 0x64, 0x5B, 0x20, 0x17,
      0xB7, 0xEC, 0x83, 0x0F, 0x8C, 0xC3, 0xC3, 0x41, 0x9B, 0x1B, 0x14, 0xAA,
      0x89, 0x0B, 0x2A, 0x8B, 0x27, 0x4C, 0x53, 0x16, 0x18, 0x5B, 0x26, 0x47,
      0x34, 0x56, 0xA1, 0x96, 0xD8, 0xC8, 0x36, 0x35, 0x2B, 0x43, 0x7A, 0x4E,
      0xB0, 0xE9, 0x24, 0xE9, 0xD5, 0x77, 0x6B, 0x52, 0x8D, 0xE1, 0xBB, 0x43,
      0x3B, 0x7C, 0x70, 0x3F, 0x30, 0x2E, 0xDE, 0x28, 0x4C, 0x9B, 0x22, 0xA6,
      0xA5, 0x02, 0xA7, 0xFD, 0x6B, 0x17, 0xAB, 0x55, 0x46, 0xB5, 0x2C, 0x86,
      0xDE, 0x88, 0x3C, 0x80, 0xF7, 0x4A, 0x9A, 0x73, 0x02, 0xE7, 0x27, 0x0B,
      0x73, 0xC5, 0x21, 0xFC, 0xC5, 0x1E, 0xB7, 0xE0, 0x2B, 0xCC, 0x68, 0xCB,
      0x25, 0x7A, 0x1F, 0xD6, 0xEA, 0x63, 0xC1, 0x95, 0x61, 0x0A, 0xB8, 0xA8,
      0xAE, 0xF2, 0x48, 0x6E, 0x18, 0xAA, 0x08, 0x10, 0x56, 0xB5, 0xF4, 0x74,
      0xFE, 0xBC, 0xBA, 0xA4, 0x85, 0x60, 0x32, 0x5C, 0x06, 0xDD, 0xBC, 0xB7,
      0x9E, 0xFA, 0x10, 0x58, 0x9A, 0x62, 0xD9, 0xD5, 0x42, 0x68, 0x06, 0x2C,
      0xC8, 0xE3, 0x69, 0x33, 0x73, 0x06, 0x6E, 0x57, 0xC8, 0x76, 0x55, 0x8F,
      0xF3, 0x18, 0x0C, 0x63, 0x22, 0x69, 0xE7, 0x7A, 0x96, 0xF3, 0x05, 0x17,
      0x5F, 0x3A, 0x2F, 0x98, 0x33, 0xBD, 0xA9, 0xC4, 0x3B, 0xAF, 0x62, 0x50,
      0x0C, 0x20, 0x2D, 0x0D, 0xBA, 0x91, 0xB2, 0x8C, 0xC6, 0x47, 0x37, 0x03,
      0x96, 0x07, 0x85, 0xC8, 0x71, 0x16, 0x04, 0xC9, 0xCC, 0x04, 0xA9, 0x8B,
      0xD1, 0x99, 0x04, 0x18, 0x9A, 0x6D, 0x62, 0xA1, 0x59, 0xE6, 0xA8, 0x3F,
      0x25, 0x90, 0x7E, 0xDC, 0xF4, 0x57, 0xA0, 0x80, 0xC7, 0x06, 0x27, 0xC6,
      0xF5, 0xB0, 0x61, 0x2A, 0xE8, 0x4D, 0x43, 0x17, 0x26, 0x1B, 0xE7, 0x28,
      0xB7, 0x86, 0xA1, 0x11, 0xB7, 0x44, 0x78, 0x04, 0x4B, 0x7C, 0xF1, 0x92,
      0xDC, 0x64, 0x90, 0x47, 0x5C, 0x9A, 0xE1, 0xE5, 0xC0, 0xFF, 0x97, 0xB8,
      0x12, 0xE0, 0x09, 0x4A, 0x63, 0xAD, 0x81, 0x8C, 0x5B, 0xE5, 0x30, 0x4A,
      0xE8, 0x98, 0x1B, 0x52, 0x86, 0x79, 0x12, 0x78, 0x71, 0x14, 0xA1, 0x40,
      0x88, 0x26, 0x1C, 0xF0, 0x74, 0x77, 0x32, 0x54, 0x8E, 0xE1, 0x05, 0x92,
      0x7D, 0xB0, 0x86, 0x08, 0x8B, 0x89, 0xDA, 0x47, 0x73, 0xF3, 0xC4, 0xCC,
      0xA0, 0xE4, 0x44, 0x31, 0xD9, 0x8A, 0x8E, 0xAB, 0x89, 0xC1, 0xD6, 0xC1,
      0xD2, 0x61, 0x09, 0xEA, 0x32, 0xCE, 0x99, 0x47, 0x3E, 0x50, 0x9B, 0x9D,
      0x47, 0x86, 0x14, 0xD2, 0x6B, 0x06, 0x38, 0x7B, 0x89, 0xD6, 0x4C, 0x67,
      0x85, 0xA5, 0x0D, 0xB4, 0xC7, 0x8F, 0xEB, 0x65, 0x78, 0x65, 0x50, 0x64,
      0x7E, 0xA3, 0x0C, 0x90, 0x29, 0x88, 0x4D, 0xD3, 0x4F, 0xC9, 0x22, 0x75,
      0x96, 0xB0, 0x91, 0xE9, 0x17, 0x66, 0xF8, 0x38, 0x9A, 0x82, 0xB0, 0x16,
      0x23, 0xD7, 0x72, 0x1F, 0x20, 0x69, 0x91, 0x11, 0x28, 0xC3, 0x09, 0x60,
      0x2F, 0x04, 0x7F, 0x44, 0x82, 0x5E, 0xF6, 0xA1, 0x2B, 0xED, 0xC7, 0x28,
      0x80, 0xAA, 0x3B, 0xB9, 0x15, 0x04, 0x90, 0x42, 0x76, 0xE3, 0xE9, 0x58,
      0x32, 0x28, 0xA7, 0x89, 0x65, 0x70, 0x67, 0x6B, 0x8D, 0xA2, 0x87, 0x23,
      0xEE, 0x17, 0x55, 0x3A, 0x15, 0x6A, 0xD5, 0x88, 0x4A, 0xF0, 0xE1, 0x06,
      0x6E, 0x51, 0x8D, 0x19, 0x95, 0x1F, 0x98, 0x07, 0x9F, 0x22, 0x44, 0x00,
      0x93, 0x69, 0x15, 0x7E, 0x19, 0x15, 0xB4, 0xFC, 0x99, 0x25, 0x4C, 0x46,
      0x91, 0xD4, 0xA6, 0x4A, 0x0A, 0xB0, 0x12, 0x9B, 0xB2, 0x35, 0xB0, 0x05,
      0x52, 0xC5, 0xB6, 0x06, 0x22, 0x0E, 0x45, 0x68, 0x33, 0x41, 0x08, 0x6F,
      0x2D, 0xB3, 0xAC, 0xC1, 0x6C, 0x8D, 0xB5, 0x63, 0x25, 0x9C, 0x6C, 0x8E,
      0x6A, 0x36, 0x0B, 0x9C, 0xBA, 0x54, 0xD0, 0x48, 0x33, 0x5E, 0x45, 0x5F,
      0xB4, 0x06, 0x41, 0x5D, 0x14, 0x71, 0x22, 0x90, 0x33, 0x25, 0xB3, 0x41,
      0x72, 0x7C, 0x76, 0xEC, 0xD2, 0xC1, 0x6D, 0x33, 0x18, 0x80, 0x55, 0x53,
      0x41, 0x96, 0xC0, 0x71, 0xF6, 0xB6, 0x80, 0x6B, 0x9E, 0x05, 0x7B, 0x55,
      0x97, 0x15, 0x8E, 0x03, 0x6C, 0x3C, 0xA9, 0x01, 0x74, 0xA8, 0x3B, 0x96,
      0x5F, 0x8C, 0x3C, 0xA7, 0x56, 0x2A, 0x3A, 0x24, 0xAE, 0x99, 0x29, 0x8A,
      0x10, 0x79, 0x1D, 0x38, 0xFA, 0x3E, 0x5F, 0x30, 0xB5, 0xCE, 0x3A, 0x84,
      0x38, 0xE6, 0x70, 0x8A, 0x15, 0x6F, 0xD5, 0x1C, 0x05, 0x5E, 0xFB, 0x5C,
      0x65, 0x75, 0x9D, 0x76, 0xB1, 0xBE, 0xBC, 0x23, 0x18, 0x96, 0x09, 0x5F,
      0x21, 0x31, 0x48, 0xAF, 0x83, 0x9B, 0x4D, 0xEA, 0xA0, 0x3C, 0x96, 0x28,
      0xD0, 0xC9, 0x43, 0xB0, 0xF1, 0x47, 0x70, 0xA9, 0xA2, 0x23, 0xFA, 0x3C,
      0xF0, 0x5C, 0x7E, 0xA9, 0x83, 0xBB, 0x64, 0x50, 0x18, 0x85, 0xA9, 0x19,
      0x67, 0x80, 0x01, 0xE9, 0x21, 0x9A, 0xEB, 0xFA, 0x71, 0xE5, 0xD3, 0xCB,
      0x5B, 0xE5, 0x85, 0xE7, 0xB9, 0x39, 0x19, 0x4C, 0x87, 0xC1, 0xEB, 0xA5,
      0xBD, 0x56, 0xA9, 0x53, 0x64, 0x5D, 0x28, 0xC2, 0x1F, 0x0E, 0x45, 0x68,
      0x6C, 0x9A, 0xC9, 0x6F, 0x82, 0x0A, 0x3A, 0xC2, 0x6F, 0x69, 0xF7, 0x05,
      0xEC, 0x92, 0x03, 0x92, 0x60, 0x1E, 0x55, 0xFC, 0x1E, 0xD2, 0xB9, 0xBD,
      0xC3, 0xC6, 0xBA, 0x00, 0x17, 0x9F, 0x9D, 0xC6, 0x87, 0xC2, 0xC0, 0x1C,
      0x29, 0xEC, 0x05, 0xD8, 0xBC, 0x5A, 0x31, 0x45, 0xC1, 0xF9, 0x36, 0xCB,
      0x8B, 0x20, 0xCB, 0x20, 0x15, 0x23, 0x3E, 0x01, 0x18, 0x4A, 0xE6, 0x8B,
      0xB0, 0xF9, 0x67, 0x28, 0xC2, 0x75, 0x21, 0x11, 0x84, 0x66, 0xA5, 0xCE,
      0x60, 0x2C, 0xC4, 0x41, 0x98, 0x81, 0xDE, 0xA6, 0x23, 0x81, 0xD7, 0x87,
      0x0C, 0x14, 0x20, 0x9B, 0x6A, 0x87, 0x4F, 0xBB, 0x49, 0x84, 0x72, 0xB3,
      0x42, 0x0A, 0x3A, 0x2A, 0x42, 0x03, 0x93, 0x88, 0x63, 0xC7, 0x61, 0x0D,
      0xF7, 0x33, 0x25, 0xB5, 0xC1, 0x2E, 0xA5, 0xD2, 0xCE, 0xC7, 0x86, 0x10,
      0xF7, 0x23, 0x7C, 0x44, 0x53, 0x53, 0x6D, 0x71, 0x80, 0xEE, 0x74, 0x25,
      0xC9, 0xDA, 0x6C, 0x43, 0x60, 0x4D, 0xC6, 0xA0, 0x74, 0xE2, 0xB2, 0x03,
      0xA0, 0xFC, 0x6B, 0x44, 0x67, 0x98, 0x79, 0xAC, 0x2D, 0x94, 0x42, 0x77,
      0x46, 0x04, 0x18, 0x09, 0xD8, 0x48, 0x20, 0x37, 0x9D, 0x14, 0x16, 0x8B,
      0xB0, 0xE8, 0x16, 0xFD, 0x24, 0x15, 0x98, 0x19, 0x5C, 0xA1, 0xC4, 0x47,
      0xC2, 0xEA, 0x5D, 0x5B, 0x45, 0x5A, 0x28, 0xE6, 0x65, 0x27, 0x08, 0x8E,
      0x24, 0x16, 0x3C, 0x6D, 0xE4, 0x6D, 0x36, 0xDC, 0x17, 0x53, 0xCA, 0xB7,
      0x45, 0x1A, 0x6A, 0xCF, 0x09, 0xAB, 0xBA, 0xA1, 0x22, 0x3E, 0x7A, 0x3B,
      0x07, 0xE8, 0x01, 0xF8, 0x40, 0x71, 0xA7, 0xA3, 0xAD, 0xE7, 0x60, 0xC7,
      0xA2, 0xA1, 0x2F, 0x4D, 0x12, 0x73, 0x45, 0x30, 0x53, 0x8D, 0x21, 0x79,
      0xEA, 0x20, 0x4A, 0xDD, 0x3A, 0x40, 0x22, 0x24, 0x67, 0x64, 0x9B, 0x42,
      0x45, 0x94, 0x2E, 0x80, 0xB5, 0x0E, 0xFA, 0xF6, 0x0B, 0x83, 0xE4, 0x59,
      0x19, 0x51, 0x10, 0x1D, 0x02, 0x5B, 0x05, 0xCC, 0x1E, 0xEF, 0xEA, 0x99,
      0x6B, 0xF1, 0xC0, 0x15, 0xA6, 0xCD, 0xC9, 0x32, 0x5E, 0xAC, 0xBC, 0x34,
      0x95, 0x92, 0xB8, 0x81, 0xCB, 0x3C, 0x3D, 0xA8, 0xC0, 0x96, 0x12, 0x79,
      0x53, 0xF0, 0xB9, 0x69, 0x75, 0x61, 0xDC, 0xDA, 0x6B, 0xA3, 0xD5, 0x66,
      0xFF, 0x3C, 0x14, 0xBF, 0x71, 0x8A, 0xDF, 0xA4, 0x78, 0x77, 0xA1, 0x2E,
      0x39, 0x0A, 0x6C, 0xD0, 0x54, 0x4D, 0x95, 0x24, 0xAF, 0xA3, 0x70, 0x69,
      0xC8, 0xAB, 0xA2, 0x4F, 0x91, 0x8F, 0xDD, 0x15, 0x98, 0x6F, 0x9B, 0x1C,
      0x64, 0x71, 0xA5, 0xC7, 0xA4, 0x95, 0x58, 0x8F, 0x79, 0xB7, 0x1F, 0xCD,
      0xBD, 0x73, 0x76, 0x40, 0x6E, 0x5D, 0xC0, 0x64},
     true,
     {0xA6, 0xA0, 0x88, 0x95, 0x51, 0x2C, 0xBF, 0x18, 0x64, 0x13, 0x4E,
      0x30, 0x22, 0x81, 0x50, 0x07, 0xC6, 0x40, 0x22, 0x9E, 0x64, 0xFF,
      0xE1, 0x22, 0xC8, 0xCA, 0x77, 0x15, 0xE2, 0xB5, 0x61, 0x4B},
     {0xE5, 0xDC, 0xCE, 0x17, 0x4C, 0x4B, 0x39, 0x53, 0x6E, 0x54, 0x8C,
      0xC3, 0x26, 0x89, 0x3C, 0x4C, 0x4C, 0xF6, 0x49, 0x69, 0x9C, 0xEE,
      0x74, 0x64, 0x76, 0xA8, 0x27, 0xCA, 0x56, 0x7D, 0x12, 0xCF}},

    {params_ml_kem768_test_mode,
     {0xE1, 0x5B, 0xD4, 0x60, 0x3F, 0x0E, 0xB6, 0x4E, 0x32, 0xB3, 0xF1,
      0xD1, 0xFA, 0x8E, 0xF6, 0xCC, 0x25, 0xD6, 0x73, 0xA1, 0xD0, 0xBB,
      0x65, 0x9C, 0xEF, 0xBA, 0x2C, 0x15, 0x37, 0x24, 0xC1, 0xE1},
     {0x11, 0xA6, 0x2A, 0x88, 0x96, 0xCF, 0xDF, 0x94, 0x33, 0x96, 0xBB, 0x8F,
      0x58, 0xD9, 0xCE, 0x2C, 0x8A, 0x4D, 0xD0, 0x7A, 0xBB, 0x73, 0x6C, 0xA3,
      0x09, 0x61, 0x42, 0x24, 0xE1, 0x3C, 0xBC, 0x2A, 0x4E, 0xEC, 0xE6, 0xA9,
      0xE1, 0x34, 0xB6, 0x56, 0x25, 0xBF, 0x73, 0x32, 0xA0, 0xB9, 0x78, 0x92,
      0x1F, 0x4B, 0x9B, 0x68, 0x97, 0x36, 0xC7, 0xA9, 0x0C, 0xD7, 0x71, 0x20,
      0xFB, 0x33, 0x12, 0x4B, 0xC2, 0x72, 0xEE, 0x9B, 0x65, 0x69, 0x52, 0x98,
      0x00, 0x77, 0xBA, 0x0A, 0x21, 0x08, 0xAA, 0x34, 0x35, 0x5C, 0x39, 0x11,
      0x18, 0xE2, 0xAD, 0xB2, 0xA1, 0xA6, 0xA6, 0x14, 0x98, 0xDA, 0x2B, 0x3E,
      0x7A, 0x6B, 0x34, 0x77, 0x89, 0x0A, 0x6E, 0xB7, 0x4C, 0xA6, 0x61, 0x61,
      0x97, 0x9B, 0x1C, 0x44, 0xD1, 0x0E, 0xCA, 0x94, 0xCC, 0x26, 0xC9, 0x3B,
      0x43, 0xE4, 0x2F, 0xC6, 0x8A, 0x93, 0xB8, 0x5A, 0xAF, 0x0F, 0x93, 0x81,
      0xDF, 0x30, 0x92, 0x91, 0xE2, 0x16, 0x0C, 0xA1, 0x96, 0x15, 0x03, 0x15,
      0x76, 0xD1, 0x44, 0x8A, 0xE4, 0xA3, 0x6D, 0xB5, 0xA7, 0xEF, 0xA9, 0x95,
      0x42, 0xDA, 0x5F, 0x63, 0x3C, 0x4A, 0x75, 0x47, 0x0F, 0x93, 0x69, 0xAF,
      0xF4, 0xD2, 0x0F, 0xD2, 0xD9, 0x6B, 0x62, 0x34, 0x87, 0x45, 0xA4, 0x6C,
      0x81, 0x05, 0x3A, 0x17, 0xFC, 0xA3, 0xFB, 0x92, 0x78, 0x71, 0xE8, 0x8E,
      0x99, 0x6C, 0x35, 0x55, 0xB6, 0x96, 0xCD, 0xCB, 0x76, 0x39, 0xE1, 0x3E,
      0x11, 0x48, 0x39, 0x7F, 0x48, 0x45, 0x36, 0x4A, 0x7C, 0xCD, 0x24, 0x5D,
      0xB2, 0x85, 0x77, 0xC6, 0x78, 0xB0, 0x61, 0x55, 0x5A, 0x8F, 0x97, 0xC7,
      0xDC, 0x4C, 0x86, 0x12, 0x01, 0xC6, 0x91, 0x06, 0xCA, 0x2F, 0x37, 0x5E,
      0x94, 0xB5, 0x97, 0xBB, 0xB0, 0x90, 0x67, 0xF4, 0xC7, 0x30, 0x60, 0x6D,
      0x2A, 0xF3, 0xB6, 0xFD, 0xB9, 0x8C, 0x14, 0xCA, 0x1A, 0x33, 0x9B, 0x3D,
      0x23, 0x17, 0x4A, 0xB1, 0x9C, 0x08, 0x49, 0x83, 0x6C, 0x33, 0x72, 0x0C,
      0x9A, 0x04, 0x82, 0x1F, 0x59, 0xB9, 0x29, 0xD7, 0x24, 0x21, 0xD7, 0x58,
      0x3E, 0x99, 0x19, 0x53, 0x13, 0x27, 0xB2, 0x96, 0xBD, 0x23, 0x25, 0xB5,
      0xCB, 0x17, 0x2F, 0xB7, 0xE6, 0x20, 0x64, 0x70, 0x91, 0xA9, 0x9A, 0x05,
      0x3E, 0x20, 0x90, 0x75, 0x88, 0x95, 0x0E, 0x30, 0x32, 0x37, 0xFA, 0x68,
      0xF5, 0x7B, 0x1A, 0x30, 0x52, 0x33, 0xB6, 0xE9, 0x31, 0xF2, 0x2A, 0xB4,
      0x67, 0x69, 0xC4, 0x69, 0x04, 0x20, 0xA9, 0x4C, 0x3C, 0x23, 0x4B, 0x56,
      0x4D, 0xEB, 0x30, 0x52, 0xC4, 0xAD, 0xB3, 0xB0, 0x8A, 0x1F, 0x45, 0x5E,
      0x2B, 0x1B, 0x81, 0x32, 0xE6, 0x23, 0x2E, 0x42, 0xCE, 0x33, 0x90, 0x41,
      0x46, 0xAB, 0x6F, 0xFB, 0x69, 0x31, 0x58, 0xAC, 0x41, 0x1D, 0xB0, 0xC4,
      0x9C, 0xA3, 0x99, 0x04, 0x98, 0xBB, 0xBE, 0xC8, 0xBB, 0x06, 0x81, 0x2A,
      0x31, 0xE6, 0x77, 0x61, 0xFC, 0x26, 0xAA, 0x86, 0x1E, 0x32, 0x89, 0x06,
      0xD5, 0xB8, 0x73, 0xCD, 0x2A, 0x8C, 0x92, 0x77, 0x82, 0x60, 0xBC, 0x84,
      0xB5, 0x07, 0x03, 0xEA, 0xC6, 0xBB, 0x10, 0x43, 0x2B, 0xC0, 0x11, 0x5A,
      0xF9, 0xFB, 0x60, 0xC8, 0xF0, 0x2D, 0x90, 0xD6, 0xA2, 0xCF, 0xF8, 0x0A,
      0x69, 0xFC, 0x0E, 0x23, 0xA3, 0x06, 0x88, 0x68, 0xA7, 0xA5, 0x22, 0xCF,
      0x96, 0x87, 0x6F, 0xD1, 0x17, 0x0B, 0xC8, 0x55, 0x2E, 0x45, 0xEB, 0x65,
      0x65, 0x67, 0x40, 0xCF, 0x2A, 0xB1, 0x06, 0xA5, 0x53, 0xCC, 0x61, 0x11,
      0x2B, 0x65, 0x9E, 0x68, 0x62, 0x34, 0x6F, 0x64, 0x35, 0x9F, 0xC6, 0xB5,
      0x8F, 0xAA, 0x70, 0x38, 0x6B, 0x95, 0x5D, 0xE9, 0x18, 0xC5, 0xB6, 0x41,
      0xF9, 0xEA, 0xA2, 0xBE, 0x55, 0x9A, 0x5B, 0xEB, 0xC4, 0x0E, 0x11, 0x3B,
      0xB6, 0xB6, 0x30, 0x58, 0xA8, 0x0C, 0x1F, 0xE1, 0x66, 0x4A, 0x81, 0x87,
      0x3B, 0xE5, 0x06, 0xF8, 0xA7, 0x95, 0xD9, 0xA8, 0xB4, 0x31, 0x83, 0x42,
      0xFE, 0xAA, 0x2A, 0x28, 0xC7, 0xAB, 0xAF, 0x1C, 0x62, 0x4D, 0x5A, 0x8D,
      0xCA, 0x35, 0x29, 0x06, 0xD0, 0x37, 0xB8, 0x83, 0xAF, 0x54, 0x29, 0x53,
      0x28, 0x18, 0x0E, 0xEB, 0x06, 0xC9, 0x10, 0xA0, 0xB7, 0x94, 0x5A, 0x14,
      0x37, 0x1C, 0x88, 0x5E, 0xAC, 0x00, 0x6E, 0x24, 0xB6, 0xA8, 0xF9, 0x59,
      0x36, 0x76, 0x3B, 0x54, 0xC8, 0x9B, 0xEB, 0x23, 0x79, 0xDE, 0x06, 0x1A,
      0xD5, 0xD0, 0xA6, 0x5D, 0x83, 0x98, 0x31, 0x0A, 0x5B, 0x88, 0x44, 0xA8,
      0x1F, 0x2C, 0x8C, 0x77, 0xE7, 0x1F, 0x97, 0x4B, 0xB0, 0x29, 0x7A, 0xBB,
      0x1C, 0x02, 0x44, 0x23, 0x85, 0x62, 0x1D, 0x60, 0x7B, 0xD3, 0x13, 0x8E,
      0x06, 0x3A, 0xC3, 0x6D, 0x72, 0x8A, 0xDE, 0x7A, 0xA6, 0xA9, 0x0B, 0xBA,
      0x32, 0xDC, 0x2F, 0xEC, 0x14, 0x98, 0xF3, 0xD6, 0x8D, 0xA3, 0x29, 0x0C,
      0x17, 0x61, 0x8E, 0x35, 0xDB, 0xC4, 0x32, 0x62, 0xB0, 0x09, 0x35, 0x7C,
      0xF6, 0xF7, 0x9B, 0x04, 0x00, 0x43, 0x16, 0xE8, 0x20, 0x76, 0x96, 0x1D,
      0xE4, 0xB4, 0x7F, 0x2E, 0x52, 0x29, 0xA9, 0x00, 0x8D, 0x4C, 0xE9, 0x02,
      0xD9, 0xF2, 0x6E, 0x50, 0x11, 0xA4, 0xB0, 0x44, 0xAB, 0x0F, 0xDC, 0x66,
      0x8D, 0xBB, 0x68, 0x13, 0x47, 0x9E, 0xCA, 0x3B, 0x0B, 0x18, 0xE0, 0x28,
      0xB1, 0x62, 0x2A, 0x9F, 0x6B, 0x89, 0x94, 0x20, 0x89, 0x9F, 0xC7, 0x24,
      0x6C, 0xA8, 0x6C, 0x00, 0xE7, 0xC4, 0xE1, 0x43, 0x43, 0x65, 0x06, 0x75,
      0x44, 0xD0, 0x2C, 0x44, 0xC8, 0x40, 0xEF, 0x34, 0xCD, 0xEA, 0x6C, 0x30,
      0x63, 0xBB, 0x64, 0xF1, 0x8B, 0x05, 0xEE, 0x37, 0x9A, 0x18, 0xF1, 0x25,
      0x51, 0x4A, 0x6A, 0x94, 0xA9, 0x5A, 0xAD, 0x31, 0x5A, 0x55, 0xF0, 0xAC,
      0xC5, 0x67, 0xBB, 0xD0, 0x70, 0x5B, 0x0A, 0xF7, 0x06, 0x43, 0xF3, 0x18,
      0xA7, 0xC7, 0x0E, 0xEE, 0x05, 0x9F, 0x52, 0xC7, 0xA6, 0x73, 0x53, 0xC8,
      0x9A, 0xB5, 0x5B, 0xB0, 0x91, 0x70, 0x5F, 0x30, 0x59, 0xC9, 0x3B, 0x7E,
      0xFB, 0x30, 0x5C, 0x77, 0x16, 0xA8, 0x6F, 0x13, 0x63, 0xE9, 0x77, 0x82,
      0x7C, 0x57, 0x71, 0x36, 0x22, 0x23, 0x6A, 0x36, 0x50, 0x14, 0xB8, 0x7E,
      0x03, 0x0B, 0xCC, 0xB3, 0x69, 0x6F, 0x42, 0xE6, 0x36, 0x52, 0x01, 0xCD,
      0xDB, 0x57, 0x78, 0x95, 0x17, 0x36, 0xBD, 0x92, 0xA4, 0xD7, 0x8B, 0xBF,
      0x3A, 0xC4, 0x93, 0x9D, 0x37, 0x72, 0xC0, 0xD5, 0x8F, 0x3F, 0x89, 0x5D,
      0xFD, 0x12, 0x4A, 0xEF, 0x4B, 0x6C, 0x45, 0x41, 0x40, 0xC0, 0x02, 0x56,
      0x8A, 0xC4, 0x29, 0x80, 0x62, 0xC1, 0xC7, 0x55, 0xB9, 0x9B, 0x51, 0x13,
      0xF0, 0xAA, 0x2D, 0xC7, 0x42, 0x07, 0x7B, 0x3B, 0x6B, 0xAE, 0x71, 0x47,
      0x42, 0xC0, 0x47, 0x89, 0xFA, 0x3B, 0xC1, 0x2B, 0x83, 0xA9, 0x24, 0x80,
      0x92, 0xA0, 0x45, 0x24, 0x51, 0xA7, 0xAC, 0xA2, 0x55, 0x04, 0xEA, 0xB2,
      0x10, 0x73, 0x74, 0x4E, 0xA7, 0x82, 0x62, 0x60, 0xAB, 0x03, 0xC8, 0x4D,
      0x22, 0x06, 0x6D, 0x21, 0x99, 0xA7, 0x2F, 0xDB, 0xC6, 0xF8, 0x18, 0x61,
      0x3F, 0xC1, 0x6E, 0x45, 0xF0, 0x0B, 0x1E, 0xDB, 0x2F, 0x07, 0xAC, 0xB5,
      0xE6, 0x53, 0x73, 0x1F, 0x8B, 0x70, 0x1B, 0xF0, 0x83, 0x3A, 0x25, 0x8F,
      0x12, 0xB6, 0x48, 0xCC, 0xE7, 0x4A, 0x0D, 0xA7, 0x9D, 0xD4, 0xCC, 0x08,
      0xB2, 0x20, 0x90, 0x42, 0xF4, 0x78, 0xE8, 0x97, 0x67, 0x78, 0x71, 0x53,
      0x8B, 0x90, 0xBB, 0x0C, 0x95, 0x43, 0x12, 0x57, 0x4F, 0x90, 0xA0, 0xB1,
      0x89, 0x89, 0x7A, 0xC0, 0x5C, 0x5A, 0x2E, 0x34, 0x5D, 0x91, 0x57, 0x4C,
      0x60, 0x51, 0x76, 0x6D, 0xAC, 0x73, 0xB7, 0xB3, 0x8A, 0x34, 0x14, 0x71,
      0x4C, 0x24, 0xB1, 0x75, 0xC3, 0x02, 0xCB, 0x09, 0x5F, 0x79, 0xDC, 0xC4,
      0x30, 0x43, 0x80, 0x69, 0x54, 0x1D, 0x6E, 0x98, 0x35, 0xEA, 0xD2, 0x27,
      0x89, 0x90, 0x0C, 0x2E, 0x56, 0xAD, 0xC6, 0x02, 0x55, 0x9D, 0x59, 0x59,
      0x06, 0xC8, 0xB5, 0xAE, 0x01, 0xA0, 0x35, 0x72, 0x18, 0x32, 0x83, 0x53,
      0x3C, 0x64, 0x06, 0x42, 0x90, 0xBE, 0xEB, 0x0A, 0x73, 0xB9, 0xE2, 0x13,
      0x7A, 0x50, 0xC0, 0x5C, 0x1B, 0xC6, 0x35, 0x71, 0x0C, 0x3A, 0x45, 0x25,
      0xE9, 0xC8, 0x70, 0x54, 0xCB, 0xBE, 0x98, 0x6B, 0x49, 0xEC, 0xC2, 0x70,
      0x6C, 0x2A, 0xC3, 0xAC, 0x1A, 0x47, 0x4C, 0xCA, 0x72, 0x32, 0x08, 0xCC,
      0xB6, 0xAA, 0x1B, 0xB7, 0xC5, 0x85, 0x1B, 0x39, 0x0D, 0xFA, 0x3B, 0x0E,
      0x09, 0xB6, 0xAE, 0x60, 0x15, 0x9D, 0x23, 0x1D, 0x59, 0xDA, 0xD2, 0x6B,
      0xF5, 0xAD, 0x61, 0x72, 0x18, 0xFD, 0x68, 0xD6, 0x15, 0x7E, 0x4A, 0x27,
      0x61, 0x22, 0x13, 0x3E, 0x14, 0xBA, 0x42, 0x08},
     true,
     {0x6C, 0x24, 0x6C, 0x21, 0x0A, 0x84, 0x50, 0xD5, 0xF2, 0x0A, 0xAD,
      0x46, 0xA2, 0x4A, 0x3F, 0x16, 0xC1, 0xE6, 0x13, 0xF4, 0x2C, 0x10,
      0x29, 0x77, 0xEF, 0xA8, 0x2C, 0x94, 0xEE, 0xFF, 0xF0, 0xEF},
     {0x66, 0xD5, 0x30, 0x7A, 0xE2, 0x6D, 0xCE, 0x8C, 0xFF, 0xFB, 0xBA,
      0x9B, 0xC0, 0xB2, 0xC6, 0x6E, 0x38, 0xB6, 0xE7, 0x75, 0x37, 0xAE,
      0x52, 0x5B, 0x3E, 0x9A, 0x18, 0xBA, 0xDB, 0xD7, 0x2F, 0xE1}},

    {params_ml_kem768_test_mode,
     {0xD1, 0x76, 0xC0, 0x83, 0x60, 0x15, 0x36, 0x2D, 0x1D, 0xEF, 0xFC,
      0x19, 0x01, 0x12, 0x7B, 0x5C, 0x41, 0xC1, 0x4A, 0xA5, 0x18, 0xBF,
      0xEE, 0x6C, 0x62, 0xF2, 0xEA, 0xEA, 0x1F, 0x22, 0x6A, 0xB5},
     {0xE6, 0x52, 0x23, 0xF4, 0xAB, 0x1C, 0x53, 0x26, 0x74, 0x49, 0xA6, 0x16,
      0xAA, 0x32, 0x39, 0x3E, 0x44, 0x1F, 0x34, 0x68, 0x87, 0x32, 0x9A, 0xC4,
      0xB9, 0xD0, 0x06, 0xF6, 0x1A, 0x25, 0xBC, 0xC3, 0x77, 0x9C, 0x74, 0x0E,
      0xED, 0x0B, 0xBD, 0xEF, 0x0A, 0x37, 0x1B, 0x95, 0x17, 0x1C, 0x70, 0xC6,
      0x0F, 0x23, 0x87, 0x26, 0x54, 0x69, 0xA8, 0xE7, 0xC6, 0x12, 0x35, 0x3F,
      0xC5, 0x19, 0x68, 0xAA, 0x43, 0xC0, 0x4D, 0x16, 0x85, 0xEB, 0x27, 0x36,
      0x4E, 0x26, 0x06, 0x5E, 0xDA, 0x53, 0xD2, 0x87, 0x9D, 0x36, 0x55, 0x2E,
      0x51, 0xEA, 0x01, 0xBD, 0x81, 0x8D, 0xD5, 0x73, 0x19, 0xDF, 0xB1, 0x27,
      0xF2, 0xA5, 0x78, 0xCD, 0x48, 0xBB, 0x60, 0x80, 0xC5, 0xAF, 0xE1, 0x3A,
      0xBA, 0x78, 0x08, 0xB5, 0x89, 0x48, 0x93, 0xBB, 0x41, 0x4A, 0x53, 0x2C,
      0x52, 0x23, 0xB9, 0x1E, 0xB1, 0xCD, 0x23, 0x92, 0x33, 0x31, 0x90, 0x05,
      0x4D, 0xFB, 0x47, 0x22, 0xF5, 0xB5, 0x48, 0xE5, 0x70, 0xFA, 0x19, 0xC7,
      0xC4, 0x85, 0x79, 0x0A, 0xC4, 0x91, 0x38, 0x92, 0x24, 0x9E, 0x63, 0xBB,
      0x32, 0x4C, 0x62, 0x0D, 0x83, 0x5E, 0x5B, 0x8C, 0x92, 0x29, 0x24, 0x79,
      0x1B, 0x08, 0x93, 0x7F, 0x13, 0x9C, 0xCB, 0xEB, 0x2D, 0x5C, 0x9A, 0x72,
      0xF3, 0x44, 0x88, 0x8B, 0xA3, 0x7B, 0x8C, 0xD7, 0xB0, 0x63, 0x49, 0xA3,
      0x10, 0xBB, 0x71, 0xB0, 0x6A, 0x3E, 0xAC, 0xC1, 0x23, 0x6A, 0xB0, 0x3A,
      0x5F, 0xF9, 0xB9, 0x23, 0xFC, 0x5F, 0xAD, 0xE5, 0x4C, 0x5A, 0x1A, 0x2E,
      0x80, 0x34, 0x24, 0xCF, 0x9C, 0x42, 0xE5, 0x04, 0x33, 0x05, 0x12, 0x7F,
      0x3E, 0xFC, 0x95, 0x6D, 0x97, 0x94, 0x6E, 0x6B, 0xB0, 0x2E, 0xB9, 0x13,
      0xD6, 0xD2, 0x16, 0x95, 0xA0, 0xA8, 0x87, 0x08, 0xB6, 0xCC, 0x56, 0x11,
      0xFE, 0xC6, 0x19, 0x32, 0x77, 0x4E, 0x46, 0x2C, 0x18, 0xF4, 0x72, 0x6F,
      0xE9, 0x92, 0xA3, 0x01, 0xB7, 0x4D, 0xE2, 0xF5, 0x6E, 0x9F, 0x79, 0x1E,
      0x90, 0x99, 0x2C, 0x3C, 0x10, 0x98, 0x60, 0x47, 0x4F, 0xBD, 0x6C, 0x5E,
      0xB8, 0x1C, 0xBF, 0xEC, 0x4A, 0x0C, 0x7F, 0x15, 0x4E, 0x5C, 0x23, 0x75,
      0xA2, 0xA3, 0x79, 0xA8, 0x3B, 0x4D, 0xA4, 0xC6, 0x79, 0x61, 0x02, 0xB2,
      0x0C, 0x55, 0x3F, 0x01, 0x5A, 0xC5, 0xAC, 0x49, 0x78, 0x51, 0x38, 0xB2,
      0x46, 0x09, 0x25, 0xBA, 0x3B, 0x15, 0x6E, 0xA3, 0x37, 0xD3, 0x12, 0x14,
      0xEB, 0x94, 0x2E, 0xAA, 0x27, 0x06, 0x2E, 0xC0, 0x73, 0x28, 0x08, 0x09,
      0x27, 0xDC, 0xCA, 0x3C, 0xFA, 0xC5, 0x1A, 0x24, 0x36, 0x45, 0xD5, 0x05,
      0x5E, 0xE0, 0xC6, 0x44, 0x42, 0x8C, 0x47, 0x15, 0x03, 0x2D, 0xB2, 0x75,
      0x7A, 0xF2, 0xAE, 0x25, 0xD7, 0x27, 0xC7, 0x8B, 0x08, 0xD7, 0xB5, 0x9D,
      0x88, 0x41, 0xB9, 0xD2, 0x61, 0x39, 0xC3, 0x71, 0x7F, 0xFE, 0xD9, 0x2E,
      0x09, 0xFC, 0x52, 0x86, 0xBC, 0x1E, 0x1D, 0x29, 0x03, 0x66, 0xE1, 0x83,
      0xFD, 0x2B, 0x55, 0x62, 0x85, 0x9F, 0xC9, 0xC3, 0x21, 0xB7, 0x31, 0x32,
      0x63, 0x5A, 0x53, 0x2D, 0xC3, 0x54, 0x10, 0x96, 0x65, 0x72, 0x25, 0x08,
      0x2C, 0xB3, 0x7D, 0xDE, 0xD9, 0x2C, 0x8C, 0x48, 0x0B, 0x5C, 0xA6, 0x8F,
      0x93, 0xAB, 0x53, 0x16, 0x8B, 0x1F, 0xF2, 0xF0, 0x8D, 0x82, 0x65, 0xAA,
      0x80, 0x35, 0x75, 0x3C, 0x96, 0x89, 0x7D, 0xD0, 0x67, 0x93, 0xA1, 0x45,
      0xEF, 0xDB, 0xBB, 0xC4, 0x51, 0x8E, 0x0C, 0x87, 0xAA, 0x90, 0xA8, 0x95,
      0xE6, 0x95, 0x5E, 0x5B, 0x62, 0x92, 0x99, 0x99, 0x14, 0x3C, 0xB1, 0x6B,
      0x09, 0xE2, 0x64, 0x33, 0xA1, 0xC0, 0xD7, 0x61, 0x05, 0x1A, 0xC3, 0x77,
      0x6E, 0xB1, 0x26, 0xA8, 0xC8, 0x90, 0x94, 0x06, 0x32, 0xB4, 0xC5, 0x0D,
      0xA7, 0xFC, 0x18, 0x14, 0x39, 0xA1, 0x1B, 0x48, 0x35, 0x54, 0xE2, 0x08,
      0xD6, 0xFA, 0xA6, 0x27, 0xB1, 0x1D, 0xC6, 0xBB, 0x50, 0x73, 0xDA, 0x15,
      0x6D, 0x2A, 0x4A, 0x08, 0x83, 0x2D, 0x25, 0x90, 0x87, 0x36, 0x72, 0x06,
      0x6B, 0x9A, 0x35, 0xC6, 0x55, 0x15, 0xB4, 0xCB, 0x56, 0x02, 0x75, 0x3B,
      0x26, 0xC9, 0xC8, 0x65, 0x71, 0x6A, 0x6B, 0xC3, 0xC0, 0x59, 0x27, 0x2F,
      0x0D, 0xF5, 0x4D, 0xB6, 0x39, 0x5F, 0x59, 0x01, 0x2A, 0xCF, 0x2A, 0x78,
      0x3B, 0xBB, 0x03, 0xCB, 0xBC, 0x3A, 0x95, 0xF6, 0x5E, 0x04, 0xA0, 0xBA,
      0x76, 0x14, 0x9A, 0x8D, 0xC4, 0xA3, 0x05, 0x90, 0x14, 0x18, 0x32, 0x48,
      0x85, 0xD4, 0x47, 0x80, 0xF6, 0x5E, 0x17, 0xE4, 0xC6, 0xAA, 0xD0, 0x83,
      0x55, 0xA5, 0x9C, 0x21, 0x55, 0x8B, 0x0D, 0x20, 0x6C, 0x73, 0xF5, 0x9B,
      0x5F, 0xE4, 0x9F, 0x8C, 0x5B, 0x86, 0xA6, 0x74, 0x7D, 0xD9, 0xE8, 0xCB,
      0x9D, 0x56, 0x0B, 0x22, 0xB5, 0x1B, 0x5A, 0x6A, 0xA8, 0x22, 0x77, 0xB6,
      0x76, 0xF0, 0x43, 0x99, 0x57, 0x04, 0x02, 0x81, 0x8A, 0x97, 0x88, 0x1E,
      0xAE, 0x8A, 0x24, 0x5E, 0x1B, 0x75, 0xFA, 0x1A, 0xC5, 0x0B, 0xB0, 0xA9,
      0x52, 0xE3, 0x19, 0xE5, 0x23, 0xA2, 0x92, 0x31, 0x6D, 0xE3, 0xE5, 0xC4,
      0xAA, 0x27, 0x12, 0x65, 0xB7, 0x9D, 0x37, 0x79, 0x5A, 0xA3, 0x59, 0x6C,
      0xAA, 0xB1, 0x07, 0x2F, 0x37, 0x75, 0xD0, 0x65, 0x02, 0x88, 0x03, 0x59,
      0xD7, 0x3C, 0x5E, 0xE0, 0xC7, 0x82, 0xEC, 0xD0, 0x4A, 0xBC, 0xB1, 0x25,
      0x5B, 0x8A, 0x50, 0x51, 0xCA, 0x51, 0xEC, 0x6C, 0x17, 0xEC, 0x95, 0x5B,
      0xB6, 0xE7, 0x15, 0x65, 0x9B, 0x4C, 0xCC, 0x4C, 0x74, 0x48, 0x43, 0x13,
      0xC4, 0xF5, 0x14, 0x9A, 0x28, 0x81, 0x78, 0xC4, 0x9F, 0x99, 0x58, 0x6A,
      0xF5, 0x47, 0x10, 0xBA, 0x54, 0x9C, 0x09, 0xCA, 0xC0, 0xCD, 0xC7, 0x6D,
      0x15, 0x18, 0x3C, 0x87, 0x48, 0x14, 0x6A, 0x07, 0xC7, 0x20, 0x49, 0x10,
      0x8E, 0x86, 0xC0, 0x44, 0x7B, 0x37, 0x1B, 0x84, 0x4C, 0x37, 0x15, 0xA9,
      0x14, 0xE3, 0x85, 0x3F, 0x41, 0x84, 0x33, 0x73, 0x81, 0x32, 0x0A, 0x02,
      0xF9, 0x4B, 0xA1, 0xD9, 0xEB, 0xBD, 0xB0, 0x03, 0x3D, 0x37, 0x72, 0x54,
      0xE4, 0xEA, 0x51, 0xAF, 0x45, 0x5A, 0x77, 0x5C, 0xB8, 0xE8, 0x53, 0x36,
      0xA7, 0x61, 0x86, 0x42, 0x39, 0x42, 0x6A, 0x44, 0x9F, 0xFC, 0xB9, 0x67,
      0xB8, 0x24, 0x61, 0xF6, 0xF5, 0xA2, 0x1D, 0x7A, 0x36, 0xDB, 0x8B, 0x54,
      0xA5, 0xAA, 0x3A, 0xEB, 0x22, 0x97, 0xC3, 0x04, 0x63, 0x06, 0x4C, 0x8A,
      0xFA, 0x12, 0x90, 0x8D, 0x4B, 0x2D, 0x1E, 0x5B, 0x0F, 0x1A, 0x23, 0xB1,
      0xD5, 0x5A, 0x5E, 0x96, 0x88, 0x6B, 0x96, 0xC3, 0x9E, 0x10, 0xC6, 0xA6,
      0x80, 0x08, 0xB0, 0x52, 0x77, 0x03, 0xEA, 0x27, 0x99, 0xF4, 0x2B, 0x50,
      0x98, 0xC4, 0x17, 0x8F, 0x47, 0x49, 0x09, 0x83, 0x79, 0x3D, 0xF4, 0xCE,
      0x2B, 0x0A, 0x2D, 0x2B, 0x56, 0x71, 0x92, 0x56, 0x20, 0x64, 0x93, 0xC3,
      0x95, 0xC5, 0x42, 0x92, 0xE5, 0x16, 0xD4, 0xF8, 0xA5, 0xC9, 0x06, 0x73,
      0x6C, 0x89, 0x81, 0x0B, 0xE0, 0x84, 0x6B, 0x8B, 0xA6, 0x54, 0xC8, 0x6A,
      0x35, 0x5B, 0x56, 0x73, 0x6B, 0x91, 0x44, 0x85, 0x32, 0x86, 0xD6, 0x5E,
      0x61, 0x41, 0x3F, 0x7D, 0xD2, 0x6D, 0x1D, 0x2A, 0xCF, 0x67, 0x28, 0xBF,
      0x6B, 0x02, 0x8F, 0x91, 0x26, 0x02, 0xCD, 0x13, 0x2E, 0x3D, 0x66, 0x03,
      0xDC, 0xBA, 0xB4, 0xCF, 0x65, 0xB0, 0xE9, 0x61, 0xA8, 0xB5, 0x97, 0xA0,
      0x81, 0x0B, 0x9F, 0x78, 0xA1, 0x4F, 0x0D, 0xFA, 0xA1, 0x08, 0x81, 0x1C,
      0xF2, 0x2C, 0x39, 0x4F, 0x21, 0xBF, 0xC5, 0x61, 0x55, 0xA6, 0x7B, 0x85,
      0x72, 0x2C, 0x75, 0x53, 0x51, 0x57, 0x6B, 0x0A, 0xB6, 0x14, 0xB7, 0xC9,
      0xC2, 0x27, 0x8D, 0xBA, 0xE2, 0x11, 0xE3, 0x38, 0x39, 0xB5, 0x53, 0x83,
      0xC5, 0x74, 0x4E, 0x62, 0x70, 0xC9, 0x2D, 0x65, 0xBA, 0x82, 0x59, 0x0A,
      0xC6, 0x13, 0xBF, 0x14, 0x93, 0x76, 0x8E, 0x7A, 0xCE, 0x60, 0xA1, 0x46,
      0x44, 0xC9, 0xA0, 0x2F, 0xB3, 0x0B, 0xB1, 0xA6, 0xB4, 0x2A, 0x6B, 0x69,
      0xA1, 0x42, 0xBF, 0x67, 0x0C, 0x48, 0xEE, 0x03, 0x07, 0x7B, 0xA7, 0xA2,
      0x96, 0x40, 0x20, 0xD5, 0x28, 0xC5, 0x1C, 0xB1, 0x7B, 0x02, 0x45, 0xAF,
      0x11, 0xC4, 0xA8, 0xEB, 0x07, 0x85, 0xAC, 0xC7, 0x38, 0xE4, 0xF0, 0xCF,
      0x0F, 0xFB, 0x83, 0x87, 0x48, 0x31, 0x4C, 0xA5, 0x1C, 0x81, 0x33, 0x1D,
      0xE2, 0x59, 0x63, 0x45, 0xB4, 0x4F, 0x44, 0xBC, 0x51, 0x6E, 0xF1, 0x23,
      0xCD, 0x89, 0x97, 0xE1, 0xFB, 0xC9, 0x3A, 0xD9, 0x5C, 0x0A, 0x9C, 0xB7,
      0x1D, 0x46, 0xC5, 0x53, 0x5A, 0x99, 0xB7, 0x51, 0x22, 0xE5, 0xE7, 0x10,
      0xDA, 0x96, 0x1B, 0xD8, 0x73, 0xE6, 0x6D, 0xDD},
     true,
     {0xE7, 0x69, 0x72, 0x47, 0x4C, 0xBC, 0x95, 0xD5, 0x50, 0x3E, 0xE2,
      0x81, 0x30, 0x13, 0x55, 0xFA, 0x5A, 0x06, 0xB2, 0xF6, 0x9F, 0xBE,
      0x13, 0xAE, 0x17, 0x4C, 0x82, 0x35, 0x33, 0xDC, 0xF1, 0xB0},
     {0xBE, 0x4A, 0x7B, 0x73, 0x9B, 0xBB, 0xFE, 0xA6, 0x2A, 0x02, 0xA5,
      0x55, 0x57, 0x14, 0x65, 0xED, 0xCF, 0xDE, 0xCE, 0xEC, 0x83, 0x84,
      0x67, 0x60, 0xA0, 0xD3, 0x99, 0x44, 0xF9, 0x92, 0x66, 0xE0}},

    {params_ml_kem768_test_mode,
     {0x4E, 0x30, 0x2E, 0xB2, 0xBB, 0x53, 0x92, 0x78, 0x2E, 0x78, 0x20,
      0x86, 0x8D, 0xED, 0xB6, 0x1F, 0x5A, 0x6A, 0xE5, 0x58, 0xCA, 0x30,
      0x7A, 0x01, 0xEC, 0xDE, 0x49, 0x70, 0xE4, 0x3E, 0xB4, 0x48},
     {0x64, 0x53, 0xB4, 0x01, 0x9A, 0x8E, 0x77, 0xC7, 0x60, 0x7A, 0xCC, 0x6D,
      0xF3, 0x37, 0x09, 0x7A, 0x9A, 0x10, 0xEA, 0x48, 0x36, 0xDB, 0xFC, 0x7C,
      0x79, 0x6A, 0x4E, 0x73, 0xA2, 0x36, 0x06, 0x9C, 0x20, 0x05, 0xC2, 0xBA,
      0x2D, 0x58, 0x43, 0xAE, 0x7A, 0x00, 0x8A, 0x33, 0x78, 0x76, 0x18, 0x2C,
      0x44, 0xB9, 0x2E, 0x66, 0xEB, 0x51, 0x46, 0x82, 0x6B, 0x24, 0x6C, 0xC7,
      0xC3, 0xF4, 0x4D, 0x40, 0xB0, 0xC8, 0x8A, 0x98, 0xC1, 0x1B, 0x87, 0x3A,
      0x29, 0xD7, 0xA4, 0xC7, 0x33, 0x65, 0x12, 0xE0, 0x04, 0x13, 0xEB, 0x51,
      0x9A, 0xA3, 0x90, 0x25, 0xAC, 0x46, 0x3C, 0x7C, 0xC2, 0x89, 0x27, 0x0A,
      0x68, 0x0B, 0x88, 0xA7, 0xFB, 0x9C, 0xFF, 0x59, 0x2F, 0xD6, 0x56, 0x26,
      0xE8, 0x31, 0xA0, 0x19, 0xEA, 0x9E, 0x4C, 0x9B, 0xB5, 0x7B, 0xD3, 0x4B,
      0x8D, 0x28, 0xB8, 0x85, 0x56, 0x16, 0xFC, 0xF1, 0x5F, 0xD6, 0x13, 0x89,
      0x47, 0xCA, 0x36, 0xD8, 0x63, 0x51, 0x0E, 0x6B, 0x3D, 0x4B, 0x22, 0xB8,
      0xF8, 0x12, 0x24, 0x99, 0x13, 0x14, 0x3F, 0x7C, 0xAD, 0xBF, 0x15, 0xAF,
      0x99, 0x1C, 0x7E, 0x36, 0xC1, 0xA6, 0x10, 0x87, 0x9D, 0x92, 0x40, 0x39,
      0x8A, 0x1A, 0x12, 0x24, 0x83, 0x7F, 0x90, 0x4C, 0x65, 0xF0, 0xC1, 0x52,
      0x7B, 0xCC, 0xCE, 0x84, 0x59, 0xB5, 0x94, 0x73, 0x9F, 0x4B, 0xC2, 0x43,
      0xD2, 0x36, 0x0C, 0xAC, 0x32, 0x0A, 0xFE, 0xB5, 0xB2, 0xF5, 0xA3, 0x10,
      0xC9, 0x0B, 0x42, 0xD3, 0xEC, 0x36, 0x9F, 0x64, 0xC0, 0xFD, 0x21, 0x18,
      0xD0, 0x35, 0x9D, 0xBC, 0x31, 0x41, 0xF6, 0x02, 0xCA, 0x8D, 0x75, 0x07,
      0xEF, 0x83, 0x0F, 0x1E, 0x47, 0x68, 0x20, 0x68, 0xAB, 0xF2, 0x25, 0x93,
      0x56, 0x3A, 0x07, 0x16, 0x53, 0xAC, 0xAF, 0xF1, 0x95, 0x06, 0xB5, 0x63,
      0x0F, 0x2B, 0x64, 0x01, 0xB7, 0x89, 0x53, 0xB1, 0x52, 0xC7, 0xA7, 0x8D,
      0x33, 0x21, 0x66, 0x95, 0xB7, 0x77, 0x8F, 0x94, 0x67, 0xFA, 0x23, 0x9E,
      0x4E, 0xD0, 0x89, 0xA1, 0xE6, 0xBC, 0xDB, 0x68, 0x2C, 0x4E, 0xC1, 0xB7,
      0xCF, 0x40, 0x7B, 0x37, 0xB0, 0x88, 0x27, 0x77, 0xB1, 0x24, 0x36, 0x06,
      0x3A, 0xC0, 0xC4, 0x7C, 0xB7, 0xA5, 0x8B, 0xB7, 0xCF, 0x34, 0xC5, 0xCD,
      0xEC, 0x29, 0x1B, 0xF1, 0xEB, 0x85, 0x1D, 0xDC, 0xBE, 0x42, 0x53, 0x5E,
      0x06, 0x39, 0x58, 0x0B, 0x19, 0x1C, 0xDB, 0xA9, 0xAB, 0x22, 0xF0, 0x24,
      0x3D, 0x13, 0x34, 0x04, 0x89, 0x64, 0x96, 0x89, 0x41, 0xE7, 0x14, 0x0C,
      0x44, 0xA1, 0x65, 0x8F, 0xE2, 0x0C, 0x8A, 0xA0, 0x23, 0xD2, 0xCB, 0x13,
      0x3D, 0x5B, 0x06, 0xFF, 0xD9, 0x9F, 0xFD, 0x47, 0x51, 0xEB, 0xEA, 0x17,
      0x0F, 0x9C, 0x30, 0xBA, 0xD8, 0xCA, 0x39, 0xC6, 0x4C, 0x7F, 0xF3, 0x57,
      0xF0, 0xFA, 0x25, 0x85, 0xBB, 0x13, 0x64, 0x56, 0x82, 0xDA, 0xE2, 0x45,
      0x18, 0x51, 0x75, 0x45, 0x17, 0x5E, 0x38, 0xD5, 0x27, 0x6F, 0xC4, 0x42,
      0xFB, 0x21, 0x43, 0xFF, 0xD2, 0xA7, 0xB6, 0x19, 0x03, 0xCD, 0x72, 0x58,
      0x3F, 0xC6, 0x4D, 0xDF, 0xA6, 0x54, 0x08, 0x1A, 0x45, 0x33, 0x45, 0x7D,
      0x36, 0xA3, 0x66, 0xBA, 0x53, 0x8B, 0x67, 0x72, 0xC5, 0xFF, 0x11, 0x67,
      0x4F, 0x0A, 0x70, 0x23, 0x40, 0xC5, 0x42, 0x35, 0x7B, 0x80, 0xB2, 0xC2,
      0x06, 0xBB, 0xAF, 0x2C, 0xF6, 0x40, 0x6F, 0x33, 0x94, 0x36, 0xA7, 0xA1,
      0x05, 0x07, 0x92, 0xAA, 0x67, 0x4C, 0x2E, 0x9C, 0x9B, 0x79, 0x86, 0x40,
      0x4C, 0x94, 0xCE, 0xEF, 0xC3, 0x43, 0xE0, 0x78, 0x11, 0x55, 0xD8, 0xB7,
      0x37, 0x97, 0x8B, 0xFA, 0xE3, 0xC8, 0xCA, 0x64, 0x72, 0x3D, 0x06, 0x56,
      0xE9, 0xB8, 0x9D, 0x23, 0x56, 0x8A, 0xC6, 0x65, 0x8F, 0x07, 0x32, 0xC3,
      0x32, 0x09, 0x6A, 0x21, 0x7C, 0xC6, 0xFA, 0x3A, 0x09, 0xFE, 0xB1, 0x65,
      0x52, 0xF5, 0xC9, 0x82, 0xCA, 0x6B, 0x54, 0x0C, 0x3A, 0xC9, 0x6C, 0x8D,
      0x0A, 0xD5, 0x76, 0x82, 0x0C, 0xC4, 0x6D, 0xD6, 0x0E, 0xC8, 0x2B, 0xA0,
      0xCF, 0xC1, 0x40, 0x73, 0x42, 0x49, 0x02, 0x57, 0x8A, 0xFE, 0xBB, 0x44,
      0xC3, 0x21, 0xA4, 0xA6, 0xB7, 0xAF, 0x97, 0x09, 0xB1, 0xD9, 0xF0, 0x5E,
      0xC9, 0xBB, 0x6B, 0x92, 0x8C, 0x93, 0x6B, 0x67, 0x05, 0x43, 0xE1, 0xB0,
      0x5E, 0x64, 0x9F, 0xA7, 0xC2, 0xB3, 0xB9, 0xFB, 0x44, 0x1C, 0x97, 0x6F,
      0xDC, 0xEB, 0x88, 0x9F, 0xF6, 0x30, 0x19, 0xF2, 0x6E, 0xB2, 0x32, 0x6F,
      0x36, 0xF8, 0x22, 0x31, 0xA6, 0x8F, 0x23, 0x0C, 0x9B, 0xF3, 0x84, 0x71,
      0x0E, 0x11, 0xCC, 0xA8, 0x58, 0x97, 0x0A, 0x33, 0x44, 0x5F, 0xF0, 0x5C,
      0xC4, 0x39, 0x9D, 0xC2, 0x8B, 0x28, 0x58, 0x57, 0x16, 0x8D, 0xD3, 0x88,
      0xF0, 0xB9, 0x5B, 0x7E, 0x3B, 0x16, 0x99, 0x83, 0x09, 0x50, 0x02, 0x2F,
      0xBA, 0x81, 0x98, 0x4A, 0xA1, 0xB1, 0x54, 0x09, 0x59, 0xEC, 0x14, 0x69,
      0xDC, 0x69, 0x6D, 0x7B, 0xB9, 0xA3, 0xC4, 0x21, 0x53, 0x54, 0x0B, 0x24,
      0xA3, 0x50, 0xCF, 0xCC, 0xBC, 0x33, 0xE5, 0xD9, 0x13, 0x68, 0x24, 0x50,
      0x03, 0x19, 0xAD, 0xEE, 0x62, 0x6B, 0xEE, 0x22, 0xA5, 0x3F, 0xF8, 0xCC,
      0xE2, 0xFB, 0x19, 0x5D, 0x38, 0x25, 0x7D, 0xDB, 0x8C, 0x30, 0x59, 0x49,
      0xD7, 0x20, 0x34, 0x2E, 0xB5, 0x8A, 0xF7, 0x22, 0x36, 0xF2, 0x27, 0x5F,
      0x98, 0xD3, 0x35, 0xDE, 0x35, 0xA0, 0xC0, 0xC6, 0x6B, 0xA7, 0x52, 0xBE,
      0xAA, 0xC3, 0x2D, 0x11, 0x3A, 0xA4, 0x3D, 0xA0, 0x92, 0xC4, 0x88, 0x8D,
      0x17, 0x50, 0xA5, 0x9B, 0xEC, 0x4E, 0xB1, 0x4A, 0x94, 0xA8, 0x4C, 0x99,
      0x47, 0x66, 0x08, 0x91, 0x45, 0x7B, 0x52, 0xF1, 0xBC, 0x3E, 0xF3, 0x3C,
      0x95, 0x1A, 0x1D, 0xE6, 0x4A, 0xCA, 0xEE, 0xC9, 0x9F, 0x77, 0x95, 0xB8,
      0xC9, 0x4A, 0x03, 0x20, 0xF4, 0x0D, 0x0E, 0xD9, 0x2E, 0xB7, 0xFA, 0x34,
      0x6E, 0x04, 0x24, 0x53, 0x99, 0x5F, 0x23, 0x94, 0xB7, 0x14, 0xA3, 0x72,
      0xE6, 0xB6, 0x78, 0xC0, 0xBB, 0x4E, 0x04, 0x67, 0x0B, 0x34, 0x7C, 0x15,
      0x67, 0x94, 0x67, 0x2F, 0xCA, 0x87, 0x96, 0x6B, 0x26, 0xF0, 0xEB, 0xBD,
      0x2B, 0x83, 0x04, 0xC9, 0xB6, 0x56, 0x04, 0x59, 0x01, 0x9E, 0xD9, 0x55,
      0x39, 0xC7, 0xC0, 0xC2, 0x09, 0x99, 0x97, 0x15, 0x21, 0xF8, 0xF7, 0x53,
      0x7B, 0x13, 0x93, 0x7A, 0xEA, 0xB1, 0xE2, 0x37, 0x36, 0x18, 0x6C, 0x67,
      0x2A, 0xF0, 0xB0, 0x84, 0xD4, 0xC5, 0xC3, 0xE1, 0x88, 0x7B, 0xDC, 0x0B,
      0xB2, 0x81, 0xB7, 0xDB, 0x92, 0x4B, 0xFD, 0xD9, 0x03, 0x4C, 0x64, 0x56,
      0xC4, 0x27, 0xB9, 0x64, 0x3C, 0x89, 0xFE, 0xBA, 0x34, 0x40, 0x94, 0x7D,
      0xD6, 0xB0, 0xB4, 0xBD, 0xD5, 0x4C, 0x38, 0x21, 0x53, 0x71, 0xE1, 0xA7,
      0x4E, 0x54, 0x3A, 0x60, 0x73, 0xCB, 0x46, 0x11, 0x8D, 0xE3, 0xB2, 0xBD,
      0x11, 0x54, 0xAC, 0x53, 0x66, 0x52, 0xBD, 0x26, 0x66, 0xF5, 0xF7, 0x10,
      0xC0, 0x3C, 0xB6, 0xAC, 0xE9, 0x48, 0xFB, 0x61, 0x69, 0x2F, 0xA3, 0xA3,
      0xE2, 0xEC, 0x4E, 0x3A, 0xD9, 0x71, 0x88, 0x44, 0x12, 0x64, 0x03, 0x42,
      0xBB, 0xB4, 0xC0, 0x9D, 0x1C, 0x11, 0x85, 0x68, 0x25, 0xD7, 0x9A, 0x8E,
      0x52, 0x49, 0x24, 0x30, 0x62, 0x03, 0x3E, 0x37, 0x9D, 0xD1, 0xEA, 0x65,
      0x26, 0x67, 0x6A, 0x24, 0x92, 0x52, 0x0B, 0xC5, 0x5A, 0x45, 0x33, 0xCE,
      0xC4, 0x7C, 0x54, 0xCD, 0xAC, 0x74, 0x83, 0x50, 0x0C, 0x07, 0x51, 0xBA,
      0x84, 0xF0, 0xCB, 0xB0, 0x20, 0xCF, 0xAD, 0x07, 0xB6, 0xF4, 0xE8, 0x93,
      0xCA, 0xE7, 0x8D, 0xA0, 0xC7, 0x08, 0x2F, 0x7A, 0x9C, 0x78, 0xD2, 0xBA,
      0xA0, 0x64, 0xC3, 0x3D, 0x20, 0x64, 0x44, 0x31, 0xBE, 0xDC, 0xB9, 0xB4,
      0x93, 0xD0, 0x5C, 0x13, 0x10, 0x4B, 0x88, 0x87, 0x03, 0x61, 0x43, 0x34,
      0x90, 0x92, 0x7B, 0x01, 0x52, 0x61, 0xB6, 0x3B, 0x1A, 0xBF, 0x2B, 0x59,
      0x77, 0x73, 0x59, 0xF7, 0x4B, 0xC2, 0x59, 0x72, 0x38, 0xDE, 0x67, 0x29,
      0x66, 0x93, 0x83, 0xE8, 0x18, 0x8C, 0x0D, 0xFA, 0x72, 0x35, 0xD9, 0x35,
      0x65, 0x7B, 0x93, 0x61, 0x01, 0xBA, 0x4D, 0xA9, 0xCB, 0x90, 0xF1, 0xAE,
      0xD0, 0xF0, 0x6C, 0x03, 0x70, 0x05, 0x74, 0xB9, 0x91, 0x59, 0xFC, 0x3E,
      0x59, 0xEB, 0x32, 0x58, 0xA9, 0xBD, 0x5F, 0x47, 0x43, 0x9B, 0x45, 0xA3,
      0x01, 0x63, 0x30, 0x9A, 0x6A, 0x29, 0xCD, 0xBB, 0xC0, 0x17, 0x84, 0x0C,
      0x56, 0x55, 0xE3, 0xE8, 0xFA, 0xFD, 0xBC, 0xA1, 0x42, 0x93, 0xB9, 0x1C,
      0x07, 0xEE, 0xAB, 0x7E, 0x6C, 0x06, 0x6A, 0x6B, 0x8B, 0xA7, 0xEC, 0x5F,
      0xAE, 0xC0, 0x35, 0x0B, 0x9C, 0x88, 0x7B, 0x18},
     true,
     {0xB7, 0x6D, 0xA0, 0x60, 0xA0, 0xED, 0x40, 0x40, 0x6C, 0xC5, 0x49,
      0xB1, 0x77, 0xBF, 0x77, 0x35, 0xB7, 0xA4, 0xD0, 0xA4, 0xFB, 0xBD,
      0x16, 0xF7, 0x83, 0xDC, 0xED, 0x0A, 0x1B, 0x42, 0x21, 0xFC},
     {0xCC, 0x54, 0xEF, 0xA4, 0xBA, 0x6B, 0x3C, 0x0B, 0x65, 0x12, 0x58,
      0xEF, 0xA6, 0xC6, 0x85, 0x0B, 0x1B, 0x31, 0xFB, 0x15, 0x9C, 0x28,
      0x2D, 0x6F, 0x35, 0x4D, 0xC1, 0x8C, 0x87, 0x49, 0xAC, 0xD7}},

    {params_ml_kem768_test_mode,
     {0x7B, 0x33, 0x4E, 0x04, 0x58, 0x96, 0xC0, 0x0F, 0x90, 0xD8, 0x11,
      0x48, 0x9D, 0x49, 0x1E, 0x8D, 0x72, 0xC4, 0xE3, 0xA2, 0x2E, 0xD8,
      0x31, 0xC0, 0x19, 0xFD, 0x4B, 0xD9, 0x67, 0xB7, 0xA8, 0x02},
     {0x40, 0x94, 0x71, 0xE1, 0x02, 0x49, 0x44, 0xD5, 0x20, 0x3F, 0x82, 0xC6,
      0x2D, 0x1C, 0xA6, 0x7E, 0x96, 0x56, 0x7F, 0x3A, 0x52, 0x5F, 0x6A, 0xA2,
      0x26, 0x83, 0xC4, 0xD5, 0x37, 0xBA, 0x87, 0xDC, 0xA6, 0x68, 0x64, 0xC8,
      0x8F, 0x29, 0x26, 0x30, 0x48, 0x4C, 0xE6, 0xC2, 0x59, 0x2B, 0xE1, 0x94,
      0xA5, 0x9B, 0x96, 0x42, 0xB0, 0xC0, 0xC1, 0xC4, 0x77, 0x1D, 0xF8, 0x77,
      0x10, 0xF1, 0x1F, 0xC8, 0x89, 0x85, 0x73, 0xBB, 0x87, 0x49, 0x65, 0xB7,
      0x74, 0xC3, 0xBC, 0xF3, 0x09, 0x20, 0x31, 0x2B, 0xC3, 0xC1, 0x94, 0x58,
      0x76, 0xAC, 0x6C, 0xC1, 0x22, 0x1C, 0x90, 0x19, 0x67, 0x70, 0x87, 0x95,
      0x1E, 0xF4, 0x30, 0x31, 0x37, 0x3B, 0xE3, 0x44, 0x92, 0xCC, 0x56, 0x38,
      0xD5, 0x30, 0xA4, 0xD0, 0x1C, 0x86, 0xF9, 0xB7, 0x02, 0x96, 0x16, 0x29,
      0x50, 0xD9, 0x2B, 0x5D, 0x97, 0x72, 0x4B, 0xC1, 0xB7, 0xE9, 0x05, 0x6C,
      0x78, 0xD1, 0x62, 0x90, 0xF3, 0xBE, 0xA4, 0x59, 0xAC, 0x21, 0x39, 0x5F,
      0xD8, 0xA6, 0x0E, 0xA8, 0x55, 0x1B, 0x5D, 0x93, 0x55, 0x10, 0x55, 0x7C,
      0xC3, 0x6C, 0xC5, 0xC4, 0x26, 0x05, 0xA4, 0x45, 0x9C, 0xD7, 0xCC, 0x60,
      0x34, 0x08, 0xC3, 0xAB, 0x23, 0x80, 0xF7, 0xF0, 0x0F, 0x44, 0xEB, 0x0D,
      0xE7, 0x45, 0x7E, 0x6F, 0x40, 0xB8, 0xDA, 0x20, 0x42, 0xF3, 0x4C, 0x18,
      0x55, 0x46, 0x06, 0xE8, 0xA2, 0x74, 0xFC, 0x8B, 0x68, 0x42, 0xF2, 0x99,
      0xC1, 0xE8, 0x00, 0x68, 0xF1, 0x18, 0x70, 0x90, 0x7D, 0x8D, 0x44, 0x26,
      0x9B, 0xB7, 0x78, 0x9A, 0xF0, 0xC8, 0x46, 0x5B, 0x72, 0xCE, 0x64, 0xB5,
      0x43, 0x1A, 0x88, 0x5A, 0x00, 0xB0, 0xC2, 0x00, 0x10, 0x2C, 0x81, 0x1E,
      0x8A, 0xDB, 0x78, 0x93, 0x50, 0x22, 0x77, 0xFB, 0x87, 0x21, 0x3A, 0xA8,
      0x90, 0x51, 0x85, 0x2E, 0x89, 0x6B, 0x3A, 0x17, 0x0A, 0x33, 0xCA, 0x8F,
      0x44, 0xA7, 0x36, 0x77, 0x56, 0x94, 0x47, 0x60, 0xB1, 0xD3, 0xC3, 0x73,
      0x18, 0xAB, 0x76, 0xEA, 0xCB, 0xB4, 0x7B, 0xA1, 0x5D, 0xBD, 0x83, 0xCF,
      0x81, 0x51, 0xBD, 0x2E, 0x36, 0x72, 0x93, 0x47, 0x02, 0xC3, 0xA2, 0x18,
      0x86, 0xB2, 0x7B, 0xC3, 0x50, 0x6D, 0xFA, 0x8C, 0xCD, 0xB7, 0x7C, 0x81,
      0x87, 0x34, 0xA1, 0x82, 0x30, 0x86, 0x80, 0x92, 0x18, 0x64, 0x31, 0x22,
      0x1E, 0x30, 0x20, 0x15, 0x73, 0x74, 0xAD, 0xD1, 0x38, 0xC8, 0xF3, 0xC6,
      0x9D, 0x77, 0x09, 0xBF, 0x60, 0xC5, 0xA0, 0x6B, 0xA3, 0xE4, 0x10, 0xAF,
      0x15, 0x6B, 0x21, 0x75, 0x02, 0x0E, 0xBD, 0x76, 0x03, 0xAE, 0x10, 0xC7,
      0x6D, 0xB7, 0xC2, 0xCE, 0x06, 0x6A, 0x98, 0xA0, 0xAD, 0x41, 0x31, 0x78,
      0x1C, 0xC3, 0x03, 0x11, 0xB3, 0x8A, 0x78, 0xCB, 0x80, 0x9B, 0xA4, 0x81,
      0x36, 0x00, 0x9C, 0x36, 0x14, 0x4B, 0x47, 0xAA, 0x20, 0x52, 0xE5, 0x9F,
      0x16, 0xE0, 0x21, 0x65, 0x61, 0xBB, 0x42, 0x06, 0x28, 0x17, 0xC6, 0x7D,
      0x81, 0xC9, 0x62, 0x23, 0xC3, 0x7D, 0xA9, 0x33, 0x2B, 0x26, 0x11, 0x52,
      0xED, 0x90, 0x35, 0x0B, 0xB2, 0xBB, 0xED, 0xBC, 0x43, 0x86, 0x41, 0xCC,
      0x08, 0x2C, 0x8E, 0x4A, 0x41, 0x81, 0x62, 0x82, 0x53, 0x73, 0x4C, 0xB2,
      0x62, 0x06, 0x1E, 0x40, 0x14, 0x3B, 0xA1, 0x1B, 0xB5, 0xAB, 0x75, 0xA8,
      0x92, 0x70, 0x63, 0x6B, 0x33, 0x0E, 0x92, 0x2C, 0x52, 0x98, 0x06, 0x8C,
      0x85, 0xC1, 0x35, 0x2F, 0xA5, 0xA1, 0x33, 0xC3, 0x90, 0x0A, 0x76, 0x2F,
      0x57, 0xB8, 0x31, 0xA9, 0x07, 0xAC, 0x37, 0xA6, 0xC9, 0xCB, 0xA9, 0xC9,
      0x15, 0x29, 0x0E, 0x10, 0xA4, 0x47, 0xA8, 0xB4, 0x54, 0xA0, 0xF7, 0x35,
      0xDD, 0xB8, 0x85, 0x2D, 0x79, 0xB0, 0xFC, 0xEB, 0x20, 0x7C, 0x25, 0x56,
      0x8F, 0x44, 0xCE, 0x95, 0x68, 0x30, 0x5A, 0x93, 0x62, 0xED, 0x58, 0x4D,
      0x77, 0x12, 0x3A, 0x37, 0x16, 0xA6, 0x67, 0xCA, 0x29, 0x52, 0xCA, 0xCF,
      0xD7, 0x38, 0x23, 0x03, 0xA6, 0x9A, 0x81, 0x7B, 0x17, 0xD8, 0x95, 0x44,
      0x1D, 0x0A, 0x9A, 0xE1, 0xF7, 0x1E, 0xCE, 0x12, 0x77, 0xC0, 0x43, 0x2A,
      0x7F, 0xA2, 0xA6, 0x3C, 0x81, 0xA5, 0x45, 0x91, 0x0B, 0x4B, 0x21, 0x2F,
      0x1D, 0x98, 0x3D, 0xC6, 0x34, 0x31, 0x3B, 0xC4, 0xB3, 0xB1, 0xB3, 0x91,
      0x90, 0x43, 0x8D, 0x47, 0xB1, 0x67, 0x47, 0xFA, 0xB4, 0x8D, 0xE4, 0x77,
      0x79, 0x11, 0xB8, 0xD3, 0xFB, 0x65, 0xFA, 0xC6, 0x8C, 0xA0, 0x96, 0x91,
      0x92, 0x67, 0x78, 0x3B, 0x04, 0x97, 0x86, 0x8A, 0x9F, 0x95, 0xE8, 0x9A,
      0x1A, 0x3B, 0x28, 0x61, 0xA3, 0xBC, 0x89, 0xCB, 0x77, 0xF6, 0x97, 0x64,
      0x63, 0xCC, 0x7B, 0x1C, 0xA6, 0xCB, 0x8D, 0x44, 0x7A, 0xA6, 0x54, 0x63,
      0x17, 0x4A, 0x38, 0x42, 0x17, 0xCF, 0xD4, 0x1B, 0x4B, 0x80, 0xB7, 0xAA,
      0x40, 0x72, 0xB1, 0x0B, 0xCC, 0xBF, 0xA7, 0x3C, 0x53, 0xE4, 0x56, 0x84,
      0xC5, 0x42, 0xB0, 0x31, 0x28, 0x8C, 0x11, 0x92, 0xAF, 0xB1, 0x07, 0x7B,
      0x4E, 0x2B, 0x56, 0x0E, 0x24, 0x81, 0x69, 0xC6, 0xB7, 0x41, 0x90, 0x43,
      0x04, 0xB0, 0x6D, 0x23, 0xC8, 0x6C, 0x94, 0x34, 0xAD, 0x64, 0xC6, 0x7D,
      0x92, 0xFB, 0x87, 0x2D, 0x28, 0xBF, 0x9F, 0x74, 0x5F, 0x2C, 0xF2, 0x10,
      0xD6, 0xD0, 0xC9, 0xC7, 0x21, 0x76, 0xA3, 0x24, 0x24, 0x47, 0xE0, 0x69,
      0x25, 0x71, 0x8F, 0xE6, 0x53, 0x0E, 0x2B, 0x50, 0x18, 0x83, 0x38, 0x31,
      0x81, 0x9C, 0x23, 0xD3, 0x40, 0x2B, 0xF9, 0xDC, 0x22, 0xFA, 0x7B, 0x1B,
      0x75, 0x4C, 0x6C, 0xD5, 0xA5, 0xC6, 0x10, 0x27, 0x53, 0x52, 0xD8, 0xB0,
      0xA0, 0x41, 0x59, 0x71, 0x3C, 0x26, 0x42, 0x2C, 0x68, 0x8E, 0xD7, 0x0B,
      0x14, 0x20, 0x9D, 0xA8, 0xEC, 0x23, 0xE8, 0xA2, 0xB2, 0x85, 0xD6, 0x1B,
      0x8C, 0x59, 0x93, 0x73, 0x97, 0x07, 0x38, 0x6B, 0x4A, 0x08, 0x4B, 0x22,
      0xF7, 0x0B, 0xA9, 0xEE, 0x30, 0x73, 0x64, 0x45, 0x21, 0xE9, 0xB6, 0x0D,
      0x77, 0x75, 0x5C, 0x30, 0xB3, 0x61, 0x80, 0xB6, 0xB6, 0xA6, 0x4A, 0x94,
      0xAC, 0xEA, 0x2A, 0xFA, 0xA7, 0x84, 0xC5, 0x3C, 0x0C, 0x78, 0xE7, 0x92,
      0x03, 0xF3, 0x6E, 0x98, 0x01, 0x6C, 0xD1, 0xD1, 0x09, 0xE1, 0x79, 0x52,
      0xA4, 0x90, 0x21, 0x57, 0x2B, 0x5F, 0x43, 0x85, 0x7F, 0xA4, 0x01, 0xBA,
      0x61, 0x71, 0x46, 0xE0, 0x94, 0x50, 0xC8, 0xF3, 0x92, 0xE4, 0x23, 0xAF,
      0x9F, 0xA2, 0x94, 0xB7, 0xE2, 0xC2, 0xD7, 0x72, 0x78, 0x4B, 0x15, 0x7F,
      0xF5, 0x91, 0xA4, 0xA1, 0x39, 0x19, 0xC5, 0x73, 0x71, 0x33, 0x12, 0x29,
      0x1B, 0xC0, 0x2F, 0x0F, 0x0C, 0x93, 0x72, 0x17, 0x96, 0x47, 0x76, 0x08,
      0xE8, 0xE6, 0x9F, 0xC6, 0x42, 0x62, 0xD4, 0x04, 0xC9, 0x5D, 0xB9, 0x8E,
      0x54, 0x8A, 0x76, 0x7A, 0x18, 0x40, 0x47, 0xA9, 0x2C, 0x8F, 0xD8, 0xAE,
      0xD9, 0x72, 0x94, 0x5E, 0x67, 0x6D, 0x94, 0x6A, 0xA0, 0xDF, 0x5A, 0x88,
      0xC7, 0xA1, 0x3D, 0x03, 0xE5, 0x01, 0xD2, 0x28, 0x48, 0x2F, 0xF9, 0x7D,
      0x62, 0xE2, 0x75, 0x29, 0x37, 0x2C, 0xCA, 0x96, 0x81, 0xF0, 0x75, 0x8C,
      0x2B, 0x50, 0x0B, 0xB5, 0xD6, 0x91, 0xD1, 0x73, 0x48, 0x59, 0xB9, 0x3E,
      0x8E, 0x17, 0x76, 0x05, 0x72, 0x93, 0x37, 0xB3, 0x98, 0x60, 0xE2, 0x56,
      0xD6, 0x1C, 0x75, 0xBD, 0x46, 0x2A, 0x66, 0x07, 0x69, 0x2B, 0xF7, 0x4A,
      0x4C, 0xCC, 0x34, 0x62, 0x40, 0x05, 0x8B, 0x76, 0x65, 0x02, 0xE7, 0x40,
      0xAF, 0x86, 0x45, 0x31, 0x30, 0x12, 0x26, 0xEB, 0xC3, 0xB8, 0xB5, 0xB1,
      0x37, 0x88, 0x46, 0x19, 0x0A, 0xAD, 0x61, 0xC9, 0xBE, 0xF1, 0x36, 0x25,
      0x37, 0x60, 0x6D, 0xEA, 0xA5, 0x31, 0xEF, 0x97, 0x58, 0x2D, 0xD1, 0x55,
      0x1E, 0x7A, 0x5F, 0x96, 0x0B, 0x8D, 0xEA, 0x7A, 0x18, 0x15, 0x27, 0x21,
      0x45, 0xB2, 0x5C, 0x7D, 0x17, 0x5C, 0xDB, 0x56, 0x8A, 0x68, 0xF0, 0x50,
      0x58, 0xF2, 0x55, 0xB5, 0xB5, 0x86, 0x86, 0x62, 0x78, 0xF3, 0xAA, 0x56,
      0x56, 0x81, 0x79, 0xDF, 0x4B, 0x84, 0x8F, 0xE5, 0x35, 0x13, 0x2B, 0x7C,
      0x52, 0xD3, 0xAA, 0x38, 0x29, 0x68, 0x6E, 0xA7, 0xCB, 0x6F, 0xA1, 0xBF,
      0x07, 0x09, 0x9F, 0x05, 0x54, 0x16, 0xB8, 0x04, 0x1A, 0xA9, 0xE2, 0x52,
      0x2C, 0xD2, 0x3F, 0xE2, 0x99, 0x81, 0xB0, 0x6B, 0x65, 0x56, 0x43, 0x2A,
      0xBE, 0xF0, 0x2E, 0x07, 0x80, 0xBE, 0xAE, 0x77, 0x7D, 0x26, 0xFC, 0x06,
      0x2D, 0x94, 0xF1, 0xBC, 0x4C, 0x68, 0x3A, 0xD2, 0xB9, 0x23, 0x03, 0xD5,
      0x32, 0x10, 0x14, 0x61, 0xFC, 0x0B, 0x81, 0x15, 0x55, 0x6C, 0x3D, 0x2F,
      0x2B, 0x85, 0x5D, 0x10, 0x09, 0x70, 0x46, 0x11},
     true,
     {0xC0, 0xF8, 0xFF, 0x46, 0xE9, 0xF5, 0x46, 0x46, 0x4D, 0x0E, 0x3C,
      0xC8, 0x3D, 0x3A, 0x05, 0xB8, 0xA2, 0xCD, 0x1A, 0xE4, 0x98, 0xEC,
      0xF7, 0x8D, 0xB8, 0xC9, 0xF1, 0xE9, 0x77, 0x43, 0x0C, 0x48},
     {0xC2, 0x6D, 0xA6, 0xA2, 0x33, 0x32, 0xB2, 0x09, 0x14, 0xF7, 0x03,
      0xE7, 0xCB, 0x23, 0x7D, 0x84, 0xF8, 0x07, 0xCC, 0x72, 0x48, 0xDD,
      0xC4, 0x75, 0x99, 0xDD, 0xB0, 0xD4, 0x0F, 0xDC, 0x1F, 0xAF}},

    {params_ml_kem768_test_mode,
     {0x94, 0x7A, 0xFE, 0x33, 0x93, 0x4E, 0x81, 0x50, 0xB0, 0x6B, 0xDD,
      0x1E, 0xAE, 0x40, 0xCF, 0x82, 0xEA, 0x99, 0xC0, 0xC0, 0x10, 0x6B,
      0x10, 0x12, 0x83, 0xEA, 0x38, 0x2E, 0xDA, 0xD9, 0x4A, 0x8E},
     {0xD0, 0xF1, 0xB5, 0x84, 0xA8, 0x7C, 0xBA, 0x74, 0x09, 0xD8, 0xB9, 0x8B,
      0x49, 0xB1, 0x33, 0x2A, 0xCA, 0x15, 0x45, 0xB2, 0x9F, 0xA2, 0xD4, 0x2B,
      0xC5, 0x37, 0xCF, 0x95, 0x9C, 0x61, 0x82, 0x30, 0x5C, 0x95, 0xE0, 0x6F,
      0x50, 0x40, 0xB4, 0x47, 0xD0, 0x07, 0x37, 0xB0, 0xBB, 0xBF, 0xD9, 0xAB,
      0x0A, 0x60, 0x62, 0xFA, 0xE2, 0x09, 0xB2, 0x49, 0x59, 0xEB, 0x83, 0xC2,
      0xF0, 0xC5, 0xA2, 0xD8, 0xAB, 0x36, 0x16, 0x1B, 0x90, 0xE2, 0xA9, 0x2C,
      0x59, 0x39, 0x05, 0x9F, 0x8B, 0x95, 0x86, 0xF0, 0x71, 0xFB, 0x1B, 0x26,
      0x12, 0x16, 0x04, 0xD5, 0x25, 0x2D, 0x13, 0x67, 0x05, 0xD3, 0x39, 0x11,
      0x70, 0x3B, 0x91, 0xDD, 0xD8, 0x5E, 0xCF, 0x80, 0xA7, 0xA1, 0x74, 0x1C,
      0x29, 0xFC, 0xB0, 0x6F, 0x32, 0x1E, 0x8B, 0xDB, 0x81, 0x66, 0xEA, 0x8C,
      0x89, 0x9B, 0x80, 0x64, 0xB6, 0x9D, 0x77, 0x92, 0x7C, 0x0B, 0x50, 0x39,
      0x89, 0x10, 0x1F, 0x1A, 0x69, 0x41, 0xF4, 0xDA, 0xB0, 0x6B, 0xAC, 0x99,
      0x73, 0x37, 0x8C, 0x2E, 0x6B, 0x3B, 0x19, 0xD9, 0x50, 0x6B, 0x2C, 0x9A,
      0x6B, 0xC2, 0x9A, 0x51, 0x58, 0x2C, 0x8B, 0x47, 0xB4, 0x63, 0x06, 0xC1,
      0xB3, 0x5A, 0x43, 0x8A, 0x09, 0x7B, 0x07, 0x40, 0x56, 0x45, 0xFC, 0x66,
      0xC2, 0x66, 0x0F, 0x4C, 0x14, 0x45, 0x7A, 0x41, 0xB9, 0x19, 0x07, 0xCB,
      0x28, 0x64, 0x33, 0x8C, 0x09, 0xC0, 0xCB, 0xAA, 0x7A, 0x87, 0x7A, 0x95,
      0x43, 0xF7, 0x75, 0x95, 0x04, 0x87, 0xDC, 0xB8, 0xCD, 0x8A, 0x10, 0x58,
      0x22, 0xB9, 0x38, 0x82, 0x27, 0x1A, 0x17, 0xB6, 0x93, 0x69, 0x12, 0xA5,
      0x5F, 0xB4, 0x10, 0x9B, 0x45, 0xB0, 0x9B, 0xB1, 0x4E, 0xAC, 0x21, 0x34,
      0xE0, 0xD7, 0x47, 0x1D, 0xE8, 0x91, 0xCA, 0x98, 0x4F, 0xD7, 0xB1, 0x94,
      0xB1, 0x52, 0x9E, 0xBD, 0xA3, 0xCE, 0x0C, 0x24, 0xC1, 0xA9, 0xA1, 0x70,
      0xD7, 0x61, 0x45, 0x21, 0x5C, 0x8C, 0xA7, 0x21, 0x7A, 0x13, 0x8C, 0x9F,
      0xB7, 0xDA, 0x83, 0x85, 0xE8, 0xA0, 0xC5, 0x6B, 0x53, 0x42, 0xDA, 0x66,
      0xF9, 0xFA, 0xB0, 0x3A, 0xAC, 0xA4, 0xE3, 0xA1, 0xBC, 0x54, 0x95, 0xC7,
      0xAB, 0x0A, 0xB0, 0xE4, 0x51, 0xA2, 0x2D, 0x48, 0x0B, 0x00, 0xFB, 0x29,
      0xCE, 0x64, 0x1C, 0x46, 0xE7, 0xA5, 0xEF, 0x16, 0x86, 0x71, 0x01, 0xA2,
      0x69, 0x11, 0xB0, 0xA4, 0x91, 0x40, 0xF7, 0x0B, 0x86, 0x84, 0x84, 0x58,
      0x50, 0xDB, 0x8E, 0x15, 0xCB, 0x7B, 0xF8, 0x42, 0xCE, 0x71, 0x24, 0x80,
      0xB1, 0x93, 0x43, 0x07, 0x76, 0x74, 0xD8, 0xA7, 0xB7, 0xC4, 0x7A, 0x5E,
      0x11, 0x70, 0xA6, 0x29, 0xE5, 0x3A, 0xDD, 0xD1, 0x0F, 0x4B, 0xB3, 0x5A,
      0x60, 0x1A, 0x84, 0x3B, 0x49, 0x39, 0x1D, 0x73, 0x37, 0xD7, 0xB3, 0xB5,
      0x92, 0x18, 0x2F, 0x3A, 0x16, 0x36, 0xE4, 0xA7, 0xCE, 0xA1, 0x46, 0x76,
      0xA0, 0xDC, 0x50, 0x84, 0x23, 0xB9, 0x4B, 0x11, 0x50, 0x41, 0xEA, 0x5F,
      0x92, 0x5B, 0xA0, 0xF2, 0xFA, 0x85, 0x36, 0xAB, 0x11, 0xE6, 0xD4, 0x51,
      0x32, 0x14, 0x50, 0xCD, 0xFA, 0x07, 0x6E, 0xAA, 0x35, 0x77, 0xD6, 0x05,
      0xA7, 0xE0, 0xB3, 0x54, 0x75, 0x4B, 0x92, 0xF1, 0xB4, 0x11, 0x48, 0x7F,
      0xD8, 0x12, 0x2E, 0x8D, 0xAC, 0x14, 0xD8, 0x46, 0x4E, 0xC4, 0x33, 0x1D,
      0xC7, 0x46, 0xB5, 0x69, 0x34, 0x51, 0xB5, 0x28, 0x78, 0x92, 0x36, 0x49,
      0x14, 0x41, 0x82, 0xC1, 0xF0, 0x3D, 0xC4, 0x28, 0x1A, 0x79, 0x91, 0xC6,
      0x0F, 0x4A, 0x40, 0x76, 0x11, 0x7B, 0x50, 0xC2, 0x16, 0x70, 0x80, 0x98,
      0x32, 0xF3, 0xA6, 0x7D, 0xC8, 0x2E, 0x44, 0x8A, 0x49, 0xAA, 0x3A, 0x20,
      0x02, 0xB1, 0xA8, 0x66, 0x7C, 0xC9, 0x47, 0x25, 0x30, 0x15, 0xA6, 0xA8,
      0x8F, 0xFC, 0x13, 0x35, 0x54, 0x74, 0x62, 0x56, 0xA1, 0x03, 0xD1, 0x1C,
      0x7B, 0xC9, 0x80, 0x13, 0x26, 0x4D, 0x5E, 0xFC, 0x15, 0xA0, 0xB2, 0x82,
      0x5B, 0x37, 0xC3, 0x56, 0xA9, 0x6F, 0x0A, 0x5B, 0x70, 0x1B, 0x4C, 0xBA,
      0xBF, 0x57, 0x64, 0x0F, 0x2A, 0x92, 0x79, 0x6C, 0x56, 0x63, 0x57, 0x0B,
      0x30, 0xFA, 0xC6, 0x5B, 0xB7, 0x9A, 0xCB, 0x59, 0xCE, 0x5C, 0x4C, 0xC7,
      0x75, 0x79, 0x20, 0x49, 0x6B, 0x85, 0xCF, 0x93, 0x89, 0x92, 0xEB, 0x32,
      0x04, 0x83, 0x60, 0x55, 0x97, 0x45, 0x36, 0x88, 0x54, 0xEC, 0x7A, 0x85,
      0xA8, 0xE8, 0x62, 0xA9, 0x73, 0xAD, 0x50, 0xD6, 0x79, 0x38, 0x26, 0x5D,
      0x35, 0x41, 0x16, 0x1B, 0xE9, 0xC4, 0x09, 0x73, 0x23, 0xF5, 0xE7, 0x7E,
      0xB8, 0xF6, 0x68, 0x87, 0xA8, 0x9F, 0xB7, 0xDA, 0x1A, 0x25, 0x85, 0xA4,
      0xDD, 0x52, 0x78, 0x2A, 0xD2, 0x21, 0xB5, 0x15, 0xA1, 0xD3, 0x77, 0x78,
      0xB5, 0x91, 0x1A, 0xC7, 0xC9, 0x0F, 0xA7, 0x72, 0x2F, 0xD5, 0xA9, 0x48,
      0x44, 0x86, 0xA6, 0x21, 0x01, 0x57, 0x18, 0x8A, 0xCB, 0x61, 0x74, 0x13,
      0x51, 0x69, 0x55, 0xF5, 0xEB, 0x91, 0xD2, 0x69, 0x1E, 0x1E, 0x38, 0xCC,
      0x2D, 0x10, 0x63, 0x83, 0x30, 0x02, 0x9D, 0x03, 0x30, 0x05, 0xA8, 0x86,
      0x06, 0xE6, 0xA9, 0x41, 0x96, 0x2F, 0x3F, 0xAC, 0x50, 0x6F, 0xDA, 0x52,
      0x11, 0x60, 0x91, 0x38, 0xE5, 0x03, 0xE3, 0x38, 0x2B, 0xD7, 0x16, 0x40,
      0x36, 0x40, 0x66, 0xA2, 0xC2, 0x77, 0x80, 0x89, 0x8A, 0xFC, 0x1C, 0xCA,
      0x95, 0x09, 0xC2, 0x3F, 0x02, 0x2A, 0xA7, 0x75, 0x46, 0x01, 0x7B, 0x19,
      0x43, 0x36, 0x0F, 0xAE, 0x24, 0xA5, 0x4A, 0x0B, 0x96, 0x5B, 0xAA, 0x79,
      0x66, 0xF4, 0x5A, 0xA7, 0xF3, 0xBA, 0xA7, 0xC6, 0x27, 0x1C, 0xD2, 0x8D,
      0x4E, 0x02, 0x63, 0xBC, 0xCB, 0xBF, 0xE4, 0xFB, 0xCD, 0xFF, 0xE0, 0x52,
      0x69, 0xB0, 0x48, 0xE9, 0x48, 0xBF, 0x2E, 0xEA, 0x1E, 0x3C, 0x18, 0x25,
      0xFB, 0xB3, 0xAA, 0x8F, 0x18, 0x2B, 0x2B, 0x7B, 0xA5, 0x28, 0x77, 0xB2,
      0x30, 0x44, 0x34, 0xDC, 0xD5, 0x61, 0xBF, 0x36, 0x5B, 0xCD, 0xC6, 0xAE,
      0x09, 0x60, 0x21, 0x1B, 0xA1, 0x4E, 0xE1, 0xF9, 0xC8, 0xCB, 0xEA, 0x37,
      0x50, 0x47, 0x2A, 0x02, 0x27, 0x1B, 0xA7, 0xE7, 0x9C, 0x4F, 0xE6, 0xC9,
      0x01, 0x9B, 0x25, 0x47, 0x09, 0x62, 0x31, 0x5C, 0x4F, 0x50, 0x33, 0x24,
      0x14, 0xF4, 0x62, 0x33, 0xA3, 0x54, 0x92, 0x89, 0x80, 0xA7, 0x75, 0x2D,
      0x7A, 0x07, 0x21, 0x5E, 0xC2, 0x41, 0x33, 0x44, 0x65, 0x21, 0xDA, 0x5E,
      0x4C, 0x50, 0xB6, 0x15, 0xAC, 0x4E, 0x78, 0xF0, 0x7B, 0x6E, 0xEA, 0x14,
      0x1A, 0xE1, 0x46, 0xFF, 0x5B, 0xC2, 0xC6, 0x31, 0x67, 0x48, 0x3C, 0xCA,
      0xD3, 0x11, 0xB8, 0x47, 0x99, 0x63, 0x39, 0xAA, 0x50, 0xB2, 0xC2, 0x37,
      0xB4, 0x48, 0x6E, 0x3B, 0x53, 0x8C, 0x75, 0xB1, 0x49, 0x69, 0x67, 0x64,
      0x65, 0xF1, 0x06, 0xB9, 0x06, 0x9A, 0x6F, 0x62, 0x79, 0x38, 0x69, 0x01,
      0x39, 0xF0, 0x11, 0xF5, 0xFA, 0x7D, 0x18, 0x32, 0xB9, 0x6A, 0x03, 0x09,
      0xFC, 0x85, 0x92, 0x0E, 0x14, 0x2C, 0x9C, 0x57, 0xCF, 0x64, 0x85, 0x2E,
      0x45, 0x64, 0x33, 0x42, 0x81, 0xBA, 0x63, 0x06, 0x8C, 0x56, 0x66, 0xA7,
      0x1B, 0x19, 0x43, 0x19, 0xE9, 0x9F, 0xB0, 0x24, 0xAD, 0x33, 0xF8, 0x79,
      0xD0, 0x8C, 0x16, 0x74, 0x3B, 0x34, 0x03, 0x8A, 0xAD, 0x20, 0xD2, 0xC8,
      0x4C, 0xB7, 0x44, 0x0E, 0xE0, 0x4C, 0x1A, 0xF4, 0xB9, 0x44, 0x22, 0x7F,
      0xE9, 0x4B, 0x5D, 0x66, 0x6C, 0x3E, 0x51, 0x52, 0x32, 0x48, 0xD9, 0x13,
      0xB9, 0xA3, 0x69, 0x3D, 0xD1, 0x14, 0x7A, 0xD7, 0x06, 0x46, 0x96, 0xCE,
      0x5B, 0x61, 0x3D, 0xFA, 0xE1, 0xB8, 0x3A, 0x20, 0x77, 0x42, 0x26, 0x4E,
      0x87, 0x8C, 0x78, 0x04, 0xD6, 0x03, 0xBD, 0xD7, 0x1E, 0x56, 0xC5, 0xCA,
      0x7F, 0x88, 0x71, 0x58, 0xE8, 0x05, 0x16, 0x1C, 0x95, 0x9F, 0x07, 0x2C,
      0xBF, 0xF8, 0x2D, 0xAE, 0x02, 0x0C, 0x80, 0x13, 0x5A, 0x8D, 0x18, 0x90,
      0xF1, 0xA9, 0xA8, 0x39, 0x03, 0x2B, 0x0D, 0x69, 0xB9, 0xA7, 0xA9, 0x77,
      0x85, 0xDA, 0x5C, 0x20, 0x16, 0x85, 0xC6, 0x83, 0x84, 0x4D, 0xDC, 0x99,
      0xF4, 0xA9, 0x7E, 0x00, 0xA0, 0x63, 0xE1, 0x51, 0x22, 0xAF, 0x22, 0xC2,
      0x74, 0x30, 0x32, 0x8E, 0xA1, 0x43, 0x5A, 0xCB, 0x07, 0xAC, 0xDA, 0x0A,
      0x3B, 0x6B, 0x78, 0x4E, 0xB1, 0x9E, 0x30, 0xE0, 0x83, 0x53, 0xCA, 0x26,
      0x11, 0x8A, 0xC4, 0xD5, 0xC4, 0xA2, 0x18, 0xA2, 0x62, 0xC4, 0xBA, 0x13,
      0x5A, 0xAC, 0x24, 0xF1, 0xEE, 0x7C, 0x5E, 0xA0, 0xE1, 0x3C, 0x86, 0x74,
      0x9E, 0x5E, 0x72, 0x54, 0x1C, 0xA6, 0xCA, 0xA1, 0xE1, 0xC0, 0x51, 0x74,
      0xB0, 0x87, 0x45, 0x43, 0x7F, 0xEE, 0xD0, 0xB9},
     true,
     {0x32, 0x0D, 0x3C, 0x28, 0x42, 0x54, 0x29, 0xFA, 0x19, 0x06, 0xE1,
      0x22, 0xAA, 0x60, 0x70, 0x1B, 0xB7, 0x38, 0xCA, 0xE4, 0x42, 0x32,
      0xF7, 0x2F, 0x49, 0xBC, 0x99, 0xC3, 0xF1, 0x79, 0x42, 0xFE},
     {0x10, 0x41, 0x77, 0xA2, 0x7A, 0x18, 0xB5, 0xF3, 0x5D, 0x2C, 0xBE,
      0x9B, 0xFA, 0xBA, 0xA2, 0xEA, 0x98, 0x7B, 0x42, 0x96, 0x94, 0x6D,
      0xEE, 0x57, 0x5B, 0x45, 0xA3, 0xA9, 0xB4, 0x4C, 0xA9, 0x9D}},

    {params_ml_kem768_test_mode,
     {0xDC, 0x85, 0x10, 0xF4, 0x55, 0x28, 0xD6, 0x98, 0x1E, 0x59, 0xC1,
      0xAA, 0x6B, 0x74, 0x3B, 0xB8, 0x44, 0x37, 0x7D, 0x73, 0x39, 0xE3,
      0x59, 0x03, 0x69, 0x29, 0xF0, 0xEE, 0xC5, 0x4F, 0xE6, 0x3C},
     {0x28, 0xC9, 0x38, 0xC9, 0x80, 0x60, 0x95, 0x4A, 0xA5, 0x57, 0x10, 0x2D,
      0x5B, 0xF3, 0x9C, 0x17, 0x44, 0x4A, 0x81, 0x11, 0x56, 0xAD, 0xB2, 0x55,
      0x45, 0x62, 0xC5, 0xAD, 0xE5, 0x66, 0x67, 0x04, 0x17, 0x1A, 0xA3, 0xB1,
      0x99, 0x64, 0x44, 0x9E, 0x1C, 0x12, 0xE7, 0x21, 0xA3, 0x02, 0x99, 0x30,
      0x7E, 0xC7, 0xAB, 0xC2, 0xCB, 0xBC, 0x82, 0x01, 0x0E, 0x3A, 0xDC, 0x6D,
      0xFE, 0xB2, 0xC4, 0x97, 0xDA, 0xBB, 0xDA, 0xC9, 0x73, 0x02, 0x1C, 0x65,
      0x35, 0x77, 0x3C, 0x12, 0x7B, 0x37, 0x93, 0x79, 0x2E, 0xBC, 0x71, 0x7D,
      0x39, 0x2B, 0xBF, 0xD4, 0x90, 0x6D, 0x4F, 0x21, 0xC6, 0x2F, 0xD6, 0xC5,
      0x65, 0x93, 0x5D, 0xA9, 0x85, 0x50, 0xA2, 0xA3, 0x10, 0x48, 0x89, 0x56,
      0x79, 0x9A, 0xB9, 0x74, 0x3B, 0x94, 0x44, 0x66, 0x58, 0xFC, 0x97, 0xA9,
      0xE1, 0xC3, 0xB2, 0x63, 0xC2, 0x30, 0x22, 0x13, 0x2D, 0x33, 0xD2, 0x78,
      0x92, 0x95, 0xC9, 0x94, 0xB6, 0xA4, 0x41, 0x00, 0x0F, 0x7F, 0xBA, 0x45,
      0xC8, 0x3A, 0x92, 0x07, 0x39, 0xA1, 0x73, 0x12, 0xAA, 0x97, 0xB1, 0x62,
      0x5F, 0xE5, 0x11, 0x8A, 0xCB, 0x15, 0x52, 0xB6, 0x47, 0xFA, 0x91, 0xC9,
      0x3C, 0x11, 0x18, 0x2D, 0x9B, 0x9A, 0xFB, 0xFC, 0xB1, 0xA0, 0xD1, 0x41,
      0x2A, 0x7B, 0xCF, 0x92, 0xFC, 0x2E, 0x1A, 0x19, 0x7E, 0xE4, 0xE7, 0x28,
      0xBB, 0x18, 0x12, 0xB8, 0x8A, 0x87, 0x72, 0x9C, 0x6E, 0x11, 0x7A, 0x45,
      0xE4, 0xE0, 0xAC, 0xAB, 0x2B, 0xC4, 0xE4, 0xA2, 0x14, 0xDB, 0xF3, 0xBB,
      0x45, 0x28, 0x31, 0x67, 0x18, 0xC4, 0x77, 0x32, 0x9D, 0xFB, 0x12, 0x29,
      0xAF, 0x9B, 0xCF, 0x0A, 0x00, 0x31, 0x7F, 0xF5, 0x26, 0xB8, 0xE4, 0x82,
      0x8C, 0xA7, 0x97, 0xB3, 0x54, 0xA0, 0xBD, 0x30, 0x4F, 0xFF, 0xFC, 0x3E,
      0x16, 0xC8, 0x6A, 0x49, 0x35, 0x5A, 0x54, 0xA0, 0x00, 0x5C, 0xB1, 0xCE,
      0x53, 0x2C, 0xBA, 0x6A, 0x19, 0x26, 0x74, 0x51, 0xAB, 0xC9, 0xC6, 0x45,
      0xA3, 0xF4, 0x9F, 0xB0, 0x75, 0x92, 0x2B, 0xA9, 0x31, 0xB1, 0xF1, 0x39,
      0xF8, 0xC6, 0x27, 0x11, 0x3B, 0x53, 0x58, 0xE0, 0x2E, 0x81, 0xE7, 0x1C,
      0x23, 0x1A, 0x41, 0xC1, 0xD6, 0x47, 0x84, 0x0B, 0x5A, 0xA9, 0x69, 0x37,
      0xB2, 0x29, 0x26, 0xD6, 0x47, 0x7E, 0x5B, 0x3A, 0x5D, 0x1C, 0x00, 0x48,
      0x5B, 0xD9, 0x1F, 0xE5, 0x67, 0x9C, 0xEE, 0xF1, 0x86, 0x70, 0x22, 0x3E,
      0x67, 0x08, 0x79, 0xEB, 0x3B, 0x02, 0xD0, 0x39, 0x58, 0xA7, 0x75, 0x1C,
      0x00, 0x75, 0x64, 0xED, 0xC3, 0x7D, 0xC0, 0x52, 0x47, 0x90, 0xC8, 0xA0,
      0x03, 0x5A, 0xB3, 0x0A, 0xC3, 0x10, 0xF8, 0x41, 0x98, 0x97, 0x59, 0x77,
      0x21, 0xD6, 0xB4, 0xEA, 0x87, 0x30, 0x04, 0x53, 0xCF, 0xE9, 0x81, 0x44,
      0x0F, 0xF0, 0xC1, 0x8A, 0x32, 0x3A, 0x01, 0x59, 0xB4, 0x7A, 0x9C, 0x26,
      0x25, 0x45, 0x35, 0x9F, 0x50, 0x31, 0xEF, 0x8B, 0x24, 0xA0, 0x08, 0x0B,
      0xDF, 0x0B, 0x55, 0xE1, 0x60, 0x63, 0x55, 0xB8, 0x8F, 0xC5, 0x78, 0x03,
      0xA3, 0xE2, 0x5A, 0xB5, 0xAB, 0xA5, 0x95, 0x82, 0x05, 0x4E, 0x18, 0x4F,
      0xD9, 0x7C, 0xC3, 0xE4, 0x63, 0x5B, 0xC7, 0x7A, 0xB7, 0x97, 0xA7, 0x48,
      0x67, 0x7B, 0x52, 0xBA, 0x76, 0xA1, 0x6A, 0x95, 0x64, 0xBA, 0x21, 0x86,
      0x55, 0xA0, 0x07, 0x0F, 0x20, 0xBD, 0x52, 0xB0, 0xA8, 0xC5, 0x12, 0xA9,
      0x47, 0x53, 0x4B, 0xBD, 0xC9, 0x9D, 0x38, 0x62, 0x1C, 0x65, 0x59, 0x70,
      0xB2, 0x09, 0xC6, 0xDC, 0x8A, 0x00, 0x33, 0x10, 0x50, 0x60, 0x13, 0x3B,
      0x60, 0x03, 0x67, 0xE6, 0x23, 0x66, 0x03, 0x08, 0x10, 0x06, 0x23, 0x87,
      0xBD, 0xD1, 0x20, 0xCE, 0xA0, 0x04, 0x24, 0xC4, 0xC3, 0x07, 0xE6, 0x4E,
      0xED, 0x5A, 0x36, 0x85, 0xF7, 0xB7, 0x8E, 0x51, 0x13, 0x8E, 0x02, 0xC4,
      0xFA, 0xF5, 0x1A, 0xE3, 0x59, 0x49, 0x9E, 0x2A, 0x97, 0xBF, 0x37, 0x99,
      0xE2, 0x16, 0x37, 0x7E, 0xC3, 0x58, 0x1A, 0xC8, 0x1C, 0x12, 0xCC, 0x8B,
      0x41, 0xAB, 0x95, 0xEA, 0x98, 0xC6, 0xD0, 0x50, 0x0B, 0xBC, 0x39, 0x60,
      0xAC, 0x05, 0xA2, 0x29, 0x27, 0x89, 0x84, 0x46, 0x02, 0xCA, 0x75, 0x28,
      0xB2, 0xB9, 0x1E, 0xAA, 0xB7, 0xA5, 0xF2, 0x13, 0x99, 0xE1, 0x39, 0x27,
      0xC5, 0x8B, 0xB0, 0x30, 0xF4, 0x92, 0x10, 0x2B, 0x43, 0x61, 0xDB, 0xB4,
      0xB3, 0x02, 0x39, 0x61, 0x11, 0x70, 0x99, 0xA7, 0x64, 0x8E, 0xB1, 0x1B,
      0x2B, 0xF2, 0x57, 0x03, 0x97, 0x2C, 0xF7, 0x78, 0xB5, 0xD8, 0xAC, 0x08,
      0x7B, 0xA5, 0x9B, 0xC5, 0x12, 0x2D, 0xBA, 0x63, 0xAD, 0x03, 0xEA, 0xBC,
      0x02, 0x03, 0x1E, 0xF6, 0x0C, 0xC7, 0x65, 0xA0, 0x47, 0x8F, 0x2C, 0x15,
      0xDF, 0x31, 0x16, 0x59, 0x03, 0xCE, 0x56, 0x68, 0x0C, 0x2A, 0x62, 0xCF,
      0x7F, 0xA8, 0x45, 0x9D, 0x26, 0xC2, 0xD3, 0xC3, 0x46, 0x61, 0x7C, 0x51,
      0x04, 0x23, 0x51, 0xB3, 0x21, 0xBE, 0xDB, 0xDC, 0x75, 0x1D, 0x90, 0x80,
      0xE0, 0xF6, 0x4D, 0x26, 0x73, 0xC1, 0x70, 0x55, 0xBF, 0x5D, 0xA5, 0x3D,
      0x74, 0xA0, 0x08, 0x83, 0x57, 0xC1, 0xBD, 0xB8, 0x58, 0x9A, 0xF5, 0x50,
      0x5C, 0x40, 0xCE, 0xEF, 0xD7, 0x14, 0x0B, 0x95, 0x78, 0x0B, 0xDC, 0x9B,
      0x31, 0x48, 0x2C, 0x75, 0xF5, 0xC6, 0xF3, 0x95, 0x26, 0x4A, 0x54, 0xC2,
      0xA5, 0x4B, 0x3E, 0xB6, 0x45, 0x4C, 0x87, 0x95, 0x10, 0xBC, 0xFB, 0x42,
      0xDC, 0xD9, 0x54, 0x8C, 0x3A, 0x53, 0x99, 0x60, 0x30, 0xCE, 0x76, 0xA8,
      0xA2, 0x43, 0x53, 0x16, 0xCC, 0x22, 0x7B, 0x59, 0x1F, 0xEA, 0x7A, 0x0C,
      0x12, 0x40, 0x5D, 0x27, 0x93, 0x6A, 0x03, 0x26, 0xA6, 0x6A, 0x6C, 0xAF,
      0x0B, 0x27, 0x1B, 0xB6, 0x43, 0x37, 0xD9, 0xDC, 0xB2, 0xE2, 0x47, 0x1D,
      0xDC, 0x63, 0x2F, 0x3E, 0x03, 0x00, 0xF9, 0x2C, 0x04, 0xB2, 0x88, 0x19,
      0x1F, 0x08, 0x56, 0xC5, 0x93, 0x76, 0xFD, 0x75, 0x57, 0x29, 0xF1, 0x95,
      0xC2, 0x64, 0x82, 0x46, 0x57, 0x72, 0x5E, 0x22, 0x1B, 0xD0, 0x17, 0x35,
      0x09, 0x82, 0xBB, 0x98, 0x20, 0x78, 0xCA, 0x7A, 0x67, 0x83, 0xC6, 0x4C,
      0x5E, 0xF4, 0x62, 0xF3, 0x29, 0xBF, 0x6B, 0xDA, 0xC6, 0x85, 0xD1, 0xBE,
      0x48, 0x3A, 0xA7, 0x8B, 0x7B, 0xA7, 0xF2, 0xA0, 0xBD, 0x0C, 0x0C, 0x3A,
      0x24, 0x34, 0x1C, 0x9A, 0xF0, 0x9C, 0x72, 0xD8, 0x26, 0x74, 0xB0, 0x84,
      0x9B, 0xB7, 0x0B, 0xD4, 0x86, 0x1E, 0x5F, 0xEC, 0x25, 0x37, 0x52, 0x23,
      0xF5, 0xE4, 0xA4, 0xE2, 0x00, 0x95, 0xF3, 0x30, 0xC4, 0x3A, 0x3A, 0xB2,
      0x3B, 0x63, 0x3A, 0x0D, 0xA0, 0x45, 0x53, 0xF6, 0x90, 0x43, 0xA0, 0xB6,
      0xAF, 0x86, 0x2A, 0x94, 0x83, 0x5A, 0xD2, 0x84, 0xC3, 0x1F, 0x82, 0xC2,
      0x3E, 0x80, 0xC2, 0xA4, 0x88, 0x87, 0x2E, 0x4B, 0x31, 0x46, 0x36, 0x09,
      0x4C, 0x70, 0xA5, 0xF7, 0x61, 0x4B, 0x11, 0x74, 0xAE, 0x46, 0xC1, 0x1F,
      0x45, 0x66, 0x12, 0x56, 0xD5, 0x66, 0xC6, 0x80, 0x19, 0x85, 0xA2, 0xA9,
      0x19, 0x90, 0x88, 0x99, 0x99, 0x63, 0x1F, 0xB5, 0xBD, 0x40, 0x72, 0xC8,
      0x84, 0xB8, 0x5B, 0xA7, 0x95, 0x21, 0xBB, 0x2C, 0x7A, 0x72, 0x9C, 0xA3,
      0xCA, 0x8C, 0x18, 0xF5, 0xEB, 0x34, 0x0E, 0xB8, 0x4B, 0x00, 0xCB, 0x99,
      0x1D, 0x17, 0x93, 0x3B, 0x32, 0x12, 0x29, 0x38, 0x28, 0xBE, 0x17, 0x34,
      0xF2, 0x1B, 0x77, 0x5B, 0xE2, 0x89, 0xF7, 0x7C, 0xA4, 0x58, 0x46, 0xB3,
      0xB6, 0xF0, 0x29, 0xD6, 0xC6, 0x10, 0xA0, 0x73, 0x8F, 0x61, 0xB5, 0xC8,
      0x0B, 0xC8, 0x08, 0xA8, 0xA5, 0x06, 0xBD, 0x10, 0x22, 0x94, 0x0A, 0xC7,
      0x55, 0xA7, 0xA4, 0x76, 0x5A, 0x1C, 0x4A, 0x36, 0xB4, 0x2D, 0xA4, 0x53,
      0xAA, 0x4B, 0x51, 0x6F, 0x78, 0xB2, 0xC4, 0x8C, 0xB9, 0x45, 0x00, 0x40,
      0x1F, 0xDC, 0x65, 0x4C, 0x1B, 0xA5, 0xC0, 0xD5, 0x8F, 0xCE, 0x89, 0x4E,
      0x45, 0x56, 0x67, 0xB2, 0x52, 0x69, 0xA4, 0x29, 0x67, 0x1B, 0x66, 0x81,
      0x14, 0x18, 0x1F, 0x06, 0x06, 0x08, 0x42, 0xD1, 0x69, 0x80, 0xF7, 0xA7,
      0x72, 0x48, 0x0F, 0x3E, 0xE4, 0x6D, 0x58, 0xC0, 0x16, 0x82, 0xAA, 0x53,
      0x8D, 0xF1, 0x81, 0x3C, 0xF9, 0x5B, 0x67, 0xD9, 0x15, 0x6D, 0x94, 0x5E,
      0x2F, 0xD8, 0x4A, 0x70, 0x03, 0xCB, 0xCF, 0x26, 0x5C, 0xDC, 0x12, 0xAA,
      0x71, 0x48, 0xB7, 0xAA, 0x55, 0xC6, 0x6F, 0x44, 0xA8, 0x2E, 0x4B, 0xCF,
      0x7D, 0x8C, 0x7B, 0x1E, 0xCF, 0x22, 0xE1, 0xBD, 0x70, 0x33, 0xC0, 0x58,
      0x8F, 0xEB, 0x6A, 0x1D, 0x55, 0x3C, 0xF8, 0xBC, 0x47, 0x7D, 0x94, 0xFF,
      0x87, 0x53, 0x23, 0x94, 0x37, 0x62, 0xAB, 0x1B},
     true,
     {0x90, 0x84, 0x5F, 0xF6, 0xC8, 0x26, 0xA3, 0x9A, 0xCD, 0x0C, 0x5E,
      0x30, 0xC1, 0xAE, 0x53, 0x1B, 0x55, 0x28, 0xF8, 0x8F, 0x12, 0xDC,
      0xA9, 0xF0, 0x35, 0x48, 0x14, 0x4D, 0x6D, 0xC6, 0x56, 0x26},
     {0xA2, 0x71, 0x8B, 0x4E, 0xB9, 0x6D, 0x59, 0x16, 0x90, 0xF6, 0x2F,
      0xDF, 0xCC, 0x26, 0x4B, 0xC4, 0x57, 0xC3, 0xA1, 0xB7, 0x55, 0xF4,
      0xCF, 0x64, 0xB3, 0x59, 0xBC, 0x94, 0x5C, 0x25, 0x4C, 0xE9}},

    {params_ml_kem768_test_mode,
     {0x62, 0xA2, 0xDA, 0x94, 0xF1, 0x09, 0xC0, 0xDE, 0xF5, 0x6D, 0xFB,
      0x27, 0x5B, 0x1A, 0x0E, 0xEA, 0xBF, 0x82, 0xAF, 0x8C, 0x6C, 0xDF,
      0xFA, 0x94, 0x08, 0x5A, 0xA9, 0x30, 0x15, 0xBC, 0x18, 0x21},
     {0x4A, 0x6C, 0x1F, 0x1A, 0x81, 0x6F, 0xB6, 0x64, 0x71, 0xE6, 0xC1, 0x8C,
      0xE6, 0x12, 0x6C, 0xD0, 0x2B, 0xA7, 0x6C, 0x73, 0x0E, 0x72, 0x15, 0x72,
      0xDC, 0x73, 0x05, 0x66, 0x49, 0x6B, 0x3A, 0xC2, 0x8F, 0x92, 0xA1, 0x5F,
      0x4C, 0x53, 0x4C, 0x48, 0x57, 0x8C, 0x18, 0xB4, 0x43, 0xF2, 0x46, 0x81,
      0xBA, 0x2B, 0x3C, 0x3B, 0x39, 0x06, 0x69, 0xA2, 0x88, 0xEF, 0x76, 0x76,
      0xC9, 0xA6, 0xB0, 0x97, 0x03, 0x40, 0xF5, 0x67, 0xC9, 0x04, 0x07, 0x61,
      0x7B, 0x28, 0xA9, 0xA0, 0x32, 0x74, 0xD8, 0x28, 0x80, 0x06, 0x28, 0x8B,
      0x0A, 0x5B, 0x68, 0x6F, 0xFC, 0xBA, 0xCB, 0x27, 0x35, 0x36, 0x94, 0x1C,
      0xD0, 0x69, 0x27, 0xBC, 0xE1, 0x09, 0xBF, 0x23, 0x30, 0xCF, 0x65, 0xA3,
      0xCD, 0x0A, 0x27, 0x06, 0xEB, 0x33, 0x15, 0x0A, 0x88, 0x50, 0x44, 0x84,
      0xFE, 0x18, 0x2B, 0xC3, 0xC7, 0x8C, 0xC5, 0xA7, 0x71, 0x18, 0x45, 0xC3,
      0xA2, 0x77, 0xC7, 0x63, 0xDC, 0xBD, 0xF1, 0x6B, 0x19, 0xF1, 0x00, 0x17,
      0x27, 0xD5, 0x67, 0x76, 0x87, 0xAB, 0xCF, 0x4B, 0x32, 0xCD, 0x16, 0x13,
      0x50, 0xEB, 0xA4, 0x20, 0x6B, 0x2E, 0x1A, 0x65, 0x84, 0x85, 0x28, 0xB7,
      0xE1, 0xA9, 0x0E, 0x04, 0x16, 0x5B, 0xEE, 0x05, 0x4F, 0x43, 0x79, 0x3C,
      0x23, 0xB0, 0x66, 0x0F, 0xE7, 0xAF, 0x3E, 0x59, 0x64, 0xDC, 0x7C, 0x27,
      0x54, 0x58, 0x04, 0x0A, 0xFC, 0x05, 0xEE, 0xC7, 0x1A, 0x57, 0xE6, 0x7C,
      0x2B, 0xB7, 0x94, 0xCE, 0x26, 0x80, 0xE7, 0xC4, 0x17, 0x0F, 0xDB, 0x67,
      0xF0, 0xF3, 0x6D, 0xBA, 0x80, 0x9F, 0x72, 0x4B, 0x80, 0x5B, 0xE6, 0x6E,
      0xEC, 0x38, 0x18, 0x1F, 0xF1, 0xB0, 0xE9, 0xF9, 0x4F, 0x79, 0x03, 0x66,
      0x1A, 0xBB, 0x17, 0xFD, 0x9B, 0xC0, 0xB0, 0xB1, 0x9B, 0xCE, 0x25, 0x60,
      0xE2, 0xC2, 0xC7, 0x64, 0xF4, 0x86, 0x3B, 0xCC, 0x40, 0xB5, 0xFC, 0x10,
      0xB6, 0x51, 0x34, 0x83, 0xA8, 0x27, 0xC1, 0x66, 0x8C, 0xCA, 0x8A, 0xA0,
      0xB6, 0x80, 0x39, 0xC8, 0xAC, 0xBE, 0x81, 0xB6, 0x39, 0xE5, 0x5A, 0xAA,
      0xCF, 0x2C, 0x76, 0x1D, 0x78, 0xA7, 0x7C, 0x26, 0x51, 0x57, 0x34, 0x71,
      0x43, 0x4A, 0x14, 0x94, 0x99, 0x43, 0xD9, 0x45, 0xBC, 0x0E, 0xD2, 0x18,
      0x7D, 0xB4, 0x42, 0xE4, 0x1B, 0x90, 0x1D, 0xBA, 0x0B, 0x14, 0x58, 0x07,
      0x10, 0x87, 0x0B, 0x81, 0xB7, 0x8D, 0x21, 0x99, 0x6A, 0xFD, 0xDC, 0x6A,
      0xCD, 0x14, 0x73, 0x67, 0x34, 0x14, 0xE5, 0xC7, 0xC3, 0x5E, 0xE6, 0x0E,
      0xDE, 0x35, 0x79, 0x22, 0x3A, 0x7A, 0x2A, 0x43, 0x66, 0x67, 0xCC, 0x51,
      0xDD, 0x36, 0x6D, 0xEA, 0xF6, 0x15, 0xCA, 0xA4, 0xCA, 0xA0, 0x73, 0xB6,
      0x07, 0x18, 0xB1, 0xE5, 0xB6, 0x2A, 0xBF, 0x50, 0xBE, 0xBF, 0x48, 0x44,
      0xEA, 0xDC, 0x2A, 0xDA, 0x98, 0xCE, 0xAA, 0x8C, 0xAE, 0xBF, 0x18, 0x7C,
      0x2F, 0x26, 0x88, 0x26, 0x73, 0x30, 0x7A, 0x13, 0xCF, 0x83, 0x90, 0x46,
      0x07, 0xA8, 0x86, 0x5A, 0x65, 0x59, 0x28, 0x66, 0x49, 0x95, 0x05, 0x05,
      0x1F, 0x10, 0xA4, 0xDE, 0x52, 0x89, 0x3E, 0x07, 0xBA, 0xBE, 0xEB, 0xAA,
      0x42, 0x72, 0xB1, 0x4E, 0xA9, 0x74, 0xB5, 0x11, 0x41, 0x01, 0x38, 0x5A,
      0xF1, 0xA0, 0x11, 0xA4, 0x88, 0xBA, 0x25, 0x17, 0x9A, 0x11, 0x8B, 0x31,
      0x2D, 0x8A, 0xC8, 0x6F, 0xB5, 0x85, 0x92, 0xD3, 0x28, 0x94, 0x1A, 0x3A,
      0x0D, 0xEA, 0x61, 0x02, 0xC8, 0x6F, 0x1E, 0x23, 0x7D, 0xDB, 0x32, 0x16,
      0x15, 0x29, 0x7A, 0x2B, 0xB6, 0x60, 0xD5, 0xE0, 0x70, 0x5A, 0x66, 0x72,
      0xFD, 0x08, 0xA5, 0xCF, 0x21, 0x2A, 0xC9, 0xB5, 0xB7, 0xA1, 0x30, 0x6C,
      0x87, 0x95, 0x2B, 0x1C, 0x11, 0xB7, 0xD4, 0xE7, 0xC5, 0xB5, 0xFC, 0xB7,
      0x03, 0x07, 0x25, 0xC2, 0xA1, 0x4A, 0xED, 0x64, 0x1D, 0x78, 0x45, 0x3E,
      0x9F, 0xE5, 0xC6, 0xE6, 0x28, 0x0D, 0x2C, 0xA5, 0x78, 0x32, 0xFB, 0xB0,
      0x91, 0xC4, 0x41, 0x83, 0x03, 0xCB, 0x4A, 0xA3, 0x2D, 0xCC, 0x3C, 0x21,
      0x1D, 0xFC, 0x8A, 0x9B, 0xB4, 0x29, 0xE7, 0x50, 0xB2, 0x25, 0x11, 0x49,
      0x72, 0x7A, 0xCB, 0xB1, 0x12, 0x60, 0x7E, 0xAB, 0x9B, 0x77, 0x44, 0x17,
      0x3D, 0x79, 0x5B, 0x06, 0xD1, 0x0E, 0xF8, 0xB1, 0xAC, 0x1A, 0xE5, 0x4D,
      0xA3, 0x0A, 0x05, 0x18, 0x8B, 0x01, 0x64, 0xB6, 0xB6, 0xE9, 0x22, 0x3E,
      0x93, 0xD3, 0x77, 0x25, 0x91, 0x63, 0xD6, 0x25, 0x30, 0xCE, 0x9A, 0x05,
      0x84, 0xF8, 0x2E, 0x59, 0x53, 0x4A, 0xC3, 0xE0, 0x61, 0xB1, 0xB4, 0x8B,
      0xAA, 0xAA, 0x9D, 0xE1, 0xF8, 0xBE, 0xC1, 0x50, 0xA6, 0x3E, 0xB5, 0x79,
      0xDC, 0x52, 0x8E, 0xDF, 0x97, 0x7E, 0x7E, 0x42, 0x17, 0x43, 0xF6, 0xA4,
      0xC5, 0xDB, 0x61, 0xDD, 0x31, 0x33, 0x4D, 0x3C, 0x63, 0xED, 0x77, 0x55,
      0x52, 0x90, 0x64, 0xD6, 0x7A, 0x4B, 0xA3, 0x78, 0x0B, 0xC7, 0xC8, 0x77,
      0x31, 0xB8, 0x65, 0x09, 0x73, 0x8C, 0x1F, 0x72, 0x15, 0x27, 0x5A, 0xCF,
      0xC6, 0xE3, 0xB6, 0x51, 0xFA, 0x91, 0x53, 0x9B, 0x2B, 0x10, 0x11, 0x14,
      0x52, 0x57, 0x06, 0x39, 0xE9, 0x66, 0x76, 0x6B, 0x64, 0x47, 0x7C, 0x3F,
      0x1E, 0x25, 0x53, 0x90, 0x84, 0x0F, 0xB9, 0x63, 0x55, 0x00, 0xF2, 0xB5,
      0x13, 0x28, 0x1E, 0x9D, 0x8B, 0x5B, 0xCC, 0xEB, 0x36, 0x0F, 0x05, 0x91,
      0x35, 0x91, 0x89, 0x9C, 0x1C, 0xBF, 0x95, 0xD2, 0x3C, 0xB8, 0xD7, 0x14,
      0xA0, 0x07, 0x99, 0x0B, 0xB3, 0x44, 0xC3, 0xA6, 0x24, 0x46, 0xA8, 0x09,
      0x45, 0xD0, 0xCA, 0x84, 0xD3, 0x72, 0x7C, 0xC8, 0xC3, 0x9E, 0x05, 0x53,
      0xA1, 0x3B, 0x70, 0x50, 0xC1, 0x2A, 0x1F, 0xB8, 0xBE, 0x65, 0x67, 0x93,
      0x0E, 0x0B, 0x6A, 0xBF, 0x6C, 0x48, 0x4F, 0xD3, 0x40, 0xCE, 0x91, 0x65,
      0x96, 0xB3, 0x6B, 0xDC, 0x40, 0x85, 0x59, 0x5C, 0xB2, 0xCF, 0x08, 0x84,
      0x67, 0xB0, 0xBA, 0x7F, 0x91, 0x5A, 0x4E, 0x95, 0x81, 0xE7, 0x26, 0x9C,
      0x9A, 0x23, 0x4F, 0x1A, 0x00, 0x4A, 0x8B, 0xD0, 0xA2, 0xD7, 0x65, 0x40,
      0x18, 0xE5, 0x5D, 0x68, 0x61, 0xB4, 0x06, 0x7A, 0x29, 0xF6, 0x42, 0x5F,
      0xF1, 0xF5, 0xA0, 0x0A, 0x80, 0x4E, 0x3B, 0x34, 0x61, 0xDA, 0xAB, 0x92,
      0x65, 0xD1, 0x5B, 0xD5, 0x30, 0xAC, 0x10, 0x68, 0x6F, 0xE1, 0x10, 0x38,
      0xE7, 0x72, 0xBC, 0x1C, 0xC4, 0x6D, 0xB5, 0xC4, 0x01, 0xFA, 0x76, 0xC1,
      0xEE, 0x70, 0x16, 0x5F, 0xB7, 0xC6, 0x7C, 0x50, 0x25, 0xC2, 0x51, 0x5E,
      0x41, 0x82, 0xB4, 0x6B, 0x92, 0x19, 0x4F, 0xF3, 0x61, 0xE2, 0x52, 0xCC,
      0xC4, 0x3B, 0x06, 0x0F, 0x50, 0xA6, 0x4C, 0x79, 0x75, 0x0B, 0xD1, 0x62,
      0xA5, 0x86, 0xC5, 0xA6, 0x30, 0x71, 0xD6, 0xE9, 0x3D, 0xAB, 0xA8, 0xBD,
      0x52, 0x55, 0x48, 0xF4, 0x44, 0xC6, 0x21, 0xA7, 0x7E, 0x3E, 0xC5, 0x83,
      0x21, 0x87, 0x88, 0xE3, 0xA7, 0x11, 0x3D, 0xE3, 0x8E, 0xB2, 0xEC, 0xCE,
      0xBB, 0x0C, 0xAD, 0x73, 0x7B, 0xAD, 0x9C, 0x21, 0xB3, 0x9D, 0xB2, 0x55,
      0x45, 0x8B, 0x8D, 0x6E, 0xE7, 0x0C, 0x25, 0x0C, 0xAC, 0xC7, 0x4A, 0x3A,
      0x19, 0xC3, 0x42, 0xB0, 0x52, 0xA0, 0xFA, 0x5B, 0x92, 0x39, 0x90, 0x13,
      0xEE, 0xEC, 0x1B, 0xEE, 0xBC, 0x8E, 0x02, 0xF8, 0x0C, 0x13, 0x70, 0x30,
      0xBA, 0x0A, 0x5D, 0xC8, 0xC3, 0x02, 0x8B, 0x55, 0x09, 0x25, 0x36, 0x7B,
      0xC7, 0x32, 0xCE, 0x0E, 0x09, 0x57, 0x0A, 0xB1, 0xAF, 0x01, 0xB7, 0xCA,
      0xCB, 0xB7, 0x8A, 0x89, 0xBC, 0xC8, 0xC6, 0x74, 0x44, 0x9B, 0x93, 0x2B,
      0x98, 0x48, 0x7A, 0xF0, 0x25, 0x79, 0x28, 0xE6, 0x4A, 0xD0, 0x69, 0x67,
      0xB3, 0x8B, 0x30, 0xA0, 0x62, 0x0F, 0xF9, 0xD4, 0x9D, 0x2D, 0x63, 0xC2,
      0xE7, 0x84, 0x7E, 0x5B, 0x26, 0x04, 0x6B, 0x45, 0x7A, 0x7B, 0xB3, 0x7C,
      0xF0, 0x12, 0xA7, 0xF1, 0x8A, 0xAE, 0x0C, 0xAA, 0x41, 0xAE, 0xF7, 0x29,
      0x37, 0xB3, 0x5A, 0xFD, 0xAB, 0x01, 0x74, 0x68, 0x8B, 0x07, 0x24, 0xB2,
      0x5D, 0x67, 0x56, 0x35, 0x36, 0x95, 0x00, 0x81, 0x93, 0x1A, 0x3A, 0xBE,
      0x89, 0xB5, 0x92, 0x97, 0xF7, 0x01, 0xF2, 0x46, 0xA6, 0x46, 0x2B, 0x60,
      0x16, 0xEA, 0x4E, 0xFD, 0x42, 0xA1, 0x5C, 0x98, 0x4E, 0x71, 0x57, 0xB5,
      0xB4, 0xA3, 0x35, 0x06, 0x48, 0x09, 0xBD, 0x77, 0x39, 0xBA, 0xA2, 0x35,
      0x19, 0x09, 0x82, 0x33, 0xC7, 0x45, 0x4E, 0x24, 0x1B, 0x23, 0xC4, 0x64,
      0x4A, 0x2A, 0xA1, 0xEB, 0x16, 0xE4, 0x56, 0xE2, 0x35, 0x67, 0xC4, 0xC3,
      0xC6, 0x66, 0x2A, 0xBF, 0xE7, 0x6F, 0x52, 0xFE, 0x97, 0xF0, 0x7F, 0x12,
      0x98, 0xBD, 0xB7, 0x0F, 0x62, 0xA5, 0x65, 0x0A},
     true,
     {0x66, 0x97, 0x3B, 0xFF, 0xA2, 0x96, 0x98, 0x2E, 0xBC, 0xE5, 0xDF,
      0x41, 0xBF, 0x3C, 0xA2, 0x50, 0xDD, 0xD3, 0x98, 0x8B, 0x75, 0x18,
      0x5D, 0x41, 0xD5, 0x49, 0xAD, 0x7C, 0x67, 0xE4, 0x64, 0xAF},
     {0xE8, 0x7B, 0x61, 0xA6, 0x49, 0x6F, 0xDA, 0x38, 0xF9, 0x48, 0xED,
      0xC5, 0xF9, 0xCA, 0x57, 0x35, 0x57, 0x9D, 0x47, 0xF6, 0x35, 0x5B,
      0x21, 0x47, 0x27, 0xEF, 0x5B, 0xEB, 0x3C, 0x13, 0xCA, 0x32}},

    {params_ml_kem768_test_mode,
     {0xF3, 0x74, 0xD3, 0xC7, 0x17, 0x2C, 0x30, 0x8D, 0x7A, 0xC5, 0xAB,
      0x1F, 0x1C, 0xE5, 0xBB, 0x97, 0x85, 0xB9, 0x8A, 0xFC, 0xBF, 0x4E,
      0x91, 0x20, 0xB4, 0x2E, 0xA8, 0x3B, 0xD3, 0xBB, 0x18, 0x67},
     {0x36, 0xC0, 0x53, 0xF7, 0x48, 0xB3, 0xA0, 0x65, 0x5C, 0x3B, 0xA8, 0x63,
      0x03, 0xE0, 0x99, 0x53, 0x89, 0x71, 0xAB, 0x05, 0xBB, 0x4B, 0x21, 0x67,
      0xD8, 0x4B, 0x4F, 0x8B, 0xE7, 0x9C, 0xCE, 0x51, 0x5F, 0x28, 0xA9, 0x0A,
      0x5A, 0xD5, 0xCB, 0xC9, 0x7B, 0x84, 0xCE, 0x10, 0xA3, 0x9B, 0xB5, 0x9A,
      0xD2, 0xFB, 0x4A, 0xED, 0x7C, 0xAD, 0x20, 0x52, 0x00, 0xA3, 0xC6, 0x88,
      0xAA, 0x32, 0xAE, 0x73, 0xF5, 0x52, 0xA3, 0x3B, 0x7D, 0xF1, 0xBA, 0x19,
      0x7F, 0x16, 0xCF, 0xA8, 0x53, 0xC7, 0xFF, 0x24, 0xC1, 0xDD, 0x58, 0x6A,
      0xDE, 0x74, 0xCB, 0x09, 0x82, 0x44, 0x9A, 0xB9, 0x81, 0x5A, 0x63, 0x67,
      0x2C, 0x32, 0x42, 0x0C, 0x18, 0x12, 0x32, 0x65, 0x54, 0xC7, 0x21, 0x9C,
      0xA7, 0x69, 0x83, 0x24, 0x11, 0x56, 0x15, 0xC4, 0x1D, 0x2E, 0x74, 0xA0,
      0x13, 0x8A, 0x42, 0x35, 0x62, 0x32, 0xD7, 0x21, 0x85, 0x49, 0x68, 0x4C,
      0xD9, 0x80, 0x0F, 0x1A, 0x84, 0x13, 0xF3, 0xD9, 0x52, 0xE7, 0x8C, 0x91,
      0x5A, 0xD5, 0x6F, 0xB0, 0xD1, 0xCC, 0x76, 0xD4, 0x8F, 0x2A, 0xBC, 0xC0,
      0xE6, 0xD7, 0x9C, 0x67, 0x19, 0x8D, 0x53, 0xB3, 0xB6, 0xFA, 0x57, 0xA4,
      0xE9, 0x14, 0x0D, 0x3C, 0x42, 0x86, 0xC9, 0x7C, 0x7C, 0xBE, 0x48, 0xC3,
      0xFC, 0xC8, 0x1C, 0x6C, 0xD1, 0x54, 0x0B, 0x12, 0xA0, 0x77, 0xBC, 0x54,
      0xF6, 0xF4, 0x67, 0x11, 0xC8, 0x44, 0xB4, 0x65, 0x59, 0x19, 0x4B, 0x55,
      0x97, 0xA8, 0x15, 0x82, 0x73, 0x07, 0xE0, 0xE5, 0x8A, 0xD7, 0xC4, 0xCF,
      0xDD, 0xE8, 0x26, 0xFC, 0x35, 0x94, 0x00, 0x2A, 0x88, 0xC8, 0x37, 0x60,
      0xB5, 0x63, 0x31, 0xA6, 0x1C, 0x54, 0x2C, 0x7B, 0x44, 0x87, 0xA8, 0x51,
      0xD7, 0xFC, 0x4D, 0x6A, 0x40, 0x72, 0xB7, 0xFA, 0x7C, 0x77, 0x64, 0x7F,
      0x8F, 0x25, 0x23, 0x37, 0x25, 0x2B, 0x1C, 0x97, 0x09, 0x51, 0x7A, 0x7E,
      0x21, 0x7B, 0x26, 0xF2, 0x69, 0x84, 0xD8, 0xD0, 0xAF, 0x75, 0xC9, 0xC1,
      0x01, 0x29, 0x77, 0xA3, 0x99, 0x6D, 0x85, 0x5A, 0x86, 0xC4, 0x72, 0x66,
      0x22, 0x4C, 0x21, 0x67, 0xE5, 0x85, 0xC9, 0x68, 0x7C, 0x0F, 0xC5, 0x25,
      0x9E, 0x5B, 0xA1, 0x0F, 0xA2, 0x3D, 0xC4, 0xA2, 0xC1, 0x63, 0xE7, 0x4E,
      0xFD, 0xA0, 0x19, 0x56, 0x96, 0x5A, 0x5D, 0xCA, 0x57, 0xBC, 0xCA, 0x2E,
      0xBD, 0x46, 0x06, 0xB8, 0x10, 0x88, 0x2A, 0x33, 0x7C, 0xB3, 0x33, 0x47,
      0x82, 0x0C, 0x93, 0x7E, 0x82, 0x1E, 0x75, 0xA5, 0xBF, 0xBC, 0xC7, 0x5B,
      0xD1, 0xE3, 0x7D, 0x81, 0x34, 0x2F, 0x82, 0x48, 0x36, 0x69, 0x45, 0xC0,
      0xAA, 0x2B, 0x98, 0xF6, 0xE2, 0x28, 0xA3, 0xAC, 0x6E, 0xF9, 0xCC, 0x28,
      0x20, 0x84, 0x70, 0xA3, 0xFA, 0x56, 0xC2, 0xF5, 0x1A, 0xF8, 0x34, 0x18,
      0xE3, 0xD1, 0x32, 0x12, 0xBC, 0x2C, 0x64, 0x30, 0x4B, 0x4E, 0xC5, 0x50,
      0x01, 0xBC, 0x36, 0x20, 0x63, 0x5C, 0x98, 0x9C, 0xC4, 0xE4, 0xEC, 0xA0,
      0x6A, 0x45, 0x3C, 0x8E, 0x56, 0x89, 0xBE, 0x8C, 0xA0, 0x10, 0xC4, 0x02,
      0x6F, 0xE2, 0x01, 0xEF, 0xAA, 0x27, 0x1C, 0x2A, 0x77, 0x25, 0x63, 0xCE,
      0x71, 0x47, 0x93, 0x2D, 0xFB, 0x72, 0x56, 0xF2, 0x89, 0x75, 0xE2, 0x82,
      0x9D, 0x45, 0x4B, 0xB2, 0xCC, 0x64, 0xAF, 0x86, 0x84, 0xB9, 0xEC, 0xAE,
      0x28, 0x5A, 0x93, 0xEC, 0x22, 0x12, 0xAF, 0x4C, 0xB6, 0x5E, 0x48, 0x44,
      0x61, 0xD2, 0xB2, 0x12, 0x1B, 0x15, 0xB3, 0x7A, 0xC9, 0x44, 0x6A, 0x83,
      0x08, 0x43, 0x7B, 0x88, 0x72, 0x71, 0x56, 0x4B, 0xC8, 0x14, 0x3A, 0x8A,
      0xF1, 0xF9, 0x12, 0xBA, 0xC5, 0x8E, 0x06, 0xAA, 0x8E, 0x03, 0x76, 0x4F,
      0xDD, 0x82, 0x31, 0x80, 0xE3, 0x0D, 0x47, 0x05, 0x6A, 0x6D, 0x21, 0x7A,
      0xBA, 0x65, 0x33, 0xAF, 0x43, 0x99, 0xBC, 0xC7, 0x98, 0xAB, 0x59, 0xAB,
      0x08, 0x80, 0x91, 0x3E, 0x14, 0x89, 0x55, 0x26, 0xAD, 0x2F, 0x06, 0xA6,
      0xD6, 0x53, 0x6A, 0x0D, 0xE7, 0x01, 0xD7, 0xDB, 0x9B, 0x2E, 0xAC, 0x74,
      0x66, 0xFB, 0x85, 0xAC, 0x00, 0x20, 0xD9, 0x61, 0xB3, 0x9B, 0x88, 0xA8,
      0xDF, 0xFC, 0x3F, 0xAC, 0x57, 0x52, 0x41, 0x70, 0xB7, 0x04, 0x90, 0x87,
      0x6F, 0x3A, 0x6F, 0xB1, 0x60, 0x20, 0xEF, 0xBB, 0xA2, 0x6E, 0x17, 0xC0,
      0x12, 0xAC, 0x42, 0x80, 0xC4, 0x6E, 0x7F, 0x64, 0x3C, 0xC3, 0x34, 0xBC,
      0xC3, 0x02, 0xBF, 0x3B, 0xAC, 0x10, 0xA3, 0x89, 0x1F, 0xF0, 0xF2, 0xBB,
      0x81, 0xD2, 0x9D, 0x8E, 0x42, 0x1A, 0x71, 0x27, 0x2C, 0x0F, 0x24, 0x39,
      0xAA, 0xF2, 0x99, 0x16, 0x46, 0x1D, 0xC2, 0x4B, 0x48, 0x99, 0x32, 0x5F,
      0xF0, 0x05, 0xC0, 0xE3, 0x00, 0x85, 0xAA, 0xAC, 0x7B, 0xF2, 0x1B, 0x58,
      0xB1, 0x14, 0xBB, 0x00, 0x25, 0x30, 0x5C, 0x8A, 0x1D, 0xAD, 0xE9, 0xA1,
      0x81, 0x13, 0x29, 0x79, 0x27, 0x13, 0xC5, 0x23, 0x6B, 0x44, 0xB5, 0x2A,
      0xB8, 0x79, 0x18, 0x56, 0xC5, 0x35, 0xDC, 0x8B, 0x0E, 0x4D, 0xD0, 0x75,
      0x36, 0x89, 0xBF, 0xE0, 0x63, 0x6D, 0xB9, 0x96, 0x97, 0xB0, 0x74, 0xBD,
      0x2D, 0x3A, 0x8C, 0x4A, 0xE6, 0x99, 0x29, 0xF0, 0x0B, 0x8B, 0x25, 0x66,
      0x17, 0xFC, 0xAA, 0x96, 0xC9, 0x80, 0xE8, 0xA6, 0x29, 0xBD, 0xA5, 0x8C,
      0xF6, 0xA7, 0x6D, 0xA4, 0x9A, 0x00, 0x40, 0x55, 0xCD, 0x4B, 0xE2, 0x1E,
      0x2C, 0x02, 0xB5, 0x53, 0x3A, 0x7F, 0xBB, 0xC4, 0x52, 0x5A, 0x95, 0x16,
      0x71, 0x75, 0x48, 0x0A, 0x7B, 0x09, 0xBA, 0xC2, 0xBA, 0xA4, 0x7A, 0x03,
      0xD4, 0x87, 0x26, 0x20, 0xC1, 0x91, 0x28, 0x79, 0x32, 0x02, 0x9C, 0x9B,
      0x53, 0x73, 0x7A, 0x69, 0x14, 0x8B, 0x7F, 0x48, 0x3C, 0x60, 0x71, 0x48,
      0xDB, 0x23, 0x59, 0x4D, 0xDB, 0x2F, 0x14, 0x9A, 0xCA, 0x20, 0xF1, 0x3A,
      0xE6, 0x93, 0xC2, 0x26, 0x5A, 0x71, 0xE9, 0x11, 0xA4, 0xD2, 0x2A, 0xCB,
      0x1E, 0xCC, 0xCC, 0x4C, 0x96, 0xB1, 0xA2, 0xC0, 0xCD, 0x45, 0xA5, 0xAC,
      0x25, 0x53, 0x3A, 0xF6, 0x82, 0x99, 0x19, 0xD9, 0x37, 0x70, 0xE4, 0x2B,
      0xEA, 0xD2, 0x30, 0xBF, 0x87, 0x39, 0xB4, 0x8C, 0x99, 0x90, 0xEC, 0xAD,
      0x9E, 0x67, 0x9C, 0x50, 0x11, 0x37, 0xC4, 0x90, 0x19, 0xC2, 0xF8, 0x57,
      0xC9, 0xD0, 0x8F, 0xD5, 0x33, 0xB0, 0x1D, 0x50, 0xAC, 0x6B, 0xF2, 0x39,
      0x86, 0x55, 0xA1, 0xCB, 0xAB, 0xC6, 0x70, 0x29, 0x9A, 0xBF, 0x46, 0xB9,
      0x59, 0x13, 0xA8, 0x45, 0x15, 0x1D, 0xE2, 0x79, 0x66, 0x66, 0xBA, 0x68,
      0x24, 0x23, 0x7A, 0xF6, 0x17, 0x89, 0x85, 0x7C, 0xCA, 0xF0, 0xAB, 0x0F,
      0x4B, 0x43, 0x0F, 0xE9, 0x66, 0xC8, 0x58, 0x18, 0xB1, 0x1E, 0x46, 0x9C,
      0xA2, 0xD7, 0xB7, 0x05, 0x83, 0x38, 0x13, 0xB8, 0xA7, 0x75, 0xC7, 0x71,
      0x38, 0xF9, 0x17, 0x00, 0xC6, 0x4B, 0xCC, 0x63, 0x60, 0x23, 0x1B, 0x15,
      0x40, 0xF7, 0x71, 0x13, 0xA3, 0x3D, 0xA0, 0x81, 0x89, 0xE4, 0x54, 0x46,
      0x54, 0x74, 0x94, 0x7E, 0xAA, 0x7F, 0x67, 0x31, 0x61, 0x3B, 0xF9, 0xAD,
      0x6B, 0xCC, 0x6D, 0x7D, 0xD1, 0xB6, 0x2E, 0x1A, 0x3A, 0xA6, 0x7C, 0x64,
      0x7A, 0x62, 0x71, 0x2B, 0xC9, 0xCF, 0x67, 0xFB, 0xB6, 0x90, 0x20, 0x5E,
      0x6E, 0x69, 0x97, 0xFD, 0xA7, 0x71, 0x9E, 0xB9, 0x99, 0x4D, 0x19, 0x95,
      0x05, 0x34, 0x2D, 0x42, 0xFB, 0x3C, 0x6A, 0xB1, 0x93, 0x68, 0x25, 0x53,
      0x49, 0x80, 0x76, 0x81, 0x02, 0x61, 0x9D, 0x79, 0xB9, 0xF1, 0x97, 0xCB,
      0x05, 0xC1, 0x6A, 0x99, 0x22, 0xB4, 0xAD, 0x75, 0x2C, 0x49, 0x26, 0x53,
      0x9A, 0xE1, 0x84, 0x75, 0x61, 0x14, 0x60, 0x1A, 0x6E, 0xD9, 0x04, 0x43,
      0xD2, 0x86, 0x9D, 0x0F, 0xDB, 0xAC, 0x6F, 0x44, 0x7D, 0x91, 0xF3, 0x7F,
      0xD4, 0xE6, 0x64, 0x9A, 0x90, 0xB7, 0x0C, 0x73, 0x18, 0x2B, 0x61, 0x62,
      0x37, 0x56, 0x10, 0xE9, 0x16, 0x6A, 0xC7, 0x66, 0x56, 0xA9, 0xB9, 0x24,
      0x2D, 0xA0, 0x2C, 0xEA, 0xEB, 0x97, 0x0A, 0x18, 0x03, 0xBD, 0x57, 0x0F,
      0x48, 0xB7, 0xB7, 0x55, 0x79, 0x28, 0xCC, 0x4A, 0xC5, 0xD4, 0x94, 0xBC,
      0xC9, 0x38, 0x60, 0x78, 0x18, 0x36, 0x75, 0xD7, 0x60, 0x32, 0x2A, 0x91,
      0xA1, 0xFA, 0xB7, 0x70, 0x19, 0x8A, 0x43, 0x7C, 0x05, 0x2E, 0xF3, 0x66,
      0xF6, 0xC9, 0x05, 0x06, 0x20, 0x10, 0xAC, 0x09, 0xAA, 0xD9, 0x59, 0x4F,
      0x58, 0x10, 0x36, 0xA7, 0xE1, 0x1B, 0x0D, 0x6C, 0x3B, 0x40, 0xBB, 0xBB,
      0xC3, 0x40, 0xCB, 0xF6, 0xD1, 0x30, 0xBF, 0xB4, 0xE7, 0xCE, 0x46, 0x96,
      0xBD, 0xB0, 0x1A, 0xBE, 0x04, 0x36, 0xAC, 0x41, 0xB2, 0x79, 0xFD, 0x57,
      0x6F, 0xE8, 0x6B, 0xE9, 0x4D, 0x21, 0x3F, 0x70},
     true,
     {0x8E, 0xF2, 0x3A, 0x2B, 0x96, 0x85, 0xF3, 0x8D, 0xD0, 0xD3, 0x23,
      0x4A, 0xA4, 0xC6, 0x95, 0x90, 0x58, 0x49, 0xD5, 0x0E, 0x34, 0x0A,
      0xBD, 0x68, 0xB8, 0x34, 0x49, 0x20, 0x2D, 0x3B, 0x05, 0xBE},
     {0x24, 0x18, 0xBB, 0x42, 0xB8, 0x9B, 0xA8, 0x75, 0x66, 0x45, 0x83,
      0xED, 0xF2, 0x41, 0x32, 0x7F, 0x37, 0x98, 0x37, 0x9B, 0xD1, 0x4B,
      0x64, 0x35, 0x10, 0x44, 0xF6, 0xC9, 0x6B, 0x3D, 0x2C, 0x27}},

    {params_ml_kem768_test_mode,
     {0xDD, 0x25, 0x2F, 0x72, 0x8F, 0xC9, 0x55, 0x3C, 0xFE, 0xE9, 0x09,
      0x24, 0x56, 0x5E, 0x98, 0x4C, 0x8E, 0x14, 0x62, 0xCD, 0xE5, 0x8A,
      0xD8, 0xC4, 0xED, 0x8D, 0xFC, 0xE9, 0x8A, 0x7F, 0x39, 0xB9},
     {0x75, 0xF9, 0x1D, 0xC8, 0x64, 0xB8, 0x19, 0xE7, 0x1C, 0xE8, 0xCA, 0x50,
      0xA7, 0xBB, 0x41, 0xAE, 0x94, 0x81, 0x8B, 0xAB, 0x31, 0xB7, 0xF8, 0x88,
      0xAC, 0xE4, 0x40, 0x71, 0xD2, 0x79, 0x53, 0x61, 0xCA, 0x2B, 0x26, 0x66,
      0x70, 0x47, 0x21, 0xB0, 0x25, 0x58, 0x21, 0x2A, 0xB4, 0x1A, 0x30, 0x0B,
      0x6D, 0x80, 0xB3, 0x32, 0xA5, 0x04, 0x48, 0xE3, 0x67, 0x86, 0xD1, 0x01,
      0xA6, 0x8A, 0x94, 0xD4, 0x23, 0x25, 0xF7, 0x18, 0xCC, 0xC4, 0xEB, 0x35,
      0x21, 0x42, 0x4C, 0x35, 0xE0, 0x2A, 0x0A, 0x1A, 0x7A, 0x35, 0x14, 0x69,
      0x6C, 0x54, 0x7E, 0x19, 0x18, 0x98, 0x2B, 0x10, 0x66, 0xA3, 0xCB, 0x63,
      0x3E, 0x2B, 0x1D, 0x23, 0x28, 0x0C, 0xDB, 0x33, 0x3A, 0xA7, 0xB3, 0xC4,
      0xBE, 0x7C, 0x18, 0xAE, 0xC4, 0x46, 0x78, 0xD6, 0x49, 0xC8, 0x77, 0x4A,
      0x4C, 0x55, 0x2E, 0x45, 0x94, 0x88, 0x01, 0x68, 0x1F, 0x88, 0x31, 0x65,
      0xEB, 0x2C, 0x5E, 0x9D, 0x43, 0xCF, 0x38, 0x62, 0x8E, 0x8C, 0x93, 0x4C,
      0x5F, 0x30, 0x81, 0x9D, 0x42, 0x87, 0xAE, 0x4A, 0x51, 0xA7, 0x97, 0xBD,
      0xB0, 0x76, 0x95, 0x34, 0x16, 0x9E, 0x29, 0x3C, 0x3A, 0x76, 0xA7, 0x63,
      0x61, 0xD7, 0x9D, 0xED, 0xA8, 0x06, 0x8F, 0xAC, 0x18, 0x7E, 0xF6, 0xB6,
      0x47, 0xC1, 0x48, 0xE9, 0x19, 0x14, 0x1F, 0x02, 0x56, 0x4B, 0x0A, 0xC4,
      0x99, 0x6B, 0x27, 0x79, 0x49, 0x15, 0xA2, 0xD1, 0xC6, 0x69, 0xDA, 0x50,
      0xDD, 0x9B, 0x6E, 0x9B, 0xB5, 0x76, 0x12, 0xCB, 0xC8, 0x37, 0x65, 0xAA,
      0x96, 0x20, 0x68, 0xB8, 0x47, 0x7D, 0x70, 0x38, 0x48, 0xDE, 0xA4, 0x56,
      0x59, 0x2A, 0x89, 0xB3, 0xCC, 0x32, 0x1C, 0x3B, 0x86, 0xC5, 0x5A, 0xB9,
      0x80, 0xBA, 0xCD, 0x2F, 0x26, 0x05, 0x39, 0x5B, 0xCF, 0x9E, 0x96, 0x5B,
      0x95, 0xD5, 0x9D, 0xD6, 0x08, 0xA8, 0xB8, 0xB8, 0x5F, 0x66, 0xC8, 0x82,
      0x2F, 0x5B, 0x2D, 0xED, 0xD3, 0x6D, 0x3F, 0x97, 0xBC, 0x15, 0x03, 0x68,
      0x6F, 0x74, 0x99, 0x10, 0x93, 0xA5, 0x79, 0x9A, 0x14, 0x6B, 0x43, 0x4C,
      0x7C, 0x2A, 0x6C, 0x57, 0x3B, 0x3A, 0xD8, 0xC4, 0x4F, 0xC5, 0x17, 0xC9,
      0x2D, 0x04, 0x2D, 0x62, 0x3A, 0x74, 0x14, 0xD5, 0x6D, 0xC4, 0x54, 0x54,
      0xCD, 0x08, 0x91, 0x9F, 0xDB, 0x8D, 0x79, 0xA7, 0x31, 0x6A, 0x4C, 0x14,
      0xAA, 0x99, 0x61, 0xDA, 0x76, 0xC9, 0xDD, 0xC4, 0xCC, 0x6C, 0xDB, 0x37,
      0xB4, 0x74, 0x83, 0x5E, 0x9A, 0x86, 0x56, 0x32, 0x12, 0xF2, 0xB3, 0x90,
      0x88, 0x75, 0x58, 0x8E, 0x24, 0x84, 0xBE, 0x8C, 0x4A, 0x49, 0xE3, 0xBC,
      0x5E, 0xC6, 0x09, 0xFB, 0x4B, 0xCD, 0x52, 0xD6, 0x2A, 0xFB, 0x1B, 0x71,
      0x3B, 0x55, 0x6C, 0xDD, 0xCC, 0x8B, 0xBD, 0x3B, 0x32, 0x2E, 0x17, 0x66,
      0xC6, 0x27, 0xA0, 0x41, 0x0A, 0x3D, 0x5E, 0x0B, 0x6F, 0x3E, 0x49, 0x92,
      0x72, 0xFC, 0xAF, 0xEA, 0xC9, 0x0C, 0xDD, 0x10, 0x0A, 0xB5, 0xD6, 0x86,
      0xF2, 0x20, 0x23, 0xF8, 0x41, 0x3F, 0x64, 0x52, 0x1D, 0x19, 0xE3, 0x8E,
      0xBA, 0xB1, 0xCA, 0x6E, 0x31, 0xCC, 0x10, 0x27, 0x81, 0x84, 0xF5, 0x08,
      0x43, 0xDA, 0x5C, 0xB1, 0xBB, 0x51, 0x64, 0xDB, 0xAB, 0x23, 0x37, 0x40,
      0x65, 0xB1, 0x04, 0x3B, 0x1A, 0x7B, 0x1A, 0x24, 0x1D, 0xDB, 0xD4, 0xBC,
      0x6E, 0xA9, 0x05, 0x7C, 0xC1, 0xA9, 0x95, 0x64, 0x4A, 0xE7, 0x32, 0x31,
      0x34, 0x84, 0x14, 0x41, 0x4C, 0x7E, 0xBC, 0x58, 0xBC, 0xB9, 0xB5, 0x38,
      0xA0, 0x91, 0x43, 0x4D, 0xC9, 0x01, 0x09, 0x04, 0xBA, 0xB0, 0x61, 0x3F,
      0xA6, 0x40, 0xC0, 0xA9, 0x03, 0xBB, 0xE5, 0x7A, 0x10, 0x0B, 0x08, 0xA4,
      0x4A, 0x70, 0x50, 0x74, 0x36, 0xC3, 0x66, 0x26, 0x8E, 0x75, 0x31, 0xC8,
      0x4A, 0x97, 0x7B, 0x9E, 0xA2, 0x26, 0x3E, 0x77, 0x0E, 0x98, 0x30, 0x65,
      0x37, 0xB5, 0x05, 0xA0, 0x12, 0x84, 0xF6, 0x53, 0xC0, 0x66, 0xBA, 0x2F,
      0x9F, 0xBC, 0x6C, 0xB3, 0x74, 0x02, 0x58, 0x89, 0x20, 0x76, 0xC0, 0x1F,
      0xA5, 0xCB, 0xBC, 0x8B, 0xE9, 0x7F, 0xF0, 0x83, 0x0B, 0xC7, 0x46, 0xBA,
      0x27, 0x55, 0x53, 0x05, 0xD1, 0x7A, 0x93, 0x13, 0x71, 0xD0, 0x48, 0x73,
      0xCE, 0x2B, 0x89, 0xF5, 0x69, 0x4B, 0xEF, 0x68, 0x51, 0x5E, 0x75, 0x52,
      0xEE, 0x03, 0xBB, 0x8E, 0x05, 0x17, 0x62, 0x27, 0x68, 0xE6, 0x02, 0x2D,
      0xFD, 0x10, 0x44, 0xCA, 0x86, 0xA7, 0x15, 0x93, 0x05, 0x30, 0xE5, 0x04,
      0xD8, 0x03, 0x15, 0x74, 0x08, 0x75, 0x72, 0x97, 0x49, 0x15, 0x88, 0x5C,
      0xD0, 0x35, 0x1A, 0xF3, 0x61, 0x0A, 0x51, 0x54, 0x55, 0x9F, 0x68, 0x84,
      0x92, 0x57, 0x9B, 0xB7, 0x23, 0xA2, 0x2F, 0x79, 0x1B, 0x08, 0x9C, 0x2C,
      0xE3, 0xA3, 0x83, 0xC0, 0x9C, 0xBE, 0xFD, 0xC1, 0x51, 0xDC, 0x26, 0x6D,
      0xCD, 0x54, 0xC1, 0x96, 0xF2, 0x5F, 0x0E, 0xE3, 0x40, 0x92, 0x10, 0x9C,
      0x6A, 0xA6, 0xCA, 0xA7, 0x63, 0x73, 0xE9, 0x12, 0x5E, 0x14, 0x3B, 0x2D,
      0x0E, 0xA0, 0x87, 0x4E, 0x4A, 0x35, 0xF0, 0x69, 0x61, 0xDD, 0x81, 0x2E,
      0x60, 0x70, 0x0F, 0x59, 0x89, 0x26, 0xFA, 0xA7, 0xC1, 0xF7, 0xCA, 0x57,
      0x9C, 0xE9, 0xAE, 0xF3, 0x55, 0x7F, 0x74, 0xE2, 0x03, 0x90, 0x59, 0x25,
      0x8A, 0x00, 0x1F, 0xC7, 0xF3, 0xB4, 0xA4, 0xEC, 0xAC, 0x63, 0x03, 0x67,
      0x2F, 0x12, 0xB9, 0xCC, 0x36, 0x6E, 0x44, 0x81, 0x24, 0xA2, 0x7C, 0xA6,
      0xFC, 0x43, 0xC6, 0x15, 0x7A, 0x37, 0x85, 0x35, 0x81, 0x7B, 0x8B, 0x0C,
      0x98, 0x18, 0x09, 0xEC, 0xD5, 0x4D, 0x92, 0xF1, 0x35, 0xD0, 0x12, 0x64,
      0x80, 0x80, 0x4E, 0xFF, 0xE4, 0x01, 0x34, 0xE6, 0xC2, 0x19, 0x0B, 0x95,
      0xBC, 0x7C, 0x5E, 0x6A, 0x72, 0xB6, 0x48, 0x7A, 0x6A, 0x45, 0x42, 0x39,
      0xDF, 0xE1, 0x0A, 0xA9, 0x96, 0x14, 0xAF, 0xA6, 0xA8, 0xF0, 0xE1, 0x2F,
      0xA8, 0x60, 0x07, 0xCE, 0x46, 0x4A, 0xED, 0x40, 0x63, 0x02, 0x5B, 0x21,
      0x3D, 0x85, 0x97, 0xD4, 0x12, 0xC9, 0x61, 0x7A, 0xB6, 0x2D, 0x95, 0x92,
      0xD4, 0x32, 0x0D, 0x7F, 0x97, 0xA6, 0xAC, 0x40, 0x92, 0xDC, 0xB3, 0x11,
      0xC2, 0x16, 0x36, 0x35, 0x81, 0x72, 0xC4, 0x3B, 0x56, 0x26, 0x64, 0x23,
      0x36, 0xA9, 0x6C, 0x14, 0x66, 0xBC, 0x30, 0x42, 0x6D, 0xFA, 0xD5, 0xC0,
      0xDE, 0xA5, 0x0F, 0x71, 0x5B, 0x19, 0x82, 0x7B, 0x19, 0x35, 0xC7, 0x9C,
      0xFA, 0x1C, 0x9C, 0xA1, 0xF7, 0x99, 0xF3, 0x95, 0x4E, 0xB1, 0xC0, 0x72,
      0x0D, 0x13, 0x4C, 0xF9, 0xA3, 0x12, 0x22, 0x63, 0xB9, 0x95, 0xE9, 0x26,
      0x8C, 0x3A, 0x1B, 0xE1, 0x8A, 0x71, 0x12, 0xEC, 0xA0, 0xC4, 0x91, 0x4F,
      0x26, 0x59, 0x73, 0x4A, 0x31, 0xA5, 0xD5, 0x70, 0x8F, 0x3A, 0x14, 0x5E,
      0x68, 0x01, 0xC1, 0x87, 0x61, 0xA2, 0x80, 0x46, 0x5C, 0x2D, 0x46, 0x00,
      0xB9, 0xD9, 0x68, 0xB3, 0xCA, 0x61, 0xEF, 0x9C, 0x76, 0x87, 0xFB, 0x15,
      0x97, 0x49, 0x95, 0x97, 0xAB, 0xAD, 0xA9, 0x30, 0xAE, 0xA4, 0x66, 0x6D,
      0xF4, 0x00, 0x29, 0xF6, 0x45, 0xA6, 0x01, 0x98, 0x46, 0x46, 0x14, 0x4D,
      0xB4, 0xCB, 0x68, 0xB3, 0x46, 0x6C, 0xD7, 0xDA, 0x15, 0x4C, 0x02, 0x75,
      0x36, 0x88, 0x6A, 0x39, 0x02, 0x34, 0xCA, 0x79, 0x2F, 0xF6, 0xAA, 0x5C,
      0x7C, 0xA2, 0x58, 0x83, 0x1A, 0xC1, 0xB0, 0xFA, 0xAF, 0xC2, 0x18, 0x90,
      0xBF, 0xE5, 0x22, 0xA7, 0x68, 0x58, 0x37, 0xA5, 0x50, 0xCB, 0x34, 0x3B,
      0x79, 0xB2, 0x95, 0x5C, 0x4A, 0xBF, 0x49, 0x39, 0x79, 0x21, 0x33, 0xAB,
      0x0D, 0x4A, 0x77, 0xC9, 0x09, 0xBE, 0x0F, 0x02, 0x33, 0xAA, 0x80, 0xAC,
      0x6B, 0x6A, 0xAB, 0x0F, 0x81, 0x02, 0xF3, 0x74, 0x98, 0x2C, 0x6B, 0xA8,
      0x85, 0x63, 0xAC, 0xF7, 0x25, 0x47, 0x60, 0x26, 0x3D, 0x7E, 0xC8, 0xC3,
      0x02, 0xF6, 0x56, 0xBC, 0x22, 0x7B, 0x3B, 0x55, 0x8B, 0xBB, 0x44, 0x43,
      0x65, 0xA2, 0xCD, 0x95, 0xB4, 0x21, 0xDA, 0x59, 0x14, 0x41, 0x7C, 0x50,
      0xE8, 0x43, 0x11, 0xC6, 0xA2, 0x21, 0x8A, 0x49, 0x16, 0x98, 0x66, 0x51,
      0x8C, 0x20, 0x16, 0x22, 0xF1, 0x09, 0x70, 0xD7, 0x5D, 0x38, 0x81, 0x57,
      0x87, 0x35, 0x5A, 0x0C, 0x61, 0x26, 0xEB, 0xF1, 0x89, 0xAB, 0x30, 0x32,
      0x1C, 0x15, 0x89, 0x8A, 0xB1, 0x17, 0xF6, 0x18, 0x3C, 0x47, 0x17, 0x4D,
      0xAA, 0xB7, 0xBB, 0x8F, 0x96, 0xC3, 0xF4, 0xF2, 0x39, 0xB6, 0x42, 0x51,
      0x6A, 0x48, 0xF0, 0x15, 0xE8, 0x38, 0xA3, 0xDB, 0xAA, 0x50, 0x0D, 0xE4,
      0x09, 0xC1, 0x3F, 0x28, 0xFC, 0xCE, 0x5F, 0x26, 0x6A, 0x98, 0xAB, 0xCB,
      0x2D, 0x92, 0xE1, 0xBE, 0x99, 0xE4, 0x38, 0xBF},
     true,
     {0xB3, 0xAB, 0x40, 0xD3, 0x58, 0x8F, 0x62, 0x81, 0xC8, 0xE5, 0x70,
      0x6A, 0x03, 0xCF, 0x74, 0x28, 0x90, 0x6D, 0x3F, 0x29, 0x7F, 0xC4,
      0x9C, 0x94, 0x30, 0x9D, 0x8A, 0x2A, 0x32, 0x68, 0x7E, 0x0F},
     {0x2F, 0x32, 0x3D, 0xFA, 0x37, 0xA7, 0x37, 0x80, 0x22, 0x27, 0xED,
      0x21, 0x01, 0x2F, 0xA0, 0xBA, 0x62, 0x4F, 0x53, 0x2F, 0x8A, 0x3D,
      0xD9, 0x79, 0xAE, 0xFC, 0xC5, 0x54, 0xC1, 0xC2, 0xBE, 0x92}},

    {params_ml_kem768_test_mode,
     {0x29, 0x7E, 0xCD, 0x18, 0xE2, 0x88, 0x0A, 0x59, 0x6F, 0x57, 0x2B,
      0x66, 0x45, 0x84, 0x10, 0xA0, 0xD8, 0x27, 0x85, 0x1E, 0xFA, 0x55,
      0xF1, 0xC9, 0xCC, 0x51, 0x3F, 0x79, 0x91, 0xF0, 0xDA, 0x0A},
     {0x19, 0xE6, 0x28, 0xA9, 0x6B, 0x03, 0x3E, 0x43, 0x58, 0xCF, 0xA8, 0xC3,
      0xC4, 0xA6, 0x42, 0xD7, 0x12, 0x7C, 0x16, 0xA5, 0x0B, 0xEE, 0xE5, 0xC5,
      0x7A, 0x25, 0x47, 0x6D, 0x8B, 0xB8, 0x6D, 0x01, 0x9D, 0xAD, 0x84, 0xC6,
      0x13, 0xD9, 0x6A, 0x38, 0x98, 0x02, 0xEE, 0xD2, 0x03, 0x1B, 0x1A, 0x07,
      0xD4, 0x78, 0x61, 0x2D, 0xB4, 0x05, 0xBB, 0xE9, 0xBB, 0x42, 0xA4, 0xBF,
      0x9E, 0x7A, 0xA9, 0x1C, 0xF2, 0x95, 0x0D, 0x00, 0x01, 0x3C, 0x4C, 0xB7,
      0xA9, 0xB0, 0xCF, 0x88, 0xA8, 0x7B, 0x19, 0xE4, 0x19, 0x95, 0xEA, 0x9E,
      0xB0, 0x96, 0x60, 0x74, 0x6B, 0x1A, 0x81, 0xA2, 0xC7, 0x4F, 0x0B, 0xBC,
      0x28, 0x28, 0x78, 0x51, 0xCC, 0xAE, 0x8C, 0xB8, 0x11, 0x8A, 0xBA, 0xA6,
      0x27, 0xF4, 0x42, 0x89, 0xA9, 0xC3, 0x55, 0xDB, 0x86, 0x30, 0x95, 0xB3,
      0x97, 0xD1, 0xC4, 0x49, 0xC2, 0x0A, 0xA6, 0x0B, 0x5F, 0x1E, 0xF9, 0x4F,
      0x90, 0x02, 0xB2, 0x12, 0xC7, 0x33, 0xD8, 0x1C, 0x5A, 0x92, 0x63, 0x1C,
      0xD3, 0xB6, 0x61, 0xB1, 0x10, 0x56, 0xC9, 0x02, 0x6E, 0x84, 0xA3, 0x4E,
      0x19, 0x83, 0x64, 0x11, 0xA7, 0x6A, 0xA5, 0x30, 0x89, 0x07, 0x11, 0x94,
      0xE2, 0xC9, 0x69, 0x6B, 0x31, 0xBE, 0x83, 0x51, 0xCB, 0x73, 0x55, 0x21,
      0xA4, 0x2C, 0xB8, 0x3C, 0x52, 0xC4, 0x21, 0x72, 0x4F, 0x2F, 0xB7, 0x6D,
      0x24, 0x5B, 0xB5, 0x57, 0x13, 0x58, 0x31, 0x4A, 0x29, 0x67, 0x60, 0x40,
      0x84, 0xF6, 0x4A, 0xD3, 0x09, 0x11, 0x60, 0x0B, 0x33, 0xAC, 0x9B, 0x23,
      0x6D, 0xB6, 0x64, 0x87, 0x05, 0x8F, 0x35, 0x61, 0x1D, 0x08, 0x94, 0x05,
      0x1F, 0x31, 0xC0, 0x32, 0x97, 0x2A, 0x03, 0xA1, 0xCA, 0x58, 0x22, 0x31,
      0x7F, 0x95, 0x3F, 0x91, 0xA7, 0x48, 0x85, 0xE2, 0x17, 0x66, 0x82, 0x4F,
      0x0F, 0xB6, 0x01, 0xC5, 0x93, 0xBC, 0x64, 0x25, 0x3E, 0x7C, 0x36, 0x63,
      0x94, 0x62, 0x72, 0x1C, 0x23, 0x2D, 0x32, 0xA0, 0x76, 0xEE, 0x24, 0x5B,
      0x9A, 0x7B, 0xC9, 0x6C, 0x0A, 0x70, 0xAC, 0xEA, 0x36, 0xF8, 0xF4, 0x1E,
      0x12, 0x42, 0xAB, 0x2D, 0x00, 0x59, 0x3B, 0xDA, 0x4C, 0x55, 0x00, 0x18,
      0xB1, 0x34, 0x5D, 0x08, 0x06, 0xA3, 0xD1, 0xE8, 0xC7, 0xA8, 0xCC, 0x29,
      0x79, 0x01, 0x27, 0x37, 0x07, 0x54, 0xA2, 0x4A, 0x73, 0xE3, 0xA4, 0xB0,
      0x8D, 0xEC, 0x3A, 0x88, 0xC0, 0x4C, 0x20, 0x22, 0x23, 0x5E, 0x55, 0xCD,
      0xCA, 0xF8, 0x09, 0x38, 0xB5, 0x79, 0xB6, 0x31, 0x23, 0xE3, 0x06, 0x60,
      0x3C, 0x78, 0x1F, 0xF3, 0xAC, 0xA1, 0x10, 0x50, 0x76, 0xAA, 0x2B, 0xBA,
      0x1D, 0x96, 0x6B, 0xC2, 0x8B, 0xB3, 0xF3, 0xA7, 0x34, 0x0D, 0x21, 0x0D,
      0x7E, 0x7A, 0x17, 0x35, 0xB0, 0x1E, 0x74, 0x9C, 0x0A, 0xE9, 0x90, 0x77,
      0x5E, 0x23, 0x06, 0xAE, 0xB8, 0x02, 0x29, 0xB9, 0x18, 0x54, 0x26, 0x08,
      0xB7, 0xE2, 0x81, 0xB6, 0xBB, 0x69, 0x6E, 0x1C, 0x82, 0x34, 0x2B, 0xA1,
      0x96, 0xD1, 0x5E, 0xBE, 0xF2, 0x0D, 0x54, 0x1C, 0xC5, 0x7C, 0xA9, 0x28,
      0xA9, 0xDB, 0x41, 0xE9, 0x72, 0xCC, 0x1C, 0xD3, 0x33, 0xBC, 0x7A, 0x64,
      0x95, 0x22, 0x47, 0xFC, 0xBC, 0x57, 0x88, 0x60, 0x19, 0x1F, 0xA6, 0x79,
      0x71, 0xA2, 0x9E, 0x40, 0xC0, 0x6B, 0x81, 0xCB, 0x5A, 0x76, 0x5C, 0x7B,
      0x26, 0x52, 0x2C, 0x19, 0xA6, 0x3C, 0x40, 0xEC, 0x31, 0xEB, 0xEC, 0xCF,
      0x7D, 0xF4, 0x4C, 0xDB, 0x8B, 0x4E, 0x7B, 0xC8, 0x05, 0xD4, 0x2A, 0x28,
      0xAC, 0xAC, 0x3E, 0xB2, 0x68, 0x4A, 0xBD, 0xF4, 0x6D, 0xB7, 0xE4, 0xC5,
      0xB0, 0x1A, 0x77, 0x0B, 0x4C, 0x0D, 0x79, 0xD7, 0x19, 0x0B, 0x97, 0x46,
      0x1B, 0xB7, 0x5C, 0xD4, 0xA7, 0x85, 0x74, 0x67, 0x7C, 0xE2, 0x6B, 0xC3,
      0x0C, 0xC2, 0x52, 0x15, 0xB1, 0xC2, 0x0C, 0x29, 0x0A, 0x15, 0x75, 0x79,
      0x0C, 0x86, 0xAE, 0xBD, 0xEC, 0x79, 0x93, 0xCC, 0x94, 0xCA, 0xE2, 0x3D,
      0x5F, 0xA2, 0x87, 0x1A, 0x18, 0x4E, 0x72, 0x18, 0x86, 0xBC, 0x92, 0x0C,
      0xAC, 0x11, 0x60, 0x7F, 0xE9, 0x68, 0x78, 0x97, 0x1F, 0x16, 0x1B, 0x78,
      0x8E, 0x56, 0x53, 0x69, 0x8A, 0x9F, 0x89, 0x81, 0x33, 0x9E, 0x58, 0x19,
      0x69, 0xC1, 0xB6, 0x1A, 0x60, 0x73, 0xC0, 0x2C, 0xC3, 0x82, 0x2A, 0x2A,
      0x2D, 0x0A, 0x6F, 0x5B, 0xAB, 0xCA, 0x0D, 0x57, 0x17, 0x5B, 0x86, 0x94,
      0x2C, 0xFA, 0xC2, 0xB0, 0xFC, 0x9F, 0x16, 0x84, 0x32, 0x4A, 0x41, 0x39,
      0xCB, 0x13, 0x42, 0xD7, 0xA6, 0xB2, 0x66, 0x43, 0xA9, 0x23, 0xFA, 0x14,
      0x98, 0x81, 0x73, 0xE5, 0x8A, 0x82, 0xB6, 0x75, 0x32, 0xC6, 0xAB, 0x89,
      0xC0, 0xD5, 0xAA, 0x1C, 0xB4, 0x7E, 0x44, 0x70, 0x3F, 0xD7, 0xAA, 0x68,
      0x16, 0x50, 0x10, 0x23, 0x22, 0x9B, 0x1E, 0x76, 0x83, 0x3D, 0x79, 0x48,
      0xF0, 0x74, 0x3B, 0xE3, 0x73, 0xBC, 0xD3, 0x61, 0x13, 0x08, 0x99, 0x01,
      0x07, 0x27, 0xCD, 0xAC, 0xD4, 0x34, 0x7B, 0x3A, 0x66, 0x45, 0x54, 0x1B,
      0xC7, 0x17, 0x33, 0x61, 0x86, 0x09, 0x53, 0xE6, 0x1F, 0x58, 0x24, 0xBA,
      0x2E, 0xCB, 0x94, 0x10, 0x13, 0x83, 0xDA, 0x6A, 0x95, 0xCB, 0x45, 0x91,
      0xC1, 0xBA, 0xAA, 0xB1, 0xB2, 0x21, 0xF4, 0x3B, 0x91, 0xD7, 0x98, 0x5A,
      0x17, 0x3A, 0x0F, 0xDA, 0x75, 0x6F, 0xE9, 0xB3, 0x8E, 0xD1, 0xEA, 0x4F,
      0x2A, 0x42, 0x8E, 0x34, 0xF3, 0x52, 0xA3, 0x7A, 0x45, 0xEA, 0x83, 0x99,
      0x8B, 0x66, 0xC4, 0xE6, 0x0B, 0x40, 0x65, 0x0A, 0x87, 0xFF, 0x51, 0x85,
      0xDC, 0xB0, 0x05, 0x5B, 0x9C, 0x01, 0x13, 0x78, 0x44, 0x24, 0x62, 0xC2,
      0x85, 0xD1, 0x8A, 0x0D, 0xBA, 0x65, 0x8D, 0xC5, 0xB1, 0x7D, 0x16, 0x20,
      0xCE, 0x56, 0x6C, 0x86, 0x87, 0x9F, 0x38, 0x75, 0x4C, 0xC8, 0xC1, 0xBA,
      0x01, 0x48, 0x27, 0xB4, 0x44, 0xCC, 0xF8, 0xE9, 0x02, 0x9A, 0x77, 0x63,
      0x31, 0x8C, 0xBF, 0xE0, 0x22, 0x0D, 0xBE, 0xD6, 0x0F, 0x67, 0x20, 0xC6,
      0xF9, 0x6B, 0x4E, 0x70, 0xC7, 0x12, 0xB5, 0x80, 0x91, 0xBE, 0x3A, 0x6C,
      0x1C, 0x46, 0x39, 0xD1, 0xAB, 0xAB, 0x09, 0x67, 0x4D, 0x2E, 0x65, 0x3D,
      0xEC, 0x76, 0x8D, 0x72, 0x8B, 0xA5, 0x93, 0xAA, 0x3C, 0x61, 0xB1, 0x7D,
      0xE1, 0x7A, 0x07, 0xB3, 0xE8, 0x55, 0x58, 0xF3, 0x8D, 0xBF, 0x84, 0x8D,
      0x65, 0xAB, 0x3A, 0x1A, 0xC1, 0x83, 0xA6, 0x2A, 0xAD, 0x4C, 0xB1, 0x83,
      0x44, 0xE7, 0x1B, 0xE4, 0xC9, 0x7F, 0xEB, 0x59, 0x00, 0x1F, 0x4A, 0xCA,
      0x46, 0x34, 0x2A, 0xAC, 0x38, 0x9B, 0x22, 0xC4, 0x7C, 0x2A, 0x21, 0x6D,
      0x9A, 0xB5, 0x0B, 0x1A, 0x02, 0x40, 0x5D, 0xA0, 0x14, 0x94, 0xC4, 0x41,
      0x84, 0x36, 0x3C, 0x31, 0xF5, 0xC3, 0x9A, 0xC9, 0x05, 0x07, 0xD9, 0x40,
      0x93, 0xF8, 0x82, 0xE6, 0x89, 0x1E, 0x24, 0x57, 0x1B, 0xF3, 0x99, 0x32,
      0xDC, 0x54, 0x82, 0x55, 0x4C, 0x12, 0xAF, 0x6A, 0x5F, 0x0B, 0x3A, 0xB4,
      0xCB, 0xD2, 0x76, 0xD6, 0xD8, 0x50, 0x89, 0x4C, 0xB4, 0xF4, 0x14, 0x19,
      0x6D, 0x51, 0x17, 0x31, 0xD2, 0x3C, 0x8E, 0x03, 0x92, 0x82, 0xF6, 0xC1,
      0x1D, 0x08, 0x9B, 0xEB, 0xFB, 0x91, 0xF0, 0xA7, 0x66, 0x93, 0x5B, 0x3E,
      0xD6, 0x64, 0x58, 0x84, 0x2B, 0x4C, 0x75, 0x75, 0x85, 0xFA, 0x66, 0x57,
      0x94, 0x75, 0x29, 0x29, 0x49, 0x30, 0xC8, 0x86, 0xA5, 0x06, 0x89, 0xCA,
      0x5D, 0xF1, 0xAA, 0x39, 0xB8, 0x34, 0x6A, 0x14, 0x15, 0x0F, 0x05, 0x1E,
      0x54, 0x62, 0xA9, 0x9E, 0x56, 0xCA, 0x38, 0x3B, 0xA5, 0xAB, 0xD4, 0x55,
      0xF0, 0x0A, 0xA9, 0xBB, 0xBA, 0x66, 0x0F, 0xCC, 0xA2, 0xC5, 0xA8, 0xA7,
      0x42, 0x99, 0x58, 0x1C, 0x27, 0x4B, 0x89, 0xE4, 0x75, 0x65, 0x52, 0x57,
      0x0A, 0x33, 0x51, 0x4C, 0xB4, 0x6C, 0xD7, 0xA3, 0xAE, 0xD3, 0x4A, 0x2D,
      0xA4, 0x2C, 0x49, 0xDE, 0x1A, 0x86, 0x45, 0xD6, 0x4A, 0xF3, 0xD6, 0x44,
      0xBE, 0x58, 0x60, 0x68, 0xA1, 0xB5, 0x80, 0x49, 0x65, 0xBB, 0x7A, 0x61,
      0x34, 0xCC, 0x8D, 0xAC, 0xD8, 0x2F, 0xF2, 0xE9, 0x54, 0x62, 0x5C, 0x92,
      0xB0, 0xD9, 0x76, 0x3E, 0x22, 0xCC, 0xF7, 0x45, 0x0A, 0x8F, 0xA7, 0x0F,
      0x57, 0x93, 0x40, 0x0D, 0x28, 0x32, 0xA0, 0x3A, 0x5A, 0x75, 0xD4, 0x04,
      0xD1, 0xE2, 0x82, 0x2D, 0xA6, 0x07, 0xF6, 0xF9, 0xC4, 0xFB, 0x51, 0x08,
      0x58, 0x97, 0x83, 0x72, 0xC6, 0x21, 0xAA, 0x72, 0x39, 0x9A, 0x34, 0xB9,
      0xE2, 0x61, 0x8F, 0x97, 0xEA, 0xB8, 0x2B, 0x56, 0xD9, 0x3E, 0x51, 0xFD,
      0x73, 0xA9, 0x0A, 0x78, 0xE2, 0xAC, 0x85, 0x82, 0x6B, 0x8E, 0x63, 0x35,
      0x33, 0x0D, 0xEB, 0x8C, 0x64, 0x4A, 0x29, 0xA1},
     true,
     {0x22, 0xBA, 0xE0, 0xE3, 0xC6, 0x22, 0x3C, 0xE2, 0xEC, 0xAA, 0x40,
      0x68, 0xB9, 0x2D, 0x33, 0x05, 0x71, 0x4B, 0xAA, 0x34, 0x39, 0x54,
      0x93, 0xDD, 0x4A, 0xB6, 0x6D, 0x57, 0x7A, 0xA6, 0xC6, 0x20},
     {0xC4, 0x4E, 0xE4, 0xE3, 0xEB, 0x80, 0xC4, 0x6D, 0x6B, 0xF5, 0xCC,
      0x3E, 0x08, 0xCB, 0x93, 0x01, 0x9C, 0x8C, 0x80, 0xDB, 0x0C, 0xBE,
      0x89, 0x70, 0x8E, 0x8A, 0x69, 0x02, 0xDE, 0x87, 0xB6, 0x99}},

};
