# (c) Copyright 2010, 2015. CodeWeavers, Inc.

"""Monitors the filesystem and notifies the registered observers of any change
to their paths of interest."""

import os
import sys
import threading
import time

import cxlog
import cxobservable
import cxutils


# These are the events you may receive
CREATED = 'created'
DELETED = 'deleted'
MODIFIED = 'modified'


#####
#
# Fallback notifier implementation
#
#####

class _Path(cxobservable.Object):
    """This is a helper class that monitors a single path."""

    observable_events = frozenset((CREATED, DELETED, MODIFIED))

    def __init__(self, path):
        cxobservable.Object.__init__(self)

        self._path = path
        self._dentries = set()
        self._digest = cxutils.FileDigest(self._path)

        try:
            self._dentries = set(os.listdir(self._path))
        except OSError:
            pass

    def emit_event_for_observer(self, event, observer, observer_data, *args):
        """Override the cxobservable.Object method."""
        observer(event, *args, observer_data)

    def poll(self):
        """Checks this path object for changes and emits the events if any."""
        if self._digest and not self._digest.modified():
            # path has not changed
            return
        old_digest = self._digest
        old_dentries = self._dentries
        self._digest = cxutils.FileDigest(self._path)
        try:
            self._dentries = set(os.listdir(self._path))
        except OSError:
            self._dentries = set()

        # Try to send the events in a logical order
        if old_digest.existed():
            if old_dentries:
                # Notify observers of the deletion of subfolders and files
                # before notifying them of the deletion of the parent directory
                for dentry in old_dentries.difference(self._dentries):
                    self.emit_event(DELETED, os.path.join(self._path, dentry))
                if not self._dentries:
                    # This is not a folder anymore
                    self.emit_event(DELETED, self._path)
            elif not self._digest.existed():
                self.emit_event(DELETED, self._path)

        if self._digest.existed():
            if self._dentries:
                if not old_dentries:
                    # Notify observers of the directory creation, before
                    # sending the notifications for its content
                    self.emit_event(CREATED, self._path)
                for dentry in self._dentries.difference(old_dentries):
                    self.emit_event(CREATED, os.path.join(self._path, dentry))
                # Don't send MODIFIED events for directories
            elif not old_digest.existed() or old_dentries:
                self.emit_event(CREATED, self._path)
            else:
                self.emit_event(MODIFIED, self._path)


class PollingFSNotifier:
    """The fallback filesystem notifier implementation.

    It works by polling the specified paths at regular intervals.
    """
    interval = 2

    def __init__(self):
        self._paths = {}
        self._lock = threading.Lock()
        self._thread = threading.Thread(target=self.worker)
        self._thread.daemon = True
        self._thread.start()

    def add_observer(self, path, observer, observer_data=None):
        """See cxfsnotifier.add_observer()."""
        with self._lock:
            if path not in self._paths:
                self._paths[path] = _Path(path)
            else:
                self._paths[path].poll()
            return self._paths[path].add_observer(cxobservable.ANY_EVENT, observer, observer_data)

    def remove_observer(self, path, observer_id):
        """See cxfsnotifier.remove_observer()."""
        with self._lock:
            try:
                self._paths[path].remove_observer(cxobservable.ANY_EVENT, observer_id)
            except KeyError:
                # Just assume this observer has been removed already to
                # simplify error handling
                pass

    def all_observers(self):
        """Return a dictionary containing all observers"""
        ret = {}
        for path, obj in self._paths.items():
            ret[path] = []
            for observers in obj.all_observers().values():
                ret[path].extend(observers)

        return ret

    def poll(self, path):
        """See cxfsnotifier.poll()."""
        pathobj = self._paths.get(path, None)
        if pathobj:
            pathobj.poll()

    def worker(self):
        """This method implements the polling loop and runs in the background
        thread.
        """
        while True:
            time.sleep(self.interval)
            paths = list(self._paths.keys())
            for path in paths:
                self.poll(path)


#####
#
# Implementation registration and selection
#
#####

# Candidate notifier implementations
_NOTIFIERS = {}

# The selected notifier implementation
_NOTIFIER = None


def add(cls, priority):
    """Registers a filesystem notifier implementation class.

    The implementation with the highest priority (a positive integer) will be
    selected, providing that the object instantiation succeeds. Otherwise the
    implementation with the next highest priority will be used.

    It is an error to register an implementation after observers have been
    added.
    """
    if _NOTIFIER:
        raise AssertionError('A cxfsnotifier implementation has already been selected')
    _NOTIFIERS[cls] = priority


# Register the fallback implementation
add(PollingFSNotifier, 0)


def get():
    """Returns the filesystem notifier implementation."""
    # pylint: disable=W0603
    global _NOTIFIER
    if not _NOTIFIER:
        for cls in sorted(_NOTIFIERS, key=_NOTIFIERS.__getitem__, reverse=True):
            try:
                _NOTIFIER = cls()
                break
            except Exception as exception: # pylint: disable=W0703
                cxlog.log("%s is not available: %s" % (cls, str(exception)))
    return _NOTIFIER


#####
#
# Wrappers for the notifier implementations
#
#####

def add_observer(path, observer, observer_data=None):
    """Adds an event observer for the specified path.

    path is the path to monitor. This can either be a file or a directory. It
      can also be a nonexistent path, the observer will then be notified when
      and if it gets created. Also note that paths are not canonicalized. This
      means two observers may in fact be watching the same filesystem objects
      but just with two different paths. They will still each receive the
      relevant events, each for its own path. If path points to a directory
      the observer receives events for the creation and deletion of files and
      folders in that directory.

    observer_data is any data the observer would like to be given back
      when it gets called.

    observer is a callable. Its signature must be of the form:
        observer(event, path, observer_data)
      Where event is one of CREATED, DELETED or MODIFIED, path is the relevant
      path and observer_data is the data that was specified in add_observer().
      Observers must not raise exceptions.

    Returns an id that can be used to remove the observer.
    """
    return get().add_observer(path, observer, observer_data)


def remove_observer(path, observer_id):
    """Removes the observer for the specified path and observer id.

    No exception is raised if the observer has already been removed.
    """
    get().remove_observer(path, observer_id)


def poll(path):
    """Synchronously checks whether there are any events to report for the
    specified path.

    Note that because this is done synchronously the observers will be called
    in the current thread.
    """
    get().poll(path)


#####
#
# Some test code
#
#####

def main():
    class FakeNotifier:
        def __init__(self):
            raise AssertionError('FakeNotifier.__init__ should not be called')

    class FakeNotifier2:
        def __init__(self):
            raise AssertionError('FakeNotifier2.__init__ should not be called')

    add(FakeNotifier, 2)
    add(FakeNotifier2, -1)

    def test_observer(event, path, _data):
        print("observer: %s: %s" % (event, path))

    def stop_observer(event, path, _data):
        print("stopper: %s: %s" % (event, path))
        if path == 'stop':
            print("Exiting")
            sys.exit(0)

    add_observer(os.getcwd(), test_observer)
    add_observer("new", test_observer)
    add_observer("stop", stop_observer)
    time.sleep(60)


if __name__ == '__main__':
    main()
