# (c) Copyright 2009-2013. CodeWeavers, Inc.

import errno
import fcntl
import hashlib
import locale
import mmap
import os
import re
import selectors
import stat
import subprocess
import sys
import time
import webbrowser

from urllib.parse import urlparse, unquote

import cxlog
import distversion
import cxobjc

class CXUtils(cxobjc.Proxy):
    pass

#####
#
# CX_ROOT
#
#####

# Points to CrossOver's root directory.
# This should be set at startup and is then considered to be constant for the
# duration of the process.
CX_ROOT = None



#####
#
# File and directory helpers
#
#####

def mkdirs(path, *args, **kwargs):
    """Similar to os.makedirs() but returns False to signal errors and logs
    errors.
    """
    try:
        os.makedirs(path, *args, **kwargs)
    except OSError as ose:
        if ose.errno != errno.EEXIST:
            cxlog.err("unable to create the '%s' directory: %s" % (cxlog.debug_str(path), ose.strerror))
            return False
    return True

PERMISSIONS_OK = 0
PERMISSIONS_NOREAD = 1
PERMISSIONS_NOEXEC = 2

def check_mmap_permissions(filename):
    try:
        f = open(filename, 'rb') # pylint: disable=R1732
    except IOError:
        return PERMISSIONS_NOREAD
    try:
        try:
            mm = mmap.mmap(f.fileno(), 1, mmap.MAP_SHARED, mmap.PROT_READ|mmap.PROT_EXEC)
        except mmap.error:
            return PERMISSIONS_NOEXEC
        mm.close()
    finally:
        f.close()
    return PERMISSIONS_OK

def check_fake_dll(path):
    if os.path.exists(path):
        with open(path, "rb") as sfile:
            sfile.seek(0x40)
            tag = sfile.read(20)
            if tag.startswith((b"Wine builtin DLL", b"Wine placeholder DLL")):
                return True
    return False

class FileDigest:
    def __init__(self, filename):
        self._filename = filename
        self._mtime, self._size, self._inode = self.stat()
        self._hash = self.hash()
        self._time = time.time()

    def stat(self):
        try:
            res = os.stat(self._filename)
            return res.st_mtime, res.st_size, res.st_ino
        except OSError:
            return None, None, None

    def hash(self):
        try:
            try:
                with open(self._filename, 'rb') as f:
                    return hashlib.file_digest(f, 'sha1').hexdigest()
            except AttributeError:
                pass

            sha1 = hashlib.sha1()
            with open(self._filename, 'rb') as f:
                while True:
                    chunk = f.read(8192)
                    if not chunk:
                        break
                    sha1.update(chunk)
            return sha1.hexdigest()
        except OSError:
            return None

    def existed(self):
        return self._mtime is not None

    def modified(self):
        mtime, size, inode = self.stat()
        if mtime != self._mtime or size != self._size or inode != self._inode:
            return True
        if mtime and mtime > self._time - 0.1:
            return self.hash() != self._hash
        return False

#####
#
# Cross-process lock
#
#####

class FLock:

    def __init__(self, name):
        """Creates and takes a lock called name using flock().

        This is used to avoid races between processes.
        """
        self._name = name

        # Create the lock directory and perform some sanity checks
        lockdir = os.path.join(os.environ.get('TMPDIR', '/tmp'),
                               ".wine-%d" % os.geteuid())
        mkdirs(lockdir, int('700', 8))
        st = os.lstat(lockdir)
        if not stat.S_ISDIR(st.st_mode):
            raise ValueError("'%s' is not owned by you" % lockdir)
        if st.st_uid != os.geteuid():
            raise ValueError("'%s' is not owned by you" % lockdir)
        if st.st_mode & int('77', 8):
            raise ValueError("'%s' must not be accessible by other users" % lockdir)

        # Perform some more sanity checks and create the lock file
        self._path = os.path.join(lockdir, name + ".lock")
        try:
            st = os.lstat(self._path)
            if not stat.S_ISREG(st.st_mode):
                raise ValueError("'%s' is not a regular file" % self._path)
        except OSError as ose:
            if ose.errno != errno.ENOENT:
                raise
        self._lock = open(self._path, 'wb') # pylint: disable=R1732

        # Take the lock
        cxlog.log("%d: Grabbing the '%s' lock" % (os.getpid(), self._path))
        try:
            fcntl.flock(self._lock, fcntl.LOCK_EX)
        except Exception as e:
            # The filesystem probably does not support locking
            cxlog.warn("unable to take the '%s' lock: %s" % (name, e))
            self._lock.close()
            self._lock = None
            raise # re-raise the exception

    def _getname(self):
        """Returns the lock name."""
        return self._name

    name = property(_getname)

    def locked(self):
        """Returns True if the lock has not been released yet."""
        return self._lock is not None

    def release(self):
        """Releases the lock.

        Note that deleting a lock carries a very high risk of causing races
        outside the scope of this class and has essentially no benefit. So the
        file created for the lock is not deleted and no support is provided
        for deleting it.
        """
        if self._lock:
            cxlog.log("%d: Releasing the '%s' lock" % (os.getpid(), self._name))
            try:
                try:
                    fcntl.flock(self._lock, fcntl.LOCK_UN)
                except Exception as e:
                    cxlog.warn("unable to release the '%s' lock: %s" % (self._name, e))
            finally:
                self._lock.close()
                self._lock = None


#####
#
# String handling
#
#####

def string_to_unicode(string):
    if isinstance(string, str):
        return string
    if isinstance(string, bytes):
        return string.decode('utf8', 'surrogateescape')
    try:
        m = memoryview(string)
        return m.tobytes().decode('utf-8', 'surrogateescape')
    except (NameError, TypeError):
        raise ValueError("string_to_unicode called with non-string") #pylint: disable=W0707

def string_to_utf8(string):
    if isinstance(string, str):
        return string.encode('utf8', 'surrogateescape')
    if isinstance(string, bytes):
        return string
    try:
        m = memoryview(string)
        return m.tobytes()
    except (NameError, TypeError):
        raise ValueError("string_to_utf8 called with non-string") #pylint: disable=W0707


string_to_str = string_to_unicode


def cmp(v1, v2):
    return (v1 > v2) - (v1 < v2)


def unescape_string(string):
    """Unescapes a string that was placed in double-quotes."""
    string = string.replace('\\\"', '\"')
    string = string.replace("\\\\", "\\")
    return string


def escape_string(string):
    """Escapes the string so it can be placed in double-quotes."""
    string = string.replace("\\", "\\\\")
    string = string.replace('"', '\\"')
    return string


def unmangle(string):
    result = []
    x = 0
    while x < len(string):
        char = string[x]
        x += 1
        if char == '+':
            result.append(' ')
        elif char == '_':
            result.append('/')
        elif char == '~':
            result.append('\\')
        elif char == '^':
            try:
                result.append(chr(int(string[x:x+2], 16)))
            except ValueError:
                # not followed by valid hex digits
                pass
            else:
                x += 2
        else:
            result.append(char)
    return ''.join(result)


_ACCELERATOR_RE = re.compile('&(.)')

def remove_accelerators(string):
    """Remove the ampersands that identify the accelerator keys on Windows.

    While only the first one is meant to be significant, they should all be
    removed, and '&&' should be turned into a regular '&'.
    """
    return _ACCELERATOR_RE.sub('\\1', string)

def md5sum_path(path):
    hashobj = hashlib.md5()
    with open(path, 'rb') as f:
        data = f.read()
        hashobj.update(data)
    return hashobj.hexdigest()

def is_valid_windows_filename(filename):
    """Checks that the given filename is valid."""
    if not filename:
        return False
    for x in filename:
        if x in "\\/*?:|<>\"'":
            return False
    return True

def sanitize_windows_filename(filename):
    """Turns a filename that may contain invalid characters into a valid
    Windows  filename."""
    result = []
    for c in filename:
        if c in '\\:*?"<>|\x7f' or ord(c) < 0x20:
            result.append('_')
        else:
            result.append(c)
    return ''.join(result)

def allow_bottlename_char(character):
    """Returns False for characters that are forbidden in bottle names."""
    # \/*?:|<>" are not valid characters in Windows paths
    # {} and space are not valid in paths stored in Debian and Solaris packages
    prohibited = "\\/*?:|<>\"'{}"
    # Spaces cause trouble in Nautilus (GNOME bug 700320).
    # So allow them but avoid them by default
    if not distversion.IS_MACOSX:
        prohibited = prohibited + " "
    return character not in prohibited

def is_valid_bottlename(bottlename):
    """Checks that the bottle name is valid."""

    if bottlename and bottlename[0] == '.':
        return False
    for x in bottlename:
        if not allow_bottlename_char(x):
            return False
    return True

@cxobjc.method(CXUtils, 'sanitizedBottleName_')
def sanitize_bottlename(name_hint):
    """Builds a valid and nice-looking bottle name based on the given hint."""

    result = []
    prev_is_edu = False # equal, dash, underscore
    for x in name_hint:
        if not allow_bottlename_char(x):
            x = '_'
        if x == '.':
            # Avoid leading dots as they cause the bottle to be hidden
            if result:
                result.append('.')
                prev_is_edu = False
        elif x == ' ' and result and result[-1] in '_ ':
            # Remove the preceding space or underscore
            result[-1] = ' '
            prev_is_edu = False
        elif x == '_':
            # Avoid consecutive and leading underscores
            if not prev_is_edu and result:
                result.append('_')
                prev_is_edu = True
        elif prev_is_edu and (x in ('=', '-')):
            # Avoid consecutive equal and dash characters and avoid underscores
            if result and result[-1] == '_':
                result[-1] = x
        else:
            result.append(x)
            prev_is_edu = False

    # Avoid trailing underscores and spaces
    bottlename = ''.join(result).rstrip("_ ")

    # The bottle name should be valid but check just to be sure
    if not is_valid_bottlename(bottlename):
        cxlog.err("sanitize_bottlename() generated an invalid bottle name: '%s'" % bottlename)
    return bottlename

def path_to_uri(path):
    result = ['file://']
    # FIXME: use filesystem encoding?
    path = string_to_utf8(path)
    for char in path:
        if char >= 0x80 or char in b'%#?*!&=()[],:;@$+ ':
            result.append('%%%02X' % char)
        else:
            result.append(chr(char))
    return ''.join(result)

def uri_to_path(uri):
    scheme, _netloc, path, _params, _query, _fragment = urlparse(uri)
    if scheme == 'file':
        return unquote(path)
    return uri

def html_escape(string):
    """Escape characters that have special meaning in html."""
    replacements = {
        '<': '&lt;',
        '>': '&gt;',
        '&': '&amp;'}
    result = []
    for char in string:
        result.append(replacements.get(char, char))
    return ''.join(result)

_HTML_BR_RE = None
_HTML_P_RE = None
_HTML_LI_RE = None
_HTML_A_RE = None
_HTML_URL_RE = None
_HTML_TAGS_RE = None
_LF_RE = None
def html_to_text(string):
    """Use some heuristics to turn an HTML string into a readable plain
    text string."""
    if string is None:
        return None

    # pylint: disable=W0603
    global _HTML_BR_RE, _HTML_P_RE, _HTML_LI_RE, _HTML_A_RE, _HTML_URL_RE
    global _HTML_TAGS_RE, _LF_RE
    if not _HTML_BR_RE:
        _HTML_BR_RE = re.compile('<br/?>(?:\\s*\n)*', re.IGNORECASE)
        _HTML_P_RE = re.compile(r'''</?(?:div|p|ol|ul)(\s+[a-z]+=(['"])(?:[^\\'"]|\.)*\2)*>(?:\s*\n)*''', re.IGNORECASE)
        _HTML_LI_RE = re.compile(r'''(\s*\n)*\s*<li(\s+[a-z]+=(['"])(?:[^\\'"]|\\.)*\3)*>''', re.IGNORECASE)
        _HTML_A_RE = re.compile(r'''</?a(\s+[a-z]+=(['"])(?:[^\\'"]|\.)*\2)*>''', re.IGNORECASE)
        _HTML_URL_RE = re.compile(r'''\shref=(['"])((?:[^\\'"]|\.)*)\1''', re.IGNORECASE)
        _HTML_TAGS_RE = re.compile(r'''(?:</?[a-z]+(\s+[a-z]+=(['"])(?:[^\\'"]|\\.)*\2)*/?>)+''', re.IGNORECASE)
        _LF_RE = re.compile(r'''\s*\n\s*\n(?:\s*\n)+''')

    # Remove carriage returns and linefeeds
    string = string.replace('\r', '').replace('\n', ' ')

    # Some tags deserve special treatment
    string = _HTML_BR_RE.sub('\n', string)    # End of line tag
    string = _HTML_LI_RE.sub('\n * ', string) # Enum tag
    string = _HTML_P_RE.sub('\n\n', string)   # 'Paragraph' tags

    # Preserve the URL for links so users can copy/paste them to their browser
    def replace_link(match):
        atag = match.group(0)
        url = _HTML_URL_RE.search(atag)
        if url:
            return "[" + url.group(2) + "] "
        return ""
    string = _HTML_A_RE.sub(replace_link, string)

    # Simply remove all other tags
    string = _HTML_TAGS_RE.sub('', string)

    # Replace the most common entities
    string = string.replace('&lt;', '<')
    string = string.replace('&gt;', '>')
    string = string.replace('&amp;', '&')
    string = string.replace('&nbsp;', ' ')

    # Finally remove excess spaces and line feeds
    string = re.sub(' +', ' ', string)
    string = _LF_RE.sub('\n\n', string).strip()
    return string

@cxobjc.method(CXUtils, 'cmdLineToArgv_')
def cmdlinetoargv(string):
    """Convert a windows-style quoted command line to an argument list. This is
    NOT equivalent to CommandLineToArgvW."""
    in_quotes = False
    pos = 0
    result = []
    arg = []
    while pos < len(string):
        char = string[pos]
        if char == ' ' and not in_quotes:
            if arg:
                result.append(''.join(arg))
                del arg[:]
        elif char == '\\' and pos+1 < len(string) and string[pos+1] in '\\"':
            pos += 1
            arg.append(string[pos])
        elif char == '"':
            in_quotes = not in_quotes
        else:
            arg.append(string[pos])
        pos += 1
    if arg:
        result.append(''.join(arg))
    return result

def argvtocmdline(argv):
    """Convert an argument list to a windows-style quoted command line."""
    result = []
    for arg in argv:
        arg = arg.replace("\\", "\\\\").replace('"', '\\"')
        if ' ' in arg:
            arg = '"%s"' % arg
        result.append(arg)
    return ' '.join(result)

@cxobjc.method(CXUtils, 'basename_')
def basename(path):
    """Get the base name of a Windows or Unix path.
    Note that Windows paths can mix forward and backward slashes.
    """
    index = max(path.rfind('/'), path.rfind('\\'))
    if index >= 0:
        return path[index+1:]
    return path

@cxobjc.method(CXUtils, 'dirname_')
def dirname(path):
    """Get the directory name of a Windows or Unix path.
    Note that Windows paths can mix forward and backward slashes.
    """
    index = max(path.rfind('/'), path.rfind('\\'))
    if index >= 0:
        return path[:index].rstrip('/\\')
    return ''


def abspath(path, cwd=None):
    path = os.path.expanduser(uri_to_path(path))
    if not cwd:
        return os.path.realpath(path)

    return os.path.realpath(os.path.join(cwd, path))


_OCTALCHAR_RE = None
def expand_octal_chars(string):
    r"""Expand escaped octal characters of the form '\040' in the specified
    string.
    """
    if string is None:
        return None
    # pylint: disable=W0603
    global _OCTALCHAR_RE
    if not _OCTALCHAR_RE:
        _OCTALCHAR_RE = re.compile('(\\\\[0-7]{3})')
    i = 0
    expanded = []
    for match in _OCTALCHAR_RE.finditer(string):
        expanded.append(string[i:match.start()])
        expanded.append(chr(int(string[match.start()+1:match.end()], 8)))
        i = match.end()
    expanded.append(string[i:])
    return ''.join(expanded)

_UNIXVAR_RE = None
def expand_unix_string(environ, string):
    """Expand references to environment variables of the form '${VARNAME}' in
    the specified string.

    Note that unlike os.path.expandvars(), if a given environment variable is
    not set it expands to an empty string.
    """
    if string is None:
        return None
    # pylint: disable=W0603
    global _UNIXVAR_RE
    if not _UNIXVAR_RE:
        _UNIXVAR_RE = re.compile(r'\${(\w+)}')
    i = 0
    expanded = []
    for match in _UNIXVAR_RE.finditer(string):
        expanded.append(string[i:match.start()])
        if match.group(1) in environ:
            expanded.append(environ[match.group(1)])
        i = match.end()
    expanded.append(string[i:])
    return ''.join(expanded)

def _int_for_version_component(comp):
    """A version component can have a non-numeric suffix (e.g. the "local" in
    "25889local").  That interferes with converting to integer. This function
    collects just the numeric prefix and converts that.
    """
    if comp.isdigit():
        return int(comp)

    n = 0
    for x in comp:
        if x.isdigit():
            n += 1
        else:
            break
    if n:
        return int(comp[0:n])
    return 0

def subtract_version(v1, v2):
    if v1:
        v1 = v1.split('.')
    else:
        v1 = []
    if v2:
        v2 = v2.split('.')
    else:
        v2 = []
    if len(v1) < len(v2):
        v1.extend(["0"] * (len(v2) - len(v1)))
    elif len(v2) < len(v1):
        v2.extend(["0"] * (len(v1) - len(v2)))

    result = []
    sign = 0
    for i in range(len(v1)):
        if sign == 0:
            sign = cmp(_int_for_version_component(v1[i]), _int_for_version_component(v2[i]))
        result.append((_int_for_version_component(v1[i]) - _int_for_version_component(v2[i])) * sign)
    return tuple(result)

def cmp_versions(v1, v2):
    if v1:
        v1 = v1.split('.')
    else:
        v1 = []
    if v2:
        v2 = v2.split('.')
    else:
        v2 = []
    if len(v1) < len(v2):
        v1.extend(["0"] * (len(v2) - len(v1)))
    elif len(v2) < len(v1):
        v2.extend(["0"] * (len(v1) - len(v2)))

    for i in range(len(v1)):
        result = cmp(_int_for_version_component(v1[i]), _int_for_version_component(v2[i]))
        if result != 0:
            return result
    return 0

def split_version(s):
    result = tuple(_int_for_version_component(x) for x in s.split('.'))
    i = len(result)-1
    while result[i] == 0 and i > 0:
        i -= 1
    return result[0:i+1]

_version_as_tuple = None

def version_as_tuple():
    # pylint: disable=W0603
    global _version_as_tuple
    if _version_as_tuple is None:
        _version_as_tuple = split_version(distversion.CX_VERSION)
    return _version_as_tuple

#####
#
# Locale functions
#
#####

def get_user_languages():
    try:
        from Foundation import NSLocale
        languages = []
        for language in NSLocale.preferredLanguages():
            if language == 'pt':
                # On Mac OS X 'pt' is identical to 'pt-BR' instead of
                # 'pt-PT' as is the case on other Unix platforms.
                languages.append('pt-BR')
            elif language == 'zh-Hans':
                # zh-Hans is what RFC 5646 specifies for Simplified Chinese but
                # everywhere else we derive the language id from the zh_CN Unix
                # locale. So convert here.
                languages.append('zh-CN')
            elif language == 'zh-Hant':
                # Same as for zh-Hans but for Traditional Chinese.
                languages.append('zh-TW')
            else:
                languages.append(language)
        return languages
    except ImportError:
        # Probe $LANGUAGE manually because getlocale() would only return
        # the first locale it contains. Normally $LANGUAGE only specifies the
        # language, not a full locale. But that should not be an issue for the
        # remainder of this function.
        locales = os.environ.get('LANGUAGE', '')
        if locales == '':
            locales, _encoding = locale.getlocale()
            if locales is None:
                locales = ''

        languages = []
        for loc in locales.split(':'):
            # The user may have set LANG='french' which we would be unable to deal
            # with. So ensure it gets normalized to something like 'fr_FR'. But we
            # must also make sure something like 'fr' does not get converted to
            # 'fr_FR'.
            normalized = locale.normalize(loc)
            if not normalized.startswith(loc + '_'):
                loc = normalized
            # Remove the encoding and modifier that we are not interested in
            loc = loc.split('.')[0].split('@')[0]

            if loc in ('C', ''):
                # These two are synonymous with English
                languages.append('en')
            elif loc == 'pt_PT':
                # This is synonymous with pt which is what we normally use
                languages.append('pt')
            elif '-' in loc:
                # In theory getlocale() may return a locale where the part
                # before the underscore already follows RFC 1766. So if that's the
                # case pass it through unmolested.
                languages.append(loc.split('_')[0])
            else:
                # This is the most common case. We transform a locale of the
                # form 'fr_FR' into what we hope will be a valid language
                # identifier like 'fr-FR'.
                languages.append(loc.replace('_', '-'))
        return languages

_SINGLETON_RE = re.compile("(-x)?-[a-z]$")
_PREFERRED_LANGUAGES = {}
_LANGUAGE_ASSOCIATIONS = {
    'zh-hans':'zh-cn',
    'zh-hant':'zh-tw',
}
def get_preferred_languages(language=None):
    """Returns a sequence of the user's preferred languages if language is
    None; and a sequence of variants otherwise.

    All languages must be identified by their RFC 1766 / RFC 3066 / RFC 5646 /
    BCP 47 language id.
    In the None case the result corresponds to the user's preferences on
    Mac OS X, while on Unix it is derived from the user's locale.
    When starting from a single language, the variants added allow broader
    matches to be established. For instance 'pt-br' gives ('pt-br', 'pt', '').
    """
    if language in _PREFERRED_LANGUAGES:
        return _PREFERRED_LANGUAGES[language]

    if language:
        languages = [language]
    else:
        languages = get_user_languages()
        if not languages:
            languages = ['en']

    # Add the user's preferred languages
    seen = set()
    preferred = []
    fallbacks = []
    for lang in languages:
        lang = lang.lower()
        if lang in seen:
            continue
        preferred.append(lang)
        seen.add(lang)

        # Compute the fallbacks
        while '-' in lang:
            lang = lang.rsplit('-', 1)[0]
            # Strip trailing singleton subtags such as '-x' as they have no
            # meaning on their own. See 'Extension Subtags' in RFC 5646 and
            # the lookup algorithm in RFC 4647.
            lang = _SINGLETON_RE.sub('', lang)
            if lang not in seen:
                fallbacks.append((lang, len(preferred)))

    # Take extra care to insert the fallbacks where they are useful and yet
    # don't override the user's preferences. Here are a few scenarios:
    #   ('en-us, 'fr')       -> ('en-us, 'en', 'fr')
    #   ('en-us, 'fr', 'en') -> ('en-us, 'fr', 'en')
    #   ('en-us, 'en-ca')    -> ('en-us, 'en-ca', 'en')
    #   ('zh-hant-tw', 'fr', 'zh-tw') -> ('zh-hant-tw', 'zh-hant', 'fr', 'zh-tw', 'zh')
    #   ('en-a-bbb-x-a-ccc-ddd') -> ('en-a-bbb-x-a-ccc-ddd', 'en-a-bbb-x-a-ccc', 'en-a-bbb', 'en')
    # See also the lookup algorithm in RFC 4647
    # Insert the fallbacks starting from the end of the array to not mess up
    # the first indices.
    for fallback in reversed(fallbacks):
        if fallback[0] in seen:
            continue
        preferred.insert(fallback[1], fallback[0])
        seen.add(fallback[0])

        if fallback[0] in _LANGUAGE_ASSOCIATIONS:
            associated_lang = _LANGUAGE_ASSOCIATIONS[fallback[0]]
            preferred.insert(fallback[1] + 1, associated_lang)
            seen.add(associated_lang)

    # Add the ultimate fallback
    preferred.append('')
    _PREFERRED_LANGUAGES[language] = preferred
    return preferred


@cxobjc.method(CXUtils, 'getLanguageValue_forLanguage_')
def get_language_value(dictionary, language=None):
    """Returns the appropriate value for the specified language from a
    dictionary mapping language names to values.

    The given dictionary should either contain a single '' key that is used
    for all languages, or only have locales as its keys for locale-specific
    values. However for compatibility with pre-10.0 releases, dictionaries
    containing both types of keys are supported too.
    Empty values are ignored.

    This function will always return a non-empty value if there is at least one
    in the dictionary.
    """
    languages = get_preferred_languages(language)
    for lang in languages:
        if lang in dictionary and dictionary[lang]:
            return (lang, dictionary[lang])

    if dictionary:
        # The dictionary is not empty, but there is no default. Just pick
        # something, predictably.
        languages = sorted(dictionary)
        for lang in languages:
            if dictionary[lang]:
                return (lang, dictionary[lang])
    return ('', '')


_LOCALE_PATH = None

def setup_textdomain():
    # pylint: disable=W0603
    global _LOCALE_PATH
    if _LOCALE_PATH is None:
        _LOCALE_PATH = os.path.join(CX_ROOT, "share", "locale")
        cxlog.log("locale path %s" % cxlog.debug_str(_LOCALE_PATH))
        if hasattr(locale, 'bindtextdomain'):
            # Use locale to set the C-level textdomain for the benefit of C
            # libraries such as GTK+. But note that bindtextdomain() is missing
            # on some platforms, like Mac OS X.
            locale.bindtextdomain('crossover', _LOCALE_PATH)
        import gettext
        gettext.bindtextdomain("crossover", _LOCALE_PATH)
        gettext.textdomain("crossover")


_GETTEXT_TRANSLATOR = None

def _translator():
    # pylint: disable=W0603
    global _GETTEXT_TRANSLATOR
    if not _GETTEXT_TRANSLATOR:
        setup_textdomain()
        languages = []
        for lang in get_preferred_languages():
            # We must convert the language id to match the naming of the
            # $CX_ROOT/share/locale folders. It's not case sensitive, but the
            # dashes must be replaced by underscores.
            languages.append(lang.replace('-', '_'))
        import gettext
        _GETTEXT_TRANSLATOR = gettext.translation("crossover", _LOCALE_PATH, languages=languages, fallback=True)

    return _GETTEXT_TRANSLATOR

def cxgettext(message):
    return string_to_unicode(_translator().gettext(message))

# So xgettext picks up the strings to translate
_ = cxgettext


#####
#
# Logging functions
#
#####


@cxobjc.method(CXUtils, 'installerLoggingPreset')
def installer_logging_preset():
    return {
        "channels": ["+relay", "+msi", "+setupapi"],
        "label": _("Install (very verbose)"),
    }


@cxobjc.method(CXUtils, 'loggingPresets')
def logging_presets():
    graphics_environment = ["VKD3D_DEBUG=trace", "VKD3D_SHADER_DEBUG=trace"]
    if distversion.IS_MACOSX:
        graphics_environment.append("MVK_CONFIG_LOG_LEVEL=4")

    window_channels = ["+event", "+win", "+x11drv"]
    if distversion.IS_MACOSX:
        window_channels = ["+event", "+win", "+macdrv"]

    return [
        {
            "channels": ["+relay"],
            "label": _("Relay (very verbose)"),
        },
        {
            "channels": ["+file", "+ntdll"],
            "label": _("File Access"),
        },
        {
            "channels": ["+font"],
            "label": _("Fonts"),
        },
        {
            "channels": ["+dinput", "+joystick", "+winmm", "+xinput"],
            "label": _("Game Controllers"),
        },
        {
            "channels": ["+d3d"],
            "environment": graphics_environment,
            "label": _("Game Graphics"),
        },
        {
            "channels": ["+event", "+key", "+keyboard"],
            "label": _("Keyboard Input"),
        },
        {
            "channels": ["+cursor", "+event"],
            "label": _("Mouse Input"),
        },
        {
            "channels": ["+iphlpapi", "+winhttp", "+wininet", "+winsock"],
            "label": _("Network Connections"),
        },
        {
            "channels": ["+print", "+psdrv", "+winspool", "+localspl"],
            "label": _("Printing"),
        },
        {
            "channels": ["+mmdevapi", "+winmm", "+driver", "+msacm", "+midi", "+dsound", "+dsound3d",
                         "+xaudio2", "+xapofx", "+dmusic", "+mci", "+pulse", "+oss", "+alsa", "+coreaudio"],
            "label": _("Sound (verbose)"),
        },
        {
            "channels": window_channels,
            "label": _("Window Behavior"),
        },
    ]


#####
#
# Dealing with subprocesses
#
#####

GRAB = 1
NULL = 2

def run(args, stdout=None, stderr=None,
        background=False, cwd=None, env=None, logprefix=""):
    """Works the same way as subprocess.Popen, but simplifies retrieving
    the output and exit code. Returns (retcode, out, err) triplets.

    If stdout or stderr are set to GRAB, then they are captured and
    returned. If they are set to NULL, then they are redirected
    to/from /dev/null instead. If the command cannot be run, then
    returns -1 as the exit code instead of raising an exception.

    Finally, the execution is timed and traced with cxlog. If stdout
    or stderr are being captured, then they are logged too. If
    logprefix is set, then the trace is prefixed with it.

    """
    start = time.time()
    if cxlog.is_on():
        if logprefix != "":
            logprefix = logprefix + ": "
        bg = ""
        if background:
            bg = " &"
        cxlog.log("%sRunning: %s%s" % (logprefix, " ".join(cxlog.debug_str(arg) for arg in args), bg))

    out_log = False
    if stdout:
        if stdout == GRAB and not background:
            subout = subprocess.PIPE
            out_log = True
        elif stdout == NULL:
            subout = subprocess.DEVNULL
        else:
            raise TypeError("invalid value for stdout")
    else:
        subout = None

    err_log = False
    if stderr:
        if stderr == GRAB and not background:
            suberr = subprocess.PIPE
            err_log = True
        elif stderr == NULL:
            if cxlog.is_on():
                suberr = None
            else:
                suberr = subprocess.DEVNULL
        else:
            raise TypeError("invalid value for stderr")
    else:
        suberr = None

    try:
        if background:
            childpid = os.fork()
            if childpid == 0:
                #child
                try:
                    subp = subprocess.Popen(args, stdout=subout, # pylint: disable=R1732
                                            stderr=suberr, close_fds=True,
                                            cwd=cwd, env=env,
                                            universal_newlines=True)
                except Exception as e:
                    sys.stderr.write(cxlog.to_str(e))
                    os._exit(1) #pylint: disable=W0212
                os._exit(0) #pylint: disable=W0212
            else:
                #parent
                _pid, retcode = os.waitpid(childpid, 0)
                out = None
                err = None
        else:
            subp = subprocess.Popen(args, stdout=subout, # pylint: disable=R1732
                                    stderr=suberr, close_fds=True,
                                    cwd=cwd, env=env,
                                    universal_newlines=True)

            retcode = None
            out = None
            err = None

            # The code below is inspired by subprocess.py in Python
            if hasattr(selectors, 'PollSelector'):
                PopenSelector = selectors.PollSelector
            else:
                PopenSelector = selectors.SelectSelector

            output = {}
            with PopenSelector() as selector:
                if subp.stdout:
                    selector.register(subp.stdout, selectors.EVENT_READ)
                    output[subp.stdout] = []
                if subp.stderr:
                    selector.register(subp.stderr, selectors.EVENT_READ)
                    output[subp.stderr] = []

                while selector.get_map() and retcode is None:
                    retcode = subp.poll()

                    for key, _event in selector.select(1):
                        data = os.read(key.fd, 32768)
                        if not data:
                            selector.unregister(key.fileobj)
                        else:
                            output[key.fileobj].append(data)

                retcode = subp.wait()

            if subp.stdout:
                out = b''.join(output[subp.stdout])
                out = out.decode(subp.stdout.encoding, subp.stdout.errors)
                out = out.replace("\r\n", "\n").replace("\r", "\n")
                subp.stdout.close()

            if subp.stderr:
                err = b''.join(output[subp.stderr])
                err = err.decode(subp.stderr.encoding, subp.stderr.errors)
                err = err.replace("\r\n", "\n").replace("\r", "\n")
                subp.stderr.close()
    except OSError as exception:
        retcode = -1
        out = ""
        err = cxlog.to_str(exception)

    if not background and cxlog.is_on():
        cxlog.log("%s%s -> rc=%d  (took %0.3f seconds)" % (logprefix, cxlog.debug_str(os.path.basename(args[0])), retcode, time.time() - start))
        if out_log:
            count = out.count("\n")
            if count > 20:
                cxlog.log("out=<%d lines>" % count)
            else:
                cxlog.log("out=[%s]" % out)
        if err_log:
            count = err.count("\n")
            if count > 20:
                cxlog.log("err=<%d lines>" % count)
            else:
                cxlog.log("err=[%s]" % err)

    return (retcode, out, err)

def system(args, background=False, cwd=None, env=None, logprefix=""):
    """Works the same way as run(), except that neither stdout nor stderr are
    captured, and only the retcode is returned.
    """
    retcode, _out, _err = run(args, background=background, cwd=cwd, env=env,
                              logprefix=logprefix)
    return retcode


def launch_url(url):
    """Open the specified URL in a web browser."""
    webbrowser.open(url)



#####
#
# Select wrapper
#
#####


# Functions and module imports are stored in the same namespace. This means a
# simple 'import select' would get overridden by our select() function, thus
# preventing it from accessing the content of the module, and necessitating
# annoying workarounds (and rightful Pylint warnings).
#
# So we just import the parts we need, renaming them in the process to avoid
# any conflict.
from select import select as select_select
from select import error as select_error

def select(iwtd, owtd, ewtd, timeout=None):
    """Same as select.select() but retries on EINTR (with the same timeout)."""
    while True:
        try:
            return select_select(iwtd, owtd, ewtd, timeout)
        except select_error as err:
            if err.args[0] == errno.EINTR:
                if timeout is not None:
                    timeout = 0
                continue
            raise



#####
#
# User directories
#
#####

def _xdg_dirs_skip_whitespace(pos, string):
    while pos < len(string) and string[pos] in ' \t':
        pos += 1
    return pos

def _internal_load_xdg_dirs():
    try:
        # Try the system's Python module first
        # pylint: disable=F0401
        import xdg.BaseDirectory
        config_file = xdg.BaseDirectory.load_first_config('user-dirs.dirs')
    except:
        # Otherwise use our builtin module
        import BaseDirectory
        config_file = BaseDirectory.load_first_config('user-dirs.dirs')
    if config_file is None:
        config_file = os.path.join(os.environ.get('HOME', '/'), ".config", "user-dirs.dirs")
    result = {}
    try:
        f = open(config_file, 'rt', encoding='utf8', errors='surrogateescape') # pylint: disable=R1732
    except:
        return {}
    try:
        for line in f:
            line = line.rstrip('\n')
            # Based on MIT code in xdg-user-dir-lookup.c in the xdg-user-dirs package.
            pos = _xdg_dirs_skip_whitespace(0, line)

            if line[pos:pos+4] != 'XDG_':
                continue
            pos += 4

            dir_pos = line.find('_DIR', pos)
            if dir_pos == -1:
                continue
            name = line[pos:dir_pos]
            pos = dir_pos + 4

            pos = _xdg_dirs_skip_whitespace(pos, line)

            if line[pos:pos+1] != '=':
                continue
            pos += 1

            pos = _xdg_dirs_skip_whitespace(pos, line)

            if line[pos:pos+1] != '"':
                continue
            pos += 1

            if line[pos:pos+6] == '$HOME/':
                pos += 6
                value_base = os.environ.get('HOME', '/') + '/'
            elif line[pos:pos+1] == '/':
                value_base = ''
            else:
                continue

            value_end = line.find('"', pos)

            if value_end != -1:
                value = value_base + line[pos:value_end].replace('\\', '')
            else:
                # Yes, the standard parser does not mind if there's no terminating quote.
                value = value_base + line[pos:].replace('\\', '')

            result[name] = value
    finally:
        f.close()
    return result

_xdg_dirs = None

def _load_xdg_dirs():
    global _xdg_dirs # pylint: disable=W0603
    if _xdg_dirs is None:
        _xdg_dirs = _internal_load_xdg_dirs()
    return _xdg_dirs

def get_download_dir():
    if not distversion.IS_MACOSX:
        xdg_dirs = _load_xdg_dirs()
        if 'DOWNLOAD' in xdg_dirs:
            return xdg_dirs['DOWNLOAD']
    path = os.path.join(os.environ.get('HOME', '/'), _('Downloads'))
    if os.path.isdir(path):
        return path
    return os.path.join(os.environ.get('HOME', '/'), 'Downloads')

def get_desktop_dir():
    if not distversion.IS_MACOSX:
        xdg_dirs = _load_xdg_dirs()
        if 'DESKTOP' in xdg_dirs:
            return xdg_dirs['DESKTOP']
    path = os.path.join(os.environ.get('HOME', '/'), _('Desktop'))
    if os.path.isdir(path):
        return path
    return os.path.join(os.environ.get('HOME', '/'), 'Desktop')


#####
#
# Locating icons
#
#####


S_MEDIUM = ('48x48', '32x32', '')

def get_icon_path(root, subdir, iconname, sizes=S_MEDIUM):
    for ext in ('.png', '.xpm'):
        for size in sizes:
            filename = os.path.join(root, size, subdir, iconname + ext)
            if os.path.exists(filename):
                return filename
    return None

def get_icon_paths(root, subdir, iconname):
    for ext in ('.png', '.xpm'):
        for size in os.listdir(root):
            filename = os.path.join(root, size, subdir, iconname + ext)
            if os.path.exists(filename):
                yield filename


#####
#
# pylint utilities
#
#####

def not_yet_implemented():
    """If a method raises NotImplementedError(), pylint regards it as an
    "abstract method" that must be implemented in all subclasses, and the
    class as an "abstract class". Sometimes this is desired, but in some cases
    we use this exception for parts of the API that we are not using now but
    plan to implement and use in the future. Those functions should instead
    raise cxutils.not_yet_implemented()."""
    return NotImplementedError()


#####
#
# umask cache
#
#####

# Retrieve the umask on startup, that is before we have multiple threads all
# trying to get it at the same time. os.umask() MUST NOT be used anywhere
# else. Should we need to modify the umask, then we will need to write a
# proper umask() wrapper.
UMASK = os.umask(0)
os.umask(UMASK)
