/*
 * Decompiled with CFR 0.152.
 */
package ghidra.framework.store.local;

import ghidra.framework.store.Version;
import ghidra.framework.store.local.LocalFileSystem;
import ghidra.framework.store.local.LocalFolderItem;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.util.ArrayList;
import java.util.NoSuchElementException;
import java.util.StringTokenizer;

class HistoryManager {
    static final String HISTORY_FILE = "history.dat";
    private LocalFolderItem item;
    private int minVersion;
    private int curVersion;
    private Version[] versions;

    HistoryManager(LocalFolderItem item, boolean create) throws IOException {
        this.item = item;
        if (create) {
            this.versions = new Version[0];
        }
    }

    private File getHistoryFile() {
        return new File(this.item.getDataDir(), HISTORY_FILE);
    }

    synchronized boolean fixHistory(int minVersion, int curVersion) throws IOException {
        this.validate();
        if (minVersion == this.minVersion && curVersion == this.curVersion) {
            return false;
        }
        if (minVersion < 1 || curVersion < minVersion) {
            throw new IllegalArgumentException();
        }
        Version[] newVersions = new Version[curVersion - minVersion + 1];
        int newIx = 0;
        int oldIx = 0;
        int version = minVersion;
        if (minVersion < this.minVersion) {
            while (version < this.minVersion && version <= curVersion) {
                newVersions[newIx++] = new Version(version++, 0L, "<Unknown>", "<Recovered>");
            }
        }
        if (version >= this.minVersion && version <= this.curVersion) {
            while (this.versions[oldIx].getVersion() < version) {
                ++oldIx;
            }
            while (version <= this.curVersion && version <= curVersion) {
                newVersions[newIx++] = this.versions[oldIx++];
                ++version;
            }
        }
        while (version <= curVersion) {
            newVersions[newIx++] = new Version(version++, 0L, "<Unknown>", "<Recovered>");
        }
        this.versions = newVersions;
        this.minVersion = minVersion;
        this.curVersion = curVersion;
        this.writeHistoryFile();
        return true;
    }

    synchronized void versionAdded(int version, long time, String comment, String user) throws IOException {
        this.validate();
        if (version != this.curVersion + 1) {
            this.item.log("ERROR! unexpected version " + version + " created, expected version " + (this.curVersion + 1), user);
            return;
        }
        this.item.log("version " + version + " created", user);
        Version ver = new Version(version, time, user, comment);
        this.appendHistoryFile(ver);
        Version[] newVersions = new Version[this.versions.length + 1];
        System.arraycopy(this.versions, 0, newVersions, 0, this.versions.length);
        newVersions[this.versions.length] = ver;
        this.versions = newVersions;
        this.curVersion = version;
        if (version == 1) {
            this.minVersion = 1;
        }
    }

    synchronized void versionDeleted(int version, String user) throws IOException {
        this.validate();
        if (this.versions.length <= 1) {
            this.item.log("ERROR! version " + version + " deleted illegally, min=" + this.minVersion + ", max=" + this.curVersion, user);
            return;
        }
        Version[] newVersions = new Version[this.versions.length - 1];
        if (version == this.versions[0].getVersion()) {
            System.arraycopy(this.versions, 1, newVersions, 0, this.versions.length - 1);
            this.minVersion = newVersions[0].getVersion();
        } else if (version == this.versions[this.versions.length - 1].getVersion()) {
            System.arraycopy(this.versions, 0, newVersions, 0, this.versions.length - 1);
            this.curVersion = newVersions[newVersions.length - 1].getVersion();
        } else {
            this.item.log("ERROR! version " + version + " deleted illegally, min=" + this.minVersion + ", max=" + this.curVersion, user);
            return;
        }
        this.item.log("version " + version + " deleted", user);
        this.versions = newVersions;
        this.writeHistoryFile();
    }

    synchronized Version[] getVersions() throws IOException {
        this.validate();
        return (Version[])this.versions.clone();
    }

    synchronized Version getVersion(int version) throws IOException {
        this.validate();
        if (version >= this.minVersion && version < this.curVersion) {
            return this.versions[version - this.minVersion];
        }
        return null;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void validate() throws IOException {
        File historyFile;
        if (LocalFileSystem.isRefreshRequired()) {
            this.versions = null;
            this.minVersion = 0;
            this.curVersion = 0;
        }
        if ((historyFile = this.getHistoryFile()).exists()) {
            Version[] oldVersions = this.versions;
            int oldMinVersion = this.minVersion;
            int oldCurVersion = this.curVersion;
            boolean success = false;
            try {
                this.readHistoryFile();
                success = true;
            }
            finally {
                if (!success) {
                    this.versions = oldVersions;
                    this.minVersion = oldMinVersion;
                    this.curVersion = oldCurVersion;
                }
            }
        } else {
            this.versions = new Version[0];
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void readHistoryFile() throws IOException {
        ArrayList<Version> list = new ArrayList<Version>();
        this.minVersion = 0;
        this.curVersion = 0;
        File historyFile = this.getHistoryFile();
        try (BufferedReader in = new BufferedReader(new FileReader(historyFile));){
            String line = in.readLine();
            while (line != null) {
                Version ver;
                try {
                    ver = this.decodeVersion(line);
                }
                catch (Exception e) {
                    throw new IOException("Bad history file: " + String.valueOf(historyFile));
                }
                int version = ver.getVersion();
                if (this.curVersion != 0 && version != this.curVersion + 1) {
                    throw new IOException("Bad history file" + String.valueOf(historyFile));
                }
                if (this.minVersion == 0) {
                    this.minVersion = version;
                }
                this.curVersion = version;
                list.add(ver);
                line = in.readLine();
            }
        }
        this.versions = new Version[list.size()];
        list.toArray(this.versions);
    }

    private void writeHistoryFile() {
        File historyFile = this.getHistoryFile();
        try {
            File tmpFile = new File(historyFile.getParentFile(), historyFile.getName() + ".new");
            tmpFile.delete();
            BufferedWriter out = new BufferedWriter(new FileWriter(tmpFile));
            for (int i = 0; i < this.versions.length; ++i) {
                out.write(this.encodeVersion(this.versions[i]));
                out.newLine();
            }
            out.close();
            File oldFile = null;
            if (historyFile.exists()) {
                oldFile = new File(historyFile.getParentFile(), historyFile.getName() + ".bak");
                oldFile.delete();
                if (!historyFile.renameTo(oldFile)) {
                    throw new IOException("file is in use");
                }
            }
            if (!tmpFile.renameTo(historyFile)) {
                if (oldFile != null) {
                    oldFile.renameTo(historyFile);
                }
                throw new IOException("file error - backup may exist");
            }
            if (oldFile != null) {
                oldFile.delete();
            }
        }
        catch (IOException e) {
            this.item.log("ERROR! failed to update history file: " + e.toString(), null);
        }
    }

    private void appendHistoryFile(Version ver) {
        File historyFile = this.getHistoryFile();
        try {
            BufferedWriter out = new BufferedWriter(new FileWriter(historyFile, true));
            out.write(this.encodeVersion(ver));
            out.newLine();
            out.close();
        }
        catch (IOException e) {
            this.item.log("ERROR! failed to update history file: " + e.toString(), null);
        }
    }

    private String encodeVersion(Version ver) {
        StringBuffer buf = new StringBuffer();
        buf.append(ver.getVersion());
        buf.append(';');
        buf.append(ver.getUser());
        buf.append(';');
        buf.append(ver.getCreateTime());
        buf.append(';');
        this.encodeString(ver.getComment(), buf);
        return buf.toString();
    }

    private Version decodeVersion(String line) throws NumberFormatException, NoSuchElementException {
        StringTokenizer st = new StringTokenizer(line, ";");
        int version = Integer.parseInt(st.nextToken());
        String user = st.nextToken();
        long time = Long.parseLong(st.nextToken());
        String comment = "";
        if (st.hasMoreTokens()) {
            comment = this.decodeString(st.nextToken());
        }
        return new Version(version, time, user, comment);
    }

    private void encodeString(String text, StringBuffer buf) {
        if (text == null) {
            return;
        }
        block6: for (int i = 0; i < text.length(); ++i) {
            char next = text.charAt(i);
            switch (next) {
                case '\n': {
                    buf.append("\\n");
                    continue block6;
                }
                case '\r': {
                    buf.append("\\r");
                    continue block6;
                }
                case ';': {
                    buf.append("\\s");
                    continue block6;
                }
                case '\\': {
                    buf.append("\\\\");
                    continue block6;
                }
                default: {
                    buf.append(next);
                }
            }
        }
    }

    private String decodeString(String text) {
        if (text == null) {
            return "";
        }
        StringBuffer buf = new StringBuffer();
        boolean controlChar = false;
        for (int i = 0; i < text.length(); ++i) {
            char next = text.charAt(i);
            if (next == '\\') {
                controlChar = true;
                continue;
            }
            if (controlChar) {
                switch (next) {
                    case 'n': {
                        buf.append('\n');
                        break;
                    }
                    case 'r': {
                        buf.append('\r');
                        break;
                    }
                    case 's': {
                        buf.append(';');
                        break;
                    }
                    default: {
                        buf.append(next);
                    }
                }
                controlChar = false;
                continue;
            }
            buf.append(next);
        }
        return buf.toString();
    }
}

