// Copyright (c) 2008, Frances Y. Kuo and Stephen Joe
// Copyright 2019 Google LLC
// All rights reserved.
//
// Use of this source code is governed by a BSD-style
// license that can be found in the LICENSE file or at
// https://developers.google.com/open-source/licenses/bsd
#ifndef JOE_KUO_SOBOL_DATA_H
#define JOE_KUO_SOBOL_DATA_H

namespace sobol_data {

const int kMaxSobolDim = 21200;
const int kMaxSobolDigits = 31;

// Calculated from https://web.maths.unsw.edu.au/~fkuo/sobol/sobol.cc with
// input data file new-joe-kuo-6.21201 by printing the values V[i] >> (32-i).
// Notice the first values of each row are the m_i from new-joe-kuo-6.21201.
const int kDirectionNumbers[kMaxSobolDim][kMaxSobolDigits] = {
  {1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1},
  {1, 3, 5, 15, 17, 51, 85, 255, 257, 771, 1285, 3855, 4369, 13107, 21845, 65535, 65537, 196611, 327685, 983055, 1114129, 3342387, 5570645, 16711935, 16843009, 50529027, 84215045, 252645135, 286331153, 858993459, 1431655765},
  {1, 3, 3, 9, 29, 23, 71, 197, 209, 627, 1907, 1369, 4109, 12327, 12407, 36949, 119041, 94979, 291587, 809225, 855325, 2565911, 7829319, 5592517, 16777681, 50332019, 50332787, 150998105, 486542605, 385885991, 1191212919},
  {1, 3, 1, 5, 31, 29, 81, 147, 433, 149, 719, 3693, 3841, 11523, 16641, 49925, 16671, 83229, 515921, 482707, 1314993, 2380693, 7205839, 2481005, 11599873, 60096515, 63897601, 191692805, 268501023, 805503005, 268501073},
  {1, 1, 1, 11, 31, 55, 61, 157, 181, 191, 1291, 3851, 6993, 7153, 16561, 16571, 17679, 184103, 514861, 908077, 983109, 2556623, 2951131, 2952667, 21303041, 63907585, 115027201, 115027211, 268453151, 268619575, 268950333},
  {1, 1, 3, 3, 25, 9, 43, 251, 449, 449, 1347, 323, 5081, 14281, 23403, 35771, 36865, 102401, 110595, 831491, 331801, 1380361, 8335403, 15675643, 16552385, 45978049, 16102723, 167293251, 450429913, 784934857, 1080781675},
  {1, 3, 5, 13, 11, 37, 31, 227, 381, 143, 241, 3889, 2255, 6625, 15395, 33895, 101617, 171539, 442933, 345725, 1298203, 844053, 7545679, 12019507, 4321485, 4264415, 130233345, 71793665, 214959135, 502268945, 1074822195},
  {1, 1, 5, 5, 17, 9, 9, 45, 237, 633, 65, 1601, 5701, 8005, 32593, 47689, 13385, 74861, 402861, 931641, 2019329, 3641345, 5509125, 3411973, 7622673, 23351305, 124063753, 158634029, 227840237, 903271033, 1716179009},
  {1, 1, 5, 5, 5, 53, 53, 113, 113, 353, 1873, 785, 4885, 533, 24853, 58661, 124261, 124193, 513313, 775985, 201985, 1185025, 3216645, 11871493, 28665093, 45424949, 75833653, 214380913, 67436913, 67436641, 1141178961},
  {1, 1, 7, 11, 19, 37, 69, 91, 103, 871, 913, 1009, 6199, 10683, 10691, 10741, 5237, 27819, 433303, 434167, 1918721, 2804481, 2478855, 16639243, 12584211, 62921509, 63022917, 37857115, 38228327, 868921959, 1556706449},
  {1, 1, 5, 1, 1, 27, 79, 35, 175, 695, 1021, 2333, 2353, 13237, 17413, 11279, 31771, 52319, 195971, 794507, 1847729, 2930161, 985141, 6498481, 32443153, 49745675, 62771999, 266304307, 451936703, 997196295, 1875902989},
  {1, 1, 1, 3, 11, 43, 75, 43, 425, 37, 1813, 1317, 2501, 12743, 12623, 4711, 56359, 253127, 120005, 779337, 2033521, 3607409, 7799633, 12002195, 3493915, 37318715, 84013147, 117665819, 252145945, 655061653, 1728271269},
  {1, 3, 5, 5, 31, 35, 113, 51, 31, 133, 1197, 1511, 6913, 1801, 23819, 61239, 11021, 162239, 245979, 312137, 541137, 1622771, 8115893, 11367317, 15663119, 28639251, 113049633, 64421987, 99680751, 744030901, 1725629373},
  {1, 3, 3, 9, 7, 49, 33, 163, 483, 681, 2023, 2129, 7169, 5123, 11267, 21513, 62471, 105521, 367649, 38051, 110051, 775849, 947175, 13964369, 24117249, 1048579, 17825795, 53477385, 422576135, 462422065, 1601175585},
  {1, 1, 1, 15, 21, 21, 77, 157, 61, 371, 1001, 2257, 1169, 7281, 30161, 30175, 55109, 55109, 154909, 810285, 1864749, 2389731, 3987321, 843585, 13522177, 20993281, 13634817, 231791887, 131104021, 724742933, 1788258637},
  {1, 3, 1, 13, 27, 49, 35, 133, 331, 475, 1389, 1707, 1461, 10927, 1749, 31409, 109391, 195317, 302367, 244561, 443703, 1268895, 5095857, 2606295, 23876465, 37079539, 98310929, 259804541, 231028491, 308059905, 2126623027},
  {1, 1, 1, 15, 7, 5, 123, 103, 287, 321, 1109, 255, 597, 11005, 32341, 10491, 96851, 12009, 159799, 209091, 733379, 3265357, 2004569, 4960971, 14421041, 51131601, 51144081, 221523135, 426099479, 187583765, 1891850603},
  {1, 3, 1, 15, 13, 25, 27, 109, 131, 897, 1687, 2495, 413, 5311, 11701, 12211, 81945, 49173, 147519, 606457, 770111, 148389, 3917099, 9257323, 19321841, 32426579, 116392625, 210773695, 183611165, 525530409, 2109135627},
  {1, 1, 5, 5, 19, 61, 87, 187, 463, 599, 225, 3331, 7029, 5501, 32705, 8641, 9287, 60649, 336219, 213775, 636115, 1228027, 1507869, 16452127, 9374257, 51848881, 69302997, 37990133, 246706179, 1002344493, 201699335},
  {1, 3, 7, 11, 23, 15, 103, 65, 67, 327, 75, 2135, 7759, 9127, 12289, 28675, 45063, 143371, 241687, 512015, 1962087, 274497, 1339459, 13939015, 32911435, 3127383, 76795471, 266982311, 117440513, 1056964611, 385875975},
  {1, 3, 7, 13, 13, 15, 69, 81, 339, 887, 733, 925, 4799, 5557, 1281, 37123, 37639, 14093, 105741, 23823, 261957, 128337, 5357651, 5030007, 2377181, 13651613, 87220159, 87055029, 46071809, 632619011, 1962999815},
  {1, 1, 3, 13, 7, 35, 63, 113, 401, 19, 221, 2999, 2387, 12527, 21249, 27905, 122115, 42765, 305415, 273187, 1898303, 213873, 2088081, 1117459, 3373021, 15741623, 79286867, 207419375, 121831425, 48168961, 1372651523},
  {1, 3, 5, 9, 1, 25, 53, 137, 331, 813, 193, 2777, 5617, 7357, 4801, 2883, 94277, 63049, 343105, 94041, 651381, 2134089, 970763, 16175469, 25120897, 49126553, 117459121, 252399357, 117723137, 1063555075, 63180805},
  {1, 3, 1, 13, 9, 35, 107, 57, 427, 201, 1429, 1825, 2683, 8371, 26049, 18243, 74305, 173389, 255945, 322019, 2072363, 2336761, 7240939, 6200457, 19526357, 55213793, 35114427, 247056883, 317657089, 913821699, 317558785},
  {1, 3, 1, 5, 27, 61, 31, 149, 287, 245, 1945, 751, 5561, 643, 17809, 19187, 55185, 60469, 464139, 92941, 339215, 3108805, 1697199, 986149, 33112937, 55493311, 66542409, 122676435, 534580993, 759153923, 1869983489},
  {1, 1, 5, 11, 19, 41, 61, 213, 5, 385, 655, 2183, 4485, 4593, 1521, 44401, 93525, 52763, 376835, 344121, 540781, 3162213, 7356725, 1131281, 5423455, 16041431, 117068245, 184502753, 449146113, 414278913, 414201093},
  {1, 3, 5, 3, 3, 13, 69, 157, 207, 521, 519, 1551, 4649, 2641, 5809, 49075, 11349, 258003, 468563, 794749, 262229, 3932269, 6029759, 262681, 19138199, 44305567, 124522393, 32771457, 82576641, 950820099, 1758826245},
  {1, 1, 7, 13, 1, 19, 1, 181, 109, 779, 161, 117, 895, 5133, 9297, 44177, 108215, 193181, 253905, 787267, 1315889, 795685, 7122365, 4564347, 10417393, 48225253, 68159983, 256912893, 449855745, 78705921, 236039431},
  {1, 3, 7, 5, 13, 19, 59, 247, 109, 861, 1087, 3827, 5237, 13649, 4565, 5447, 100179, 70937, 351553, 38007, 1815871, 4165931, 7705745, 2689009, 14259835, 51958071, 108727705, 20394381, 496592817, 26022771, 397466007},
  {1, 1, 3, 9, 25, 29, 41, 3, 471, 445, 803, 3503, 755, 2655, 21197, 64125, 117815, 115965, 50117, 904425, 1494077, 2522279, 5114915, 14040609, 1307823, 22165547, 85722447, 95687667, 181160689, 310149809, 437081331},
  {1, 3, 5, 13, 23, 1, 55, 151, 485, 951, 155, 1481, 7019, 3193, 1261, 52263, 1393, 70353, 32531, 943365, 1267515, 1322307, 3993929, 10834683, 24932735, 12181349, 128974935, 137367853, 277874673, 733013587, 468792693},
  {1, 3, 7, 3, 13, 59, 17, 43, 109, 629, 849, 359, 2493, 3267, 18869, 33359, 30355, 146687, 82177, 841479, 1994069, 1872887, 5875193, 5455465, 12861685, 2737411, 121590569, 205576463, 503136081, 906207283, 586490519},
  {1, 3, 1, 3, 5, 53, 69, 255, 265, 463, 153, 3115, 8039, 3463, 8205, 57519, 8773, 26319, 519617, 388601, 964369, 2013163, 8117597, 6413811, 27422973, 14973271, 107273539, 125225963, 335552209, 1006665331, 67143985},
  {1, 1, 5, 5, 23, 33, 13, 175, 379, 895, 167, 1077, 6983, 4999, 9837, 13901, 21185, 225857, 227011, 382197, 384017, 4133995, 4185103, 4578435, 32797835, 24671625, 44490395, 237300883, 292313521, 1038365585, 1689052853},
  {1, 1, 7, 7, 1, 61, 123, 139, 403, 341, 825, 3903, 1679, 5585, 3437, 45557, 86787, 103683, 293149, 491297, 1658751, 659839, 1224063, 7321209, 935541, 51366251, 115595115, 221530157, 479930449, 868080049, 1258412471},
  {1, 1, 7, 9, 13, 61, 49, 223, 243, 885, 951, 719, 6647, 5243, 25709, 33885, 79619, 67853, 490257, 400153, 1714949, 2246443, 3322167, 700681, 5617531, 7173659, 1051179, 15741511, 506479281, 777034417, 1047561655},
  {1, 3, 3, 5, 3, 55, 33, 55, 65, 965, 323, 2377, 4337, 10767, 10933, 59119, 126983, 77833, 118799, 765987, 1003565, 430315, 2453701, 5427993, 18796615, 53982813, 131335677, 228867883, 89938289, 11804371, 1270180563},
  {1, 3, 1, 15, 31, 13, 49, 245, 337, 1011, 1137, 1343, 6383, 9565, 23553, 54501, 91393, 259843, 206081, 697103, 211743, 233741, 4828465, 15163893, 13300817, 16925939, 117444977, 219043391, 17830895, 858793053, 722469121},
  {1, 3, 5, 15, 31, 59, 63, 97, 185, 923, 1773, 679, 4711, 1267, 27111, 29561, 67393, 126403, 310725, 954575, 1117663, 3474043, 1641599, 15695009, 25865721, 22746203, 36353325, 158093415, 480958887, 431050931, 246098855},
  {1, 3, 1, 11, 11, 11, 77, 249, 121, 715, 1385, 931, 4403, 4387, 6005, 28545, 122433, 48323, 30913, 783563, 1894731, 3333195, 2673677, 11045945, 19518521, 10093579, 73204649, 186584419, 117900915, 522126179, 690562357},
  {1, 3, 1, 11, 27, 43, 71, 9, 317, 647, 1621, 823, 4783, 535, 4699, 27189, 34545, 111987, 41393, 470875, 1698571, 1375003, 6485847, 4250041, 29371789, 37820215, 54789925, 122422439, 71573119, 569126247, 1269586955},
  {1, 1, 7, 15, 21, 11, 81, 45, 437, 997, 1891, 43, 1961, 8343, 16037, 25185, 72593, 206961, 331095, 428319, 1149957, 2927643, 3843105, 6184157, 15635685, 26052437, 128599667, 31835899, 317276409, 618160839, 782523541},
  {1, 3, 7, 3, 25, 31, 65, 79, 381, 951, 267, 2743, 3853, 7451, 3917, 64971, 29201, 48243, 161687, 809587, 897609, 3700207, 2473329, 4750271, 24707117, 25887111, 67044443, 124455559, 304157085, 684141227, 1920532925},
  {1, 3, 1, 1, 19, 11, 3, 205, 277, 111, 2045, 2773, 3935, 11511, 11671, 7977, 80561, 248371, 16049, 466769, 1744579, 3578235, 3250003, 14887197, 4632933, 930719, 10138125, 145970629, 351985103, 679305799, 914048007},
  {1, 1, 5, 9, 19, 21, 29, 157, 253, 509, 185, 2741, 3519, 9641, 16873, 29153, 98737, 230353, 99669, 32857, 1672515, 1806661, 5489933, 14898509, 238989, 27494381, 98747561, 208358181, 526467631, 88435769, 2033804345},
  {1, 3, 7, 11, 1, 33, 89, 185, 381, 319, 683, 3983, 5509, 6581, 22917, 35765, 102289, 18323, 348695, 678171, 1711441, 1974609, 3555433, 8831177, 508077, 34390511, 35276667, 171068511, 41178133, 115893253, 1334930549},
  {1, 3, 3, 3, 15, 9, 79, 71, 207, 989, 1785, 2865, 3629, 8979, 6441, 58837, 87205, 51031, 90775, 404831, 1423955, 4040053, 4569347, 16106899, 8746179, 16213393, 116721141, 242552181, 116984313, 801580743, 1103957709},
  {1, 3, 7, 11, 15, 39, 119, 27, 407, 781, 1597, 81, 4365, 12573, 4397, 15777, 26805, 125575, 495315, 77335, 1097539, 1191035, 1186171, 9623031, 9109627, 44105937, 77661137, 245698325, 456720969, 295248201, 1205412873},
  {1, 1, 3, 1, 11, 31, 97, 225, 27, 863, 341, 3923, 4981, 12349, 4195, 28719, 37069, 36989, 389751, 144053, 1934615, 3575051, 8319869, 14671757, 14539823, 23882795, 117440553, 83888447, 117444753, 452993921, 1291847143},
  {1, 1, 1, 3, 23, 43, 57, 177, 7, 843, 1547, 2489, 2505, 733, 17191, 16139, 53261, 70141, 242245, 948039, 1739219, 3113703, 6931237, 3850077, 25842083, 53973071, 80491223, 108789077, 311761061, 663303033, 1907576003},
  {1, 3, 7, 7, 17, 17, 37, 71, 443, 265, 1561, 1905, 3767, 9627, 14511, 64661, 85453, 223207, 82371, 82635, 675789, 4120909, 2993953, 759235, 2543087, 46065365, 21546933, 100912149, 369676947, 438985887, 441470387},
  {1, 3, 1, 5, 27, 63, 123, 213, 383, 177, 1911, 151, 801, 2137, 22197, 54035, 17669, 154655, 313093, 116489, 1102111, 1526059, 5096727, 2515753, 26940819, 26053861, 72374155, 238173011, 157110781, 819253509, 221322545},
  {1, 1, 3, 5, 11, 43, 53, 133, 503, 675, 1057, 2787, 2401, 14365, 2115, 14479, 129117, 51245, 84999, 596993, 373767, 2096191, 218137, 13599785, 10247459, 23866551, 113248253, 247465071, 20977125, 893388689, 1371537863},
  {1, 3, 5, 5, 29, 17, 47, 173, 479, 417, 1059, 4069, 8101, 509, 5681, 55887, 42509, 19327, 228353, 689155, 297989, 2855941, 693277, 4453393, 14992431, 31276205, 98842079, 128581025, 494665763, 360973285, 1822690213},
  {1, 3, 3, 11, 3, 1, 109, 9, 69, 929, 1507, 227, 3947, 3555, 2657, 34637, 78761, 196325, 50177, 594947, 1891331, 1350667, 171011, 3456001, 19805293, 21365769, 10466373, 233326497, 147689955, 97606883, 1224545131},
  {1, 1, 1, 5, 17, 39, 23, 5, 343, 113, 881, 593, 8053, 9473, 31031, 2823, 130389, 65351, 487681, 413953, 1799425, 3211525, 4011793, 9897767, 27350295, 15419141, 36503639, 127205745, 380699249, 944047953, 1088622197},
  {1, 3, 1, 5, 25, 15, 31, 103, 499, 817, 819, 2865, 5493, 5961, 25215, 47311, 52599, 116387, 33537, 164099, 623873, 2726149, 4820761, 16490255, 19588895, 65172327, 70627571, 1814577, 285323827, 923074097, 657906805},
  {1, 1, 1, 11, 11, 17, 63, 105, 183, 881, 721, 2161, 7003, 9979, 24097, 51727, 57081, 203623, 331009, 701697, 2039553, 3817227, 2116875, 8184593, 32341311, 48651113, 108783031, 57515633, 456394705, 399771505, 1870402651},
  {1, 1, 5, 11, 9, 29, 97, 231, 363, 745, 1241, 2621, 1507, 8289, 27669, 52249, 48143, 230627, 443585, 642113, 708933, 1663563, 5316553, 12020957, 31151137, 14247079, 53034027, 254708777, 401605273, 441452925, 642779555},
  {1, 1, 5, 15, 19, 45, 41, 7, 383, 361, 473, 2333, 6759, 16251, 19109, 22769, 93119, 9335, 275905, 674497, 1015109, 3414095, 1319891, 9200109, 2937961, 53773383, 6765631, 192880297, 214843673, 88557149, 206226471},
  {1, 3, 7, 7, 31, 19, 83, 137, 221, 105, 187, 1215, 5215, 7959, 16155, 21787, 33025, 40117, 302273, 518851, 674247, 1098183, 6875999, 7718483, 28295443, 45315401, 109772829, 120521385, 445579387, 377948031, 464199583},
  {1, 1, 1, 3, 23, 15, 111, 223, 83, 201, 1865, 1577, 3643, 6031, 20375, 2535, 1239, 83419, 194497, 719553, 1060545, 2991811, 1962711, 15417807, 6805295, 12376991, 126796051, 31703305, 400933769, 242861801, 1563804411},
  {1, 1, 5, 13, 31, 15, 55, 25, 161, 809, 985, 2813, 4853, 4839, 10967, 25839, 122961, 106825, 204865, 532673, 1172677, 1927373, 8017759, 11034959, 17325431, 63472985, 87586017, 186586473, 349492505, 937119805, 1724138549},
  {1, 1, 3, 13, 25, 47, 39, 87, 257, 41, 2025, 2267, 8181, 10657, 2183, 19423, 59823, 188873, 270401, 567105, 1548483, 3893965, 1566937, 7264623, 22721383, 1553687, 18873537, 59467369, 298108585, 499567131, 410829621},
  {1, 1, 1, 11, 21, 53, 125, 249, 293, 101, 533, 533, 4575, 10689, 11465, 14073, 22109, 261953, 167057, 77713, 730577, 1618971, 4753669, 2537765, 15080813, 42926409, 108941685, 194695733, 483955909, 81483397, 432490127},
  {1, 1, 7, 11, 11, 7, 57, 79, 323, 477, 621, 3027, 5599, 3495, 17755, 43141, 8131, 239671, 498097, 539729, 1706039, 2756475, 5383963, 14555415, 13283145, 63579135, 86424819, 123505837, 170849021, 423505955, 1791477487},
  {1, 1, 5, 5, 17, 13, 81, 3, 131, 989, 29, 65, 4929, 5061, 2009, 59989, 110199, 20103, 90257, 1041489, 829045, 929141, 315393, 2297885, 10686465, 58544211, 54301075, 48255757, 362968333, 725184625, 2109999985},
  {1, 1, 7, 13, 23, 7, 65, 251, 475, 541, 2045, 579, 7113, 9803, 25459, 53981, 51495, 91807, 2609, 700113, 1776503, 3859197, 4788231, 14905367, 25415729, 28602411, 84242603, 92213869, 68072429, 471776755, 1143658105},
  {1, 3, 5, 1, 9, 43, 3, 149, 11, 805, 479, 129, 6541, 11677, 26287, 57767, 50873, 172935, 312817, 2931, 562837, 4175217, 1228825, 1654811, 27770963, 47202437, 68370587, 150421909, 433832431, 796510673, 782129469},
  {1, 1, 3, 13, 31, 13, 13, 255, 487, 589, 365, 999, 1961, 8755, 10585, 36241, 105579, 119859, 304273, 543249, 760755, 3550269, 7697423, 14269469, 586813, 29914159, 43596823, 14605981, 298710461, 88635415, 577684953},
  {1, 3, 3, 1, 5, 63, 89, 91, 127, 509, 1103, 551, 5741, 10473, 28315, 64173, 115527, 129467, 232113, 788947, 447859, 3342001, 4240405, 3712015, 17132649, 39302219, 33692719, 167898637, 235390431, 235817879, 1779359133},
  {1, 1, 3, 3, 1, 19, 123, 127, 237, 101, 141, 1983, 4447, 10709, 8535, 51991, 117067, 143433, 438961, 193585, 643443, 1528147, 5630417, 15917379, 17173259, 52660495, 38891837, 169177237, 45134973, 677943183, 432476239},
  {1, 1, 5, 7, 23, 31, 37, 243, 289, 517, 765, 4025, 3003, 9019, 11675, 50353, 114279, 84573, 45937, 561617, 1012917, 486807, 1458887, 2844239, 16382773, 42756547, 12885009, 240154421, 53920877, 1052525641, 554292203},
  {1, 1, 5, 11, 17, 53, 117, 183, 491, 661, 2037, 2025, 2007, 10173, 5953, 42713, 58683, 156455, 97745, 894673, 1042421, 2125883, 8363969, 12697829, 12484453, 29819719, 112852155, 173805061, 128447973, 78659289, 1684001063},
  {1, 1, 1, 5, 1, 13, 13, 209, 345, 349, 541, 397, 425, 11437, 29233, 20993, 101069, 236453, 290897, 897617, 703889, 2614037, 222033, 12466525, 21398621, 15090881, 97285385, 261674189, 456394381, 578030557, 1088689017},
  {1, 1, 3, 15, 1, 57, 115, 7, 33, 501, 749, 143, 1059, 3557, 32509, 55535, 94795, 44933, 507409, 908465, 597235, 1067679, 6542033, 16097641, 7980035, 25029815, 23526641, 242279845, 337441181, 88775359, 617793267},
  {1, 3, 1, 11, 7, 43, 81, 207, 175, 191, 797, 1307, 733, 13661, 31345, 64019, 26249, 96029, 402981, 8311, 1567893, 1306071, 1689171, 16368503, 6675741, 59523347, 28070107, 202010867, 276878449, 295246551, 1823737625},
  {1, 3, 1, 1, 15, 27, 63, 255, 49, 771, 997, 1451, 6967, 12737, 19065, 36589, 120773, 88539, 23317, 240495, 342101, 2343357, 4750683, 14992111, 872787, 54427459, 23302525, 165859855, 434655761, 473018775, 1785546611},
  {1, 3, 5, 3, 27, 61, 105, 171, 305, 579, 1893, 2095, 7421, 2689, 22431, 12983, 131045, 129307, 158613, 268399, 1206865, 994751, 347487, 5997249, 29939989, 65537967, 20660581, 149700863, 104056561, 181520339, 743265081},
  {1, 1, 5, 3, 1, 3, 57, 249, 149, 539, 167, 2011, 485, 9083, 29573, 4483, 88751, 242555, 314813, 923373, 513009, 1319095, 5360325, 11796959, 7111725, 10255493, 126659873, 63970575, 446186787, 662160615, 1337643113},
  {1, 1, 3, 5, 5, 57, 15, 13, 159, 15, 1487, 3245, 6927, 12043, 28991, 36673, 74207, 191601, 472189, 204997, 1893111, 343057, 8030425, 12819837, 20119659, 65732857, 84082739, 117506795, 50529795, 655294977, 1764951555},
  {1, 1, 1, 11, 7, 11, 105, 141, 225, 83, 1431, 1311, 7877, 733, 4577, 35707, 50955, 175431, 433965, 603677, 2094517, 1067391, 3757011, 3267319, 13406245, 34410577, 29118837, 122941463, 127004995, 363576899, 1539640521},
  {1, 3, 3, 5, 27, 59, 121, 101, 271, 23, 581, 3249, 2847, 11469, 17001, 51475, 109547, 82513, 145517, 1026567, 117751, 3923161, 2049287, 16267039, 3718949, 6933977, 544859, 69169459, 266334953, 370728893, 372840955},
  {1, 3, 5, 9, 11, 49, 51, 59, 115, 863, 133, 2999, 7999, 7677, 16467, 58125, 31593, 18201, 384613, 896807, 1919313, 21461, 21263, 2297125, 20670799, 34019695, 119432463, 32450467, 294212473, 244568187, 479401243},
  {1, 1, 7, 1, 23, 45, 125, 71, 419, 127, 1815, 2369, 2379, 969, 26311, 59835, 120777, 254617, 400101, 145701, 1347923, 3490645, 1929243, 7344185, 30453841, 61903075, 49312039, 5747635, 72390219, 220044493, 1563582247},
  {1, 1, 3, 5, 23, 5, 105, 109, 75, 463, 679, 901, 2695, 11409, 7871, 40383, 53035, 137909, 174701, 465813, 1585527, 1728657, 5423115, 3391489, 6930437, 27344937, 123846815, 174342187, 502252827, 1009762921, 554353835},
  {1, 1, 7, 15, 7, 11, 67, 121, 453, 867, 1407, 3073, 5125, 1029, 25657, 5193, 87251, 261179, 318733, 237741, 978803, 1828347, 3145739, 11534375, 28311599, 57671845, 99614905, 89129527, 210765403, 678430237, 649076137},
  {1, 3, 7, 3, 9, 13, 31, 27, 449, 907, 1693, 3173, 5345, 691, 31731, 24577, 122929, 172075, 8581, 484719, 389075, 3650303, 8176645, 3345457, 4820043, 30557215, 107316525, 84504375, 224121961, 31916025, 1677729413},
  {1, 3, 1, 15, 19, 39, 39, 89, 15, 887, 1781, 419, 1121, 3369, 20637, 35781, 112427, 118593, 179693, 481351, 2053621, 918867, 5641223, 16143403, 6956155, 54936629, 12860835, 158602339, 332873913, 1048659439, 2040191813},
  {1, 1, 1, 1, 1, 33, 73, 145, 379, 563, 779, 3163, 1563, 7131, 9595, 2643, 47707, 143105, 327389, 284981, 925381, 1299653, 2971333, 131173, 28473101, 6786389, 7762207, 260370743, 529299367, 194239959, 1336674679},
  {1, 3, 1, 15, 15, 43, 29, 13, 483, 435, 1961, 2527, 3797, 537, 2573, 46627, 102079, 135761, 100909, 670583, 832325, 440515, 606155, 15295575, 8379233, 2051425, 39583959, 169607495, 76808933, 4983507, 2110003249},
  {1, 1, 7, 3, 19, 27, 85, 131, 431, 771, 1291, 781, 4901, 9021, 19877, 64475, 71121, 18833, 260629, 219893, 1113155, 670663, 5939535, 13021015, 5293145, 47234223, 19993915, 26736079, 429474679, 516345281, 1707363913},
  {1, 3, 3, 3, 5, 35, 23, 195, 349, 487, 1257, 3677, 365, 13499, 4049, 57961, 39525, 63331, 424325, 264735, 1792327, 57679, 5900361, 1733103, 3294275, 56534159, 82875841, 45125163, 140186845, 434339073, 992217865},
  {1, 3, 3, 7, 9, 27, 39, 59, 297, 775, 1521, 2869, 4881, 16375, 26265, 19393, 12005, 125615, 480837, 76655, 708871, 79627, 1890565, 16384263, 29533443, 43511639, 31589557, 224832747, 31934037, 599215145, 1215316437},
  {1, 1, 3, 9, 11, 17, 13, 241, 157, 983, 403, 977, 1311, 12969, 26615, 13183, 96091, 197935, 341469, 104717, 1347335, 3315981, 6588167, 10137397, 24079137, 35454901, 55471529, 46513427, 382235879, 30224781, 348853331},
  {1, 3, 7, 15, 25, 57, 33, 189, 213, 23, 337, 2193, 7697, 6455, 5979, 38787, 40623, 65439, 318285, 351119, 409859, 2967299, 3623685, 12030725, 12500805, 2092889, 96206161, 255770083, 430895365, 415103949, 15781461},
  {1, 1, 7, 1, 9, 55, 73, 83, 217, 535, 435, 3669, 6463, 11035, 17717, 10071, 26065, 13295, 438229, 647181, 336643, 327429, 1161493, 7646003, 31667565, 27345807, 56819965, 44368475, 451872215, 447416073, 1279596275},
  {1, 3, 3, 13, 19, 27, 23, 113, 249, 943, 1681, 1653, 6715, 15449, 24141, 35793, 1231, 261227, 388397, 915199, 1687303, 4050177, 7659807, 2979119, 28687195, 3829533, 31773093, 79499627, 206370421, 197572041, 315802495},
  {1, 3, 5, 3, 23, 43, 3, 253, 479, 995, 1121, 3939, 7337, 7409, 9061, 56693, 16283, 229893, 33661, 294919, 1475585, 2526223, 2204675, 12502055, 2266207, 36099153, 78605779, 247141399, 55899933, 817962223, 1063586093},
  {1, 1, 5, 5, 11, 5, 45, 117, 217, 543, 1763, 3943, 6511, 4833, 26931, 30983, 66959, 254867, 323021, 37245, 1661953, 338945, 4312095, 8270865, 26147841, 43502689, 90885229, 59408331, 264176007, 1045496603, 1631786723},
  {1, 3, 3, 7, 29, 37, 33, 123, 147, 243, 1153, 1413, 897, 10643, 26783, 42731, 28345, 221157, 265813, 494823, 344071, 770059, 3522577, 3653689, 6504533, 63062255, 108085271, 139575583, 278316749, 253347401, 992039301},
  {1, 3, 1, 15, 5, 5, 37, 227, 223, 459, 897, 3715, 1665, 11919, 25989, 31621, 29093, 150627, 277855, 290123, 1654785, 573443, 2605057, 8568847, 11550725, 66109445, 66076709, 93569251, 345063647, 38879691, 611926913},
  {1, 1, 7, 5, 5, 39, 63, 255, 135, 487, 1473, 321, 2631, 15941, 1093, 55399, 33279, 16959, 254919, 437543, 765953, 2469889, 1691655, 10604549, 24653829, 37228583, 54104127, 175845631, 377139335, 692990439, 1800385985},
  {1, 3, 1, 7, 9, 7, 87, 249, 217, 599, 993, 995, 7201, 4647, 14313, 14119, 2871, 230873, 421689, 754295, 1213441, 2395139, 1278977, 14523399, 30057481, 18848775, 98163799, 188245241, 126477529, 382918231, 862783457},
  {1, 1, 3, 13, 9, 47, 7, 225, 363, 247, 1121, 3425, 3747, 13805, 19817, 25743, 41127, 38657, 115019, 16727, 607233, 3326977, 5721091, 14382093, 30860297, 13466671, 87776263, 206980321, 206335339, 582720759, 104906849},
  {1, 3, 7, 13, 19, 13, 9, 67, 9, 737, 561, 1043, 2039, 5693, 14883, 28701, 53273, 159859, 225433, 1036369, 799489, 3895555, 6094599, 12314893, 16880915, 25233165, 43121929, 103980355, 110519561, 1008551393, 1962934577},
  {1, 3, 5, 5, 19, 59, 7, 41, 319, 677, 657, 2291, 1621, 1845, 21539, 35243, 88247, 101977, 195375, 422901, 1757441, 3258627, 1685253, 11091717, 12548883, 2438459, 1159943, 24170793, 466910271, 78976421, 1085916049},
  {1, 1, 5, 3, 31, 63, 15, 43, 207, 789, 1369, 633, 3245, 9931, 28631, 39255, 68183, 250515, 521847, 263597, 783809, 3560897, 1217989, 12887235, 892383, 49376127, 34435151, 254111851, 139817359, 7943125, 1459057305},
  {1, 1, 7, 9, 13, 39, 3, 47, 497, 169, 1017, 777, 6351, 8257, 30149, 42207, 66699, 68743, 144777, 71025, 1159489, 1505217, 7955143, 6644169, 11477581, 38476903, 95129667, 234467439, 333312049, 920367849, 910577849},
  {1, 3, 1, 7, 21, 17, 97, 19, 415, 905, 905, 1739, 5305, 10719, 15325, 40553, 7689, 34315, 178695, 129617, 2051649, 3097923, 5070657, 14478279, 8358101, 39952209, 15767329, 231735763, 22057183, 154186057, 1374713801},
  {1, 3, 7, 1, 3, 31, 71, 111, 165, 127, 713, 1819, 1391, 6009, 18443, 55319, 79935, 161895, 243821, 191415, 1195457, 888003, 1152199, 3456705, 644163, 59120351, 127577351, 203356079, 29360229, 734004031, 1497367817},
  {1, 1, 5, 11, 1, 61, 83, 119, 203, 847, 553, 3657, 6957, 12243, 5177, 24037, 27707, 146479, 118035, 204007, 1279809, 36417, 6587333, 6202059, 30972865, 47153917, 11087635, 264948535, 313195659, 961703055, 111489897},
  {1, 3, 3, 13, 9, 61, 19, 97, 47, 35, 1497, 75, 7435, 5973, 3681, 36149, 39723, 225177, 122359, 208203, 1509185, 3343427, 1382339, 2832205, 31023817, 45174781, 119184211, 203255713, 152296943, 10038371, 187760793},
  {1, 1, 7, 7, 15, 29, 63, 95, 417, 469, 1001, 729, 799, 8575, 27703, 47253, 37591, 244791, 17881, 595869, 91841, 2076225, 6368967, 13743303, 5623119, 12792925, 47089023, 61292831, 481404257, 758134037, 267764521},
  {1, 3, 1, 9, 25, 9, 71, 57, 213, 385, 725, 3975, 4821, 12437, 29901, 46557, 56051, 186573, 338329, 756533, 1404721, 1756371, 3201425, 7467865, 10924809, 56364345, 18733399, 120419753, 148729925, 808620561, 2130466725},
  {1, 3, 5, 13, 31, 47, 101, 57, 39, 341, 1501, 3351, 3505, 15505, 24475, 2219, 79817, 93653, 107715, 597697, 1592465, 4050259, 1989685, 12966941, 30972175, 27264287, 18887477, 6500841, 459522263, 463978661, 409814669},
  {1, 1, 3, 3, 31, 57, 125, 173, 365, 551, 1173, 1341, 3575, 2903, 32035, 41413, 130385, 218969, 484553, 216355, 1147153, 4030865, 888979, 13471283, 15244559, 8064297, 88624461, 63539101, 396293533, 1006656695, 1812763717},
  {1, 3, 7, 1, 13, 57, 67, 157, 451, 707, 1901, 303, 723, 8093, 6113, 50989, 44391, 96649, 458559, 783239, 1790577, 988307, 6451191, 12827761, 18322205, 25439497, 31290163, 209662861, 476298259, 203710035, 1773994077},
  {1, 1, 1, 7, 21, 13, 105, 89, 429, 965, 1237, 2293, 5557, 9651, 21665, 18081, 37517, 88765, 363321, 764497, 2092241, 94161, 5339025, 8091927, 4151557, 41879837, 88022393, 46052649, 113042941, 347686421, 1235437381},
  {1, 1, 5, 9, 17, 51, 45, 119, 157, 141, 1701, 2885, 5441, 9165, 7493, 44375, 97065, 28667, 480873, 267281, 455569, 1127089, 8135573, 11539993, 11538433, 59822115, 3166333, 173109479, 515281293, 234599357, 1081250933},
  {1, 3, 7, 7, 13, 45, 91, 9, 129, 741, 501, 1351, 1539, 4619, 4625, 44593, 109167, 68285, 13293, 487617, 1422833, 3010771, 1146839, 4214039, 28049437, 48496669, 105119787, 131334265, 471072849, 979107893, 1665926213},
  {1, 3, 7, 1, 23, 57, 67, 141, 151, 571, 973, 479, 5475, 15789, 18091, 47965, 97471, 92105, 59643, 211503, 1137905, 3927443, 1337527, 297969, 33332231, 31958025, 124275763, 163247261, 450371047, 593767851, 300708861},
  {1, 1, 3, 11, 17, 47, 93, 107, 375, 157, 389, 1741, 8191, 5143, 20613, 30859, 123377, 153455, 7147, 191401, 736849, 283697, 1360627, 6316379, 13770689, 43166079, 131513133, 124656027, 10371495, 754303405, 20529941},
  {1, 3, 3, 5, 11, 21, 43, 51, 169, 915, 189, 391, 6639, 369, 24335, 44841, 56607, 137071, 258333, 847959, 512561, 3636403, 1381907, 10034869, 1778779, 35987173, 108473691, 149522339, 46072281, 293666947, 1704920013},
  {1, 1, 5, 3, 15, 55, 101, 67, 455, 625, 1477, 1853, 217, 1823, 31811, 54211, 28113, 102679, 75227, 105573, 966097, 2224561, 6060533, 10572691, 801759, 33878631, 86367093, 94135603, 448647543, 770814145, 1618605397},
  {1, 3, 5, 9, 1, 23, 29, 47, 345, 595, 765, 3111, 2185, 10333, 3453, 59019, 34785, 154827, 345381, 372183, 1103185, 1897075, 495125, 16554937, 7989329, 38698727, 125972749, 58067839, 117019913, 675700323, 182207085},
  {1, 3, 7, 7, 5, 49, 29, 155, 323, 589, 1285, 3999, 1355, 5507, 21121, 57493, 111537, 190967, 28735, 299081, 249937, 1980723, 4222647, 3992663, 16104405, 18839105, 89899053, 265011243, 436158675, 265961629, 1061242261},
  {1, 3, 3, 7, 5, 41, 127, 61, 261, 717, 45, 1071, 2095, 10403, 12721, 15757, 97715, 52561, 404945, 143329, 404593, 2271571, 6536403, 12823383, 17286229, 46598233, 62311439, 41239693, 242697621, 855606685, 854704797},
  {1, 3, 7, 7, 17, 23, 117, 67, 129, 1009, 1261, 415, 4619, 14307, 19317, 49875, 92585, 259895, 291405, 763957, 1108177, 3819699, 1165207, 2941335, 20173633, 21089639, 33354821, 242150643, 156804177, 976267073, 200446973},
  {1, 1, 3, 13, 11, 39, 21, 207, 123, 305, 101, 1805, 5775, 6177, 22895, 28523, 91697, 138795, 383607, 512645, 644049, 3865969, 256051, 8773021, 8232411, 16097655, 42071141, 25320543, 528813067, 1048994817, 1268283381},
  {1, 1, 3, 9, 29, 3, 95, 47, 231, 73, 467, 359, 2829, 979, 19579, 14517, 75765, 222213, 460205, 818643, 943797, 91981, 6335191, 9432381, 27163457, 42894583, 80895323, 174408691, 134882699, 517562245, 1635141631},
  {1, 3, 1, 9, 1, 29, 117, 21, 441, 259, 1863, 3277, 475, 16383, 24767, 57507, 32243, 171955, 322335, 977405, 1653885, 432119, 7700981, 5476309, 6469589, 40030025, 127282977, 169610545, 63737149, 776209863, 1431094195},
  {1, 3, 1, 13, 21, 39, 125, 211, 439, 723, 1095, 1581, 1851, 11163, 9283, 36673, 104543, 149517, 227789, 989225, 1702709, 3396775, 1442005, 6438545, 3018561, 51838563, 89014649, 241618271, 76886099, 55612335, 361435403},
  {1, 1, 7, 3, 17, 63, 115, 89, 49, 773, 1195, 503, 7945, 16241, 23315, 41273, 71969, 241147, 332455, 666987, 1635997, 3946925, 1199731, 8868535, 3728861, 50314979, 40444679, 120162261, 46727213, 103874641, 1414725663},
  {1, 3, 7, 13, 11, 33, 101, 107, 63, 73, 1219, 3381, 8093, 6567, 4469, 49051, 115843, 246045, 457557, 961875, 553045, 2560591, 7124051, 7852593, 22172231, 27416037, 85859449, 263193855, 217326811, 697719989, 1830864047},
  {1, 1, 5, 5, 13, 57, 63, 135, 437, 177, 787, 983, 4987, 1379, 379, 63551, 55409, 108757, 395567, 858527, 892861, 2999677, 83441, 5270257, 25208521, 18988797, 48395291, 27567219, 36678873, 343354493, 1111490935},
  {1, 1, 3, 7, 27, 63, 93, 47, 417, 483, 1367, 2079, 237, 4269, 6641, 23661, 14359, 141457, 309587, 379477, 219213, 1995861, 7708727, 5278803, 7440327, 53194867, 126819393, 234161379, 399838629, 578688879, 966849731},
  {1, 1, 3, 1, 23, 29, 1, 191, 49, 23, 735, 1711, 7021, 6859, 5457, 29423, 81583, 165041, 151011, 673601, 685709, 1563141, 4230967, 3775253, 1624011, 41816393, 21631021, 63041691, 331870597, 71308155, 1624192971},
  {1, 1, 3, 15, 25, 55, 9, 101, 219, 607, 1551, 1839, 7565, 8517, 28355, 16353, 121787, 172963, 174797, 1026933, 387637, 2417877, 290775, 8738395, 21305413, 53341787, 78393373, 171839497, 335262127, 259666811, 2020398003},
  {1, 3, 1, 7, 7, 19, 51, 251, 393, 307, 575, 1973, 2931, 665, 22813, 24881, 33561, 81313, 298435, 817149, 1617997, 2577335, 3164021, 15656603, 18722067, 8070463, 62285631, 120469959, 203358261, 746389743, 1948321323},
  {1, 3, 3, 3, 25, 55, 17, 75, 337, 3, 1535, 405, 6929, 6713, 6627, 50009, 69387, 63781, 184755, 651293, 446989, 1471031, 4865783, 7157151, 9112325, 34940171, 47559493, 121380263, 208539013, 904279391, 1664808619},
  {1, 1, 1, 13, 25, 17, 65, 45, 479, 413, 579, 3247, 3143, 10059, 27591, 15191, 36039, 34603, 70977, 232535, 189381, 3649437, 5470677, 1319865, 1339405, 19181621, 99381301, 136671417, 477917515, 199405625, 61083951},
  {1, 1, 7, 7, 27, 49, 99, 161, 213, 727, 1623, 1623, 7553, 9869, 8585, 32155, 85229, 214563, 451227, 927865, 997677, 2112037, 3614451, 16616275, 30195735, 13025341, 90390655, 61358101, 398409961, 297255747, 1739998003},
  {1, 3, 5, 1, 23, 5, 43, 41, 251, 857, 1803, 1693, 4359, 7943, 11521, 31535, 95613, 154475, 58629, 490763, 1024853, 1239279, 2597649, 3458685, 2555923, 11337777, 56033319, 141099053, 362479911, 904070061, 850200151},
  {1, 3, 3, 7, 11, 61, 39, 87, 383, 835, 1967, 1269, 7841, 15533, 13097, 58775, 62353, 38157, 336173, 811777, 1926133, 745671, 3339927, 10974043, 7500823, 14863377, 20681787, 261293267, 186049875, 366220751, 1351631907},
  {1, 1, 3, 15, 13, 7, 29, 7, 505, 923, 1683, 3979, 7449, 9425, 25739, 40341, 36571, 64549, 181663, 232817, 879565, 1263509, 6932167, 6715723, 33372097, 18320507, 126288745, 175633755, 300400717, 338182471, 1785315639},
  {1, 3, 7, 1, 5, 31, 47, 157, 445, 501, 2015, 41, 617, 12215, 19471, 25629, 128105, 138491, 150575, 118279, 1271341, 3270575, 2419267, 2731725, 26844121, 21819867, 90012539, 66861857, 420479425, 319815873, 1504708459},
  {1, 1, 3, 7, 1, 43, 9, 147, 115, 605, 1047, 1147, 1737, 3609, 18615, 34753, 34359, 100057, 72253, 282531, 808837, 573213, 776647, 3039811, 30982989, 10321143, 89541965, 263762815, 193852823, 64720081, 192746139},
  {1, 3, 3, 13, 5, 1, 119, 211, 455, 1001, 815, 2521, 5165, 3219, 14583, 15235, 85197, 24357, 189721, 343535, 152757, 799215, 296775, 15107393, 17668681, 18380789, 12130595, 114945815, 522761691, 731245797, 838025843},
  {1, 1, 3, 5, 13, 19, 3, 243, 75, 843, 651, 11, 1417, 8843, 27279, 59025, 106125, 171549, 57269, 891973, 925445, 2759621, 5769287, 7879233, 8936513, 6834247, 82395255, 162119935, 506118279, 472859079, 975978727},
  {1, 3, 7, 7, 1, 19, 91, 249, 357, 589, 1087, 1369, 6297, 8705, 17943, 22057, 70213, 191047, 457231, 203871, 1414509, 1938447, 893507, 785611, 482013, 20427983, 26476575, 87294069, 217842121, 157548649, 1717830123},
  {1, 1, 1, 9, 1, 25, 109, 197, 279, 411, 1275, 2187, 3531, 11331, 3291, 59347, 48599, 36311, 62869, 546589, 372773, 1405445, 2618053, 15391437, 27438661, 12495997, 115781673, 245772513, 325480675, 422473935, 1211257063},
  {1, 3, 1, 15, 23, 57, 59, 135, 191, 75, 1947, 4009, 7311, 837, 15697, 9311, 23153, 141961, 320057, 823597, 2068725, 448807, 4931909, 15628611, 30427987, 28372549, 5361703, 240234659, 47206419, 194924951, 1678954663},
  {1, 1, 5, 15, 29, 21, 39, 253, 383, 349, 407, 875, 735, 13357, 4075, 53543, 47957, 8699, 234189, 857659, 359781, 2344541, 806465, 16509387, 28186185, 1222121, 9380883, 146922729, 38249635, 768580969, 173737771},
  {1, 3, 3, 5, 19, 45, 61, 151, 199, 981, 903, 3401, 3293, 2923, 27681, 58483, 9455, 126293, 126561, 65555, 460405, 2165279, 5839559, 12265033, 22771551, 6377977, 69467185, 69637347, 476890355, 578260169, 889393243},
  {1, 3, 5, 13, 9, 61, 107, 141, 141, 1, 479, 593, 83, 15255, 28003, 2495, 88993, 204971, 106791, 483691, 763469, 237791, 8004353, 15597313, 22044957, 17885961, 124905295, 235973497, 93490209, 735780149, 322388715},
  {1, 3, 1, 11, 27, 25, 85, 105, 309, 979, 1275, 1873, 1175, 13305, 23677, 27007, 33879, 21651, 238967, 556425, 1482733, 2085079, 7507205, 5025543, 15197983, 31146805, 89351993, 186136845, 258596193, 130164087, 919844143},
  {1, 3, 3, 11, 19, 7, 115, 223, 349, 43, 1019, 3601, 6581, 9373, 16725, 46289, 112781, 26313, 262803, 392097, 1174493, 3555927, 49415, 6055175, 28514591, 58838315, 125691711, 24626627, 413454737, 971542615, 1102919055},
  {1, 1, 7, 9, 21, 39, 123, 21, 275, 927, 939, 2467, 4965, 4647, 18983, 55725, 122773, 157059, 172441, 762265, 1401845, 2368021, 1582083, 1813517, 18992137, 63947779, 24847407, 169073801, 343291135, 959797195, 1021658087},
  {1, 1, 7, 13, 15, 41, 47, 243, 303, 437, 607, 387, 1669, 12675, 17301, 45495, 29949, 133421, 275273, 952171, 1470133, 4079037, 3653635, 8994825, 32227347, 15515677, 115163155, 16040023, 307839379, 379699833, 1290487523},
  {1, 1, 1, 7, 7, 3, 15, 99, 409, 719, 1587, 2819, 2819, 5893, 11017, 22285, 113929, 211837, 216671, 849661, 373933, 693861, 6225925, 5439491, 27852803, 54460447, 117637139, 234029167, 290652581, 590676803, 779681879},
  {1, 3, 3, 15, 27, 49, 113, 123, 113, 67, 469, 3585, 1539, 11779, 6671, 14875, 97841, 223857, 220795, 8817, 1972803, 176085, 2883585, 262147, 17039363, 36438031, 117702683, 223084593, 454295665, 650379387, 72089713},
  {1, 3, 7, 11, 3, 23, 87, 169, 119, 483, 199, 2945, 6531, 14727, 11915, 53635, 18327, 37335, 168489, 621559, 509027, 819015, 2211841, 5554179, 10338311, 66338827, 34390019, 113098775, 437174359, 773111977, 1717158007},
  {1, 1, 5, 15, 7, 17, 109, 229, 179, 213, 741, 1857, 65, 15429, 9679, 44103, 79441, 129325, 463269, 957683, 265749, 1931813, 5459969, 2166785, 25628677, 29446159, 128577543, 127234065, 55840877, 398971109, 1447383219},
  {1, 1, 5, 13, 11, 17, 25, 135, 403, 557, 1433, 2881, 8001, 3781, 14285, 26187, 95057, 201561, 142279, 1006547, 1399405, 333913, 2920449, 3608577, 21172229, 60641293, 40488971, 30519313, 407900185, 859082887, 213217683},
  {1, 3, 1, 1, 1, 61, 67, 215, 189, 945, 1243, 3681, 1251, 6561, 33, 57697, 34333, 41507, 6903, 171677, 919953, 417147, 4688897, 4359171, 22840321, 55149569, 77362177, 243043389, 135781443, 1026454743, 1230652605},
  {1, 1, 7, 13, 17, 33, 9, 221, 429, 217, 1679, 1377, 3553, 3815, 23661, 60337, 9153, 252009, 194685, 408013, 1481145, 2060015, 7304193, 16616449, 25142279, 34649101, 26858513, 72004641, 290317321, 531735773, 1985033645},
  {1, 1, 3, 11, 27, 3, 15, 93, 93, 865, 1049, 3873, 5729, 13027, 9579, 40315, 16483, 116335, 519805, 184317, 2079937, 2317497, 3412993, 7707649, 17808387, 49273867, 82000923, 59378691, 136037391, 1016923229, 1322308701},
  {1, 3, 7, 7, 25, 41, 121, 35, 373, 379, 1547, 481, 5795, 8935, 23463, 60729, 19273, 156121, 7299, 210389, 1172507, 3123947, 2587649, 740355, 4615175, 2282503, 125801497, 95237161, 208624761, 423920675, 164240757},
  {1, 3, 3, 9, 11, 35, 45, 205, 241, 9, 59, 1505, 7267, 9443, 4841, 25771, 125763, 48269, 320493, 465297, 432553, 2066715, 1276929, 5430275, 16929795, 48094217, 28335115, 208749603, 296928301, 431445197, 59880689},
  {1, 3, 1, 7, 3, 51, 7, 177, 53, 975, 89, 1553, 5715, 497, 23703, 50707, 109059, 229911, 493249, 748037, 2009855, 1556009, 1545985, 8065795, 29557505, 55121671, 130793731, 243729715, 140377863, 575351217, 520880437},
  {1, 1, 3, 5, 27, 1, 113, 231, 299, 759, 861, 3569, 4145, 14835, 27733, 28811, 114321, 257985, 191287, 62235, 739431, 3301325, 693505, 8997633, 7196931, 38764805, 76248859, 85187329, 23668593, 544804327, 1865760811},
  {1, 3, 3, 15, 25, 29, 5, 255, 139, 891, 2031, 3697, 5427, 15923, 8831, 60297, 92077, 110005, 119439, 801435, 316715, 1641023, 1163009, 9324803, 22244611, 10941711, 122507033, 144399133, 255191813, 745504767, 1658274187},
  {1, 3, 1, 1, 13, 9, 109, 193, 419, 95, 17, 1873, 5235, 13489, 3281, 52189, 43129, 108605, 247825, 776499, 2046095, 1854849, 2146561, 12569859, 805121, 50081025, 5505293, 41280265, 445390701, 37704129, 2018578595},
  {1, 1, 7, 9, 3, 7, 29, 41, 135, 839, 867, 2769, 1201, 2231, 19321, 10067, 67671, 222893, 113529, 407799, 1850359, 72179, 1870081, 1803009, 5998343, 52010249, 81832195, 127928071, 421578525, 887612201, 1693143431},
  {1, 1, 7, 9, 25, 49, 123, 217, 113, 909, 215, 2545, 817, 10743, 18905, 42729, 43521, 76331, 90793, 440001, 1153469, 592711, 6954241, 13382913, 4220167, 34115337, 53062937, 67857713, 229513595, 326321113, 2018964849},
  {1, 1, 7, 3, 23, 15, 43, 133, 217, 327, 901, 3761, 4785, 8919, 18995, 20263, 9919, 57851, 231893, 140361, 1105367, 3406421, 7780097, 14007041, 3093255, 45164291, 41904407, 181822735, 112608043, 165851525, 170892761},
  {1, 1, 3, 3, 13, 53, 63, 123, 477, 711, 1387, 1297, 5489, 4371, 24851, 9661, 67525, 197487, 178027, 316589, 1750519, 3341435, 2254593, 7397633, 18127107, 38054659, 66670349, 130203957, 265554751, 257441659, 344354013},
  {1, 1, 3, 15, 7, 29, 75, 119, 181, 957, 247, 849, 3441, 6835, 2975, 7863, 109293, 147227, 435623, 820965, 199245, 73511, 6105857, 8743681, 13598979, 2291471, 118743815, 190183709, 427688267, 1000407927, 749487541},
  {1, 1, 1, 11, 27, 25, 109, 151, 267, 99, 1461, 1425, 3601, 16273, 19707, 56427, 84329, 133149, 366663, 522651, 522323, 3853957, 1240833, 1851137, 6329601, 28549387, 30888731, 146693913, 15654765, 432552343, 31538187},
  {1, 3, 7, 15, 5, 5, 53, 145, 11, 725, 1501, 2673, 6995, 8439, 16191, 61045, 126165, 43493, 336833, 489211, 2031461, 740845, 5858049, 11270403, 12998919, 34013455, 115057925, 153091845, 21426485, 705761681, 1679541515},
  {1, 3, 7, 1, 9, 43, 71, 229, 157, 607, 1835, 1001, 507, 14431, 1609, 11473, 118227, 93967, 383853, 473349, 2075431, 2895171, 5272769, 13812675, 20437831, 4538689, 16107977, 190730475, 411709447, 973860901, 1796460893},
  {1, 3, 3, 13, 25, 1, 5, 27, 471, 349, 127, 1593, 7787, 11643, 12357, 54465, 121177, 129453, 147091, 668319, 1205685, 1231463, 7575105, 3746627, 29385923, 5433293, 85175641, 236739009, 214224965, 678072539, 1954509079},
  {1, 1, 1, 1, 23, 37, 9, 221, 269, 897, 1685, 2857, 5193, 9705, 22537, 59423, 6189, 153601, 243813, 849957, 957385, 3930237, 7560001, 15765569, 30312769, 39098689, 131228247, 121368421, 88080457, 37748893, 1514144077},
  {1, 1, 3, 3, 31, 29, 51, 19, 311, 553, 1969, 1529, 2697, 15435, 18699, 13591, 119045, 34603, 271339, 907999, 899249, 2989609, 8147265, 2972353, 5728963, 39520195, 126852191, 177207901, 72155251, 1001324627, 1395589623},
  {1, 3, 7, 5, 5, 55, 17, 39, 475, 671, 1529, 153, 2619, 13711, 19597, 51357, 113295, 144825, 469903, 104163, 1141911, 3480385, 6747073, 9715267, 6034375, 62130373, 129250757, 59097207, 50085969, 567034087, 1858256923},
  {1, 1, 7, 1, 1, 35, 47, 27, 437, 395, 1635, 745, 7449, 15199, 5769, 7561, 7323, 42663, 226195, 211245, 633459, 2476347, 6772417, 11729345, 6075591, 40303425, 78859457, 175644387, 317407343, 395133019, 681558133},
  {1, 1, 7, 3, 13, 23, 43, 135, 327, 139, 389, 2393, 6841, 1743, 28075, 56165, 43855, 257107, 207951, 121951, 1238195, 3493917, 3422145, 16525505, 1163207, 29787075, 4722893, 84495191, 224971883, 1021267143, 1960393863},
  {1, 3, 7, 3, 9, 25, 91, 25, 429, 219, 513, 3337, 5371, 1199, 12331, 46833, 57665, 73059, 519025, 94149, 792291, 3659433, 7388225, 9233603, 22507591, 43726147, 12696777, 46736985, 28955675, 610870489, 1055693933},
  {1, 1, 3, 5, 13, 29, 119, 201, 277, 157, 2043, 2249, 6985, 16123, 6509, 51109, 18533, 255935, 249345, 287517, 1858629, 2364883, 8027457, 6636481, 27849795, 24384325, 61130061, 241518813, 171709495, 192267401, 1276462549},
  {1, 3, 5, 3, 29, 57, 13, 17, 167, 739, 1031, 3353, 6331, 12125, 23851, 2981, 62321, 78517, 338201, 249551, 62059, 2274927, 7008321, 11016515, 17312325, 36202435, 105422557, 100208633, 148485197, 586755281, 1233419751},
  {1, 3, 3, 5, 29, 21, 95, 27, 255, 679, 1531, 985, 7963, 5499, 9325, 11749, 76205, 155543, 445971, 685943, 1454207, 2798339, 2520897, 12999107, 12402883, 60537285, 102415325, 10620117, 286526495, 444308699, 829144127},
  {1, 3, 7, 15, 9, 5, 21, 71, 61, 961, 1201, 137, 5931, 4975, 8871, 41905, 57501, 88205, 476271, 300149, 46537, 1332649, 4512577, 3624771, 9533383, 6515023, 117159497, 13864517, 486767701, 822252679, 956786173},
  {1, 3, 5, 13, 15, 57, 33, 93, 459, 867, 223, 185, 7547, 861, 13029, 20087, 112305, 192681, 444165, 479523, 1520139, 3651223, 6975553, 13307459, 6628677, 36766925, 21354191, 79311353, 480597089, 869878941, 2086956171},
  {1, 1, 1, 15, 17, 43, 127, 191, 67, 177, 1073, 1513, 3513, 7833, 20567, 5961, 35699, 159575, 84871, 389963, 950585, 1737081, 2264001, 3972289, 25793217, 40076751, 22792529, 40047979, 500418879, 829835775, 322924803},
  {1, 1, 1, 15, 23, 7, 21, 199, 75, 293, 1611, 3881, 2201, 15625, 8951, 65359, 51423, 49213, 442575, 835907, 803389, 1623267, 639169, 10764353, 5762241, 41512399, 103235671, 217938887, 511118677, 334099335, 1792602379},
  {1, 3, 7, 13, 15, 39, 21, 149, 65, 741, 319, 2969, 7547, 10735, 7797, 39431, 129583, 37389, 457277, 921481, 1180221, 1442503, 5639873, 8524867, 14815303, 39460685, 3334735, 153503975, 67665237, 492308309, 485228673},
  {1, 3, 7, 11, 23, 13, 101, 89, 277, 519, 711, 889, 299, 15119, 22003, 37919, 46085, 181325, 373841, 818189, 2096639, 754271, 2982465, 12557379, 9085255, 63487563, 71769815, 267938509, 212483877, 680173465, 1081082325},
  {1, 3, 7, 15, 19, 27, 85, 203, 441, 97, 1895, 2361, 2907, 1647, 6519, 24219, 3731, 235997, 267811, 351185, 1122041, 4153839, 1565633, 3685443, 1993671, 54958415, 12004691, 17357531, 206174485, 737865483, 629719417},
  {1, 3, 1, 3, 29, 25, 21, 155, 11, 191, 197, 3401, 3115, 12985, 12283, 2933, 87105, 66365, 96675, 836659, 19943, 1715245, 7227201, 1400387, 7456193, 19129795, 34070237, 55561305, 417250133, 880767323, 1295887691},
  {1, 1, 7, 5, 27, 11, 81, 101, 457, 675, 1687, 697, 5033, 15407, 25437, 15779, 19523, 118249, 310989, 587713, 1424187, 630671, 6138305, 3384641, 21120071, 4969797, 73158619, 47531723, 252985105, 212736805, 1507458313},
  {1, 3, 1, 5, 25, 5, 65, 193, 41, 567, 781, 3641, 2379, 13433, 12221, 4017, 104733, 182041, 277881, 231473, 555087, 3647669, 7865921, 5768003, 33032001, 51921221, 72928985, 202991685, 57273089, 190656769, 1264401769},
  {1, 3, 1, 5, 11, 15, 113, 77, 411, 695, 1111, 1641, 2155, 7017, 31389, 18787, 129175, 87209, 445413, 542067, 468783, 1630031, 8037569, 1490243, 17263169, 46705093, 116351691, 77624527, 388217649, 451117965, 929397979},
  {1, 1, 3, 9, 11, 53, 119, 171, 55, 297, 509, 3237, 4085, 10903, 30685, 62039, 30529, 28971, 296367, 31899, 1793941, 924561, 547569, 4704689, 24508755, 44153113, 38121243, 154966821, 178230599, 14389563, 698020231},
  {1, 1, 1, 1, 11, 39, 113, 139, 165, 347, 595, 2925, 1869, 14261, 30517, 63711, 14067, 199949, 303431, 579217, 510535, 1240599, 3544177, 4286737, 21537905, 20111729, 69551899, 692535, 198592353, 166336411, 1393586453},
  {1, 3, 7, 11, 9, 17, 101, 13, 81, 325, 1733, 1429, 6039, 12499, 9431, 59333, 77261, 254449, 479673, 249877, 1315361, 4018449, 5331857, 1851731, 27288087, 23089435, 81819929, 157340961, 258203925, 954487229, 2103759297},
  {1, 3, 1, 1, 21, 43, 115, 9, 113, 907, 645, 3757, 1431, 8501, 19677, 62017, 113887, 127479, 457717, 233005, 685279, 3625649, 6815569, 11936467, 25824753, 12217105, 60699909, 130247963, 490970979, 264433945, 1556647969},
  {1, 1, 7, 3, 9, 25, 117, 197, 159, 471, 475, 2589, 6957, 14387, 9847, 6149, 47109, 38945, 252049, 166363, 1272659, 3853559, 1390865, 806321, 31155511, 20512627, 11526425, 219777801, 274400005, 327084021, 918552847},
  {1, 3, 1, 9, 11, 21, 57, 207, 485, 613, 1661, 4021, 4887, 8053, 29173, 9599, 97377, 88613, 400107, 772817, 1898793, 2898969, 5567153, 13531411, 28133009, 41106841, 67350299, 58328357, 535348521, 615357791, 996889685},
  {1, 1, 7, 7, 27, 55, 49, 223, 89, 85, 1523, 693, 2157, 3635, 1363, 19447, 34587, 67341, 473491, 389837, 1467449, 3819623, 5053745, 831697, 3530647, 61368951, 5803787, 229132583, 462030145, 184370095, 968959209},
  {1, 1, 5, 3, 19, 41, 45, 51, 447, 299, 1355, 2637, 1293, 13585, 14743, 50935, 27669, 136209, 54279, 984155, 861463, 1407047, 7128177, 5755313, 3136885, 3819635, 59925251, 98774841, 470653821, 236059395, 1116806543},
  {1, 3, 1, 13, 1, 33, 117, 143, 313, 187, 1073, 397, 5607, 4437, 21553, 63029, 389, 3201, 514683, 808669, 800311, 1451813, 1207217, 16329171, 12097393, 3651165, 90930961, 224102673, 156470117, 1039155551, 30989353},
  {1, 1, 7, 7, 5, 11, 65, 97, 377, 377, 1501, 2181, 4501, 4339, 31923, 64937, 118439, 68741, 311389, 671821, 508781, 3065545, 7127505, 5036241, 912087, 62503063, 120713493, 44615963, 287861041, 1027345, 1791603753},
  {1, 3, 1, 1, 21, 35, 95, 65, 99, 23, 1239, 3661, 3951, 6725, 16077, 39889, 104423, 128331, 496909, 319071, 1771603, 3428123, 5479889, 10106099, 18571825, 66435761, 107008005, 98635795, 237137999, 576761937, 283955507},
  {1, 1, 5, 9, 3, 37, 95, 167, 115, 425, 867, 1365, 4157, 2113, 27981, 21975, 84545, 170643, 290803, 307663, 815861, 1496047, 7828017, 3443697, 1420885, 14872057, 131567635, 212414517, 29052943, 608759863, 1146843203},
  {1, 3, 3, 13, 1, 37, 27, 189, 81, 679, 773, 717, 6279, 5063, 13761, 51277, 4049, 82655, 108137, 230269, 499859, 3809921, 6612881, 11582163, 4823091, 21018397, 63492113, 224923669, 471486507, 730443885, 402903105},
  {1, 1, 3, 11, 1, 61, 99, 233, 429, 969, 49, 1269, 4309, 8967, 3343, 2829, 33553, 51047, 34589, 607009, 1969005, 1593221, 7499473, 13922065, 20117459, 29880155, 129302545, 172032045, 130089043, 490930265, 709558717},
  {1, 1, 1, 7, 25, 63, 99, 165, 245, 793, 1143, 757, 4941, 14341, 2051, 38941, 34851, 247815, 325721, 104825, 211341, 823203, 3405457, 3922577, 11168721, 27897143, 54525961, 213909551, 96469107, 171966677, 977273189},
  {1, 1, 5, 11, 11, 43, 55, 65, 71, 283, 273, 829, 1141, 4737, 17807, 59551, 7303, 160155, 507245, 491803, 754591, 2457741, 3049937, 1416881, 22523637, 29279195, 27279387, 35733563, 132300903, 581091569, 1219150071},
  {1, 1, 5, 5, 9, 3, 101, 251, 355, 379, 1611, 3101, 5597, 4481, 14465, 60061, 97943, 229057, 9591, 676727, 163607, 2204231, 6465137, 10271345, 21132917, 30913909, 121389081, 209534995, 268255285, 1059471531, 1126482419},
  {1, 1, 1, 15, 21, 63, 85, 99, 49, 749, 1335, 1957, 4653, 16101, 29163, 15793, 22883, 68833, 309119, 730437, 1552257, 157139, 6973681, 7549265, 12449553, 41498687, 41630725, 65573935, 323734597, 5964947, 1103754593},
  {1, 1, 5, 13, 27, 9, 121, 43, 255, 715, 289, 61, 6037, 11297, 28585, 52719, 41181, 247541, 520303, 741755, 2052295, 3675069, 7617521, 15056049, 21509493, 46674653, 72934411, 184034329, 223964201, 769275, 2028793167},
  {1, 3, 1, 5, 27, 19, 17, 223, 77, 571, 1415, 3837, 3695, 15653, 3305, 8831, 48679, 236061, 358067, 921257, 456743, 1038547, 7032369, 13029779, 651633, 55000117, 45308939, 68236323, 309365761, 819291279, 324623869},
  {1, 1, 5, 3, 13, 59, 125, 251, 195, 551, 1737, 845, 53, 3297, 16231, 53945, 15383, 62569, 312503, 685399, 650347, 2248357, 4614257, 1423473, 16457141, 66025299, 49355805, 259384363, 518986797, 659496139, 1633709075},
  {1, 3, 3, 15, 13, 27, 49, 105, 389, 971, 755, 2013, 1919, 2471, 26211, 23905, 38023, 1765, 293349, 327201, 1053519, 3339527, 3305809, 8064787, 462835, 15794559, 32840733, 36344875, 487887873, 594248857, 1855077717},
  {1, 3, 5, 15, 23, 43, 35, 107, 447, 763, 253, 893, 4903, 8673, 16739, 45475, 94647, 21151, 377575, 458579, 1148215, 1558241, 180593, 9159219, 32397429, 44182815, 133880839, 57048091, 524262515, 466177179, 440982735},
  {1, 3, 5, 11, 21, 3, 17, 39, 497, 407, 611, 3541, 2359, 8265, 18047, 27785, 18359, 234957, 296627, 376701, 753971, 1476111, 4818673, 13534355, 25804789, 49790907, 36185029, 200919795, 296370945, 208330583, 446546145},
  {1, 1, 7, 13, 15, 31, 113, 17, 23, 507, 1995, 597, 6133, 4107, 21617, 38539, 81539, 166613, 462149, 78427, 1068871, 1243023, 947345, 14717169, 17868087, 64548413, 16750815, 253881551, 421689153, 300015489, 442613799},
  {1, 1, 7, 15, 3, 15, 31, 153, 423, 79, 503, 1413, 2597, 11819, 12339, 8711, 126491, 68123, 41549, 226195, 2077731, 2063443, 5792273, 2523121, 7213431, 59400703, 2255315, 233974751, 456479535, 536355625, 199491927},
  {1, 1, 7, 9, 19, 25, 23, 171, 505, 923, 1989, 613, 7821, 2811, 18133, 14807, 123989, 245283, 160391, 886685, 212063, 4140745, 7089457, 2580753, 4650039, 58063833, 7591107, 7213897, 180625191, 408947067, 1252880393},
  {1, 1, 5, 9, 21, 27, 121, 223, 133, 87, 697, 1589, 157, 15193, 11797, 22489, 122951, 71541, 63947, 259241, 1695251, 4038021, 4218225, 7276497, 7668725, 29827609, 46744517, 241552459, 514806633, 303023375, 1625862549},
  {1, 1, 5, 5, 9, 19, 107, 99, 319, 765, 1461, 1901, 1373, 16377, 22969, 11173, 60575, 102551, 395287, 256187, 408249, 1039361, 971441, 15026801, 29837909, 32078549, 9566553, 253885251, 488658299, 816307571, 803726831},
  {1, 1, 3, 3, 19, 25, 3, 101, 171, 729, 187, 349, 3973, 6351, 14559, 37223, 25021, 75463, 469833, 1043647, 1840165, 1227951, 2609105, 12666417, 28568595, 52197683, 11796547, 225268937, 194318707, 305914645, 239375451},
  {1, 1, 3, 1, 13, 23, 85, 93, 291, 209, 37, 3965, 829, 15679, 7869, 19769, 102203, 249209, 426265, 483647, 1499869, 2826993, 5958481, 5829041, 22994835, 17876177, 92319325, 92308295, 142674213, 595959053, 1636152371},
  {1, 1, 1, 15, 25, 25, 77, 253, 333, 947, 1073, 133, 5501, 13245, 14899, 36517, 128797, 256353, 268161, 108481, 1980319, 4130333, 4393809, 15405137, 11212209, 60114335, 44787401, 98780745, 109595421, 327092557, 2063416733},
  {1, 1, 3, 9, 17, 29, 55, 47, 255, 305, 2037, 3229, 1149, 9215, 7925, 63717, 113873, 64235, 478947, 868027, 1244677, 3177313, 7437937, 10452337, 14441619, 57179353, 50885185, 168358733, 91801927, 330878463, 380773423},
  {1, 3, 3, 9, 29, 63, 9, 103, 489, 939, 1523, 1029, 7991, 8815, 6365, 17849, 30547, 170661, 412099, 808581, 2021615, 4135135, 5743857, 6368531, 20118067, 5454009, 119269581, 83681999, 213885817, 344165175, 1670286841},
  {1, 3, 7, 15, 7, 31, 89, 175, 369, 339, 595, 2213, 3831, 12395, 16475, 13619, 97483, 94573, 13627, 705469, 1113847, 1523807, 1822065, 16538771, 28147127, 34657983, 27211479, 179346159, 181476201, 376548255, 1257352641},
  {1, 3, 7, 13, 25, 5, 71, 207, 251, 367, 665, 1173, 4271, 11883, 18825, 10141, 128393, 50891, 342163, 628815, 375787, 2361741, 3367537, 9153331, 28054391, 13385757, 113004233, 2924405, 129384567, 1027817695, 1716404395},
  {1, 3, 3, 3, 21, 25, 75, 35, 31, 321, 1603, 205, 5111, 10303, 10887, 56449, 77709, 209031, 513391, 870267, 1366805, 2206631, 2568881, 1562483, 1599379, 40559539, 119767365, 15590889, 475447355, 908017875, 442777999},
  {1, 1, 1, 9, 11, 1, 65, 5, 11, 329, 535, 3245, 3621, 14877, 6661, 2567, 61997, 20005, 289369, 288519, 254293, 1646611, 3337169, 14532145, 27487761, 29271545, 48623963, 23306449, 52166673, 586416341, 1800667387},
  {1, 1, 5, 3, 19, 13, 17, 43, 379, 485, 383, 1405, 5605, 1753, 31759, 58383, 21545, 187461, 105607, 279895, 1273281, 3015115, 2231601, 1706801, 6165717, 1985651, 53634115, 185063133, 11534337, 256901211, 451936523},
  {1, 3, 5, 13, 13, 9, 85, 147, 489, 787, 1133, 2933, 6655, 5193, 7113, 38977, 105173, 66505, 269727, 320821, 68919, 2907025, 751025, 8826899, 4166069, 20229469, 9181661, 39174265, 438243397, 585945219, 221032569},
  {1, 3, 1, 1, 5, 51, 37, 129, 195, 297, 1783, 749, 4199, 6749, 13253, 16729, 15615, 21625, 130213, 349583, 1052229, 689331, 3702929, 7966643, 24936977, 29606385, 21933397, 43681987, 410316917, 615419985, 2106806483},
  {1, 1, 3, 15, 19, 57, 59, 181, 455, 697, 2033, 2381, 317, 1775, 11203, 36951, 85597, 144719, 50569, 135459, 635765, 2609797, 5102801, 1541873, 31496211, 22847391, 11200835, 264920425, 182571083, 1042345989, 1047302199},
  {1, 3, 7, 1, 27, 9, 65, 145, 325, 189, 201, 237, 959, 6027, 23461, 7487, 63893, 70597, 1293, 254121, 189201, 1075189, 6974609, 6875347, 20804439, 6581265, 66344011, 127257977, 407123057, 810722369, 454837557},
  {1, 3, 1, 15, 31, 23, 19, 5, 485, 581, 539, 141, 3127, 8349, 18475, 64483, 50435, 124279, 113537, 503281, 1889833, 260679, 6836177, 14570963, 9825745, 50881343, 83376719, 1679847, 232309827, 556467253, 1349805141},
  {1, 1, 7, 13, 11, 15, 65, 83, 185, 847, 831, 2917, 2549, 5691, 14433, 4991, 25379, 31013, 187863, 585573, 178795, 4112291, 2744497, 2616337, 11318391, 41859709, 494299, 80550111, 305077361, 413291715, 560033225},
  {1, 3, 5, 7, 7, 55, 73, 15, 303, 511, 1905, 1173, 4463, 10761, 10275, 36331, 76515, 193741, 288267, 375699, 1323563, 1552661, 1726321, 15078771, 13778805, 16385175, 118115031, 133629255, 29760601, 374512831, 834149407},
  {1, 3, 5, 9, 7, 21, 45, 15, 397, 385, 597, 381, 6703, 3625, 23149, 47595, 80889, 16345, 190339, 247665, 828917, 3802361, 395121, 4070995, 33161877, 43415257, 88522071, 134184549, 312384573, 404313183, 1637016765},
  {1, 3, 7, 3, 23, 13, 73, 221, 511, 883, 1265, 3811, 3781, 9357, 17433, 55589, 65863, 19911, 360019, 404057, 708289, 2953331, 5389061, 10043855, 519187, 17172287, 96383067, 74128625, 361742629, 894530329, 586548587},
  {1, 1, 3, 11, 1, 51, 73, 185, 33, 975, 1441, 3651, 903, 11789, 11025, 29987, 8565, 100259, 325887, 915679, 1739633, 3217475, 6761925, 15819021, 5879575, 3048511, 67361885, 218435791, 390885693, 223691653, 1887225045},
  {1, 3, 3, 9, 19, 59, 21, 39, 339, 37, 143, 2463, 1469, 8953, 2939, 101, 39281, 129143, 428105, 837697, 817447, 3919425, 3155101, 96519, 24138359, 467989, 14748111, 268292207, 27780905, 349634971, 2138627575},
  {1, 1, 7, 1, 31, 33, 19, 167, 117, 635, 639, 2175, 2431, 14457, 1651, 52089, 54355, 565, 417357, 968303, 136997, 2070197, 4856981, 9673429, 29274707, 48671573, 33425747, 103279989, 91743583, 1035269491, 972313865},
  {1, 1, 1, 3, 5, 13, 59, 83, 355, 349, 1967, 1215, 3999, 8415, 21917, 34143, 70099, 153085, 475593, 183505, 1603231, 566385, 4417149, 11709893, 21769717, 11275031, 99367889, 13899857, 429689727, 987292951, 1603309727},
  {1, 1, 1, 5, 19, 3, 53, 133, 97, 863, 983, 2511, 1663, 10655, 2203, 9221, 105513, 11335, 56363, 640151, 1037329, 3905429, 6020165, 9753173, 23784725, 18525521, 127049799, 106690887, 163101993, 502909337, 204472517},
  {1, 3, 1, 13, 9, 41, 91, 105, 173, 97, 625, 3267, 5989, 13355, 22587, 25375, 84791, 22357, 32739, 974283, 1736663, 1176575, 1604445, 2249727, 676213, 66487281, 84355549, 145075181, 170220335, 888132573, 318701585},
  {1, 1, 5, 3, 7, 49, 115, 133, 71, 231, 1063, 1519, 831, 12891, 12053, 15501, 129675, 95941, 240175, 924665, 2026101, 2927725, 3466709, 7138741, 6681425, 13628119, 45314371, 229298029, 146311551, 968608049, 1325121979},
  {1, 1, 7, 5, 17, 43, 47, 45, 497, 547, 757, 315, 8039, 1785, 23719, 31983, 3461, 192181, 342327, 515951, 1837741, 4168447, 2657717, 7243005, 30651347, 44082929, 32766045, 252835055, 466701403, 208743569, 1175750045},
  {1, 3, 5, 15, 21, 61, 123, 191, 249, 31, 631, 2079, 4781, 12719, 18209, 6311, 2723, 51405, 158773, 689635, 521449, 2191477, 6550181, 14837463, 3959185, 10376787, 95064913, 262388337, 489923647, 26319995, 1851602181},
  {1, 3, 7, 9, 17, 7, 11, 185, 127, 169, 1951, 1583, 7725, 3405, 15843, 42151, 104729, 158569, 96731, 642361, 1820619, 1560857, 3750957, 10330983, 6695347, 21111957, 40254941, 123923283, 107740287, 522155253, 1481743491},
  {1, 1, 5, 13, 11, 11, 9, 49, 29, 125, 791, 447, 4143, 13323, 3307, 18861, 118305, 108155, 112375, 1000355, 523179, 3778929, 869805, 1132741, 13891505, 33658905, 129362383, 170394479, 162122805, 1022966925, 1655823817},
  {1, 1, 1, 15, 31, 41, 13, 167, 273, 429, 57, 131, 3479, 6271, 18401, 19053, 101379, 91259, 44225, 740547, 237211, 3150655, 3988333, 10641917, 1322677, 54279419, 2407371, 232059077, 482817057, 644398259, 423263381},
  {1, 3, 5, 3, 27, 7, 35, 209, 65, 265, 1393, 4019, 7221, 10559, 2477, 20145, 13045, 130681, 278067, 978551, 1411703, 3590351, 578781, 14475839, 8304241, 54706687, 111377119, 26321147, 43224159, 404069437, 177511485},
  {1, 3, 1, 13, 31, 19, 53, 143, 135, 9, 1021, 3915, 2853, 16027, 9131, 41785, 56049, 189223, 465841, 171317, 343347, 1525035, 3166157, 14409135, 28696117, 32645553, 72235467, 60810199, 278987801, 1059165395, 618892771},
  {1, 1, 7, 13, 31, 5, 115, 153, 143, 957, 623, 303, 6031, 14121, 8463, 5905, 107831, 142189, 77203, 612697, 1837663, 1498337, 4397693, 1361541, 30499571, 55587609, 36750547, 177217889, 234029119, 13959197, 1609761139},
  {1, 1, 5, 11, 25, 19, 29, 31, 297, 943, 443, 2087, 1599, 6827, 14349, 10259, 47141, 67591, 317601, 928251, 1694057, 3095225, 8295589, 15554021, 29311889, 51256479, 60439389, 74093935, 54526057, 1002438819, 2118123917},
  {1, 3, 3, 5, 21, 11, 127, 81, 479, 25, 699, 263, 1117, 329, 32391, 53643, 28333, 238485, 389547, 366681, 1928091, 2342653, 1940845, 13374711, 14419127, 40119001, 77352393, 84207215, 88981595, 8306925, 987709875},
  {1, 1, 3, 11, 25, 31, 97, 19, 195, 781, 705, 2579, 3063, 9693, 27009, 16779, 9881, 32995, 43493, 924185, 577359, 2648015, 7446589, 8921149, 28466679, 61771775, 21706437, 2157283, 106971253, 744669343, 1571176471},
  {1, 1, 5, 5, 31, 11, 75, 207, 197, 885, 2037, 1211, 4615, 1083, 31939, 17865, 112073, 74313, 320205, 586607, 1658075, 2766259, 4111469, 10896333, 19678385, 52417073, 37276635, 174691279, 132665383, 921358579, 1142067545},
  {1, 1, 1, 11, 9, 23, 29, 231, 307, 17, 1497, 163, 2263, 4127, 30277, 63187, 100833, 252175, 131465, 564081, 179395, 701263, 6257165, 3232317, 13856053, 51903487, 127330781, 28218091, 65556585, 975908939, 1085657495},
  {1, 1, 5, 11, 11, 43, 111, 233, 307, 523, 1259, 999, 3487, 555, 4557, 45505, 53241, 14477, 273923, 1007557, 337649, 479801, 875205, 4494821, 17995281, 14855839, 31240015, 199072599, 20746323, 802148533, 1478570367},
  {1, 1, 7, 5, 1, 21, 107, 229, 343, 933, 217, 1803, 1855, 9393, 23871, 57223, 106131, 71861, 440759, 371361, 1957711, 1349631, 2152957, 291517, 16583219, 23814833, 41155341, 194540817, 239702815, 745372641, 1580515819},
  {1, 1, 1, 11, 3, 21, 125, 131, 405, 599, 1469, 723, 3615, 1751, 6205, 30081, 70319, 163275, 264221, 842199, 371361, 1531159, 7536965, 15009437, 14615125, 57241535, 116367639, 244185897, 428201313, 190663527, 1325842769},
  {1, 3, 5, 5, 9, 39, 33, 81, 389, 151, 811, 2559, 8133, 11095, 30419, 35335, 6673, 33307, 75487, 982835, 313001, 3560449, 3268325, 13702855, 330833, 35084249, 28039949, 226402051, 205854037, 1003818397, 1582563473},
  {1, 1, 7, 7, 7, 1, 59, 223, 265, 529, 2021, 1475, 4047, 15729, 12477, 53573, 59971, 104341, 43037, 432491, 604463, 1367571, 4990077, 6204685, 12215987, 27786995, 100139275, 189280013, 193538903, 573484971, 152096917},
  {1, 3, 1, 3, 9, 23, 85, 181, 47, 265, 49, 3547, 4541, 691, 893, 40539, 55417, 74383, 348979, 2113, 760243, 3833103, 2779277, 5629951, 31309557, 40091583, 124405533, 46861611, 512601953, 350370777, 371405035},
  {1, 3, 5, 11, 19, 23, 9, 7, 157, 299, 1983, 695, 5333, 12279, 14141, 42785, 57101, 185691, 349529, 941543, 1551389, 2354413, 8005429, 4536695, 18955921, 48175127, 114542871, 219297547, 315854101, 498146795, 2062673033},
  {1, 3, 1, 5, 15, 5, 21, 105, 29, 339, 1041, 3179, 2413, 13299, 12507, 61169, 90399, 221011, 404775, 839627, 1880189, 2609203, 6500917, 219151, 2157845, 3233529, 13621019, 114301729, 395529529, 376783661, 1196656057},
  {1, 1, 1, 1, 5, 33, 65, 85, 111, 705, 479, 103, 1767, 11351, 25399, 22323, 66207, 260519, 514531, 554033, 339067, 2517073, 18989, 10671237, 21515381, 13062741, 19239697, 221249845, 336759621, 1018503617, 1444092987},
  {1, 1, 1, 7, 9, 35, 77, 87, 151, 321, 101, 1619, 7375, 9687, 3985, 41331, 100765, 257439, 499345, 921597, 1274019, 659851, 7625165, 212461, 13027189, 2032883, 83329309, 178834735, 484012921, 169376683, 1193335859},
  {1, 1, 5, 7, 17, 1, 51, 197, 175, 811, 1229, 1731, 7359, 4019, 7129, 3387, 84175, 189509, 519223, 843097, 239249, 3087223, 7505285, 6783133, 29724625, 62294067, 68237077, 38689037, 32084263, 236484529, 1636598899},
  {1, 3, 3, 15, 23, 37, 85, 185, 239, 543, 731, 3119, 6085, 9713, 27061, 36597, 57447, 62291, 282583, 871321, 176981, 3454057, 3764373, 10891623, 19379095, 12969939, 14669579, 60528937, 352539961, 117729757, 1929418187},
  {1, 3, 1, 7, 7, 55, 111, 109, 289, 439, 243, 847, 7877, 15091, 8121, 4733, 50677, 97453, 19071, 891671, 373393, 1921865, 6017797, 5806695, 9676757, 15837275, 129974035, 54253339, 424726883, 615516609, 721044973},
  {1, 1, 7, 11, 17, 53, 35, 217, 259, 853, 1667, 3087, 7175, 15553, 321, 65499, 87931, 21837, 230155, 593661, 1771511, 2867557, 4735005, 6344149, 31881651, 17171743, 61526045, 134672393, 521990167, 789516477, 813920415},
  {1, 3, 1, 9, 1, 63, 87, 17, 73, 565, 1091, 1119, 3333, 8643, 18631, 54231, 109657, 96365, 215171, 395639, 583843, 3201357, 1127373, 8317303, 24082165, 64443669, 131674133, 256487467, 212357187, 824119421, 972714245},
  {1, 1, 3, 3, 11, 41, 1, 57, 295, 263, 1029, 995, 7887, 12933, 27265, 15745, 93619, 217343, 236319, 653521, 1309485, 1304599, 8343925, 14757533, 8241431, 4530487, 117424151, 107823157, 528760861, 1004290221, 79741331},
  {1, 1, 5, 1, 27, 45, 109, 161, 411, 421, 1395, 2031, 4487, 3331, 19727, 269, 113439, 181595, 54223, 675437, 370455, 4088661, 7971997, 11495061, 21417649, 16099861, 24313887, 65994809, 92733521, 151191557, 554631583},
  {1, 3, 5, 11, 25, 35, 47, 191, 339, 417, 1727, 343, 6101, 263, 22797, 61723, 56637, 136973, 224621, 145809, 1551099, 2022833, 317869, 8494727, 6852849, 21472087, 49479709, 13828159, 2687003, 190513283, 1297678463},
  {1, 1, 5, 15, 21, 1, 93, 251, 351, 217, 1767, 1063, 6423, 2659, 26017, 52863, 101751, 98923, 220837, 285565, 1729259, 3859785, 1452909, 14154981, 9172977, 65780699, 41622545, 119696429, 355062873, 912077839, 1000041851},
  {1, 3, 3, 11, 3, 7, 75, 155, 313, 211, 491, 2383, 5493, 6177, 26977, 56953, 84173, 17897, 448305, 313859, 1786605, 1655817, 3263989, 9590631, 3764887, 46665815, 537631, 93889563, 283755639, 29953079, 1177203749},
  {1, 3, 3, 5, 11, 9, 101, 161, 453, 913, 1067, 1103, 2021, 6305, 4655, 58733, 10731, 105723, 144495, 976115, 560959, 4036989, 4308285, 12841527, 22378551, 32340761, 78097431, 175369261, 80854137, 798215381, 238877921},
  {1, 1, 3, 1, 15, 45, 127, 141, 163, 727, 1597, 3347, 207, 12837, 5411, 63329, 79383, 82489, 422567, 71269, 1458757, 2213567, 2180053, 7950781, 12045719, 6107829, 125266963, 154006585, 379907187, 673121321, 1086350767},
  {1, 3, 3, 7, 1, 33, 63, 73, 73, 341, 1691, 1615, 7749, 97, 7661, 55587, 23583, 148505, 378067, 881863, 855235, 3988789, 5930445, 15345047, 33241335, 52896411, 55684125, 64574477, 356236299, 110070973, 449264805},
  {1, 3, 5, 13, 15, 39, 53, 235, 77, 99, 949, 339, 1229, 3559, 26283, 42209, 94893, 76711, 434749, 284527, 1876533, 4122151, 1027221, 7601247, 4959249, 50641329, 12796939, 36260899, 198763609, 1072663719, 1984971881},
  {1, 1, 5, 13, 31, 17, 97, 13, 215, 301, 1927, 2491, 8043, 3647, 11231, 21853, 38871, 187579, 381991, 617589, 2070363, 1603301, 2039581, 15932613, 21039553, 20913481, 6936267, 188332389, 334219101, 448475401, 22966547},
  {1, 1, 7, 1, 1, 37, 91, 93, 441, 251, 1131, 3259, 4443, 8925, 14167, 55899, 81743, 160553, 78771, 697083, 987201, 800645, 6175285, 2961829, 25570499, 43417285, 90361949, 103457121, 369839391, 2795913, 1571352597},
  {1, 3, 7, 9, 25, 5, 105, 69, 81, 943, 1459, 3227, 3513, 89, 27167, 63107, 65983, 112603, 405327, 532451, 1015445, 3063765, 4472485, 3322215, 16548803, 6596805, 122688837, 210863457, 416520525, 1043409297, 1627111733},
  {1, 3, 7, 11, 31, 43, 13, 209, 27, 1017, 501, 3279, 3849, 13161, 11837, 8577, 18589, 16115, 232195, 516497, 778647, 2269919, 4482669, 14967407, 22745283, 28914759, 79033795, 42919879, 113969969, 715910077, 1299661039},
  {1, 1, 7, 15, 1, 33, 31, 233, 161, 507, 387, 4011, 2971, 14045, 2201, 53639, 13451, 156373, 179887, 790251, 1216465, 2436429, 5694901, 16247685, 29088707, 46901963, 54240349, 3367773, 307273051, 427258157, 1778106397},
  {1, 3, 3, 5, 5, 53, 33, 177, 503, 627, 1927, 3331, 729, 381, 4315, 48903, 51479, 8531, 460171, 641133, 428229, 372729, 2083749, 5687607, 21754567, 4639561, 104366409, 159944289, 332044661, 82476453, 1796145331},
  {1, 1, 7, 11, 7, 61, 119, 31, 457, 229, 1875, 755, 5355, 7005, 14301, 63497, 43035, 194597, 309369, 165999, 289567, 448785, 6970349, 8032949, 24948291, 44813263, 58922971, 165860433, 396053291, 2817451, 1111490773},
  {1, 1, 5, 15, 19, 5, 53, 201, 157, 885, 1057, 1335, 459, 14063, 14941, 52533, 48123, 117879, 359467, 785591, 1764887, 3872691, 6772461, 2894269, 8310465, 8456651, 8763847, 133390585, 69117753, 702792605, 300243977},
  {1, 3, 7, 9, 1, 35, 51, 113, 249, 425, 1009, 2599, 4265, 10729, 23391, 39483, 12489, 125293, 373167, 845555, 944275, 3648827, 2301117, 9863135, 3049411, 43172037, 46788829, 25053895, 217056119, 813165373, 2096254197},
  {1, 3, 5, 7, 21, 53, 37, 155, 119, 345, 631, 3499, 3433, 12827, 25909, 38259, 80359, 95879, 63209, 619417, 720943, 3080237, 2034221, 10556711, 1130049, 48716363, 43354049, 16050025, 290398001, 1029217679, 645495971},
  {1, 3, 5, 7, 15, 31, 109, 69, 503, 595, 1879, 1219, 7929, 5019, 25397, 617, 2297, 241051, 94071, 256597, 1265797, 1988521, 1872229, 1333239, 15555009, 66732363, 8050011, 182703683, 366695697, 805263865, 266163203},
  {1, 3, 3, 1, 25, 35, 65, 131, 403, 705, 503, 3211, 5993, 14045, 17791, 57011, 81049, 147423, 119097, 616637, 1210495, 3959709, 3335789, 303527, 20099655, 10609485, 81492949, 44997479, 366247781, 295149007, 1710386519},
  {1, 3, 7, 7, 19, 33, 11, 153, 45, 633, 499, 2467, 745, 57, 31793, 52021, 72623, 234209, 174087, 313415, 1288587, 3342041, 4256349, 11201303, 21415363, 23774667, 125886159, 92287613, 209750791, 428333533, 8608961},
  {1, 3, 3, 5, 11, 3, 29, 93, 487, 33, 703, 2075, 3753, 5405, 28923, 9897, 96157, 261859, 471695, 1035989, 959959, 2154789, 5254061, 3403207, 25120711, 18420041, 107413959, 131319127, 451991409, 630203793, 712289363},
  {1, 1, 3, 15, 21, 53, 107, 179, 387, 927, 1757, 247, 2731, 14201, 20865, 55683, 128903, 42745, 398157, 998605, 374593, 728315, 6711509, 4602941, 16076871, 47171147, 115376473, 220207049, 195051647, 1067499559, 1176617199},
  {1, 1, 3, 7, 21, 45, 51, 147, 175, 317, 361, 2455, 3835, 13817, 25353, 49939, 122655, 36177, 178525, 886753, 1747707, 1511051, 7993149, 3675965, 5778503, 6825283, 25714649, 11028465, 133890087, 351862887, 873005219},
  {1, 1, 1, 7, 7, 13, 15, 243, 269, 795, 1965, 1671, 235, 15099, 17421, 27649, 62475, 169005, 249045, 435331, 1822073, 3805699, 1553845, 14663997, 33311301, 36489667, 131426115, 198080465, 118489171, 24117383, 898630001},
  {1, 1, 3, 5, 19, 33, 57, 115, 443, 537, 627, 4083, 2011, 297, 12491, 81, 118495, 30355, 22617, 301077, 100455, 3824537, 6577933, 5213749, 30053703, 28279361, 90918111, 162262901, 26341429, 1010118839, 2027524511},
  {1, 3, 3, 9, 3, 39, 25, 61, 185, 717, 1049, 1431, 4633, 1661, 26823, 51145, 35065, 258399, 136663, 289339, 332855, 1173931, 5970605, 4324383, 31129799, 1516997, 126570191, 38649283, 201216085, 849850465, 136523805},
  {1, 3, 7, 3, 7, 37, 107, 153, 7, 269, 1581, 1439, 1497, 4857, 21733, 40313, 79411, 111193, 20119, 579197, 942267, 3940959, 3795613, 5665711, 8324803, 10835535, 129362395, 144079593, 295709751, 90318029, 489118827},
  {1, 1, 7, 3, 7, 41, 91, 41, 145, 489, 1245, 367, 27, 15293, 613, 35049, 19391, 94641, 158919, 363883, 482675, 1750759, 1765565, 14897197, 22445763, 10681287, 15403995, 244384485, 487844871, 714325197, 1239202877},
  {1, 1, 5, 9, 7, 7, 105, 81, 403, 407, 283, 539, 5771, 8879, 9323, 18685, 15041, 240335, 116443, 266217, 1388505, 469789, 5498325, 680933, 11040705, 38679885, 101068115, 198061667, 158182453, 963437581, 623060343},
  {1, 1, 7, 9, 27, 55, 29, 77, 193, 963, 949, 3671, 6155, 16285, 8175, 22305, 61017, 72923, 489871, 504451, 567049, 3902459, 8181717, 2417373, 25443011, 42236877, 127159623, 43810259, 84343857, 800017617, 1354065269},
  {1, 1, 5, 3, 25, 51, 107, 63, 403, 917, 815, 2963, 563, 9303, 3065, 44079, 127537, 80509, 280185, 698909, 2079395, 2395861, 3998045, 3737029, 22884609, 6520071, 43350285, 261954879, 285582095, 616945139, 1740922175},
  {1, 1, 7, 3, 7, 61, 19, 51, 457, 599, 535, 3003, 7187, 2805, 11005, 28977, 70787, 97977, 226373, 19775, 879989, 1015209, 6721861, 8407829, 18694915, 43809031, 93619483, 132939569, 206451983, 203286983, 851281029},
  {1, 3, 7, 1, 23, 51, 105, 153, 239, 215, 1847, 1291, 2993, 4145, 22975, 21605, 94669, 61151, 137467, 698573, 166681, 4131625, 8347797, 13133751, 31841539, 41602829, 41159435, 119905591, 325503285, 882366805, 1435509835},
  {1, 1, 3, 5, 27, 23, 79, 49, 495, 45, 1935, 659, 3123, 16273, 21043, 56673, 73169, 122545, 330335, 831197, 1226531, 1020805, 4341053, 15564677, 909575, 63873281, 52064023, 106941699, 174404387, 816158061, 1063767507},
  {1, 1, 1, 11, 11, 47, 55, 133, 495, 999, 1461, 2543, 2067, 3411, 31481, 29197, 13865, 10873, 4699, 497237, 141513, 1178379, 295125, 8493885, 29474053, 41729295, 97627919, 198942979, 126711067, 565612345, 387711027},
  {1, 1, 3, 15, 27, 51, 93, 17, 355, 763, 1675, 2435, 5267, 1105, 22617, 28629, 33237, 187339, 192411, 645041, 804429, 3498733, 2648517, 14055013, 15036679, 25911051, 1773335, 265542927, 308371761, 925133277, 111194391},
  {1, 3, 1, 3, 1, 3, 79, 119, 499, 17, 995, 2107, 7505, 12279, 17553, 17751, 7665, 205209, 445849, 195021, 1304711, 196593, 6583245, 7039319, 22312197, 64685327, 129834245, 151611663, 157000523, 678036859, 1707905487},
  {1, 1, 1, 1, 15, 43, 45, 17, 167, 973, 799, 3795, 6483, 307, 5971, 477, 120421, 140843, 332315, 67125, 118387, 3431317, 2293053, 12467141, 2631941, 39345925, 64524043, 44669231, 244915985, 430719421, 1659629843},
  {1, 1, 1, 3, 27, 49, 89, 29, 483, 913, 2023, 835, 6051, 8291, 3617, 2301, 48231, 102491, 397519, 938489, 46071, 2330661, 1244141, 7845061, 8946693, 36959239, 107060255, 76639293, 494091317, 827843801, 633330823},
  {1, 1, 3, 3, 5, 11, 75, 7, 41, 851, 611, 563, 6243, 9121, 2789, 18403, 118497, 231101, 230961, 943623, 1656033, 3744613, 7237077, 3170725, 22696967, 50351111, 16794633, 83997703, 50496607, 16899115, 1561249029},
  {1, 3, 1, 3, 7, 57, 39, 123, 257, 283, 507, 3491, 737, 13479, 23521, 47841, 48855, 10485, 129109, 256087, 310201, 1768621, 7700341, 12020615, 25969669, 41174031, 112991235, 121967669, 320083003, 439491743, 1245586845},
  {1, 3, 3, 11, 27, 23, 113, 229, 187, 299, 133, 3247, 7457, 1509, 9709, 17197, 48705, 29951, 82215, 121425, 1731965, 2060915, 5774021, 14794271, 29553671, 63140871, 55395351, 22461499, 517725213, 548275385, 1274131839},
  {1, 1, 3, 13, 9, 63, 101, 77, 451, 169, 337, 3855, 1379, 7073, 3435, 64499, 125709, 241979, 174567, 231417, 1300751, 360163, 5456397, 10930653, 10738695, 27401225, 36586501, 137862155, 453610561, 147625137, 1874136151},
  {1, 3, 7, 3, 3, 59, 45, 195, 229, 415, 409, 1431, 6561, 1505, 8749, 8549, 107037, 185915, 187001, 73475, 1260085, 2853831, 5469445, 10306799, 23192579, 5325839, 105636895, 206699575, 108164129, 915930159, 641008721},
  {1, 3, 5, 3, 11, 19, 71, 93, 43, 857, 369, 2127, 2913, 803, 25833, 3949, 61605, 134273, 216051, 988273, 2004975, 1510691, 6811853, 16748255, 14857217, 13796367, 132187167, 268424223, 351313003, 257348625, 1566412087},
  {1, 3, 7, 9, 19, 33, 115, 19, 241, 703, 247, 3275, 7105, 2497, 15175, 12481, 80071, 18929, 158645, 196503, 796573, 1741001, 7559141, 1508119, 12390403, 24317957, 129200143, 263655429, 144535615, 57544855, 1573671229},
  {1, 3, 5, 11, 5, 35, 21, 155, 463, 1005, 1073, 1087, 5569, 4931, 28481, 48083, 78313, 42515, 340593, 789433, 754867, 2373803, 6998685, 279295, 13586433, 48689159, 65359889, 165867535, 83873793, 705097751, 638341531},
  {1, 3, 7, 3, 25, 15, 109, 83, 93, 69, 1189, 2023, 5505, 129, 15245, 12447, 55485, 15603, 410889, 345723, 1761919, 533679, 3813165, 13312719, 30228483, 59097103, 58998789, 79970307, 468140041, 204193903, 1346060777},
  {1, 3, 5, 7, 5, 21, 93, 133, 135, 167, 903, 2075, 2433, 10627, 23949, 9355, 75423, 250615, 474303, 506893, 16169, 134473, 6580629, 12681015, 22200321, 41271307, 129712145, 44548105, 46022729, 881705169, 1324270067},
  {1, 1, 7, 7, 3, 59, 121, 161, 285, 815, 1769, 3705, 1345, 15169, 30279, 711, 85827, 51579, 336185, 152033, 1079773, 4082671, 3721513, 16584633, 25759745, 5967873, 105738247, 127234055, 217395203, 613855291, 437555321},
  {1, 3, 1, 1, 3, 47, 103, 171, 381, 609, 185, 373, 7233, 12099, 17601, 63681, 30915, 3055, 333991, 930539, 1288893, 180385, 3622649, 10504501, 24530945, 50688003, 128471041, 215535617, 391671811, 341487663, 1505669223},
  {1, 3, 3, 15, 23, 33, 107, 131, 441, 445, 689, 2059, 2881, 12867, 20163, 44751, 46551, 56673, 374443, 106947, 507897, 2509565, 3874289, 8669899, 30232577, 39317507, 78213123, 264450063, 346599447, 902484001, 278425707},
  {1, 3, 3, 11, 7, 53, 101, 167, 435, 803, 1255, 3781, 3777, 15427, 7235, 6219, 110919, 19445, 454565, 624615, 186355, 2151779, 7455143, 16236805, 8572929, 26898435, 84897795, 229470219, 493465607, 46829621, 245141605},
  {1, 1, 5, 11, 15, 59, 41, 19, 135, 835, 1263, 505, 2913, 9697, 26469, 27883, 28207, 37147, 279817, 392115, 884455, 710051, 7368847, 3477977, 12489729, 62696449, 38517765, 95513611, 113845263, 250334267, 1444584489},
  {1, 1, 7, 11, 21, 49, 23, 219, 127, 961, 1065, 385, 7073, 6177, 231, 7787, 5493, 247121, 46455, 664955, 2074079, 2164385, 4162633, 14778401, 13011969, 54180865, 57844743, 74642443, 220433429, 1038220337, 1646726167},
  {1, 3, 5, 15, 7, 47, 117, 217, 45, 731, 1639, 733, 2017, 8035, 19109, 20143, 5607, 162511, 480469, 870969, 360205, 2428283, 2305991, 16337469, 22627329, 64998403, 24218629, 91556879, 225993735, 320590895, 67937397},
  {1, 1, 7, 11, 27, 57, 91, 87, 81, 35, 1269, 1007, 273, 1585, 2903, 45819, 66827, 30505, 457003, 809959, 494817, 2971315, 3930693, 15401375, 7670529, 2566401, 42543879, 208398091, 257745179, 61256505, 1005208923},
  {1, 1, 3, 11, 15, 37, 53, 219, 193, 937, 1899, 3733, 273, 9617, 23987, 1115, 27167, 58933, 224773, 334443, 1470001, 1200121, 2501179, 10029349, 21962497, 28521729, 76612867, 182151947, 75706639, 618330405, 204290869},
  {1, 3, 5, 3, 13, 11, 27, 19, 199, 393, 965, 2195, 497, 3347, 32213, 45747, 21405, 89019, 179403, 842659, 231575, 3008697, 679669, 6472739, 14867713, 12548867, 75533061, 241132291, 389928717, 243562763, 880391451},
  {1, 3, 1, 3, 5, 1, 37, 173, 413, 1023, 553, 409, 4689, 7795, 23313, 10259, 17813, 69297, 496309, 547357, 1845837, 2689391, 2396153, 11682761, 8836353, 65270019, 119272705, 48487683, 475729157, 291155713, 1711257893},
  {1, 3, 1, 7, 15, 29, 123, 95, 255, 373, 1799, 3841, 4913, 14995, 25009, 10071, 112863, 189293, 428331, 328687, 370511, 1240165, 3618935, 11033777, 14143233, 52845827, 102644481, 118660871, 470204175, 491697437, 668911483},
  {1, 3, 5, 13, 21, 57, 51, 17, 511, 195, 1157, 1831, 6705, 7539, 26773, 53181, 76101, 168393, 50211, 91137, 1670639, 3343379, 3616501, 11254903, 6640897, 5634307, 28865285, 149475597, 401129237, 907772729, 1466549555},
  {1, 1, 1, 15, 29, 19, 7, 73, 295, 519, 587, 3523, 2609, 8625, 23025, 15999, 57933, 97859, 99031, 656249, 1765687, 1372535, 1782779, 5422099, 30645505, 18191105, 126977793, 243845391, 531212061, 529172243, 1015925511},
  {1, 1, 5, 13, 13, 35, 115, 191, 123, 535, 717, 1661, 2289, 5425, 5013, 33181, 98781, 62707, 450019, 303023, 889451, 6887, 7880509, 11754701, 3247361, 8692481, 41218821, 72761101, 194968333, 973510435, 999443827},
  {1, 3, 3, 5, 23, 21, 47, 251, 379, 921, 1119, 297, 5329, 13043, 23251, 40309, 42023, 198917, 323455, 219595, 2040235, 1647401, 3761935, 7417465, 29321985, 53268739, 126870275, 97378053, 204232471, 953419029, 789685039},
  {1, 3, 3, 9, 29, 53, 121, 201, 135, 193, 523, 2943, 7889, 15731, 23827, 1049, 7533, 86437, 267177, 482489, 1000951, 1639857, 5200827, 12345935, 11329793, 35586307, 93936899, 41544969, 366908189, 1061295413, 1891903353},
  {1, 1, 1, 7, 29, 45, 125, 9, 99, 867, 425, 601, 6993, 16145, 22833, 19319, 13677, 96349, 268813, 118297, 370451, 3962323, 6189721, 1912873, 7584001, 31691009, 7973121, 139271431, 376239901, 935159597, 1077609853},
  {1, 3, 1, 9, 13, 15, 67, 181, 109, 293, 1305, 3079, 5041, 15283, 24433, 56601, 89789, 200671, 70259, 150213, 636893, 3815477, 7343497, 1431287, 32073985, 25266947, 54695681, 178727177, 80789773, 944886031, 1603325251},
  {1, 3, 3, 9, 5, 35, 15, 209, 305, 87, 767, 2795, 3257, 2555, 13659, 26561, 98781, 179451, 500743, 375865, 1629417, 691823, 2294071, 2426787, 8716737, 52104771, 16988739, 133143497, 355225029, 77457635, 861171791},
  {1, 3, 3, 11, 27, 57, 113, 123, 179, 643, 149, 523, 6041, 299, 22747, 4675, 18371, 129889, 460521, 253411, 371515, 3262283, 555277, 8662355, 19397185, 10926275, 113872579, 213914315, 264340315, 725652729, 1413828657},
  {1, 1, 3, 15, 11, 17, 67, 223, 63, 657, 335, 3309, 5385, 249, 30187, 23239, 94787, 232393, 262651, 577863, 1457655, 2799561, 6235639, 7661525, 30457537, 15864257, 105790915, 248164943, 309771723, 13972817, 1585475587},
  {1, 1, 1, 9, 25, 29, 109, 159, 39, 513, 571, 1761, 6601, 15513, 19785, 29057, 42641, 134037, 277157, 408279, 513615, 3235817, 5450819, 12395865, 13433153, 24077761, 46351297, 64596553, 230703321, 155207261, 507527213},
  {1, 1, 3, 1, 5, 63, 75, 19, 455, 601, 123, 691, 649, 5529, 7899, 19081, 127373, 235175, 523459, 398779, 1966015, 1178241, 6650211, 8872203, 30302401, 57228993, 67837891, 163803201, 347687109, 608793343, 1412186123},
  {1, 1, 1, 3, 21, 5, 45, 169, 377, 513, 1951, 2565, 5833, 15161, 8681, 62859, 55709, 208525, 458165, 983425, 1553617, 3682025, 1166935, 5661773, 15285569, 63418433, 111862849, 88885827, 16057813, 107873093, 328089709},
  {1, 1, 3, 11, 3, 33, 119, 69, 253, 907, 805, 1449, 377, 2569, 31659, 2787, 92139, 131545, 33231, 214525, 1158613, 2453011, 167789, 9011809, 19308481, 34583361, 41654595, 56336587, 312682307, 274902113, 1604674615},
  {1, 1, 5, 13, 31, 15, 17, 7, 499, 61, 687, 1867, 5449, 15065, 4413, 8421, 25719, 197127, 138841, 1032799, 487979, 1545557, 2997623, 13397235, 9957953, 31364033, 73303749, 115546445, 441534943, 858927311, 147958865},
  {1, 3, 7, 11, 17, 33, 73, 77, 299, 243, 641, 2345, 633, 2091, 30703, 54563, 83497, 242745, 11377, 936997, 1934659, 205979, 2384073, 6956369, 1519681, 11613507, 48956615, 63435467, 344942161, 236867681, 2067401737},
  {1, 1, 7, 11, 9, 35, 31, 235, 359, 647, 379, 1161, 4169, 13737, 31231, 32979, 35793, 212811, 369527, 78163, 2093103, 1045391, 4862611, 201409, 30346433, 19865537, 57900231, 94482635, 270568393, 748896099, 698721119},
  {1, 3, 3, 15, 31, 25, 5, 67, 33, 45, 437, 4067, 7801, 9339, 8011, 53495, 100183, 151281, 411661, 604299, 1014233, 3373653, 1592381, 6943899, 13653057, 61985731, 133659459, 25239119, 215471711, 689992665, 1420176709},
  {1, 1, 3, 11, 7, 17, 37, 87, 333, 253, 1517, 2921, 3001, 8761, 7707, 64979, 107743, 79321, 125837, 153375, 1453813, 1891461, 2899013, 15790769, 27622465, 41574081, 93220931, 116293195, 497493063, 244640081, 156538213},
  {1, 1, 7, 15, 7, 15, 107, 189, 153, 769, 1521, 3427, 7961, 15129, 28559, 63863, 15071, 63079, 180643, 826549, 1433041, 1807593, 7531753, 9051707, 29620161, 8161857, 15046471, 181981007, 492317511, 115058639, 1951258923},
  {1, 3, 5, 13, 5, 61, 113, 37, 293, 393, 113, 43, 921, 14491, 13261, 18421, 10957, 73829, 55689, 618653, 782429, 2699281, 3092873, 104083, 15855425, 33260099, 91676229, 184899917, 92438341, 743437821, 452991281},
  {1, 1, 1, 15, 29, 43, 107, 31, 167, 147, 301, 1021, 7721, 4505, 5881, 24663, 91909, 185139, 60675, 572679, 813359, 579707, 6012517, 528437, 29545921, 36299457, 129292481, 3303887, 523429469, 170608235, 615458091},
  {1, 1, 1, 13, 3, 1, 35, 93, 195, 181, 2027, 1491, 4825, 9161, 32105, 2613, 90267, 258969, 263131, 670229, 1688187, 755517, 5181363, 7739803, 24395201, 58739777, 133731777, 140102221, 311125955, 159157185, 1416803171},
  {1, 3, 3, 3, 13, 33, 77, 199, 153, 221, 1699, 3671, 3225, 8427, 15403, 14555, 115493, 99305, 380997, 356415, 954849, 3854789, 5321515, 3962639, 15374401, 16616131, 113224003, 65595971, 378093517, 414972641, 312327949},
  {1, 3, 5, 13, 7, 49, 123, 155, 495, 681, 819, 809, 2137, 11115, 2781, 2485, 71535, 178889, 487907, 761747, 1809015, 305729, 5380667, 7404097, 7119809, 15841731, 52200133, 197453645, 162280903, 591912305, 1648416571},
  {1, 3, 5, 15, 27, 61, 117, 189, 183, 887, 617, 4053, 6121, 3675, 16989, 20151, 100211, 229573, 118781, 290165, 1986767, 4133423, 5436209, 3666653, 11778113, 27264067, 39862981, 81799887, 266385243, 849513597, 1277615413},
  {1, 1, 1, 7, 31, 59, 125, 235, 389, 369, 447, 1039, 6909, 5181, 16373, 13555, 74699, 206839, 136017, 770871, 1927361, 2480557, 2481851, 5279419, 23497745, 19765969, 55438257, 41479095, 242794767, 67743019, 862894957},
  {1, 3, 5, 1, 5, 39, 115, 89, 249, 377, 431, 3747, 61, 5135, 3953, 26045, 10305, 149587, 94199, 134293, 5541, 791725, 2558139, 11268503, 26450513, 17861331, 114346933, 2087281, 534308117, 996732183, 460192547},
  {1, 1, 1, 5, 7, 47, 59, 157, 77, 445, 699, 3439, 1645, 2877, 18485, 17265, 22355, 45419, 212471, 118961, 839697, 1298681, 242559, 15661579, 18792721, 23842705, 7043057, 106946741, 144462103, 516798783, 1487322923},
  {1, 1, 3, 5, 11, 21, 19, 75, 11, 599, 1575, 735, 605, 5053, 5559, 41265, 72983, 180497, 198927, 921935, 418807, 754987, 281147, 12898355, 25161585, 29328049, 29610931, 200414005, 298026779, 505756421, 1449773859},
  {1, 3, 5, 3, 19, 13, 41, 69, 199, 143, 1761, 3215, 197, 13439, 7633, 57663, 5655, 25137, 399925, 767553, 1127251, 1683275, 5858669, 12430563, 21783857, 9831603, 103171797, 204095027, 295052035, 217078589, 1511144825},
  {1, 3, 5, 7, 19, 43, 25, 41, 41, 11, 1647, 2783, 5757, 10687, 25393, 62651, 7831, 36487, 63109, 40821, 1359357, 1474463, 4650459, 13499107, 13094865, 20695539, 127077813, 111926839, 42585347, 126496539, 72174409},
  {1, 3, 1, 9, 19, 45, 111, 97, 405, 399, 457, 3219, 3429, 9359, 21333, 32373, 57735, 58041, 523187, 42181, 683161, 3839579, 5552621, 8864063, 27777425, 43668915, 85888721, 87262201, 126200067, 694979357, 922164607},
  {1, 1, 3, 1, 23, 15, 65, 121, 59, 985, 829, 2259, 4181, 2221, 25335, 52373, 94507, 244347, 494797, 242165, 1211183, 1486573, 7140961, 4100935, 13840113, 31226577, 97421843, 175872113, 256645895, 217397535, 895409521},
  {1, 1, 3, 7, 17, 13, 107, 229, 75, 551, 1299, 2363, 7277, 13837, 10711, 52691, 124333, 123233, 399615, 25345, 1594711, 1545731, 7801839, 7844439, 31115633, 38071345, 56707827, 212984183, 239512321, 489293597, 1705947995},
  {1, 1, 5, 5, 21, 57, 23, 199, 509, 139, 2007, 3875, 3965, 13293, 15569, 10385, 100337, 75293, 205363, 213715, 1202769, 3290279, 3467379, 4478367, 10236177, 7659121, 17519413, 104283637, 42585349, 587889449, 1064191815},
  {1, 3, 1, 11, 19, 53, 15, 229, 215, 741, 695, 823, 7597, 4239, 13253, 37703, 31703, 153561, 236803, 424433, 2040747, 3208881, 2113067, 6285539, 29748657, 6074899, 133358001, 229179, 336687107, 468815877, 940847135},
  {1, 3, 7, 1, 29, 3, 17, 163, 417, 559, 549, 319, 389, 11023, 14147, 8909, 112193, 125639, 279077, 163439, 1774117, 1662691, 7172577, 11606851, 20079441, 65808691, 2374423, 199023409, 104140813, 698314803, 1832210529},
  {1, 3, 1, 13, 17, 9, 47, 133, 365, 7, 1937, 1071, 3413, 6031, 20741, 46849, 54037, 192829, 148331, 145825, 1095377, 21267, 3241253, 2120499, 27338769, 34143283, 33581393, 234881597, 503382017, 369295417, 1442906175},
  {1, 3, 5, 7, 19, 37, 55, 163, 301, 249, 689, 2327, 6005, 13167, 14049, 16875, 113447, 249561, 472667, 505431, 866897, 3942357, 7019621, 396115, 14550545, 53610579, 68813141, 263882295, 364272643, 795937813, 1425290343},
  {1, 3, 5, 13, 11, 23, 61, 205, 257, 377, 615, 1457, 2925, 5687, 6305, 40289, 49087, 86787, 310641, 210737, 1378133, 2627117, 1147779, 5475573, 9150033, 45389267, 107829, 189129309, 530613275, 438535207, 354653293},
  {1, 3, 5, 1, 23, 37, 13, 75, 331, 495, 579, 3367, 7325, 10447, 12385, 13805, 72547, 31233, 45617, 719487, 1540959, 1238051, 1288655, 5027131, 17826545, 66062131, 78655989, 223410353, 160465927, 5422101, 1175829597},
  {1, 1, 1, 9, 1, 23, 49, 129, 475, 543, 883, 2531, 2845, 15453, 9573, 43501, 117221, 42003, 199701, 843005, 289855, 420923, 70399, 1086399, 27550449, 30127281, 6553329, 89414393, 350557201, 247772167, 696106017},
  {1, 3, 1, 5, 23, 59, 51, 35, 343, 695, 219, 369, 317, 791, 21989, 34921, 59699, 163983, 76175, 89455, 909755, 1532763, 745831, 844621, 13216945, 58076851, 112608625, 47069141, 110836743, 425901067, 493087779},
  {1, 3, 3, 1, 27, 17, 63, 97, 71, 507, 1929, 613, 1021, 13279, 3807, 48789, 55199, 201405, 437483, 259181, 171011, 2293015, 719597, 10811425, 8321777, 5687859, 34909715, 68995025, 7672907, 885435233, 234904911},
  {1, 1, 5, 1, 21, 31, 11, 109, 247, 409, 1817, 2173, 7789, 15997, 2969, 43229, 106457, 64739, 304839, 499305, 542387, 1168709, 3265757, 12000305, 22743793, 61677073, 21328149, 93930065, 411568197, 84467023, 1336532251},
  {1, 1, 3, 15, 23, 9, 7, 209, 301, 23, 147, 1691, 2101, 4021, 13391, 32691, 51523, 145789, 426707, 431485, 1997641, 2426555, 6471071, 12668543, 26824785, 42558705, 18021299, 103119167, 4936391, 1016529113, 623484791},
  {1, 1, 7, 5, 5, 19, 37, 229, 249, 277, 1115, 2309, 3973, 13965, 15227, 7257, 76865, 146671, 234409, 250129, 223461, 2890073, 4976135, 14610313, 21443601, 47972817, 124545335, 259800789, 262525141, 942411587, 1046349589},
  {1, 1, 1, 5, 5, 63, 5, 249, 285, 431, 343, 2467, 7301, 2325, 29293, 51225, 9977, 236563, 461833, 418909, 644097, 2123059, 6592859, 8627399, 26359377, 2613361, 47591921, 109448373, 343824341, 753926895, 2003881813},
  {1, 1, 1, 11, 7, 45, 35, 75, 505, 537, 29, 2919, 2653, 7381, 25069, 55191, 37499, 197433, 206679, 394583, 223757, 2619165, 4640049, 7878939, 24805649, 43825169, 10615889, 167047931, 4473175, 669801469, 398488947},
  {1, 3, 5, 15, 11, 39, 15, 63, 263, 9, 199, 445, 7205, 11103, 521, 1163, 72583, 12979, 16859, 669563, 1991299, 312077, 5423523, 15812897, 5954033, 59399443, 130179573, 157166175, 137000155, 217814103, 958971935},
  {1, 3, 3, 3, 27, 63, 53, 171, 227, 63, 1049, 827, 4781, 5847, 27135, 3207, 16015, 6571, 79553, 58127, 1905999, 3923387, 6677581, 15017567, 9792433, 34583411, 79222291, 8455219, 351528779, 122952655, 494452805},
  {1, 1, 3, 13, 7, 11, 115, 183, 179, 937, 1785, 381, 2397, 11629, 13775, 60097, 49091, 238823, 404167, 29283, 1407847, 4012589, 8346829, 15471873, 20386993, 16583953, 85795347, 9545821, 165752663, 358730843, 1579659267},
  {1, 3, 1, 11, 13, 15, 107, 81, 53, 295, 1785, 3757, 8013, 5967, 20109, 34895, 46801, 212715, 247495, 366149, 1480225, 3233659, 7704677, 5296345, 17486545, 63634931, 40016145, 173696603, 509740381, 928129663, 703672379},
  {1, 3, 3, 13, 11, 5, 109, 243, 3, 505, 323, 1373, 1077, 3007, 9807, 4681, 77023, 152425, 162153, 815999, 325215, 1171229, 3632407, 13380833, 6123697, 51902227, 130588595, 6301277, 413213147, 589333621, 1868992541},
  {1, 3, 3, 11, 21, 51, 17, 177, 381, 937, 1263, 3889, 2309, 5263, 12367, 59119, 74017, 244647, 287997, 871989, 733761, 2169029, 1075571, 1445085, 17696337, 10885651, 98467443, 253908379, 149392581, 335972163, 399639649},
  {1, 3, 5, 9, 27, 25, 85, 193, 143, 573, 1189, 2995, 2349, 12631, 6745, 47661, 32759, 119349, 355857, 354013, 1980979, 4087537, 7217585, 8792991, 6550641, 50207059, 98179797, 74108089, 298261579, 523876329, 2005666885},
  {1, 3, 5, 11, 13, 9, 81, 21, 159, 953, 91, 1751, 5749, 2167, 20089, 30447, 72801, 27949, 304989, 1029113, 563699, 2665413, 6172335, 11515371, 21761393, 52171539, 76812085, 56369915, 98876381, 365210361, 430672961},
  {1, 1, 3, 3, 27, 61, 11, 253, 391, 333, 1105, 635, 2941, 1493, 6431, 54703, 7679, 178953, 82783, 901553, 361251, 3475073, 3673861, 9310999, 12461809, 33190225, 104258451, 120013875, 413217099, 299980269, 2124631163},
  {1, 3, 3, 15, 9, 57, 95, 81, 419, 735, 251, 1141, 709, 11231, 9263, 42667, 115325, 250877, 17571, 21725, 279991, 155139, 1223199, 9774897, 2876081, 12875379, 104098611, 62155487, 375670745, 775687241, 826862639},
  {1, 1, 5, 9, 31, 39, 59, 13, 319, 807, 1241, 2433, 3779, 9671, 23371, 34399, 74209, 39077, 97457, 507311, 968609, 2842789, 8348371, 2600551, 27179853, 14010781, 107437361, 44291709, 124806363, 821168595, 1745978199},
  {1, 3, 3, 5, 27, 13, 107, 141, 423, 937, 2027, 3233, 3715, 8517, 1609, 55767, 59509, 64423, 180965, 579335, 139497, 417979, 3581149, 13233627, 9020277, 58648431, 109387223, 263450041, 80488263, 354731753, 384552311},
  {1, 3, 3, 9, 9, 25, 125, 23, 443, 835, 1245, 847, 7551, 9373, 27737, 50203, 51935, 59719, 412755, 208017, 1578233, 3791953, 5184927, 10967313, 3543677, 30279751, 70137335, 248659221, 502701013, 174567501, 114330409},
  {1, 1, 7, 15, 17, 17, 83, 107, 411, 285, 847, 1571, 5159, 7727, 249, 16157, 128331, 153447, 270461, 24753, 1007705, 2974671, 4564273, 9169633, 22966469, 2023301, 127135507, 49668635, 110708829, 241847421, 640501031},
  {1, 1, 3, 13, 29, 61, 37, 81, 349, 727, 1453, 1957, 3307, 14327, 29325, 13623, 121427, 213627, 129971, 146455, 1004819, 2094353, 690287, 7157147, 4347325, 35277421, 109304567, 157550009, 102623169, 877562585, 932582177},
  {1, 3, 7, 11, 31, 13, 59, 77, 273, 591, 1265, 1533, 2075, 9941, 10381, 56945, 63133, 128919, 446453, 635079, 700799, 2937809, 3731571, 8124155, 2098637, 27267055, 81818419, 10498999, 518031571, 568517329, 270782327},
  {1, 1, 7, 7, 13, 17, 25, 25, 187, 329, 347, 1473, 6963, 503, 25945, 42405, 10615, 109199, 496023, 411239, 2052613, 906419, 4177485, 15826987, 13143605, 58822573, 54833107, 199580083, 123714113, 603117021, 479568925},
  {1, 3, 7, 7, 5, 51, 37, 99, 221, 153, 503, 2583, 2431, 5869, 16941, 38061, 125743, 26173, 79375, 121389, 1377271, 3822375, 5583049, 2897877, 21965861, 49947223, 49446803, 78570587, 267274569, 566163039, 1746873345},
  {1, 3, 1, 13, 19, 27, 11, 69, 181, 479, 1183, 3229, 171, 15845, 3963, 28907, 64693, 45153, 69049, 933063, 1785003, 3978329, 2059005, 3726679, 25931509, 9258079, 114602709, 101148273, 480587847, 883937503, 1982180375},
  {1, 3, 3, 13, 23, 21, 103, 147, 323, 909, 947, 315, 6407, 6557, 32073, 12271, 2553, 196983, 292441, 520721, 399921, 3490783, 6767709, 52689, 9984117, 8736839, 49440983, 89055249, 288244299, 910487377, 987137099},
  {1, 3, 1, 3, 23, 1, 31, 59, 93, 513, 45, 2271, 2607, 8717, 22795, 7205, 49773, 251467, 57209, 275801, 868735, 1786767, 6078563, 1884873, 7986317, 43995175, 62552885, 76679583, 60261315, 1037523325, 369474579},
  {1, 3, 5, 1, 7, 43, 109, 59, 231, 41, 1515, 2385, 4531, 11029, 959, 56591, 54537, 80649, 29999, 136053, 1002877, 1647971, 5271293, 9482267, 7229173, 60945775, 96743633, 52200637, 308609603, 496936415, 918749281},
  {1, 3, 1, 5, 31, 57, 49, 223, 283, 1013, 11, 701, 7339, 13541, 22875, 39555, 113033, 247963, 449943, 925729, 253993, 2843575, 5679093, 16345639, 27245269, 35487647, 6592597, 66319225, 217699403, 929644253, 1828274205},
  {1, 1, 5, 1, 19, 53, 55, 31, 31, 299, 495, 693, 2299, 7207, 19803, 13959, 126093, 117439, 338841, 57021, 1862613, 2815057, 7163325, 7584447, 31725557, 57936381, 13906641, 197437173, 91535959, 981401313, 894713963},
  {1, 3, 3, 9, 5, 33, 77, 253, 427, 791, 731, 1019, 5031, 9757, 7081, 40011, 90435, 225111, 372755, 143371, 1724589, 2053021, 3249393, 2432777, 13092037, 52418343, 121412631, 258653077, 135962969, 840970101, 539095081},
  {1, 3, 7, 11, 1, 9, 119, 203, 53, 877, 1707, 3499, 7847, 6653, 1965, 43457, 46163, 10079, 230521, 451785, 761335, 2219347, 1870525, 6892193, 4111861, 13050247, 19993299, 129835799, 254896205, 494078805, 1544769603},
  {1, 1, 3, 7, 13, 39, 55, 159, 423, 113, 1653, 3455, 6447, 10735, 8429, 4941, 35399, 98041, 274101, 955589, 722237, 347323, 2394163, 8793641, 25453517, 5916069, 123942455, 108603283, 471597521, 840425835, 554061939},
  {1, 1, 3, 5, 21, 47, 51, 59, 55, 411, 931, 251, 2607, 1095, 31317, 215, 19075, 121473, 118409, 293937, 154973, 1998441, 5254649, 1574113, 529829, 4726725, 92799447, 250144081, 524843273, 466201899, 319655255},
  {1, 3, 7, 3, 31, 25, 81, 115, 405, 239, 741, 455, 1655, 1349, 26421, 23889, 80149, 2843, 346479, 930693, 211655, 692889, 6353279, 10820993, 32300605, 56973127, 7604531, 144489311, 16071443, 515755301, 456794973},
  {1, 1, 5, 1, 31, 3, 101, 83, 479, 491, 1779, 2225, 6331, 3951, 20435, 3839, 42009, 179241, 224335, 842885, 130333, 1652521, 3577865, 7400395, 2274277, 7589485, 48777425, 220738549, 185131291, 864746775, 1352285513},
  {1, 3, 3, 3, 9, 37, 107, 161, 203, 503, 767, 3435, 2863, 15973, 4049, 47257, 70739, 243019, 233077, 679443, 243317, 3931765, 7302349, 7147033, 33496693, 35335335, 84620439, 106954719, 532320533, 132286745, 996297087},
  {1, 3, 7, 9, 1, 27, 61, 119, 233, 39, 1375, 4089, 2603, 10637, 9781, 21723, 15615, 100141, 311071, 448785, 1788611, 2632889, 3101445, 15842827, 20864981, 62902383, 104608659, 229956597, 451179277, 687646991, 1723938121},
  {1, 1, 5, 9, 1, 31, 45, 51, 369, 587, 383, 2813, 7043, 4383, 7667, 63959, 72247, 45001, 130343, 335717, 1807571, 1072885, 5320797, 3972479, 8746165, 524477, 87558673, 183036861, 209244933, 874181931, 1372347769},
  {1, 3, 7, 5, 31, 7, 49, 119, 487, 591, 1627, 53, 819, 15645, 53, 58071, 92857, 123517, 46995, 234233, 1351389, 1936141, 5619177, 14150831, 31340389, 15515903, 3808979, 120718521, 520068371, 116051235, 1054845245},
  {1, 1, 7, 1, 9, 47, 1, 223, 369, 711, 1603, 1917, 4579, 13247, 25265, 63579, 103591, 243017, 167659, 24913, 1895995, 1198153, 605241, 12756343, 23232037, 52436381, 132135507, 236990517, 434207493, 1008830779, 115669333},
  {1, 3, 5, 3, 21, 37, 111, 17, 483, 739, 1193, 2775, 4743, 7285, 9927, 12997, 128071, 128963, 240561, 819995, 1407933, 434721, 7807267, 16343753, 20852685, 45518567, 96592241, 104704223, 63991825, 705703961, 1781993579},
  {1, 3, 3, 7, 17, 11, 51, 117, 455, 191, 1493, 3821, 7507, 15613, 26817, 22733, 3155, 67189, 199585, 294015, 316577, 1586853, 6408311, 16677371, 21928605, 27233583, 93446967, 253944571, 175255565, 832458791, 833032263},
  {1, 1, 5, 9, 23, 39, 99, 181, 343, 485, 99, 1931, 7247, 12503, 12995, 14407, 117713, 130029, 292017, 262703, 1305201, 1106783, 6202805, 6711449, 4680597, 56870053, 39617297, 104524381, 172101651, 729403411, 360190063},
  {1, 3, 1, 7, 29, 49, 31, 71, 489, 527, 1763, 2909, 3731, 3597, 4483, 31369, 6551, 160663, 458889, 180029, 585115, 2414401, 3484633, 1692455, 10299941, 25478367, 117733205, 240938555, 317505545, 852901917, 858374267},
  {1, 1, 5, 11, 5, 5, 73, 189, 321, 57, 1191, 3685, 2995, 3311, 16035, 24421, 48935, 88771, 134711, 422619, 1675071, 3037231, 1067569, 7552431, 12979717, 21895037, 70147281, 148013375, 196025345, 33107001, 518667277},
  {1, 1, 5, 15, 13, 45, 125, 207, 371, 415, 315, 983, 7943, 15047, 12515, 5729, 62439, 177643, 262643, 422369, 1505337, 4184485, 2145193, 4600509, 23869797, 42802613, 26297553, 258314907, 31247369, 145523713, 699393049},
  {1, 3, 3, 5, 25, 59, 33, 31, 239, 919, 1859, 2709, 3587, 9997, 14561, 7791, 90303, 236157, 318355, 1019537, 1172405, 2027477, 1966705, 4851247, 19273221, 26623487, 26614359, 121513913, 30559237, 568214559, 561173621},
  {1, 3, 5, 13, 27, 61, 23, 115, 61, 413, 1275, 3559, 3927, 3013, 23527, 15147, 126773, 65055, 411233, 981513, 1489703, 3823947, 5029525, 15348357, 30849813, 62434135, 61923473, 63191697, 398316575, 114086969, 1315802155},
  {1, 3, 7, 15, 5, 59, 101, 81, 47, 967, 809, 3189, 6723, 13085, 10533, 23277, 27367, 108333, 264551, 988567, 1188857, 1294205, 4785787, 5571739, 27460397, 40582831, 6117875, 215078643, 295693321, 264277047, 675648513},
  {1, 1, 5, 11, 31, 15, 39, 25, 173, 505, 809, 2677, 7247, 11595, 1967, 13945, 73297, 160073, 24561, 986847, 1644247, 3691307, 5722659, 6135455, 3941885, 50546893, 49657281, 54840015, 499419339, 767010787, 1306099483},
  {1, 1, 5, 9, 19, 13, 95, 89, 511, 127, 1395, 2935, 35, 12443, 28351, 54811, 55769, 148691, 486813, 601903, 1063637, 3383913, 1975797, 9675321, 25442949, 765557, 125941057, 191076941, 359038279, 279940713, 473801043},
  {1, 1, 5, 5, 31, 45, 9, 57, 91, 303, 1295, 3215, 7715, 3963, 27519, 52759, 54989, 116459, 388587, 375619, 1709681, 2428865, 342089, 5525529, 6194741, 12500085, 108436545, 115404865, 450957899, 782435961, 681741621},
  {1, 3, 3, 3, 19, 15, 113, 187, 217, 489, 1285, 1803, 6531, 13225, 14557, 18333, 5341, 139585, 407559, 190513, 1189927, 1641443, 5732575, 11713465, 30286093, 27824343, 83872199, 169064655, 66746335, 477146691, 1379009405},
  {1, 1, 3, 1, 13, 29, 57, 139, 255, 197, 537, 2183, 2347, 7307, 7289, 28063, 69271, 231567, 34747, 497921, 1878097, 18851, 901083, 3857533, 5950077, 64707525, 52118599, 12409029, 331138753, 881078105, 1258352461},
  {1, 3, 1, 15, 11, 7, 53, 255, 467, 9, 757, 3167, 139, 3593, 23327, 25781, 29437, 213577, 354931, 914141, 1341189, 3620567, 1601135, 12118461, 29013485, 11706247, 104318277, 92127683, 229476303, 289770363, 1787191129},
  {1, 3, 3, 15, 21, 13, 9, 189, 359, 323, 49, 333, 6911, 12073, 4813, 23569, 121763, 31215, 11131, 467799, 106757, 3072373, 8020015, 4992167, 7054573, 17424847, 35331399, 47444291, 359716825, 456230385, 1477943069},
  {1, 3, 7, 11, 7, 37, 21, 119, 401, 157, 1659, 1069, 471, 14265, 11273, 42269, 70969, 13635, 510935, 994421, 980231, 826599, 1450313, 15853379, 15282517, 29452159, 12056003, 156280263, 108125787, 7501001, 512915785},
  {1, 1, 5, 7, 17, 33, 115, 229, 149, 151, 2027, 279, 5795, 14651, 4511, 35637, 32359, 247003, 459193, 805787, 1052519, 3434069, 5693, 185241, 27595397, 48709429, 104560065, 115829059, 504552261, 53115517, 1493921143},
  {1, 1, 5, 15, 5, 49, 77, 155, 383, 385, 1985, 945, 7239, 4971, 31791, 6557, 74211, 15651, 44327, 345465, 1897329, 1605031, 123371, 943835, 11304125, 50234653, 3838017, 213984715, 142026449, 591073997, 1786391321},
  {1, 3, 7, 3, 7, 55, 85, 41, 357, 527, 1715, 1619, 4107, 8633, 1529, 17077, 120361, 35089, 471347, 745835, 1694111, 3455741, 4390741, 16626477, 30748837, 534375, 115868611, 46709583, 446257499, 16263035, 96212233},
  {1, 1, 3, 1, 21, 45, 115, 21, 199, 967, 1581, 3807, 3435, 6667, 32409, 19327, 57903, 155951, 285489, 806187, 556533, 255137, 4654235, 8457869, 32184637, 24329253, 97583431, 82007109, 197108441, 1020579689, 35019623},
  {1, 1, 3, 7, 21, 39, 117, 191, 169, 73, 413, 3417, 2279, 3019, 13801, 22169, 86883, 59813, 149395, 324717, 165343, 3168227, 5171399, 4430075, 4483821, 3312093, 101252039, 56061123, 227573209, 93555451, 1188153185},
  {1, 1, 1, 13, 1, 31, 57, 195, 231, 321, 367, 1027, 8139, 6027, 15051, 53127, 33939, 31637, 482799, 893241, 1030169, 2173127, 4237941, 4170853, 2997061, 19368165, 20030789, 204823753, 143996997, 945109611, 1741996157},
  {1, 3, 7, 3, 11, 29, 47, 161, 71, 419, 1721, 437, 7039, 13721, 21977, 11525, 13077, 32563, 150797, 139687, 1958637, 3253589, 4238839, 12384255, 25283597, 37484975, 56566339, 136805967, 490005719, 210138833, 864616515},
  {1, 1, 7, 3, 11, 9, 43, 65, 157, 1, 1851, 823, 403, 14251, 21997, 11845, 9413, 167911, 3809, 382095, 1408279, 2559779, 7879281, 11342441, 4902325, 15854741, 104374467, 258362311, 180297559, 607565125, 1038225479},
  {1, 1, 1, 5, 21, 15, 31, 101, 293, 299, 127, 1321, 6975, 13867, 16667, 30671, 13015, 149869, 461033, 965427, 1790039, 1160025, 7665601, 15962303, 29556085, 64334445, 35836101, 159473985, 48673617, 391228123, 1943375883},
  {1, 1, 7, 1, 27, 1, 11, 229, 241, 705, 43, 1475, 2043, 4779, 30029, 30535, 118225, 96207, 82881, 576491, 29555, 1668027, 2675137, 1773789, 17502517, 28510149, 43600323, 121997765, 418207559, 902697029, 724373543},
  {1, 3, 7, 1, 5, 15, 73, 183, 193, 55, 1345, 49, 1463, 15001, 28985, 64839, 39903, 122109, 353567, 363981, 693015, 2113213, 5752023, 5852283, 33017133, 47773503, 124335811, 4843085, 287289049, 417417163, 2040934429},
  {1, 3, 3, 3, 19, 3, 55, 21, 169, 663, 1675, 137, 3855, 7273, 17085, 4397, 9581, 19741, 425289, 1017731, 1455611, 3759901, 1086573, 9089335, 4101893, 42376207, 124512199, 240292943, 135671903, 1007845583, 1091910715},
  {1, 1, 1, 13, 7, 21, 69, 67, 373, 965, 1273, 2279, 5915, 347, 21483, 59879, 1845, 189747, 203047, 296913, 2014203, 1292519, 8227019, 11994397, 21038493, 44643589, 84765125, 215193801, 102668739, 786862177, 1453841433},
  {1, 1, 7, 7, 21, 23, 17, 43, 341, 845, 465, 3355, 7707, 7635, 7701, 48313, 110251, 249357, 87663, 163481, 1301171, 3772471, 5559995, 13332521, 20867613, 51799445, 42168067, 102865155, 311800585, 624899851, 1073724229},
  {1, 3, 5, 5, 25, 5, 81, 101, 233, 139, 359, 2057, 4823, 2481, 4979, 415, 110691, 193575, 410731, 412727, 949523, 1263145, 3897793, 15721143, 20327669, 17294735, 53701377, 18167561, 523899661, 54995217, 192594229},
  {1, 1, 3, 11, 15, 39, 55, 3, 471, 765, 1143, 3941, 2431, 11603, 8177, 4317, 88169, 23945, 205033, 317757, 1375169, 1010755, 2135709, 4227035, 32775205, 44315677, 106178823, 253558031, 128221443, 932971275, 1887558923},
  {1, 1, 7, 15, 9, 57, 81, 79, 215, 433, 333, 3855, 3811, 5203, 2709, 19249, 104935, 205915, 492083, 555709, 1916665, 1206127, 5948511, 8210181, 2507309, 18412965, 12821251, 141389067, 171297045, 1061068549, 277786485},
  {1, 1, 5, 5, 19, 45, 83, 31, 209, 363, 701, 1303, 7235, 1011, 26903, 10815, 41449, 23931, 258361, 492297, 606015, 800313, 6960955, 12671933, 30560605, 41090853, 42567937, 73873665, 148074247, 767641913, 2051402527},
  {1, 3, 7, 5, 1, 13, 55, 163, 435, 807, 287, 2031, 3547, 9137, 11825, 49211, 87419, 150495, 517885, 621629, 203781, 2023025, 5671745, 6827713, 14164501, 54690647, 57339139, 77630729, 480294173, 1067614489, 321229107},
  {1, 3, 3, 7, 3, 3, 17, 197, 39, 169, 489, 1769, 511, 1489, 18773, 7889, 71613, 82261, 258183, 21431, 1250621, 1832951, 5451147, 16062731, 17003581, 32614047, 54122247, 265987851, 227208463, 956737311, 1847196445},
  {1, 1, 3, 5, 29, 43, 87, 161, 289, 339, 1233, 2353, 415, 851, 22833, 37875, 125319, 239745, 153777, 102367, 1894451, 284961, 408199, 9537795, 23626093, 6000093, 15318279, 123661057, 341214993, 136990015, 577592099},
  {1, 3, 3, 9, 21, 9, 77, 1, 453, 167, 1643, 2227, 4627, 977, 5973, 36703, 39511, 135251, 412719, 337099, 1000535, 3827197, 6349493, 4811317, 29515877, 44460487, 51278599, 194414341, 455600921, 157798701, 1877403929},
  {1, 1, 7, 1, 15, 7, 67, 33, 193, 241, 1031, 2339, 5827, 12579, 805, 55279, 96109, 146297, 463149, 735495, 1460323, 2456019, 7887301, 2183565, 33015653, 64414661, 31372291, 118154245, 161199123, 978000899, 2114489471},
  {1, 3, 1, 11, 1, 63, 45, 65, 265, 661, 849, 1979, 6411, 6689, 17319, 65321, 119415, 129117, 121203, 553979, 1516459, 338791, 7864027, 981625, 32827933, 31259735, 42523653, 126514183, 29281285, 788458515, 1819913257},
  {1, 3, 1, 13, 19, 49, 3, 11, 159, 213, 659, 2839, 1979, 14177, 12327, 62191, 15209, 206839, 370881, 916077, 621289, 1548875, 2744665, 14002577, 28684981, 65023671, 127951877, 115383297, 283231255, 90403845, 2116478031},
  {1, 3, 5, 11, 9, 29, 27, 227, 253, 449, 1403, 3427, 8115, 9569, 3427, 63777, 50367, 107695, 185025, 1004277, 1379291, 1706395, 2158297, 5300853, 16513429, 22860327, 19975169, 151301127, 267187229, 596452401, 953121855},
  {1, 1, 3, 1, 7, 3, 77, 143, 277, 779, 1499, 475, 6451, 5315, 1089, 12743, 67525, 102861, 226187, 628693, 1048395, 905185, 1945485, 10510893, 6449477, 22631141, 105844743, 255291397, 320819211, 980406279, 1070182481},
  {1, 1, 1, 5, 11, 23, 87, 131, 393, 849, 193, 3189, 6935, 14147, 1987, 13511, 56513, 174529, 9017, 767853, 1293775, 535555, 4751779, 14256439, 24615229, 24355501, 84537349, 124563457, 91017231, 645779459, 2001981563},
  {1, 3, 5, 11, 3, 3, 89, 9, 449, 243, 1501, 1739, 4299, 5825, 7363, 54337, 18133, 115781, 192287, 552315, 1505683, 2263729, 948731, 212657, 20586525, 16748055, 71233537, 255086599, 35491863, 68423727, 436514901},
  {1, 3, 1, 9, 29, 29, 113, 15, 65, 611, 135, 3687, 5755, 13953, 14727, 1675, 7055, 167335, 410059, 880493, 25311, 2603873, 7585729, 5395177, 17204021, 44744183, 109756421, 31211525, 292405273, 655278137, 1384103941},
  {1, 1, 1, 9, 21, 19, 39, 151, 395, 501, 1339, 959, 2725, 15873, 16897, 45569, 115209, 186901, 382483, 12839, 1688215, 3508107, 3901429, 14717755, 11646399, 66628773, 22806529, 61079553, 492568577, 648806409, 170131477},
  {1, 3, 7, 1, 7, 35, 45, 33, 119, 225, 1631, 1695, 1459, 2817, 13571, 59655, 81665, 219399, 124707, 169773, 125729, 3737975, 7875553, 6719839, 32497567, 49769139, 77529089, 56557571, 368246791, 608239617, 432209927},
  {1, 1, 1, 3, 25, 55, 37, 79, 167, 907, 1075, 271, 4059, 11521, 29441, 60673, 123139, 52505, 507191, 586021, 1604943, 511911, 5249163, 15974195, 24307727, 23886043, 82771969, 115539969, 307691521, 922550275, 28114969},
  {1, 3, 5, 13, 5, 13, 53, 165, 437, 67, 1705, 3177, 8095, 129, 30851, 49285, 39821, 20101, 279181, 945589, 1532453, 3272245, 241859, 11499049, 30577897, 15577887, 3391489, 2899971, 361021445, 948256781, 2040479749},
  {1, 3, 3, 13, 27, 57, 95, 55, 443, 245, 1945, 1725, 1929, 11905, 19587, 21123, 108941, 172443, 155577, 213471, 1147319, 223547, 5147253, 4794905, 22736445, 18577417, 79380481, 257769475, 258588675, 65126413, 606093339},
  {1, 3, 1, 9, 5, 33, 109, 35, 99, 827, 341, 2401, 2411, 11137, 20611, 3713, 12937, 181637, 491937, 897261, 161699, 1260771, 4936891, 5122261, 16045793, 3027179, 27246593, 217137155, 20692993, 374423561, 682704901},
  {1, 1, 5, 9, 7, 33, 43, 39, 87, 799, 635, 3481, 7159, 7617, 31169, 29637, 90825, 68167, 78305, 793707, 1444967, 1148183, 7552351, 10708923, 27730393, 7005495, 69996545, 139218945, 6066181, 609513481, 61739015},
  {1, 3, 1, 1, 31, 15, 45, 27, 337, 113, 987, 2065, 2529, 10945, 16451, 63553, 6849, 57439, 394319, 285805, 2069723, 2796817, 7314481, 10975259, 13909969, 29896353, 85209089, 32772099, 469504001, 198569985, 1974792223},
  {1, 1, 5, 9, 5, 15, 105, 123, 479, 289, 1609, 2177, 4629, 10433, 17857, 48709, 116809, 223813, 332623, 413225, 183867, 3278495, 7306593, 2328841, 5778753, 35300949, 99618817, 101732353, 319852549, 862998537, 1999695877},
  {1, 3, 5, 11, 31, 47, 97, 87, 385, 195, 1041, 651, 3271, 12993, 14659, 16325, 40267, 35295, 30959, 922401, 946071, 2988481, 1190147, 16363217, 21240395, 56615815, 71643137, 112676867, 156479493, 60616715, 579727391},
  {1, 1, 3, 7, 17, 3, 101, 55, 87, 629, 1687, 1387, 2745, 15041, 7361, 15939, 102087, 183249, 191811, 1009445, 519543, 3798423, 5840565, 11939287, 9858731, 66463481, 38473729, 104804353, 325414915, 602673159, 1873096721},
  {1, 3, 5, 5, 7, 21, 9, 237, 313, 549, 1107, 117, 6183, 9281, 16707, 53829, 66501, 151111, 50389, 436041, 1564973, 2374521, 2637413, 14639763, 2074165, 65659751, 87240705, 34951171, 432492549, 572616709, 2107789319},
  {1, 1, 3, 9, 9, 5, 55, 201, 487, 851, 1103, 2993, 4055, 14657, 2241, 20419, 45769, 118089, 26821, 650103, 524681, 344615, 6178323, 4147983, 11033073, 38923543, 107941889, 198258689, 175607811, 673288201, 673271817},
  {1, 1, 5, 9, 31, 19, 59, 7, 363, 381, 1167, 2057, 5715, 11329, 10561, 56005, 85321, 169439, 16467, 469499, 663751, 1613995, 5857725, 1222607, 20898633, 31731987, 100716545, 196874241, 361025541, 686600201, 274165791},
  {1, 3, 3, 15, 23, 63, 19, 227, 387, 827, 487, 1049, 7471, 11585, 11587, 37699, 18511, 39383, 306047, 740307, 396451, 1209539, 4746619, 9156519, 22027353, 24099439, 11530241, 5533699, 467939331, 587051023, 450113559},
  {1, 3, 1, 5, 23, 25, 61, 245, 363, 863, 963, 3583, 6475, 6849, 32067, 46913, 37957, 205143, 20057, 19197, 1432501, 2412971, 6820255, 5068163, 11048255, 4046219, 105246721, 205213699, 190812161, 285487109, 12046359},
  {1, 1, 5, 1, 5, 27, 81, 85, 275, 49, 235, 3291, 1195, 7361, 31041, 49861, 125761, 17477, 114267, 465041, 1490325, 3733459, 5497329, 271915, 32678811, 64141931, 52023297, 33755137, 383995909, 72880129, 1339617285},
  {1, 1, 5, 7, 23, 53, 85, 107, 511, 779, 1265, 1093, 7859, 11233, 9569, 21029, 38055, 143991, 489493, 907381, 833739, 1829151, 8374763, 2832977, 16328421, 35847123, 113417217, 121232385, 55596037, 17945607, 1580411927},
  {1, 3, 3, 1, 9, 21, 75, 219, 59, 485, 1739, 3845, 1109, 11617, 32419, 26467, 56993, 230569, 138805, 719403, 437947, 2246171, 7265989, 4542059, 18490469, 35072309, 97944577, 13018115, 227857411, 624733185, 556723209},
  {1, 3, 5, 1, 13, 41, 19, 143, 293, 391, 2023, 1791, 4399, 10657, 30691, 30693, 104161, 16109, 222985, 819059, 1158959, 2367493, 287527, 4286407, 18441631, 32599503, 76317697, 49591299, 451093509, 70044673, 2066725901},
  {1, 3, 7, 15, 21, 13, 21, 195, 215, 413, 523, 2099, 2341, 993, 6819, 3687, 89455, 93621, 508717, 342901, 1654051, 1828919, 4261949, 7535787, 1818003, 62663749, 7257089, 32887811, 387560455, 253547535, 707234837},
  {1, 1, 1, 3, 29, 51, 47, 57, 135, 575, 943, 1673, 541, 9697, 19169, 52961, 75811, 198333, 385299, 395023, 1523481, 3009511, 1348255, 7206607, 28209769, 43462205, 48466945, 117437441, 173362177, 992558083, 2140759069},
  {1, 3, 5, 1, 9, 13, 113, 175, 447, 115, 657, 4077, 5973, 8033, 27107, 15781, 95265, 152105, 174765, 149137, 398479, 2794527, 7595987, 4437937, 32061773, 42688565, 17155073, 59827203, 231896069, 259195905, 274365449},
  {1, 1, 1, 11, 17, 41, 37, 95, 297, 579, 911, 2207, 2387, 9889, 10977, 41569, 9067, 210673, 433289, 94853, 337919, 2807241, 2713059, 13815343, 19239871, 46873395, 1469441, 145523713, 98132993, 422794251, 501496849},
  {1, 3, 5, 3, 23, 11, 23, 231, 93, 667, 711, 1563, 7961, 5921, 23523, 11621, 40931, 120759, 150187, 268279, 199111, 1265853, 7655931, 5117735, 11254907, 45572473, 118410241, 100506627, 102460421, 580361219, 390913047},
  {1, 1, 7, 3, 17, 59, 13, 181, 141, 991, 1817, 457, 1711, 8737, 15969, 1703, 26019, 92273, 86939, 375981, 1981909, 2785133, 3481983, 615673, 11265513, 33700495, 29379585, 214729729, 80555015, 61764611, 1892848657},
  {1, 3, 3, 5, 31, 59, 81, 205, 245, 537, 1049, 997, 1815, 6625, 25187, 35171, 40677, 53183, 413147, 553073, 28781, 2142357, 6369465, 2028729, 23615237, 13990647, 88974337, 192465923, 327760899, 413807621, 1485571103},
  {1, 3, 7, 5, 17, 13, 9, 79, 17, 185, 5, 2211, 6263, 7137, 12643, 1255, 21285, 244785, 482285, 362025, 1096303, 3287921, 452057, 3121125, 19128515, 15433175, 90088449, 102413315, 386524167, 93828101, 533425169},
  {1, 3, 7, 13, 7, 53, 61, 145, 13, 285, 1203, 947, 2933, 2017, 24291, 42791, 33965, 185447, 132949, 65053, 802481, 344685, 6785213, 10124755, 14860115, 28582933, 30966785, 218622979, 17531911, 568460301, 1764463623},
  {1, 1, 7, 3, 31, 19, 69, 217, 47, 441, 1893, 673, 4451, 6817, 27617, 46567, 3171, 65343, 221171, 332581, 1173881, 3886735, 5482905, 1130437, 19573697, 26274307, 29793281, 45583361, 301923335, 961176579, 413651999},
  {1, 1, 1, 1, 25, 9, 23, 225, 385, 629, 603, 3747, 4241, 9185, 32353, 13985, 22561, 243577, 403945, 892151, 1350657, 638305, 3478933, 5214651, 3552899, 15890929, 111334401, 22000641, 113112065, 78384129, 1122423833},
  {1, 3, 1, 9, 5, 37, 31, 237, 431, 79, 1521, 459, 2523, 6433, 11427, 57953, 80553, 112933, 233285, 489919, 2022541, 1508687, 5248559, 7256209, 12430827, 34487611, 1463297, 46842883, 276775937, 78441481, 1844192261},
  {1, 3, 7, 3, 9, 43, 105, 179, 5, 225, 799, 1777, 4893, 3297, 6371, 23847, 62819, 193641, 446411, 468681, 1505299, 395301, 2792065, 14081471, 16235921, 60908221, 105708545, 13341699, 339860487, 172258307, 519322633},
  {1, 1, 3, 1, 29, 45, 29, 159, 267, 247, 455, 847, 3909, 7713, 9633, 38563, 111009, 11005, 325837, 443645, 1164863, 3330923, 3922071, 559527, 2403119, 32237413, 49402881, 170230785, 8217603, 815326209, 1851288605},
  {1, 1, 3, 7, 25, 21, 121, 57, 467, 275, 719, 1521, 7319, 11681, 8993, 56739, 10087, 59961, 497269, 525977, 207769, 2417779, 5767027, 41647, 29384721, 866935, 4541441, 65770497, 493937667, 1030763527, 1110570009},
  {1, 3, 1, 3, 11, 35, 119, 123, 81, 979, 1187, 3623, 4293, 14945, 675, 64801, 115363, 119275, 199619, 995287, 120603, 1541873, 3672051, 9982083, 3403911, 37284581, 49740801, 191298563, 384105473, 269790211, 721394699},
  {1, 1, 1, 7, 15, 25, 121, 235, 25, 487, 873, 1787, 1977, 11873, 2081, 13665, 52647, 193647, 208121, 1009049, 560907, 3625785, 3613255, 2122505, 8301851, 49637657, 63425537, 264825857, 49075201, 829805575, 1188002831},
  {1, 1, 1, 11, 3, 7, 17, 135, 345, 353, 383, 4011, 2573, 7185, 13809, 58545, 35547, 69395, 268567, 725249, 190263, 1140329, 8234513, 8738671, 12958427, 47234717, 74258177, 48562433, 378222337, 366635275, 1315049731},
  {1, 3, 7, 15, 27, 13, 97, 123, 65, 675, 951, 1285, 6559, 16177, 17011, 4951, 20735, 79883, 387133, 661521, 452747, 3746289, 6339187, 576935, 23550645, 59438479, 77544705, 6099203, 456624903, 618197263, 1586066203},
  {1, 3, 7, 3, 7, 1, 71, 19, 325, 765, 337, 1197, 2697, 2257, 22547, 16343, 119699, 143895, 385585, 395831, 2044451, 3349301, 7882989, 14412577, 2401181, 28844249, 57094401, 101922563, 440345351, 549336323, 4189447},
  {1, 3, 5, 1, 31, 37, 11, 71, 169, 283, 83, 3801, 7083, 1713, 20531, 28821, 47537, 165391, 219669, 602715, 849495, 3496793, 5502283, 9317091, 3773609, 9858875, 48043265, 45570819, 172054277, 217995521, 1041910559},
  {1, 1, 3, 15, 17, 29, 83, 65, 275, 679, 1749, 4007, 7749, 13297, 24017, 58355, 18399, 238977, 302733, 1011171, 300337, 1876355, 107511, 16115557, 25898039, 57864437, 68144385, 143762177, 249655555, 89998095, 122959633},
  {1, 1, 3, 1, 21, 11, 41, 95, 237, 361, 1819, 2783, 2383, 7185, 3121, 54643, 84593, 64133, 473755, 176793, 1461711, 3891773, 5807961, 11051019, 21800623, 16955679, 55401217, 14866689, 519933699, 1020460289, 596580629},
  {1, 3, 7, 11, 29, 57, 111, 187, 465, 145, 605, 1987, 8109, 12433, 18003, 42679, 118875, 242061, 203401, 11167, 1070987, 1093249, 5332353, 1227821, 32863219, 8053053, 101042433, 129752835, 308759303, 769605899, 2121220381},
  {1, 1, 3, 3, 19, 15, 55, 83, 357, 1001, 643, 1517, 6529, 8209, 17585, 36179, 70547, 85891, 497823, 899719, 1246947, 2008789, 2135193, 8812147, 3275613, 4977233, 27254529, 247973633, 200778499, 952571139, 586504467},
  {1, 3, 1, 5, 29, 35, 73, 23, 77, 619, 1523, 1725, 8145, 5937, 30643, 56401, 124821, 196493, 294035, 844505, 388807, 2543133, 2674395, 7940579, 20180813, 52952193, 27251457, 110074115, 529467137, 141516037, 682400541},
  {1, 1, 5, 5, 19, 23, 7, 197, 449, 337, 717, 2921, 315, 4177, 12433, 7893, 7413, 174915, 435143, 23831, 1998805, 3703985, 4443233, 12052349, 31482873, 63126507, 31119105, 165693185, 396395781, 320897285, 1226838291},
  {1, 3, 5, 9, 7, 63, 117, 97, 97, 813, 1925, 2817, 1579, 7313, 26131, 39253, 125433, 226135, 133583, 656229, 147761, 2092625, 2505629, 12659221, 7526865, 43065467, 86834945, 98710787, 210411781, 127869193, 1867214087},
  {1, 1, 1, 11, 31, 7, 25, 235, 231, 133, 1007, 1371, 1553, 4241, 17713, 51953, 112859, 110031, 405847, 53577, 1891099, 2465367, 1709365, 8786623, 702251, 66908961, 91113217, 265967873, 496675585, 159295243, 984892703},
  {1, 1, 7, 5, 19, 7, 47, 171, 267, 243, 1331, 567, 6033, 13553, 24817, 23223, 129237, 210883, 339543, 1021471, 1154747, 1070075, 6853571, 2806019, 2111303, 60552929, 20858625, 180776705, 38726407, 201509637, 1740878611},
  {1, 1, 5, 1, 7, 49, 55, 89, 109, 735, 1455, 3193, 6239, 6321, 28849, 2901, 79153, 244823, 467425, 711207, 157193, 3003229, 8202127, 8046879, 29365161, 9813327, 125956353, 28997377, 130423045, 838020353, 1950654215},
  {1, 1, 1, 7, 1, 61, 9, 103, 3, 929, 1481, 2927, 2957, 4945, 29649, 33585, 79383, 65105, 327149, 433625, 61911, 1398227, 4369457, 6638489, 24603103, 40843901, 99859713, 122325249, 152364801, 95191815, 413606145},
  {1, 1, 5, 13, 17, 21, 75, 49, 255, 1019, 1161, 2133, 1177, 12145, 30929, 39189, 89437, 45505, 158917, 667867, 1895457, 1772591, 2293611, 3522297, 25934469, 25018153, 29267201, 259574529, 195342597, 198475533, 1407548177},
  {1, 3, 1, 3, 13, 15, 41, 247, 211, 409, 1163, 523, 2635, 15825, 22771, 62065, 49139, 81245, 354687, 173049, 1207943, 909251, 4770601, 10128091, 11686459, 47244475, 66293505, 174041347, 344618753, 699093763, 1774131469},
  {1, 3, 7, 7, 21, 59, 91, 149, 479, 391, 681, 2311, 6249, 3505, 32467, 16695, 116311, 176965, 98251, 209643, 694949, 4128463, 5805687, 3396569, 28767767, 7449817, 65911553, 151444227, 259396871, 324689671, 68683541},
  {1, 1, 5, 11, 27, 53, 21, 211, 197, 815, 719, 1605, 255, 8785, 12561, 17845, 3931, 127723, 484101, 455013, 744387, 1056021, 2074143, 14511167, 28425941, 33407439, 94267649, 142264577, 127391493, 320373003, 1871254299},
  {1, 1, 3, 3, 9, 33, 59, 3, 323, 1, 101, 1135, 8105, 8561, 13681, 20819, 117875, 199481, 37905, 695339, 1907795, 202163, 7888689, 6097909, 20107071, 13651961, 41999617, 228577025, 186980099, 634658563, 1990163721},
  {1, 3, 3, 1, 29, 5, 17, 141, 51, 991, 841, 327, 3859, 10897, 31347, 37971, 89809, 86125, 49173, 180289, 1556733, 2802115, 3096623, 7815929, 21679031, 1990019, 14831873, 51836163, 165914883, 933349121, 1537268509},
  {1, 3, 1, 5, 11, 19, 23, 89, 175, 173, 165, 2881, 1881, 3249, 27507, 57137, 115509, 39611, 219523, 80359, 341417, 1993791, 831357, 216117, 21039281, 60233737, 37951233, 230640387, 407856385, 1007149829, 2020907275},
  {1, 1, 1, 15, 13, 51, 87, 39, 495, 611, 1341, 1531, 7029, 593, 12849, 59793, 4031, 70461, 133763, 690151, 61303, 2834271, 1122163, 16162477, 11098859, 21698533, 68501249, 252924673, 24984321, 426279695, 861499149},
  {1, 1, 3, 11, 13, 55, 75, 185, 57, 61, 1917, 2051, 5965, 14193, 22257, 22803, 87099, 71229, 130695, 231387, 2034665, 1649161, 2478189, 3289517, 31641971, 61412221, 19475201, 90664705, 522827011, 246261003, 303368461},
  {1, 1, 5, 5, 7, 53, 11, 217, 213, 933, 921, 3607, 5175, 7281, 28337, 15829, 53781, 107639, 162693, 969723, 1419433, 3098757, 932501, 11878153, 22743399, 1773767, 131535105, 8480513, 256769285, 712190213, 427585799},
  {1, 3, 3, 5, 17, 53, 103, 251, 369, 781, 1319, 3717, 4439, 6769, 9043, 41267, 91221, 184673, 238757, 981431, 1861195, 445249, 4648445, 6968695, 8879957, 71079, 109261057, 227635971, 334176003, 553934085, 1282314513},
  {1, 3, 5, 13, 1, 39, 25, 235, 321, 773, 251, 3111, 6397, 3569, 16787, 37013, 123581, 97841, 260535, 370857, 841243, 2626417, 6293973, 2628875, 15323959, 51187725, 40043265, 141503747, 236259077, 426349837, 1382676737},
  {1, 1, 7, 3, 31, 5, 25, 29, 325, 385, 1313, 127, 4705, 16369, 31601, 60311, 121267, 185583, 330101, 455433, 380301, 3482645, 1945585, 4978897, 18259151, 4167889, 18393345, 89769217, 154364167, 233226499, 1850643743},
  {1, 1, 5, 15, 15, 27, 15, 85, 239, 243, 1633, 3473, 2621, 13201, 10161, 40437, 101823, 120575, 324203, 301375, 1455429, 1247679, 5016163, 15854385, 8161505, 14518957, 126846721, 227382017, 68622597, 980595471, 984205071},
  {1, 3, 3, 3, 9, 19, 113, 13, 137, 165, 25, 2957, 7549, 12849, 26995, 22707, 16947, 120313, 314435, 947041, 457629, 2631609, 4873589, 3272233, 12812477, 27366157, 121093889, 117159171, 30253315, 767667459, 1858285833},
  {1, 3, 1, 3, 11, 21, 3, 97, 417, 183, 1205, 1437, 247, 8721, 18163, 37809, 57491, 253883, 406981, 691251, 1770673, 1590577, 3918919, 12931301, 19987117, 27755015, 7190785, 106221315, 36541697, 882855171, 1567740683},
  {1, 1, 7, 3, 17, 21, 125, 55, 67, 387, 385, 2323, 887, 9361, 8625, 17111, 41555, 203809, 293349, 622701, 622759, 4030707, 1081907, 3178673, 28344451, 49389287, 112900353, 264149249, 151323399, 451041027, 213909265},
  {1, 3, 5, 5, 29, 11, 103, 223, 233, 641, 133, 415, 1297, 5105, 29491, 6165, 111797, 68077, 61019, 486935, 1657583, 2424537, 4589265, 13075157, 16090191, 41916321, 103726849, 89387779, 314430213, 676680965, 1052761373},
  {1, 3, 3, 11, 1, 9, 5, 189, 235, 1007, 1363, 3985, 889, 4049, 3347, 51123, 14459, 120689, 516185, 471573, 132781, 2867035, 794527, 7331427, 20059297, 40536361, 107971841, 158982403, 74807555, 299179787, 1094171393},
  {1, 3, 7, 9, 23, 19, 19, 183, 269, 403, 1643, 3559, 5189, 7249, 28307, 56023, 73081, 176487, 154435, 591427, 81415, 383645, 6763011, 1242875, 5234231, 43422901, 11048705, 196456195, 264084231, 254337289, 1346973463},
  {1, 3, 7, 3, 29, 45, 17, 69, 475, 149, 1291, 2689, 7625, 9969, 26451, 25783, 16371, 181997, 13053, 350401, 1092821, 1868843, 1287845, 3687291, 27378993, 65881625, 12665089, 120931587, 193319175, 374685443, 1318445341},
  {1, 3, 7, 3, 27, 37, 41, 73, 253, 1001, 431, 1111, 7887, 5873, 16659, 4439, 125011, 233899, 517237, 710649, 624857, 3520877, 1800601, 13154527, 19056935, 19454207, 35178753, 96429315, 529005831, 361298179, 833252635},
  {1, 1, 7, 5, 3, 7, 87, 143, 289, 495, 631, 3011, 6151, 1937, 24657, 50327, 86357, 82867, 55927, 1037767, 124671, 241393, 2179391, 4900775, 22202259, 7194551, 35189505, 133774081, 272970503, 998907141, 1410724611},
  {1, 1, 1, 13, 5, 45, 17, 167, 23, 975, 801, 1975, 6833, 7433, 6585, 37257, 89157, 8653, 24677, 888377, 1579727, 2346815, 5064199, 11247961, 2452047, 44744329, 35042241, 193187777, 320963009, 253253197, 1045074885},
  {1, 3, 1, 11, 7, 21, 39, 23, 213, 429, 1301, 2059, 197, 13993, 6747, 55337, 35651, 51423, 346973, 778303, 1695983, 2859069, 5677653, 14263405, 31484899, 13739117, 2898881, 190582595, 50107713, 793058123, 1074723015},
  {1, 3, 3, 15, 3, 57, 121, 133, 29, 711, 1961, 2497, 189, 3913, 6155, 39915, 35911, 185947, 376161, 504897, 1891421, 1641749, 5095119, 8851265, 14033769, 20561093, 118691649, 139412803, 492549059, 251121743, 1068292675},
  {1, 1, 3, 5, 11, 55, 115, 137, 233, 673, 985, 2849, 5911, 4553, 8713, 2491, 98125, 84291, 81135, 652059, 1668497, 1820305, 6339449, 101457, 4922025, 29239263, 54813761, 68199489, 536469059, 669652037, 646949451},
  {1, 1, 7, 15, 29, 45, 1, 241, 329, 323, 925, 2821, 3331, 11753, 3369, 52511, 115271, 85589, 356821, 15161, 1516889, 217825, 1088907, 11547221, 9232141, 56535131, 38032961, 12695617, 28598983, 701863247, 1674494813},
  {1, 1, 5, 7, 13, 31, 81, 105, 199, 145, 195, 1365, 5119, 5913, 27049, 57597, 127439, 56901, 348919, 327977, 1377089, 1170559, 1631961, 4213195, 17440685, 21733943, 80728129, 104210369, 452638277, 487551687, 1831005005},
  {1, 3, 7, 11, 3, 55, 11, 31, 117, 343, 1265, 1837, 2451, 1033, 14571, 32799, 120259, 210395, 251983, 189843, 1445959, 2044301, 7721039, 11899209, 8845509, 14371435, 37016001, 261235523, 478860231, 1037537867, 2122510659},
  {1, 1, 3, 7, 29, 57, 61, 179, 429, 591, 177, 1945, 2159, 2313, 16201, 6139, 61903, 205013, 427361, 205765, 817307, 3115045, 8309735, 4815337, 16663089, 19939799, 133293761, 232540481, 216316355, 309831, 2051165},
  {1, 3, 5, 11, 23, 49, 101, 137, 339, 323, 1035, 1749, 7737, 985, 30507, 29005, 35907, 139215, 183257, 68605, 1969009, 291611, 7320363, 4608771, 24812173, 62948577, 498753, 188712643, 203360965, 572424395, 1453155159},
  {1, 3, 1, 13, 21, 35, 55, 79, 19, 269, 1055, 2651, 7083, 1145, 7771, 40921, 53829, 119885, 235627, 217247, 442791, 1860555, 5695093, 9416359, 30061443, 56069251, 24488385, 208515779, 24403265, 930408909, 1230414037},
  {1, 3, 3, 11, 9, 9, 95, 167, 437, 361, 1185, 4083, 603, 13705, 19739, 27995, 75715, 257873, 384081, 474823, 950191, 239037, 3708881, 12097881, 22285211, 38012755, 30157761, 149696323, 70005827, 627414603, 81077961},
  {1, 1, 1, 7, 31, 61, 77, 65, 489, 657, 691, 2423, 4147, 13785, 505, 30169, 4623, 156183, 315957, 22133, 384697, 3134977, 8332025, 7824059, 6808639, 45574843, 73940545, 262672449, 99110465, 452583111, 1708810207},
  {1, 3, 5, 7, 21, 37, 87, 191, 311, 453, 2013, 829, 2619, 14121, 31275, 51869, 121871, 228365, 154637, 693359, 1635351, 160799, 2125693, 16262693, 4007557, 16904211, 20166081, 198236099, 417279813, 544037191, 2092265813},
  {1, 1, 5, 9, 17, 47, 35, 101, 5, 813, 1157, 1279, 7365, 13161, 21689, 65389, 41985, 80921, 19463, 527467, 1240301, 3360125, 4128309, 13253549, 2210007, 8137133, 60074817, 100896577, 192650565, 1025735497, 1923761617},
  {1, 1, 5, 3, 11, 35, 113, 199, 369, 721, 901, 1471, 7801, 4329, 32441, 39565, 123147, 16643, 472331, 798569, 444831, 72041, 3246169, 12753085, 18953527, 20872177, 114882497, 198543937, 174040645, 258154307, 1275133387},
  {1, 3, 1, 5, 9, 61, 83, 157, 391, 739, 1957, 2123, 4341, 14697, 30363, 51305, 2829, 153361, 31029, 853883, 860629, 2852207, 4788091, 13340237, 13542771, 39187181, 132695361, 170970051, 488771265, 799507269, 1843727561},
  {1, 3, 5, 11, 19, 19, 111, 225, 383, 219, 997, 717, 7505, 12521, 22827, 50125, 29571, 125067, 150715, 847543, 1923833, 2220647, 70435, 13997933, 20044469, 29316105, 26234433, 75368899, 87439557, 924724043, 945675603},
  {1, 3, 1, 11, 13, 63, 35, 127, 209, 831, 501, 3017, 3507, 6633, 32139, 13017, 91523, 175253, 79543, 956411, 1964695, 1949353, 1198247, 1677709, 10174913, 6833611, 92984769, 184056387, 208198081, 752448331, 1335607373},
  {1, 3, 7, 9, 29, 7, 11, 163, 81, 563, 1445, 3215, 6377, 13417, 17851, 3855, 69505, 163973, 70079, 691907, 1620683, 3325929, 7309291, 5228093, 26755719, 19973361, 134169281, 242567747, 510077639, 940187465, 297472477},
  {1, 3, 7, 11, 25, 3, 39, 195, 491, 45, 839, 4021, 4899, 10873, 20971, 62687, 36739, 261249, 385979, 5887, 741771, 3086195, 3736469, 8716767, 28641645, 203979, 54858433, 111373507, 81859911, 650553291, 41036633},
  {1, 3, 7, 15, 13, 5, 67, 143, 117, 505, 1281, 3679, 5695, 15017, 6619, 58847, 62119, 103733, 69597, 731675, 1439431, 2225789, 797569, 12897433, 29782743, 59707095, 109522369, 182012739, 476419143, 1040927311, 1113041357},
  {1, 3, 7, 9, 9, 19, 21, 221, 147, 763, 683, 2211, 589, 15673, 25723, 21775, 28577, 193649, 336203, 85117, 593077, 584939, 1987891, 15302755, 1934939, 28678005, 15837505, 63372995, 167534791, 496889033, 1067926217},
  {1, 1, 3, 5, 21, 47, 53, 109, 299, 807, 1153, 1209, 7961, 1897, 17193, 30443, 96877, 69757, 471127, 643197, 635109, 3895347, 6230383, 5805897, 28086401, 18585281, 20548929, 252836033, 183568323, 514860613, 1733538901},
  {1, 3, 7, 11, 9, 31, 45, 43, 505, 647, 1127, 2681, 4917, 8985, 20011, 56911, 52963, 241905, 333383, 27477, 1055555, 2971297, 7048463, 7387551, 15079121, 8995373, 123362497, 66022467, 26173383, 312095691, 331593801},
  {1, 1, 5, 15, 31, 41, 63, 113, 399, 727, 673, 2587, 5259, 14825, 4409, 32125, 87847, 201655, 451553, 147751, 1156457, 2660199, 3756047, 3929289, 9666243, 50385619, 61130305, 85984705, 39880645, 857779791, 1185092831},
  {1, 1, 1, 13, 17, 53, 35, 99, 57, 243, 1447, 1919, 2831, 10521, 2201, 15769, 29733, 133177, 445149, 601899, 589707, 1902961, 6326667, 10653023, 17775255, 23871799, 10254017, 165787713, 420091585, 63634381, 1524992465},
  {1, 3, 7, 11, 23, 51, 13, 9, 49, 449, 997, 3073, 4407, 8585, 10923, 42911, 28387, 209711, 147307, 541685, 498705, 3222665, 3331593, 10852749, 12906345, 10035423, 131418177, 149020227, 175566407, 243955531, 556752471},
  {1, 3, 5, 7, 23, 33, 89, 41, 415, 53, 697, 1113, 1489, 11433, 10987, 7213, 20207, 239599, 154345, 485569, 316529, 2356407, 3970909, 6747537, 32996417, 19129241, 113696833, 165466563, 72397637, 70328391, 1370559575},
  {1, 1, 3, 7, 1, 13, 29, 13, 255, 749, 77, 3463, 1761, 11785, 14873, 23595, 112943, 210793, 122613, 462277, 887717, 3925815, 1310053, 454469, 12773151, 13253353, 57873729, 119613889, 115592515, 898565447, 1271239617},
  {1, 3, 3, 7, 13, 15, 93, 191, 309, 869, 739, 1041, 3053, 10809, 12459, 64651, 51823, 75381, 353015, 1034181, 1544695, 3231213, 2936685, 11819323, 22081625, 15351525, 100762049, 130150211, 519292739, 190225095, 335481421},
  {1, 3, 5, 13, 5, 19, 109, 211, 347, 839, 893, 2947, 7735, 9993, 20091, 25245, 32421, 197309, 388955, 135333, 1776411, 1576355, 4382287, 8528197, 1305531, 62422447, 115380929, 9755843, 219627461, 715687245, 1851310789},
  {1, 3, 1, 13, 27, 3, 119, 157, 485, 99, 1703, 3895, 573, 14041, 5275, 62921, 127461, 247459, 131627, 569983, 737893, 1898077, 6198971, 14516271, 16941311, 37653637, 127843649, 112954179, 41112257, 964937677, 801840091},
  {1, 3, 7, 11, 1, 23, 123, 105, 31, 359, 275, 1775, 3685, 11577, 2139, 1167, 93411, 49593, 210447, 872003, 792193, 3875783, 2963615, 13528763, 10981239, 11896445, 42179649, 202827203, 229125063, 667497163, 1156131265},
  {1, 3, 3, 5, 27, 11, 125, 3, 413, 199, 2043, 2895, 2945, 697, 31707, 53419, 31981, 249635, 312371, 35893, 679099, 453989, 2446415, 7172739, 30053383, 34935257, 7154369, 240215363, 287772611, 476202181, 852549083},
  {1, 3, 3, 3, 15, 49, 121, 159, 233, 543, 193, 4007, 321, 3817, 15003, 50155, 68075, 23351, 403465, 686081, 1812615, 112897, 4319287, 5782041, 5349327, 47188569, 22035905, 11545667, 433101251, 691139139, 760315855},
  {1, 1, 3, 5, 9, 47, 87, 1, 51, 1011, 1595, 2239, 6467, 6649, 30041, 39867, 16621, 239457, 393239, 131167, 393257, 393707, 7995499, 1180627, 22938567, 6424315, 120204865, 62541505, 160601795, 276535493, 1459430857},
  {1, 3, 7, 9, 1, 33, 87, 137, 469, 749, 1413, 805, 6817, 6601, 15051, 44015, 96929, 133689, 460601, 964479, 1848161, 1086205, 2539893, 1623277, 180333, 44486633, 50326593, 4696899, 221751623, 202790729, 1154813761},
  {1, 3, 1, 13, 19, 45, 95, 227, 29, 677, 1275, 3395, 4451, 3001, 30971, 60473, 42085, 209003, 78597, 916311, 752475, 1970645, 3436861, 9893699, 6912843, 59890475, 109048513, 186730819, 319359809, 963853517, 486862547},
  {1, 1, 7, 5, 7, 63, 33, 71, 443, 561, 1311, 3069, 6943, 1753, 23689, 62175, 20013, 8047, 109735, 795049, 1279519, 641635, 1166937, 15889479, 15178629, 10959863, 51798977, 122983617, 88185415, 882542021, 209694023},
  {1, 1, 1, 13, 9, 37, 23, 69, 13, 415, 1479, 1197, 861, 14681, 19273, 61257, 42277, 206113, 334989, 203999, 1649325, 453125, 6627975, 16558031, 13033637, 6057861, 109965121, 161273153, 497714625, 594206669, 1101204809},
  {1, 3, 3, 13, 27, 21, 13, 233, 105, 777, 345, 2443, 1105, 1321, 20923, 36203, 106725, 148387, 144813, 694405, 369873, 1174753, 34177, 1664561, 3615459, 35186617, 58806337, 210584643, 397624387, 372729933, 977169627},
  {1, 1, 7, 11, 23, 13, 21, 147, 221, 549, 73, 2729, 6279, 9897, 30249, 56799, 129443, 30335, 164757, 871661, 1099083, 1226325, 2107629, 13884913, 22784097, 57701455, 9277121, 228251969, 507715399, 338557003, 2014815319},
  {1, 1, 7, 7, 25, 27, 15, 45, 227, 39, 75, 1191, 3563, 12329, 29065, 46383, 12287, 138945, 74227, 597255, 1677781, 3601259, 5941039, 1969811, 6574767, 50721315, 71796289, 128048577, 170925767, 913362887, 682853593},
  {1, 1, 5, 7, 13, 49, 99, 167, 227, 13, 353, 1047, 8075, 985, 11945, 29997, 36479, 258261, 154569, 547147, 256511, 3849211, 4615877, 4891177, 24603711, 52231699, 98678465, 246920641, 515001925, 500396359, 393595597},
  {1, 1, 3, 13, 31, 9, 27, 7, 461, 737, 1559, 3243, 53, 169, 13865, 40219, 108565, 115143, 223297, 777987, 866671, 4101557, 3476713, 8032095, 25715187, 18331181, 51518145, 201223745, 338056515, 647652045, 734041567},
  {1, 3, 1, 1, 21, 41, 97, 165, 171, 821, 587, 2137, 2293, 9225, 15227, 19257, 124953, 234077, 65873, 394809, 967389, 3373811, 1892589, 12821043, 5861137, 28008877, 128640577, 244417987, 402646593, 965443521, 1806707029},
  {1, 3, 1, 11, 17, 41, 29, 187, 87, 599, 1467, 1395, 5931, 14249, 21291, 54121, 41715, 110297, 182481, 380949, 839747, 184399, 5772015, 13178355, 15898171, 38647715, 93045185, 263915203, 203684033, 473969611, 2084594129},
  {1, 1, 1, 9, 9, 49, 89, 205, 409, 453, 61, 1923, 1257, 9529, 19785, 59385, 86481, 74577, 392937, 418241, 942037, 2759489, 3656157, 5733893, 11492443, 11656849, 130243649, 75059521, 74441281, 258637897, 918195145},
  {1, 3, 7, 3, 9, 43, 89, 143, 431, 83, 1243, 1795, 3599, 3433, 29739, 46543, 85051, 119873, 80739, 810225, 1688503, 148775, 7779435, 11455987, 998347, 8603815, 24399553, 31193283, 520646087, 1065982915, 1665685961},
  {1, 3, 5, 13, 3, 25, 59, 219, 43, 223, 797, 2651, 6015, 6809, 20619, 47565, 79541, 129355, 514369, 766147, 1199955, 780563, 1038935, 1768757, 16630899, 21580407, 25777601, 220252099, 526321349, 867769933, 489294787},
  {1, 1, 5, 15, 7, 55, 65, 207, 213, 311, 1287, 1269, 6467, 12489, 27993, 23757, 47351, 3807, 457295, 304041, 467655, 3682013, 4877071, 1071919, 2556653, 16318571, 74792641, 178216513, 251479237, 884102863, 1287615559},
  {1, 3, 7, 11, 21, 57, 31, 183, 351, 857, 911, 1683, 7155, 7529, 29419, 26671, 78323, 239965, 95089, 70135, 1043695, 2162055, 318833, 12333831, 4866011, 40207243, 80186049, 214161091, 158233415, 706709195, 1539917781},
  {1, 3, 5, 11, 27, 1, 21, 47, 387, 383, 1593, 115, 3805, 10169, 15499, 41309, 13971, 79443, 344281, 352989, 1181703, 3897595, 2836871, 6004049, 17161051, 3379413, 34642113, 36545475, 384935621, 424537931, 2062848475},
  {1, 3, 1, 1, 13, 23, 87, 173, 181, 619, 1653, 3931, 6073, 9817, 18635, 35513, 105497, 105733, 511791, 542031, 701397, 442333, 603299, 3167597, 13582115, 15201457, 128936513, 159459395, 234843585, 725276225, 139041741},
  {1, 1, 7, 5, 17, 43, 37, 61, 307, 621, 1785, 55, 115, 3161, 9449, 25007, 4541, 203017, 177411, 983933, 268261, 3933307, 7185653, 8605249, 30315391, 59061547, 103967297, 246191041, 142673991, 312179525, 1436293841},
  {1, 3, 7, 15, 25, 61, 123, 15, 237, 671, 1473, 467, 1907, 10809, 6091, 25455, 17591, 44817, 168245, 883, 1147703, 1223061, 2259351, 10009865, 25862475, 30620763, 126437569, 68412995, 411181767, 751552335, 1277072857},
  {1, 1, 7, 5, 29, 57, 75, 237, 85, 699, 159, 3577, 4771, 14089, 27257, 32591, 105149, 191493, 211985, 214035, 42069, 3152973, 1853299, 3292999, 27176321, 12580395, 7967809, 195227713, 314661319, 113167557, 483918685},
  {1, 1, 1, 11, 25, 19, 51, 1, 147, 31, 895, 2617, 625, 841, 16377, 35033, 7059, 128001, 343051, 437371, 1969273, 1791387, 3994807, 13595719, 342705, 29343673, 53708097, 158436801, 247817409, 589667019, 730311513},
  {1, 3, 7, 5, 29, 15, 115, 175, 395, 391, 1141, 1827, 1181, 10521, 19851, 18975, 122205, 73749, 155655, 778283, 352279, 4055075, 680687, 140861, 15917451, 3682325, 29995713, 70702019, 387218631, 1028521029, 515243485},
  {1, 3, 5, 7, 17, 7, 11, 193, 89, 243, 561, 3787, 4551, 15225, 5003, 41677, 115327, 250265, 486047, 1012451, 411065, 2305009, 1161051, 8046601, 25850259, 56202063, 45252929, 139104067, 535650245, 858989383, 974983121},
  {1, 3, 1, 11, 7, 57, 7, 125, 403, 947, 1261, 409, 8083, 1433, 29531, 51529, 108595, 58511, 239233, 466895, 1641077, 1191851, 3775131, 1019125, 3502929, 9492923, 78146625, 254801347, 345646017, 108996171, 1110024007},
  {1, 1, 5, 13, 21, 63, 115, 233, 231, 921, 1747, 3635, 2519, 4553, 5353, 16205, 67189, 143757, 299399, 103371, 1600529, 2763983, 3046769, 5719083, 14403867, 6148335, 96392001, 118057153, 485842245, 879973837, 1316947925},
  {1, 1, 5, 11, 3, 27, 15, 91, 505, 591, 1451, 3881, 2997, 137, 11289, 5213, 48179, 146683, 359491, 926055, 1182995, 1249329, 7540775, 12726563, 17480049, 15629821, 99339201, 69896001, 445912773, 280880587, 1415460675},
  {1, 1, 3, 11, 21, 9, 109, 153, 317, 533, 593, 3967, 2797, 4473, 1353, 14267, 9203, 176493, 279393, 542309, 117409, 415621, 3962765, 3809833, 16769767, 42843125, 71238977, 33108033, 182961731, 904895307, 1895608533},
  {1, 3, 3, 13, 9, 57, 121, 245, 219, 867, 967, 791, 7095, 15465, 14251, 50987, 122837, 179233, 267249, 859137, 1141965, 1748195, 1043803, 13144735, 20355071, 22508031, 118265409, 26000835, 516120515, 402473165, 1073516745},
  {1, 1, 1, 9, 29, 21, 99, 35, 375, 959, 329, 4087, 7171, 13017, 329, 33225, 107409, 91941, 388717, 15579, 1284539, 2970351, 6277015, 927745, 21640927, 34575275, 83265473, 137540801, 87272129, 208257865, 1972931805},
  {1, 1, 1, 9, 11, 17, 17, 97, 89, 135, 631, 3809, 3253, 13929, 10777, 9177, 81329, 184691, 210153, 712617, 677193, 1017985, 146367, 9862415, 645337, 31596285, 14565185, 79562049, 236077761, 526067401, 2145447243},
  {1, 1, 1, 15, 21, 51, 91, 249, 459, 801, 757, 2353, 2033, 3929, 5241, 58809, 21879, 121069, 253899, 119251, 988481, 396579, 5664297, 4213037, 13698777, 29297449, 57608001, 255029185, 357279553, 1017275471, 1567841109},
  {1, 3, 5, 9, 23, 29, 77, 53, 399, 767, 1817, 2171, 1629, 10585, 22059, 57597, 111857, 162047, 337637, 210165, 283389, 366583, 1410407, 5141121, 11899955, 21578805, 29351489, 265270467, 495972549, 491701065, 1915543511},
  {1, 1, 3, 5, 29, 5, 43, 121, 17, 859, 1479, 3785, 6641, 809, 19673, 35963, 25085, 196837, 446189, 597203, 282721, 2846985, 8368435, 1823199, 8541041, 40392025, 44187969, 229393729, 207411651, 65401541, 570829661},
  {1, 1, 3, 7, 7, 61, 45, 109, 371, 833, 91, 153, 4553, 297, 13529, 38459, 88959, 218111, 205013, 347333, 85125, 1279051, 2131289, 6766595, 20955313, 84113, 19613761, 65366977, 55428931, 297050055, 1456841031},
  {1, 1, 3, 11, 7, 55, 81, 123, 389, 139, 1933, 891, 1789, 10937, 809, 827, 108083, 74687, 471839, 678169, 320403, 2603565, 373395, 15341221, 32817091, 36189333, 128034753, 134929985, 136346563, 1029705803, 1428348871},
  {1, 3, 7, 15, 25, 17, 93, 165, 503, 717, 1553, 1475, 1627, 6649, 10603, 17807, 80983, 106865, 371001, 617269, 197501, 3399695, 41237, 12788681, 31155339, 62022371, 13765185, 7996099, 34516039, 396310863, 1569346521},
  {1, 1, 1, 13, 13, 63, 13, 225, 357, 571, 33, 4073, 3795, 10585, 20089, 50153, 38805, 89461, 239111, 348757, 1737337, 3086317, 1548803, 6644249, 33155441, 13610875, 127819073, 204859969, 307803585, 730361677, 1137232717},
  {1, 1, 3, 11, 1, 31, 107, 145, 407, 961, 501, 2987, 103, 13337, 12009, 11371, 55187, 154617, 449591, 427043, 1266761, 2763807, 1580361, 2155565, 4335459, 48249151, 111212865, 236548289, 295302979, 387909707, 472106433},
  {1, 1, 7, 1, 23, 63, 49, 193, 173, 281, 25, 2465, 5927, 9273, 31865, 30927, 83289, 112751, 223159, 1043689, 111881, 845701, 8165633, 7548769, 14924409, 34627487, 47300929, 242615361, 520504647, 560646465, 1674480471},
  {1, 1, 7, 1, 1, 1, 85, 77, 273, 693, 349, 1239, 4503, 3673, 21257, 15279, 104601, 53369, 138377, 714381, 1463605, 1290921, 1191085, 16299973, 8698287, 49206031, 116322369, 213349185, 469232199, 733559873, 1768473281},
  {1, 1, 5, 11, 7, 61, 9, 121, 25, 357, 1443, 405, 7827, 6313, 3833, 49773, 60051, 144895, 315813, 746657, 1613777, 2915681, 3814813, 3558875, 17429357, 4705931, 102976833, 14980417, 236933701, 16524235, 1588892231},
  {1, 1, 7, 13, 11, 53, 11, 207, 145, 211, 1703, 1081, 2117, 5481, 3353, 55839, 93557, 257075, 10941, 556211, 2078247, 890537, 7819067, 6848079, 9820833, 36510381, 124072129, 14378305, 63638983, 346425037, 289316683},
  {1, 1, 3, 11, 27, 23, 19, 9, 297, 279, 1481, 2273, 6387, 4313, 27321, 28395, 112403, 114659, 36031, 1007067, 98433, 229825, 7766559, 6194129, 23432921, 31623291, 63603521, 52942657, 471501507, 911190347, 697933147},
  {1, 3, 3, 5, 15, 45, 3, 41, 305, 87, 1815, 3461, 5349, 15893, 11159, 45143, 104025, 125267, 164169, 843087, 2002189, 2661389, 7179363, 4189987, 12084233, 14606441, 35178033, 29291059, 138821779, 1051592341, 1389126431},
  {1, 3, 3, 13, 9, 37, 79, 125, 259, 561, 1087, 4091, 793, 14829, 615, 36279, 28561, 58581, 356449, 706843, 1403897, 3316591, 7091157, 11465603, 22893231, 3216149, 37698897, 167458867, 421079795, 736964957, 1214764313},
  {1, 3, 5, 7, 31, 55, 7, 145, 347, 929, 589, 2783, 5905, 12669, 17991, 16369, 36635, 42459, 133723, 762219, 932861, 2671351, 23125, 11971473, 13010315, 16548693, 64595281, 171882771, 439113909, 300168535, 1758352655},
  {1, 1, 7, 15, 3, 25, 1, 181, 13, 243, 653, 2235, 7445, 3053, 9557, 35507, 84059, 207311, 302197, 1010749, 1643617, 3722745, 7844983, 11769449, 5288551, 50068401, 116803441, 142763665, 119416439, 22199071, 1194424083},
  {1, 3, 5, 5, 17, 53, 65, 7, 33, 583, 1363, 1313, 2319, 9093, 28479, 43793, 78841, 166229, 242385, 885269, 878147, 242549, 5590603, 13748103, 26527501, 42487827, 75752625, 220297331, 286224597, 193200437, 1195199233},
  {1, 3, 3, 7, 27, 47, 97, 201, 187, 321, 63, 1515, 7917, 685, 30903, 44471, 8155, 13767, 253507, 63101, 536261, 2650703, 865109, 11465731, 29430207, 66065377, 51393873, 227557427, 514857971, 219168279, 1022383371},
  {1, 1, 3, 5, 23, 9, 3, 165, 61, 19, 1789, 3783, 3037, 12317, 27253, 40823, 15057, 161995, 183373, 727599, 1224337, 39809, 1825111, 575993, 10127515, 41652537, 105158001, 35102609, 378636723, 281547669, 500717319},
  {1, 3, 1, 13, 15, 43, 125, 191, 67, 273, 1551, 2227, 5253, 13389, 1431, 43189, 26321, 240987, 350959, 610193, 1058179, 2092423, 7566749, 2691219, 11500359, 11876681, 97917393, 155339059, 282227825, 985584797, 1746263327},
  {1, 1, 1, 13, 25, 53, 107, 33, 299, 249, 1475, 2233, 907, 6757, 11693, 58645, 31033, 173261, 90961, 378847, 1731813, 2049495, 3165421, 9122591, 1498253, 8647287, 90509009, 47411697, 239962321, 576677757, 544451849},
  {1, 3, 5, 1, 23, 37, 85, 17, 207, 643, 665, 2933, 5199, 1109, 11967, 54737, 7805, 85715, 422993, 885657, 1009365, 45611, 8039511, 7603125, 28574889, 6554395, 47973745, 40663891, 260344469, 452293041, 974919431},
  {1, 1, 7, 7, 25, 57, 59, 41, 15, 751, 751, 1749, 7053, 2325, 30197, 46995, 17427, 103189, 105269, 201007, 608701, 3618931, 6252763, 3780195, 25780473, 16572865, 15326641, 44291633, 236404631, 12536855, 1894681353},
  {1, 3, 3, 1, 13, 25, 127, 93, 281, 613, 875, 2223, 6345, 14589, 15975, 13239, 13917, 115729, 341037, 140411, 1317929, 1788981, 8042113, 2876671, 15821035, 29010165, 131471377, 33177587, 69885619, 71737169, 1844728093},
  {1, 1, 5, 3, 29, 55, 79, 249, 43, 317, 533, 995, 1991, 717, 5853, 10609, 4407, 220185, 140331, 158827, 1342485, 1795271, 6604201, 14961225, 20599943, 12443939, 42240657, 157332625, 268515445, 490897971, 1128732941},
  {1, 3, 3, 15, 17, 49, 79, 31, 193, 233, 1437, 2615, 819, 2077, 32143, 18231, 10867, 118285, 329277, 626235, 191019, 2820845, 7669125, 4896873, 15895139, 12882775, 27371601, 133953171, 437818931, 876143999, 1504335105},
  {1, 1, 5, 15, 25, 3, 123, 145, 377, 9, 455, 1191, 3953, 16221, 5893, 10161, 15707, 256157, 30383, 213455, 967149, 3992709, 4085245, 11085075, 2271763, 59208573, 79253457, 183370033, 426660149, 398632927, 1704512777},
  {1, 3, 5, 3, 15, 19, 41, 231, 81, 393, 3, 19, 2409, 15117, 7271, 11505, 99295, 48267, 136879, 994245, 150299, 2181765, 3670181, 1573207, 28837575, 6819061, 110639697, 57165171, 451435637, 871907155, 637179167},
  {1, 1, 3, 1, 27, 43, 113, 179, 7, 853, 947, 2731, 297, 9453, 32485, 41847, 55253, 91271, 466623, 206765, 432015, 1828851, 3646889, 15118655, 24547919, 40034069, 34753169, 79897585, 5321971, 75776337, 1181620491},
  {1, 1, 1, 11, 29, 39, 53, 191, 443, 689, 529, 3329, 7431, 5701, 16477, 2005, 120511, 79497, 211611, 1022417, 428051, 1411135, 5966525, 6634029, 10431157, 1963379, 31833041, 29274833, 8985425, 235374395, 1493224717},
  {1, 3, 7, 5, 3, 29, 19, 67, 441, 113, 949, 2769, 4169, 6917, 29159, 31699, 18265, 236687, 431801, 647919, 1924327, 453701, 6795053, 12475105, 19905189, 61898637, 66306769, 142138899, 489618007, 338610389, 20380947},
  {1, 3, 5, 11, 11, 55, 85, 169, 215, 815, 803, 2345, 3967, 14693, 31679, 35537, 110775, 247183, 268459, 332713, 1822277, 2099891, 2434683, 13097647, 15287429, 10604291, 35215281, 42793715, 452196437, 549535803, 21039899},
  {1, 1, 7, 9, 5, 45, 111, 5, 419, 375, 303, 1725, 4489, 10901, 23749, 56243, 80061, 125929, 171129, 265067, 1383297, 1683375, 120403, 12223667, 19039889, 18259813, 101683505, 139980593, 510233879, 842291801, 845616917},
  {1, 3, 5, 15, 29, 43, 79, 19, 23, 417, 381, 541, 4923, 3733, 17615, 497, 66707, 83129, 300231, 223051, 1671471, 3564379, 3237437, 15132913, 12515209, 268575, 95694801, 59534099, 34366613, 67943935, 341197069},
  {1, 1, 3, 15, 3, 31, 117, 39, 117, 305, 1227, 1223, 143, 15293, 14877, 64919, 122523, 224703, 102355, 901929, 1368971, 221617, 7299837, 13705839, 23471123, 41168467, 72893393, 6347505, 259067187, 1062355263, 406196499},
  {1, 1, 5, 9, 5, 47, 87, 239, 181, 353, 1561, 3313, 1921, 15205, 2085, 3249, 19645, 17633, 448635, 897395, 901411, 206041, 1763277, 2520093, 162725, 46939541, 104368337, 83936721, 426923605, 886183257, 1746403605},
  {1, 3, 3, 1, 3, 15, 53, 221, 441, 987, 1997, 2529, 8059, 2981, 28559, 17975, 130845, 182463, 332251, 732713, 1900113, 3533757, 4144767, 8028505, 18056253, 49316671, 127046641, 96201107, 244592595, 1051039729, 546274067},
  {1, 1, 7, 11, 15, 57, 111, 139, 137, 883, 1881, 2823, 5661, 573, 21269, 11987, 72639, 164323, 454245, 817219, 1345663, 1538405, 6498799, 5447533, 3454427, 53539145, 14770929, 62749393, 340511159, 571414683, 1226232607},
  {1, 3, 5, 5, 21, 11, 5, 13, 27, 973, 587, 1331, 1373, 6829, 13271, 20689, 119865, 8177, 57423, 539681, 1176593, 3359935, 2317481, 12347991, 1502391, 44310817, 55717521, 179200371, 258930549, 608409109, 656518405},
  {1, 1, 7, 11, 29, 51, 93, 29, 217, 221, 55, 2477, 1979, 13077, 11213, 17971, 37919, 152625, 427119, 283833, 589121, 857949, 5360249, 11026211, 30543849, 656535, 44178321, 186021425, 297927319, 746483067, 1490255117},
  {1, 3, 3, 13, 3, 11, 49, 75, 379, 371, 1441, 793, 7633, 8069, 8359, 60343, 68657, 211071, 119023, 82349, 1942423, 58991, 6778703, 4472189, 18159621, 60834117, 19034353, 183725011, 109837395, 769613917, 1085608723},
  {1, 1, 1, 13, 19, 45, 89, 249, 91, 649, 1695, 915, 5619, 10157, 26733, 27349, 109913, 17575, 324457, 139493, 69501, 1925231, 3875613, 9561219, 17851079, 45639391, 30905809, 75732017, 283953265, 1035778557, 2132732163},
  {1, 3, 1, 7, 7, 29, 1, 77, 313, 895, 519, 771, 295, 8365, 16127, 31189, 65307, 62771, 215377, 961965, 1351721, 1581165, 5825083, 780051, 28798671, 30277067, 68718513, 167705907, 328689905, 117254327, 1173200663},
  {1, 3, 1, 15, 5, 3, 1, 57, 331, 109, 485, 2853, 6831, 6261, 28831, 51189, 21843, 135089, 160367, 547237, 1193125, 3437887, 1351513, 1832569, 31545185, 41663339, 120486289, 173925267, 103811601, 928080575, 1842531605},
  {1, 1, 1, 15, 17, 3, 35, 99, 245, 971, 839, 2509, 2803, 341, 20461, 49013, 79835, 258277, 403727, 133975, 1102271, 2075529, 642903, 3102147, 14007729, 58449887, 79513969, 46241841, 42096401, 1017049599, 771922689},
  {1, 3, 3, 3, 9, 37, 57, 251, 325, 317, 529, 1313, 6379, 229, 3919, 11511, 72543, 204021, 459065, 69965, 1784639, 2536529, 5168801, 11757365, 27944517, 13319839, 39872017, 243193683, 441675859, 79531507, 1146424601},
  {1, 1, 1, 15, 25, 59, 1, 119, 95, 15, 795, 2375, 6463, 1341, 10461, 33365, 3931, 165229, 229175, 910165, 133963, 1202475, 5224515, 16546839, 23619883, 15560515, 73873457, 259728753, 257162929, 773943359, 1579278089},
  {1, 3, 1, 5, 1, 49, 117, 21, 47, 179, 863, 85, 1669, 15605, 11959, 7669, 42169, 52533, 431669, 380417, 1866401, 2972491, 5568983, 14703763, 30709609, 61671849, 75041137, 145444147, 292500113, 918474389, 1050559249},
  {1, 3, 7, 3, 9, 37, 19, 221, 455, 973, 571, 1427, 817, 1773, 21927, 52371, 53055, 105381, 390201, 855079, 689305, 759163, 286665, 9845943, 19788055, 48986125, 38039697, 253980531, 333514103, 232541395, 1320549657},
  {1, 1, 1, 15, 17, 9, 67, 213, 127, 887, 1299, 2913, 7451, 10277, 18509, 53, 30939, 106213, 403461, 161847, 1325089, 1972611, 2394771, 10508799, 2337677, 1506311, 128980273, 30432241, 124787921, 720391295, 2142275329},
  {1, 3, 1, 13, 27, 27, 41, 43, 171, 623, 691, 391, 4885, 7485, 20215, 7381, 32881, 129455, 282815, 486837, 11063, 1992351, 7814547, 8637551, 30935339, 38274697, 5778737, 140012275, 142130865, 244911261, 624541451},
  {1, 3, 1, 13, 17, 17, 123, 239, 143, 227, 1151, 519, 6543, 9661, 5151, 59477, 39121, 252133, 170677, 351951, 1490907, 3755347, 7548911, 4710483, 17361307, 53076195, 17795825, 171671859, 206740913, 916595261, 1063572225},
  {1, 3, 7, 5, 7, 63, 97, 39, 101, 555, 1057, 381, 7891, 4373, 15343, 11571, 9945, 53355, 275643, 557037, 892139, 3226801, 3008039, 11274949, 23285761, 22079335, 16397937, 264663123, 522632215, 224047733, 1317944087},
  {1, 3, 5, 1, 3, 27, 85, 129, 161, 875, 1945, 3541, 695, 15541, 1151, 6321, 41885, 27879, 358543, 132585, 1823965, 607333, 3714623, 3549101, 17280841, 12713827, 3385329, 114210515, 321771669, 726360625, 971356947},
  {1, 3, 3, 5, 21, 59, 25, 183, 35, 25, 987, 1459, 181, 1885, 28519, 30279, 18121, 48345, 73071, 902925, 1362331, 3744135, 8151173, 1615127, 11799, 60030105, 31156049, 88956019, 231438963, 790515349, 752283653},
  {1, 3, 5, 13, 1, 15, 127, 237, 349, 337, 1491, 2383, 7811, 15405, 31551, 8257, 12481, 253909, 228603, 255803, 1788689, 488673, 1935013, 629223, 17753803, 52132623, 68456785, 258909555, 347438517, 596857341, 767537169},
  {1, 3, 5, 5, 31, 5, 109, 51, 409, 733, 1395, 3207, 6049, 15821, 14263, 43329, 65225, 135115, 157265, 151889, 1720295, 2907757, 5512785, 2055127, 16866739, 44666925, 72594449, 111209587, 154549301, 30109653, 47255567},
  {1, 1, 5, 7, 13, 35, 113, 25, 263, 389, 299, 2521, 1783, 15413, 22637, 54337, 59843, 46937, 223999, 548869, 450773, 3547523, 4579361, 11511415, 11233037, 28655131, 35650833, 256074353, 513910229, 222782135, 141750301},
  {1, 3, 7, 11, 15, 47, 97, 73, 55, 75, 113, 2695, 1023, 4013, 10223, 19907, 103751, 194771, 474307, 911901, 1639989, 1189491, 2167215, 3306861, 24510059, 12377211, 109107697, 40258515, 526570423, 1062909115, 1007628319},
  {1, 3, 1, 1, 3, 13, 69, 211, 289, 483, 1335, 787, 677, 12669, 28391, 38341, 89805, 43463, 154697, 208393, 878119, 4164213, 3717359, 3608435, 27807199, 34823481, 77907953, 193068595, 349960945, 323921553, 1388810259},
  {1, 1, 3, 3, 17, 7, 37, 77, 505, 137, 1113, 345, 2975, 11125, 12653, 45255, 62663, 25821, 329035, 400545, 1208209, 1967149, 7609853, 1683837, 24239421, 20954427, 45307089, 206239441, 335086291, 907951795, 499740673},
  {1, 1, 1, 13, 3, 11, 95, 199, 453, 109, 479, 3725, 239, 3173, 3789, 61637, 72265, 64455, 288895, 544147, 1655851, 2563193, 5973169, 7289995, 7725177, 47987155, 30340113, 190454705, 85355665, 140223549, 509915155},
  {1, 1, 7, 15, 19, 53, 3, 145, 359, 863, 347, 3833, 3043, 3733, 5565, 43843, 87115, 187087, 371017, 385935, 1346693, 1459243, 7253339, 11006855, 25695045, 6305999, 118512817, 227604337, 456257879, 749889535, 1551323139},
  {1, 1, 7, 15, 25, 63, 127, 129, 125, 195, 155, 2211, 8153, 15421, 5973, 56131, 89675, 260677, 434371, 771291, 1033917, 1520321, 5240967, 11086511, 29495023, 37358581, 12979057, 32911857, 394662903, 552844895, 406581257},
  {1, 1, 7, 13, 9, 49, 121, 115, 73, 119, 1851, 727, 47, 6061, 9613, 8771, 54345, 96085, 467781, 164765, 623223, 1917677, 3286907, 13298911, 11925963, 4703235, 109441713, 117248241, 116239287, 254766397, 1693126681},
  {1, 3, 3, 13, 13, 11, 71, 7, 45, 591, 133, 2407, 5563, 16365, 12351, 53831, 31681, 91201, 239743, 700851, 70251, 1315441, 4996755, 3557361, 6278683, 27284975, 17061425, 147164179, 6179507, 835186685, 323530781},
  {1, 1, 1, 13, 23, 29, 87, 89, 501, 71, 1759, 1119, 687, 13901, 1293, 19461, 78857, 220179, 236585, 341003, 1557549, 787625, 312611, 3591435, 25549123, 41412051, 77572913, 92769201, 263596337, 179749949, 1353711623},
  {1, 1, 7, 7, 13, 7, 13, 183, 53, 951, 1877, 3991, 6771, 5101, 2269, 4099, 45059, 151569, 241691, 806969, 1454251, 3887105, 5779819, 612225, 26104139, 31078183, 87809457, 220360977, 16158647, 930046839, 2030043165},
  {1, 3, 7, 11, 7, 1, 27, 47, 61, 21, 919, 961, 1091, 10901, 24127, 27139, 55815, 228891, 263725, 422407, 792107, 1120849, 5181097, 5788003, 9105813, 61739039, 41301905, 180494643, 520503895, 361488635, 2067529751},
  {1, 3, 5, 5, 1, 27, 1, 5, 63, 157, 1297, 1049, 5893, 4213, 18503, 36993, 74633, 58261, 298895, 311685, 1992169, 3298491, 173321, 3565165, 12480493, 2751425, 27324305, 7517139, 32131029, 261232405, 326189073},
  {1, 3, 7, 9, 19, 33, 17, 133, 425, 797, 1721, 153, 119, 5237, 22671, 33155, 87941, 233871, 501125, 602845, 269697, 2106221, 5971849, 3150237, 12123757, 41862931, 115690897, 100840659, 493746135, 949169209, 745422851},
  {1, 3, 3, 7, 13, 37, 1, 215, 509, 1003, 61, 2353, 7511, 14645, 19535, 62855, 33931, 82305, 100793, 540341, 1458627, 3202937, 6551607, 12208457, 14621, 29951011, 79891057, 189993363, 322773715, 641919799, 1442693149},
  {1, 1, 7, 1, 29, 19, 31, 79, 199, 555, 1209, 1603, 6089, 253, 14837, 32131, 60293, 242305, 186263, 734955, 743555, 3655739, 7084183, 7884581, 30533743, 25797805, 70389361, 216426833, 205133303, 1045930449, 566575117},
  {1, 3, 1, 1, 5, 31, 111, 127, 333, 429, 1863, 3925, 5411, 2757, 27967, 33413, 8845, 46977, 374427, 597499, 1730179, 2802545, 4355793, 4405491, 445793, 29776319, 91866641, 55993395, 173163793, 837729009, 76759061},
  {1, 1, 7, 5, 5, 5, 123, 191, 47, 993, 269, 4051, 2111, 9405, 27661, 44643, 27041, 87481, 217585, 872719, 651531, 2673571, 4476925, 4038993, 31351159, 38008363, 45320913, 266206737, 89184375, 766574005, 1555266581},
  {1, 1, 5, 15, 1, 9, 87, 5, 47, 463, 865, 1813, 7357, 3909, 14085, 9825, 22379, 81013, 86869, 863539, 263617, 658643, 4026171, 3623869, 29625225, 41687385, 119802545, 198466769, 290205077, 946703007, 571249681},
  {1, 3, 1, 3, 23, 63, 123, 83, 511, 777, 63, 1285, 4537, 13413, 12087, 50469, 125167, 31987, 242515, 463367, 839247, 1936947, 4879973, 14936035, 22325121, 46344165, 92834129, 47887667, 368081681, 714166611, 703646727},
  {1, 3, 3, 7, 27, 25, 31, 65, 441, 529, 1815, 1893, 323, 10917, 29023, 50983, 44267, 6071, 445925, 739923, 1111685, 1897445, 2467765, 3214867, 31840385, 10549503, 66269777, 86076979, 522164499, 489694967, 709985291},
  {1, 3, 7, 5, 11, 19, 7, 5, 397, 811, 755, 2883, 4217, 285, 6327, 20259, 70697, 30071, 198503, 853515, 927337, 2708209, 5200159, 3607143, 23995855, 2058453, 14766833, 158364019, 179350007, 693194837, 101880859},
  {1, 3, 1, 13, 9, 21, 13, 7, 271, 539, 1769, 3243, 5325, 11773, 16839, 1381, 83873, 20205, 272961, 99337, 601267, 1668315, 4021351, 13618741, 11273255, 50598089, 125043569, 214195795, 309130609, 280294301, 379427865},
  {1, 1, 7, 1, 31, 13, 47, 131, 181, 457, 1559, 2663, 6653, 1069, 11813, 56995, 89893, 15459, 96489, 651699, 1237911, 1230441, 7113957, 3152675, 24065891, 7431553, 129233745, 235588913, 158991671, 213794065, 1492438031},
  {1, 3, 3, 7, 29, 55, 25, 203, 419, 91, 437, 1159, 5691, 4941, 14687, 24167, 114603, 187569, 339275, 61133, 1216823, 2316135, 4176279, 8542233, 13757579, 54951759, 3587697, 131778707, 324866995, 446851063, 1982622733},
  {1, 1, 3, 13, 29, 19, 71, 217, 337, 329, 501, 939, 2205, 3229, 901, 33767, 74537, 39537, 322503, 691923, 626229, 2196397, 3125581, 15324433, 422447, 23143977, 108935121, 195429905, 451562483, 656294813, 439723021},
  {1, 1, 3, 1, 1, 27, 17, 201, 97, 285, 1269, 4043, 2207, 8029, 4365, 53479, 58661, 146477, 391551, 818677, 2031621, 852037, 1246169, 9897681, 27725791, 55249835, 106632849, 247819281, 47006963, 547351473, 1117447185},
  {1, 1, 1, 1, 3, 41, 13, 199, 141, 129, 1515, 3129, 5969, 12437, 6293, 42213, 33445, 30375, 220429, 92961, 724803, 349081, 4618493, 7477759, 22066333, 42524957, 7947985, 60715761, 471494353, 450591953, 2000768019},
  {1, 3, 3, 9, 3, 17, 119, 41, 271, 933, 877, 701, 2197, 5381, 6631, 64807, 108581, 249647, 321813, 278811, 1931533, 3557619, 5572705, 12766833, 12403657, 17652481, 91977489, 123154451, 16725395, 1054859033, 593392659},
  {1, 1, 1, 7, 15, 47, 3, 195, 115, 821, 725, 843, 6071, 13021, 23533, 33317, 86755, 260971, 79123, 338719, 382815, 4111775, 6245593, 6456313, 11635967, 48729987, 13140625, 120118417, 408022257, 916695863, 457489439},
  {1, 3, 5, 15, 17, 33, 85, 65, 297, 571, 1123, 2743, 5727, 10509, 7023, 57569, 119651, 180837, 41533, 891505, 1790565, 1962909, 1783583, 7405287, 31614715, 54623731, 110702833, 91560819, 524668597, 358012479, 1188123649},
  {1, 1, 5, 11, 27, 15, 37, 235, 415, 293, 1439, 2739, 4171, 14525, 4285, 609, 126703, 175343, 118787, 864361, 1863799, 1757291, 6189545, 1225219, 13908615, 30821207, 6582673, 128574769, 27927413, 684362363, 744889355},
  {1, 3, 7, 7, 1, 55, 71, 35, 307, 11, 401, 1881, 933, 12021, 16807, 54563, 3115, 138173, 35723, 821667, 1297823, 3918927, 4160551, 12360509, 20503573, 57200385, 50582897, 131962067, 464585303, 708911895, 1480686609},
  {1, 3, 1, 11, 21, 37, 3, 177, 119, 339, 559, 3991, 3437, 1357, 17255, 65509, 13543, 204721, 252841, 18103, 1078917, 3334811, 7922615, 12084435, 7955003, 43267953, 8666737, 63188531, 45396657, 297033755, 190714885},
  {1, 3, 3, 9, 17, 17, 97, 119, 301, 169, 157, 3267, 2261, 7805, 24679, 935, 67685, 101053, 90005, 261317, 1713363, 1776649, 198357, 7423625, 10897095, 49135809, 50617425, 70281619, 294681459, 1011876121, 1234390017},
  {1, 3, 3, 9, 29, 3, 111, 101, 355, 869, 375, 2609, 7377, 3965, 20175, 53119, 5341, 14233, 400527, 869603, 1944209, 3979927, 4559417, 4797347, 20476525, 19806101, 92506065, 137207795, 504604819, 146772377, 1117054541},
  {1, 3, 5, 9, 7, 21, 123, 99, 343, 693, 1927, 1605, 4923, 4141, 26055, 6761, 90493, 70299, 29097, 419071, 1444127, 2151139, 5793329, 14257651, 30332057, 47246975, 58926289, 266954739, 231720405, 792763193, 843082583},
  {1, 1, 3, 5, 13, 31, 99, 17, 75, 385, 1539, 1553, 7077, 4957, 6429, 52575, 84377, 157849, 111491, 953295, 1094613, 3005039, 1116589, 1288727, 25128669, 64646033, 34566033, 206387889, 228689555, 956382421, 1358974557},
  {1, 3, 3, 5, 31, 35, 107, 11, 407, 1019, 1317, 3593, 7203, 7693, 24519, 34351, 3185, 231835, 213407, 948367, 476591, 1969747, 5157439, 16364049, 25727341, 24656543, 91179057, 39957971, 414619923, 512896437, 545773647},
  {1, 3, 3, 13, 17, 33, 99, 245, 401, 957, 157, 1949, 1571, 9629, 31079, 10351, 34425, 199061, 475581, 1048319, 1343401, 4145285, 4604897, 4801217, 7163697, 15024431, 47371345, 71109971, 264440659, 880648893, 486902081},
  {1, 3, 1, 11, 27, 15, 11, 109, 429, 307, 1911, 2701, 861, 13325, 3999, 15453, 26527, 22039, 317979, 174639, 425529, 1971785, 3466799, 9863195, 22645753, 33267657, 101100721, 267310067, 524743761, 420395611, 1139799371},
  {1, 1, 5, 13, 13, 35, 55, 255, 311, 957, 1803, 2673, 5195, 13853, 293, 42409, 3825, 196097, 485935, 1595, 1012427, 828963, 1714633, 13974879, 25739469, 2512127, 56421969, 13442353, 352272021, 777017405, 1563596381},
  {1, 1, 1, 11, 19, 3, 89, 37, 211, 783, 1355, 3567, 7135, 13413, 28037, 53741, 27575, 149263, 189239, 784221, 468737, 1555871, 5209187, 15833023, 14031387, 19381435, 30328433, 37510193, 254605681, 901538939, 2105418179},
  {1, 1, 5, 5, 21, 49, 79, 17, 509, 331, 183, 3831, 855, 7717, 8613, 1257, 38521, 218969, 495453, 244643, 1295085, 2319321, 5768615, 4672251, 16111075, 44628291, 110801201, 217238001, 250324981, 1066973941, 1012644805},
  {1, 3, 7, 5, 29, 19, 85, 109, 105, 523, 845, 3385, 7477, 14797, 15199, 51291, 40017, 68041, 250575, 1012409, 1646297, 3335733, 6587623, 2499345, 7710925, 20786185, 95029297, 206441555, 409904407, 624334037, 544620621},
  {1, 1, 1, 7, 25, 17, 125, 131, 53, 757, 253, 2989, 2939, 5021, 18373, 47853, 53947, 171205, 495957, 521201, 770239, 1720345, 49985, 6014497, 27902241, 39963575, 68438993, 51819601, 255029841, 190145335, 378381641},
  {1, 3, 3, 9, 19, 23, 105, 39, 351, 677, 211, 401, 8103, 3613, 7383, 47887, 96381, 252887, 226139, 649885, 1741107, 1924915, 4082369, 15695687, 8333517, 16512243, 56529105, 109168819, 524348563, 885059321, 616664643},
  {1, 3, 5, 1, 5, 11, 17, 3, 405, 469, 1569, 2865, 3133, 10765, 12607, 2969, 124373, 142041, 317911, 1039261, 876375, 226841, 3286561, 5826493, 3270589, 61763969, 68634577, 19623315, 300393365, 921593681, 872407637},
  {1, 1, 3, 13, 15, 5, 117, 179, 139, 145, 477, 1137, 2537, 9453, 11757, 44479, 22161, 247515, 108281, 779217, 475599, 3187911, 2459237, 1577849, 23697053, 37473893, 37209489, 255260241, 234676371, 1026328605, 1781493343},
  {1, 1, 7, 9, 5, 21, 9, 93, 211, 963, 1207, 3343, 4911, 6725, 22149, 41259, 52341, 114881, 277721, 646757, 1298161, 3821519, 7494639, 7265987, 8880651, 43092635, 120951473, 96333809, 331926391, 900319609, 44661205},
  {1, 1, 1, 9, 13, 43, 17, 53, 81, 793, 1571, 2523, 3683, 10677, 3989, 14605, 15797, 64977, 334295, 73789, 1155105, 3351005, 7087029, 12280991, 5795495, 50294647, 76522257, 115846833, 37492529, 64848057, 228436189},
  {1, 3, 3, 13, 25, 21, 5, 59, 489, 987, 1941, 171, 6009, 2117, 11647, 60063, 4377, 43997, 394569, 587897, 1329335, 2955269, 7342351, 2981881, 25804415, 19636261, 67046705, 235228883, 79950899, 76803101, 2133433545},
  {1, 3, 3, 7, 1, 39, 89, 171, 403, 467, 1767, 3423, 2791, 773, 18727, 11631, 73715, 207813, 198099, 498965, 1639823, 1635871, 8057607, 2061603, 13432059, 28888979, 86172593, 75654643, 388136563, 984818167, 1536554449},
  {1, 1, 3, 9, 19, 49, 91, 125, 163, 1013, 89, 2849, 6785, 1421, 9805, 31743, 25237, 156743, 83037, 274223, 1957873, 3879607, 7809673, 12364109, 31772029, 12374461, 123258065, 163285265, 528311699, 328333465, 433226307},
  {1, 1, 5, 9, 9, 11, 15, 241, 43, 297, 1719, 1541, 1821, 2413, 1309, 44953, 49493, 209221, 210775, 478067, 1177413, 2848415, 5056357, 11665443, 22939945, 17963769, 108934929, 222983665, 324426709, 296662873, 1982597977},
  {1, 3, 7, 15, 29, 23, 103, 239, 191, 33, 1043, 3649, 6579, 10877, 31975, 45579, 109659, 81257, 68227, 1021947, 1516747, 916523, 1533429, 4820615, 11064237, 23126375, 121504145, 69355187, 89017239, 1071799231, 532143693},
  {1, 3, 3, 9, 21, 51, 123, 55, 223, 645, 1463, 4021, 5891, 9869, 8967, 52943, 60637, 196977, 66207, 72119, 691747, 1760091, 225169, 7704611, 2064329, 10357095, 32114129, 143528179, 302671059, 577127097, 1614488389},
  {1, 1, 5, 7, 3, 41, 27, 235, 391, 303, 2021, 3187, 7607, 11013, 1957, 21001, 115675, 8431, 272045, 378591, 302855, 2391539, 7130923, 3367089, 8072103, 31862219, 45178737, 208260049, 501007541, 542412023, 1381368531},
  {1, 1, 1, 9, 5, 49, 49, 29, 377, 251, 1887, 1017, 1301, 14141, 18941, 24253, 107573, 73913, 385453, 831965, 2020801, 785605, 1195831, 1834723, 9796525, 48261633, 612145, 78330737, 94094097, 821259481, 433167445},
  {1, 1, 3, 3, 13, 41, 27, 47, 223, 23, 517, 3227, 6731, 8869, 19989, 14703, 50975, 236921, 82061, 947879, 1380659, 3651275, 1665475, 10505233, 25292847, 32021879, 49786097, 136026065, 249054771, 826887987, 616237277},
  {1, 1, 7, 1, 31, 25, 47, 9, 511, 623, 2047, 1263, 1511, 165, 19461, 4427, 76253, 89339, 78997, 155787, 68069, 2415579, 5537219, 8876059, 12841819, 56871475, 5362033, 217127793, 454627703, 915114481, 1185895887},
  {1, 1, 3, 15, 15, 23, 53, 1, 261, 595, 85, 241, 7047, 845, 31605, 16687, 104467, 118139, 430083, 512097, 987317, 487545, 5641071, 11392505, 24852261, 33982523, 31758257, 220720593, 132834707, 945273343, 1736615519},
  {1, 3, 3, 11, 17, 5, 81, 73, 149, 781, 2035, 3163, 4247, 10965, 31655, 64783, 70623, 244853, 77345, 799261, 1350149, 1319817, 3380433, 4175967, 10594423, 34223427, 103903313, 219411379, 389216051, 142276603, 1972292801},
  {1, 3, 7, 7, 29, 59, 49, 79, 397, 901, 1105, 2191, 6277, 14997, 26703, 21979, 115699, 77865, 322831, 893805, 1595867, 807073, 1642529, 1897325, 27549475, 51202521, 112785617, 99744403, 6489719, 440808151, 1561127373},
  {1, 3, 3, 11, 13, 27, 25, 173, 107, 73, 1265, 585, 5251, 5709, 2279, 49551, 30047, 155369, 417599, 679717, 2057593, 327271, 3892005, 15984581, 27518149, 45395231, 42455057, 189006227, 44349715, 377844923, 1416434781},
  {1, 1, 7, 15, 29, 23, 73, 229, 235, 887, 1469, 4073, 2591, 10861, 22293, 12075, 22867, 35257, 432739, 289237, 282753, 479239, 7598139, 13694857, 8362333, 11762819, 45039601, 106900817, 373892887, 759623295, 1646369869},
  {1, 1, 3, 9, 17, 15, 83, 173, 207, 879, 1701, 1509, 11, 7197, 27941, 15183, 83349, 37, 379203, 785295, 1292329, 2779531, 6875731, 7838945, 18197969, 5530615, 131378033, 172476625, 220992979, 33385273, 1690654529},
  {1, 1, 3, 5, 5, 37, 65, 161, 39, 421, 1153, 2007, 5355, 6901, 10613, 36079, 22489, 85425, 157017, 50349, 1551341, 1684923, 1029657, 175461, 21401739, 25652823, 68726481, 69766289, 382171891, 211884085, 492956629},
  {1, 1, 7, 11, 23, 37, 5, 11, 9, 499, 17, 157, 5747, 14365, 26629, 57867, 31255, 216627, 41537, 700049, 622519, 3936613, 618759, 7730141, 1482441, 59878559, 115470769, 74324113, 469686615, 336743099, 1953981767},
  {1, 3, 7, 13, 25, 9, 49, 7, 39, 945, 1349, 1759, 1441, 6005, 29855, 31035, 74361, 136429, 522709, 82925, 538523, 3293147, 3828821, 15133201, 18745011, 6160493, 99760017, 192566227, 307149431, 567469725, 1211820489},
  {1, 1, 5, 3, 21, 15, 113, 81, 265, 837, 333, 3625, 6133, 45, 11773, 633, 47999, 106873, 503643, 526605, 1430805, 593349, 6118937, 8641873, 17194117, 14145305, 86165713, 100569169, 263361045, 336249299, 406358085},
  {1, 3, 1, 11, 13, 27, 73, 109, 297, 327, 299, 3253, 6957, 109, 16959, 59389, 59263, 255585, 326511, 699157, 1015609, 1732133, 3427691, 4319575, 19195089, 13664201, 6378737, 114362419, 164207569, 787590491, 506638173},
  {1, 1, 3, 13, 19, 39, 123, 73, 65, 5, 1061, 2187, 5055, 10725, 4037, 17423, 52625, 209575, 459771, 345359, 77245, 2615237, 1024681, 3777017, 30696359, 64944627, 96834001, 157270737, 410135603, 694381949, 830861507},
  {1, 1, 3, 1, 11, 31, 21, 115, 453, 857, 711, 495, 549, 3093, 13213, 6031, 58509, 80783, 81795, 592561, 758487, 2451433, 4050365, 956619, 18496803, 43739761, 96988433, 41601009, 386034291, 146242769, 1870406875},
  {1, 3, 7, 7, 15, 29, 79, 103, 47, 713, 1735, 3121, 6321, 9325, 17727, 39211, 90755, 157067, 512697, 593611, 802027, 2417787, 7874221, 15930051, 20651501, 30857061, 14936337, 253950451, 160997015, 881442199, 689653599},
  {1, 1, 5, 5, 29, 9, 97, 33, 471, 705, 329, 1501, 1349, 813, 13677, 61769, 12937, 88705, 113845, 778429, 1481837, 912539, 7221197, 5540253, 33440481, 60509929, 115692177, 9896465, 462925909, 709759701, 1799248973},
  {1, 3, 3, 1, 21, 9, 111, 209, 71, 47, 491, 2143, 1797, 469, 7007, 12175, 116869, 111233, 408725, 1013171, 1271773, 4150067, 5722259, 8688639, 20952923, 45212241, 76245873, 27738163, 440241267, 843901393, 31781573},
  {1, 3, 3, 3, 11, 39, 21, 135, 445, 259, 607, 3811, 5449, 8133, 2127, 6639, 81415, 231967, 120371, 675361, 2004547, 1516241, 5044663, 10956435, 6015239, 61655597, 46037169, 108346483, 500661939, 510531347, 933084635},
  {1, 1, 7, 9, 11, 25, 113, 251, 395, 317, 317, 91, 1979, 4053, 18021, 36539, 73221, 65055, 98821, 590357, 1916615, 3079815, 4969305, 7189193, 14923511, 29706407, 89014609, 56875089, 60309047, 444104505, 1280029147},
  {1, 3, 1, 9, 3, 21, 103, 133, 389, 943, 1235, 1749, 7063, 1725, 327, 17309, 120845, 206879, 235577, 538659, 2091209, 2631857, 8247939, 14626543, 20798529, 41663907, 4088689, 226730419, 520515025, 941330681, 251978835},
  {1, 1, 3, 7, 1, 11, 5, 15, 497, 477, 479, 3079, 6969, 1973, 7341, 45551, 106507, 90117, 106511, 385081, 270379, 3432893, 4547017, 12265059, 15246587, 22311597, 31742289, 65237105, 167073523, 295227159, 6005457},
  {1, 1, 3, 3, 15, 39, 105, 131, 475, 465, 181, 865, 3813, 9917, 9037, 54607, 1295, 20235, 94003, 357197, 675687, 1457735, 6791061, 10280641, 29742077, 17326009, 42554257, 134181073, 96730515, 1046662355, 544248671},
  {1, 1, 7, 9, 19, 63, 123, 131, 415, 525, 457, 2471, 3135, 7813, 18389, 16667, 19717, 239367, 292643, 539007, 392679, 3272331, 1824985, 13967021, 13173355, 6492787, 46339921, 36009745, 77707255, 931907449, 2058188483},
  {1, 3, 7, 15, 25, 35, 123, 45, 341, 805, 485, 4049, 7065, 7549, 22527, 51563, 61643, 116189, 130535, 54439, 1926697, 665569, 3853833, 14097945, 14989229, 19128421, 42434001, 178917075, 469918103, 395555103, 1616531273},
  {1, 1, 1, 5, 29, 9, 47, 227, 51, 867, 1873, 1593, 2271, 13981, 22885, 23005, 61129, 167377, 78421, 932275, 1573103, 1572871, 3670103, 12059973, 26740461, 31982019, 99105713, 262682257, 405275665, 213408373, 1155204941},
  {1, 1, 7, 15, 31, 9, 71, 117, 285, 711, 837, 1435, 6275, 2933, 1989, 26555, 110531, 197451, 43085, 264067, 247913, 1603849, 1632619, 363737, 18445231, 9539311, 18649585, 107396305, 290273591, 505045439, 1538543567},
  {1, 3, 1, 1, 5, 19, 79, 25, 301, 415, 1871, 645, 3251, 2525, 17239, 55165, 126277, 105945, 210271, 948115, 1692733, 56649, 5920579, 6323955, 8540113, 63494967, 30692465, 169672307, 27082961, 741791409, 1302855765},
  {1, 3, 1, 3, 17, 51, 99, 185, 447, 43, 523, 219, 429, 4293, 6175, 11277, 81223, 169293, 55543, 182783, 478141, 610155, 7521431, 11888639, 10446031, 41532057, 110928177, 222792243, 243639345, 128773459, 787990465},
  {1, 3, 1, 13, 29, 13, 51, 93, 7, 995, 757, 3017, 6865, 14077, 17055, 33325, 90313, 72897, 306417, 490479, 1732033, 3424355, 3982287, 14139585, 23872701, 63658333, 37440945, 175123411, 496036113, 202430109, 1618094925},
  {1, 1, 3, 15, 7, 25, 75, 17, 155, 981, 1231, 1229, 1995, 1885, 5749, 6591, 88387, 251075, 463997, 963311, 897933, 3265599, 1642121, 333675, 22999425, 26480415, 43787985, 226780785, 449578835, 27534399, 1335776343},
  {1, 3, 5, 3, 27, 45, 71, 73, 225, 763, 377, 1139, 2863, 1653, 6775, 48089, 64839, 38359, 169161, 552051, 1664229, 2117077, 2506023, 9130613, 32661463, 9301203, 100045361, 165081363, 330555829, 550548851, 1001562571},
  {1, 1, 3, 1, 1, 39, 69, 113, 29, 371, 1051, 793, 3749, 5197, 17197, 20687, 73933, 133189, 404347, 473353, 1508773, 2715121, 5885919, 4794535, 30827261, 37576977, 114458545, 220876593, 214887699, 6248913, 318597969},
  {1, 1, 3, 13, 23, 61, 27, 183, 307, 431, 1345, 2757, 4031, 9389, 24197, 7263, 88641, 258899, 402897, 404847, 260027, 2250103, 5160427, 4454133, 185249, 32750083, 97087857, 237038993, 289123283, 423139965, 49955911},
  {1, 3, 7, 5, 5, 59, 117, 197, 303, 721, 877, 723, 1601, 213, 11119, 47563, 126305, 28961, 106423, 493225, 143457, 1495043, 7311693, 6222873, 25681103, 46073949, 85933969, 81357907, 110568119, 298263701, 143918037},
  {1, 3, 5, 1, 27, 33, 99, 237, 485, 711, 665, 3077, 5105, 4109, 21151, 26825, 55717, 40247, 310445, 989927, 89233, 515265, 3338123, 6339269, 29957585, 60743309, 110355953, 138585875, 164895445, 839124113, 1769268299},
  {1, 1, 3, 1, 13, 9, 103, 201, 23, 951, 2029, 165, 2093, 15981, 8109, 57647, 43437, 5289, 382645, 770811, 1649253, 490339, 7305611, 1096281, 12204705, 4009345, 59834609, 26813009, 230259155, 689488209, 150648925},
  {1, 3, 5, 13, 5, 29, 55, 85, 221, 677, 611, 3613, 4567, 10397, 10023, 46969, 46249, 140777, 473761, 212907, 1966633, 42697, 369257, 4939935, 23112705, 36303699, 17213905, 65254579, 13039445, 942372861, 1727483989},
  {1, 1, 1, 1, 7, 61, 9, 233, 261, 561, 953, 4023, 2443, 12197, 28965, 31133, 40173, 82795, 61073, 1035965, 763269, 2356073, 8269501, 14061765, 1874683, 37643335, 74166161, 217433873, 2277873, 1000532017, 704507607},
  {1, 3, 3, 13, 1, 17, 103, 71, 223, 213, 833, 1747, 6999, 1141, 14071, 49663, 24041, 7541, 380957, 594027, 36971, 12395, 7221969, 6918181, 4628383, 54798363, 11882193, 49145747, 159945395, 237539325, 1710897105},
  {1, 3, 5, 15, 25, 53, 57, 187, 25, 695, 1207, 4089, 2877, 653, 18223, 45609, 101291, 96565, 340993, 633925, 1346567, 1954997, 738995, 4842411, 5385933, 53260089, 114460657, 129025779, 452231957, 355334239, 1414505289},
  {1, 1, 7, 1, 7, 31, 87, 129, 493, 519, 1555, 1155, 4637, 2245, 14653, 27435, 103661, 130867, 367619, 939107, 1152037, 3896425, 4968027, 9198159, 15365399, 16920137, 119610353, 105992625, 49262263, 803951185, 643397079},
  {1, 1, 1, 15, 21, 17, 23, 29, 19, 255, 927, 1791, 3093, 14101, 31037, 58221, 113507, 100921, 27141, 707099, 1149649, 3841831, 1553555, 4699931, 32706939, 24579121, 94844017, 232195953, 536721009, 134024927, 1146359493},
  {1, 1, 3, 9, 17, 33, 95, 129, 175, 461, 287, 2633, 2325, 10605, 16589, 32751, 12453, 86069, 247357, 797299, 1674989, 2568187, 2256145, 1641867, 6982117, 33260833, 17335121, 17744241, 215646995, 20105177, 1194436929},
  {1, 3, 5, 7, 23, 19, 63, 209, 249, 583, 1373, 2039, 2225, 2789, 4047, 3913, 114531, 133819, 354567, 482107, 1717637, 1681789, 3864859, 5328721, 24994243, 60340957, 474097, 25247059, 108725237, 141207703, 257097927},
  {1, 3, 3, 5, 5, 19, 79, 241, 459, 355, 1455, 3313, 3639, 12653, 27207, 63103, 113057, 166769, 476991, 149779, 1183733, 2258383, 8040319, 9379947, 22800325, 45655219, 73036209, 196207219, 1075859, 768676597, 1796276565},
  {1, 1, 7, 9, 21, 41, 97, 119, 129, 769, 1541, 3495, 7741, 2373, 4093, 33899, 75173, 196193, 423893, 859805, 172635, 3276333, 799925, 6083929, 22885195, 15589889, 20008081, 89618385, 61687543, 1020608345, 690787013},
  {1, 1, 7, 11, 9, 29, 35, 255, 141, 937, 1763, 41, 1393, 9805, 30685, 53515, 743, 109437, 446697, 844735, 1456611, 2840009, 6591533, 4490319, 11960901, 63409941, 46309137, 178965937, 245897239, 448268187, 543773017},
  {1, 3, 7, 1, 13, 51, 61, 157, 177, 847, 1829, 3539, 285, 5365, 19487, 30827, 55333, 252521, 131823, 624545, 1577369, 3600573, 8147379, 3093417, 9689479, 6643601, 14153329, 199532755, 536641975, 278182481, 247407069},
  {1, 1, 1, 15, 21, 13, 9, 55, 397, 19, 1495, 1255, 7235, 1909, 22885, 2717, 94243, 102073, 240985, 625349, 1156427, 89057, 534119, 286267, 18080035, 16084647, 15970737, 262269681, 339383537, 422773183, 1137477829},
  {1, 1, 7, 7, 25, 37, 53, 237, 319, 197, 269, 1205, 1485, 1229, 4013, 13675, 32619, 170605, 489569, 686089, 370705, 1042915, 6170681, 9758841, 5499321, 26672945, 93641425, 206615121, 357626071, 816855639, 1668509257},
  {1, 1, 5, 15, 23, 17, 35, 247, 323, 807, 233, 3681, 4407, 4765, 20709, 19129, 44003, 125995, 131301, 217703, 82667, 1634279, 8108739, 5066685, 28310213, 38165387, 16510961, 95230225, 411243413, 617143551, 1451995463},
  {1, 1, 3, 7, 9, 59, 85, 105, 493, 763, 1639, 391, 1451, 1269, 4421, 37727, 43435, 240685, 65759, 120617, 408557, 2688065, 3437591, 8251579, 11721315, 19574415, 37116209, 112703217, 522765043, 69743415, 36346073},
  {1, 3, 3, 9, 15, 33, 5, 253, 129, 625, 1527, 2793, 6057, 13293, 22399, 57167, 7533, 507, 73853, 482641, 2016673, 2914685, 2853133, 756987, 16789189, 35142173, 111575441, 210410995, 106228243, 706598425, 2079306079},
  {1, 3, 1, 1, 7, 47, 21, 161, 235, 83, 397, 3563, 5953, 2133, 27055, 34317, 104101, 84347, 465987, 948513, 1834949, 380583, 8277983, 9119753, 27346335, 19717005, 83258289, 59733075, 289273649, 713058705, 813787095},
  {1, 3, 7, 11, 3, 41, 25, 117, 375, 779, 1297, 3715, 8117, 13845, 21247, 48555, 127151, 251879, 44381, 279917, 819177, 1354747, 4549047, 1752965, 6593911, 50529865, 7942193, 24840083, 397219767, 98335451, 690445779},
  {1, 1, 3, 7, 31, 19, 103, 173, 475, 189, 2035, 2921, 1107, 8359, 26959, 15005, 130365, 135781, 351449, 152021, 613127, 1941461, 583447, 8102365, 14311635, 8302069, 61440845, 104511445, 73796919, 1017831251, 2086155459},
  {1, 1, 7, 3, 25, 7, 93, 255, 307, 113, 1893, 2233, 6919, 7183, 3615, 36409, 47729, 187131, 151537, 552383, 1950081, 1098801, 5406227, 9198915, 32741439, 25416057, 21127421, 34293221, 333284339, 420026263, 195694549},
  {1, 3, 5, 15, 9, 57, 79, 143, 165, 5, 1389, 193, 693, 2923, 8405, 39855, 62801, 244907, 182127, 434321, 563117, 254463, 1794555, 16212123, 4385783, 56060763, 57651165, 163163503, 106610545, 431142323, 1634663885},
  {1, 3, 5, 1, 29, 45, 91, 49, 189, 461, 439, 1283, 7835, 11319, 20333, 17375, 118463, 80995, 41963, 977037, 396843, 3338219, 5904155, 13968513, 24194817, 28977049, 126073861, 112959079, 344322065, 256473501, 105562201},
  {1, 1, 3, 13, 11, 61, 41, 231, 373, 695, 395, 915, 5393, 10915, 10951, 31501, 43415, 223909, 354551, 1025351, 1103849, 1319207, 1532537, 14803705, 11528417, 10857299, 23847813, 38694221, 340676119, 143324473, 1340930135},
  {1, 3, 7, 11, 5, 51, 67, 53, 483, 95, 1943, 247, 5653, 2475, 21973, 60973, 22481, 145447, 32765, 958777, 1942471, 1530093, 7658245, 11128093, 2766157, 52792847, 56464693, 107804735, 267588819, 898146551, 2089292617},
  {1, 3, 7, 5, 5, 57, 45, 235, 137, 793, 1069, 1661, 1557, 2987, 30677, 55341, 101855, 19515, 336847, 26675, 927757, 3000595, 5393087, 6851539, 25787227, 34182179, 30256493, 200380559, 244774323, 94765625, 901322441},
  {1, 3, 5, 3, 25, 55, 103, 177, 81, 861, 1151, 143, 7655, 8527, 5917, 1471, 92925, 192867, 50289, 433669, 441915, 1339303, 6536339, 5462745, 15864765, 34113309, 66752245, 177921143, 388398033, 833073631, 1163374429},
  {1, 1, 3, 1, 21, 41, 67, 131, 253, 431, 1269, 3181, 3429, 5195, 13335, 28589, 51163, 230435, 27119, 188957, 1786425, 3228175, 6659633, 13443607, 23669955, 1780323, 126677773, 148859277, 194448439, 173663797, 2033840329},
  {1, 3, 1, 1, 21, 7, 77, 221, 257, 663, 71, 2949, 2481, 11811, 29189, 64459, 31831, 179195, 185777, 462319, 1978435, 3264839, 7476185, 10892085, 25049007, 49386287, 48008893, 36109503, 162220789, 154157821, 2033335233},
  {1, 3, 5, 3, 3, 23, 45, 107, 299, 739, 1013, 3, 3165, 13691, 27957, 43439, 58653, 215545, 299429, 122535, 1690981, 650017, 1820385, 3342199, 27585197, 33302983, 129366709, 23680799, 148585169, 138529919, 188988231},
  {1, 1, 5, 1, 3, 37, 109, 37, 243, 983, 1221, 1691, 3869, 15483, 19255, 9387, 51479, 69101, 475071, 280787, 159555, 1572277, 6308989, 13728383, 31319205, 25962903, 909973, 16586333, 535820881, 947751285, 1271339335},
  {1, 1, 5, 5, 31, 7, 5, 193, 397, 867, 1495, 3435, 7441, 16227, 12935, 25291, 50003, 137593, 60341, 368303, 847711, 1616115, 6540101, 4152349, 17819921, 11321331, 40162373, 145313741, 494625041, 479047473, 2057688155},
  {1, 1, 1, 1, 17, 59, 97, 233, 389, 597, 1013, 1631, 483, 10503, 26575, 47199, 34687, 213935, 176165, 258155, 1552511, 143675, 3354723, 9148275, 13702777, 3818833, 20998629, 86844741, 277190549, 177841429, 214683205},
  {1, 1, 1, 11, 7, 41, 107, 53, 111, 125, 1513, 1921, 7647, 15743, 6591, 23999, 34229, 45485, 441907, 601677, 384751, 2196465, 6781791, 5639319, 13495023, 31996097, 23567293, 64017157, 521076469, 137518111, 919178195},
  {1, 3, 3, 3, 31, 29, 117, 3, 365, 971, 1139, 2123, 5913, 1651, 28709, 23049, 130897, 189437, 370215, 538235, 943669, 1106887, 5961349, 8739001, 24062073, 41249963, 27046373, 250995375, 244488087, 318118591, 2024304195},
  {1, 1, 1, 13, 23, 3, 1, 167, 475, 639, 1811, 3841, 3081, 6739, 22759, 64655, 90451, 138353, 192801, 297731, 1644513, 3724633, 3716237, 3701961, 26307923, 35844079, 102401357, 169250749, 269477173, 825093881, 1163805891},
  {1, 1, 5, 3, 5, 47, 65, 123, 275, 783, 95, 119, 7591, 15887, 14431, 17915, 17589, 131119, 354609, 49507, 680445, 3549353, 7793053, 1305893, 16556285, 33339165, 51491829, 120171637, 198008785, 518357463, 1155804993},
  {1, 3, 1, 15, 13, 33, 93, 237, 467, 431, 705, 4013, 4035, 3559, 13869, 31867, 124249, 96991, 383, 261891, 1616267, 2200037, 2421125, 7337515, 20618139, 19464597, 71240301, 84900439, 62510517, 478274899, 746683097},
  {1, 3, 5, 1, 19, 7, 101, 231, 155, 737, 1381, 3343, 2051, 16231, 22253, 13695, 96031, 84557, 433277, 682303, 373113, 517513, 6587475, 14374499, 24828457, 47650673, 61281445, 214264263, 52274833, 241261341, 1126399575},
  {1, 1, 5, 9, 15, 49, 45, 163, 433, 765, 2031, 201, 2589, 1307, 29335, 14795, 13759, 16337, 127475, 691755, 1712789, 2787963, 7578923, 11050713, 7809307, 19512715, 132208749, 119845421, 312716721, 1012229309, 682091211},
  {1, 3, 7, 9, 19, 41, 31, 89, 93, 623, 105, 745, 4409, 3667, 9669, 32813, 78579, 32341, 416355, 264441, 177547, 2084779, 2914585, 10534843, 12804159, 17918375, 71830781, 157823647, 167313395, 248016501, 633934815},
  {1, 1, 5, 1, 11, 45, 127, 85, 389, 439, 829, 477, 7965, 15643, 25175, 11659, 3895, 110405, 331591, 839313, 609687, 814027, 8357633, 3299935, 24175043, 17365499, 50862861, 130561005, 320385073, 759712693, 336157903},
  {1, 3, 3, 15, 13, 41, 1, 207, 435, 585, 311, 1725, 2737, 3459, 31653, 22057, 48509, 251223, 283239, 559855, 274353, 3603953, 6438347, 13203137, 3896159, 11227055, 36206973, 34382527, 430782967, 657946355, 958489553},
  {1, 3, 3, 3, 13, 49, 21, 31, 197, 799, 1411, 2959, 7133, 1179, 8277, 56649, 78321, 161487, 215631, 503723, 186089, 1995447, 466449, 2773441, 6814701, 7653799, 19688373, 228711071, 458281687, 653297279, 724676433},
  {1, 3, 1, 3, 7, 43, 9, 141, 133, 579, 1059, 93, 957, 5723, 7509, 17227, 21429, 92237, 389201, 44939, 1096827, 2368851, 4024261, 1088009, 6185135, 32236035, 49223357, 46758927, 202989301, 40246335, 1775818707},
  {1, 3, 7, 1, 15, 51, 23, 213, 381, 851, 699, 2261, 3419, 1943, 21197, 35709, 130779, 206425, 273889, 782581, 1976351, 919971, 1325749, 12070257, 32597271, 18169621, 106379357, 216094807, 507454835, 409662813, 419113411},
  {1, 3, 5, 9, 25, 35, 67, 141, 35, 409, 1423, 365, 1645, 6395, 30997, 39055, 60535, 116503, 316717, 93049, 1736671, 678333, 362691, 14666601, 46303, 59854195, 11362173, 79469007, 340507121, 349565749, 731270109},
  {1, 3, 3, 11, 15, 33, 27, 181, 93, 87, 1761, 3511, 1353, 371, 22597, 10345, 69497, 178461, 506683, 764173, 795599, 1031587, 4982769, 3189795, 19019665, 60541387, 106981957, 216711983, 453235991, 513665207, 761714771},
  {1, 3, 5, 3, 25, 63, 111, 137, 321, 819, 705, 1547, 7271, 1967, 125, 39711, 72733, 143107, 150809, 626493, 834387, 3033735, 2849309, 13948955, 18570461, 35701245, 10475381, 59374583, 455731665, 69741535, 1346009949},
  {1, 3, 1, 1, 5, 57, 99, 59, 411, 757, 1371, 3953, 3695, 14591, 15389, 43931, 97623, 130699, 407951, 147613, 1196705, 2522745, 6070055, 6757301, 31380987, 39065977, 127146989, 141310439, 44327861, 394060701, 227949265},
  {1, 3, 5, 11, 11, 21, 25, 147, 239, 455, 709, 953, 7175, 431, 1533, 41247, 25557, 124801, 277431, 171159, 640669, 2509365, 7398221, 7382143, 19865831, 14216233, 134072221, 196576423, 313250417, 585342615, 917179855},
  {1, 3, 3, 15, 5, 53, 91, 205, 341, 63, 723, 1565, 7135, 9695, 20893, 44569, 29405, 73375, 345515, 445165, 1399031, 2177499, 2361769, 9542561, 20996031, 4201089, 12589821, 163595783, 306206711, 348199955, 1505981401},
  {1, 1, 7, 15, 11, 21, 99, 79, 63, 593, 2007, 3629, 5271, 14735, 21119, 31449, 12893, 108433, 211607, 679917, 969717, 1194677, 3744219, 15066433, 23767375, 53737833, 112648509, 162722213, 152881395, 250880667, 1579009991},
  {1, 3, 3, 1, 9, 21, 45, 175, 453, 435, 1855, 2649, 6959, 12991, 11485, 47897, 59987, 51087, 287915, 1027995, 1324921, 3524247, 1152061, 14214501, 27142579, 34008873, 54065869, 28496039, 446445367, 574922397, 274614485},
  {1, 1, 3, 15, 15, 33, 121, 121, 251, 431, 1127, 3305, 4199, 14447, 23167, 52061, 13013, 66437, 459915, 115663, 1304327, 651157, 1310437, 12311053, 17453139, 23793505, 23799117, 70596997, 26945847, 904645147, 734152915},
  {1, 1, 1, 9, 31, 15, 71, 29, 345, 391, 1159, 2809, 345, 3155, 6535, 19887, 103735, 252305, 68237, 547581, 1823559, 2562167, 993289, 4336629, 6664307, 20914079, 49229261, 229706365, 417411893, 904531453, 1590673611},
  {1, 3, 7, 1, 23, 29, 95, 151, 327, 727, 647, 1623, 2971, 10903, 13453, 15037, 121051, 121217, 520639, 440257, 1003893, 263181, 5748141, 6021373, 19525069, 17853025, 113363765, 68337031, 492314835, 1004118557, 559507291},
  {1, 1, 7, 7, 9, 29, 79, 91, 127, 909, 1293, 1315, 5315, 11607, 3711, 56009, 75173, 53811, 204683, 804809, 1409433, 4017277, 6934471, 203283, 29602965, 27328057, 81877333, 192164325, 502594643, 877443731, 1791768133},
  {1, 1, 5, 11, 13, 37, 89, 73, 149, 477, 1909, 3343, 525, 11723, 12487, 23931, 91693, 3303, 145819, 903391, 2015015, 4019371, 7703867, 11437299, 10125977, 9341327, 126062949, 220395837, 111239825, 256856063, 1343795529},
  {1, 3, 5, 7, 5, 59, 55, 255, 223, 459, 2027, 237, 4205, 8811, 101, 1055, 70889, 218471, 450213, 335501, 1359221, 1833541, 6386961, 6581017, 8672143, 8402211, 25422653, 218257807, 422500849, 322380603, 420116161},
  {1, 1, 1, 7, 27, 11, 95, 65, 325, 835, 907, 3801, 3787, 6823, 23295, 37231, 99177, 51129, 78729, 28861, 1078443, 3603587, 140757, 7898353, 26709131, 35884781, 119978797, 132613813, 182798261, 881170899, 286247887},
  {1, 1, 1, 11, 27, 33, 99, 175, 51, 913, 331, 1851, 4133, 603, 32007, 17599, 2597, 137953, 90259, 66485, 801429, 83465, 6259635, 3244157, 19958161, 57513111, 102085957, 188409693, 106250261, 638227583, 66433871},
  {1, 3, 5, 5, 13, 37, 31, 99, 273, 409, 1827, 3845, 5491, 14583, 13917, 28783, 96107, 35883, 322339, 515369, 346881, 2023203, 6284911, 9322733, 23121743, 28918877, 80010813, 3537015, 333396465, 713229017, 342902985},
  {1, 1, 3, 7, 23, 19, 107, 85, 283, 523, 509, 451, 421, 2171, 5031, 28925, 50221, 6013, 78617, 946441, 789271, 1059173, 4544977, 748111, 1970349, 10252335, 38673333, 87165373, 489601495, 282199027, 830240331},
  {1, 3, 5, 7, 13, 9, 51, 81, 87, 619, 61, 2803, 5271, 12319, 3437, 31855, 2601, 213295, 493319, 865613, 1963483, 85393, 3923129, 8191775, 18167037, 42609149, 10097909, 195439351, 222480593, 565377755, 1775360585},
  {1, 1, 1, 15, 9, 45, 35, 219, 401, 271, 953, 649, 6847, 12559, 20239, 52495, 110433, 42427, 278059, 233989, 848449, 2809379, 687993, 9812635, 29407023, 45719633, 6961085, 150881349, 149315573, 28045851, 1299269341},
  {1, 1, 7, 11, 9, 45, 17, 219, 169, 837, 1483, 1605, 2901, 13371, 16007, 10265, 63401, 138667, 42571, 844759, 162853, 3802131, 2827667, 10910045, 25822319, 48633723, 23472829, 33168109, 345907187, 808896703, 1463377093},
  {1, 1, 7, 7, 21, 43, 37, 33, 291, 359, 71, 2899, 7037, 13003, 12071, 26809, 34149, 173231, 301637, 340991, 609871, 2495245, 6937709, 13251245, 2808249, 36814111, 21976037, 232954749, 205082771, 564105459, 208240473},
  {1, 3, 3, 13, 31, 53, 37, 15, 149, 949, 551, 3445, 5455, 14351, 301, 50761, 97199, 56561, 485063, 168187, 83529, 3357527, 1929387, 8541825, 15725351, 17818289, 13622173, 149938631, 403691383, 667926545, 7308995},
  {1, 3, 1, 5, 19, 45, 81, 223, 193, 439, 2047, 3879, 789, 155, 133, 24763, 69091, 73125, 316535, 491471, 391729, 269875, 3885153, 5266573, 14156525, 64488511, 6827189, 125333503, 69776341, 554294521, 944365127},
  {1, 1, 7, 3, 11, 63, 35, 61, 255, 563, 459, 2991, 3359, 9551, 18159, 39657, 99809, 5113, 186877, 150921, 60927, 3296721, 1871105, 3115033, 17565229, 41157397, 92018101, 239867221, 330086867, 4002903, 167720519},
  {1, 1, 5, 9, 13, 49, 47, 185, 239, 221, 1533, 3635, 2045, 8043, 12231, 13563, 35951, 108963, 331591, 150225, 196907, 4124489, 2803967, 1268947, 15861221, 39721447, 3809429, 75105949, 166091089, 695886717, 1297541705},
  {1, 3, 7, 3, 25, 37, 127, 223, 51, 357, 483, 3837, 6873, 7971, 28085, 2301, 17833, 117051, 215619, 748777, 1151453, 3988089, 5668071, 1364861, 7508503, 37432871, 25451357, 177635743, 146743411, 294965119, 1137006293},
  {1, 1, 7, 9, 31, 37, 113, 31, 387, 833, 1243, 1543, 5535, 4207, 21039, 42473, 21611, 258681, 463463, 360047, 579121, 2791345, 4617235, 16167373, 12891085, 37990741, 122001461, 255195733, 118301139, 18712669, 863773779},
  {1, 3, 1, 9, 23, 59, 119, 221, 73, 185, 2007, 2885, 2563, 9559, 1309, 21163, 106031, 141817, 305753, 982069, 1803399, 2866327, 2741319, 15570889, 1734759, 56931453, 133605533, 104174839, 224892277, 758997, 539509443},
  {1, 1, 1, 13, 7, 33, 53, 179, 67, 185, 1541, 1807, 4659, 5047, 24703, 41423, 122723, 65265, 524147, 747619, 2013189, 1145113, 6482171, 10300931, 5360921, 47534801, 57636709, 236460101, 164112533, 204056601, 1283172691},
  {1, 3, 1, 11, 31, 37, 23, 215, 269, 357, 207, 645, 4219, 13799, 28925, 43371, 24877, 210933, 498527, 366937, 1740661, 1059831, 5347515, 1887465, 24462407, 10670917, 55598621, 205022231, 56139125, 818476887, 1706655435},
  {1, 3, 3, 13, 19, 27, 107, 55, 91, 71, 1695, 1815, 89, 1955, 22101, 9785, 58127, 81693, 231185, 225145, 868629, 3078721, 1822325, 13752261, 29376237, 33833779, 80467637, 138543567, 363989975, 546284081, 1237454415},
  {1, 1, 3, 15, 3, 19, 35, 247, 49, 529, 1523, 3317, 6151, 10559, 12655, 11757, 54021, 77593, 365873, 120689, 1876453, 2647243, 6000055, 281005, 24440287, 14033321, 118395517, 246911909, 96692471, 670253467, 1583055071},
  {1, 1, 7, 7, 23, 25, 107, 139, 483, 503, 1277, 243, 7879, 5023, 2319, 17801, 28165, 118285, 247315, 987709, 1945217, 4118417, 8172869, 1954503, 19389389, 6701421, 40658997, 78646453, 170946003, 997218259, 735180379},
  {1, 3, 3, 13, 3, 15, 11, 197, 135, 839, 985, 275, 5527, 11327, 4493, 31881, 2959, 58509, 176549, 820209, 1750903, 614361, 2878853, 9711331, 28991749, 55240813, 118480149, 136946871, 189388631, 882429905, 883475551},
  {1, 3, 5, 3, 25, 47, 95, 21, 113, 307, 1001, 3065, 295, 1759, 24013, 11183, 126349, 198803, 185, 190461, 942943, 252639, 258301, 766563, 20995823, 47057001, 76166845, 102125735, 5120497, 795060639, 40519389},
  {1, 1, 3, 9, 19, 19, 99, 213, 363, 449, 735, 2851, 2521, 7747, 28375, 27389, 46467, 177285, 18569, 437697, 1002055, 2256725, 1102043, 12244185, 33094977, 25694571, 12070917, 19419309, 22569751, 88691645, 1519094607},
  {1, 1, 3, 9, 5, 49, 63, 61, 157, 857, 497, 2801, 6987, 15847, 4799, 50189, 105667, 147667, 413255, 968769, 1648879, 780627, 289071, 13876639, 15914207, 47662181, 115992701, 217238677, 58614519, 378164573, 1757112537},
  {1, 1, 1, 9, 1, 41, 109, 119, 499, 939, 867, 3675, 8023, 12735, 15087, 55503, 81223, 104287, 133191, 5171, 439361, 431521, 877753, 369665, 25470953, 39529589, 123548837, 108334805, 394919829, 146318941, 1266299733},
  {1, 3, 1, 1, 13, 33, 109, 123, 289, 3, 1271, 2773, 4265, 11907, 29525, 6875, 81223, 159187, 38903, 360787, 1662349, 1359403, 6031749, 16445421, 5634431, 22039175, 86466301, 146769183, 401758965, 220939133, 183282393},
  {1, 3, 1, 11, 9, 57, 83, 221, 95, 43, 1189, 457, 7133, 11947, 24741, 64699, 40397, 137667, 242895, 873245, 376599, 1286177, 3121337, 6905895, 17919895, 55637571, 124065117, 118061071, 396209781, 476742967, 1038593245},
  {1, 1, 7, 3, 11, 49, 33, 219, 229, 289, 685, 3359, 4495, 11247, 11759, 1353, 59969, 142041, 473811, 884919, 1106117, 1384735, 41423, 9709283, 15622697, 6191309, 93810597, 184377397, 59369363, 657036247, 674772295},
  {1, 3, 1, 3, 19, 43, 67, 193, 41, 771, 407, 81, 3891, 8887, 16061, 5835, 80069, 240457, 284617, 268417, 342067, 3446847, 6810125, 3964029, 25466235, 32677013, 121739501, 25077591, 310535349, 155645535, 923698119},
  {1, 1, 7, 11, 5, 29, 51, 175, 297, 539, 1, 2245, 6439, 5791, 12687, 21289, 128841, 63559, 65219, 319621, 1003093, 3447571, 385069, 3515179, 21635619, 3501105, 113139597, 208501189, 362772275, 691458079, 642176457},
  {1, 3, 7, 15, 21, 33, 117, 183, 511, 489, 1283, 3281, 5979, 10631, 32605, 45325, 95941, 160335, 257231, 782299, 249137, 1408269, 5047331, 10675541, 6270795, 10350973, 57243181, 250688247, 389793203, 1055267539, 603931097},
  {1, 3, 7, 5, 9, 3, 125, 147, 359, 549, 369, 3049, 2405, 7483, 19589, 2173, 92751, 33863, 279421, 859319, 1704813, 984301, 4420751, 2494463, 31719399, 48073115, 130395805, 77307247, 465081203, 246969529, 1786406597},
  {1, 3, 5, 7, 19, 5, 65, 97, 483, 377, 1523, 1457, 2995, 8023, 17917, 27215, 114889, 135249, 202967, 350719, 1846063, 1408973, 3843955, 8315845, 28945771, 38201109, 66148013, 258414423, 432142769, 247577691, 1916934615},
  {1, 1, 5, 1, 11, 21, 41, 113, 277, 131, 1475, 1043, 2367, 1839, 24783, 21899, 18631, 97493, 325615, 676839, 547823, 322363, 5766117, 10395225, 5873, 4490189, 59807941, 237947957, 421167121, 434309077, 1387424079},
  {1, 3, 3, 1, 15, 17, 101, 69, 443, 865, 817, 1421, 5231, 5463, 23237, 13041, 97115, 54417, 324283, 203723, 325467, 995029, 3873939, 6487247, 23477947, 22194561, 116730925, 248582983, 44274807, 300646109, 1544683283},
  {1, 1, 3, 3, 3, 55, 95, 99, 75, 195, 1929, 3931, 5855, 11255, 24375, 14997, 49297, 94873, 35757, 449933, 945585, 853057, 4228649, 15156307, 2737141, 56116861, 14705557, 203164085, 104072215, 66925079, 1649208607},
  {1, 3, 1, 3, 19, 23, 93, 213, 241, 551, 1307, 585, 7729, 7099, 4157, 36083, 56637, 189569, 339341, 1016751, 1219203, 223903, 7204705, 11896001, 14322227, 46303327, 51873805, 56198847, 156563573, 368847551, 780954887},
  {1, 3, 1, 11, 23, 15, 53, 249, 467, 519, 95, 741, 409, 15179, 32061, 16179, 116661, 174357, 276797, 9063, 1542943, 663685, 4858293, 12591437, 14249687, 30175007, 48315549, 124476735, 38002485, 241821623, 314149123},
  {1, 1, 1, 15, 29, 37, 43, 203, 233, 877, 77, 1933, 2729, 5819, 22687, 52471, 65209, 183063, 92979, 1043205, 126969, 2471171, 2696899, 2386563, 19330323, 51596055, 129657357, 25078941, 47700213, 130929339, 937191177},
  {1, 3, 7, 11, 27, 39, 43, 161, 255, 15, 1463, 833, 495, 2535, 13125, 21621, 58969, 114433, 50957, 364921, 647131, 3168161, 3339269, 9740773, 2648835, 12097, 102865613, 176198983, 201730675, 881028183, 664283415},
  {1, 1, 7, 11, 3, 53, 81, 67, 375, 823, 1903, 3061, 395, 10735, 20727, 24881, 83345, 122905, 73743, 368711, 319621, 24709, 2974597, 16015821, 6499783, 23657645, 111760989, 67441653, 255555635, 64978079, 1598411023},
  {1, 1, 1, 1, 15, 37, 93, 233, 247, 501, 1321, 3275, 5409, 11915, 24783, 2071, 82935, 153113, 160303, 31291, 1330903, 2579105, 8314863, 16197327, 18049501, 57815395, 69423797, 149138893, 181170453, 447071477, 1469936411},
  {1, 3, 3, 7, 7, 11, 5, 105, 139, 983, 1239, 531, 3881, 13163, 7629, 22609, 7549, 100885, 47657, 318015, 102975, 95805, 7879125, 15200005, 23821425, 55056099, 128466437, 24646863, 404278615, 995676795, 1224370459},
  {1, 1, 5, 3, 19, 49, 107, 227, 361, 101, 355, 2649, 7383, 4727, 17511, 32563, 60701, 214865, 379595, 30933, 133633, 1435635, 5969515, 1630717, 33139963, 28011321, 115491757, 258433061, 509428849, 168882135, 653434647},
  {1, 1, 7, 5, 25, 41, 101, 121, 209, 293, 1937, 2259, 5557, 2979, 26527, 20945, 55551, 43999, 495199, 991715, 1141895, 2737511, 5655859, 12394511, 20276789, 11380359, 112412949, 139383773, 429234323, 444775601, 71180053},
  {1, 1, 3, 7, 7, 1, 9, 13, 463, 1019, 995, 3159, 107, 11919, 29287, 28501, 101397, 54485, 140227, 691943, 1679531, 3269057, 7823361, 3432953, 32696173, 5597561, 86624117, 14976421, 247614871, 288045267, 1175510299},
  {1, 3, 5, 11, 5, 35, 127, 97, 261, 789, 807, 807, 6257, 6811, 10829, 47959, 5821, 130007, 294517, 655549, 1947107, 1396651, 3073003, 13803817, 11936877, 20774835, 24525173, 233989135, 482004753, 839363191, 1736996609},
  {1, 1, 7, 5, 11, 13, 45, 91, 417, 101, 1973, 3645, 2107, 11295, 21143, 28849, 67551, 87885, 108255, 532395, 851365, 390483, 6155611, 5994603, 16472755, 1561157, 14316429, 191892949, 533752051, 1041366417, 378666759},
  {1, 1, 3, 7, 5, 63, 57, 49, 203, 157, 115, 1393, 8117, 4147, 5551, 2101, 51125, 9175, 106489, 753699, 1867919, 3244253, 6783999, 950453, 18844467, 16422507, 41533197, 202768109, 430347255, 639100659, 442127129},
  {1, 3, 5, 5, 3, 43, 15, 155, 127, 489, 1165, 3701, 4867, 8495, 2485, 36023, 24019, 94157, 196169, 13893, 239289, 3795461, 5827211, 3725459, 7844283, 27360237, 90241197, 258727671, 114110577, 504123161, 966065927},
  {1, 1, 7, 7, 29, 29, 69, 215, 415, 367, 371, 1901, 6075, 9903, 22391, 10353, 117213, 97759, 383851, 641851, 74009, 333877, 7661597, 9853377, 27718919, 53358877, 58270301, 100604213, 509976115, 1036297363, 1046333713},
  {1, 1, 1, 3, 11, 33, 89, 149, 433, 705, 1437, 1597, 505, 15835, 31647, 4919, 20597, 174809, 284651, 558439, 852435, 1131023, 2410023, 249483, 26401299, 7399335, 371853, 197111133, 442149877, 89278391, 1427430175},
  {1, 3, 5, 1, 13, 37, 19, 119, 5, 581, 2037, 1633, 2099, 9071, 24949, 8663, 53495, 215083, 327467, 527197, 881013, 100695, 7416675, 2407419, 18275471, 41963669, 37024637, 91470743, 401979697, 22897821, 1706300169},
  {1, 3, 7, 13, 5, 49, 103, 245, 215, 515, 133, 2007, 1933, 2003, 32637, 49205, 84511, 224371, 33055, 570113, 1864143, 3872337, 3744457, 3270287, 7089761, 8185575, 82359877, 144425951, 273291091, 200714417, 2067855625},
  {1, 3, 1, 9, 1, 3, 25, 197, 253, 387, 1683, 2267, 221, 387, 17597, 45299, 111511, 12903, 329781, 198763, 391243, 4159523, 5438333, 5197713, 27309185, 56988759, 65117381, 110806431, 226850901, 291135029, 908389653},
  {1, 3, 5, 15, 21, 9, 73, 201, 405, 999, 437, 3877, 6045, 4915, 6317, 16407, 6681, 28655, 516623, 618047, 611887, 2119571, 4459481, 7932831, 27522147, 48568691, 122895949, 243744015, 378483569, 306311155, 88107281},
  {1, 1, 3, 1, 31, 55, 25, 83, 421, 395, 1807, 2129, 7797, 4403, 16015, 34389, 123923, 224225, 78397, 516635, 1502925, 2195127, 1007901, 14936061, 12329075, 5546203, 25198221, 35629741, 370690423, 951597557, 1741229315},
  {1, 1, 3, 1, 23, 21, 121, 183, 125, 347, 143, 3685, 4317, 12003, 30223, 40341, 107795, 233065, 394895, 737951, 1183201, 1069607, 1965645, 3064701, 13354695, 26299547, 80703325, 110157741, 60543543, 844930549, 1588444427},
  {1, 3, 3, 3, 17, 45, 17, 223, 267, 795, 1815, 1309, 155, 5231, 213, 3601, 118137, 196779, 5779, 390623, 1560473, 2814721, 2474497, 16237885, 26002703, 28204077, 109303421, 212655351, 450942391, 1012990111, 1113731341},
  {1, 1, 1, 15, 17, 59, 5, 133, 15, 715, 1503, 153, 2887, 3351, 28423, 30295, 54457, 87099, 398997, 164231, 1529139, 587873, 8308945, 14956997, 6207355, 13253081, 83932461, 247452453, 451376373, 670491099, 593920261},
  {1, 1, 1, 1, 27, 13, 119, 77, 243, 995, 1851, 3719, 4695, 6551, 12023, 53431, 11191, 194733, 14991, 372401, 1197607, 3368613, 1246721, 6652769, 18190029, 39949733, 108506869, 97487829, 317704085, 472838933, 1085206799},
  {1, 3, 1, 5, 31, 49, 43, 165, 49, 609, 1265, 1141, 505, 10459, 21949, 41235, 102875, 80609, 327563, 892357, 786631, 2883847, 4456679, 7079655, 13894819, 14423527, 123473741, 39604831, 64776437, 273425593, 1801213707},
  {1, 1, 7, 13, 11, 63, 21, 253, 229, 585, 1543, 3719, 4141, 4931, 6687, 17457, 83223, 70653, 357549, 680343, 785563, 1637595, 4445543, 4238641, 7233373, 7323471, 123357109, 162398205, 182989587, 932818873, 1947973383},
  {1, 3, 7, 11, 23, 27, 17, 131, 295, 895, 1493, 1411, 3247, 10711, 2965, 309, 29689, 156781, 510537, 672299, 1224285, 1058741, 50349, 10158599, 14617861, 5708973, 49092437, 64715767, 251118099, 199168535, 1076735259},
  {1, 1, 5, 9, 29, 7, 97, 15, 113, 445, 859, 1483, 1121, 14587, 13647, 58035, 80919, 12715, 183017, 854315, 990113, 2200923, 6340083, 6635861, 22070681, 62444843, 108679717, 161521261, 156098641, 337064189, 545911577},
  {1, 3, 1, 9, 13, 49, 99, 107, 323, 201, 681, 3071, 5281, 8619, 77, 35827, 62999, 247467, 186591, 756485, 1889561, 1844905, 2750995, 4655719, 20404249, 13139819, 94480933, 31902703, 382672981, 584585845, 74060569},
  {1, 1, 1, 15, 9, 19, 61, 161, 7, 87, 587, 2199, 2811, 9615, 10247, 31991, 127545, 178019, 244173, 473295, 1255047, 4070913, 3430285, 5955721, 9004717, 53148265, 76181365, 227366373, 264845973, 420854235, 394644765},
  {1, 3, 3, 15, 15, 19, 95, 45, 299, 829, 981, 3479, 487, 15047, 7381, 45297, 85237, 99053, 137953, 432341, 657423, 3606621, 2054991, 10967763, 23532729, 27071693, 16061733, 204939383, 185013079, 912781587, 1580601107},
  {1, 1, 1, 9, 3, 37, 7, 19, 227, 13, 397, 513, 1257, 7323, 32735, 8599, 104479, 130853, 421239, 622001, 580857, 3927305, 6172023, 3712651, 1649727, 26885583, 84427453, 257200509, 275626805, 768521021, 1663743767},
  {1, 1, 5, 15, 15, 13, 17, 111, 135, 929, 1145, 811, 1801, 4987, 15567, 23091, 70417, 214581, 175199, 718983, 37353, 476109, 5710947, 4255447, 32128077, 510699, 79134581, 92404973, 263080209, 759876219, 968827659},
  {1, 3, 1, 3, 27, 57, 31, 19, 279, 103, 693, 631, 3409, 13227, 10061, 14323, 10653, 151337, 62035, 919729, 1930649, 4097629, 4971445, 4892855, 21053073, 62119483, 77232133, 142654383, 272486741, 408901503, 3273999},
  {1, 1, 1, 1, 15, 13, 67, 83, 23, 799, 1735, 2063, 3363, 10975, 7527, 26743, 72375, 221561, 394983, 182677, 292625, 60077, 7282829, 10102133, 11608941, 439921, 18490869, 183117445, 114989205, 753058645, 199904539},
  {1, 3, 3, 7, 3, 1, 61, 31, 41, 533, 2025, 4067, 6963, 2175, 29045, 42305, 23117, 252097, 371931, 294899, 1080609, 3485859, 5370235, 10146599, 6790821, 41273233, 70117845, 216319367, 414636183, 1008172251, 651014175},
  {1, 1, 5, 7, 17, 27, 81, 79, 107, 205, 29, 97, 4883, 1567, 15079, 22339, 72521, 76875, 299237, 617443, 337217, 3947889, 3020647, 16676587, 26861103, 48127845, 128896509, 183354069, 330018609, 600293523, 303976469},
  {1, 1, 1, 5, 19, 49, 91, 201, 283, 949, 651, 3819, 5073, 683, 27839, 13127, 108739, 173277, 312899, 201909, 737659, 694821, 2272359, 14454989, 7910377, 65292139, 132698725, 19341805, 102050645, 782143985, 316682247},
  {1, 1, 7, 9, 11, 13, 73, 197, 37, 219, 1931, 3369, 6017, 3051, 10031, 24001, 61891, 235333, 504191, 528831, 687267, 2299339, 2668101, 9649385, 26389395, 1028927, 36698509, 176357693, 288347251, 863892925, 181202951},
  {1, 1, 7, 15, 11, 7, 75, 205, 7, 819, 399, 661, 6487, 16103, 16023, 1985, 11973, 67657, 158957, 918185, 624775, 1829361, 5306441, 2345653, 14950663, 47604481, 123816157, 119822661, 483702323, 169687387, 1510330375},
  {1, 3, 3, 3, 27, 37, 95, 41, 307, 165, 1077, 3485, 563, 8351, 18309, 5185, 49737, 190033, 128479, 700089, 1148483, 3963741, 7035175, 11938035, 32746915, 33976989, 5503309, 254846007, 482930679, 411627807, 1535700999},
  {1, 3, 5, 3, 21, 49, 57, 179, 109, 627, 1789, 431, 2941, 13987, 25197, 55111, 70085, 84039, 348119, 276303, 390597, 998319, 6489509, 2562543, 24167641, 27214487, 79623461, 186412703, 92551377, 879603391, 1451569169},
  {1, 1, 7, 5, 11, 19, 43, 137, 149, 679, 1543, 245, 1381, 13059, 17247, 24129, 103247, 83421, 202961, 299873, 1275859, 1290731, 3125737, 136413, 28186535, 18672979, 108018525, 105934477, 361792563, 170993265, 1328664583},
  {1, 3, 5, 5, 15, 3, 69, 81, 135, 159, 1363, 3401, 6355, 10831, 18789, 46407, 7235, 255185, 475977, 799551, 1977623, 77169, 3954325, 119985, 7437395, 31468877, 7683469, 252122743, 76088433, 1002554521, 2051100683},
  {1, 3, 5, 1, 9, 61, 49, 53, 319, 25, 1647, 1297, 615, 4519, 3253, 9159, 66887, 54751, 7019, 502831, 812131, 2100433, 2557683, 7541985, 9206507, 27556281, 37312621, 86130087, 146210417, 475258845, 2008772621},
  {1, 3, 5, 11, 31, 43, 9, 101, 71, 919, 335, 3147, 5823, 2039, 11461, 9287, 25165, 117597, 423033, 732259, 257051, 1490409, 3337293, 1608893, 29593161, 50300245, 129861349, 5753879, 525554641, 457913879, 1009307675},
  {1, 3, 1, 1, 15, 5, 29, 109, 511, 945, 867, 3677, 6915, 13135, 14213, 42115, 82319, 98953, 285599, 83627, 424643, 1492689, 2285435, 117117, 21383135, 54843061, 5791933, 23623063, 305688373, 910805021, 773505051},
  {1, 3, 3, 15, 17, 49, 91, 111, 215, 29, 1879, 97, 2505, 15675, 20429, 12929, 100229, 235651, 56751, 175101, 988829, 2412261, 1801103, 351665, 2115643, 55757527, 109705389, 8355615, 112970871, 280511795, 1225277453},
  {1, 3, 1, 13, 19, 61, 11, 111, 163, 777, 533, 1113, 5339, 10319, 17861, 61571, 119171, 56717, 463791, 640701, 804365, 3011537, 5280115, 10933771, 1963991, 46506173, 47555357, 254868407, 176745909, 996764945, 436846599},
  {1, 1, 7, 9, 17, 55, 117, 91, 455, 289, 557, 913, 4455, 15479, 28279, 51585, 91779, 219295, 215729, 633311, 2087405, 3781925, 4222275, 12871923, 2395855, 43679857, 133464445, 249005957, 170887219, 450885597, 742178845},
  {1, 3, 1, 7, 25, 19, 123, 37, 1, 277, 717, 2965, 4469, 7443, 19613, 61955, 48649, 249363, 34365, 869945, 1914527, 2578343, 3267459, 6708963, 11821147, 28736363, 121539501, 58641775, 141394293, 755686011, 692322317},
  {1, 3, 7, 3, 19, 23, 87, 235, 209, 457, 2041, 2893, 1805, 2563, 20029, 29701, 87041, 254985, 60477, 433205, 1522713, 2485339, 5001159, 13382447, 33341339, 16597619, 129324333, 32946703, 215084147, 72329471, 1557135391},
  {1, 3, 3, 5, 5, 43, 23, 61, 351, 791, 59, 2009, 2909, 4307, 19357, 48129, 85007, 164867, 511029, 511061, 351423, 2807161, 6784349, 5545833, 2027507, 35057667, 11678989, 4856655, 293506167, 1035910777, 1689255961},
  {1, 1, 3, 7, 5, 1, 27, 231, 385, 257, 1261, 2701, 1807, 15399, 31287, 4869, 81157, 199943, 55575, 162061, 797653, 754467, 5986975, 9161195, 2469459, 49668769, 113830669, 217767749, 42749687, 262935763, 1877409817},
  {1, 3, 1, 1, 27, 19, 87, 253, 131, 685, 1743, 3983, 2651, 5087, 10869, 19715, 37647, 32029, 89889, 474909, 1329951, 192293, 7185759, 2986649, 9557925, 57410425, 105130069, 59517351, 489970069, 40187357, 1366622223},
  {1, 3, 7, 11, 21, 17, 11, 81, 191, 641, 1821, 3005, 7251, 10175, 2917, 58405, 110377, 3327, 250231, 93533, 1762083, 3239825, 5830619, 733827, 5559203, 32886973, 63329613, 43124663, 306339059, 69098263, 344378393},
  {1, 3, 3, 5, 15, 31, 41, 213, 55, 931, 1953, 49, 6037, 13043, 11805, 9121, 41903, 255529, 355061, 1013003, 1431899, 2879065, 5747001, 14415131, 733215, 42847379, 82763709, 61934351, 202751927, 712187897, 1352576019},
  {1, 1, 7, 15, 7, 27, 65, 223, 113, 79, 1875, 911, 5445, 6339, 11055, 53473, 92261, 156453, 377289, 188683, 1893265, 3314155, 7999985, 6476713, 15465973, 381879, 35868309, 1223453, 481424147, 203915579, 1015219211},
  {1, 3, 7, 7, 23, 55, 51, 167, 495, 25, 1585, 3447, 799, 7367, 24837, 36005, 94053, 163941, 277605, 851489, 1925693, 4189005, 6932603, 3828895, 17034449, 66645781, 15191301, 126999063, 190940115, 470320795, 1278784539},
  {1, 1, 3, 7, 27, 15, 95, 193, 337, 415, 975, 3085, 967, 295, 17975, 28261, 122149, 189049, 234309, 849485, 442979, 120719, 1819801, 11646101, 30562255, 44894465, 115414781, 174288581, 418187959, 58271059, 1433127943},
  {1, 1, 7, 15, 19, 7, 93, 41, 433, 551, 401, 3169, 3971, 15071, 29767, 4001, 117413, 125937, 20093, 836735, 2075695, 3718327, 5624897, 11350843, 24451519, 59412325, 64914493, 151235413, 172874675, 298068123, 1078918175},
  {1, 1, 7, 11, 13, 15, 53, 69, 433, 59, 1117, 3359, 6231, 7511, 4711, 33953, 47329, 46055, 41433, 264703, 1088819, 459983, 7804845, 15316687, 26851753, 28455037, 35010405, 43667765, 522101203, 705659935, 894422017},
  {1, 1, 7, 3, 23, 5, 115, 201, 225, 109, 1903, 3897, 6265, 8443, 28655, 6881, 23593, 22573, 311879, 979653, 1941275, 349343, 3536427, 252881, 23186419, 9723863, 126634509, 48818589, 299984371, 137533751, 1547136027},
  {1, 1, 1, 11, 17, 1, 39, 143, 361, 659, 1105, 23, 4923, 367, 24727, 47399, 28653, 221283, 151791, 881025, 1561381, 1373771, 4349213, 14330419, 3328985, 38031857, 45776165, 164304901, 412084949, 866345951, 1200329733},
  {1, 1, 1, 9, 27, 57, 85, 227, 261, 119, 1881, 3965, 6999, 11831, 10871, 31015, 431, 157933, 479435, 701611, 509517, 4108919, 5552721, 1060035, 14091123, 45160297, 73607565, 122888037, 378114933, 615339613, 251999247},
  {1, 3, 7, 7, 15, 7, 107, 17, 315, 49, 1591, 905, 7789, 11667, 26557, 28901, 78629, 252861, 438373, 686585, 568955, 548373, 3077027, 4330521, 29841923, 61187267, 46811085, 192928591, 73685235, 1007540859, 655223811},
  {1, 3, 1, 7, 29, 3, 47, 237, 157, 769, 839, 3199, 3195, 5695, 14485, 8611, 59177, 253367, 262661, 560701, 1244831, 4126491, 6192175, 2305361, 19219733, 54548025, 63783573, 42880807, 83761813, 217511899, 1618947081},
  {1, 1, 3, 15, 25, 39, 63, 15, 111, 857, 881, 1505, 7671, 4567, 23895, 25957, 81261, 85419, 198665, 264245, 1751181, 2791789, 4495515, 1386655, 7225735, 19550737, 113502301, 163830277, 23330487, 610420443, 150578181},
  {1, 1, 7, 1, 3, 35, 41, 215, 99, 895, 1025, 1483, 4707, 10399, 8791, 60129, 74795, 71613, 377857, 177219, 732393, 570697, 4465797, 6419219, 16183573, 18923281, 92087061, 174524677, 12054547, 102987221, 237548559},
  {1, 3, 5, 1, 1, 31, 25, 247, 113, 841, 397, 1825, 6969, 12907, 6509, 15847, 40295, 188791, 443705, 948547, 65817, 930523, 1054039, 13703739, 7800127, 26411695, 70723517, 163795839, 153306673, 601908285, 1164178437},
  {1, 1, 3, 5, 19, 41, 49, 243, 225, 973, 241, 175, 1041, 13083, 23615, 63461, 4391, 183157, 156347, 1004431, 1677525, 457187, 739875, 15194959, 33270233, 7857259, 128517413, 166142573, 439948631, 475239793, 755311631},
  {1, 1, 1, 7, 15, 15, 105, 141, 83, 75, 1675, 3523, 5219, 8591, 10903, 54311, 4385, 113445, 189101, 609003, 1203395, 523405, 7143353, 13486385, 9807705, 30126057, 59126949, 71717861, 322345941, 452440787, 1328389147},
  {1, 1, 7, 5, 13, 27, 47, 199, 445, 841, 959, 1157, 2209, 5067, 19375, 24353, 24559, 61179, 32693, 834445, 1769109, 2617007, 4651791, 6993977, 26536895, 42334263, 92975805, 80280093, 427653171, 62819633, 726320129},
  {1, 3, 5, 15, 23, 31, 31, 81, 85, 33, 785, 2639, 7799, 8471, 28325, 7399, 98601, 127421, 240813, 151197, 88627, 2904811, 357295, 364647, 12181129, 56315149, 66744293, 238966775, 174195537, 447465747, 1928879123},
  {1, 1, 5, 13, 21, 3, 47, 99, 235, 943, 1731, 2467, 7891, 13679, 12119, 11811, 35619, 158523, 331965, 704733, 237073, 3415985, 4122069, 1413841, 15047673, 3687801, 17853093, 226441381, 356502225, 839115737, 1467700241},
  {1, 1, 1, 3, 17, 53, 85, 219, 73, 131, 1339, 875, 1191, 10439, 967, 19495, 57125, 220339, 195515, 400859, 1703173, 3736907, 308445, 11261753, 6364721, 58364541, 106079877, 22854709, 131126357, 217115031, 1005708293},
  {1, 1, 5, 7, 17, 63, 113, 7, 185, 557, 749, 3563, 4973, 4099, 30447, 5987, 53161, 116459, 143777, 418539, 1742137, 1882643, 1336547, 11778867, 14837541, 4898671, 111110101, 72666749, 28265105, 630536755, 1589621781},
  {1, 3, 3, 15, 15, 21, 43, 111, 155, 689, 345, 423, 3597, 2695, 32057, 16493, 25649, 96729, 226147, 37257, 573169, 2837341, 3661775, 15975347, 32295421, 25117499, 63888133, 163657695, 318239687, 435219651, 1613922115},
  {1, 1, 5, 1, 15, 29, 93, 5, 361, 713, 695, 3937, 425, 14039, 17163, 34127, 52915, 32853, 111227, 101887, 1260055, 1628859, 6256611, 8794013, 2824151, 6547571, 43721069, 179132445, 194902209, 701228869, 1636169179},
  {1, 3, 7, 7, 13, 41, 115, 175, 315, 937, 123, 2841, 4457, 15543, 18121, 24905, 130993, 42059, 429179, 145321, 1274897, 1232317, 3899031, 8774049, 11839495, 55688675, 97077613, 184348063, 272868803, 743204939, 1375601105},
  {1, 1, 3, 11, 25, 5, 103, 53, 423, 811, 657, 399, 7257, 6287, 22139, 48329, 40565, 19927, 204927, 991685, 286787, 2793525, 7184461, 1040607, 24946437, 3270447, 25467077, 176525645, 338432839, 1019492303, 491635029},
  {1, 1, 1, 1, 1, 13, 101, 211, 383, 325, 97, 1703, 4429, 3335, 571, 56171, 39227, 49883, 190551, 660647, 622017, 989025, 2255747, 5233123, 19343469, 22803739, 60496261, 24463965, 286515397, 548239557, 1090853701},
  {1, 3, 7, 9, 31, 45, 83, 157, 509, 701, 841, 1105, 3643, 3635, 15753, 2265, 61599, 111941, 396251, 766449, 369939, 2341151, 1260495, 16214955, 19562811, 16288225, 35960861, 120692023, 438694851, 87681733, 1622757059},
  {1, 1, 1, 7, 1, 9, 69, 17, 129, 281, 1161, 2945, 7693, 7327, 32139, 48619, 124285, 232919, 146131, 561215, 1020067, 2887867, 2823011, 5214755, 9655755, 33187991, 103751965, 103235693, 66787525, 751913155, 500249285},
  {1, 3, 7, 1, 11, 29, 51, 143, 77, 433, 1723, 2317, 5641, 9127, 12009, 1993, 106167, 244161, 293315, 49713, 1615457, 2406091, 88867, 5971537, 16396195, 55032715, 8080573, 41077567, 130648259, 124179533, 1621829847},
  {1, 1, 1, 1, 21, 43, 13, 67, 177, 505, 1629, 1267, 4885, 15463, 27675, 48683, 38331, 174159, 296665, 561411, 2034593, 1299839, 1834755, 11953959, 32286257, 24243307, 4532469, 91947997, 417136837, 177146309, 528698449},
  {1, 1, 3, 11, 27, 63, 111, 47, 233, 781, 453, 1679, 3209, 7599, 12283, 52425, 96245, 33109, 501185, 257849, 1305049, 2494271, 1104791, 11136919, 24091261, 5634479, 32542853, 35250125, 492316103, 362390991, 1179377495},
  {1, 1, 3, 13, 29, 27, 119, 141, 493, 971, 461, 1159, 633, 7855, 10491, 58505, 33971, 156063, 234153, 769945, 1966811, 3434367, 8167833, 2482627, 23996453, 56654975, 103811333, 202407373, 353408583, 300990665, 630413649},
  {1, 1, 3, 15, 23, 5, 79, 215, 163, 149, 1805, 2399, 61, 7111, 31419, 38441, 125169, 60817, 61603, 656269, 67713, 18645, 2169467, 16419823, 10624173, 51717787, 112420101, 93697181, 510283463, 728148939, 1004165979},
  {1, 3, 5, 13, 19, 5, 1, 39, 409, 561, 709, 829, 1357, 2223, 5417, 50603, 11711, 11649, 118203, 861619, 877805, 116047, 4605915, 2542863, 15127951, 22869423, 52214669, 63747663, 269305537, 242127809, 208147399},
  {1, 3, 3, 13, 19, 43, 9, 177, 449, 447, 73, 2107, 5669, 12391, 15513, 64493, 19571, 129281, 78629, 349447, 948203, 3260875, 6013893, 14697775, 32161969, 11748539, 13102741, 142495775, 105038151, 609639361, 1573616735},
  {1, 3, 5, 1, 23, 13, 63, 109, 203, 593, 829, 4017, 6881, 13255, 27721, 55883, 9843, 207645, 375611, 554877, 1810363, 854025, 8333807, 9906695, 6197315, 38462651, 29626365, 161235615, 469519297, 833376717, 1133420739},
  {1, 1, 5, 7, 3, 9, 53, 175, 391, 169, 1283, 3793, 4451, 451, 25931, 11103, 110805, 143381, 372791, 200783, 905389, 2011297, 250527, 10629753, 22401567, 35848697, 107286925, 180031605, 200669121, 1019858499, 402388951},
  {1, 1, 5, 7, 29, 43, 9, 5, 209, 77, 927, 2941, 8145, 13975, 2987, 29135, 124789, 124427, 60969, 604727, 849551, 3368531, 4389143, 6355389, 9750683, 2637907, 15287069, 90090141, 249107905, 965664579, 1260435145},
  {1, 3, 5, 15, 17, 49, 5, 143, 131, 771, 1685, 925, 2175, 12491, 25641, 27451, 68381, 134663, 482827, 823887, 115453, 2569989, 8003389, 14421691, 12487775, 37588333, 30380717, 81178119, 192290625, 941874883, 1854121925},
  {1, 1, 3, 11, 27, 27, 27, 159, 161, 1015, 1587, 4049, 1983, 1051, 18475, 9817, 128565, 58101, 392773, 135717, 325345, 2835287, 2613373, 10183477, 1125615, 36105877, 31613037, 111566085, 492985159, 482281807, 343207895},
  {1, 3, 1, 3, 23, 57, 119, 67, 481, 577, 389, 3319, 5325, 15975, 4057, 32399, 99481, 172729, 447439, 200237, 1505989, 346111, 3340683, 15936655, 7175781, 13114011, 71828773, 131614367, 41471685, 974653391, 1170767187},
  {1, 3, 5, 1, 19, 39, 87, 61, 329, 657, 1773, 31, 1707, 12299, 21657, 39099, 11507, 133753, 397945, 911969, 1576571, 157275, 255211, 13657463, 19472765, 61154029, 5834565, 90912967, 405794497, 306661837, 1892033351},
  {1, 1, 3, 1, 5, 25, 15, 241, 131, 815, 1751, 3029, 8039, 15355, 19851, 49177, 24255, 26623, 214139, 915557, 1005815, 4118809, 3361569, 646545, 7172819, 33078373, 116884829, 257741957, 38065223, 659357765, 1033165001},
  {1, 3, 3, 13, 27, 13, 77, 87, 437, 57, 621, 1031, 7891, 27, 19289, 30333, 129651, 115817, 515443, 572719, 1899877, 3103907, 5869259, 5811223, 22457317, 29014757, 127045413, 210931431, 254955207, 789719489, 1778474327},
  {1, 3, 1, 13, 23, 51, 117, 37, 331, 745, 605, 3179, 4713, 10911, 5945, 60975, 124247, 130469, 146501, 348619, 324807, 994697, 1516999, 1308887, 25980217, 20067623, 64812181, 120983023, 8017349, 31951297, 145801811},
  {1, 1, 5, 5, 19, 17, 99, 167, 87, 721, 737, 789, 2165, 11999, 20171, 40527, 17047, 15137, 451695, 490201, 2011081, 1156289, 3400919, 7080235, 20240063, 47212039, 64686989, 9391789, 62339521, 391970497, 763214791},
  {1, 3, 5, 13, 1, 51, 119, 211, 165, 299, 1327, 3053, 3343, 5515, 10441, 18395, 34239, 187763, 501961, 209801, 1077509, 3086571, 84489, 13982513, 22035979, 51048189, 35546301, 158598375, 125048385, 264050753, 199542485},
  {1, 1, 5, 15, 29, 45, 17, 129, 67, 345, 1553, 2705, 7369, 3063, 12075, 46575, 78109, 63739, 510223, 88387, 1372443, 3813145, 4125479, 3765019, 30342043, 43385907, 65148957, 175259005, 374486465, 457312971, 759024329},
  {1, 1, 1, 9, 23, 7, 13, 209, 7, 407, 317, 3077, 7287, 5211, 587, 36763, 115187, 244285, 232721, 865659, 1288131, 84685, 1701985, 7204779, 14393847, 44433301, 68320717, 261636325, 144219205, 185124557, 1221308883},
  {1, 1, 1, 5, 9, 59, 89, 3, 487, 451, 505, 2499, 7563, 2619, 19963, 30459, 94783, 92219, 21265, 650551, 833929, 1650953, 824149, 16331159, 11909913, 49981189, 64263477, 242932069, 247615813, 800750017, 1132143181},
  {1, 3, 1, 7, 21, 1, 21, 203, 101, 417, 1389, 2751, 1397, 16247, 13145, 46479, 77789, 100851, 283187, 993887, 618009, 1391179, 7503219, 4195079, 29360989, 20973619, 113262149, 180366527, 96522437, 943816139, 2093151057},
  {1, 3, 7, 13, 7, 31, 3, 247, 349, 485, 1259, 549, 6321, 6295, 17161, 65385, 129435, 228597, 303673, 863813, 1790713, 576411, 5051479, 7842121, 27579403, 21837539, 130017629, 78514975, 181580227, 189400641, 2090149595},
  {1, 1, 7, 7, 27, 33, 107, 197, 293, 729, 1753, 2571, 103, 6147, 15611, 3485, 98033, 110765, 378399, 652865, 2036315, 1508087, 915643, 2530771, 3799457, 18595401, 77928757, 168247317, 84644547, 497430723, 175916103},
  {1, 3, 5, 9, 25, 35, 5, 253, 137, 213, 2041, 3387, 1809, 8751, 19065, 11947, 123675, 116963, 71081, 745659, 1843087, 3134555, 11647, 12197899, 4612209, 30355583, 44362293, 185351503, 60619713, 3534533, 592388173},
  {1, 1, 7, 13, 15, 35, 67, 83, 295, 175, 839, 2831, 839, 5907, 8859, 56797, 90747, 3885, 18357, 222141, 1919261, 3116105, 6106537, 9502033, 24928105, 31593137, 77740853, 257047509, 73580099, 861053001, 869838419},
  {1, 3, 3, 11, 3, 17, 55, 141, 247, 991, 117, 3799, 1221, 759, 25689, 65421, 3157, 28541, 1439, 80029, 427435, 3094197, 4987513, 1231475, 23958325, 44248259, 49180485, 220524607, 175235655, 621537479, 913269071},
  {1, 1, 5, 1, 11, 37, 87, 233, 457, 653, 899, 2933, 3105, 9335, 11275, 27503, 99091, 37489, 469931, 912357, 317487, 452899, 7813655, 11090049, 17566667, 55842195, 28586589, 58993245, 66857409, 149271621, 148262623},
  {1, 1, 3, 15, 3, 31, 67, 167, 437, 9, 651, 1109, 1139, 5715, 6027, 22265, 68657, 226853, 217521, 56469, 772649, 1558355, 3540843, 5042673, 15457659, 3903137, 117749597, 28565909, 458694343, 689842251, 1613757135},
  {1, 1, 3, 1, 7, 63, 67, 17, 11, 883, 1855, 1941, 4751, 4843, 715, 17305, 62975, 71613, 151961, 83621, 1220191, 4072193, 3641837, 10268673, 30168387, 24463661, 99862493, 139100325, 530718535, 154571845, 1449929931},
  {1, 3, 7, 9, 19, 33, 113, 117, 495, 39, 1795, 2561, 5519, 10603, 11481, 46089, 105359, 35225, 338111, 344059, 1445263, 3667085, 4295921, 14891877, 15933439, 5510965, 87822317, 119280007, 512511811, 914733125, 784151247},
  {1, 1, 7, 5, 1, 3, 103, 37, 201, 223, 1101, 877, 6483, 1507, 26971, 15085, 110723, 245635, 8633, 123593, 1346659, 269339, 5236213, 3494491, 6666559, 2209009, 57934509, 118243189, 382007875, 103066049, 171783901},
  {1, 1, 5, 9, 29, 49, 51, 33, 439, 917, 861, 1321, 2135, 12379, 2875, 47247, 121739, 27015, 432035, 373913, 676911, 3491069, 7274467, 5361151, 32033963, 18864685, 32927645, 44969541, 98982849, 977987277, 1915450825},
  {1, 1, 3, 3, 1, 5, 17, 93, 217, 619, 613, 1357, 6095, 12059, 91, 60169, 50317, 184719, 76703, 746115, 1186551, 3743435, 5158097, 6960091, 890319, 2987661, 27755821, 17228933, 265731911, 939733447, 551648973},
  {1, 3, 1, 11, 3, 21, 5, 41, 15, 175, 843, 2937, 6849, 13847, 27929, 39039, 19713, 16653, 75035, 106359, 762683, 1558005, 2533273, 10387037, 23265943, 43193931, 72296653, 17632255, 516288837, 490438215, 1829146823},
  {1, 3, 3, 7, 9, 57, 55, 127, 79, 287, 445, 2205, 7989, 8303, 5721, 41309, 93449, 228639, 114483, 564093, 129881, 356473, 1921849, 7558011, 3792543, 37584887, 55327469, 244086127, 151264071, 75830219, 101104325},
  {1, 1, 7, 13, 23, 17, 93, 129, 157, 135, 1747, 1813, 4183, 11019, 18427, 32013, 105995, 95749, 496135, 635415, 81539, 2654999, 4538709, 14402005, 24559707, 62865477, 14406461, 52742981, 143919683, 531398217, 383094731},
  {1, 1, 1, 5, 31, 59, 99, 33, 425, 329, 887, 367, 1761, 11887, 7083, 58779, 19983, 153117, 505357, 608829, 885279, 196147, 3463939, 8088301, 11014345, 66649207, 106216981, 69968397, 496643013, 496417857, 1519422811},
  {1, 1, 7, 9, 17, 53, 77, 139, 435, 387, 49, 3649, 1773, 11631, 28923, 2269, 124943, 174091, 235567, 88175, 1366169, 3197197, 1460365, 339823, 1424315, 25955959, 116132157, 153903149, 421840451, 199502541, 72861645},
  {1, 3, 3, 15, 21, 57, 45, 161, 331, 719, 273, 3479, 4173, 1975, 18633, 49341, 57345, 106515, 270347, 696359, 73811, 1073601, 7659825, 8593799, 16377325, 13024539, 31969189, 76417823, 140145479, 464944323, 2093565529},
  {1, 1, 3, 9, 3, 3, 105, 201, 373, 877, 919, 1263, 6649, 8415, 28267, 42521, 68551, 64985, 63581, 50663, 1978563, 2371775, 7835919, 12408645, 6142249, 26935439, 42015061, 260899181, 436421831, 515449421, 225990991},
  {1, 3, 1, 15, 13, 43, 13, 99, 73, 163, 353, 3569, 5601, 9207, 6969, 47999, 83781, 81227, 511865, 278243, 138577, 2038551, 284185, 5672191, 9254971, 56378075, 9988221, 48093087, 204350277, 276812995, 791839689},
  {1, 3, 7, 3, 5, 9, 69, 177, 449, 47, 781, 1125, 4245, 14607, 15321, 14617, 38981, 11355, 69243, 1167, 1859923, 2055083, 6574021, 12018459, 20114871, 15266791, 36020621, 164225327, 111959619, 924688847, 610145497},
  {1, 1, 1, 5, 3, 45, 1, 123, 409, 903, 205, 2057, 7637, 447, 10043, 26971, 110543, 65985, 369379, 370227, 1984033, 2534983, 7785469, 4927195, 13726747, 41567167, 6126781, 116891725, 482779717, 573451713, 1464553415},
  {1, 3, 5, 9, 19, 47, 87, 135, 481, 799, 101, 3409, 2241, 10615, 8825, 40315, 112587, 244185, 114145, 535553, 369777, 3822807, 7205925, 1616547, 7576915, 18211675, 4130333, 117921375, 387402817, 361819077, 2087822791},
  {1, 3, 1, 13, 3, 25, 15, 27, 181, 967, 669, 2577, 7249, 983, 17913, 43263, 104903, 202177, 162263, 885781, 1219805, 1072139, 5283221, 737547, 23380163, 53137803, 129817949, 247296927, 527799365, 434310593, 1198432711},
  {1, 1, 7, 3, 31, 5, 103, 53, 1, 911, 1209, 3697, 6685, 12751, 25787, 63517, 91333, 240593, 348867, 131125, 1179715, 655747, 7209077, 2229711, 31327819, 65930055, 112340349, 122057965, 310248003, 674641095, 2019810243},
  {1, 1, 3, 1, 5, 5, 49, 135, 281, 747, 761, 2973, 7963, 15027, 14939, 57065, 32719, 30671, 449623, 384371, 1858445, 816127, 5184017, 11086679, 16681191, 63020089, 34448781, 232187541, 229942599, 679745221, 1128230089},
  {1, 3, 3, 5, 19, 61, 125, 199, 299, 515, 1365, 369, 7027, 5459, 27257, 53485, 103627, 46785, 107587, 175471, 1865573, 902797, 3835901, 10490539, 4993731, 36058057, 96643037, 158309015, 162473543, 338660297, 1280240095},
  {1, 3, 1, 7, 5, 41, 63, 229, 283, 571, 147, 447, 657, 10671, 28009, 3039, 12109, 222291, 426475, 706837, 810867, 1785049, 1771317, 4652941, 27648257, 63710967, 98290773, 121849359, 145605189, 933387851, 1121716545},
  {1, 3, 1, 11, 5, 15, 55, 7, 259, 61, 27, 1429, 5631, 13339, 18393, 19439, 68577, 24619, 494381, 263441, 274273, 3451669, 6474275, 2809529, 20560475, 29320413, 23257693, 220317159, 142604741, 832179015, 106587585},
  {1, 1, 5, 1, 3, 53, 51, 253, 155, 553, 1293, 3735, 6567, 14819, 20395, 43279, 25379, 9641, 61499, 888945, 1077283, 2805993, 2298599, 2895895, 24393269, 13695585, 27279781, 146882037, 90548929, 417955013, 1295768151},
  {1, 3, 5, 9, 5, 41, 21, 159, 101, 785, 1981, 3799, 7693, 16375, 14121, 14427, 53035, 9455, 2091, 456815, 899133, 3070259, 3460051, 8140871, 1640485, 2429131, 61938341, 5968991, 198277569, 992137925, 1472621137},
  {1, 3, 7, 7, 9, 3, 95, 105, 129, 213, 1215, 1027, 5699, 10491, 22697, 47721, 88353, 135999, 518665, 376321, 247423, 2803371, 5616655, 8918077, 19337365, 35035085, 62033829, 113172967, 262827203, 958092363, 1523169877},
  {1, 1, 3, 3, 29, 13, 9, 253, 449, 321, 341, 2879, 171, 4763, 30187, 23529, 117293, 116787, 245263, 472619, 881255, 3822131, 7697195, 6379423, 7688381, 46454141, 76925765, 146524933, 472350151, 1008723143, 372044881},
  {1, 3, 7, 11, 21, 11, 75, 35, 43, 965, 675, 2217, 7175, 15387, 19769, 26473, 48109, 244923, 51385, 42981, 831869, 1108437, 7412715, 11243473, 25775191, 7790493, 129909485, 35932775, 423368643, 32821575, 489500361},
  {1, 1, 5, 15, 31, 5, 29, 137, 311, 751, 47, 1367, 5921, 8079, 11787, 26431, 24365, 99305, 25523, 436431, 1130347, 2227309, 3974153, 9445273, 590737, 27073111, 104139221, 253574093, 79395137, 10083147, 1012637003},
  {1, 1, 3, 15, 17, 1, 45, 69, 55, 649, 835, 569, 7615, 14955, 30907, 40361, 50913, 193191, 432027, 868087, 1352311, 135125, 3327535, 14255577, 28740135, 51877797, 56005997, 9545733, 106900679, 634032971, 1817794269},
  {1, 3, 1, 13, 31, 7, 23, 15, 391, 145, 1845, 1825, 1403, 2819, 31257, 22543, 113767, 152637, 42385, 629905, 326265, 135041, 2760263, 10107407, 13270547, 37419985, 41406781, 161868471, 422172869, 193270465, 1928139739},
  {1, 1, 3, 15, 5, 9, 79, 77, 105, 399, 1933, 2503, 4781, 11543, 19499, 42073, 161, 205171, 152251, 210309, 195099, 4098011, 6576133, 10701403, 21287157, 30369403, 9485445, 229356893, 4770375, 582499787, 302762057},
  {1, 3, 1, 3, 17, 47, 19, 13, 107, 475, 759, 2933, 3761, 1255, 3225, 56735, 61545, 88175, 16293, 194485, 666627, 435433, 6158133, 7047497, 30749475, 39505059, 92471005, 157642527, 62987845, 521636687, 1814818261},
  {1, 1, 7, 11, 3, 7, 121, 209, 397, 877, 293, 847, 7039, 2355, 6123, 46797, 7597, 139581, 248985, 290447, 1051739, 2778615, 5731423, 6578199, 32450509, 39297833, 58668997, 25507541, 392256707, 776823503, 370113631},
  {1, 1, 1, 15, 29, 45, 5, 109, 335, 461, 143, 931, 4045, 13159, 32619, 4059, 93541, 1707, 466431, 830535, 1488111, 677213, 6967067, 3364509, 25376549, 6512131, 10941989, 52644573, 189655493, 1072052811, 535532121},
  {1, 3, 1, 7, 11, 57, 73, 89, 201, 173, 803, 3953, 5205, 15071, 3897, 6975, 73517, 81853, 123079, 265763, 494195, 2470675, 1717719, 57169, 17600031, 23823551, 55215629, 264552687, 134563525, 617547595, 1103696079},
  {1, 1, 5, 11, 11, 33, 37, 29, 263, 1019, 657, 1453, 7807, 7003, 1531, 30127, 26025, 114677, 7039, 970591, 1467887, 3335445, 4492925, 1481327, 25998167, 55437085, 133775757, 182010245, 339980353, 1032277711, 1510992095},
  {1, 3, 3, 13, 31, 25, 37, 47, 261, 607, 1703, 2603, 417, 10367, 4009, 29757, 4451, 51261, 383967, 248159, 154429, 3686899, 8110813, 10634369, 15417517, 21910703, 25357413, 208429487, 23906887, 658281281, 727135827},
  {1, 1, 1, 1, 31, 61, 45, 115, 275, 239, 1989, 1897, 4329, 14119, 26523, 18267, 53035, 239413, 9963, 619839, 630193, 58237, 3268353, 10183395, 5533083, 30369123, 107393405, 58908861, 110473541, 49939909, 1476765659},
  {1, 3, 5, 3, 31, 3, 11, 173, 335, 579, 1193, 2219, 7875, 6331, 23113, 32299, 126817, 255585, 307573, 998197, 1536803, 281133, 760485, 3004903, 21666593, 13999085, 52340693, 45936839, 268749377, 273519823, 1886203211},
  {1, 1, 7, 9, 29, 45, 13, 67, 399, 177, 1293, 3865, 2225, 7095, 27419, 63773, 101295, 106663, 117327, 977919, 1845393, 247809, 6334887, 12838935, 11097371, 12203691, 99274013, 151464157, 209189059, 737215693, 1475600833},
  {1, 1, 7, 11, 11, 51, 121, 227, 469, 905, 929, 2635, 4165, 13527, 5707, 52381, 81133, 89397, 379517, 1191, 1030505, 4113067, 1193163, 3242891, 31083713, 52742715, 6768309, 42201949, 200301507, 862996943, 552860503},
  {1, 3, 7, 9, 13, 39, 55, 167, 23, 147, 1603, 2083, 4645, 4183, 8489, 57113, 27887, 172071, 296549, 1010321, 422657, 936961, 1726053, 2267773, 2318525, 4695387, 127528357, 5664703, 479631427, 416980549, 834554449},
  {1, 1, 3, 15, 27, 53, 11, 155, 157, 629, 259, 3009, 4605, 3151, 3643, 29689, 114081, 226605, 375931, 496377, 177141, 3960995, 4199223, 15857105, 25719695, 53990903, 133296717, 5183437, 45562055, 269342539, 303027415},
  {1, 3, 1, 7, 15, 47, 51, 1, 259, 603, 887, 2833, 6581, 12439, 15217, 40215, 67413, 122225, 174881, 855325, 96567, 3408401, 7386701, 6011409, 14609039, 63955591, 14582573, 5549903, 172570373, 313659659, 1489390859},
  {1, 3, 5, 3, 1, 47, 91, 43, 361, 571, 29, 1453, 4269, 9479, 19409, 41171, 90329, 138871, 40765, 764725, 1929997, 4159567, 5953849, 7149883, 11814727, 9326855, 75779533, 215663727, 446106881, 360468239, 314390805},
  {1, 1, 3, 9, 11, 51, 55, 23, 415, 277, 1423, 3475, 1527, 16347, 2803, 37681, 25983, 79785, 233749, 119105, 1389961, 2478369, 2126203, 16101813, 19855389, 11222721, 55416197, 135768085, 3770631, 860429, 597523207},
  {1, 1, 3, 11, 29, 49, 101, 75, 299, 709, 805, 4037, 4389, 14967, 4179, 30289, 83037, 209275, 249659, 880407, 1216337, 236557, 187683, 4657631, 17637023, 20984959, 33979405, 225627981, 236671239, 747113743, 1984504593},
  {1, 1, 7, 3, 23, 1, 37, 51, 379, 771, 1301, 3717, 6673, 6047, 18003, 57685, 3101, 23777, 58911, 641655, 1649321, 1737229, 6855941, 14272355, 12077375, 53759179, 70737645, 241747549, 275753219, 638127367, 327812363},
  {1, 1, 5, 3, 23, 11, 125, 177, 375, 665, 951, 1577, 2603, 4683, 30835, 16791, 49081, 11169, 522773, 562715, 482011, 3288837, 2595879, 14384693, 5321719, 6171945, 10271597, 11376309, 267246337, 58721543, 1535321859},
  {1, 1, 1, 1, 1, 5, 71, 255, 21, 459, 467, 2083, 5415, 14171, 563, 9523, 81779, 38579, 145463, 288893, 1579073, 1300955, 7920017, 3833493, 2266261, 13091793, 32955141, 28414165, 234832133, 239262469, 682266885},
  {1, 1, 5, 13, 23, 29, 109, 157, 363, 971, 549, 647, 1177, 2183, 7635, 13559, 13079, 222717, 186243, 780455, 1064887, 1928449, 6452653, 4403011, 8132669, 61534471, 15636453, 141799533, 358774529, 879214857, 20147459},
  {1, 1, 3, 9, 7, 15, 101, 3, 365, 213, 745, 1155, 6099, 2659, 1043, 34737, 126143, 101797, 471985, 889835, 712633, 2041787, 5194911, 8430963, 31450433, 54785765, 49083061, 82879077, 212629767, 497351437, 640966411},
  {1, 3, 5, 15, 15, 19, 47, 179, 303, 521, 1279, 219, 2415, 1739, 31857, 6195, 6261, 102689, 283805, 982713, 2095165, 2042073, 2386439, 1052669, 21293173, 50298313, 54513381, 263126711, 183008513, 634919171, 188434203},
  {1, 3, 3, 13, 27, 11, 83, 165, 369, 989, 261, 3933, 4809, 10639, 28305, 43541, 127835, 146737, 394733, 79445, 1279603, 2789099, 365487, 6754863, 13261223, 11022595, 108581037, 97717311, 471302919, 416856833, 2048638743},
  {1, 1, 3, 11, 31, 59, 1, 185, 53, 703, 1471, 2935, 1107, 8899, 13875, 44593, 45437, 133753, 55413, 280103, 83499, 1290007, 6999621, 12226161, 9457529, 2249869, 118449013, 259612901, 344800007, 594855695, 551965971},
  {1, 3, 3, 7, 25, 3, 81, 27, 93, 521, 433, 2859, 5861, 7215, 6097, 27765, 98449, 113319, 422881, 166983, 1526889, 746971, 4459619, 11489939, 9109273, 33099331, 81866773, 207304447, 158677255, 142535435, 495072021},
  {1, 3, 3, 11, 29, 15, 49, 167, 315, 927, 543, 3473, 4307, 10715, 2993, 20053, 34941, 153467, 335461, 347455, 380885, 163589, 3054969, 12409233, 32317247, 59101153, 113921581, 149501911, 189482247, 851637511, 1801021713},
  {1, 3, 1, 3, 29, 33, 53, 15, 183, 691, 703, 1311, 3393, 9671, 19121, 24503, 116561, 187691, 268747, 888647, 135605, 611929, 353197, 4645097, 21950321, 10042287, 95287893, 189466223, 4580613, 992094479, 1773034777},
  {1, 3, 5, 13, 23, 49, 3, 11, 1, 357, 1407, 415, 7211, 7187, 6449, 37683, 83703, 246909, 37199, 890129, 690621, 2168967, 2627799, 1046885, 27570929, 30613925, 107559413, 243073287, 4295937, 591435521, 1707140867},
  {1, 3, 7, 15, 1, 25, 91, 113, 323, 371, 189, 925, 1181, 7015, 14353, 56337, 11665, 80431, 119627, 318905, 1382551, 2517009, 6449669, 10971915, 33337431, 55315447, 91493581, 8669999, 362579715, 49144067, 1737471261},
  {1, 3, 3, 3, 17, 59, 119, 199, 115, 223, 877, 2193, 193, 9711, 8209, 38101, 130389, 186535, 482121, 403601, 565049, 1028301, 103401, 6247427, 1104283, 65286003, 5359293, 22961855, 470706439, 5160207, 691239197},
  {1, 1, 1, 5, 5, 35, 31, 59, 437, 411, 37, 2405, 3797, 13655, 5907, 51347, 125975, 164927, 240729, 459945, 740789, 802547, 6683297, 5733955, 17919039, 11956591, 788397, 147090765, 122471685, 431817473, 1031745793},
  {1, 3, 1, 13, 9, 37, 1, 241, 59, 157, 1785, 1223, 563, 14195, 20817, 23351, 60799, 161507, 223719, 925691, 480067, 217641, 8335707, 13127859, 27972805, 28461037, 89146981, 236203719, 344240901, 87824641, 2091854605},
  {1, 3, 5, 13, 3, 21, 25, 95, 15, 745, 85, 701, 5361, 12383, 19377, 35635, 32247, 205961, 449699, 877283, 189757, 2517505, 1094695, 5574039, 24081767, 46669563, 83362989, 175638239, 222871809, 78191873, 1072358167},
  {1, 3, 7, 1, 31, 33, 111, 195, 35, 913, 2013, 2951, 6611, 12595, 28977, 50577, 125087, 178829, 79791, 431581, 1394061, 3427541, 7331143, 15564175, 33174157, 31864685, 128581893, 95552519, 429310211, 310341901, 1273856771},
  {1, 3, 5, 1, 19, 3, 75, 119, 111, 409, 951, 1457, 4957, 4647, 3473, 38323, 124987, 54145, 438421, 1033885, 1887089, 2752817, 2228471, 10617269, 33164591, 20581455, 22416269, 185754159, 127309057, 539448589, 2005951239},
  {1, 3, 1, 15, 19, 59, 3, 155, 237, 657, 1967, 3323, 6235, 14467, 18065, 58135, 100893, 87709, 279213, 378597, 486125, 3128171, 5409403, 12454717, 31869333, 17847549, 116149925, 32394983, 472287493, 715534083, 1895178007},
  {1, 1, 5, 1, 3, 19, 45, 105, 377, 881, 167, 2255, 4483, 13603, 3347, 52055, 26395, 157329, 115845, 256731, 994979, 3094299, 5379123, 14561813, 18051217, 20477965, 93078965, 97141669, 230819585, 553281285, 1116335895},
  {1, 1, 7, 7, 13, 13, 99, 89, 201, 279, 161, 2483, 6001, 13063, 32179, 38869, 15417, 184723, 168583, 79241, 1044591, 1937451, 6566869, 7362367, 15252449, 28843527, 107490581, 52972845, 10019587, 187186947, 377035025},
  {1, 1, 7, 3, 13, 17, 97, 129, 137, 377, 1519, 183, 3725, 12959, 3731, 54741, 94461, 184475, 109211, 82867, 1841395, 1026523, 8260347, 1464461, 14216697, 1026035, 114151573, 216010205, 316690179, 803317511, 1138984721},
  {1, 1, 7, 9, 9, 5, 45, 135, 115, 181, 1685, 3505, 4387, 6347, 28083, 14965, 116951, 23307, 206599, 648023, 411981, 3376365, 3263747, 13495855, 6753771, 17371249, 75049181, 121992309, 270000387, 871317773, 1360955157},
  {1, 1, 1, 1, 19, 35, 69, 113, 305, 419, 949, 2969, 247, 6787, 755, 45683, 13715, 195841, 331541, 509715, 1751531, 1061347, 1659121, 497059, 12679907, 38152149, 4526685, 199850405, 428297989, 1003650821, 243387671},
  {1, 1, 5, 13, 23, 61, 13, 139, 501, 811, 67, 1501, 6493, 10999, 31635, 63479, 116023, 73757, 450563, 1007655, 1482817, 3465619, 2597649, 2238149, 2682347, 26701239, 129043197, 166136397, 517733633, 170467081, 2069619971},
  {1, 1, 3, 13, 15, 41, 27, 217, 293, 13, 145, 2631, 6991, 3883, 19315, 60817, 75995, 136197, 132135, 574585, 1127487, 1272935, 3640535, 12385595, 6544117, 40307089, 32179557, 247806005, 329372423, 299209481, 944020227},
  {1, 3, 3, 13, 15, 37, 71, 123, 285, 49, 627, 1283, 5993, 7783, 19409, 60565, 49851, 229893, 211467, 888381, 200229, 51995, 6510171, 2911329, 29660245, 14185151, 42144165, 255008303, 37353735, 351731457, 660651779},
  {1, 3, 3, 11, 9, 25, 11, 1, 203, 353, 1347, 1999, 2799, 6619, 22673, 4373, 83869, 98767, 275963, 545993, 1595111, 362265, 6622247, 5249777, 31866265, 31880577, 55241173, 149934935, 44387079, 1016167687, 1102117637},
  {1, 3, 5, 1, 7, 49, 101, 231, 499, 63, 1977, 2207, 7829, 9023, 2545, 49491, 10043, 188373, 101711, 845719, 2061437, 732301, 6022665, 16182487, 11285085, 56151387, 47991461, 118538079, 11075841, 363467021, 2080193811},
  {1, 1, 7, 1, 17, 15, 115, 139, 381, 943, 623, 4037, 2971, 15163, 16787, 6197, 116639, 5187, 436221, 42461, 878013, 1061179, 2208613, 7311809, 23755243, 17496449, 74969789, 41281749, 399870211, 23072517, 916541709},
  {1, 1, 3, 5, 13, 55, 23, 87, 139, 795, 1669, 1375, 1185, 10679, 27571, 25553, 50035, 193239, 291709, 635017, 1522665, 2079157, 5867709, 5380867, 12421989, 5414383, 134193157, 192997069, 97246471, 1011405057, 613996801},
  {1, 3, 3, 5, 5, 45, 97, 253, 241, 333, 645, 555, 7867, 8211, 4337, 7957, 118739, 199519, 55383, 349259, 1504431, 183707, 6819647, 2141007, 21030833, 57866685, 10541813, 61528263, 506393351, 714061577, 1915519753},
  {1, 3, 5, 1, 1, 1, 89, 27, 407, 509, 1433, 609, 2355, 3467, 6449, 22579, 96795, 205267, 32115, 221771, 1184377, 4134641, 2057347, 11112883, 29998755, 12068033, 104829181, 103712887, 297841921, 490790157, 1732273941},
  {1, 3, 7, 1, 27, 29, 5, 157, 495, 811, 1293, 1143, 827, 1507, 12977, 31185, 61535, 95305, 177243, 937487, 852551, 3621541, 3412837, 3847115, 4587261, 43443813, 98496613, 258537575, 446442243, 268802317, 814184711},
  {1, 1, 3, 3, 25, 49, 127, 111, 191, 3, 845, 1383, 2521, 14055, 3763, 18897, 108405, 4943, 199123, 682349, 416737, 2605105, 6013869, 9748987, 14633549, 14011943, 22695845, 77973357, 143174919, 272205063, 2047355157},
  {1, 1, 5, 7, 5, 51, 101, 155, 237, 461, 831, 3091, 3851, 8515, 20563, 47319, 88285, 105307, 183145, 185619, 1029473, 4136395, 614375, 10206901, 27554845, 29878717, 35844677, 40952677, 183813377, 487950083, 907436817},
  {1, 3, 7, 1, 29, 35, 105, 91, 285, 705, 131, 395, 6011, 4131, 30737, 49361, 114399, 214351, 294249, 919327, 319321, 3339963, 3950475, 7610833, 18722557, 49921277, 71743749, 104246311, 57343235, 673892109, 1819995905},
  {1, 3, 5, 3, 13, 21, 83, 173, 221, 827, 1775, 1931, 6727, 9835, 3937, 58851, 108905, 123115, 357743, 193625, 644203, 66727, 4102113, 1508729, 25364853, 26017841, 94711301, 212936599, 135642113, 160523279, 655469593},
  {1, 1, 3, 5, 3, 25, 95, 115, 205, 569, 1447, 933, 6425, 1551, 803, 59617, 106659, 79657, 232959, 215053, 256237, 1235019, 2869891, 10218037, 21536075, 917491, 85530973, 222774109, 153615367, 913323009, 1519629327},
  {1, 1, 7, 9, 31, 3, 17, 175, 145, 447, 1321, 1069, 6527, 15603, 3043, 59685, 65959, 250861, 137821, 475255, 901165, 1786159, 3916093, 14631415, 26070175, 51888261, 20019181, 118170021, 293172227, 384949261, 361804803},
  {1, 1, 3, 3, 23, 1, 79, 51, 421, 419, 873, 3939, 1801, 8423, 24163, 15137, 12901, 66353, 520911, 214573, 1911437, 386915, 4547273, 14952399, 28534289, 14953071, 122149013, 223420557, 112147463, 760525831, 2127442971},
  {1, 1, 5, 1, 3, 39, 15, 85, 169, 669, 919, 397, 5579, 3019, 1699, 60903, 18091, 158305, 518209, 383777, 1608107, 955491, 1314543, 2313101, 14394243, 44785841, 25381693, 255460309, 510002177, 285443077, 2140777495},
  {1, 3, 5, 1, 21, 61, 87, 217, 251, 619, 1091, 4009, 229, 4295, 23137, 37219, 20011, 145335, 203767, 763149, 461079, 3097385, 2249853, 6872437, 2772367, 20761175, 113951533, 199113039, 478862337, 240333837, 373644289},
  {1, 1, 1, 11, 23, 55, 85, 121, 363, 867, 315, 447, 3373, 10703, 26083, 35811, 118121, 162593, 523641, 849371, 1234483, 1159737, 5887765, 7167837, 18309417, 34944435, 96682037, 141276765, 530588677, 81832975, 299994131},
  {1, 3, 3, 13, 29, 19, 89, 85, 137, 469, 1873, 2765, 3975, 5475, 27425, 5541, 123947, 113511, 247241, 527583, 1150151, 2120899, 1381799, 12931931, 9697551, 34845565, 41780877, 118282119, 42755079, 626867201, 1981078545},
  {1, 3, 7, 13, 19, 63, 61, 77, 67, 361, 11, 1787, 4703, 4123, 19873, 13025, 23011, 127081, 79097, 927203, 671639, 309817, 901711, 3622521, 2212877, 33017801, 129820677, 178124279, 131572739, 126077953, 1715117071},
  {1, 1, 3, 11, 7, 15, 127, 105, 179, 857, 1671, 3647, 3389, 8111, 26019, 21345, 40301, 261681, 261217, 260225, 782647, 3924545, 7063199, 15435925, 2825617, 32645411, 19428213, 255122589, 74793991, 806765583, 1423678475},
  {1, 1, 1, 7, 19, 21, 99, 161, 499, 519, 1287, 2973, 479, 8643, 18595, 9379, 91621, 86845, 111891, 614249, 362823, 1030481, 1469057, 6388393, 10457843, 1083845, 74813725, 129925317, 106093573, 878097411, 196963351},
  {1, 1, 3, 13, 29, 51, 95, 251, 55, 519, 1955, 2881, 5951, 9171, 31139, 47905, 86635, 130855, 329769, 731161, 911397, 2836577, 8329673, 675277, 6784999, 38115037, 119837517, 29071077, 467522567, 810374153, 2045232145},
  {1, 1, 3, 11, 23, 63, 121, 237, 175, 311, 701, 1539, 2383, 12875, 21155, 12513, 35757, 200865, 5681, 676391, 773759, 3900949, 2940041, 13207859, 12052057, 58419945, 26005301, 17271509, 503698439, 624020495, 1259514907},
  {1, 1, 7, 5, 5, 45, 73, 97, 5, 153, 715, 2037, 3353, 16111, 25443, 50725, 4459, 114543, 338055, 31411, 1572243, 680935, 5352627, 6575577, 4101827, 66452307, 84345853, 132737565, 65907715, 364067841, 1113406489},
  {1, 1, 1, 3, 13, 7, 67, 173, 425, 843, 1497, 2729, 5193, 12431, 8739, 30435, 107617, 22123, 49341, 579821, 1749899, 1967379, 5162041, 9281327, 1581307, 1979963, 87355325, 123302877, 3793925, 870188039, 840104969},
  {1, 1, 7, 1, 23, 3, 119, 11, 77, 141, 1905, 2269, 4269, 14903, 25571, 39013, 3311, 173557, 513677, 679633, 1217413, 4151803, 3537015, 5028235, 5204575, 33062647, 75575885, 132996269, 167625731, 224859141, 1299301387},
  {1, 1, 7, 15, 1, 23, 79, 251, 439, 603, 405, 2449, 6383, 4291, 26787, 39525, 33121, 128959, 144821, 405185, 1500677, 1002273, 1673365, 5143107, 2823643, 53221677, 60562045, 229398981, 177521667, 226817035, 1870611485},
  {1, 3, 7, 11, 29, 27, 47, 255, 47, 661, 1967, 1007, 3689, 519, 8545, 39073, 126501, 77547, 641, 456185, 1819841, 1881393, 1593067, 6815845, 22547409, 49809367, 41427125, 267942991, 422073347, 720983047, 1627067393},
  {1, 3, 7, 5, 19, 39, 35, 115, 417, 373, 291, 329, 603, 10235, 29761, 45377, 38091, 52825, 262849, 441005, 339061, 1492231, 7900407, 12700041, 9824079, 17645449, 3514893, 183168631, 207400963, 414552073, 1293684751},
  {1, 3, 1, 9, 11, 33, 27, 193, 207, 423, 1311, 1369, 7307, 5403, 6593, 59335, 44491, 208985, 296183, 139417, 2038455, 114317, 2927865, 13756945, 27952295, 64654969, 52014029, 175556279, 207663109, 35753989, 899903503},
  {1, 1, 3, 11, 9, 29, 83, 17, 497, 493, 329, 3141, 5935, 9203, 1603, 40257, 45133, 144223, 283087, 863913, 907895, 872339, 1554255, 16454483, 8175383, 20611045, 47967853, 51182437, 505466887, 35778575, 698413061},
  {1, 3, 1, 5, 31, 51, 29, 171, 51, 493, 1621, 3501, 4091, 6235, 28609, 41543, 30791, 235605, 219597, 219579, 1124817, 3138981, 7880523, 9387471, 32977735, 64043361, 54041837, 203145015, 493580293, 715804681, 410112027},
  {1, 1, 5, 9, 21, 43, 105, 207, 245, 363, 1191, 699, 1139, 13859, 2115, 53703, 24771, 258503, 350145, 522239, 387037, 1984427, 2697025, 9200817, 26012853, 27786693, 37747845, 19903941, 124727297, 1011838989, 2047807489},
  {1, 1, 3, 11, 19, 5, 81, 119, 247, 169, 1337, 45, 6565, 14663, 15555, 52545, 76109, 21061, 139363, 158875, 214293, 239449, 909575, 5661739, 24178591, 6482495, 8174669, 127447821, 70373383, 90394639, 868560927},
  {1, 3, 1, 11, 3, 51, 3, 101, 159, 11, 253, 299, 5043, 9971, 17985, 22599, 54729, 118613, 291957, 777765, 1600163, 668181, 7799285, 5309995, 2821649, 52889381, 118038229, 157239911, 478138373, 790073351, 661917703},
  {1, 3, 1, 5, 11, 53, 85, 39, 57, 645, 2007, 1039, 3627, 7971, 20673, 29383, 95815, 111937, 414307, 35455, 1282637, 3406391, 3660215, 13549085, 22543585, 17592693, 67585205, 83038919, 121581573, 446473, 255258639},
  {1, 3, 5, 3, 17, 61, 97, 165, 415, 357, 283, 601, 5505, 5007, 10305, 12227, 10057, 131799, 386015, 423515, 1438375, 1524629, 3792027, 2409873, 7227695, 45472627, 89587661, 25499295, 408276993, 994226191, 501133317},
  {1, 3, 7, 3, 9, 51, 49, 85, 3, 227, 137, 309, 243, 9643, 31041, 57921, 106189, 22735, 378209, 22551, 710903, 3246441, 3926053, 7789327, 15111399, 38520537, 85752077, 100473559, 156995587, 777236495, 502812693},
  {1, 1, 5, 3, 11, 59, 11, 131, 409, 703, 455, 123, 6727, 10251, 11971, 583, 66889, 255053, 174445, 794685, 1384661, 2384351, 3285325, 16361081, 30616885, 47429745, 78584901, 121845909, 233787393, 201461767, 103518239},
  {1, 3, 7, 9, 25, 49, 21, 171, 287, 379, 667, 313, 713, 14975, 20545, 56129, 109895, 155083, 178371, 273719, 1801537, 1196681, 1747973, 15528749, 12038991, 52085499, 126861869, 147341439, 145240067, 65597445, 1610215429},
  {1, 1, 3, 9, 7, 35, 47, 3, 367, 581, 1627, 1665, 3905, 3887, 29059, 26497, 103055, 220565, 28845, 1003753, 1746909, 1855721, 897051, 3151185, 13373623, 23782595, 56628525, 148918621, 190889991, 719339533, 2074329099},
  {1, 3, 1, 1, 29, 57, 35, 55, 255, 653, 823, 2197, 6179, 5467, 18305, 37767, 25219, 245663, 421507, 785873, 1849841, 658449, 230643, 6329261, 28605819, 40139913, 72047005, 240128439, 47202309, 905035789, 1880145945},
  {1, 3, 7, 15, 17, 15, 117, 83, 359, 163, 115, 2999, 5373, 13487, 21377, 17793, 66433, 154511, 303533, 357611, 1824057, 1492801, 5621997, 13819829, 25357009, 20165395, 27906773, 178004063, 257703939, 965492739, 539246605},
  {1, 1, 5, 3, 21, 61, 35, 97, 71, 687, 207, 2917, 1049, 6287, 13699, 10375, 42633, 117395, 346007, 599001, 286887, 1253381, 4628641, 5219345, 7528571, 8121427, 63119677, 120132637, 66437121, 545964039, 1651851265},
  {1, 1, 1, 15, 13, 15, 125, 81, 263, 661, 417, 3243, 1669, 4703, 27011, 5763, 45453, 162195, 266389, 325091, 1611307, 314149, 1375771, 6866187, 19713897, 28852819, 23775189, 203100093, 181780485, 166707211, 1194344457},
  {1, 1, 7, 3, 3, 19, 111, 193, 443, 339, 659, 1211, 1557, 3279, 5635, 44549, 29197, 165381, 300565, 915017, 591391, 2710073, 6188069, 3372597, 26844061, 17180003, 99187429, 16758173, 424411139, 300679175, 122945567},
  {1, 3, 1, 3, 27, 3, 3, 173, 391, 213, 803, 3281, 3207, 2339, 13313, 5127, 21505, 158749, 373813, 1027077, 713899, 72925, 3495899, 7730825, 5363351, 46416165, 47311917, 156801607, 145752069, 370147343, 1139802143},
  {1, 1, 5, 15, 19, 1, 7, 211, 157, 603, 403, 1387, 1583, 1851, 12291, 4103, 102405, 110605, 299047, 888837, 1208541, 184635, 7672673, 15865125, 11228749, 32283897, 40672101, 114224757, 385875969, 754974731, 452984839},
  {1, 3, 5, 13, 17, 53, 125, 13, 339, 723, 521, 413, 5801, 10451, 8705, 4611, 86533, 11789, 470545, 813621, 737917, 1325581, 5500755, 305363, 30637065, 17195933, 8139945, 179880659, 325320705, 20185091, 442761221},
  {1, 1, 3, 13, 29, 9, 99, 77, 141, 609, 1533, 983, 2039, 51, 19969, 18945, 105987, 30221, 253469, 284169, 1806947, 3938893, 2245261, 3880033, 13818877, 9554391, 87677431, 177788467, 400818177, 705429505, 1397489667},
  {1, 1, 3, 11, 21, 55, 5, 51, 423, 309, 525, 3715, 3025, 15055, 12545, 52993, 4867, 104715, 471317, 472887, 312581, 3728179, 5317799, 2677813, 25437453, 8380803, 47231697, 163627471, 228655105, 362741761, 1976631299},
  {1, 1, 3, 7, 9, 21, 77, 171, 239, 341, 1653, 1093, 2273, 10723, 257, 59137, 115459, 8455, 303369, 14613, 1264973, 2699179, 7773679, 14169173, 18082165, 19291973, 58799585, 177064163, 5308417, 259194881, 76611587},
  {1, 1, 1, 15, 31, 15, 23, 35, 317, 869, 1917, 1415, 4313, 3687, 27905, 43265, 65281, 68367, 162591, 925967, 1878807, 2965795, 4026941, 16430181, 17143933, 38003847, 52674009, 121301863, 255655937, 183173121, 2062876673},
  {1, 1, 1, 5, 21, 25, 99, 167, 439, 453, 473, 431, 6665, 4989, 2433, 62849, 74881, 121477, 206229, 347801, 1837283, 1563175, 3965751, 2116421, 32237913, 66847023, 3937417, 95682813, 230965249, 35405825, 1556365313},
  {1, 1, 7, 9, 31, 47, 81, 83, 345, 43, 1363, 1885, 3155, 3185, 12161, 50817, 56199, 115593, 413855, 169903, 158417, 1617619, 4676569, 8726955, 7674835, 58293981, 77214163, 30151409, 434913281, 588529665, 216907783},
  {1, 3, 7, 1, 31, 17, 61, 185, 341, 129, 547, 717, 2071, 9991, 10881, 19587, 44679, 202625, 333983, 721297, 196029, 1811001, 1776341, 12240129, 5347491, 34006861, 93644439, 250878855, 339492865, 172277763, 1888337927},
  {1, 3, 1, 13, 23, 61, 77, 217, 455, 77, 1263, 1601, 3501, 14953, 29313, 9603, 30849, 25997, 229015, 185533, 1004237, 3077721, 6135367, 2809549, 25270383, 8035265, 86770989, 264519401, 120340481, 804569091, 1252278273},
  {1, 1, 7, 7, 19, 19, 1, 229, 431, 943, 1069, 1949, 1289, 15729, 23425, 31361, 61575, 205191, 424851, 184723, 55681, 493925, 448559, 4012335, 19118253, 5320989, 127084425, 113059569, 212418561, 93405185, 828424199},
  {1, 1, 3, 5, 1, 35, 97, 251, 487, 459, 1265, 1739, 165, 10365, 4737, 57985, 101507, 136581, 522881, 304035, 1078753, 384891, 4571751, 15207243, 31821681, 63155531, 62042661, 122094077, 327237633, 825311233, 1056292867},
  {1, 3, 5, 3, 11, 25, 79, 175, 383, 545, 187, 197, 4329, 3363, 9857, 35459, 77189, 237187, 34699, 975769, 1261263, 3704111, 2212863, 16444833, 22783547, 25174085, 31679849, 261163427, 139804673, 847757315, 1509179397},
  {1, 1, 3, 3, 29, 9, 63, 55, 175, 277, 431, 2549, 2629, 6409, 17345, 56641, 78787, 251203, 498013, 133705, 1927295, 2537591, 3303535, 8219093, 31559407, 24200117, 119360901, 195448009, 301740033, 697577473, 1565904899},
  {1, 1, 3, 15, 17, 21, 79, 139, 99, 135, 1763, 1805, 3471, 5439, 15553, 21953, 107715, 134991, 266193, 783317, 1448975, 2614475, 5713059, 2789191, 11114147, 64575053, 47194703, 51641343, 8138753, 75821057, 1523388419},
  {1, 1, 3, 9, 9, 15, 35, 119, 289, 835, 769, 3843, 4119, 4421, 23105, 49857, 80451, 141001, 401353, 128847, 1616483, 883255, 7634913, 4751107, 12681025, 34764483, 112557783, 20009605, 426782721, 59805697, 1307611139},
  {1, 1, 1, 5, 19, 19, 67, 199, 307, 815, 1367, 1231, 3927, 6593, 28609, 27841, 34497, 81093, 8915, 925523, 1869571, 1397127, 6770803, 4379503, 21123223, 39990543, 133102743, 12798721, 322637825, 235753473, 837562369},
  {1, 1, 3, 1, 29, 51, 121, 209, 431, 47, 1115, 907, 2535, 9755, 8001, 55361, 105795, 107329, 174173, 217459, 43449, 1369105, 6198767, 15828463, 5361563, 63266251, 126843431, 90440411, 112488449, 244600833, 665128963},
  {1, 1, 3, 5, 17, 1, 5, 119, 121, 223, 1719, 1291, 3947, 15891, 14785, 38721, 3651, 179525, 463697, 1029441, 757189, 562359, 2683961, 4985887, 14064759, 22282699, 33295019, 194260307, 155480065, 255070209, 848678915},
  {1, 3, 1, 15, 29, 25, 3, 131, 373, 307, 645, 3513, 1289, 1987, 7873, 57411, 85697, 79567, 108765, 552537, 1627075, 2110147, 8349365, 1089651, 7808069, 65087353, 3700553, 186137731, 273174529, 144191491, 974753793},
  {1, 3, 3, 11, 29, 45, 105, 179, 331, 465, 891, 1315, 403, 3057, 31041, 39875, 75843, 80715, 335709, 977901, 1152681, 3374835, 6060555, 170129, 7704507, 22548195, 38273875, 203960753, 376991745, 268988419, 375984131},
  {1, 1, 5, 13, 17, 59, 77, 127, 485, 855, 1147, 3093, 891, 9869, 25793, 49473, 77637, 261837, 491089, 296187, 958605, 1974975, 5600805, 3093655, 18350267, 44565589, 70779835, 73927117, 91754497, 563646465, 1801998341},
  {1, 1, 1, 7, 23, 27, 31, 203, 285, 463, 827, 685, 1349, 15051, 12737, 27201, 30785, 239047, 227543, 258395, 961759, 1851147, 8066269, 863631, 19464315, 4918509, 42408965, 86573195, 409030657, 234303489, 619458561},
  {1, 1, 1, 5, 29, 5, 107, 195, 31, 425, 19, 2865, 3869, 11153, 4417, 6849, 68417, 161989, 485597, 83909, 606635, 3742979, 3984863, 12434793, 15751379, 61873649, 44504669, 16590801, 281628673, 821891073, 871141377},
  {1, 1, 7, 5, 7, 47, 1, 73, 307, 347, 393, 2205, 7709, 15121, 27457, 30785, 43719, 113349, 391879, 322031, 924609, 1879945, 4147059, 3118235, 3593161, 15605725, 38865117, 118820049, 192229377, 426315777, 246984711},
  {1, 1, 1, 13, 15, 61, 25, 131, 113, 369, 1995, 2527, 4475, 1745, 5665, 4449, 25569, 161645, 430639, 408605, 586809, 390307, 6350289, 9972881, 1342571, 62215935, 84499739, 262268145, 50904065, 355496961, 915465217},
  {1, 1, 1, 1, 31, 63, 21, 253, 307, 281, 859, 3319, 6721, 2891, 14881, 6753, 92641, 128289, 495231, 878367, 1945909, 1907677, 3657747, 2413561, 12694971, 49012567, 97626849, 142172715, 474545153, 71793665, 5624833},
  {1, 1, 3, 11, 1, 17, 5, 183, 301, 979, 651, 1685, 6343, 10067, 7521, 44129, 20899, 20971, 100193, 290609, 74021, 3221975, 1227597, 5410035, 19285419, 24366901, 19944743, 216169971, 83264513, 1018938369, 1257870339},
  {1, 1, 5, 15, 23, 45, 99, 145, 263, 507, 1381, 3425, 2215, 1815, 4705, 59297, 48421, 19951, 46583, 262925, 937283, 278833, 7641063, 8964123, 33304517, 30301185, 109226887, 47676663, 72670209, 409940993, 1901595653},
  {1, 3, 1, 5, 11, 63, 85, 203, 411, 881, 1369, 1237, 4657, 6541, 23457, 63139, 119329, 154277, 445227, 72863, 2024117, 1197163, 218299, 10577553, 24876089, 1333237, 66093265, 101885037, 382004225, 920304643, 1241682945},
  {1, 3, 3, 13, 17, 53, 121, 201, 269, 983, 215, 3187, 7121, 6111, 16993, 53347, 129507, 257453, 355953, 213141, 169881, 2508841, 5728557, 14825591, 27355383, 30831571, 115314801, 53388031, 179405825, 858428419, 1930863619},
  {1, 3, 5, 15, 15, 5, 13, 143, 3, 313, 1677, 1093, 3295, 3387, 7585, 53027, 18213, 103535, 440559, 626341, 79597, 3882927, 494947, 14687577, 5951661, 45601125, 106099903, 99394267, 441777153, 298406915, 1773401093},
  {1, 1, 3, 13, 3, 23, 73, 17, 257, 965, 239, 1271, 2803, 7327, 12577, 46177, 107875, 230637, 241507, 213687, 1865961, 1293297, 316705, 11011877, 5151631, 3599703, 83190867, 30180159, 105303041, 483236865, 1440375811},
  {1, 3, 5, 13, 9, 57, 115, 37, 41, 467, 135, 1403, 3811, 4741, 9185, 31075, 30693, 197869, 274985, 732825, 1590675, 1949957, 5645577, 9485683, 13342759, 58513051, 5276611, 222346021, 320070657, 539468803, 1984826373},
  {1, 3, 7, 15, 9, 33, 39, 203, 351, 367, 1355, 1403, 3685, 4757, 3425, 41955, 61927, 153647, 72233, 381825, 591303, 1045419, 4759103, 1062607, 31319019, 42215451, 125185157, 260015605, 276423681, 886031363, 2023480327},
  {1, 3, 5, 11, 31, 3, 113, 123, 203, 421, 1821, 3151, 2375, 4419, 26145, 10019, 15973, 253291, 278783, 821923, 849041, 1128795, 8135467, 6239685, 21143037, 4032751, 82870951, 139889059, 301513729, 895306755, 498467845},
  {1, 1, 1, 7, 21, 63, 99, 23, 133, 79, 991, 1755, 4989, 4709, 22369, 20577, 24545, 85287, 411701, 315295, 916931, 1817527, 5355493, 8793455, 21502143, 53311803, 77867293, 219906629, 154768385, 955202561, 1020939265},
  {1, 3, 5, 1, 25, 63, 113, 239, 49, 443, 173, 1261, 3201, 10599, 29473, 29539, 3813, 222049, 472249, 685535, 212049, 2337935, 4521297, 3442395, 31714573, 5812493, 67227937, 173688967, 276608001, 337734659, 603640837},
  {1, 3, 3, 13, 3, 25, 101, 169, 23, 585, 327, 1327, 111, 10059, 16097, 45219, 91683, 141357, 41123, 704761, 700997, 2781705, 5757815, 7446377, 4239527, 6419919, 61426191, 150228459, 33340417, 758985731, 293544963},
  {1, 3, 3, 5, 19, 1, 33, 89, 437, 213, 1933, 1741, 2603, 5625, 481, 60131, 21731, 49189, 226291, 349281, 1053185, 3055353, 3619797, 15940085, 9678189, 32746669, 108559179, 32303897, 178650113, 1014486019, 390804483},
  {1, 3, 1, 3, 15, 15, 25, 139, 73, 335, 237, 2461, 3101, 14951, 18593, 17251, 64481, 60899, 70831, 249583, 950073, 3855723, 8333161, 8956271, 4914637, 45072509, 41142845, 242305159, 429239297, 348556291, 1173558273},
  {1, 3, 5, 1, 31, 15, 31, 187, 177, 659, 1339, 3767, 4975, 7123, 21409, 11043, 82341, 3489, 469567, 960367, 513471, 1768539, 6937681, 11415475, 28731291, 12283159, 123924559, 106727603, 123399169, 193948675, 245124101},
  {1, 3, 1, 3, 25, 19, 47, 89, 107, 107, 649, 683, 3123, 11085, 3873, 49763, 110241, 64547, 142585, 905971, 620175, 1605689, 1540555, 3179275, 17991465, 39554827, 48268627, 91138029, 505525249, 765541379, 2066607105},
  {1, 3, 7, 9, 15, 21, 101, 25, 11, 625, 1555, 675, 3893, 5805, 29281, 22947, 25127, 231913, 217967, 553973, 1107141, 3006137, 5306219, 5274705, 20679155, 47851203, 106637781, 47284493, 368806913, 49644547, 2033722375},
  {1, 1, 1, 5, 7, 49, 123, 21, 439, 369, 207, 535, 4619, 14665, 23137, 62241, 46753, 84517, 198503, 1045713, 1252251, 4021749, 5924055, 5639505, 13990063, 50404279, 76254699, 159746665, 220515329, 1001411585, 1980849153},
  {1, 1, 5, 7, 1, 25, 103, 185, 99, 239, 1093, 1561, 6177, 4039, 21665, 7649, 122085, 152039, 511777, 155257, 851079, 2701785, 5059139, 4531343, 14389157, 52485433, 54819137, 265584999, 53449729, 519885825, 1088179205},
  {1, 3, 7, 5, 29, 21, 43, 103, 343, 973, 1561, 2975, 7467, 7947, 1761, 53987, 12839, 156965, 405309, 503413, 2075787, 68039, 7930551, 3766189, 24632441, 2656831, 122389131, 3783339, 413778945, 31736835, 1816476679},
  {1, 1, 7, 9, 19, 3, 13, 23, 461, 813, 1191, 985, 559, 3317, 25201, 1425, 56055, 34649, 147203, 1003795, 1402749, 1125767, 2873853, 12306973, 6503287, 45282217, 59012607, 35157285, 438795521, 552437505, 1687813383},
  {1, 3, 5, 5, 27, 31, 79, 15, 365, 901, 1949, 117, 3619, 13311, 5841, 26707, 28469, 170741, 249867, 577583, 1765407, 3420255, 12509, 16233077, 2741101, 53381, 103033075, 160357295, 282271489, 600792835, 719445253},
  {1, 3, 5, 7, 5, 33, 67, 199, 425, 189, 1691, 3099, 815, 1677, 23729, 25779, 47605, 130007, 126485, 373265, 182803, 2876087, 4069369, 6349997, 17518763, 13104747, 50688959, 230312797, 85529345, 986953475, 1135278341},
  {1, 1, 7, 11, 13, 29, 73, 137, 265, 601, 445, 3893, 2511, 8047, 16977, 10961, 81271, 144763, 444061, 949133, 1063865, 2270393, 1379673, 9407497, 27579565, 19109157, 96500703, 165315199, 368113921, 787478785, 1204918535},
  {1, 1, 3, 1, 13, 5, 57, 101, 357, 391, 335, 601, 1359, 1065, 23409, 46609, 67187, 58897, 37021, 693141, 659593, 3122421, 5373237, 13165655, 12486239, 5801609, 3906975, 182879449, 315869441, 192525569, 1457583363},
  {1, 1, 1, 1, 25, 57, 27, 115, 31, 873, 611, 2125, 447, 13585, 6321, 42065, 124017, 60817, 326281, 367529, 2024587, 2886883, 881935, 12885113, 3944275, 57765885, 51753167, 237150721, 323472129, 928621313, 573067009},
  {1, 3, 3, 13, 27, 17, 73, 11, 359, 33, 1153, 271, 4537, 15141, 1681, 39539, 72211, 212445, 52619, 267937, 832761, 451163, 7134039, 12800689, 8056977, 58063423, 24782153, 33143733, 313374977, 459774723, 908760835},
  {1, 3, 7, 3, 11, 63, 103, 61, 59, 629, 1629, 3279, 3919, 3177, 30289, 42611, 19319, 231187, 496027, 432527, 61079, 1596045, 6260491, 1821701, 25058221, 44953759, 7053695, 166111929, 489805569, 444812547, 824691463},
  {1, 1, 5, 15, 3, 63, 85, 193, 381, 165, 175, 3247, 2501, 4209, 13233, 45489, 28053, 122591, 75859, 53615, 1778501, 3707761, 498445, 7293845, 27716159, 15245951, 16662101, 193338209, 235279617, 193102081, 826948869},
  {1, 1, 5, 15, 1, 33, 59, 219, 487, 193, 1557, 703, 2907, 7953, 26257, 41393, 8949, 255199, 276945, 362865, 944427, 787307, 3430327, 8019473, 4984549, 16387663, 4593899, 93721281, 346192641, 308452097, 305571589},
  {1, 1, 7, 3, 9, 3, 105, 95, 389, 991, 21, 3841, 6983, 285, 29969, 23761, 18487, 195219, 473561, 597843, 420953, 1240111, 6345941, 5902127, 3762373, 8914481, 95948119, 31989549, 196618497, 107486977, 1984447751},
  {1, 1, 1, 1, 1, 31, 25, 137, 117, 67, 1283, 1963, 6591, 15541, 4721, 43377, 78897, 250001, 308689, 511823, 1252937, 1595097, 1376805, 175091, 28425939, 4961019, 51815087, 114172101, 514742529, 772196609, 982660353},
  {1, 3, 5, 11, 7, 15, 127, 89, 453, 777, 1827, 2311, 7661, 11833, 11249, 29875, 87701, 225243, 351191, 262143, 1281519, 3501481, 7638133, 14322873, 30346643, 63456215, 64017149, 95485033, 529071873, 141351171, 669271301},
  {1, 1, 7, 13, 19, 29, 79, 165, 223, 453, 2039, 3961, 6467, 5481, 11825, 30033, 3959, 94045, 445251, 766669, 1747967, 3151285, 3872175, 52693, 20617799, 59521769, 88833779, 55830265, 283168513, 933504257, 1875664647},
  {1, 3, 3, 7, 17, 41, 43, 157, 323, 3, 1001, 2109, 4513, 12127, 721, 50323, 68563, 55351, 437697, 724057, 601307, 3800493, 8031763, 8754131, 20763545, 15151661, 93377745, 141911727, 394373889, 1018043651, 253936387},
  {1, 1, 5, 9, 31, 57, 3, 217, 113, 271, 1663, 1367, 6949, 8165, 10545, 48177, 33813, 139801, 392143, 711401, 155027, 1192841, 2731841, 9667935, 17954319, 23051783, 110258549, 117042133, 96240385, 397564161, 1936912645},
  {1, 1, 7, 15, 27, 35, 81, 235, 61, 205, 525, 311, 6357, 2527, 15121, 26033, 75159, 60351, 312011, 983411, 1521633, 3510491, 2298061, 1946301, 1939037, 46381767, 20631237, 201934671, 411029761, 412642049, 1011399943},
  {1, 3, 1, 9, 19, 29, 71, 207, 321, 1011, 1615, 1333, 3459, 6681, 26673, 50195, 49009, 193625, 467523, 372717, 1341591, 17439, 2825649, 7492835, 17290879, 38157061, 101151443, 79151465, 454084353, 841764611, 1629798657},
  {1, 3, 7, 7, 3, 57, 41, 19, 25, 397, 565, 1837, 7625, 11813, 10129, 3091, 47191, 113815, 180947, 834505, 1003673, 98851, 7035497, 15512157, 26211813, 34322397, 38302873, 260567221, 388802817, 1054774019, 1826068743},
  {1, 3, 3, 1, 27, 47, 31, 79, 441, 961, 1255, 423, 2405, 913, 2737, 23475, 128019, 35921, 295115, 958687, 766703, 2643487, 2537289, 2041585, 28942679, 2287895, 129297589, 169669057, 188129025, 996331267, 787101955},
  {1, 3, 3, 13, 3, 29, 69, 227, 85, 201, 395, 3199, 3869, 13099, 30577, 43571, 116883, 253949, 256723, 971373, 473013, 167539, 4451109, 13293017, 10660251, 49960591, 107588493, 254697467, 71550209, 981099267, 815256835},
  {1, 3, 3, 7, 29, 61, 99, 7, 27, 227, 945, 873, 475, 4363, 3953, 30899, 26771, 230359, 278125, 388397, 1513267, 2373399, 615467, 11515795, 30956833, 19631225, 91589003, 179610971, 506365697, 591635203, 1962469123},
  {1, 3, 5, 13, 19, 21, 57, 149, 217, 443, 565, 453, 5487, 10981, 8241, 34547, 86613, 102333, 508835, 23813, 206601, 1768933, 215369, 9427339, 3509765, 944821, 88940895, 116307829, 486114561, 7991555, 1942318341},
  {1, 3, 3, 1, 9, 27, 47, 191, 35, 395, 1429, 4079, 6871, 8013, 20017, 41555, 89747, 145457, 511161, 6411, 28031, 978895, 6662035, 4912155, 7636229, 51088127, 38307847, 62033565, 470522625, 506979075, 877555459},
  {1, 3, 5, 15, 5, 43, 9, 79, 279, 563, 1125, 985, 8117, 4099, 21041, 16179, 10869, 38047, 444789, 492347, 22329, 850207, 2312103, 15999267, 18981525, 26944265, 15072293, 52007123, 244772097, 631370499, 1134510853},
  {1, 3, 5, 1, 13, 41, 21, 117, 287, 667, 701, 1483, 8167, 13283, 7601, 3603, 124949, 76241, 141949, 601145, 1750053, 748741, 2448879, 7622059, 9339597, 58297403, 21796023, 91012531, 83546369, 20914435, 591166213},
  {1, 3, 1, 3, 15, 15, 59, 5, 383, 509, 1657, 3977, 7697, 10941, 1841, 46771, 113457, 104691, 346559, 772127, 768075, 587797, 3897839, 8796397, 11459625, 23163065, 54272321, 97616525, 163559169, 603285763, 989446913},
  {1, 3, 1, 1, 17, 29, 19, 23, 377, 45, 981, 1631, 3557, 6749, 12785, 52819, 74417, 42609, 277409, 921101, 825955, 345639, 646729, 12305885, 25744197, 51338063, 91262613, 207069613, 180741889, 174002947, 1775420673},
  {1, 3, 3, 9, 9, 51, 9, 193, 345, 361, 1679, 3333, 713, 5387, 20177, 57011, 111411, 235161, 341369, 871971, 23129, 1819185, 1582825, 4141433, 20625023, 24909365, 110947705, 214674107, 166378753, 695581955, 71994627},
  {1, 3, 5, 5, 17, 45, 97, 17, 385, 349, 105, 2245, 7295, 14393, 23089, 3699, 28757, 163701, 465697, 895549, 1708625, 3999457, 6826545, 14330797, 30621145, 22747125, 57508943, 201631689, 536204545, 622007555, 1501459717},
  {1, 3, 7, 3, 19, 51, 35, 99, 79, 301, 1563, 399, 5879, 14675, 9105, 18963, 130583, 75859, 98019, 880291, 337075, 1798451, 1522591, 1224445, 18228939, 27667295, 25079527, 156748515, 324501249, 963870467, 1961923335},
  {1, 1, 7, 15, 13, 53, 55, 203, 417, 161, 2033, 1845, 6763, 3585, 2481, 9969, 106903, 25855, 327869, 567685, 1342695, 29787, 5311505, 8889809, 26993057, 21049317, 38535067, 19101617, 91837185, 7278337, 640371975},
  {1, 1, 3, 3, 7, 23, 11, 43, 241, 309, 1453, 3147, 2619, 3163, 10993, 2705, 76115, 223571, 156919, 866727, 2017691, 2114811, 4633697, 7124261, 16063101, 4309019, 90558411, 65786027, 236783361, 463283969, 1182256387},
  {1, 1, 1, 11, 17, 1, 17, 137, 443, 465, 993, 3217, 7879, 14607, 11697, 3505, 16849, 140699, 431969, 943025, 1279905, 4053145, 5101899, 256865, 28561233, 23377569, 54185175, 256427007, 394355969, 1030036737, 986877697},
  {1, 1, 7, 13, 29, 49, 71, 217, 291, 793, 135, 21, 2503, 11091, 30097, 56817, 119063, 125597, 375981, 562113, 539991, 3567017, 2860947, 12854249, 11722903, 9775653, 9975319, 44721123, 66228993, 726629633, 267656967},
  {1, 3, 1, 11, 31, 51, 121, 227, 377, 157, 1457, 1317, 5625, 6217, 2033, 34867, 42609, 210939, 168303, 873059, 1981513, 3034291, 6184361, 11419277, 2006913, 51207413, 125553737, 213242489, 514319617, 670273283, 1124826369},
  {1, 1, 3, 7, 23, 61, 47, 93, 79, 617, 1805, 2403, 5513, 16335, 20273, 62161, 83635, 73239, 455591, 716237, 1400447, 3443213, 2029343, 15022233, 6466973, 40679443, 33736473, 13656575, 393663233, 340740353, 498176771},
  {1, 3, 5, 11, 23, 25, 41, 11, 495, 587, 1223, 3107, 1469, 15223, 14673, 48531, 36341, 1435, 221991, 699977, 70873, 3326299, 4393535, 3910779, 774711, 17267059, 48568813, 50929511, 168776961, 46731011, 1642228485},
  {1, 3, 7, 7, 9, 1, 1, 49, 23, 723, 1761, 3717, 7375, 10875, 26673, 31283, 72343, 195735, 465977, 39249, 872913, 13537, 3677223, 1478051, 14525201, 56419701, 11560799, 180958571, 374015233, 695145731, 571224839},
  {1, 3, 3, 11, 25, 37, 57, 63, 309, 603, 183, 285, 1663, 5627, 4561, 13939, 89875, 77211, 221737, 909173, 320425, 998063, 2325125, 13082219, 12736071, 2251277, 96232463, 78318507, 357399809, 596445955, 1346334979},
  {1, 3, 7, 11, 19, 25, 25, 201, 391, 257, 529, 1645, 1, 15111, 14193, 1075, 119927, 205979, 375331, 23369, 986569, 769753, 6022039, 12285361, 8854497, 54340637, 83229073, 250558263, 393966849, 426085635, 1876837127},
  {1, 3, 3, 9, 11, 43, 91, 65, 5, 959, 301, 1015, 6343, 3453, 4593, 26771, 59731, 219513, 78331, 1031931, 1909963, 2419441, 454357, 490927, 3087805, 61592007, 102958583, 175968749, 247027457, 407063811, 1872464131},
  {1, 3, 3, 11, 17, 17, 103, 37, 77, 973, 575, 439, 49, 3639, 27065, 58763, 19899, 211395, 30793, 391625, 807551, 392813, 4581893, 15658037, 19936599, 30580127, 126561689, 198985871, 162169153, 460000579, 469587395},
  {1, 1, 5, 7, 1, 15, 107, 237, 231, 967, 923, 1101, 6715, 1713, 24169, 42457, 41357, 253775, 105161, 976999, 1338899, 3832485, 264159, 10071903, 7944883, 22501045, 99310787, 135579433, 95549377, 828102593, 1312730309},
  {1, 3, 1, 15, 9, 33, 29, 211, 245, 601, 1783, 887, 1209, 11785, 9657, 13259, 2633, 99271, 393297, 561257, 2016293, 2716763, 6794685, 6823281, 18464303, 42072223, 85949745, 228936049, 505900865, 176905283, 851151809},
  {1, 3, 3, 7, 21, 43, 27, 89, 27, 141, 865, 367, 1379, 4063, 12409, 53787, 17915, 159439, 381389, 712947, 1832291, 3436849, 7347843, 4378773, 18076137, 18455159, 124021963, 126260119, 90480833, 862659907, 277707331},
  {1, 3, 7, 7, 15, 17, 15, 15, 131, 649, 1955, 3289, 3983, 10689, 19913, 28539, 22799, 33615, 478423, 363881, 225655, 949175, 4765387, 16363057, 2103579, 64156033, 71675143, 125938713, 441627329, 284855363, 1396655815},
  {1, 3, 1, 5, 17, 7, 125, 69, 359, 981, 1345, 933, 5281, 7113, 11817, 55819, 95049, 76365, 31561, 228687, 391445, 2101517, 6000671, 6083197, 9933353, 33102941, 98636873, 29693825, 121702337, 845782851, 1229556289},
  {1, 1, 5, 9, 17, 7, 41, 207, 497, 1015, 493, 891, 3563, 3541, 3289, 33369, 94637, 25537, 270809, 212079, 308001, 367879, 1237385, 4825599, 9660117, 61432883, 78031123, 86972029, 468148801, 558989249, 581743941},
  {1, 3, 5, 11, 27, 3, 47, 31, 303, 1007, 2047, 2203, 6257, 8369, 10713, 34443, 73869, 27587, 262467, 816491, 1068343, 3218695, 6806903, 10194775, 23865543, 23277091, 2833353, 46284681, 407526849, 34526659, 874983877},
  {1, 1, 1, 15, 25, 15, 89, 51, 217, 357, 1133, 1917, 213, 3365, 17001, 2601, 97193, 50631, 282577, 219079, 1088737, 1354299, 346721, 15586029, 20141621, 63462645, 122983485, 181447693, 97048641, 369032001, 1715457345},
  {1, 1, 5, 13, 29, 23, 123, 207, 429, 805, 819, 2357, 6313, 11019, 6953, 51913, 6621, 122821, 136661, 616831, 1480915, 66535, 7139797, 15437885, 8507787, 47804957, 17390017, 195558355, 227751233, 596201281, 2072912581},
  {1, 1, 3, 7, 19, 15, 41, 73, 279, 11, 1089, 3107, 7737, 15953, 8409, 26377, 118475, 217615, 324123, 927255, 1624593, 4073169, 1748847, 14331715, 21989385, 53202587, 23549025, 254770777, 41380801, 242950337, 68399683},
  {1, 3, 5, 7, 7, 15, 41, 73, 493, 457, 1731, 1139, 2513, 12373, 28377, 59083, 125037, 116751, 509983, 452647, 509969, 2922609, 1644949, 15337601, 32779403, 63783707, 111009689, 103673421, 512537921, 445567043, 613339845},
  {1, 3, 5, 9, 17, 5, 55, 155, 173, 1005, 529, 3175, 7667, 4747, 3705, 9947, 37405, 238593, 244745, 1018925, 1875071, 2647059, 4342917, 5236309, 29911753, 52722959, 101867163, 226524675, 115667265, 808813891, 1682317893},
  {1, 1, 7, 7, 5, 21, 105, 31, 205, 847, 1033, 3167, 2347, 8499, 27321, 16281, 102623, 7951, 39181, 241453, 785233, 2081591, 10597, 2648327, 20737009, 60175671, 11308627, 138722427, 172534081, 207991233, 1272917063},
  {1, 3, 5, 3, 11, 17, 59, 189, 179, 1007, 33, 3287, 4813, 8177, 3961, 45739, 17101, 230155, 476435, 930617, 422179, 1598949, 4830523, 3294519, 11953353, 10364495, 117224117, 2532345, 3710657, 201129411, 1486160453},
  {1, 3, 3, 13, 27, 47, 47, 171, 413, 875, 1081, 1259, 7139, 8645, 1433, 47963, 91627, 121477, 34691, 671671, 1581255, 521971, 1874789, 1442195, 14549473, 41026435, 20843835, 141173133, 387852609, 503248067, 360876099},
  {1, 3, 5, 7, 25, 21, 51, 29, 361, 293, 51, 1119, 1453, 5283, 9273, 48811, 125277, 173455, 152449, 449213, 1779083, 2271317, 4335681, 11057725, 17296219, 23763863, 20246021, 63091899, 454918209, 165723971, 39945925},
  {1, 3, 7, 7, 29, 55, 103, 199, 511, 341, 1957, 3987, 2855, 1279, 28633, 36091, 119615, 13711, 469637, 462479, 490207, 991663, 3585735, 5388525, 16055069, 40357867, 120508687, 172949335, 143589825, 450300739, 1212176199},
  {1, 1, 1, 9, 23, 51, 61, 63, 391, 37, 55, 3771, 6517, 15913, 4249, 1129, 22281, 154241, 302751, 236987, 173557, 3055879, 397727, 8654925, 19788879, 63107075, 53602013, 113238545, 159102913, 292766529, 1914414017},
  {1, 1, 1, 9, 3, 19, 13, 147, 453, 855, 1321, 189, 5043, 11215, 5385, 48825, 11225, 26657, 75307, 590331, 1988645, 3696875, 8357949, 9255199, 29055761, 631797, 111652907, 14644967, 9347905, 869116865, 384454721},
  {1, 3, 3, 13, 23, 3, 87, 155, 401, 981, 607, 3413, 995, 6473, 12553, 29179, 30699, 160549, 386031, 329083, 540447, 2424195, 2385385, 13533133, 29010983, 7987965, 112875, 114724945, 319407681, 949428035, 399668675},
  {1, 3, 1, 9, 29, 47, 95, 123, 421, 353, 1867, 2609, 2569, 14083, 24745, 51243, 51513, 170849, 181285, 423879, 1913143, 1370995, 5419197, 8720761, 27170611, 30555705, 4232353, 145537083, 285850433, 999047747, 572264641},
  {1, 1, 5, 13, 25, 39, 29, 111, 125, 545, 1493, 2371, 6361, 6307, 31161, 745, 114173, 97381, 123569, 488559, 44949, 2078407, 662309, 7312937, 30952589, 48544331, 84281073, 167645931, 190991681, 378435137, 1272067909},
  {1, 3, 3, 11, 13, 31, 87, 75, 27, 393, 921, 3655, 3343, 16349, 30361, 1755, 98955, 84387, 343797, 905447, 551087, 2873859, 4894403, 13684689, 29249249, 714559, 113553511, 31070005, 183903425, 29796419, 2106836035},
  {1, 1, 5, 9, 19, 19, 7, 129, 223, 715, 433, 1627, 4463, 2951, 825, 43737, 68317, 193697, 369595, 155227, 34223, 3928697, 4551335, 3845715, 1380761, 55055107, 133818711, 43256879, 47973825, 958707905, 1847958085},
  {1, 1, 7, 1, 31, 13, 49, 33, 89, 43, 1529, 725, 3809, 3427, 31049, 9897, 99903, 260329, 509367, 719445, 1075673, 2944249, 3308881, 16749955, 11398673, 35198717, 49931609, 149861707, 373278785, 49444033, 1020731463},
  {1, 1, 7, 3, 1, 27, 45, 9, 309, 875, 659, 2661, 553, 7069, 20425, 39081, 2255, 111083, 8041, 894659, 91605, 2835233, 4653005, 6260323, 4641147, 54663469, 85415313, 22837349, 458823489, 244176065, 1687800519},
  {1, 1, 7, 15, 13, 37, 61, 19, 125, 683, 1227, 2255, 1455, 9339, 23449, 54681, 122559, 168871, 226085, 961149, 1530021, 191835, 1394421, 7651747, 7389491, 9814599, 47832151, 81062083, 239364417, 771022017, 1939085511},
  {1, 3, 5, 7, 19, 7, 71, 21, 465, 645, 1885, 873, 7405, 1913, 15465, 15179, 35693, 220015, 129771, 339919, 1879519, 794989, 7442633, 8901277, 9016661, 58579521, 78502693, 169530993, 165469761, 743725891, 1816422725},
  {1, 3, 1, 11, 11, 35, 79, 61, 79, 57, 1603, 3719, 6323, 16371, 15065, 16827, 92601, 39779, 49523, 882699, 1493111, 3479621, 5191735, 2923297, 26516427, 56872223, 77252315, 123128843, 78401985, 838652995, 1206656705},
  {1, 1, 7, 1, 29, 57, 85, 21, 205, 37, 2045, 683, 4901, 8223, 16313, 53961, 28639, 130729, 412149, 396577, 1904509, 1159453, 911909, 1855021, 21901685, 48518499, 25256877, 47493463, 180727105, 898801473, 1071118663},
  {1, 1, 5, 13, 31, 31, 65, 131, 259, 535, 967, 3943, 2605, 2089, 19481, 19273, 82797, 151461, 403255, 86295, 1948425, 2141147, 922459, 8220927, 25809727, 890975, 40804341, 207325041, 24369857, 326841281, 1782750149},
  {1, 1, 7, 9, 27, 61, 39, 243, 207, 41, 1909, 3279, 1331, 4635, 9625, 2169, 62847, 202337, 168819, 549669, 1744719, 974795, 5094663, 13451121, 32486477, 56832855, 115739931, 262551891, 189955521, 172120129, 640819015},
  {1, 3, 3, 5, 11, 63, 105, 19, 169, 95, 773, 3175, 1869, 1797, 3481, 17419, 29323, 223661, 286899, 144135, 1900833, 2937131, 1750257, 693623, 5235581, 57715983, 51880149, 117047565, 438190145, 27955651, 1109027651},
  {1, 3, 3, 15, 13, 33, 107, 197, 153, 795, 1477, 105, 4965, 991, 31593, 53835, 64923, 192679, 313333, 710937, 1430899, 774093, 4446577, 14678755, 33320141, 2522817, 75568669, 40187863, 500250689, 207334595, 178040643},
  {1, 3, 7, 11, 11, 37, 23, 149, 197, 3, 1035, 3857, 553, 1059, 8361, 10219, 105215, 205091, 16499, 108861, 1843503, 101165, 6705549, 4649691, 6107907, 43421913, 80898961, 238225883, 179265345, 136147651, 1244810439},
  {1, 3, 1, 3, 17, 29, 89, 189, 193, 59, 1477, 3517, 2565, 7739, 21577, 30411, 96953, 229163, 392937, 999605, 1408929, 1820309, 1850313, 10724947, 2562061, 32345621, 56592637, 236167859, 323782209, 313209027, 343430977},
  {1, 1, 1, 9, 23, 3, 25, 163, 469, 305, 1791, 3393, 6141, 8119, 17641, 15049, 1753, 191265, 138111, 634027, 1732081, 570043, 7321197, 1100185, 16309511, 65610057, 25260437, 39491695, 478656321, 647010881, 1299264961},
  {1, 3, 5, 7, 7, 41, 19, 101, 179, 487, 1071, 2761, 8043, 5103, 26345, 47355, 111645, 55791, 461439, 535969, 1101771, 3411069, 6304923, 4151071, 24219047, 21789361, 20386931, 134293879, 107170497, 401044419, 793834949},
  {1, 1, 7, 9, 1, 21, 101, 103, 349, 85, 1841, 1033, 4473, 3563, 19497, 46121, 102079, 75425, 210153, 220301, 1398157, 685071, 6419541, 10632413, 2897449, 45684417, 44792241, 93010243, 382402881, 662262593, 1746857543},
  {1, 1, 3, 13, 23, 61, 39, 27, 479, 13, 45, 1371, 7897, 10637, 24377, 23977, 26827, 116901, 47423, 817285, 348911, 2052163, 1790231, 9843285, 20910549, 20600003, 123343697, 10096773, 301353409, 485658945, 1968896323},
  {1, 1, 5, 9, 17, 61, 71, 55, 355, 99, 1695, 3053, 839, 959, 32585, 16281, 7453, 7521, 27513, 596149, 944815, 2514335, 2581035, 10055163, 14620423, 66148245, 103268159, 171996071, 15859393, 913767105, 1008119237},
  {1, 1, 3, 1, 7, 27, 87, 221, 327, 241, 461, 3177, 5933, 8299, 27945, 43529, 109243, 117033, 79855, 933283, 90879, 3583237, 3412543, 6507561, 24363461, 19188785, 100126597, 30702691, 500679617, 169760321, 2067971395},
  {1, 3, 7, 9, 5, 41, 111, 245, 447, 263, 1363, 1767, 6331, 3355, 18841, 61691, 61567, 18657, 485117, 409777, 357575, 425405, 7274327, 4610143, 1006171, 51989375, 110122403, 95059171, 219058625, 877593539, 1608293063},
  {1, 3, 3, 13, 15, 11, 15, 169, 429, 149, 1965, 2477, 7733, 2499, 11017, 31435, 33659, 227877, 505079, 209331, 1225351, 2249009, 7796437, 1728909, 7888901, 1010085, 63422269, 112161675, 364812609, 110968643, 920300867},
  {1, 1, 5, 15, 15, 47, 25, 33, 469, 701, 773, 2747, 1533, 14633, 23737, 58761, 111341, 206119, 442535, 471463, 1375681, 465273, 985293, 7919637, 20323693, 20174771, 45896885, 193413153, 276381505, 567650625, 903638853},
  {1, 3, 1, 5, 19, 57, 37, 75, 423, 11, 685, 2487, 1779, 8797, 5017, 11899, 1993, 202989, 376875, 178833, 1544685, 1475955, 651343, 2199747, 8721493, 24313007, 1011787, 83654741, 445738945, 175072451, 901206337},
  {1, 3, 1, 5, 19, 41, 67, 99, 333, 991, 953, 3221, 939, 4197, 14137, 35851, 111001, 255901, 52187, 353361, 662843, 4009835, 1078901, 15480599, 19252049, 30261645, 49038403, 242807709, 247506497, 786607811, 440836417},
  {1, 3, 1, 15, 11, 39, 25, 1, 159, 679, 465, 1611, 5799, 2537, 11081, 9611, 35129, 47319, 179395, 107871, 1087793, 2359145, 7664215, 12677855, 32126345, 22566499, 89213231, 97446993, 420848833, 781442755, 1034892609},
  {1, 1, 5, 11, 5, 37, 37, 7, 101, 703, 235, 23, 2209, 12799, 14345, 57977, 51181, 201139, 520541, 887517, 883053, 4081631, 1216861, 15880711, 20756163, 45650191, 50571081, 100845975, 73480769, 834625217, 164163013},
  {1, 1, 7, 3, 11, 23, 71, 215, 45, 269, 1539, 3625, 5773, 6889, 18041, 4601, 33439, 147547, 441171, 750207, 349807, 2852607, 3968997, 15588485, 10942475, 32060913, 8695797, 267558369, 469276225, 18406977, 1212737479},
  {1, 3, 5, 15, 27, 33, 105, 109, 205, 653, 821, 435, 1087, 2495, 4041, 61387, 92941, 98487, 344019, 466937, 1901057, 2294405, 766773, 9584149, 18211341, 58224459, 106130631, 241011079, 93441089, 426901571, 1312994885},
  {1, 1, 3, 5, 11, 39, 53, 213, 41, 385, 1425, 25, 5553, 12523, 12505, 62745, 20811, 224637, 83, 1005295, 1202797, 2820765, 5242721, 3201305, 13666473, 58637249, 39683177, 67219251, 226800961, 952621889, 387205955},
  {1, 3, 5, 15, 29, 49, 13, 253, 505, 407, 985, 2569, 6727, 4761, 7897, 35243, 8125, 63063, 183253, 1038441, 1511525, 3141669, 1249505, 11276591, 12579745, 64296529, 70851119, 174660193, 388455745, 427388611, 1442384197},
  {1, 1, 1, 3, 29, 17, 69, 47, 25, 819, 1145, 2479, 1183, 3343, 21865, 10889, 40969, 245563, 179525, 237385, 1586829, 2069815, 3064449, 14323531, 27234449, 29463639, 13822551, 161751991, 446459201, 693413313, 845473089},
  {1, 3, 1, 15, 25, 61, 43, 55, 279, 579, 361, 355, 6101, 3143, 11273, 17691, 98537, 208567, 12241, 876101, 1977987, 2345807, 89519, 3802251, 13029025, 29151851, 100824573, 228500095, 434088129, 880848323, 880822081},
  {1, 3, 5, 11, 3, 59, 125, 101, 451, 495, 1711, 3443, 3625, 15579, 489, 14843, 26061, 8051, 492235, 773987, 1349813, 2507661, 659947, 512759, 11530775, 21467515, 109767841, 263297939, 480939969, 280359107, 923414213},
  {1, 3, 1, 11, 25, 61, 49, 219, 23, 795, 481, 3609, 3691, 15419, 25177, 12875, 85481, 142067, 264785, 169797, 1626041, 563939, 3453871, 6087939, 17714665, 21370577, 22200067, 258403011, 169743041, 571106243, 1728288065},
  {1, 3, 7, 5, 9, 59, 49, 233, 345, 143, 181, 3587, 3041, 1219, 23241, 54971, 33423, 44029, 231873, 946291, 680873, 4004529, 216657, 1256055, 14678573, 41870875, 82721225, 252961115, 309558593, 810413251, 898243143},
  {1, 3, 7, 13, 9, 31, 39, 137, 261, 919, 1367, 3145, 4659, 5875, 8377, 53899, 79407, 57973, 10305, 271703, 1103871, 228945, 4476717, 6097311, 19704239, 26409393, 68668059, 267513595, 243889217, 337508035, 1143441095},
  {1, 1, 3, 3, 27, 43, 95, 65, 301, 915, 31, 451, 7743, 7277, 20073, 11337, 85531, 195355, 222275, 126691, 605879, 1719401, 1904005, 4818587, 19813543, 56360955, 133361207, 33351269, 526685761, 862883777, 399207619},
  {1, 3, 1, 5, 23, 37, 53, 31, 203, 453, 71, 1585, 6011, 16369, 17209, 13707, 62969, 20125, 107807, 284445, 1097997, 1880055, 6808467, 9335613, 13833455, 28571289, 39139555, 184695449, 76970433, 558557251, 113727681},
  {1, 1, 5, 1, 15, 47, 91, 227, 297, 45, 1415, 3647, 7811, 14015, 16841, 22185, 54797, 226425, 407319, 921367, 1708803, 3043211, 8108449, 12540165, 26491439, 29458503, 114657915, 14655319, 487701313, 483547713, 263871173},
  {1, 1, 1, 1, 29, 27, 93, 121, 169, 69, 1361, 2907, 1867, 7017, 10377, 47017, 49849, 233753, 75269, 114179, 6725, 1880705, 7923617, 15478045, 22019401, 43177859, 43373043, 261051313, 407059777, 247551553, 1559339841},
  {1, 3, 1, 7, 23, 53, 77, 41, 25, 873, 1333, 3889, 3239, 1771, 27033, 19819, 119145, 106303, 267807, 159245, 759397, 3844833, 944833, 12205137, 32412333, 61313897, 79867007, 247987155, 167593281, 410846531, 1517722177},
  {1, 1, 1, 7, 31, 27, 87, 81, 167, 343, 1981, 2499, 7749, 15747, 30473, 18137, 23561, 246719, 13831, 242179, 1273471, 2290393, 3026831, 11905119, 4430405, 52085739, 88299661, 36102939, 420482241, 189808321, 561016385},
  {1, 3, 5, 13, 1, 17, 97, 37, 81, 645, 1167, 3547, 7769, 10731, 25961, 15675, 83469, 139317, 440329, 743433, 1488985, 1812733, 7579849, 10811549, 7818167, 21067011, 26462049, 136216003, 105364289, 121801283, 1611117765},
  {1, 1, 7, 5, 9, 17, 31, 55, 151, 463, 1041, 2303, 4015, 3737, 169, 29385, 5631, 86941, 270481, 236217, 691655, 754607, 7868175, 3191975, 29407193, 44586167, 80695079, 244906593, 389788865, 384112321, 563358023},
  {1, 1, 3, 11, 31, 9, 81, 213, 95, 215, 2031, 2129, 4299, 3021, 31545, 45833, 4251, 188627, 518023, 201857, 691641, 2535709, 482407, 1418831, 14407383, 3635065, 2716675, 233194565, 274813121, 739507905, 2115723971},
  {1, 1, 1, 3, 25, 25, 115, 229, 101, 441, 783, 1729, 7905, 2375, 23353, 9689, 105833, 181371, 508801, 740737, 1853179, 2463645, 1345725, 11897393, 21412759, 34491449, 54413561, 193716783, 135267393, 827384129, 1630537921},
  {1, 1, 5, 9, 3, 19, 73, 35, 379, 493, 1333, 1647, 13, 197, 9833, 50009, 99901, 103249, 179611, 171179, 1002705, 2571819, 4699987, 5013013, 23104061, 29304839, 104546133, 50837821, 525702593, 218627649, 598355013},
  {1, 1, 7, 9, 3, 55, 99, 43, 281, 9, 73, 2477, 8183, 11055, 13113, 25257, 76751, 183281, 19099, 669599, 908251, 3538979, 4063249, 11928545, 21365665, 19006037, 12719535, 239212663, 128610113, 475958081, 1408207175},
  {1, 3, 7, 13, 25, 19, 27, 195, 469, 175, 355, 1861, 7255, 15377, 31225, 31371, 68959, 50229, 362993, 277883, 2057507, 3237883, 1356733, 2228263, 10617835, 26083709, 23206367, 178394777, 476973121, 420617923, 833080263},
  {1, 1, 3, 11, 15, 19, 115, 31, 413, 835, 697, 879, 6515, 13465, 5257, 5705, 107035, 217011, 388599, 448891, 876923, 1154039, 2686805, 9401803, 32158161, 37297623, 89850011, 62984193, 372863553, 504588737, 256298307},
  {1, 3, 3, 15, 3, 61, 105, 201, 151, 739, 49, 3963, 2573, 3303, 17065, 55291, 3723, 123031, 353195, 16629, 78081, 1287041, 2319727, 6188763, 23113865, 18142771, 109012997, 31036159, 181597889, 669508547, 27871555},
  {1, 3, 5, 7, 23, 5, 11, 215, 19, 591, 509, 2887, 1631, 4391, 9993, 3627, 14125, 102111, 309311, 974333, 1097283, 1920639, 6725355, 6120295, 18523285, 56831119, 2106167, 262521759, 430310849, 205935299, 1237828165},
  {1, 3, 3, 3, 25, 1, 109, 5, 363, 545, 1745, 503, 827, 4677, 5001, 53643, 81019, 22139, 345457, 812361, 144997, 2969181, 8077907, 4151481, 14657609, 33540383, 14729571, 192364797, 25703873, 805852995, 1027134787},
  {1, 1, 3, 15, 1, 45, 121, 141, 497, 745, 1825, 2041, 2561, 8153, 11769, 45753, 29947, 184183, 267385, 667205, 1025617, 2209365, 3172265, 2803217, 28919609, 60245697, 18964409, 261057313, 385834817, 586663233, 256254787},
  {1, 3, 1, 11, 29, 7, 71, 241, 7, 39, 1379, 2479, 7483, 7195, 11945, 43083, 64393, 25235, 77493, 195807, 1644591, 2226249, 5740751, 8108815, 31930683, 44836775, 50277267, 132080083, 87279809, 771946179, 1974681025},
  {1, 1, 7, 11, 3, 27, 39, 97, 339, 217, 1409, 1569, 4761, 1567, 2873, 33273, 91215, 154643, 128187, 340435, 1079407, 1841257, 4692251, 12147441, 23064409, 14069897, 124181633, 234585511, 403817921, 726833473, 320762183},
  {1, 1, 5, 15, 11, 53, 87, 213, 297, 923, 393, 717, 3297, 16123, 30489, 3705, 3565, 241623, 383923, 1031021, 815775, 4092189, 7046929, 15876435, 19908273, 49791477, 98484809, 206459875, 406574913, 328383937, 1858568517},
  {1, 1, 1, 11, 27, 41, 121, 49, 225, 379, 1305, 319, 2461, 5445, 15241, 37049, 75161, 105267, 321123, 269521, 125393, 3908537, 2954937, 10788355, 11020897, 41830599, 39043509, 121568733, 327660353, 19480001, 1544365377},
  {1, 1, 5, 5, 25, 3, 121, 23, 47, 843, 1679, 1427, 6393, 4199, 23433, 41881, 12365, 193949, 29217, 780635, 1478113, 69359, 907975, 16629331, 19432199, 9865467, 107250545, 53840399, 483955905, 435509185, 380093125},
  {1, 1, 5, 13, 17, 3, 17, 25, 161, 487, 121, 361, 1375, 10745, 15289, 51401, 67485, 53, 430441, 107483, 1327305, 3330081, 4714953, 3748543, 3887873, 1638289, 121648791, 182722209, 346840897, 930915009, 906259781},
  {1, 1, 7, 3, 3, 37, 7, 245, 107, 107, 745, 2415, 2131, 11419, 1801, 8137, 102543, 182811, 326971, 988525, 194959, 2943229, 7231571, 16316707, 29292753, 1402183, 11954267, 260183747, 426653633, 219562433, 581344071},
  {1, 1, 5, 3, 3, 23, 67, 91, 281, 387, 465, 905, 883, 9775, 22969, 32937, 111085, 20699, 164027, 1046607, 1627115, 555027, 8329265, 16500939, 27921561, 37059281, 24778459, 212212983, 143543105, 478830017, 396032837},
  {1, 3, 7, 15, 25, 55, 123, 49, 23, 983, 1903, 2589, 2073, 7823, 30009, 56635, 59791, 158039, 18929, 175903, 495379, 2922985, 6623967, 15402335, 18967223, 34190293, 90424049, 78015335, 130322625, 205632195, 1339764423},
  {1, 1, 5, 11, 25, 17, 63, 229, 267, 175, 1759, 1947, 479, 11089, 19721, 54889, 74973, 125811, 15649, 524809, 1133079, 578109, 5431155, 6182247, 31979047, 64166003, 43948727, 178829369, 254517185, 22638785, 572634821},
  {1, 3, 7, 3, 11, 37, 83, 95, 415, 1003, 1175, 2361, 2117, 9809, 11545, 15003, 89919, 242747, 433443, 645645, 513627, 1472215, 1072743, 8911939, 900607, 8171601, 4110093, 233422265, 238272193, 1032012739, 1762870471},
  {1, 3, 1, 9, 5, 39, 51, 129, 249, 161, 1981, 2755, 8057, 13641, 29353, 34587, 27129, 202033, 119341, 231487, 1132555, 337945, 645297, 16518121, 1739877, 3410811, 127147457, 22287761, 262406977, 190582595, 1987335233},
  {1, 1, 7, 1, 15, 47, 9, 197, 199, 549, 1091, 2853, 2331, 4535, 10105, 58537, 85359, 227737, 373751, 493607, 178257, 2156621, 3635567, 5548413, 6123291, 2725997, 50590947, 20447519, 338175681, 723000897, 103390663},
  {1, 3, 3, 13, 15, 21, 23, 111, 463, 719, 1667, 377, 5039, 10723, 14937, 21195, 129419, 53973, 265447, 659613, 1152175, 2410791, 5927607, 2786887, 17050027, 33797009, 100994887, 78058603, 201682625, 965038787, 1668731331},
  {1, 1, 3, 7, 23, 47, 39, 47, 307, 949, 1651, 2525, 5835, 1425, 8761, 10025, 121819, 16767, 75439, 808071, 1492879, 3883271, 1345627, 11893277, 11266075, 44032725, 68793651, 81637305, 37803457, 371469377, 639390147},
  {1, 3, 3, 9, 23, 47, 111, 39, 251, 1001, 179, 3985, 535, 15435, 13785, 10763, 35659, 175953, 10239, 314023, 434935, 3708911, 8260947, 1875457, 18182427, 5290329, 16353935, 170693795, 436252609, 926491459, 18610627},
  {1, 1, 3, 13, 5, 45, 51, 123, 205, 651, 1583, 1691, 1631, 11975, 20361, 33001, 81131, 58517, 452477, 900229, 132299, 3791715, 148949, 12682723, 25426663, 12092067, 37652023, 180401711, 495499841, 421994433, 657019075},
  {1, 1, 7, 9, 1, 29, 59, 27, 389, 497, 1459, 1633, 521, 14037, 5081, 48121, 95583, 110705, 320313, 1036437, 2021027, 1874147, 1312989, 477785, 17279483, 19645465, 112250929, 16241021, 234799169, 174092609, 862027719},
  {1, 1, 3, 3, 3, 23, 35, 247, 371, 729, 931, 681, 1777, 8353, 19397, 37509, 114455, 236055, 453727, 47179, 1745247, 3985883, 5723151, 1832373, 26190751, 34710653, 120240269, 4063573, 133839345, 706365393, 1839958227},
  {1, 3, 3, 1, 19, 15, 17, 191, 495, 643, 319, 37, 5691, 7049, 3309, 31271, 122807, 85149, 68303, 677499, 1679149, 1713491, 518491, 3283919, 30967891, 36776729, 100459383, 64416749, 137921841, 534950291, 1526609779},
  {1, 3, 5, 11, 5, 31, 123, 243, 335, 573, 113, 209, 4825, 7783, 25957, 65439, 91025, 151671, 459841, 86979, 1540991, 4189695, 3543731, 12005313, 1483965, 29994453, 102399533, 21895987, 363744209, 819912019, 29614805},
  {1, 3, 7, 7, 29, 19, 25, 191, 89, 515, 55, 3013, 4523, 12913, 9133, 25287, 15283, 85019, 169425, 721279, 583773, 4139075, 899117, 15099919, 466819, 60134649, 119626407, 185260693, 520812593, 668487571, 763931255},
  {1, 1, 3, 3, 15, 3, 35, 37, 339, 7, 697, 359, 4553, 1431, 5029, 47581, 50967, 122359, 25043, 663903, 606959, 755161, 2493039, 7018211, 27994757, 4939115, 74824285, 69886395, 79685713, 1063163569, 28992979},
  {1, 3, 1, 1, 9, 15, 33, 77, 161, 13, 255, 1187, 6587, 11715, 5485, 18063, 42549, 60605, 189021, 845947, 360405, 1427489, 6457717, 12372233, 11242155, 25198343, 60331927, 83657823, 88664753, 464990323, 999897457},
  {1, 3, 7, 7, 25, 57, 61, 171, 231, 43, 1219, 903, 5623, 4781, 13661, 57399, 96755, 11355, 239957, 603093, 1774313, 785023, 2731971, 12379479, 28091215, 11774299, 61270347, 239232065, 237401329, 893801907, 1330861239},
  {1, 1, 5, 15, 29, 47, 117, 23, 213, 907, 1423, 369, 4529, 9651, 17797, 50701, 111569, 30331, 335129, 1028867, 769873, 4007259, 7296977, 1727015, 2313867, 13194029, 76181981, 247764423, 527825073, 888570577, 1341296085},
  {1, 1, 5, 7, 15, 55, 105, 249, 401, 37, 1885, 3779, 3441, 9057, 25733, 16837, 118865, 87763, 9227, 209979, 840709, 568405, 7791813, 5874865, 27259785, 65320391, 85978389, 262043197, 345712785, 102198385, 787177813},
  {1, 1, 5, 3, 3, 27, 49, 89, 335, 561, 1235, 3251, 2731, 12711, 7277, 10205, 87665, 110263, 267271, 138247, 1270893, 3290117, 1611195, 10168037, 16346879, 1441255, 98910487, 245781627, 418054129, 256093809, 1300849013},
  {1, 1, 1, 15, 29, 49, 37, 173, 25, 743, 1321, 821, 5899, 9213, 877, 10037, 13045, 154267, 400393, 771101, 1133633, 2534553, 480605, 3358531, 15117085, 15937145, 28077631, 62903641, 463397329, 290871185, 1841001457},
  {1, 1, 7, 3, 1, 41, 61, 209, 275, 925, 521, 3029, 1569, 9277, 8005, 181, 8787, 241815, 353805, 636469, 622153, 3933765, 5779447, 14281801, 27700117, 45116081, 104000821, 7267513, 423266641, 804707089, 467600215},
  {1, 3, 5, 13, 17, 1, 11, 171, 441, 119, 1589, 299, 157, 11439, 29109, 39935, 38929, 1969, 256661, 496517, 1149087, 2892799, 4464645, 2286411, 19178465, 55504839, 39072345, 254462963, 23490065, 1017785459, 1202599317},
  {1, 1, 5, 9, 13, 33, 27, 77, 363, 939, 1103, 2135, 1759, 5429, 23357, 42901, 102065, 65565, 467593, 1021589, 687103, 2770393, 1739863, 10921743, 32769491, 12323243, 3410131, 6555225, 286526769, 198445745, 1559518517},
  {1, 3, 7, 1, 17, 39, 49, 201, 49, 803, 2003, 1193, 7415, 13847, 25885, 47647, 45875, 162237, 439197, 61075, 1998981, 2502085, 1482597, 1517815, 25546119, 1284917, 73345067, 231105283, 342141041, 226599763, 819696311},
  {1, 1, 5, 5, 17, 49, 39, 19, 311, 801, 1441, 3263, 7973, 14181, 24277, 17493, 31121, 232465, 437909, 523701, 1083315, 2752263, 2215739, 13651453, 7416461, 52734603, 120366417, 1747721, 46856401, 192084145, 1891845397},
  {1, 1, 3, 9, 9, 27, 59, 89, 81, 473, 1369, 3121, 7929, 10905, 7205, 33189, 112471, 88925, 75413, 860847, 1278639, 1502245, 1084589, 9129101, 3653421, 27743813, 48928525, 135181901, 351191761, 144766033, 681631955},
  {1, 3, 3, 5, 17, 35, 35, 239, 379, 431, 501, 3561, 2059, 9679, 26221, 2687, 113783, 246329, 83597, 938887, 1696215, 99507, 5213543, 10542499, 4542633, 51859749, 18576879, 164240667, 225589905, 792085075, 960450291},
  {1, 3, 5, 15, 25, 29, 113, 179, 269, 891, 301, 2017, 7513, 9379, 31909, 13263, 10065, 239859, 235677, 890001, 31173, 2026167, 3612377, 5837031, 3969929, 27080637, 11749821, 215086615, 523790481, 877169619, 1172835285},
  {1, 3, 1, 11, 17, 35, 49, 149, 135, 661, 1691, 3169, 3765, 9003, 11925, 25967, 111381, 205687, 202885, 516799, 1097189, 1511977, 6242003, 6529137, 17196055, 31548901, 83278633, 147382655, 426463601, 295038547, 111649425},
  {1, 3, 7, 15, 5, 21, 53, 241, 475, 271, 683, 2351, 2181, 6333, 5973, 13239, 68883, 15763, 478089, 36505, 1319633, 313045, 5398239, 3419675, 26274583, 41921155, 56405785, 153458801, 110346609, 557799475, 1489478039},
  {1, 1, 7, 13, 25, 33, 71, 153, 221, 507, 2017, 2401, 7545, 8489, 5925, 49125, 86611, 80473, 203925, 821381, 653523, 3493453, 3223249, 14201871, 15007845, 66257309, 75045293, 66666141, 357660241, 993118673, 1516047063},
  {1, 1, 7, 5, 1, 49, 87, 1, 179, 331, 1597, 3713, 809, 11109, 5573, 31285, 22323, 253425, 65645, 106325, 1423171, 1600309, 6452127, 16478271, 520929, 5060125, 42253837, 37105, 360618513, 598619601, 1503063255},
  {1, 3, 1, 5, 5, 61, 93, 39, 479, 977, 1099, 1291, 7049, 2797, 3077, 15063, 117109, 213625, 290929, 319865, 410489, 508771, 1067803, 6346877, 8478471, 62264287, 72614901, 34348785, 529806449, 941939187, 1921899345},
  {1, 3, 1, 13, 19, 41, 57, 77, 5, 117, 125, 115, 3969, 1345, 23205, 57511, 116469, 155825, 156135, 824397, 1625413, 161241, 2140465, 9886193, 1850137, 5430551, 95425221, 15728957, 82849169, 1022420339, 779106385},
  {1, 1, 1, 9, 15, 9, 57, 7, 219, 41, 767, 23, 5771, 14175, 9229, 10141, 63445, 65565, 510395, 481501, 950389, 1736915, 4817327, 16220933, 10649827, 26314147, 66166375, 135507763, 393863729, 712519217, 1379480177},
  {1, 3, 7, 9, 17, 61, 1, 59, 227, 349, 63, 189, 3871, 7919, 25181, 42463, 14419, 197397, 88637, 1028169, 1272981, 1818111, 6928279, 7454849, 20858307, 31049369, 90332019, 164971755, 39577649, 283807123, 423830327},
  {1, 3, 5, 5, 9, 29, 33, 203, 413, 701, 1129, 2103, 1889, 8377, 25509, 15687, 64241, 84409, 101229, 769753, 1955317, 3220239, 8109801, 10368929, 28107885, 14822035, 110176181, 147489109, 94024401, 110628147, 1361702741},
  {1, 1, 3, 1, 9, 17, 69, 115, 123, 1001, 1, 2893, 3957, 8593, 15221, 4581, 30455, 16309, 523637, 837861, 310193, 2952583, 4008095, 16542421, 3690077, 55075513, 78970465, 136931861, 245920209, 988348305, 1841881619},
  {1, 1, 3, 1, 31, 41, 83, 91, 113, 195, 1121, 2665, 6815, 1189, 12701, 58421, 53463, 212405, 90467, 394461, 706559, 1088847, 4310445, 4676639, 5180917, 6164085, 62004267, 54404433, 193336145, 292213585, 946339507},
  {1, 1, 1, 13, 3, 59, 13, 123, 95, 103, 1689, 2809, 5049, 4055, 4741, 55549, 39349, 73561, 514487, 682111, 1838065, 2695847, 3546939, 97179, 24775701, 29259829, 120938381, 99807203, 199122257, 799948625, 731020497},
  {1, 1, 1, 15, 21, 41, 11, 167, 375, 589, 207, 1631, 1597, 8091, 22997, 51981, 60725, 37147, 415521, 790373, 1867183, 1309203, 964395, 13511073, 21928003, 56880827, 50833105, 88007319, 419250225, 307969969, 940871953},
  {1, 3, 5, 5, 1, 33, 57, 89, 157, 921, 1353, 2777, 461, 14567, 9621, 6463, 80113, 83161, 126629, 899685, 149453, 458349, 2685449, 10925325, 21637197, 28512461, 106200793, 73407859, 39866577, 450910419, 56727957},
  {1, 1, 5, 1, 25, 5, 51, 247, 1, 577, 463, 3741, 303, 16059, 15901, 41613, 70737, 211797, 284093, 977521, 1343463, 2308243, 7350461, 1226797, 6528155, 26553161, 79838883, 107662527, 66813361, 530002065, 1701393973},
  {1, 1, 7, 5, 13, 7, 17, 87, 51, 987, 835, 93, 5203, 3973, 6381, 37557, 67283, 98801, 334113, 314659, 508789, 1545723, 4816135, 6276695, 4882815, 39225697, 95519119, 90744353, 279815153, 943131153, 634117367},
  {1, 1, 7, 7, 3, 27, 7, 1, 135, 171, 231, 3349, 4459, 2925, 16141, 19861, 32083, 43123, 351279, 306487, 897883, 2872829, 6497611, 15813759, 12040939, 18846793, 61742503, 157682281, 193077937, 769566865, 1300383351},
  {1, 1, 5, 5, 9, 51, 71, 153, 115, 315, 265, 2207, 4127, 12631, 20701, 22461, 67025, 32209, 399213, 815639, 1678867, 3661477, 4673375, 8330063, 25893077, 38981379, 69255723, 263255291, 308160849, 329169969, 1838137013},
  {1, 1, 3, 15, 23, 59, 35, 121, 425, 921, 1255, 2123, 5811, 15937, 15149, 8997, 91287, 255931, 270123, 376503, 870383, 148357, 43573, 16216493, 24080787, 8673119, 43291679, 142116701, 274302257, 601709041, 1286704627},
  {1, 3, 7, 7, 11, 21, 45, 57, 269, 395, 555, 783, 6677, 2889, 8949, 12295, 117107, 215291, 303655, 930201, 456081, 1509245, 609129, 5333983, 9823471, 31895443, 69020361, 45255653, 258777841, 768798003, 1586124951},
  {1, 3, 5, 7, 31, 19, 73, 35, 465, 349, 1429, 863, 4707, 6121, 20973, 53447, 61841, 237627, 493115, 822431, 1442181, 3354943, 5334373, 14618241, 23891617, 56238523, 69822407, 223303397, 518411793, 1004410803, 1914411125},
  {1, 3, 3, 9, 25, 27, 119, 159, 195, 949, 19, 73, 4511, 15711, 5533, 61919, 97047, 257365, 56037, 308399, 405187, 936835, 5075247, 14373337, 3715215, 57097229, 52791939, 238587851, 327855345, 196603091, 1040462899},
  {1, 3, 3, 7, 9, 59, 47, 57, 91, 749, 1579, 1297, 2445, 5167, 3029, 10527, 120631, 114203, 498933, 771511, 2032859, 502597, 2465815, 5071385, 28975703, 25799509, 17688369, 224569691, 428589425, 100538707, 160141075},
  {1, 3, 3, 3, 31, 57, 19, 203, 61, 927, 1477, 2863, 1305, 11673, 3653, 44167, 36663, 71551, 477987, 508069, 428223, 257791, 6551137, 10850115, 14989473, 28612099, 49763901, 246770933, 505165329, 377799731, 862810067},
  {1, 3, 7, 11, 29, 13, 3, 111, 351, 79, 1863, 2213, 3273, 7049, 30597, 42887, 94707, 196855, 275249, 402097, 1328871, 2087051, 1755235, 11839747, 1240619, 43977097, 24581957, 5582733, 255415985, 628747411, 435435607},
  {1, 3, 3, 9, 7, 23, 47, 237, 121, 877, 441, 119, 2723, 3989, 4381, 39207, 69191, 15813, 395467, 607603, 2008947, 1223025, 6942981, 3314073, 2953885, 11104515, 91304071, 4862473, 7756497, 205414707, 316804467},
  {1, 3, 3, 11, 17, 23, 63, 177, 231, 363, 1451, 33, 2169, 7251, 13141, 60255, 17095, 131143, 190045, 572667, 1277243, 638765, 2918875, 215663, 9312631, 30286669, 84637717, 181510039, 425503857, 840767955, 251550291},
  {1, 1, 5, 11, 31, 41, 93, 229, 39, 1009, 1061, 433, 2393, 15401, 11061, 8117, 40257, 235599, 485963, 1044037, 883041, 149761, 5477651, 6592421, 10478201, 55438133, 39052685, 243820205, 502744721, 478655057, 1174096853},
  {1, 1, 5, 15, 31, 37, 25, 135, 135, 897, 33, 3713, 7663, 8079, 21421, 32813, 84545, 96843, 220619, 171481, 681253, 383315, 7813795, 15699293, 14870269, 19867845, 23375403, 116900683, 62868177, 541056849, 1574715829},
  {1, 1, 5, 7, 17, 49, 43, 89, 411, 731, 1431, 3893, 1635, 7063, 7933, 50957, 52161, 28483, 496325, 858093, 448815, 3686877, 5473143, 15074687, 24054971, 28121369, 81078783, 116876803, 84272305, 901008529, 1760422133},
  {1, 1, 1, 13, 29, 27, 5, 77, 283, 913, 789, 817, 3309, 475, 30693, 51581, 14917, 129353, 62041, 717807, 1429553, 3977313, 2765135, 2191845, 27406393, 66837429, 11006457, 85339743, 169751185, 1037490769, 917196945},
  {1, 1, 3, 1, 19, 21, 67, 77, 423, 551, 5, 1057, 5469, 7859, 549, 35293, 36423, 112453, 309855, 406993, 1928271, 552025, 6634603, 13657939, 31575641, 34510333, 96088457, 248738167, 329099921, 101058257, 1781166995},
  {1, 1, 5, 1, 1, 21, 99, 237, 215, 759, 1505, 1983, 1517, 8923, 27813, 26525, 42305, 9925, 164949, 973777, 1018407, 22265, 8262171, 16242435, 4290813, 18829091, 10802345, 4221287, 138717393, 766839601, 815681941},
  {1, 3, 5, 7, 19, 61, 73, 215, 165, 127, 205, 259, 7755, 15395, 13949, 39487, 39617, 103499, 484679, 511841, 1163461, 2695011, 5357633, 6832995, 10993689, 45302079, 71320255, 114578543, 123962513, 476360947, 53750517},
  {1, 1, 5, 9, 15, 23, 17, 111, 471, 751, 1923, 775, 6901, 13095, 24677, 17261, 79169, 137805, 476379, 340339, 684269, 3835891, 6276435, 4411027, 14870175, 61122555, 83816505, 179224443, 242730225, 103206065, 764022933},
  {1, 1, 7, 1, 25, 5, 63, 141, 461, 687, 1589, 1559, 7719, 11349, 23373, 13477, 128707, 146245, 99909, 699073, 1097115, 2876249, 7794545, 6673499, 14988993, 5118571, 27434163, 185641929, 266450705, 667402161, 1845131831},
  {1, 1, 1, 3, 11, 63, 11, 27, 253, 439, 297, 1315, 829, 3765, 31973, 55653, 49861, 108871, 401871, 51955, 1360103, 2690087, 2716689, 808347, 5533, 20520383, 101879513, 230777, 517340881, 818003185, 853064849},
  {1, 3, 1, 1, 9, 47, 127, 179, 173, 809, 241, 35, 7355, 5049, 17789, 38743, 49605, 77133, 238157, 629995, 554139, 1054287, 1643153, 7248677, 27692677, 30157255, 85159231, 265648757, 205271889, 171185011, 1606681329},
  {1, 3, 1, 11, 19, 63, 93, 1, 205, 977, 303, 3409, 6529, 10927, 5045, 12271, 86277, 93447, 385815, 724243, 722193, 877565, 1681593, 5040341, 16669467, 5275925, 49911357, 54765803, 442560689, 334208083, 1504750289},
  {1, 3, 7, 9, 31, 63, 41, 79, 477, 91, 1801, 3487, 6885, 13341, 2629, 1351, 13059, 37125, 438019, 130331, 1870677, 3139507, 1863289, 6929511, 6774141, 6905843, 116383681, 203091809, 480948945, 786308147, 565687575},
  {1, 1, 3, 7, 15, 59, 9, 101, 459, 247, 549, 2855, 5765, 7785, 4229, 22549, 105735, 253187, 12035, 215335, 1165109, 538505, 662255, 7385699, 21500937, 37116411, 123342609, 50202613, 464651665, 40773297, 929299},
  {1, 1, 7, 3, 13, 59, 71, 123, 93, 517, 1453, 2389, 4429, 5053, 21925, 27621, 34307, 223751, 219665, 641591, 318067, 2927255, 7162689, 7469545, 31456985, 38680385, 506361, 117744873, 313306769, 647009041, 1972424343},
  {1, 1, 5, 3, 19, 21, 77, 53, 81, 879, 1653, 1637, 3667, 2623, 18781, 9837, 125441, 205319, 155143, 420377, 1395201, 3090017, 4456293, 9386427, 27942193, 58821081, 50567559, 99655083, 127451665, 67472529, 69700981},
  {1, 1, 1, 15, 17, 57, 65, 53, 407, 765, 417, 497, 5009, 2175, 19221, 10797, 41861, 91019, 82837, 695173, 994437, 2946897, 3087635, 9971625, 16383357, 10920325, 41645205, 214827579, 206455505, 372160849, 1618102865},
  {1, 3, 3, 7, 31, 13, 5, 203, 263, 17, 119, 1607, 6773, 11195, 15077, 64863, 27015, 128139, 9875, 9105, 808697, 930687, 1133971, 1254845, 29960427, 60717699, 460329, 265750823, 400902065, 717989683, 1010106771},
  {1, 3, 3, 13, 19, 13, 13, 147, 93, 735, 689, 781, 655, 6853, 29677, 13959, 53383, 123265, 60831, 730297, 1483713, 3025191, 6703337, 12982803, 22761029, 56378609, 13813195, 16532849, 120853841, 363116563, 1250723251},
  {1, 1, 1, 1, 1, 25, 63, 159, 493, 987, 71, 1249, 5859, 11717, 317, 57157, 77957, 215173, 255877, 32925, 799163, 3427195, 3051153, 10417703, 9162867, 23100429, 105622655, 262835905, 135769137, 475114449, 933316337},
  {1, 1, 1, 15, 13, 23, 61, 61, 5, 947, 1853, 3331, 467, 8081, 5725, 15669, 6021, 106635, 242569, 696235, 752265, 2409697, 1750129, 8978375, 20520105, 32627023, 18418343, 247402781, 232215441, 554534385, 866126705},
  {1, 1, 3, 9, 19, 61, 65, 189, 95, 309, 283, 1725, 5683, 15463, 27965, 37261, 45191, 213901, 115871, 335769, 1890701, 3180745, 1006043, 1395265, 28642535, 9737193, 43955423, 199749139, 403534705, 850225041, 374843123},
  {1, 1, 7, 5, 9, 33, 35, 75, 475, 831, 1445, 1485, 5047, 9631, 20173, 60909, 95107, 171649, 322197, 788405, 850823, 3732047, 658391, 2088851, 8361545, 33052593, 104264611, 158635307, 308974481, 93283345, 2113699127},
  {1, 1, 3, 15, 11, 23, 59, 87, 433, 221, 685, 3113, 4095, 13819, 15341, 22461, 46727, 93835, 183431, 497579, 1471895, 3423883, 4306909, 16685569, 21117417, 13452509, 57727179, 36034271, 245250449, 140654289, 2115166643},
  {1, 1, 7, 15, 25, 29, 67, 17, 349, 353, 1321, 563, 57, 533, 13909, 63045, 9379, 259947, 85349, 707393, 1552199, 2873285, 2150257, 5308101, 33152189, 1332631, 71845245, 260267577, 348592337, 150968433, 1589000919},
  {1, 3, 3, 3, 5, 43, 109, 217, 15, 185, 1895, 1015, 1831, 10623, 6605, 10863, 85927, 249519, 401833, 725447, 990553, 742869, 5150235, 15512253, 23410299, 2601843, 92047835, 4489219, 339253937, 905575987, 240902835},
  {1, 1, 7, 1, 1, 47, 81, 185, 59, 691, 191, 3709, 1535, 13347, 23661, 51901, 76771, 157989, 459581, 386251, 706357, 3423141, 222623, 5580087, 450291, 6563921, 89272675, 251647991, 270151217, 576588817, 342252855},
  {1, 1, 5, 1, 23, 57, 83, 217, 457, 771, 1877, 2789, 8143, 4797, 8205, 57349, 19041, 92197, 418019, 455517, 2066287, 4089309, 2110693, 6352839, 25574289, 7004105, 27514491, 60781385, 83057169, 332486033, 375930549},
  {1, 1, 3, 7, 23, 35, 79, 49, 227, 205, 1523, 3873, 4843, 10505, 22333, 47701, 2151, 190115, 501115, 707167, 1251187, 1138077, 4956415, 2384425, 1189087, 47671861, 126822343, 251496109, 89969393, 928399953, 1536380787},
  {1, 1, 1, 1, 17, 43, 121, 95, 205, 35, 189, 2061, 1693, 13273, 5701, 31381, 130277, 167653, 263797, 987343, 992733, 765715, 8215241, 4579519, 21717289, 29517537, 134149961, 9193485, 143350673, 882835729, 764119441},
  {1, 1, 1, 15, 31, 49, 83, 249, 433, 497, 1949, 1845, 5215, 5971, 4109, 15165, 49637, 67243, 441787, 726381, 1057999, 3159005, 5467357, 13361077, 6488633, 53674577, 52363851, 246359911, 374685841, 789384209, 1128957361},
  {1, 3, 1, 1, 21, 53, 73, 211, 265, 929, 923, 279, 3621, 9469, 11909, 40807, 46117, 225645, 272113, 655313, 507325, 2059623, 4699341, 15308173, 33550815, 44186739, 133051369, 35401921, 441722097, 16000051, 696010641},
  {1, 3, 7, 7, 1, 57, 13, 45, 467, 705, 371, 1345, 1647, 3411, 5677, 30911, 47331, 242091, 337341, 965957, 2033833, 3481449, 4974663, 4217621, 9272799, 36556325, 46612675, 257388279, 306102513, 638410579, 1326067511},
  {1, 3, 1, 11, 27, 29, 117, 163, 143, 669, 489, 3913, 7891, 9031, 21245, 24623, 52837, 50983, 193343, 532817, 1646521, 1557175, 1213435, 2303793, 551221, 12564317, 44467415, 249806787, 137991761, 938493651, 770773617},
  {1, 3, 7, 15, 27, 15, 77, 217, 107, 839, 1517, 1543, 357, 10365, 16933, 16551, 100003, 149603, 141799, 273107, 217729, 181829, 2645759, 8613123, 18771681, 37594875, 132948785, 102350785, 486519569, 182691507, 693307415},
  {1, 1, 1, 5, 31, 17, 107, 245, 345, 939, 1453, 3645, 6865, 16173, 11637, 6981, 93157, 176225, 325371, 808677, 1131255, 3419729, 116373, 16393311, 26542505, 56557617, 111742597, 53546553, 528826769, 892962321, 1955717457},
  {1, 3, 5, 5, 9, 61, 43, 97, 453, 917, 945, 2143, 5473, 5611, 10677, 46591, 125601, 186089, 490557, 187209, 965551, 551029, 2234633, 11626737, 29752261, 35340459, 89381829, 202271223, 260876113, 453614515, 792938837},
  {1, 1, 5, 11, 3, 33, 71, 97, 137, 549, 1605, 3839, 4883, 2677, 13565, 61893, 18017, 210927, 294455, 1024813, 1552235, 2367813, 7064053, 9964225, 29408577, 29990283, 79306023, 68827561, 317616913, 720820657, 188414837},
  {1, 3, 1, 11, 29, 23, 85, 47, 225, 633, 1613, 1297, 1415, 15813, 9133, 50151, 37285, 130407, 81721, 814363, 2083649, 1677651, 7899605, 6044261, 8269353, 44918101, 68107347, 62720673, 347805585, 631839315, 1073251377},
  {1, 1, 3, 3, 9, 19, 57, 107, 79, 449, 1951, 753, 6317, 10377, 31397, 26773, 82855, 43303, 247973, 936255, 1762621, 569671, 6614475, 11254861, 21638083, 54152661, 121718449, 70544813, 345204017, 179770961, 1650656275},
  {1, 1, 1, 5, 21, 3, 39, 187, 299, 517, 1313, 741, 7259, 4197, 12829, 5541, 128549, 193505, 63025, 763447, 245331, 2503319, 3995415, 12008521, 24399853, 28131345, 44227007, 135612561, 527120849, 867523089, 764268785},
  {1, 1, 5, 13, 1, 39, 39, 41, 381, 123, 1257, 3185, 493, 3723, 4101, 63965, 75169, 118569, 295925, 98355, 950275, 2654229, 6258753, 229631, 14976509, 60000637, 107845481, 109285615, 383031505, 268820177, 148335509},
  {1, 3, 7, 7, 3, 37, 15, 161, 129, 169, 555, 3605, 4287, 15831, 5165, 64047, 78883, 90219, 148095, 551961, 697443, 3722453, 3183709, 1992269, 17451151, 10688849, 75413555, 190112163, 8923121, 924357651, 1390977591},
  {1, 3, 7, 15, 15, 23, 81, 163, 257, 791, 505, 1903, 2703, 11919, 23181, 51887, 114403, 71907, 176883, 966795, 938125, 1565087, 3343141, 3950331, 2641309, 17617811, 68220811, 131507859, 390260945, 578517555, 1548327351},
  {1, 3, 7, 7, 27, 63, 17, 147, 111, 851, 1533, 1365, 5359, 3315, 8781, 36991, 128419, 179947, 127847, 175235, 299165, 3208975, 7520331, 6275839, 19416865, 55460025, 43963899, 181527495, 44303121, 600078483, 1850593463},
  {1, 3, 7, 1, 15, 5, 61, 143, 385, 261, 1019, 1705, 1737, 14485, 11765, 45159, 32803, 65581, 347571, 107425, 117217, 464379, 751061, 779865, 22521567, 35958685, 54929733, 131739025, 88486129, 67016403, 1161246039},
  {1, 3, 5, 5, 25, 17, 49, 229, 431, 567, 1613, 3431, 2139, 2981, 2845, 60935, 115425, 139625, 18093, 968101, 2054837, 1595009, 1878603, 4626947, 10743377, 9809179, 64976719, 22931801, 114078737, 472421555, 402160629},
  {1, 3, 5, 11, 17, 57, 71, 241, 31, 1007, 1695, 2965, 149, 14125, 15877, 49991, 105633, 227495, 14245, 86709, 380643, 1776181, 3009507, 15604363, 21142083, 1070665, 111134281, 261210905, 115835697, 639841683, 511384469},
  {1, 1, 3, 11, 7, 49, 39, 101, 5, 501, 1491, 3061, 225, 12255, 14549, 62605, 87207, 30255, 438763, 56509, 1727387, 620097, 8018809, 8025857, 7579751, 48402753, 68215181, 246913323, 124289393, 602083601, 48856531},
  {1, 3, 5, 7, 17, 35, 37, 97, 415, 15, 1349, 997, 2949, 4511, 4413, 31319, 92169, 164019, 414349, 860839, 536457, 2997909, 3115651, 6088211, 16661473, 25571249, 30241945, 128520275, 73425, 593199059, 928625109},
  {1, 3, 1, 5, 25, 35, 99, 183, 161, 59, 1363, 515, 3767, 3641, 11461, 19295, 126173, 18769, 253845, 796047, 1176255, 2016291, 1233957, 10735439, 25103071, 21029911, 64830899, 181052061, 369730737, 82940595, 1221422321},
  {1, 1, 7, 15, 7, 15, 127, 137, 281, 67, 139, 2315, 3517, 13371, 7341, 57093, 73067, 44915, 413571, 421019, 172011, 1439997, 2734189, 11209455, 12990679, 22168543, 109302681, 63002623, 431068721, 872056369, 200739927},
  {1, 1, 5, 15, 23, 49, 19, 79, 425, 805, 1035, 429, 7707, 14195, 23301, 32533, 13705, 105075, 471195, 784309, 1914855, 870467, 1513885, 6605585, 21458535, 60872449, 133147919, 110981711, 470581553, 920368209, 1425885493},
  {1, 3, 5, 3, 21, 25, 123, 11, 425, 475, 961, 2995, 7405, 5449, 8237, 55375, 92409, 79447, 523145, 764045, 382695, 3289207, 2330717, 10596671, 13456141, 51256231, 10836769, 115897789, 509945745, 204600755, 1675981141},
  {1, 1, 7, 1, 21, 1, 75, 231, 451, 595, 719, 2369, 5907, 1227, 7493, 3253, 20555, 183293, 437137, 44205, 598759, 714203, 5007927, 16447239, 25497547, 18037221, 44876039, 167440391, 207188337, 994402193, 650830519},
  {1, 1, 1, 9, 21, 57, 45, 255, 19, 79, 481, 3363, 3451, 8399, 10101, 54629, 6189, 45749, 493065, 276997, 400929, 3202595, 8136255, 12407627, 1906309, 13029751, 113961911, 122914235, 322283409, 999327153, 1324374577},
  {1, 1, 7, 13, 31, 49, 95, 69, 483, 427, 37, 4047, 7057, 9111, 26301, 57173, 92107, 63025, 452123, 55853, 1568315, 2642601, 6176487, 9816919, 13531761, 7250987, 41646701, 205332787, 456613585, 264099697, 1196161687},
  {1, 3, 3, 11, 3, 61, 87, 79, 499, 91, 771, 1987, 2017, 3381, 23581, 64607, 2655, 120351, 261127, 482361, 1342515, 982035, 7927159, 5481487, 12527559, 40853959, 4028933, 155992977, 314859953, 235200435, 218993043},
  {1, 3, 1, 7, 5, 57, 1, 121, 155, 225, 501, 477, 6555, 9863, 11781, 41287, 13997, 49587, 82953, 48157, 1195069, 2129093, 8119559, 8431261, 19221569, 21927185, 68238583, 4613195, 297321937, 279262803, 1078435633},
  {1, 3, 7, 11, 27, 49, 83, 213, 61, 283, 1599, 3205, 2525, 8553, 32477, 27855, 62651, 23903, 353551, 667413, 598807, 3837817, 1221193, 15755975, 10931283, 46754129, 4517673, 149646389, 79566705, 254207187, 566109783},
  {1, 1, 1, 9, 9, 49, 3, 51, 141, 33, 301, 2167, 587, 15067, 16917, 31413, 120077, 141301, 361749, 143117, 430975, 2401071, 3439769, 1613541, 10905521, 58188459, 23650855, 64955775, 201530705, 838063089, 913149489},
  {1, 1, 1, 11, 7, 55, 99, 81, 191, 553, 953, 3753, 6731, 1093, 485, 21453, 111677, 193431, 88347, 444675, 1338167, 4171013, 5817851, 16313861, 32315645, 16708069, 53327127, 240231097, 284602641, 864476945, 1872440881},
  {1, 1, 3, 3, 11, 59, 57, 235, 297, 197, 853, 1411, 3799, 7527, 18885, 43685, 32175, 240959, 94495, 268095, 1639229, 2157423, 1186213, 5027857, 16070713, 63187567, 95892539, 190799507, 301443793, 524214961, 1373605363},
  {1, 3, 5, 3, 7, 7, 5, 201, 393, 95, 91, 3273, 6285, 10661, 2877, 54335, 103961, 7255, 113115, 776659, 1986449, 3820445, 4395541, 14426211, 7103271, 23225901, 75456585, 157041769, 363734833, 920083, 1729002837},
  {1, 1, 5, 7, 17, 57, 87, 3, 413, 915, 659, 369, 3593, 14429, 15853, 9469, 11929, 216539, 513757, 14301, 1835451, 529887, 3732697, 8160503, 7243599, 5015077, 47811917, 142581361, 392811185, 509472657, 1139660565},
  {1, 3, 7, 1, 31, 31, 45, 115, 417, 427, 745, 4087, 953, 1119, 30877, 32503, 83787, 233077, 300875, 823507, 20073, 1053223, 466909, 14113471, 19123429, 33682083, 7724565, 116087995, 32033041, 89331187, 508078103},
  {1, 3, 7, 3, 29, 43, 45, 221, 41, 641, 451, 173, 2999, 12103, 27509, 59495, 74667, 169207, 74441, 389359, 536177, 635497, 5991381, 8174829, 22097951, 46972657, 57547619, 95042235, 415106225, 530736211, 224125431},
  {1, 1, 3, 11, 25, 57, 117, 201, 135, 787, 1525, 3879, 3247, 8907, 17093, 46357, 107919, 11575, 180941, 645597, 1165881, 2953749, 990155, 3896527, 5779121, 44375939, 68973779, 93291663, 278768081, 560659441, 689172851},
  {1, 1, 7, 11, 3, 35, 69, 157, 331, 615, 573, 2169, 3575, 289, 28245, 40317, 63131, 151959, 294599, 326503, 1687585, 1884409, 6504631, 1524427, 15320937, 62245549, 13521867, 184348701, 440707921, 221128721, 184702711},
  {1, 3, 3, 13, 15, 51, 67, 127, 265, 495, 103, 3145, 2685, 15919, 16509, 41495, 94415, 10361, 403019, 134383, 127079, 348379, 2036141, 10220155, 28864587, 39442573, 38384841, 81878275, 164421617, 793200147, 304361043},
  {1, 3, 5, 11, 31, 27, 65, 57, 153, 465, 1163, 467, 4103, 4713, 11845, 28511, 9753, 22559, 335043, 510831, 1854581, 994525, 8093173, 10978381, 12234999, 43670287, 99455347, 191348269, 137929265, 793678771, 831740661},
  {1, 3, 7, 3, 23, 31, 9, 51, 239, 417, 1597, 229, 2865, 15199, 5293, 19511, 13451, 18295, 173123, 999387, 275325, 3314951, 1145731, 12133653, 29875273, 63622377, 12568445, 145644275, 46257233, 185613267, 886085015},
  {1, 3, 5, 3, 11, 45, 123, 217, 31, 765, 1009, 2001, 3645, 9407, 5181, 58583, 94921, 189815, 35159, 375289, 715103, 3181509, 5027947, 9919313, 8686901, 4665613, 119802401, 175900899, 108796305, 192705683, 2085478229},
  {1, 3, 3, 9, 5, 23, 117, 83, 237, 1017, 251, 1187, 2631, 5151, 16165, 22823, 5135, 41309, 378593, 204219, 1892857, 222295, 3708273, 16454989, 13139175, 565567, 99396947, 69715451, 522786353, 443133107, 2057439475},
  {1, 1, 1, 7, 23, 55, 97, 141, 501, 305, 467, 4061, 2369, 15973, 17357, 31741, 77245, 230331, 71595, 634003, 1021621, 2714169, 2093785, 13493021, 1450783, 7383489, 40975605, 142875833, 42740657, 179574609, 1220926609},
  {1, 1, 7, 5, 31, 51, 125, 191, 219, 495, 37, 3337, 813, 241, 2845, 28685, 129243, 262009, 38715, 202415, 981241, 3149499, 3311463, 14085547, 4930433, 39720381, 47366417, 222022893, 338333137, 487886097, 807736663},
  {1, 3, 1, 1, 11, 39, 93, 109, 285, 147, 1297, 737, 4051, 7223, 37, 615, 56077, 173269, 119783, 252347, 109449, 3424793, 4883009, 14874207, 28293565, 19867317, 48312063, 139870507, 113285009, 735907987, 659465393},
  {1, 3, 1, 15, 13, 17, 57, 171, 463, 163, 609, 1681, 7583, 9231, 20373, 31719, 41741, 189403, 447457, 169141, 365189, 2065367, 7774227, 9953527, 27092789, 28094069, 68107539, 115897107, 380564369, 493817491, 1517619313},
  {1, 3, 1, 1, 23, 5, 51, 5, 205, 415, 419, 989, 4239, 10943, 3125, 53223, 58253, 49365, 521723, 480409, 1094295, 819993, 4534777, 14471459, 1700015, 30474105, 67392267, 237978627, 411801777, 685169491, 734414705},
  {1, 1, 3, 15, 3, 13, 65, 145, 387, 59, 395, 1067, 4143, 5649, 8069, 8541, 90367, 233395, 231479, 899737, 1556901, 3718573, 1915071, 8591431, 28633663, 40161135, 55955531, 78269621, 38752305, 305234737, 1210326899},
  {1, 3, 1, 13, 9, 59, 121, 127, 95, 71, 1541, 1423, 1753, 8041, 17661, 64207, 52317, 58105, 339237, 843671, 170437, 1887147, 6692819, 8775011, 19083761, 57702715, 18525285, 90376045, 490569681, 589666163, 97037841},
  {1, 1, 3, 15, 7, 5, 69, 167, 181, 991, 1189, 4017, 5935, 6669, 28501, 12653, 106143, 224883, 234035, 669201, 652849, 3447323, 828953, 10132955, 14911929, 55348661, 122330867, 4817073, 373854673, 551070097, 1876460915},
  {1, 3, 5, 7, 23, 41, 53, 21, 47, 261, 1231, 2011, 133, 2247, 25581, 31831, 57193, 23571, 222571, 405261, 612705, 2362329, 4823331, 15002505, 16875435, 15477319, 133199041, 37356899, 127557233, 144093747, 503997141},
  {1, 1, 1, 5, 17, 47, 77, 19, 331, 609, 1893, 3965, 3123, 9093, 26501, 27053, 49693, 123769, 25133, 1043715, 1237777, 2422103, 7730615, 16440725, 9866977, 3493169, 49465727, 266295593, 16742353, 430570065, 877734833},
  {1, 3, 1, 3, 9, 39, 103, 231, 249, 75, 373, 107, 1823, 10801, 30405, 18719, 71549, 240695, 378789, 895795, 188203, 1569059, 7636397, 9428159, 32389689, 51573535, 14448355, 29575333, 417471697, 263328307, 780078193},
  {1, 3, 3, 7, 1, 51, 35, 111, 137, 879, 1221, 225, 4285, 2287, 16541, 4471, 111535, 16147, 192869, 588071, 1865039, 1365947, 1991789, 6025179, 1599177, 19718277, 79933601, 237323667, 487629777, 911450771, 1234256979},
  {1, 1, 7, 9, 23, 17, 75, 245, 409, 163, 395, 3731, 7111, 6845, 12709, 9581, 48091, 142517, 170611, 217085, 205007, 1033401, 6526253, 13688607, 31861655, 1238311, 129610115, 132821369, 244551217, 44904881, 645263863},
  {1, 1, 3, 13, 29, 47, 75, 153, 497, 621, 1691, 3187, 2125, 10533, 22349, 58077, 106687, 217201, 489449, 155891, 673927, 3268797, 4284821, 8692161, 15344903, 46460927, 132278329, 198858497, 121528625, 473060273, 2084785747},
  {1, 1, 7, 7, 9, 7, 55, 159, 255, 417, 1335, 643, 3843, 3733, 30197, 2797, 39131, 156347, 343149, 713171, 912059, 1858203, 530219, 8146805, 28147299, 23731119, 52825735, 225587857, 280111345, 634559409, 1788824759},
  {1, 3, 3, 1, 21, 41, 7, 21, 5, 679, 1655, 95, 5699, 5785, 16469, 19807, 105919, 118389, 257521, 680677, 1433803, 853545, 2995041, 5515579, 31144027, 193947, 50685719, 25979469, 150956529, 1040569203, 553147315},
  {1, 1, 1, 13, 19, 7, 85, 7, 195, 357, 1097, 2893, 2913, 9635, 20621, 53477, 80333, 244625, 438063, 845387, 66689, 4137395, 6130431, 7224577, 6014861, 37405009, 34588285, 210590383, 364620561, 479488337, 2045014161},
  {1, 1, 5, 9, 25, 33, 41, 155, 39, 655, 1993, 3117, 3639, 7977, 25237, 5885, 24057, 19445, 124917, 171997, 1568693, 1085223, 6369275, 13823835, 5360893, 49750137, 40925627, 86077573, 340030737, 206894417, 1899766005},
  {1, 1, 1, 13, 3, 63, 121, 247, 151, 673, 609, 285, 2299, 7663, 8677, 40197, 105997, 172049, 449599, 392307, 2047213, 3722531, 964315, 3510757, 14938805, 22396465, 76319527, 100637219, 269332881, 222578257, 538794801},
  {1, 3, 7, 11, 17, 13, 49, 253, 245, 21, 273, 993, 911, 863, 3157, 12519, 95563, 7199, 349861, 1047753, 1683581, 635089, 7598569, 15233625, 21599197, 33906349, 108030931, 161363171, 443425681, 642934931, 130756215},
  {1, 1, 5, 5, 23, 1, 121, 95, 225, 9, 1237, 1183, 6461, 559, 20461, 38453, 101865, 255065, 521787, 911821, 806205, 3284819, 8133789, 6677517, 33119961, 43745315, 110271121, 68780171, 156432081, 444720497, 2042501973},
  {1, 3, 7, 13, 3, 7, 121, 151, 233, 561, 281, 3583, 897, 1767, 7965, 23415, 111467, 202585, 302135, 463835, 1458509, 1550179, 7789669, 7059445, 15051445, 56368355, 28042077, 156416291, 233676305, 623644531, 1238703255},
  {1, 1, 7, 7, 9, 47, 107, 41, 25, 569, 1697, 2299, 6231, 12209, 1637, 38301, 83131, 248571, 439149, 818299, 1247591, 2094861, 1195645, 7736533, 11972509, 1245671, 123412075, 53545637, 304685713, 685193009, 958303735},
  {1, 3, 7, 7, 27, 43, 59, 37, 31, 51, 503, 149, 4043, 11847, 23973, 20967, 24395, 129043, 433071, 850399, 1753407, 176385, 576059, 16305263, 17999283, 49900113, 46721823, 213158715, 289184241, 552153811, 67244343},
  {1, 3, 3, 11, 5, 1, 119, 181, 47, 641, 685, 4017, 637, 16251, 20141, 1303, 71359, 255631, 428177, 167605, 1961403, 42521, 8026747, 13364333, 21540921, 10640461, 121697857, 36182359, 251143761, 628672147, 257495379},
  {1, 3, 3, 7, 11, 1, 101, 7, 239, 747, 307, 1721, 5979, 4367, 25493, 28023, 23455, 12675, 451743, 1004933, 1138673, 1888987, 193267, 2791007, 32193335, 19492333, 38534735, 62423795, 168885937, 690065939, 2068044339},
  {1, 3, 5, 7, 1, 63, 19, 151, 469, 333, 1587, 2453, 897, 4711, 23965, 63687, 76121, 208259, 333453, 928907, 640943, 1102563, 3211233, 15297289, 11573343, 4571273, 84269221, 190866987, 123851441, 801384915, 183794837},
  {1, 3, 1, 5, 11, 61, 21, 253, 91, 993, 1347, 1993, 5607, 13031, 11237, 1591, 100573, 27521, 49815, 510369, 460177, 1319121, 712807, 7348285, 17424711, 36613909, 75833699, 191196507, 362781873, 1005989651, 325460209},
  {1, 3, 5, 5, 1, 39, 65, 71, 189, 389, 1437, 1055, 6439, 3989, 13493, 14751, 129289, 227201, 165517, 637083, 752109, 3609427, 7554561, 9301265, 2160849, 17390691, 115034107, 141163137, 213966929, 715706739, 1564035829},
  {1, 1, 3, 3, 19, 15, 93, 3, 339, 165, 1675, 3953, 2145, 12113, 19389, 56925, 97647, 106639, 357271, 262555, 1706377, 904999, 3331039, 5843137, 17328991, 25396221, 85102053, 75938125, 254462769, 964438161, 992794259},
  {1, 1, 3, 13, 13, 45, 5, 175, 211, 993, 705, 2761, 3023, 13633, 5733, 7917, 97903, 3969, 270217, 472193, 849369, 583155, 8050735, 7760373, 23296373, 34123861, 40811587, 231280621, 416236945, 776047377, 358921843},
  {1, 1, 3, 1, 19, 39, 121, 29, 287, 87, 281, 3491, 7107, 13007, 32005, 24725, 71135, 256525, 422423, 659003, 1262141, 4139545, 472003, 8017387, 14064525, 17550855, 89512735, 260896395, 480654609, 1025950641, 1589596595},
  {1, 1, 7, 1, 29, 49, 103, 187, 39, 923, 51, 1533, 3249, 4399, 24989, 58213, 49051, 99341, 95241, 500749, 1614875, 662679, 1236019, 15541839, 10082679, 54118057, 131450021, 202013507, 489013425, 1038364241, 617931927},
  {1, 1, 5, 5, 5, 43, 25, 107, 453, 955, 115, 57, 4589, 14573, 11981, 35789, 124169, 86025, 503833, 348183, 1929253, 1298671, 5705977, 13710047, 30618687, 37171453, 117194233, 62393233, 82319537, 680007441, 1503160661},
  {1, 1, 3, 5, 21, 45, 103, 99, 183, 987, 1207, 1697, 8033, 13703, 5501, 16261, 103775, 249097, 284945, 145185, 670491, 3227007, 6011715, 11908719, 18524275, 55645045, 115767397, 75575739, 121703665, 52925329, 861673875},
  {1, 1, 5, 7, 11, 23, 9, 17, 261, 749, 1957, 935, 6283, 8625, 23749, 21357, 40361, 26379, 261911, 224547, 1721629, 141077, 3745433, 8021473, 24113209, 8706363, 121147511, 127704837, 185202897, 702136049, 1471991861},
  {1, 1, 1, 9, 9, 51, 69, 225, 265, 323, 1161, 2993, 7305, 2249, 19133, 33277, 17581, 33029, 130821, 480031, 1350441, 4131685, 2760325, 8517615, 11608229, 15477237, 28982197, 187737925, 325632529, 574767505, 1361270545},
  {1, 3, 1, 9, 23, 19, 57, 205, 503, 489, 1499, 3277, 817, 11931, 7837, 31671, 37949, 26893, 416523, 806207, 1028397, 990009, 6757515, 4142101, 899439, 8620497, 104108053, 179211383, 39311665, 682385747, 426611089},
  {1, 3, 3, 5, 1, 7, 49, 1, 313, 123, 643, 2027, 1469, 3585, 16349, 58783, 42095, 148033, 294869, 9163, 1674461, 3785301, 7322245, 14267191, 840367, 43325583, 83155369, 67170165, 434742961, 903544883, 1698620499},
  {1, 3, 7, 11, 27, 47, 95, 111, 27, 213, 465, 3693, 3661, 7531, 21901, 49239, 88571, 38095, 363231, 884603, 388523, 89675, 2298463, 820433, 9907077, 25194913, 81530337, 31722519, 213136529, 446993747, 660967511},
  {1, 1, 7, 9, 3, 37, 115, 189, 31, 613, 1393, 1229, 4767, 12425, 6853, 20813, 117995, 213701, 25687, 920825, 1492727, 115953, 4066747, 4298441, 9371813, 47513953, 90244659, 167590645, 357507985, 163120465, 754221431},
  {1, 1, 3, 3, 25, 17, 99, 47, 161, 931, 959, 1293, 7095, 8325, 23557, 54973, 26511, 129231, 251741, 802501, 681951, 2859363, 4081893, 3971143, 13388931, 56593737, 33982739, 73112073, 130211441, 1015174737, 1038058739},
  {1, 1, 1, 7, 23, 9, 11, 51, 205, 419, 479, 1497, 2493, 13921, 22285, 20413, 109229, 29003, 379995, 406877, 256559, 1164015, 3132137, 9089783, 1782963, 25949565, 4701977, 267271485, 50482801, 720041233, 861062225},
  {1, 3, 1, 9, 17, 29, 51, 79, 159, 435, 477, 413, 3815, 5589, 16101, 16127, 100749, 160845, 148685, 67697, 343679, 3082867, 4482811, 16698711, 32743321, 59323113, 63207115, 138596489, 349153361, 920721491, 2084168177},
  {1, 3, 7, 5, 7, 23, 99, 43, 169, 665, 403, 1163, 4337, 1335, 20333, 53031, 1371, 3521, 192835, 1006843, 919575, 22671, 7870557, 9422957, 13859759, 6530663, 2056885, 96012611, 83577521, 547955251, 1287024023},
  {1, 3, 1, 5, 25, 27, 125, 249, 421, 267, 1259, 4089, 59, 9377, 31317, 3727, 6829, 117569, 251845, 793415, 295025, 1409181, 1147081, 8093767, 11568119, 47087677, 80184463, 7635549, 138144817, 154885907, 866348081},
  {1, 3, 3, 1, 27, 37, 91, 17, 123, 597, 1749, 3449, 6503, 11043, 31557, 17127, 80831, 73029, 394959, 719865, 1041535, 1946781, 7064831, 10624137, 9987441, 23372469, 12036699, 267012143, 132098001, 48235539, 572536563},
  {1, 3, 7, 7, 23, 41, 19, 245, 109, 569, 547, 1917, 7943, 2697, 32517, 15, 47467, 36675, 340435, 635469, 1298743, 3199785, 4533161, 16531893, 31375615, 63000497, 12049283, 181435429, 199697073, 166285139, 535092983},
  {1, 3, 7, 7, 9, 1, 123, 105, 329, 435, 2013, 2745, 347, 11045, 15125, 54911, 120811, 63587, 350637, 555909, 455815, 866373, 7780813, 12101999, 7567889, 64968157, 133584407, 68215689, 84664657, 319379219, 1523594295},
  {1, 1, 1, 13, 29, 53, 51, 67, 105, 89, 1887, 3543, 963, 8159, 1893, 4085, 64957, 239969, 52465, 328617, 1090959, 1425631, 1654829, 10830829, 30328003, 64800539, 37736087, 141608667, 216665201, 345174513, 1188808881},
  {1, 1, 5, 3, 5, 27, 41, 67, 67, 883, 973, 1785, 901, 14969, 1901, 18429, 29961, 186287, 386617, 20639, 1574533, 1082407, 2192607, 7030039, 6919737, 38034061, 113120969, 239222549, 187315537, 346207761, 1492789717},
  {1, 3, 3, 13, 17, 11, 117, 115, 163, 939, 79, 641, 4365, 2267, 12205, 63735, 54943, 76969, 41573, 526983, 1487289, 2830903, 1879823, 9977967, 32850107, 35440917, 77388937, 125914183, 289138033, 553689491, 1526730067},
  {1, 1, 3, 3, 9, 5, 41, 123, 149, 9, 1533, 3939, 5995, 12701, 325, 24509, 112175, 80239, 495533, 13489, 1866677, 2396615, 5597689, 9914637, 31702609, 35983695, 109222967, 231778521, 189766225, 190665905, 134511155},
  {1, 1, 1, 15, 31, 1, 101, 229, 191, 965, 61, 2671, 4177, 15779, 23117, 46677, 103901, 3107, 132723, 154517, 986305, 1794489, 1059651, 5369625, 22484297, 66861731, 100059053, 211248023, 461080913, 968393073, 1688760209},
  {1, 1, 5, 15, 1, 25, 49, 185, 33, 697, 1043, 2639, 7819, 3171, 8901, 46021, 21977, 67555, 134909, 53293, 1658989, 3592309, 3305933, 14266869, 24605311, 19330947, 19373103, 189108711, 305237425, 761431153, 1575183413},
  {1, 3, 5, 13, 19, 9, 111, 49, 47, 847, 1865, 717, 5287, 13417, 9477, 56047, 79625, 51561, 148671, 508981, 121899, 588845, 2520187, 2432659, 12245853, 46208873, 75221467, 254045429, 177159185, 212986707, 714923509},
  {1, 3, 7, 11, 5, 61, 63, 111, 171, 735, 2003, 73, 5701, 647, 12557, 6215, 30203, 144175, 138849, 164105, 1928467, 986963, 7214559, 14490939, 24588967, 49684109, 7142161, 59615579, 5540529, 357241107, 1256278999},
  {1, 3, 1, 11, 1, 49, 121, 79, 431, 671, 1241, 1161, 2057, 263, 14493, 9063, 87933, 49071, 251517, 624949, 198981, 1443747, 8363427, 5400395, 686909, 59417517, 116751221, 62272779, 93202961, 872308499, 1097043729},
  {1, 3, 3, 1, 1, 23, 75, 15, 117, 641, 313, 1525, 2041, 1409, 381, 16959, 78671, 148389, 436917, 832299, 616231, 2783035, 1526465, 16634309, 1513333, 40669753, 111494645, 67596349, 153908241, 551290323, 1879908115},
  {1, 3, 5, 11, 15, 57, 13, 67, 139, 131, 1339, 2419, 7945, 11877, 1773, 63759, 32489, 98479, 141603, 935197, 1683209, 1161087, 3008279, 15668487, 25244975, 63615879, 29964957, 212150289, 529486385, 454537907, 1548128533},
  {1, 3, 1, 1, 19, 39, 97, 83, 297, 595, 1611, 5, 4753, 3435, 7933, 3287, 130013, 135525, 352047, 459595, 1892229, 4160567, 4643957, 1668791, 2211031, 24774753, 1967237, 37630535, 45234481, 10117331, 1423961233},
  {1, 3, 1, 9, 7, 49, 125, 101, 383, 717, 63, 2295, 3873, 13461, 27309, 44175, 119085, 84973, 443643, 889597, 700553, 4108537, 5493219, 14946961, 1275275, 51247195, 113435349, 28888721, 29529457, 269417715, 832518289},
  {1, 1, 3, 3, 15, 29, 89, 77, 269, 689, 229, 1207, 7311, 8663, 5813, 2741, 2367, 197679, 459051, 785257, 1491869, 3134177, 4244129, 673773, 18106713, 39507963, 89450995, 102797379, 347405521, 429732337, 96382067},
  {1, 1, 1, 1, 1, 61, 25, 255, 203, 233, 271, 987, 2277, 8735, 9309, 30533, 71037, 132717, 460461, 825425, 1356789, 3326691, 1619879, 4604221, 22003899, 54708423, 107826865, 234605867, 350450545, 373666865, 127035089},
  {1, 1, 5, 7, 21, 27, 63, 79, 337, 133, 1453, 3633, 6157, 15875, 20621, 57493, 93849, 102123, 391209, 640591, 1794739, 140267, 142677, 15152289, 2801201, 61949453, 85346353, 226091727, 71709009, 219033489, 1063454293},
  {1, 3, 1, 7, 7, 55, 31, 81, 203, 709, 1743, 1677, 4247, 11411, 16293, 55175, 68125, 241315, 403131, 597827, 1814683, 3113941, 1217839, 13833689, 26717899, 14013793, 50995427, 69923199, 163774577, 589285715, 2077249521},
  {1, 1, 3, 3, 29, 51, 37, 17, 487, 325, 1393, 1433, 3467, 2851, 8085, 64805, 5535, 75311, 326585, 616263, 802473, 3779989, 6671307, 5539273, 3686341, 39043157, 100097927, 223101423, 204977041, 856165905, 1380578867},
  {1, 1, 7, 9, 3, 41, 99, 177, 241, 869, 739, 2729, 4585, 14801, 30013, 35005, 53771, 81509, 483191, 325333, 296391, 4051757, 5254997, 5297177, 5203151, 35054613, 33879597, 106392717, 176173745, 58748433, 1988126743},
  {1, 1, 3, 1, 9, 43, 97, 65, 99, 295, 1693, 2083, 3241, 4073, 17741, 15805, 14031, 240365, 440941, 573655, 712109, 1167109, 6755295, 14899403, 20724665, 50680519, 56592645, 141461037, 280143185, 116692913, 302826771},
  {1, 1, 1, 9, 5, 39, 67, 119, 235, 543, 795, 2773, 3195, 6273, 28757, 20909, 93453, 47013, 484457, 792427, 548991, 778403, 2843071, 15852347, 15607471, 7201873, 22130383, 93457325, 79580721, 754619601, 2143736881},
  {1, 1, 5, 5, 21, 41, 89, 1, 85, 81, 57, 2951, 1531, 10101, 17061, 36509, 127401, 259881, 220905, 186229, 769861, 1395181, 6154393, 14477437, 15817733, 55882763, 8879351, 130085697, 385079057, 261469937, 130323253},
  {1, 1, 1, 7, 3, 35, 127, 69, 39, 265, 1643, 2973, 267, 12365, 2203, 14647, 72303, 146649, 90545, 1010497, 732925, 2414015, 8008537, 503739, 475421, 27902375, 38881861, 66186439, 92612885, 763867933, 1892983893},
  {1, 3, 1, 1, 21, 57, 99, 205, 119, 477, 1771, 1989, 2761, 12573, 26171, 63605, 93419, 146903, 448443, 543567, 1346301, 1610039, 6911513, 1935855, 11905333, 6951343, 28734871, 140219955, 78239325, 901667023, 253188469},
  {1, 1, 3, 13, 1, 59, 93, 125, 279, 935, 1877, 2061, 4845, 7835, 29495, 42095, 72413, 140215, 135535, 32997, 373655, 1295699, 2008497, 13631173, 31050975, 61391795, 107926411, 61824909, 246369837, 784956981, 1573468343},
  {1, 1, 7, 9, 7, 45, 69, 99, 273, 35, 1579, 2137, 7175, 6999, 2223, 51039, 33977, 8059, 177393, 805839, 1163739, 1130965, 2235, 1145685, 18807425, 63945563, 67285169, 60066869, 536265413, 147206357, 473369363},
  {1, 1, 7, 7, 29, 21, 127, 91, 9, 383, 787, 1783, 601, 5047, 25199, 45535, 76985, 1141, 259575, 535291, 774793, 45129, 5829019, 9843529, 22860417, 2700533, 124331699, 44997585, 45193933, 466710341, 449669939},
  {1, 1, 7, 13, 7, 29, 35, 219, 43, 581, 2043, 2211, 6169, 12173, 19995, 39223, 10089, 58847, 393913, 600175, 1738749, 2541601, 6023529, 1849751, 12303317, 42420745, 100258571, 231079067, 349326381, 685175405, 469119155},
  {1, 3, 5, 13, 29, 29, 39, 63, 411, 645, 415, 2383, 1989, 11411, 28327, 14541, 39711, 229811, 335099, 148331, 993137, 2762077, 6591905, 1822279, 15998097, 37795309, 93843871, 54546221, 339084925, 1022524599, 1088412145},
  {1, 1, 7, 9, 15, 9, 87, 95, 321, 709, 543, 3831, 2453, 4167, 19215, 47775, 13241, 31995, 446073, 1047163, 545569, 2463133, 3553787, 2501043, 29538553, 48887685, 31553311, 138246905, 194387069, 750922725, 1946563059},
  {1, 3, 1, 5, 31, 25, 5, 85, 239, 487, 1613, 3937, 4661, 3535, 2847, 29629, 38651, 89459, 35961, 327787, 1506643, 3594099, 3393161, 1771381, 7263015, 7371695, 108232427, 2577305, 394148781, 280366695, 1733439157},
  {1, 3, 5, 11, 27, 41, 3, 201, 39, 91, 1997, 237, 5639, 14703, 2143, 49469, 4607, 69493, 107121, 823627, 170773, 1573731, 7928417, 14903577, 17598847, 51675283, 9733281, 139308205, 419549829, 192803511, 2025097745},
  {1, 1, 7, 3, 27, 49, 87, 71, 473, 247, 1007, 47, 475, 5413, 24523, 41239, 41897, 26321, 174137, 210883, 1140561, 39269, 763539, 10388305, 23168973, 44139829, 65377329, 199732167, 260608197, 165113629, 689177363},
  {1, 3, 7, 15, 9, 57, 81, 149, 287, 333, 1911, 3417, 1081, 8995, 18887, 23821, 71581, 187829, 478435, 260223, 668719, 1530955, 6925321, 1033951, 14647625, 13930315, 114655791, 220695549, 99535037, 489582583, 1307271923},
  {1, 1, 5, 1, 3, 63, 43, 151, 97, 431, 961, 1019, 5153, 2407, 17039, 46815, 75259, 240951, 404013, 181669, 1216673, 3366421, 2644619, 7418233, 30642803, 33355357, 132344027, 140985745, 387060589, 781084229, 1990892977},
  {1, 1, 5, 5, 27, 21, 127, 161, 507, 311, 129, 3489, 1133, 3241, 19347, 52967, 65931, 175955, 354813, 550319, 2092001, 1223331, 672381, 12596493, 30677467, 31623639, 19039107, 259567471, 455261709, 342330813, 1644505137},
  {1, 3, 7, 15, 21, 33, 117, 83, 497, 667, 1399, 931, 1695, 8171, 18839, 18413, 60445, 43765, 347199, 357535, 606859, 1607685, 256187, 1460397, 3877357, 61441465, 56073517, 250716913, 235119797, 975559623, 267477715},
  {1, 1, 1, 13, 3, 39, 53, 27, 193, 993, 671, 1871, 7579, 11457, 963, 20935, 106063, 125267, 27237, 956077, 751535, 3583109, 3087371, 8680759, 8338353, 15565165, 54892161, 194223747, 266250757, 87509517, 1174468629},
  {1, 1, 5, 11, 7, 39, 81, 107, 195, 387, 849, 395, 1317, 6487, 4975, 27935, 101755, 15165, 241213, 606877, 474027, 1484797, 7192313, 14183473, 159603, 34440493, 120756415, 81496681, 288936701, 428538277, 1030724593},
  {1, 3, 3, 3, 3, 15, 45, 127, 279, 111, 331, 357, 4637, 4697, 13939, 34341, 35721, 240465, 520417, 372877, 675575, 1412049, 596469, 12990661, 13606785, 9994791, 50042003, 214015039, 492290637, 944660735, 942020919},
  {1, 1, 3, 9, 21, 49, 47, 97, 61, 101, 181, 1867, 1201, 14099, 7655, 57359, 23517, 144371, 295475, 611351, 758849, 2157731, 460219, 2985299, 19028819, 51018157, 2404395, 131181317, 54972589, 527064085, 484389559},
  {1, 1, 5, 11, 25, 19, 51, 51, 101, 451, 545, 101, 7497, 9141, 29099, 14103, 45547, 217117, 472803, 759317, 1099353, 4128449, 3552935, 7690317, 29227771, 21284691, 94094639, 152748843, 10446013, 507190445, 1263362801},
  {1, 1, 1, 3, 13, 53, 119, 81, 377, 245, 765, 251, 3757, 16045, 2459, 17655, 112175, 87581, 35063, 93723, 1688065, 1635859, 7057183, 5423115, 18532267, 51070189, 79550647, 97570179, 80171741, 766715405, 1106115445},
  {1, 1, 1, 3, 5, 61, 65, 37, 331, 925, 1439, 3219, 2843, 11397, 3915, 25175, 60399, 22173, 130175, 337411, 1609223, 429675, 3560413, 11276007, 6570769, 52187121, 97405625, 156103351, 483990245, 639097565, 1963219861},
  {1, 3, 5, 9, 23, 31, 95, 155, 83, 641, 1129, 135, 477, 1623, 27887, 40221, 52735, 57783, 292025, 864045, 288069, 3248081, 2672561, 15533219, 7673327, 63061401, 132919855, 174359497, 149813437, 222641959, 668513009},
  {1, 1, 3, 9, 9, 61, 93, 11, 331, 585, 799, 1417, 1533, 463, 10207, 60799, 16573, 69555, 300463, 967971, 1440091, 3736573, 4461425, 8698859, 20375945, 13880539, 120706571, 128679449, 352995373, 728893541, 1636274359},
  {1, 1, 7, 7, 21, 51, 61, 29, 467, 935, 11, 3357, 1087, 12337, 18051, 39527, 96617, 182069, 43615, 814829, 1926919, 1362227, 117525, 11779181, 23398257, 37555487, 25237089, 229586003, 363931013, 84108621, 387222035},
  {1, 3, 3, 11, 1, 39, 103, 153, 351, 893, 1823, 835, 2149, 4203, 18039, 38925, 38265, 197977, 82643, 136801, 1256965, 3797683, 5082649, 10159663, 25035873, 57278497, 112598855, 144313029, 399126813, 727101207, 1081331831},
  {1, 1, 1, 9, 31, 13, 61, 235, 369, 359, 835, 2067, 2697, 15289, 17555, 30791, 110063, 71671, 409041, 763087, 845379, 1725069, 2208651, 9528657, 28100865, 37983089, 25242315, 209476175, 490139437, 123163965, 615887029},
  {1, 1, 7, 1, 15, 1, 107, 27, 201, 451, 1521, 313, 3195, 3847, 19311, 51711, 122841, 67027, 200777, 336499, 1244781, 3697377, 5274219, 10570037, 21269243, 61738095, 131206021, 52297837, 31853077, 417516981, 1085447251},
  {1, 1, 5, 13, 1, 27, 63, 137, 355, 489, 2039, 1015, 2519, 13797, 27243, 951, 66827, 246651, 473559, 171349, 591605, 323971, 4742789, 16681859, 8429521, 55427445, 59103709, 266945663, 380304245, 477988349, 137558481},
  {1, 1, 7, 9, 29, 33, 23, 197, 49, 555, 1087, 3447, 7299, 15513, 29395, 9927, 37609, 63035, 276683, 507479, 670937, 4110807, 6534015, 12920117, 4197685, 64118205, 122682009, 61357771, 112723557, 521692973, 1243168147},
  {1, 3, 5, 11, 7, 37, 55, 63, 443, 573, 1715, 631, 3405, 6155, 12855, 29069, 64895, 60629, 338253, 877791, 680873, 2423053, 5851521, 10836255, 17518573, 19513357, 30890359, 117359685, 314151389, 785853719, 235086705},
  {1, 3, 3, 3, 31, 35, 51, 167, 225, 617, 2007, 2555, 6819, 12709, 25579, 5045, 47753, 86257, 410333, 905081, 869601, 1971141, 733643, 782559, 22672329, 3053257, 32274705, 19150951, 98324549, 158090655, 1846314263},
  {1, 1, 1, 13, 15, 5, 73, 85, 109, 43, 1067, 3941, 1125, 10269, 14811, 63063, 35727, 61491, 344905, 673143, 923847, 4105147, 349435, 5059965, 76721, 8033895, 105573963, 2885227, 254565677, 169084333, 269854901},
  {1, 1, 7, 11, 17, 3, 127, 145, 279, 19, 1007, 3287, 4751, 12507, 87, 31695, 16377, 89753, 437955, 965605, 770261, 1327771, 887681, 14532849, 11847469, 27184301, 33873205, 1687825, 205699285, 714100805, 1683027795},
  {1, 3, 7, 3, 19, 1, 117, 111, 193, 435, 47, 1801, 529, 8547, 16935, 12845, 57853, 37209, 145729, 267235, 83603, 32361, 8374963, 412533, 13557541, 56721059, 4407031, 244911213, 102842333, 993263031, 1769403251},
  {1, 3, 3, 13, 1, 19, 101, 19, 469, 187, 207, 1243, 8153, 3273, 32371, 18565, 24297, 167551, 523103, 179021, 1519583, 792077, 3399599, 5355909, 29104357, 63292017, 58810139, 119619999, 79544429, 619737727, 187776439},
  {1, 3, 1, 5, 11, 51, 69, 189, 453, 775, 241, 3331, 4067, 14759, 25135, 9213, 77595, 99283, 522701, 56009, 645367, 182451, 1792323, 8444577, 10975251, 40350789, 7680649, 208089173, 47136293, 601072983, 153805973},
  {1, 1, 1, 1, 23, 55, 113, 133, 497, 731, 391, 2777, 3529, 955, 11863, 22671, 57855, 233119, 167433, 54789, 1604187, 4069907, 1429495, 4421077, 28595917, 43971427, 16261551, 217516229, 366695101, 923387669, 1698739957},
  {1, 3, 1, 11, 5, 49, 59, 35, 261, 949, 325, 3595, 7433, 11099, 22359, 6477, 84155, 193693, 392735, 777775, 1475077, 572937, 8262263, 1065467, 18499219, 57002573, 99966963, 218092061, 300069837, 11674743, 840379189},
  {1, 3, 5, 9, 13, 37, 103, 219, 329, 865, 1787, 2497, 7249, 9877, 32075, 22453, 45967, 54135, 5635, 934403, 1539585, 1783417, 7167643, 1528311, 33215413, 18879563, 58980775, 209138427, 51583645, 636991471, 1293966961},
  {1, 3, 7, 9, 11, 33, 19, 255, 191, 935, 1115, 1901, 1577, 9623, 18191, 61373, 81693, 33619, 229645, 44827, 1849157, 1489301, 5062069, 14714029, 29381665, 3554895, 9349207, 130693593, 450250077, 562477927, 1391186291},
  {1, 1, 5, 7, 29, 23, 77, 43, 283, 143, 1211, 73, 2835, 10235, 21463, 31247, 46843, 31953, 110367, 412937, 1714047, 2762613, 6648337, 4348429, 12852033, 46003307, 63547301, 256780497, 88756885, 148805445, 1126057553},
  {1, 1, 7, 3, 3, 27, 35, 173, 453, 425, 1225, 3023, 2159, 8433, 11395, 23911, 70249, 15857, 262401, 870201, 149361, 2054575, 826363, 13074239, 25708247, 60410113, 78234829, 122581707, 127829813, 900153133, 1171835091},
  {1, 1, 1, 5, 27, 21, 35, 25, 71, 145, 1545, 523, 4527, 7655, 23727, 59327, 44255, 45595, 114317, 16047, 596701, 2498051, 200545, 14818995, 9975887, 44334197, 8712677, 243123277, 30393237, 495181109, 306932309},
  {1, 1, 5, 3, 13, 49, 61, 157, 113, 775, 763, 1785, 225, 11851, 6839, 5967, 30971, 170517, 81287, 693151, 191139, 4122363, 2413159, 9176041, 27001225, 12323275, 42731351, 246683461, 418672989, 480030997, 439220593},
  {1, 1, 3, 1, 31, 57, 97, 229, 291, 777, 213, 4067, 921, 8203, 10615, 39311, 29885, 69147, 55433, 179083, 760023, 1379499, 5432789, 5746627, 21979123, 4109989, 60653043, 73069069, 324387789, 42701877, 1308066615},
  {1, 1, 5, 1, 25, 13, 125, 123, 263, 207, 119, 3111, 3841, 843, 9527, 719, 24891, 109079, 33943, 149379, 1279195, 2552045, 6648189, 4223637, 31330685, 35831037, 107312667, 15395869, 120976493, 722144373, 1903627697},
  {1, 1, 7, 7, 25, 57, 81, 129, 31, 133, 1869, 2949, 5563, 14965, 10251, 44151, 121993, 199349, 510099, 477119, 995471, 1733631, 1560961, 14373159, 25605027, 47993955, 42446813, 26766511, 188326773, 32038589, 665371091},
  {1, 3, 3, 7, 3, 51, 33, 127, 281, 425, 1253, 405, 7941, 8799, 29423, 9357, 57673, 208741, 149561, 184465, 966035, 1533193, 4235419, 4071823, 12330699, 66856083, 85424019, 105995657, 44059981, 568360743, 1013034551},
  {1, 1, 3, 9, 3, 63, 93, 173, 255, 609, 49, 111, 7785, 15865, 9443, 823, 83197, 143971, 468725, 608933, 724679, 2504467, 3626377, 9795363, 15646215, 62628177, 58368091, 213570431, 370427501, 969694973, 219451575},
  {1, 1, 1, 3, 17, 59, 113, 55, 155, 789, 1335, 177, 3071, 1851, 27655, 42015, 77007, 130861, 280891, 828093, 1849083, 2974737, 5544969, 151511, 27720233, 4774131, 57573769, 194348121, 175373605, 336134245, 22347157},
  {1, 3, 7, 15, 15, 23, 35, 35, 131, 623, 47, 437, 1337, 9891, 15543, 37181, 74477, 69861, 185781, 957869, 965801, 1344545, 5715089, 5504157, 16789125, 16296879, 458359, 209866397, 144593117, 456006519, 653972083},
  {1, 3, 7, 5, 29, 57, 39, 31, 111, 271, 59, 1473, 949, 3899, 6727, 38685, 80621, 82927, 88883, 247055, 1171817, 3221437, 2644645, 11049525, 2223425, 36234515, 42216515, 236980333, 413145101, 262200503, 463570227},
  {1, 1, 3, 11, 17, 19, 41, 229, 259, 691, 1455, 3023, 7455, 9711, 12207, 13871, 26669, 218401, 402211, 839461, 532275, 192275, 1748629, 1519433, 22490253, 54142565, 11023877, 97786101, 371469077, 625258709, 520452439},
  {1, 3, 5, 11, 29, 13, 9, 165, 499, 355, 1415, 1395, 7595, 15571, 4183, 59677, 126991, 121765, 80551, 293907, 1510495, 4005067, 8142229, 3219449, 29886469, 63152457, 16954089, 255075345, 94655653, 60037959, 1565788049},
  {1, 3, 1, 9, 5, 5, 25, 247, 185, 241, 1325, 3133, 7471, 2649, 27363, 52117, 73307, 16559, 445163, 272387, 43055, 133337, 8063035, 15931959, 7842099, 22166347, 81248393, 5916347, 507833637, 825852911, 151445},
  {1, 3, 3, 11, 17, 29, 57, 61, 51, 203, 993, 1837, 3785, 15163, 1191, 35165, 116329, 209961, 118899, 179179, 1472703, 3318931, 332901, 14572785, 5729587, 48311995, 62880303, 150791349, 350472125, 303403479, 2073840631},
  {1, 1, 7, 7, 21, 57, 79, 165, 277, 133, 93, 1055, 7169, 15685, 25115, 44167, 26201, 198437, 26543, 679511, 595889, 29783, 6954251, 8060579, 17032651, 16787257, 52999811, 29575091, 470874893, 306911693, 2103711539},
  {1, 1, 5, 3, 5, 17, 25, 177, 95, 323, 367, 1359, 4915, 6409, 13603, 51319, 36379, 185701, 212543, 465759, 1125847, 4053647, 3956337, 10216209, 7333885, 39612293, 1779945, 129392651, 418170021, 39411501, 1798476689},
  {1, 1, 1, 1, 11, 25, 115, 45, 373, 221, 1483, 591, 6561, 4527, 21679, 239, 120591, 180527, 149221, 754403, 1270917, 3876375, 3580619, 13232827, 13704109, 39835301, 92799771, 109582313, 277379437, 381204645, 194608821},
  {1, 3, 5, 3, 5, 23, 69, 77, 313, 473, 1037, 4045, 3969, 5445, 28955, 54693, 31007, 146477, 103599, 393049, 1931463, 3657555, 1513903, 10615127, 24321555, 8269211, 62300295, 225795323, 246592797, 329117935, 1350732657},
  {1, 3, 1, 5, 1, 15, 73, 83, 439, 463, 203, 361, 6835, 1061, 11323, 47653, 59067, 163427, 375991, 57201, 728283, 3846037, 959533, 10137413, 8593713, 6318363, 120762261, 66670367, 507434325, 985241983, 1960773717},
  {1, 1, 3, 11, 21, 5, 89, 233, 405, 253, 773, 3901, 6085, 5677, 6603, 24967, 42141, 197153, 451367, 459771, 1277503, 323271, 4480683, 897243, 28521475, 64010763, 64605987, 107999131, 426906509, 717400429, 1725158711},
  {1, 1, 3, 9, 15, 53, 71, 29, 101, 599, 1073, 705, 4507, 12779, 25863, 26463, 21197, 192995, 345081, 1023863, 1772601, 1235999, 7935243, 6408049, 1003651, 5640567, 48125957, 143607729, 303843605, 795668933, 1506082647},
  {1, 1, 3, 1, 3, 9, 27, 97, 207, 859, 417, 735, 2179, 5071, 26831, 1839, 91565, 201451, 369765, 106099, 1675373, 3706955, 5320905, 12686265, 27334339, 56155049, 77810561, 258888989, 465588997, 54554485, 113284375},
  {1, 1, 1, 3, 13, 63, 65, 125, 195, 611, 649, 2221, 3143, 143, 1487, 15887, 122287, 13933, 255335, 564289, 1466483, 2990091, 2892005, 1728889, 11297291, 9512395, 125877185, 75205085, 463449093, 141507701, 1194321173},
  {1, 3, 3, 15, 17, 57, 99, 119, 243, 407, 1229, 813, 5245, 1893, 4699, 43301, 82521, 17453, 21819, 892639, 1606925, 1658237, 29305, 9080405, 30782967, 53427283, 131188875, 182440419, 222068525, 187476623, 1825404343},
  {1, 1, 1, 5, 27, 27, 49, 13, 313, 287, 473, 2629, 3509, 11371, 21127, 23839, 13231, 108843, 330493, 133969, 659235, 3985851, 2302175, 5792241, 25781195, 6513059, 101359683, 246976445, 204444173, 316274165, 684295989},
  {1, 1, 7, 7, 23, 3, 75, 59, 245, 689, 1215, 2375, 3325, 1593, 26179, 65335, 47609, 112933, 103341, 772585, 1113929, 3616697, 6280599, 11864015, 13364593, 4554741, 20373415, 3602503, 209348509, 345950237, 541038963},
  {1, 3, 1, 5, 21, 51, 43, 107, 91, 611, 1405, 677, 2087, 9565, 24843, 63461, 35099, 168707, 396547, 21765, 1365809, 2501977, 7576249, 4126497, 11774095, 40308747, 125892737, 144015727, 333504261, 305088703, 4522261},
  {1, 3, 7, 11, 9, 27, 81, 101, 449, 201, 1507, 2217, 6767, 8059, 4583, 11581, 90541, 98561, 363291, 352541, 1533723, 2106339, 4452247, 722263, 32093533, 7305027, 101198329, 182293273, 338415845, 747756391, 530364563},
  {1, 1, 3, 9, 13, 41, 21, 195, 421, 315, 347, 2621, 2359, 9247, 18671, 4975, 103437, 58627, 381211, 646927, 117051, 2706725, 1389631, 16368853, 31671457, 32164943, 11227385, 131357021, 223089893, 227834997, 133988503},
  {1, 1, 5, 7, 31, 45, 77, 229, 455, 575, 1087, 1147, 2273, 13773, 4491, 54599, 105979, 76033, 252173, 298279, 819027, 1681203, 7739729, 6605133, 22916469, 38704577, 113665555, 95985947, 63897421, 439167853, 1970004017},
  {1, 1, 1, 1, 9, 5, 87, 19, 207, 545, 1435, 495, 1299, 4947, 4151, 37631, 8751, 71183, 18951, 319003, 165489, 970465, 5935005, 9447243, 6059133, 43414053, 89077673, 77698065, 330412965, 800316741, 588748693},
  {1, 1, 3, 3, 15, 9, 63, 67, 219, 735, 1911, 2361, 6503, 11977, 9219, 20599, 42173, 210953, 251917, 169987, 2045961, 3504225, 7145705, 195485, 2984893, 47731315, 23361585, 7973723, 45276613, 940818541, 1618786839},
  {1, 3, 1, 9, 31, 27, 103, 153, 81, 939, 461, 2753, 697, 537, 259, 24373, 96347, 245903, 397969, 166537, 1219301, 1759043, 6433959, 4320725, 19923607, 7344127, 70261719, 164638343, 240127581, 879795487, 588368501},
  {1, 3, 3, 9, 21, 53, 49, 211, 415, 817, 321, 3775, 2921, 9473, 5555, 49973, 68505, 95883, 443795, 798371, 579271, 2335565, 6029781, 12320843, 28049479, 8654721, 94635291, 226242215, 281829229, 70531999, 1067316407},
  {1, 1, 7, 3, 23, 55, 15, 51, 435, 1013, 73, 3967, 4575, 13099, 31719, 24223, 66473, 72833, 26501, 866221, 1605029, 1768393, 5240785, 3533991, 10082423, 57784097, 5078829, 9762209, 470519221, 757951365, 2058957779},
  {1, 1, 3, 7, 5, 27, 43, 225, 267, 21, 1261, 603, 6913, 4421, 28795, 33639, 118077, 112269, 159119, 868501, 811697, 687475, 3502909, 11918943, 2228579, 62260237, 4851243, 196743619, 496391597, 207790093, 187428791},
  {1, 1, 7, 13, 25, 31, 101, 109, 237, 91, 1587, 1987, 2795, 6991, 591, 61007, 47561, 23951, 198039, 117153, 1620951, 3578771, 6601435, 15395741, 12471769, 34474529, 77732857, 141166477, 430580965, 1044039989, 320033939},
  {1, 1, 3, 13, 23, 51, 91, 89, 287, 39, 1513, 463, 6135, 10469, 2107, 21287, 55389, 257927, 277129, 184497, 104841, 3410971, 6393305, 4432933, 32954771, 1181177, 32120693, 176566015, 336728277, 759866621, 551550551},
  {1, 3, 3, 1, 9, 43, 125, 157, 369, 495, 1849, 785, 6357, 6557, 29387, 16933, 12729, 188291, 445855, 532389, 464775, 2556931, 4396391, 9025297, 1665355, 62968279, 91208939, 123563835, 279496877, 900426735, 631052727},
  {1, 3, 1, 13, 5, 25, 107, 139, 367, 239, 1671, 1239, 7027, 5291, 21255, 42077, 10411, 142411, 122179, 215407, 298597, 3557073, 587613, 16559621, 21110269, 34042205, 85238649, 83296745, 332616933, 881580711, 807905941},
  {1, 3, 5, 13, 11, 13, 35, 177, 45, 939, 251, 59, 333, 13105, 29683, 7669, 4767, 183619, 183517, 921799, 138813, 201307, 862955, 2268525, 24777297, 47045449, 54290767, 203630815, 182787133, 905565311, 2063319025},
  {1, 3, 5, 7, 29, 57, 109, 227, 435, 739, 423, 1941, 3345, 12731, 15527, 59197, 24335, 2505, 30175, 823159, 1181971, 2393077, 6819401, 1553825, 30070261, 18152847, 109330431, 232722229, 510615805, 69567959, 716011249},
  {1, 3, 3, 9, 23, 51, 19, 207, 69, 99, 955, 519, 7305, 2415, 9039, 33357, 90921, 71499, 21201, 868449, 1844513, 275293, 1092559, 11972621, 30430417, 45430037, 102278323, 139533177, 69691341, 446498023, 53527607},
  {1, 1, 5, 13, 17, 1, 67, 201, 61, 403, 1059, 2915, 2419, 12773, 30171, 14887, 24795, 38987, 458839, 912895, 264493, 3366787, 6278379, 5871253, 1928441, 31102401, 77426273, 119484631, 357529733, 792267869, 1703398673},
  {1, 3, 1, 11, 17, 19, 25, 27, 207, 299, 143, 1955, 5669, 2301, 17579, 40261, 8827, 192077, 218587, 103637, 1114979, 1595181, 4403525, 5829073, 16356477, 21961641, 121369927, 175324035, 527788573, 317540111, 377213301},
  {1, 1, 5, 3, 25, 57, 45, 255, 489, 1011, 1699, 2637, 5279, 12211, 3831, 9503, 88875, 252997, 319683, 372207, 356067, 2981961, 5275819, 12681461, 2000601, 62948687, 26448593, 118797281, 380425285, 65448581, 1464763921},
  {1, 3, 3, 15, 7, 47, 113, 33, 511, 907, 1815, 1741, 2091, 13857, 32019, 11093, 94201, 155213, 521165, 41797, 228747, 1919495, 2283073, 7047993, 23969461, 47746935, 80316533, 5284667, 108252885, 645926895, 511821399},
  {1, 3, 3, 5, 5, 27, 95, 3, 353, 253, 947, 393, 1815, 14551, 2047, 16845, 35977, 162887, 388059, 407517, 987541, 379321, 5931139, 3113739, 5474069, 24678227, 19108449, 216439045, 335601797, 507287831, 1200989463},
  {1, 1, 5, 11, 29, 19, 63, 117, 293, 861, 2039, 9, 5999, 6909, 435, 46111, 979, 204341, 18575, 605077, 1743549, 2758007, 592739, 8025203, 6244521, 35976771, 38720625, 211837783, 153716997, 164401565, 470027857},
  {1, 3, 7, 3, 15, 63, 107, 173, 509, 817, 99, 2825, 131, 7917, 29075, 37693, 46357, 15025, 196741, 365981, 1567449, 48439, 975739, 10612495, 6808517, 39070555, 13853261, 156448863, 451237877, 442966719, 270380691},
  {1, 3, 1, 1, 29, 49, 33, 153, 119, 777, 1315, 3581, 5675, 4043, 1407, 1701, 57363, 222943, 223115, 554687, 1186983, 3119759, 4043665, 11806739, 1445501, 50849423, 121306285, 141911673, 383522549, 126131495, 1841087637},
  {1, 3, 5, 15, 13, 11, 17, 147, 327, 305, 367, 3237, 5423, 13757, 19795, 16093, 99671, 148281, 20887, 1042077, 417251, 3961069, 1331573, 12792771, 3526785, 61650191, 78317609, 161023719, 442203621, 93449055, 503764945},
  {1, 1, 5, 13, 1, 39, 35, 29, 25, 751, 1365, 2243, 8181, 7063, 7095, 47175, 47027, 3155, 360223, 435457, 2021177, 3989783, 3456447, 12264921, 11834335, 22819077, 40026039, 195562113, 131617053, 612932677, 519664561},
  {1, 3, 7, 11, 25, 53, 11, 111, 289, 755, 1201, 691, 3679, 3725, 21251, 51325, 10517, 247481, 473859, 1307, 71029, 3008285, 5115203, 13182301, 28941923, 44616173, 42718957, 55006927, 146590581, 280710271, 1518466323},
  {1, 1, 1, 11, 11, 37, 33, 211, 395, 691, 1817, 861, 6485, 12077, 14131, 54095, 53367, 5117, 173321, 336143, 721239, 3533237, 6715161, 764857, 4484419, 23894931, 85294651, 238858435, 123797165, 918547789, 1825663605},
  {1, 3, 3, 11, 21, 3, 111, 171, 305, 561, 1501, 2011, 7841, 10931, 8255, 39925, 62705, 122001, 61727, 780085, 625933, 4057033, 5174155, 9618383, 20039307, 24265493, 55563891, 121520781, 171473677, 454605303, 1600655095},
  {1, 3, 7, 9, 9, 59, 109, 113, 31, 915, 103, 1861, 2779, 10619, 3423, 25317, 67733, 11099, 267287, 758845, 1221711, 1002711, 8365529, 1856985, 32212461, 21856983, 29661245, 40090857, 142071477, 878298599, 155476371},
  {1, 1, 1, 1, 7, 25, 61, 97, 103, 723, 1145, 3105, 371, 339, 14383, 32583, 70807, 169431, 178705, 409091, 206339, 2548335, 1698337, 5804041, 21082515, 10664847, 55803045, 203743233, 261864149, 817971781, 2039762837},
  {1, 1, 7, 13, 17, 9, 113, 51, 233, 209, 1117, 211, 6969, 2347, 17039, 33943, 6961, 135895, 45895, 375247, 64151, 536293, 7037723, 10257311, 16810379, 429277, 106006459, 208971589, 16631853, 638520021, 1198731891},
  {1, 1, 5, 9, 25, 43, 21, 217, 327, 735, 197, 1063, 799, 801, 22715, 35919, 10963, 159671, 97487, 861549, 769975, 838671, 5967089, 3446085, 30785527, 32639233, 87691509, 123603123, 466767765, 259654541, 1901474705},
  {1, 1, 7, 13, 9, 13, 73, 33, 415, 923, 863, 1999, 5383, 8119, 15831, 28247, 74705, 186327, 335327, 205307, 1026503, 3721623, 526377, 15223089, 8211781, 39800781, 90266581, 122243829, 58815765, 645994389, 444626195},
  {1, 3, 1, 5, 7, 33, 51, 185, 289, 967, 1277, 1011, 767, 15505, 25595, 39629, 102739, 131259, 163801, 812107, 1916605, 3404619, 4265007, 2813873, 20951687, 8707141, 486333, 114137107, 508990773, 447429647, 983328021},
  {1, 3, 3, 13, 21, 11, 105, 235, 343, 1021, 2009, 2251, 3865, 6923, 10255, 42005, 127153, 91991, 370131, 944229, 1954907, 3849221, 3781813, 13393167, 22112943, 17211645, 53082843, 177893125, 75634221, 332320727, 954434423},
  {1, 3, 5, 9, 29, 11, 33, 17, 149, 155, 1739, 3039, 7015, 2401, 19963, 47245, 35159, 86847, 251979, 591333, 1161299, 2209839, 1038499, 15178501, 31795449, 49676277, 131899109, 150242595, 192930645, 399795663, 1939129105},
  {1, 3, 7, 7, 17, 13, 89, 177, 297, 267, 545, 3861, 329, 13267, 22479, 405, 78453, 105813, 178387, 503875, 522807, 2056839, 6486015, 15273613, 29556547, 25402667, 83953975, 162858981, 317279005, 158575831, 690916531},
  {1, 3, 5, 15, 27, 33, 1, 231, 181, 557, 447, 379, 7845, 1295, 17671, 5349, 73463, 87705, 302273, 92507, 1351231, 3519057, 4507535, 3417867, 570337, 62708353, 105450431, 64560313, 168262973, 571564071, 1335155889},
  {1, 1, 5, 13, 3, 63, 59, 33, 263, 877, 1867, 1383, 641, 7139, 8383, 695, 82739, 165075, 480989, 984413, 1185113, 2266939, 1119193, 4104791, 28680701, 64449685, 129413843, 237415661, 133156109, 166581877, 1812814705},
  {1, 3, 7, 5, 13, 51, 9, 113, 223, 605, 1189, 4063, 6925, 9563, 16367, 62229, 63253, 143607, 266795, 303405, 321355, 2882847, 208233, 3434423, 9062035, 40288465, 114443511, 12973325, 466281885, 747277815, 774315699},
  {1, 1, 1, 13, 5, 35, 83, 107, 295, 231, 265, 5, 4087, 6407, 28871, 33047, 111799, 6075, 360747, 492845, 1275697, 1804209, 8275533, 5346965, 33042955, 49034843, 131252169, 155418989, 389791845, 850072821, 1310934869},
  {1, 1, 5, 1, 7, 25, 95, 137, 97, 987, 1753, 2781, 1369, 6903, 7015, 27175, 66899, 247231, 72097, 689187, 1022977, 2399459, 7597151, 5190733, 3441211, 48205683, 118458551, 48860833, 401256733, 87860677, 1242386097},
  {1, 1, 5, 13, 19, 61, 77, 229, 193, 165, 811, 249, 79, 10719, 18903, 40375, 38835, 201011, 234285, 533535, 1924107, 2544891, 355503, 15126275, 30690837, 53907403, 92897697, 109880533, 36608325, 1037971349, 1610533969},
  {1, 3, 7, 7, 27, 9, 119, 193, 459, 43, 1989, 2959, 3595, 6341, 23955, 19677, 128885, 85525, 319865, 622131, 1135129, 3408539, 7200453, 7366985, 23189743, 59310169, 19862673, 219750207, 173082373, 511697599, 986006483},
  {1, 1, 5, 11, 5, 43, 35, 33, 25, 581, 897, 351, 4201, 3971, 4191, 31959, 101715, 39541, 65847, 284221, 1047057, 1994459, 4613095, 15561443, 6296127, 39105337, 104234851, 209057629, 535353421, 627872181, 1274325745},
  {1, 1, 7, 11, 21, 29, 53, 45, 359, 197, 313, 3825, 6717, 4077, 6131, 50735, 92177, 105233, 209391, 914987, 1390091, 2862691, 4095961, 5571535, 20907999, 27067879, 86050539, 188287715, 21320173, 1008038989, 2110495859},
  {1, 1, 1, 15, 3, 45, 99, 133, 357, 315, 1159, 241, 2463, 11253, 4611, 48927, 109751, 135065, 68713, 212519, 1380869, 1977895, 1505843, 13759829, 1568501, 51631907, 59489937, 224609823, 292737669, 193581373, 460841813},
  {1, 1, 7, 11, 9, 33, 111, 85, 443, 601, 447, 337, 6471, 7029, 15811, 64479, 35505, 60433, 237811, 896943, 2072281, 1027551, 3648221, 214987, 33422769, 52014715, 70547337, 265817807, 281568613, 200848893, 881450451},
  {1, 3, 7, 9, 13, 33, 25, 31, 9, 729, 1763, 4077, 7575, 7877, 29251, 58589, 129717, 75803, 494835, 113551, 488191, 2109785, 6020899, 7277447, 19528517, 26452511, 16215345, 58136943, 370765829, 827848831, 1112386899},
  {1, 3, 5, 13, 13, 37, 29, 103, 53, 229, 591, 1073, 1323, 14405, 8227, 62589, 33911, 171291, 96243, 898971, 513235, 97081, 3346815, 15053667, 4543905, 35880299, 83248197, 213971367, 480658133, 456798815, 165605137},
  {1, 1, 5, 1, 17, 33, 15, 183, 473, 297, 2003, 93, 4955, 1787, 26559, 35143, 47219, 61951, 319031, 71607, 1463289, 3544349, 7725115, 369655, 28107701, 58632271, 20325981, 164597257, 283872181, 582867429, 815392657},
  {1, 1, 5, 13, 5, 29, 113, 161, 267, 451, 1193, 149, 273, 11809, 17931, 6719, 92243, 244819, 357883, 824979, 1433263, 4076487, 973309, 7961505, 3882035, 40693115, 91747727, 229686919, 224953725, 325951069, 253191473},
  {1, 1, 1, 9, 17, 39, 47, 233, 165, 373, 955, 2891, 7523, 7235, 16383, 45063, 44279, 178943, 166711, 752785, 1778837, 461339, 3730299, 14602571, 8278869, 18962457, 36733929, 65580505, 25742693, 736518565, 480736469},
  {1, 1, 1, 3, 7, 21, 115, 205, 153, 449, 339, 2073, 1077, 5749, 13187, 29519, 23959, 172821, 354165, 491431, 1604597, 3045615, 7546975, 14813943, 22400165, 17593483, 30282747, 27922635, 255216685, 710320877, 171380213},
  {1, 1, 7, 13, 9, 39, 117, 187, 37, 753, 227, 3519, 7391, 5751, 439, 33399, 123697, 74423, 224959, 400561, 393423, 419965, 4237623, 3221639, 1403045, 2289197, 47573661, 205639685, 297393077, 149513685, 2046376851},
  {1, 1, 1, 9, 5, 19, 41, 161, 141, 195, 1719, 3321, 5, 12877, 22643, 14063, 44375, 149471, 20515, 243661, 1831019, 1347663, 7082075, 8024205, 29877, 63797611, 107971995, 65512563, 489684141, 262174349, 442582005},
  {1, 3, 7, 11, 21, 13, 83, 55, 299, 75, 1905, 3765, 4685, 12297, 21019, 28349, 125909, 53145, 328879, 551643, 1223757, 4025397, 2559481, 10460785, 16676507, 64304939, 5021299, 256155223, 166652813, 571939103, 47041779},
  {1, 1, 7, 3, 3, 23, 111, 243, 187, 297, 1061, 2515, 977, 9555, 23007, 1495, 125969, 112185, 63209, 610413, 1479261, 1789665, 4064113, 14159939, 24113739, 11639853, 108140787, 174729869, 201722765, 637705973, 1665620211},
  {1, 3, 7, 3, 29, 11, 103, 177, 225, 875, 1649, 1401, 6383, 8309, 31747, 59805, 48629, 48721, 489271, 258797, 1897045, 3923523, 2550383, 394685, 23148915, 649991, 31107113, 152857759, 50837349, 1049262399, 1346807123},
  {1, 3, 5, 3, 3, 41, 71, 3, 373, 757, 701, 2825, 1521, 13217, 15083, 17085, 115735, 15637, 360865, 119027, 1976937, 694905, 1612671, 5671971, 16720147, 52175159, 106311143, 64978775, 157129821, 771254687, 190415281},
  {1, 1, 5, 3, 11, 5, 103, 227, 209, 723, 1543, 3895, 6345, 7901, 32035, 31535, 75411, 152541, 32809, 819439, 772977, 3703785, 6663195, 15567885, 23213557, 14335205, 25188619, 161826115, 415954669, 219511373, 2058023665},
  {1, 1, 5, 1, 9, 51, 77, 67, 359, 937, 557, 993, 3871, 3577, 3355, 9775, 84979, 214431, 120751, 431317, 1700159, 4004837, 4860965, 4290363, 1972499, 46179663, 80476217, 24258035, 169502117, 849804557, 1972035153},
  {1, 3, 7, 1, 1, 15, 121, 239, 29, 113, 1123, 3877, 6941, 14129, 12587, 65437, 49621, 221971, 172719, 813529, 585011, 3850513, 7076567, 4550247, 31909829, 37054263, 108915115, 49130479, 22019949, 488530559, 1594306803},
  {1, 1, 5, 1, 27, 61, 83, 113, 185, 601, 947, 3933, 381, 13869, 11987, 7535, 33747, 28383, 209213, 453375, 1714549, 1511891, 1883847, 6660383, 19588789, 9463919, 98531707, 231378003, 256544813, 747469453, 292743665},
  {1, 1, 5, 3, 5, 37, 97, 31, 81, 367, 747, 1811, 5313, 14151, 8247, 34887, 66963, 123901, 431335, 1045203, 662671, 3532313, 3744283, 5360385, 18675969, 63009641, 67003259, 228162329, 330721069, 837972005, 1505710321},
  {1, 3, 5, 9, 27, 61, 87, 31, 185, 521, 837, 959, 5001, 3957, 15347, 54733, 92695, 204767, 461485, 219775, 241073, 507445, 4317067, 2654583, 30966563, 56528577, 129730771, 207653739, 338609549, 948255855, 725748465},
  {1, 3, 5, 3, 11, 61, 37, 19, 107, 749, 1345, 3829, 6701, 4315, 16127, 25717, 37975, 27957, 218473, 463351, 223363, 2124925, 3714249, 9664135, 20665223, 50085011, 63464915, 37811877, 370258189, 978564951, 1859615217},
  {1, 3, 1, 15, 13, 45, 101, 113, 243, 963, 1861, 3283, 1419, 12131, 29055, 60997, 93747, 238513, 34663, 904923, 220603, 2325407, 4830837, 1388241, 27238207, 4747813, 76853481, 224463017, 269824741, 876456679, 2101368917},
  {1, 1, 7, 1, 11, 63, 17, 117, 271, 819, 677, 669, 1991, 12511, 18295, 38455, 119537, 42107, 128229, 959965, 1335251, 3034011, 5573489, 1239721, 15042895, 51725099, 68406521, 93971253, 505129893, 795812501, 1537646931},
  {1, 1, 1, 13, 13, 33, 41, 73, 187, 537, 993, 3147, 1013, 16063, 32071, 6599, 119511, 195707, 43747, 915455, 1972655, 1112495, 4680765, 5781307, 29672271, 51529117, 36226487, 178613577, 98444701, 1029896165, 44435893},
  {1, 3, 1, 1, 25, 21, 107, 81, 117, 917, 113, 349, 4475, 9149, 5107, 58077, 35379, 195679, 477871, 700275, 1653909, 522147, 1231883, 4282251, 1960287, 60791643, 102169141, 89234191, 249734677, 794161919, 1048292245},
  {1, 1, 1, 11, 21, 21, 29, 251, 125, 681, 141, 2893, 5843, 14359, 26279, 33719, 118455, 225533, 221175, 469475, 1078675, 1200581, 4810767, 15964431, 5226139, 62533667, 101104141, 131498229, 101353461, 965401621, 909788437},
  {1, 3, 3, 1, 5, 41, 85, 163, 387, 29, 1593, 221, 2769, 10809, 31499, 40781, 31809, 154955, 254539, 485479, 244947, 42477, 5654481, 11190615, 3982543, 36684571, 74372431, 195747247, 322073213, 839569023, 792406839},
  {1, 3, 5, 11, 1, 17, 69, 127, 273, 449, 1855, 2971, 7031, 10583, 4999, 62213, 76231, 163021, 161731, 54271, 348067, 1853425, 5076795, 2996063, 7844409, 36748065, 33077885, 60859285, 475825205, 353675415, 1854252433},
  {1, 1, 5, 7, 1, 61, 9, 211, 123, 563, 111, 1883, 5801, 2191, 4151, 22423, 28355, 236489, 162651, 933603, 969399, 3834101, 3637305, 8094409, 8095269, 61702281, 25658723, 40274161, 288206029, 927294597, 638583665},
  {1, 1, 3, 11, 11, 51, 1, 81, 405, 803, 2017, 161, 5429, 731, 21935, 47079, 47429, 73545, 100433, 175049, 502411, 1979999, 3431283, 6055949, 23042739, 46281903, 6782003, 3260661, 79734541, 126115733, 332678135},
  {1, 1, 7, 9, 15, 55, 65, 51, 459, 485, 1539, 3135, 2929, 7867, 18239, 32135, 6465, 166723, 499905, 559437, 459603, 828717, 1805417, 12916223, 700965, 15980397, 32462915, 155318117, 204400717, 237653845, 1396638067},
  {1, 1, 7, 11, 3, 45, 15, 7, 331, 417, 1813, 4009, 1341, 10965, 5171, 44191, 5825, 104897, 415433, 916167, 979737, 3236845, 5301689, 12584171, 28498619, 781127, 44561147, 57214667, 243104941, 311758061, 1198695795},
  {1, 1, 1, 5, 9, 29, 89, 121, 277, 509, 1989, 1293, 4787, 16097, 3195, 28223, 109511, 230851, 245319, 513499, 1227271, 2508543, 3662531, 4445555, 17194027, 47080499, 276477, 221741043, 520071221, 922786829, 2063706389},
  {1, 1, 3, 9, 17, 45, 97, 197, 339, 943, 1377, 2947, 5833, 7, 22919, 25079, 121797, 185419, 290767, 772459, 2070079, 3932915, 8322973, 13508189, 7347507, 908989, 2427403, 101137625, 219492461, 383997733, 1826246775},
  {1, 1, 7, 9, 15, 61, 75, 233, 401, 705, 825, 2521, 3787, 14387, 30911, 32055, 115905, 139459, 328769, 713543, 1195085, 2227403, 2438319, 3556359, 5019455, 24207343, 17731421, 53972097, 474717109, 229193925, 1318866451},
  {1, 1, 7, 15, 25, 57, 3, 43, 361, 459, 1551, 1859, 6787, 2293, 10563, 2159, 29825, 205957, 337563, 940727, 397717, 3144777, 146867, 12493877, 15629245, 4107505, 46949561, 57083263, 438795045, 625017277, 689373395},
  {1, 3, 3, 11, 11, 35, 91, 65, 43, 509, 1829, 1149, 4801, 4109, 18275, 48221, 90497, 29569, 425873, 553369, 1342385, 2629883, 2121797, 14999855, 19390963, 56911939, 28768047, 149957115, 420506877, 355469103, 1601215031},
  {1, 3, 5, 9, 15, 3, 81, 109, 231, 481, 417, 2505, 315, 6693, 14291, 14509, 31111, 162703, 330633, 148153, 1593323, 3771971, 5255929, 15912219, 18802559, 11316367, 65532845, 132091383, 310438901, 362791455, 1545345297},
  {1, 1, 3, 9, 3, 7, 107, 221, 297, 543, 149, 579, 927, 79, 22167, 62247, 40325, 20619, 436125, 7589, 1993193, 2821527, 4891839, 1345465, 9757583, 45377429, 46361037, 20216061, 52048373, 977939125, 1783253783},
  {1, 3, 1, 11, 17, 3, 81, 137, 157, 587, 741, 1277, 2631, 3953, 18203, 51741, 89475, 187013, 480899, 60301, 1425555, 2360743, 5932619, 2709717, 18116231, 43020699, 86902441, 202520971, 399649637, 859186287, 263878485},
  {1, 1, 7, 5, 13, 43, 117, 19, 495, 185, 1105, 605, 5249, 11099, 27023, 38279, 62593, 4239, 394907, 600229, 1340823, 3260373, 7773085, 9780139, 31632671, 50015643, 112491519, 32168109, 76517309, 192785781, 1122784307},
  {1, 1, 7, 9, 23, 55, 91, 213, 21, 779, 857, 2047, 7813, 10053, 31907, 11103, 103809, 64643, 454553, 406205, 1184489, 2326847, 7708499, 12800245, 3093659, 25915873, 121377439, 88351027, 536635325, 343193229, 481131315},
  {1, 1, 1, 1, 27, 7, 39, 181, 63, 519, 1073, 3147, 4111, 363, 6367, 50487, 99335, 101383, 308253, 787509, 875589, 1066215, 4556233, 11990189, 31343811, 64276533, 69590109, 88134745, 370800181, 609955109, 2081370389},
  {1, 3, 7, 9, 15, 61, 7, 139, 495, 805, 1545, 3789, 2411, 3989, 12691, 49261, 110597, 69643, 12289, 618503, 962625, 3018865, 5886181, 5837527, 23689215, 57384523, 2915541, 94058103, 432347925, 910683423, 1050945171},
  {1, 1, 3, 1, 25, 11, 23, 241, 167, 607, 479, 153, 7787, 13929, 16043, 43487, 88069, 112643, 296983, 276541, 784485, 391411, 1296665, 153813, 18108413, 61493851, 76668965, 173613275, 329987029, 304723245, 527046935},
  {1, 3, 5, 15, 29, 35, 45, 71, 457, 297, 883, 3021, 5361, 15427, 6687, 36773, 18439, 186377, 157719, 768037, 1660959, 3967121, 6806003, 12114855, 32542213, 58999183, 55137959, 185586325, 412125533, 260891287, 646768433},
  {1, 3, 1, 7, 29, 27, 93, 241, 427, 89, 1185, 37, 3863, 14095, 14951, 36069, 93955, 134409, 408855, 815421, 623391, 3336999, 6457917, 14821835, 16043455, 10814723, 41227993, 208208053, 42679077, 606308119, 451087957},
  {1, 3, 1, 5, 5, 45, 51, 15, 235, 889, 1649, 2331, 2713, 10943, 32151, 6261, 116707, 213931, 389739, 521811, 1935709, 2625341, 2811865, 15869555, 21746895, 56311357, 82655051, 225093185, 260954733, 221607111, 448853877},
  {1, 1, 3, 11, 11, 15, 71, 85, 135, 163, 139, 1147, 1043, 3195, 27135, 48055, 24805, 103913, 520433, 328789, 801045, 3473863, 2782609, 3577433, 4048881, 40223681, 71815849, 87946001, 492109797, 447662341, 738182871},
  {1, 3, 5, 13, 3, 43, 71, 131, 473, 933, 569, 2491, 7751, 1865, 14971, 56413, 32743, 191083, 498349, 920313, 1391933, 9153, 4353059, 13301307, 32423607, 6146173, 97438581, 14119371, 114535445, 348417263, 1847970449},
  {1, 1, 7, 9, 21, 37, 105, 227, 329, 509, 1319, 307, 1557, 14625, 26635, 45423, 28385, 78627, 96187, 939467, 1387287, 2491141, 540875, 10305795, 20956441, 7258601, 84886479, 240534823, 181716733, 776043869, 1449839283},
  {1, 1, 3, 13, 15, 1, 25, 93, 335, 953, 769, 4039, 369, 10727, 13255, 60759, 62821, 146863, 381049, 729419, 1204807, 628299, 3183345, 16333043, 30009519, 17697345, 39980315, 38939833, 301869869, 737339685, 1331587703},
  {1, 3, 7, 5, 17, 21, 59, 89, 437, 679, 437, 1543, 7663, 5005, 1939, 39821, 35941, 197223, 86311, 78019, 2039413, 4091483, 4819595, 1578441, 2830031, 21977041, 123841749, 235032239, 520222869, 425467903, 1811680915},
  {1, 1, 7, 15, 27, 49, 125, 13, 397, 877, 1087, 2191, 4711, 9065, 18427, 22751, 22113, 123493, 346553, 499675, 1596435, 3710099, 2571331, 13190371, 31262769, 8164005, 104180069, 48489339, 359154261, 133712429, 1450707475},
  {1, 1, 7, 5, 15, 47, 115, 125, 187, 31, 1003, 2575, 5397, 3883, 3583, 29511, 82145, 2991, 290937, 337605, 730097, 318727, 7445869, 12711613, 21821081, 8871557, 123029831, 12244893, 145730397, 771826229, 212460083},
  {1, 1, 7, 11, 15, 1, 127, 207, 383, 707, 183, 1053, 3123, 14071, 27031, 18951, 21153, 158561, 57317, 1044947, 686497, 3814677, 6464573, 12371617, 11101613, 53094751, 107819893, 253366341, 293484437, 959082581, 55266323},
  {1, 3, 3, 1, 31, 53, 15, 19, 477, 245, 777, 1613, 5813, 7443, 9215, 7461, 1697, 52651, 232817, 425199, 1435577, 3731961, 7401895, 7746083, 33175415, 11555243, 108413483, 85185709, 68738157, 634969975, 1203351415},
  {1, 3, 1, 11, 23, 59, 65, 23, 493, 157, 1389, 2833, 4535, 3907, 5887, 8309, 109347, 120805, 37221, 416473, 372171, 1377049, 4407143, 9615035, 15722435, 25963487, 74790913, 58491273, 95987397, 598644199, 583524565},
  {1, 1, 7, 1, 19, 7, 51, 135, 327, 441, 1841, 3091, 3451, 14381, 17523, 16559, 28257, 54123, 391597, 592453, 1520017, 1204445, 2859493, 4104331, 16115199, 66251445, 62993977, 44360247, 358368421, 259302173, 2089061459},
  {1, 1, 7, 7, 3, 37, 29, 249, 437, 319, 1693, 945, 7639, 5923, 18559, 39287, 95777, 178925, 386161, 921567, 1060539, 771191, 4873619, 4558737, 7666135, 54189335, 132484641, 184193065, 359064389, 431779813, 791005139},
  {1, 3, 7, 15, 7, 61, 81, 127, 383, 99, 23, 3833, 3973, 7651, 31647, 23269, 77605, 172205, 404581, 263151, 339095, 3560585, 3925349, 15376321, 20359373, 18034107, 102298379, 199837101, 248862317, 754160631, 374215667},
  {1, 3, 1, 7, 7, 21, 119, 185, 243, 619, 1363, 2033, 4835, 5089, 26011, 3037, 55331, 18025, 85549, 459111, 1043937, 1392227, 76669, 13258441, 26382505, 2945755, 24777261, 61472835, 323879669, 52563023, 1995380885},
  {1, 3, 1, 1, 3, 27, 63, 145, 271, 735, 695, 3981, 3049, 5433, 9547, 52269, 37731, 229743, 240229, 535097, 1736293, 1351331, 4587249, 16283877, 28442197, 44297343, 3012111, 176534335, 472491389, 673727167, 444085685},
  {1, 3, 3, 1, 3, 29, 79, 211, 279, 819, 501, 3665, 1455, 10455, 25559, 22949, 119905, 247467, 483373, 330303, 2081305, 2801241, 4902829, 11275313, 18901615, 7875223, 60601017, 129597997, 28679333, 690557431, 1834610775},
  {1, 1, 3, 3, 31, 61, 113, 5, 411, 91, 489, 3257, 5939, 6715, 143, 759, 92261, 180513, 280757, 337967, 870487, 1644659, 6993973, 8038553, 32524435, 4007975, 86810949, 205262297, 252992341, 743755301, 1365078167},
  {1, 1, 5, 1, 23, 11, 103, 89, 377, 441, 43, 967, 3383, 8717, 15347, 60175, 2339, 93231, 157025, 113665, 390157, 3593243, 4674679, 3996879, 906205, 47349141, 102810933, 54938719, 43626517, 541912509, 430728209},
  {1, 1, 5, 13, 29, 39, 97, 189, 197, 621, 1755, 333, 6783, 9711, 25319, 5319, 6243, 193443, 52019, 193545, 830587, 455747, 1719707, 1736371, 28826101, 43111791, 10803241, 1491077, 23495013, 209648085, 1566523345},
  {1, 1, 5, 13, 27, 17, 97, 197, 351, 799, 335, 765, 5329, 12549, 27699, 37407, 92771, 108387, 499445, 48435, 692495, 2770403, 1962993, 777941, 29336365, 62530367, 130070967, 149310387, 161739165, 653034637, 1640796209},
  {1, 1, 5, 11, 29, 17, 9, 211, 127, 633, 1187, 3965, 4145, 12667, 20175, 9959, 108899, 175973, 133439, 31015, 335231, 1847077, 3173277, 12298603, 1883277, 4134911, 100930855, 192688141, 224657629, 265057269, 533573937},
  {1, 1, 7, 5, 27, 29, 65, 115, 287, 325, 461, 5, 899, 2027, 27087, 7351, 11105, 228783, 139565, 754847, 210615, 707813, 4805853, 15943255, 1176091, 18970411, 89565277, 51914649, 361252149, 550019749, 1405682451},
  {1, 1, 1, 5, 27, 17, 31, 13, 231, 627, 1163, 649, 1693, 9975, 3895, 16695, 8551, 74595, 334389, 114835, 2003441, 678359, 7839137, 9999721, 6194769, 28740915, 105875331, 155694985, 107406157, 1072492389, 908979317},
  {1, 3, 1, 15, 7, 49, 113, 123, 427, 603, 347, 2785, 7129, 4645, 12243, 93, 44131, 234977, 129405, 564899, 1414383, 3307517, 1800697, 10427265, 25656289, 13208475, 100959127, 198753011, 10918685, 951397007, 1514324277},
  {1, 1, 3, 7, 1, 33, 113, 105, 411, 939, 205, 3965, 4361, 4649, 26955, 9967, 41573, 50149, 66787, 633503, 1000087, 650223, 2402797, 657817, 14716247, 11833835, 123906151, 206884655, 108810717, 114055677, 1191039927},
  {1, 1, 1, 1, 5, 21, 35, 159, 275, 929, 1193, 3205, 4787, 3515, 2991, 3959, 63335, 169127, 57891, 378299, 782781, 723757, 7266049, 7947483, 7451271, 56441523, 67007357, 164676137, 159222965, 133648229, 1591449109},
  {1, 1, 1, 5, 1, 21, 29, 191, 275, 233, 1239, 515, 4349, 14989, 26351, 14315, 107627, 231295, 3923, 45903, 1518543, 903709, 5147157, 11553975, 6333933, 19426053, 25420099, 175875103, 285596637, 1047011373, 1361673029},
  {1, 1, 5, 11, 27, 43, 111, 83, 153, 577, 1537, 149, 231, 839, 8467, 52027, 67135, 222361, 138709, 759269, 458993, 2284069, 1626903, 2750715, 18939819, 13850671, 88906261, 1424337, 510497509, 112422325, 1412426433},
  {1, 3, 5, 13, 21, 19, 57, 69, 87, 163, 271, 3535, 1057, 8517, 15, 56073, 79915, 83583, 87623, 313185, 1414631, 1894415, 3942869, 8902965, 28514865, 59953017, 44384279, 82071183, 350790477, 330419631, 2105707585},
  {1, 3, 3, 13, 17, 49, 65, 45, 457, 241, 391, 2033, 2507, 7771, 25755, 17241, 121821, 210387, 490947, 380923, 1902011, 3626423, 3147083, 4413371, 8786141, 45309351, 119953649, 236933157, 385838613, 394762983, 1897871175},
  {1, 1, 5, 7, 11, 19, 79, 133, 341, 761, 27, 3905, 4137, 14363, 11283, 38091, 14175, 205333, 140253, 951677, 1245761, 774771, 5158711, 13542619, 31003121, 54998863, 4240883, 266808385, 339413213, 580453877, 483703873},
  {1, 3, 3, 13, 19, 1, 11, 139, 249, 245, 1393, 2151, 2857, 1665, 25127, 35017, 45453, 220851, 196289, 183759, 1843729, 2344469, 3218967, 10584079, 13992627, 25136061, 48794399, 248623307, 451397453, 109066495, 1350626503},
  {1, 1, 3, 15, 11, 7, 127, 47, 385, 1007, 713, 2235, 5489, 8755, 16675, 64779, 28697, 108433, 500685, 397913, 1473401, 2856249, 1374359, 1239461, 24525407, 16151825, 36368303, 28296361, 355588941, 258573237, 2064439239},
  {1, 3, 5, 13, 19, 21, 21, 167, 405, 655, 1653, 889, 7367, 4177, 3391, 7865, 107275, 151935, 520001, 702955, 174407, 3752709, 3307779, 15861789, 18538179, 40962315, 120002909, 86451783, 16337221, 481511855, 2015981761},
  {1, 1, 5, 3, 19, 63, 99, 39, 89, 415, 951, 2863, 6569, 3797, 13711, 56779, 62767, 175089, 482765, 737761, 1812757, 3462409, 5738143, 14141957, 470081, 24506937, 53990239, 136565007, 112937325, 852207533, 401657921},
  {1, 1, 1, 13, 31, 29, 119, 35, 311, 839, 1749, 941, 7487, 2385, 14239, 36795, 18507, 158007, 131229, 740795, 1149621, 874853, 2054121, 12558529, 10265523, 6743311, 26818733, 217893239, 207251845, 58590509, 161742021},
  {1, 3, 7, 3, 17, 3, 97, 143, 465, 345, 1457, 2201, 5329, 359, 20939, 3145, 92153, 136341, 104095, 422569, 874895, 1335157, 3683143, 8287043, 18871787, 6000643, 36279483, 149270877, 322346077, 569808199, 1027263171},
  {1, 3, 7, 11, 1, 15, 3, 115, 335, 567, 1749, 1811, 3491, 15939, 1483, 35673, 65753, 195613, 292767, 912261, 87989, 2224973, 5987137, 9939649, 17804691, 16575089, 111406669, 179233885, 508096517, 84207207, 452252867},
  {1, 1, 3, 13, 3, 21, 7, 141, 149, 571, 1877, 473, 2143, 9569, 10495, 20667, 4553, 84275, 401537, 211867, 781557, 3900363, 1194135, 13798137, 28449467, 33015691, 119427813, 121561671, 37777445, 599835053, 2126577351},
  {1, 3, 3, 11, 23, 61, 47, 179, 297, 453, 181, 3405, 2981, 13409, 14871, 28393, 70349, 238453, 416393, 577211, 1117373, 2971493, 6670423, 8424383, 764631, 47781599, 85355527, 76650579, 512478573, 226778303, 673539527},
  {1, 3, 1, 13, 1, 43, 5, 201, 371, 1003, 367, 2709, 7675, 14973, 13783, 50841, 102319, 178807, 30355, 270737, 1921707, 896923, 1077545, 8835093, 6043425, 50483907, 25027641, 152440259, 436343541, 41839423, 1348679749},
  {1, 3, 3, 15, 29, 17, 19, 241, 495, 317, 1135, 2227, 6457, 4783, 21371, 15369, 101181, 250257, 212107, 303491, 669945, 782415, 3281109, 3019499, 19969837, 10919557, 107317191, 216574453, 40615373, 82927495, 1415658311},
  {1, 3, 3, 7, 29, 9, 57, 95, 261, 531, 1717, 3389, 7991, 3793, 25535, 38521, 35533, 101817, 61211, 967451, 938275, 381877, 5077283, 3877441, 4995323, 39723663, 32923781, 23794983, 290004389, 133934191, 403738311},
  {1, 1, 1, 5, 31, 43, 73, 119, 499, 589, 1529, 3337, 4097, 15641, 27511, 39499, 28555, 110335, 212237, 439565, 892199, 3081549, 3689141, 13981267, 18442571, 46732563, 110620235, 37236483, 268983709, 1009311037, 1365869253},
  {1, 1, 7, 9, 29, 43, 127, 91, 243, 979, 1325, 2835, 2787, 9445, 24839, 24283, 49517, 53759, 85271, 521017, 1213201, 1017309, 4614093, 15890557, 1706659, 47488097, 76986989, 97975419, 287921029, 180815869, 1595463875},
  {1, 1, 7, 5, 9, 3, 115, 199, 219, 901, 747, 1077, 3197, 2443, 6307, 59563, 13469, 57939, 179227, 820281, 228397, 1991737, 2647517, 5873051, 2980073, 14432075, 18854287, 104669817, 465918125, 639831381, 2145094211},
  {1, 3, 5, 1, 3, 43, 7, 117, 297, 313, 1043, 1579, 5099, 13289, 6591, 36121, 22411, 33587, 91657, 414261, 1722889, 1616499, 8261291, 469987, 10075065, 43000341, 133988069, 235022647, 98185221, 171636111, 1782480449},
  {1, 1, 7, 11, 29, 33, 15, 121, 131, 579, 317, 1871, 1121, 11653, 32719, 27595, 97133, 247229, 327187, 897587, 751205, 1530431, 2515625, 12363565, 26142851, 65483965, 4031191, 116831903, 167285613, 176783789, 1134235843},
  {1, 1, 5, 9, 25, 25, 43, 89, 355, 1011, 1385, 2901, 6387, 1653, 20071, 35675, 88559, 38523, 488979, 467475, 440833, 2078295, 7019145, 12173309, 3941623, 56484111, 98640369, 185357803, 132577461, 68106077, 2003307969},
  {1, 1, 1, 9, 5, 47, 61, 165, 85, 757, 1397, 1177, 1657, 4899, 19715, 7211, 74619, 103795, 446319, 862045, 1981019, 1702407, 5779287, 2846119, 9858247, 8766699, 53205075, 211902153, 98832701, 324288917, 651472325},
  {1, 1, 3, 9, 11, 49, 15, 139, 261, 613, 931, 1299, 2777, 2835, 32739, 15851, 52281, 152311, 321889, 915279, 564341, 1858685, 2921675, 16759847, 27220769, 33084269, 52692263, 107219849, 136061581, 333194485, 286592071},
  {1, 1, 1, 5, 3, 55, 83, 227, 125, 581, 1607, 1171, 6681, 14463, 31291, 57195, 71259, 187455, 174321, 46153, 1741925, 3544173, 3546323, 8230519, 10728133, 50632293, 81173095, 79145285, 231637453, 714794821, 525743429},
  {1, 3, 5, 13, 5, 55, 3, 247, 493, 155, 1073, 3743, 5719, 4019, 20059, 19289, 39227, 182015, 132727, 392005, 1637237, 1234793, 986635, 10845897, 9951791, 25630789, 64031809, 268376645, 431443957, 823392583, 1245392833},
  {1, 1, 7, 1, 11, 23, 13, 75, 399, 847, 499, 1643, 6977, 3699, 11011, 45227, 84285, 211447, 328817, 447049, 714555, 557497, 8345073, 1154409, 2752197, 49755173, 41955391, 1751993, 117696685, 546277557, 144133443},
  {1, 3, 1, 9, 11, 41, 47, 131, 313, 627, 481, 2469, 3281, 979, 25859, 32745, 113919, 45811, 318705, 267495, 1712261, 1694325, 4611879, 13344329, 7510671, 60037055, 51072691, 180086825, 19109437, 161679191, 584280005},
  {1, 3, 5, 13, 29, 3, 65, 101, 11, 29, 1807, 153, 1487, 16109, 23239, 3929, 8971, 6175, 483887, 475713, 1580767, 2333503, 8119657, 9207635, 32759981, 22720495, 79261893, 127050619, 321762757, 466761503, 1700751169},
  {1, 1, 5, 9, 13, 31, 83, 195, 351, 355, 467, 3871, 3085, 4441, 21607, 843, 12911, 84699, 410535, 733533, 101525, 3838973, 4031761, 9049301, 3001629, 5780561, 47271675, 225730635, 4437565, 853307453, 638886721},
  {1, 3, 5, 3, 19, 21, 111, 179, 143, 361, 1619, 1547, 3409, 6905, 16151, 46889, 27883, 240593, 514493, 266571, 1497261, 3415845, 5007825, 12794239, 2612489, 58470341, 46477023, 158135283, 279666669, 643260319, 608233025},
  {1, 1, 5, 9, 31, 1, 93, 199, 491, 135, 1627, 2559, 1389, 14561, 22551, 49227, 84399, 201563, 370357, 511335, 137895, 1422709, 1567357, 10966409, 23604445, 44883201, 116597307, 169470387, 516968765, 596650717, 248593217},
  {1, 3, 3, 9, 25, 53, 3, 105, 39, 445, 259, 1045, 1129, 9153, 6151, 63465, 125677, 135639, 90019, 420687, 932529, 1409303, 7194541, 7646043, 12462177, 47748891, 84015035, 262613243, 244258045, 789032543, 31566919},
  {1, 1, 5, 9, 19, 63, 71, 9, 73, 435, 1377, 4015, 1821, 6453, 23039, 21643, 62863, 16155, 469561, 133185, 459969, 886255, 3108723, 5175625, 26860191, 56672789, 65951531, 145391319, 502976861, 1042226701, 551941057},
  {1, 3, 7, 13, 19, 13, 37, 247, 391, 23, 1491, 1257, 6395, 237, 8471, 56889, 45513, 73307, 243937, 61699, 729927, 3740949, 3293073, 983425, 30343349, 14617511, 88808961, 99681555, 2320213, 202707519, 2134864195},
  {1, 1, 3, 3, 19, 55, 109, 23, 227, 747, 729, 2221, 727, 2209, 10175, 21531, 60713, 86941, 313965, 56633, 1959691, 2298117, 6106917, 5753429, 89735, 29395927, 42636549, 70754327, 478451813, 257624365, 1765920199},
  {1, 1, 5, 11, 25, 21, 75, 37, 219, 355, 1005, 1895, 7039, 5225, 17807, 14811, 34863, 149273, 27255, 903487, 1670473, 2638699, 5454281, 4454909, 29051667, 12877397, 11982521, 216620463, 49334229, 405536749, 1895833793},
  {1, 3, 5, 13, 11, 43, 9, 67, 87, 797, 1077, 245, 4521, 11845, 29263, 13929, 100363, 198623, 135033, 846597, 1611527, 2540409, 24985, 3400491, 24502503, 8806967, 3731131, 106077327, 294969501, 99202383, 672327105},
  {1, 3, 5, 3, 15, 29, 127, 237, 277, 373, 1859, 3083, 587, 1123, 5259, 36665, 65403, 114737, 343713, 977339, 1762797, 3015035, 2553911, 13777175, 10123793, 65353621, 68007045, 196253085, 125732613, 45257511, 818032705},
  {1, 1, 7, 15, 13, 7, 103, 53, 13, 965, 1497, 775, 3439, 1501, 2919, 61787, 34573, 262105, 267371, 216981, 54977, 2452227, 2180335, 11849655, 4501995, 36707053, 28422585, 157217979, 346383445, 415201341, 868253379},
  {1, 3, 3, 15, 17, 13, 97, 169, 67, 953, 189, 2739, 1459, 10543, 7667, 46393, 80413, 94961, 147239, 144295, 49619, 1511411, 6165257, 1628743, 2572215, 65820385, 114957709, 151135185, 114164037, 1068379479, 1596432839},
  {1, 1, 5, 1, 17, 39, 15, 127, 327, 989, 1471, 3235, 2801, 15311, 23483, 57707, 4895, 143603, 153515, 823965, 114105, 938969, 1789489, 15869179, 19146349, 38956149, 66658895, 42545253, 57373677, 562135749, 1803583297},
  {1, 1, 1, 15, 5, 37, 55, 155, 47, 463, 1851, 3467, 2765, 9359, 32395, 41739, 38683, 157685, 45987, 564119, 1154229, 3166349, 4198657, 13079129, 2158637, 10100565, 130780339, 151034461, 285330461, 688869413, 1292558149},
  {1, 3, 3, 15, 1, 13, 93, 239, 291, 115, 365, 61, 395, 15853, 26583, 41465, 26829, 167889, 340599, 998535, 442863, 657997, 6266597, 115629, 26100483, 43141167, 22140217, 121928259, 173844693, 145565695, 949451975},
  {1, 1, 5, 1, 19, 27, 61, 95, 105, 369, 1557, 961, 6917, 3621, 15631, 62107, 79199, 138883, 343193, 626325, 544387, 174909, 3133839, 7050187, 31839119, 34984995, 43088271, 127810823, 516316621, 1050577709, 9396417},
  {1, 3, 3, 9, 7, 35, 115, 53, 111, 345, 1145, 1687, 3401, 12107, 16419, 34841, 112685, 196743, 241549, 862389, 309757, 1004907, 6275485, 13779103, 5914851, 41535389, 30825519, 138333329, 397706125, 245506199, 417747399},
  {1, 1, 1, 5, 7, 31, 63, 19, 373, 79, 1369, 3037, 2835, 4439, 15475, 39787, 27435, 149903, 190853, 272025, 1491625, 821461, 993579, 11066333, 8159199, 62476615, 57280849, 263072313, 192836021, 120744341, 1178826949},
  {1, 3, 7, 9, 11, 17, 29, 33, 331, 447, 1981, 3333, 6535, 6557, 30711, 17609, 104761, 84879, 62849, 444607, 1982455, 173251, 7212033, 13427777, 17296715, 41990567, 102454005, 223147515, 274425957, 28413599, 1278447555},
  {1, 3, 3, 5, 11, 41, 29, 43, 365, 279, 1919, 945, 179, 1987, 15803, 46697, 98829, 173195, 226201, 578727, 593639, 80585, 4678643, 6668453, 4178057, 21775783, 88920553, 227599837, 432973877, 75105287, 212028743},
  {1, 3, 1, 13, 7, 7, 25, 33, 103, 367, 1267, 763, 5691, 8643, 31579, 25865, 27087, 66183, 221829, 1010305, 870143, 234155, 8250269, 5583849, 8075173, 20101541, 69415221, 220084397, 248398213, 833541095, 933136453},
  {1, 3, 1, 5, 11, 15, 3, 213, 511, 211, 1069, 4047, 3335, 12729, 26943, 43497, 97503, 169231, 361241, 609553, 581941, 1638283, 6690093, 9325621, 23914019, 5103981, 50084609, 166916575, 199523125, 1072437935, 1319940549},
  {1, 1, 3, 1, 5, 11, 27, 201, 361, 537, 679, 3709, 293, 2997, 16143, 14747, 36825, 67343, 346383, 69401, 1284869, 533463, 1894819, 13379859, 28879837, 2455931, 116379671, 63430599, 476133517, 124387245, 713162823},
  {1, 1, 3, 1, 25, 15, 19, 185, 369, 577, 1625, 655, 2363, 3861, 26839, 23691, 61561, 75279, 173587, 120357, 612869, 289367, 7136891, 55867, 3779859, 56081333, 82658349, 8181163, 129696901, 429226749, 391605191},
  {1, 1, 5, 5, 1, 47, 61, 45, 411, 597, 955, 1007, 3775, 5809, 7087, 47083, 75551, 48647, 483843, 577029, 1349195, 1410655, 8223417, 16221835, 8943737, 48000577, 44805577, 96225303, 219921333, 283794253, 95832897},
  {1, 1, 5, 3, 27, 51, 101, 167, 429, 333, 1703, 3541, 2947, 3695, 20515, 26411, 109423, 32769, 229397, 491565, 2064411, 295093, 3506555, 1868095, 32014597, 626163, 74488385, 143633489, 480092757, 1058478965, 793417281},
  {1, 3, 5, 5, 1, 53, 17, 63, 141, 215, 1223, 3129, 635, 15919, 24899, 60969, 23147, 74759, 318483, 967711, 1537107, 1682453, 5133659, 718149, 2998417, 38424031, 27445425, 19908833, 126138581, 189180791, 443478849},
  {1, 3, 3, 1, 23, 31, 25, 11, 195, 241, 995, 3941, 573, 13855, 7739, 18969, 19405, 213839, 241997, 161257, 407791, 1784001, 5917037, 10897535, 7741209, 18660091, 49871551, 205155133, 410395757, 939405895, 1308264263},
  {1, 3, 3, 7, 17, 13, 71, 203, 465, 479, 1857, 1493, 8067, 7113, 5071, 11305, 92445, 51529, 332999, 65143, 1943461, 3337869, 6003567, 1140229, 6029159, 4484127, 88422289, 183355783, 379149621, 920648431, 1257050823},
  {1, 1, 5, 3, 11, 57, 9, 59, 225, 691, 425, 2423, 6031, 6631, 10275, 4459, 46063, 216257, 68805, 294087, 330147, 99249, 1734559, 9322105, 23850327, 45401069, 19193657, 153418481, 203553429, 144405525, 838784705},
  {1, 3, 7, 1, 29, 57, 103, 123, 401, 807, 471, 2759, 5113, 15937, 855, 793, 102969, 216263, 425799, 373115, 383709, 3771677, 107503, 16584189, 24680065, 54597153, 126642447, 79211851, 221544749, 643807807, 465570627},
  {1, 3, 1, 1, 3, 1, 67, 123, 157, 655, 519, 323, 1853, 15041, 6087, 57849, 93567, 5707, 321497, 275407, 1645065, 1832485, 521763, 14748653, 7760385, 46242789, 51374851, 217785411, 141899037, 992606815, 349050693},
  {1, 1, 7, 5, 11, 11, 105, 135, 247, 689, 1141, 2347, 7113, 9347, 26931, 30491, 56749, 233795, 418505, 409445, 578583, 1158221, 4727905, 13569367, 11914223, 24864953, 13218391, 4988281, 183267501, 55853941, 225269443},
  {1, 1, 3, 11, 15, 37, 87, 3, 209, 575, 1521, 3863, 3893, 211, 2467, 46139, 105577, 220741, 361425, 423267, 122885, 2662549, 1925439, 15836133, 25190999, 58534717, 72676643, 24498673, 51356925, 673444117, 1538502343},
  {1, 3, 1, 3, 29, 55, 115, 31, 19, 195, 985, 3275, 363, 9801, 9455, 521, 19935, 221769, 406339, 453573, 1227589, 2824657, 925397, 11442877, 7670503, 24952609, 82415909, 217680023, 520134629, 320590159, 2078886085},
  {1, 1, 3, 9, 13, 31, 57, 251, 201, 275, 1751, 389, 1463, 13159, 8099, 3531, 126761, 223943, 495447, 942685, 1243919, 2999201, 3051015, 10847241, 11294057, 40965795, 107627365, 39141745, 27919365, 68058869, 258871623},
  {1, 3, 5, 15, 19, 51, 127, 255, 397, 243, 29, 3007, 7845, 4687, 539, 62905, 81035, 252749, 372949, 285565, 687905, 2476377, 704939, 9715921, 1099715, 57361453, 113296003, 120283949, 103753533, 7931751, 1838256065},
  {1, 1, 7, 15, 9, 37, 39, 217, 509, 137, 1123, 3361, 6323, 5323, 26539, 8619, 35213, 10185, 399967, 313679, 1016149, 1711519, 5025191, 12389515, 14830329, 51937359, 80974457, 1610229, 209101685, 1010732069, 1742716227},
  {1, 3, 7, 5, 25, 3, 93, 203, 345, 581, 261, 2811, 4829, 6977, 12039, 537, 127129, 143587, 249323, 757545, 1304851, 2116953, 627575, 6814783, 8758967, 12775065, 516963, 38082803, 39803389, 104298335, 844131011},
  {1, 1, 7, 1, 15, 41, 51, 227, 447, 893, 1209, 3865, 5229, 4277, 30703, 39099, 58429, 24935, 480101, 351279, 1416265, 826397, 675089, 12061819, 29680763, 53635571, 123761111, 247926343, 510877165, 1013122861, 1528389571},
  {1, 1, 1, 5, 31, 19, 23, 195, 359, 853, 595, 337, 2503, 16371, 7275, 30571, 127531, 132911, 495933, 647173, 90169, 1957941, 1024217, 1270467, 991361, 51899439, 8612269, 1470981, 376202149, 98839525, 251406661},
  {1, 3, 7, 5, 5, 13, 89, 157, 351, 777, 151, 3565, 4219, 7423, 27971, 23625, 7305, 157091, 51639, 733215, 954379, 2666727, 5836973, 9040863, 24335309, 37248283, 55283785, 227976017, 119974645, 12634103, 104869443},
  {1, 1, 1, 5, 7, 1, 9, 89, 175, 909, 1523, 2295, 7949, 6739, 11011, 23387, 59275, 129007, 359589, 702503, 1222659, 2185299, 7665845, 7383675, 4511745, 11558729, 18219435, 90310193, 66195901, 620110677, 2047075653},
  {1, 3, 5, 15, 27, 17, 11, 235, 19, 105, 457, 465, 3819, 11335, 31635, 65289, 106955, 215469, 515773, 358959, 170545, 1658501, 448173, 15707511, 22371939, 24165819, 59732321, 39374777, 189261493, 902683735, 1701438913},
  {1, 3, 1, 13, 3, 41, 85, 221, 451, 613, 543, 2265, 6831, 1725, 9623, 12457, 79839, 104487, 367841, 155143, 709135, 3669583, 3185297, 2034987, 33437469, 21534303, 18722197, 26016123, 279399557, 245748511, 1736328773},
  {1, 1, 7, 7, 3, 29, 9, 197, 455, 665, 343, 1811, 5395, 393, 22415, 44395, 75581, 116385, 202213, 390019, 1552043, 1908143, 6355909, 4765119, 8498029, 33070629, 123269341, 74879607, 173247141, 638240909, 989793859},
  {1, 1, 3, 13, 29, 55, 71, 95, 475, 615, 2029, 123, 413, 16127, 26683, 37403, 32745, 117603, 419503, 389813, 158689, 1823193, 7330045, 11069641, 4455099, 3527193, 37013299, 210462029, 245836413, 1006729637, 573170631},
  {1, 1, 5, 9, 15, 61, 9, 51, 105, 271, 511, 2801, 693, 11839, 29499, 57595, 57839, 57579, 179381, 667307, 301979, 1155449, 5841223, 5350389, 16868441, 29386215, 27994319, 28651533, 178350509, 905153349, 915915073},
  {1, 1, 7, 13, 29, 9, 105, 59, 377, 635, 717, 4033, 6963, 10541, 7591, 55755, 110237, 98091, 471663, 804011, 1786291, 3377761, 1384039, 15917537, 5890627, 22654003, 93515401, 228395139, 261574933, 161082429, 901466051},
  {1, 1, 1, 13, 7, 13, 59, 17, 335, 355, 77, 3665, 7003, 9521, 11839, 48267, 125691, 113703, 515573, 686763, 1058281, 290047, 3482661, 12181061, 22491427, 33723475, 76184641, 36184463, 52960821, 370692109, 1469709765},
  {1, 3, 1, 1, 23, 43, 51, 209, 151, 365, 1021, 2859, 3937, 2899, 8403, 35193, 21839, 63467, 424749, 199085, 2049677, 3506351, 6062541, 8356795, 16941647, 29525385, 69311999, 14450969, 454404621, 835520503, 230706885},
  {1, 1, 3, 3, 31, 41, 111, 107, 171, 433, 1233, 505, 2971, 6927, 12675, 16075, 57673, 121965, 203505, 41839, 1288261, 174189, 6990229, 7026879, 21470731, 30782115, 80866113, 37768465, 287923381, 259596149, 2061260743},
  {1, 3, 7, 13, 17, 25, 127, 195, 257, 551, 1867, 2145, 3695, 14567, 28515, 20713, 77129, 122347, 27891, 482403, 822533, 1726869, 1871151, 16059837, 10025341, 62414431, 96701749, 255765377, 12909061, 632218359, 679358531},
  {1, 1, 5, 13, 13, 45, 39, 195, 55, 991, 1981, 1043, 5875, 581, 29111, 60459, 34719, 107119, 17599, 957183, 1007989, 831237, 1225017, 3805545, 4838907, 51886193, 81480845, 256039275, 226038917, 371715005, 440960961},
  {1, 3, 3, 11, 25, 31, 91, 153, 415, 449, 1301, 563, 7755, 10671, 27075, 5097, 25165, 148325, 39927, 321109, 958237, 2483015, 3501813, 5362887, 26003199, 53823809, 26759877, 146499889, 27669989, 735474647, 94309063},
  {1, 1, 3, 5, 31, 63, 1, 157, 229, 949, 971, 137, 6589, 8387, 14115, 13051, 31305, 232875, 57981, 270969, 236855, 451399, 3998535, 9903063, 25160041, 17925335, 5278695, 243969793, 365816333, 406184565, 1309171015},
  {1, 3, 7, 15, 25, 7, 89, 133, 73, 497, 1361, 613, 455, 1005, 25911, 29545, 13337, 233577, 20287, 869741, 564111, 71775, 1882331, 2418891, 19110843, 5760239, 3323877, 253621771, 18630437, 709668319, 2053453251},
  {1, 3, 3, 1, 13, 5, 119, 93, 175, 511, 1923, 763, 7573, 7545, 16439, 10265, 49053, 65327, 509623, 198759, 363477, 875771, 4957533, 11507577, 11487733, 1258917, 118114171, 99176299, 507812797, 855879935, 1322524743},
  {1, 1, 3, 15, 27, 59, 49, 205, 497, 485, 117, 2523, 4495, 15153, 28783, 971, 27513, 123489, 17069, 187765, 1606367, 3800535, 357139, 7089583, 27928311, 15412953, 52030577, 65904487, 57599413, 23675661, 1489415879},
  {1, 3, 7, 9, 15, 47, 111, 31, 363, 11, 475, 2931, 6813, 1259, 22835, 6617, 119945, 79151, 98277, 891881, 1135577, 1336089, 5110349, 11067525, 28172949, 26223933, 21692035, 151450409, 7392861, 443191927, 2005329987},
  {1, 1, 5, 5, 1, 35, 95, 225, 17, 991, 809, 2601, 6455, 13803, 31163, 24875, 59951, 7527, 471267, 349673, 1259473, 1295159, 3822251, 1667557, 13963839, 1380435, 25826045, 113130029, 493225221, 1069207173, 370812097},
  {1, 1, 5, 5, 15, 1, 1, 171, 433, 887, 1813, 3431, 2471, 7803, 23891, 39539, 67191, 95167, 473397, 303919, 1653619, 3219289, 1591223, 5207781, 22901451, 59945693, 59653145, 139081573, 435040533, 766396677, 676837121},
  {1, 3, 3, 15, 1, 15, 43, 179, 15, 949, 1881, 1027, 6989, 8955, 26331, 64049, 52949, 159161, 371631, 185629, 155493, 245109, 2820505, 2807931, 24283235, 27583457, 61914331, 103553489, 525079325, 658786199, 1823316231},
  {1, 3, 7, 13, 1, 3, 49, 183, 373, 175, 1733, 913, 929, 1065, 22623, 4881, 68625, 254099, 500715, 480017, 2054951, 3810277, 6825131, 4834645, 1318891, 16656971, 16457779, 56739899, 81810621, 614519071, 199273219},
  {1, 3, 5, 7, 15, 51, 107, 115, 323, 357, 167, 2069, 7541, 9601, 3199, 46929, 69971, 163357, 273905, 793405, 794429, 1846133, 4996277, 3501619, 3584957, 35158027, 131603471, 223886939, 476604077, 813233439, 367572225},
  {1, 1, 3, 5, 5, 21, 31, 107, 21, 299, 1937, 43, 3673, 8155, 27275, 46515, 50321, 123571, 218559, 742447, 1387557, 2765925, 7490611, 12530929, 28705431, 13241945, 45485631, 209848185, 267787597, 299025461, 9908999},
  {1, 3, 5, 11, 9, 55, 35, 113, 29, 99, 161, 1607, 8141, 4951, 2083, 62705, 59219, 38897, 114607, 455605, 1849597, 2354343, 4973711, 9202825, 33463607, 32699637, 8698675, 53652701, 503831677, 517316199, 1296322305},
  {1, 3, 7, 15, 25, 7, 113, 179, 213, 19, 1717, 1027, 2021, 11263, 2291, 41137, 35281, 42865, 397495, 88997, 1858799, 3837121, 2626627, 13073769, 11291363, 55723257, 14734163, 35996453, 333132733, 343759175, 1957954307},
  {1, 1, 5, 1, 31, 33, 85, 111, 67, 95, 2013, 2217, 871, 5329, 14327, 21779, 107511, 215451, 297453, 726191, 2064871, 4176117, 1882573, 14732201, 29859475, 52759747, 61801317, 70196495, 436295653, 360419917, 1091046657},
  {1, 1, 1, 7, 7, 63, 67, 145, 495, 419, 1945, 3437, 6255, 151, 13947, 25939, 74611, 203253, 353593, 5761, 1998989, 155335, 2535133, 10909677, 6824591, 24995503, 57703397, 208454681, 180203557, 885998037, 740433157},
  {1, 3, 5, 7, 17, 37, 97, 187, 215, 399, 1603, 2195, 5923, 769, 8151, 28017, 6515, 349, 223279, 930519, 878171, 2917993, 2605841, 12087185, 31334829, 44707013, 8814165, 194781079, 267130821, 992773711, 1743593729},
  {1, 1, 3, 9, 25, 1, 119, 193, 385, 861, 2005, 2769, 675, 767, 27579, 16595, 81905, 59199, 13819, 503107, 1706021, 772255, 8284275, 7207215, 24483967, 32833483, 72397553, 260184585, 271204597, 705572693, 1914143495},
  {1, 3, 1, 15, 19, 7, 5, 227, 173, 383, 289, 461, 579, 3689, 7671, 1169, 106743, 75869, 446013, 849393, 968507, 3235321, 8190715, 5584117, 14625967, 23205343, 117831465, 51164543, 349690005, 477259087, 1537207045},
  {1, 3, 1, 11, 1, 37, 93, 239, 465, 891, 1479, 921, 4439, 15265, 7063, 16081, 94775, 183001, 17703, 484855, 1425191, 1788165, 3247679, 9787753, 9907457, 19223815, 33406677, 204623967, 516150405, 750181199, 1726648069},
  {1, 1, 1, 13, 27, 61, 99, 69, 279, 655, 1853, 1593, 6319, 9003, 20963, 41683, 61747, 8575, 516209, 494971, 286057, 3274035, 3355853, 6854225, 4526195, 45585971, 63707853, 67899429, 455372613, 284444645, 1452023045},
  {1, 1, 1, 11, 5, 7, 19, 7, 387, 303, 321, 931, 5809, 16029, 4903, 24435, 55251, 22169, 63139, 914429, 947053, 2618353, 7887357, 12699961, 18164367, 33518065, 76566759, 225144751, 201172013, 765493229, 1629490437},
  {1, 1, 1, 15, 21, 55, 43, 107, 217, 687, 19, 3225, 3419, 9991, 8427, 13779, 94195, 172221, 317435, 653421, 1797429, 3431853, 5100031, 1241101, 11375965, 35790255, 42949465, 161182113, 455636853, 598265653, 1153011461},
  {1, 1, 7, 5, 7, 55, 79, 41, 317, 357, 859, 1205, 191, 9395, 30195, 64275, 41013, 50043, 486397, 813257, 2064561, 3899591, 7373055, 13271631, 15106241, 14780211, 100046693, 122927613, 49151557, 50689797, 1137454851},
  {1, 1, 3, 11, 3, 43, 7, 133, 115, 995, 1205, 1055, 4153, 10481, 16319, 3123, 60881, 220125, 438437, 471133, 244609, 2355355, 6688873, 13742741, 25711395, 8216197, 30092247, 168356371, 404707117, 975959389, 1973762311},
  {1, 1, 7, 11, 31, 57, 53, 9, 459, 223, 1969, 3513, 7033, 8505, 31263, 46931, 68309, 187605, 358585, 948343, 1645015, 3316467, 3594761, 1725561, 2215487, 51967787, 5414491, 70156699, 314476765, 212728477, 879401219},
  {1, 1, 3, 7, 17, 11, 115, 255, 281, 97, 1685, 2039, 2845, 11637, 31719, 47411, 123537, 159889, 176943, 40025, 321205, 2852809, 6706263, 13727107, 31422663, 48589325, 16180931, 147109695, 11850429, 854183149, 609396487},
  {1, 3, 7, 1, 23, 41, 69, 199, 53, 105, 657, 1453, 4429, 1101, 18183, 18065, 37521, 90527, 183589, 500823, 880327, 199677, 2236843, 7807155, 16163187, 47359135, 90354535, 267910311, 60645485, 391980975, 1788345091},
  {1, 3, 1, 5, 11, 33, 91, 131, 191, 73, 823, 117, 1053, 127, 29283, 18193, 99863, 247991, 12145, 361703, 2027433, 4063493, 1349993, 6584455, 4549493, 26486155, 105474599, 182902645, 511833485, 866596007, 1231714565},
  {1, 3, 7, 11, 7, 3, 21, 65, 187, 103, 1393, 1797, 6673, 1409, 30607, 42929, 65009, 113525, 306305, 314045, 1010403, 3147099, 7256777, 4474785, 26737775, 35982199, 199467, 107684947, 477060125, 633813631, 1171870979},
  {1, 3, 7, 1, 31, 25, 25, 161, 299, 275, 417, 2267, 6861, 1255, 21059, 59729, 60401, 152863, 366477, 465047, 758235, 1689347, 2810649, 6926421, 4772759, 21655817, 17315915, 89725901, 79374877, 180172519, 1657578755},
  {1, 3, 5, 13, 5, 11, 61, 155, 115, 1001, 747, 889, 3235, 5709, 9983, 7857, 128467, 122103, 358543, 384433, 1503995, 611249, 699797, 10966591, 30398825, 2073759, 29221345, 138221595, 229591125, 814309983, 1863485697},
  {1, 3, 7, 7, 7, 1, 97, 177, 507, 273, 1781, 3455, 5123, 15607, 20027, 26385, 59537, 24281, 370585, 412095, 1229715, 2893187, 1220713, 4924023, 8349255, 63106821, 56194765, 191812705, 197247237, 261603351, 1612428547},
  {1, 1, 7, 5, 1, 7, 59, 49, 147, 343, 97, 3517, 5611, 8705, 26279, 29747, 4949, 87099, 491163, 218517, 975045, 3961751, 342369, 739969, 18414847, 47735791, 1256737, 185064999, 100880181, 584692269, 1593820419},
  {1, 1, 5, 13, 21, 29, 13, 21, 503, 515, 1217, 3905, 5513, 15849, 4767, 48115, 50679, 183031, 420255, 368039, 1557991, 3296863, 1773069, 8609981, 28959895, 6416147, 85975291, 193804875, 38614493, 36514653, 50028801},
  {1, 3, 1, 9, 9, 39, 65, 111, 385, 757, 583, 2225, 2039, 2817, 10231, 47025, 60311, 111035, 394123, 341637, 1024927, 3909373, 2775631, 11183367, 1916221, 63268943, 66106117, 119362879, 443457541, 755074511, 1039631109},
  {1, 3, 3, 15, 23, 17, 63, 169, 503, 949, 849, 461, 6799, 669, 847, 35121, 10389, 218873, 471833, 726799, 1047853, 2411303, 8215765, 595531, 14473419, 6448895, 72230245, 117529043, 264079461, 645326463, 1223930119},
  {1, 1, 1, 3, 1, 41, 63, 159, 251, 457, 521, 1653, 623, 3287, 7451, 24243, 105875, 18513, 129815, 575291, 1549693, 1419729, 3691733, 7137199, 23795563, 18275479, 26186805, 183935385, 15765285, 201230549, 1775210757},
  {1, 1, 7, 3, 9, 1, 41, 37, 441, 921, 1415, 2955, 5841, 1451, 23531, 48531, 88629, 37597, 390943, 80643, 1057115, 645959, 227939, 16516091, 12322085, 38539221, 81532247, 92013209, 298066893, 788841973, 1307361027},
  {1, 1, 5, 11, 23, 29, 89, 185, 413, 357, 1131, 2369, 3835, 6233, 28535, 5555, 4311, 249905, 520977, 489763, 541491, 70075, 6459039, 6895759, 20016177, 44061447, 111754409, 233896191, 255595125, 1041546317, 689799937},
  {1, 1, 5, 15, 27, 35, 17, 73, 315, 911, 1761, 797, 5349, 3219, 30267, 37299, 41015, 160853, 440341, 845829, 1927175, 2758811, 628761, 7953353, 5826799, 29803087, 51567, 189887913, 274741517, 404804629, 1075877121},
  {1, 3, 7, 11, 21, 9, 119, 233, 249, 901, 189, 3625, 2691, 16201, 1271, 3313, 72913, 234549, 79379, 792083, 1408533, 2292407, 4220539, 15474919, 11314887, 16379843, 46837569, 808991, 208389045, 808364431, 2048614147},
  {1, 3, 3, 13, 29, 61, 105, 145, 187, 79, 609, 321, 4289, 3933, 471, 3697, 82933, 161083, 73239, 737847, 852515, 1835667, 4930121, 3878121, 19162735, 29124243, 80211955, 169968687, 141547389, 280420303, 2112812295},
  {1, 3, 1, 15, 19, 63, 13, 185, 115, 219, 1021, 1205, 4273, 11521, 25807, 19313, 54839, 121533, 109661, 835689, 1880995, 941075, 5967345, 11086989, 21009883, 13406879, 84702763, 257525011, 115211613, 731540223, 1834663173},
  {1, 1, 3, 3, 23, 31, 93, 153, 87, 947, 1039, 469, 4047, 8869, 639, 44083, 721, 106805, 178369, 890977, 1737875, 2667987, 5025621, 7520429, 22922233, 53954843, 56738005, 216235979, 305266053, 937333213, 1317689607},
  {1, 1, 1, 1, 9, 1, 85, 3, 15, 995, 455, 2769, 6781, 16203, 3915, 46963, 39283, 115091, 430171, 379843, 182151, 794489, 4616025, 12217389, 29716113, 64094895, 104472047, 197263553, 209589485, 654188213, 1109558533},
  {1, 1, 3, 3, 13, 7, 55, 215, 185, 367, 765, 441, 4497, 1521, 10447, 7603, 29361, 45749, 133467, 808269, 774857, 544073, 3494535, 12463661, 29086867, 66707315, 47587379, 243400547, 410915901, 565110269, 65987335},
  {1, 1, 1, 5, 1, 31, 13, 95, 417, 735, 975, 3407, 4871, 16133, 13519, 45715, 34579, 170039, 251099, 773197, 292579, 2143189, 7929039, 8713389, 5888353, 11772257, 75625033, 63702715, 468463861, 59532221, 556565253},
  {1, 1, 3, 3, 5, 43, 111, 107, 419, 515, 1075, 3597, 1187, 4143, 6043, 42419, 127121, 93589, 366419, 605553, 689225, 3079365, 2536933, 7345973, 33522565, 7835163, 562505, 121614233, 403015733, 822331029, 872399111},
  {1, 1, 3, 13, 31, 51, 83, 163, 489, 887, 863, 599, 9, 13861, 7671, 18323, 125297, 232059, 282325, 897757, 1244741, 3908245, 7338911, 14638805, 14971201, 26832921, 73255159, 109179783, 291722253, 848527053, 2082573063},
  {1, 3, 3, 3, 19, 27, 91, 115, 103, 969, 593, 3667, 1867, 15433, 16455, 19985, 23509, 93941, 203333, 1037517, 331293, 1670709, 3058545, 488567, 30707379, 30439681, 129616221, 74141839, 306484741, 784591855, 203949319},
  {1, 3, 3, 13, 7, 25, 47, 141, 57, 553, 1785, 1709, 7453, 2209, 23487, 63217, 71541, 139963, 242893, 153575, 1785453, 2363075, 1226643, 15628939, 7905179, 58108879, 66140695, 172410187, 102715693, 407070495, 243607303},
  {1, 3, 1, 13, 11, 13, 71, 219, 5, 451, 2043, 1605, 6439, 12203, 5699, 55601, 112855, 90207, 457537, 715115, 13581, 1490245, 2079811, 14093657, 31551437, 18050627, 118616893, 91088405, 221877477, 221591079, 1761900293},
  {1, 3, 1, 13, 5, 57, 61, 223, 401, 413, 321, 1365, 619, 12477, 5407, 14705, 22551, 6847, 41551, 155843, 51999, 3366325, 6094655, 1053551, 7403499, 52730727, 108605841, 157448571, 418708277, 878261343, 532890373},
  {1, 3, 1, 5, 25, 57, 89, 211, 195, 455, 1165, 3979, 6313, 5751, 23731, 57617, 29687, 96855, 351683, 89979, 2090363, 2487281, 6262709, 7030673, 10930131, 48385761, 128043727, 118706325, 281085773, 101684231, 1088051461},
  {1, 1, 1, 9, 31, 23, 71, 145, 89, 285, 1593, 1171, 5685, 15459, 14107, 46883, 33379, 169643, 154733, 873289, 1774601, 883839, 4989979, 11426831, 22781219, 5921217, 43417843, 1540969, 419356221, 605089141, 122692613},
  {1, 3, 7, 7, 9, 41, 65, 251, 65, 137, 1577, 3027, 5555, 2865, 32423, 4897, 39009, 51305, 450599, 692955, 1963571, 4142681, 4633239, 9453291, 26292827, 23402465, 102002485, 232523959, 280760229, 273989391, 1115999235},
  {1, 1, 5, 13, 27, 5, 125, 21, 171, 647, 983, 2921, 6623, 5695, 24379, 11491, 64295, 225767, 339425, 27987, 1526295, 3237391, 972385, 224753, 21110457, 23032871, 54535405, 51949473, 381876069, 365727669, 1373920257},
  {1, 1, 1, 13, 15, 9, 117, 197, 123, 953, 1191, 3657, 5757, 15957, 31847, 36835, 121891, 115887, 517045, 1048567, 1572615, 2121551, 5741841, 4545775, 23008309, 9966311, 40645647, 25977231, 47501357, 440261773, 354360325},
  {1, 1, 3, 7, 29, 13, 5, 175, 395, 127, 679, 255, 6055, 7639, 18235, 18019, 29537, 65633, 420467, 738263, 1681663, 1117701, 3272309, 9516681, 3113273, 21203665, 74417785, 90410169, 169584693, 44955829, 1853930503},
  {1, 3, 7, 15, 15, 51, 77, 147, 319, 147, 1775, 3983, 3175, 5723, 23075, 1377, 19297, 11361, 157681, 486605, 1254027, 3801769, 4922297, 1317197, 13827049, 54802097, 43761113, 134454581, 266202293, 790734215, 1497459715},
  {1, 3, 3, 3, 7, 11, 119, 41, 43, 153, 975, 679, 3081, 10359, 12675, 16737, 59237, 193445, 9825, 827365, 277185, 364263, 6015833, 9952111, 27512285, 10896537, 50973223, 120672965, 486744589, 336690759, 897664007},
  {1, 1, 5, 13, 3, 7, 65, 67, 63, 399, 1561, 2789, 2083, 12289, 14775, 8611, 67111, 5607, 59833, 683105, 1202607, 1092769, 8074457, 13651025, 25159591, 44575415, 9171529, 243631079, 189590037, 869781933, 1011121153},
  {1, 1, 7, 3, 19, 53, 103, 67, 35, 865, 161, 93, 2533, 3851, 9979, 41123, 67109, 107309, 148725, 891635, 1668845, 3137709, 7709061, 8233479, 7546243, 3418431, 91742847, 232119265, 16358797, 852562101, 184415235},
  {1, 1, 1, 11, 31, 9, 29, 189, 199, 817, 1571, 395, 345, 3777, 21775, 15843, 78819, 119977, 253865, 682839, 1108719, 1583463, 83741, 15726623, 7673889, 45081261, 75790127, 99592787, 98678509, 590226269, 909605893},
  {1, 3, 5, 11, 31, 3, 9, 67, 277, 735, 181, 2777, 3009, 7233, 24223, 14881, 20131, 134305, 480937, 249181, 735919, 759217, 7932723, 6485589, 30476331, 44167379, 66715859, 212218915, 448143933, 76986879, 1314290689},
  {1, 1, 3, 3, 17, 7, 17, 3, 375, 933, 237, 3919, 5409, 3355, 14491, 22051, 93025, 254629, 185591, 324549, 1899007, 209281, 382033, 4792107, 30735751, 25279221, 58304799, 41034489, 380000077, 261795701, 1212136455},
  {1, 3, 3, 5, 9, 27, 19, 77, 221, 3, 1965, 309, 3001, 15977, 16079, 6945, 108325, 205219, 382883, 427369, 1167493, 2229067, 5186727, 517977, 21101323, 50810703, 127423539, 80772987, 248089789, 281728095, 1854624775},
  {1, 1, 5, 1, 3, 33, 35, 133, 37, 709, 627, 1705, 2525, 4307, 24043, 11683, 60263, 223339, 227745, 173831, 190273, 536931, 705551, 6512623, 12647129, 31079215, 30391087, 50150089, 366048237, 540504341, 383702017},
  {1, 1, 7, 3, 25, 21, 105, 55, 375, 681, 881, 1299, 5879, 459, 19971, 62435, 38565, 34477, 39231, 279815, 1361251, 3922181, 1816697, 15178343, 27147139, 4519377, 104889137, 213606341, 97745333, 325545669, 687815683},
  {1, 3, 7, 1, 13, 7, 113, 103, 313, 515, 1041, 3683, 4619, 5093, 31151, 33889, 28577, 121647, 90607, 250685, 1825567, 791909, 4521687, 7639249, 24728055, 52148065, 27516205, 109739475, 533778693, 229449759, 1442237443},
  {1, 1, 3, 7, 19, 43, 83, 37, 39, 133, 1759, 1171, 1521, 13717, 8007, 47651, 39457, 175713, 344701, 927789, 71717, 2124003, 3942541, 6978219, 18152117, 54405965, 120758391, 208192151, 214187213, 482666317, 290194439},
  {1, 1, 7, 13, 7, 35, 15, 155, 293, 1001, 157, 3883, 405, 1797, 1175, 47203, 76709, 215011, 111997, 391693, 628329, 2746981, 3361715, 4681539, 7570735, 59344369, 111577763, 69704335, 401452477, 91271405, 1714435075},
  {1, 1, 3, 3, 13, 19, 125, 49, 333, 387, 339, 1815, 4503, 7359, 14059, 11555, 109217, 175525, 15019, 1015337, 1427067, 1255083, 4298575, 14908601, 9514549, 57671697, 68164505, 72357937, 152057333, 602977429, 99621895},
  {1, 1, 3, 13, 19, 19, 105, 225, 151, 27, 1251, 885, 4815, 7863, 31371, 28579, 48481, 37803, 299561, 138645, 69103, 3473363, 4354933, 7924181, 32540981, 15953171, 114253957, 235130953, 429978373, 513689173, 745417735},
  {1, 1, 1, 5, 7, 59, 17, 145, 77, 117, 1355, 1429, 2301, 16177, 4687, 8483, 49507, 127975, 441965, 81045, 499399, 1870099, 984655, 5172111, 5016385, 1012131, 86645111, 5636843, 191526157, 565858653, 1898105861},
  {1, 3, 3, 13, 5, 31, 119, 167, 459, 727, 1799, 2537, 695, 13637, 20527, 12321, 30181, 98731, 159871, 204725, 804521, 248745, 4113829, 1056609, 15335625, 3279495, 37356293, 76159435, 334909637, 1047678495, 194925575},
  {1, 3, 3, 3, 27, 51, 107, 85, 267, 57, 1279, 823, 6247, 3603, 27059, 10913, 32805, 22053, 401085, 836517, 1300973, 1534307, 7029057, 16171983, 12599405, 35866921, 27612713, 222845245, 247100213, 21657255, 1214473223},
  {1, 1, 7, 15, 29, 17, 67, 197, 215, 465, 109, 3461, 5269, 15287, 19331, 36835, 94693, 183777, 147875, 366475, 1959617, 1605155, 1505853, 14654047, 10259567, 62328255, 60230079, 49753085, 531054157, 379916485, 47526915},
  {1, 1, 3, 5, 11, 15, 123, 53, 293, 797, 1105, 1777, 6509, 217, 26207, 17251, 79969, 173027, 29281, 284601, 1774021, 3401251, 1851247, 8291255, 5407499, 59540147, 48013231, 101891363, 83394765, 616292733, 462750727},
  {1, 3, 3, 13, 3, 5, 109, 53, 203, 693, 871, 135, 369, 11149, 2007, 23969, 9509, 74731, 473593, 329123, 460167, 1734671, 3340417, 13290179, 28944429, 63140457, 9784415, 182873743, 377390317, 786682735, 1983482887},
  {1, 3, 5, 15, 17, 43, 81, 235, 119, 817, 1777, 261, 8049, 4251, 18571, 49761, 60643, 114533, 147247, 75689, 392935, 3699817, 7366081, 14536575, 14945907, 23669319, 6846875, 178158617, 426731037, 165540567, 1628007425},
  {1, 1, 3, 7, 7, 13, 87, 99, 481, 931, 1507, 651, 5267, 8281, 2135, 62627, 100769, 168097, 184297, 212131, 238061, 228397, 908615, 5895425, 2903173, 8010797, 15568741, 36801247, 82090245, 465786093, 1316561927},
  {1, 3, 1, 13, 27, 43, 77, 225, 341, 163, 933, 429, 4943, 7781, 27311, 37057, 119111, 172879, 457665, 250205, 349147, 882747, 4757847, 12863945, 29407907, 7408039, 35327701, 247410235, 12393381, 763568095, 117182469},
  {1, 1, 7, 1, 1, 49, 85, 211, 449, 479, 1395, 787, 5653, 14891, 5195, 25539, 88261, 219343, 463171, 526067, 69367, 1510585, 4232487, 10022301, 21457421, 38881589, 6867443, 186414145, 32191965, 561598485, 1126445059},
  {1, 1, 5, 9, 25, 13, 49, 85, 125, 85, 1281, 3365, 4305, 11791, 20163, 2883, 116295, 126787, 213579, 1031679, 1693419, 2584311, 2974487, 15098479, 11003243, 12083943, 75149915, 246976621, 22519069, 162660101, 2016456705},
  {1, 3, 1, 13, 3, 31, 117, 39, 43, 151, 663, 669, 1571, 5207, 5211, 16065, 39623, 224335, 390105, 234713, 1812875, 2282381, 3352229, 8703489, 22617849, 43956979, 28632281, 9789009, 261241525, 137281335, 840339461},
  {1, 3, 7, 15, 17, 7, 79, 163, 37, 841, 1799, 1787, 4501, 3785, 16463, 43329, 93377, 171969, 109135, 979301, 1949313, 2347389, 7157283, 13470275, 13855573, 63084725, 21687075, 75902883, 111029277, 334646303, 1756098563},
  {1, 1, 3, 9, 1, 23, 67, 191, 449, 931, 1521, 2705, 887, 7037, 13791, 55235, 15681, 34639, 109779, 839253, 360621, 986617, 6623231, 1478881, 25687927, 21781299, 92565013, 246855379, 293458405, 273087805, 1863462919},
  {1, 1, 1, 1, 5, 13, 55, 161, 419, 577, 1703, 2589, 2651, 2873, 20775, 13507, 71235, 94659, 512839, 252615, 1995373, 85647, 5757601, 5920583, 13897061, 35813907, 50646433, 78668239, 52689301, 875996365, 1207152645},
  {1, 3, 3, 3, 5, 19, 37, 169, 69, 1003, 1755, 3101, 1469, 8583, 28483, 36673, 59461, 38853, 229571, 663641, 675907, 1880099, 6533591, 9687969, 29478989, 42628459, 104473799, 234492093, 230059277, 175045383, 1985638407},
  {1, 1, 1, 1, 11, 33, 105, 79, 283, 91, 299, 835, 3193, 5593, 6943, 13635, 62787, 124739, 48585, 2295, 1423723, 760285, 745157, 11075885, 3211357, 10553941, 130095551, 204024299, 109272429, 309823677, 1058557957},
  {1, 3, 3, 13, 25, 21, 81, 213, 465, 475, 331, 457, 61, 9511, 23203, 10817, 121669, 74699, 233411, 564071, 1128251, 3294647, 3335611, 15740473, 30161853, 65675039, 53283567, 198121757, 209652141, 288190663, 2124247047},
  {1, 1, 3, 11, 1, 11, 77, 95, 455, 949, 1999, 1833, 1275, 5631, 3979, 12995, 11585, 33229, 378455, 509513, 1536283, 2433413, 5622969, 14983163, 25989093, 3468663, 107445481, 86561993, 493593781, 169417621, 314912775},
  {1, 1, 1, 1, 15, 25, 51, 137, 275, 451, 1179, 3595, 5177, 7105, 27183, 8579, 6531, 165251, 212365, 775303, 1214593, 3555183, 2713985, 11860581, 18357917, 45935037, 5202863, 125606707, 432046637, 410463581, 564183045},
  {1, 3, 3, 3, 3, 59, 79, 143, 393, 583, 349, 3039, 7079, 14245, 22111, 63105, 16773, 49285, 394629, 603581, 481465, 3628433, 6247959, 502229, 27326803, 33719781, 119245977, 211780203, 529981333, 323164095, 455950343},
  {1, 1, 7, 9, 21, 11, 123, 105, 53, 297, 803, 4025, 5421, 14527, 22611, 27779, 50821, 162695, 69255, 413857, 437741, 3928863, 7585639, 13198787, 24134897, 10207359, 251103, 199060069, 218588333, 414004645, 622149635},
  {1, 3, 7, 11, 21, 15, 103, 109, 311, 321, 1217, 2777, 5457, 1823, 11075, 12417, 5505, 104325, 92035, 388005, 1737977, 1061923, 3708013, 9809071, 5005251, 36290011, 31617891, 80599613, 448450557, 695892359, 1386692611},
  {1, 3, 5, 11, 19, 31, 79, 89, 295, 413, 817, 499, 3699, 14411, 7347, 57473, 3971, 165761, 63109, 906425, 1346801, 3686471, 5226517, 12665171, 31868299, 36359697, 26133013, 145277741, 50495653, 430836327, 1330626561},
  {1, 1, 1, 5, 11, 3, 81, 13, 315, 841, 1543, 411, 6883, 6347, 10835, 2179, 18307, 197255, 451201, 29333, 1642199, 1448495, 6982561, 11770815, 19257365, 38015509, 15370837, 92965773, 216441669, 54717349, 628342789},
  {1, 3, 3, 11, 23, 43, 23, 131, 17, 517, 995, 2687, 7443, 15085, 4415, 27393, 55557, 224525, 420609, 763653, 641857, 787885, 8246807, 5278503, 9583337, 35244217, 105042157, 124870091, 254131685, 439491711, 511639559},
  {1, 1, 1, 1, 11, 57, 73, 9, 123, 905, 1763, 1789, 3701, 7131, 25819, 45315, 86787, 24835, 248585, 509231, 1957179, 353179, 3691369, 12650623, 19806961, 4698171, 98284175, 659505, 472797293, 796298165, 102432773},
  {1, 1, 3, 5, 9, 53, 99, 229, 43, 207, 625, 1583, 6727, 15249, 13815, 45315, 75521, 187139, 23811, 103207, 590601, 1223971, 3042017, 4900761, 26697455, 19369421, 131502361, 131900447, 364890069, 236643821, 1390870535},
  {1, 1, 7, 7, 17, 39, 91, 1, 297, 711, 225, 513, 7391, 291, 5635, 58883, 45573, 16905, 204319, 782853, 1307157, 1912503, 1996587, 3503765, 18486127, 35461571, 133634781, 79369885, 85290565, 117805061, 1551630339},
  {1, 1, 7, 11, 7, 55, 111, 129, 423, 521, 1807, 3015, 1449, 12321, 847, 64003, 17925, 225797, 181777, 344633, 1817089, 783967, 106149, 12227399, 12546333, 34094041, 77976615, 78177651, 47136013, 191615645, 1066663939},
  {1, 3, 7, 3, 13, 9, 125, 187, 11, 485, 647, 275, 3495, 11989, 5951, 37889, 41985, 130061, 424971, 697363, 928879, 2399297, 4418941, 11527667, 7605581, 33476637, 57310081, 140743067, 81458837, 1053795967, 359661571},
  {1, 1, 3, 11, 11, 25, 49, 33, 361, 105, 271, 3841, 4837, 2437, 30181, 49153, 16385, 212995, 475147, 245771, 49177, 1097777, 4898849, 11452777, 7815273, 55034127, 54120193, 185848549, 431737221, 183678437, 1879048193},
  {1, 3, 5, 1, 27, 15, 119, 35, 159, 273, 1489, 3157, 5433, 3337, 26859, 22529, 129027, 79877, 104449, 620571, 985103, 1378423, 7546915, 15091871, 14731537, 28302801, 82478165, 89951545, 256156937, 951443691, 1740636161},
  {1, 3, 5, 13, 23, 31, 97, 145, 41, 605, 1455, 59, 5389, 5527, 14447, 22529, 104451, 10245, 272397, 75799, 1509407, 3971169, 571537, 10373161, 15637085, 29072815, 86779963, 70167821, 244342167, 410513519, 1178599425},
  {1, 1, 7, 9, 7, 41, 61, 193, 353, 879, 1805, 581, 5447, 11177, 7331, 21505, 74753, 193543, 521225, 363527, 427049, 869437, 2665665, 14407009, 14278511, 46461709, 17667653, 128448839, 314884009, 329371811, 298844161},
  {1, 1, 7, 11, 29, 19, 55, 207, 361, 759, 63, 2255, 2119, 14671, 21783, 50177, 107521, 173063, 277515, 586781, 467987, 1336375, 6161615, 1336681, 16533239, 11525183, 67003599, 154238023, 33000783, 473684247, 1840250881},
  {1, 3, 1, 13, 17, 7, 73, 179, 103, 23, 917, 1205, 4925, 1691, 5419, 43521, 11779, 202241, 198157, 999953, 524807, 4129353, 5467827, 12324455, 2845207, 35425685, 73932469, 121708861, 395541659, 478266155, 137625601},
  {1, 3, 5, 3, 15, 3, 9, 109, 227, 861, 867, 3529, 1535, 489, 22873, 34305, 17923, 236037, 294403, 707087, 169475, 2384393, 2066029, 5263075, 11095389, 35842403, 96101321, 218820607, 517921769, 135846745, 889978881},
  {1, 3, 3, 9, 15, 15, 95, 193, 385, 997, 1525, 1865, 1425, 4079, 14771, 16641, 259, 156931, 95497, 51471, 135439, 2125663, 5845953, 8487041, 32735461, 10454773, 47287881, 33655441, 98992367, 1018619571, 2041774081},
  {1, 1, 3, 5, 5, 29, 49, 171, 171, 623, 1167, 3743, 1809, 12009, 7043, 2817, 19201, 134915, 323333, 884485, 1093917, 641329, 2696107, 5546411, 25994095, 51210127, 129624991, 104702481, 4543465, 262720131, 2139291649},
  {1, 3, 7, 5, 23, 11, 87, 183, 299, 555, 1857, 489, 3505, 9161, 28763, 20225, 78595, 152327, 375045, 127255, 829707, 2064727, 7230391, 4048939, 15239467, 34281537, 72994025, 105104049, 331567305, 430231387, 177930241},
  {1, 3, 5, 9, 19, 21, 85, 127, 337, 439, 1183, 1891, 1877, 4373, 10451, 2817, 42243, 43781, 523017, 171795, 1067285, 801109, 3991423, 7615569, 3255991, 35792287, 83632739, 109305941, 105873429, 109892051, 1831010305},
  {1, 3, 7, 13, 27, 17, 29, 83, 463, 385, 1167, 3453, 4523, 4759, 9321, 10497, 76035, 188167, 511245, 893723, 299281, 2851101, 5965651, 15078095, 9977985, 26592655, 10964605, 239685803, 528938391, 987224937, 1388642305},
  {1, 1, 3, 7, 21, 59, 65, 83, 177, 763, 317, 2913, 7527, 5967, 17167, 257, 124673, 182531, 97031, 19221, 464187, 2117953, 2416979, 15790001, 7180795, 26857533, 14244449, 28096103, 45577807, 533505039, 1402142721},
  {1, 1, 7, 15, 13, 27, 49, 35, 253, 101, 1699, 355, 2181, 10859, 24221, 47361, 119553, 231175, 33039, 613645, 1006363, 985393, 7750435, 16565245, 23391077, 36862883, 38942819, 46539653, 519873387, 954855837, 113836033},
  {1, 1, 5, 1, 17, 17, 81, 91, 349, 655, 1373, 2225, 945, 899, 31801, 56577, 85761, 112901, 251649, 972561, 1050385, 3392849, 6216539, 9281117, 198031, 11003485, 22914993, 19031217, 127587459, 137079609, 1932328961},
  {1, 3, 7, 11, 5, 1, 81, 53, 215, 587, 167, 4045, 5671, 5597, 13529, 56065, 69891, 170247, 222475, 636677, 2054401, 3276625, 6977333, 2858967, 438091, 15117735, 26619597, 1679655, 82160349, 183880665, 133496833},
  {1, 3, 5, 15, 1, 9, 59, 235, 315, 195, 909, 2237, 505, 10415, 28145, 7553, 123267, 214661, 253583, 31873, 1103753, 2848955, 8077419, 7159995, 29540931, 2737165, 73451581, 144131449, 32240943, 190371953, 945864705},
  {1, 1, 1, 3, 9, 31, 41, 43, 275, 921, 25, 671, 5737, 11241, 4193, 24961, 119937, 58497, 161667, 134281, 253087, 4069545, 4364459, 13871507, 11692825, 58288537, 104394271, 108196329, 32298857, 5307873, 1440759809},
  {1, 3, 3, 13, 29, 13, 95, 213, 317, 995, 1489, 3779, 3043, 8569, 28823, 19073, 44419, 236675, 308621, 990621, 1824909, 747487, 1846357, 16221373, 2489955, 14952273, 108424259, 142418275, 243882745, 530304535, 1114685441},
  {1, 1, 7, 5, 9, 49, 125, 241, 87, 153, 1673, 3849, 7253, 1715, 11627, 47489, 116353, 254599, 515717, 450697, 25521, 1210365, 5738097, 8904407, 12073241, 33914377, 20901257, 194242261, 81637939, 368164843, 20201473},
  {1, 1, 3, 9, 27, 27, 19, 223, 63, 463, 1095, 1395, 6643, 11589, 2145, 37249, 102785, 103811, 69513, 499611, 502939, 491155, 2867807, 14142143, 22362703, 3046087, 67106547, 162070131, 497482437, 1033685473, 862732289},
  {1, 1, 3, 15, 21, 17, 45, 23, 357, 11, 1307, 1791, 2481, 2123, 24341, 11393, 8065, 105347, 97679, 279189, 1390481, 3249325, 2855575, 2246629, 32315275, 46023835, 35114623, 181818161, 295487691, 19325589, 815022081},
  {1, 3, 5, 15, 31, 53, 117, 51, 433, 193, 1239, 3329, 2403, 12745, 32219, 13953, 17027, 259461, 505487, 789407, 1460917, 1613813, 1150643, 9413169, 20331585, 64958807, 125883521, 66890211, 278813001, 496599643, 828162049},
  {1, 1, 5, 9, 7, 27, 9, 115, 417, 579, 83, 173, 4717, 15665, 27463, 33665, 73857, 2693, 517513, 736135, 988571, 3682953, 5990387, 13180961, 1171907, 27025875, 92858925, 149895405, 501498545, 70433735, 2137473025},
  {1, 3, 5, 7, 9, 9, 31, 35, 249, 567, 331, 905, 5101, 14817, 14255, 64385, 7811, 40581, 60295, 402313, 962953, 2331295, 1334179, 7324793, 26542007, 60867531, 38063113, 184011373, 2515809, 719772975, 1690288129},
  {1, 3, 7, 3, 1, 61, 29, 129, 119, 421, 1597, 2987, 3041, 7629, 23451, 61057, 61827, 253831, 283779, 307073, 1315773, 4099229, 6995713, 7607543, 5803301, 46638269, 22094891, 136898657, 534217293, 288416283, 808173569},
  {1, 1, 7, 9, 13, 1, 99, 105, 107, 509, 989, 2259, 1009, 6827, 8903, 8321, 82305, 81799, 516233, 890765, 526465, 1588451, 7506921, 5101035, 21075837, 52415581, 57457491, 76630641, 187000363, 716749127, 1991720961},
  {1, 3, 5, 15, 11, 29, 85, 29, 265, 105, 2035, 3349, 3543, 13903, 10213, 33665, 96131, 70277, 113551, 460939, 494237, 1393621, 3754909, 3677065, 32306153, 16282483, 127655317, 83527255, 110929871, 617191013, 140099585},
  {1, 3, 1, 1, 25, 19, 53, 139, 467, 485, 491, 3067, 7353, 13861, 25819, 47233, 1923, 9857, 420993, 319897, 1412499, 232117, 3358987, 1681235, 19116133, 52377195, 19256443, 196197945, 301030821, 29655643, 115326977},
  {1, 1, 5, 3, 3, 43, 41, 185, 45, 463, 351, 2823, 2519, 6705, 11395, 46465, 94337, 57989, 277123, 641667, 1477035, 3585705, 6408505, 14987181, 32938063, 59600863, 47094663, 231792983, 517192625, 294446851, 1774141441},
  {1, 3, 7, 13, 11, 15, 87, 221, 427, 673, 1631, 599, 3259, 10691, 31283, 29313, 62851, 66695, 25229, 907403, 1446031, 194775, 7336285, 385835, 31835169, 48336351, 125112535, 194522683, 278546243, 515295667, 1099939841},
  {1, 3, 5, 11, 9, 9, 15, 49, 275, 335, 1613, 3587, 5309, 14849, 26475, 23489, 72515, 243909, 223051, 55369, 1368777, 3102799, 6981873, 9541715, 4004751, 61541389, 97023043, 131260285, 62811713, 437906347, 1656008705},
  {1, 3, 7, 9, 29, 13, 79, 225, 381, 781, 1411, 2761, 7157, 14983, 19717, 53057, 107715, 155207, 209225, 699869, 2078029, 1833999, 2579489, 7566525, 24472909, 2485955, 5098889, 219106357, 61791943, 525790789, 1264881665},
  {1, 1, 7, 11, 29, 25, 117, 183, 101, 651, 653, 3157, 445, 14389, 23293, 10433, 11585, 261959, 252235, 735069, 1340761, 4155957, 7461623, 2689957, 6877771, 65435597, 63960085, 65616637, 320511989, 176071101, 257994753},
  {1, 1, 1, 3, 5, 33, 73, 155, 473, 387, 591, 2045, 5965, 16299, 31499, 36545, 96705, 39233, 334147, 647365, 1584353, 2225673, 7121627, 3077017, 26449731, 10697999, 12624317, 136654605, 521631595, 204841803, 1393569793},
  {1, 3, 1, 7, 11, 33, 29, 21, 491, 937, 729, 4075, 975, 2461, 18991, 3905, 27971, 181185, 31431, 613451, 2061537, 4087133, 7761877, 11829419, 26968425, 23167257, 82870443, 131633039, 16100829, 441888239, 277049345},
  {1, 3, 7, 15, 29, 39, 105, 111, 173, 943, 69, 295, 8175, 13037, 26131, 54465, 52291, 161479, 331471, 68701, 564199, 511273, 3699631, 15655021, 26420079, 42835973, 8185831, 37599407, 271150637, 78152787, 469757953},
  {1, 1, 5, 15, 7, 5, 97, 147, 105, 887, 443, 2595, 5889, 10753, 1619, 18113, 106433, 255941, 45903, 531271, 1146309, 1399073, 7916499, 4999209, 4606391, 19324283, 107924579, 81772609, 488203201, 791987475, 1713893377},
  {1, 3, 3, 15, 11, 45, 87, 207, 353, 909, 1847, 323, 2283, 12885, 16415, 12865, 49859, 102339, 141263, 100427, 998125, 1016599, 5813519, 9862817, 11157837, 8798455, 74203907, 233031723, 520518293, 709983, 1724903425},
  {1, 1, 5, 3, 19, 33, 43, 79, 115, 653, 359, 2873, 4609, 12439, 6339, 37441, 3265, 93509, 111939, 735955, 105441, 421227, 1238287, 14847539, 10959181, 13015719, 59474553, 160007617, 12225623, 719779075, 1012084737},
  {1, 3, 7, 9, 17, 61, 49, 227, 291, 69, 1753, 3899, 483, 3187, 29041, 43201, 72899, 182855, 499017, 726353, 1502717, 2121969, 5990563, 6818147, 9587077, 35544601, 52228347, 105626403, 533038131, 978096177, 35934209},
  {1, 3, 5, 3, 25, 35, 61, 211, 393, 199, 691, 1779, 6295, 13371, 15817, 19777, 34627, 97605, 513091, 294233, 1378275, 4070653, 1054867, 4340041, 28072583, 12438899, 60132019, 78618199, 234468987, 1010871817, 332722177},
  {1, 3, 7, 5, 7, 23, 37, 91, 245, 915, 579, 867, 6193, 1063, 17363, 53825, 130371, 206151, 358853, 44359, 1473367, 2459109, 4342811, 9378997, 18468179, 6653187, 29198627, 251430641, 445092967, 662556435, 435982337},
  {1, 3, 7, 7, 23, 51, 41, 63, 375, 3, 159, 1889, 4419, 1687, 17977, 36673, 88643, 156615, 361543, 389207, 221683, 192745, 2978943, 2020663, 22736451, 35994335, 80942113, 190158467, 502237655, 706777209, 1793372161},
  {1, 1, 1, 7, 13, 11, 53, 43, 317, 325, 1749, 2423, 4123, 8595, 20773, 17857, 90561, 240961, 235335, 992077, 1822411, 3028213, 4970219, 15506429, 25285125, 66376725, 45596471, 185149147, 395418579, 1015721445, 1164455937},
  {1, 1, 7, 7, 9, 9, 61, 113, 437, 213, 1407, 645, 4345, 807, 30411, 34241, 75585, 214215, 370375, 424777, 580425, 2004989, 7909041, 15873781, 16113813, 26339007, 29895237, 108803385, 516518887, 984469771, 156774401},
  {1, 3, 3, 11, 17, 39, 17, 113, 391, 385, 581, 2023, 7449, 10153, 22033, 39361, 48067, 19779, 4043, 763089, 263911, 1659601, 6244913, 10439367, 22489281, 2829701, 94808231, 95331545, 277653865, 517011793, 514355201},
  {1, 1, 3, 5, 29, 31, 101, 215, 379, 377, 1113, 2855, 7147, 14377, 25515, 38337, 118977, 102211, 112453, 518621, 1381471, 569253, 1338903, 5210939, 27631545, 3282841, 25505895, 235772715, 213680489, 956148971, 897060865},
  {1, 3, 5, 5, 13, 3, 121, 125, 227, 969, 11, 1115, 5657, 9209, 6117, 44481, 108611, 233797, 435653, 889037, 1257411, 1952953, 1922621, 10776099, 31512073, 65749963, 13878299, 205182937, 58576441, 88794021, 1637388289},
  {1, 3, 7, 15, 29, 17, 33, 123, 317, 301, 749, 1365, 5619, 605, 1613, 48065, 88003, 81351, 335951, 813277, 19409, 1346785, 5000763, 13657981, 3944301, 9336877, 108434837, 200728371, 499499549, 154069645, 1906847745},
  {1, 3, 1, 15, 7, 53, 125, 249, 219, 655, 105, 2825, 1649, 12783, 19777, 34625, 14019, 27073, 385487, 334407, 370421, 2011069, 5210553, 16285723, 4227791, 37193513, 113871561, 123045041, 299533359, 111409665, 871182337},
  {1, 1, 7, 1, 25, 53, 19, 53, 157, 373, 1855, 495, 5065, 9465, 2313, 32449, 86081, 186695, 205505, 658905, 1773557, 1631699, 291317, 11438301, 5606325, 5443327, 58116399, 87678089, 131329337, 24768457, 460673025},
  {1, 3, 5, 13, 3, 57, 57, 161, 431, 415, 1859, 1033, 6349, 1577, 31579, 50497, 66627, 231621, 473293, 879043, 267129, 139513, 4673505, 10830959, 15847263, 25370627, 132033225, 253295885, 488480745, 696816923, 418844673},
  {1, 1, 7, 5, 23, 63, 29, 221, 13, 965, 1997, 2265, 1583, 10491, 9551, 24257, 49345, 203207, 17989, 752727, 2026879, 233693, 957213, 1635405, 1311749, 1596301, 9512345, 215858671, 385704507, 369148303, 1871384577},
  {1, 1, 3, 13, 31, 25, 23, 61, 285, 5, 2005, 879, 795, 13299, 19685, 31937, 110785, 224835, 200269, 840799, 271833, 2402263, 6779901, 9624157, 5411269, 20191125, 91779503, 165871451, 6545459, 730456101, 543002625},
  {1, 1, 7, 1, 21, 45, 121, 89, 263, 543, 1333, 2711, 219, 10823, 26139, 34625, 8129, 26183, 409153, 571733, 2041965, 2884793, 982425, 14180167, 15747807, 12774901, 68549719, 12797723, 234254215, 566399067, 10702849},
  {1, 1, 3, 3, 27, 13, 19, 117, 161, 457, 1541, 295, 4953, 12125, 14503, 43169, 97505, 238371, 455587, 576763, 1559597, 697395, 2591765, 412865, 28468905, 46918565, 117997383, 204474873, 498089853, 665863559, 611365889},
  {1, 3, 5, 3, 7, 63, 13, 247, 439, 681, 977, 2537, 6923, 10323, 7349, 50721, 117027, 195941, 235107, 357031, 737311, 2474093, 2474071, 10633687, 24920649, 31605809, 27969609, 148686315, 152395443, 93325717, 1080321},
  {1, 3, 5, 9, 3, 51, 81, 251, 349, 983, 581, 2515, 2281, 2849, 31915, 62625, 110179, 90405, 204841, 530787, 1110035, 1183793, 8122459, 12996733, 4174775, 53867557, 75944947, 211363721, 176353409, 120616523, 714099713},
  {1, 3, 5, 3, 11, 63, 47, 137, 303, 627, 91, 2269, 7097, 2145, 31059, 12129, 122787, 174885, 30755, 663147, 563231, 907343, 3794985, 6605135, 31197971, 38379451, 63964477, 246420121, 470812033, 747646771, 739277825},
  {1, 1, 3, 15, 13, 17, 53, 27, 133, 13, 117, 1837, 4103, 5843, 29153, 33441, 72097, 9443, 456303, 933933, 1525809, 3196949, 5277819, 4905317, 4684205, 10332757, 52930957, 29686631, 350256755, 1026582593, 1060160513},
  {1, 1, 5, 13, 21, 33, 37, 253, 465, 209, 309, 49, 3209, 15677, 14569, 34977, 67617, 173861, 434605, 21301, 1294849, 3092997, 809565, 11235953, 31154289, 55404309, 1437329, 138570025, 251546909, 50069705, 712770561},
  {1, 1, 7, 15, 13, 21, 33, 203, 499, 141, 1155, 3893, 1663, 2115, 27459, 5729, 83105, 226791, 61039, 543341, 162357, 4099585, 6018603, 7403027, 21975853, 9819171, 68048149, 138440991, 261364771, 643563747, 220517377},
  {1, 3, 5, 11, 21, 9, 39, 157, 257, 273, 1257, 1831, 515, 7969, 20133, 28449, 121571, 204389, 449067, 989877, 792105, 400967, 7121469, 14180961, 5650865, 41434057, 93606343, 92016547, 345771265, 249853573, 1367647233},
  {1, 1, 3, 13, 19, 29, 15, 189, 103, 219, 1395, 517, 7425, 6585, 15865, 54113, 26529, 113443, 263213, 752179, 666941, 4033839, 5103069, 5678279, 3778491, 46547411, 100092645, 170874273, 56327257, 278734233, 1699607553},
  {1, 1, 5, 11, 21, 31, 49, 151, 39, 537, 1783, 3449, 6915, 223, 11587, 33249, 118817, 135525, 95467, 166261, 710463, 3457809, 378679, 3949639, 10065337, 23346199, 58993753, 78553315, 114818879, 545993571, 1272970241},
  {1, 3, 3, 11, 7, 63, 69, 31, 27, 911, 1903, 2821, 7977, 12949, 32257, 36577, 126883, 103971, 315819, 620839, 837919, 128293, 4186495, 13828219, 16711791, 25363343, 114493605, 122231753, 480579317, 82922209, 2077588481},
  {1, 1, 7, 9, 25, 45, 23, 233, 511, 595, 1383, 1721, 6789, 12055, 21179, 23009, 113697, 180135, 70121, 277113, 1907981, 1912631, 6583049, 4245983, 16226419, 51216327, 3532121, 146699941, 403252727, 90400731, 729756673},
  {1, 1, 7, 13, 1, 27, 123, 49, 439, 683, 501, 641, 1947, 6111, 25423, 417, 109857, 245351, 377069, 690913, 1697083, 1013083, 1066449, 2362135, 21009803, 18262933, 121033441, 69418683, 475579455, 86443823, 947383297},
  {1, 3, 3, 5, 1, 23, 57, 241, 243, 593, 2039, 1617, 2209, 5171, 9675, 2529, 26403, 20259, 374629, 113249, 1280823, 1165657, 1713553, 15892307, 6978417, 8534039, 88418929, 242263425, 407103315, 213610731, 2127813633},
  {1, 1, 1, 7, 5, 19, 83, 55, 481, 125, 177, 1021, 1139, 11403, 23099, 13089, 5281, 25953, 472679, 832613, 1421619, 4126067, 2004247, 2711041, 32426973, 49875409, 2479261, 175534483, 124465579, 183707803, 1768147969},
  {1, 1, 3, 5, 29, 39, 33, 217, 461, 907, 733, 3795, 4811, 12939, 27715, 58657, 55265, 159715, 229605, 898557, 903559, 2860417, 7980089, 14525933, 2508203, 50596029, 58431603, 98348651, 361162667, 772701603, 1688099841},
  {1, 3, 7, 3, 7, 11, 39, 165, 495, 147, 999, 1827, 817, 603, 9293, 10081, 111075, 33191, 386787, 318759, 1039787, 2823879, 6144197, 10141711, 246771, 40289031, 39469123, 264819217, 275082299, 801776813, 763259905},
  {1, 3, 7, 15, 25, 53, 35, 15, 431, 733, 1213, 2907, 8087, 3939, 27363, 13345, 76259, 192615, 368111, 803833, 24981, 1737155, 893039, 15573455, 7430781, 23651741, 41051067, 185230583, 487459587, 615929539, 1412338689},
  {1, 3, 7, 13, 13, 9, 33, 27, 485, 183, 455, 3341, 2555, 4985, 8793, 50785, 95779, 30951, 419245, 623277, 2036905, 1287361, 5728379, 11196869, 20236951, 24890983, 27658157, 253105691, 133302105, 951996729, 1961518081},
  {1, 1, 1, 15, 25, 47, 75, 21, 205, 15, 1639, 3067, 1295, 11693, 16903, 56417, 29409, 157793, 292399, 534905, 1920143, 1843691, 4212405, 10099629, 23657903, 1827335, 38512667, 267315759, 282738573, 803361639, 1236696065},
  {1, 1, 1, 15, 3, 31, 93, 57, 43, 185, 251, 1899, 7885, 10829, 3609, 32993, 13281, 82465, 335407, 689059, 1154239, 1555197, 2449049, 15431499, 25048665, 64541339, 30776139, 195173485, 371862445, 759572409, 1019132929},
  {1, 1, 3, 1, 29, 9, 69, 223, 221, 537, 365, 3411, 5771, 15279, 5309, 37153, 57569, 198115, 108129, 290877, 421289, 2051301, 3725887, 13225597, 6741561, 66289357, 107360627, 239976427, 42079375, 319440413, 232973313},
  {1, 1, 7, 5, 1, 5, 125, 243, 213, 1003, 1571, 3355, 3981, 8781, 25993, 51425, 60641, 60967, 220709, 964001, 1530021, 767709, 1196691, 4815605, 21080651, 25398787, 76104763, 205607277, 179338349, 491127657, 1327420417},
  {1, 1, 1, 13, 7, 19, 53, 243, 301, 75, 1183, 2723, 6687, 13, 16581, 2913, 27617, 246369, 195117, 793895, 1185971, 3515541, 3912275, 1409549, 30710315, 19266175, 17847683, 174198783, 53094701, 977432101, 1565953025},
  {1, 3, 1, 13, 17, 51, 91, 239, 437, 191, 1065, 2495, 5755, 3405, 8299, 12449, 89443, 58593, 515885, 921585, 1001875, 1791163, 5449551, 12547221, 11627807, 16171465, 124635999, 150095643, 330027373, 887037707, 169184257},
  {1, 1, 5, 5, 11, 59, 21, 169, 299, 123, 1845, 2199, 2157, 14461, 10327, 43553, 37985, 179237, 49445, 269163, 1771419, 1477301, 160649, 156683, 14724763, 11318229, 31235767, 38318029, 271763613, 650345143, 407972865},
  {1, 3, 7, 7, 19, 47, 51, 179, 41, 19, 1347, 2325, 8063, 5993, 15653, 62433, 101923, 119143, 311463, 197171, 326543, 2698195, 5293523, 14745673, 11207411, 328739, 84476405, 105319839, 464594121, 978015685, 695467009},
  {1, 1, 1, 9, 25, 27, 7, 133, 223, 533, 719, 353, 7093, 8285, 10375, 38753, 114081, 174113, 368681, 755577, 1008571, 1300135, 1142565, 13559935, 20129973, 65698863, 14961665, 76306325, 213576509, 710616999, 559000577},
  {1, 3, 5, 15, 31, 5, 67, 39, 441, 495, 977, 3699, 1435, 11385, 14567, 10145, 44771, 173797, 367023, 778367, 906661, 3686563, 620807, 1346265, 32980367, 30843889, 10150035, 46642939, 376233433, 270467975, 241521665},
  {1, 1, 3, 15, 15, 39, 25, 33, 91, 523, 249, 4035, 769, 5181, 9691, 20897, 38113, 86435, 497967, 957167, 695687, 4095673, 1070529, 7533115, 7978347, 64493209, 325219, 41310625, 93963997, 732917563, 2140232705},
  {1, 1, 3, 3, 3, 57, 83, 187, 423, 165, 161, 3453, 2241, 981, 8429, 12961, 37921, 191395, 427811, 330275, 171417, 1621491, 3595099, 8558151, 30248517, 48642817, 98033565, 210702881, 55256501, 956787917, 1716986881},
  {1, 1, 7, 15, 1, 17, 57, 189, 283, 11, 823, 3505, 7025, 11879, 15441, 61537, 18081, 36007, 158063, 897441, 745649, 519577, 552925, 7295867, 10743979, 59446167, 77511441, 181988177, 451900039, 697657521, 600802305},
  {1, 1, 3, 11, 1, 41, 7, 255, 385, 339, 607, 1405, 1473, 13697, 9491, 5729, 93473, 213667, 52139, 263649, 103305, 3826343, 940319, 10481505, 23464947, 13540351, 97058845, 225644961, 233518369, 334724595, 1520526337},
  {1, 1, 7, 15, 5, 9, 91, 99, 211, 233, 51, 2663, 1165, 9283, 18495, 19489, 102433, 71783, 498479, 304933, 1194985, 138299, 7713987, 8031475, 21921673, 20120659, 92079943, 146080301, 421468899, 49640927, 710759425},
  {1, 1, 3, 7, 21, 37, 13, 91, 39, 27, 1021, 2813, 5937, 6645, 3403, 20833, 2273, 110947, 502503, 814133, 1779397, 797805, 5143675, 3345415, 10819451, 32181789, 12172829, 18549521, 344924117, 596859883, 1609917441},
  {1, 3, 1, 1, 29, 29, 5, 69, 399, 665, 1407, 3921, 2653, 11753, 18925, 20897, 11747, 18401, 422497, 282621, 1202045, 2153509, 6249637, 1060335, 7953785, 3984799, 76033201, 196788157, 166928713, 495412493, 2092688385},
  {1, 3, 7, 15, 13, 41, 39, 1, 437, 549, 161, 2315, 5631, 8335, 22661, 35169, 66851, 141607, 68143, 489325, 235849, 1523719, 3424289, 15352213, 28000325, 59885249, 40421291, 28761119, 38033007, 1051537765, 64476161},
  {1, 1, 3, 1, 7, 17, 115, 61, 69, 955, 475, 3763, 8035, 927, 17893, 60321, 127649, 144035, 234785, 271719, 234353, 360979, 2210333, 9911973, 27376923, 61914171, 71745171, 182825475, 535635583, 382797253, 131322881},
  {1, 3, 1, 13, 21, 59, 81, 145, 463, 145, 1941, 2777, 7453, 14229, 11281, 36065, 37987, 209889, 495597, 697717, 709211, 1488497, 4937329, 13098543, 25915249, 50892725, 97454137, 20651389, 343358517, 209127409, 1157872641},
  {1, 1, 1, 15, 15, 11, 27, 165, 461, 395, 1645, 3611, 7463, 12379, 26787, 5473, 7713, 19489, 229551, 988335, 796395, 3186299, 7980741, 5772909, 25956779, 45766349, 83647931, 111763783, 534318779, 873270403, 1725783041},
  {1, 1, 7, 9, 29, 19, 27, 123, 21, 149, 1643, 4001, 7207, 6769, 4647, 35361, 7393, 25255, 111081, 41661, 1063923, 1242747, 8096475, 12572405, 25930613, 28957003, 118173185, 221769607, 272442385, 1029131207, 1738365953},
  {1, 1, 1, 11, 13, 9, 103, 139, 185, 587, 591, 1113, 2223, 11667, 32671, 53281, 8673, 182433, 22187, 413229, 642281, 3024647, 7042027, 3431577, 30936107, 54867247, 97383929, 83761423, 55717235, 670981055, 499520513},
  {1, 3, 1, 1, 31, 13, 19, 93, 229, 125, 1471, 2369, 3055, 10277, 28563, 20897, 40675, 160929, 25697, 335423, 267245, 312627, 6058941, 8782213, 18481373, 44696287, 31066465, 111025039, 530722501, 50219635, 1902814209},
  {1, 3, 7, 5, 7, 53, 99, 175, 161, 851, 617, 4027, 2357, 11199, 1931, 4833, 53283, 153447, 264101, 47527, 2006997, 3194691, 7244687, 10215617, 26157555, 10904073, 107522971, 129016597, 490520927, 491390635, 1411253249},
  {1, 3, 5, 11, 3, 31, 111, 179, 237, 845, 539, 1057, 259, 3417, 26637, 42785, 3043, 258341, 228587, 756451, 2080895, 2314575, 3847635, 15673805, 30087917, 51567675, 101399809, 88467363, 381246265, 143031789, 640830465},
  {1, 1, 5, 3, 21, 49, 125, 119, 463, 403, 737, 1811, 3941, 13015, 29081, 30177, 47905, 25957, 33955, 432117, 1333585, 344349, 8262551, 1859567, 13825267, 42578561, 94427635, 34326981, 7623159, 353266489, 86144001},
  {1, 3, 5, 13, 5, 29, 69, 251, 313, 357, 663, 1097, 3307, 12845, 28495, 49825, 39907, 106533, 92781, 245157, 34557, 1476453, 2167195, 12546777, 8066949, 6897527, 19187881, 48320971, 506323149, 560615087, 1394484225},
  {1, 3, 3, 5, 29, 17, 89, 15, 411, 409, 2013, 757, 4085, 12521, 11131, 64737, 127203, 223459, 355173, 1029693, 684529, 2009977, 6046127, 1998587, 1419897, 18956221, 83519125, 168245077, 487641929, 955649435, 768513025},
  {1, 1, 1, 15, 7, 51, 3, 193, 493, 133, 381, 2027, 227, 6635, 12931, 51297, 20641, 75809, 426031, 199463, 2017619, 1629667, 5835809, 1780941, 21401381, 8455517, 97062091, 150930819, 471280011, 994518499, 1410333697},
  {1, 1, 1, 15, 7, 23, 99, 203, 323, 1007, 1465, 2887, 2215, 1787, 22069, 48481, 57633, 2465, 263471, 818151, 494839, 4159107, 2374699, 5278819, 10482767, 38413593, 3122535, 232744583, 107377115, 577200789, 1495886849},
  {1, 1, 5, 9, 29, 59, 77, 151, 509, 313, 415, 3977, 5431, 8019, 8571, 19233, 94817, 181029, 408361, 108605, 539483, 3793325, 3245303, 11158813, 21849689, 17975615, 59841129, 158508311, 353230771, 349650587, 1418181633},
  {1, 3, 1, 15, 19, 13, 57, 217, 87, 119, 25, 1149, 5667, 3765, 6959, 29025, 58979, 58273, 208111, 922931, 797037, 460697, 2670777, 7913591, 5062999, 45368057, 83216797, 222578115, 52752021, 939601039, 1130595329},
  {1, 3, 7, 13, 19, 31, 119, 3, 457, 117, 905, 361, 1483, 12405, 27005, 60641, 20963, 70567, 423149, 379763, 1963519, 2610903, 6133923, 11110441, 28400341, 13302313, 133568969, 144203755, 415791637, 582044701, 238152705},
  {1, 3, 5, 11, 15, 35, 61, 77, 119, 51, 1753, 2765, 1091, 10573, 23595, 60001, 43107, 196005, 357867, 334255, 634435, 2933661, 1432237, 12785623, 30344083, 58393337, 59836461, 177538339, 498807917, 1067788043, 174291969},
  {1, 3, 3, 7, 1, 35, 17, 93, 197, 511, 1253, 3031, 2739, 15127, 15147, 15585, 71971, 101923, 204967, 364065, 282179, 1752753, 4437885, 4836453, 2277023, 47240261, 19569335, 189573587, 181588855, 334503435, 441498625},
  {1, 3, 3, 1, 11, 55, 55, 107, 161, 75, 129, 2195, 2023, 4877, 25797, 43361, 34467, 9123, 303329, 2923, 1305815, 316311, 2636107, 11498689, 31840235, 24953633, 46254899, 78339015, 260666733, 938486629, 936414209},
  {1, 3, 5, 7, 23, 19, 113, 167, 167, 271, 1303, 125, 5057, 1323, 5165, 34209, 15075, 95205, 150695, 82679, 1031155, 3955153, 8186183, 3722631, 13211567, 21766199, 60910109, 73511393, 489697163, 774898317, 1034976257},
  {1, 1, 5, 3, 21, 31, 11, 119, 215, 483, 1535, 407, 6485, 15401, 30297, 22433, 120737, 147429, 248035, 82165, 520191, 2635243, 5994775, 11534455, 30408707, 57673695, 19926711, 233836533, 49291657, 464539513, 1685097473},
  {1, 3, 5, 9, 21, 5, 77, 95, 443, 247, 913, 605, 365, 7465, 19707, 26977, 109411, 35173, 482793, 602805, 884581, 230637, 949183, 9167195, 778647, 22472689, 158013, 54079053, 438763017, 410025563, 101762049},
  {1, 3, 1, 7, 17, 59, 9, 35, 391, 767, 1493, 475, 4725, 7529, 31579, 49633, 101027, 240097, 183527, 87217, 345947, 289961, 3896131, 2955687, 30087839, 7556213, 133151163, 152047829, 168829641, 890271227, 1286659073},
  {1, 3, 3, 7, 31, 21, 61, 31, 421, 179, 273, 771, 5745, 10575, 32765, 16225, 119971, 86371, 227623, 975871, 485877, 108957, 2041151, 1353989, 30480403, 11062513, 92660067, 140228433, 342895023, 248539741, 1773167617},
  {1, 3, 5, 15, 27, 13, 125, 55, 423, 1021, 497, 3521, 6903, 15111, 8285, 6113, 68579, 97573, 507567, 441083, 828909, 900061, 4982743, 6520711, 15723741, 48958993, 16838049, 199994583, 341586855, 693911229, 683967489},
  {1, 1, 5, 9, 13, 31, 105, 93, 421, 709, 643, 1079, 1533, 9149, 10799, 10401, 100257, 114917, 58985, 973101, 30335, 3655817, 3685693, 10503749, 260261, 28731811, 128955223, 203119773, 6801885, 705968719, 1392258049},
  {1, 3, 1, 11, 19, 29, 53, 199, 319, 247, 655, 3039, 6411, 12267, 14245, 15857, 4275, 163505, 457307, 679683, 1018669, 4025125, 796023, 3830031, 8598055, 43069663, 86891695, 253921787, 533106075, 851556949, 1277523713},
  {1, 3, 1, 11, 9, 57, 5, 91, 469, 149, 259, 329, 5433, 6941, 15093, 55441, 12179, 234961, 443227, 726297, 1375497, 1736981, 7928299, 10177605, 23038981, 23665747, 125582329, 228224361, 175824717, 578020293, 1770646273},
  {1, 3, 1, 5, 5, 51, 59, 25, 455, 367, 1623, 441, 3155, 11695, 20767, 48529, 75571, 189681, 458741, 288533, 1372419, 4036907, 6115657, 1859223, 4621151, 6718183, 97617705, 30056227, 134382175, 692775279, 387795201},
  {1, 3, 7, 7, 11, 49, 113, 95, 91, 389, 605, 1973, 2051, 2315, 22229, 53585, 26163, 155319, 405335, 337691, 1306881, 832769, 7846703, 10662891, 6421397, 34589773, 47899461, 109719731, 107523675, 874756101, 1551314689},
  {1, 3, 5, 3, 19, 11, 99, 135, 433, 781, 1473, 885, 1105, 3573, 3739, 23889, 83731, 205141, 112563, 779011, 7995, 451379, 5395895, 5002113, 17284797, 24499441, 130104837, 151490113, 127511589, 1067629451, 1593947905},
  {1, 3, 1, 11, 3, 25, 9, 227, 433, 723, 317, 139, 6627, 8067, 28439, 22193, 66611, 20017, 498555, 943379, 1354025, 1397017, 4666707, 5855873, 5244483, 1686189, 83788731, 169905139, 441747891, 518951879, 371806977},
  {1, 1, 1, 9, 9, 9, 5, 63, 241, 215, 1991, 2949, 3943, 775, 31511, 18257, 51057, 141169, 439929, 602137, 446489, 2895893, 3387567, 10031201, 19935303, 31897495, 60176501, 131289207, 177561207, 503633767, 413642497},
  {1, 1, 3, 7, 17, 49, 35, 167, 131, 107, 1295, 2465, 4577, 11147, 29833, 60689, 11537, 238515, 104343, 892929, 2072609, 4136979, 4399319, 5742995, 9134971, 3860287, 120890321, 176716241, 8572219, 35112057, 548001537},
  {1, 1, 5, 1, 5, 25, 119, 129, 391, 743, 1069, 2957, 349, 6891, 13635, 21969, 9937, 54709, 445745, 332821, 1580041, 1281063, 1895569, 2356695, 27748215, 53284701, 61726621, 55440685, 406628507, 488545171, 1589612801},
  {1, 3, 1, 7, 9, 31, 63, 253, 215, 51, 1347, 2361, 3125, 13049, 28461, 28817, 52819, 184977, 128983, 5657, 868911, 4037167, 5800589, 6326855, 9778115, 33484979, 1324265, 45121349, 144241609, 176796957, 1937476865},
  {1, 1, 7, 9, 3, 31, 21, 163, 255, 47, 259, 535, 5461, 3349, 30649, 41073, 42065, 251991, 454297, 417299, 1717775, 1725029, 5403187, 14596815, 23964639, 14207571, 118132263, 65503397, 308130277, 312447369, 442446593},
  {1, 3, 3, 13, 17, 33, 87, 47, 243, 709, 929, 3943, 3107, 3421, 13721, 28657, 98483, 173395, 412189, 167553, 771473, 358887, 1866111, 15012195, 23106133, 52851537, 110688279, 185692819, 535133069, 377577225, 2018885377},
  {1, 3, 5, 11, 25, 61, 61, 173, 397, 735, 2005, 3355, 8121, 11593, 27697, 10033, 33811, 28181, 401051, 786313, 1619341, 231917, 5874077, 15384733, 19415183, 34194565, 2834507, 257507561, 228942649, 1018510049, 94723329},
  {1, 3, 5, 15, 17, 43, 63, 231, 275, 311, 1277, 2669, 7307, 2099, 9755, 18257, 76595, 26037, 446271, 460929, 1684123, 3579631, 889751, 5941379, 16845319, 38407565, 41278365, 144728379, 218568643, 867134795, 1470060289},
  {1, 3, 5, 3, 25, 43, 71, 191, 9, 121, 1873, 3747, 7491, 14055, 24293, 41649, 21683, 219893, 15315, 773769, 1105051, 2961815, 7670031, 12438809, 31950665, 65803169, 87602899, 102610259, 278078967, 641165173, 1630916353},
  {1, 3, 5, 13, 17, 35, 113, 113, 385, 941, 39, 2705, 1225, 5167, 1373, 10129, 39955, 210805, 435037, 542593, 641171, 1748897, 1025825, 5510673, 31049245, 64460471, 118128129, 49373785, 338457471, 420035757, 1039655169},
  {1, 3, 5, 5, 7, 35, 19, 105, 487, 71, 139, 627, 4187, 3183, 713, 21937, 10547, 60533, 91669, 339351, 1652627, 681155, 1169593, 4560919, 8993015, 7080763, 22400867, 30459819, 391211423, 14374649, 865958145},
  {1, 1, 5, 13, 29, 29, 103, 5, 157, 869, 1675, 423, 6689, 10697, 5303, 38385, 41361, 71701, 96253, 855181, 49293, 2342839, 25685, 15599053, 32898101, 54207099, 123218807, 189262193, 49362969, 244918919, 1460081921},
  {1, 1, 5, 1, 29, 31, 61, 111, 473, 963, 685, 1483, 2383, 8109, 8495, 54577, 17009, 19797, 142449, 95629, 1677455, 1778157, 4545279, 11820681, 19948211, 10150909, 118007739, 144636255, 486090525, 710091391, 148225793},
  {1, 1, 5, 3, 19, 13, 95, 113, 217, 59, 1353, 1647, 3617, 3271, 2321, 56689, 11409, 236021, 299411, 619139, 69021, 2644623, 3983041, 5487465, 3270763, 31570745, 17521407, 60349233, 511147767, 171800321, 1882685697},
  {1, 3, 5, 7, 25, 35, 59, 131, 309, 445, 415, 93, 1453, 8789, 30201, 62033, 89939, 226773, 384023, 321673, 2003091, 567019, 7897715, 7953189, 24464141, 39825327, 40036077, 238776701, 92102149, 613905289, 509326081},
  {1, 1, 5, 1, 5, 43, 71, 241, 123, 189, 831, 3469, 8093, 6187, 32721, 1617, 119377, 197173, 197265, 253077, 1399739, 3792535, 6169185, 13569707, 14935693, 21881295, 75312669, 266206125, 499342971, 307566241, 199434497},
  {1, 3, 7, 5, 25, 31, 123, 171, 319, 379, 889, 2365, 4881, 12225, 16609, 5457, 98707, 74135, 11157, 366217, 2064047, 2336395, 1076859, 1511215, 12905739, 20098121, 111486221, 146088289, 487013361, 374288625, 738179841},
  {1, 3, 1, 11, 27, 43, 121, 63, 291, 591, 811, 1995, 4777, 2083, 31385, 43889, 93267, 29937, 288731, 453579, 1115995, 3381033, 4498767, 2848211, 28306751, 49102715, 78467579, 117459417, 188472339, 647798249, 1101594369},
  {1, 1, 5, 11, 27, 53, 85, 187, 461, 823, 703, 399, 6925, 11517, 28697, 34609, 8465, 163349, 87259, 1014347, 610661, 3600197, 7596875, 14752317, 17025447, 21802543, 16215679, 59701149, 123402829, 145141673, 1439890177},
  {1, 1, 3, 5, 13, 11, 33, 121, 93, 717, 1275, 3877, 4247, 5845, 26909, 61201, 433, 211475, 498197, 902749, 1718875, 1777489, 4062057, 413517, 6763581, 13336747, 9735541, 142710279, 499425605, 17609453, 968243457},
  {1, 3, 1, 9, 7, 5, 47, 199, 367, 561, 185, 2855, 5997, 2699, 7581, 54321, 51859, 246641, 130201, 14295, 328821, 3218815, 4222871, 4331103, 26023201, 57204873, 86614295, 211286877, 278189915, 994104525, 1434584833},
  {1, 1, 5, 9, 23, 11, 71, 201, 61, 729, 1011, 3529, 663, 1413, 25675, 61649, 33713, 52245, 94073, 558663, 1458139, 4122455, 4031769, 7330573, 526121, 30415171, 77810841, 234670471, 535982933, 835703099, 1365149953},
  {1, 3, 7, 13, 27, 21, 11, 127, 281, 487, 1217, 3129, 5541, 3129, 17783, 45137, 122899, 117719, 367677, 126283, 1229541, 606011, 3881327, 5100137, 12977911, 40841137, 28637577, 123507701, 270176905, 692427303, 87998209},
  {1, 1, 5, 9, 1, 29, 85, 193, 213, 743, 1473, 611, 391, 9405, 21137, 16113, 89393, 54293, 518617, 377681, 200781, 2498629, 7495697, 3973509, 14279031, 26108113, 97265971, 205502871, 322083469, 1018993601, 1248624897},
  {1, 3, 3, 3, 31, 63, 37, 147, 39, 351, 79, 3069, 2441, 8901, 8777, 2833, 7251, 211987, 268755, 691535, 1338959, 3174485, 2125923, 9916695, 7869039, 40687071, 113057037, 52466361, 63349749, 323526521, 379867905},
  {1, 1, 7, 7, 25, 49, 55, 47, 441, 343, 1267, 1123, 5917, 14395, 10579, 42225, 19473, 1143, 149879, 604873, 1573089, 313351, 2213919, 14662121, 16083847, 21858755, 18272979, 225900877, 420589195, 31842339, 558657281},
  {1, 1, 7, 1, 13, 55, 55, 123, 103, 773, 125, 2145, 4743, 13347, 2589, 53329, 15409, 172919, 241937, 944349, 878439, 3306503, 3615787, 3538295, 26731573, 40588365, 98448913, 53097783, 373927603, 1018621965, 854604545},
  {1, 3, 7, 3, 9, 33, 25, 183, 469, 213, 291, 75, 6725, 6847, 26745, 51953, 54675, 63543, 154067, 210905, 2074193, 2307113, 4738119, 9819269, 10952197, 50269939, 68208507, 180962645, 71159343, 60617481, 584611585},
  {1, 3, 3, 7, 15, 43, 7, 79, 171, 21, 1767, 2537, 4285, 12007, 24039, 13009, 20243, 212787, 250935, 687071, 434139, 1069687, 2799359, 7567003, 33360229, 23046999, 63330777, 206028237, 187984503, 503965527, 1495699201},
  {1, 3, 7, 13, 9, 61, 125, 23, 227, 879, 215, 1635, 2835, 883, 15939, 23665, 123763, 185463, 65917, 664153, 2092621, 2673357, 7914375, 2814259, 31978367, 21034183, 125301203, 133750243, 70566851, 128470387, 1560844033},
  {1, 1, 5, 13, 25, 45, 63, 43, 183, 829, 149, 989, 987, 3819, 12181, 22801, 79473, 205685, 209181, 923209, 485373, 592303, 2716987, 13005287, 15929645, 38983333, 110050221, 202946795, 517485947, 54022533, 1291365121},
  {1, 1, 3, 7, 19, 27, 35, 83, 135, 459, 785, 131, 2655, 3329, 3009, 32113, 106545, 56403, 499287, 580803, 1138635, 1314259, 3892707, 3493111, 6257979, 7035233, 76468723, 19939439, 130046961, 514594193, 1767691009},
  {1, 1, 7, 5, 11, 41, 9, 219, 475, 985, 1329, 3787, 1975, 4679, 8627, 9585, 11697, 96311, 221557, 48987, 574073, 3352761, 1051979, 1804075, 147593, 64798945, 118998971, 144428487, 495116951, 992637667, 61496065},
  {1, 1, 7, 3, 1, 17, 91, 155, 3, 763, 1879, 233, 215, 2955, 25993, 5073, 3889, 258935, 390099, 304721, 1772225, 613867, 2540395, 11660883, 32094123, 1921575, 108721305, 218183079, 172621947, 374763705, 320740097},
  {1, 1, 1, 11, 25, 11, 23, 227, 453, 775, 1935, 3833, 4583, 269, 705, 50897, 31793, 169745, 453467, 505801, 1852379, 964551, 2121619, 5074837, 14413559, 33540671, 119786377, 92360439, 46030653, 1029572593, 1338663681},
  {1, 3, 3, 11, 7, 25, 105, 21, 449, 555, 1275, 3475, 5503, 15617, 813, 7793, 128435, 141907, 161691, 263127, 636905, 718489, 3152101, 6650097, 1804539, 14347819, 119537923, 18876335, 245380721, 337669725, 1499501825},
  {1, 3, 7, 13, 31, 37, 25, 255, 233, 663, 1155, 1563, 4775, 7449, 29949, 2033, 86067, 162263, 8349, 813519, 221141, 1953449, 7637103, 15165785, 1792007, 46931027, 106297131, 166830327, 428879657, 74449165, 1227915009},
  {1, 1, 3, 1, 23, 51, 51, 137, 63, 809, 349, 2789, 6953, 10605, 18959, 5265, 121937, 153107, 38705, 350663, 824163, 261571, 2652761, 11616015, 27657177, 6423853, 38142773, 71700377, 51777085, 26112671, 817236225},
  {1, 3, 3, 13, 21, 45, 15, 161, 393, 229, 437, 2967, 4019, 3893, 21305, 27569, 16371, 76563, 23517, 99013, 386781, 1049343, 5986865, 12870553, 13726453, 47551621, 85218887, 154093667, 92828837, 199381161, 1139679489},
  {1, 1, 3, 7, 5, 11, 15, 211, 287, 131, 1847, 2569, 7881, 15669, 31037, 2769, 57809, 77715, 486423, 1014741, 2091995, 3255551, 6365795, 11219727, 11471475, 1396487, 26388857, 264482681, 500053061, 246819341, 1045320449},
  {1, 3, 3, 15, 27, 19, 85, 251, 221, 639, 665, 3729, 5771, 7873, 28005, 58705, 61043, 118227, 315167, 516043, 914403, 2853029, 4025931, 36397, 24573711, 20014217, 54010465, 59553179, 256509809, 166601781, 2027584257},
  {1, 3, 7, 15, 15, 47, 93, 215, 343, 85, 1401, 1375, 2949, 13661, 25453, 49713, 47507, 29079, 511775, 489439, 531423, 3424493, 5584487, 1173479, 8035557, 47443689, 93982319, 19006517, 137245005, 901386429, 143283969},
  {1, 1, 1, 9, 7, 51, 53, 217, 471, 389, 551, 1141, 1767, 2237, 17797, 20401, 46289, 242481, 431097, 647095, 1287939, 2555141, 6833513, 8843655, 30741333, 3569175, 45233733, 148436919, 329148685, 1014567509, 1151952129},
  {1, 1, 7, 9, 3, 29, 65, 29, 223, 591, 1719, 1049, 7643, 3853, 29867, 50033, 14993, 171959, 145177, 360051, 1866541, 191761, 7818093, 495823, 11776767, 21681671, 56145129, 123011211, 30883101, 452552507, 1373100289},
  {1, 1, 1, 11, 13, 41, 85, 29, 451, 387, 1783, 3733, 8033, 4711, 31643, 32177, 114353, 175185, 332635, 206461, 726809, 3567973, 7229837, 3709811, 8404403, 38433159, 128825573, 4132849, 289357111, 939067275, 1535764737},
  {1, 3, 1, 11, 11, 57, 75, 153, 7, 373, 2011, 271, 469, 3267, 18969, 34193, 9907, 213137, 113659, 160827, 1363753, 2673979, 29449, 7489239, 16002693, 31440715, 111992063, 214025093, 154493043, 115672073, 970697985},
  {1, 1, 5, 3, 19, 43, 7, 243, 385, 293, 923, 843, 4895, 469, 8421, 9681, 90769, 61621, 211699, 435107, 928027, 3607415, 8105827, 12729297, 9081589, 33075595, 5436827, 253472879, 428940965, 285416693, 650579713},
  {1, 3, 1, 15, 29, 47, 17, 125, 471, 927, 349, 3859, 3059, 11483, 14791, 13457, 111443, 235985, 62751, 124909, 787775, 1387873, 2179501, 14323431, 16917455, 17146541, 89827299, 12401187, 103865035, 183841015, 2051625217},
  {1, 3, 1, 11, 17, 17, 111, 109, 9, 213, 1313, 3903, 4411, 4329, 28277, 23249, 23891, 236785, 146299, 425121, 87041, 1010719, 1064189, 15701113, 4368357, 3708401, 51992591, 199952395, 4128921, 781546181, 1863973121},
  {1, 3, 3, 15, 1, 55, 47, 69, 143, 789, 1149, 3833, 5053, 6949, 10569, 27953, 37875, 256243, 25919, 277297, 2083879, 1244287, 2395349, 4001151, 27573317, 32783469, 8928073, 150091757, 446762901, 904470585, 557140225},
  {1, 3, 5, 7, 11, 15, 79, 83, 123, 937, 1115, 2775, 3041, 11869, 21167, 2321, 25587, 146677, 267863, 587771, 643103, 1699967, 5017731, 6950955, 6877753, 36041035, 100488711, 22083121, 297864109, 1048400447, 1825570561},
  {1, 3, 7, 13, 9, 47, 45, 221, 139, 923, 1661, 1379, 2485, 7233, 6035, 43153, 71731, 93623, 192541, 664697, 778303, 1253437, 3842285, 3563963, 32120907, 24863053, 76967187, 99826341, 133026961, 206644003, 140710145},
  {1, 1, 3, 3, 11, 55, 77, 3, 87, 693, 1991, 1145, 2783, 16207, 24569, 51313, 2513, 19987, 198003, 204091, 1329159, 1595485, 452691, 15964295, 28170405, 60141047, 57730537, 196356047, 373761791, 434588969, 1551978241},
  {1, 1, 5, 11, 3, 35, 91, 9, 391, 927, 101, 1839, 3755, 10345, 16907, 61777, 76369, 226037, 411291, 106867, 363027, 3353131, 1719833, 1503959, 25251791, 1157045, 13820639, 121910779, 445901689, 522657211, 1873892609},
  {1, 3, 5, 3, 5, 49, 79, 91, 205, 443, 1369, 197, 2537, 11219, 17765, 9841, 16691, 19733, 374579, 315573, 350753, 1073791, 4863691, 10394365, 17461259, 56054153, 48526997, 181499993, 213545923, 1058878869, 672091905},
  {1, 1, 3, 15, 9, 7, 25, 25, 357, 247, 477, 421, 7679, 5987, 30079, 13169, 12497, 83795, 18719, 329593, 537271, 962697, 8315657, 16113045, 12061991, 49410093, 33209493, 17849871, 48452403, 924235727, 102601985},
  {1, 1, 5, 3, 29, 5, 89, 117, 481, 491, 371, 389, 7101, 2253, 23617, 14001, 93937, 86773, 479027, 896429, 2002869, 3776937, 1049957, 1463761, 31892891, 62416067, 51078261, 10805389, 99217373, 972259729, 2022174977},
  {1, 1, 5, 13, 29, 59, 17, 181, 511, 291, 1991, 3499, 8177, 5559, 30045, 4561, 114673, 176853, 338621, 549421, 467595, 2601441, 867653, 11350031, 14062259, 5304887, 42489691, 58557985, 391945191, 899397325, 909846785},
  {1, 3, 3, 11, 23, 31, 117, 217, 241, 115, 749, 945, 1897, 12253, 8473, 46129, 52019, 115827, 431291, 347815, 1603983, 1556389, 305033, 12949089, 2545635, 20067037, 59832833, 94101977, 70631757, 416934441, 975795457},
  {1, 1, 7, 15, 25, 47, 31, 1, 165, 311, 635, 3629, 1593, 8305, 30033, 35121, 51121, 118711, 109503, 555945, 984991, 3051471, 5693329, 5832533, 26393703, 12957931, 71416925, 245479625, 197230881, 1016723841, 713665793},
  {1, 3, 5, 9, 3, 17, 101, 237, 379, 503, 49, 929, 1687, 3865, 26723, 33713, 83155, 13813, 9113, 448243, 867713, 385749, 7157085, 13982443, 32696071, 2342849, 125921105, 79345927, 45053577, 102268947, 963352833},
  {1, 3, 5, 5, 15, 41, 1, 239, 53, 215, 1733, 827, 579, 4089, 6579, 40849, 10963, 47989, 232341, 491775, 461241, 845489, 6506911, 13033189, 6457383, 1530485, 2116971, 255645043, 213605289, 354198403, 818395905},
  {1, 3, 1, 15, 15, 21, 35, 21, 403, 257, 1475, 2403, 4705, 11553, 203, 21201, 96755, 138929, 129567, 428031, 1119365, 29139, 6255685, 13169923, 1834289, 7050963, 106248147, 4560753, 357012177, 89578075, 965841153},
  {1, 3, 5, 11, 9, 53, 113, 9, 447, 511, 543, 3141, 7389, 11249, 431, 24881, 5619, 1077, 248155, 7545, 339109, 961473, 2643097, 583119, 24435983, 35594799, 112175733, 259156109, 9418593, 360505151, 154520833},
  {1, 3, 5, 9, 9, 11, 55, 93, 325, 411, 305, 2573, 6871, 12339, 6435, 57713, 93267, 37461, 326617, 257529, 1633691, 324999, 578797, 4511093, 8074891, 7125921, 3418557, 249441447, 431519395, 319085523, 1595228417},
  {1, 3, 3, 7, 31, 27, 21, 113, 99, 853, 365, 589, 3731, 10875, 12767, 42481, 48403, 53843, 267543, 767087, 849291, 2860997, 3862753, 5959923, 4229, 64835805, 80091261, 188709635, 386959819, 1026991407, 1097665793},
  {1, 3, 1, 7, 15, 27, 31, 17, 275, 93, 1161, 2619, 1329, 7307, 587, 44561, 89587, 187153, 275799, 546815, 193931, 3896815, 465089, 16042115, 31463565, 2530713, 14862155, 135480225, 386956475, 717795835, 648621313},
  {1, 3, 5, 9, 17, 47, 49, 237, 27, 193, 1237, 591, 5151, 5521, 31583, 60625, 96659, 227189, 173913, 908385, 670399, 3642753, 6266461, 3586987, 1547409, 21603493, 37315647, 78723471, 241198177, 732619439, 590569729},
  {1, 3, 5, 3, 13, 1, 27, 87, 43, 977, 305, 3293, 2475, 14571, 18321, 36721, 89235, 169045, 231923, 720829, 37777, 4149931, 5855815, 9350939, 11994849, 2982689, 78350925, 179606395, 137534235, 671068961, 34456321},
  {1, 1, 5, 7, 15, 13, 101, 1, 291, 807, 1711, 2277, 5573, 11051, 13133, 5425, 67185, 88213, 290359, 1040191, 2049981, 1651861, 3212881, 6843315, 30266007, 44249311, 67656149, 32481109, 456688059, 715519965, 2133714177},
  {1, 3, 3, 1, 9, 3, 65, 81, 415, 733, 1527, 2747, 6069, 159, 7095, 38353, 27571, 70611, 414737, 671865, 1646995, 1694097, 4095649, 10464943, 23650381, 15869703, 80176395, 42297317, 462588975, 595382759, 1527196929},
  {1, 3, 3, 15, 27, 1, 71, 49, 231, 851, 2039, 613, 1899, 2537, 14511, 49201, 72339, 57715, 284767, 939115, 538257, 3778967, 3879457, 671543, 19874723, 41271847, 213525, 208064955, 166229433, 245823295, 1547674881},
  {1, 1, 1, 11, 3, 41, 55, 23, 247, 1011, 581, 2363, 2745, 1337, 20931, 22481, 126385, 236145, 75099, 310067, 1466841, 3183943, 550855, 10319079, 8093379, 64254421, 2317291, 93404649, 506975273, 298286387, 615846657},
  {1, 1, 3, 11, 17, 61, 67, 255, 143, 357, 945, 3407, 5817, 4155, 23851, 25489, 128561, 12115, 512923, 899745, 1450957, 2141971, 2414447, 2348351, 20768597, 1485665, 68219551, 66311273, 331215179, 450709083, 1436674817},
  {1, 3, 5, 3, 23, 1, 75, 247, 265, 413, 1899, 2565, 6629, 15655, 16117, 58993, 102451, 113557, 89843, 818279, 1696849, 196411, 4087207, 16439385, 31361197, 8698683, 69531477, 258636373, 229523735, 192223909, 601084673},
  {1, 1, 1, 9, 11, 49, 11, 189, 223, 177, 1457, 1931, 163, 15905, 17297, 11473, 124177, 225969, 253945, 863611, 1651009, 2749307, 3438413, 9977359, 20017025, 33320673, 74655355, 230108467, 332606097, 313594977, 402774273},
  {1, 3, 7, 13, 17, 1, 111, 189, 343, 961, 427, 2507, 2393, 8653, 6353, 44945, 87091, 160567, 485629, 876129, 523089, 760639, 1263437, 11665191, 15478705, 49532475, 44648635, 2383689, 371816317, 662515329, 2122438913},
  {1, 3, 7, 13, 23, 61, 59, 51, 313, 963, 791, 3681, 5637, 3965, 9263, 3633, 84915, 133431, 497181, 234407, 176365, 1449835, 560579, 14710569, 7309491, 5118919, 18607729, 253343797, 525783981, 386476671, 1812702977},
  {1, 3, 7, 7, 21, 53, 127, 141, 499, 859, 337, 2835, 3195, 4351, 32369, 45009, 122771, 85591, 447543, 277221, 1377637, 3879215, 5638109, 12966531, 9206379, 63507521, 5046371, 131014187, 19865455, 350441921, 594360577},
  {1, 1, 7, 5, 1, 5, 53, 63, 497, 535, 35, 305, 4395, 9757, 13193, 56753, 81233, 45047, 172789, 494705, 170613, 3552293, 6298831, 11592833, 24778535, 40212371, 32133409, 129255579, 212824717, 283851545, 1926449921},
  {1, 1, 5, 13, 13, 31, 59, 229, 211, 745, 1453, 3677, 3005, 7703, 23907, 1841, 20529, 20629, 24221, 180157, 1288431, 203787, 3087573, 570595, 931321, 6740669, 1854637, 208062701, 61591463, 555053651, 703969537},
  {1, 3, 5, 5, 7, 63, 17, 197, 493, 861, 499, 3015, 6349, 1815, 7437, 47185, 11763, 162837, 166517, 876151, 1418095, 586849, 2727157, 16262269, 1621261, 25295555, 61308535, 43738877, 347886119, 1056265949, 821762305},
  {1, 1, 1, 13, 13, 37, 29, 189, 253, 1017, 321, 3145, 407, 7547, 17099, 48113, 117009, 227857, 250653, 103869, 872149, 172141, 3809293, 860621, 19669833, 13887345, 41315705, 193494695, 290533899, 241252027, 619809025},
  {1, 3, 3, 3, 23, 53, 69, 77, 175, 17, 1831, 841, 3851, 1295, 32107, 58673, 70483, 171731, 390803, 817319, 607973, 2549845, 7239901, 13896063, 32541025, 4810775, 65567353, 9753883, 70511295, 994397947, 1506900225},
  {1, 3, 7, 13, 13, 39, 107, 237, 389, 729, 635, 3717, 3041, 3169, 14987, 43633, 12147, 18551, 103901, 138557, 1863927, 382523, 5195293, 1872693, 32951369, 41886059, 85272309, 227701457, 290792465, 1048395099, 1926697217},
  {1, 1, 3, 1, 25, 7, 69, 35, 495, 49, 659, 2783, 6051, 13875, 23927, 21201, 87057, 206739, 250065, 970281, 1643511, 2875925, 86547, 15858335, 4236577, 45456995, 77835151, 6240883, 194432515, 578751399, 1345052417},
  {1, 3, 7, 5, 5, 25, 49, 7, 193, 493, 93, 657, 1515, 13975, 14155, 14577, 78035, 9047, 393109, 654517, 1318089, 1947233, 6573687, 13629425, 3736477, 51709741, 102042753, 25759579, 255015847, 709823867, 452133633},
  {1, 3, 1, 1, 11, 15, 113, 45, 21, 595, 731, 3397, 4117, 9711, 16625, 57585, 39987, 161809, 435889, 716283, 793183, 3981889, 1918685, 14162629, 31115651, 38934763, 79935605, 80143013, 443369279, 768541793, 1309782273},
  {1, 3, 3, 9, 19, 19, 59, 7, 105, 579, 599, 2859, 97, 14717, 15361, 60433, 100147, 60083, 62073, 884451, 1540035, 3021227, 7431607, 7111737, 4642259, 55727303, 14128251, 267619921, 350718765, 86347729, 1165093633},
  {1, 1, 1, 5, 27, 49, 113, 5, 367, 563, 1397, 2805, 3021, 3111, 20671, 7857, 689, 198673, 318165, 671659, 30017, 2150529, 1265589, 6193599, 27951747, 65372421, 103514181, 257393245, 121801271, 307554767, 885760257},
  {1, 3, 3, 15, 27, 51, 99, 167, 109, 365, 1959, 1523, 6959, 14405, 18191, 47633, 77779, 107251, 101919, 934443, 1135203, 2853619, 6573943, 16339069, 27136125, 406967, 79551907, 268133215, 322541685, 301394719, 459524865},
  {1, 3, 1, 5, 21, 51, 125, 67, 123, 45, 1657, 51, 4825, 14081, 31049, 20177, 7987, 216945, 229237, 177381, 739299, 1529293, 7491955, 15505739, 16563069, 62044425, 63140067, 11593417, 143917041, 1070186809, 723873537},
  {1, 1, 5, 7, 21, 59, 21, 249, 77, 793, 1687, 2561, 2241, 4321, 7477, 29745, 9073, 164853, 465335, 907109, 1012427, 352677, 935913, 166717, 26886729, 42254695, 132763249, 83611889, 434818321, 1070193573, 348124417},
  {1, 1, 1, 7, 15, 35, 71, 29, 267, 611, 1813, 1823, 7039, 3299, 9919, 5137, 114385, 71313, 165335, 217023, 1319763, 2581175, 2347405, 10774235, 29563891, 39451717, 46435439, 184713519, 446208627, 903540431, 114946305},
  {1, 3, 7, 11, 21, 59, 109, 213, 371, 785, 659, 1687, 4827, 6017, 19619, 2257, 42259, 230711, 45403, 859813, 1857899, 661181, 2611653, 10948099, 6322753, 22268643, 34570023, 134795915, 440488497, 553566963, 863049985},
  {1, 1, 3, 11, 27, 17, 1, 55, 367, 939, 333, 127, 5105, 2405, 28139, 57361, 15665, 247347, 24795, 749227, 1621089, 1101265, 5062695, 934399, 21873307, 40779837, 80923375, 18008865, 263017333, 642288667, 1942704897},
  {1, 1, 7, 13, 5, 35, 59, 133, 509, 573, 625, 3857, 7935, 5279, 3727, 28145, 108561, 7639, 96093, 816565, 1072723, 4183467, 660085, 4567885, 22736493, 23909473, 56250465, 100186319, 403536431, 805736959, 946448641},
  {1, 1, 1, 7, 11, 47, 127, 157, 19, 403, 151, 1143, 7407, 8985, 32521, 33745, 97617, 145201, 150263, 77051, 456927, 3298959, 2684429, 16502403, 6851907, 32140743, 99377671, 91219007, 242467465, 726377561, 10174721},
  {1, 3, 1, 1, 5, 13, 105, 123, 63, 139, 1569, 1983, 563, 7175, 27705, 40849, 128019, 188497, 88881, 604021, 718813, 3367641, 1707531, 13792143, 26288699, 15607505, 25079535, 115109987, 261673751, 823236361, 528723713},
  {1, 1, 3, 13, 9, 35, 105, 227, 145, 21, 1369, 57, 393, 2921, 18511, 7241, 35657, 176763, 176709, 404033, 1524731, 1606209, 3212011, 16011081, 25182749, 60572161, 56117873, 265830753, 66177889, 315648967, 2031126337},
  {1, 3, 1, 7, 17, 61, 99, 187, 261, 281, 437, 2219, 5999, 1857, 18001, 50041, 82443, 162441, 422863, 67657, 1428597, 1975835, 772851, 1911469, 20007361, 46288429, 113378755, 68817255, 306581033, 17694793, 772960321},
  {1, 3, 3, 5, 1, 59, 67, 45, 451, 439, 2005, 3607, 3, 7167, 14227, 30745, 63483, 62619, 211149, 874457, 2018531, 3063339, 3667685, 6418139, 21371119, 30669053, 130162895, 167948155, 157078551, 867060203, 373701569},
  {1, 3, 3, 3, 29, 19, 25, 251, 275, 733, 1749, 4021, 871, 3227, 13701, 46905, 123099, 24619, 136651, 643781, 963915, 39489, 6630099, 1369931, 4908245, 49534029, 23968237, 81045967, 258155251, 184487661, 1273574721},
  {1, 3, 3, 13, 27, 53, 57, 243, 491, 521, 1921, 1037, 5013, 5703, 15261, 45225, 73275, 162027, 217413, 552387, 1518445, 2856465, 7287531, 16453251, 23436929, 14073049, 34534421, 18248605, 199646351, 963190581, 1203216449},
  {1, 3, 1, 11, 13, 57, 1, 15, 123, 533, 785, 335, 1423, 14269, 3483, 45177, 16875, 190681, 436163, 1037141, 1145585, 408089, 1502887, 3279859, 12699357, 5831209, 60664535, 131170151, 114373365, 920872739, 1788983105},
  {1, 3, 7, 13, 15, 55, 5, 139, 385, 47, 1981, 1291, 7397, 12925, 29445, 44329, 50667, 23087, 30533, 202071, 1181519, 1716269, 5869619, 12415481, 7541575, 31133733, 14881987, 26987741, 219012437, 836668189, 1599132097},
  {1, 1, 7, 1, 23, 23, 59, 93, 117, 57, 63, 3047, 4849, 11637, 25311, 31113, 2473, 197887, 456265, 703327, 49647, 1558643, 6613157, 6756621, 4807073, 10129175, 35394959, 26551801, 146371533, 149534887, 684877889},
  {1, 1, 7, 13, 19, 37, 25, 203, 477, 447, 1345, 3485, 2099, 13347, 11621, 26521, 54041, 163119, 112325, 222939, 1005405, 2565169, 3554323, 1998133, 27002423, 53031897, 25910837, 39882635, 55478507, 165513933, 1407028673},
  {1, 1, 7, 3, 11, 23, 81, 17, 41, 735, 1149, 3253, 7665, 8291, 22293, 16777, 13593, 81583, 91723, 827075, 678767, 152585, 4670473, 2952529, 15455383, 13673013, 82242621, 23733065, 24161227, 1060746493, 937502145},
  {1, 1, 5, 3, 15, 9, 57, 167, 463, 493, 747, 1947, 6471, 1111, 31619, 50233, 73913, 30237, 483275, 376775, 1581793, 2368609, 1225887, 5739719, 23069925, 4540435, 62183331, 246543471, 221687503, 406118555, 476920129},
  {1, 1, 5, 15, 7, 15, 107, 205, 325, 167, 1749, 927, 3589, 6127, 7617, 52521, 68217, 145677, 101447, 302287, 793959, 2391379, 3340213, 3584893, 10109759, 29017725, 2772855, 194955517, 446386951, 140469625, 273797057},
  {1, 1, 1, 13, 21, 25, 83, 147, 411, 399, 1423, 2279, 3661, 7591, 17429, 51561, 81465, 112553, 121925, 478941, 198737, 1490299, 7396219, 7258643, 27355991, 47603543, 9606527, 76726773, 383024671, 1052851437, 177198529},
  {1, 1, 1, 9, 5, 17, 69, 205, 243, 647, 473, 1717, 1977, 10725, 2913, 35017, 107561, 205833, 157505, 400461, 1855833, 894797, 4848549, 14981947, 29911535, 49237489, 50792045, 72818113, 509806445, 325528841, 1354464193},
  {1, 1, 3, 5, 5, 37, 103, 15, 485, 641, 1761, 3755, 6997, 10985, 11773, 61865, 89673, 41707, 436301, 52557, 587389, 145167, 8352615, 14266125, 14202873, 36397785, 97218499, 59569565, 497632545, 406878437, 1187053121},
  {1, 1, 5, 13, 9, 51, 87, 195, 97, 807, 1801, 961, 6341, 4307, 29105, 54953, 13593, 239885, 505285, 240193, 1360347, 2283263, 7562827, 7152185, 19501663, 14546897, 21418633, 24692669, 87585243, 234774649, 589486017},
  {1, 3, 1, 13, 9, 35, 83, 61, 387, 817, 951, 3993, 7831, 8479, 23941, 25913, 98427, 100265, 480581, 268369, 1377003, 981243, 3649077, 5258075, 12968041, 31301791, 43593153, 95844447, 18442343, 967194893, 1114036801},
  {1, 1, 7, 11, 19, 47, 75, 37, 91, 337, 953, 1169, 163, 2259, 24713, 37017, 112153, 237775, 214083, 268635, 1181655, 1052371, 6450813, 4246115, 29517897, 22657489, 133435913, 81002859, 294926299, 907294273, 2010927681},
  {1, 1, 1, 11, 13, 15, 83, 171, 159, 87, 619, 2973, 2653, 13725, 12499, 1385, 126825, 39705, 216515, 89541, 400967, 1535947, 16899, 7749159, 21924495, 11107315, 123772965, 236938661, 36181893, 835221627, 213443521},
  {1, 3, 5, 3, 5, 63, 119, 25, 343, 269, 553, 2183, 959, 3825, 22189, 36473, 47371, 117229, 125003, 221149, 561111, 3468463, 1460849, 6541935, 10062581, 58682401, 104189311, 112088983, 317014137, 354446805, 2128641345},
  {1, 1, 5, 15, 5, 37, 89, 109, 497, 1013, 265, 669, 1859, 2647, 3445, 61593, 127545, 101677, 434247, 827981, 2029773, 1162977, 4157573, 15511833, 5296253, 30102689, 56653525, 52161579, 451593695, 1072751453, 901144385},
  {1, 3, 3, 9, 21, 21, 15, 245, 107, 649, 367, 1601, 7279, 15783, 4943, 47353, 46011, 166907, 337217, 869453, 1351245, 3810183, 7356445, 10994051, 28262705, 9946887, 53168857, 179192167, 30428191, 1059684615, 763530305},
  {1, 3, 3, 15, 5, 41, 125, 113, 159, 161, 1191, 3491, 3531, 55, 20857, 59353, 73531, 203339, 18759, 982621, 1453041, 656581, 750617, 5194903, 15932681, 40758767, 13067675, 1750787, 399541199, 915124001, 747995713},
  {1, 3, 5, 9, 21, 57, 21, 195, 99, 193, 1915, 2923, 6349, 15085, 24929, 36073, 52459, 118669, 352705, 312013, 1540049, 1842077, 5008171, 1063531, 15780265, 33844131, 4564915, 171194373, 378871925, 1061489785, 598321729},
  {1, 1, 1, 11, 31, 11, 73, 141, 361, 621, 1021, 2067, 5115, 12665, 26845, 19033, 12361, 41321, 148035, 301271, 1472707, 3181009, 3808181, 761585, 26084709, 42185429, 6599451, 83028563, 295911505, 440035205, 1755191617},
  {1, 1, 1, 3, 29, 11, 43, 61, 209, 923, 1753, 1937, 843, 205, 8367, 54233, 59113, 79609, 260939, 147413, 1666627, 4011251, 2523413, 7961417, 15711619, 30435057, 58107993, 28766163, 143798725, 1040605159, 610934977},
  {1, 1, 1, 5, 15, 33, 119, 209, 215, 973, 1775, 815, 6693, 7957, 14517, 6057, 105769, 66345, 37325, 841927, 582377, 492959, 5032809, 5821983, 14446261, 29433383, 56979799, 196980749, 16528557, 293367181, 1943334209},
  {1, 1, 1, 5, 17, 57, 27, 147, 489, 59, 1439, 2279, 445, 11791, 19739, 63081, 4345, 211033, 95181, 379481, 568433, 4155635, 5153243, 13087713, 19478051, 6101191, 121113199, 44567845, 20458679, 710425827, 1071171777},
  {1, 3, 1, 7, 11, 55, 1, 83, 305, 17, 1909, 405, 2325, 5293, 28559, 37161, 43755, 124409, 304335, 258771, 915071, 2561785, 91979, 2080841, 20234841, 22988205, 9095005, 10803549, 441470021, 546359271, 1364284993},
  {1, 3, 3, 7, 11, 27, 103, 157, 455, 1005, 2033, 3145, 1919, 15723, 25197, 26489, 64731, 228587, 108623, 902739, 588739, 869791, 1460613, 9190751, 26822549, 8803673, 309809, 141376343, 166953251, 295478373, 434513217},
  {1, 1, 5, 11, 15, 51, 37, 131, 503, 1007, 1795, 2421, 1335, 7413, 21741, 4729, 59241, 47661, 499203, 738823, 10779, 667261, 6269691, 3119727, 25276615, 32510235, 75570381, 186702927, 53209837, 17230149, 160170433},
  {1, 1, 3, 1, 23, 63, 69, 83, 419, 283, 583, 123, 7725, 2243, 8403, 36777, 110553, 176219, 126473, 644639, 716327, 1843789, 990955, 14388827, 5310771, 36203231, 2569059, 132680949, 57130235, 683926795, 576904385},
  {1, 1, 5, 5, 27, 45, 109, 17, 299, 65, 351, 947, 1165, 10723, 2053, 31785, 217, 185581, 55821, 325139, 646661, 2486853, 7188169, 6378051, 6920489, 13478871, 48679147, 97141893, 492633371, 1072998301, 537303617},
  {1, 1, 3, 3, 23, 61, 115, 253, 1, 931, 1481, 3187, 441, 14735, 27207, 45321, 86585, 6907, 393739, 56863, 1081893, 2850411, 6175301, 3535849, 6278715, 30217249, 27385467, 196978337, 295257543, 808255455, 121847489},
  {1, 1, 5, 3, 25, 11, 83, 141, 359, 343, 901, 1629, 731, 12841, 14357, 26393, 107529, 179181, 375307, 101905, 931363, 3902027, 4437573, 15169343, 25759183, 41368045, 42376037, 142787171, 393406465, 63828221, 1215546305},
  {1, 1, 3, 9, 7, 45, 97, 3, 299, 217, 663, 1527, 6379, 4527, 26147, 19737, 119993, 47323, 398849, 270863, 838197, 3678761, 6371899, 6804035, 11040209, 37092495, 94414511, 117844003, 298342167, 92579739, 634217537},
  {1, 1, 7, 9, 11, 53, 9, 203, 337, 713, 1517, 719, 4587, 11443, 26905, 57113, 39129, 59663, 368641, 794627, 942093, 3874881, 8134803, 10658041, 19718785, 64512949, 88041543, 80226211, 498869211, 882138977, 865382465},
  {1, 1, 7, 9, 11, 41, 125, 213, 237, 377, 361, 3231, 4223, 3263, 12655, 49081, 105913, 89791, 260097, 374787, 2070545, 1071157, 6744205, 6457765, 25201297, 58331073, 31634423, 136299447, 166457335, 1977751, 354333761},
  {1, 3, 7, 7, 7, 33, 99, 19, 117, 273, 985, 107, 3831, 10135, 19423, 31225, 88315, 109119, 486415, 625695, 1543193, 246875, 4498475, 13682045, 28616201, 46268225, 27838403, 115524223, 275570015, 705506439, 1407545921},
  {1, 1, 5, 15, 25, 41, 13, 125, 449, 169, 1149, 4021, 5663, 3077, 19163, 34745, 40553, 225949, 75527, 993041, 482561, 340853, 5216693, 2398089, 4412353, 30912661, 34149565, 220521559, 507872493, 185636979, 1014988865},
  {1, 3, 5, 9, 25, 57, 47, 103, 269, 51, 1805, 2503, 6687, 8065, 12045, 21305, 88651, 254637, 232193, 392449, 2044177, 1662823, 2994095, 15057861, 11475019, 38641461, 115554991, 82737287, 478467305, 933332021, 1530593985},
  {1, 3, 5, 7, 3, 35, 87, 225, 189, 229, 931, 3293, 1347, 1427, 3269, 15961, 124251, 225389, 177423, 542491, 1522955, 4135279, 8218617, 2762405, 1673053, 9778091, 109074485, 155814763, 132223371, 695904045, 197404481},
  {1, 1, 1, 3, 5, 31, 61, 19, 247, 9, 1667, 343, 559, 2703, 3763, 46137, 110521, 93529, 249099, 783629, 1182999, 789797, 889147, 15590159, 14452449, 34097435, 38788079, 79232359, 354000791, 186104587, 347211841},
  {1, 3, 5, 15, 31, 19, 57, 187, 109, 121, 1287, 2269, 659, 16235, 1273, 46041, 119195, 66445, 319239, 505095, 1399611, 2839361, 4067651, 15417845, 13550257, 63456735, 16106677, 148758107, 156976211, 786381585, 167107649},
  {1, 1, 1, 3, 5, 47, 59, 243, 255, 97, 1959, 1723, 1347, 3019, 26989, 3929, 91801, 29353, 395019, 858893, 1368871, 781091, 7704027, 11335815, 4929721, 47778111, 41265731, 100363083, 344748531, 902373301, 1579130433},
  {1, 3, 3, 15, 29, 35, 75, 67, 497, 731, 193, 3307, 3579, 12005, 7209, 34841, 25707, 235147, 292103, 456453, 241467, 1751347, 4527531, 8158697, 21741443, 64355801, 103315555, 183460387, 179994605, 156642417, 2049609921},
  {1, 1, 5, 9, 13, 35, 79, 213, 51, 983, 1927, 1793, 5037, 5463, 965, 55209, 42617, 258925, 289537, 883973, 1102091, 1995527, 1003965, 775211, 9735343, 4991023, 8340377, 146922517, 90523247, 379996365, 1431452609},
  {1, 1, 7, 11, 5, 41, 7, 83, 15, 411, 1775, 3515, 6755, 3249, 16425, 23129, 72649, 221199, 315267, 949133, 1362577, 3196639, 6265595, 4320199, 1447971, 18742263, 43497795, 2599995, 171489865, 1005903473, 1011644609},
  {1, 3, 5, 1, 19, 61, 3, 19, 395, 819, 1331, 179, 5225, 5333, 3601, 17929, 6123, 4557, 83849, 436875, 1750165, 3564171, 5117451, 15712227, 15564715, 8978219, 11795307, 224783729, 50980557, 249941001, 1529723585},
  {1, 1, 3, 9, 7, 5, 87, 15, 387, 609, 1465, 277, 987, 8377, 903, 2329, 58633, 58363, 54913, 850319, 1501341, 1664927, 5296311, 9212203, 26177881, 46027073, 49823373, 23478867, 100267761, 742292655, 77203009},
  {1, 1, 1, 3, 15, 11, 123, 107, 355, 333, 285, 1801, 6989, 1549, 25791, 32425, 30633, 40761, 133259, 823943, 225923, 2259939, 7233683, 1040827, 3236373, 22209477, 35384209, 116503717, 493605733, 425593207, 1729000769},
  {1, 1, 7, 13, 27, 13, 73, 111, 481, 227, 1091, 365, 5713, 5087, 27217, 4297, 108985, 197199, 457349, 195731, 1543093, 726177, 694327, 9312521, 26519851, 52935099, 20421605, 126357449, 311967047, 865619641, 501766849},
  {1, 3, 3, 15, 1, 55, 95, 213, 377, 405, 139, 1867, 2175, 4217, 28813, 45113, 17803, 199979, 236935, 247705, 9391, 1830567, 5821533, 14314817, 24596205, 60906915, 68057091, 72465239, 54988705, 1030818901, 1919224641},
  {1, 3, 5, 11, 21, 43, 109, 155, 181, 901, 1951, 507, 4389, 10815, 3141, 48361, 24251, 257389, 206211, 14989, 1980291, 1563317, 4107443, 4945005, 9457261, 24373959, 129698515, 222237837, 405349447, 165888541, 952200001},
  {1, 1, 1, 15, 17, 11, 43, 215, 501, 19, 259, 3479, 6381, 6927, 31247, 11817, 107577, 223033, 213639, 17305, 2068611, 1562323, 1100255, 2571821, 3207883, 28611387, 89020351, 21799669, 250175383, 176336951, 2065007297},
  {1, 3, 5, 15, 19, 61, 75, 41, 391, 95, 865, 1441, 7993, 13979, 24663, 19593, 31771, 191869, 115591, 668555, 664213, 1054387, 848177, 9746159, 19274119, 47939241, 43355417, 111777345, 364428051, 948736223, 204460481},
  {1, 3, 1, 3, 21, 15, 115, 213, 1, 645, 777, 1517, 2543, 11223, 3633, 34745, 37755, 122697, 408971, 1040525, 1758855, 1243627, 5966845, 12631801, 1979549, 8099617, 29815397, 89046343, 195670815, 972980953, 1042962241},
  {1, 3, 5, 3, 9, 57, 39, 211, 407, 65, 1795, 2805, 2799, 8691, 1987, 54201, 117979, 38877, 15243, 279697, 507025, 2460863, 2275611, 14062303, 13305593, 19033883, 94912461, 18319207, 437892459, 109150443, 880594497},
  {1, 1, 3, 13, 17, 55, 47, 113, 29, 139, 1301, 3303, 1129, 13947, 29821, 12297, 72729, 201131, 187013, 251801, 1218735, 1913287, 3646841, 14966821, 32350579, 64545565, 112798095, 195390129, 526540371, 673539269, 723471553},
  {1, 1, 3, 13, 5, 35, 97, 151, 477, 409, 1397, 3399, 4421, 15929, 6163, 4073, 15113, 20699, 377477, 41357, 1732027, 2198665, 5952319, 10299717, 11323921, 14261325, 59393071, 236929293, 38002193, 548237483, 1453298753},
  {1, 3, 1, 9, 21, 51, 99, 133, 149, 763, 623, 173, 4311, 11081, 1095, 3577, 73483, 240633, 20865, 503437, 617403, 2511531, 7574445, 3474829, 18382435, 38750151, 34720645, 246304655, 507371185, 372913327, 36587713},
  {1, 3, 7, 15, 13, 3, 99, 3, 195, 907, 1335, 1355, 7977, 5773, 32383, 23305, 60715, 172991, 66183, 839573, 1933243, 271003, 7646187, 5278811, 31671315, 57297327, 70311635, 202701809, 450369973, 892889767, 1926804673},
  {1, 1, 3, 9, 17, 43, 43, 217, 475, 917, 1373, 1677, 4871, 9619, 16657, 27641, 97369, 158283, 359297, 552601, 323251, 3701731, 1330641, 8680707, 25615949, 40200213, 95987157, 114683695, 230845947, 569138937, 470074177},
  {1, 3, 3, 7, 31, 31, 55, 11, 73, 693, 25, 417, 1195, 6225, 32279, 30329, 42667, 79323, 289263, 321447, 53735, 829999, 4290131, 1786513, 31192877, 56966753, 73285833, 169199523, 256469369, 1016172031, 1320834881},
  {1, 3, 5, 9, 21, 57, 127, 149, 79, 379, 1609, 2543, 6473, 16033, 27191, 14185, 42219, 97437, 90977, 7597, 1285617, 3492375, 7834205, 11384807, 12216419, 61283969, 89253943, 246035953, 43008585, 314344431, 1803410625},
  {1, 1, 5, 1, 13, 9, 81, 153, 297, 789, 1749, 2819, 3961, 11231, 24927, 52585, 103065, 76253, 187561, 590629, 311233, 3387961, 2555409, 15007233, 9212541, 52097661, 25882667, 52396721, 171349015, 768660391, 1657510721},
  {1, 3, 5, 3, 23, 61, 45, 43, 43, 133, 1481, 1543, 2991, 13739, 10287, 52825, 35771, 127645, 482731, 834095, 860661, 2675733, 7537907, 8803555, 13068237, 51682033, 27765183, 45624743, 159371331, 926574199, 383919937},
  {1, 1, 3, 9, 25, 43, 31, 177, 337, 193, 1083, 1, 991, 9725, 8379, 44953, 87497, 58427, 461153, 657201, 432339, 1528887, 756825, 1222121, 14998873, 20772499, 104617897, 175843383, 38590021, 599283923, 141040193},
  {1, 3, 5, 11, 13, 33, 65, 83, 421, 149, 409, 2443, 7423, 8847, 29599, 10969, 48923, 81181, 364835, 808245, 1956329, 3473465, 1900635, 11468941, 1638909, 6358113, 94700739, 199688823, 235081255, 899367975, 1191959361},
  {1, 1, 5, 11, 11, 1, 23, 225, 77, 585, 1505, 2525, 739, 10915, 25733, 53817, 125881, 117069, 421027, 899171, 124489, 3701039, 1918809, 11749221, 9872977, 45896457, 104741909, 237056075, 5685067, 787855629, 1173729345},
  {1, 3, 7, 13, 7, 55, 3, 223, 415, 521, 1865, 2349, 5663, 7455, 16569, 61145, 6747, 141103, 250917, 221439, 574831, 1813835, 5055431, 4185927, 2532529, 19276689, 112502645, 247561335, 404541047, 911474225, 1321570369},
  {1, 1, 7, 13, 1, 45, 121, 49, 463, 99, 1061, 2559, 5087, 13389, 11035, 65369, 104073, 9279, 84581, 918569, 860661, 3036017, 270745, 856007, 8821707, 37031245, 12876007, 111725479, 499935173, 670031939, 946480833},
  {1, 3, 7, 11, 31, 51, 35, 235, 385, 1023, 1771, 2013, 5437, 4877, 22119, 25481, 73419, 63391, 402147, 732071, 314603, 3196763, 6163763, 7395257, 543431, 22354195, 21982837, 265301605, 10451893, 860998191, 1257643457},
  {1, 3, 3, 11, 21, 3, 11, 119, 81, 737, 1093, 2377, 4055, 1121, 15767, 54553, 98091, 859, 60899, 81581, 1385211, 1947507, 2788735, 712425, 33334233, 24995805, 80848737, 198998015, 110672041, 746836671, 1025776193},
  {1, 1, 5, 13, 9, 3, 83, 217, 387, 249, 1047, 1861, 4103, 15367, 24545, 21977, 82329, 96493, 172837, 828385, 1464651, 3029851, 4498801, 15838331, 25056513, 3207679, 9804029, 208737583, 405487007, 776158505, 1305308225},
  {1, 3, 3, 1, 5, 37, 43, 183, 383, 463, 937, 1165, 1481, 959, 17047, 49769, 42331, 218267, 39849, 388925, 1045213, 2204419, 1751871, 8486327, 14540663, 32318705, 117698133, 174522001, 450960855, 1034379231, 1506781633},
  {1, 1, 3, 5, 7, 43, 127, 243, 81, 1021, 165, 753, 4711, 12965, 22049, 42649, 65545, 126699, 523117, 854895, 1783123, 4060471, 5698027, 13502569, 10877093, 16201885, 9500761, 59011887, 434389613, 77213001, 273048897},
  {1, 1, 5, 5, 3, 61, 65, 53, 425, 89, 5, 1467, 1395, 9579, 8961, 2105, 15161, 117101, 466861, 494827, 605557, 1973001, 176589, 11790497, 3975793, 24116173, 85037715, 10881627, 472520739, 698504889, 21121089},
  {1, 3, 7, 13, 11, 35, 123, 21, 83, 689, 667, 1203, 5959, 15697, 26885, 48729, 119275, 187343, 435685, 38643, 1306747, 2963763, 6835565, 1241131, 5648457, 38818707, 23338123, 17188271, 120288681, 854869373, 1059388993},
  {1, 1, 5, 13, 9, 49, 41, 101, 291, 339, 1067, 657, 4453, 1137, 21131, 1353, 17577, 33277, 130405, 349985, 427513, 3557665, 2319309, 5767499, 14156667, 60294755, 91754473, 126358557, 174598409, 967314851, 746076865},
  {1, 3, 3, 3, 17, 61, 11, 213, 27, 805, 1691, 1057, 6011, 11941, 18883, 17721, 82827, 36091, 495083, 932265, 1929797, 221107, 4861565, 11291347, 19650077, 43214931, 1019993, 126806707, 221810653, 500143659, 1541296961},
  {1, 3, 1, 7, 3, 51, 5, 63, 121, 3, 245, 2631, 3737, 16121, 26803, 61161, 18091, 140345, 103407, 690107, 915547, 3808157, 2758215, 2622145, 19701515, 46822957, 113990255, 45460385, 501117233, 919724331, 1059876545},
  {1, 3, 1, 1, 23, 51, 79, 19, 161, 107, 609, 3489, 3389, 4035, 2427, 10569, 13243, 205049, 38569, 864303, 121691, 1166311, 155339, 11554521, 8100403, 17522649, 128633865, 200498245, 271994283, 830006995, 1014309057},
  {1, 3, 1, 1, 17, 11, 101, 101, 373, 63, 1641, 285, 1333, 165, 14025, 31241, 29963, 6249, 341609, 278505, 1182947, 3787469, 3230349, 10326733, 25433079, 37361377, 54582869, 39634605, 181396621, 269062145, 1336259009},
  {1, 1, 1, 5, 1, 51, 83, 137, 45, 1019, 821, 867, 6055, 10443, 9857, 63769, 13241, 95849, 412717, 158313, 1673691, 2155163, 796257, 4461525, 13017731, 54829533, 114239019, 262372895, 405331203, 287804217, 1953598657},
  {1, 3, 1, 5, 17, 23, 25, 181, 429, 495, 317, 3219, 5963, 13945, 9969, 55673, 39979, 178793, 389997, 147689, 1165055, 3723409, 2253917, 12749941, 310983, 43908277, 98335643, 135753555, 340027377, 473305673, 940740033},
  {1, 3, 7, 3, 3, 15, 123, 191, 369, 177, 1697, 2113, 3889, 5201, 21839, 24361, 57387, 230111, 95723, 276027, 1072471, 4189123, 3512327, 10426665, 17504329, 15854585, 29046505, 230509593, 202036537, 749939367, 762154561},
  {1, 3, 1, 11, 21, 39, 51, 139, 271, 605, 1007, 3513, 3365, 3781, 6799, 39241, 107915, 34137, 47203, 898349, 728911, 920523, 2099267, 5793943, 31189797, 56900503, 108595425, 8587757, 356188957, 454779559, 1518417217},
  {1, 1, 7, 5, 13, 19, 47, 165, 249, 405, 255, 1295, 4513, 14395, 5587, 39785, 66489, 114751, 119981, 763941, 1491019, 1233127, 6312429, 12210817, 30454925, 51175287, 5407335, 52533481, 480426787, 210826955, 353455937},
  {1, 1, 3, 7, 5, 17, 99, 1, 393, 31, 621, 797, 6113, 16003, 32043, 44841, 35129, 161019, 373423, 629037, 445033, 2096315, 7606665, 2501281, 6376679, 35964805, 91078901, 228540537, 132672011, 490115555, 761257409},
  {1, 3, 5, 13, 11, 21, 65, 81, 147, 443, 775, 3671, 7029, 11749, 3339, 1993, 98123, 51837, 250469, 518835, 958813, 661129, 6992361, 13413979, 19106323, 59685999, 12201263, 197062477, 16554109, 613394323, 1036923073},
  {1, 3, 7, 1, 23, 33, 99, 177, 161, 577, 1729, 617, 3465, 11787, 17577, 39913, 116475, 253935, 45481, 931183, 1418489, 3270603, 1615273, 1633481, 2254713, 16081193, 32318913, 98167521, 38360291, 433811457, 2054814273},
  {1, 1, 5, 7, 15, 15, 53, 193, 97, 255, 1223, 545, 5153, 873, 24525, 28265, 92137, 217293, 418799, 595303, 1315015, 449517, 3206617, 1165369, 21334775, 38219375, 91098313, 39716409, 454173873, 200418533, 1860362689},
  {1, 3, 5, 1, 7, 57, 47, 121, 383, 835, 1709, 2363, 4731, 12163, 7001, 61113, 63547, 222061, 299881, 619071, 868913, 3971655, 3369697, 14433943, 9999579, 61673541, 89436451, 143909187, 3763979, 471934177, 366699201},
  {1, 3, 3, 11, 19, 33, 63, 99, 387, 95, 783, 1009, 6373, 4021, 7685, 51465, 75851, 67787, 212899, 726955, 589337, 3747335, 1466011, 360427, 4744647, 17028663, 58184201, 137567357, 463150509, 538653805, 1859855809},
  {1, 1, 1, 15, 25, 33, 73, 135, 335, 785, 935, 1927, 5847, 10501, 7719, 51737, 130313, 204313, 16807, 840177, 44553, 2657809, 6220399, 8384423, 7737465, 64762303, 77747935, 179695807, 198320093, 175256127, 848710209},
  {1, 1, 5, 3, 27, 45, 71, 215, 489, 157, 1189, 2577, 6901, 10219, 3025, 26697, 32825, 114797, 196203, 1023283, 664101, 3926655, 6149807, 7447265, 27694021, 37385661, 7526329, 244241149, 291527139, 223740665, 584766785},
  {1, 1, 7, 7, 21, 3, 97, 225, 101, 159, 293, 2789, 7955, 14829, 1209, 20473, 30345, 58095, 125615, 310333, 1333787, 333433, 8276649, 25245, 33162039, 27395149, 68553709, 176294347, 5379685, 15615601, 1330794945},
  {1, 3, 1, 5, 23, 41, 83, 63, 361, 195, 1707, 2081, 5363, 6327, 179, 42041, 120155, 229817, 380397, 264047, 1614849, 3304475, 5371047, 1692801, 14927035, 40734323, 109387017, 108280075, 5393615, 909811163, 91486913},
  {1, 1, 3, 1, 21, 51, 59, 67, 175, 363, 825, 2971, 3321, 8837, 11805, 16425, 41737, 80651, 435625, 689341, 960523, 1996819, 673931, 5470487, 20625939, 25542465, 10981763, 230414273, 428861101, 311601061, 975190209},
  {1, 3, 7, 1, 19, 3, 15, 21, 429, 675, 1589, 2615, 2575, 1537, 7139, 9241, 17835, 251199, 18729, 233323, 1974299, 2302055, 7069805, 467349, 10248379, 63334141, 15509183, 200998327, 468099593, 364379707, 874959553},
  {1, 3, 3, 5, 21, 29, 17, 115, 345, 397, 523, 1699, 7043, 11173, 3023, 1337, 88715, 123867, 413357, 184301, 1966373, 1898329, 1839547, 1485329, 28995237, 32756531, 34173259, 119937675, 215352285, 601243767, 200994113},
  {1, 1, 5, 7, 19, 63, 99, 175, 91, 433, 153, 3749, 517, 13667, 7423, 27721, 57081, 201501, 453231, 999931, 1941303, 1677691, 5441943, 9979459, 18815945, 50593025, 87442717, 61591405, 53791691, 743389943, 628350529},
  {1, 3, 7, 3, 25, 23, 53, 149, 65, 551, 1231, 365, 6637, 15137, 16319, 41033, 11371, 245279, 334891, 798689, 1483023, 1508685, 2022333, 441753, 370607, 11028871, 95130309, 215365493, 290285177, 165669175, 304375745},
  {1, 3, 7, 13, 5, 45, 11, 151, 323, 31, 1749, 409, 6753, 10503, 14991, 8041, 45915, 161567, 186597, 264701, 513333, 3867395, 6089567, 3003531, 25053159, 26385613, 70345697, 160459129, 13088463, 208904871, 88112577},
  {1, 3, 7, 3, 5, 21, 29, 117, 321, 341, 1811, 3619, 4337, 12255, 8629, 51113, 3419, 70735, 154603, 431805, 611597, 3836773, 2828733, 11672713, 13219357, 20415771, 35096715, 67276281, 508405351, 774697741, 968229697},
  {1, 3, 7, 3, 7, 3, 5, 221, 407, 671, 1763, 3669, 2353, 8175, 23489, 41257, 39963, 127087, 508395, 238655, 919835, 1718141, 5449989, 6152431, 25108823, 37332843, 27877709, 23503209, 129546519, 432119689, 1510443457},
  {1, 1, 3, 7, 11, 55, 53, 185, 247, 35, 1823, 513, 1379, 11827, 20069, 55225, 50169, 1163, 192495, 373283, 1810319, 2304173, 7544321, 672559, 28269419, 37907895, 129501337, 38276827, 487070891, 929583245, 794318529},
  {1, 3, 3, 5, 29, 51, 73, 191, 185, 961, 881, 2019, 5651, 1019, 15587, 46713, 17275, 127291, 17901, 204005, 667531, 920449, 3004087, 9037313, 17469865, 9081961, 29037451, 36813947, 11846099, 5919323, 1405769281},
  {1, 3, 7, 13, 7, 55, 59, 5, 417, 829, 453, 2339, 587, 13283, 797, 62265, 69307, 109007, 19685, 192831, 1665455, 5811, 406621, 5493817, 31550853, 34475405, 70329547, 227599491, 152303723, 39713573, 884034881},
  {1, 3, 7, 3, 11, 41, 75, 85, 65, 149, 1583, 529, 2707, 11479, 7109, 59065, 74715, 35215, 159083, 380979, 1236913, 3079859, 5060717, 15321321, 24201261, 37623847, 50222329, 142448667, 444053519, 596040557, 28360513},
  {1, 3, 7, 9, 13, 57, 37, 243, 91, 613, 665, 171, 1631, 13737, 2377, 11801, 73707, 35567, 42657, 508085, 951713, 1141645, 4628475, 11531827, 25875565, 50394017, 2440019, 178529431, 19804033, 197544497, 2126139201},
  {1, 1, 3, 7, 5, 43, 97, 53, 477, 793, 999, 3647, 2555, 7371, 19295, 55033, 8697, 219355, 241903, 805997, 1779859, 4041465, 6311421, 4678629, 30293553, 18981295, 36219767, 168699155, 45864403, 150077447, 356675649},
  {1, 1, 7, 1, 1, 9, 99, 253, 317, 817, 1559, 2081, 2529, 14611, 15997, 50729, 73977, 182543, 350441, 153897, 1074833, 3619787, 4782997, 8041173, 264329, 44260703, 129243689, 252432457, 100433035, 387657429, 841332289},
  {1, 3, 3, 1, 5, 41, 57, 121, 387, 441, 709, 1511, 7045, 8409, 13297, 5705, 69867, 231787, 119465, 896701, 598929, 2836433, 328625, 1403499, 14727249, 55376301, 36025183, 30548717, 283455761, 754525289, 235995201},
  {1, 1, 1, 13, 29, 57, 63, 183, 327, 473, 1943, 213, 3973, 16289, 2739, 22537, 1513, 151289, 83109, 725749, 1909649, 2565879, 2777599, 5807791, 18270209, 59693199, 91423885, 180945965, 164708217, 546130363, 684676161},
  {1, 3, 7, 9, 25, 15, 75, 185, 335, 881, 1041, 3339, 4471, 6823, 21121, 48985, 21403, 114095, 267105, 59809, 2018967, 523747, 5781265, 13320855, 26013833, 16250297, 121673107, 96534879, 85539663, 416361849, 1300737601},
  {1, 3, 3, 13, 23, 3, 57, 117, 511, 927, 771, 3229, 949, 15487, 11963, 19081, 16923, 64699, 277861, 29167, 877243, 3631793, 7210797, 2710727, 12881463, 11714891, 119525061, 16337773, 459108359, 457942579, 1096217025},
  {1, 1, 3, 7, 27, 19, 55, 207, 331, 705, 1945, 797, 7125, 10493, 16585, 53577, 16265, 18475, 138543, 1038259, 1250219, 4080047, 6231543, 4191411, 13575321, 40445697, 25305381, 158640829, 520128981, 505060481, 938097217},
  {1, 3, 1, 1, 29, 7, 91, 93, 459, 93, 1501, 1927, 6415, 16255, 9823, 56857, 112603, 113497, 261225, 425061, 1502383, 713907, 5936917, 2634067, 9089701, 66664021, 62733631, 256335559, 118452855, 780055623, 375211329},
  {1, 1, 5, 5, 31, 11, 97, 179, 505, 807, 877, 4003, 4377, 8851, 4239, 17705, 31545, 96285, 33773, 105847, 1633411, 2498793, 431979, 7397505, 26517583, 21323797, 114140875, 155296961, 518479195, 267571703, 17928129},
  {1, 1, 3, 5, 11, 25, 17, 131, 23, 95, 311, 1429, 2029, 13091, 23739, 46649, 120777, 116715, 256637, 552403, 2012113, 941385, 3094987, 2200495, 29918983, 37071871, 83768413, 80221477, 164105387, 914911651, 202902209},
  {1, 1, 3, 11, 13, 27, 33, 127, 481, 117, 1127, 1619, 6493, 8507, 6615, 17369, 38921, 172299, 100275, 502229, 38099, 3137801, 8372711, 677417, 19634445, 37552079, 115805547, 265494437, 366805907, 112941695, 861381697},
  {1, 3, 1, 13, 19, 27, 89, 101, 27, 235, 1579, 1701, 4421, 16037, 16239, 12921, 19323, 231209, 168757, 608859, 173283, 3719521, 3607533, 883891, 10704851, 37149011, 79851501, 863981, 533815053, 443659063, 1187649985},
  {1, 3, 1, 15, 1, 15, 3, 117, 317, 475, 1691, 2423, 5519, 1703, 2969, 58889, 18635, 87481, 318167, 809417, 1857143, 2609707, 7250509, 2146069, 21325043, 33367875, 6957583, 227275335, 316042383, 290610385, 1059221953},
  {1, 1, 3, 1, 13, 15, 19, 37, 237, 467, 1321, 453, 2169, 13313, 31499, 33433, 123545, 41803, 398713, 344149, 1344199, 3368555, 8283677, 6001285, 16752635, 40554609, 6911293, 67392241, 153012025, 295323379, 1389028289},
  {1, 1, 3, 15, 29, 55, 31, 199, 85, 285, 967, 367, 3941, 151, 20587, 9945, 44009, 138027, 95095, 981445, 1830655, 668183, 6559391, 1556221, 10603349, 22956367, 56286327, 219636509, 83438783, 732656675, 998197825},
  {1, 3, 7, 15, 7, 13, 31, 35, 117, 543, 1179, 3441, 3039, 11225, 30229, 5865, 111163, 259647, 274711, 567887, 772421, 171607, 7459371, 5884845, 17331191, 1446835, 78893481, 155915703, 130095153, 613622637, 1043060417},
  {1, 1, 3, 15, 3, 43, 1, 63, 353, 395, 1775, 3493, 5175, 13193, 25343, 42729, 41017, 165483, 262583, 963675, 208355, 4140041, 2012311, 16538825, 22508787, 19930055, 128410525, 93137471, 262887297, 842378599, 894086081},
  {1, 3, 3, 15, 17, 25, 57, 205, 411, 83, 1877, 2093, 5599, 12115, 8751, 15145, 20075, 76107, 164023, 163545, 1167857, 2389041, 801909, 4918403, 25526219, 26764013, 119267941, 149515447, 280346507, 614192343, 1257277377},
  {1, 1, 1, 11, 15, 9, 115, 99, 85, 887, 987, 4015, 7077, 3739, 21505, 51929, 1417, 261305, 437203, 647383, 872017, 2739323, 7546091, 14720173, 25843391, 35835955, 74857847, 222260621, 379053043, 317246857, 828654785},
  {1, 3, 1, 11, 25, 39, 127, 37, 329, 273, 1531, 3211, 7115, 15501, 26575, 17001, 32859, 146217, 399923, 709073, 2050655, 480503, 616733, 4133665, 10053593, 19172323, 47013075, 243230867, 212374917, 5890535, 539372353},
  {1, 1, 5, 13, 15, 3, 3, 101, 431, 645, 493, 723, 8083, 1423, 14879, 11817, 65481, 251533, 401717, 425303, 1890171, 3659515, 8135949, 16294791, 17779821, 49435829, 8445371, 55484683, 251814775, 73392663, 683955777},
  {1, 3, 3, 5, 31, 35, 37, 131, 259, 849, 325, 3403, 3627, 3295, 30885, 18537, 39115, 247387, 98461, 78167, 1385931, 786429, 2538347, 7078827, 17693065, 17575757, 80625955, 52777299, 337757351, 241405165, 1806158145},
  {1, 3, 7, 1, 9, 3, 31, 201, 379, 907, 1005, 3333, 7457, 2533, 30357, 6361, 79035, 149631, 373017, 33985, 1867467, 3236263, 8228817, 3908275, 10819971, 53043093, 58636557, 244413321, 181857021, 175758125, 2016899521},
  {1, 3, 1, 9, 7, 7, 95, 103, 121, 157, 895, 2683, 5839, 12403, 14327, 49481, 85483, 51977, 130225, 47183, 1602431, 127687, 2621839, 8173297, 25176533, 6017015, 32603779, 156146935, 82093339, 309524255, 1065433665},
  {1, 3, 7, 3, 13, 5, 55, 233, 3, 855, 859, 1115, 3883, 8041, 3353, 57705, 95867, 53215, 54107, 939205, 1776077, 2555807, 1045489, 4023227, 30966015, 13272227, 115769363, 167678435, 491752961, 330683185, 356323009},
  {1, 1, 5, 9, 3, 55, 99, 79, 263, 831, 1579, 205, 5673, 1999, 14879, 14361, 105833, 30605, 14769, 288859, 1791823, 1419451, 5573383, 7009359, 21399191, 66199331, 19981637, 195355617, 77347879, 964968327, 726983361},
  {1, 3, 1, 5, 17, 25, 85, 19, 189, 141, 877, 667, 4461, 11915, 23247, 63065, 74251, 152137, 256445, 670041, 1843553, 586925, 5327755, 5580197, 22398325, 36633957, 43600259, 97569301, 497931891, 803481351, 218895937},
  {1, 1, 5, 5, 1, 35, 15, 219, 469, 725, 1793, 3683, 3661, 15627, 30197, 40153, 121865, 240029, 26141, 626009, 621531, 3552695, 30019, 13522973, 6195805, 50405097, 90150971, 76795637, 78004755, 368531773, 1659999297},
  {1, 1, 7, 5, 27, 3, 41, 153, 431, 487, 759, 1345, 6735, 9937, 26277, 42985, 114681, 115135, 458653, 285891, 716011, 3655601, 2403793, 15568967, 22115167, 50108399, 29940201, 68882247, 26808025, 834870685, 1199785665},
  {1, 1, 1, 11, 11, 13, 41, 121, 265, 465, 1447, 5, 3407, 1907, 10037, 42921, 81641, 138265, 392083, 474963, 1829077, 1270945, 7623249, 1097489, 19479561, 13745599, 64304925, 240399767, 370392987, 145022381, 1066537793},
  {1, 3, 5, 9, 15, 63, 5, 7, 407, 83, 365, 3687, 7721, 6973, 16967, 9145, 65019, 126061, 515889, 1023943, 922471, 1296861, 2240175, 15657567, 8707403, 9086021, 57355503, 178617601, 152210469, 251126767, 1541422529},
  {1, 1, 7, 7, 5, 41, 75, 155, 417, 565, 1199, 1111, 2823, 10703, 22561, 52937, 73097, 178047, 514271, 633949, 1473473, 1377987, 4334083, 2727929, 4650685, 17158151, 92156463, 164586751, 372624039, 94946217, 1604119617},
  {1, 3, 7, 5, 7, 43, 39, 185, 105, 327, 1977, 1137, 3261, 10583, 11661, 60281, 98283, 233119, 302749, 598863, 1331427, 553151, 7253649, 14097153, 2814479, 29087041, 35208169, 33629781, 306357039, 436598101, 280351425},
  {1, 3, 7, 7, 19, 19, 103, 137, 169, 273, 1357, 3413, 7647, 10531, 32489, 57897, 8555, 188159, 435295, 307035, 779227, 284399, 4531745, 12661569, 13415257, 54883893, 36393933, 67129735, 104256923, 174831697, 1697199937},
  {1, 1, 3, 13, 13, 3, 81, 23, 161, 295, 735, 2031, 1027, 15513, 20165, 15209, 79993, 79163, 53653, 828885, 933835, 3056329, 4846703, 13761577, 13720927, 48697015, 84814295, 59619051, 71571793, 927204237, 762607809},
  {1, 1, 5, 1, 15, 1, 91, 35, 375, 207, 1417, 1115, 2237, 11749, 8509, 65241, 98937, 123453, 339865, 858583, 65529, 3337923, 2849931, 6687823, 9981127, 50863713, 66172435, 89252853, 350581789, 950539061, 624491201},
  {1, 3, 7, 3, 25, 51, 49, 219, 195, 417, 1523, 3953, 5739, 7499, 27071, 13513, 20843, 94159, 461883, 187729, 502907, 305049, 871523, 11734155, 17046617, 17957883, 71698585, 143269523, 467012227, 953840711, 2093135169},
  {1, 1, 3, 11, 23, 29, 19, 81, 421, 633, 513, 547, 7545, 29, 11909, 55417, 123129, 213627, 335123, 997839, 2058709, 2371259, 6207641, 3238013, 2099313, 5474809, 27913947, 185857697, 206197317, 544040525, 636863809},
  {1, 1, 1, 7, 13, 61, 33, 243, 221, 231, 111, 879, 2861, 1795, 27531, 601, 29961, 72617, 210239, 225557, 1323813, 923913, 8146411, 11472613, 5175615, 42270439, 120825783, 126364357, 12928267, 212989187, 777016513},
  {1, 3, 7, 3, 19, 21, 1, 141, 159, 605, 969, 3013, 6583, 2447, 19919, 29865, 77643, 84431, 155259, 1044763, 971549, 4053865, 1364773, 6149127, 4338693, 39988401, 82219245, 206431663, 103875351, 24996471, 979735617},
  {1, 3, 7, 3, 31, 9, 91, 83, 29, 873, 929, 43, 2253, 12539, 23951, 28825, 77131, 240399, 476539, 218903, 1822977, 2805555, 455579, 6073509, 17137185, 37463945, 76901107, 56357205, 449658467, 184230343, 1536140097},
  {1, 1, 5, 3, 31, 15, 87, 105, 319, 973, 1489, 3417, 3377, 15749, 2357, 59737, 53625, 176573, 227291, 1045767, 2080055, 3344671, 4921761, 7103927, 12657797, 18404073, 124245297, 184884137, 233651421, 12142061, 1881747521},
  {1, 1, 3, 15, 7, 23, 3, 81, 383, 419, 713, 997, 6873, 593, 285, 44633, 1881, 139963, 478039, 184607, 894239, 2379083, 4723097, 12883127, 6579659, 44889969, 22165005, 260045617, 17061161, 304446885, 1310301121},
  {1, 3, 3, 1, 29, 13, 29, 101, 441, 693, 2039, 2951, 5921, 12129, 12053, 8761, 51931, 133499, 337625, 268053, 1667365, 1044773, 7382429, 3639553, 17009037, 10038799, 34621727, 23092249, 64843401, 566647133, 31354433},
  {1, 1, 3, 15, 9, 29, 97, 117, 421, 433, 1017, 125, 3607, 9415, 6843, 48425, 108297, 92395, 311, 72977, 194325, 3876649, 2350029, 5261533, 13691497, 19090753, 21110021, 124102863, 292214047, 580808643, 356394817},
  {1, 3, 5, 9, 11, 13, 75, 155, 413, 75, 109, 1599, 6161, 16115, 12621, 23481, 77771, 177309, 378705, 153347, 1820949, 1645395, 6436179, 10661957, 2412995, 58160389, 113515111, 215668889, 183059499, 182476133, 1033909185},
  {1, 3, 3, 3, 11, 13, 49, 225, 401, 599, 1815, 1643, 7853, 13305, 25195, 6169, 85723, 26635, 352059, 131843, 193829, 106777, 1512329, 708681, 10104847, 53981199, 7739379, 186150405, 292041521, 218644803, 1726495425},
  {1, 3, 7, 5, 15, 11, 27, 95, 387, 931, 549, 2179, 3397, 15883, 16563, 36313, 38475, 237295, 248253, 87559, 1917443, 120387, 2896503, 9624363, 413131, 36234093, 108241771, 230178413, 84442899, 1004055803, 1028249025},
  {1, 1, 7, 3, 9, 39, 121, 5, 453, 27, 1747, 657, 2593, 1289, 12577, 25561, 111513, 163551, 429979, 895505, 69135, 4023825, 2111101, 7226989, 20862883, 3651435, 106444521, 91274409, 493591585, 708833945, 1513825729},
  {1, 3, 7, 5, 25, 25, 109, 49, 185, 985, 631, 803, 3865, 8955, 17901, 63929, 108171, 34079, 185949, 1017361, 1420817, 633397, 5295785, 10417121, 9685793, 21125935, 114660347, 35674177, 327481043, 642699653, 1724166465},
  {1, 1, 3, 13, 3, 59, 47, 49, 139, 275, 1471, 2995, 5593, 14011, 18741, 50233, 55433, 86491, 372693, 925211, 771635, 3822199, 1873657, 10816355, 26776027, 24818887, 129391355, 53649393, 341548147, 495487325, 2135613377},
  {1, 1, 5, 15, 29, 11, 97, 225, 245, 291, 1873, 2365, 767, 3419, 14943, 36841, 34409, 90381, 250487, 549893, 1874995, 3521609, 1496313, 1693053, 23815835, 2202185, 35564677, 218839223, 127717859, 906324647, 612970433},
  {1, 3, 3, 5, 15, 17, 19, 209, 359, 891, 1375, 2003, 7247, 5299, 28841, 45705, 130987, 68539, 194589, 178183, 88121, 2455563, 6035705, 1046815, 3283699, 58510055, 100911611, 6080999, 377115131, 1039299521, 1534760385},
  {1, 3, 7, 7, 9, 55, 105, 35, 77, 47, 1023, 13, 2901, 847, 10265, 1209, 38891, 210559, 306207, 684033, 1708095, 2338849, 143387, 9736549, 32378903, 37041271, 48975957, 11132669, 149450311, 743974785, 1133047489},
  {1, 3, 7, 7, 7, 5, 65, 233, 141, 277, 1333, 2357, 443, 7257, 21979, 9545, 38587, 144527, 281535, 443023, 638349, 2851465, 7900449, 11330133, 27733453, 48062701, 105981261, 228685379, 418591521, 9505955, 320037441},
  {1, 3, 5, 11, 13, 63, 41, 87, 193, 737, 1085, 2317, 7869, 10149, 12163, 9721, 82235, 107405, 398899, 317829, 1041831, 3837601, 1402127, 4666217, 14114777, 40187541, 33837557, 91825445, 125822429, 423173307, 417270081},
  {1, 3, 1, 1, 7, 57, 75, 235, 461, 857, 155, 2679, 5925, 2565, 10881, 45721, 101435, 135193, 451097, 38799, 1115521, 1946579, 5845827, 6468341, 4006673, 17327347, 31124015, 85878973, 397298125, 650910729, 1906628417},
  {1, 1, 7, 15, 13, 41, 63, 135, 433, 387, 1943, 2249, 5469, 11679, 28661, 24633, 37737, 48543, 19287, 720789, 326273, 1265079, 6238623, 15308201, 9972843, 62541023, 65104049, 114175957, 151690135, 257802317, 1277277633},
  {1, 3, 3, 13, 5, 3, 103, 161, 367, 649, 789, 1179, 4163, 5699, 16787, 30825, 11755, 58091, 417077, 281229, 1815211, 2226879, 7591897, 10718631, 9555713, 21373421, 45295219, 192232059, 467813115, 732747675, 1405192513},
  {1, 3, 7, 7, 31, 13, 45, 141, 113, 769, 1035, 457, 6709, 14989, 27311, 18425, 55163, 159855, 501503, 742295, 1700741, 2185957, 3658629, 15517033, 511545, 47092275, 109297169, 154718381, 202399045, 800883671, 1812282177},
  {1, 1, 3, 1, 1, 43, 119, 145, 111, 593, 1139, 417, 637, 4437, 17285, 22585, 5945, 37563, 96345, 803737, 1291171, 27599, 2704649, 8616103, 11992025, 10877963, 57021001, 207739525, 192794461, 259706429, 1534340929},
  {1, 3, 5, 9, 9, 33, 19, 99, 201, 685, 1793, 2621, 6857, 8769, 5623, 45145, 19035, 189037, 63345, 64897, 187833, 2260619, 6950715, 14664145, 25608613, 27247785, 102549445, 121952945, 139286553, 96327311, 187335233},
  {1, 3, 5, 5, 23, 43, 27, 189, 325, 415, 215, 1253, 3599, 1215, 10093, 22121, 51787, 52573, 463837, 747935, 546227, 911587, 1444565, 8339181, 26787703, 15285263, 28094621, 71415079, 314643575, 271401397, 535238849},
  {1, 1, 3, 13, 11, 35, 113, 173, 503, 19, 1459, 503, 5363, 3967, 13945, 29353, 65161, 179067, 110229, 685971, 637355, 1077929, 6103717, 1138399, 13510187, 550731, 99738399, 58797211, 93526359, 526351985, 1884897217},
  {1, 1, 5, 11, 31, 49, 13, 173, 199, 623, 1231, 2495, 6581, 7957, 25321, 24345, 48313, 130397, 49043, 1003655, 295049, 1452677, 1227365, 8340031, 13894039, 47449847, 96207063, 211552317, 18622237, 161240929, 1890873537},
  {1, 3, 1, 9, 23, 3, 79, 149, 505, 937, 1839, 3701, 1673, 8589, 8031, 54489, 30395, 212457, 375025, 128671, 153787, 2593175, 1604157, 6364945, 10277729, 30479863, 47722861, 157894609, 317840357, 545260167, 964703553},
  {1, 3, 3, 5, 21, 27, 107, 11, 505, 407, 177, 3593, 4729, 12773, 11685, 19225, 7291, 168123, 398173, 315549, 173747, 2000627, 1210483, 6044145, 32398079, 433369, 50656241, 251864273, 337285277, 602581885, 2073353409},
  {1, 3, 1, 11, 29, 49, 79, 53, 61, 895, 2035, 563, 5613, 6065, 6207, 6681, 13083, 104105, 356595, 430741, 1694857, 2743175, 1454317, 1847781, 12964503, 14292779, 99286539, 125338949, 465995865, 282983447, 678869185},
  {1, 1, 3, 7, 1, 53, 3, 215, 99, 865, 1749, 3533, 4305, 1243, 28463, 26873, 88681, 38635, 439903, 147769, 485085, 2939243, 162191, 10766075, 24607433, 3891133, 94676869, 250523849, 165704387, 689416695, 519850305},
  {1, 1, 1, 13, 31, 59, 115, 53, 403, 909, 847, 103, 4967, 10623, 30073, 10409, 77161, 5337, 287029, 461543, 297283, 3671403, 676733, 3698523, 9688069, 44307063, 71398415, 164150623, 347518455, 309173969, 1467183169},
  {1, 1, 7, 5, 27, 1, 119, 83, 457, 81, 395, 811, 6221, 14337, 541, 36281, 67001, 48479, 376893, 477667, 94793, 3131215, 4914043, 2561553, 25611993, 19708803, 129225203, 121472245, 199071657, 85965461, 57796929},
  {1, 1, 5, 5, 5, 53, 83, 117, 269, 327, 875, 101, 3343, 715, 26339, 3401, 67593, 47181, 391965, 554685, 301549, 2988875, 3617773, 4192277, 8706159, 56729363, 102221181, 68075655, 16051203, 61211611, 857621185},
  {1, 1, 1, 11, 31, 39, 121, 147, 305, 383, 1211, 1897, 7647, 11687, 18907, 5385, 21481, 234825, 71443, 236199, 847839, 2664273, 3333051, 7866841, 19982567, 55856531, 15077937, 241806871, 209633903, 315231107, 862125761},
  {1, 3, 3, 15, 23, 53, 17, 85, 395, 503, 61, 1745, 4713, 4641, 13787, 12281, 74683, 9195, 83031, 925503, 1043965, 4152697, 8095677, 2803059, 6401039, 13314021, 8004537, 131166337, 497877401, 295006627, 976177857},
  {1, 1, 7, 7, 27, 1, 105, 29, 287, 37, 959, 975, 4427, 4705, 10175, 44873, 2137, 249999, 144287, 127907, 33481, 699713, 2294549, 2727047, 22754525, 29628455, 58256519, 185972595, 399830121, 694361175, 1658390081},
  {1, 3, 3, 5, 7, 63, 57, 199, 27, 107, 1095, 3923, 6969, 713, 11619, 49689, 77243, 176811, 200157, 453167, 909303, 2475265, 6670095, 1146099, 26178803, 50235599, 13892123, 97864817, 182835169, 1031497003, 1714548673},
  {1, 3, 5, 1, 5, 49, 85, 45, 449, 45, 49, 3419, 1109, 455, 15917, 64569, 98571, 223933, 174841, 95597, 374601, 47661, 6381173, 7678841, 5038965, 5186921, 26504547, 226765773, 314678271, 775987205, 244747329},
  {1, 1, 1, 5, 13, 15, 39, 27, 467, 85, 1537, 3055, 1977, 8829, 25231, 48153, 103065, 21897, 279197, 997301, 66551, 173695, 509507, 16319387, 4458365, 44959753, 108404199, 57549953, 461528453, 620155271, 664223809},
  {1, 1, 1, 15, 1, 47, 23, 121, 147, 547, 1865, 1491, 779, 3515, 12667, 21321, 40457, 142345, 171415, 676537, 1309911, 911391, 4372705, 14694491, 31751051, 32176209, 93743099, 88373699, 395981283, 991056851, 700604353},
  {1, 3, 3, 1, 19, 5, 77, 101, 1, 721, 1149, 2967, 4925, 11889, 16655, 42041, 93627, 143307, 328729, 67259, 583117, 2595925, 2372685, 15140345, 11140809, 7971813, 90188079, 230159941, 150281705, 114895879, 1811401537},
  {1, 1, 1, 7, 1, 35, 95, 239, 127, 855, 1031, 455, 7631, 6039, 21983, 10217, 7593, 131945, 81247, 1033529, 1071579, 967703, 5891319, 5948887, 10036863, 35831471, 33620975, 31524519, 53027583, 325145271, 60359873},
  {1, 3, 7, 9, 23, 43, 75, 105, 335, 223, 1825, 3217, 413, 7473, 30005, 27353, 125819, 54303, 223409, 305151, 1961795, 2624595, 5782625, 4517255, 3939479, 41651321, 18311129, 166783093, 430728809, 801841037, 504842433},
  {1, 1, 5, 15, 29, 9, 43, 145, 223, 523, 511, 323, 5955, 11141, 22533, 15945, 50681, 208797, 90935, 633445, 616657, 3972579, 6223849, 7486055, 11182627, 893223, 4809227, 207418763, 420314605, 456010445, 1887593921},
  {1, 1, 3, 1, 13, 61, 93, 133, 461, 233, 383, 693, 7347, 3165, 27493, 13737, 117241, 124187, 331833, 1021557, 2043605, 2089861, 5862749, 12376469, 2500785, 28930023, 116428781, 153066315, 123509333, 507285725, 733373761},
  {1, 3, 7, 1, 13, 45, 113, 207, 53, 1007, 815, 1145, 2937, 289, 22195, 31033, 24635, 60719, 167001, 425637, 821061, 3413161, 5174039, 13236461, 7011287, 34630439, 39392193, 73654641, 404663881, 823114155, 1878668993},
  {1, 3, 5, 5, 19, 17, 113, 89, 19, 1023, 1625, 3277, 697, 5187, 15433, 64953, 88491, 90077, 79293, 871803, 2018537, 3262377, 2504049, 7802059, 468935, 3466977, 5546789, 263283857, 212390107, 1005843729, 1976689857},
  {1, 1, 3, 13, 21, 15, 15, 197, 409, 391, 1993, 2475, 3189, 4431, 29585, 35257, 41673, 58715, 311733, 454253, 2057191, 1246903, 3280413, 1676113, 27272047, 36528433, 49260435, 96272189, 370964631, 845547641, 1017780929},
  {1, 1, 5, 5, 31, 7, 111, 231, 187, 543, 45, 3863, 3811, 4573, 4437, 43337, 1721, 127645, 160317, 56679, 994271, 3844343, 391727, 12315155, 2708279, 6633029, 99581215, 76655371, 141436517, 696650653, 1559944897},
  {1, 3, 3, 7, 19, 7, 123, 23, 79, 513, 189, 3663, 1291, 13257, 8949, 33673, 114843, 196059, 219327, 270459, 1049039, 3653843, 5909215, 2224871, 4856201, 42556949, 34157671, 60024915, 221727121, 254289213, 1924677953},
  {1, 1, 5, 13, 3, 53, 109, 133, 157, 223, 651, 3059, 6055, 14455, 26903, 12521, 52521, 145837, 356949, 376251, 972397, 39349, 4560429, 16742245, 8719175, 50137747, 68547627, 204475375, 330308543, 344994335, 1401948353},
  {1, 1, 7, 1, 23, 63, 59, 229, 17, 199, 643, 637, 7631, 13647, 7399, 59465, 1961, 227503, 310361, 934831, 586359, 4191651, 2314861, 5399545, 22855119, 58999483, 55362949, 17484039, 32694727, 274803487, 2122941121},
  {1, 1, 1, 3, 1, 51, 119, 67, 335, 543, 913, 3565, 4795, 13405, 7463, 35241, 114809, 198665, 116507, 645433, 412491, 2413983, 1359515, 11625063, 7513335, 65007449, 127024901, 11026579, 328785285, 250695935, 634636993},
  {1, 1, 5, 3, 31, 5, 91, 97, 23, 223, 837, 1353, 1929, 12043, 10039, 63417, 47273, 10861, 495003, 744167, 631581, 2224435, 3942665, 4185775, 27476663, 15483485, 120340865, 152622369, 200261939, 575419343, 447504577},
  {1, 3, 5, 7, 19, 3, 79, 171, 301, 687, 1545, 355, 4709, 12965, 16797, 63481, 34139, 248541, 504831, 993211, 811835, 3911495, 301027, 9438181, 30534535, 13872321, 128585307, 82707885, 262275261, 856035061, 824057537},
  {1, 3, 5, 11, 11, 49, 111, 123, 251, 569, 1605, 401, 5439, 13519, 8847, 19977, 98027, 181037, 79507, 364003, 1932553, 1954055, 7352627, 12040099, 14473777, 6626893, 120923417, 216342823, 151246103, 757233783, 593144641},
  {1, 3, 1, 3, 3, 53, 7, 55, 369, 633, 181, 4037, 2993, 15815, 8661, 32057, 72907, 2489, 58427, 793259, 919341, 522607, 1955647, 14739337, 14562417, 66651357, 104139485, 256182409, 338759391, 222788237, 1465198401},
  {1, 1, 1, 13, 31, 29, 75, 167, 279, 597, 539, 1791, 8013, 4387, 9717, 40889, 102633, 262137, 437, 104103, 1658661, 2225427, 3119023, 7753391, 6913341, 56713011, 12115543, 210047349, 383191371, 138218525, 1067394113},
  {1, 1, 5, 7, 17, 15, 99, 183, 211, 49, 225, 3143, 4537, 13141, 23375, 65497, 98201, 4045, 396511, 352233, 1016087, 380203, 1828847, 2416731, 27861241, 59598729, 133720783, 242301665, 513884717, 430551623, 1031961921},
  {1, 1, 3, 5, 3, 59, 25, 149, 467, 69, 1939, 1007, 2765, 4693, 29815, 16825, 95577, 109195, 291165, 439547, 392467, 1574177, 6506429, 346523, 18582109, 5365451, 109324807, 172059765, 175019005, 805276575, 1722857025},
  {1, 3, 1, 3, 17, 33, 119, 189, 447, 251, 879, 177, 5395, 13487, 9587, 9897, 56235, 178505, 154971, 345209, 1585977, 3067679, 6984485, 4638919, 919667, 26127575, 16921465, 142182299, 373788295, 455734891, 545549761},
  {1, 3, 3, 7, 15, 31, 115, 3, 21, 817, 475, 1849, 6041, 12541, 18701, 57977, 58779, 138459, 396543, 206759, 716055, 4133147, 3690859, 11834877, 29300665, 13025811, 44942017, 233229345, 285265045, 110724181, 792406337},
  {1, 1, 5, 13, 31, 33, 7, 115, 361, 587, 1919, 1007, 3537, 7493, 19357, 63321, 7961, 83805, 269813, 970151, 1832249, 3768607, 5972475, 8996145, 26592899, 283063, 126473847, 92642425, 471968301, 293978485, 401600449},
  {1, 3, 7, 13, 23, 35, 15, 111, 123, 633, 805, 1983, 2109, 14477, 4985, 26617, 35355, 125567, 39861, 939071, 800267, 512631, 1991399, 7838643, 31332561, 18774733, 100644215, 261628837, 64622421, 76967409, 1044295489},
  {1, 3, 3, 11, 25, 13, 11, 205, 97, 893, 927, 1291, 4007, 13593, 29693, 14377, 6427, 225627, 159603, 478385, 293381, 4108803, 4677333, 12922617, 22009557, 59836135, 133366195, 24580607, 179506193, 172848117, 431572033},
  {1, 3, 5, 15, 9, 13, 121, 89, 215, 823, 1389, 1581, 8095, 4707, 16061, 45385, 99595, 140925, 140599, 682145, 1459253, 1000497, 2894913, 16309711, 6325519, 38638997, 61961765, 258663543, 222166171, 209972197, 1147119297},
  {1, 3, 1, 3, 23, 39, 83, 23, 47, 941, 1419, 2389, 5699, 7519, 5829, 43113, 29611, 35321, 295227, 220095, 1778751, 1862715, 6325439, 14713863, 20762021, 60385507, 74286909, 191928123, 200313399, 662287741, 1200462401},
  {1, 3, 1, 9, 23, 43, 79, 237, 93, 203, 695, 225, 5645, 3591, 16775, 9433, 95371, 81449, 310673, 446079, 970931, 2169591, 6263141, 6967253, 19198819, 56219599, 61904313, 26568869, 42708527, 63695647, 605331905},
  {1, 3, 5, 3, 15, 19, 89, 129, 375, 125, 225, 1323, 2267, 11607, 17937, 38025, 72875, 19677, 230651, 962471, 1718955, 3579377, 8249321, 3460095, 6920677, 44247449, 91646755, 148681955, 342469359, 627222905, 1039551553},
  {1, 3, 3, 1, 31, 37, 93, 133, 377, 959, 707, 621, 7179, 15493, 30287, 6569, 78459, 120203, 305497, 1038455, 1287085, 2450565, 2245005, 12106497, 10288231, 17753627, 45362901, 232678723, 502902733, 737343351, 298545345},
  {1, 3, 7, 13, 5, 13, 15, 1, 37, 525, 1641, 2829, 6139, 4069, 19187, 6121, 94715, 71023, 233621, 241645, 1479589, 846583, 2117017, 13792061, 12328069, 26498225, 26555989, 13406307, 497392733, 931185707, 2112076865},
  {1, 3, 3, 9, 17, 3, 67, 97, 375, 845, 403, 973, 3919, 2275, 31627, 35337, 88587, 122699, 453585, 338169, 123531, 1401563, 5759897, 2781663, 16978373, 51974475, 26056581, 121286391, 159313099, 520517363, 380507201},
  {1, 1, 3, 3, 25, 7, 91, 67, 271, 465, 481, 3477, 5229, 241, 8411, 37849, 87353, 8107, 290459, 808929, 1663407, 1431763, 8164187, 7983431, 15028441, 51024937, 120730941, 152176885, 456788361, 413803203, 760357185},
  {1, 1, 1, 11, 1, 41, 109, 115, 75, 787, 309, 2887, 179, 9073, 13895, 59401, 85657, 188937, 143187, 262009, 1125265, 3732101, 4973419, 13247987, 16072523, 24969693, 22744079, 88408955, 487923977, 831475151, 114134465},
  {1, 3, 3, 15, 11, 31, 113, 91, 303, 907, 1933, 2167, 7799, 11821, 20659, 17193, 104763, 115179, 423063, 41699, 2088087, 2145241, 1057299, 12852999, 12090131, 35530789, 106123551, 11510335, 484075925, 183755067, 1513765825},
  {1, 3, 1, 15, 27, 17, 21, 41, 99, 137, 1397, 929, 5819, 11977, 6201, 4009, 38683, 30201, 20663, 761779, 1228425, 3601053, 6651553, 6379323, 7189713, 35033805, 90542857, 266113891, 343912721, 741706177, 1649213761},
  {1, 1, 7, 13, 21, 29, 47, 239, 287, 305, 899, 2711, 1723, 3315, 199, 56329, 77193, 20271, 53269, 1031789, 595861, 415127, 2269943, 3509127, 7784121, 59268395, 11870719, 103523235, 410743195, 646634479, 303635777},
  {1, 1, 1, 3, 31, 21, 101, 149, 107, 761, 1197, 1703, 4803, 8411, 10649, 31497, 73417, 131929, 12539, 399783, 1770413, 2665165, 4952829, 8288211, 3005921, 8719221, 122192271, 48271771, 459059139, 195399121, 769638721},
  {1, 1, 5, 15, 23, 45, 109, 221, 85, 619, 169, 1013, 3305, 9451, 26189, 10873, 96169, 14653, 124343, 248367, 1796533, 3955429, 4788405, 8966445, 15602611, 61944497, 49765037, 208013249, 312879923, 589168789, 2020261313},
  {1, 3, 5, 13, 7, 57, 19, 153, 231, 627, 565, 1595, 6309, 5037, 25505, 57013, 81543, 76497, 166417, 64323, 1841277, 728351, 2519405, 10519707, 10966983, 53248089, 26446119, 165710273, 77104945, 861050021, 439915121},
  {1, 1, 7, 7, 1, 45, 43, 79, 271, 59, 219, 2255, 1785, 7919, 24061, 4029, 39925, 213747, 365523, 403405, 615265, 4055839, 5908939, 356723, 3412759, 41251095, 124145811, 1658533, 333369507, 904631145, 1272252721},
  {1, 3, 7, 5, 31, 57, 57, 231, 33, 227, 531, 679, 1141, 85, 19777, 34133, 61703, 125267, 450585, 821587, 1289309, 2103669, 3179923, 13881845, 30569135, 14496199, 97842203, 256061289, 475239481, 933879013, 1082081713},
  {1, 1, 3, 15, 11, 59, 59, 169, 459, 693, 907, 1191, 3783, 12809, 6263, 29733, 105949, 255127, 52091, 259671, 1561047, 3716455, 2886645, 5840983, 17899873, 44050135, 68674851, 180337051, 456292997, 971485723, 516837937},
  {1, 1, 7, 13, 19, 21, 105, 65, 267, 141, 1547, 781, 7295, 13565, 17775, 49653, 55229, 119251, 1273, 495967, 1493489, 162581, 5283205, 2487135, 17161369, 50229623, 19765097, 150006691, 11638617, 479906275, 1394964113},
  {1, 3, 3, 5, 31, 63, 97, 155, 477, 661, 329, 797, 2539, 4061, 10537, 5493, 55975, 251863, 116889, 709955, 1024091, 908141, 1541111, 5273193, 25984841, 19123949, 5449785, 127838303, 173416761, 600458837, 1155433361},
  {1, 3, 3, 7, 11, 17, 119, 89, 71, 103, 1043, 413, 6035, 12829, 11559, 61045, 89759, 98263, 109691, 69975, 91133, 4123435, 789933, 10832619, 17114323, 43719103, 96978129, 55201647, 267052889, 674243227, 149966097},
  {1, 3, 1, 9, 5, 19, 53, 185, 103, 629, 2015, 1257, 5163, 10581, 13449, 58197, 103463, 247637, 296597, 628561, 1528903, 2461553, 5552037, 1006499, 1454817, 13161299, 64100717, 65193831, 194960097, 392938261, 1417516753},
  {1, 1, 1, 5, 23, 35, 25, 129, 179, 959, 677, 2249, 6315, 12151, 3459, 37853, 57357, 187765, 57905, 758723, 294631, 2821397, 7055645, 3161831, 4415947, 47720249, 94446053, 258253391, 418413283, 348247743, 1542331985},
  {1, 1, 1, 1, 9, 47, 93, 45, 35, 45, 265, 2065, 6225, 25, 27135, 46693, 47613, 97173, 164341, 702557, 922107, 1193769, 5931457, 2815623, 19956009, 9340181, 105284277, 145129989, 456299725, 539071467, 750529361},
  {1, 3, 1, 11, 21, 53, 127, 163, 311, 667, 597, 1561, 4515, 23, 9551, 37725, 94783, 198261, 388599, 700609, 1041641, 3990651, 7212039, 10149339, 10337671, 4649017, 113473285, 172965959, 62997763, 924421971, 1987604561},
  {1, 1, 3, 3, 7, 47, 105, 211, 241, 95, 389, 899, 6001, 8129, 19889, 5125, 16325, 182039, 418327, 825179, 1802611, 1389573, 5647391, 8642021, 28769571, 40331537, 30951439, 256655413, 87586045, 1036873829, 745898225},
  {1, 1, 3, 15, 29, 45, 9, 27, 483, 799, 269, 1811, 4493, 7109, 22149, 51221, 38165, 146263, 404827, 211521, 5185, 3926029, 4593695, 1370327, 23374435, 21735761, 123349407, 138879433, 15927481, 513220769, 2134350321},
  {1, 1, 3, 3, 29, 5, 57, 205, 187, 615, 1677, 3987, 4577, 8799, 16311, 59005, 61405, 194807, 216183, 793281, 1271513, 1861181, 7611049, 10328911, 19685315, 34502321, 41286047, 64640037, 26832675, 409841699, 1382849009},
  {1, 1, 5, 3, 15, 5, 91, 101, 319, 445, 1261, 2039, 4071, 8249, 11611, 31717, 14445, 197777, 200375, 14155, 707545, 1094263, 8228353, 963555, 31262393, 38824033, 125595667, 54530915, 415242101, 792751575, 1572864305},
  {1, 3, 7, 11, 19, 17, 1, 185, 153, 579, 1001, 2031, 2295, 16335, 24771, 26685, 103439, 28147, 508727, 663263, 313805, 810621, 5541517, 8582765, 20941175, 1796957, 20010803, 99168387, 382067715, 683307471, 2044359473},
  {1, 3, 3, 15, 13, 45, 93, 185, 319, 667, 1085, 93, 577, 11551, 11355, 27645, 48367, 127351, 400691, 730449, 998241, 3831961, 1725373, 9360603, 13842415, 54267601, 107001697, 17484165, 422050843, 97522895, 959493649},
  {1, 1, 7, 13, 3, 61, 45, 191, 51, 981, 1151, 2715, 2503, 4147, 4587, 43341, 97581, 256691, 231737, 776527, 470361, 2482321, 2491227, 14910711, 5703481, 48765731, 9708031, 7354827, 507047759, 560532295, 1191796721},
  {1, 1, 3, 3, 27, 17, 71, 141, 57, 981, 1033, 333, 4639, 15885, 1039, 60597, 8893, 32343, 234615, 298695, 1396557, 2720987, 5619513, 11453909, 8088753, 16179421, 65579017, 254603243, 405388201, 192597795, 1399177361},
  {1, 3, 3, 15, 21, 55, 33, 123, 357, 893, 829, 4045, 5027, 11727, 13357, 63933, 84535, 226423, 227923, 693321, 744187, 2986885, 1333527, 11405553, 28543137, 16407929, 1799625, 263333831, 529929323, 327820465, 1156226193},
  {1, 1, 1, 9, 31, 47, 27, 223, 311, 205, 179, 3411, 4019, 10997, 28115, 46165, 1485, 257237, 514749, 435915, 2094171, 2549431, 5500723, 15200619, 1274497, 46143391, 65961687, 100892751, 315135529, 239858031, 694463857},
  {1, 3, 5, 1, 3, 39, 15, 7, 501, 641, 735, 295, 2005, 12641, 19779, 39685, 53647, 100241, 22973, 14535, 1426003, 853651, 6244299, 430137, 6598957, 27635899, 11473427, 111620281, 45444269, 193068583, 572711217},
  {1, 3, 3, 1, 15, 1, 75, 243, 329, 267, 1323, 2285, 5389, 11881, 15737, 61733, 9319, 87319, 158493, 591571, 1473781, 359303, 1641895, 3948757, 10305559, 60343551, 105891121, 79612017, 234145069, 584293181, 1712689937},
  {1, 1, 3, 3, 13, 17, 101, 99, 209, 939, 1147, 3221, 5159, 3435, 183, 14125, 78173, 238391, 342519, 133457, 1497165, 1069217, 2785751, 14000853, 4990327, 3468975, 94142409, 143902235, 439804111, 822348123, 17611921},
  {1, 1, 1, 1, 27, 43, 29, 179, 179, 659, 807, 313, 4165, 963, 11317, 12429, 113493, 256949, 224213, 286287, 864511, 4070817, 8341455, 16288439, 8010031, 34213243, 80415749, 160764969, 511252823, 17262865, 1879146577},
  {1, 1, 3, 13, 9, 51, 125, 245, 381, 555, 1383, 3887, 2045, 12829, 12029, 6389, 63605, 124503, 93273, 720341, 1521111, 362409, 2206505, 455513, 22864655, 34644611, 73444243, 262975601, 286240465, 801110457, 1674836273},
  {1, 1, 1, 9, 29, 39, 55, 127, 235, 617, 1553, 3133, 7735, 14725, 16733, 27797, 23285, 21461, 386141, 846281, 451795, 3214483, 5440691, 15752487, 757541, 40372493, 94374185, 153098883, 337643425, 480272849, 1306579601},
  {1, 1, 3, 5, 15, 9, 47, 217, 89, 987, 1083, 1045, 4745, 12915, 13719, 46157, 80605, 253879, 426737, 874323, 2003789, 2846179, 7816301, 830549, 30604527, 57545455, 25106217, 241054901, 22829655, 819505859, 428946065},
  {1, 3, 3, 7, 23, 3, 35, 79, 45, 435, 1549, 2645, 2831, 10359, 10041, 52317, 97383, 8439, 498203, 226635, 1405167, 4096655, 5095539, 6766737, 22921471, 58639049, 99500841, 181491883, 383572051, 297699093, 223670929},
  {1, 1, 7, 15, 31, 61, 25, 223, 511, 319, 487, 1677, 739, 7097, 18417, 7269, 41285, 140051, 162459, 7891, 962513, 957653, 814619, 3815211, 33005523, 35383755, 31198529, 39585871, 16519997, 726940749, 1173132305},
  {1, 1, 7, 5, 19, 21, 123, 237, 299, 367, 1341, 1449, 2949, 8629, 11051, 30805, 14893, 112339, 429873, 113375, 1139025, 1085831, 5479977, 4334263, 11151179, 51555937, 19312645, 118577025, 142540065, 275846319, 517947473},
  {1, 3, 7, 7, 31, 53, 125, 33, 257, 719, 1297, 895, 5095, 10237, 12309, 36981, 37079, 196691, 325595, 529363, 1834201, 723633, 473797, 9136965, 25897307, 46066821, 53407379, 51078259, 46421553, 184945753, 2017601841},
  {1, 3, 1, 5, 31, 59, 73, 211, 97, 209, 1289, 4033, 6143, 14275, 7997, 56141, 110263, 68981, 390425, 824587, 240415, 1439525, 6189423, 5075381, 27890477, 48730397, 56128693, 113224395, 122793687, 367962449, 265739089},
  {1, 1, 5, 7, 31, 5, 75, 105, 389, 985, 9, 4033, 1185, 7821, 19083, 39541, 96877, 5009, 62451, 556827, 814857, 2859879, 6733069, 16645721, 25059629, 51930541, 19262261, 87961045, 41961241, 370997663, 1340441425},
  {1, 1, 1, 15, 11, 39, 73, 253, 275, 813, 25, 3441, 2493, 5873, 3739, 57989, 58925, 53333, 396795, 241439, 2036491, 1886069, 3655057, 1826695, 10977705, 53445573, 49513237, 119901673, 368025829, 124694271, 353599825},
  {1, 3, 7, 1, 31, 19, 119, 5, 109, 397, 1329, 3347, 5941, 12449, 2533, 17349, 64983, 83411, 250909, 928019, 1542951, 570235, 4525913, 16201537, 13362601, 25061109, 83191927, 47768609, 196990269, 891259681, 1305696113},
  {1, 1, 1, 1, 5, 59, 61, 175, 435, 985, 65, 3781, 5425, 15073, 16361, 31685, 128485, 21589, 431573, 73489, 1389359, 3492665, 5030739, 10477079, 32056789, 44405341, 52108369, 266329093, 93968741, 208592957, 1899533585},
  {1, 3, 5, 7, 31, 13, 53, 87, 69, 305, 1455, 273, 2197, 4277, 24423, 20885, 19295, 248593, 315195, 182299, 469025, 2824217, 2680851, 7866721, 21973181, 62840811, 130697381, 100418169, 9236449, 395337667, 191904337},
  {1, 3, 3, 15, 13, 13, 91, 171, 71, 583, 15, 3599, 6801, 10041, 26097, 57893, 126727, 110679, 146259, 273425, 642049, 4115551, 406639, 2790907, 6575163, 32342179, 121471395, 159552221, 288786293, 724482885, 778840625},
  {1, 3, 3, 5, 5, 13, 91, 225, 63, 69, 1795, 341, 461, 5015, 9471, 59421, 101951, 192055, 189113, 650777, 1798697, 2609791, 7714437, 8046339, 20467985, 31126607, 131823697, 114115633, 510027923, 923449851, 609686801},
  {1, 3, 7, 5, 21, 55, 109, 39, 459, 925, 229, 2855, 5807, 2117, 31739, 60245, 33711, 106387, 273017, 422425, 624147, 611913, 7361195, 651055, 20619889, 30746393, 62023971, 247800715, 463951625, 120709399, 1975910961},
  {1, 1, 3, 3, 1, 5, 17, 177, 401, 727, 1555, 3097, 1243, 5933, 14579, 53877, 105485, 21143, 5111, 775709, 1360409, 2303525, 2378389, 10639301, 24656451, 38103879, 53146197, 108338055, 463929401, 538536111, 139416913},
  {1, 1, 7, 3, 19, 19, 37, 87, 105, 73, 197, 4067, 6237, 10553, 9207, 50085, 37277, 210259, 110135, 231967, 902671, 2275865, 2126379, 14282701, 19221029, 25508145, 98872759, 12219353, 428747813, 311578835, 157985649},
  {1, 1, 3, 15, 1, 55, 119, 115, 441, 3, 1003, 1631, 197, 12929, 25385, 39877, 34149, 234967, 81371, 664093, 1153563, 2661955, 6643295, 1347189, 19035327, 15697023, 128025443, 170690297, 76977613, 383169677, 71861809},
  {1, 3, 7, 11, 31, 1, 119, 49, 467, 647, 685, 2771, 3689, 11049, 26787, 17637, 92751, 20947, 80631, 891283, 1062813, 2010875, 302853, 12978559, 13011667, 46946321, 38661983, 74169965, 6802069, 708996183, 568477169},
  {1, 1, 1, 11, 19, 19, 21, 73, 459, 935, 615, 371, 1099, 14407, 10375, 40285, 6237, 27189, 121407, 281735, 72367, 38601, 3990325, 1018159, 29289011, 53160091, 129528319, 166056935, 102090363, 263399003, 624144369},
  {1, 3, 5, 13, 15, 3, 107, 179, 259, 677, 1101, 315, 7673, 14639, 11241, 15485, 15975, 117169, 477777, 409995, 824455, 3956103, 6948335, 8894687, 3418585, 35716465, 97759775, 161004157, 186587411, 1009611357, 822604273},
  {1, 1, 7, 9, 15, 21, 93, 25, 349, 23, 1087, 27, 5691, 12997, 29301, 2133, 5525, 131091, 281885, 218243, 730017, 3700113, 6449501, 7729497, 23393187, 25433883, 26812247, 111707799, 184522841, 230279401, 308634993},
  {1, 3, 3, 5, 7, 43, 1, 195, 69, 753, 1315, 2629, 3259, 5595, 19439, 9645, 23551, 219767, 341945, 837275, 885903, 447149, 7346111, 2720945, 4158925, 26876327, 43781937, 257694439, 433337439, 278691763, 1316266769},
  {1, 3, 5, 5, 31, 9, 75, 217, 217, 197, 1925, 2033, 3585, 15219, 20251, 1549, 11695, 193777, 189561, 779675, 1420461, 2833127, 6035757, 333445, 2341041, 29817297, 23593461, 203953157, 227021415, 378016463, 933777233},
  {1, 1, 5, 11, 17, 31, 3, 209, 315, 49, 949, 2267, 4611, 4375, 16431, 57245, 35709, 4145, 507583, 588437, 387235, 368791, 2982301, 9388967, 3917573, 55134697, 51007631, 20540391, 446773739, 255658739, 164477169},
  {1, 1, 7, 9, 17, 35, 13, 115, 119, 553, 1527, 2857, 3599, 391, 25101, 27101, 93045, 72883, 355485, 239261, 1045655, 2140601, 1563631, 7573203, 11855189, 63037819, 36761405, 37859363, 523996211, 258905025, 1704354769},
  {1, 3, 3, 15, 13, 59, 17, 177, 301, 719, 909, 1663, 5033, 1129, 529, 26085, 49543, 205271, 507603, 431761, 279223, 2843285, 1108013, 8910905, 22354491, 16210345, 50591955, 69515981, 391000549, 188876005, 228752689},
  {1, 1, 7, 5, 15, 13, 99, 157, 379, 975, 1019, 2251, 3807, 10621, 351, 1717, 96061, 256659, 468401, 729475, 1896585, 108975, 8001657, 16562311, 5155563, 29473447, 106684839, 256194563, 411164449, 802358035, 611666545},
  {1, 3, 3, 13, 5, 57, 5, 31, 361, 981, 883, 3723, 2259, 5151, 11783, 1725, 118623, 236727, 75569, 442265, 870845, 2864545, 3182763, 12701869, 13455801, 64393223, 98079407, 259538703, 510214627, 404203003, 745208561},
  {1, 1, 1, 13, 1, 43, 125, 19, 77, 509, 1817, 3795, 1863, 8401, 27253, 58757, 108565, 171861, 46361, 544149, 426639, 553193, 8135919, 1203497, 7159681, 8739709, 89019543, 47928947, 439768781, 764687225, 1851038065},
  {1, 1, 5, 7, 19, 41, 21, 151, 89, 189, 769, 1937, 4497, 13607, 24691, 48221, 104973, 236721, 23027, 1023639, 677285, 4030345, 5670083, 3694013, 28184305, 18920629, 127791509, 105547909, 3253811, 580632999, 1289775441},
  {1, 1, 1, 9, 21, 9, 1, 195, 31, 907, 1719, 1549, 809, 13629, 16597, 30901, 29205, 3861, 468381, 853121, 1937085, 3783365, 1952503, 2800587, 16726679, 63551579, 102773905, 223622789, 506857273, 332803201, 11458833},
  {1, 1, 1, 3, 21, 61, 103, 219, 311, 849, 523, 21, 4533, 6367, 3935, 43453, 9789, 223541, 1591, 825985, 569249, 290723, 6692511, 11820923, 32069997, 7297831, 116588641, 197991017, 182271003, 300377787, 1549219345},
  {1, 1, 7, 9, 7, 33, 77, 19, 489, 933, 1729, 1813, 6741, 10701, 7, 27637, 67037, 161043, 48061, 94859, 900757, 3144609, 5530247, 3520045, 32084089, 8590133, 97367345, 170691905, 280541001, 504217283, 1493484305},
  {1, 1, 1, 5, 23, 53, 43, 63, 453, 209, 1313, 2847, 2641, 13783, 14983, 45949, 84477, 149461, 318737, 561059, 1399761, 1130759, 7558619, 550905, 33317853, 48079077, 46294155, 222392997, 508500187, 991251379, 692653233},
  {1, 3, 7, 7, 15, 45, 83, 241, 509, 659, 213, 221, 5205, 6019, 18945, 9389, 21415, 30739, 368379, 344163, 1724001, 611647, 5036821, 7514721, 13189799, 6505841, 9522593, 192725809, 54807847, 464482853, 472735249},
  {1, 1, 5, 9, 25, 43, 37, 9, 191, 505, 765, 295, 953, 1045, 11203, 24565, 105453, 80113, 15645, 887741, 1407743, 541553, 5875157, 7036635, 1110285, 37615217, 56892211, 132165853, 458220473, 468867351, 111824689},
  {1, 3, 7, 11, 5, 49, 45, 177, 379, 695, 355, 1711, 7747, 497, 7597, 6565, 2391, 94451, 277879, 108649, 1632077, 709993, 8215013, 7731263, 16771587, 53382719, 72256899, 75271199, 281131549, 378900401, 817490993},
  {1, 1, 5, 13, 23, 47, 101, 145, 301, 207, 195, 2225, 8093, 15345, 14097, 54693, 24677, 108337, 211385, 655603, 268395, 2317577, 5397789, 6625321, 31717595, 61980423, 19828989, 57264673, 182436645, 568232693, 314352593},
  {1, 3, 7, 13, 9, 9, 55, 223, 343, 921, 1825, 3281, 2627, 855, 27651, 46717, 64815, 231571, 147281, 783141, 1110509, 2670147, 3233291, 16314235, 9080853, 31822957, 57300453, 258883479, 65984163, 844452863, 1499589201},
  {1, 1, 7, 1, 21, 1, 67, 149, 433, 111, 577, 3675, 495, 9043, 23613, 32877, 72341, 99155, 241973, 16889, 1341045, 210439, 4152865, 14443021, 22471371, 16738261, 68804759, 74593179, 143787439, 772946993, 1065016689},
  {1, 3, 1, 13, 9, 39, 37, 73, 117, 559, 1131, 2511, 7599, 8393, 24747, 24389, 58959, 212981, 357137, 695741, 402371, 351793, 3379941, 11491217, 11569435, 41590607, 118350467, 152316211, 134300645, 523079175, 1891247153},
  {1, 3, 3, 7, 11, 15, 85, 229, 7, 21, 1649, 739, 375, 13991, 27053, 11709, 117207, 260375, 392891, 1001207, 1171011, 1272425, 2361865, 13172675, 21067217, 21082653, 34873223, 212534051, 160382939, 958335489, 75887729},
  {1, 1, 5, 5, 15, 41, 49, 117, 173, 825, 1343, 377, 1789, 12519, 30667, 34173, 2317, 38737, 443345, 887083, 332237, 2225725, 5210721, 3273593, 30064349, 66536571, 93341869, 1098801, 467649651, 1009278383, 1954070225},
  {1, 1, 7, 15, 9, 11, 97, 99, 347, 729, 9, 1703, 1177, 5189, 9061, 55093, 24181, 4851, 134267, 506981, 336455, 2829909, 2321951, 14835311, 23559941, 24952405, 107460371, 114802477, 41137897, 62891633, 1204251953},
  {1, 1, 5, 11, 15, 25, 99, 63, 89, 675, 561, 215, 8111, 3955, 24635, 51437, 104205, 221521, 364383, 65835, 238533, 3920495, 1206795, 3222277, 11822831, 32095109, 129598859, 176785371, 293845247, 468691383, 1065110705},
  {1, 1, 1, 1, 7, 53, 99, 193, 233, 731, 733, 1883, 7783, 14413, 14003, 32149, 105133, 127925, 477205, 866739, 961, 192527, 2265189, 3174773, 26620463, 46075945, 111014151, 203885411, 394474993, 76616671, 1996428401},
  {1, 3, 5, 7, 31, 23, 45, 153, 337, 293, 443, 2301, 5135, 7455, 13123, 18461, 14767, 229009, 191579, 606523, 1818587, 3700769, 5032149, 6068661, 31905809, 60914447, 76320153, 59344787, 179144923, 383234095, 738768273},
  {1, 3, 1, 3, 23, 53, 23, 165, 53, 875, 1543, 1035, 4247, 5101, 28445, 16469, 36375, 243381, 397567, 368739, 1942633, 1981563, 5590177, 15852857, 7234703, 66543811, 129114199, 58867387, 122566289, 756420881, 1266852209},
  {1, 1, 1, 15, 13, 41, 77, 93, 205, 743, 1101, 1413, 2371, 7183, 12337, 7965, 99301, 258709, 52475, 781817, 1726437, 2184073, 3539945, 3908361, 19888355, 31754569, 128183209, 148382663, 123049739, 93771725, 865627793},
  {1, 1, 3, 15, 17, 63, 25, 101, 147, 149, 1207, 3525, 2661, 9539, 11145, 30317, 96261, 72343, 403195, 81389, 1043443, 1894285, 1581001, 15896391, 17034641, 30887499, 110520481, 195244009, 178488743, 240563853, 867698193},
  {1, 3, 1, 9, 17, 5, 3, 35, 389, 909, 1017, 2803, 5243, 13025, 8851, 49701, 48815, 153461, 256789, 124453, 498801, 1829111, 278983, 3912569, 18142897, 26459837, 100810839, 37411567, 471028509, 693471375, 1785594225},
  {1, 1, 7, 15, 19, 27, 69, 91, 71, 547, 1421, 831, 6969, 5517, 28233, 55317, 27461, 105971, 20283, 171647, 998615, 3963289, 8057831, 9168835, 7196639, 51485153, 27342051, 49051613, 76205793, 536081757, 1642156593},
  {1, 1, 3, 3, 17, 45, 55, 63, 263, 819, 1211, 2739, 655, 13269, 22281, 9653, 56773, 74935, 336439, 37485, 1073873, 3210147, 6552859, 13912555, 9263999, 32118967, 14302127, 212497843, 100326281, 421777829, 1919270641},
  {1, 3, 1, 5, 23, 13, 81, 251, 83, 551, 491, 1029, 3561, 357, 23393, 43637, 93607, 1653, 78777, 83043, 924617, 1192125, 6273791, 13783815, 22132795, 65101143, 132758473, 76964053, 344316193, 45014485, 385636497},
  {1, 3, 1, 13, 25, 27, 93, 143, 407, 403, 1395, 1733, 3187, 1917, 31453, 64277, 96599, 234677, 452465, 416237, 1433343, 2681481, 218707, 2371347, 13892863, 61251423, 116236185, 78224879, 124683513, 343844353, 2115683729},
  {1, 1, 7, 13, 3, 21, 85, 113, 483, 461, 1343, 561, 2081, 10857, 24253, 37573, 95573, 106931, 224249, 380143, 837329, 2867913, 4886229, 13641319, 8062969, 38528259, 11134581, 63010061, 55011037, 935924265, 73229137},
  {1, 1, 1, 1, 11, 11, 53, 135, 25, 163, 1729, 617, 1533, 10881, 16041, 9125, 123141, 168117, 448437, 774783, 2078207, 4118249, 7161947, 8251549, 6425519, 21974741, 130228597, 102903625, 149928309, 699383021, 184134481},
  {1, 1, 5, 1, 3, 49, 125, 139, 77, 891, 815, 3431, 4875, 12513, 4595, 42469, 18861, 81009, 144405, 44839, 1914181, 2698177, 4899903, 7572585, 6837479, 47431595, 34948699, 258068935, 450906925, 523014799, 365244849},
  {1, 1, 1, 1, 27, 63, 111, 109, 421, 425, 345, 1613, 5447, 1357, 32413, 64405, 23637, 83253, 362869, 169967, 360651, 3482867, 6724705, 2143561, 29750349, 62772381, 40951193, 70539155, 155339401, 1012820209, 826979729},
  {1, 3, 5, 3, 17, 5, 37, 171, 259, 281, 1003, 2901, 3241, 15557, 21415, 53, 14079, 10417, 323871, 278389, 271129, 3225425, 5754159, 15883239, 24442053, 67054103, 75275297, 26930517, 293909345, 439730067, 1217680913},
  {1, 1, 5, 11, 15, 55, 75, 199, 493, 215, 1625, 2345, 7873, 2325, 11003, 54965, 91277, 197809, 42207, 460011, 323371, 334599, 3587339, 8355409, 25969243, 30688445, 43142757, 163797141, 137426017, 692276111, 843430481},
  {1, 3, 7, 1, 21, 33, 23, 5, 495, 941, 1185, 475, 5799, 15161, 10677, 8069, 33271, 200755, 441661, 414777, 615733, 1821523, 4148081, 11385795, 2251273, 63856461, 24238815, 113689971, 165083237, 461545881, 210414065},
  {1, 1, 5, 9, 31, 37, 37, 29, 217, 389, 297, 3097, 7319, 2601, 15307, 24261, 46157, 259505, 250845, 81915, 851761, 2564393, 1721273, 14957981, 9717057, 40021437, 110836349, 256165123, 415055933, 202500775, 1668199377},
  {1, 3, 7, 5, 7, 45, 111, 167, 297, 275, 1669, 2489, 1511, 15753, 1289, 34885, 127047, 245171, 330745, 783275, 1803049, 2719075, 5940643, 8606021, 14365375, 16963185, 80043877, 51515011, 148517501, 766920613, 1294647889},
  {1, 3, 1, 7, 3, 45, 19, 11, 189, 199, 1227, 2647, 1897, 9077, 17189, 61301, 100023, 84789, 248891, 14135, 1174817, 82287, 2902511, 7461153, 14440795, 28236719, 2413723, 109762037, 30328057, 989786577, 1466858929},
  {1, 1, 1, 13, 15, 39, 19, 179, 147, 341, 283, 3029, 7599, 8937, 18761, 46725, 78405, 186293, 392441, 131259, 1570339, 2120223, 515807, 10053519, 28302857, 723079, 49015249, 226783379, 155961485, 246087397, 52461841},
  {1, 3, 3, 9, 3, 11, 41, 255, 365, 835, 921, 389, 919, 15223, 14541, 31037, 28887, 35927, 326005, 895039, 1207871, 1981045, 7189027, 14632665, 32001647, 17457117, 73856089, 146453187, 328634099, 456123745, 2060315025},
  {1, 1, 3, 3, 5, 37, 29, 203, 313, 271, 1207, 487, 3711, 3811, 26757, 56749, 4661, 22423, 267383, 674105, 1028633, 918041, 5773839, 1386365, 18214611, 56195875, 52809931, 147539027, 148697263, 419794409, 571938993},
  {1, 3, 7, 9, 19, 53, 49, 139, 351, 537, 1681, 1595, 5399, 13839, 28685, 50525, 100887, 212563, 126837, 426559, 1954817, 1547373, 4156463, 7515531, 3745029, 60711517, 40398031, 2504643, 122691731, 254812769, 393255153},
  {1, 3, 1, 1, 15, 35, 21, 37, 247, 891, 1855, 1243, 3137, 10381, 30379, 57365, 65295, 71605, 409821, 514555, 784439, 2660441, 1063065, 10053747, 20416575, 28309971, 30245319, 4227949, 296131249, 687199039, 514461809},
  {1, 3, 7, 5, 9, 47, 91, 25, 479, 337, 781, 3545, 1045, 9491, 22853, 51885, 21623, 56659, 15865, 988709, 2078763, 342127, 776213, 12286083, 18647781, 2055201, 40008301, 192659185, 352490631, 212331489, 1780062929},
  {1, 1, 5, 15, 19, 31, 81, 5, 117, 923, 565, 2443, 7383, 1795, 11685, 12013, 53045, 155921, 414971, 926647, 1322643, 4053485, 4778921, 6319713, 16237983, 29598097, 102966551, 5170547, 256138175, 712134473, 944434545},
  {1, 3, 3, 5, 17, 15, 21, 245, 489, 889, 2047, 2737, 7445, 14785, 13401, 7717, 114695, 93687, 135673, 668205, 1694091, 2619777, 4108153, 29325, 13437693, 34197355, 50080037, 240068185, 303597109, 476375389, 55242321},
  {1, 1, 1, 15, 19, 45, 67, 117, 299, 607, 953, 743, 6863, 12123, 6701, 37901, 8597, 127125, 399739, 836263, 257697, 854463, 974737, 4916087, 21766331, 12952069, 117186603, 43119547, 368471767, 665993953, 2077914801},
  {1, 1, 3, 1, 1, 43, 19, 129, 345, 861, 209, 2387, 7205, 7131, 8235, 37005, 6669, 164055, 430485, 899197, 1117855, 550023, 5749725, 736421, 8445769, 52160101, 46967767, 68952657, 44689127, 546439567, 1186910577},
  {1, 3, 5, 1, 1, 13, 75, 99, 333, 157, 23, 1217, 1857, 15479, 16031, 22781, 11039, 228881, 197725, 926949, 1909913, 802559, 5213031, 9048273, 6426721, 55408051, 111409253, 5359661, 195286915, 234052555, 723251793},
  {1, 3, 3, 11, 7, 61, 119, 89, 491, 401, 227, 1739, 3807, 16003, 2875, 32429, 97999, 181015, 477719, 827259, 703873, 2548155, 645885, 8821415, 8737861, 55461919, 23460479, 228168771, 159887455, 588492503, 931140593},
  {1, 3, 7, 15, 13, 55, 3, 159, 405, 593, 975, 361, 2563, 6061, 28087, 53909, 123327, 755, 107539, 1004257, 20123, 979879, 1127123, 15308585, 287613, 23831243, 108521789, 21375503, 180878905, 1059795691, 1886195665},
  {1, 1, 3, 13, 19, 5, 5, 9, 119, 41, 33, 1111, 4443, 4663, 28841, 59453, 28549, 146391, 401593, 377135, 299905, 1613017, 3597869, 4315763, 12816829, 30233997, 40975171, 158777103, 504241051, 701901941, 1456365617},
  {1, 1, 7, 7, 25, 59, 125, 255, 49, 947, 1673, 2947, 6369, 2267, 8813, 52493, 28437, 80723, 121843, 455093, 791959, 2449097, 1506819, 9413173, 32051551, 19183485, 58840991, 211410357, 293488455, 418670233, 294409713},
  {1, 1, 5, 15, 25, 25, 111, 193, 217, 193, 821, 2779, 69, 2957, 27043, 39765, 47917, 248497, 18139, 482941, 1760405, 258043, 7490677, 7839125, 11102197, 38915585, 42244271, 185088417, 177431633, 307334983, 438972145},
  {1, 3, 5, 7, 21, 19, 51, 157, 203, 487, 1745, 1875, 911, 14071, 7557, 52925, 78519, 171601, 207227, 236209, 194975, 766167, 7842945, 16346167, 29582979, 28771117, 79759871, 102963803, 509552939, 1017600201, 1488294897},
  {1, 1, 5, 9, 3, 15, 55, 73, 313, 245, 1061, 1929, 3035, 607, 11563, 41757, 95565, 75985, 342941, 100199, 1358235, 1169355, 1477829, 14130293, 27774913, 47338193, 40309677, 227730047, 47070091, 131728951, 1299891921},
  {1, 1, 5, 7, 3, 57, 105, 121, 461, 43, 803, 1801, 4059, 2157, 17547, 34453, 38221, 182257, 199635, 359271, 871573, 1850261, 2771149, 5036089, 17918143, 1990183, 58893461, 25977895, 223187929, 537454647, 1294036305},
  {1, 3, 7, 7, 19, 11, 1, 121, 499, 841, 601, 3515, 2969, 13697, 8917, 4533, 83655, 2755, 502347, 50255, 1659223, 2031885, 5526933, 5649719, 29639789, 32784981, 22661471, 187616701, 26462765, 625343857, 1431896305},
  {1, 3, 3, 3, 13, 35, 113, 231, 391, 689, 697, 2871, 7387, 715, 27005, 7261, 128071, 68039, 172751, 876225, 484207, 1946373, 996779, 2168195, 1019629, 32815333, 89382451, 10077055, 206065879, 250107601, 1616398385},
  {1, 1, 1, 13, 19, 5, 17, 43, 175, 291, 987, 1917, 7635, 15655, 10689, 24397, 42549, 154437, 291017, 417239, 1418353, 2449181, 7670143, 13301163, 9236431, 57862311, 69403569, 103090687, 262869779, 606027853, 1883849745},
  {1, 1, 7, 15, 19, 37, 121, 243, 125, 623, 1231, 29, 2325, 5147, 21435, 37821, 18877, 124227, 267083, 573647, 2096345, 642421, 3981607, 11067225, 12493155, 26060027, 93086177, 204106089, 371362735, 82639791, 156449041},
  {1, 3, 5, 15, 25, 27, 57, 187, 77, 401, 1489, 2977, 5415, 3381, 2551, 11461, 127919, 59969, 197571, 710477, 1722343, 231197, 7469335, 9844457, 13943229, 18399653, 24202789, 40291107, 18834545, 414351659, 795997521},
  {1, 1, 1, 7, 1, 1, 85, 27, 115, 559, 9, 2365, 711, 5733, 2819, 51269, 24189, 43333, 454595, 494405, 2013917, 3514129, 3757407, 9852519, 1931395, 28846725, 44186817, 11157667, 322563921, 493757407, 1084920593},
  {1, 3, 1, 15, 9, 29, 61, 113, 169, 349, 591, 1061, 6041, 7613, 23691, 19109, 69055, 79685, 372803, 134477, 228577, 3127129, 3119557, 4668189, 12181081, 2761131, 27043601, 67134181, 20021865, 559868847, 835171345},
  {1, 1, 5, 1, 13, 45, 49, 227, 345, 563, 87, 3597, 3961, 7205, 8441, 7749, 28277, 64833, 236613, 928601, 1783913, 235333, 1209111, 14743517, 25935615, 35548531, 20253313, 59707749, 43976657, 797121629, 720879377},
  {1, 1, 1, 5, 3, 21, 121, 183, 463, 83, 1365, 539, 1485, 10063, 24867, 35725, 38749, 35269, 236097, 591047, 545217, 3620917, 2244771, 2179179, 11715535, 30556841, 41469463, 247418585, 423183075, 537445847, 619692657},
  {1, 3, 5, 5, 3, 61, 101, 237, 41, 147, 1907, 3049, 7583, 8283, 6099, 35581, 34751, 21825, 80073, 891863, 1204201, 2177065, 7252185, 727293, 9402983, 60292887, 53153509, 26280211, 24583359, 1049838575, 1368078161},
  {1, 3, 1, 15, 31, 57, 19, 155, 445, 805, 1793, 207, 1975, 3357, 14281, 52101, 65879, 225989, 192707, 410715, 1423685, 3250223, 8267967, 11471281, 33217161, 16604725, 133200411, 227981171, 207237345, 563236437, 1722162865},
  {1, 1, 7, 13, 9, 39, 27, 73, 165, 345, 543, 4095, 133, 10469, 11573, 18789, 79237, 202435, 151113, 988757, 514131, 388223, 1664149, 582921, 28444477, 12469963, 102287195, 188863033, 312340825, 462807649, 1143857329},
  {1, 1, 7, 15, 17, 57, 99, 81, 359, 367, 1057, 1173, 4225, 15127, 2615, 24589, 23501, 133699, 115019, 168141, 543941, 2413671, 2065653, 13425963, 20340715, 55947773, 124173161, 90346693, 427921667, 737781235, 46678929},
  {1, 3, 5, 3, 31, 23, 113, 111, 495, 947, 1625, 1195, 2053, 1509, 1347, 30245, 42591, 106945, 323663, 39499, 1015259, 2315341, 4775155, 7671147, 26375887, 11852325, 133458855, 58695969, 438487177, 200219287, 1082739057},
  {1, 1, 5, 5, 9, 47, 25, 63, 455, 107, 771, 3815, 3827, 16287, 11615, 11533, 4941, 47297, 392641, 475869, 1027835, 650877, 6297283, 5469923, 16085975, 22197343, 11648651, 52534975, 481790403, 454551891, 1352543153},
  {1, 1, 7, 9, 17, 61, 51, 215, 63, 123, 1253, 3927, 721, 9647, 3283, 43725, 55133, 203971, 76621, 142029, 1175513, 2593335, 4361827, 7296819, 21039975, 23251545, 12269563, 266786693, 291235635, 166595159, 1986665649},
  {1, 1, 5, 15, 11, 17, 83, 255, 473, 107, 681, 763, 7855, 8043, 31503, 53021, 126413, 28609, 171211, 969695, 905965, 4033087, 6185723, 11713493, 978519, 13558029, 22309143, 218819275, 272408903, 407034355, 537356913},
  {1, 3, 1, 7, 7, 31, 37, 5, 253, 155, 2017, 609, 1421, 14927, 25241, 33197, 12087, 189125, 505291, 345923, 529347, 1052153, 5772601, 8226985, 24976079, 65023829, 78433101, 136201161, 169277195, 226137197, 1813269969},
  {1, 3, 3, 13, 31, 25, 21, 241, 431, 193, 681, 2265, 5091, 11479, 21443, 20813, 104895, 46279, 130625, 691795, 508269, 2825385, 8245205, 11571643, 8204997, 24147541, 24198941, 52551559, 279236723, 388459023, 393471185},
  {1, 3, 5, 5, 15, 9, 49, 255, 157, 995, 631, 1995, 3605, 9085, 24245, 6085, 86535, 52033, 339657, 639963, 159197, 3271117, 3543451, 11534745, 9437407, 45089859, 57673863, 202375177, 519049745, 181407521, 64017265},
  {1, 3, 3, 7, 19, 31, 85, 153, 493, 951, 451, 1587, 6609, 3681, 13205, 63029, 68295, 30023, 374219, 116959, 1867203, 3928025, 1950629, 8702713, 2132115, 23407671, 24129455, 184983837, 238197613, 315412497, 976527345},
  {1, 1, 5, 1, 17, 41, 107, 231, 307, 361, 575, 3239, 3443, 16159, 20625, 30093, 44021, 80449, 357189, 567877, 1206253, 3352559, 5506947, 16475743, 26904373, 13824307, 95950019, 37084367, 179199683, 1060189981, 2072481585},
  {1, 1, 7, 9, 31, 49, 93, 79, 181, 117, 1241, 3645, 4901, 12599, 13247, 54925, 26413, 85187, 447693, 409667, 252757, 3551713, 7482443, 4318337, 20042889, 10451277, 41738281, 123075457, 400263299, 572514795, 2114324881},
  {1, 3, 3, 9, 7, 31, 127, 201, 11, 199, 1851, 23, 5667, 8159, 20951, 48781, 90607, 256327, 464453, 829259, 1668987, 651683, 5076213, 8332471, 7595043, 56610391, 121082699, 267861647, 337018435, 766815003, 1729167793},
  {1, 3, 3, 7, 3, 37, 29, 189, 65, 461, 769, 321, 6577, 16223, 16865, 10989, 34071, 177607, 353739, 951887, 249593, 2809041, 8067689, 6381429, 2145657, 16481093, 114621621, 83094133, 278776219, 1023510501, 2099602001},
  {1, 1, 5, 11, 1, 13, 91, 167, 33, 111, 1445, 1047, 2479, 12623, 22893, 25037, 97221, 147521, 417103, 523093, 510945, 2327199, 5972563, 10806925, 27995315, 54771425, 68063339, 55612795, 528003923, 413774353, 800789937},
  {1, 1, 3, 1, 3, 1, 47, 185, 329, 903, 1651, 3005, 907, 1255, 8303, 22477, 103821, 65991, 32581, 495951, 1553221, 1053667, 7772797, 6319029, 9817763, 57541911, 96601697, 189931271, 128848339, 895102595, 686447377},
  {1, 3, 5, 13, 19, 31, 5, 233, 265, 769, 1303, 2503, 2229, 14019, 20257, 30525, 123415, 119041, 288513, 669447, 1983787, 3912521, 143765, 8645661, 13002661, 63459379, 39990467, 54406121, 247835359, 373593333, 1801636145},
  {1, 3, 7, 3, 27, 11, 67, 195, 5, 661, 125, 3761, 7211, 16043, 7267, 3325, 5247, 167171, 25871, 39175, 951047, 738095, 8383471, 7054345, 6760089, 40115049, 86565861, 12640991, 126425967, 809319375, 108339537},
  {1, 1, 1, 3, 27, 13, 115, 25, 473, 417, 1751, 2223, 2099, 5913, 14273, 23253, 79349, 107781, 201479, 84767, 872193, 3507487, 4861229, 2314517, 18092229, 64696499, 11730219, 69613679, 322209957, 292082701, 1380093873},
  {1, 3, 7, 15, 13, 53, 99, 115, 225, 737, 1621, 539, 4131, 471, 31865, 41645, 49815, 214275, 481539, 870161, 1740553, 2270039, 4644263, 7960685, 32118829, 48098001, 41209503, 123329655, 23217339, 521103349, 152141553},
  {1, 1, 5, 5, 25, 19, 39, 207, 153, 569, 1755, 2477, 3065, 7383, 29919, 59853, 33837, 193281, 463105, 496397, 1714951, 3980611, 4249987, 1866501, 1854981, 38823871, 20490825, 188888981, 157970275, 995810875, 497498001},
  {1, 3, 5, 11, 13, 59, 33, 3, 435, 273, 701, 3819, 7291, 11803, 26111, 157, 100623, 28417, 456455, 203033, 619799, 357653, 1073647, 13675063, 22494237, 55260017, 86931375, 93866383, 353462455, 18605843, 1494935025},
  {1, 1, 3, 9, 29, 19, 71, 59, 93, 1019, 887, 83, 4675, 7541, 26821, 26661, 13541, 221447, 24845, 449809, 918839, 2872115, 1494903, 7851585, 23415831, 4271875, 47328447, 80229535, 10955065, 564223177, 598830321},
  {1, 3, 1, 3, 21, 53, 71, 73, 43, 321, 1581, 1399, 4043, 12995, 16825, 5693, 32535, 252677, 375567, 90385, 633145, 1339667, 1899421, 15526455, 893285, 18653569, 123904371, 207611007, 388093983, 763753365, 748517937},
  {1, 3, 7, 15, 3, 13, 37, 11, 93, 873, 1193, 3481, 451, 15869, 17879, 57893, 97071, 122627, 257795, 69407, 1635633, 1441577, 6078783, 11902921, 9689669, 17919197, 40655165, 266020455, 497739929, 380130779, 2066259153},
  {1, 3, 1, 11, 31, 19, 101, 57, 129, 753, 853, 463, 6757, 11083, 8667, 56477, 64895, 146437, 285703, 695323, 1936447, 181273, 5569653, 1357077, 14427669, 28958033, 74968587, 246082353, 78588023, 683478095, 249161137},
  {1, 3, 5, 15, 25, 41, 25, 197, 235, 609, 905, 993, 3233, 1935, 24661, 2413, 54791, 188417, 352259, 221197, 57365, 1974397, 2416737, 14377103, 18112885, 12214309, 119253605, 184246917, 519759883, 31060689, 1721759569},
  {1, 3, 1, 5, 21, 7, 53, 107, 473, 77, 1135, 1045, 4933, 5615, 15931, 2733, 78463, 51205, 161801, 75793, 772115, 3283041, 6604919, 8521997, 29792737, 52017931, 78204193, 173267705, 172412347, 246618927, 1918317841},
  {1, 3, 7, 11, 3, 9, 105, 183, 151, 527, 425, 975, 4073, 913, 2793, 35957, 85623, 250371, 205319, 36383, 1841701, 1330789, 8136339, 8648499, 31354579, 39213557, 88420851, 266136397, 387927725, 703079245, 987298865},
  {1, 1, 7, 13, 19, 61, 81, 9, 413, 851, 1723, 1113, 1453, 8635, 3353, 6461, 76277, 62595, 356745, 932239, 1966729, 1944989, 668029, 2536793, 15306231, 37580111, 14582165, 139232961, 304327007, 244509229, 717010001},
  {1, 3, 7, 15, 19, 53, 83, 31, 441, 343, 575, 935, 4543, 1303, 12567, 63341, 93167, 199299, 250499, 366223, 68233, 1028767, 7380811, 14951541, 32007595, 7833691, 70145147, 205251779, 324829963, 240422763, 2057960625},
  {1, 1, 1, 5, 29, 19, 119, 75, 3, 591, 845, 649, 1717, 13695, 26905, 30509, 87029, 141189, 458625, 336281, 1872007, 1753475, 4073863, 7085151, 19629795, 54842561, 3929141, 12479233, 14971867, 970007885, 1163105361},
  {1, 1, 7, 9, 5, 53, 127, 191, 15, 773, 1433, 2899, 21, 4977, 17839, 41525, 1837, 99203, 396941, 653209, 243601, 678123, 5966827, 2924659, 24175481, 50734629, 112176327, 51866097, 394663101, 789202299, 1585624945},
  {1, 1, 5, 9, 21, 9, 99, 115, 397, 99, 725, 3835, 973, 1219, 21159, 48893, 124557, 42881, 139917, 652161, 1324717, 3255991, 3675095, 10973569, 1675823, 61011297, 36257015, 86968601, 240195503, 246896403, 740001905},
  {1, 3, 5, 3, 7, 39, 29, 93, 303, 913, 981, 3549, 5225, 10907, 393, 56957, 26487, 242561, 127887, 913811, 1068203, 3102593, 6995777, 12021723, 3247461, 57895401, 88509721, 101483965, 235249263, 749513133, 1704370321},
  {1, 3, 3, 11, 9, 25, 105, 101, 1, 867, 389, 2241, 773, 14123, 10015, 32509, 86767, 175495, 170375, 874373, 806069, 1800397, 2354561, 15752997, 694903, 29571329, 52241613, 188493457, 184636335, 1020639883, 190348753},
  {1, 1, 5, 1, 1, 37, 117, 213, 391, 779, 1851, 1485, 1277, 5607, 819, 3853, 24413, 6017, 146565, 455829, 1340833, 3303409, 1674433, 14550227, 33441503, 56245671, 30442849, 120100497, 474584147, 587462215, 1453885201},
  {1, 3, 7, 1, 3, 5, 43, 47, 483, 367, 749, 1693, 4961, 15257, 3775, 35573, 97071, 173187, 210061, 1023647, 109185, 1986983, 2970811, 1946831, 1502803, 66077153, 32329377, 201541205, 475459693, 875305147, 1132216593},
  {1, 3, 3, 1, 27, 11, 21, 83, 437, 379, 1041, 393, 5611, 2421, 31739, 19973, 16799, 141959, 281229, 483223, 1740943, 3638265, 3948799, 5881953, 6111671, 64832069, 42453477, 200294447, 531219921, 507004375, 1918398801},
  {1, 3, 5, 7, 19, 1, 79, 63, 53, 201, 1159, 2501, 6327, 11317, 9537, 34117, 94263, 42881, 257675, 866183, 765597, 3222659, 7698107, 12818057, 14798005, 28622291, 56991073, 24041515, 288542369, 242859549, 1784093969},
  {1, 3, 5, 13, 9, 37, 61, 217, 427, 913, 1311, 3503, 5473, 10583, 19723, 9581, 23679, 19329, 285569, 906397, 1275793, 2742681, 1282765, 8848095, 14584949, 30267187, 54841707, 28978589, 249231211, 961767951, 1194789809},
  {1, 1, 3, 9, 11, 29, 121, 175, 141, 515, 925, 837, 6011, 10419, 32157, 62941, 80901, 144007, 22157, 782727, 1928633, 1252053, 4670043, 14749673, 31160127, 3437353, 55385289, 137304463, 104938279, 928338417, 1724581777},
  {1, 3, 5, 9, 27, 57, 97, 175, 365, 367, 1737, 3845, 1257, 12243, 2201, 51037, 123447, 174209, 296581, 1022095, 1736093, 3967117, 7731915, 12363625, 6068051, 66037245, 111337785, 169452877, 127668551, 807154365, 508315793},
  {1, 3, 3, 9, 23, 1, 53, 123, 127, 333, 1335, 707, 5747, 6541, 9809, 52165, 65591, 174983, 414085, 977307, 824997, 2316521, 684799, 7801643, 17355553, 65418827, 91241079, 197845039, 242251265, 456478749, 994691185},
  {1, 3, 1, 9, 17, 37, 101, 41, 91, 61, 433, 979, 4345, 12351, 10829, 19437, 23751, 45925, 86053, 768757, 1798497, 1885441, 8006493, 13582255, 20342265, 21365565, 17433031, 248647005, 381033811, 482481097, 1196344689},
  {1, 3, 3, 13, 3, 21, 15, 49, 257, 99, 1793, 2987, 5233, 11625, 28069, 53045, 91367, 115815, 132321, 226927, 580417, 1499939, 7278533, 5511005, 31188615, 46227557, 103278983, 259225749, 462973093, 52721985, 1377589617},
  {1, 1, 7, 11, 21, 13, 89, 11, 135, 153, 783, 2893, 6815, 12007, 15605, 48813, 33669, 177955, 372143, 506089, 445249, 1053037, 3228575, 1625667, 13793237, 25253107, 32174153, 10104451, 17509107, 625681257, 2042732433},
  {1, 3, 7, 13, 5, 61, 73, 5, 269, 699, 925, 2925, 5919, 5841, 24875, 14901, 93439, 205347, 483169, 733881, 839913, 1176445, 4578769, 7610185, 18149455, 18747465, 69711073, 254166155, 260627589, 563817079, 70597393},
  {1, 3, 5, 5, 25, 45, 43, 93, 15, 927, 1253, 319, 1173, 14559, 20221, 61485, 61767, 5089, 230697, 669421, 493913, 1522031, 6358921, 3414595, 9861515, 38431801, 72083043, 70237537, 26590595, 393778569, 34376497},
  {1, 1, 3, 3, 27, 45, 9, 103, 447, 627, 1239, 3869, 2169, 49, 17917, 20981, 9669, 60903, 311527, 717175, 1700673, 2988293, 6276595, 16501819, 26721615, 29589067, 76215217, 181009029, 362797925, 677111993, 1199819793},
  {1, 3, 7, 7, 11, 9, 1, 1, 1, 527, 825, 3295, 623, 2095, 10537, 48237, 53175, 18595, 394155, 278391, 1217141, 1733389, 4748549, 3338277, 32108971, 17037277, 104436099, 215995883, 255214131, 575623221, 914999601},
  {1, 3, 3, 11, 21, 11, 59, 165, 33, 743, 1461, 1535, 6393, 1301, 17823, 20229, 38927, 146535, 407911, 855097, 1693639, 3903823, 6271785, 542829, 16526099, 19364369, 30976195, 43383117, 156929161, 16773595, 1704114481},
  {1, 1, 7, 3, 19, 43, 47, 245, 469, 551, 1447, 1963, 169, 1481, 31925, 19573, 14373, 180067, 447975, 681839, 1757767, 2983427, 1257209, 14206665, 24360275, 43578707, 83212823, 3090773, 239928965, 329057713, 1200178289},
  {1, 1, 3, 1, 11, 21, 51, 7, 251, 199, 1153, 767, 6417, 3417, 30171, 15797, 12957, 144999, 93669, 104103, 1124849, 2451071, 843379, 7301847, 28090363, 49307917, 41148611, 260352309, 277342725, 290738623, 1973205809},
  {1, 3, 7, 1, 31, 5, 41, 103, 447, 263, 211, 2029, 8021, 4705, 10579, 4661, 23231, 191331, 375021, 938403, 148449, 3341941, 6649427, 16073467, 16287035, 23792463, 11279761, 80358585, 128610805, 516076839, 1814760977},
  {1, 1, 3, 5, 17, 25, 55, 75, 393, 107, 2017, 2389, 1685, 14021, 9161, 14181, 13269, 195175, 457825, 152061, 147309, 2298899, 3970063, 2381109, 2221607, 17650341, 133405785, 80298417, 179991665, 987849405, 1187872401},
  {1, 1, 1, 9, 13, 1, 75, 237, 205, 461, 689, 2531, 2839, 13925, 23351, 41605, 24269, 237989, 59373, 704809, 1686213, 1254431, 2843785, 8424577, 18617753, 33083109, 70204407, 227709619, 255561481, 191402379, 1062675057},
  {1, 3, 7, 1, 23, 39, 33, 189, 157, 571, 239, 1053, 1559, 1685, 23059, 45413, 17823, 210723, 435437, 5035, 1221571, 3775581, 107521, 10116601, 33142383, 51593147, 58082577, 139583051, 227497473, 1009759855, 1413504017},
  {1, 3, 3, 3, 27, 61, 71, 121, 49, 157, 1341, 1707, 2417, 11689, 26507, 51381, 13151, 236903, 144367, 703159, 2076625, 1761291, 5791789, 4546957, 19505881, 64185497, 43377535, 143434597, 127258469, 1064730031, 421420785},
  {1, 3, 7, 7, 19, 63, 47, 53, 95, 791, 1467, 1273, 2045, 755, 8555, 60477, 83039, 226659, 166059, 598063, 2059203, 2455619, 321769, 1788355, 27555939, 10272743, 72796805, 26035313, 476868535, 527676351, 1841581393},
  {1, 1, 3, 15, 27, 33, 21, 253, 317, 153, 1509, 1765, 3809, 601, 5907, 42261, 82557, 229223, 319851, 859703, 1903741, 878745, 7004633, 16479177, 16192845, 29677937, 23717601, 250393109, 109238445, 799747447, 1959854865},
  {1, 3, 5, 15, 11, 17, 97, 91, 165, 199, 1751, 2135, 1315, 3077, 29995, 58629, 124927, 70561, 15971, 426559, 695373, 745709, 1464831, 13241089, 11181195, 34620451, 48320427, 248169695, 323118265, 1060557447, 42201521},
  {1, 3, 1, 5, 3, 33, 93, 49, 39, 743, 341, 2549, 7603, 3369, 30889, 39637, 36023, 40229, 458729, 130407, 1340117, 2489841, 2835413, 13176627, 17802883, 21648553, 33676041, 112428039, 129929181, 519225541, 2105078161},
  {1, 1, 3, 13, 3, 5, 87, 63, 293, 785, 1591, 675, 3915, 2209, 18201, 51413, 124981, 10151, 98537, 954735, 2070225, 506299, 2385803, 8647321, 4057677, 25046339, 67045319, 29038135, 500373965, 491355093, 1018295153},
  {1, 3, 3, 11, 3, 15, 69, 231, 241, 127, 429, 2201, 8173, 12549, 25745, 9893, 100919, 69607, 116711, 198511, 1137859, 50153, 2944507, 1628357, 2579395, 53804489, 69275781, 177363065, 223047937, 1063663109, 359022417},
  {1, 1, 5, 11, 15, 39, 3, 29, 125, 685, 643, 1385, 829, 7347, 28793, 3005, 112789, 154337, 389167, 515323, 1644267, 3810527, 1611489, 1811153, 27978105, 7986135, 48337917, 130144209, 333407863, 1040727789, 1958228689},
  {1, 1, 7, 15, 27, 15, 59, 237, 299, 773, 1097, 3875, 6503, 7129, 28495, 33013, 39597, 120611, 229867, 708263, 1330259, 401335, 70385, 4833127, 20416913, 44546629, 80178199, 103819939, 530434805, 542855219, 1981889649},
  {1, 3, 5, 13, 9, 17, 31, 227, 69, 443, 1633, 525, 1659, 14681, 15209, 42517, 34103, 21281, 387297, 721533, 1650629, 374427, 4002887, 9170265, 22830807, 39390613, 74015681, 56680671, 202535565, 847059237, 1140174609},
  {1, 3, 5, 5, 13, 51, 69, 173, 111, 433, 279, 2145, 2091, 9741, 24881, 13381, 102359, 26657, 487529, 462969, 984007, 1127121, 4447361, 3679515, 11040101, 3916555, 131219525, 40813847, 356702121, 229756669, 174020945},
  {1, 3, 1, 7, 7, 35, 55, 51, 357, 99, 1789, 333, 2073, 10151, 14527, 22125, 33871, 170853, 18347, 822051, 1774431, 1969547, 743647, 3553433, 2984207, 43855177, 78902625, 158253517, 234676595, 255736955, 909260049},
  {1, 3, 3, 7, 13, 41, 101, 87, 425, 701, 1143, 2733, 6473, 8667, 17419, 32477, 101535, 38759, 140843, 37537, 391253, 3000305, 8136915, 10226077, 4200961, 33038131, 21866553, 115685301, 288809231, 582692687, 1869333649},
  {1, 1, 5, 5, 25, 29, 63, 31, 385, 537, 563, 607, 6723, 9251, 6531, 16029, 15965, 206497, 346017, 935853, 1265769, 3551163, 6561803, 2187293, 22209157, 51995479, 6252059, 204959791, 426288447, 604044399, 1879019185},
  {1, 3, 5, 5, 9, 63, 111, 131, 239, 723, 705, 2805, 6579, 12691, 17521, 22013, 29719, 177633, 113129, 512701, 750859, 3496235, 1074143, 15391475, 9453823, 35472797, 101499737, 311767, 402418343, 2042333, 1776637585},
  {1, 3, 1, 7, 31, 55, 101, 225, 477, 271, 611, 3179, 7859, 9835, 2165, 30909, 128839, 28901, 1003, 853051, 30987, 1642873, 6692637, 3771305, 32315499, 50064631, 55814647, 76964383, 234409319, 870695001, 2092899953},
  {1, 1, 3, 3, 5, 15, 81, 127, 391, 333, 419, 1091, 5997, 12315, 31521, 3965, 102133, 60711, 81383, 673193, 1418019, 555365, 7008035, 15398819, 21165841, 34153311, 40495703, 240759041, 530004215, 547189813, 370375281},
  {1, 3, 5, 15, 23, 7, 35, 109, 181, 295, 825, 419, 969, 15753, 9365, 5877, 4327, 118433, 83363, 114979, 192283, 64287, 2435025, 15482585, 16920435, 11111949, 76709983, 264865677, 34574501, 763201105, 238651633},
  {1, 3, 5, 5, 25, 23, 69, 177, 325, 359, 1577, 619, 6233, 11753, 8103, 10741, 24623, 38049, 81321, 853869, 724771, 3398465, 62797, 7355377, 12338563, 48004829, 99443799, 133474781, 3765733, 75683107, 1662325873},
  {1, 3, 5, 11, 31, 13, 79, 61, 241, 1011, 1961, 949, 6211, 497, 7099, 885, 13375, 204257, 416679, 90603, 1683201, 3546451, 5623721, 8206765, 10431463, 23312589, 132613785, 124290631, 330281989, 1021875159, 1459467409},
  {1, 3, 5, 3, 25, 19, 67, 235, 337, 1015, 1485, 355, 3653, 12735, 14503, 7853, 82927, 114465, 412015, 27501, 808255, 3657543, 4814599, 10329405, 2669691, 14275561, 76028383, 255899921, 507564307, 269494355, 569415409},
  {1, 3, 5, 7, 31, 23, 35, 231, 147, 15, 263, 1995, 431, 5941, 18931, 24901, 130975, 4961, 460331, 229611, 1951531, 2972991, 2095899, 14855679, 21587827, 45189291, 20402327, 221149779, 469281145, 660838831, 2046561905},
  {1, 3, 3, 7, 1, 35, 37, 7, 85, 443, 715, 743, 2189, 12537, 17427, 13365, 87231, 231719, 38315, 322029, 1803551, 1675585, 3546091, 13282593, 18097799, 7186687, 72135595, 149669441, 210855621, 281655367, 2052726193},
  {1, 1, 3, 1, 7, 41, 1, 209, 121, 929, 661, 3999, 955, 5123, 31115, 12829, 8765, 149991, 154597, 71915, 1697293, 626237, 5238293, 4947549, 3068421, 16503889, 58808123, 1253839, 33282911, 224788807, 290849457},
  {1, 1, 3, 5, 11, 43, 127, 125, 107, 293, 273, 2071, 3003, 11631, 7769, 39597, 38773, 127975, 123489, 514727, 167455, 2299507, 4947633, 6394679, 32160689, 49188829, 55389539, 159310431, 92401235, 535267861, 53152561},
  {1, 1, 1, 13, 13, 29, 39, 217, 111, 779, 1287, 1675, 4201, 4869, 20403, 11141, 105469, 106149, 365865, 168681, 1720841, 448051, 1844877, 7711571, 8576975, 4636955, 19286087, 208486741, 326167241, 424485239, 1910226289},
  {1, 1, 3, 15, 25, 53, 25, 135, 389, 925, 1971, 663, 7545, 2673, 7725, 47765, 100357, 73383, 269739, 520501, 254505, 3714653, 2688563, 2989569, 20290721, 39999751, 28661187, 135911765, 455931533, 474521193, 291273521},
  {1, 1, 5, 13, 3, 59, 97, 91, 357, 45, 947, 3031, 8095, 6269, 13975, 16549, 56973, 117857, 322089, 462711, 1837103, 3881037, 1644567, 7117121, 23267617, 22632775, 823107, 158110259, 137212801, 866821451, 721014577},
  {1, 1, 5, 15, 25, 31, 1, 171, 375, 939, 507, 3591, 1089, 13605, 2813, 20549, 129221, 1953, 490795, 611629, 149507, 2656325, 3139703, 14293139, 13556263, 58367943, 12474507, 25611981, 23387097, 855803545, 969294769},
  {1, 1, 3, 7, 25, 21, 41, 131, 147, 737, 9, 1603, 1859, 11573, 28397, 19589, 38053, 13031, 11555, 239157, 37929, 1088621, 3349687, 10773719, 11929549, 37143781, 110284007, 115301127, 180189785, 774310849, 1036250481},
  {1, 3, 3, 9, 21, 9, 59, 27, 169, 875, 711, 1389, 2899, 7937, 4173, 43509, 29543, 248807, 143525, 368185, 586765, 809999, 3011679, 8383845, 8842663, 4328771, 50432929, 179607343, 251896725, 1037552737, 839352593},
  {1, 1, 5, 9, 13, 29, 71, 39, 51, 337, 1067, 2661, 1203, 5967, 19249, 46029, 67861, 74081, 188653, 513337, 1736857, 429011, 6816627, 15610767, 7514605, 34659527, 132373057, 85662079, 394339387, 287376733, 1349985041},
  {1, 3, 7, 1, 17, 21, 43, 79, 181, 741, 1901, 3445, 7171, 2109, 1589, 12741, 14215, 112483, 440493, 90541, 874929, 2769807, 1862523, 1950393, 18199417, 10901145, 51041281, 21414359, 81525193, 505658393, 1653337873},
  {1, 1, 3, 9, 23, 37, 105, 51, 227, 775, 1265, 2987, 2197, 13903, 28891, 50669, 112797, 117671, 147181, 490107, 1341089, 822933, 1341767, 8167399, 23208875, 35087965, 65383959, 39561905, 520510211, 571468015, 1274065521},
  {1, 1, 1, 13, 23, 47, 111, 41, 93, 261, 75, 2155, 4301, 11517, 16101, 49125, 9829, 180901, 518825, 619955, 35771, 3917715, 2211637, 4018113, 5164225, 48614751, 125978911, 173163073, 57822321, 130944241, 121018609},
  {1, 1, 3, 3, 27, 27, 123, 125, 501, 775, 413, 1065, 7607, 15065, 26013, 23829, 86245, 184295, 359463, 908471, 693175, 2666423, 1241073, 6975481, 5050131, 51636649, 8233941, 98967779, 399519837, 686838817, 1112547409},
  {1, 3, 7, 3, 27, 11, 59, 87, 207, 743, 1765, 2969, 913, 8101, 11583, 16869, 28687, 159523, 48047, 651495, 152487, 3582647, 5733659, 16044739, 31829595, 47349817, 74222565, 174626405, 233997473, 750753051, 1506937169},
  {1, 3, 3, 1, 23, 7, 113, 17, 285, 993, 695, 2399, 5019, 4779, 28917, 23933, 126663, 186983, 10861, 190907, 916643, 3083213, 5341677, 4270201, 21294021, 37035427, 35437179, 117041127, 122119159, 477963065, 12834481},
  {1, 3, 1, 5, 11, 51, 49, 139, 213, 435, 1475, 2209, 6695, 12981, 9851, 19685, 106751, 233445, 171433, 603759, 832391, 2081533, 5799911, 2111537, 960895, 23255927, 14295245, 29947595, 255985937, 105792519, 1063954385},
  {1, 3, 5, 7, 1, 63, 31, 151, 173, 767, 1453, 1497, 6911, 9597, 25551, 46725, 130351, 5345, 155307, 1008885, 1938307, 148731, 7018315, 4304049, 21624067, 28936825, 69811781, 225516587, 482808633, 131428179, 541578065},
  {1, 1, 7, 7, 21, 53, 39, 159, 389, 231, 309, 359, 7701, 14819, 5175, 32285, 117165, 2467, 28003, 495081, 217737, 2177747, 1312555, 13388921, 28810939, 17805569, 65142683, 10504865, 87713375, 894368771, 1303086769},
  {1, 1, 1, 1, 11, 47, 83, 29, 247, 89, 369, 2727, 3103, 14421, 17369, 58821, 19893, 35301, 504997, 216367, 647819, 1664223, 4250113, 11477787, 32857037, 25271757, 33406499, 188213243, 211700649, 18436101, 2088099633},
  {1, 3, 1, 5, 25, 25, 111, 245, 239, 755, 113, 1765, 3583, 917, 403, 9317, 25567, 123365, 5289, 741501, 1492653, 722411, 1873457, 6107763, 22955143, 17492077, 71760457, 54728731, 486091873, 518487183, 601715089},
  {1, 3, 3, 3, 5, 59, 85, 151, 463, 591, 743, 3767, 121, 2927, 11031, 47981, 39535, 176935, 324079, 1031913, 381335, 3027361, 336027, 10887547, 32645427, 22537019, 134131627, 110371973, 34292435, 591366291, 758779313},
  {1, 3, 5, 9, 11, 39, 77, 161, 275, 233, 1991, 2683, 6545, 2423, 32113, 11853, 66551, 191009, 374501, 157247, 684451, 3085953, 6663197, 5579911, 4861325, 25202667, 101409279, 221162797, 457836667, 86921877, 1074405105},
  {1, 3, 5, 11, 5, 57, 13, 229, 329, 757, 1863, 3959, 4243, 7265, 15599, 54485, 94255, 184929, 314407, 253425, 1039029, 1918457, 7998497, 4113821, 6324033, 11834563, 40270723, 114805167, 369971741, 404543107, 1918631217},
  {1, 1, 1, 1, 1, 23, 19, 67, 453, 593, 2011, 1813, 4695, 8903, 9623, 59813, 94309, 203309, 105661, 326557, 381835, 1833871, 5751431, 12420001, 10044469, 33967463, 110615241, 123860579, 408523483, 506829395, 137142737},
  {1, 3, 3, 7, 1, 29, 103, 255, 493, 647, 1709, 4065, 4199, 949, 28829, 46573, 26591, 260959, 423955, 756613, 1807017, 833259, 2297459, 12855049, 18519059, 18452753, 50634725, 202120379, 8666281, 827610457, 447543953},
  {1, 1, 7, 9, 3, 55, 53, 33, 5, 223, 423, 3347, 7647, 7211, 25157, 45653, 11341, 118651, 264085, 4999, 461243, 257665, 8050421, 10488793, 7154435, 20258155, 120421335, 151170155, 530103343, 896762193, 183336369},
  {1, 3, 5, 13, 3, 43, 79, 255, 471, 573, 1007, 2119, 6731, 10047, 23179, 55973, 6103, 187081, 380537, 391583, 1432455, 1741307, 7050779, 1859075, 20930953, 3008315, 38890107, 82704111, 17709835, 387319535, 1845413361},
  {1, 1, 1, 3, 7, 39, 55, 61, 53, 377, 435, 401, 3307, 12621, 14299, 54893, 24837, 97133, 121727, 907675, 1770163, 3641763, 1716777, 13602593, 11374789, 54308991, 89855885, 247506751, 162711681, 495399759, 329599377},
  {1, 3, 3, 7, 21, 31, 67, 17, 243, 425, 747, 2995, 1389, 2557, 18415, 49821, 59991, 125711, 395187, 20625, 51371, 1252767, 7496757, 13264343, 20533125, 42549791, 59952415, 49925945, 156587897, 408993267, 1242247345},
  {1, 3, 1, 3, 3, 39, 75, 11, 447, 249, 1135, 1011, 1657, 10767, 19501, 59893, 115407, 60605, 176791, 673423, 1842579, 813263, 344127, 12501019, 14434429, 43403259, 37539039, 71325821, 508685483, 543213857, 2013844145},
  {1, 3, 1, 11, 17, 51, 117, 129, 17, 143, 785, 103, 5049, 14703, 28479, 12661, 16007, 187181, 501951, 931741, 1800871, 4111609, 3445237, 10935245, 5349267, 51148429, 109999939, 117035029, 142728971, 397366387, 1207931121},
  {1, 3, 7, 5, 13, 17, 75, 255, 75, 661, 1175, 477, 1811, 1479, 15783, 25733, 102311, 222763, 520625, 933017, 1091725, 4177063, 6569219, 516031, 20052129, 55082355, 125756321, 68088535, 288799195, 753318131, 4657745},
  {1, 3, 7, 9, 11, 57, 101, 77, 431, 247, 997, 3657, 5117, 6815, 3841, 12949, 105983, 187227, 239709, 377247, 1234069, 1499633, 1188321, 783811, 25533947, 54443745, 59571389, 220095137, 1995875, 276003693, 694007057},
  {1, 1, 5, 1, 17, 21, 101, 183, 209, 69, 299, 1585, 6381, 12983, 10053, 20661, 6061, 81913, 88605, 263581, 293033, 3214329, 2147835, 11810173, 22622049, 21919751, 112107357, 31393017, 267861243, 447038793, 2085825937},
  {1, 1, 7, 3, 5, 13, 21, 63, 83, 857, 749, 1251, 5363, 9629, 16563, 36077, 60197, 57643, 257279, 392321, 640937, 1640169, 3926931, 11122111, 22351197, 20836745, 69328239, 22703935, 244428649, 298898487, 1356956081},
  {1, 3, 3, 9, 3, 59, 9, 45, 55, 489, 137, 2423, 2661, 12111, 4375, 48405, 63271, 184303, 78653, 182151, 1416375, 1597181, 3199945, 15997307, 20736037, 63409709, 10132795, 139571961, 523167691, 358628395, 916511473},
  {1, 1, 5, 9, 23, 9, 41, 177, 447, 671, 1631, 3115, 4215, 14435, 8743, 56629, 47749, 51433, 349877, 534171, 1603733, 2635501, 6681405, 14684579, 4542979, 55744059, 81351231, 222291331, 107068999, 646178515, 621454129},
  {1, 3, 7, 11, 19, 23, 15, 221, 413, 783, 1247, 2343, 4397, 3145, 32043, 54989, 78183, 130539, 344191, 739847, 993843, 3858027, 6667945, 10238505, 19554163, 33474227, 79447587, 220023289, 65008861, 239214423, 822771441},
  {1, 3, 3, 1, 31, 55, 31, 87, 333, 849, 1777, 343, 5199, 1507, 11621, 17269, 98831, 110207, 133141, 507419, 995867, 3627611, 3625571, 151417, 14718853, 63230605, 119617803, 15138771, 143091751, 1035148273, 1169535473},
  {1, 3, 7, 3, 17, 57, 63, 63, 111, 977, 631, 3019, 2953, 14273, 29209, 35021, 23279, 244859, 384087, 609797, 1816125, 2175507, 5399139, 3314251, 29440837, 36847555, 52859399, 70767661, 139975749, 580321045, 1395405105},
  {1, 3, 1, 13, 9, 39, 87, 15, 397, 185, 701, 1487, 3807, 13727, 19883, 57349, 98519, 258509, 184057, 662533, 494635, 615435, 553995, 9131385, 28623949, 60022145, 8892339, 168371507, 69329115, 528621439, 692201585},
  {1, 3, 7, 1, 17, 57, 57, 157, 119, 181, 899, 353, 3603, 15041, 7421, 10781, 114047, 170011, 436117, 150533, 1836085, 2849797, 6939873, 8911947, 10869913, 36829479, 97074653, 91764423, 12125197, 306137993, 37174161},
  {1, 1, 7, 3, 29, 13, 29, 191, 105, 373, 961, 1991, 5531, 6793, 29497, 29821, 8237, 85691, 415455, 182297, 272425, 2721793, 1095763, 16472485, 1821617, 18370125, 131949227, 177733543, 369513677, 280452989, 924026609},
  {1, 3, 3, 11, 7, 61, 65, 39, 215, 187, 191, 1651, 2481, 3951, 24965, 44213, 75759, 76415, 363679, 634115, 1455929, 879925, 1620827, 9363275, 27026687, 17893323, 100467031, 199827173, 398657515, 751392809, 1035463345},
  {1, 1, 7, 5, 25, 11, 105, 23, 257, 771, 1359, 2837, 7821, 12223, 28033, 13589, 127869, 46299, 243929, 160541, 470839, 3566933, 731043, 3255917, 16514983, 19422819, 110451041, 68719801, 11902115, 253417709, 1387544529},
  {1, 3, 5, 11, 3, 3, 23, 139, 407, 885, 1679, 2979, 8149, 14281, 12487, 59661, 104663, 30857, 179647, 918815, 2046775, 3676947, 5655855, 6433507, 28345553, 5970683, 117491583, 114613809, 23157613, 158662523, 1948222385},
  {1, 3, 7, 3, 21, 45, 13, 85, 249, 1015, 2023, 1429, 965, 7091, 31721, 28005, 99519, 47707, 143447, 1042177, 660265, 1609009, 4176761, 9427957, 32977179, 2545531, 45725137, 113749873, 144513711, 199975205, 1927404497},
  {1, 1, 1, 13, 19, 5, 119, 47, 91, 285, 211, 2607, 4287, 9197, 455, 1741, 62453, 30605, 452401, 243471, 1546025, 46403, 2391923, 11695519, 28204873, 33321655, 29136755, 263059883, 149724441, 299471987, 328787313},
  {1, 3, 1, 1, 9, 59, 25, 137, 121, 287, 577, 3325, 2365, 8823, 5033, 32693, 4511, 210461, 377429, 167173, 707847, 4187941, 7149885, 5014613, 20885315, 36473725, 114190041, 69843537, 418841979, 861741477, 1664580945},
  {1, 3, 3, 13, 25, 63, 99, 43, 15, 855, 245, 3189, 59, 5181, 21299, 42925, 111527, 80047, 183929, 606493, 591651, 987487, 3737551, 15053035, 26595347, 59077793, 118939361, 75264167, 162510897, 305302567, 8334033},
  {1, 3, 5, 11, 7, 9, 41, 157, 359, 773, 1347, 2049, 4589, 13731, 32133, 26789, 113615, 40697, 459103, 712027, 2040829, 706301, 517745, 486203, 4549353, 24001767, 21337405, 104209017, 294791727, 1042948937, 1276516753},
  {1, 1, 7, 11, 31, 37, 83, 105, 183, 375, 79, 1821, 1989, 15199, 22207, 46741, 64453, 159979, 434935, 242267, 1137505, 2606023, 4695669, 11634275, 16417627, 54690987, 100370729, 163944049, 127188963, 390506227, 385299537},
  {1, 1, 5, 3, 23, 37, 127, 9, 467, 651, 993, 69, 6943, 4093, 20871, 56781, 2485, 160329, 354175, 33755, 1414737, 307115, 516821, 7325623, 5366599, 29336917, 7302305, 44611251, 40716945, 283564931, 741993425},
  {1, 1, 3, 15, 31, 49, 123, 149, 211, 371, 1825, 3011, 485, 1251, 17343, 30309, 101285, 185007, 475891, 765259, 702277, 1192335, 6477977, 1829127, 22501807, 65355573, 118647303, 250475913, 504538391, 754372195, 659291057},
  {1, 1, 1, 15, 11, 33, 127, 251, 89, 317, 1869, 219, 2275, 14201, 27063, 23485, 2133, 14221, 183923, 649431, 282181, 1747451, 3917847, 12019933, 3029113, 28863297, 42589335, 91008751, 33449197, 320461811, 326690609},
  {1, 1, 5, 5, 19, 5, 81, 35, 233, 95, 9, 863, 725, 11095, 16217, 15925, 111517, 128985, 312473, 21599, 652521, 1426085, 4163647, 5011829, 24512971, 25369061, 28104603, 250773593, 56867795, 259325349, 223517553},
  {1, 1, 1, 15, 23, 47, 51, 43, 169, 637, 865, 57, 1509, 1683, 7587, 55333, 75861, 46541, 36659, 319307, 676939, 1088455, 1618975, 9956525, 8924921, 52147693, 124566133, 199417433, 13551087, 607587887, 1761028369},
  {1, 3, 1, 3, 7, 7, 117, 187, 273, 303, 717, 3091, 2083, 3315, 647, 12821, 16327, 133037, 490103, 982859, 1558131, 2631841, 5555695, 14999117, 31897947, 19176257, 68550871, 140647615, 504245607, 824054915, 2028874673},
  {1, 1, 5, 15, 13, 27, 23, 227, 145, 547, 1783, 987, 6895, 7135, 11023, 35781, 38405, 125609, 320243, 839233, 1882719, 772299, 4095447, 1119749, 13976295, 409883, 27870383, 251871867, 346780539, 805406171, 1772405553},
  {1, 1, 5, 11, 21, 39, 57, 203, 477, 17, 985, 1729, 4297, 7483, 13263, 53221, 7077, 146089, 284919, 297561, 167539, 2309797, 4504463, 5298577, 9554373, 27610277, 53433501, 244052533, 364808359, 145160371, 1675839633},
  {1, 3, 7, 9, 3, 49, 71, 45, 143, 967, 39, 583, 2123, 5165, 17437, 26669, 112095, 82075, 480733, 796375, 1096157, 1109427, 8325409, 10666347, 5030043, 45663299, 57397043, 213013599, 135428537, 278939449, 1830815217},
  {1, 1, 1, 9, 21, 51, 71, 163, 441, 709, 397, 445, 6167, 7753, 11513, 63965, 4589, 156925, 237461, 495049, 601807, 1834891, 2476439, 10612141, 18710145, 46713537, 103935505, 260072155, 25349317, 5496157, 1322062161},
  {1, 1, 7, 7, 27, 35, 5, 181, 449, 53, 621, 3401, 5263, 4557, 9141, 36109, 13565, 133339, 387931, 655583, 1236951, 4021729, 5170513, 1580349, 7049145, 19631825, 125142661, 63086403, 77962257, 382453457, 1029125649},
  {1, 1, 5, 7, 7, 37, 83, 111, 485, 881, 465, 3371, 5603, 371, 29393, 29589, 117661, 77081, 324891, 422219, 2019777, 3996135, 5515635, 6696241, 17818693, 56193869, 23351543, 195039807, 278927111, 534773973, 1474342449},
  {1, 3, 1, 15, 7, 47, 41, 245, 377, 823, 309, 3929, 2159, 13917, 13365, 19277, 54527, 237661, 450267, 468555, 1669739, 1639965, 5251649, 12026517, 27230875, 57574953, 124292829, 151754451, 203063969, 438238289, 364526353},
  {1, 3, 7, 7, 7, 29, 25, 141, 19, 611, 79, 2689, 109, 12321, 8345, 51885, 26927, 180667, 467475, 198995, 1100617, 3672589, 8278641, 7947055, 23210639, 46819899, 117328389, 221987641, 290810669, 884225813, 2009396113},
  {1, 1, 1, 13, 3, 53, 113, 151, 381, 791, 137, 3185, 3567, 211, 597, 9077, 66061, 146365, 345297, 557535, 1193561, 193093, 1864395, 16564065, 263379, 33259541, 107579893, 951011, 27169775, 164751281, 667417489},
  {1, 1, 3, 9, 7, 53, 87, 89, 491, 861, 467, 3763, 2025, 4187, 9637, 20197, 112653, 248319, 121557, 1040339, 610905, 218227, 1260645, 9774511, 15289745, 15842311, 88741967, 44991805, 195013535, 314887273, 1107882161},
  {1, 1, 7, 1, 27, 33, 71, 41, 63, 1011, 741, 1135, 175, 3739, 21493, 30037, 50381, 217275, 49373, 96991, 1710413, 1726739, 3104549, 6632283, 13275199, 49468641, 80723851, 120090515, 520608383, 127714977, 1351903089},
  {1, 3, 3, 5, 9, 19, 55, 175, 325, 55, 1193, 1423, 2049, 9633, 17515, 57037, 102503, 204847, 90417, 166989, 1226479, 2365259, 3872059, 3429969, 25696835, 53364981, 87677515, 62974349, 421934949, 214579911, 649782385},
  {1, 1, 3, 1, 27, 55, 69, 103, 401, 707, 825, 399, 6799, 13199, 6295, 14597, 29925, 24815, 353021, 795599, 550395, 3965265, 5798195, 13368141, 11271495, 35402197, 50314939, 97029411, 273190027, 281337347, 1226123889},
  {1, 3, 7, 3, 19, 63, 25, 151, 17, 159, 1673, 615, 6317, 13261, 26267, 22045, 124103, 15083, 118327, 424775, 2055931, 3395453, 4196611, 10343037, 23061563, 1837605, 10837795, 204726305, 197599017, 629889223, 1916095729},
  {1, 3, 7, 9, 27, 1, 77, 129, 423, 647, 707, 2579, 3525, 6723, 31615, 34917, 79783, 152619, 316029, 641231, 198637, 532825, 3068173, 9634795, 6137275, 47329095, 11826791, 202819265, 469366567, 915603971, 2121210033},
  {1, 3, 3, 7, 7, 31, 35, 241, 309, 369, 895, 3683, 4795, 11319, 451, 46277, 46839, 164175, 335795, 171971, 639723, 3224439, 4925189, 3395825, 643229, 12194003, 2940383, 198064223, 55437843, 867201415, 1808796337},
  {1, 3, 5, 7, 17, 7, 117, 141, 267, 713, 569, 1915, 4369, 7793, 30853, 51597, 62047, 145241, 434131, 846413, 1839939, 1854745, 7180713, 10860983, 25106917, 58244205, 19575319, 155433805, 377004677, 855808841, 839297073},
  {1, 3, 7, 1, 29, 61, 81, 73, 413, 13, 1977, 3229, 5853, 8451, 15539, 44069, 91415, 131531, 276405, 677577, 254705, 3594013, 7366533, 9504801, 30047025, 42794709, 51741329, 148472001, 301906223, 576243455, 947877201},
  {1, 3, 7, 1, 5, 45, 109, 21, 431, 487, 2019, 2647, 927, 16015, 10711, 57221, 83303, 226795, 86261, 985553, 1902561, 922433, 8186713, 11663843, 14908235, 43104615, 132588451, 248786843, 460867323, 601925091, 2006022385},
  {1, 3, 1, 3, 11, 19, 37, 183, 451, 377, 269, 3993, 3229, 4899, 26561, 52901, 26847, 253789, 108183, 406215, 123495, 2929473, 987475, 5190719, 5835901, 44396073, 167125, 6324497, 440895615, 558815341, 1183867409},
  {1, 3, 1, 11, 5, 19, 121, 55, 57, 117, 687, 83, 3047, 1367, 17595, 40837, 45623, 108333, 204319, 596585, 1782183, 1466309, 749635, 16757237, 25621105, 26538883, 50609823, 251256323, 501931411, 826607215, 2100947377},
  {1, 3, 1, 7, 17, 31, 41, 219, 239, 963, 199, 2895, 5599, 7639, 17201, 4421, 64415, 243581, 143027, 96061, 866715, 4098725, 5251327, 893123, 21996215, 1419091, 11313963, 62026443, 13722499, 667187109, 1086649201},
  {1, 3, 3, 5, 27, 53, 71, 183, 509, 771, 1809, 1539, 2229, 4893, 17115, 9725, 97767, 44431, 327121, 1040511, 1401513, 2685395, 4047371, 2932697, 4386903, 50541325, 84336119, 12936537, 258849161, 564830439, 1200582321},
  {1, 1, 3, 9, 9, 9, 13, 49, 265, 643, 1929, 859, 497, 9797, 27771, 38333, 107173, 102671, 360917, 451709, 1799813, 1540593, 3418669, 3642949, 6412319, 18558293, 6125871, 51844093, 345309937, 1063856919, 1644134705},
  {1, 3, 7, 11, 19, 39, 115, 139, 207, 903, 963, 1849, 4403, 6229, 10021, 15021, 111391, 251899, 422335, 59815, 1000099, 2153207, 4034143, 7612395, 20154275, 1736951, 15958541, 36472343, 96831673, 754176369, 2033140753},
  {1, 3, 7, 13, 3, 57, 99, 223, 503, 423, 1755, 807, 1885, 213, 18723, 47325, 40135, 11211, 288089, 270199, 1092261, 1790359, 3780243, 15704995, 5973939, 41118319, 51441683, 170395417, 229119361, 508042687, 610826737},
  {1, 3, 7, 15, 11, 15, 111, 193, 243, 599, 593, 3385, 5393, 15073, 17777, 58157, 13295, 216859, 241915, 645951, 131227, 788651, 5555765, 15345639, 4546107, 65318149, 106843533, 240010365, 258097693, 360021389, 1109735953},
  {1, 1, 5, 3, 19, 63, 121, 207, 99, 435, 1961, 2747, 6405, 3971, 23481, 20485, 25245, 135801, 425791, 343359, 1896107, 856541, 7246267, 14335151, 1947479, 42672301, 35185215, 242396521, 424681719, 114316293, 678451761},
  {1, 3, 5, 13, 9, 29, 79, 131, 415, 49, 229, 1003, 3263, 12975, 15987, 15589, 43735, 123625, 264345, 578677, 187025, 1823923, 5010255, 9102843, 27579701, 46459473, 26383239, 189168819, 119848027, 379775271, 1469395249},
  {1, 1, 3, 7, 1, 41, 127, 155, 29, 73, 963, 659, 2741, 3465, 2595, 2797, 29093, 115471, 136667, 877685, 1998237, 2610387, 7242151, 4922329, 21605869, 5722943, 19296399, 257004225, 27887213, 642831871, 775751601},
  {1, 1, 3, 5, 23, 23, 93, 233, 113, 521, 427, 1557, 6917, 12953, 22441, 12333, 34765, 39519, 219257, 503075, 1023403, 2093497, 5739453, 6993901, 13991509, 54374295, 26479625, 204735825, 289149605, 356786949, 52754897},
  {1, 1, 5, 13, 5, 25, 85, 191, 387, 69, 955, 243, 4473, 9813, 21711, 62589, 36213, 103721, 193809, 617065, 1992757, 2562137, 1988739, 12159471, 26437921, 52149823, 115516591, 200407661, 471303281, 380038563, 1158518449},
  {1, 3, 3, 7, 1, 53, 95, 65, 231, 995, 539, 2103, 5513, 14087, 28655, 55477, 10471, 239887, 410963, 37477, 458785, 2031667, 3997837, 8847555, 31261551, 59048159, 35983219, 254346413, 344925651, 855451043, 1642686705},
  {1, 3, 5, 3, 17, 13, 19, 227, 197, 91, 1437, 1121, 3307, 6903, 3297, 45477, 119199, 153209, 194871, 59565, 312377, 3740799, 302287, 16313745, 16815551, 57017601, 79823733, 183901303, 395720331, 246549237, 1155693713},
  {1, 1, 5, 11, 31, 29, 109, 171, 257, 783, 861, 9, 4895, 1859, 10909, 54485, 57549, 26841, 145271, 631603, 1184297, 2375225, 1607223, 4930253, 12299003, 12557681, 68923693, 72936803, 238054239, 852385945, 1062619153},
  {1, 1, 7, 13, 5, 47, 61, 5, 363, 351, 1525, 823, 2883, 12435, 17629, 373, 27789, 151515, 222833, 955169, 1299987, 2649617, 5398177, 5530391, 11219187, 12727361, 93639907, 165431935, 148894071, 691202777, 1614184177},
  {1, 1, 5, 11, 9, 3, 69, 159, 371, 477, 1223, 1973, 2757, 413, 31223, 37693, 79125, 120425, 67223, 695141, 778807, 479817, 4371147, 8169359, 29012633, 8827843, 74715145, 59897313, 284469441, 1044508315, 1492400849},
  {1, 1, 3, 5, 23, 45, 43, 195, 423, 829, 1673, 1563, 6633, 14775, 21097, 6453, 66557, 105535, 11065, 950435, 610833, 4193871, 7572223, 12701619, 5508953, 51932957, 117261447, 257480245, 364474531, 563897381, 2069142065},
  {1, 1, 3, 3, 13, 9, 107, 209, 49, 609, 1047, 3691, 7483, 4269, 7557, 2829, 26781, 127743, 373823, 8249, 758317, 684631, 1744589, 6674277, 31053133, 13431755, 90616855, 226934111, 221220745, 273523041, 1773004337},
  {1, 1, 3, 15, 3, 43, 73, 161, 53, 813, 325, 3439, 7009, 8691, 11711, 30949, 52229, 202447, 141523, 850551, 901695, 1750573, 195205, 2579353, 32706353, 14058345, 129204547, 106201709, 259887127, 460554419, 1315333233},
  {1, 1, 3, 3, 23, 45, 99, 61, 407, 15, 1515, 1557, 953, 8567, 13729, 35829, 67357, 96127, 476351, 508067, 363785, 1202999, 5852513, 15647075, 25019091, 3818191, 118834721, 175693245, 109136235, 1734589, 901165777},
  {1, 1, 5, 9, 31, 35, 117, 57, 227, 923, 1373, 1811, 3405, 11979, 10149, 44645, 32813, 216217, 270965, 921139, 1035551, 2837297, 2762109, 189951, 9612295, 50666793, 132314711, 90160881, 309961407, 424066489, 47077553},
  {1, 1, 3, 9, 15, 53, 105, 209, 153, 67, 1477, 667, 3077, 4911, 3871, 62901, 62181, 4879, 505685, 46459, 1654073, 1386253, 2140045, 12467269, 32767999, 63591961, 28056175, 46184889, 129024379, 970226403, 1383922993},
  {1, 1, 3, 3, 21, 53, 93, 101, 183, 1023, 3, 3041, 5815, 9043, 5801, 31765, 130717, 17279, 441663, 741537, 684305, 1429761, 682825, 14230107, 20281971, 6351751, 57147733, 167253395, 397943519, 173556653, 298356113},
  {1, 3, 3, 5, 17, 49, 127, 161, 321, 869, 1369, 923, 3235, 711, 30007, 37541, 26631, 140495, 187217, 730485, 696621, 228675, 8283613, 16295813, 5792297, 44786053, 65266423, 124254687, 130648851, 627679059, 995676465},
  {1, 1, 3, 3, 15, 17, 97, 229, 389, 159, 1075, 2001, 7905, 15191, 14693, 48565, 49549, 227679, 61695, 654907, 1776437, 896341, 5546889, 7345945, 11304947, 40586303, 43451125, 22872869, 123545563, 784550841, 531711889},
  {1, 1, 5, 11, 5, 5, 121, 173, 95, 173, 1883, 3915, 1439, 9981, 24375, 22765, 112085, 181097, 292631, 9065, 1089841, 3595077, 6804865, 10009507, 22914593, 24395647, 37939191, 146875083, 71682873, 826477571, 1682797361},
  {1, 3, 3, 1, 31, 53, 29, 189, 37, 623, 217, 949, 3959, 7189, 25427, 333, 112583, 56527, 429653, 505851, 398649, 2548537, 139745, 10789673, 6923715, 66120661, 106868817, 112350107, 536465753, 741896279, 876106193},
  {1, 3, 5, 9, 21, 45, 101, 23, 355, 729, 797, 2317, 2931, 7433, 29175, 11421, 951, 58921, 460445, 833897, 1268017, 1736521, 3552571, 13869807, 21085405, 9818841, 45202337, 24217119, 110932773, 602129827, 651594065},
  {1, 3, 7, 1, 1, 63, 63, 155, 237, 865, 1169, 43, 7335, 6445, 7979, 44013, 13543, 157899, 450709, 707125, 1479507, 3304419, 7183583, 14661017, 24227045, 28226349, 65181655, 27800603, 461732857, 157674271, 302405681},
  {1, 3, 7, 7, 11, 51, 37, 199, 503, 991, 319, 3013, 7885, 12837, 32419, 7901, 27847, 136843, 90643, 102719, 1032007, 2624481, 1827971, 3400939, 21380603, 37667243, 6407825, 88067993, 254437529, 771146223, 441400273},
  {1, 3, 7, 7, 27, 31, 101, 243, 37, 811, 1909, 109, 6455, 7903, 11821, 61093, 116399, 261723, 311731, 292463, 742379, 385249, 1463431, 14017625, 24866591, 40148105, 67104825, 19342155, 87042195, 856715113, 1429231697},
  {1, 1, 3, 13, 23, 21, 89, 99, 243, 605, 1017, 1871, 1101, 12825, 8227, 43629, 40933, 240335, 373905, 900643, 1562953, 2909117, 6855935, 15416751, 2053225, 57899869, 45811315, 197460689, 328930077, 778782239, 1032407409},
  {1, 3, 3, 13, 19, 3, 51, 59, 501, 605, 385, 2189, 3229, 7981, 31407, 13117, 77367, 116079, 309209, 396983, 244863, 3094919, 6443103, 12620017, 30970809, 32020125, 76591809, 263822593, 13208865, 876122051, 582200913},
  {1, 1, 1, 1, 25, 11, 127, 215, 295, 237, 1245, 3657, 7803, 3897, 655, 41949, 102805, 162349, 400733, 323493, 829559, 3020259, 7925955, 9417203, 17503001, 32208553, 27416597, 237283511, 220807301, 1060277019, 1917143825},
  {1, 1, 7, 7, 5, 9, 63, 129, 143, 417, 795, 3409, 2847, 5887, 3093, 15397, 75597, 201627, 395323, 703841, 1311197, 1073283, 4661469, 5136843, 11608109, 39773503, 126935197, 193968619, 304066387, 774540337, 1064486545},
  {1, 3, 3, 13, 7, 57, 67, 57, 5, 847, 1185, 3349, 4841, 11457, 8857, 65453, 57999, 249375, 248121, 498531, 179141, 1901223, 4823061, 1996881, 11335683, 8075565, 118038065, 9460005, 29773325, 109782949, 1451452369},
  {1, 1, 3, 3, 9, 53, 51, 43, 85, 437, 13, 2543, 3651, 15493, 767, 37101, 87413, 29679, 196319, 165565, 571985, 2592223, 3713063, 3402113, 20519857, 52866641, 17930371, 129920431, 287257057, 194071099, 600559729},
  {1, 1, 7, 9, 1, 49, 97, 115, 133, 1011, 1399, 2653, 7765, 13999, 12097, 59445, 43037, 23867, 104309, 879077, 936157, 3786941, 5317135, 16111353, 30627271, 31139339, 79522921, 42777825, 369832371, 472478509, 922981201},
  {1, 1, 5, 1, 3, 27, 123, 107, 389, 401, 1759, 1333, 1371, 5277, 14865, 47373, 123853, 77977, 444477, 991087, 1586119, 55503, 1980015, 10024801, 23346933, 40012035, 30566345, 183003871, 50786881, 665837509, 122228177},
  {1, 1, 5, 1, 13, 23, 3, 123, 137, 821, 399, 1671, 3095, 3121, 31387, 20189, 27333, 105097, 342045, 397601, 1966923, 864655, 8016447, 4294413, 2263761, 51370851, 61335691, 260781363, 172062053, 234084719, 12370161},
  {1, 1, 5, 3, 7, 35, 57, 237, 509, 753, 1783, 2815, 6495, 13283, 7091, 2773, 28405, 70633, 438751, 117163, 1913207, 1220749, 3046153, 12950737, 11525405, 64585947, 38112099, 102341563, 3651543, 833598631, 1271192945},
  {1, 1, 7, 11, 5, 37, 77, 109, 7, 969, 1087, 3705, 1695, 14223, 28959, 33381, 110565, 136651, 120215, 360737, 1891393, 3871377, 7575361, 2186603, 33480357, 17756411, 117177045, 141577531, 183793443, 600505403, 1916760081},
  {1, 3, 1, 11, 25, 5, 25, 163, 179, 185, 671, 1031, 4537, 11601, 9323, 7725, 87815, 183309, 523807, 622133, 2000497, 645781, 4295599, 1412303, 14820557, 60877499, 49576107, 153948317, 255525125, 586101799, 1134398961},
  {1, 1, 3, 7, 17, 25, 49, 221, 183, 619, 1953, 343, 4523, 14883, 6833, 25141, 113277, 119551, 285691, 691685, 667117, 509341, 13793, 1773579, 4493927, 29562725, 52772723, 170995047, 385725671, 771523477, 735996657},
  {1, 3, 7, 5, 27, 19, 59, 153, 11, 807, 513, 3019, 6875, 5307, 8405, 47829, 130735, 208219, 264881, 25967, 776943, 1844655, 2964477, 5006895, 15113483, 32271989, 48787391, 147040823, 505907791, 2664017, 1905792305},
  {1, 1, 1, 13, 25, 41, 21, 109, 321, 135, 497, 1235, 5177, 5167, 18609, 60821, 117405, 69949, 513649, 475365, 1996261, 3635177, 5602097, 252845, 6619371, 46333997, 130351479, 100605109, 369959547, 359862525, 1390782897},
  {1, 1, 7, 5, 21, 53, 25, 197, 411, 503, 1009, 1921, 4305, 2633, 31415, 8013, 119285, 6123, 189401, 998321, 201577, 82677, 1384905, 5195751, 946715, 21886533, 125509429, 102589053, 190723349, 835593035, 2042511761},
  {1, 3, 5, 1, 25, 45, 27, 227, 271, 903, 639, 3805, 657, 8683, 29585, 33413, 24319, 42105, 217205, 380517, 1362769, 2106375, 2921711, 10116347, 2962787, 35112715, 124633769, 56051973, 435743799, 20627437, 549758545},
  {1, 1, 5, 3, 9, 49, 37, 35, 351, 491, 851, 2983, 31, 5619, 6919, 3093, 123429, 34761, 155167, 823717, 1134821, 362537, 6498431, 9808371, 17550015, 16034055, 46855203, 257147611, 104260711, 488509907, 658950641},
  {1, 1, 5, 3, 11, 49, 33, 153, 393, 1017, 1561, 2795, 4435, 12589, 22349, 54733, 8701, 164729, 98175, 564071, 1600613, 3176997, 4613845, 3440405, 27081765, 34814981, 98057911, 218087183, 466321593, 536190889, 1840891441},
  {1, 1, 1, 15, 17, 29, 49, 245, 217, 359, 1133, 393, 3317, 415, 16407, 54069, 46917, 252941, 166035, 206765, 664793, 2004605, 6369945, 9022021, 7068939, 21123665, 130128621, 108176137, 19266243, 980233723, 1521612817},
  {1, 1, 3, 5, 3, 9, 95, 63, 319, 319, 1009, 19, 6453, 16279, 6975, 37781, 33765, 171855, 399129, 873655, 557813, 4028971, 450163, 9203291, 2824331, 33888485, 120359239, 245210841, 299009411, 48009795, 1867100849},
  {1, 1, 5, 9, 3, 25, 67, 95, 369, 237, 285, 2409, 671, 5143, 121, 3781, 30653, 10745, 94069, 600431, 352613, 2459191, 1324627, 13310693, 16073273, 12862449, 28950661, 227270579, 364398699, 975922045, 2092761617},
  {1, 1, 3, 1, 9, 49, 35, 87, 317, 185, 445, 2263, 7923, 10183, 26615, 30949, 62917, 167951, 279197, 495549, 932829, 2938719, 5548011, 6034761, 5994381, 28049057, 30106507, 63922495, 110574979, 910898739, 1546338289},
  {1, 3, 3, 11, 9, 59, 29, 135, 129, 477, 353, 3571, 1057, 16329, 23523, 35533, 82023, 198991, 99871, 902061, 1138399, 2503025, 2673459, 11261213, 27221113, 51897517, 108686495, 21968381, 469129277, 1013231631, 919706545},
  {1, 1, 1, 15, 13, 11, 19, 5, 133, 827, 1799, 1893, 1939, 1101, 12147, 31757, 82501, 4109, 313747, 695217, 452303, 3814703, 2235345, 7820513, 22604903, 42849067, 59651153, 136566631, 516252849, 968084295, 7868625},
  {1, 1, 3, 3, 15, 49, 33, 185, 511, 1013, 41, 3499, 6235, 7643, 16725, 20309, 21805, 250847, 394047, 747771, 1163093, 2070357, 2726165, 2759075, 32774953, 15920877, 86762871, 241705159, 344460751, 581457809, 536377649},
  {1, 1, 5, 11, 27, 45, 89, 157, 63, 137, 2047, 1275, 4995, 625, 6111, 26365, 39893, 123177, 467159, 596791, 1273305, 3794269, 1280929, 6529187, 16163397, 11313179, 122217879, 61191463, 134892373, 234333723, 663133233},
  {1, 3, 7, 11, 3, 1, 121, 1, 341, 33, 1895, 3033, 3929, 10257, 21037, 43949, 53599, 140859, 354815, 554103, 61253, 2459517, 7744397, 15987497, 27488349, 61371467, 55753605, 261045757, 432473021, 683113873, 1174113041},
  {1, 3, 3, 11, 7, 11, 117, 5, 115, 287, 335, 3415, 5397, 15065, 19121, 60605, 128863, 113903, 286095, 699539, 400063, 1960625, 5518481, 16385151, 2139939, 25481515, 24565043, 61604177, 327951229, 707549917, 103025393},
  {1, 3, 3, 13, 21, 25, 15, 125, 277, 125, 801, 3761, 2623, 11333, 16867, 61949, 98391, 86719, 507529, 745857, 1117365, 2909107, 2760753, 3645537, 106609, 35545629, 15019373, 69659091, 224798729, 423248407, 2130115473},
  {1, 3, 5, 11, 19, 33, 21, 71, 499, 747, 1515, 185, 1759, 14623, 895, 53013, 39671, 132185, 370319, 325023, 1503653, 673153, 3434715, 4282671, 13965791, 4087199, 92046605, 114265771, 326088227, 853534283, 1676344337},
  {1, 3, 7, 1, 29, 35, 9, 203, 277, 299, 1509, 2017, 2897, 14175, 1643, 21221, 107335, 198203, 358789, 814489, 1331871, 3892981, 2307375, 15536809, 30944463, 25498217, 39425637, 252475165, 493719155, 983030439, 1120696977},
  {1, 3, 5, 11, 7, 47, 111, 197, 459, 941, 1619, 2119, 2191, 11049, 6811, 39341, 98519, 253433, 410767, 883851, 1855403, 1863851, 6535617, 12331919, 7814209, 40666583, 29918763, 23633451, 48464429, 30192927, 1466282993},
  {1, 1, 5, 9, 7, 43, 103, 115, 87, 269, 1235, 77, 5887, 1611, 29041, 20741, 18285, 20009, 285061, 833179, 1533607, 1939379, 6012775, 5623571, 23187065, 21285271, 62646849, 93172059, 179379047, 340350053, 1740856657},
  {1, 1, 5, 7, 1, 61, 83, 225, 179, 81, 1145, 2403, 1485, 8967, 20607, 41285, 96789, 222105, 53387, 322973, 349833, 786671, 587165, 2144151, 381133, 64976189, 1372223, 115169057, 157443011, 296535507, 1596042673},
  {1, 3, 3, 1, 25, 47, 27, 157, 359, 803, 1683, 1995, 6445, 13113, 17899, 43069, 10167, 179007, 417413, 796045, 2091443, 1479927, 1699177, 2019059, 15180047, 43606631, 99249151, 81082105, 90925069, 1042527367, 1836619057},
  {1, 3, 1, 7, 21, 37, 43, 119, 245, 49, 1581, 2275, 3311, 4087, 29765, 36245, 24927, 28173, 277635, 275593, 1684657, 4142535, 6713291, 10116081, 7731509, 52423361, 54945807, 52962387, 365047699, 987890273, 1268909745},
  {1, 1, 3, 13, 5, 33, 49, 191, 455, 105, 665, 3855, 3207, 2671, 32203, 34933, 40229, 116351, 104577, 373377, 823949, 3571917, 3767187, 763275, 13940637, 51407357, 40489747, 216835115, 267274011, 778302767, 1692157297},
  {1, 3, 1, 1, 25, 63, 19, 217, 17, 353, 947, 1951, 4097, 9041, 11921, 58301, 89055, 177261, 17029, 42373, 1147827, 4108287, 5464429, 3968293, 6655261, 53295295, 14048019, 192716613, 350005813, 974967277, 162700145},
  {1, 3, 1, 7, 21, 31, 113, 97, 347, 993, 1799, 3831, 3711, 6193, 1235, 39757, 55, 9149, 255619, 511625, 1601931, 1822365, 859957, 15415527, 6277741, 28171235, 86322731, 49377259, 138581077, 898880279, 1914719473},
  {1, 1, 1, 5, 3, 63, 11, 203, 425, 445, 1361, 531, 1265, 1755, 11685, 2389, 53725, 150861, 171529, 202255, 729635, 1961519, 1487407, 13428349, 11271369, 20687789, 69019567, 186677853, 367386143, 824569, 434644913},
  {1, 3, 1, 7, 13, 29, 23, 85, 57, 467, 1835, 133, 7961, 4175, 2445, 5413, 58463, 180621, 6659, 131601, 1650185, 577051, 1696329, 11615885, 22615615, 62142311, 34084865, 189863725, 381492483, 1043558337, 997295825},
  {1, 1, 1, 15, 23, 27, 37, 5, 123, 913, 1293, 1633, 3113, 5413, 26407, 57933, 101285, 184541, 307715, 559643, 406063, 1835521, 3693265, 2104887, 13715629, 63528649, 20283901, 206410389, 72581833, 8301195, 560571793},
  {1, 1, 5, 13, 27, 1, 121, 151, 303, 931, 375, 3679, 1863, 12301, 30907, 40061, 21365, 99193, 419329, 314887, 89629, 1994365, 96843, 5019331, 3676215, 28380019, 13392811, 50888067, 411610313, 27495775, 1762966641},
  {1, 3, 1, 9, 31, 9, 49, 203, 177, 937, 1503, 933, 5867, 12533, 13621, 5213, 40783, 23053, 355341, 1047555, 1049637, 1711109, 4068375, 3898429, 33205517, 66039107, 15527049, 15229151, 103701657, 156919729, 504257745},
  {1, 3, 3, 15, 1, 41, 23, 191, 191, 931, 837, 3553, 2611, 4735, 18105, 24565, 44463, 239983, 183307, 670741, 1057805, 3503211, 7494675, 4861355, 16094695, 50421825, 45017749, 170014271, 211919827, 872867197, 229639057},
  {1, 1, 5, 7, 27, 49, 51, 111, 435, 195, 1229, 711, 7145, 14571, 31707, 34581, 102693, 214201, 453643, 748551, 316421, 1539175, 7980147, 14159095, 27626727, 62616953, 101972051, 150964933, 501782943, 593149511, 1706397681},
  {1, 1, 7, 7, 3, 41, 59, 203, 291, 903, 1727, 2757, 1463, 6287, 31535, 49237, 109109, 34139, 481291, 268311, 866317, 456719, 3352695, 3455111, 10320243, 20216939, 54502849, 83274611, 250338915, 37053403, 2083338193},
  {1, 1, 7, 13, 23, 5, 75, 3, 207, 525, 411, 2133, 2231, 477, 7155, 47645, 3797, 200123, 303105, 876547, 1859593, 3006591, 1941679, 2580939, 2531417, 18539199, 50832921, 189431475, 288328785, 546233223, 1131124945},
  {1, 3, 5, 7, 13, 19, 111, 225, 489, 83, 1177, 4077, 4617, 14413, 7133, 51757, 64783, 32009, 26371, 599809, 572199, 1722211, 2105285, 9662413, 15650223, 19843893, 9947369, 22859253, 183426609, 217491857, 760143953},
  {1, 3, 1, 7, 9, 59, 3, 113, 379, 803, 1289, 3347, 4127, 6669, 14867, 2429, 28503, 255549, 174851, 578837, 2000175, 3913503, 3829205, 12851119, 30936063, 26049645, 133449543, 98746915, 59795209, 435462135, 897425073},
  {1, 3, 7, 3, 31, 37, 87, 79, 399, 749, 995, 1611, 3137, 12543, 31955, 20933, 101671, 9979, 312583, 817435, 1678097, 758067, 6613283, 777979, 16130153, 3696295, 74656903, 78130597, 243584459, 887177871, 993603889},
  {1, 1, 5, 7, 21, 59, 49, 45, 511, 639, 1033, 2169, 3265, 15001, 10745, 31613, 58813, 249385, 55563, 911625, 258319, 2730333, 6629737, 6139363, 19885635, 45513629, 113202557, 52237853, 108864565, 68506677, 1989537297},
  {1, 1, 5, 1, 25, 19, 23, 203, 11, 883, 1031, 4087, 5059, 11321, 21675, 41229, 15317, 13433, 280333, 341765, 831, 2685307, 1338191, 2084303, 15734247, 15973235, 108908387, 201022791, 450612445, 417357599, 232816881},
  {1, 3, 7, 5, 11, 27, 33, 205, 163, 289, 501, 3505, 1515, 1895, 15889, 61269, 27727, 11819, 306433, 71951, 832311, 3112741, 3975673, 1211903, 4989981, 15824401, 129373485, 252595255, 107856139, 481556789, 246377617},
  {1, 3, 1, 1, 23, 7, 39, 239, 29, 119, 1499, 2071, 6495, 12107, 5339, 62757, 67655, 166525, 106245, 162571, 1352459, 862067, 2742091, 6764985, 14457331, 11738071, 65986595, 33612427, 248843983, 652312095, 1964635633},
  {1, 3, 1, 1, 23, 29, 55, 181, 327, 905, 427, 1033, 427, 3687, 5367, 32981, 10199, 15389, 302853, 431371, 446225, 1153891, 234361, 14061683, 5813221, 51408927, 112554261, 228809807, 359104027, 990893331, 2117488913},
  {1, 3, 3, 7, 21, 27, 115, 127, 393, 855, 1291, 2121, 381, 9995, 29757, 9877, 56767, 31055, 376515, 541505, 493023, 682719, 6680315, 15336413, 7447411, 10166295, 32301853, 65155241, 173802039, 309606465, 681846289},
  {1, 3, 5, 1, 25, 13, 15, 183, 269, 1005, 1531, 3451, 3975, 9479, 23695, 36053, 13495, 205209, 501061, 542549, 661857, 2442147, 2981387, 7565209, 629129, 16407735, 20416359, 195401795, 54890355, 658080651, 1165250353},
  {1, 3, 7, 7, 19, 31, 111, 97, 33, 179, 1343, 2069, 977, 5043, 9129, 59861, 38383, 67691, 76739, 437079, 1158395, 793819, 6861509, 5517093, 15056399, 39885811, 16686993, 118578421, 149080031, 974302853, 943230929},
  {1, 3, 1, 5, 17, 57, 99, 129, 379, 829, 837, 1845, 3613, 7351, 19291, 8229, 107079, 8829, 424129, 357965, 831589, 2651087, 6453421, 11745535, 4582369, 57814113, 133111513, 170391201, 124588939, 19336391, 801876177},
  {1, 3, 3, 5, 31, 23, 119, 229, 135, 389, 9, 705, 6697, 15441, 5303, 2141, 71559, 259231, 87233, 376523, 2034523, 2544355, 6766849, 9308067, 14356585, 32544125, 111081901, 149177701, 12446941, 124194907, 2039583121},
  {1, 1, 1, 11, 25, 31, 105, 95, 5, 931, 789, 375, 7543, 9957, 28627, 12445, 16213, 216189, 257095, 93397, 951419, 2569365, 3478827, 9789081, 16502999, 14709049, 123952531, 257156859, 409404881, 907368567, 415479793},
  {1, 1, 7, 15, 21, 17, 19, 103, 389, 545, 1725, 2867, 4251, 3829, 6907, 29565, 69557, 75963, 323, 915137, 1289813, 2900223, 6585675, 7102209, 28942437, 60892241, 46260319, 256883751, 516676753, 410256079, 1004824497},
  {1, 3, 7, 7, 15, 37, 97, 65, 337, 409, 1649, 2869, 7929, 8905, 21989, 53501, 2831, 121867, 29379, 911563, 136129, 2902949, 1731565, 14797501, 24901845, 7084157, 118626329, 85952565, 115584597, 592306281, 54356433},
  {1, 3, 5, 3, 11, 15, 69, 29, 353, 207, 233, 411, 2047, 10303, 31655, 11381, 18391, 15065, 72775, 873031, 553067, 1433777, 7127353, 14844877, 22676947, 22874565, 66654959, 234300899, 154362331, 320577859, 90313521},
  {1, 3, 3, 7, 27, 43, 125, 107, 69, 981, 215, 1955, 3589, 597, 12703, 40717, 27207, 109151, 117955, 23503, 1632623, 420073, 3238751, 15792297, 23501521, 42536155, 21705375, 246483505, 16203041, 799761747, 1685713585},
  {1, 1, 7, 9, 25, 13, 109, 73, 227, 663, 1115, 285, 471, 3359, 15787, 32597, 24165, 76603, 336581, 952013, 1528273, 3554817, 7859957, 10381055, 26081435, 28858847, 128166169, 94643523, 200856563, 850606303, 1797214801},
  {1, 3, 7, 5, 1, 45, 7, 79, 441, 149, 701, 1457, 6595, 14829, 20865, 58237, 112799, 79339, 332225, 529989, 1392449, 2809451, 4350515, 5761805, 31572689, 25608289, 44322669, 123133023, 343599873, 327413573, 86876561},
  {1, 3, 7, 15, 15, 47, 83, 239, 295, 23, 1085, 813, 1209, 3573, 2855, 48781, 66663, 213403, 272203, 1025099, 1813739, 563799, 6456043, 13862611, 15905011, 61993497, 110583689, 168668725, 413422953, 847497067, 1861640017},
  {1, 1, 3, 15, 13, 7, 59, 67, 255, 537, 1841, 3857, 6821, 15175, 13997, 60645, 50717, 172623, 67651, 970953, 1891939, 544311, 2513527, 16090859, 2080525, 46099349, 131148621, 3072617, 316436555, 547633521, 2125754513},
  {1, 3, 1, 1, 9, 57, 59, 21, 21, 41, 1693, 2805, 7953, 1389, 14105, 24237, 41439, 187245, 348485, 517717, 180085, 1445463, 2778745, 5169009, 4704229, 39949857, 71817913, 158047469, 118602001, 807395253, 823770065},
  {1, 3, 5, 15, 19, 49, 107, 117, 99, 607, 145, 53, 1863, 9383, 12029, 39861, 52095, 37001, 207563, 248543, 2055781, 567327, 4557033, 6215943, 1628947, 57233013, 1412625, 164926779, 299799995, 725078665, 1334426833},
  {1, 3, 3, 13, 1, 39, 5, 141, 503, 265, 281, 1785, 2673, 6597, 6333, 47309, 17167, 169551, 25417, 714325, 1753163, 2455593, 1194201, 13432859, 13555797, 48978669, 6806693, 190369565, 285524393, 712041329, 1572886033},
  {1, 1, 5, 3, 3, 19, 3, 181, 169, 269, 955, 2399, 3157, 11053, 8563, 53805, 949, 45497, 327759, 42591, 2042871, 82007, 409761, 11485565, 8569601, 14926967, 64048355, 245320209, 269549257, 81211551, 270081905},
  {1, 3, 3, 13, 11, 1, 95, 43, 179, 507, 443, 209, 3239, 14239, 21829, 59157, 73439, 256079, 411849, 199903, 727149, 4059227, 3141815, 1003783, 28061935, 41758959, 25694949, 254935635, 472253379, 182615441, 1310388881},
  {1, 1, 7, 9, 3, 17, 99, 179, 445, 479, 1897, 1507, 5753, 4757, 2135, 44269, 28069, 66075, 4421, 146519, 1413837, 2544743, 8015023, 8179065, 32687691, 44054725, 24962871, 186624485, 416837265, 788510187, 406478513},
  {1, 3, 3, 1, 9, 51, 29, 13, 295, 291, 927, 85, 5707, 7447, 32319, 17941, 21687, 53279, 93125, 282333, 560879, 2270321, 6528073, 5131275, 19271103, 15720747, 27094305, 89487983, 388245499, 200026235, 2101634257},
  {1, 1, 1, 3, 13, 11, 21, 157, 213, 327, 1071, 591, 2639, 15405, 6617, 51325, 34749, 247469, 472015, 1043265, 1513039, 1277817, 5296025, 4787097, 21188699, 65631187, 3043899, 9055883, 25072169, 568632541, 1018567793},
  {1, 3, 5, 1, 7, 25, 55, 47, 495, 681, 727, 2707, 2955, 705, 7489, 61613, 90623, 67305, 249541, 1000523, 777333, 1693539, 7127987, 3551163, 11839213, 35721939, 69672463, 199612767, 245421813, 259757365, 1895397553},
  {1, 1, 3, 9, 17, 3, 73, 67, 465, 367, 1473, 3195, 7825, 5299, 1817, 9653, 2605, 254063, 278597, 389333, 90751, 1237253, 4517767, 11384621, 6346603, 46825949, 71011471, 67280813, 404201111, 115794373, 195331377},
  {1, 1, 1, 1, 19, 31, 77, 253, 71, 599, 1601, 871, 2243, 6699, 13013, 6533, 24829, 16973, 237261, 985311, 1932115, 845641, 2976089, 16005323, 27769483, 44092789, 8865475, 239029055, 76336639, 273653281, 1643710449},
  {1, 1, 7, 9, 21, 1, 71, 115, 5, 65, 767, 925, 7901, 10761, 19431, 30941, 73285, 130843, 513349, 941121, 1172445, 1273115, 3302303, 6040017, 1509621, 40893235, 114762737, 8846313, 182302277, 889619003, 59084113},
  {1, 3, 1, 7, 23, 31, 31, 15, 105, 391, 585, 2995, 2635, 10607, 24951, 62613, 86231, 167517, 211139, 120267, 1777611, 2785083, 5424907, 9029037, 11126659, 7310165, 108730919, 267006871, 267406635, 2933891, 141932977},
  {1, 3, 3, 1, 19, 25, 71, 211, 41, 197, 787, 225, 6781, 813, 10117, 23949, 108015, 131599, 401733, 813383, 220997, 4173123, 5965295, 7605309, 6844145, 24940271, 132694525, 259049177, 51833329, 513310777, 1287445649},
  {1, 3, 3, 3, 17, 29, 3, 153, 231, 643, 1151, 447, 3699, 9625, 26677, 49645, 52431, 229871, 441383, 551333, 1244841, 1863871, 1587333, 14637539, 17238879, 4352267, 112700523, 216280631, 223749629, 787364065, 981086321},
  {1, 1, 5, 9, 1, 25, 71, 21, 395, 297, 557, 3841, 233, 1877, 4569, 941, 100221, 148489, 373669, 595517, 20661, 1492651, 4826201, 15422975, 13985253, 47232777, 66583453, 249703221, 173321433, 307409365, 315641233},
  {1, 1, 3, 13, 1, 45, 115, 61, 5, 937, 173, 2109, 2927, 9599, 9155, 23285, 78757, 57823, 425185, 484197, 165793, 3357119, 3793505, 3300225, 14862181, 57212017, 80691793, 30038963, 380458947, 217764407, 1406474577},
  {1, 1, 3, 3, 15, 21, 61, 121, 253, 285, 1083, 3545, 5537, 6773, 2629, 14381, 1421, 149487, 109487, 345003, 1731233, 1077945, 6104757, 13096513, 11102065, 63504679, 88018021, 136118789, 155000809, 601963305, 709962513},
  {1, 3, 3, 15, 13, 63, 33, 77, 49, 849, 1795, 2771, 5481, 9833, 603, 33469, 38871, 12959, 225579, 743289, 1360827, 3653261, 6315833, 4257837, 28530765, 8638159, 123720639, 130277581, 72815773, 246629575, 2032270769},
  {1, 1, 7, 5, 1, 39, 113, 237, 225, 1005, 1687, 2297, 3213, 2605, 14669, 52397, 111277, 128683, 367017, 599477, 1561003, 3226109, 2767129, 12212541, 15423857, 31415067, 30364645, 172318361, 4290257, 592522897, 1595004689},
  {1, 1, 3, 1, 11, 1, 39, 23, 67, 441, 1235, 2545, 3139, 15901, 29243, 24605, 88341, 18719, 196525, 529263, 1982397, 341667, 4349867, 7557495, 24834365, 32478919, 67468653, 2919335, 13219233, 224175903, 1035060881},
  {1, 3, 1, 3, 15, 49, 39, 57, 311, 345, 525, 223, 4923, 6311, 25275, 64021, 48263, 81309, 343271, 216947, 1665685, 4020451, 1682213, 10724547, 14317285, 14822841, 30382179, 231861607, 335420371, 1052142607, 1930317425},
  {1, 1, 5, 7, 9, 13, 69, 11, 349, 423, 1773, 1055, 1001, 9359, 17025, 20517, 6157, 80969, 408555, 484917, 433817, 4169721, 3941207, 6943553, 5888771, 1175265, 11501899, 5545349, 2604475, 165723069, 1743972209},
  {1, 1, 1, 13, 15, 63, 89, 207, 335, 591, 1223, 2701, 55, 12471, 13127, 34389, 58229, 2045, 19681, 8931, 668835, 2722989, 4826603, 9107443, 13816283, 42833379, 44210793, 76390963, 243132635, 261168787, 984725329},
  {1, 1, 3, 5, 15, 19, 83, 67, 407, 113, 1961, 779, 5803, 12417, 21751, 49485, 81029, 127871, 228009, 790123, 989247, 4012135, 6907927, 14633187, 26434053, 34228621, 94682135, 225942503, 409385733, 1065189411, 829745969},
  {1, 3, 3, 1, 21, 53, 81, 95, 405, 427, 1047, 2443, 4153, 5843, 22511, 15957, 125527, 97215, 347877, 898145, 908297, 2206829, 6579267, 8179033, 32587007, 52679195, 20829615, 161970909, 70072039, 825715251, 325304881},
  {1, 1, 7, 7, 7, 25, 115, 155, 453, 537, 741, 2379, 2343, 16035, 19587, 44325, 27781, 143579, 141035, 393971, 115741, 2276471, 7392295, 12598305, 3047181, 14460425, 44049767, 4406555, 274581511, 535829863, 1009813041},
  {1, 3, 3, 11, 27, 21, 111, 121, 503, 437, 803, 3399, 5303, 10163, 18199, 56037, 95959, 132479, 32495, 591471, 416257, 113211, 1127061, 15903619, 18742857, 41695127, 15496139, 209932467, 363880279, 718208851, 1622933489},
  {1, 1, 5, 13, 19, 27, 7, 81, 259, 545, 965, 743, 4533, 8813, 21253, 39245, 81485, 26089, 240417, 978223, 1178151, 890371, 5189125, 4202087, 7040829, 58722337, 83903643, 79971209, 354184633, 45487305, 1825464305},
  {1, 1, 5, 5, 1, 59, 37, 11, 105, 343, 75, 1319, 6317, 9593, 1699, 62493, 73205, 5593, 461481, 7421, 854311, 2927913, 2473807, 2211973, 9355635, 4047831, 122110611, 28988249, 409781661, 787677903, 394389233},
  {1, 3, 1, 9, 13, 9, 115, 131, 387, 1023, 253, 693, 5191, 12777, 10565, 1005, 70671, 213613, 192941, 432433, 1108741, 3700079, 5689839, 3280935, 21737675, 46027465, 78517809, 229508395, 282473909, 583168273, 1379298001},
  {1, 3, 1, 15, 7, 35, 111, 195, 287, 305, 533, 1901, 3363, 10085, 30791, 18557, 13863, 249469, 214059, 489339, 406327, 988523, 8238935, 5016923, 4339365, 18598161, 50560433, 99671967, 191590041, 550800003, 2040478033},
  {1, 1, 3, 9, 27, 51, 21, 77, 413, 925, 717, 791, 4147, 585, 5649, 51693, 90717, 128079, 135845, 101311, 411951, 398097, 5910329, 15370865, 13165153, 42754481, 27470891, 117974863, 71536349, 209579717, 625713137},
  {1, 3, 3, 5, 25, 59, 79, 249, 185, 567, 71, 1997, 7373, 2327, 18637, 1413, 38527, 34351, 203873, 83757, 635671, 541027, 3768253, 15952893, 25737611, 421643, 90974265, 259258441, 250704443, 497250929, 1803141201},
  {1, 3, 3, 11, 15, 21, 97, 99, 391, 57, 1973, 29, 7451, 2529, 25737, 4781, 35519, 222735, 166575, 525499, 1117441, 1858405, 7626543, 10253003, 29178461, 23190801, 5141985, 21479271, 438831069, 851989549, 692593233},
  {1, 3, 7, 5, 7, 59, 93, 5, 287, 469, 1639, 3637, 5465, 14431, 32265, 55429, 66159, 135179, 93537, 342435, 1024823, 433417, 4990769, 2024979, 3544777, 31440979, 133680441, 62542797, 361296211, 504272741, 1372867793},
  {1, 1, 3, 11, 3, 1, 71, 75, 427, 299, 811, 3697, 3529, 5433, 26957, 2589, 81037, 249071, 269159, 666087, 420565, 1973491, 5014775, 4670879, 1430815, 62461631, 20142757, 100682781, 371805573, 177028033, 605634193},
  {1, 3, 1, 9, 19, 59, 37, 255, 165, 1005, 19, 2851, 4309, 455, 9485, 26277, 107519, 82157, 78765, 152175, 388599, 3719297, 4417771, 3266825, 32876953, 35838559, 52869727, 259943633, 425315379, 910486561, 1507075249},
  {1, 1, 1, 5, 1, 55, 15, 233, 133, 47, 1831, 713, 2601, 1017, 3201, 37149, 111549, 244877, 8041, 375917, 1633739, 3253955, 8030429, 14092641, 31361555, 577115, 17244925, 40385197, 122401925, 612362829, 1114229265},
  {1, 1, 5, 5, 21, 55, 127, 69, 377, 41, 25, 2295, 7595, 4733, 11615, 14013, 114821, 168825, 502633, 383657, 1471083, 1330659, 6754385, 16773213, 2241445, 34559733, 33192763, 76169319, 486727817, 261387691, 1217673201},
  {1, 1, 5, 3, 23, 5, 7, 181, 161, 775, 1095, 2271, 6637, 14489, 6873, 16349, 55645, 225353, 333935, 142699, 2055617, 2665635, 4206201, 15134517, 8480587, 37581195, 32208363, 231679785, 274698557, 1039110069, 1371865457},
  {1, 3, 5, 9, 9, 15, 5, 133, 357, 21, 127, 2685, 6299, 4363, 17573, 49029, 24863, 169385, 436141, 702053, 670179, 336585, 3150353, 12730089, 14566857, 6784227, 63340769, 28674479, 201163367, 813356609, 1465528657},
  {1, 3, 3, 9, 13, 39, 51, 223, 201, 401, 1839, 2461, 7633, 6039, 10445, 53925, 116991, 116463, 146093, 659577, 1455099, 2890479, 5697035, 16603997, 22678293, 9256915, 126033585, 191632197, 315887099, 489441921, 1183586641},
  {1, 1, 5, 1, 9, 21, 19, 249, 227, 359, 255, 2895, 4117, 2073, 27687, 61469, 84261, 82873, 250477, 390837, 122457, 2422751, 7602821, 9207783, 25013243, 31923931, 50284651, 78398385, 279180157, 500792491, 1919371633},
  {1, 1, 5, 15, 5, 61, 113, 161, 95, 3, 877, 2775, 293, 6655, 4023, 197, 31957, 62009, 31395, 685369, 610633, 255997, 6297117, 5224019, 20488943, 59270265, 71410115, 136848489, 359291115, 824344763, 407875313},
  {1, 3, 7, 1, 7, 55, 73, 39, 295, 403, 985, 2315, 1667, 13525, 1453, 3869, 92175, 237035, 8549, 986595, 336363, 3387901, 1617251, 793819, 18026343, 10152669, 127689759, 78834527, 304016529, 876659417, 110364465},
  {1, 1, 5, 1, 27, 1, 85, 195, 11, 713, 1841, 3895, 3131, 2193, 17607, 62349, 37253, 16057, 493549, 266663, 1850701, 253137, 7326591, 12954039, 15219661, 33172069, 18684459, 236447767, 404918245, 227121907, 1044957937},
  {1, 3, 5, 13, 25, 1, 119, 97, 239, 167, 1393, 1753, 6989, 12155, 12509, 20213, 127519, 86121, 492841, 435317, 580861, 839131, 4054861, 8676187, 28164475, 53528101, 17942045, 62544337, 509137591, 40485249, 1839928849},
  {1, 1, 7, 15, 31, 21, 41, 255, 425, 445, 165, 2097, 5627, 4971, 13207, 6661, 57173, 24507, 178979, 460203, 778481, 3985485, 2445555, 6951301, 29143849, 43365529, 108164077, 83844903, 260598887, 861063187, 1424995217},
  {1, 1, 1, 15, 13, 33, 81, 105, 453, 197, 13, 1547, 7381, 8709, 15103, 47213, 61989, 19709, 139363, 746529, 303349, 471165, 2928869, 16502825, 21557641, 22786865, 133339479, 102364073, 375892129, 860528659, 1769584177},
  {1, 1, 3, 11, 11, 33, 107, 123, 483, 367, 121, 995, 1911, 8205, 22577, 15581, 9853, 126095, 247271, 169711, 219509, 678015, 4745351, 1812711, 31094907, 22198989, 45569127, 221053003, 265619177, 451897301, 578239409},
  {1, 1, 1, 9, 9, 43, 71, 49, 273, 431, 1705, 3313, 4259, 16291, 14345, 50917, 69869, 333, 31781, 393317, 1935207, 1275979, 7264437, 6309941, 7596083, 39173189, 30865637, 215574687, 171732815, 477320365, 276413329},
  {1, 1, 1, 7, 3, 1, 43, 213, 97, 547, 1559, 1149, 2791, 3751, 887, 41685, 109429, 148669, 310699, 561839, 210677, 2233983, 6850217, 390757, 21558159, 48559067, 42793177, 15846851, 315998347, 893191459, 960881489},
  {1, 1, 3, 15, 25, 47, 49, 251, 425, 35, 295, 3767, 6305, 9633, 5045, 15197, 20077, 238511, 178403, 668029, 880491, 3463501, 6016431, 7083061, 7940007, 24003835, 33801619, 103340805, 441306789, 446562217, 1000719345},
  {1, 3, 3, 1, 5, 55, 91, 245, 27, 981, 331, 555, 6553, 11017, 15289, 14077, 55583, 118735, 410085, 121585, 1193035, 2376039, 6656417, 2540847, 2295097, 62084655, 92460551, 96113533, 49452253, 370069477, 1061958449},
  {1, 1, 3, 7, 1, 23, 23, 155, 223, 565, 1005, 3211, 3847, 7479, 3643, 23605, 37509, 262111, 342955, 966181, 678131, 2933579, 3280879, 13746459, 19316737, 3753449, 59851399, 220022459, 244996691, 619704351, 1547238193},
  {1, 1, 5, 1, 17, 7, 47, 95, 35, 779, 1685, 2099, 7505, 15425, 18089, 12893, 85021, 15177, 314605, 615149, 1250379, 2759939, 3317035, 12552391, 3868783, 62672193, 128313383, 185460349, 111244421, 882924757, 1721765553},
  {1, 3, 3, 7, 3, 63, 83, 151, 211, 147, 611, 1171, 1681, 7687, 13423, 30085, 46775, 10111, 137251, 926135, 683995, 2413895, 1937875, 6531911, 2845943, 28943287, 46360807, 15690917, 154740211, 707751187, 1086848177},
  {1, 3, 3, 1, 3, 27, 107, 117, 497, 537, 195, 3075, 2753, 1665, 19399, 54685, 11303, 206399, 42213, 215351, 2071015, 2514767, 7549345, 14870373, 6050229, 8213007, 132757327, 182262885, 178208501, 520835771, 778941617},
  {1, 1, 1, 7, 23, 5, 103, 209, 117, 845, 1243, 1283, 4253, 9723, 20937, 8853, 86541, 56909, 353451, 360571, 1024625, 3274083, 7058205, 14885473, 19168017, 31993319, 77564735, 45910905, 275222223, 136485445, 140009137},
  {1, 3, 1, 1, 5, 49, 7, 13, 419, 125, 287, 1599, 8161, 1275, 24661, 64821, 97087, 131757, 59813, 580089, 1512797, 3951931, 2346753, 12299095, 5990801, 36011931, 70802803, 114972437, 45408479, 452719689, 1601263217},
  {1, 3, 3, 3, 13, 63, 23, 183, 39, 979, 1301, 2349, 905, 15805, 30151, 27535, 87837, 208441, 515185, 604399, 913889, 1074133, 8079133, 15011357, 2334697, 47037127, 17693859, 64977615, 425740323, 136537473, 629219469},
  {1, 1, 3, 9, 17, 11, 97, 189, 189, 511, 1779, 2077, 6891, 11623, 23949, 6939, 87607, 216173, 473299, 528823, 1721189, 3991211, 3665387, 9501547, 23140137, 14780577, 115063135, 2787413, 48668109, 118212119, 1067087669},
  {1, 1, 7, 11, 13, 45, 15, 37, 11, 853, 915, 1569, 6103, 10633, 3137, 45187, 84231, 240137, 48153, 843839, 1452115, 4065449, 6117751, 13945573, 28024479, 57996973, 92844923, 13666593, 187020235, 967566295, 509005613},
  {1, 3, 3, 5, 15, 61, 91, 255, 131, 821, 1755, 1501, 2663, 1747, 941, 3931, 59061, 124265, 484055, 429473, 729983, 1175205, 5858741, 15434729, 30412007, 8808213, 31581175, 15807881, 46912961, 1040277551, 1022318341},
  {1, 1, 3, 7, 19, 19, 65, 95, 499, 239, 2023, 3185, 4649, 3861, 3767, 45423, 1759, 168381, 200573, 262889, 1440193, 317379, 7564249, 14672449, 8173677, 59403069, 13993483, 53447231, 465932139, 536925281, 493239725},
  {1, 3, 5, 15, 15, 63, 55, 93, 127, 303, 171, 1763, 4991, 9479, 9917, 21371, 93429, 219119, 465, 198061, 1293669, 587005, 8378279, 14902577, 12602701, 22839345, 84707457, 110268541, 258117629, 530547527, 125190133},
  {1, 3, 7, 5, 31, 53, 111, 35, 433, 163, 1903, 3991, 3585, 643, 21941, 6507, 8405, 137133, 521567, 848033, 2042743, 1585281, 793377, 3742195, 8939845, 63650789, 129691229, 241643707, 510391797, 577334367, 1796986837},
  {1, 3, 1, 9, 27, 39, 67, 89, 487, 349, 587, 1723, 4311, 11321, 25785, 7027, 32997, 213963, 121247, 358693, 634989, 928409, 3838827, 1194289, 21121343, 53169717, 119476433, 223984501, 321643731, 679422751, 1367577421},
  {1, 3, 5, 7, 1, 63, 23, 237, 507, 689, 1341, 441, 1721, 843, 20335, 21471, 15549, 26751, 101881, 49907, 335065, 4035237, 1370791, 2108341, 18041051, 478091, 20352943, 136219367, 520073093, 541955429, 1888599573},
  {1, 1, 3, 3, 31, 63, 83, 103, 25, 799, 1379, 1817, 3809, 12285, 16673, 48451, 48007, 151565, 219929, 739117, 472933, 3795865, 6037077, 1398707, 31105913, 21472913, 32844603, 165326231, 370989523, 398066567, 319973197},
  {1, 1, 5, 3, 25, 29, 99, 193, 21, 549, 33, 3109, 4135, 10071, 32355, 58823, 18063, 114715, 112949, 200563, 1106427, 1502357, 1352427, 8473283, 32174243, 25686631, 8828395, 162914545, 116479413, 1025914889, 669406933},
  {1, 3, 1, 7, 13, 27, 83, 189, 121, 167, 379, 1503, 7955, 13189, 313, 45939, 95205, 21707, 235153, 675375, 33605, 2891737, 7432719, 10851175, 9078633, 27535529, 105558413, 80191241, 146058391, 376626455, 1348186717},
  {1, 3, 5, 15, 25, 19, 83, 87, 257, 237, 709, 1169, 1561, 7117, 4785, 34403, 117189, 199311, 56337, 484667, 1916773, 167833, 2921061, 4795083, 30944379, 48791859, 91917303, 77500151, 509382435, 481460727, 349387917},
  {1, 1, 1, 7, 9, 55, 21, 5, 439, 367, 403, 2311, 6243, 8349, 13127, 63887, 7711, 3391, 247673, 311291, 434881, 11927, 282667, 8884449, 28791213, 64893897, 97409429, 52662857, 350520079, 61968473, 1146686525},
  {1, 3, 7, 3, 5, 35, 51, 67, 453, 767, 29, 3293, 6665, 11459, 2799, 30943, 84669, 88189, 44025, 489719, 219853, 4172969, 3454993, 3479783, 5441585, 7589759, 46958883, 96109903, 359107933, 984206165, 303379829},
  {1, 3, 3, 3, 5, 19, 59, 7, 367, 683, 783, 1317, 7119, 6129, 19525, 58763, 67093, 206121, 356177, 330151, 1108573, 618369, 4588549, 11766885, 17648993, 7042765, 59441087, 206407089, 255232915, 1034992739, 1599286861},
  {1, 1, 5, 5, 5, 19, 61, 67, 381, 291, 875, 2179, 2481, 9325, 11253, 39659, 78551, 239787, 98387, 16803, 888917, 592791, 6001773, 14181543, 15410541, 50124977, 128329953, 257177971, 52027787, 282472931, 1168541229},
  {1, 3, 5, 5, 7, 47, 107, 9, 141, 667, 1989, 821, 3909, 1733, 10187, 45335, 55213, 234975, 293179, 571633, 1133389, 3857265, 3151211, 3626203, 14675117, 23042847, 49378699, 207137491, 437294051, 470281613, 1055461261},
  {1, 1, 7, 7, 31, 61, 1, 71, 477, 689, 1539, 3617, 8105, 6535, 3293, 33595, 56055, 10345, 242773, 915509, 1163223, 1905967, 2528153, 11059567, 3031279, 9159005, 59820571, 9126943, 227463737, 970600495, 548833509},
  {1, 1, 5, 5, 23, 9, 103, 197, 241, 249, 297, 3607, 6217, 1673, 30103, 53167, 128991, 38715, 297459, 564081, 810603, 1155889, 773415, 7526719, 24836871, 53627559, 43182041, 57554555, 112371455, 125941737, 1508727805},
  {1, 3, 1, 5, 23, 15, 115, 105, 365, 51, 825, 2687, 359, 16325, 15083, 30551, 86829, 186075, 523827, 285681, 1743213, 1385257, 1948091, 1857691, 29886597, 11993011, 75911577, 183025877, 479088367, 519468725, 824343485},
  {1, 1, 3, 11, 29, 45, 65, 251, 169, 189, 1243, 2345, 1345, 14471, 25631, 59071, 130559, 64637, 48753, 881279, 142675, 1752935, 7583669, 9016899, 24978619, 24765229, 101851891, 231252775, 226855753, 413267469, 858647845},
  {1, 1, 5, 9, 7, 63, 81, 167, 309, 539, 1169, 3949, 4193, 12047, 1491, 38695, 78287, 161051, 272063, 982009, 595533, 973643, 7305137, 16579543, 21389877, 51504507, 59058971, 148843223, 201435425, 643279121, 899769477},
  {1, 3, 1, 9, 29, 33, 89, 167, 67, 73, 1885, 477, 5745, 13365, 6819, 6855, 74509, 125595, 118207, 628195, 1525607, 3371287, 6444809, 2958033, 29717099, 17740299, 92298827, 169071463, 408185723, 310248149, 474410477},
  {1, 3, 7, 9, 9, 49, 95, 13, 157, 997, 1725, 935, 7543, 6349, 18277, 587, 75797, 215213, 378067, 820271, 462063, 3031297, 1386895, 1789699, 1880419, 63280123, 89527761, 11692885, 481592551, 334005035, 983258013},
  {1, 1, 5, 5, 11, 59, 97, 17, 303, 469, 93, 2761, 7395, 9021, 24299, 41815, 39727, 1499, 391731, 881901, 1899617, 235555, 1931479, 13756417, 14564183, 53423219, 39382959, 98346557, 277055431, 72532453, 1643044637},
  {1, 1, 7, 3, 27, 63, 71, 99, 407, 139, 711, 2589, 4715, 5405, 3277, 11547, 29879, 28137, 470609, 669753, 1337805, 179293, 7763033, 3980709, 17706049, 41686981, 11899927, 142369221, 375099671, 891411299, 1224639581},
  {1, 3, 7, 3, 11, 15, 49, 57, 271, 493, 1165, 2839, 8191, 2609, 14759, 63951, 124445, 203709, 112121, 556921, 464253, 3624011, 6646831, 11564241, 30912463, 56520339, 70733745, 127060765, 165443723, 733694001, 826384941},
  {1, 1, 1, 7, 21, 15, 71, 245, 413, 473, 1321, 1165, 1027, 6983, 12867, 4359, 31119, 115871, 103609, 440295, 1807681, 2329157, 5675775, 6075107, 27683487, 17223063, 895779, 211756997, 523417933, 485013593, 283518261},
  {1, 1, 5, 3, 15, 21, 19, 197, 401, 627, 2047, 2761, 5807, 5751, 28025, 16755, 118887, 187979, 225557, 217765, 1913823, 1655341, 5167647, 9049903, 24836269, 11208997, 92395779, 258634793, 534678949, 548264627, 1873213077},
  {1, 1, 3, 3, 5, 57, 19, 209, 341, 165, 489, 455, 231, 14385, 12457, 58579, 46119, 121805, 298777, 183, 1525975, 3662397, 7343659, 10946435, 20533283, 33928751, 130415897, 144890965, 139722523, 695668767, 154352749},
  {1, 3, 3, 11, 13, 63, 79, 129, 17, 315, 1881, 1069, 177, 12013, 29567, 11647, 51325, 246137, 122745, 865663, 324417, 688717, 5012123, 2203303, 20094965, 48175923, 29332171, 84200615, 413821475, 188168633, 278784141},
  {1, 1, 3, 7, 31, 29, 51, 235, 475, 375, 617, 437, 6379, 8505, 23079, 12495, 101151, 65469, 384765, 9061, 831319, 1566237, 3252817, 1925113, 29682949, 41314787, 128928755, 89448333, 352086435, 748107489, 556623501},
  {1, 1, 3, 7, 27, 3, 3, 137, 203, 959, 363, 371, 2899, 13491, 22979, 48199, 82511, 14429, 83069, 148513, 153729, 3358913, 357067, 4985373, 20502725, 54483105, 13070449, 63912945, 185831825, 25927073, 336172613},
  {1, 3, 3, 3, 9, 1, 57, 7, 363, 537, 713, 2417, 509, 7747, 22135, 39599, 38685, 189049, 502705, 800683, 1819287, 2899671, 5417705, 4152057, 8850347, 42159967, 36291087, 101204003, 518681669, 924717245, 2014494997},
  {1, 3, 3, 3, 13, 21, 79, 121, 487, 921, 113, 281, 2853, 14855, 19747, 13319, 130125, 75481, 208113, 901167, 155787, 1009817, 6253067, 11905969, 25407675, 50598599, 37035607, 182802891, 65889105, 667815873, 12229957},
  {1, 1, 1, 11, 3, 53, 89, 123, 307, 585, 567, 1925, 505, 15935, 20419, 51655, 109135, 59871, 250357, 792617, 380071, 625879, 2735637, 3589849, 7163579, 37542145, 56291399, 48838839, 477728273, 420991393, 2055825477},
  {1, 1, 3, 3, 15, 45, 77, 197, 499, 683, 1405, 3573, 981, 14135, 19763, 31399, 12303, 150365, 116217, 445501, 1322647, 200355, 278595, 8994869, 21021313, 48611647, 10076235, 228411779, 8216305, 249193361, 603909061},
  {1, 1, 1, 11, 27, 31, 61, 191, 29, 601, 373, 2011, 6193, 3599, 4387, 46599, 87887, 92767, 258805, 803889, 1695421, 3724935, 1185905, 12405055, 21871463, 1138171, 52295021, 191701291, 272738841, 25409745, 900593765},
  {1, 3, 5, 9, 7, 13, 1, 193, 469, 603, 1315, 3329, 3761, 8355, 10425, 57779, 64933, 17679, 283735, 801385, 459935, 3814399, 146559, 16536811, 4893773, 25031161, 72176435, 115963159, 468945741, 639561443, 58678581},
  {1, 1, 3, 9, 29, 61, 103, 17, 117, 251, 2029, 2963, 3763, 16117, 6627, 58375, 65103, 166749, 208755, 936123, 676491, 1051057, 2927539, 15314771, 8902045, 12588823, 55344253, 37333257, 30380839, 348295789, 376077469},
  {1, 3, 1, 3, 7, 51, 91, 145, 497, 657, 871, 3707, 5905, 10449, 14901, 13611, 13973, 98027, 430741, 992877, 476329, 2377417, 1573187, 12181175, 480191, 34558041, 85171593, 64452419, 54506391, 425305563, 1439866461},
  {1, 1, 3, 1, 3, 53, 23, 149, 461, 333, 1809, 1315, 1815, 8223, 13297, 6051, 9223, 192461, 375131, 498805, 1998239, 2370281, 5309319, 12029059, 28080011, 34553415, 30099309, 81348749, 197578181, 378341307, 1751353109},
  {1, 1, 1, 7, 15, 31, 3, 47, 443, 829, 1305, 893, 4191, 9681, 32661, 45419, 108951, 216943, 49689, 620413, 1283749, 2351241, 2345981, 1148033, 16946559, 23226791, 118167155, 261782009, 312057443, 495056915, 1768876429},
  {1, 3, 1, 3, 27, 43, 51, 221, 295, 825, 649, 2953, 6203, 8237, 20253, 16251, 13749, 193067, 99733, 763761, 1469577, 3428577, 6262623, 2168921, 12102091, 29829215, 43554423, 252867477, 245077959, 242338259, 1083576093},
  {1, 3, 1, 3, 9, 35, 41, 195, 249, 225, 387, 3789, 1499, 2559, 28413, 6459, 107061, 46763, 458005, 649315, 1904677, 2068515, 3846213, 16076851, 8269895, 20542285, 111431831, 113852085, 134870101, 123807767, 1736163669},
  {1, 1, 5, 15, 19, 29, 13, 115, 333, 787, 787, 723, 2987, 6227, 10865, 59043, 53255, 109899, 272857, 295649, 1802271, 360563, 2523349, 10977703, 33194909, 28868841, 118785089, 209223785, 241992769, 865464755, 1311654149},
  {1, 3, 5, 13, 5, 59, 5, 251, 79, 387, 11, 3167, 6619, 13317, 18979, 7943, 51533, 216287, 400627, 653603, 1684541, 2860095, 4199493, 12668293, 29435337, 28346073, 41585453, 173834945, 285184839, 712672109, 1154628893},
  {1, 1, 7, 11, 31, 51, 43, 1, 189, 519, 1945, 2129, 4365, 14059, 3139, 9159, 99919, 55385, 161657, 448173, 648233, 2278457, 701491, 10169627, 22580977, 5347771, 76002279, 19783683, 434264749, 601451097, 1623014069},
  {1, 1, 7, 5, 31, 9, 43, 19, 151, 533, 1061, 3849, 6871, 6941, 14935, 13423, 124191, 216313, 490167, 68529, 235051, 14909, 4429353, 5171973, 28824415, 176987, 84759039, 115643881, 14134799, 599063945, 254675005},
  {1, 3, 7, 5, 19, 57, 7, 129, 25, 353, 17, 1739, 6513, 399, 28835, 13191, 92621, 144349, 107647, 43437, 520739, 3993089, 4782659, 12278623, 21946527, 43100143, 54748501, 228713499, 229932025, 76111633, 576806885},
  {1, 3, 5, 15, 25, 15, 37, 125, 39, 239, 271, 65, 2189, 10449, 11815, 33295, 56925, 135551, 58161, 169659, 1700409, 453143, 3560083, 7036481, 3001773, 62695957, 120950187, 250451099, 32742567, 750047401, 1286480541},
  {1, 3, 7, 15, 19, 57, 47, 245, 509, 945, 385, 3987, 3585, 14711, 9655, 15279, 70045, 4477, 5429, 825657, 2035211, 2534009, 2578119, 935859, 8619287, 49258863, 94397709, 215181787, 253585793, 518087669, 370521605},
  {1, 1, 3, 13, 21, 31, 13, 81, 9, 489, 1321, 63, 1363, 2219, 19541, 30571, 71959, 229485, 41623, 833915, 500009, 3070239, 4642607, 15242647, 10382983, 56156263, 15523889, 92614001, 489881353, 906108679, 82988069},
  {1, 1, 5, 7, 3, 57, 25, 147, 23, 553, 889, 307, 6429, 15807, 12861, 57019, 22711, 225323, 331665, 951649, 1883963, 3893551, 5232525, 15424845, 12785267, 6999135, 113229517, 263232455, 29732337, 336636703, 2134861765},
  {1, 1, 3, 15, 29, 21, 99, 237, 151, 881, 675, 3625, 1159, 11759, 21347, 9223, 120655, 158813, 276213, 521399, 825659, 1377621, 6119687, 3620185, 14159107, 29268709, 19444643, 41808897, 165244333, 836899065, 813311669},
  {1, 1, 7, 1, 9, 13, 111, 239, 235, 609, 1569, 3271, 2837, 13807, 7301, 32715, 73815, 117993, 352787, 58991, 1727251, 3081481, 4234173, 12026193, 26903427, 10651239, 123046985, 242516807, 328570785, 815594503, 636989061},
  {1, 3, 1, 15, 7, 59, 27, 81, 129, 9, 647, 3595, 1877, 1067, 1859, 19271, 12493, 3035, 356985, 876981, 1254673, 3431289, 4707171, 12769031, 18792135, 27103177, 41082969, 95221287, 63657765, 296903241, 36605077},
  {1, 3, 7, 1, 3, 25, 119, 57, 145, 441, 1045, 789, 215, 1265, 9369, 40179, 106277, 151181, 16731, 773749, 1780531, 4189521, 4230491, 4058855, 10595127, 39627195, 76231459, 140090449, 428521235, 266010111, 1399008333},
  {1, 3, 7, 3, 17, 25, 87, 211, 441, 229, 223, 2795, 7241, 7007, 20575, 52351, 110269, 107325, 344249, 238371, 830751, 2898729, 5550913, 4709371, 10848467, 36023865, 33843161, 146891579, 377902313, 429526221, 1781957541},
  {1, 1, 3, 1, 13, 1, 55, 227, 389, 141, 1097, 2487, 7603, 4161, 5025, 55171, 7367, 127949, 516059, 270203, 1824567, 515353, 5221777, 6375271, 1606403, 4621839, 23691497, 60002465, 121015491, 980346983, 1705722637},
  {1, 1, 3, 5, 15, 29, 29, 145, 233, 209, 891, 89, 8097, 2897, 26685, 24347, 38167, 179725, 341567, 216017, 1168607, 2272067, 4208567, 4016231, 7359039, 9678053, 70554099, 45673127, 210973439, 831728163, 153073661},
  {1, 1, 3, 1, 29, 53, 19, 95, 161, 359, 435, 3313, 4955, 7965, 21015, 50767, 57023, 207389, 504731, 932299, 1635907, 1081205, 2576661, 9709035, 18132945, 49752689, 48482995, 69750941, 194309959, 609278649, 381333789},
  {1, 3, 5, 9, 19, 3, 109, 77, 29, 937, 1663, 125, 2453, 1069, 20639, 47839, 5661, 230047, 384151, 435933, 287321, 1952639, 5224787, 10239067, 12907711, 38461473, 16376799, 58699851, 137300955, 977755337, 1099254637},
  {1, 3, 7, 13, 5, 23, 43, 231, 347, 591, 1963, 2491, 4045, 16029, 8149, 15563, 90357, 126541, 373047, 388811, 168673, 3244361, 5198293, 14421521, 8810765, 64861073, 44744121, 208027679, 397560451, 373918067, 1337372173},
  {1, 1, 5, 1, 13, 3, 75, 211, 419, 929, 901, 3453, 8121, 799, 8897, 29475, 131047, 114411, 207927, 948803, 72421, 2747233, 5973361, 7888801, 18080387, 56058915, 59944731, 36762031, 23011937, 884018275, 619912581},
  {1, 1, 7, 15, 11, 11, 123, 111, 309, 415, 1071, 975, 2009, 12945, 19617, 25251, 10919, 35945, 302333, 1018833, 1955273, 599305, 7556445, 9887279, 5061089, 2554061, 32336245, 86600851, 123933783, 726077999, 1610577245},
  {1, 1, 1, 7, 31, 35, 81, 255, 89, 643, 451, 513, 497, 11751, 24215, 15183, 89279, 88415, 416857, 815181, 210777, 1199875, 7745305, 12146507, 10360757, 26154889, 129949555, 193118775, 306900713, 526558181, 587390885},
  {1, 3, 5, 5, 25, 17, 5, 165, 139, 929, 1927, 1353, 7427, 9719, 17087, 33695, 51229, 76383, 416859, 478543, 261999, 2365307, 2498483, 10288781, 6012251, 58703569, 86259083, 238367349, 524626621, 925718181, 2102728437},
  {1, 3, 5, 1, 21, 55, 79, 85, 333, 847, 1305, 851, 5057, 8361, 18269, 37019, 97877, 226191, 33599, 497355, 1886145, 2942061, 1293423, 5033331, 15834569, 23781803, 102197925, 66578539, 145631007, 538600771, 1438292093},
  {1, 3, 7, 15, 27, 17, 55, 125, 395, 223, 271, 781, 1639, 10569, 11143, 33775, 30077, 225565, 80917, 44497, 256339, 1055793, 2794687, 2190677, 21218645, 46871749, 87593383, 135027657, 87818107, 930469585, 957198829},
  {1, 1, 7, 9, 7, 33, 127, 85, 209, 339, 483, 241, 2523, 14951, 6855, 47343, 123647, 104857, 424155, 422993, 1016419, 4151385, 7391431, 2824511, 27720205, 883865, 20362723, 198389757, 243936637, 1048476829, 1557543413},
  {1, 1, 3, 9, 5, 19, 9, 183, 435, 343, 1105, 3139, 7617, 1311, 267, 64311, 4367, 211261, 414867, 1011523, 1746293, 4116483, 1257489, 15454449, 3158997, 23663707, 87233429, 75880395, 333759913, 745523193, 1776140965},
  {1, 1, 5, 1, 15, 53, 11, 63, 113, 241, 855, 3123, 4777, 3495, 23345, 54659, 33383, 8939, 353335, 906049, 1489367, 1993285, 7952085, 7318075, 28491175, 57329657, 127158881, 86758347, 383670353, 354866291, 1812675909},
  {1, 3, 1, 5, 19, 29, 119, 205, 167, 683, 289, 1629, 4977, 8981, 6867, 23175, 113581, 160891, 312659, 18389, 1842263, 418425, 895583, 6629241, 8228345, 40032563, 1144347, 181223335, 393094779, 608581125, 673105389},
  {1, 3, 1, 1, 31, 63, 95, 159, 267, 231, 863, 3385, 5315, 7267, 13757, 36251, 115989, 175947, 188407, 172881, 1034493, 2097669, 7547637, 4527873, 20898757, 61875829, 50504819, 6381957, 379464777, 749887119, 958022949},
  {1, 3, 5, 11, 19, 21, 53, 41, 125, 179, 533, 1279, 3759, 7073, 13905, 37123, 89829, 130351, 364917, 461273, 1834695, 539163, 415487, 2731555, 7977173, 59398751, 9552929, 243682829, 205933275, 472551495, 421476205},
  {1, 3, 5, 9, 17, 7, 27, 67, 97, 809, 1423, 2743, 2859, 16121, 329, 8819, 86277, 223343, 57079, 98783, 1920217, 1917449, 5650097, 14008035, 7297551, 14580657, 1567221, 98139105, 419571291, 207250207, 862911853},
  {1, 3, 1, 15, 1, 41, 59, 155, 509, 51, 1827, 3739, 3879, 13369, 30821, 23083, 44725, 194571, 19129, 752211, 146799, 1437381, 1653233, 7835583, 31416493, 11069209, 89334153, 127972757, 94984819, 958058915, 1593677261},
  {1, 3, 3, 7, 21, 31, 7, 13, 347, 919, 1225, 497, 5051, 3769, 20211, 57671, 110381, 21945, 101589, 238175, 960705, 91557, 210919, 4790965, 21809565, 32415059, 67450999, 37910325, 348966131, 526491445, 1083930573},
  {1, 3, 7, 13, 31, 9, 127, 195, 123, 387, 3, 3593, 6623, 9827, 29319, 36079, 128381, 27805, 355799, 30929, 442827, 2776009, 3620721, 1918713, 22303953, 2873153, 6427947, 119020585, 207096209, 173553413, 1678264901},
  {1, 1, 3, 9, 7, 27, 95, 211, 287, 189, 1683, 1999, 7641, 14983, 4699, 44375, 55759, 95549, 14611, 350657, 1081081, 3591629, 3476329, 9078317, 13073991, 49522877, 74976149, 194516819, 185078529, 167269433, 894986245},
  {1, 1, 5, 3, 7, 21, 29, 189, 101, 423, 885, 3275, 6569, 11023, 22265, 31699, 14151, 129835, 257077, 1010125, 1140847, 2680483, 7114523, 15530355, 24635745, 38811863, 116656517, 75236419, 154416745, 483161483, 1459922341},
  {1, 3, 5, 3, 9, 33, 79, 75, 327, 975, 287, 3025, 2157, 7301, 24447, 39071, 25245, 123167, 478237, 79699, 120295, 3265953, 1176937, 15458485, 32645445, 32875509, 4761499, 89652219, 501998419, 590058489, 1194021197},
  {1, 3, 3, 15, 31, 27, 63, 1, 71, 119, 1151, 517, 6131, 11055, 179, 42759, 95021, 79289, 202269, 965061, 518897, 3310589, 518491, 6989265, 32912885, 35914037, 62653263, 117183629, 1839253, 1051987129, 1995228757},
  {1, 3, 7, 11, 23, 15, 101, 247, 349, 735, 673, 997, 6451, 229, 32103, 32367, 73213, 95005, 502289, 804693, 2021317, 1518031, 1358153, 15916399, 19269153, 53760195, 32012035, 119861909, 127269295, 970451097, 518763197},
  {1, 3, 5, 15, 7, 1, 51, 135, 207, 741, 1831, 1235, 4747, 11915, 22009, 57235, 92357, 222831, 189233, 597957, 29547, 1989061, 6012525, 11164597, 16158639, 42971801, 126786241, 166734377, 138237433, 565676139, 1576057829},
  {1, 3, 1, 13, 9, 31, 19, 221, 465, 681, 627, 2595, 5617, 14201, 30355, 20167, 62573, 159355, 411995, 164703, 101313, 1467057, 6034015, 690063, 11185879, 56408445, 101651065, 28515683, 312552927, 875118925, 811856765},
  {1, 1, 3, 1, 13, 49, 55, 155, 11, 885, 1275, 3591, 2217, 6659, 30885, 46251, 82999, 96717, 190331, 777691, 1043047, 3317657, 512649, 2651897, 9146471, 5190741, 64446221, 181386003, 11039877, 846310607, 454069781},
  {1, 1, 7, 11, 27, 57, 93, 95, 243, 63, 1405, 2049, 7689, 15943, 18503, 58799, 121855, 9049, 296921, 239305, 335563, 1673899, 2523817, 11188737, 24648349, 22616231, 118986159, 85004535, 144256265, 51843701, 1404524581},
  {1, 1, 7, 7, 5, 11, 47, 189, 467, 631, 1665, 2717, 4285, 2087, 1435, 60311, 59407, 154681, 407317, 932559, 931317, 3677349, 5083735, 6194013, 8600237, 55761659, 3849547, 104491339, 113502865, 363528089, 1418772037},
  {1, 1, 3, 11, 7, 27, 127, 3, 231, 757, 435, 2545, 3537, 9127, 19915, 39287, 124239, 112701, 72465, 191173, 511473, 1105149, 1233625, 15381493, 7211071, 4066925, 108260619, 139750759, 69989449, 952295225, 183555365},
  {1, 1, 5, 13, 5, 29, 85, 127, 339, 875, 497, 1573, 6553, 11983, 18029, 22139, 43223, 85771, 113275, 47827, 1263195, 2835843, 6798489, 9766273, 8158857, 13555459, 32781507, 182518719, 204811537, 325287151, 1915820101},
  {1, 3, 1, 1, 21, 3, 15, 91, 231, 683, 1529, 2651, 4147, 13437, 23861, 45003, 130165, 113419, 235319, 42075, 1187413, 2453381, 2967729, 12855525, 12638849, 55475419, 93129677, 71242249, 448733519, 220816395, 1358008637},
  {1, 3, 1, 7, 27, 17, 19, 179, 243, 223, 1037, 1501, 5935, 2259, 25185, 64995, 26341, 60331, 134321, 911833, 1685315, 633781, 6412345, 3737889, 17550141, 49274775, 4678611, 78121, 209932577, 165657603, 476317829},
  {1, 1, 3, 15, 11, 19, 127, 27, 483, 219, 583, 2555, 531, 3451, 17875, 54215, 103535, 200765, 378773, 585153, 520177, 2974461, 8061121, 14614593, 18285561, 14616085, 37945841, 253695569, 256313465, 264194433, 1167533541},
  {1, 1, 1, 13, 31, 39, 89, 149, 363, 741, 1355, 4067, 3171, 6783, 1799, 64047, 100095, 91487, 334227, 196633, 589877, 2424851, 196755, 4391405, 1901279, 11469845, 25496809, 96012049, 295778621, 393030109, 2104295669},
  {1, 1, 3, 11, 25, 51, 45, 235, 379, 123, 1701, 725, 1991, 7471, 9833, 15539, 113799, 221869, 493425, 582171, 1999397, 3397191, 2385413, 4553233, 6892921, 32108599, 100899099, 207496849, 30360941, 105229779, 1046385077},
  {1, 1, 5, 13, 15, 47, 13, 201, 263, 57, 375, 2963, 7475, 15929, 13775, 43327, 6111, 101147, 206811, 142873, 1926717, 2434647, 3846855, 7528233, 14073227, 61898113, 118225713, 140568689, 120508475, 824757401, 777127149},
  {1, 1, 3, 1, 29, 29, 11, 161, 345, 253, 97, 255, 7267, 2379, 3933, 40923, 116247, 6541, 388283, 558603, 1608235, 3607165, 8350267, 14845711, 17617731, 37868359, 83473681, 100473953, 366810281, 5072815, 1728794277},
  {1, 3, 1, 15, 3, 47, 11, 69, 347, 747, 795, 2401, 3367, 2383, 6125, 58491, 105493, 253579, 339769, 872977, 1377837, 2712113, 6093319, 4624053, 31984097, 35364665, 120389747, 37164897, 323292909, 73493847, 1410539445},
  {1, 1, 7, 3, 1, 49, 101, 47, 71, 761, 1503, 2619, 191, 8895, 873, 46451, 100423, 132009, 145777, 827907, 459287, 535147, 7215641, 1185301, 3930355, 44387353, 120167593, 179056141, 83574853, 113615555, 1203653909},
  {1, 3, 3, 5, 25, 41, 93, 85, 427, 109, 1675, 2409, 4317, 9233, 30283, 951, 86925, 261113, 391575, 878359, 272167, 963443, 6676947, 35245, 3451159, 17497221, 72782019, 88215803, 220074887, 1010703333, 1144431133},
  {1, 1, 3, 9, 11, 3, 67, 159, 425, 751, 887, 1415, 403, 15977, 10739, 57799, 98223, 130045, 389395, 305421, 744761, 1543953, 3301853, 4775987, 15655145, 29385669, 70460781, 250634665, 362216731, 531884837, 999104621},
  {1, 1, 5, 13, 9, 1, 9, 103, 481, 601, 931, 1957, 5763, 7095, 27141, 17707, 823, 76363, 325051, 931615, 693535, 3798295, 1546217, 13397139, 6554975, 61487677, 115663359, 36089501, 370109869, 608216447, 1946647221},
  {1, 1, 3, 15, 29, 13, 43, 33, 297, 269, 1041, 1411, 3461, 12043, 10045, 539, 68887, 215885, 302005, 163095, 839427, 2385165, 935771, 4425087, 22622035, 20317719, 98437581, 45489471, 220341077, 266763575, 157471061},
  {1, 3, 5, 3, 3, 3, 5, 7, 185, 753, 133, 1561, 5595, 13777, 25795, 59495, 1453, 144767, 213277, 130329, 2076433, 1643847, 7933737, 6947723, 9831047, 14288107, 22675567, 104989413, 423241787, 773743509, 1575630061},
  {1, 3, 5, 5, 1, 19, 29, 145, 163, 149, 619, 2603, 7757, 10035, 10189, 27835, 59349, 180175, 388027, 1048343, 876829, 3678023, 358847, 194173, 19620687, 50571157, 78147489, 29599279, 132470797, 485817463, 1045668917},
  {1, 3, 7, 15, 27, 15, 111, 173, 135, 117, 157, 2601, 7919, 12111, 22795, 30711, 18573, 206269, 484949, 95505, 90893, 2568981, 3542887, 2818729, 17008967, 10070259, 132255023, 157425169, 433731725, 576073137, 1006011957},
  {1, 3, 1, 1, 29, 27, 65, 31, 101, 715, 289, 3643, 2335, 6789, 23397, 16683, 629, 124107, 224759, 209683, 173853, 2698523, 3792137, 2880343, 15940805, 10617547, 114166413, 34216677, 185479407, 299248347, 2087897021},
  {1, 3, 1, 3, 11, 45, 71, 109, 321, 423, 1695, 169, 3075, 12423, 11391, 607, 42461, 257563, 120981, 188033, 1174159, 1640121, 87743, 5871327, 1938681, 20468813, 118611027, 178113157, 25414957, 13861189, 626727093},
  {1, 1, 3, 9, 13, 51, 35, 121, 203, 279, 433, 2725, 7951, 2105, 27333, 11755, 49847, 219981, 198387, 632459, 323461, 1296265, 2204811, 10071421, 26627789, 16331211, 53133011, 158747913, 392783307, 89305139, 1881502509},
  {1, 1, 1, 15, 23, 31, 25, 105, 501, 441, 1511, 3133, 2811, 10595, 21779, 30727, 67823, 57471, 419537, 15509, 781973, 3118739, 6604015, 8229963, 14489167, 52903961, 1492911, 80796677, 195872841, 563723041, 720610597},
  {1, 1, 5, 13, 7, 1, 97, 193, 121, 993, 1347, 1903, 1883, 6583, 24535, 19727, 47551, 212059, 24859, 512657, 1642627, 3099847, 2515055, 6037639, 10559887, 42771965, 50081333, 27993233, 329256501, 849370653, 1503675605},
  {1, 1, 7, 9, 7, 29, 17, 41, 101, 447, 1289, 387, 1891, 2723, 26091, 14007, 114127, 136377, 374107, 357777, 1203615, 1950095, 2154327, 9159531, 10442761, 65510971, 12939349, 59684969, 214377041, 456070697, 1676117317},
  {1, 1, 3, 3, 3, 53, 81, 81, 177, 165, 195, 3413, 8177, 3817, 8453, 62123, 74935, 188045, 364025, 545937, 1368247, 605855, 3025295, 7980879, 9513947, 10439765, 25703007, 148066287, 280255383, 976215563, 551871069},
  {1, 3, 7, 15, 15, 31, 23, 31, 337, 439, 1773, 63, 5351, 5491, 1767, 25647, 32893, 241757, 76693, 389, 1662901, 538013, 239941, 2271675, 16730785, 59320783, 53387457, 99823365, 421469401, 73076789, 1457502053},
  {1, 3, 1, 11, 5, 15, 23, 75, 437, 553, 429, 2705, 3625, 13851, 19865, 54099, 28741, 19563, 138877, 56991, 1236881, 3239637, 6696769, 16310743, 16802727, 1336451, 113254903, 3208487, 490802037, 261293523, 1609693653},
  {1, 3, 3, 9, 13, 15, 33, 235, 215, 415, 1737, 1409, 2101, 14623, 14717, 46747, 18901, 104649, 237755, 1009819, 1551769, 3929331, 4609005, 7074221, 798949, 28447843, 3832295, 113689371, 527073097, 666472335, 1980262693},
  {1, 3, 7, 7, 13, 51, 101, 217, 175, 813, 1639, 4009, 1625, 4991, 17525, 4043, 103669, 205837, 248893, 552599, 1235389, 251135, 948039, 353601, 327055, 4633241, 107223739, 16542607, 527510785, 822562135, 201183237},
  {1, 1, 5, 13, 23, 33, 29, 175, 39, 673, 557, 3239, 5129, 11049, 27227, 57031, 81183, 67755, 521451, 116145, 498355, 3866571, 5566857, 948261, 29268443, 49122603, 63295073, 128109211, 464413615, 454574005, 709363645},
  {1, 3, 7, 13, 1, 37, 33, 139, 493, 891, 1883, 2525, 5741, 15795, 5875, 10295, 20157, 56909, 23463, 797759, 228235, 3423207, 6837557, 1347703, 16683813, 587617, 131071055, 175780227, 39949541, 167895337, 194209429},
  {1, 3, 1, 15, 15, 27, 127, 111, 147, 363, 725, 3077, 4341, 9131, 24635, 32295, 44317, 181899, 448681, 555309, 1026449, 2970093, 4718373, 8493577, 28336745, 25518807, 83537403, 69679507, 309308445, 331855729, 227906933},
  {1, 1, 7, 3, 17, 25, 59, 135, 177, 635, 73, 3455, 3083, 6009, 13033, 37923, 71415, 133241, 186465, 782563, 721071, 164085, 918909, 4228731, 21729917, 4240867, 51377001, 247324521, 376756347, 634286639, 1492786413},
  {1, 1, 1, 5, 15, 53, 93, 161, 215, 459, 1087, 179, 2235, 8885, 15309, 3755, 60647, 246943, 39915, 289641, 1947287, 3619459, 3729687, 9483881, 8743529, 57781981, 118266489, 169960665, 441759639, 60333875, 2006436189},
  {1, 1, 7, 13, 7, 17, 75, 173, 449, 855, 103, 2739, 3421, 11811, 18805, 35803, 37991, 105721, 173231, 689897, 1413555, 2046653, 4287783, 15512351, 5241945, 66310405, 33879241, 108547903, 301661133, 734399679, 1770178613},
  {1, 1, 7, 9, 5, 11, 53, 75, 247, 249, 1201, 953, 2455, 4589, 6027, 17031, 36479, 258089, 222955, 938915, 722621, 2217439, 2020165, 15740493, 24044979, 59777063, 61925799, 26065929, 81307855, 725066341, 368961085},
  {1, 1, 5, 13, 27, 51, 119, 39, 137, 11, 1435, 3773, 3889, 6081, 11829, 26811, 4967, 199131, 46059, 948605, 1035705, 4112309, 4160765, 8371619, 24965085, 4321905, 86292111, 17751327, 78106927, 587605627, 227981757},
  {1, 1, 5, 5, 5, 35, 1, 197, 501, 185, 1039, 1563, 6421, 14373, 25655, 47327, 31215, 50315, 313699, 505907, 1212341, 1914587, 7540803, 1544515, 3043951, 36614753, 30282217, 235002359, 353197115, 469902673, 732050413},
  {1, 1, 3, 13, 31, 55, 115, 183, 483, 655, 1351, 3203, 725, 3299, 22579, 31063, 74495, 78541, 397223, 620321, 2007685, 3559593, 4349589, 14897113, 9843821, 55829517, 86679753, 133571031, 513654429, 1009096409, 1481600885},
  {1, 3, 5, 11, 31, 31, 83, 59, 395, 21, 1881, 2821, 2251, 11781, 26265, 17507, 44405, 241087, 513509, 505573, 939909, 1563561, 810553, 3829417, 20216343, 24295655, 41610843, 203810285, 394430383, 902617271, 1018243005},
  {1, 3, 7, 13, 21, 19, 103, 21, 403, 443, 1951, 55, 985, 15983, 15087, 6575, 25549, 24685, 103207, 790955, 2014357, 1488541, 1032799, 10447517, 24992465, 44996845, 101045533, 136530227, 491283993, 471337741, 1968752485},
  {1, 1, 5, 15, 29, 11, 51, 53, 255, 183, 1475, 1491, 259, 387, 10303, 2639, 128431, 108139, 370217, 649663, 1345669, 652937, 2601655, 3328865, 31578917, 64240153, 78098833, 78691633, 525442737, 466588621, 1110953413},
  {1, 3, 5, 7, 21, 37, 45, 39, 479, 637, 1325, 3753, 3319, 7403, 31759, 32015, 35821, 53231, 166825, 932919, 537371, 3415915, 7177185, 9614317, 16720791, 23912499, 72051551, 65005849, 473416233, 644320237, 2141694373},
  {1, 1, 3, 5, 7, 43, 89, 53, 269, 187, 995, 141, 119, 8139, 29699, 54871, 108127, 37261, 284079, 918953, 686857, 2195227, 6250355, 8387323, 6017885, 8999785, 119166447, 254283929, 68687529, 685270593, 598272421},
  {1, 1, 1, 5, 1, 53, 3, 23, 379, 223, 1889, 4035, 1437, 12425, 9051, 44807, 57439, 114863, 183659, 433191, 385291, 1281381, 3291117, 52049, 11303245, 52391467, 47916741, 116220039, 226152407, 701862085, 621482589},
  {1, 3, 1, 13, 3, 31, 61, 43, 249, 449, 901, 1921, 3495, 8599, 5263, 34191, 104941, 217131, 196651, 791333, 1395461, 2001671, 5794581, 13916675, 7550999, 61210203, 87814375, 86505177, 75491029, 730840597, 1890569813},
  {1, 1, 3, 5, 3, 25, 35, 133, 25, 597, 915, 3663, 5147, 11831, 24269, 61931, 83175, 227133, 393519, 492909, 414515, 1920789, 3593695, 9868023, 4652555, 63206741, 1430421, 104313281, 504060805, 372418327, 1458663701},
  {1, 1, 1, 9, 21, 27, 93, 93, 217, 299, 1881, 3647, 4825, 7989, 24121, 20227, 98839, 73727, 194535, 233323, 905533, 2198871, 6000899, 12418031, 32042661, 64037091, 120104521, 177025595, 121001619, 778608815, 1331534157},
  {1, 3, 1, 15, 5, 15, 49, 129, 315, 631, 2037, 1567, 4043, 15589, 30905, 39171, 117045, 43707, 177513, 629607, 1719057, 4052259, 5450199, 13316709, 22988333, 39921951, 98449105, 18090169, 417080007, 832836423, 864295965},
  {1, 3, 3, 7, 25, 5, 123, 51, 47, 471, 1563, 3947, 7975, 3681, 9611, 22055, 119229, 15593, 119013, 906275, 882451, 3289421, 8199033, 11417197, 32533693, 57466161, 61194009, 232219205, 528669627, 870955373, 152000493},
  {1, 3, 7, 15, 1, 17, 73, 245, 465, 95, 95, 1159, 1319, 4675, 8841, 6627, 111157, 173021, 430245, 674043, 182071, 3285879, 3811787, 14599031, 11671361, 5766957, 17565677, 236194317, 467936713, 251931211, 830741541},
  {1, 1, 3, 15, 5, 51, 35, 71, 423, 651, 753, 173, 2131, 15799, 29601, 24595, 1687, 100573, 500517, 1017087, 2066749, 3248393, 3297157, 4313309, 32387169, 34371107, 93788475, 46148469, 314462157, 667848331, 1654618677},
  {1, 1, 1, 1, 3, 53, 83, 187, 445, 827, 1549, 979, 5363, 1701, 2149, 64891, 91719, 258655, 406895, 550637, 898335, 1336093, 8179697, 5869839, 16142677, 13448439, 64091565, 119150297, 245760711, 317090843, 1011497501},
  {1, 1, 7, 9, 3, 15, 65, 161, 37, 233, 771, 3749, 727, 6857, 17175, 8191, 66319, 197129, 140523, 538789, 1585717, 3393083, 5303015, 9758523, 10266191, 37589165, 11171855, 199425785, 41937739, 450992081, 1791068717},
  {1, 1, 7, 7, 27, 29, 107, 247, 249, 353, 773, 3677, 7273, 5419, 29397, 32251, 80615, 202489, 67877, 265825, 1791023, 275493, 1533037, 3108019, 1808247, 731707, 114449275, 29408559, 183846789, 759084015, 762950421},
  {1, 3, 3, 7, 31, 49, 87, 159, 145, 497, 1715, 2115, 5035, 6431, 7245, 62763, 79717, 206265, 511781, 396389, 1350699, 3693585, 5790829, 3575195, 378871, 33167821, 100403337, 193728905, 184827133, 903108839, 1282727669},
  {1, 3, 3, 5, 7, 31, 51, 117, 101, 617, 557, 2551, 6589, 13295, 31975, 63295, 40493, 75561, 382119, 465209, 1246269, 3746905, 1754167, 15576091, 10055183, 21949219, 40887617, 50785039, 13227649, 956296021, 1266363653},
  {1, 1, 3, 3, 15, 27, 125, 163, 169, 893, 1771, 25, 5787, 10267, 10297, 37987, 47351, 94941, 209833, 89069, 1532977, 1498191, 1262637, 4887715, 22929067, 49897677, 59862739, 147402381, 249973553, 423661835, 1677050309},
  {1, 1, 1, 5, 9, 47, 85, 65, 289, 783, 1105, 4035, 4111, 2589, 24575, 39951, 22703, 214095, 299179, 31471, 1860801, 1712871, 6961407, 3152367, 25315841, 14768611, 129540021, 68244565, 239464807, 999916737, 993364637},
  {1, 3, 3, 13, 23, 33, 7, 49, 301, 531, 1713, 2755, 5543, 8153, 24099, 16567, 39773, 77801, 36911, 131449, 175043, 932049, 1484835, 223419, 19077269, 25307339, 2813349, 160048413, 9844851, 742291637, 836907725},
  {1, 1, 5, 9, 7, 39, 101, 67, 417, 923, 757, 1537, 5553, 12233, 20881, 18931, 29111, 50107, 34351, 187689, 368965, 2281887, 1218637, 9427307, 13212381, 43964671, 37113455, 145507359, 383415655, 342022767, 130246813},
  {1, 1, 5, 1, 19, 7, 25, 123, 125, 183, 573, 3317, 6867, 871, 17631, 51503, 128559, 191851, 482791, 572141, 170733, 2883251, 4667053, 13039415, 1562569, 41571295, 130866395, 105497301, 327678749, 949725397, 1320890741},
  {1, 1, 3, 15, 19, 13, 117, 41, 129, 715, 1525, 2257, 2179, 10807, 23271, 62495, 15183, 92077, 13285, 69097, 1016047, 2547387, 6001359, 11526863, 2781925, 144335, 124894847, 230893645, 271581501, 126881197, 1496357877},
  {1, 3, 1, 5, 25, 53, 19, 169, 289, 569, 1135, 1967, 7001, 15883, 15113, 7075, 83989, 201307, 294243, 548143, 577659, 3937941, 4974579, 2423287, 30706599, 37403313, 133316669, 235331091, 492720029, 796178403, 1698823541},
  {1, 3, 7, 15, 7, 37, 127, 147, 415, 313, 1541, 1889, 3763, 16199, 12681, 52291, 33269, 92893, 444645, 545085, 1796175, 3038961, 1440609, 12336333, 27296787, 13365107, 94223703, 188341645, 125466157, 1007217827, 373680565},
  {1, 1, 3, 9, 1, 35, 95, 137, 237, 951, 899, 3177, 6073, 10655, 31687, 57823, 106991, 31597, 232419, 884983, 152573, 1332661, 988435, 992763, 1471569, 10400945, 34552955, 4535999, 26479665, 18634613, 1649152965},
  {1, 1, 5, 5, 29, 57, 45, 253, 297, 529, 1553, 467, 8035, 15675, 21691, 44807, 92031, 86667, 376163, 732523, 1531615, 1356259, 1355723, 12317647, 4409791, 48516255, 110603933, 227648361, 424031193, 447118393, 605434981},
  {1, 3, 7, 15, 25, 41, 59, 81, 87, 985, 1001, 2369, 661, 7551, 11829, 64827, 94309, 201853, 43493, 977059, 305535, 3394261, 1232363, 484337, 21638539, 62277231, 3100655, 153471515, 481773113, 717021175, 1082106853},
  {1, 1, 7, 9, 27, 21, 7, 233, 309, 67, 701, 2737, 4261, 2467, 15691, 49735, 68255, 155209, 343659, 804733, 882399, 1500649, 4875083, 3116851, 24011797, 43872327, 83426383, 4570699, 110509989, 948836785, 215090581},
  {1, 3, 7, 1, 19, 55, 47, 155, 333, 101, 517, 1991, 4619, 10435, 27241, 12131, 5557, 166941, 174059, 1000629, 1122765, 3974693, 7190177, 15319711, 2315723, 15694595, 133007889, 137070617, 137074305, 997111835, 1296763653},
  {1, 1, 7, 3, 23, 35, 7, 125, 157, 537, 933, 3281, 4975, 8969, 27581, 39595, 52839, 665, 251297, 986149, 1101273, 1228389, 2355943, 5118659, 11515599, 30861739, 49962295, 57686257, 169987419, 111309051, 368444013},
  {1, 1, 3, 7, 19, 53, 81, 103, 461, 435, 777, 335, 5261, 12249, 9695, 56591, 4367, 78029, 438445, 702841, 905591, 4151855, 4223657, 5832655, 27188381, 33052675, 3560089, 259560719, 209095127, 120178945, 815898205},
  {1, 3, 1, 7, 19, 9, 75, 245, 355, 37, 1855, 1339, 3107, 7251, 16543, 32335, 16365, 94123, 81889, 39521, 18523, 2607629, 887359, 13407053, 24675023, 47810225, 5736937, 116365793, 333422113, 344567565, 906041221},
  {1, 1, 1, 3, 5, 35, 39, 223, 113, 423, 1423, 713, 6113, 349, 24147, 46455, 53439, 218479, 409389, 707311, 1052749, 19981, 6022741, 13642347, 23132513, 64179933, 79858403, 209098071, 385135139, 1053025157, 1884917133},
  {1, 3, 1, 1, 15, 31, 11, 75, 499, 345, 1253, 2629, 2551, 7483, 25395, 1015, 50013, 157899, 407527, 1005553, 1861997, 2104353, 7640249, 734673, 6518379, 64645091, 34071507, 6810497, 148457545, 873228113, 591702053},
  {1, 1, 3, 11, 25, 25, 3, 211, 185, 45, 1865, 1805, 3303, 11091, 529, 2355, 64215, 164189, 277217, 208811, 133631, 1142893, 7319641, 10645627, 8147947, 22828559, 7404227, 206279697, 454141697, 349226147, 1470174469},
  {1, 3, 1, 1, 9, 21, 7, 165, 107, 641, 1083, 2805, 2099, 5855, 18477, 59595, 111333, 82683, 149351, 71223, 1874155, 2850849, 3833943, 6128117, 29917595, 50693469, 49450623, 128027741, 197634197, 145676311, 1944962389},
  {1, 3, 5, 3, 9, 21, 77, 103, 505, 277, 335, 797, 3869, 2957, 1979, 103, 94557, 20399, 164397, 725731, 2041155, 3316091, 7974785, 7959179, 5001715, 33051737, 126894079, 84665363, 318005851, 126910429, 1974507629},
  {1, 3, 5, 15, 31, 23, 77, 247, 303, 891, 1261, 3233, 3495, 13111, 13185, 26451, 106773, 258207, 506209, 162989, 641245, 4097863, 6771113, 12796653, 17393073, 28961599, 121863311, 261590057, 250063029, 494704091, 143481813},
  {1, 3, 5, 11, 11, 35, 49, 229, 149, 931, 881, 775, 2949, 3141, 29157, 39675, 47973, 63199, 53797, 971057, 876753, 3134243, 5634867, 10659235, 16286773, 53706475, 103910945, 256674487, 151552059, 604293987, 113452013},
  {1, 1, 3, 5, 19, 57, 23, 95, 347, 221, 195, 3561, 1481, 2063, 3979, 46663, 81759, 109197, 484975, 781565, 1005171, 3969377, 4745517, 13227281, 26490255, 41485, 2975843, 221113055, 365380481, 727060889, 1754927877},
  {1, 3, 5, 3, 13, 1, 23, 173, 431, 29, 421, 3235, 2751, 4447, 28283, 27239, 861, 253103, 507149, 854791, 173887, 763705, 2569199, 14659489, 19158095, 5277003, 76515301, 143231621, 61189253, 746037793, 1022992661},
  {1, 1, 5, 13, 23, 3, 1, 9, 327, 855, 1251, 2997, 6129, 4223, 11555, 45431, 44895, 213867, 1803, 565521, 2086193, 3302195, 417727, 11661897, 28138229, 31406105, 118250007, 119970991, 30253905, 864792753, 537785925},
  {1, 3, 7, 13, 29, 21, 37, 229, 217, 353, 1239, 3955, 491, 12183, 14777, 4771, 6549, 80701, 402183, 878339, 1763107, 2198291, 6923539, 7409199, 3834735, 1672025, 62621329, 261849177, 401446965, 566924707, 935329749},
  {1, 1, 5, 5, 1, 33, 103, 187, 183, 939, 1873, 2633, 6143, 15405, 17353, 55075, 32407, 15899, 25347, 24343, 1233183, 1007369, 5212153, 15604821, 33488401, 29709827, 134029055, 245409393, 103597439, 678716711, 1686933757},
  {1, 1, 1, 9, 21, 27, 71, 129, 499, 279, 1181, 4053, 2485, 1961, 30603, 21479, 75839, 93775, 205575, 696075, 167709, 1463149, 3238347, 13205557, 8094669, 31360119, 75138091, 246061523, 21271679, 928376741, 38006525},
  {1, 1, 3, 15, 21, 37, 45, 201, 221, 187, 727, 1241, 6171, 1383, 22277, 36731, 64103, 255965, 190725, 827151, 880939, 3860811, 132015, 12364243, 27699789, 53755293, 16671347, 145433133, 406252941, 557854319, 608220469},
  {1, 3, 7, 5, 21, 17, 67, 177, 323, 601, 633, 865, 6131, 10329, 8689, 52179, 100277, 257437, 357903, 1035803, 1125911, 649757, 1095387, 9396133, 29420035, 53733455, 124498671, 169183261, 347126771, 161795975, 389900493},
  {1, 3, 5, 9, 15, 45, 71, 43, 359, 651, 103, 403, 3249, 11769, 6567, 11679, 91469, 26671, 373255, 293393, 279103, 13929, 724585, 7668549, 3393745, 18956213, 36762185, 54943827, 471685103, 203169289, 265603773},
  {1, 3, 3, 13, 3, 23, 101, 145, 367, 999, 1489, 3673, 2959, 10855, 16029, 15243, 107461, 257817, 12815, 818701, 1390141, 834095, 6664735, 13035361, 27173461, 9659691, 19031839, 253371713, 367316117, 861681831, 379902293},
  {1, 3, 7, 3, 13, 43, 123, 87, 55, 1015, 141, 2917, 6567, 16025, 25555, 6999, 39805, 49293, 114185, 821775, 1956357, 679425, 3768933, 7788077, 5177117, 57151239, 40013339, 130257121, 31230507, 897177941, 1654348205},
  {1, 3, 1, 3, 17, 7, 21, 161, 41, 889, 1315, 1897, 639, 15451, 3049, 6179, 75669, 153723, 205317, 513547, 1753633, 3337799, 2819615, 13157127, 30816007, 48635005, 37611395, 208976361, 353094169, 872952811, 1509471077},
  {1, 3, 5, 15, 27, 33, 55, 17, 81, 431, 325, 909, 3547, 10121, 17815, 38495, 50605, 220079, 513537, 786953, 1991171, 2989665, 4160035, 6420391, 13432561, 65266135, 54633267, 255682733, 130896419, 205533057, 205547405},
  {1, 1, 3, 1, 15, 37, 43, 137, 203, 191, 1129, 1585, 435, 3177, 769, 46899, 60023, 101661, 471051, 790537, 2043943, 2191445, 6230067, 5083229, 20034133, 33493619, 85170759, 129345933, 350667907, 245005463, 898022669},
  {1, 3, 7, 11, 21, 23, 125, 41, 17, 951, 465, 3691, 3465, 13247, 13779, 64567, 50845, 101101, 235521, 727047, 239657, 2422879, 6899751, 391439, 15886553, 27553715, 38243229, 182621091, 371566473, 359649233, 97066277},
  {1, 3, 3, 1, 31, 23, 43, 101, 405, 739, 1061, 2955, 5643, 16137, 8763, 53095, 107549, 89289, 410627, 1047561, 1516597, 2980977, 4643991, 9139531, 30997765, 25897119, 23372005, 151188113, 187195419, 761138269, 1743438701},
  {1, 1, 5, 1, 19, 33, 99, 109, 203, 65, 395, 2775, 1373, 2557, 5875, 55031, 51167, 135371, 478215, 566285, 1559595, 1899621, 5575863, 15160469, 13594491, 3423565, 87036573, 189307095, 8427587, 804508357, 1389459981},
  {1, 3, 3, 3, 27, 51, 79, 63, 331, 365, 1071, 1661, 4549, 8561, 1719, 35423, 67149, 166953, 27649, 979977, 1113105, 767069, 21685, 2688401, 15048063, 63684129, 77657551, 124019435, 216235639, 543315113, 1119488221},
  {1, 3, 3, 9, 3, 17, 53, 161, 141, 489, 1325, 1709, 1381, 5093, 171, 7335, 125501, 33801, 330891, 65669, 1007787, 1931219, 4223255, 14541075, 12902751, 21358147, 1042363, 133052483, 366739827, 209453597, 1205665485},
  {1, 1, 7, 15, 9, 3, 95, 237, 197, 949, 7, 1837, 729, 10111, 6637, 42859, 30823, 250777, 64653, 352899, 124309, 3490949, 3754903, 256507, 31136243, 54168465, 100826207, 266322359, 276527809, 64652447, 1612310533},
  {1, 1, 3, 3, 19, 31, 57, 173, 483, 861, 1001, 1919, 3389, 11777, 20693, 22523, 65095, 209597, 24201, 481681, 390573, 2802643, 585915, 1871909, 5337703, 2192951, 56048449, 81676559, 375880271, 672179547, 369043005},
  {1, 3, 1, 9, 27, 13, 113, 177, 75, 925, 949, 119, 4759, 7775, 23033, 32419, 91989, 116283, 470671, 138389, 1349783, 1229775, 526399, 4106501, 371399, 22715563, 124923569, 225246917, 242104197, 632183843, 1010760981},
  {1, 1, 7, 15, 23, 15, 65, 61, 137, 653, 1843, 323, 379, 15157, 29885, 40619, 35079, 208569, 99981, 522909, 220581, 972795, 6476987, 6024463, 3915491, 45064805, 34107097, 158003417, 411711959, 557477795, 407242845},
  {1, 3, 3, 7, 29, 3, 11, 205, 347, 745, 1477, 3929, 5749, 4735, 29435, 15943, 9597, 212521, 477061, 212359, 1719485, 2768065, 3490495, 7177077, 4602035, 26418835, 7599407, 90091291, 79698521, 557874585, 1489003109},
  {1, 3, 5, 9, 1, 11, 111, 15, 7, 69, 45, 3607, 1099, 9203, 21301, 57403, 101797, 106655, 454791, 703647, 518277, 1467317, 2040437, 4001405, 6359951, 29342275, 133869025, 97809785, 307245009, 527453383, 330341957},
  {1, 3, 3, 3, 23, 3, 83, 173, 73, 485, 681, 1867, 3839, 11823, 13339, 65063, 74589, 24617, 379521, 326277, 1880761, 2914449, 6107711, 3423175, 1299163, 21811887, 127703621, 92787301, 152751157, 280873313, 1927523285},
  {1, 1, 3, 11, 31, 43, 107, 127, 465, 389, 1595, 427, 1571, 5885, 29569, 22355, 14455, 40413, 261249, 763021, 1917857, 2360217, 3582589, 12030811, 1783555, 42834445, 97566913, 93804081, 15225423, 955041327, 111505469},
  {1, 1, 7, 9, 27, 25, 117, 27, 287, 391, 279, 3247, 35, 12973, 5483, 34887, 18911, 45961, 214667, 940189, 877491, 3544803, 2202317, 10884789, 25185789, 48593981, 36888549, 44256153, 382912623, 354008645, 482336957},
  {1, 3, 7, 11, 19, 55, 45, 127, 245, 945, 305, 3907, 2455, 3163, 31, 33199, 50637, 119629, 113281, 496257, 1934469, 3130071, 5953249, 8478343, 28213071, 33339903, 121976781, 209663965, 198975345, 533991789, 647284933},
  {1, 1, 7, 11, 15, 17, 65, 15, 37, 207, 1447, 3027, 2281, 6557, 16717, 6347, 26087, 179641, 192649, 903565, 1532315, 1001863, 3732017, 13976375, 22007601, 44701525, 116498185, 135331115, 27011259, 374097611, 1252278253},
  {1, 1, 1, 13, 5, 27, 33, 213, 29, 603, 1171, 3235, 2255, 2017, 30999, 18431, 25071, 65711, 226435, 582291, 1276333, 390379, 6302291, 3942251, 16205501, 24524409, 31692929, 41439901, 449183467, 645900401, 805035181},
  {1, 3, 1, 5, 11, 1, 73, 233, 69, 125, 397, 297, 3337, 6191, 31055, 36591, 19245, 81867, 388291, 1037661, 415307, 2838031, 5808871, 8633083, 29248091, 41626603, 106359727, 98495175, 71487153, 655159101, 1581040069},
  {1, 1, 1, 15, 1, 1, 65, 145, 201, 917, 1891, 2999, 4069, 10413, 15819, 11719, 12287, 57423, 13505, 627667, 294519, 3885695, 5626463, 12785319, 23924619, 59931045, 126292781, 246877167, 417669347, 598561533, 113971341},
  {1, 3, 5, 13, 15, 51, 115, 167, 311, 375, 1069, 2595, 3337, 753, 11903, 45743, 102125, 163631, 262595, 253657, 1373553, 387585, 8374837, 10495853, 2434909, 41096935, 8123613, 45539459, 404406983, 842967393, 2138622493},
  {1, 1, 3, 1, 1, 23, 69, 125, 147, 915, 1945, 411, 979, 13863, 30443, 54663, 68895, 5869, 424267, 761927, 1698121, 1528871, 2573539, 11662597, 27092265, 23098715, 104117693, 97939289, 43245669, 192211441, 535230101},
  {1, 3, 1, 11, 5, 1, 93, 23, 135, 93, 1689, 23, 3519, 4491, 24673, 5075, 101109, 34171, 117837, 13647, 1738223, 629843, 6503489, 7695765, 3392807, 11962823, 125312041, 212020829, 128798625, 345876723, 905319301},
  {1, 1, 7, 3, 11, 59, 93, 153, 487, 475, 1191, 1455, 5963, 8259, 18811, 64295, 45535, 149033, 24129, 906841, 231513, 2583391, 1037271, 5418681, 8803673, 61237125, 61039477, 121398801, 134470161, 599058281, 1436732229},
  {1, 1, 3, 1, 13, 15, 55, 71, 433, 33, 491, 1835, 5695, 10509, 347, 34343, 95519, 180941, 39243, 536523, 1464137, 3998037, 4654909, 10416731, 4886695, 16463637, 12590385, 243076109, 468255303, 49498021, 1287757853},
  {1, 1, 1, 15, 19, 1, 23, 47, 235, 101, 1057, 901, 5477, 7079, 30885, 38267, 94535, 255519, 114881, 76865, 976339, 1700193, 7198685, 1441, 31104887, 4773887, 73184843, 227083011, 42547377, 670791575, 1029268421},
  {1, 1, 5, 13, 11, 43, 119, 77, 441, 121, 783, 827, 1757, 12751, 31593, 14787, 47607, 168315, 329675, 313421, 485089, 934181, 129815, 3778023, 21778375, 18674865, 95853257, 161234335, 295654561, 1058512667, 108818821},
  {1, 3, 7, 11, 19, 17, 37, 225, 329, 231, 515, 1541, 7371, 6355, 10905, 55235, 6229, 213181, 269633, 242241, 1732323, 760147, 7564215, 13128791, 22002969, 41616929, 119474327, 111221173, 210709513, 995058555, 1435343653},
  {1, 1, 5, 13, 7, 11, 35, 215, 345, 577, 147, 2803, 3291, 4631, 5329, 11699, 115575, 191291, 116299, 382273, 1682137, 3169537, 6128581, 11148387, 2983159, 10227469, 75832281, 170076857, 418346549, 276555083, 1963772373},
  {1, 1, 3, 9, 21, 55, 113, 251, 25, 221, 1445, 3385, 1589, 4109, 29897, 43235, 52215, 71901, 136899, 778947, 1137633, 3429187, 349997, 5231699, 29968331, 41380323, 102186255, 92082555, 280043627, 599359119, 1842157741},
  {1, 1, 5, 7, 9, 45, 5, 33, 331, 285, 1101, 3131, 2713, 5653, 3823, 1935, 100271, 21739, 241857, 396507, 776779, 3706627, 679319, 14899253, 29940711, 50040083, 20295373, 268233587, 513735907, 1062397625, 1188583565},
  {1, 3, 7, 7, 5, 39, 43, 167, 481, 629, 777, 1827, 4653, 403, 4781, 15883, 46053, 4253, 294349, 694095, 863689, 299273, 5620613, 8112635, 14645427, 33377055, 69120301, 191562791, 12920781, 496728935, 947252789},
  {1, 3, 3, 7, 31, 33, 31, 159, 313, 673, 1425, 663, 5819, 1297, 26627, 61335, 17085, 7753, 306117, 337733, 1830875, 3356889, 708765, 15970355, 28406295, 8279183, 87522617, 227719065, 507361011, 159841717, 1686780621},
  {1, 3, 3, 1, 19, 61, 117, 93, 373, 491, 1031, 757, 4185, 771, 7265, 65331, 59669, 167929, 494915, 889285, 1877319, 33675, 7637147, 16509107, 27031581, 6319277, 88782911, 101925175, 342805917, 564217707, 24625525},
  {1, 1, 7, 9, 3, 45, 65, 223, 437, 41, 1139, 2733, 5963, 2709, 25429, 14043, 4327, 196409, 80331, 770117, 1734263, 2939807, 1532977, 14445863, 15884279, 173133, 3531143, 79128661, 283098735, 1032179963, 274097341},
  {1, 3, 5, 11, 21, 27, 31, 127, 255, 761, 1865, 1319, 6583, 9235, 10717, 37675, 11525, 227999, 74437, 351695, 561781, 2028517, 380613, 8237669, 19102787, 46698623, 12582377, 250626389, 66183145, 804215903, 211910309},
  {1, 1, 1, 5, 21, 1, 63, 43, 413, 557, 567, 2893, 8017, 2307, 29525, 21339, 54087, 256607, 111947, 909267, 1231479, 1268609, 6164025, 8782847, 29577091, 29212801, 37603263, 179130687, 202236013, 761733823, 1289819317},
  {1, 1, 7, 3, 31, 1, 15, 235, 215, 395, 1971, 469, 5275, 431, 5349, 12475, 15399, 26137, 350401, 17869, 490315, 3666089, 4132809, 263541, 24399281, 42378721, 1091303, 37086693, 13926933, 696637503, 751844693},
  {1, 1, 1, 13, 25, 59, 71, 245, 389, 279, 1293, 89, 6551, 10285, 14495, 8015, 35087, 69103, 388035, 831823, 161269, 1875453, 5847455, 7775595, 7441713, 54684511, 99631479, 57007337, 41053135, 565517457, 520557629},
  {1, 1, 5, 5, 9, 63, 17, 229, 425, 939, 877, 3689, 7229, 6707, 30771, 56343, 120095, 73067, 101699, 854751, 121553, 576419, 4974003, 12271903, 29868325, 57131127, 74892567, 261833411, 138956357, 365889065, 249511445},
  {1, 3, 7, 7, 11, 29, 43, 41, 25, 237, 1585, 3735, 2617, 7541, 26243, 2711, 30109, 211213, 227917, 891201, 1856751, 2197957, 5813011, 12928847, 16150115, 9304711, 35348041, 12639739, 25772563, 780085237, 1465988941},
  {1, 1, 7, 9, 21, 5, 69, 231, 361, 39, 1695, 3043, 2973, 5487, 12857, 4579, 97591, 169337, 424779, 298963, 1690227, 1117331, 7348497, 3279835, 19352961, 29078669, 115561929, 140345311, 485800289, 366054219, 237914245},
  {1, 1, 5, 3, 17, 63, 91, 217, 407, 133, 1373, 4021, 1737, 10043, 4561, 47763, 74711, 4603, 363077, 665291, 846969, 2841785, 6573531, 8070769, 2726743, 56020483, 72080867, 112553663, 50503765, 157154667, 260189781},
  {1, 3, 7, 9, 31, 13, 101, 231, 175, 457, 89, 2167, 2725, 8505, 375, 60031, 98189, 100973, 334147, 220745, 691695, 1159051, 2233249, 1575901, 19603683, 66380815, 48554329, 145565607, 422809383, 194309481, 724455837},
  {1, 1, 3, 15, 31, 11, 27, 211, 347, 291, 1881, 3091, 3307, 5117, 13341, 61283, 8495, 230069, 433789, 54685, 155289, 1486529, 6689209, 9991905, 9861865, 2989891, 15639965, 93892761, 508520663, 633383259, 1621301533},
  {1, 3, 5, 5, 13, 25, 5, 197, 237, 135, 635, 1175, 5085, 14737, 10807, 6791, 43397, 201399, 270675, 515715, 1407143, 68563, 7983995, 6787875, 28911705, 35150097, 132439981, 172367439, 303963527, 1059724369, 1664245853},
  {1, 3, 3, 9, 7, 63, 107, 127, 147, 477, 1813, 2619, 8089, 2651, 26549, 18339, 39325, 31377, 486451, 993417, 829861, 1541817, 2639173, 10466097, 31092343, 42403283, 73597973, 115677195, 440082645, 637881015, 1290215061},
  {1, 1, 5, 11, 15, 45, 27, 133, 45, 621, 707, 2679, 5929, 19, 9599, 21943, 45559, 107027, 304021, 392861, 1504143, 547245, 1863991, 2887739, 29061491, 24319901, 128750229, 27624347, 42897869, 188669005, 1598168053},
  {1, 3, 7, 9, 21, 37, 41, 255, 69, 1009, 1999, 367, 6177, 10017, 3549, 13491, 41581, 37941, 328699, 517771, 1803931, 5319, 6628137, 7521487, 8042071, 29559849, 44300613, 130454035, 224833615, 162850155, 1711494397},
  {1, 1, 1, 15, 19, 55, 73, 189, 423, 983, 1811, 2551, 4765, 12077, 18205, 11475, 124623, 206711, 414329, 176281, 1253565, 1150203, 2141059, 12195145, 13980829, 38676529, 71752157, 189674447, 37078843, 741241219, 1929788333},
  {1, 1, 5, 7, 17, 13, 25, 225, 463, 471, 631, 1811, 5797, 3235, 32253, 53203, 71103, 95123, 36537, 410523, 876195, 2050967, 4773399, 15215609, 11244957, 4844277, 15242289, 84666671, 97684053, 1008205791, 340938261},
  {1, 3, 7, 1, 29, 7, 123, 187, 331, 735, 1757, 1115, 2077, 15725, 2183, 47847, 17829, 106613, 15059, 193427, 592265, 1054961, 6142657, 1978241, 10497877, 61041535, 120573357, 156682991, 510814795, 629100665, 306543725},
  {1, 3, 7, 9, 17, 61, 111, 93, 21, 1003, 1905, 3719, 2111, 11845, 6427, 57039, 92853, 231093, 219675, 251535, 513931, 3189409, 4328039, 6862723, 9968869, 16640131, 50260121, 138125605, 169841431, 718449405, 2086827293},
  {1, 3, 7, 7, 17, 21, 51, 59, 115, 723, 2039, 2833, 5969, 5737, 18407, 58951, 50373, 89525, 165461, 344979, 803739, 3310765, 1547001, 14059097, 4523529, 38138157, 13506179, 59101471, 74870335, 323029633, 1606367933},
  {1, 3, 3, 13, 9, 47, 95, 233, 13, 281, 1049, 619, 405, 16205, 20097, 24747, 89661, 22897, 235191, 380815, 1530297, 1282965, 6394555, 3529651, 13021143, 22341487, 105475629, 56226935, 331582939, 943561055, 744800941},
  {1, 3, 7, 13, 9, 41, 11, 171, 453, 713, 587, 1669, 2489, 10277, 18599, 37767, 118565, 97909, 524255, 190367, 1632703, 1268877, 8062537, 5411759, 19151103, 35973373, 120766727, 244303695, 219505235, 1003238713, 558675597},
  {1, 3, 3, 13, 21, 41, 123, 173, 511, 399, 859, 1515, 5773, 12535, 26289, 22539, 126125, 102353, 430327, 322451, 438407, 3794381, 1165999, 12988313, 24453589, 1302665, 117515057, 56385863, 248620337, 138316891, 181395845},
  {1, 1, 7, 15, 11, 3, 113, 111, 73, 7, 1191, 2573, 7713, 465, 27615, 52151, 28391, 128561, 456917, 419225, 336313, 3258027, 1006833, 10214755, 108185, 12972317, 92083247, 50876855, 273400151, 146403593, 1042129693},
  {1, 1, 7, 15, 5, 5, 39, 11, 489, 13, 1041, 1639, 7879, 11899, 6899, 22239, 53607, 137809, 432469, 158871, 1225379, 3074761, 3654689, 9090483, 3978275, 13746783, 129615937, 50774621, 10806585, 340766089, 1867619365},
  {1, 1, 5, 9, 27, 31, 15, 237, 401, 795, 1675, 2361, 7333, 12507, 14627, 15471, 118023, 37395, 234071, 813965, 379037, 2968205, 6165663, 15854919, 20385021, 18497161, 41064211, 103337579, 90359621, 1065975857, 935273045},
  {1, 3, 1, 7, 21, 53, 31, 81, 189, 683, 1283, 419, 7585, 9207, 15053, 16275, 128925, 192595, 434489, 562063, 1735347, 4010417, 1860683, 1588403, 24106309, 61402321, 116005529, 229865403, 38027017, 593784599, 904703333},
  {1, 3, 5, 11, 21, 1, 49, 251, 403, 489, 1235, 429, 4855, 4081, 17575, 1015, 88421, 210167, 179037, 692871, 975799, 3946167, 4317357, 8921201, 4809651, 25012189, 131852959, 129022801, 308218555, 864632873, 1500497453},
  {1, 3, 1, 15, 29, 33, 77, 53, 105, 731, 749, 2677, 3967, 7967, 18723, 23231, 8837, 10515, 426193, 413335, 1916055, 3805931, 7030875, 6543719, 29808717, 52395439, 364819, 114740785, 303815429, 793564849, 1956020053},
  {1, 3, 3, 11, 9, 47, 11, 95, 137, 923, 599, 1585, 1969, 9625, 19171, 7375, 67797, 56625, 341137, 739075, 2063137, 1692481, 5196101, 2126219, 29602309, 26817781, 29921795, 155390751, 39785615, 47120741, 1084593021},
  {1, 1, 1, 5, 7, 7, 85, 49, 339, 883, 261, 2125, 3831, 9797, 16395, 57903, 15351, 151863, 47507, 211737, 420109, 577367, 3435991, 5659925, 7492705, 63951615, 78722459, 268074057, 141348527, 82943501, 2139874045},
  {1, 3, 3, 3, 5, 9, 33, 99, 75, 889, 101, 2099, 6635, 11511, 21573, 30291, 115613, 125841, 434233, 40223, 1939775, 3304727, 7767861, 2590041, 21343731, 8020651, 70380589, 17409449, 334458237, 773519703, 761366837},
  {1, 1, 5, 11, 1, 11, 79, 49, 7, 131, 471, 1235, 3287, 14777, 12053, 36627, 1647, 78835, 389877, 391443, 1975605, 3601165, 389939, 914505, 5408249, 64879933, 115736929, 265166685, 151401579, 466510155, 647131117},
  {1, 1, 5, 15, 9, 9, 83, 15, 21, 899, 1785, 2349, 3471, 6723, 1405, 515, 115103, 58963, 373809, 686355, 1319735, 134965, 2931069, 7644887, 27099141, 5276571, 57839395, 62624353, 437618393, 1052420439, 204331557},
  {1, 3, 5, 11, 1, 7, 121, 223, 509, 859, 1037, 491, 5529, 481, 17029, 56643, 85517, 167095, 71933, 642835, 1840409, 2360611, 3430785, 3157687, 4518573, 55390687, 28604525, 6594219, 199798223, 241889523, 758733949},
  {1, 1, 7, 5, 17, 35, 91, 171, 113, 65, 423, 2371, 5105, 12827, 31087, 24535, 1527, 183569, 119583, 92951, 1726725, 2343721, 989121, 7143003, 30183231, 66096209, 80381625, 23564283, 462985701, 925415917, 378038357},
  {1, 1, 3, 3, 21, 47, 55, 11, 159, 51, 263, 2113, 661, 9147, 28929, 59947, 9999, 97941, 484337, 261135, 879641, 2667533, 3843305, 3439877, 14276561, 25795693, 15312435, 74021883, 431075493, 791986259, 1219622613},
  {1, 1, 1, 9, 19, 7, 43, 223, 207, 787, 543, 2141, 4247, 7369, 29031, 19191, 4391, 221687, 86495, 817173, 170005, 4126841, 321541, 4897005, 23087489, 55393717, 42579423, 47135905, 321754531, 995677017, 414906061},
  {1, 1, 7, 11, 11, 51, 121, 9, 211, 905, 687, 889, 1827, 13395, 3507, 18175, 54503, 255697, 220625, 964625, 157721, 2627651, 6809735, 15858069, 23247163, 15530337, 85364883, 215244381, 104752993, 750599017, 65868613},
  {1, 3, 1, 7, 15, 23, 5, 139, 469, 569, 33, 3477, 5391, 13665, 17011, 50975, 31413, 255347, 60313, 184853, 367141, 3190279, 7959213, 4855335, 781519, 11082071, 58751875, 202387953, 531697851, 91956381, 886911277},
  {1, 1, 1, 15, 29, 29, 29, 201, 63, 1019, 97, 1671, 9, 4617, 19833, 15931, 79039, 32567, 504633, 809495, 1521163, 2964035, 6359671, 5691913, 17702769, 42062011, 103174969, 211117427, 419496375, 223416671, 472980125},
  {1, 1, 5, 15, 25, 5, 67, 225, 189, 919, 1471, 1451, 5017, 16189, 31555, 60367, 107255, 25971, 223249, 763395, 450075, 416381, 2585219, 2940899, 19351353, 51888725, 50806937, 189419491, 437254611, 348962477, 1345644813},
  {1, 3, 5, 5, 15, 51, 89, 221, 149, 863, 43, 2381, 1767, 8037, 5319, 18231, 65989, 172343, 371795, 651777, 48649, 2978387, 2486003, 13701819, 10946609, 6643537, 32349111, 215179137, 463979631, 815263009, 2143772157},
  {1, 3, 3, 1, 15, 17, 5, 77, 69, 27, 1883, 63, 5987, 1497, 3723, 3983, 77653, 96881, 292379, 334353, 840251, 2619947, 6946323, 5648955, 24539829, 32165929, 31915061, 99236897, 444212307, 867327749, 827751437},
  {1, 3, 7, 11, 7, 5, 113, 229, 123, 709, 1531, 641, 6655, 14923, 22947, 28159, 87909, 91029, 210713, 960477, 154583, 577751, 370403, 7525957, 2264759, 44300493, 103912691, 102387809, 514868257, 219875129, 1829973317},
  {1, 3, 1, 13, 21, 15, 45, 175, 81, 499, 1113, 587, 7573, 11689, 15651, 57903, 124757, 130419, 104147, 709979, 586209, 1816247, 6531673, 11570987, 308957, 30823915, 127234181, 263630775, 206549983, 651865205, 691947581},
  {1, 3, 1, 1, 29, 43, 101, 37, 131, 757, 465, 743, 2737, 8063, 23967, 49143, 107605, 84371, 3295, 356555, 1855205, 2579095, 1352211, 1600333, 24218391, 28617127, 32974465, 102086347, 474842897, 633985269, 1619601797},
  {1, 1, 7, 13, 9, 21, 39, 177, 51, 691, 2047, 1519, 6137, 5271, 8703, 21751, 97943, 171281, 417751, 471775, 311011, 111257, 1950395, 11278877, 6536065, 33619685, 89963533, 219128235, 162898473, 929951365, 1127553253},
  {1, 1, 3, 3, 5, 55, 63, 21, 3, 317, 461, 527, 2673, 16211, 6721, 31819, 5455, 14165, 496817, 793567, 467297, 3211701, 1299015, 13295301, 13050207, 35545739, 73706865, 191068971, 194609309, 451974579, 519390645},
  {1, 3, 5, 5, 5, 47, 7, 241, 387, 589, 323, 203, 7241, 14761, 13287, 53447, 50693, 32759, 285075, 513867, 1783489, 1948381, 1090643, 1582317, 13115327, 65641573, 132421929, 106318499, 296604935, 1003852257, 617450461},
  {1, 3, 5, 3, 23, 63, 55, 61, 231, 1023, 1315, 1181, 243, 7389, 25639, 53783, 72709, 46199, 300693, 315477, 1346797, 100997, 2986207, 11334129, 16569205, 44725873, 70817463, 241586549, 531027071, 37534497, 2117786813},
  {1, 1, 7, 13, 31, 43, 41, 81, 127, 887, 1513, 4055, 1361, 2443, 6963, 36655, 65095, 113361, 14999, 809929, 260081, 3940019, 3920063, 6893769, 5951677, 58454315, 44497289, 121546091, 433011653, 60368049, 754298965},
  {1, 1, 1, 5, 7, 43, 43, 33, 323, 911, 1373, 3053, 6503, 513, 6457, 35723, 66399, 11063, 237299, 686681, 382049, 3861681, 1807691, 13631005, 30894589, 43969983, 64305979, 175900089, 87295819, 292825143, 651973229},
  {1, 1, 7, 11, 25, 61, 21, 149, 205, 349, 1433, 1587, 149, 7275, 5465, 39003, 44207, 63057, 402993, 128195, 621683, 784571, 7270715, 2417251, 29935363, 12619655, 91495397, 70723207, 168252573, 806472843, 881230517},
  {1, 3, 5, 5, 11, 9, 31, 217, 119, 511, 209, 3325, 2023, 2877, 463, 14567, 9189, 56951, 507027, 563909, 1141627, 2632753, 7423059, 785481, 1918885, 18847171, 99629427, 143145449, 340171447, 865043705, 897315741},
  {1, 3, 5, 15, 21, 47, 89, 41, 347, 849, 1375, 3311, 807, 11443, 27643, 65295, 15557, 236055, 6681, 775503, 1684809, 2022467, 6077463, 4664445, 28576883, 31119937, 115045957, 224211061, 62628369, 55550257, 2143110277},
  {1, 1, 5, 7, 29, 43, 123, 191, 321, 373, 447, 2145, 1221, 2071, 12689, 29899, 45167, 14227, 158905, 1034583, 794077, 1392713, 7389397, 4112707, 21512827, 6211233, 57331707, 155342891, 161929353, 320420123, 28028133},
  {1, 3, 5, 15, 1, 21, 43, 141, 461, 779, 1109, 2915, 909, 8585, 19859, 29967, 126101, 24055, 513881, 83163, 2064475, 2664981, 2048703, 6166315, 21876277, 2669959, 15326325, 4149263, 15990511, 121958565, 416851709},
  {1, 3, 3, 11, 5, 17, 57, 13, 393, 661, 1761, 2455, 43, 8593, 20505, 2731, 67869, 110001, 523633, 1014223, 992967, 3927679, 1894987, 2816999, 7883875, 61654335, 76005521, 114246737, 211897227, 156232135, 787113709},
  {1, 3, 5, 1, 31, 47, 65, 249, 77, 513, 851, 2381, 3447, 693, 7729, 24347, 129501, 259831, 215159, 129753, 1446629, 1176179, 314903, 3980811, 13535695, 57469877, 118157759, 219499569, 412370239, 529452791, 451893053},
  {1, 3, 5, 15, 31, 19, 83, 47, 369, 697, 1815, 819, 7573, 9245, 8013, 14067, 97869, 206775, 191033, 494021, 1352161, 4142617, 4054773, 1973971, 19120151, 18269009, 72305609, 42031743, 333270443, 290384403, 1104274285},
  {1, 3, 5, 5, 11, 25, 27, 151, 107, 339, 299, 3869, 3393, 5661, 2947, 1615, 107829, 18103, 140243, 310213, 1987179, 3880469, 1834581, 11776729, 11555081, 12326737, 56753463, 160574887, 500172315, 1053825229, 642787885},
  {1, 1, 3, 1, 1, 59, 85, 57, 175, 465, 239, 3115, 7157, 7035, 11463, 7399, 7767, 161493, 298323, 943967, 1016173, 1821015, 8194351, 6283353, 9396891, 19441633, 115697377, 186580655, 266523757, 205055445, 1070860149},
  {1, 1, 7, 5, 31, 41, 53, 149, 121, 743, 189, 159, 5289, 2945, 1179, 33855, 31159, 74225, 152287, 609241, 339795, 664363, 5687627, 5535927, 25483921, 62394119, 53150825, 140315315, 354368159, 921389485, 617709117},
  {1, 3, 3, 15, 23, 51, 83, 25, 159, 163, 61, 713, 4529, 5253, 1603, 14159, 90997, 185137, 217557, 811221, 86865, 2098969, 8333635, 2648401, 6343177, 37746935, 11737711, 102700775, 431434675, 400774813, 565001357},
  {1, 3, 5, 11, 7, 29, 15, 177, 507, 695, 1305, 1863, 7525, 3063, 27433, 4155, 52493, 202007, 44093, 703573, 55503, 846137, 748459, 10132629, 9856917, 17739691, 52163833, 266338559, 203424273, 895499523, 685800709},
  {1, 1, 3, 11, 5, 41, 115, 227, 409, 951, 591, 4003, 7717, 4369, 15637, 40771, 29007, 20597, 234297, 554959, 1692383, 3788101, 2724177, 6477315, 11610201, 58641461, 9925777, 260736527, 348175207, 815983635, 749317085},
  {1, 1, 7, 11, 23, 55, 71, 135, 51, 411, 2003, 2375, 6823, 1711, 4443, 47567, 31399, 99025, 105617, 60749, 1424485, 3128133, 1044261, 12065681, 22700369, 2777241, 58286141, 4449429, 248862605, 790253161, 1571478133},
  {1, 3, 1, 3, 31, 47, 31, 233, 243, 3, 313, 1649, 6955, 13679, 32327, 58263, 27285, 174803, 285789, 674125, 1483629, 1262445, 6275995, 6984685, 24547697, 26055619, 78090335, 12292829, 71401869, 482444901, 518638197},
  {1, 1, 3, 11, 29, 9, 1, 79, 247, 677, 685, 3107, 5987, 9675, 29523, 18047, 71271, 98069, 319385, 836183, 1375311, 1118999, 605081, 13760125, 9006887, 50093675, 86062173, 203158945, 501904657, 589634905, 1570590085},
  {1, 1, 1, 7, 25, 31, 39, 241, 483, 839, 1143, 437, 2317, 2437, 173, 45011, 113967, 50839, 142097, 971491, 389889, 2179917, 3214227, 4187373, 24544405, 12876261, 13305015, 91819691, 471932507, 912352003, 1713985581},
  {1, 1, 5, 1, 17, 19, 83, 57, 39, 479, 715, 1911, 1091, 10937, 22145, 2683, 95199, 233427, 472927, 216183, 1799429, 558913, 6007651, 4456745, 22283109, 16516025, 30674285, 247731665, 206845427, 570709679, 909384333},
  {1, 1, 7, 1, 27, 45, 35, 55, 477, 719, 217, 3349, 7717, 6853, 9699, 7631, 23319, 166289, 268795, 930613, 1908527, 1294149, 7780661, 9476527, 8952089, 47884963, 26847627, 153186139, 319090827, 516945805, 1939421293},
  {1, 3, 1, 11, 9, 39, 25, 223, 303, 713, 151, 2611, 4629, 5855, 31729, 23467, 73165, 132275, 41429, 547307, 940297, 3826435, 8189409, 10749445, 28064187, 40520649, 64954345, 85637695, 347930185, 36401483, 29209573},
  {1, 1, 1, 11, 13, 35, 53, 39, 167, 779, 1673, 1221, 6281, 15113, 32027, 31071, 79959, 207703, 255869, 827695, 666917, 2989431, 1949521, 14547597, 32160329, 2993715, 35758123, 114423479, 283733087, 960048093, 2114697021},
  {1, 1, 5, 9, 19, 63, 89, 113, 199, 107, 1015, 835, 2879, 9499, 25597, 27107, 19583, 189427, 390295, 912997, 948237, 2643995, 1996815, 12982609, 12040625, 31656525, 25915697, 172972085, 493672505, 92303143, 1870674085},
  {1, 1, 7, 3, 19, 37, 15, 23, 449, 641, 1811, 3407, 6775, 6283, 31157, 7603, 28639, 71153, 240665, 83833, 759871, 3147865, 7456845, 13813891, 32024943, 45848901, 76300797, 160123477, 380301673, 959731823, 68444773},
  {1, 1, 3, 1, 19, 15, 31, 99, 511, 897, 1693, 2093, 955, 15897, 26693, 37363, 75151, 148309, 203539, 543725, 1494045, 2100317, 726033, 3648597, 12496739, 22852211, 49593027, 125617669, 252496427, 723599211, 641565805},
  {1, 1, 5, 1, 5, 15, 47, 19, 441, 541, 1621, 3877, 6407, 15991, 1931, 51855, 81671, 35763, 91391, 6563, 115761, 803925, 7621825, 6982995, 16604531, 18762651, 50307883, 72170473, 334146077, 434144649, 1802907829},
  {1, 3, 5, 9, 21, 61, 15, 77, 265, 351, 879, 3835, 6555, 2349, 23235, 30143, 69749, 157335, 390783, 722979, 1129507, 2518033, 7007383, 3028447, 954121, 27492005, 24271977, 123325665, 506022183, 663175141, 1518693085},
  {1, 1, 5, 11, 25, 37, 29, 181, 341, 641, 1213, 1319, 6359, 6231, 32573, 41059, 104863, 97075, 49301, 161643, 1778187, 2361923, 6412811, 16133819, 18561423, 38431643, 117546153, 92040605, 361305557, 497636671, 537073813},
  {1, 1, 1, 7, 1, 37, 87, 123, 33, 913, 111, 2613, 4895, 12595, 26633, 22539, 81967, 231671, 303825, 812795, 1259019, 674345, 6863377, 3719035, 332367, 31134745, 60079103, 16242105, 487018729, 222275779, 1169488997},
  {1, 3, 5, 3, 27, 11, 45, 89, 183, 241, 1355, 783, 3343, 239, 8643, 36255, 11429, 126391, 81781, 351929, 327201, 1261159, 1556159, 1058657, 29072683, 64221125, 109954157, 1692781, 131259709, 376831153, 253475381},
  {1, 3, 7, 7, 9, 35, 67, 187, 233, 577, 1445, 3063, 6039, 16233, 1453, 16963, 98861, 70869, 189845, 1042155, 1498685, 3202609, 2023969, 11061987, 6444143, 14018771, 26447977, 41481501, 307503275, 354188627, 136080749},
  {1, 1, 3, 13, 27, 11, 23, 15, 95, 63, 1931, 911, 8149, 6833, 3051, 19743, 64343, 251477, 451455, 556093, 357945, 4088445, 2083517, 2715149, 2786669, 9158809, 110972021, 222582007, 101900599, 437224589, 1560099741},
  {1, 3, 3, 5, 29, 49, 125, 117, 47, 143, 423, 3215, 3605, 3677, 17155, 42447, 57109, 182033, 203903, 367275, 330383, 411067, 1084411, 13440705, 19279157, 19946101, 104119437, 57505351, 461292763, 347702781, 2124358189},
  {1, 3, 1, 1, 31, 1, 123, 195, 83, 893, 1947, 339, 2927, 7183, 15443, 40255, 17125, 256819, 408575, 67881, 1615787, 1346037, 2490241, 1921897, 29964967, 37017661, 107918401, 15368053, 331868333, 171745121, 449242997},
  {1, 1, 7, 13, 31, 15, 91, 207, 39, 275, 439, 2617, 3093, 11041, 24997, 43651, 7855, 91473, 231959, 598889, 802997, 1065705, 5575157, 15623941, 23828977, 41909469, 91033099, 251334571, 433793727, 932148259, 2069086909},
  {1, 1, 5, 3, 3, 41, 13, 67, 361, 497, 25, 3807, 3551, 9681, 21043, 1855, 75159, 106451, 479741, 95393, 1967763, 2881443, 6450669, 2333707, 13805407, 43352879, 40798777, 175672693, 364876963, 587248093, 2018773325},
  {1, 3, 3, 3, 11, 27, 103, 59, 427, 327, 1705, 29, 8127, 1641, 20847, 31111, 130661, 101073, 310329, 344881, 1566353, 3183021, 4032841, 2614289, 4095149, 58186011, 111334787, 238590577, 24745203, 349740977, 424049805},
  {1, 3, 7, 5, 3, 37, 81, 137, 225, 101, 187, 3067, 2491, 12687, 16227, 8959, 37477, 37685, 193527, 681765, 2037159, 1184647, 5059983, 10793127, 7909683, 51378117, 12008745, 194175505, 450579173, 423245825, 819116885},
  {1, 3, 5, 15, 15, 33, 69, 223, 225, 771, 1917, 2293, 2889, 12083, 23971, 55343, 91365, 202103, 392569, 647989, 1476499, 38795, 808193, 15530459, 16739277, 47368207, 119533523, 202896679, 354909669, 486820817, 869533909},
  {1, 1, 3, 5, 11, 9, 121, 81, 203, 425, 1189, 2011, 3041, 3247, 739, 9967, 98375, 179253, 153207, 181565, 1457339, 587991, 8386103, 1367837, 25220875, 66733803, 47170949, 264852179, 236026225, 740302281, 1189088005},
  {1, 3, 1, 1, 13, 9, 39, 169, 437, 571, 1481, 3355, 3895, 8975, 31031, 43895, 68469, 108595, 199871, 924347, 1259655, 3763441, 3523699, 1688843, 4941381, 19800251, 5122709, 81472725, 233909149, 301013125, 1918460725},
  {1, 3, 1, 11, 1, 43, 35, 35, 293, 11, 657, 1415, 5021, 14463, 17945, 31099, 52077, 12339, 54485, 546403, 641685, 1829761, 7269593, 3831071, 9026781, 19841219, 107629145, 126670183, 343363785, 820348259, 322855013},
  {1, 1, 5, 5, 13, 47, 91, 15, 159, 23, 971, 3575, 757, 13477, 31757, 37971, 101871, 109971, 71579, 237155, 157585, 2450369, 6153333, 13853981, 23979205, 38772553, 131197741, 198243751, 162604419, 108075843, 267461453},
  {1, 1, 7, 1, 5, 63, 69, 27, 71, 129, 123, 3767, 89, 7865, 1189, 17571, 70831, 166545, 233819, 171307, 275393, 864319, 479453, 13848789, 15643507, 32535189, 22821525, 114547339, 508752615, 1065422563, 1061004125},
  {1, 3, 3, 5, 23, 1, 83, 3, 487, 491, 217, 2583, 3889, 15009, 9227, 29823, 58709, 107793, 298431, 136417, 1689707, 2864157, 3398737, 2747837, 25068953, 33636419, 72337945, 81321643, 412776975, 755120317, 866308733},
  {1, 3, 5, 15, 25, 1, 73, 107, 245, 191, 1449, 571, 1403, 6953, 17457, 36187, 39645, 248215, 476697, 826019, 846559, 3060831, 2196589, 2403383, 17603593, 7557619, 21289029, 116712409, 178961059, 947307455, 769066189},
  {1, 3, 3, 3, 27, 19, 25, 105, 207, 857, 1161, 3657, 2107, 7955, 517, 16563, 45213, 133681, 499545, 785441, 1088761, 2677315, 1229447, 8478569, 5254131, 27007543, 118539599, 203611933, 331837041, 636416703, 696070405},
  {1, 3, 3, 9, 21, 29, 5, 103, 219, 35, 3, 1635, 4815, 15797, 29839, 18887, 10373, 253521, 269811, 379259, 160067, 1326715, 4377129, 10010289, 25046841, 20524953, 70305145, 260986901, 225236791, 155488329, 422489757},
  {1, 1, 7, 7, 3, 63, 75, 77, 13, 57, 603, 2333, 7761, 14397, 10875, 33119, 92215, 92369, 308925, 91873, 2052181, 3930665, 5116663, 3210923, 1037639, 20854669, 17159487, 11820215, 484834203, 202948757, 519846829},
  {1, 3, 7, 13, 3, 11, 5, 255, 1, 947, 1695, 1927, 7447, 7407, 20797, 63203, 129181, 17653, 241119, 178229, 999401, 3864911, 2464553, 11264187, 8970461, 7005965, 59106533, 185578965, 421811981, 155952751, 1028013941},
  {1, 1, 5, 1, 1, 21, 105, 73, 429, 973, 1801, 3943, 6161, 1309, 3359, 48599, 15591, 232403, 379903, 1003367, 1409123, 4071287, 5768511, 15354091, 22489475, 54968151, 46037937, 120187467, 397205139, 700896273, 50882189},
  {1, 1, 3, 15, 27, 9, 9, 129, 117, 545, 9, 1983, 6351, 10925, 27337, 59787, 25823, 82645, 279709, 479417, 9715, 361767, 3920231, 11080387, 13998079, 53690351, 64758105, 166241189, 394129983, 1021143647, 163130173},
  {1, 3, 3, 5, 5, 5, 13, 155, 503, 875, 1243, 2259, 3445, 11953, 6517, 37347, 95213, 80657, 378719, 273971, 52939, 2720003, 5653861, 3036101, 5783129, 32540129, 105679513, 83330031, 339976839, 451526611, 758038621},
  {1, 1, 7, 3, 29, 21, 121, 147, 413, 423, 1887, 2429, 2765, 16335, 3071, 30391, 13431, 24817, 383993, 739191, 1469267, 2755383, 1702821, 7543343, 26024417, 28581141, 22943503, 58495931, 187270577, 536461829, 1590276741},
  {1, 1, 7, 9, 5, 53, 41, 137, 463, 583, 1627, 1731, 6675, 3703, 8177, 19163, 8623, 115569, 246547, 700859, 816875, 3685191, 1539391, 4835273, 6662477, 57793945, 52718745, 165983849, 516159885, 663379651, 8530293},
  {1, 3, 5, 11, 31, 29, 67, 159, 425, 25, 1457, 139, 5019, 701, 7357, 62787, 82637, 54615, 397405, 288685, 1980511, 1296885, 6273597, 12630987, 24069447, 14343415, 31255773, 49113465, 455184215, 674223195, 890273949},
  {1, 3, 1, 5, 25, 15, 123, 123, 245, 859, 249, 2175, 2137, 5765, 4873, 60859, 22909, 5331, 134939, 915239, 1955193, 3525041, 3353729, 8310047, 30686541, 15340299, 59137905, 9139763, 520104859, 285241175, 956361005},
  {1, 1, 3, 5, 23, 1, 111, 111, 111, 469, 1473, 1777, 3579, 13503, 2569, 4907, 15695, 189301, 208919, 157473, 881483, 1559521, 4582469, 1899677, 21374599, 29534791, 7288567, 113234765, 470963325, 351967851, 459500597},
  {1, 1, 7, 3, 17, 23, 51, 23, 499, 135, 713, 3317, 807, 9589, 11349, 56307, 80943, 77009, 249561, 147899, 1140553, 187081, 518461, 640545, 4888621, 48610587, 126354475, 182981753, 117979071, 305669139, 255369661},
  {1, 1, 1, 15, 9, 51, 75, 159, 359, 773, 1521, 2913, 5901, 3047, 14649, 56827, 91439, 107511, 389657, 694307, 1805421, 1336569, 5735893, 3116997, 19034599, 15422071, 125040967, 194650075, 11151241, 80897587, 1300628453},
  {1, 1, 3, 1, 13, 61, 117, 195, 49, 267, 57, 1769, 3621, 9415, 29443, 30031, 79687, 7157, 409011, 978803, 1929939, 735723, 2703213, 7068627, 22336213, 57012443, 41926375, 155280547, 311748857, 203919913, 1282779813},
  {1, 3, 7, 11, 3, 25, 33, 31, 315, 191, 359, 3399, 2481, 13831, 20205, 15859, 637, 122101, 88665, 413881, 1049955, 3239631, 3629353, 5314449, 15196933, 27599237, 110881381, 181843763, 325276841, 375715831, 705155045},
  {1, 3, 3, 5, 31, 43, 35, 125, 291, 51, 1469, 3857, 1707, 2641, 32137, 63403, 120893, 118417, 379679, 918569, 549457, 4123289, 5050343, 10921029, 16368913, 1707079, 108893015, 241415597, 104916435, 220217207, 1621108493},
  {1, 3, 5, 1, 25, 11, 113, 137, 211, 159, 1667, 939, 6439, 5337, 32059, 49775, 1333, 137495, 16183, 818303, 867053, 634867, 1903031, 15505525, 5335373, 9566529, 84909345, 227446957, 151892587, 681218693, 929963757},
  {1, 3, 3, 11, 31, 61, 99, 49, 383, 343, 395, 51, 6931, 16039, 5901, 9891, 48733, 166641, 395345, 488117, 173951, 806133, 1504147, 13076545, 31861533, 16793097, 31025865, 149695945, 334744093, 561604639, 1559970613},
  {1, 1, 3, 5, 9, 63, 63, 49, 405, 915, 1505, 2141, 6749, 7799, 17313, 24699, 75423, 190277, 288583, 495055, 702585, 798373, 5683059, 7536715, 21692861, 12388179, 72549267, 110298147, 220407673, 410604795, 1811644837},
  {1, 3, 7, 11, 15, 11, 49, 161, 155, 869, 121, 301, 6561, 4279, 15233, 37083, 9757, 21317, 315465, 1477, 1185913, 3444443, 5971887, 8349789, 22174359, 44002719, 17594891, 112268511, 144512129, 802886443, 993876165},
  {1, 1, 5, 13, 19, 13, 103, 59, 503, 293, 701, 2527, 5327, 13927, 5025, 33483, 15455, 146883, 257475, 90077, 1759839, 84209, 7089409, 5916381, 29919463, 44350059, 16223729, 262116293, 215713477, 737728787, 1499276245},
  {1, 1, 7, 1, 1, 37, 55, 155, 485, 399, 855, 2677, 5927, 9657, 2795, 19903, 113351, 60225, 46027, 969823, 1432803, 1410489, 4960161, 8532559, 2085097, 30044701, 79731751, 239960785, 270968931, 361473221, 488454861},
  {1, 1, 1, 5, 19, 15, 121, 69, 385, 75, 1567, 2649, 5601, 12981, 15903, 18599, 9463, 169031, 454211, 496477, 144893, 1108971, 743243, 7221039, 2219165, 48511437, 53278219, 260138479, 211835091, 110574897, 1926565645},
  {1, 1, 1, 11, 19, 21, 45, 59, 505, 737, 15, 1383, 1177, 8375, 15557, 51, 66895, 186055, 81741, 327873, 1927775, 419211, 1477877, 3966459, 27320255, 55162729, 106894909, 88079275, 391821401, 498987023, 1913977765},
  {1, 1, 7, 13, 29, 19, 123, 127, 469, 773, 733, 3289, 8087, 5803, 27897, 26587, 91807, 103745, 5831, 224731, 425821, 2820073, 2573317, 9536903, 3001747, 42210883, 108346167, 108156433, 379509921, 132067635, 632623941},
  {1, 3, 3, 11, 19, 55, 101, 67, 485, 939, 607, 1521, 6161, 12235, 16499, 13439, 47797, 80193, 33089, 866761, 117757, 3263111, 1082389, 5205431, 757293, 35274717, 111026083, 96951375, 531918445, 558449505, 1082468341},
  {1, 3, 5, 13, 29, 31, 31, 9, 453, 151, 1055, 3873, 405, 12877, 29829, 33459, 22365, 195911, 494667, 152659, 966225, 3498901, 1396747, 7680363, 8893177, 66139109, 41139203, 216127467, 183496739, 359035787, 1316386253},
  {1, 3, 5, 1, 17, 1, 17, 131, 107, 1003, 1749, 1849, 6207, 2153, 21275, 15199, 12549, 242503, 19655, 248135, 1428455, 3473559, 4606181, 2783721, 9227249, 30432367, 89421311, 110731737, 21017779, 583647349, 1089719181},
  {1, 3, 7, 15, 7, 25, 51, 143, 51, 517, 1841, 1771, 5389, 4633, 11123, 63807, 40805, 42949, 35405, 95557, 1735147, 1793757, 8221389, 8923041, 179743, 5935031, 26431325, 22437471, 298636271, 280987189, 94963069},
  {1, 3, 7, 11, 23, 7, 89, 95, 403, 361, 835, 585, 2783, 8091, 5141, 20291, 105133, 240453, 476361, 311261, 1919557, 1145803, 93873, 74249, 12684403, 26387869, 46246187, 133659777, 214178369, 515668255, 1773454533},
  {1, 3, 1, 9, 1, 53, 115, 11, 493, 587, 305, 3605, 1711, 4169, 20013, 48499, 20829, 134723, 219847, 837463, 25683, 3364317, 4045433, 12035063, 1387901, 62084307, 4560539, 186711553, 236988419, 199234755, 480282189},
  {1, 3, 7, 3, 17, 59, 55, 251, 49, 759, 1227, 3685, 7765, 1445, 20385, 37563, 5645, 184901, 182849, 933963, 1581397, 1445829, 7104057, 6770523, 21375033, 26479137, 107273327, 114257907, 173092827, 77148799, 1793951789},
  {1, 1, 5, 7, 29, 55, 19, 157, 129, 927, 893, 1235, 1955, 8153, 2865, 54987, 112367, 165699, 250441, 261831, 1365457, 2794825, 6663463, 8708023, 31971881, 65191431, 110929045, 245218353, 495182227, 787689295, 1425009421},
  {1, 3, 1, 15, 21, 35, 81, 53, 175, 939, 1635, 3597, 747, 14011, 12867, 56463, 32757, 184259, 74049, 616527, 681077, 292611, 1632219, 8477377, 20349153, 3572169, 52529575, 3971133, 382897945, 809953369, 993042213},
  {1, 3, 7, 1, 27, 61, 91, 73, 405, 677, 603, 2715, 7099, 941, 24523, 28031, 60069, 131141, 362051, 241605, 795983, 3992845, 7441355, 8208043, 7955947, 45318789, 52660473, 103781409, 270304247, 1022336573, 1710882333},
  {1, 3, 5, 9, 13, 45, 35, 167, 57, 483, 735, 2777, 7847, 6257, 13109, 58435, 57085, 147143, 149071, 21195, 1291987, 2352909, 7372693, 7205307, 14275149, 33519549, 112130731, 256382377, 26753691, 796394507, 1296608301},
  {1, 3, 5, 7, 1, 3, 97, 13, 159, 533, 1791, 1061, 981, 10795, 26165, 64787, 108205, 262087, 257217, 346459, 670173, 2128163, 1919747, 3526977, 33542623, 27604457, 30004815, 173259043, 266825045, 697207799, 803045781},
  {1, 1, 5, 13, 27, 5, 125, 25, 251, 221, 1909, 197, 6987, 11537, 15287, 55015, 114039, 232131, 143683, 268501, 1483335, 204635, 5557687, 7053693, 1024783, 48785123, 10690843, 252699477, 396098515, 290697081, 1513839501},
  {1, 3, 5, 5, 27, 15, 1, 131, 375, 923, 81, 3153, 6071, 2515, 23729, 4171, 30461, 3015, 387139, 1045061, 1611501, 2784563, 6739837, 6023221, 19802425, 40270203, 2493535, 194120977, 181538873, 1004430603, 1706439589},
  {1, 3, 3, 9, 9, 23, 71, 13, 465, 261, 937, 1549, 5993, 11325, 15065, 29867, 121517, 148161, 304579, 280279, 1683937, 19213, 7427871, 14613783, 21286867, 34284455, 45720971, 46997783, 54217371, 266569799, 1494647213},
  {1, 3, 1, 3, 7, 63, 17, 129, 435, 23, 215, 2251, 1561, 9703, 26483, 52511, 123157, 239171, 23757, 1017413, 208509, 2900275, 1212831, 8695221, 11132597, 9145077, 18670073, 210660243, 272888249, 545451289, 946386085},
  {1, 1, 3, 1, 5, 53, 77, 109, 9, 605, 371, 2081, 6023, 7145, 15837, 64531, 92319, 193989, 107843, 867787, 557051, 3120755, 5022307, 5303079, 21140923, 53602989, 87659763, 244574025, 456861987, 735916115, 705784141},
  {1, 3, 7, 11, 27, 39, 115, 47, 259, 337, 1857, 3465, 1549, 7747, 8525, 60643, 125069, 239301, 29641, 559697, 1228541, 324113, 1207957, 4739921, 27098923, 53957935, 42696079, 237094619, 7751101, 55826239, 1548093749},
  {1, 3, 7, 7, 29, 29, 75, 77, 29, 661, 899, 3137, 2661, 15271, 28093, 14275, 121709, 155077, 509381, 1034063, 522491, 1659461, 7325295, 9097835, 801883, 439197, 103355475, 57568299, 40319785, 15559207, 1987302757},
  {1, 1, 1, 3, 3, 3, 11, 219, 39, 757, 1465, 249, 7445, 7013, 15187, 38175, 102855, 19143, 233925, 400321, 1211977, 702017, 8218241, 9841661, 18033431, 5413391, 35715743, 65418947, 242100843, 477519373, 1420176749},
  {1, 3, 7, 15, 15, 1, 39, 245, 427, 1003, 1493, 1913, 6435, 14787, 13481, 33627, 59421, 175301, 474189, 891213, 553955, 1083993, 541311, 3720989, 24742569, 32545135, 369983, 268234565, 155331873, 536236899, 1205215685},
  {1, 1, 7, 3, 3, 37, 5, 97, 343, 833, 1379, 1551, 5403, 1843, 5877, 9027, 25903, 250433, 418121, 483673, 1934575, 576003, 2050751, 11200065, 8721067, 51713149, 80836237, 141220809, 490961785, 264692543, 122501157},
  {1, 3, 1, 1, 3, 17, 17, 163, 339, 691, 1707, 1845, 5941, 4259, 24531, 7647, 82293, 194243, 59855, 932933, 137043, 1928319, 3185669, 9815057, 18052825, 15516225, 36948207, 83248051, 281104093, 390216321, 874432565},
  {1, 1, 1, 1, 27, 21, 85, 221, 71, 949, 1753, 391, 6349, 15901, 20811, 4111, 13223, 21319, 412359, 787421, 303719, 3861587, 5358691, 4089065, 7822351, 4295279, 87215009, 11262439, 238711515, 842300293, 1739353005},
  {1, 1, 1, 5, 31, 19, 45, 99, 469, 783, 1747, 3807, 5889, 9485, 13715, 35167, 33351, 24647, 290243, 491985, 784505, 575543, 2500661, 14916071, 26712169, 57698393, 109470597, 61452979, 220299443, 889118413, 1262415245},
  {1, 3, 1, 9, 23, 21, 39, 25, 421, 713, 461, 2857, 5023, 5341, 6409, 3643, 86061, 233347, 276871, 542337, 1446303, 1161681, 2323587, 8047755, 1477991, 11133515, 37331991, 233516153, 478496967, 443256399, 190482909},
  {1, 3, 7, 5, 25, 19, 59, 147, 387, 857, 375, 3103, 1261, 13697, 25675, 36943, 59269, 166533, 315527, 639375, 2053557, 1445113, 3433513, 11703753, 13278611, 52760393, 38472469, 147526799, 209796519, 971839869, 1292880733},
  {1, 3, 5, 5, 31, 21, 49, 251, 463, 441, 473, 3487, 3915, 11151, 17721, 62363, 40381, 65927, 180867, 964225, 1257663, 3495911, 1052749, 946557, 3683147, 27555607, 82581321, 72100241, 332155877, 775201419, 211124309},
  {1, 1, 3, 9, 15, 47, 81, 219, 143, 141, 81, 1705, 5847, 3437, 30521, 13595, 58479, 166277, 260491, 734865, 1598869, 2856371, 1714245, 2676989, 22088575, 60321783, 17677247, 141694273, 67246823, 668869439, 992624989},
  {1, 1, 7, 3, 25, 19, 97, 41, 77, 105, 1337, 695, 7589, 8587, 7509, 29363, 53551, 65921, 321161, 340355, 1029293, 2231715, 3008551, 15636763, 8440791, 29318495, 104730081, 246287791, 479336605, 55452759, 1143611573},
  {1, 1, 5, 13, 3, 11, 61, 19, 139, 667, 963, 1567, 5715, 7079, 15967, 15127, 127207, 57475, 431747, 302221, 750265, 2560679, 5084869, 5965529, 2078529, 55383129, 45356149, 64196321, 330404861, 598053853, 1801022133},
  {1, 1, 5, 5, 5, 29, 67, 57, 477, 173, 1163, 727, 823, 15635, 17705, 16347, 90271, 80515, 385675, 17051, 559235, 1819117, 1198667, 2059811, 28074867, 58941989, 6939125, 73314357, 458423201, 673674579, 1847445829},
  {1, 3, 7, 11, 13, 39, 57, 193, 73, 617, 535, 1623, 4581, 4451, 2589, 21795, 91085, 216453, 503177, 18567, 455569, 665795, 7181231, 11911599, 24265087, 66930017, 91211357, 19614615, 184930421, 23557839, 1691427093},
  {1, 1, 5, 5, 9, 27, 75, 127, 325, 413, 1669, 1749, 8045, 16199, 12237, 23011, 82799, 113795, 76939, 831639, 492957, 3862745, 2478853, 11856151, 22788459, 19918635, 51811371, 265221203, 337427529, 1027833975, 1594498789},
  {1, 1, 3, 1, 17, 23, 27, 189, 319, 953, 347, 909, 4401, 12791, 25077, 48627, 41615, 80773, 306307, 771743, 1382833, 3051761, 5574487, 16496041, 5656499, 27182421, 123388511, 64308679, 475448097, 997590879, 2135590725},
  {1, 1, 3, 3, 17, 51, 37, 79, 301, 607, 885, 1169, 3275, 3327, 20013, 45011, 100527, 213125, 5761, 634779, 540061, 1613959, 6387529, 10824871, 15812025, 54328831, 105557127, 166257853, 157520557, 598383231, 201162229},
  {1, 3, 5, 3, 21, 9, 99, 213, 387, 889, 575, 3591, 5377, 2981, 23989, 32339, 104285, 51079, 191365, 972935, 55983, 2122405, 4051383, 8120101, 18557099, 52406337, 120364061, 172031603, 493761339, 1008872827, 967442733},
  {1, 3, 3, 13, 11, 7, 23, 255, 279, 853, 453, 2377, 8123, 15393, 9669, 20099, 47101, 33409, 83079, 408989, 1110181, 2163381, 5562701, 11689525, 25256455, 36577459, 66447639, 100923581, 343416051, 1071130091, 351850253},
  {1, 3, 1, 7, 11, 9, 109, 35, 405, 449, 1967, 2943, 3485, 5031, 14273, 18011, 6941, 156163, 326153, 759297, 873987, 3973715, 2250461, 9849447, 3353703, 35246201, 30997797, 44933087, 506109793, 794003707, 1796237637},
  {1, 3, 3, 1, 25, 27, 43, 115, 435, 675, 1937, 1477, 4831, 9417, 7017, 315, 110605, 8705, 164363, 763415, 280109, 2601593, 7370313, 1097721, 7178761, 52202011, 96640619, 145206545, 419228771, 119289735, 299141837},
  {1, 1, 7, 13, 19, 45, 83, 241, 487, 215, 1453, 209, 4061, 1765, 15623, 11191, 90455, 209409, 219655, 826901, 314943, 1291845, 2528995, 3904329, 25934557, 58993055, 100437719, 185240267, 184452123, 3128761, 1269298221},
  {1, 1, 7, 7, 21, 31, 95, 9, 287, 1005, 1933, 3405, 6913, 7733, 18975, 34855, 118839, 125441, 284173, 933383, 1793577, 1566261, 1490635, 8796785, 26878967, 39691819, 133112803, 144470447, 180180739, 652625009, 1002339021},
  {1, 1, 1, 11, 13, 11, 25, 39, 283, 57, 255, 2809, 5767, 6127, 6705, 15883, 9823, 13319, 291853, 56351, 1807421, 1555515, 4404281, 15672625, 14835347, 38273249, 112725987, 207094921, 306724101, 293100531, 1380481493},
  {1, 3, 1, 11, 1, 51, 73, 181, 261, 215, 385, 2777, 5169, 12431, 23563, 16719, 103477, 257027, 453637, 943123, 863261, 2219051, 6544619, 15426891, 14058505, 9523259, 5559943, 37491591, 124006281, 637827537, 233064293},
  {1, 3, 3, 9, 9, 39, 123, 197, 501, 679, 109, 3369, 4817, 8855, 7997, 49955, 69933, 192513, 405507, 512023, 1560593, 2289681, 5329071, 1773971, 103001, 55767799, 38033263, 145340727, 148530065, 665551191, 1484965637},
  {1, 1, 5, 1, 29, 61, 15, 183, 453, 999, 1211, 3217, 8035, 5153, 19975, 49735, 3063, 228099, 454415, 412939, 806659, 3253505, 3722077, 7668631, 24324529, 14441313, 114694523, 132076461, 491841443, 650573513, 591962061},
  {1, 3, 7, 11, 11, 21, 51, 45, 379, 793, 289, 309, 1229, 7159, 581, 45939, 1613, 86277, 522505, 572673, 1848623, 3796789, 1473823, 7391981, 3712091, 25419775, 97472019, 44686115, 495272377, 7900063, 1075306277},
  {1, 1, 3, 9, 17, 11, 75, 67, 289, 191, 1083, 2949, 6063, 6611, 21595, 26591, 60823, 66821, 48395, 858895, 1752333, 3470617, 1816057, 15987083, 21336817, 14062273, 101652495, 36131145, 532366233, 480641601, 980302629},
  {1, 3, 7, 3, 27, 45, 59, 193, 485, 277, 27, 1219, 2389, 15875, 6273, 26459, 69821, 55045, 428289, 707841, 731927, 2498829, 2240259, 15188363, 14435035, 51200677, 128238497, 232414143, 164745893, 635845331, 873639765},
  {1, 1, 5, 3, 31, 29, 65, 197, 115, 163, 9, 783, 5573, 2833, 12603, 29423, 17351, 152835, 368397, 68877, 1944367, 3730183, 4213043, 5220861, 68177, 48586115, 102300305, 214217983, 68725159, 449698061, 1602687453},
  {1, 1, 3, 7, 5, 53, 115, 181, 175, 749, 1335, 1151, 2131, 12467, 15811, 14639, 119991, 86789, 429317, 635143, 1985827, 2831629, 1721735, 15632649, 7882855, 19548497, 130820773, 24656241, 366853353, 758486761, 1855192421},
  {1, 1, 1, 9, 27, 39, 11, 1, 443, 677, 777, 1857, 7459, 3177, 3875, 48271, 3495, 255751, 319247, 5901, 1373493, 3619625, 7548299, 5782677, 5915863, 57583023, 30893767, 150315141, 171411499, 899196029, 415542893},
  {1, 1, 7, 7, 17, 3, 23, 161, 105, 603, 1991, 3845, 465, 11467, 2077, 16291, 81743, 224513, 29965, 673539, 1025853, 1875285, 4616067, 4789367, 24616205, 58520021, 55352039, 223083975, 380681341, 698996175, 769663413},
  {1, 1, 3, 13, 5, 23, 39, 35, 399, 795, 265, 207, 1823, 15069, 31839, 36087, 103479, 18693, 312079, 1044755, 1762601, 2531101, 4839729, 13663317, 16082313, 50066947, 93820849, 142203813, 367805663, 1069373081, 968847421},
  {1, 1, 1, 1, 29, 61, 89, 193, 41, 99, 315, 1021, 6109, 12507, 19973, 7763, 52239, 133671, 322023, 772667, 422883, 495415, 7901671, 1302447, 20921301, 35277497, 737127, 164994475, 108706549, 636734951, 768243093},
  {1, 1, 5, 3, 13, 57, 119, 251, 215, 695, 1521, 4081, 2481, 657, 855, 13847, 88471, 193187, 411309, 7807, 1539663, 1600849, 7457553, 4328157, 15340181, 52433059, 56260463, 222459959, 523917591, 388593297, 932178973},
  {1, 1, 7, 3, 25, 5, 3, 133, 111, 385, 773, 1027, 4327, 3031, 3537, 14859, 11967, 117857, 93545, 737187, 1519835, 1856781, 1778487, 4044233, 3780011, 39253083, 115446317, 102248853, 155152309, 799114499, 1860865813},
  {1, 3, 7, 5, 5, 27, 43, 117, 479, 83, 1421, 2791, 6551, 6231, 10353, 18827, 25149, 166501, 204647, 565171, 1938661, 2724137, 4089711, 11489785, 23400665, 20453239, 131692049, 251398925, 65277253, 1036802083, 795670613},
  {1, 1, 1, 13, 3, 29, 35, 71, 85, 821, 1671, 3057, 797, 13683, 7025, 60411, 88479, 94759, 74987, 501437, 1502415, 1354101, 3340053, 6109367, 919715, 9993177, 122031115, 28444099, 370545389, 696665347, 64637557},
  {1, 3, 7, 1, 1, 47, 117, 233, 141, 993, 1381, 2551, 1031, 11765, 18429, 15635, 122397, 21861, 159267, 515391, 433609, 2540047, 42655, 10903403, 15770431, 33432371, 73426905, 90693661, 499911623, 6581611, 91511773},
  {1, 3, 1, 3, 13, 3, 77, 29, 35, 807, 1109, 695, 5605, 5477, 449, 10363, 47213, 133923, 337965, 45999, 15093, 4101727, 5824107, 14718733, 29192309, 20625335, 98008545, 187909951, 207510867, 949274303, 638703373},
  {1, 1, 1, 15, 21, 37, 117, 105, 273, 311, 1287, 1415, 1221, 1847, 19487, 43911, 75527, 255527, 443497, 860271, 88147, 3058187, 8228471, 7462839, 18279889, 21081293, 109945525, 254556599, 114630337, 55014853, 1265579077},
  {1, 3, 1, 11, 21, 61, 107, 225, 335, 501, 1995, 2399, 5475, 12613, 18877, 19443, 99613, 19683, 449573, 665895, 1175515, 3213413, 4605987, 11462657, 15472783, 2735101, 131353405, 199738897, 317122399, 77532123, 2124149181},
  {1, 3, 3, 1, 31, 41, 27, 205, 103, 837, 639, 2007, 2759, 12471, 1457, 19659, 40893, 143969, 45675, 479921, 783859, 2508821, 7792895, 7186801, 33041183, 32448393, 48803613, 153381877, 191544773, 373937827, 837944277},
  {1, 1, 7, 13, 29, 39, 71, 245, 105, 235, 1277, 1515, 6129, 15947, 26653, 11683, 16495, 158625, 246247, 856123, 1532105, 3688477, 8239175, 16492799, 20822797, 28659503, 60225181, 15554739, 535289445, 247534703, 1247009237},
  {1, 1, 7, 5, 7, 13, 87, 251, 315, 1017, 587, 2917, 5911, 2919, 29715, 38575, 45751, 207457, 90479, 4337, 939383, 2438193, 6208705, 14531825, 15806403, 16522549, 63763895, 150677073, 125275517, 535161697, 1480841269},
  {1, 1, 1, 3, 7, 19, 81, 243, 177, 917, 2023, 2365, 7465, 4901, 29841, 49115, 45359, 39463, 422309, 764197, 1622129, 600715, 2799613, 10306995, 21419163, 34553129, 8891079, 188430447, 51280099, 523547679, 87112781},
  {1, 3, 5, 15, 1, 31, 15, 147, 285, 1003, 1757, 47, 6925, 1197, 19633, 6875, 130157, 58535, 414377, 59179, 1901633, 2526869, 7770897, 10604583, 12497405, 57144415, 52902809, 105844487, 535633387, 395117951, 256990701},
  {1, 1, 5, 7, 27, 25, 47, 209, 85, 403, 1399, 2331, 3663, 595, 13407, 13079, 102055, 87267, 61289, 414433, 813011, 3757457, 6601099, 4483435, 32394077, 21130341, 29385881, 202500037, 281068225, 808366437, 204397381},
  {1, 3, 5, 9, 7, 25, 7, 139, 389, 817, 1153, 1421, 5735, 9577, 10269, 16243, 75005, 180711, 313391, 883361, 351571, 2708105, 1453313, 10977903, 29568823, 25925015, 55892587, 92152409, 158133619, 805511347, 734229773},
  {1, 1, 1, 9, 5, 61, 49, 117, 389, 541, 433, 1405, 2575, 223, 7265, 48587, 111295, 175335, 236271, 459059, 509555, 3504319, 5360899, 1042251, 12143011, 13286143, 40666763, 51217745, 4033365, 619097731, 2079541909},
  {1, 1, 5, 7, 15, 1, 81, 207, 435, 843, 835, 3797, 7637, 5333, 31115, 24335, 43207, 22947, 410025, 793717, 1035491, 30095, 243465, 6370121, 18449073, 17138809, 130975135, 95322451, 151954955, 674862677, 1263235437},
  {1, 3, 7, 11, 13, 3, 47, 249, 301, 715, 2015, 3049, 8155, 10989, 26051, 48303, 32037, 231525, 461481, 784743, 972501, 3994045, 2737931, 11985475, 25010837, 51896669, 13511131, 63240469, 355487647, 291514581, 1000375357},
  {1, 1, 7, 7, 3, 33, 119, 113, 381, 575, 367, 41, 3317, 11727, 4351, 44711, 115655, 33441, 189933, 453617, 176347, 3867865, 5253179, 4114915, 7002913, 53534597, 95573291, 117002875, 287855841, 226906261, 11668677},
  {1, 3, 3, 13, 9, 3, 51, 37, 173, 137, 533, 1827, 631, 10047, 6267, 23647, 120789, 55009, 53223, 878463, 1885125, 3887793, 1492079, 4801739, 7476903, 42508563, 110079757, 258941317, 61779517, 736932665, 1814702645},
  {1, 3, 3, 11, 17, 39, 61, 245, 13, 139, 1281, 1319, 1233, 13629, 32269, 51523, 43037, 77857, 336289, 938923, 1832521, 2904471, 7089715, 10985267, 11706405, 10872579, 86633961, 154726139, 331941987, 1014317459, 544571341},
  {1, 1, 1, 7, 15, 17, 91, 109, 163, 609, 11, 3251, 7653, 14035, 31755, 46063, 56567, 97959, 509025, 713573, 2058227, 3451045, 2668159, 5424949, 33057843, 4258341, 95490945, 181060943, 12371829, 131711577, 1775460885},
  {1, 3, 3, 15, 13, 21, 55, 231, 385, 133, 1833, 2637, 6935, 14303, 26745, 7227, 108509, 115937, 478693, 901439, 1773907, 1255817, 3184381, 9439795, 1297915, 59257927, 62839083, 25029257, 367398053, 992093947, 884653397},
  {1, 1, 1, 7, 17, 41, 125, 141, 89, 823, 1411, 3637, 6211, 13323, 6111, 30999, 42087, 194151, 13153, 866811, 1535223, 2209419, 2903603, 2274967, 3409105, 17647209, 126927503, 126009925, 527946937, 46530277, 1033456037},
  {1, 1, 1, 11, 1, 21, 9, 43, 97, 685, 1223, 1491, 121, 1793, 2397, 40931, 76799, 95975, 72045, 432371, 537115, 992775, 482893, 5969779, 284899, 58210489, 94539209, 220431267, 425563807, 944480539, 106992573},
  {1, 3, 5, 5, 17, 17, 5, 223, 129, 865, 1839, 1137, 6391, 4377, 9233, 299, 74765, 260007, 462307, 784751, 1417735, 667139, 4092465, 811915, 4560191, 17849289, 92178699, 237080905, 290635539, 899860447, 1878765197},
  {1, 3, 7, 15, 21, 55, 5, 77, 341, 637, 1853, 1435, 1195, 9283, 21257, 17947, 1213, 158373, 31661, 683767, 1289729, 3362399, 5942907, 12311099, 19758403, 26927731, 9756853, 27506889, 344319961, 444882051, 1031037093},
  {1, 3, 5, 1, 9, 49, 43, 211, 127, 655, 1397, 1235, 5279, 2351, 24229, 40931, 85613, 143911, 417063, 389759, 1839111, 3379213, 1323233, 2036053, 8065053, 19175351, 23277925, 25559693, 283249853, 73539431, 1704114357},
  {1, 3, 5, 3, 25, 29, 13, 229, 25, 675, 837, 2753, 2125, 9863, 11293, 48163, 113613, 214247, 433317, 376171, 1249283, 3715123, 7000107, 2920839, 10048037, 31913415, 4020071, 200326267, 291720761, 962389063, 1958200997},
  {1, 3, 5, 7, 23, 43, 127, 1, 163, 237, 337, 3019, 7747, 16227, 2881, 36683, 63133, 214439, 27553, 859629, 1652793, 3274773, 7907571, 8407485, 3574607, 29892455, 125660477, 207828145, 74084521, 1039173931, 389355109},
  {1, 3, 5, 5, 25, 9, 43, 171, 421, 521, 1885, 337, 7873, 6347, 13181, 60931, 16685, 249703, 321059, 46695, 1997839, 3009597, 5393529, 1762559, 4287695, 32252155, 34959023, 128163959, 347889837, 986315983, 648070261},
  {1, 3, 5, 5, 7, 47, 107, 173, 163, 191, 625, 3389, 2833, 7945, 24787, 50719, 40325, 171879, 333667, 788025, 231701, 2466121, 1055591, 4290549, 17575597, 56269667, 10812739, 135805967, 37168319, 561468517, 1951218365},
  {1, 1, 7, 3, 27, 57, 27, 209, 253, 815, 301, 1633, 3945, 5051, 28851, 41823, 112535, 158561, 59177, 278177, 119075, 4035365, 654307, 2152019, 31591921, 41400999, 51419879, 150717503, 107601997, 878748289, 1029309045},
  {1, 3, 7, 9, 9, 51, 103, 213, 437, 189, 1857, 1331, 5551, 10641, 27405, 10803, 102717, 71461, 10219, 361447, 1201957, 1935685, 2508087, 6601635, 19694755, 28777359, 17163941, 58320749, 50937251, 277836851, 1718603085},
  {1, 1, 5, 5, 15, 1, 25, 105, 117, 347, 161, 3369, 3589, 12903, 23559, 46423, 66983, 211171, 9131, 779057, 1306923, 1233671, 943103, 11631459, 23579381, 44723427, 71216423, 221132531, 109577065, 364394685, 374887397},
  {1, 1, 1, 5, 3, 61, 93, 51, 81, 281, 1383, 745, 4137, 2005, 3635, 28191, 119271, 259431, 285411, 1013037, 677135, 1167627, 3487581, 6433507, 2984471, 16693937, 56645859, 214914823, 364037379, 214023453, 1294450893},
  {1, 3, 7, 5, 13, 57, 111, 211, 303, 477, 359, 4019, 6779, 5129, 22035, 49775, 79237, 256933, 350183, 550971, 110871, 4137801, 1203465, 9051285, 17786479, 13451841, 96554729, 379873, 519722707, 86521677, 422110605},
  {1, 1, 1, 7, 17, 29, 113, 113, 201, 531, 749, 739, 2879, 3315, 18733, 32755, 104367, 34599, 313761, 934267, 2014467, 2380079, 509895, 6042639, 339301, 21013327, 42148437, 93847149, 432908577, 588950359, 1191519941},
  {1, 3, 7, 13, 21, 55, 21, 183, 359, 75, 377, 2211, 4281, 14317, 28307, 39487, 80805, 45029, 402415, 670003, 1520393, 3042127, 7480737, 7492861, 31485433, 31922579, 78475485, 245806075, 417117171, 183497677, 302630029},
  {1, 3, 7, 1, 21, 1, 49, 213, 317, 75, 113, 1741, 7963, 12785, 11571, 3087, 50757, 152229, 84067, 587179, 1175439, 2892935, 5464275, 15999475, 30122053, 48325811, 108014499, 127691109, 192138195, 858190173, 1592697101},
  {1, 3, 7, 9, 11, 31, 29, 101, 261, 141, 715, 2727, 8187, 2075, 32433, 12395, 57117, 6437, 169003, 318309, 1704589, 2602991, 5177411, 14876955, 8061427, 22216869, 84216613, 180683449, 114897745, 70333003, 1345009285},
  {1, 3, 7, 3, 23, 9, 17, 143, 385, 211, 593, 241, 6567, 10777, 6677, 20579, 57757, 47653, 7265, 52589, 290699, 1005503, 2373865, 4802875, 22960269, 19406163, 52373183, 254196833, 422607411, 375377243, 858807949},
  {1, 1, 3, 15, 3, 17, 117, 99, 91, 793, 989, 2421, 5643, 16103, 9759, 62295, 64583, 68773, 61101, 179633, 1301643, 2018555, 6645805, 3020841, 21527491, 50706723, 29390219, 247723893, 74116805, 1032386909, 2006211285},
  {1, 3, 7, 11, 23, 43, 107, 35, 421, 431, 743, 853, 7939, 12169, 4199, 23591, 34005, 237733, 61801, 834877, 1953673, 3052161, 5409849, 13345967, 1515273, 15211469, 12767943, 206706645, 152451579, 699556633, 1564403885},
  {1, 1, 1, 11, 21, 53, 17, 203, 123, 395, 59, 929, 255, 7585, 10945, 35403, 42767, 165415, 30573, 697319, 525715, 579471, 4169373, 3941961, 25686641, 3281889, 101982043, 253534897, 235458363, 555375775, 618553773},
  {1, 3, 3, 15, 17, 57, 57, 133, 67, 71, 1685, 903, 4079, 15423, 26495, 36119, 36101, 210593, 286629, 849827, 1770119, 3144431, 1228531, 1392589, 28143477, 15060407, 115376657, 244487317, 259440413, 453420253, 1633869365},
  {1, 1, 1, 15, 3, 47, 95, 39, 405, 407, 1557, 3001, 6225, 15187, 5663, 10951, 4583, 241383, 123561, 556985, 210613, 2343597, 4982405, 13524935, 27323585, 4632335, 7136687, 238908951, 343370293, 278227613, 1618844053},
  {1, 3, 5, 5, 13, 47, 33, 61, 375, 1023, 1981, 2773, 2375, 11321, 17731, 51215, 76229, 224039, 472163, 73139, 1480065, 201387, 776931, 13745833, 9622725, 12751103, 61292275, 169868345, 299988355, 592790349, 1216835661},
  {1, 3, 5, 9, 17, 59, 117, 95, 493, 149, 1269, 2865, 369, 2109, 24601, 57035, 117645, 114151, 91759, 685559, 680093, 1942183, 7566201, 2205991, 11225491, 25586827, 39548847, 38128551, 377668155, 539256199, 2141507885},
  {1, 3, 5, 13, 17, 63, 67, 247, 95, 721, 67, 305, 6179, 15399, 32559, 40695, 74117, 134823, 89515, 582335, 106377, 1430425, 4683693, 6872989, 14445587, 64326397, 68995347, 178642221, 207610821, 781293389, 108393173},
  {1, 1, 5, 1, 3, 21, 41, 15, 453, 475, 2017, 3193, 5903, 897, 4237, 61331, 42559, 131619, 27759, 164213, 755895, 376495, 4509993, 11581151, 31468429, 8162915, 103002999, 196297241, 264657755, 271838291, 151538477},
  {1, 1, 5, 3, 15, 41, 1, 141, 441, 575, 155, 3791, 7711, 11231, 24611, 44859, 73467, 180511, 244177, 112593, 1657039, 1543899, 6487559, 13141723, 19117973, 43726045, 22608161, 220561193, 315519241, 181812885, 1320572469},
  {1, 3, 7, 1, 17, 53, 27, 169, 31, 437, 963, 1793, 7777, 1917, 29311, 18699, 11241, 246777, 325719, 708443, 978911, 627401, 7424583, 11431861, 28673987, 38306449, 26709295, 238919291, 245088055, 939427325, 1276522029},
  {1, 3, 3, 13, 9, 27, 77, 87, 329, 885, 749, 1713, 6013, 6921, 629, 44239, 40009, 8109, 12915, 697947, 194913, 2975715, 8156693, 6528751, 9165567, 27293871, 104947731, 91220455, 200510603, 1004660991, 1397240605},
  {1, 3, 5, 13, 3, 7, 53, 27, 353, 267, 925, 2141, 439, 15175, 30851, 38427, 104505, 133659, 443103, 292817, 982745, 4140227, 4689145, 216479, 22937553, 40474691, 110842271, 135224709, 333202481, 817386741, 1008208181},
  {1, 3, 3, 13, 17, 57, 35, 101, 265, 901, 1825, 2159, 6149, 5967, 24023, 12435, 101497, 39101, 64147, 869827, 880627, 3101385, 4674619, 7044187, 2104799, 34246643, 38775541, 127411235, 412132061, 517668577, 1974215717},
  {1, 1, 5, 11, 13, 51, 99, 111, 193, 415, 1541, 2929, 5045, 3147, 12587, 11643, 108635, 241503, 469465, 670915, 1209553, 3197133, 4140209, 3306647, 29049973, 34235059, 117615043, 185020671, 373433529, 17432997, 280366805},
  {1, 3, 7, 11, 15, 9, 33, 17, 511, 815, 299, 1077, 6171, 10451, 15039, 14387, 50329, 134137, 163741, 594641, 1922527, 1141467, 2232539, 6320421, 29904905, 28873645, 86236491, 65767993, 240355213, 771350337, 2003815045},
  {1, 1, 1, 15, 25, 63, 51, 137, 449, 951, 1051, 1101, 4725, 2463, 7355, 50699, 53243, 90331, 439637, 971999, 506325, 2584549, 4150039, 140155, 4424221, 20285133, 118391971, 47353607, 249004061, 807769501, 1074612805},
  {1, 1, 1, 7, 27, 63, 29, 179, 317, 521, 1459, 827, 6599, 13459, 15439, 5587, 41883, 173435, 316317, 957901, 863953, 3556747, 119777, 11542483, 28124459, 49854377, 121731525, 215393769, 179834197, 510148161, 1686677765},
  {1, 3, 3, 15, 17, 31, 37, 191, 229, 245, 181, 941, 5761, 1849, 31599, 8507, 42697, 223357, 332497, 546119, 1613013, 969107, 1860557, 16299571, 8969895, 43912247, 4836447, 23709155, 402408579, 283271909, 1376966653},
  {1, 1, 1, 9, 27, 45, 67, 239, 481, 615, 1667, 3751, 8141, 10013, 2125, 61423, 105451, 191339, 140915, 619089, 1516611, 292993, 1376737, 996135, 26211501, 19168325, 50520985, 196930795, 536119215, 758696383, 1807820589},
  {1, 1, 1, 1, 13, 51, 117, 135, 73, 151, 1291, 2541, 1411, 3767, 26949, 9559, 2235, 106795, 107963, 922839, 362097, 347851, 3788757, 1350095, 33175901, 2338189, 6830723, 48777137, 506385625, 202053427, 1970404181},
  {1, 3, 1, 9, 7, 11, 21, 187, 243, 857, 1951, 865, 7273, 2041, 8155, 59731, 37065, 177119, 520595, 854877, 1184765, 2750875, 1768825, 16486765, 12981095, 26481125, 62470199, 192788627, 480512259, 21788305, 1471981437},
  {1, 1, 3, 3, 19, 33, 89, 115, 455, 137, 707, 1867, 4221, 2433, 9119, 699, 20363, 29817, 258909, 494029, 1351247, 676979, 8260649, 1117657, 27563807, 60110105, 31020933, 261606547, 429946675, 601254709, 687236253},
  {1, 1, 3, 11, 5, 3, 121, 1, 71, 951, 603, 3873, 723, 3285, 19289, 63079, 68139, 233545, 510133, 190923, 469953, 2616407, 1881131, 2071757, 15056385, 7981037, 25185663, 206100249, 158911643, 213470091, 1986864541},
  {1, 3, 7, 15, 21, 1, 117, 17, 455, 519, 731, 3003, 5741, 9557, 29163, 47363, 125481, 12761, 131737, 217283, 513379, 3713087, 4089955, 11458941, 5405169, 51907661, 23898469, 166946835, 288252775, 1057448681, 881417165},
  {1, 1, 3, 13, 25, 5, 43, 147, 209, 895, 255, 1231, 241, 487, 15593, 4303, 89019, 195017, 13555, 1030235, 1662719, 3258937, 4970141, 10566287, 8439445, 21600361, 51806665, 247040839, 63736525, 465675503, 7403429},
  {1, 1, 3, 13, 7, 1, 89, 187, 217, 927, 2029, 3521, 2777, 8103, 22819, 25179, 35259, 61081, 54611, 783941, 841799, 3136067, 280225, 15437591, 5541301, 39415643, 66512163, 113928307, 520446029, 359225077, 2138116869},
  {1, 1, 7, 11, 7, 33, 3, 73, 5, 489, 227, 2259, 7031, 6425, 26135, 5323, 24395, 200253, 463581, 903369, 342087, 4075049, 3338887, 11005487, 1496875, 46454409, 31801325, 186744585, 28177983, 927650765, 598402605},
  {1, 3, 3, 7, 31, 19, 97, 201, 455, 819, 945, 2771, 8083, 8711, 2835, 55227, 98489, 151837, 5977, 51545, 1057349, 4134415, 8265283, 1185789, 24200517, 32637007, 96600537, 195433885, 522855385, 531761493, 304796405},
  {1, 1, 1, 5, 15, 45, 43, 157, 245, 967, 877, 2289, 4499, 9891, 18827, 28955, 119195, 28635, 219999, 1025629, 1230443, 1266185, 5577379, 14731799, 21031653, 57715579, 123740435, 87570905, 57160669, 555227861, 324542021},
  {1, 3, 1, 7, 21, 59, 123, 63, 231, 485, 1781, 1211, 4597, 5269, 1607, 36699, 118569, 25535, 137309, 578259, 316521, 2072165, 5592785, 8249937, 16495491, 29257783, 61624665, 233756955, 112741191, 896170533, 1034582157},
  {1, 1, 1, 13, 23, 39, 105, 55, 249, 991, 1625, 3089, 3825, 4275, 29139, 3819, 36891, 113691, 30167, 493141, 405329, 119071, 1258957, 14273951, 9063765, 6860303, 114126043, 160261883, 384988329, 629036781, 1763368725},
  {1, 3, 3, 1, 29, 29, 55, 169, 13, 895, 1355, 1101, 6063, 12935, 23215, 5827, 4025, 130301, 56479, 716375, 548943, 1077109, 4208639, 109479, 10924621, 27637709, 16299107, 252405469, 141614385, 578541369, 1526035189},
  {1, 1, 5, 5, 31, 49, 99, 137, 209, 1017, 1179, 3931, 637, 14131, 19285, 39399, 81179, 73519, 268983, 249421, 2013431, 3094329, 190887, 267259, 12149779, 13274577, 86431541, 96137171, 144312769, 777389851, 995267669},
  {1, 1, 1, 1, 3, 11, 119, 11, 215, 337, 243, 3883, 3807, 7335, 11901, 39463, 86395, 237739, 37755, 339353, 1747349, 2411769, 7143789, 8436521, 31493239, 48247545, 56532549, 25634801, 62673169, 986905051, 1848578069},
  {1, 3, 7, 3, 7, 27, 71, 225, 219, 367, 1213, 2739, 1185, 10175, 21313, 60351, 40249, 12361, 115765, 126601, 1791389, 1795737, 1187303, 10881841, 11863505, 20380059, 22864257, 146358159, 185139045, 972171255, 342799749},
  {1, 3, 7, 13, 7, 49, 23, 223, 61, 1011, 797, 1335, 6711, 5961, 5605, 3631, 37001, 256105, 305979, 207253, 45751, 2544877, 6512025, 1611515, 33538721, 61621491, 115298197, 231917969, 75427967, 1011644831, 929859757},
  {1, 3, 3, 11, 19, 37, 1, 149, 39, 661, 929, 2125, 2299, 5181, 28083, 41683, 46889, 141917, 328405, 348045, 873643, 1810339, 303935, 15270805, 14522371, 58362547, 87747559, 135635257, 179243539, 79234097, 1525663293},
  {1, 3, 3, 13, 13, 9, 67, 21, 463, 279, 529, 523, 6705, 11011, 31695, 30803, 102873, 130621, 440659, 690335, 1257915, 4085641, 2783659, 8396285, 4050561, 28630311, 13334145, 130263023, 162400793, 99244289, 1566470261},
  {1, 3, 1, 5, 13, 1, 123, 3, 291, 625, 1949, 2713, 5917, 10343, 13627, 37291, 108281, 105247, 186527, 1011343, 119715, 687553, 5535053, 14310781, 26029183, 32009047, 7482123, 239153687, 128785701, 284642925, 2003514405},
  {1, 1, 3, 9, 27, 41, 3, 207, 103, 265, 811, 549, 6109, 313, 8889, 471, 45451, 179977, 434551, 139921, 1255575, 3612841, 845569, 4176001, 23534543, 3889393, 39462715, 30947887, 522844107, 522080211, 1121377437},
  {1, 3, 3, 13, 23, 43, 99, 33, 279, 463, 955, 793, 4113, 10615, 16957, 53351, 8441, 82989, 30195, 967813, 952237, 977645, 960271, 7131645, 16470217, 28291741, 36968519, 10369963, 177095421, 482267179, 320061317},
  {1, 1, 5, 7, 11, 49, 79, 45, 17, 937, 359, 1037, 1099, 3003, 31561, 16191, 113563, 180623, 440501, 849053, 307871, 1795013, 5374491, 6469619, 13496931, 30896685, 75291419, 147109513, 309490437, 250584743, 1925590437},
  {1, 1, 1, 7, 3, 45, 111, 35, 109, 983, 53, 4057, 7349, 3599, 2209, 22463, 95675, 228555, 40765, 724885, 2086579, 781213, 4856341, 15233011, 32974885, 45742579, 120841579, 242887663, 83481149, 593590951, 1170984517},
  {1, 3, 7, 11, 9, 43, 27, 9, 85, 529, 1497, 347, 759, 12449, 11373, 36639, 107721, 260649, 334781, 881815, 398257, 2255845, 536135, 8113679, 12360019, 15342467, 129927761, 135958041, 117021975, 617713895, 2006450509},
  {1, 1, 3, 9, 17, 1, 49, 31, 367, 813, 1385, 2025, 773, 4679, 4543, 14883, 17403, 214073, 201687, 818075, 456363, 2615499, 5693157, 2600649, 4413419, 35570267, 90557747, 195255759, 352080725, 273110377, 868216421},
  {1, 1, 5, 15, 15, 9, 43, 97, 239, 995, 1037, 841, 4167, 12113, 23765, 35119, 4427, 90991, 512829, 977545, 1404847, 2276241, 4375103, 14158357, 31162229, 685043, 97317819, 52222397, 168883719, 760632943, 783954861},
  {1, 3, 5, 9, 29, 53, 123, 49, 221, 113, 1157, 73, 6087, 1363, 11029, 2599, 120921, 85867, 159611, 248455, 1068695, 555401, 861759, 9284423, 26070467, 5715167, 96162875, 14710157, 335526501, 976280875, 519641637},
  {1, 3, 1, 13, 3, 15, 69, 199, 279, 919, 5, 161, 4817, 15031, 121, 23791, 37497, 101391, 292023, 41361, 107185, 2503587, 7648773, 5696337, 30930401, 66832835, 128938979, 154914779, 302534349, 266189263, 695497989},
  {1, 3, 1, 9, 3, 31, 117, 77, 393, 241, 645, 3181, 1067, 15879, 2037, 60599, 128249, 9071, 96499, 334489, 1923745, 1647507, 6777007, 8738907, 10238267, 14920415, 65648767, 132891753, 298282985, 1025216067, 961546325},
  {1, 3, 3, 15, 3, 63, 57, 33, 117, 789, 941, 1301, 5865, 12693, 3523, 45971, 5225, 29917, 126289, 600981, 612753, 652943, 520251, 6647935, 24752199, 34992287, 62591607, 230870203, 42294527, 991829009, 676742285},
  {1, 1, 5, 13, 3, 61, 51, 151, 175, 305, 95, 1557, 6567, 7841, 13903, 48603, 97739, 251455, 387103, 926465, 226563, 3180401, 3069241, 10857817, 11962151, 32720053, 52675491, 176046581, 153220359, 590397781, 944299565},
  {1, 3, 3, 5, 15, 25, 127, 79, 245, 767, 645, 3933, 1357, 12579, 4067, 58891, 91481, 114525, 60955, 648973, 1074991, 1214805, 1291673, 14736723, 27123325, 59481587, 18143247, 152717407, 245720417, 139404317, 19916149},
  {1, 3, 5, 11, 21, 31, 13, 251, 127, 231, 1795, 2627, 1191, 3363, 23543, 28963, 4825, 218235, 176089, 151819, 1418541, 772411, 6079993, 8685857, 21424769, 22154437, 79293197, 15826409, 427418885, 233608857, 2076374213},
  {1, 1, 3, 5, 7, 49, 121, 57, 131, 481, 1879, 525, 5225, 337, 1957, 10703, 25163, 202089, 304443, 383253, 247095, 2733315, 2830755, 13212169, 7540543, 61048781, 42896427, 49059403, 73442859, 883615519, 600779293},
  {1, 1, 5, 13, 9, 55, 27, 37, 211, 125, 119, 3373, 251, 12357, 13975, 61691, 65259, 113919, 89119, 32011, 1493789, 3839309, 2942859, 7940785, 30431603, 62631701, 59794955, 69840761, 7117019, 324109573, 106116061},
  {1, 3, 3, 15, 1, 51, 91, 119, 233, 993, 203, 1635, 1167, 6327, 29119, 38755, 51321, 237117, 231889, 586519, 1539865, 3147637, 2142009, 15643279, 27683547, 3615745, 57362125, 78601937, 363675873, 1056864889, 806240085},
  {1, 1, 7, 1, 13, 5, 23, 253, 121, 989, 1105, 3321, 3221, 6073, 21185, 17063, 121483, 161613, 265079, 217367, 482615, 627045, 3680683, 16297595, 20897991, 65233011, 92137283, 39838495, 159854283, 426992235, 718503709},
  {1, 1, 3, 15, 13, 49, 121, 247, 247, 133, 485, 1067, 7875, 411, 7647, 32787, 2971, 23097, 191057, 606219, 606243, 3326035, 6996125, 8929345, 17580467, 7717063, 54251305, 36429261, 308214629, 60678625, 1180421893},
  {1, 3, 7, 13, 31, 37, 127, 241, 145, 133, 53, 267, 2029, 3703, 16123, 25835, 99577, 98073, 224795, 2061, 763923, 1910829, 3004455, 8448427, 6961167, 62019911, 85316345, 171458755, 219168853, 67662109, 410574533},
  {1, 3, 1, 15, 15, 9, 15, 89, 35, 367, 401, 61, 1953, 7873, 17861, 3023, 92233, 73583, 41013, 116761, 1935407, 866421, 6764751, 4462633, 8608193, 47872375, 112422807, 225179283, 64463867, 72218815, 444360349},
  {1, 1, 1, 1, 1, 41, 71, 249, 213, 779, 1385, 1767, 999, 15151, 16647, 27635, 13627, 22523, 285787, 107547, 2006067, 1487885, 6227055, 517247, 21186057, 31806663, 96850189, 128028449, 148484889, 147702513, 858238133},
  {1, 3, 7, 13, 31, 23, 123, 235, 343, 949, 309, 3777, 3587, 853, 19779, 15891, 81385, 84057, 136795, 611341, 1528865, 4097101, 2460725, 6215113, 5738419, 62573239, 80686243, 144244569, 341557675, 173644465, 877739165},
  {1, 1, 3, 13, 29, 35, 5, 37, 63, 757, 303, 1579, 3443, 243, 11873, 4815, 104539, 142793, 489139, 191007, 133649, 911899, 5130775, 44733, 25148051, 33304509, 76896733, 103266173, 121794109, 728363567, 780981541},
  {1, 3, 1, 9, 19, 49, 81, 53, 11, 901, 1857, 147, 3103, 14019, 21, 57575, 51801, 88175, 419059, 708105, 657967, 948843, 7561887, 8511193, 26574859, 9011955, 3968793, 18908065, 465734309, 775814939, 196258565},
  {1, 3, 7, 13, 3, 39, 99, 99, 45, 91, 1567, 551, 3129, 4809, 29057, 51559, 123273, 79305, 308603, 609809, 843305, 2982453, 3908205, 8139747, 19406697, 40944001, 48188225, 136531583, 321867251, 976922331, 1235842013},
  {1, 3, 7, 3, 3, 27, 17, 231, 377, 381, 1479, 2525, 2595, 2799, 25737, 8303, 9913, 152329, 297909, 36365, 571925, 3622479, 1892077, 3543647, 31579319, 28865301, 106501611, 184709089, 14105921, 899893711, 1156430773},
  {1, 3, 5, 15, 15, 25, 103, 215, 301, 59, 1417, 981, 7579, 12731, 22329, 45503, 89401, 116715, 484445, 657465, 204543, 631389, 4610801, 12475931, 7723241, 38915031, 67499135, 3267609, 125268757, 194700375, 1371109285},
  {1, 1, 1, 13, 5, 31, 61, 31, 349, 925, 1301, 685, 435, 11567, 10715, 37291, 89499, 122875, 24695, 301159, 1947757, 4079131, 2967293, 12181035, 22363359, 41483687, 114261503, 211773297, 215045457, 83304797, 1866677941},
  {1, 1, 7, 9, 19, 57, 109, 1, 37, 377, 1015, 2273, 6741, 3191, 15949, 63911, 107611, 245389, 49311, 445625, 2014679, 489543, 2410611, 11298879, 33094123, 14920765, 102954455, 140738927, 203856917, 527024315, 1813410085},
  {1, 3, 3, 13, 3, 23, 103, 127, 11, 59, 1847, 1175, 425, 3423, 20643, 13531, 36825, 92157, 189811, 509809, 1607257, 2814001, 7024857, 13578525, 17027397, 64663785, 44495985, 104591311, 29489829, 774473845, 1544931717},
  {1, 3, 3, 7, 3, 11, 105, 141, 55, 217, 1427, 477, 667, 9403, 11905, 45967, 10425, 26157, 18393, 464165, 1049285, 2184279, 6229143, 13389157, 16443119, 44997497, 5425267, 62053241, 199516453, 131060431, 552212389},
  {1, 3, 3, 5, 3, 27, 11, 187, 495, 907, 1925, 445, 6639, 8159, 15225, 13103, 63001, 229229, 468827, 961057, 2020053, 2896901, 4226117, 12696593, 28353693, 60815963, 30386655, 251034013, 96933817, 902250463, 675387157},
  {1, 3, 1, 5, 27, 31, 77, 213, 73, 343, 1123, 3609, 2431, 15329, 32165, 59055, 24297, 157327, 239775, 323001, 27377, 195691, 1225767, 7732299, 7019645, 7411093, 113335207, 141289093, 330934039, 48339007, 47505965},
  {1, 1, 7, 5, 1, 11, 105, 139, 485, 1007, 709, 3509, 5231, 11717, 31433, 61735, 31435, 127981, 430419, 368947, 1027265, 2318663, 304545, 910043, 8822205, 47425523, 102920007, 122778125, 285052579, 1026670363, 564841213},
  {1, 1, 3, 15, 23, 49, 95, 169, 399, 1019, 19, 2013, 5311, 7951, 22609, 24575, 107035, 12073, 519377, 281137, 999415, 721685, 1910143, 14047749, 32933677, 62372285, 120103859, 3044301, 133433849, 701266999, 708100309},
  {1, 3, 1, 7, 13, 3, 29, 203, 209, 701, 1791, 2615, 5351, 4237, 12565, 15615, 4681, 65167, 238237, 770731, 16321, 2894099, 2208009, 291375, 13104807, 58809389, 32427921, 181164369, 385499275, 989502119, 615513917},
  {1, 3, 1, 15, 27, 11, 91, 31, 205, 205, 1683, 901, 5129, 6049, 11865, 18263, 66153, 56751, 163285, 47917, 1620709, 531717, 2360673, 5008315, 16152223, 47697633, 115456411, 17712539, 492593307, 17291235, 328765405},
  {1, 1, 7, 7, 27, 59, 21, 3, 209, 79, 769, 4013, 2041, 2645, 11561, 31783, 113739, 111661, 464593, 936237, 1650885, 175435, 6574369, 1916575, 21853013, 800375, 55147127, 21829611, 207167167, 422926939, 1007274381},
  {1, 3, 7, 11, 5, 45, 39, 243, 185, 871, 795, 1845, 8043, 6285, 20991, 5547, 114441, 261849, 352061, 205819, 1464655, 701813, 7993093, 15946695, 1966749, 56231389, 90571099, 210633993, 302122003, 29251421, 809646589},
  {1, 1, 5, 7, 13, 7, 15, 165, 349, 179, 789, 1269, 3787, 5429, 26567, 62203, 123723, 60703, 438965, 911099, 911685, 4027145, 2743619, 8406111, 7204705, 27852043, 93624647, 87392521, 390773467, 282315125, 1882024733},
  {1, 3, 3, 13, 31, 23, 75, 41, 177, 735, 1889, 4039, 3519, 15003, 965, 36551, 121209, 37645, 204691, 97965, 580321, 1103133, 6234935, 6258123, 23447349, 2231559, 92050365, 144679737, 336052349, 341067387, 1719975461},
  {1, 3, 1, 7, 15, 57, 15, 139, 27, 469, 1003, 691, 7893, 9643, 30983, 37059, 125689, 97119, 400701, 237289, 747135, 3145077, 4461949, 16462229, 8212779, 11186441, 62944685, 131873035, 318739113, 235264537, 541322357},
  {1, 3, 1, 13, 23, 27, 3, 237, 233, 875, 99, 883, 6167, 5463, 6245, 9975, 62745, 46607, 506775, 180901, 336109, 1987949, 6960995, 9846907, 16073137, 7685709, 106185101, 93733209, 443215281, 118963011, 1126504981},
  {1, 1, 5, 13, 25, 57, 79, 51, 147, 619, 1147, 279, 6583, 1939, 477, 6935, 58939, 93007, 128927, 549435, 1634643, 2556901, 3170517, 6447309, 26730565, 54256165, 102720873, 127576273, 188475573, 620384723, 122845253},
  {1, 3, 5, 5, 31, 61, 125, 163, 213, 699, 583, 3865, 615, 9707, 11651, 63179, 27961, 108091, 405879, 504317, 1831675, 2923743, 3291153, 12134891, 10464009, 7876377, 14409839, 91875709, 42654797, 222593357, 709942213},
  {1, 1, 5, 1, 5, 21, 93, 239, 31, 641, 777, 27, 5247, 8993, 21053, 48735, 20395, 5455, 398739, 380479, 1943879, 2726351, 3676173, 4110553, 32656231, 41955267, 14564449, 151263633, 52696359, 929848407, 788287117},
  {1, 3, 7, 9, 1, 13, 61, 57, 503, 453, 83, 3271, 2845, 1121, 18639, 46587, 95913, 189593, 512063, 721275, 152039, 2963647, 5782651, 6937757, 21604083, 55427233, 69421097, 50614571, 294110899, 583712485, 754336861},
  {1, 1, 7, 5, 29, 53, 13, 219, 379, 441, 821, 3179, 4877, 2535, 7557, 15287, 101531, 116941, 286355, 742447, 1870663, 3363807, 2909273, 11808085, 8373527, 18065711, 83964425, 87906051, 234671845, 424490531, 729532165},
  {1, 1, 7, 13, 9, 53, 17, 183, 265, 393, 587, 2753, 6453, 7135, 24737, 21727, 81563, 205933, 25307, 170923, 32111, 936067, 6858541, 11216063, 13601955, 17605777, 77721487, 240790863, 300008989, 152131655, 1667996805},
  {1, 1, 1, 13, 11, 23, 73, 109, 393, 1013, 559, 755, 7291, 6631, 26509, 48359, 86555, 168523, 462487, 295721, 761977, 128767, 1663895, 10722619, 12727423, 47159709, 71590997, 160771525, 317606617, 930653771, 473254453},
  {1, 3, 1, 5, 5, 15, 107, 103, 355, 307, 1559, 837, 5413, 5285, 17489, 21079, 43977, 81647, 204767, 570791, 1774941, 2358509, 2129833, 9089509, 22475325, 62059225, 10337059, 136530087, 99246663, 667148387, 703860525},
  {1, 1, 5, 7, 17, 21, 21, 23, 109, 709, 1947, 3585, 3629, 4669, 949, 11711, 4939, 161167, 281301, 1022567, 206127, 3540343, 7986533, 16414235, 5292743, 46286425, 126438543, 29450039, 272065615, 274565591, 1953543629},
  {1, 3, 7, 1, 9, 33, 85, 147, 467, 259, 1913, 199, 7399, 9551, 22387, 36923, 93977, 196601, 270263, 61731, 1778459, 1438479, 7950209, 15066701, 15431229, 15021463, 63893565, 198212897, 414018073, 309407525, 1153284021},
  {1, 3, 5, 11, 15, 53, 23, 41, 249, 515, 1161, 2467, 1299, 7449, 2613, 60911, 90409, 160587, 383449, 961137, 143123, 56597, 1728879, 11146131, 931065, 19421463, 73466377, 25893373, 489366375, 561081583, 1179480333},
  {1, 1, 5, 5, 5, 29, 91, 139, 487, 545, 321, 3035, 4545, 6747, 21673, 3167, 54587, 201199, 284695, 612471, 1322255, 2121529, 3331557, 10656745, 18130247, 53443659, 128378545, 58043247, 497444385, 972280823, 722648981},
  {1, 1, 3, 13, 23, 49, 95, 103, 25, 119, 469, 2515, 2551, 841, 25089, 61511, 2475, 35433, 436371, 811253, 1371447, 258885, 8046545, 14821519, 10312269, 8682195, 30474785, 187632537, 201536351, 979701435, 536675181},
  {1, 1, 5, 7, 11, 31, 31, 197, 245, 715, 257, 4043, 8099, 11531, 5617, 43823, 2747, 60783, 83861, 994429, 708881, 721257, 5884787, 9601239, 15565385, 46695855, 25002705, 107580061, 130922325, 276883407, 1679811653},
  {1, 1, 3, 3, 19, 7, 9, 179, 103, 995, 191, 179, 3843, 5215, 27639, 55219, 51995, 63065, 311805, 946605, 623881, 2111759, 829801, 8535401, 13212789, 48598417, 78456805, 200750701, 525176433, 950651969, 588493717},
  {1, 3, 1, 7, 23, 59, 25, 65, 399, 211, 1453, 3511, 7203, 16015, 32197, 41431, 50713, 37327, 421853, 349425, 1547789, 1668647, 7065243, 12385093, 26582101, 13681395, 22266789, 27187973, 143418737, 499959763, 2118994837},
  {1, 3, 3, 5, 9, 35, 109, 67, 197, 449, 643, 519, 5751, 15551, 11331, 7515, 119449, 51837, 84923, 16939, 1700409, 1376851, 8221409, 14396267, 28928563, 14946217, 54527673, 130026049, 339741433, 901790485, 1044386933},
  {1, 3, 5, 3, 1, 17, 53, 201, 265, 351, 467, 911, 1117, 7183, 20371, 6075, 110873, 210875, 94897, 311599, 1409579, 3749439, 4069947, 12069475, 28846549, 44558933, 6119505, 211549707, 517334269, 368953125, 1581329125},
  {1, 1, 7, 7, 27, 17, 93, 81, 227, 619, 1191, 735, 815, 12615, 2719, 52483, 109403, 54685, 294001, 542125, 1725487, 2561047, 7656611, 7468441, 14407477, 8249609, 97576505, 209448409, 325604593, 587429369, 1557896277},
  {1, 3, 1, 15, 19, 3, 83, 75, 343, 297, 1019, 3469, 4383, 13299, 29755, 14235, 31161, 172735, 506677, 5157, 1827901, 2796637, 3167397, 13563145, 17643343, 55145121, 95646227, 54291293, 322329189, 121248037, 961211557},
  {1, 1, 5, 3, 13, 55, 119, 169, 85, 595, 299, 2469, 5625, 2877, 16117, 13247, 25035, 7695, 218065, 768371, 905269, 766065, 1519807, 11522063, 1643409, 35082581, 8969515, 168588379, 318322183, 306007415, 1542835885},
  {1, 1, 3, 5, 15, 17, 61, 161, 47, 393, 143, 867, 5517, 9495, 12795, 45835, 36699, 233721, 238651, 966717, 1982503, 3883047, 3195027, 13975989, 29540559, 57919333, 10112693, 108317139, 330422021, 866883869, 501193541},
  {1, 3, 5, 1, 27, 31, 113, 125, 251, 687, 969, 1473, 2245, 6355, 13655, 9859, 1465, 236827, 288627, 1033777, 294289, 1892343, 7816567, 8978921, 11994049, 65209359, 43843835, 87360175, 250152193, 481233929, 77279157},
  {1, 1, 1, 5, 5, 37, 29, 133, 443, 899, 277, 2353, 7223, 4459, 19159, 35811, 14971, 253787, 217983, 924727, 1535383, 263119, 4602215, 1557737, 15087469, 48480459, 81153715, 160058077, 392515965, 304499513, 985580229},
  {1, 1, 3, 9, 19, 27, 53, 145, 195, 167, 2045, 447, 1803, 1895, 8431, 40355, 83419, 139033, 404215, 605753, 1962389, 50651, 1792787, 804521, 8076761, 28773259, 40758637, 97449949, 194731785, 133195625, 374361845},
  {1, 1, 3, 9, 5, 27, 91, 147, 233, 451, 475, 27, 4629, 16181, 16437, 20895, 1483, 229897, 468823, 300527, 438457, 3263669, 6791549, 11340775, 15097177, 16462229, 96780085, 182322763, 168696919, 1032985559, 638768653},
  {1, 3, 5, 3, 29, 17, 53, 167, 433, 689, 1131, 2985, 1553, 11697, 6993, 40647, 71337, 47787, 418321, 280947, 1122195, 2435007, 3619893, 7953479, 3562571, 32243905, 66322599, 190060475, 136055163, 827599467, 1939092381},
  {1, 3, 3, 13, 21, 43, 69, 229, 399, 525, 179, 237, 7017, 5703, 17653, 38743, 79257, 255117, 342099, 197735, 124041, 3306411, 396535, 13243501, 14954155, 17739825, 53190019, 67643147, 521106109, 773821187, 1912620741},
  {1, 1, 3, 15, 13, 39, 75, 163, 229, 875, 197, 477, 3667, 15501, 15801, 13991, 32203, 10665, 421521, 706411, 1679253, 321517, 966733, 8798507, 705193, 60277979, 61096623, 70835969, 20638099, 187382523, 732104893},
  {1, 1, 7, 15, 15, 51, 81, 187, 487, 673, 865, 1915, 1009, 5935, 8097, 45855, 85019, 88365, 404249, 334569, 1466533, 429471, 5955825, 10313881, 23470775, 51373835, 133867665, 189196447, 147625397, 994678167, 807148485},
  {1, 3, 5, 5, 7, 3, 63, 77, 495, 815, 391, 2321, 1007, 15661, 30715, 20195, 113993, 79931, 123063, 644901, 668597, 1828513, 231547, 10909757, 27421529, 18668705, 132390359, 134174309, 445316699, 152348217, 95862813},
  {1, 1, 7, 3, 17, 25, 83, 173, 173, 911, 1373, 2957, 4549, 15977, 17695, 38523, 32491, 54365, 301813, 39087, 1812915, 1715881, 338643, 15337167, 4304509, 28835515, 123657839, 69880663, 389543387, 701417749, 1888120221},
  {1, 1, 7, 13, 13, 23, 77, 147, 497, 1003, 1687, 1795, 1393, 1881, 8479, 12027, 83627, 176861, 151803, 1032879, 1318277, 2070235, 6136529, 12971935, 28935985, 58975081, 74082741, 45921503, 19632403, 505737253, 507558685},
  {1, 3, 7, 11, 27, 43, 97, 25, 61, 457, 203, 2573, 5943, 15021, 4003, 38739, 126153, 46073, 132925, 989733, 843189, 3250431, 6629075, 15929863, 1051371, 50772833, 38075459, 227239445, 413886251, 623791489, 1561992349},
  {1, 3, 3, 13, 9, 37, 37, 25, 219, 889, 1535, 2791, 4531, 13679, 12663, 5235, 51097, 249309, 261747, 533051, 1798335, 979159, 2980395, 11328369, 32445207, 1856949, 83710065, 123275685, 171779457, 189352641, 1315511573},
  {1, 1, 3, 1, 17, 7, 51, 123, 89, 887, 1467, 1645, 3767, 6383, 30837, 30087, 90539, 203705, 308879, 711579, 1243069, 2538165, 7357205, 16471319, 10706653, 35505933, 42222163, 93407061, 124908249, 88109123, 96533333},
  {1, 3, 3, 1, 21, 47, 5, 151, 83, 257, 569, 2711, 637, 12569, 16893, 25327, 115193, 185821, 495503, 979599, 1454237, 2988755, 3017653, 4628613, 19555087, 16459523, 122726397, 18431451, 190409131, 193567223, 1962175613},
  {1, 3, 7, 1, 31, 37, 73, 3, 115, 919, 1817, 2483, 4811, 15245, 4375, 34091, 9401, 181001, 289671, 228229, 439459, 2974091, 7140329, 10860509, 26509241, 46206383, 15827097, 267228821, 439264595, 55827733, 1887270365},
  {1, 1, 1, 5, 1, 39, 39, 231, 9, 733, 455, 3383, 4777, 7235, 12631, 5843, 124011, 52651, 470415, 587395, 347053, 617921, 4825889, 11456383, 14009463, 61765317, 10009969, 188052143, 317121961, 297897705, 878141621},
  {1, 1, 7, 9, 13, 25, 55, 25, 73, 59, 1699, 929, 755, 1279, 5583, 63603, 5867, 50669, 331407, 710551, 1443259, 1145293, 4845471, 9032499, 11864465, 47632285, 35956015, 46317545, 466433153, 315449529, 644685813},
  {1, 3, 5, 3, 9, 49, 79, 55, 479, 179, 1159, 4079, 3503, 11603, 12361, 35119, 50057, 130203, 74113, 1042839, 1722251, 3705525, 4882785, 2123449, 15999733, 5646041, 127509913, 25550953, 470791861, 1050854167, 1160372805},
  {1, 1, 5, 9, 21, 45, 31, 163, 377, 817, 219, 147, 2581, 12769, 30783, 36299, 62267, 241231, 440459, 735119, 572335, 670861, 7934133, 5362135, 439227, 20199201, 76632173, 235852859, 290159763, 86597349, 1377255837},
  {1, 3, 1, 7, 15, 27, 39, 189, 493, 259, 1663, 1213, 961, 11089, 16079, 56843, 118329, 191055, 71565, 537497, 951181, 1590441, 6182923, 6822351, 22305633, 45034001, 5499051, 188233507, 235087019, 361450245, 1726288669},
  {1, 1, 5, 9, 5, 41, 13, 153, 313, 337, 1027, 1267, 4249, 13071, 27043, 3403, 88203, 188879, 408715, 1047967, 1504395, 1777815, 883755, 30147, 9598411, 57520873, 119927741, 238805879, 92790949, 1031782565, 1152356933},
  {1, 3, 7, 3, 13, 11, 23, 255, 51, 527, 317, 3217, 5037, 12723, 17411, 44251, 111721, 121737, 393093, 670099, 228009, 2843801, 4953145, 1747989, 31812945, 42635099, 10784243, 231381879, 437100673, 635801709, 1643734069},
  {1, 1, 5, 1, 25, 57, 83, 97, 233, 513, 1283, 2675, 4111, 4111, 32141, 23319, 60395, 59519, 272003, 174227, 203939, 1805737, 1315471, 12064611, 5040587, 8353929, 4426173, 135889505, 465152905, 865125515, 58571509},
  {1, 3, 3, 15, 25, 33, 103, 81, 155, 189, 139, 1179, 2691, 15119, 13959, 59267, 87081, 220589, 443009, 710303, 1613963, 3004637, 1274583, 10293937, 11712819, 48954105, 97833941, 192577757, 48530385, 225232577, 1311171861},
  {1, 3, 3, 1, 25, 55, 67, 19, 19, 9, 437, 579, 4273, 10733, 7125, 53551, 49945, 107357, 131215, 747395, 1973149, 2122661, 4415197, 5037581, 5491127, 65392575, 106304305, 245409279, 185851559, 17900951, 878810253},
  {1, 1, 1, 7, 23, 41, 47, 5, 435, 749, 595, 199, 3941, 7199, 4795, 31355, 31051, 92555, 261261, 901521, 1386127, 876229, 3347043, 13214321, 25322099, 14755361, 96231545, 220324611, 467454413, 572038205, 340280229},
  {1, 3, 1, 15, 5, 49, 35, 9, 199, 703, 1769, 3269, 5689, 13063, 22771, 41643, 66569, 126927, 422789, 192899, 1159859, 1394361, 5283559, 11908829, 967849, 54552159, 24685487, 148224203, 163295165, 275892821, 1304290053},
  {1, 1, 5, 5, 21, 55, 125, 55, 63, 149, 1167, 3577, 1051, 3921, 20545, 40407, 111003, 26367, 144135, 617751, 157493, 3962171, 585061, 9753321, 7756803, 22603549, 11185695, 227651649, 325995039, 238519851, 865627277},
  {1, 3, 7, 13, 29, 53, 107, 193, 163, 339, 1335, 1573, 5809, 5681, 29487, 64843, 115897, 242889, 363275, 260383, 1367351, 2102633, 269311, 9723785, 31757901, 33256601, 110335571, 109321059, 221518315, 417098597, 131906973},
  {1, 1, 1, 9, 17, 9, 91, 177, 211, 453, 1807, 1881, 6051, 225, 6021, 38047, 66459, 104667, 141059, 991499, 1977123, 2259201, 5401999, 3467001, 11678139, 55972365, 24138975, 240247865, 194088159, 91608463, 456667533},
  {1, 1, 1, 13, 15, 1, 29, 43, 181, 105, 1945, 2313, 6429, 2901, 6221, 13279, 47067, 21787, 49415, 696605, 283927, 1940855, 7210345, 12924907, 13595371, 22115603, 56229011, 25808199, 176933543, 227215791, 1238583981},
  {1, 3, 5, 11, 29, 55, 115, 115, 187, 1013, 697, 1885, 121, 12387, 32443, 14507, 43689, 69963, 250121, 783635, 794917, 2014021, 1147773, 10973925, 32235547, 36070859, 32515831, 217430635, 230262873, 230021525, 1516073749},
  {1, 1, 1, 7, 19, 51, 21, 107, 55, 125, 1655, 2281, 3293, 15749, 27521, 16391, 29019, 189819, 42765, 252181, 1191197, 2938187, 7696857, 1644153, 14027707, 61517525, 45586623, 4455399, 304495239, 350629059, 1819703533},
  {1, 1, 7, 9, 19, 9, 81, 93, 139, 401, 193, 73, 5159, 9323, 6019, 54715, 81291, 230861, 195343, 336649, 27927, 3677451, 6090087, 5928313, 16889615, 24629771, 21813395, 159729469, 229209901, 550727709, 1388049029},
  {1, 1, 7, 9, 15, 51, 115, 69, 247, 599, 1163, 2251, 1211, 8827, 15581, 10599, 17291, 75453, 294159, 226581, 238869, 3728733, 3376091, 4060389, 421153, 49131789, 110091109, 25978005, 368224341, 809607539, 1283066885},
  {1, 1, 7, 9, 5, 39, 75, 185, 321, 911, 849, 843, 6791, 10407, 10513, 44687, 34987, 77725, 299791, 650527, 1563925, 4037965, 895751, 8673291, 4102741, 19738247, 32890593, 235289961, 427681825, 700543239, 975669653},
  {1, 1, 5, 5, 15, 9, 21, 47, 459, 681, 2001, 1545, 5939, 7073, 29043, 55155, 13979, 22639, 382727, 808717, 666943, 944943, 243069, 15304925, 18856087, 49680123, 66792179, 193030505, 258689615, 105917305, 2026912525},
  {1, 3, 1, 11, 13, 25, 53, 97, 487, 797, 567, 3757, 5597, 6313, 18531, 3379, 46777, 174767, 225537, 513795, 310027, 165727, 370019, 8941805, 6861435, 23164421, 6381307, 210636143, 70912939, 119155033, 1932714525},
  {1, 1, 3, 3, 29, 55, 11, 219, 325, 591, 2015, 383, 2595, 11855, 22501, 43143, 130603, 215609, 303629, 580115, 1403413, 3240573, 2578981, 2088779, 332445, 37115801, 113930985, 62524581, 201993969, 936533635, 1824157781},
  {1, 1, 1, 5, 15, 57, 33, 125, 323, 749, 1843, 4019, 2075, 6673, 6957, 1455, 131035, 254107, 123407, 781837, 1736239, 1195643, 2378311, 9499249, 32024419, 35196673, 76436941, 255227413, 306446783, 712246423, 244636973},
  {1, 1, 5, 7, 19, 7, 47, 239, 51, 107, 1081, 467, 5493, 7617, 10355, 45651, 76475, 242095, 403461, 444437, 116745, 1136665, 7809065, 3078613, 326005, 65366935, 30752825, 186146187, 225771235, 283960681, 1688604957},
  {1, 3, 1, 1, 11, 3, 67, 199, 175, 421, 935, 309, 4449, 6363, 9183, 20739, 104681, 213231, 309259, 981009, 698397, 3061837, 4204569, 5671225, 9853667, 42958549, 65313539, 95278627, 446488353, 305775553, 728995253},
  {1, 1, 1, 7, 9, 33, 3, 219, 481, 513, 417, 1267, 2863, 765, 18431, 20155, 86011, 188747, 24589, 925711, 1155131, 319609, 6185109, 3563871, 21766619, 16196475, 17983673, 50770625, 390773163, 1069794717, 647051325},
  {1, 3, 1, 1, 19, 1, 89, 109, 415, 105, 487, 3241, 7465, 9233, 16307, 45043, 47737, 143471, 469067, 670281, 1437423, 2219667, 6351879, 7081101, 2704735, 63132189, 33125983, 102983155, 144929467, 304234393, 913446941},
  {1, 1, 3, 13, 9, 43, 25, 231, 383, 789, 1855, 691, 3465, 2387, 11715, 51387, 112939, 53577, 299075, 1012443, 57057, 4105447, 1266781, 2293161, 32907235, 26828525, 55186805, 145037191, 353837817, 348748477, 208515509},
  {1, 3, 3, 3, 13, 39, 63, 107, 33, 265, 437, 117, 3179, 5543, 28179, 33707, 121065, 197709, 165069, 507859, 1655525, 3551401, 6897725, 13058191, 15352307, 38158911, 82693463, 250009737, 397788569, 602651445, 22490645},
  {1, 3, 3, 13, 21, 5, 31, 111, 321, 425, 253, 3501, 3209, 15429, 18383, 24603, 124633, 192717, 280131, 431447, 1681399, 643773, 5796537, 14501623, 11783955, 3594519, 86376351, 48800219, 284205311, 970839021, 1353708301},
  {1, 3, 5, 9, 1, 27, 117, 187, 433, 459, 1999, 1069, 4857, 8591, 26343, 35299, 27145, 137771, 184907, 268235, 847345, 3663051, 2678297, 9015007, 2066369, 41867073, 20354683, 165794363, 307230789, 453941073, 1292382117},
  {1, 1, 7, 3, 15, 43, 11, 193, 391, 341, 1203, 1259, 7265, 1783, 13161, 9471, 6283, 140637, 211013, 695745, 1096269, 2826117, 1753711, 15005213, 8270659, 1642577, 53654261, 43556447, 52666477, 1073528479, 228037093},
  {1, 1, 7, 1, 5, 15, 45, 143, 193, 985, 1105, 3483, 2421, 9687, 22347, 10299, 109643, 247245, 355655, 9679, 1319677, 2866939, 4268605, 10412775, 13329139, 21406811, 130215025, 232874571, 516901621, 402749949, 962699717},
  {1, 3, 7, 13, 21, 17, 79, 231, 487, 663, 1101, 1025, 5779, 14681, 29181, 33871, 53785, 14297, 217547, 978775, 2085955, 3711685, 914321, 7184865, 6037729, 53371547, 22487139, 54764649, 102619479, 686245383, 68633197},
  {1, 1, 3, 9, 15, 19, 55, 219, 27, 963, 1513, 1017, 3907, 12279, 32655, 9363, 94827, 189609, 283079, 371925, 968917, 2583449, 5803277, 6615157, 4968589, 18796791, 42160819, 22908393, 279949433, 877908969, 1133877205},
  {1, 3, 7, 3, 31, 27, 17, 1, 51, 861, 529, 1225, 6395, 15787, 5231, 41123, 66569, 213865, 27589, 643009, 511197, 3649535, 4934043, 1543593, 24748019, 19327459, 109989683, 15423809, 156400405, 56848433, 417846373},
  {1, 3, 3, 11, 27, 7, 101, 143, 21, 191, 1437, 2393, 4097, 14319, 6977, 22127, 60809, 163613, 352069, 392405, 982889, 4115187, 2007517, 4444019, 3975677, 14237403, 35171211, 107145355, 417637189, 237875095, 570996901},
  {1, 3, 3, 3, 25, 35, 105, 141, 433, 269, 1469, 2939, 5387, 7373, 7863, 21931, 21881, 209229, 207309, 831047, 1993929, 3839479, 1124311, 13196275, 30574951, 9132303, 29914313, 112179029, 522098163, 610513205, 551361629},
  {1, 3, 7, 5, 5, 21, 23, 11, 217, 357, 1847, 101, 1161, 5297, 14879, 14571, 65529, 181513, 212803, 34647, 1205479, 1997269, 2253645, 1455911, 31236015, 62868325, 50069507, 110889771, 246685275, 795615893, 483132509},
  {1, 3, 1, 3, 25, 23, 81, 217, 505, 161, 1667, 1343, 1751, 2463, 26431, 3891, 114601, 177199, 257865, 1020231, 1213805, 2211367, 5836403, 12291763, 30218011, 61384009, 75869233, 14139361, 353507433, 372378041, 537002421},
  {1, 1, 3, 1, 17, 51, 125, 205, 385, 351, 731, 2911, 2749, 2689, 27031, 2971, 21499, 93129, 70095, 547803, 1063497, 3783251, 1622783, 225091, 19232757, 5289037, 67263521, 138696635, 304204899, 745706541, 411188381},
  {1, 1, 5, 5, 5, 17, 31, 171, 477, 671, 167, 1797, 8047, 10123, 4325, 32503, 30411, 103487, 101447, 141895, 1269235, 251477, 1511165, 3679043, 3507869, 39210881, 28531427, 185099357, 262516909, 248708763, 530260357},
  {1, 1, 7, 1, 11, 23, 123, 161, 99, 1007, 765, 1965, 5395, 16193, 17751, 2875, 50907, 177293, 133447, 722625, 1766521, 989725, 4867663, 6328921, 2917041, 4817131, 52660879, 255629809, 516668623, 486715517, 111322317},
  {1, 3, 1, 9, 13, 11, 111, 217, 31, 753, 377, 2267, 7893, 7195, 24999, 6483, 101289, 153071, 50115, 247111, 493401, 1422961, 1462959, 11416501, 30050807, 12947219, 81912337, 174811435, 197717561, 135521929, 921257717},
  {1, 3, 1, 9, 21, 53, 127, 121, 151, 395, 1447, 1411, 5179, 12043, 27607, 17651, 105289, 26799, 456771, 519263, 1989463, 281629, 601327, 8580493, 28482845, 13569081, 72868661, 118780069, 351336453, 660176921, 1378569285},
  {1, 1, 5, 3, 11, 37, 97, 139, 113, 835, 229, 3741, 827, 5527, 5779, 61675, 32491, 256207, 77121, 351557, 351323, 1965171, 2614353, 5350671, 14341225, 32036091, 100569503, 24746073, 426224137, 401975573, 223424725},
  {1, 1, 7, 7, 27, 55, 11, 55, 429, 269, 1179, 233, 1053, 10225, 16703, 27883, 122555, 40973, 283841, 735197, 1408633, 129053, 6056121, 13633979, 10693583, 33872377, 3207543, 105887591, 413044435, 338266837, 171656509},
  {1, 1, 1, 3, 15, 9, 67, 119, 95, 753, 511, 2507, 3953, 6403, 27635, 61083, 20651, 111691, 82633, 118593, 1063007, 1624073, 3819721, 15898809, 17279127, 10308709, 29737613, 196891567, 385965529, 452155501, 1260635829},
  {1, 3, 3, 7, 27, 57, 25, 27, 249, 515, 193, 4043, 2017, 751, 10949, 4295, 73545, 55101, 202953, 699469, 1393623, 2142227, 4034609, 7614919, 4952825, 13791967, 1955777, 140574255, 527900133, 405618435, 1780214405},
  {1, 3, 1, 9, 31, 57, 67, 21, 177, 573, 1835, 2015, 6201, 2383, 31087, 18387, 29641, 213103, 36547, 189525, 452943, 4124729, 5247147, 3641619, 488199, 23123001, 10621505, 130401967, 396752165, 860268873, 313273221},
  {1, 1, 5, 1, 19, 3, 89, 243, 69, 387, 1905, 3465, 2775, 7713, 30081, 599, 122651, 91327, 152899, 933849, 1426637, 139543, 4293017, 2512891, 28514433, 36747311, 19481219, 235655837, 505987527, 522003403, 1105544749},
  {1, 1, 3, 3, 9, 59, 15, 89, 85, 605, 881, 263, 2551, 797, 16541, 49663, 45371, 227353, 19021, 600263, 642545, 3935521, 5328207, 6639359, 24742223, 41782851, 39638669, 75296209, 351312955, 936422111, 628442941},
  {1, 3, 7, 11, 25, 41, 59, 139, 405, 571, 1147, 2963, 4175, 12901, 6309, 64175, 39289, 196697, 330317, 863959, 388067, 2959729, 5937653, 9452299, 21425929, 2746133, 40676493, 24423281, 20301027, 982476983, 1315683933},
  {1, 3, 1, 5, 29, 29, 11, 243, 183, 281, 807, 1, 7079, 10079, 13865, 50175, 70505, 98847, 90447, 322639, 1113583, 435033, 7437773, 12712697, 23393391, 42003149, 68605879, 161262253, 225337065, 697534559, 788430293},
  {1, 3, 7, 5, 5, 1, 89, 55, 423, 157, 675, 1849, 241, 6467, 15459, 12187, 65257, 43849, 418499, 136791, 222067, 1633587, 6224237, 633089, 9763675, 45644897, 123335175, 144503083, 522905977, 264568813, 1023187093},
  {1, 1, 7, 11, 15, 63, 89, 109, 501, 549, 317, 3043, 1151, 3895, 19851, 44347, 16747, 115725, 443469, 731217, 859993, 3888447, 1477095, 15433751, 17006791, 9405151, 110658289, 113810385, 412133985, 721332413, 1013608853},
  {1, 3, 1, 15, 7, 23, 97, 97, 225, 737, 1117, 3325, 209, 14169, 10813, 12559, 51001, 182559, 408773, 28865, 2022865, 2390903, 3542875, 6026715, 17016043, 12578391, 124157327, 91217251, 254730067, 213405223, 309915325},
  {1, 3, 7, 13, 13, 39, 91, 153, 395, 879, 1041, 3753, 5577, 1985, 25247, 57995, 104153, 234153, 303083, 663087, 544549, 2647837, 4704679, 3342625, 5570561, 27067879, 40045747, 230100419, 190782139, 997666485, 1991873245},
  {1, 1, 1, 3, 17, 15, 113, 143, 101, 901, 1119, 1819, 3577, 3441, 31511, 12731, 84955, 12651, 436969, 672063, 461573, 1509207, 2226629, 3401827, 7449575, 4906957, 61618605, 231794439, 323309531, 328020109, 323187901},
  {1, 3, 1, 11, 15, 27, 21, 37, 287, 121, 451, 1353, 2173, 299, 18791, 5779, 62025, 47055, 518561, 208289, 2000685, 3877147, 5724503, 447981, 29627407, 55232121, 49859479, 79515271, 20007721, 884737881, 1513912021},
  {1, 3, 3, 5, 23, 1, 49, 145, 315, 769, 99, 1385, 5961, 9121, 1465, 33543, 75097, 14749, 134635, 263909, 600855, 795515, 4573995, 2090417, 18662687, 56114633, 22432599, 72635571, 394945611, 241011859, 2128237405},
  {1, 3, 3, 13, 21, 39, 39, 233, 271, 113, 615, 2257, 3765, 5921, 313, 16647, 46489, 157085, 375011, 1036023, 1119541, 1982817, 5282159, 8873797, 7092471, 35503149, 5718919, 122574527, 116682131, 673425107, 1740387645},
  {1, 3, 7, 7, 25, 45, 11, 237, 83, 203, 929, 1819, 2679, 11583, 30091, 39035, 13513, 141161, 39073, 463919, 1547015, 1108745, 3055571, 8294145, 29126165, 65613311, 56684193, 158887097, 46036409, 649601869, 2076767221},
  {1, 1, 1, 7, 21, 63, 85, 251, 133, 991, 1515, 2547, 6051, 7279, 3569, 60943, 56715, 42939, 135597, 555379, 1783613, 1686323, 3089881, 4909931, 32208349, 19942765, 35108269, 36209373, 12492241, 1019074135, 1967390869},
  {1, 3, 7, 15, 11, 19, 87, 17, 313, 283, 1021, 2743, 4855, 13741, 17955, 42019, 73561, 193545, 362729, 420269, 1438493, 3350889, 5672711, 3344227, 27755953, 7009827, 109007717, 124916833, 435560843, 767981217, 2278269},
  {1, 1, 7, 13, 29, 13, 61, 93, 81, 91, 995, 907, 4847, 2599, 20041, 34271, 5131, 111613, 416939, 467311, 2047023, 2604143, 584751, 1723683, 26261169, 453661, 87711077, 111753345, 131843777, 878788927, 1390370389},
  {1, 1, 3, 11, 13, 45, 103, 33, 243, 109, 2029, 121, 231, 16179, 13741, 5287, 79275, 119577, 448805, 946419, 816143, 3122245, 6380775, 2985145, 2357011, 67035039, 25782491, 83796829, 108645365, 974145955, 2140174693},
  {1, 3, 5, 9, 9, 5, 73, 225, 199, 723, 611, 1909, 2345, 10257, 9909, 35615, 121817, 49403, 102635, 697443, 329759, 3475563, 6142107, 931853, 24180789, 54024301, 110904491, 129127803, 173496763, 235249503, 205835101},
  {1, 1, 3, 11, 7, 5, 33, 89, 431, 899, 803, 3173, 6131, 16097, 20561, 61975, 7163, 115417, 255589, 835577, 1913395, 908883, 4311619, 12851797, 13551413, 60545101, 14641531, 28371457, 78063615, 330125531, 1760921421},
  {1, 3, 3, 7, 7, 47, 23, 47, 411, 69, 239, 661, 5591, 10457, 24245, 65471, 47993, 235645, 298089, 361329, 1183257, 2478609, 243289, 16705117, 856363, 16593661, 120372083, 31721781, 788847, 67403023, 598519693},
  {1, 1, 5, 15, 25, 35, 87, 23, 115, 939, 1579, 119, 4001, 13791, 9729, 239, 104619, 61119, 458669, 261103, 1692057, 3314377, 1186129, 13911845, 2388821, 42554661, 107314393, 191321719, 498073381, 335229415, 2006828037},
  {1, 3, 5, 11, 25, 45, 29, 195, 369, 237, 735, 155, 123, 4415, 32255, 48595, 91657, 65611, 267561, 842807, 598423, 1397247, 4812561, 14369407, 32938567, 7642685, 15038605, 77836229, 324796561, 462162041, 252479381},
  {1, 3, 3, 9, 13, 53, 15, 77, 313, 75, 529, 925, 5679, 14585, 19889, 3063, 92665, 19165, 269543, 122983, 600727, 10957, 5893595, 16233387, 26484577, 35611599, 115845655, 23311213, 26759055, 446299723, 506615885},
  {1, 1, 7, 15, 15, 27, 105, 13, 31, 669, 563, 1809, 4321, 7797, 4177, 21895, 103675, 147741, 304873, 624505, 1512605, 3108487, 257479, 7129789, 2458763, 11164289, 5481519, 124855051, 383252559, 326384147, 1276432589},
  {1, 1, 5, 9, 3, 29, 111, 177, 33, 235, 1951, 1561, 141, 4803, 16327, 30835, 91019, 181967, 68907, 824889, 912787, 939965, 6020439, 14252299, 8341553, 39946721, 41903439, 146405447, 84625793, 79308905, 711371253},
  {1, 1, 1, 7, 9, 41, 1, 149, 95, 933, 115, 1619, 771, 8189, 8781, 58815, 63643, 18427, 314733, 850159, 285331, 373451, 1608575, 1433949, 28107699, 16639025, 89794541, 216808877, 286611459, 50982927, 1492970973},
  {1, 1, 5, 3, 13, 41, 33, 159, 355, 159, 1243, 1439, 6571, 14397, 31321, 343, 112411, 232031, 431969, 902755, 673467, 2446539, 2257813, 10581237, 11986289, 45669565, 79948433, 130867213, 213072979, 881153979, 1933697373},
  {1, 1, 7, 11, 9, 15, 91, 145, 457, 255, 1449, 611, 1449, 2521, 28949, 18655, 41627, 195709, 312173, 848631, 1213573, 1127101, 3860239, 996467, 19359253, 31952831, 10215321, 236748343, 490082403, 910315999, 33710877},
  {1, 3, 7, 5, 27, 57, 35, 99, 447, 287, 743, 1163, 4379, 7361, 3831, 48699, 57337, 31977, 11235, 903081, 965527, 293529, 3615389, 3868561, 22629785, 27450465, 58720093, 120012309, 71173551, 140388173, 537551469},
  {1, 3, 3, 7, 15, 53, 41, 83, 133, 571, 1739, 531, 2921, 11527, 21941, 4775, 68585, 189085, 399273, 819833, 910995, 717211, 680601, 16510699, 27643929, 19215109, 131374861, 199962743, 238307245, 683576259, 1520401253},
  {1, 1, 1, 13, 9, 27, 39, 113, 429, 447, 595, 3171, 5245, 4095, 14847, 51667, 85867, 189771, 142247, 15163, 274593, 962265, 5588183, 3689655, 17346173, 55327557, 51829197, 180068067, 486850493, 775591545, 1328083237},
  {1, 1, 3, 7, 19, 19, 21, 101, 489, 1011, 265, 3899, 3225, 11701, 5193, 26455, 15355, 55129, 291305, 612213, 1292685, 4069291, 5227575, 570843, 24545129, 34420775, 15041857, 173313111, 114462575, 341610011, 635677357},
  {1, 3, 7, 3, 15, 25, 103, 213, 441, 215, 1483, 263, 3561, 7915, 7969, 58239, 33929, 180217, 64549, 895473, 1949343, 2930093, 1764355, 8312123, 10948349, 32703597, 8323129, 233505663, 502604737, 1051158191, 1557509013},
  {1, 3, 3, 3, 11, 47, 97, 29, 489, 867, 1347, 2155, 4871, 8001, 18305, 48983, 55097, 200541, 342765, 187253, 941441, 3244999, 4281447, 994635, 28172121, 10183437, 91465733, 71147753, 151286519, 1018287291, 1914333325},
  {1, 3, 1, 9, 25, 15, 61, 17, 343, 775, 1765, 3803, 4577, 8437, 12605, 56383, 52089, 88831, 311267, 654835, 925077, 1546635, 8214531, 11504045, 9513953, 36859139, 113736329, 165304783, 463108223, 284620399, 1678204525},
  {1, 1, 5, 3, 11, 39, 69, 23, 23, 65, 1967, 2429, 1703, 6671, 14981, 15751, 110923, 179231, 70433, 311909, 299705, 3226547, 979173, 1298913, 17199527, 51704565, 25975739, 195327797, 383826521, 915257043, 1166447573},
  {1, 1, 5, 15, 23, 59, 125, 51, 225, 439, 2019, 2589, 7781, 13111, 2911, 27155, 74859, 202767, 289773, 274657, 1678685, 1241107, 3584145, 6181375, 12112573, 15333301, 23884723, 163753671, 38014981, 675038073, 1236028965},
  {1, 1, 1, 3, 1, 31, 37, 245, 203, 305, 821, 367, 5211, 9791, 21777, 65231, 122059, 82427, 188649, 268079, 2002165, 106595, 7561431, 9593337, 2826607, 57877855, 102474301, 177694957, 452006657, 139798263, 1478753493},
  {1, 1, 5, 9, 9, 31, 97, 25, 271, 83, 343, 2461, 1805, 14383, 10059, 47579, 116779, 140079, 435435, 887411, 2079813, 2637943, 6406259, 13812133, 11256357, 63258777, 6375595, 136433103, 251452205, 678466141, 1301819653},
  {1, 1, 5, 13, 15, 33, 127, 109, 137, 963, 961, 1647, 7881, 8133, 22359, 63403, 64315, 24175, 497583, 954365, 1292919, 3012661, 3844107, 9697723, 818057, 28879055, 68086277, 20428399, 130334271, 97634693, 596082605},
  {1, 1, 3, 7, 25, 31, 123, 241, 283, 1, 1781, 23, 971, 6485, 127, 6203, 74907, 215593, 155561, 563455, 216149, 1852957, 7715567, 10636833, 31130895, 60629727, 95481733, 130949389, 8075339, 658837133, 86979197},
  {1, 1, 5, 15, 15, 27, 25, 145, 395, 679, 979, 571, 1585, 14787, 7465, 31823, 6027, 126463, 225389, 539961, 1700301, 76087, 6137659, 3753425, 15760985, 11318197, 49944357, 175728671, 33958169, 668081239, 670435637},
  {1, 1, 5, 7, 13, 11, 7, 131, 511, 597, 379, 1513, 6267, 16039, 1503, 44691, 80139, 43535, 251109, 310251, 119385, 3875145, 2179333, 5676033, 9141491, 37112553, 9970119, 226238881, 255821225, 114018745, 656751573},
  {1, 1, 1, 13, 15, 49, 73, 217, 353, 577, 1913, 1127, 961, 11557, 24993, 9159, 80571, 138459, 348327, 639037, 191687, 3980067, 1535827, 14913979, 16003435, 340435, 95910397, 115052391, 460302863, 644993459, 1546520813},
  {1, 3, 3, 9, 7, 3, 105, 141, 377, 687, 1917, 485, 983, 11149, 23303, 15627, 52121, 42349, 349095, 728813, 1365877, 880903, 8123335, 8586907, 26635445, 52821739, 39391799, 233181061, 52304555, 244974485, 1814440669},
  {1, 1, 3, 15, 11, 7, 117, 179, 505, 67, 1817, 913, 5757, 1981, 1637, 45519, 116411, 170873, 175521, 514749, 2017257, 2539875, 6408481, 12452487, 23221689, 41680167, 128002059, 193472871, 526210415, 693611703, 1845116941},
  {1, 1, 1, 7, 5, 29, 3, 43, 223, 295, 1895, 3425, 5355, 5155, 17197, 20711, 10027, 23515, 288365, 305059, 267007, 1763169, 180581, 7744449, 13754081, 8618385, 81822583, 140873137, 130129133, 877051459, 1638075845},
  {1, 1, 7, 9, 21, 59, 121, 245, 73, 233, 1527, 869, 4145, 7995, 6473, 55183, 57163, 238461, 427375, 358191, 216009, 3439399, 4262159, 12308379, 22317139, 22297773, 45556771, 127663699, 233287041, 183948407, 241250101},
  {1, 1, 5, 13, 17, 21, 89, 179, 495, 669, 453, 2603, 5969, 6161, 4743, 9499, 99355, 113327, 180271, 82659, 769407, 829819, 3854185, 4776897, 5456363, 37273735, 108154809, 22699215, 436461243, 187230365, 1598088317},
  {1, 1, 7, 11, 25, 21, 103, 131, 391, 249, 1633, 2603, 2207, 8987, 15487, 21827, 105131, 224205, 117101, 538087, 718463, 3122037, 5625637, 10578825, 28746287, 41579, 71313009, 35503521, 500504077, 28391665, 471676453},
  {1, 3, 7, 9, 13, 45, 99, 251, 115, 597, 1505, 2421, 1231, 10015, 24295, 38403, 55049, 197577, 364399, 7527, 1806831, 3344593, 3926773, 2558861, 13214667, 29942515, 117894799, 70641581, 284392105, 9570961, 1815804757},
  {1, 1, 5, 5, 31, 49, 17, 67, 463, 813, 1491, 3309, 7881, 8109, 7289, 23095, 37627, 12319, 287015, 366013, 2081351, 1771963, 5901049, 13697953, 23305755, 52380913, 74860243, 71783643, 270157575, 763473323, 1569858605},
  {1, 3, 1, 15, 23, 35, 123, 21, 169, 499, 95, 603, 1829, 7865, 26313, 28007, 92825, 185055, 238757, 19889, 1781349, 2217445, 954939, 15463355, 9252057, 43587377, 67376525, 171305963, 408042987, 803093955, 1094586493},
  {1, 1, 7, 1, 9, 29, 45, 65, 95, 97, 673, 3673, 2969, 2317, 22209, 62407, 19195, 250845, 442151, 59043, 1875031, 3407359, 307827, 8498005, 17368407, 6364939, 117042563, 215838659, 259950615, 158907251, 1346070381},
  {1, 1, 3, 7, 29, 33, 121, 17, 331, 487, 1901, 1951, 5383, 9375, 4029, 22807, 113547, 46873, 316009, 978491, 1643363, 2351587, 2922155, 11809473, 4203865, 41454283, 90558717, 161178593, 301378137, 13599451, 1737019829},
  {1, 3, 7, 9, 25, 43, 91, 147, 141, 401, 1647, 2697, 4645, 7179, 31857, 35199, 120073, 37945, 296431, 23731, 1436225, 285157, 1144909, 11696531, 11946163, 1179253, 102246175, 161276575, 314805209, 370073807, 1725278101},
  {1, 3, 5, 11, 9, 31, 127, 105, 39, 883, 1635, 919, 5069, 2875, 24519, 33587, 90121, 172171, 28905, 922215, 1161157, 2752185, 4815935, 13383085, 22795685, 17662957, 88051401, 145917467, 99193497, 1049076265, 3497461},
  {1, 1, 5, 9, 1, 63, 73, 135, 95, 503, 385, 3903, 545, 12635, 27569, 11423, 44747, 44607, 178667, 87483, 494453, 3782303, 118141, 16382793, 23581921, 3985015, 32323493, 157422023, 526259889, 115487071, 1142356373},
  {1, 1, 3, 11, 27, 31, 47, 173, 55, 339, 1255, 1947, 793, 14133, 13963, 32467, 76347, 184585, 282661, 903909, 1738193, 3251113, 5695371, 13655989, 22515557, 52276873, 23003469, 30613079, 59846943, 463286297, 545492909},
  {1, 1, 3, 15, 17, 33, 113, 249, 401, 743, 1307, 3123, 627, 1253, 13285, 41039, 5659, 128889, 448033, 163367, 1093755, 3181739, 3618099, 3514315, 5359917, 28953085, 74601837, 13691341, 234207035, 1050920087, 1794540861},
  {1, 1, 3, 1, 9, 7, 39, 65, 281, 107, 833, 193, 2987, 12267, 31335, 2579, 56811, 13705, 401135, 747107, 920301, 886721, 1488359, 11684035, 15919745, 44075199, 34989611, 97610257, 292552021, 1040615193, 674297061},
  {1, 1, 7, 3, 15, 21, 99, 211, 39, 179, 587, 1169, 6455, 8225, 2049, 7863, 27259, 10781, 441765, 299169, 900947, 722097, 3393357, 1243401, 3557445, 31312469, 64812191, 263981741, 312229047, 515117547, 413936173},
  {1, 3, 5, 13, 5, 13, 123, 1, 223, 273, 731, 2967, 4793, 4229, 26031, 60315, 114617, 177451, 194095, 857191, 1598159, 2143145, 2535743, 15622357, 29483287, 29500305, 87534457, 31912063, 36071439, 883840637, 2037294253},
  {1, 1, 1, 1, 3, 17, 7, 23, 225, 757, 743, 1257, 2047, 12509, 25467, 55931, 90099, 159187, 333747, 804785, 1318695, 2882837, 580393, 9411551, 20828487, 11875933, 74519479, 8377309, 258820947, 275777297, 1775257597},
  {1, 1, 7, 15, 29, 3, 15, 113, 227, 675, 1295, 2777, 2921, 5485, 2577, 35551, 75411, 9941, 495889, 982515, 203017, 3186009, 2531711, 15339313, 7750645, 13485529, 7522103, 205249995, 374787311, 1013149019, 2028453261},
  {1, 3, 7, 13, 19, 21, 85, 129, 45, 599, 317, 1513, 4953, 10383, 25253, 5567, 40337, 205041, 336467, 865273, 549155, 3073903, 2649371, 13170367, 32467005, 61627235, 51785251, 227212155, 346555885, 995630475, 978909893},
  {1, 1, 7, 11, 13, 47, 127, 67, 219, 131, 905, 2005, 851, 15243, 5777, 21911, 33075, 40117, 36117, 515435, 1933061, 591729, 5064173, 11244269, 327013, 65532319, 34069015, 221625897, 447301501, 206359671, 1282926021},
  {1, 1, 5, 3, 23, 57, 57, 189, 153, 37, 955, 2049, 1295, 15119, 27213, 42207, 68851, 138487, 506073, 442849, 105255, 2573115, 2951519, 9784307, 10555239, 2151489, 114982535, 185343549, 299117409, 1040999491, 1275370549},
  {1, 3, 7, 11, 13, 61, 3, 241, 269, 789, 1595, 2369, 4843, 11347, 21543, 59483, 57521, 64305, 81013, 788395, 591159, 3015017, 5942695, 11327583, 3217535, 8763521, 74799431, 184355737, 321858677, 168765073, 939735653},
  {1, 1, 5, 5, 25, 21, 19, 237, 3, 605, 1343, 3965, 3511, 7889, 27759, 803, 90739, 153111, 271359, 758051, 1018391, 491081, 4002363, 12504809, 1144363, 38679573, 23631795, 195363037, 2037263, 954816413, 1523034573},
  {1, 3, 1, 15, 21, 15, 123, 5, 345, 945, 283, 1313, 335, 2085, 19505, 32655, 36849, 198999, 377373, 661499, 1985077, 2999829, 3523107, 3709923, 3978579, 28437033, 128736807, 221916221, 396743819, 219298283, 890168477},
  {1, 1, 3, 3, 5, 21, 123, 89, 67, 11, 1247, 1155, 287, 13455, 5693, 3007, 21747, 84465, 81621, 197363, 696879, 1919553, 7076607, 4538049, 14709629, 48985049, 125194701, 158363913, 213681249, 187009459, 759616021},
  {1, 3, 3, 13, 1, 53, 101, 27, 387, 379, 19, 751, 2445, 11737, 975, 52115, 113105, 248981, 513531, 433195, 1475111, 3978847, 6690401, 633349, 13298445, 18947093, 14283225, 74458371, 30282515, 873079853, 1260556957},
  {1, 3, 3, 3, 9, 29, 81, 117, 443, 145, 1619, 1813, 8125, 5829, 28617, 36463, 25809, 133461, 451477, 301055, 465951, 25659, 3941511, 2319777, 15963927, 55578049, 3748547, 73973895, 500954063, 1029994835, 220473037},
  {1, 1, 5, 15, 27, 15, 83, 83, 61, 715, 1655, 1631, 3457, 2727, 2163, 23411, 82195, 157047, 265269, 474165, 1565705, 2057277, 498853, 3830987, 10294561, 62944049, 71381473, 25767055, 144651957, 163921709, 1545440389},
  {1, 3, 1, 5, 11, 11, 121, 7, 135, 883, 927, 1817, 6839, 12361, 24119, 13811, 102513, 41879, 350327, 569905, 1922061, 3271711, 2139269, 11357337, 3712941, 48768905, 52650871, 192034837, 336860919, 657204533, 2086963181},
  {1, 3, 7, 11, 23, 59, 39, 165, 109, 355, 1303, 381, 5697, 275, 3771, 37939, 63761, 148017, 330933, 192049, 1409029, 1802305, 8028347, 4096311, 1671401, 230081, 132548067, 35161803, 222410241, 970322317, 276937845},
  {1, 3, 5, 11, 11, 5, 81, 157, 55, 435, 613, 127, 4087, 3791, 21627, 32627, 6257, 225267, 496305, 1013805, 1417219, 2154603, 7102511, 6201565, 25551085, 29418675, 128968517, 127238105, 207645937, 131103189, 1446051029},
  {1, 3, 7, 15, 13, 37, 83, 195, 207, 771, 51, 3685, 6389, 1229, 11101, 3495, 113105, 204369, 400465, 69091, 681647, 1945225, 971765, 4516089, 7809517, 63319845, 61798867, 259737391, 440359031, 918666007, 1382860717},
  {1, 3, 7, 13, 31, 3, 9, 13, 487, 95, 77, 809, 5809, 12887, 29933, 3727, 121745, 46321, 417619, 737653, 801709, 4040351, 7616431, 15841581, 21072065, 27605379, 123947203, 230345139, 151804325, 630372595, 876845797},
  {1, 1, 3, 7, 25, 9, 13, 29, 353, 659, 1785, 3825, 3729, 13109, 12973, 24967, 50579, 22737, 8017, 810471, 771723, 1702543, 6287255, 3356075, 4118369, 16547503, 129017075, 249020227, 526565511, 662722231, 729830765},
  {1, 1, 1, 5, 21, 3, 97, 1, 245, 917, 29, 1429, 8141, 7569, 32493, 19831, 50515, 142483, 149335, 1002223, 167577, 1728727, 5879091, 2846343, 27601775, 26069191, 98737183, 183778583, 325149819, 884443455, 1954958301},
  {1, 3, 1, 9, 19, 13, 13, 109, 377, 1007, 1737, 1939, 1419, 1145, 5065, 27823, 92209, 31255, 177691, 819825, 1001147, 1636071, 5378375, 1062707, 22460973, 20028391, 43689969, 211332893, 535124799, 999736267, 313645229},
  {1, 1, 7, 9, 27, 57, 53, 69, 423, 43, 1629, 1003, 1473, 10809, 5659, 2907, 67731, 164885, 47927, 229433, 1285823, 1393559, 8207327, 7125565, 13077, 57897979, 26531617, 108434759, 406753899, 816035673, 1968914173},
  {1, 1, 1, 9, 1, 45, 11, 231, 351, 155, 663, 2783, 3491, 5725, 25207, 15619, 11955, 70483, 430587, 149667, 1491103, 2977249, 3369921, 6743585, 16334677, 32468273, 112355169, 207141005, 382765515, 708688477, 1249592445},
  {1, 1, 1, 3, 15, 25, 77, 89, 231, 813, 657, 2603, 4885, 1383, 14499, 35315, 99027, 257523, 394417, 986297, 1060279, 824271, 6180691, 4269893, 2475283, 17876411, 129188281, 71287699, 61530365, 889901405, 14556325},
  {1, 3, 5, 5, 9, 21, 101, 181, 449, 491, 737, 803, 659, 11771, 545, 33463, 59057, 113971, 128415, 209139, 1573015, 759295, 6562831, 5858587, 33426585, 50463559, 123079057, 188879077, 113649869, 631649319, 729177189},
  {1, 3, 7, 9, 7, 19, 27, 199, 265, 329, 1031, 1235, 3191, 10071, 16281, 37975, 28657, 84721, 365591, 490213, 1435789, 1462189, 3233281, 4991575, 20166411, 34540485, 9337421, 131285537, 217900425, 901790183, 1044426805},
  {1, 1, 7, 11, 27, 55, 3, 127, 503, 1003, 1041, 1953, 5835, 4851, 13485, 19023, 41267, 114613, 123477, 457085, 1880753, 1549757, 3836585, 907097, 25291157, 30251671, 88597395, 122236057, 25383093, 103531707, 1960107685},
  {1, 1, 7, 15, 5, 45, 97, 61, 221, 497, 1949, 3163, 4707, 8441, 1437, 7751, 22195, 223509, 343121, 859627, 1589143, 4069355, 6364783, 10960791, 6740923, 49207887, 65996817, 137429029, 327483919, 498024223, 1832888301},
  {1, 3, 5, 1, 3, 35, 107, 9, 473, 971, 227, 2225, 3999, 3095, 18879, 14427, 93297, 114931, 242555, 858673, 1350837, 2433181, 3435247, 1526363, 11633321, 44925733, 39096071, 51089869, 314550073, 786687041, 125404885},
  {1, 1, 1, 9, 21, 59, 21, 1, 41, 435, 575, 491, 1839, 1095, 9727, 47131, 72563, 232435, 285307, 517687, 1893537, 364755, 1309211, 13880027, 3891825, 33808873, 52096933, 253618633, 196695177, 179056225, 1404892213},
  {1, 3, 5, 9, 13, 29, 123, 251, 465, 701, 1105, 829, 573, 11503, 11861, 35263, 82193, 12595, 453779, 325295, 89047, 695857, 1145597, 5146263, 21674287, 63596667, 37174799, 99795703, 200033701, 158659451, 1273335205},
  {1, 3, 3, 13, 27, 59, 29, 111, 225, 973, 561, 1481, 835, 9261, 13831, 4067, 87281, 144917, 76347, 893681, 2059869, 858235, 6395941, 2640111, 3709343, 35597787, 9970235, 133198945, 3269019, 499700909, 80507421},
  {1, 1, 1, 7, 17, 3, 97, 211, 333, 315, 571, 3523, 7305, 6461, 20139, 16683, 76243, 153747, 179061, 979823, 185169, 555031, 7960577, 15935835, 8489073, 66353789, 123662437, 25946623, 340907775, 183624513, 945361389},
  {1, 3, 7, 11, 31, 21, 105, 247, 113, 863, 1767, 381, 4623, 8935, 7911, 48107, 1201, 99505, 287029, 699129, 208763, 557075, 1540277, 3965135, 10191821, 35619753, 88771491, 103519045, 320831721, 827835129, 274275029},
  {1, 1, 5, 7, 29, 45, 17, 155, 69, 17, 655, 1983, 6385, 6177, 7961, 42879, 24147, 222295, 187485, 840995, 1251815, 413819, 5845033, 4862211, 24989739, 45044061, 132207217, 239351039, 189200051, 184755403, 35333917},
  {1, 3, 3, 15, 31, 15, 63, 81, 309, 115, 393, 3445, 689, 13963, 18887, 53995, 23249, 37749, 5305, 98033, 1686369, 1602609, 6268031, 1969479, 14204617, 3311807, 26159959, 159660779, 388137529, 655792993, 1380491221},
  {1, 1, 5, 1, 19, 39, 127, 61, 357, 53, 195, 2745, 7853, 5753, 3669, 38759, 121491, 148055, 481435, 333985, 725745, 4156417, 4492339, 1629519, 23727791, 65495929, 86239951, 193163759, 500182931, 80747895, 1253050749},
  {1, 3, 7, 7, 17, 51, 57, 145, 451, 365, 1517, 909, 4265, 10737, 9579, 539, 116305, 213457, 126329, 687727, 1397825, 2485519, 812979, 15783889, 16660283, 58893191, 118599591, 39384291, 256831315, 475188793, 1007688189},
  {1, 1, 3, 13, 3, 37, 121, 103, 257, 47, 1685, 2951, 5753, 15379, 8899, 707, 78419, 167793, 295675, 504169, 430931, 3698531, 7997285, 13791135, 30964829, 33101883, 46338493, 215322311, 348304625, 994349333, 646475605},
  {1, 1, 5, 7, 31, 63, 61, 197, 97, 773, 133, 1517, 3093, 14879, 22941, 58287, 107923, 131895, 256285, 896161, 1811185, 1662835, 3398959, 7663291, 32166167, 12481151, 55729623, 96827199, 448503301, 307949075, 1982017701},
  {1, 1, 5, 1, 3, 9, 27, 53, 97, 663, 1915, 409, 471, 1391, 24853, 47087, 44339, 13879, 335643, 140209, 1819391, 3055929, 1731571, 11166811, 7213029, 63435525, 66082847, 127787925, 373953489, 674640347, 1498843125},
  {1, 1, 1, 7, 21, 53, 69, 5, 187, 571, 2023, 997, 323, 12059, 7071, 3211, 76659, 57459, 253109, 879339, 900719, 3084639, 6958495, 13885505, 11379325, 21325089, 55733915, 226027993, 490229005, 1024685945, 474419909},
  {1, 3, 3, 1, 7, 59, 55, 157, 101, 123, 1301, 3709, 4673, 3897, 28791, 18371, 80337, 171349, 179383, 126965, 1924453, 3417937, 76099, 16744655, 23625889, 8487731, 1803431, 63803787, 63814891, 208838261, 1426514333},
  {1, 3, 7, 5, 5, 23, 39, 139, 365, 415, 1481, 3415, 6323, 11109, 5719, 5603, 70769, 64657, 458939, 86143, 188877, 2446553, 4309045, 147659, 22593621, 25576871, 12996053, 154126317, 169631507, 502165805, 120734813},
  {1, 3, 5, 3, 5, 11, 23, 143, 243, 229, 183, 3367, 3187, 8151, 28351, 61819, 27121, 98739, 69689, 922995, 1004497, 221937, 5886065, 168221, 23450451, 32313773, 31659545, 116807853, 1462081, 733687169, 1264801877},
  {1, 3, 7, 9, 5, 37, 29, 23, 437, 827, 985, 2879, 7611, 1391, 19087, 12603, 98545, 96817, 464695, 315239, 507903, 2824839, 2295837, 329899, 7809889, 55962911, 7130525, 212197301, 274319433, 744813665, 1919815861},
  {1, 3, 3, 5, 7, 9, 5, 143, 217, 757, 1697, 2459, 453, 8679, 4513, 12087, 16657, 199093, 230995, 261565, 393303, 3114375, 1924341, 5080151, 21786615, 30680443, 68510025, 159557443, 101794653, 589212447, 557421605},
  {1, 3, 5, 5, 11, 33, 3, 143, 293, 921, 185, 2461, 5547, 12247, 28591, 34203, 85489, 85811, 437567, 640369, 1100775, 1713, 8068153, 16035051, 18394467, 49905627, 60873151, 260278625, 351647729, 657822321, 1754246389},
  {1, 3, 7, 5, 3, 53, 43, 179, 235, 417, 1307, 1367, 3695, 12809, 1807, 31619, 75313, 131921, 215355, 896121, 1847011, 3353489, 6647061, 5876829, 3147399, 27353865, 120694001, 29455185, 417059303, 1065935565, 711219533},
  {1, 3, 1, 11, 15, 43, 115, 229, 157, 25, 687, 3347, 271, 5777, 8557, 36135, 24881, 247767, 309017, 301097, 1810021, 1940117, 5656531, 14023687, 11370291, 3912589, 15643197, 37493785, 193212799, 931409759, 1257569005},
  {1, 3, 7, 5, 27, 37, 55, 135, 209, 47, 1603, 957, 5785, 11141, 10407, 63443, 126801, 40209, 375355, 1008225, 1461955, 1540013, 8004889, 14726543, 18283901, 25308813, 27436139, 139914323, 140791079, 390347453, 43310445},
  {1, 1, 1, 15, 17, 17, 103, 29, 489, 493, 119, 1707, 3463, 1815, 32055, 28875, 100275, 197043, 184509, 745343, 604355, 421813, 2769347, 10132707, 18521935, 41642301, 36275381, 263164321, 242200105, 35644937, 34680885},
  {1, 3, 7, 11, 17, 13, 115, 145, 77, 515, 1911, 477, 5997, 8731, 3143, 57547, 103697, 104689, 286005, 79991, 866047, 91065, 3024231, 15288415, 13810217, 10804001, 111308931, 121807207, 411770257, 758456225, 1145431093},
  {1, 3, 1, 13, 31, 41, 73, 91, 231, 1, 455, 2023, 4691, 3613, 16329, 10175, 14577, 80919, 470111, 770229, 1337671, 2263723, 4427129, 189121, 15027839, 28914229, 20692537, 173821773, 431314699, 56007907, 444698749},
  {1, 1, 5, 15, 15, 39, 17, 117, 131, 657, 1939, 2245, 2575, 195, 25209, 39239, 89139, 202103, 175189, 973985, 1029449, 605935, 6455751, 1724345, 23181895, 35597313, 6640083, 146252757, 399006061, 237749679, 866548037},
  {1, 3, 7, 15, 5, 51, 69, 141, 499, 931, 1165, 2119, 1703, 10867, 28443, 45923, 57809, 229905, 374289, 371083, 2086793, 287699, 1603447, 81465, 2944501, 32902427, 45600269, 242143129, 448840333, 934197933, 1612362661},
  {1, 1, 1, 15, 13, 45, 45, 103, 115, 177, 651, 2545, 1417, 5349, 3385, 9423, 5235, 21299, 64893, 1024643, 827559, 2662119, 335181, 613357, 19567879, 17435065, 70434071, 28068059, 369840487, 645059779, 706824493},
  {1, 3, 3, 1, 1, 41, 117, 15, 225, 861, 843, 2775, 4543, 6275, 14671, 56331, 60369, 201237, 385879, 299411, 1013659, 127383, 8006517, 7806575, 24745103, 33366817, 24028977, 254292353, 467909549, 921041017, 1453314661},
  {1, 3, 5, 15, 5, 35, 87, 193, 341, 55, 1131, 945, 6865, 11271, 18705, 30167, 1521, 84499, 424373, 166539, 1586585, 2913985, 977183, 12062215, 21404461, 16657909, 90691831, 178137699, 532078677, 945434159, 526819685},
  {1, 3, 5, 9, 13, 35, 71, 197, 79, 351, 3, 3939, 1105, 12455, 28921, 62855, 71089, 224403, 316211, 1012367, 1571209, 1962705, 8055131, 9836437, 23724145, 50200909, 69076661, 262541543, 187053941, 813855239, 2096008901},
  {1, 3, 1, 13, 9, 23, 89, 165, 59, 257, 1369, 161, 6255, 2997, 19175, 26947, 98961, 165111, 185631, 56195, 485045, 3444711, 7136583, 6362465, 26697607, 49793355, 102960579, 31130653, 39136507, 661868445, 743791133},
  {1, 3, 5, 3, 5, 41, 107, 231, 111, 207, 1865, 2079, 5891, 2487, 5863, 33835, 97169, 202451, 315993, 980371, 1619859, 3110889, 1094017, 8928625, 30818049, 18548967, 66350461, 67533901, 251786177, 411223449, 131896469},
  {1, 3, 7, 15, 3, 3, 105, 235, 263, 991, 367, 1885, 1769, 7805, 11909, 32855, 110833, 166833, 23473, 248973, 1394357, 1424031, 1590089, 5091425, 9482593, 58159297, 130250163, 166415011, 118843871, 23701519, 1342205965},
  {1, 3, 3, 5, 15, 59, 67, 247, 77, 367, 1641, 1959, 1921, 5939, 17355, 10355, 104625, 217301, 148627, 16533, 137621, 812421, 8108449, 8313299, 30140613, 64145671, 92262021, 85477023, 527912001, 837639101, 141147157},
  {1, 1, 7, 1, 3, 53, 37, 5, 221, 779, 1353, 1633, 2769, 6355, 8505, 2967, 78963, 101973, 250559, 453009, 1792163, 3059647, 7787231, 3559015, 22100065, 26831759, 39082787, 59910979, 139452737, 39924879, 158311285},
  {1, 1, 7, 13, 11, 13, 73, 227, 115, 523, 355, 3127, 7545, 8409, 22335, 435, 128179, 125557, 293779, 918401, 925067, 1478307, 593441, 10957701, 4076477, 42268005, 84351809, 265031335, 86775835, 452729661, 834934589},
  {1, 1, 5, 11, 21, 15, 91, 115, 427, 683, 461, 2433, 6313, 4595, 24401, 35463, 53491, 82775, 3633, 135571, 1150133, 3639189, 8351477, 2058909, 11933069, 20245163, 119345579, 260074651, 128399761, 766327783, 1467698101},
  {1, 3, 7, 5, 29, 21, 57, 215, 423, 717, 1455, 705, 6835, 4503, 26077, 46751, 79313, 122961, 436987, 1039495, 832671, 437731, 831093, 14432857, 16760147, 66069285, 15476847, 50887105, 369007265, 965986691, 266912949},
  {1, 1, 1, 15, 3, 33, 25, 227, 381, 477, 1023, 2751, 2229, 631, 16903, 30363, 68211, 188179, 55517, 216973, 1928375, 344779, 5427361, 11505675, 8204727, 30758485, 32510705, 68602523, 8966733, 518903961, 1376770021},
  {1, 3, 1, 11, 9, 17, 59, 73, 53, 671, 251, 1729, 7593, 12473, 22533, 62407, 55665, 213495, 325177, 328591, 1890227, 641417, 7402511, 7312823, 22587237, 19116021, 132428583, 172300283, 514665371, 324369991, 1574537565},
  {1, 3, 3, 1, 3, 35, 37, 173, 459, 143, 135, 3871, 2689, 8007, 4379, 35011, 61745, 228501, 372247, 283025, 138389, 4180179, 62583, 2977057, 30932137, 41484105, 93253953, 239795599, 94166069, 905373029, 1052646021},
  {1, 3, 5, 9, 23, 19, 43, 45, 493, 509, 1851, 1615, 5675, 13793, 6973, 10439, 10385, 169395, 417331, 867477, 808333, 2669021, 2387051, 11093319, 17282455, 41836273, 12191081, 120979941, 165552103, 949354959, 2080536781},
  {1, 3, 3, 15, 5, 17, 77, 85, 451, 753, 579, 1057, 4851, 6017, 4195, 20667, 102481, 110261, 395609, 208651, 837931, 3059551, 3807551, 4863033, 23617383, 45276273, 120692471, 11749249, 250992759, 914213457, 1558821741},
  {1, 3, 3, 5, 31, 29, 81, 159, 103, 391, 15, 899, 4623, 5957, 31961, 11967, 81873, 114165, 438387, 113925, 2025747, 2603867, 4895597, 3075497, 33214361, 51097873, 64697197, 8310905, 343724587, 1038815139, 778277949},
  {1, 1, 1, 7, 17, 57, 81, 17, 177, 633, 49, 2793, 5229, 5995, 9491, 44611, 13875, 232659, 216725, 839439, 580363, 3781939, 446915, 14444675, 12867979, 39819475, 123540635, 100041647, 276150561, 826929621, 752952309},
  {1, 1, 7, 15, 17, 19, 65, 57, 189, 239, 1229, 929, 2681, 12845, 29311, 19619, 69235, 2101, 341393, 53023, 251681, 4002583, 2074955, 3145951, 5243301, 24118787, 131073707, 265295851, 303053615, 540023797, 2131798861},
  {1, 3, 1, 11, 13, 47, 61, 203, 383, 875, 943, 139, 4217, 8279, 1047, 21211, 97713, 253463, 105817, 341771, 349445, 1846131, 7824129, 907065, 8134245, 1846697, 97977957, 76743071, 360785685, 663712073, 7288645},
  {1, 3, 7, 13, 23, 7, 1, 69, 47, 537, 1325, 3101, 685, 14057, 19953, 30991, 127921, 26097, 397555, 228637, 445721, 1952127, 5149079, 11562709, 2151223, 61902863, 23925015, 189457767, 327232291, 75853299, 794645565},
  {1, 3, 3, 1, 1, 7, 39, 77, 47, 755, 527, 2985, 5433, 15095, 27741, 28367, 66033, 245909, 340215, 334099, 518453, 1681177, 3296531, 12861349, 16565213, 44646553, 116477031, 25478043, 127826613, 162844739, 110186469},
  {1, 1, 7, 5, 23, 57, 79, 155, 81, 937, 1071, 3929, 1655, 3831, 17351, 33163, 89363, 141909, 76251, 851981, 2031623, 65613, 6750293, 2293783, 26804379, 30476173, 16191159, 231808469, 457128073, 797908185, 957346709},
  {1, 3, 7, 1, 3, 41, 13, 235, 207, 487, 1883, 2247, 1231, 2751, 15615, 31387, 110417, 38993, 269919, 192529, 1421343, 1749039, 1863905, 16232745, 17052393, 57422917, 107624641, 231238065, 192728401, 610634801, 2037858069},
  {1, 1, 7, 1, 21, 57, 95, 191, 119, 483, 283, 2221, 5665, 14819, 26097, 5767, 127731, 138965, 178111, 578567, 234499, 1018973, 298001, 14476377, 25984669, 38056749, 124163435, 41020427, 278642577, 365593479, 1484112821},
  {1, 3, 1, 1, 9, 59, 27, 51, 393, 31, 925, 715, 7705, 14885, 28767, 43203, 23089, 115511, 112787, 214043, 1268761, 619645, 7744533, 6663551, 13305021, 16377363, 38464193, 98215711, 122135527, 621738437, 339880141},
  {1, 1, 3, 3, 3, 61, 109, 131, 113, 249, 1331, 2521, 2973, 6375, 20093, 58479, 34483, 171537, 177397, 334357, 149035, 783911, 2744937, 3931975, 5333451, 18505745, 37301647, 53672255, 14100045, 708146819, 1009475333},
  {1, 3, 7, 9, 31, 37, 125, 245, 237, 245, 111, 379, 7495, 15531, 2325, 57055, 20049, 212753, 64631, 537117, 209451, 3426887, 156319, 8261367, 7586175, 24559701, 79282421, 126010209, 388855669, 303037619, 613179717},
  {1, 3, 7, 13, 21, 21, 57, 21, 449, 969, 417, 2999, 509, 639, 7797, 9375, 75185, 131281, 252723, 160223, 1883983, 1280419, 745207, 13912827, 22503387, 7881827, 58311781, 15353923, 398719701, 879117403, 1598707653},
  {1, 3, 7, 7, 7, 29, 11, 175, 55, 705, 891, 863, 3021, 10071, 10267, 30755, 111537, 169297, 303321, 891865, 1683683, 2837121, 1555081, 4721497, 1166143, 23349769, 106466713, 133010499, 442139805, 899057989, 602806309},
  {1, 1, 3, 13, 19, 17, 127, 57, 449, 579, 337, 899, 1235, 11269, 4245, 49479, 32979, 145073, 502907, 246617, 557927, 1202093, 581335, 12199619, 5009425, 48206439, 79731057, 213000837, 482829907, 593216335, 512894493},
  {1, 1, 1, 11, 29, 61, 35, 75, 249, 683, 287, 45, 3277, 7521, 2073, 27903, 75507, 65075, 125241, 497755, 1525335, 1539977, 5471357, 839039, 15666889, 52678841, 15029091, 20586823, 27717899, 268647659, 2008255581},
  {1, 3, 5, 5, 15, 25, 77, 63, 63, 801, 1387, 1533, 2185, 10899, 28381, 31903, 73137, 204691, 405311, 980181, 1681271, 579567, 5504181, 11752561, 4128015, 49226425, 43016475, 260942659, 233428177, 321151243, 1448357173},
  {1, 3, 1, 1, 21, 49, 3, 249, 419, 575, 87, 3749, 2523, 16125, 9483, 8987, 59281, 190007, 267411, 775751, 1283563, 3669937, 283567, 8950721, 15518633, 11058905, 2489435, 56650209, 463237755, 1034096833, 1762228765},
  {1, 1, 1, 11, 21, 43, 85, 211, 449, 439, 1495, 1841, 4765, 15253, 1467, 53739, 120691, 75891, 326809, 160851, 168913, 4116435, 1935113, 15734519, 20018629, 58442793, 51105007, 266848047, 243824287, 137568545, 1070445005},
  {1, 3, 3, 15, 3, 37, 31, 243, 187, 995, 1103, 2723, 1523, 15967, 28649, 56599, 30161, 215829, 360441, 1000141, 517971, 439461, 6802301, 5089581, 1427845, 1228345, 94249677, 263714117, 42837449, 656836583, 388040373},
  {1, 1, 5, 11, 9, 11, 17, 87, 335, 125, 1079, 1657, 1237, 8059, 29833, 41719, 15347, 144279, 328817, 304591, 168265, 2887831, 5406469, 4449713, 6692627, 815901, 26956327, 102051863, 208853825, 491418703, 1784070069},
  {1, 3, 1, 3, 3, 41, 35, 37, 33, 61, 505, 3203, 5, 101, 8571, 56363, 36785, 223095, 82833, 543829, 1845727, 1259937, 1792275, 2237595, 3685807, 40730611, 121893345, 262465619, 28108415, 433890273, 1570930733},
  {1, 1, 3, 11, 9, 11, 85, 235, 261, 473, 109, 2127, 5745, 6389, 7431, 28467, 112627, 93297, 32733, 106703, 394313, 1853491, 4970673, 3254403, 29739427, 55549135, 49003749, 7797759, 281312071, 235430781, 1494777741},
  {1, 1, 5, 15, 3, 55, 77, 97, 17, 193, 1267, 3063, 6531, 9797, 8639, 58579, 50131, 69943, 484437, 887245, 1697889, 3665971, 6213707, 2675107, 8097875, 33648993, 59447745, 191695933, 525930771, 150435397, 1615976605},
  {1, 1, 5, 5, 25, 41, 79, 83, 485, 697, 149, 1023, 89, 6115, 15227, 35795, 87411, 72215, 395487, 70467, 118347, 1208845, 4228829, 4372371, 582819, 66265431, 98853797, 197620791, 53636609, 310784685, 138065557},
  {1, 1, 3, 15, 1, 9, 73, 251, 33, 599, 1017, 353, 4305, 16033, 29663, 15315, 18995, 4401, 471769, 127439, 1951323, 2075179, 4786905, 2225927, 3665733, 5496199, 7582051, 197879875, 228774579, 686917453, 301445565},
  {1, 3, 7, 15, 3, 1, 89, 39, 125, 337, 1445, 3131, 3685, 9849, 25829, 7015, 27185, 251953, 405105, 312269, 1934583, 3362411, 5471909, 13420483, 11817083, 60396631, 32529217, 14707075, 282669155, 232619783, 2043275101},
  {1, 3, 7, 3, 19, 1, 63, 179, 349, 135, 185, 2977, 2527, 15087, 18133, 41871, 126929, 85777, 335549, 487493, 1617623, 1053965, 2242365, 16603595, 711149, 7850375, 89856675, 47562637, 206103777, 812587227, 341151477},
  {1, 1, 3, 3, 23, 7, 91, 221, 325, 723, 345, 81, 8077, 5501, 8453, 64503, 39155, 228465, 428533, 875969, 1598457, 3019557, 6839067, 71759, 20699625, 12427151, 62555635, 244875967, 135835415, 322735151, 1880485165},
  {1, 1, 3, 9, 7, 3, 13, 173, 479, 161, 1989, 3255, 2069, 6717, 559, 26211, 17843, 92017, 114527, 418245, 1663325, 3142523, 2287975, 4985141, 3043439, 25572503, 114528301, 165550283, 25835591, 1055260869, 645839885},
  {1, 3, 3, 5, 9, 61, 93, 203, 277, 367, 1141, 981, 4745, 12625, 21003, 54779, 104497, 59957, 170131, 565459, 31327, 3902295, 1745377, 12371411, 7515861, 14315067, 118966735, 222383667, 181125491, 829399993, 1841766525},
  {1, 3, 5, 5, 27, 17, 5, 211, 403, 701, 5, 3091, 4611, 5615, 23667, 54819, 8273, 255123, 236767, 631361, 672599, 1464631, 4873545, 14505413, 10503883, 13935343, 49097833, 71679317, 375950329, 785553981, 1062647733},
  {1, 1, 3, 1, 21, 61, 125, 77, 57, 463, 1499, 791, 2087, 2805, 18829, 40151, 130707, 40369, 380023, 394951, 131527, 838519, 6510567, 10894387, 2543405, 23479253, 24135217, 202165209, 163834891, 844719767, 748795005},
  {1, 3, 5, 3, 11, 41, 125, 231, 119, 837, 831, 1331, 7439, 2381, 3759, 51, 49809, 223539, 515929, 361181, 1348431, 2805055, 274413, 11580233, 24842491, 37204901, 40754813, 21647961, 8319427, 324242693, 208531997},
  {1, 3, 1, 11, 19, 59, 117, 107, 443, 699, 315, 1491, 2581, 15871, 17159, 50299, 73521, 90455, 119705, 285525, 1425517, 4190035, 1906673, 4516637, 2539965, 49070301, 34424533, 110608963, 271173029, 318146185, 979660933},
  {1, 3, 5, 11, 5, 9, 121, 35, 209, 877, 527, 3493, 4657, 16093, 17589, 6327, 7089, 233459, 306417, 278979, 973299, 427387, 4079009, 188039, 23591327, 28735749, 46194891, 59568427, 417509295, 947846847, 806308301},
  {1, 1, 7, 15, 9, 43, 119, 29, 381, 479, 1443, 3171, 5053, 9625, 21161, 5375, 112563, 77365, 432561, 268999, 1107177, 988433, 3890499, 9411351, 15734101, 53660205, 133460277, 66130539, 223294803, 146215403, 1270343517},
  {1, 1, 3, 5, 15, 21, 31, 223, 83, 399, 1529, 3605, 6343, 10469, 10099, 2587, 675, 219361, 384163, 306789, 623467, 2266197, 6347969, 480653, 28872073, 54143879, 127553351, 137878733, 301033675, 494012313, 1084159005},
  {1, 1, 3, 5, 5, 45, 23, 123, 353, 971, 85, 3069, 3245, 6569, 13241, 2927, 4003, 222625, 419107, 304943, 761287, 3662893, 3802229, 11687415, 27596009, 43847459, 56213239, 204129783, 396684819, 58289611, 1867061373},
  {1, 1, 1, 3, 25, 49, 5, 77, 491, 881, 993, 1195, 7677, 5709, 10807, 30723, 1379, 110499, 235937, 169727, 569443, 441351, 975911, 7636305, 6688583, 50583907, 26604425, 49782859, 322680727, 7790157, 1943408653},
  {1, 3, 3, 3, 5, 49, 127, 255, 183, 583, 1599, 987, 7281, 7149, 28507, 35643, 94049, 10789, 227749, 896867, 82779, 3446845, 6142113, 14412265, 1805129, 12025297, 37090309, 240448487, 450675247, 402699681, 344341421},
  {1, 1, 5, 1, 13, 55, 55, 157, 197, 25, 1971, 3161, 3903, 8919, 13563, 39619, 22307, 168935, 151275, 523119, 37965, 819257, 5865491, 6258847, 15302771, 39879201, 125798431, 220629485, 234128201, 537576245, 2059043605},
  {1, 3, 7, 9, 3, 37, 79, 193, 25, 103, 843, 2651, 6341, 2653, 24337, 52943, 53601, 673, 294311, 568369, 837571, 1216037, 4032127, 9637755, 21995637, 33579493, 43573869, 259487123, 385855159, 434236171, 951778893},
  {1, 1, 7, 3, 25, 49, 99, 139, 45, 211, 2033, 2331, 7037, 7177, 1755, 32203, 9571, 173477, 169773, 526719, 2023267, 4187745, 3487469, 5464731, 649577, 24788023, 27701465, 15843051, 223426451, 163382761, 1103586461},
  {1, 3, 7, 3, 5, 19, 127, 135, 403, 221, 141, 1065, 3935, 2745, 25979, 57979, 47905, 24033, 163117, 599395, 1625465, 2452089, 7519833, 5333821, 17009051, 66231319, 60495891, 202130989, 12895655, 166232425, 2106763373},
  {1, 1, 3, 3, 31, 23, 111, 37, 261, 7, 835, 2379, 7927, 8181, 23751, 47187, 61027, 138209, 240549, 585209, 1033033, 3931937, 7371227, 3367151, 4612141, 57927021, 68396269, 66448385, 260799739, 565652877, 1936813437},
  {1, 3, 7, 15, 1, 39, 79, 3, 103, 427, 1917, 809, 5039, 689, 1939, 36619, 75297, 84705, 331553, 618367, 16965, 3807521, 1743805, 4141889, 31288197, 5683595, 23868659, 89239901, 442836175, 164054865, 1031372877},
  {1, 1, 1, 15, 29, 37, 39, 243, 149, 353, 763, 3405, 5751, 9441, 6653, 48663, 1187, 165859, 151341, 862883, 372223, 3144973, 7805405, 7429075, 24825643, 66872537, 35630107, 121846861, 495674735, 1027186783, 49038285},
  {1, 3, 3, 11, 1, 57, 125, 151, 445, 423, 841, 2265, 5017, 15863, 13057, 16775, 95777, 99365, 26605, 166199, 1980891, 1409839, 2446797, 4074035, 24237373, 34418279, 105436843, 94321803, 298309541, 242045007, 201832037},
  {1, 3, 5, 13, 11, 49, 61, 159, 211, 917, 561, 1903, 3985, 11117, 28969, 7327, 61153, 186147, 257191, 252465, 922311, 1950079, 3947909, 4332045, 18126099, 50843387, 4815853, 30249199, 222397167, 455322259, 2083812525},
  {1, 3, 5, 13, 29, 5, 35, 51, 91, 291, 9, 3713, 3341, 4551, 12085, 31887, 33441, 44579, 178535, 698663, 1570655, 1528201, 7868053, 12446557, 13016565, 5226799, 35832627, 69095983, 118215677, 629545371, 1975702661},
  {1, 3, 3, 1, 1, 39, 111, 141, 319, 179, 1709, 1605, 5063, 13279, 10003, 14515, 59937, 233829, 8295, 569571, 578245, 1388161, 1348211, 12135237, 27425069, 40050155, 116134335, 233671853, 325684593, 891005325, 2134070901},
  {1, 1, 3, 9, 7, 59, 91, 41, 343, 475, 1669, 2311, 5141, 12661, 25847, 4963, 71267, 204577, 316463, 286581, 14293, 1998541, 7034431, 4743781, 5726805, 2499859, 79897645, 211188347, 437320319, 656796221, 167231853},
  {1, 3, 5, 9, 9, 11, 49, 221, 1, 243, 791, 229, 503, 373, 19189, 17751, 90145, 127587, 36643, 379963, 359289, 3104647, 921183, 13915, 11928337, 21889873, 18484395, 20578397, 52313979, 485633727, 40823837},
  {1, 1, 5, 11, 17, 13, 45, 57, 215, 491, 1601, 2183, 3713, 429, 22007, 45219, 124003, 229799, 355617, 444263, 801231, 3896535, 908291, 15932421, 16111141, 4952567, 49069669, 103936367, 119358095, 435762829, 940287533},
  {1, 1, 3, 11, 31, 61, 23, 237, 261, 955, 1085, 1541, 2601, 909, 7749, 55079, 1571, 177825, 308269, 276265, 130403, 2946445, 2016355, 3053055, 6993169, 49068715, 103161951, 183575747, 382948287, 1065590271, 355118157},
  {1, 1, 3, 9, 13, 11, 121, 173, 177, 551, 1757, 2745, 2265, 4611, 743, 36379, 49059, 180385, 155631, 140159, 1915377, 1509007, 92415, 1091723, 17594661, 10378291, 15397987, 250948555, 326347345, 297910465, 1007393077},
  {1, 1, 3, 15, 23, 43, 107, 239, 463, 369, 1857, 1073, 1247, 1029, 22557, 36935, 20387, 242785, 449321, 671529, 251749, 991709, 6035609, 3959217, 219727, 41302659, 9531027, 12383773, 56721947, 118989431, 1947872157},
  {1, 1, 3, 11, 23, 35, 89, 93, 41, 941, 1141, 2339, 1423, 8007, 28685, 35535, 26851, 162145, 459629, 478241, 1563757, 2352383, 3974479, 1599155, 17172831, 46489103, 21490857, 4764329, 454090681, 432355555, 1932487861},
  {1, 3, 5, 13, 29, 7, 79, 15, 59, 145, 1237, 2215, 1257, 12621, 31101, 61559, 39521, 203171, 98343, 2343, 1855069, 207073, 7294321, 1664581, 33411975, 59971543, 35098797, 264647431, 517961407, 580759943, 1142423789},
  {1, 1, 3, 7, 13, 55, 57, 229, 205, 1009, 341, 3901, 5189, 957, 32587, 22043, 8483, 236577, 61217, 800227, 2073549, 2011063, 1177399, 1541799, 12952715, 43799959, 29570295, 150092895, 536188503, 807843025, 1651033069},
  {1, 3, 7, 11, 1, 1, 41, 7, 365, 407, 1609, 1423, 6483, 5171, 32519, 45355, 77921, 157921, 293861, 468855, 1009379, 341387, 4244469, 12660547, 5879597, 49063303, 86877693, 137966445, 7442805, 616297025, 1618830405},
  {1, 3, 7, 3, 17, 31, 125, 27, 125, 335, 1395, 2639, 329, 2549, 14449, 34319, 96865, 253089, 68909, 931511, 1283933, 714979, 6903617, 926059, 33399765, 25847501, 132277065, 114489207, 286000007, 360451515, 1004850893},
  {1, 3, 3, 7, 19, 45, 11, 73, 123, 179, 1685, 3385, 2379, 3387, 16793, 47255, 87713, 64485, 177121, 200381, 382827, 127217, 4145023, 13808713, 28032357, 1217043, 12204211, 30310745, 73172493, 476676303, 1733096773},
  {1, 3, 7, 5, 31, 25, 47, 153, 121, 453, 935, 3953, 2081, 12145, 24979, 34683, 5281, 221089, 183979, 858869, 937927, 3315133, 3314023, 3472555, 8435479, 4687181, 51154207, 250904995, 185896147, 939361809, 526975101},
  {1, 1, 7, 13, 25, 11, 65, 3, 277, 237, 1129, 1801, 4165, 9065, 18747, 8939, 108451, 175141, 367011, 267747, 1956313, 3611063, 6209953, 10376435, 19192871, 27582227, 8280251, 154408695, 103028227, 916745097, 1049934717},
  {1, 1, 7, 7, 13, 5, 37, 253, 507, 645, 1355, 3401, 6707, 6329, 11237, 4455, 14691, 12709, 507753, 1035107, 723263, 67343, 5988183, 15185377, 12051923, 23907297, 115906687, 251540993, 483214015, 267776823, 1357659853},
  {1, 1, 3, 15, 17, 49, 3, 233, 407, 451, 69, 3859, 3171, 12303, 21031, 9883, 127651, 84513, 503977, 988847, 294195, 1900865, 8241551, 12348581, 14585293, 14498531, 57728249, 261163621, 248323881, 74515801, 156360565},
  {1, 1, 3, 3, 9, 53, 119, 117, 401, 903, 1449, 3639, 4083, 2095, 22085, 31007, 104227, 102177, 31397, 697071, 1002759, 3864893, 5295099, 11478043, 29912197, 21385991, 19168389, 68519997, 331582697, 938399227, 1819505077},
  {1, 3, 7, 15, 5, 61, 117, 193, 137, 431, 195, 4019, 3047, 5049, 14281, 23231, 126241, 171425, 342881, 348411, 1658135, 359279, 59339, 5081579, 5323549, 41376573, 51373973, 61239713, 220246103, 548980827, 2136187981},
  {1, 1, 1, 15, 17, 19, 29, 83, 449, 257, 1105, 1949, 1749, 3459, 6343, 22203, 61795, 116579, 47533, 942255, 830737, 2833179, 1384265, 3138439, 23563171, 25153331, 82078111, 246211727, 167993609, 583532769, 187198485},
  {1, 1, 1, 15, 23, 39, 61, 219, 109, 365, 863, 1813, 6673, 15999, 5101, 16319, 6371, 24547, 492589, 492009, 1067049, 2005075, 5982337, 7546939, 31693655, 15535461, 115432971, 82343771, 375046141, 42306227, 796826501},
  {1, 1, 5, 5, 13, 11, 37, 151, 365, 719, 1233, 2425, 1285, 1721, 1205, 33015, 6115, 113511, 454127, 55655, 1757233, 61459, 69757, 8622307, 5001653, 61837359, 51860603, 19625239, 311562515, 875175975, 2081612733},
  {1, 3, 3, 3, 7, 53, 109, 153, 45, 425, 1741, 1229, 4405, 8071, 25155, 50339, 59745, 116837, 60453, 584737, 1392667, 16487, 4800547, 12075391, 22626579, 22102847, 84362487, 42191759, 512948301, 1069197421, 1341036933},
  {1, 3, 1, 1, 1, 13, 39, 49, 413, 77, 1367, 2553, 5563, 7659, 3467, 51379, 116577, 223655, 106019, 204067, 162863, 2704477, 6835295, 16535007, 8654679, 17344621, 59237303, 117374057, 319154493, 765694973, 1010771269},
  {1, 1, 5, 9, 3, 49, 23, 11, 445, 121, 1505, 877, 4137, 1809, 2429, 615, 84067, 261095, 510947, 194161, 1947159, 1904213, 7802501, 6821515, 12439907, 26562867, 51023951, 18429331, 259942115, 591452799, 541946557},
  {1, 1, 1, 11, 21, 13, 93, 33, 493, 805, 775, 2939, 2961, 13625, 31879, 58515, 120099, 15971, 359913, 67875, 397831, 3122791, 2137787, 4575951, 29155423, 38315757, 106013909, 183463751, 299619771, 679118933, 1708009341},
  {1, 1, 7, 5, 1, 59, 63, 131, 373, 23, 337, 2107, 5315, 4889, 22851, 64395, 10659, 97573, 78699, 270507, 1929753, 2370153, 7339549, 7316179, 31185117, 4271135, 11504505, 61314773, 343322175, 282305425, 2031326509},
  {1, 1, 3, 13, 21, 47, 15, 131, 353, 793, 1891, 1757, 5793, 1147, 23697, 62295, 109283, 49121, 243371, 879983, 890405, 2938481, 3871485, 5828551, 8751483, 29228721, 95250427, 243546491, 60872473, 464412999, 1716245},
  {1, 3, 5, 13, 7, 59, 25, 135, 259, 109, 1835, 429, 8153, 7355, 145, 62871, 94945, 164067, 165031, 132477, 23061, 45583, 7108117, 12198829, 16526475, 52740753, 129419995, 224412259, 391117161, 476782151, 203493333},
  {1, 3, 3, 13, 9, 47, 121, 89, 89, 635, 1079, 2353, 4803, 11369, 12653, 33847, 103137, 167717, 469739, 616179, 392861, 3266759, 561483, 11469771, 3925865, 1960609, 33306111, 157649665, 168861519, 988467551, 1140802445},
  {1, 3, 5, 9, 23, 39, 49, 231, 105, 603, 613, 2021, 6073, 11819, 10595, 64435, 31969, 9571, 286243, 930725, 1247145, 1960863, 2467749, 14712327, 29331433, 24168883, 64063887, 94608915, 370577913, 1048094421, 148697173},
  {1, 3, 7, 7, 7, 19, 19, 155, 347, 387, 1459, 3793, 619, 14437, 2455, 3507, 113121, 3617, 463209, 634537, 1436861, 27093, 4689885, 2644493, 20420949, 66645589, 104260951, 55981097, 57219923, 236153277, 1627218045},
  {1, 1, 1, 15, 21, 35, 19, 185, 483, 425, 479, 3429, 5403, 10791, 14219, 42691, 66659, 26403, 49453, 199083, 134569, 3457013, 3246911, 12632817, 3564367, 6365805, 114502459, 142058097, 470260977, 818544485, 486882485},
  {1, 1, 3, 11, 5, 51, 105, 63, 493, 677, 1457, 2865, 5619, 9321, 19583, 51219, 77155, 33057, 519917, 844467, 481433, 1504175, 6212877, 13569395, 23569759, 48186395, 12828915, 185423089, 159046063, 213906061, 1904812877},
  {1, 1, 3, 3, 23, 1, 77, 177, 263, 289, 1567, 3837, 5359, 3269, 16023, 32595, 355, 237985, 158309, 629937, 994959, 3207151, 3810763, 8533381, 13483343, 17394429, 85946339, 105360565, 43226747, 778891965, 124825373},
  {1, 1, 7, 3, 13, 61, 79, 77, 51, 953, 1417, 795, 4467, 2981, 25131, 32043, 94243, 245669, 27501, 1019179, 307847, 1258901, 30131, 291145, 11006207, 47322075, 82179177, 2915685, 180372899, 598178753, 70634717},
  {1, 1, 5, 13, 23, 13, 29, 185, 337, 7, 149, 3609, 8119, 9545, 16579, 39115, 70755, 86695, 51943, 450605, 957315, 3690663, 6577827, 14545667, 9283781, 34487739, 26227987, 46706725, 41552007, 1054233649, 514478445},
  {1, 3, 1, 5, 23, 9, 123, 15, 99, 55, 1021, 3709, 1521, 15189, 22193, 19247, 40673, 112487, 493799, 1034621, 738183, 4113801, 3960409, 15021917, 5609617, 38043187, 112346983, 30363243, 137783127, 1019861883, 208840301},
  {1, 3, 7, 9, 13, 41, 39, 45, 49, 181, 1587, 3213, 1037, 14775, 3333, 59567, 86881, 199905, 177063, 243583, 2026131, 3781013, 1652227, 4829899, 21968119, 30055673, 61728907, 207993399, 502317581, 511930443, 640781893},
  {1, 1, 1, 7, 29, 55, 59, 31, 411, 601, 191, 283, 3211, 7951, 7919, 23755, 120483, 136611, 341285, 1035315, 192685, 595189, 4695753, 13805509, 13814863, 65216365, 37009285, 125297885, 94598905, 914241105, 238479733},
  {1, 1, 7, 7, 21, 47, 7, 193, 343, 831, 1267, 3289, 1015, 13093, 2717, 39623, 100835, 242341, 469865, 225531, 250277, 1382393, 5401263, 4364981, 4746929, 28140269, 19539295, 196224997, 246696875, 720929207, 211888157},
  {1, 3, 7, 1, 17, 9, 97, 19, 279, 827, 1699, 3573, 3137, 3535, 17791, 1243, 8801, 35169, 411055, 237619, 907403, 1736339, 5071025, 5136529, 16818357, 8266101, 84403219, 237105483, 194602797, 514199553, 1256911301},
  {1, 1, 5, 11, 27, 15, 103, 135, 35, 625, 1575, 97, 7013, 13353, 19333, 36439, 43491, 39143, 237665, 970477, 165785, 2047449, 5341289, 229581, 25854935, 15304357, 58625359, 119049415, 191542531, 925675127, 67980349},
  {1, 3, 3, 7, 17, 13, 49, 135, 435, 743, 1799, 2655, 4839, 2893, 31153, 53727, 67105, 245541, 115809, 622975, 185231, 2054091, 3024901, 11758045, 29669921, 10287337, 52364017, 57083961, 297736291, 596985931, 1781109725},
  {1, 1, 5, 1, 3, 41, 1, 195, 53, 803, 1575, 2939, 3873, 10495, 5211, 29363, 11939, 45799, 348075, 629921, 383119, 114931, 2534625, 2475283, 8586665, 38691265, 21045909, 154870071, 333059229, 297692229, 149586021},
  {1, 3, 1, 15, 19, 19, 37, 59, 355, 483, 685, 3899, 4645, 15127, 3479, 28747, 80705, 24263, 96589, 136653, 554485, 3375811, 2465841, 15884501, 22224997, 56451627, 58894113, 195570835, 93692701, 957996793, 197329445},
  {1, 1, 5, 3, 25, 9, 9, 229, 101, 631, 1165, 4091, 3723, 10655, 9463, 58843, 29507, 11975, 127689, 653279, 1093115, 2803419, 3588151, 12468335, 15319293, 53168035, 112609313, 104159165, 436029769, 804150281, 1872118645},
  {1, 3, 5, 15, 5, 13, 91, 61, 19, 469, 1675, 3331, 3121, 3435, 4111, 19819, 51905, 170179, 517189, 836315, 1027015, 2753153, 108747, 16431273, 25249971, 40225761, 118139349, 30785143, 412958665, 216467609, 374053301},
  {1, 1, 7, 1, 31, 61, 23, 83, 165, 551, 1097, 3825, 5385, 4723, 3635, 9683, 112707, 68165, 182351, 901853, 981699, 1634989, 2106045, 7375683, 33114797, 51572039, 114811043, 7076267, 463651105, 68859925, 1699353845},
  {1, 3, 7, 15, 9, 31, 11, 121, 503, 855, 561, 1647, 1229, 1147, 15997, 34223, 100929, 141761, 338241, 932055, 1432909, 2057717, 5638703, 2879301, 19759865, 19564383, 11374413, 163617619, 402621473, 615565771, 1254965525},
  {1, 3, 7, 13, 21, 47, 41, 195, 197, 719, 1263, 3609, 7515, 2659, 30713, 7895, 91585, 189505, 405699, 347855, 1758277, 3005367, 5086929, 14867203, 14984581, 21207217, 103304327, 215958185, 274330261, 440603135, 1671013989},
  {1, 1, 1, 7, 31, 61, 101, 101, 479, 571, 605, 301, 6633, 15587, 23665, 47959, 36163, 230723, 441413, 438993, 1889859, 1284063, 1599087, 295893, 20401861, 52159595, 114664279, 189232755, 71706353, 763250423, 1394244853},
  {1, 3, 7, 3, 25, 39, 35, 225, 135, 463, 53, 709, 5129, 4135, 10421, 14927, 100417, 241729, 518477, 812767, 683605, 2746029, 6826983, 1438725, 9227393, 46610283, 119871727, 251740099, 378366709, 653512379, 1383118437},
  {1, 1, 5, 13, 19, 55, 107, 15, 163, 287, 673, 899, 5197, 4619, 3465, 48711, 127299, 28103, 428103, 694217, 664913, 2012357, 7779097, 7671933, 4829337, 36706271, 4652929, 148132107, 450358225, 874872287, 1675832341},
  {1, 3, 3, 5, 21, 49, 15, 105, 283, 877, 1875, 1079, 3431, 13053, 26599, 22803, 125889, 191685, 368323, 144479, 1428187, 1273261, 3523383, 11788169, 10335387, 24491849, 134173521, 253451593, 327535347, 298367325, 282878749},
  {1, 1, 7, 1, 1, 1, 95, 113, 119, 575, 1159, 2325, 6895, 12177, 4369, 45487, 20163, 234053, 214223, 481603, 598851, 340637, 6125839, 6428373, 3836433, 34245945, 18315227, 107742341, 485229199, 287592051, 1280110797},
  {1, 1, 1, 11, 25, 25, 83, 207, 301, 729, 1947, 2321, 3621, 15707, 11303, 30971, 79683, 245955, 305865, 435535, 1069419, 124449, 7813673, 5077747, 3326019, 19940969, 112542311, 7003847, 272490705, 678621137, 1002238197},
  {1, 1, 5, 5, 7, 63, 83, 105, 211, 175, 1817, 2883, 5385, 7437, 24865, 30895, 92867, 97095, 221903, 650445, 110577, 2420333, 6991503, 345921, 17582665, 40299911, 110139569, 196830543, 236567391, 740280827, 471629613},
  {1, 3, 7, 5, 23, 39, 19, 211, 151, 295, 573, 223, 5065, 6345, 23187, 49563, 62401, 87617, 30923, 848349, 29001, 937501, 6231605, 10674929, 744265, 9210803, 112225125, 130884791, 321437083, 257804481, 1838269309},
  {1, 1, 7, 11, 15, 31, 89, 123, 57, 695, 685, 1799, 659, 9929, 22933, 2503, 67011, 185413, 1605, 309465, 863937, 2922023, 328329, 4365071, 6684165, 5952003, 100736925, 258908381, 151846703, 86247495, 1850001709},
  {1, 1, 7, 7, 19, 17, 27, 137, 117, 141, 1481, 869, 7061, 3073, 19671, 38131, 54723, 138437, 455369, 943069, 523383, 1743481, 5829375, 3350183, 30201767, 12121619, 27203255, 17209759, 341619563, 447207317, 39169053},
  {1, 3, 3, 11, 9, 19, 123, 93, 39, 517, 883, 3769, 2267, 8089, 6617, 43391, 128321, 181957, 157005, 378591, 495041, 2731549, 424555, 12380765, 10139275, 42454841, 23920287, 245307753, 157880175, 1064256811, 1965775245},
  {1, 3, 1, 7, 9, 61, 51, 241, 319, 853, 1239, 899, 105, 1677, 29351, 51043, 93761, 148551, 129733, 495303, 1899631, 1718281, 2070615, 7719069, 20572395, 29603901, 3513709, 139297711, 108364319, 67147901, 380129133},
  {1, 1, 7, 15, 13, 59, 85, 175, 223, 87, 905, 3175, 3405, 3489, 18475, 5739, 120003, 151621, 86977, 143315, 1602913, 970851, 2175045, 14159937, 18936361, 5763175, 21058997, 109580483, 113782011, 934810793, 1076388861},
  {1, 1, 1, 15, 1, 55, 79, 97, 315, 605, 851, 4015, 3689, 9371, 31523, 53363, 103235, 39235, 120013, 953183, 1344117, 300129, 7621823, 13612473, 16328683, 21644201, 28688585, 64636663, 333274121, 885340373, 715206005},
  {1, 1, 5, 15, 1, 39, 91, 27, 211, 881, 1375, 2307, 5791, 10185, 23093, 59111, 92483, 121159, 250949, 267231, 983013, 3485013, 3318253, 9046693, 1650199, 61006077, 61221757, 138787161, 339998007, 148983911, 790454221},
  {1, 3, 1, 5, 3, 17, 59, 219, 105, 623, 21, 2843, 3427, 4799, 3793, 43223, 11457, 9799, 357191, 171081, 2049495, 3576665, 3775853, 8678047, 11945725, 33054475, 73010289, 100008085, 144825, 948571375, 1209196469},
  {1, 3, 3, 7, 21, 55, 17, 29, 397, 93, 1981, 4047, 935, 5971, 14589, 32095, 116673, 68421, 469057, 827739, 1529053, 3601215, 6512127, 10157687, 26092423, 11511623, 36842613, 139387769, 59714141, 981166619, 843400421},
  {1, 1, 3, 9, 5, 57, 63, 27, 373, 815, 167, 205, 367, 4945, 30041, 49471, 126019, 102081, 237391, 690647, 1888499, 377101, 756005, 3067523, 26136709, 33650361, 120008899, 228673589, 145847375, 179745723, 351393101},
  {1, 1, 5, 9, 7, 3, 69, 35, 197, 309, 1729, 3735, 1523, 10427, 26253, 33983, 100547, 160583, 170307, 602453, 66765, 1114883, 6178281, 10467011, 17566207, 48085611, 33442005, 202921117, 238407325, 315248059, 663410149},
  {1, 1, 3, 7, 7, 49, 35, 189, 297, 311, 2025, 305, 3863, 14393, 2533, 41511, 75203, 70849, 217409, 840137, 2046463, 1586945, 1734587, 2870675, 32550821, 47489863, 76865955, 207154869, 504857687, 977775831, 1646421293},
  {1, 3, 3, 9, 17, 31, 5, 17, 167, 601, 909, 3149, 2533, 12123, 25325, 13935, 112577, 32197, 219087, 760899, 220925, 3999099, 1650651, 13187397, 1722967, 63182975, 16870807, 200259903, 91217873, 977959067, 1194745077},
  {1, 3, 5, 3, 11, 41, 69, 199, 79, 611, 133, 3519, 5955, 4609, 27403, 20955, 51393, 79171, 231241, 51661, 428671, 1231703, 6673805, 506241, 4647229, 7158659, 49151477, 15892093, 311188551, 587559369, 1406128141},
  {1, 3, 3, 13, 3, 17, 53, 165, 361, 797, 1447, 869, 6707, 6541, 32249, 32175, 107137, 81797, 470667, 570777, 1237655, 868503, 2786127, 8813731, 25771855, 16851485, 75196587, 257850233, 4567659, 620873059, 343815133},
  {1, 3, 1, 1, 29, 47, 17, 45, 473, 199, 1595, 3095, 3635, 6965, 21859, 37979, 46977, 137607, 409475, 791967, 1156757, 3333583, 7737743, 15377667, 15580405, 54475317, 70597601, 53129373, 293813459, 350578825, 682859293},
  {1, 1, 3, 9, 1, 15, 59, 163, 91, 811, 1087, 1707, 6743, 12643, 29901, 12095, 114563, 42369, 257935, 31123, 236429, 451749, 7767381, 8214173, 1116445, 50380777, 14895445, 251939713, 249154893, 915827339, 1583248933},
  {1, 1, 1, 3, 19, 21, 7, 209, 121, 821, 709, 1085, 5333, 7689, 28355, 19259, 54403, 181379, 106625, 124053, 1292979, 2076589, 1526623, 11334235, 5285959, 49151535, 86749495, 216546095, 129351459, 1017954129, 1136295229},
  {1, 3, 1, 15, 5, 27, 115, 31, 37, 79, 1347, 155, 3709, 13251, 32151, 19419, 28033, 143239, 352141, 937115, 312209, 2909637, 665465, 13953691, 33149317, 60175709, 65500573, 186362315, 273553337, 534913169, 1314818933},
  {1, 3, 7, 15, 27, 27, 127, 231, 137, 205, 1665, 1461, 299, 2797, 879, 17923, 59521, 70273, 327809, 644997, 482477, 2007497, 1579929, 14835655, 21661535, 28411291, 48203831, 238329537, 410625595, 409911349, 1445130589},
  {1, 1, 1, 7, 13, 3, 127, 13, 253, 481, 1435, 1895, 2665, 7611, 17761, 65335, 128899, 161411, 297605, 99715, 449177, 3635705, 4369523, 3167335, 19090587, 57729497, 109653713, 70705447, 199396073, 953281687, 1069852533},
  {1, 1, 3, 7, 7, 21, 71, 247, 301, 183, 1785, 331, 4835, 2251, 4493, 20095, 86147, 248193, 174721, 49289, 182427, 74733, 466169, 15506499, 19990381, 19286039, 7328569, 75427573, 48052877, 947212187, 982747109},
  {1, 3, 7, 9, 9, 1, 77, 169, 103, 647, 1959, 1847, 5803, 3421, 15915, 48939, 86657, 241281, 523911, 925851, 984979, 2896591, 4348083, 11974837, 21825993, 25771561, 86504697, 259289157, 529034891, 53700625, 1663633661},
  {1, 3, 1, 7, 19, 17, 81, 45, 263, 549, 1607, 2177, 1117, 14427, 16451, 43955, 117633, 130183, 266629, 292253, 2076343, 1267, 7477263, 547037, 15198379, 32197005, 32277903, 13272543, 258052193, 622173813, 1878896821},
  {1, 1, 7, 15, 27, 25, 27, 27, 33, 813, 1667, 253, 2749, 927, 29707, 64115, 2947, 28805, 289153, 608133, 495279, 1580713, 3724973, 2490007, 14687983, 11491673, 19567227, 264707527, 245578085, 512305333, 1974040805},
  {1, 1, 7, 3, 17, 29, 13, 67, 417, 303, 19, 3809, 7225, 12775, 3933, 16991, 28291, 228997, 347021, 796055, 885439, 1950135, 1632985, 7366567, 31755245, 61848013, 10469447, 247873915, 165401877, 547368723, 1851809381},
  {1, 1, 1, 11, 13, 41, 77, 217, 281, 659, 1099, 3047, 1619, 525, 4313, 22479, 86147, 112259, 47241, 249499, 677299, 2318751, 7749315, 2526763, 9617953, 58312429, 85247985, 218321693, 498496043, 443804227, 1973461245},
  {1, 3, 3, 9, 23, 47, 5, 33, 219, 531, 77, 2307, 1893, 8335, 8281, 29047, 70401, 101125, 513295, 985861, 789761, 2398043, 6246699, 8368025, 11102885, 47380843, 131034777, 27568227, 504780613, 1046336071, 1052803781},
  {1, 3, 7, 3, 3, 35, 27, 249, 159, 495, 431, 3001, 1475, 11505, 15693, 37239, 76545, 70401, 252685, 212229, 640805, 1191261, 4934095, 9333357, 5078481, 29226865, 52656615, 246176689, 77823607, 145589679, 36755693},
  {1, 1, 5, 9, 21, 49, 43, 159, 465, 959, 179, 993, 121, 11569, 21027, 59515, 66819, 174855, 401155, 820487, 1850651, 163145, 5545929, 34799, 22099741, 44123341, 76498311, 153048251, 178770627, 245486913, 1389199677},
  {1, 3, 1, 5, 1, 61, 9, 221, 231, 55, 191, 2829, 3331, 8911, 15109, 53343, 94977, 242439, 332551, 139531, 1781567, 2911091, 727503, 2183773, 13206777, 55176145, 126466419, 226843673, 257682377, 326136219, 2079327573},
  {1, 1, 7, 1, 7, 35, 67, 97, 159, 191, 935, 3151, 6397, 10751, 1835, 2067, 8451, 210181, 234767, 978181, 3885, 116997, 5132775, 280925, 20797569, 24624601, 112112129, 165494627, 158085893, 550260693, 1938075461},
  {1, 1, 1, 7, 15, 39, 127, 163, 437, 333, 829, 753, 8151, 13239, 523, 5891, 19203, 24323, 409349, 631553, 843065, 880945, 6057309, 9711091, 15417127, 711847, 75276171, 178227509, 460304153, 394366565, 35496981},
  {1, 1, 3, 13, 9, 25, 73, 155, 445, 239, 2035, 15, 5243, 15531, 1733, 14639, 121603, 17665, 261387, 858387, 1373451, 2452859, 2548489, 9737611, 11480725, 13988141, 32937705, 106176869, 113068381, 976732819, 602717093},
  {1, 3, 7, 15, 5, 25, 3, 55, 117, 57, 783, 1509, 7043, 13159, 8557, 16943, 72449, 250625, 294145, 850715, 1394963, 1925425, 4498225, 3667227, 32137171, 31195261, 13893883, 31721801, 389816673, 76828835, 1594643445},
  {1, 3, 5, 1, 21, 55, 89, 119, 199, 79, 161, 1597, 3263, 3335, 5757, 9647, 34305, 112131, 210443, 756247, 848413, 198199, 4421317, 4084265, 530369, 24123967, 10037631, 6581957, 418951801, 487448179, 1838608213},
  {1, 3, 7, 5, 27, 23, 85, 113, 111, 211, 389, 1513, 2759, 7945, 931, 62763, 50689, 141825, 297483, 497169, 357921, 285307, 4431579, 6304397, 21424653, 33652259, 71495907, 214295317, 457954439, 341306289, 1743745133},
  {1, 1, 1, 7, 1, 5, 17, 177, 357, 619, 5, 2583, 621, 2973, 28845, 9943, 107523, 254979, 275461, 822287, 1520647, 130075, 5624979, 10474503, 31806625, 46004435, 132556317, 194165315, 375913287, 221068183, 1358805901},
  {1, 3, 7, 13, 11, 21, 47, 99, 421, 279, 1541, 1305, 4571, 6127, 20735, 17387, 78849, 70657, 437251, 267281, 1920003, 1207301, 5489725, 10503523, 7415389, 61255723, 12268819, 169572329, 4313177, 1008763681, 102417941},
  {1, 3, 5, 5, 23, 43, 19, 137, 425, 409, 1625, 2671, 4385, 3197, 25753, 32223, 2049, 83971, 497679, 604189, 391173, 3995717, 5392559, 12146875, 22649547, 57851243, 122048221, 166813183, 308561471, 393319523, 1127632109},
  {1, 1, 7, 5, 27, 17, 57, 15, 383, 181, 951, 2115, 5237, 1495, 9671, 54891, 14339, 120837, 321547, 919569, 612391, 518171, 4221053, 2660705, 1501771, 29772509, 123217709, 82160883, 422917437, 1058481769, 453486053},
  {1, 3, 3, 11, 9, 1, 53, 127, 375, 499, 1487, 121, 1465, 3175, 24337, 53831, 16385, 16389, 311309, 1032223, 311315, 278583, 5718037, 12075401, 16597789, 51037737, 77450215, 29115723, 340707093, 394135519, 640855141},
  {1, 3, 7, 11, 29, 35, 67, 129, 221, 439, 1159, 3501, 7741, 8885, 11381, 20707, 47105, 30723, 10247, 870411, 1071133, 3987491, 3131459, 9291905, 22857949, 7883191, 98172039, 89056685, 486286909, 550369973, 1731548277},
  {1, 3, 5, 11, 29, 59, 23, 117, 343, 637, 1825, 1687, 2823, 11641, 3311, 23603, 26625, 239619, 47109, 264203, 1390621, 997435, 1968151, 10512501, 6183255, 28117629, 10571553, 25812631, 457270023, 523834745, 1378837743},
  {1, 1, 5, 11, 1, 35, 103, 155, 233, 575, 1761, 503, 4175, 6105, 29737, 32681, 43009, 26625, 395269, 88075, 346113, 4167715, 3573863, 2189467, 30021865, 31947327, 72204001, 1669623, 10283087, 300404697, 113089577},
  {1, 3, 3, 1, 5, 63, 27, 71, 245, 433, 1779, 2475, 5479, 4705, 10795, 34247, 48129, 27651, 123907, 78849, 924677, 998463, 142363, 15643719, 30629109, 33131953, 41169651, 183825835, 268929383, 90482273, 1324912171},
  {1, 3, 5, 7, 29, 45, 117, 5, 393, 849, 843, 3131, 6995, 9979, 28907, 30115, 4609, 199171, 408069, 552455, 1740317, 298541, 6684277, 4251141, 11865993, 49891665, 66522443, 2530875, 7454035, 230679803, 1021797099},
  {1, 3, 5, 9, 27, 29, 69, 5, 395, 561, 1531, 409, 2779, 8785, 16405, 27315, 93697, 70147, 89605, 41481, 568859, 918045, 1392197, 6045189, 9814923, 33102897, 68950011, 205518745, 422239451, 436420689, 93357589},
  {1, 3, 1, 9, 15, 29, 85, 3, 331, 19, 1941, 567, 6957, 747, 1627, 11347, 17153, 170755, 158977, 724745, 988431, 2122525, 1295189, 10869507, 11892811, 43531027, 51854997, 113713975, 92875821, 834238443, 409582427},
  {1, 1, 3, 9, 27, 45, 47, 127, 133, 921, 1817, 2231, 6333, 14371, 12799, 9831, 48385, 4353, 422147, 839433, 1246491, 1776429, 7911215, 1220479, 10643845, 12561049, 40889369, 197220791, 139672509, 261750051, 1698694911},
  {1, 1, 5, 15, 31, 7, 125, 13, 455, 159, 331, 3629, 4705, 11261, 3657, 36307, 70401, 199425, 259845, 176399, 1387807, 1140999, 6188413, 12715789, 20300999, 12692895, 90187, 45540653, 256293217, 912290045, 1742335817},
  {1, 1, 5, 9, 11, 53, 51, 35, 87, 885, 1975, 3899, 1013, 7667, 32385, 33635, 84737, 166145, 310021, 196361, 1833227, 2191157, 4343091, 1031971, 13890903, 27188853, 42677943, 125489723, 295447797, 373882099, 460345729},
  {1, 1, 1, 3, 7, 45, 107, 177, 193, 765, 731, 139, 5563, 623, 16485, 54999, 4865, 61185, 53505, 388355, 1309959, 3279149, 6998891, 3173809, 32268225, 56472061, 49877467, 66007435, 495109307, 216756591, 738697573},
  {1, 1, 5, 9, 17, 53, 117, 69, 385, 587, 1483, 149, 2769, 3013, 18183, 10173, 107265, 254209, 423173, 920329, 566033, 343861, 2172277, 12204869, 3363457, 13908299, 88206027, 258739093, 23855569, 739521733, 470028295},
  {1, 1, 5, 11, 5, 3, 25, 153, 351, 749, 801, 3077, 3209, 11189, 25241, 14115, 87297, 260865, 398085, 548619, 1280773, 2428675, 4635417, 13646233, 7409247, 37267949, 86637601, 74081541, 237385609, 1046276789, 115623321},
  {1, 1, 7, 9, 1, 47, 41, 247, 135, 163, 899, 1517, 5647, 10595, 32531, 12497, 90369, 193793, 118023, 967433, 1219841, 390447, 6139177, 13402103, 27306887, 58821539, 59416195, 235682541, 365133071, 923389027, 1561645075},
  {1, 3, 5, 11, 5, 61, 111, 215, 251, 279, 825, 2155, 3527, 173, 10973, 59257, 76033, 64771, 211717, 820491, 1160453, 380733, 2546543, 6240215, 15878139, 61056023, 35144761, 128238955, 58051271, 950852525, 2024860637},
  {1, 3, 5, 11, 25, 15, 71, 83, 135, 231, 1415, 3761, 7513, 8337, 28979, 43615, 107393, 258947, 26757, 220299, 1112217, 2959759, 7053255, 5247187, 30052615, 24515175, 74199047, 72151601, 230230745, 15807249, 928188595},
  {1, 3, 5, 13, 19, 5, 55, 165, 141, 119, 1891, 2255, 4735, 16217, 26195, 50527, 109441, 164227, 178053, 561037, 1105043, 2161029, 2981815, 1069093, 32755981, 9876215, 44749283, 4116815, 1057023, 881784025, 355681491},
  {1, 1, 7, 15, 23, 59, 59, 191, 1, 855, 453, 2619, 5013, 14749, 24335, 44339, 45953, 260737, 140679, 524943, 2078359, 3502011, 3932603, 11511871, 4319361, 36927703, 71867973, 236652987, 20252181, 503127069, 1680832655},
  {1, 1, 1, 13, 15, 41, 51, 147, 229, 495, 1191, 867, 1525, 581, 29713, 26391, 122753, 60289, 280961, 225933, 752271, 1560745, 5682099, 13551635, 19974245, 39758191, 111319591, 80456931, 428177781, 325956549, 1375401361},
  {1, 1, 1, 9, 29, 5, 59, 127, 105, 417, 301, 2249, 6335, 3513, 17373, 52977, 101505, 121473, 36993, 134025, 1628317, 998021, 144571, 5166335, 22193385, 53261601, 96465325, 70627913, 209792575, 453575737, 1454192733},
  {1, 1, 3, 7, 21, 27, 109, 143, 63, 347, 1429, 2889, 2597, 10243, 9913, 22687, 57729, 130689, 262275, 910983, 1518997, 3010715, 1520621, 15369231, 18275519, 10097883, 95263253, 251538889, 26665893, 1041447555, 1741237561},
  {1, 3, 5, 5, 7, 3, 125, 147, 313, 351, 1163, 415, 5615, 5571, 7089, 55621, 117121, 161667, 438661, 401797, 1154951, 116355, 7628029, 1532435, 9603769, 32702943, 101646347, 21170207, 530783855, 983508291, 1642943537},
  {1, 3, 3, 3, 31, 43, 101, 93, 9, 671, 135, 333, 2169, 11169, 7403, 50707, 66945, 123267, 51331, 254595, 580255, 1865643, 5848037, 15910621, 32667529, 38424863, 58455815, 147880141, 299879417, 143775521, 1374622827},
  {1, 1, 7, 13, 15, 33, 125, 155, 227, 827, 1047, 2441, 3007, 10881, 19969, 63805, 41857, 75137, 92807, 518541, 1084047, 3262113, 1627389, 735771, 14135907, 12143291, 39921815, 101009417, 363022655, 212496897, 391997825},
  {1, 3, 3, 5, 31, 33, 29, 249, 159, 797, 1475, 841, 6933, 6417, 25629, 61865, 49281, 168835, 453251, 245893, 1949343, 3692705, 404125, 2807417, 1945375, 31280285, 55463235, 172822217, 378082709, 625288593, 1494814365},
  {1, 3, 3, 15, 11, 55, 11, 117, 149, 911, 1589, 3133, 6477, 6123, 10471, 41099, 115329, 148867, 506755, 289679, 1869963, 231607, 4477579, 13794293, 32206869, 55513359, 94048949, 217863869, 100760013, 39859563, 1176521063},
  {1, 3, 3, 9, 27, 37, 1, 119, 509, 969, 831, 3771, 2093, 13621, 31737, 43269, 114049, 158339, 46723, 76169, 1649563, 2622885, 5009281, 5075447, 11126141, 63370569, 68182207, 29681211, 140799405, 904778933, 1817349497},
  {1, 1, 1, 1, 9, 23, 119, 109, 487, 753, 1673, 2163, 3349, 4741, 29971, 3407, 56193, 235905, 369793, 386433, 875401, 1468567, 2971895, 16651245, 12469351, 25544561, 9149193, 220406259, 9249685, 864276997, 1311799443},
  {1, 3, 3, 7, 25, 7, 67, 9, 461, 631, 651, 2271, 5663, 2621, 3953, 20975, 86657, 163459, 436355, 259207, 1000857, 1757575, 5419715, 14963593, 11345229, 46277879, 48999947, 184137567, 139217055, 617339069, 1741710577},
  {1, 1, 5, 11, 13, 31, 29, 255, 371, 517, 845, 3649, 1187, 10061, 22887, 58417, 85889, 1409, 239493, 51851, 935821, 2947231, 6006173, 3604351, 2614003, 58360709, 109871565, 90540481, 418549795, 794275021, 1639220711},
  {1, 3, 5, 13, 29, 1, 11, 137, 151, 249, 167, 1243, 997, 11023, 11875, 42315, 6529, 156803, 88197, 542605, 1127581, 676993, 1783947, 5696777, 4519447, 55092089, 76408103, 163081563, 407200869, 617362831, 1081344995},
  {1, 1, 5, 5, 5, 55, 103, 71, 255, 1023, 209, 1005, 2147, 11527, 17863, 6661, 114305, 227713, 35205, 306821, 1665669, 2781879, 1801191, 11369415, 33102719, 15766143, 116266321, 210583149, 305605859, 473404295, 1508810567},
  {1, 1, 3, 3, 31, 39, 7, 151, 353, 775, 1313, 1257, 4197, 2625, 9571, 27269, 95361, 257409, 484227, 70019, 536479, 3519143, 7302023, 4107031, 15188705, 34199943, 70380961, 213829225, 452743909, 833931457, 66519011},
  {1, 1, 1, 3, 7, 17, 3, 127, 501, 503, 1879, 2329, 3049, 10603, 2111, 33189, 44161, 31361, 340353, 348035, 475783, 3265169, 20355, 1804287, 984693, 31917687, 97685975, 121225113, 317240937, 829688811, 208943807},
  {1, 3, 3, 7, 13, 59, 93, 13, 375, 483, 1991, 2257, 3003, 1699, 4339, 51827, 129, 205443, 290179, 181895, 1219725, 983227, 915677, 13922701, 10839031, 38094179, 50728263, 162896721, 181317947, 807984419, 402399603},
  {1, 3, 7, 15, 27, 41, 59, 225, 405, 807, 1545, 2581, 1173, 14137, 3413, 39299, 10817, 212291, 265543, 591439, 1920347, 887017, 383099, 13359137, 7157845, 27171047, 57764041, 160405589, 255418965, 552413049, 964755477},
  {1, 1, 1, 3, 9, 23, 37, 123, 465, 1023, 1065, 1455, 5107, 3839, 20451, 11461, 52801, 47041, 207041, 635843, 1412297, 3751255, 5377125, 2145339, 32394641, 45111103, 10895977, 28656751, 330325683, 128179263, 1966989219},
  {1, 1, 1, 11, 19, 55, 91, 121, 317, 199, 215, 3031, 7223, 11891, 21463, 64921, 74177, 166337, 29121, 927947, 1036755, 1196663, 5166107, 2901049, 21755261, 12299783, 45021207, 166320663, 432494327, 146981939, 2039994519},
  {1, 3, 7, 11, 19, 5, 5, 115, 399, 219, 71, 1465, 281, 14451, 26807, 42541, 30657, 192579, 225351, 791243, 57299, 2898245, 5346885, 14286515, 7522895, 51621915, 122483335, 148593401, 291538521, 1005105843, 2137125239},
  {1, 3, 5, 13, 3, 33, 75, 35, 19, 559, 761, 947, 7479, 15325, 31453, 20561, 22593, 240323, 144325, 1008973, 2011459, 858721, 7293451, 6499043, 4797267, 48247663, 91976761, 197767667, 304424439, 360523037, 134024221},
  {1, 3, 3, 13, 23, 47, 99, 73, 331, 353, 401, 1737, 6235, 13781, 5547, 56443, 34369, 185411, 124099, 509773, 739543, 4110575, 3156515, 12310153, 17790859, 46202913, 129040977, 124324617, 331029147, 552929685, 894114155},
  {1, 3, 3, 13, 21, 37, 41, 205, 87, 399, 51, 3175, 7403, 12875, 21129, 7079, 3777, 249795, 310467, 621773, 1115221, 3045349, 7114345, 16050701, 5518999, 48261327, 22551411, 133786407, 89142443, 437670667, 1826276681},
  {1, 3, 5, 11, 15, 47, 33, 39, 465, 871, 277, 2351, 695, 1953, 24293, 20595, 102209, 161219, 272325, 351819, 963407, 1756911, 4673121, 1654503, 17007249, 7331751, 7295189, 235156719, 155654391, 166792289, 290789541},
  {1, 1, 7, 11, 13, 15, 115, 59, 469, 715, 191, 1927, 905, 13463, 29005, 46789, 53953, 180929, 275783, 670411, 1987149, 454223, 1881395, 11783547, 15939861, 35383563, 41921791, 3704647, 461871689, 732597591, 416722701},
  {1, 3, 5, 9, 13, 55, 79, 17, 265, 887, 905, 3985, 6907, 3379, 20055, 58569, 71489, 202563, 250821, 213449, 133837, 2000247, 6662927, 6133201, 9055561, 17028663, 99824073, 152950609, 467157051, 399370867, 921233175},
  {1, 1, 7, 11, 21, 29, 23, 109, 17, 427, 1623, 2219, 3857, 3709, 25033, 63823, 86209, 77249, 139079, 323659, 1820501, 4135645, 7243095, 5546797, 19381457, 11765355, 33507351, 26597611, 290323409, 225193021, 730082953},
  {1, 3, 5, 15, 19, 27, 113, 15, 25, 63, 1885, 2693, 5301, 9385, 14137, 26097, 4161, 60483, 282181, 873807, 962899, 3473243, 703281, 3986383, 1700441, 44651263, 17322141, 17029445, 496686581, 14740073, 1692984953},
  {1, 3, 3, 11, 17, 5, 73, 143, 79, 957, 461, 1709, 4909, 2285, 18113, 8401, 34369, 144451, 376899, 686155, 1798993, 1947461, 3456777, 11237199, 20805519, 41817213, 133600397, 25138925, 49715821, 65048621, 2078859777},
  {1, 1, 3, 7, 9, 9, 101, 127, 137, 755, 1359, 1965, 83, 13335, 27763, 7941, 65729, 165441, 305603, 387783, 1448393, 332873, 586021, 10936127, 9381705, 30553139, 87958031, 20160749, 99906067, 747719895, 1331723571},
  {1, 1, 1, 3, 13, 61, 95, 61, 295, 615, 555, 2163, 8155, 14043, 21465, 46741, 41921, 6593, 214977, 1005507, 1343309, 2243709, 2558751, 14222205, 16533095, 38194599, 26861163, 164362803, 480722715, 279751323, 2089460505},
  {1, 1, 1, 13, 29, 19, 111, 17, 373, 153, 1703, 2199, 7209, 15845, 1879, 7493, 88385, 12865, 497089, 920013, 1620829, 3516115, 4675247, 4629713, 9702837, 10782553, 48771943, 157947607, 158915177, 935356965, 1676375703},
  {1, 3, 1, 13, 21, 51, 49, 51, 255, 151, 207, 1915, 7629, 2705, 8739, 7467, 105537, 21827, 447169, 1031885, 269397, 4183923, 2413041, 10769523, 21855551, 62611287, 42587919, 24583483, 380910605, 764312273, 924508259},
  {1, 3, 7, 5, 21, 21, 23, 193, 467, 739, 519, 2315, 2953, 10633, 9163, 6007, 31809, 103747, 383943, 506565, 1481301, 42069, 7176151, 4547713, 19492243, 36084771, 53789127, 9030347, 350968009, 406144585, 1196204939},
  {1, 3, 1, 5, 23, 19, 23, 247, 93, 297, 1089, 2349, 4683, 13609, 7615, 18647, 19137, 111043, 243265, 36293, 1032535, 2841683, 1386967, 2346679, 21561245, 2962153, 111645953, 165958765, 272278539, 793476457, 445441919},
  {1, 1, 3, 3, 21, 39, 19, 71, 93, 1, 133, 3531, 7503, 2819, 24211, 1739, 91073, 136769, 147523, 321859, 862037, 903527, 536531, 13425287, 11627037, 42295105, 117338693, 173341835, 485642255, 444622147, 953057363},
  {1, 3, 5, 13, 9, 43, 31, 111, 493, 739, 705, 2715, 3613, 11877, 27945, 46053, 10433, 49475, 403397, 559309, 989641, 1133803, 2533343, 10422575, 13237293, 35023907, 114941697, 153402843, 533752157, 724163877, 260997865},
  {1, 1, 7, 13, 27, 59, 103, 129, 53, 531, 1379, 1441, 5341, 14937, 5079, 39881, 81985, 120513, 63687, 499149, 1765467, 3489019, 5799847, 11040065, 30499957, 56405971, 55189027, 62814177, 410333085, 504114841, 1445752087},
  {1, 1, 3, 3, 11, 63, 91, 95, 433, 393, 715, 809, 591, 4141, 17417, 54107, 33601, 4801, 434243, 14787, 511947, 897919, 6000795, 15759263, 21212401, 48465353, 123125387, 41838185, 502604815, 686481773, 1105463497},
  {1, 3, 5, 1, 7, 25, 25, 175, 205, 803, 183, 1441, 1279, 2753, 20001, 56677, 44865, 133827, 40517, 426177, 1234375, 456409, 5259737, 6141935, 23853837, 29610211, 18436855, 146119521, 153148223, 834783873, 152736225},
  {1, 1, 5, 3, 13, 23, 77, 25, 133, 137, 1907, 1313, 2463, 14339, 13, 57757, 95041, 69313, 367173, 365635, 199629, 2347223, 6902157, 16544729, 9262661, 13589961, 4009651, 55100257, 53167711, 1008693699, 179839693},
  {1, 1, 5, 9, 23, 35, 1, 119, 111, 61, 403, 1815, 1985, 5651, 10883, 55943, 99041, 88097, 418405, 307305, 192759, 228355, 3564577, 8809687, 8334671, 23002845, 47096307, 17769271, 117226785, 520839155, 1256231203},
  {1, 3, 1, 7, 21, 43, 115, 7, 107, 719, 759, 1521, 467, 8735, 29785, 63821, 76321, 48739, 317409, 126759, 1543605, 931851, 4982803, 1775655, 9926795, 7948399, 5435287, 131629969, 337418547, 301180799, 708810169},
  {1, 1, 3, 13, 19, 17, 51, 141, 399, 569, 703, 2221, 2809, 13355, 1907, 15837, 87969, 223521, 446307, 497773, 1653171, 1265201, 3836435, 10040045, 11509295, 48479833, 133234335, 60454093, 78542169, 897089483, 193926739},
  {1, 1, 5, 15, 15, 53, 57, 31, 481, 69, 1439, 4049, 6727, 11307, 20683, 63517, 102433, 91873, 514917, 230255, 626095, 1007125, 337433, 1640127, 31574529, 8549285, 56947007, 30642929, 205281191, 627957259, 1639031403},
  {1, 1, 1, 3, 13, 27, 9, 255, 363, 131, 1745, 2489, 6451, 6585, 12873, 35405, 105761, 220257, 503073, 292643, 1241005, 2242107, 7143977, 14426847, 23694091, 7779107, 14186417, 36223641, 473493715, 1029594841, 8217641},
  {1, 3, 5, 1, 17, 31, 113, 135, 449, 915, 1017, 2317, 6821, 5483, 30707, 45279, 103329, 49827, 210789, 51297, 654833, 1786431, 2293265, 7212583, 305121, 17537971, 106626585, 167314733, 378081349, 661071691, 434058131},
  {1, 3, 5, 1, 13, 47, 25, 53, 413, 545, 1777, 3049, 7527, 9689, 25935, 9919, 121633, 96995, 242981, 489441, 858605, 1349135, 3680889, 4978325, 5441469, 31025537, 110857489, 48200393, 38244807, 575837305, 1136961391},
  {1, 3, 7, 11, 17, 39, 13, 131, 295, 517, 1755, 2977, 6267, 12351, 8957, 17765, 101921, 259427, 276967, 185771, 1325745, 3956231, 4433517, 11539043, 17588807, 16611877, 55647291, 70712321, 214939163, 213065439, 472474717},
  {1, 1, 7, 5, 27, 57, 47, 21, 125, 429, 1169, 1717, 5455, 16359, 29065, 6671, 95137, 232673, 71143, 711717, 333563, 1665817, 6688015, 8305653, 2641373, 5826509, 110987441, 77879893, 395471087, 1032885575, 716963625},
  {1, 1, 5, 5, 21, 15, 79, 241, 83, 515, 859, 2351, 3125, 7465, 30475, 19759, 99041, 10657, 223781, 879205, 341237, 655663, 2458479, 12400465, 7977459, 35596707, 46870459, 20798927, 504086293, 392839241, 1036247147},
  {1, 3, 1, 9, 11, 5, 81, 11, 7, 221, 141, 3329, 3435, 323, 18999, 54735, 4385, 237155, 278369, 261609, 340203, 3461413, 7626033, 14642475, 827943, 36494525, 62677805, 133881889, 240513291, 425203363, 192431767},
  {1, 1, 1, 15, 7, 57, 87, 251, 63, 561, 929, 1367, 2511, 14527, 9335, 38775, 125921, 200417, 511201, 402415, 1230823, 824601, 4631927, 2047451, 18074335, 13018577, 86702977, 88624823, 96617391, 677805151, 683926359},
  {1, 3, 3, 9, 23, 37, 59, 105, 179, 515, 235, 2445, 433, 13039, 27005, 48829, 71841, 43811, 465123, 806057, 1246391, 2187525, 3575643, 3142409, 12768915, 59964899, 93015371, 78145005, 510983569, 757035919, 18377757},
  {1, 1, 1, 1, 23, 37, 103, 31, 89, 921, 1687, 831, 387, 10237, 1241, 19295, 7393, 190305, 376481, 683553, 1133303, 4004613, 7947079, 5852479, 4292473, 17302137, 11010359, 106433759, 171448163, 461907933, 1177049849},
  {1, 3, 3, 7, 25, 23, 57, 251, 309, 579, 603, 807, 7383, 8579, 4025, 16757, 99297, 172003, 227683, 895527, 826297, 855351, 1597273, 14477211, 19491541, 58998883, 75908539, 158024967, 512742583, 912384547, 803377369},
  {1, 1, 3, 15, 23, 59, 29, 33, 467, 641, 1271, 2915, 2549, 14767, 26557, 43483, 4449, 210337, 115939, 42671, 7287, 2071835, 7854909, 10479937, 11259699, 11283297, 40671895, 8112579, 14634197, 644271823, 1864918173},
  {1, 1, 7, 13, 1, 57, 23, 129, 321, 75, 189, 4087, 5011, 4355, 25759, 37153, 113121, 202593, 259943, 830189, 1533217, 859417, 4837687, 2135905, 4795361, 49138539, 119274141, 213915159, 499646643, 366488099, 635965695},
  {1, 1, 5, 1, 21, 57, 25, 183, 37, 669, 259, 1381, 877, 10245, 16643, 61035, 97825, 165921, 290341, 383713, 2022389, 2849177, 5471673, 15718551, 21289093, 24406717, 96554659, 225531077, 347926797, 568768549, 95727907},
  {1, 1, 7, 5, 11, 11, 85, 141, 393, 957, 1745, 2243, 1681, 5583, 16527, 12017, 56545, 108705, 434791, 260965, 1425323, 4061611, 6091509, 11936493, 25157545, 65144669, 63095601, 80104931, 157238449, 1060293743, 472746927},
  {1, 1, 5, 15, 23, 31, 5, 169, 287, 527, 1831, 2937, 7533, 9739, 24305, 2239, 53921, 178209, 72677, 880303, 575607, 2645695, 2528677, 6903433, 24220287, 43492463, 87870791, 177083993, 340571597, 479382891, 1099129233},
  {1, 1, 7, 1, 7, 13, 3, 243, 189, 309, 607, 3659, 6369, 7649, 24255, 55373, 31201, 116129, 182183, 363873, 512807, 1658285, 2135203, 1097363, 11786781, 11305045, 132415871, 206668715, 33260033, 877692353, 1893702559},
  {1, 1, 1, 3, 3, 59, 103, 209, 287, 913, 1223, 1063, 7715, 6073, 26697, 25671, 86881, 210657, 336801, 343971, 1734499, 3338651, 6010567, 12092273, 24175871, 24698481, 86329127, 123063879, 171761283, 646934745, 1084611049},
  {1, 3, 7, 5, 19, 19, 117, 191, 275, 637, 991, 2199, 2921, 10553, 21211, 25981, 36705, 216995, 25511, 2277, 1632819, 2091187, 2023829, 9521631, 28491059, 23209117, 19132479, 81106615, 305449481, 308365017, 475941883},
  {1, 3, 3, 5, 29, 59, 17, 13, 127, 57, 1405, 3181, 2237, 1795, 21419, 43421, 130593, 155683, 457699, 612069, 48957, 2802331, 1208817, 5005805, 5209183, 39339289, 90191773, 208886221, 535270813, 561225571, 1020561163},
  {1, 1, 1, 15, 25, 41, 11, 117, 463, 425, 305, 1441, 4307, 7967, 17529, 4043, 70561, 94177, 429665, 766895, 2068985, 3818377, 3488939, 310229, 16904623, 29176585, 86495889, 37680961, 435667955, 667703935, 1377800409},
  {1, 3, 5, 5, 19, 53, 69, 73, 453, 611, 1583, 1721, 6303, 10561, 18527, 48973, 59297, 13155, 294821, 917861, 1333811, 3321493, 6896037, 10083689, 17781221, 3174787, 118576911, 131117721, 483600447, 109942369, 1116788159},
  {1, 1, 7, 11, 15, 61, 87, 69, 463, 771, 819, 469, 8165, 8897, 29657, 55161, 545, 117793, 383911, 300715, 617647, 3561373, 6248439, 14813989, 19796783, 31400547, 39363603, 5482933, 163164357, 736539553, 377404473},
  {1, 1, 5, 1, 15, 25, 23, 47, 287, 457, 1219, 473, 4127, 3349, 9425, 41541, 45729, 80225, 430885, 499361, 4911, 2122681, 4471223, 15308559, 32537279, 48697769, 126537827, 86056889, 98452607, 784822901, 811220849},
  {1, 3, 7, 5, 17, 17, 33, 161, 239, 231, 241, 1297, 4879, 12761, 20939, 65261, 110753, 191587, 30887, 190949, 237617, 2341553, 7272385, 5104577, 24957647, 18077767, 94693713, 21334705, 267291823, 974895033, 153401003},
  {1, 3, 3, 9, 19, 53, 95, 89, 117, 333, 1815, 2217, 7779, 8213, 4667, 58395, 110817, 214563, 424547, 811561, 880051, 883413, 4430975, 13427961, 1162645, 26582381, 36557175, 116888073, 522174083, 387763445, 1377204955},
  {1, 3, 3, 9, 17, 7, 41, 99, 371, 797, 729, 2851, 2003, 4463, 20793, 54315, 8737, 74403, 452899, 371945, 973617, 3227367, 1518601, 14212291, 30655635, 8982397, 25338489, 112681923, 367701491, 1044669391, 1990691417},
  {1, 3, 5, 5, 23, 39, 19, 235, 163, 365, 141, 791, 455, 2761, 9115, 53351, 31137, 222499, 225509, 448805, 950839, 1590727, 7959603, 5401739, 15881539, 1562317, 111545773, 166067383, 251561063, 463448169, 725451515},
  {1, 3, 3, 3, 9, 27, 29, 139, 165, 867, 2023, 1333, 3771, 10451, 9141, 41177, 37729, 32291, 6307, 411235, 4713, 1971835, 4798525, 12445739, 17106949, 12155523, 108035783, 176339029, 482390171, 750294707, 1214688917},
  {1, 1, 3, 7, 3, 11, 125, 157, 355, 519, 187, 3381, 1151, 1629, 25247, 42797, 70753, 257697, 286819, 4263, 763491, 1129323, 3549213, 227517, 25585987, 43678631, 120908059, 47548501, 371753119, 736466301, 944816831},
  {1, 3, 3, 3, 21, 25, 37, 155, 257, 311, 961, 1945, 1383, 5679, 7857, 7183, 126817, 245411, 45091, 924131, 1597877, 152569, 5711877, 9443387, 1792417, 18246487, 36478369, 214707577, 201058119, 100280527, 1496449041},
  {1, 3, 3, 5, 29, 11, 49, 125, 171, 605, 1923, 2781, 2555, 5063, 5075, 43301, 74081, 212515, 105507, 552357, 1739389, 1880939, 5796369, 12683997, 2660043, 48088765, 80581539, 55342141, 115211291, 499574759, 1421115571},
  {1, 3, 5, 9, 27, 1, 27, 149, 253, 205, 1299, 2901, 2891, 975, 7641, 8115, 62561, 202275, 82213, 622313, 14907, 2582241, 2961979, 12914421, 5079709, 26513389, 80888307, 77614709, 40622379, 231107375, 1542427961},
  {1, 3, 5, 3, 31, 7, 49, 215, 81, 791, 1485, 837, 5051, 1947, 7521, 25723, 3553, 134179, 2981, 514275, 615743, 933095, 877073, 14345911, 30890865, 31184439, 13289709, 147045029, 396500763, 401478267, 1129206209},
  {1, 3, 5, 7, 23, 25, 69, 13, 3, 859, 441, 3577, 1687, 6559, 8687, 46757, 9057, 205475, 272421, 217127, 1369463, 1147769, 1596005, 14133869, 12476323, 53100155, 19594585, 46693657, 396452983, 168251583, 1701211215},
  {1, 1, 1, 9, 1, 59, 3, 31, 251, 187, 617, 2607, 4635, 6121, 8565, 8871, 107937, 223905, 109025, 541417, 465825, 1241051, 1632867, 10646143, 10143643, 13150427, 128204617, 91885199, 481692475, 676201801, 1857697493},
  {1, 3, 3, 9, 29, 37, 127, 87, 153, 633, 1691, 2729, 3167, 3219, 21237, 25573, 71393, 165027, 129699, 898921, 94973, 430661, 2237791, 13915639, 3207289, 24916121, 115365499, 247682313, 512562303, 648879987, 1733522581},
  {1, 1, 5, 13, 19, 63, 93, 235, 299, 621, 405, 663, 6639, 12265, 9303, 42719, 84641, 8993, 346597, 675245, 888627, 873695, 379197, 3212555, 86731, 53701709, 119042485, 223540727, 274538191, 669191497, 241245239},
  {1, 1, 3, 9, 25, 11, 9, 231, 101, 335, 1793, 1497, 7069, 4171, 30199, 63, 45153, 155873, 119075, 464681, 1013049, 758507, 8340841, 3124679, 27218565, 65089327, 113005857, 184737081, 364720701, 298502187, 1719496279},
  {1, 1, 1, 1, 5, 19, 17, 217, 165, 413, 925, 1409, 6559, 14537, 22057, 44331, 54369, 86177, 253409, 927841, 1709605, 3040755, 1743729, 1989049, 1480645, 22741629, 89850813, 126287969, 81854975, 561709865, 1034910665},
  {1, 1, 3, 7, 11, 51, 45, 217, 57, 795, 951, 2933, 6705, 137, 30525, 9679, 58465, 15969, 323683, 309991, 256427, 1325267, 5822285, 9850361, 575257, 38647995, 24473111, 264452885, 464312145, 200432873, 1601476893},
  {1, 1, 3, 15, 27, 47, 35, 125, 363, 619, 1027, 2861, 3923, 10459, 16789, 27277, 7393, 136353, 210211, 573935, 693627, 2425295, 3434179, 11096285, 28487883, 66204491, 96587939, 63920397, 424775603, 391027579, 716888245},
  {1, 1, 7, 7, 13, 37, 33, 29, 385, 851, 143, 119, 7345, 4251, 25121, 31609, 97953, 3681, 516775, 615527, 1705197, 1784005, 241345, 5054013, 25634337, 56317619, 116101871, 2953623, 304449489, 406364539, 668264577},
  {1, 3, 1, 1, 17, 25, 119, 7, 365, 397, 601, 2087, 6903, 15345, 14671, 37889, 76321, 151971, 514657, 642081, 759153, 310777, 339671, 2472551, 27205261, 31319405, 37332921, 54945287, 12365143, 832547345, 59295279},
  {1, 3, 1, 13, 19, 51, 41, 139, 133, 723, 25, 2621, 1257, 7037, 9527, 50037, 26209, 171235, 390497, 808749, 2015219, 2281683, 93321, 3850987, 17698533, 59587955, 46259769, 25779293, 439881801, 69736349, 1430800791},
  {1, 1, 5, 11, 5, 59, 119, 75, 397, 545, 1095, 585, 3271, 1049, 123, 33029, 99681, 182305, 168293, 58283, 407013, 4000219, 7136663, 6345003, 12950573, 30246849, 62535399, 40427497, 55280871, 799316985, 897167771},
  {1, 1, 7, 11, 9, 27, 21, 197, 177, 31, 453, 2457, 2733, 7787, 1923, 24639, 105697, 257633, 4007, 132395, 1695657, 3873147, 3901685, 11109349, 27966865, 18460031, 34707045, 238343801, 422297549, 337474699, 1553787811},
  {1, 1, 7, 13, 29, 13, 91, 91, 243, 279, 601, 1699, 7169, 4727, 7815, 29099, 117409, 254817, 120359, 179885, 228925, 1546861, 5859003, 5601147, 24493331, 22335351, 762169, 27404547, 333555233, 446033495, 1131897383},
  {1, 3, 7, 5, 1, 35, 27, 235, 163, 913, 1479, 769, 7179, 1983, 25977, 55373, 44641, 157603, 329383, 882789, 685921, 229571, 7288763, 1770827, 1583427, 44997233, 1858663, 101102881, 322928427, 1048113055, 1526704409},
  {1, 3, 5, 11, 9, 33, 99, 141, 301, 109, 1785, 129, 1707, 5181, 4797, 9979, 90593, 235683, 112869, 213099, 514473, 3685697, 3496131, 2769773, 9431437, 18082317, 73521177, 164692769, 402329163, 342698589, 1722624477},
  {1, 1, 1, 13, 3, 47, 89, 43, 293, 87, 1689, 3885, 7747, 5607, 477, 31887, 56161, 234145, 249249, 998637, 511011, 2392143, 5576889, 2349515, 5708613, 2411767, 9097785, 230159693, 235870179, 162208263, 367611133},
  {1, 1, 5, 1, 9, 21, 73, 37, 45, 621, 1855, 3691, 4899, 2191, 13459, 23543, 99105, 17953, 186789, 730401, 1341609, 2656629, 5360297, 14748997, 1434317, 51764877, 125207647, 222726539, 403738691, 141730799, 1851566067},
  {1, 1, 1, 1, 7, 39, 61, 125, 341, 905, 213, 1755, 241, 13407, 8791, 10165, 52641, 162145, 206113, 142497, 1381351, 3012551, 2359261, 15208349, 30035637, 25668777, 100035189, 52446267, 229595793, 528874047, 1326555063},
  {1, 1, 1, 1, 19, 31, 79, 19, 55, 875, 1017, 1787, 4879, 533, 15029, 52295, 88481, 182945, 307169, 282977, 660019, 14463, 4037807, 9936371, 21815255, 111307, 15923033, 205509211, 145222703, 276901557, 1589711509},
  {1, 3, 1, 1, 9, 59, 113, 71, 113, 649, 561, 71, 5253, 783, 7389, 19361, 52001, 130787, 160801, 576481, 556009, 773083, 4236241, 1530663, 18948753, 40919657, 102342801, 136800039, 400455333, 844852591, 1520522557},
  {1, 1, 1, 11, 5, 39, 61, 225, 291, 907, 795, 1099, 597, 11829, 15137, 42865, 50513, 246609, 250385, 284443, 455957, 2217271, 114989, 5673297, 45171, 5337339, 82063819, 124563803, 27960677, 773357957, 1588627345},
  {1, 3, 1, 5, 25, 11, 71, 155, 271, 309, 1981, 1253, 463, 1133, 20833, 48625, 98545, 109267, 174609, 428821, 1735433, 2210107, 2563415, 6653387, 24622495, 34068101, 29202125, 82941013, 506108991, 378320957, 2108463537},
  {1, 3, 5, 9, 7, 41, 87, 241, 457, 899, 1493, 3675, 3025, 10607, 22569, 52813, 65809, 160499, 509589, 575705, 2041111, 1680153, 8293639, 6738273, 2380985, 19293715, 114143653, 88847435, 234517569, 663735391, 559595129},
  {1, 3, 7, 13, 7, 37, 37, 103, 281, 915, 1259, 4049, 559, 173, 4123, 63767, 108273, 222931, 349623, 478589, 1766167, 3143957, 161621, 9819063, 1330281, 20168387, 63464891, 112076961, 393619647, 844482717, 392972203},
  {1, 3, 7, 15, 13, 57, 9, 51, 39, 549, 1431, 2887, 1081, 4643, 16331, 14221, 122769, 54835, 428215, 984287, 875805, 1783561, 2541433, 3561923, 29414903, 50941621, 70012423, 127083383, 424472393, 242675059, 775062459},
  {1, 3, 5, 7, 13, 1, 101, 125, 25, 713, 1423, 513, 3323, 9951, 7163, 20969, 46513, 253939, 175541, 1015959, 186653, 1542449, 3990837, 12484365, 32752585, 1930201, 16658655, 180361425, 531209835, 793223503, 335475723},
  {1, 1, 7, 15, 11, 25, 25, 3, 47, 531, 1529, 471, 6191, 10051, 29671, 49085, 83185, 59953, 448631, 608735, 1745179, 2571529, 3260777, 16028659, 25439135, 43839619, 20623721, 184422631, 235397087, 337729907, 2131375991},
  {1, 1, 3, 5, 23, 51, 117, 141, 55, 275, 761, 1923, 6267, 2291, 3701, 26615, 44465, 135473, 119123, 597365, 1404167, 2757923, 1909573, 2575837, 1986119, 49955107, 36783273, 237718611, 110806539, 844711107, 2093609701},
  {1, 1, 7, 9, 15, 19, 111, 65, 137, 373, 1753, 3591, 1137, 11639, 28591, 27265, 122385, 12913, 29687, 827161, 909343, 2842627, 6742047, 8675537, 28876921, 36298821, 42770473, 182331927, 423750881, 114055719, 1466982975},
  {1, 3, 1, 15, 29, 5, 67, 13, 425, 961, 453, 2481, 1407, 3479, 23303, 30407, 129777, 185459, 252273, 274815, 863245, 1760309, 6882387, 3740925, 19297401, 26916753, 82078197, 3005793, 488682479, 260199815, 1982856023},
  {1, 1, 5, 3, 19, 39, 39, 123, 351, 77, 1339, 1765, 3767, 1907, 13627, 23877, 48753, 2097, 159413, 1020755, 5635, 911927, 608887, 1026635, 22206063, 21267517, 96915787, 198590293, 112769351, 931010979, 2056656011},
  {1, 3, 5, 9, 23, 7, 103, 177, 221, 197, 561, 2121, 7231, 12053, 30127, 29849, 78065, 154195, 34293, 853657, 1578503, 3185207, 5707319, 15649313, 33247149, 38487733, 19801665, 69363033, 160544751, 107055429, 1215227071},
  {1, 1, 5, 7, 15, 1, 3, 123, 197, 493, 171, 2425, 3865, 4061, 31883, 2491, 112273, 196305, 181685, 923831, 562719, 60945, 849491, 14191115, 6443573, 52099069, 46128795, 238717129, 503091529, 274184973, 541585467},
  {1, 1, 3, 13, 29, 33, 99, 67, 327, 969, 1793, 1871, 1839, 13059, 7605, 16797, 72305, 48305, 286931, 371293, 598669, 3747249, 1671123, 5804307, 33400087, 57525593, 2756785, 77960447, 367051487, 533330707, 399944485},
  {1, 3, 5, 11, 25, 53, 25, 93, 303, 623, 1889, 1471, 1213, 14459, 8527, 25095, 2129, 234291, 354165, 1020667, 1068169, 2103429, 3412169, 9280365, 1784895, 56761535, 50545265, 259181551, 472367565, 59624203, 1911571679},
  {1, 1, 1, 13, 15, 3, 115, 3, 289, 743, 1855, 359, 2375, 13765, 19711, 40765, 111601, 186321, 483441, 345181, 301471, 910995, 6634979, 8589651, 32318545, 34463575, 21715855, 158963671, 12306903, 803949877, 1847085455},
  {1, 1, 7, 11, 27, 51, 85, 163, 219, 871, 637, 2011, 5981, 587, 17521, 17333, 129617, 151089, 52887, 303579, 1457291, 880547, 5967781, 7209363, 29711595, 1515223, 34875821, 63440747, 325586029, 746076603, 2087211809},
  {1, 3, 5, 1, 21, 59, 49, 39, 305, 513, 2017, 285, 5817, 13123, 27765, 46741, 75025, 108211, 478165, 980945, 1886853, 3810187, 7893473, 8527799, 21224161, 5042737, 72390513, 164500205, 41606953, 841083603, 473224933},
  {1, 3, 3, 7, 21, 39, 71, 163, 423, 845, 783, 397, 7319, 10677, 13407, 47471, 84465, 29395, 96627, 435063, 1163141, 1330327, 6777591, 11997523, 6319735, 36958653, 54366719, 135352381, 503725671, 693232613, 645549103},
  {1, 3, 7, 5, 21, 59, 99, 179, 473, 687, 1393, 723, 2245, 2933, 25943, 7769, 34257, 116179, 23287, 822549, 1776005, 627851, 1298835, 6086499, 12372489, 53398431, 54892737, 68972643, 265502677, 554838277, 516048583},
  {1, 1, 5, 9, 5, 45, 71, 189, 165, 555, 643, 2289, 3133, 12319, 22209, 1533, 127057, 179825, 235157, 633689, 533653, 3766717, 335255, 3711917, 8933237, 29148539, 129339763, 247092897, 280593389, 516334127, 532478321},
  {1, 1, 3, 9, 7, 43, 1, 155, 323, 169, 339, 2561, 4049, 4953, 5289, 8783, 84881, 239121, 189459, 450169, 1674135, 975803, 6946993, 1894795, 16202419, 55385241, 14639299, 184773425, 413382497, 597817417, 661022489},
  {1, 3, 1, 11, 15, 5, 25, 201, 267, 891, 561, 501, 575, 15147, 1743, 45237, 24689, 225331, 418929, 172763, 1770655, 3172021, 8109961, 12271609, 18005883, 21085867, 30752033, 198981605, 467084559, 1030598427, 701548127},
  {1, 3, 5, 13, 25, 27, 105, 205, 165, 795, 975, 943, 7413, 10299, 14839, 54895, 111825, 156211, 365301, 38525, 80521, 902827, 4174521, 4380061, 30391221, 60130603, 110345439, 185123071, 231461925, 416929803, 1483370631},
  {1, 1, 5, 1, 17, 43, 69, 103, 449, 917, 103, 945, 513, 709, 11647, 28065, 115601, 176849, 319605, 865169, 1540737, 3448763, 4878485, 16722167, 27871313, 19536805, 109276343, 114861377, 270656401, 589320469, 935096463},
  {1, 1, 3, 15, 23, 51, 23, 7, 159, 743, 177, 3457, 415, 1775, 25353, 36385, 81265, 96625, 108819, 1183, 1955719, 2761891, 3518631, 7133303, 24205679, 30255191, 12250689, 158379633, 257276399, 875600671, 297579417},
  {1, 3, 5, 13, 9, 63, 121, 19, 165, 449, 1523, 1959, 6901, 12281, 29149, 45999, 75473, 127763, 254293, 15229, 1814681, 2028175, 5211049, 5377091, 11173301, 41653425, 38893795, 22709783, 178029349, 491688809, 720890733},
  {1, 3, 7, 11, 17, 19, 9, 155, 373, 753, 1313, 2205, 3571, 16317, 16151, 15325, 73393, 110163, 418551, 927067, 1440385, 2301859, 3326713, 7425195, 19797477, 17673281, 115313713, 21114285, 420411171, 665378861, 766418055},
  {1, 3, 3, 7, 15, 43, 65, 183, 407, 123, 1151, 375, 3461, 6673, 12985, 21005, 117969, 88979, 436499, 326487, 949919, 4091547, 2710769, 11530823, 22846183, 38216011, 58828271, 15575943, 449916789, 821986081, 973904073},
  {1, 3, 7, 7, 9, 1, 87, 247, 489, 123, 1677, 1947, 7961, 13497, 27919, 28993, 111313, 36179, 407223, 514647, 1543897, 2663793, 5750631, 7885127, 32448953, 15000235, 8545981, 127315883, 439156041, 314142665, 256575647},
  {1, 3, 3, 7, 19, 21, 95, 227, 217, 133, 69, 1535, 699, 3521, 29255, 34733, 92913, 5075, 23219, 23191, 151235, 599653, 6645551, 14257491, 4034601, 11023125, 48435829, 222427535, 411171819, 610400337, 387288919},
  {1, 3, 5, 3, 7, 57, 45, 251, 407, 81, 1259, 2425, 2217, 13097, 12773, 14643, 74833, 106707, 296981, 458099, 2011223, 296393, 6410557, 12150539, 25982375, 34188289, 92546811, 263956361, 291858841, 215555577, 225790869},
  {1, 1, 1, 11, 23, 37, 13, 229, 467, 591, 1521, 469, 3763, 2289, 14233, 24053, 105841, 7249, 330577, 377115, 1137479, 1121909, 2968413, 11237141, 23350243, 60391903, 885729, 5672901, 297374147, 900963905, 97114953},
  {1, 3, 5, 1, 27, 53, 105, 5, 85, 765, 1973, 2597, 5725, 1063, 18145, 961, 111249, 45363, 463701, 167889, 1706827, 592197, 331641, 304085, 2995109, 36942061, 24761573, 219765813, 165724237, 977394103, 953662449},
  {1, 3, 7, 1, 21, 47, 115, 95, 403, 3, 1593, 3379, 7371, 15553, 12503, 57979, 58609, 155347, 123319, 1039857, 894149, 2493919, 990531, 12385679, 20610563, 34790323, 3997001, 215245827, 268236347, 452769329, 1286140551},
  {1, 1, 3, 1, 1, 35, 121, 29, 379, 245, 919, 2673, 3503, 14197, 31193, 8355, 49457, 149777, 495699, 331121, 686929, 1197043, 7782153, 8590669, 13673131, 2583941, 119169351, 267346017, 222636895, 220461669, 2015315305},
  {1, 3, 5, 11, 19, 49, 97, 7, 195, 1013, 1671, 3415, 2009, 13389, 4837, 27453, 26193, 105235, 470549, 133787, 1924675, 3653697, 1440625, 6349687, 4005811, 36644645, 51254103, 137010279, 532566697, 631259613, 1163597141},
  {1, 1, 5, 13, 9, 15, 115, 97, 463, 449, 303, 2681, 1215, 12559, 15685, 21321, 38449, 224433, 257429, 193885, 103513, 2448223, 40803, 6002673, 32653087, 11218289, 5648223, 61187241, 263859087, 1021121631, 1476353397},
  {1, 3, 5, 13, 23, 5, 113, 193, 419, 301, 1121, 317, 5503, 4683, 25519, 65, 127409, 201203, 199477, 706781, 554695, 2080117, 2551137, 1280977, 25859731, 17870653, 25614769, 239550317, 441246223, 723343323, 51212927},
  {1, 3, 3, 7, 15, 29, 45, 97, 323, 475, 143, 1173, 4033, 8939, 31849, 3575, 122065, 102643, 522707, 733367, 513631, 2841069, 3768413, 7110865, 30572915, 22053323, 77693343, 216106437, 355504305, 986949403, 178168921},
  {1, 1, 7, 7, 21, 1, 101, 143, 197, 409, 855, 1753, 5211, 3763, 11139, 37309, 60273, 44657, 404471, 761879, 40645, 27985, 5138005, 11929279, 5012053, 10084937, 122544711, 111468649, 190132043, 121421539, 1697877427},
  {1, 1, 3, 13, 25, 33, 55, 45, 381, 349, 991, 535, 4823, 3701, 31629, 48037, 62897, 228753, 19, 747037, 1354057, 856945, 8237639, 10355389, 29803053, 35596813, 34106607, 74991239, 140401351, 742816485, 371159869},
  {1, 3, 1, 11, 17, 51, 27, 57, 409, 551, 949, 365, 8093, 10831, 19697, 39437, 34705, 168627, 523089, 597019, 1612353, 709955, 931403, 4803145, 3252937, 52324823, 109168709, 63595581, 123432653, 994757759, 1688834017},
  {1, 3, 5, 3, 31, 33, 81, 49, 91, 865, 469, 2115, 377, 8237, 31907, 38239, 45009, 184595, 409461, 1008563, 888527, 1099217, 2854465, 12821185, 18416363, 8155057, 112734469, 156235027, 74125961, 689117309, 626858291},
  {1, 1, 3, 7, 29, 59, 57, 17, 121, 889, 1557, 1797, 5001, 14209, 21355, 59739, 92881, 11857, 274419, 640759, 1661133, 1813739, 7200329, 346657, 13239145, 24381193, 4721861, 116396757, 251666521, 148916817, 656459387},
  {1, 1, 5, 9, 11, 45, 89, 87, 397, 785, 525, 1593, 5251, 12449, 23579, 54265, 70641, 127697, 36341, 181849, 498011, 1126141, 7794633, 11458823, 27939325, 65417217, 1004765, 193377417, 521992083, 967009777, 2056477963},
  {1, 3, 5, 11, 5, 31, 19, 47, 207, 331, 91, 1691, 5171, 53, 15945, 33349, 39025, 237587, 273877, 587931, 976853, 957679, 2345859, 16375263, 17553759, 32529403, 68548779, 186716715, 272457347, 343429573, 589907129},
  {1, 1, 1, 15, 11, 41, 91, 177, 505, 871, 815, 3673, 5631, 9915, 1133, 37861, 98225, 230001, 314769, 38687, 718299, 753913, 5463947, 1146241, 13551753, 50166711, 27089631, 53788041, 277576495, 836288523, 1098367837},
  {1, 3, 5, 5, 25, 63, 53, 231, 55, 51, 481, 303, 1859, 11973, 28557, 22045, 16881, 19219, 58261, 277781, 629449, 511183, 7260069, 3570167, 18117735, 10073859, 115672945, 100336927, 448732147, 982590901, 1385783005},
  {1, 1, 5, 3, 27, 11, 37, 91, 363, 411, 1131, 3369, 377, 6585, 7353, 42949, 124113, 14417, 392533, 815763, 631755, 146651, 1267637, 13902251, 16200731, 56510059, 123146363, 245944537, 41518089, 582062921, 1196405833},
  {1, 3, 1, 9, 31, 63, 83, 23, 405, 941, 119, 1471, 2509, 15507, 29239, 49613, 76177, 141171, 492977, 696473, 4943, 2520527, 110467, 14410695, 18401957, 3707805, 51974407, 136462223, 179873501, 927808259, 2027767431},
  {1, 1, 5, 1, 11, 63, 117, 237, 407, 231, 1425, 71, 8005, 4023, 9029, 59819, 90513, 17457, 65685, 1011697, 1541595, 715119, 4099813, 8337725, 22605799, 32180951, 37699457, 202090455, 101475829, 185671943, 2079279637},
  {1, 1, 5, 7, 1, 9, 43, 87, 351, 63, 1075, 3381, 5447, 2437, 24983, 26905, 111921, 237201, 409653, 121175, 1262161, 2895577, 1312187, 13083623, 26402703, 22353135, 126834243, 196932485, 331261431, 160476469, 478254055},
  {1, 3, 7, 5, 5, 35, 33, 89, 251, 819, 1735, 2625, 6363, 6837, 27603, 26669, 48113, 70291, 186743, 436245, 247637, 610771, 2532241, 14763849, 2527211, 13180611, 123615767, 244349585, 89978283, 714996293, 1861721827},
  {1, 3, 7, 13, 29, 63, 51, 245, 371, 791, 907, 3499, 3033, 8443, 20023, 1805, 28625, 114675, 455607, 975261, 1162317, 349391, 1783427, 13457509, 157923, 26634279, 53721467, 17759163, 372729769, 88366411, 934766791},
  {1, 1, 5, 7, 13, 15, 109, 197, 451, 709, 929, 3193, 5727, 11185, 29479, 1671, 22577, 255185, 390197, 296631, 362845, 1370719, 2037757, 10536053, 17472723, 23076213, 105824689, 190660969, 269402799, 617143329, 426979191},
  {1, 1, 7, 13, 19, 23, 97, 9, 359, 635, 777, 39, 893, 2531, 13563, 19295, 105233, 48241, 165495, 698557, 1169219, 664135, 463825, 10239001, 13073687, 29069259, 99595097, 248903927, 223414989, 132593811, 1481073963},
  {1, 1, 5, 1, 31, 63, 55, 7, 157, 877, 991, 1317, 1595, 2019, 21435, 52255, 90609, 237553, 176885, 733105, 1786191, 3326063, 3043751, 10495703, 7638701, 37146717, 37049839, 113975061, 286093227, 171746675, 788448395},
  {1, 1, 5, 3, 19, 37, 23, 13, 335, 431, 483, 615, 2431, 505, 26245, 63323, 116145, 32465, 217365, 716147, 282563, 3925621, 5775239, 8950525, 18063295, 20097983, 86138835, 15532151, 19727567, 105586121, 924549109},
  {1, 3, 7, 5, 5, 9, 37, 65, 303, 423, 1907, 2661, 7213, 2975, 29045, 16243, 34961, 59443, 342935, 220533, 1696981, 1923193, 4892565, 14156369, 7184191, 63900151, 7174371, 251734197, 419492125, 845381935, 2000598789},
  {1, 3, 1, 5, 13, 37, 115, 217, 227, 159, 707, 1387, 943, 4935, 5503, 35171, 18609, 238131, 297137, 545141, 488029, 3162453, 2632611, 8303305, 28126195, 64158991, 72496691, 259127611, 114482655, 687205239, 1329152783},
  {1, 3, 7, 9, 19, 15, 87, 233, 453, 159, 169, 1077, 2129, 413, 19773, 629, 83889, 45043, 138359, 816377, 919971, 1176863, 7730503, 16342425, 24148181, 41072911, 16317241, 7622981, 515641889, 662452173, 306985805},
  {1, 1, 5, 15, 29, 39, 37, 243, 233, 365, 1843, 2219, 1255, 15287, 603, 13511, 72977, 234897, 207189, 983327, 802797, 1430807, 371029, 14658979, 24835033, 20688829, 79114883, 219214395, 280072983, 834634055, 1879151563},
  {1, 1, 3, 3, 31, 53, 33, 125, 497, 597, 127, 1829, 3905, 2611, 4263, 40971, 39601, 96529, 292371, 244307, 618607, 1742085, 6575409, 6551853, 26200929, 58428901, 84025807, 148645333, 268741361, 22083139, 230296311},
  {1, 3, 5, 9, 11, 47, 71, 215, 383, 321, 1445, 135, 5953, 8791, 22073, 16537, 103153, 158547, 390005, 1023161, 776763, 1548607, 4057975, 3309543, 28467183, 37511121, 102177589, 162255383, 240901969, 1034476711, 1846462793},
  {1, 3, 3, 13, 15, 7, 7, 133, 401, 459, 1117, 3165, 4105, 11943, 22431, 56821, 26929, 57683, 131859, 77501, 982975, 2873623, 3837271, 8261429, 33247169, 33697115, 52456909, 76270573, 257284281, 351761719, 704263727},
  {1, 1, 7, 9, 31, 39, 19, 7, 19, 401, 79, 3641, 6815, 1489, 7537, 49467, 50385, 212209, 60215, 192409, 1801327, 1607447, 6089539, 2601335, 5169091, 35572481, 77535391, 56738601, 293264975, 1017937825, 905445025},
  {1, 3, 7, 7, 17, 11, 91, 205, 251, 321, 515, 3521, 311, 3169, 271, 34749, 111953, 207603, 166615, 290871, 423073, 4179739, 3454283, 5294941, 15798795, 27196113, 87846867, 26610545, 309465127, 666768913, 1202879519},
  {1, 3, 3, 7, 29, 15, 5, 153, 83, 603, 1373, 997, 4939, 9811, 243, 5375, 123665, 65363, 381011, 634007, 507373, 3320863, 1356885, 2024585, 14763363, 63976715, 113699053, 260857557, 233644379, 456121731, 1161052739},
  {1, 1, 3, 11, 21, 47, 25, 221, 237, 177, 535, 2819, 6213, 7877, 26795, 36609, 45361, 89585, 519443, 609915, 1343013, 1426463, 4693001, 13970445, 4322525, 62581985, 22157927, 97843699, 38058293, 607751285, 729072635},
  {1, 3, 7, 3, 31, 1, 69, 73, 47, 653, 139, 1649, 7183, 1293, 26507, 38415, 56785, 77875, 132631, 822419, 260079, 3557393, 1210453, 10856601, 23255487, 34031997, 20811003, 36745793, 213835743, 890757181, 544741019},
  {1, 1, 1, 13, 17, 41, 23, 73, 115, 509, 787, 3733, 1871, 171, 29967, 39941, 37201, 21073, 254609, 936125, 888481, 1177625, 1181735, 14883001, 9631939, 58104141, 63459139, 8675557, 113451359, 813152539, 1800582047},
  {1, 3, 5, 1, 9, 7, 61, 23, 105, 381, 1421, 2887, 3717, 643, 26375, 57991, 38321, 221587, 295189, 582545, 729849, 3844119, 6912013, 9463975, 22698201, 26802221, 61636285, 208553367, 504124149, 700870771, 1209260151},
  {1, 3, 3, 3, 19, 3, 101, 117, 393, 83, 1255, 3331, 6481, 8661, 20855, 28875, 65137, 257651, 136403, 554835, 663907, 2217491, 3651125, 12382757, 8921817, 66958339, 108935383, 3384563, 486325601, 1013237701, 1739220839},
  {1, 3, 5, 11, 21, 13, 111, 193, 51, 899, 159, 1989, 7931, 10511, 3933, 447, 78353, 202387, 313653, 57179, 1229797, 2457117, 5572191, 16380625, 23557635, 57900019, 16763343, 71226421, 210271211, 393512287, 1883604173},
  {1, 1, 5, 15, 23, 35, 49, 139, 397, 145, 597, 1847, 7077, 715, 20227, 42183, 52145, 91921, 131669, 478495, 1230055, 885267, 7622209, 5164763, 19494685, 10426945, 2440997, 247761831, 342328341, 756006011, 2125055603},
  {1, 3, 3, 3, 17, 3, 87, 233, 35, 317, 337, 237, 6901, 3439, 20033, 10307, 93297, 90323, 262003, 882323, 1063137, 942995, 2919559, 11108153, 16053715, 45112749, 35992257, 165040925, 310618469, 58879583, 563841905},
  {1, 3, 5, 3, 11, 35, 13, 171, 7, 963, 1443, 1501, 7617, 963, 25453, 62589, 115249, 35219, 142389, 407283, 815419, 2452915, 4045757, 13462971, 10149975, 21422355, 91137427, 200665165, 316502609, 113769107, 1649981885},
  {1, 1, 1, 5, 11, 9, 39, 175, 409, 411, 1407, 2743, 4255, 989, 15823, 1707, 91793, 259505, 475345, 88277, 1316603, 2484921, 6146711, 602975, 8067081, 62892523, 71151023, 8535079, 124291951, 351382477, 705139167},
  {1, 1, 7, 13, 27, 55, 63, 239, 355, 417, 2007, 2299, 2921, 1637, 10687, 60615, 1905, 149233, 297175, 721181, 641899, 4058247, 7461615, 6565215, 24773363, 6360369, 112653639, 138509163, 85988921, 77597333, 1340103279},
  {1, 1, 7, 9, 5, 61, 57, 73, 263, 307, 2003, 1763, 639, 5885, 14709, 16985, 31761, 175025, 363383, 714425, 288437, 3795341, 2257897, 13724601, 17263607, 53975747, 41826915, 250591443, 152283823, 132548269, 75537829},
  {1, 1, 3, 3, 21, 55, 19, 249, 509, 533, 1361, 1397, 2777, 15523, 4389, 13339, 53169, 98193, 420531, 513939, 716773, 2438023, 4628099, 2996521, 25902925, 45480261, 77736449, 175431429, 485767849, 822533843, 2011975445},
  {1, 3, 5, 15, 9, 3, 91, 237, 451, 299, 1541, 4083, 879, 7859, 21585, 14833, 32273, 102227, 135157, 209343, 1591737, 1625235, 2612459, 5880893, 23491891, 55293627, 119541845, 185340611, 221120383, 36050403, 1976439777},
  {1, 1, 7, 3, 31, 47, 49, 231, 123, 391, 1633, 2567, 5577, 1631, 27951, 22913, 15601, 33617, 74647, 108147, 410031, 3443615, 6415585, 12649655, 31921515, 16109335, 1768721, 235639255, 60398873, 160280015, 1120328799},
  {1, 3, 7, 13, 11, 13, 1, 111, 183, 87, 839, 1915, 5523, 3677, 13065, 38225, 28369, 122675, 236983, 531133, 403131, 810141, 3575953, 4722911, 23138343, 40890983, 22508663, 259115, 252896643, 254725453, 815720985},
  {1, 1, 3, 7, 15, 15, 63, 241, 167, 345, 653, 701, 4725, 12911, 11545, 24475, 59249, 250833, 308211, 645399, 1200895, 763071, 1500591, 1921121, 33083447, 5599689, 105255709, 43138509, 433688997, 636664991, 1197991529},
  {1, 1, 3, 7, 25, 15, 49, 235, 331, 639, 965, 1117, 7147, 3789, 3309, 20255, 85905, 94769, 293619, 1013463, 1406313, 1815743, 2382241, 16746107, 27865787, 55920175, 78048437, 128989773, 518377403, 844725981, 1631828189},
  {1, 3, 5, 7, 7, 63, 93, 241, 253, 31, 951, 3723, 3359, 7303, 191, 36427, 129041, 11539, 290709, 16855, 1673975, 2710959, 5539821, 8649377, 7648237, 31758223, 124606471, 163809211, 384794991, 581146199, 189373103},
  {1, 3, 7, 9, 9, 59, 5, 107, 181, 413, 1269, 3121, 1929, 11921, 8931, 47459, 129841, 224691, 459671, 896473, 173305, 3168171, 771989, 5647003, 27435653, 30774925, 110681413, 23348385, 234408505, 681110881, 1066152339},
  {1, 3, 1, 15, 25, 27, 13, 47, 295, 111, 1287, 2551, 4887, 4145, 17063, 42037, 128273, 65395, 332497, 678207, 240169, 1852811, 2670845, 2129535, 27876055, 5337215, 116184887, 87503911, 149169927, 659265185, 2055392951},
  {1, 3, 3, 13, 17, 17, 21, 17, 491, 845, 1463, 1305, 1375, 16149, 19331, 25043, 105809, 101939, 378643, 938461, 1156321, 284545, 4307909, 16746017, 17966811, 8397821, 29615687, 43296809, 117582671, 243367461, 1009906643},
  {1, 3, 5, 1, 27, 5, 93, 139, 283, 711, 1141, 1743, 5001, 8851, 19351, 12275, 54257, 87955, 476053, 960593, 1799147, 1235349, 6479853, 5166779, 23342859, 20326775, 9113477, 3070559, 351042521, 144182531, 493364135},
  {1, 1, 1, 1, 23, 25, 51, 63, 429, 735, 201, 3785, 6677, 16375, 19681, 17857, 107057, 96145, 421297, 538097, 506727, 1615721, 4066627, 15041871, 23985597, 9096431, 67307545, 99909657, 486379749, 615431527, 699045841},
  {1, 3, 3, 3, 9, 63, 71, 147, 463, 465, 1163, 1045, 6967, 12537, 31853, 38391, 128785, 137267, 503859, 53395, 487609, 3215599, 5158231, 16015107, 3696895, 61769921, 82681435, 152707333, 462845287, 248035273, 1379516989},
  {1, 3, 7, 1, 5, 51, 79, 239, 389, 3, 601, 3787, 7635, 16295, 1681, 63971, 101393, 243923, 438071, 645073, 855861, 463331, 3138335, 4476383, 10568501, 51179219, 50570377, 201352731, 113094819, 775186935, 306306849},
  {1, 3, 1, 3, 5, 31, 103, 89, 321, 971, 783, 3685, 1155, 10353, 2167, 35423, 34129, 105523, 454545, 530771, 276789, 4139215, 5380439, 521609, 32092721, 37515099, 122959071, 100351445, 89216371, 351290273, 1118801063},
  {1, 1, 5, 15, 25, 19, 93, 59, 361, 217, 1141, 597, 5877, 15961, 1593, 22925, 67249, 156785, 323573, 72639, 26345, 974947, 7198573, 8819499, 21243737, 20008457, 12033669, 265959301, 165772101, 872142121, 1613682441},
  {1, 3, 1, 9, 25, 59, 69, 89, 477, 89, 487, 237, 5625, 9579, 30421, 21883, 81393, 175315, 235633, 875993, 860969, 1199083, 6091637, 16679209, 16913965, 49021833, 6030231, 162269213, 32774473, 665584667, 19670981},
  {1, 1, 3, 7, 1, 5, 13, 225, 9, 981, 1081, 1407, 6855, 15215, 21713, 62313, 56273, 182225, 65331, 766327, 171633, 1560437, 647005, 12745649, 8205113, 7029093, 29771017, 53496463, 418105911, 247134303, 293120161},
  {1, 1, 7, 15, 11, 13, 119, 109, 151, 245, 1195, 3741, 755, 8047, 15431, 21001, 88145, 85457, 453463, 698815, 1985531, 3047293, 3752295, 13520701, 14781191, 20265349, 65359291, 191204589, 363038435, 704625311, 1173411991},
  {1, 3, 7, 3, 17, 47, 107, 137, 99, 255, 1597, 3281, 5779, 13487, 15061, 19199, 105041, 95955, 231031, 1007763, 1409441, 2994943, 856891, 8920473, 25667539, 41773551, 95467565, 27510753, 444421187, 933748863, 1422668581},
  {1, 1, 3, 3, 9, 39, 77, 227, 511, 839, 1375, 3887, 25, 14763, 13259, 217, 59825, 79953, 374163, 881715, 366841, 754519, 5539101, 12576755, 6765263, 63294423, 49094191, 74153087, 265332681, 639735803, 1307638299},
  {1, 3, 5, 7, 17, 3, 87, 61, 439, 287, 709, 4085, 4251, 8945, 28203, 24011, 74065, 175667, 257877, 245911, 822689, 883411, 4545575, 15316013, 32128263, 45604559, 47011733, 209099269, 372489611, 156812961, 1938962523},
  {1, 3, 1, 1, 29, 25, 49, 101, 209, 359, 285, 1593, 4161, 2943, 23225, 6381, 44401, 142995, 266545, 67377, 1601005, 2521929, 5118977, 9921685, 28446049, 50892567, 3553901, 190321673, 432558513, 851272303, 830898121},
  {1, 1, 3, 13, 1, 45, 87, 7, 491, 399, 905, 1403, 4791, 7419, 14355, 47767, 16721, 66705, 26259, 336605, 581681, 2204125, 2650119, 7573751, 9769115, 2379839, 123556633, 216405163, 479914279, 697436331, 1020943267},
  {1, 1, 7, 15, 13, 25, 111, 197, 297, 301, 499, 4007, 2235, 7681, 4641, 32447, 108177, 6065, 478199, 494751, 1667517, 2386921, 8040575, 9695381, 17576409, 12358365, 129668963, 44660823, 88770891, 524454065, 1412010737},
  {1, 1, 3, 3, 27, 41, 97, 83, 405, 353, 1609, 201, 1503, 10673, 29377, 20445, 87953, 233393, 262387, 545971, 1587691, 170841, 6606897, 774211, 31895685, 64770129, 68080569, 134517657, 522870447, 822168001, 2010001201},
  {1, 1, 7, 3, 9, 47, 65, 191, 207, 545, 377, 3011, 7361, 3467, 14073, 46769, 39217, 173393, 463799, 207347, 235129, 4018271, 7010385, 2648111, 19364351, 41167153, 124496585, 89746387, 373948305, 553304507, 481996425},
  {1, 1, 7, 5, 7, 39, 9, 91, 187, 949, 1829, 161, 3689, 4145, 32675, 23263, 44273, 247665, 370519, 985301, 809271, 1758167, 3323929, 4012203, 30472619, 62515301, 95431317, 7788017, 28259577, 2482881, 1994131},
  {1, 1, 5, 9, 29, 9, 83, 113, 77, 673, 613, 3645, 6671, 8583, 27701, 18615, 86289, 126417, 415605, 170777, 1867693, 3820537, 7385699, 7992833, 25967613, 23297489, 11822901, 118697741, 144138815, 452894839, 741853957},
  {1, 3, 5, 9, 29, 13, 127, 247, 285, 845, 463, 539, 4441, 1867, 12469, 16213, 5681, 28243, 401781, 95289, 1805293, 2183005, 1552911, 3933703, 12500653, 39253117, 31007455, 255223755, 238413737, 992302075, 123489445},
  {1, 3, 7, 15, 1, 29, 47, 157, 239, 595, 563, 1103, 3431, 2849, 28125, 19969, 123473, 146483, 307511, 98047, 667505, 545357, 2845311, 10490445, 2398943, 24558179, 15965475, 199842111, 318983671, 910450353, 244337997},
  {1, 1, 1, 15, 25, 13, 1, 131, 57, 257, 2021, 169, 7603, 10721, 21675, 63171, 122833, 260273, 153297, 1020063, 1308457, 2210941, 5416689, 13406099, 13796745, 17708209, 41646741, 157230201, 314097027, 731170705, 2092447771},
  {1, 3, 5, 3, 5, 19, 31, 57, 275, 381, 775, 681, 1145, 12237, 5141, 29375, 67057, 225843, 223797, 118419, 1725237, 4091971, 7151343, 11117545, 16772323, 65815981, 51161431, 57439129, 230065449, 142253245, 483364997},
  {1, 3, 5, 13, 27, 13, 47, 201, 267, 581, 1563, 3845, 951, 7209, 27253, 19755, 5009, 208051, 28981, 846301, 1812907, 470365, 7239903, 10226681, 7401691, 34345781, 42258443, 82458037, 316705895, 128352729, 2144442917},
  {1, 3, 5, 15, 19, 35, 57, 17, 61, 273, 967, 3029, 1747, 1753, 31321, 23711, 41681, 204083, 380149, 254239, 720931, 851571, 7931593, 9438977, 29235501, 7207681, 20056695, 132225957, 318481379, 1064817577, 746869449},
  {1, 1, 1, 5, 13, 13, 7, 177, 335, 393, 1401, 1411, 4703, 8259, 1281, 39835, 101201, 106385, 59953, 789205, 678845, 967037, 5119223, 9290497, 14264575, 65861689, 12258409, 135526067, 237617871, 481192947, 973182961},
  {1, 1, 3, 15, 25, 27, 27, 121, 183, 105, 663, 1375, 6987, 7151, 13763, 39323, 3089, 13777, 351891, 696735, 301865, 245355, 1316043, 2040105, 13347719, 34152377, 127792263, 17921775, 377409371, 606133599, 1684683219},
  {1, 3, 7, 5, 15, 1, 81, 129, 455, 163, 675, 81, 3735, 14409, 7269, 16425, 2065, 44467, 488503, 745973, 297983, 3399633, 7235457, 5506929, 32642775, 24016531, 45470675, 105910945, 473603847, 774420825, 1797950581},
  {1, 3, 3, 11, 13, 7, 79, 157, 165, 663, 229, 3539, 1837, 6485, 30729, 42157, 15473, 247987, 176883, 433723, 1263741, 3652823, 2158047, 9006861, 32004949, 31711495, 34105141, 214099875, 272463261, 144824645, 710713977},
  {1, 1, 5, 15, 9, 9, 9, 47, 133, 863, 43, 1461, 511, 13991, 24781, 19221, 22033, 195409, 472949, 13503, 951673, 270073, 2261433, 13918143, 14502709, 27230383, 31753051, 237602213, 179936655, 622364599, 1296344157},
  {1, 3, 1, 7, 31, 33, 103, 13, 159, 689, 1353, 4025, 6051, 7683, 1741, 30047, 51665, 142483, 241393, 1028887, 334383, 1042545, 3056855, 4680989, 13526863, 32312065, 102866009, 178006345, 301891507, 673751347, 933353533},
  {1, 1, 3, 11, 5, 45, 71, 219, 475, 585, 1207, 3163, 4661, 4713, 12729, 30445, 45937, 171889, 270451, 424763, 1483381, 2117853, 6909335, 5609419, 8799147, 56309881, 30185639, 246953419, 270497957, 843175897, 499957161},
  {1, 3, 7, 5, 5, 53, 101, 227, 129, 521, 91, 1129, 4683, 11235, 24697, 45055, 13009, 213395, 212663, 846997, 24117, 2464613, 6783925, 11916307, 26672177, 16004921, 130858859, 200854841, 421413371, 855637267, 1437862185},
  {1, 1, 3, 13, 1, 43, 7, 1, 73, 857, 1713, 185, 1685, 2369, 24187, 40419, 90033, 147825, 204627, 821917, 1643569, 3104091, 3982807, 12958833, 23672249, 59754249, 66447713, 185333257, 433472101, 619530801, 1207292043},
  {1, 1, 7, 7, 21, 7, 13, 177, 503, 1003, 1091, 2411, 1433, 9063, 13901, 3329, 82609, 57617, 21751, 891895, 1543909, 2857719, 202909, 15487201, 8286471, 41407355, 29799091, 37107867, 338279433, 986626551, 53017181},
  {1, 1, 7, 1, 7, 41, 99, 203, 325, 249, 1763, 545, 2981, 14125, 7815, 11385, 2865, 189105, 109335, 332721, 359031, 3666393, 3112179, 10737915, 6395989, 65390921, 47750323, 58618673, 439431893, 836246365, 206046039},
  {1, 3, 7, 11, 3, 11, 95, 137, 325, 701, 1177, 1631, 4483, 2955, 30229, 25577, 97297, 124051, 146551, 471675, 1589747, 1980123, 730767, 5752473, 12535381, 30721325, 12461897, 39969903, 247327923, 83021627, 1172490213},
  {1, 1, 7, 7, 17, 45, 77, 103, 143, 97, 1963, 3635, 1539, 10491, 23483, 22767, 19729, 72081, 304279, 455639, 1354401, 1790045, 5510621, 4327991, 30679615, 11930833, 117045019, 240302115, 156321235, 611610187, 1364300651},
  {1, 1, 7, 15, 7, 5, 81, 63, 243, 55, 39, 207, 2315, 8285, 8155, 11631, 44561, 126449, 315991, 816735, 1993143, 2410613, 4268481, 1886959, 2509347, 61454023, 43617751, 246331871, 209505563, 205111309, 1595565579},
  {1, 3, 5, 15, 23, 19, 115, 9, 125, 851, 161, 3767, 3513, 1855, 11139, 1719, 76721, 235667, 144661, 42335, 1437095, 230979, 4260227, 2056729, 31844909, 38762435, 45321649, 49082631, 318348297, 155946095, 1699578419},
  {1, 3, 7, 11, 11, 23, 15, 13, 235, 5, 1039, 1425, 6485, 5539, 8967, 64809, 125745, 259155, 307415, 322363, 675195, 698311, 5491423, 3182109, 10081147, 22209365, 74786655, 154413793, 371255301, 1004567123, 529158423},
  {1, 3, 5, 7, 19, 11, 83, 135, 45, 905, 1081, 1857, 3185, 13555, 21365, 38143, 28569, 5035, 331773, 378063, 599371, 1984483, 8314411, 12374751, 5887797, 32836305, 97327809, 149874409, 67056377, 454187771, 1934158909},
  {1, 1, 5, 1, 25, 27, 119, 109, 167, 847, 1539, 2653, 797, 11185, 23501, 22389, 4489, 252521, 153517, 680009, 1421137, 1325427, 4736575, 13332197, 22556479, 49018551, 94537003, 176326565, 470061733, 310556393, 2082403045},
  {1, 1, 7, 7, 11, 3, 51, 97, 277, 557, 207, 3645, 825, 8521, 26653, 60071, 92233, 215785, 339775, 982095, 806211, 1010683, 3543115, 13332601, 5894349, 46330485, 66629895, 39949909, 522268305, 95965041, 601541813},
  {1, 3, 3, 15, 17, 35, 57, 7, 267, 549, 97, 243, 1137, 10311, 6737, 19077, 80953, 21643, 355883, 652231, 1282633, 2040699, 6758401, 2728015, 2941139, 22978605, 89220121, 244490347, 332610329, 168790671, 699645705},
  {1, 1, 1, 15, 23, 33, 27, 203, 415, 1023, 1145, 1881, 7715, 4413, 3727, 5185, 100329, 156281, 365577, 582983, 1427935, 1982057, 8185891, 7600179, 13011159, 28596455, 21983201, 116386785, 195964699, 179251253, 1470696327},
  {1, 1, 3, 3, 13, 47, 63, 13, 75, 505, 595, 2911, 4029, 14187, 23151, 42877, 7513, 143737, 402923, 911563, 1784261, 3205495, 1320039, 15105061, 25382387, 74945, 121091451, 164566727, 31216565, 200211123, 1305138775},
  {1, 1, 5, 15, 23, 5, 11, 65, 147, 675, 1961, 2177, 727, 15077, 23759, 10195, 66601, 165241, 415645, 974919, 212191, 1907181, 265523, 14153145, 9807611, 36065851, 133494625, 209008729, 132811791, 296725741, 2053238023},
  {1, 3, 5, 9, 9, 39, 69, 229, 341, 627, 1331, 3139, 3921, 9219, 14887, 4659, 32537, 182331, 165021, 105921, 677585, 1197519, 1838925, 15174509, 32051245, 19673371, 15627611, 154613291, 332764297, 1007133659, 1573230719},
  {1, 1, 7, 3, 1, 35, 49, 71, 165, 83, 719, 2771, 6475, 7821, 16709, 4449, 3945, 58153, 438799, 34379, 844873, 3046235, 5037417, 669967, 14418301, 26476571, 128714679, 123995195, 144971411, 506739253, 45354909},
  {1, 3, 5, 5, 23, 15, 3, 57, 465, 77, 121, 3767, 6841, 13601, 12035, 14075, 119305, 225051, 294109, 736589, 976847, 1400295, 6533995, 16526145, 12319209, 28617237, 36260209, 64757503, 304523025, 895405993, 1289909755},
  {1, 1, 7, 3, 3, 23, 45, 131, 287, 941, 713, 415, 6865, 14209, 29555, 55493, 11529, 200857, 171311, 437579, 979915, 2970223, 2388853, 378331, 33032551, 19372805, 128592145, 19679143, 168871929, 528838153, 410930507},
  {1, 3, 5, 11, 29, 35, 55, 75, 225, 779, 569, 1795, 1377, 12765, 19081, 47287, 81385, 198987, 341693, 703043, 2003269, 3960523, 6716719, 8278883, 32128441, 33351539, 64859681, 223328203, 234050041, 757130709, 298009041},
  {1, 3, 7, 3, 31, 47, 127, 89, 157, 737, 1395, 3615, 7923, 14731, 15797, 40061, 34009, 60459, 25711, 587595, 1830087, 3187159, 900263, 12931809, 9307045, 40712281, 57843707, 44366007, 17018811, 505884115, 1653149837},
  {1, 1, 1, 11, 21, 37, 21, 59, 9, 141, 193, 3095, 3435, 12371, 26931, 61861, 64537, 135449, 347817, 60739, 1841501, 3929325, 282765, 2928211, 28984289, 21084261, 60165081, 187164063, 530129091, 432451611, 1256039499},
  {1, 1, 3, 7, 13, 51, 15, 153, 77, 1013, 651, 3949, 6229, 14297, 1039, 46139, 96329, 131577, 9051, 998607, 839493, 895339, 5091319, 10238513, 19958677, 56180173, 68522115, 143201605, 80031677, 223441985, 1030202023},
  {1, 3, 3, 13, 7, 43, 95, 61, 217, 3, 549, 739, 123, 3661, 15375, 13919, 131049, 14075, 251771, 171717, 1119583, 1015923, 7202295, 3267141, 4747969, 61390395, 128607885, 195667819, 226178083, 318760357, 463628759},
  {1, 3, 5, 9, 13, 37, 101, 89, 55, 413, 1089, 775, 7575, 13063, 31393, 29583, 17593, 113547, 263677, 478273, 455125, 3617101, 2067437, 2764913, 18493919, 41464437, 35836233, 20594047, 211864255, 368068047, 113038297},
  {1, 1, 3, 9, 25, 63, 119, 143, 499, 145, 603, 2067, 4713, 13457, 14053, 117, 57865, 3241, 383019, 516801, 1293265, 1896935, 4407039, 7774343, 8134731, 63201513, 90301667, 11297995, 429880481, 47111689, 1821224253},
  {1, 1, 5, 9, 7, 23, 57, 253, 115, 591, 2003, 63, 7615, 11493, 28519, 47087, 58025, 40121, 292605, 890049, 1962319, 2835071, 8242097, 51205, 29038859, 24488775, 125160827, 82705895, 120164871, 937593277, 42243295},
  {1, 1, 7, 3, 7, 53, 121, 33, 233, 645, 1093, 1697, 7213, 2603, 10743, 51303, 100825, 206649, 98751, 821835, 841295, 2085965, 1863585, 6015193, 25655297, 5773837, 38558989, 255469353, 162997445, 833406147, 1042501183},
  {1, 3, 5, 7, 13, 31, 17, 125, 93, 969, 159, 1529, 7165, 7371, 8707, 56953, 18393, 48795, 104525, 913743, 820309, 1134839, 6643177, 2670421, 9857893, 50260225, 64010423, 210211665, 371033461, 629773043, 1332011787},
  {1, 3, 3, 1, 13, 9, 91, 25, 171, 843, 1635, 2043, 1043, 15893, 11409, 53689, 21033, 145931, 130891, 593865, 558549, 3607505, 7192467, 9377585, 24573603, 5993811, 120550507, 107806051, 420609803, 384557517, 56612873},
  {1, 3, 5, 7, 13, 19, 89, 97, 203, 923, 1109, 2061, 463, 11703, 8925, 56015, 7993, 21355, 391677, 339791, 1747925, 2970235, 8383649, 13910345, 30956435, 55738131, 62726301, 165571861, 522630615, 346869471, 645598197},
  {1, 3, 5, 11, 5, 21, 79, 237, 195, 649, 717, 211, 919, 12855, 3045, 39659, 41529, 203947, 325597, 1021251, 1471965, 3374973, 1842903, 12246405, 23305131, 24909233, 81399397, 200438795, 488882591, 42029983, 2069951037},
  {1, 1, 1, 15, 13, 19, 21, 69, 393, 257, 1263, 309, 3209, 8403, 24467, 6467, 117017, 173081, 274841, 76615, 1351237, 895067, 1557165, 8429549, 28673233, 41936873, 108203783, 218258493, 451279169, 920030059, 752812027},
  {1, 1, 1, 11, 7, 27, 59, 117, 379, 353, 943, 2513, 3869, 4567, 12989, 13139, 50553, 16809, 294361, 753155, 279055, 1808915, 1691235, 14113357, 31049635, 19960505, 17854983, 72699913, 46440981, 564621999, 1185200693},
  {1, 1, 1, 3, 13, 43, 11, 15, 149, 237, 1555, 71, 2357, 15773, 21419, 40571, 4137, 177305, 124569, 257547, 525829, 2249251, 2794003, 5414503, 27197389, 48868021, 126569579, 28778223, 106016861, 1070591749, 1950707603},
  {1, 3, 1, 9, 19, 23, 59, 215, 15, 921, 1729, 249, 3785, 7171, 1233, 3449, 63577, 118987, 340361, 240129, 1731083, 3385887, 48755, 13736527, 13474487, 55175233, 91352697, 233843329, 335886337, 677454859, 744949017},
  {1, 1, 1, 7, 7, 37, 63, 205, 75, 599, 951, 2513, 3347, 2497, 8019, 5433, 81865, 56025, 319753, 111119, 261647, 3596845, 6639111, 9446133, 11059043, 20772271, 115741663, 119864713, 140813739, 878818937, 308590555},
  {1, 3, 3, 15, 27, 17, 25, 201, 23, 699, 1525, 465, 1115, 12299, 14747, 40363, 16153, 59035, 15771, 434183, 630787, 1466377, 4972641, 7151633, 32481439, 45736563, 10609597, 94822761, 510194051, 881246115, 1552865043},
  {1, 1, 1, 3, 29, 59, 115, 233, 107, 815, 291, 3821, 7325, 7381, 21445, 33917, 68713, 41833, 182953, 958475, 778261, 598067, 6627435, 892929, 5464499, 12771511, 67626603, 15723957, 72410597, 184841677, 1807828141},
  {1, 3, 1, 11, 11, 33, 107, 171, 421, 893, 587, 3373, 4101, 3885, 25383, 12035, 121257, 240763, 397145, 1018883, 816147, 203817, 7007283, 5549299, 12158125, 28427301, 2288403, 260961285, 36292589, 937478517, 457846351},
  {1, 3, 3, 7, 5, 23, 43, 51, 357, 77, 1327, 2995, 1321, 1571, 26419, 23603, 129241, 7003, 437723, 531471, 840733, 635919, 4930579, 1295387, 31446493, 18306325, 26540215, 266970267, 467645249, 418958171, 919221931},
  {1, 3, 7, 9, 27, 57, 101, 51, 215, 215, 469, 303, 723, 2903, 30569, 42631, 100601, 128907, 247295, 773377, 2065667, 289025, 438061, 3374571, 30785055, 51905279, 61342029, 233308311, 23596395, 678955775, 898132241},
  {1, 3, 3, 13, 1, 7, 63, 205, 143, 321, 1439, 253, 2667, 1271, 11761, 55631, 91001, 169419, 377787, 1003269, 423193, 2137375, 346967, 9001413, 5131191, 33021369, 4031223, 103580549, 181180259, 857446671, 2077925145},
  {1, 1, 7, 9, 3, 7, 7, 15, 503, 875, 1619, 1715, 5047, 5665, 5503, 17745, 73417, 65593, 6223, 962817, 1832203, 1174335, 1771855, 16179991, 5025487, 45237331, 67381035, 158329355, 124397039, 35201465, 1606878695},
  {1, 1, 7, 15, 19, 49, 65, 31, 245, 371, 377, 2963, 6185, 5519, 10743, 33231, 15161, 81017, 311615, 670983, 1121051, 3195145, 1310009, 904583, 21547645, 15934587, 35977345, 116932923, 432491185, 1054559559, 1183490671},
  {1, 1, 7, 3, 25, 27, 115, 51, 299, 451, 285, 1709, 6153, 14881, 17861, 22071, 88393, 164457, 248575, 191243, 1758993, 128803, 734059, 7514619, 18637555, 7861595, 54237573, 29421301, 130530577, 676646857, 386869677},
  {1, 3, 1, 5, 21, 21, 127, 185, 325, 995, 213, 3279, 4769, 15943, 2589, 29567, 32889, 162347, 454841, 853773, 432397, 806685, 7079255, 55057, 29335789, 23257883, 41919517, 233006055, 124500153, 508677615, 1736717157},
  {1, 3, 7, 5, 21, 9, 63, 59, 159, 743, 663, 2965, 97, 8993, 25633, 29033, 74185, 84299, 339471, 481549, 240909, 2876721, 1389847, 16473491, 23395799, 9690143, 84809807, 61709933, 190289497, 829923993, 511937417},
  {1, 3, 7, 13, 3, 35, 59, 101, 21, 659, 1531, 3995, 795, 2143, 21749, 52715, 79289, 31979, 9247, 948101, 963995, 4186267, 5564371, 3869437, 32126605, 44077771, 129939027, 200413363, 231009283, 136031239, 1207356589},
  {1, 3, 3, 15, 27, 29, 95, 1, 501, 425, 417, 2351, 7877, 4127, 3633, 23347, 19513, 260811, 118619, 87175, 509315, 1605253, 811943, 2044505, 20731549, 36872401, 122806313, 75771143, 121135885, 34415767, 1701256649},
  {1, 3, 5, 7, 7, 49, 55, 19, 329, 467, 425, 1609, 6987, 16123, 26879, 42883, 31769, 145723, 447069, 996239, 365471, 837017, 3773327, 15795883, 10059585, 53029099, 113544369, 222240129, 78654291, 716856883, 1971779895},
  {1, 1, 1, 15, 17, 21, 13, 13, 85, 7, 677, 3739, 5491, 6299, 29957, 55765, 95769, 33897, 231401, 964487, 239513, 1667229, 194805, 10314245, 13927037, 22538991, 103640909, 18550451, 116591051, 52109619, 109401437},
  {1, 1, 1, 7, 31, 21, 1, 5, 193, 659, 979, 3409, 3151, 6615, 7445, 8151, 100345, 135529, 479737, 94607, 489879, 3835037, 6854073, 13313661, 24726249, 62987291, 52569723, 37642969, 270055511, 580240847, 1608401693},
  {1, 3, 1, 1, 11, 61, 27, 205, 263, 805, 955, 3469, 1233, 1609, 15329, 13353, 39433, 130507, 96137, 521865, 1009299, 2218677, 2055827, 6200853, 16248687, 30821757, 102862675, 18122293, 361799033, 659328017, 931720969},
  {1, 3, 3, 9, 3, 29, 59, 75, 149, 557, 663, 3887, 3369, 3397, 10611, 9511, 118633, 25051, 405243, 397441, 1846939, 3560837, 5621747, 5099731, 29372925, 12923509, 61756495, 140892679, 265822913, 88396925, 1115776395},
  {1, 1, 7, 13, 29, 21, 101, 139, 99, 411, 569, 2343, 6901, 1685, 20599, 49543, 109929, 25849, 504063, 467589, 184981, 1520557, 2964365, 14408931, 28611659, 34997555, 116669665, 148064447, 413124781, 774891997, 491520207},
  {1, 3, 3, 15, 11, 3, 87, 89, 5, 293, 291, 1405, 1489, 9877, 32505, 32263, 23529, 11275, 485883, 425863, 202131, 2914203, 2215087, 5148033, 11288221, 4530973, 2859371, 185870805, 521926777, 171585805, 2101495953},
  {1, 1, 5, 5, 19, 45, 89, 5, 381, 253, 1339, 707, 4645, 14177, 29441, 8965, 3657, 143433, 316397, 254861, 479131, 268165, 8193521, 9254685, 29234069, 18795445, 90663059, 38799275, 365957709, 30413113, 565202073},
  {1, 3, 7, 15, 27, 45, 25, 177, 81, 229, 1339, 2143, 6547, 6841, 23449, 14813, 35977, 182155, 314031, 194695, 575875, 1530773, 4443617, 7422953, 30052281, 10032813, 99156531, 112265815, 303370555, 689379553, 1352888033},
  {1, 1, 1, 3, 27, 23, 81, 157, 53, 513, 1435, 277, 2353, 3545, 21461, 51479, 16777, 224105, 371065, 252811, 1480595, 2896287, 658633, 8313285, 32240397, 7284489, 4755443, 116379709, 55997369, 852728961, 188657405},
  {1, 3, 1, 3, 3, 17, 75, 139, 283, 881, 1157, 2081, 937, 14549, 10215, 13053, 103017, 226043, 477737, 301451, 307099, 842393, 4662739, 9897235, 7893267, 64467625, 101650525, 258994809, 314532769, 339179645, 712136815},
  {1, 1, 7, 9, 25, 27, 27, 133, 21, 559, 225, 613, 6931, 11785, 23413, 35757, 88201, 241513, 162863, 129, 902033, 3790755, 1703891, 12754381, 26532173, 8542327, 37625673, 65617741, 66152171, 284948353, 2083703261},
  {1, 1, 3, 13, 19, 9, 65, 49, 453, 779, 621, 1151, 1807, 13269, 6515, 17113, 123753, 190425, 345483, 237701, 888219, 2053265, 1447849, 6849833, 8748045, 12907907, 90067557, 252099287, 310473943, 309775677, 302645003},
  {1, 1, 1, 13, 21, 49, 39, 79, 119, 401, 903, 493, 3779, 7389, 29425, 28091, 119465, 242761, 84633, 556165, 1432989, 2012601, 7757007, 8142695, 32685183, 51041577, 50449023, 16775637, 150983627, 526470757, 1782092057},
  {1, 3, 1, 3, 23, 57, 59, 213, 463, 839, 1201, 1951, 5197, 13035, 29657, 35517, 79033, 239467, 153737, 44779, 1309935, 3507025, 1747459, 4187789, 15578663, 28122815, 126741241, 164480711, 3358741, 763130307, 341778945},
  {1, 3, 7, 7, 3, 49, 29, 181, 367, 101, 1277, 3329, 3563, 10373, 29757, 62555, 102409, 247947, 3711, 300015, 1063739, 3634921, 7807493, 9896589, 26623943, 34216749, 15460789, 7774105, 258111779, 257040845, 1982202197},
  {1, 3, 1, 7, 31, 31, 117, 213, 373, 57, 1095, 2733, 3431, 3915, 7665, 44459, 62457, 216939, 108601, 343535, 1526311, 467703, 1148525, 10916557, 19941165, 15852465, 116023503, 23538277, 232678735, 860891283, 1722830073},
  {1, 1, 7, 5, 9, 25, 47, 117, 355, 495, 1367, 2579, 5617, 787, 27655, 18885, 40697, 213529, 22607, 451373, 1317985, 3144001, 2430567, 13074077, 25352907, 2589431, 84772255, 146574507, 369989097, 328698507, 56552383},
  {1, 1, 1, 3, 9, 39, 113, 87, 107, 477, 891, 2273, 4239, 7521, 147, 1737, 65833, 93049, 489577, 38187, 1525985, 737359, 2010121, 15234175, 31006067, 64378805, 68957027, 51877977, 429628615, 869564953, 1571017467},
  {1, 1, 1, 3, 13, 61, 81, 225, 113, 441, 889, 1915, 3897, 15179, 4053, 5925, 62265, 212873, 468553, 669035, 84709, 2198357, 3237929, 9656553, 16034233, 8338001, 95361169, 177420691, 363688593, 657452771, 777582445},
  {1, 1, 5, 3, 15, 47, 59, 187, 475, 197, 1381, 33, 4605, 1487, 14359, 33769, 30809, 19737, 190637, 46635, 1940455, 1935463, 5955651, 13721635, 11460995, 46817341, 46474909, 6333017, 480538229, 213637959, 437041023},
  {1, 3, 7, 15, 23, 45, 53, 215, 129, 465, 795, 53, 7077, 9517, 2663, 55397, 52937, 38555, 286095, 339879, 2063727, 3782773, 5597549, 6086479, 30006985, 49962073, 103307459, 224861405, 285958029, 134672149, 362605231},
  {1, 1, 7, 13, 25, 49, 105, 255, 245, 153, 1093, 2123, 2823, 5125, 17483, 49003, 27721, 191417, 298751, 815141, 133041, 1821673, 4509025, 13474199, 7340573, 3445937, 52058845, 207438211, 4119919, 976148557, 1902538611},
  {1, 1, 1, 13, 31, 5, 7, 243, 255, 231, 1663, 1007, 7573, 405, 29183, 11367, 70217, 169561, 217753, 286373, 579383, 3175661, 595215, 9066859, 22103607, 11340671, 47148423, 143315639, 216119309, 607605325, 477288615},
  {1, 1, 5, 13, 15, 15, 115, 91, 63, 1013, 1713, 1945, 6397, 14213, 24417, 40807, 64521, 81321, 165533, 688485, 784551, 3904711, 3402619, 15214019, 10373031, 28522957, 30099785, 109106369, 509279733, 475994669, 2114676681},
  {1, 1, 5, 3, 19, 49, 91, 25, 43, 601, 25, 2405, 1973, 629, 497, 12625, 42505, 95785, 126765, 448875, 1270907, 665465, 4887843, 10024289, 21577091, 39296801, 36420401, 212070429, 435912605, 370697373, 1752052793},
  {1, 1, 3, 5, 13, 45, 11, 81, 251, 545, 1155, 1409, 7187, 847, 2835, 32909, 86873, 44537, 134283, 204717, 348005, 3343957, 5238595, 3940633, 31630579, 25083801, 3956715, 84742521, 50552059, 755686775, 821082875},
  {1, 3, 1, 13, 27, 57, 1, 103, 465, 809, 1727, 3721, 3347, 3791, 17247, 8377, 125017, 230299, 49449, 118309, 178915, 498513, 5985097, 3786079, 13821177, 52116481, 116164967, 183207777, 428974075, 738565591, 628532087},
  {1, 3, 3, 15, 25, 31, 91, 91, 119, 205, 1431, 703, 5327, 7323, 30415, 61955, 129593, 132251, 493179, 717095, 830881, 522727, 5688067, 15619891, 16194735, 49819125, 49956335, 67849895, 374102535, 674581251, 1070962775},
  {1, 3, 5, 11, 19, 39, 79, 243, 109, 463, 1869, 2133, 4139, 10461, 14793, 24025, 80713, 152107, 447085, 282723, 1564523, 288111, 7199031, 6277387, 9910389, 12204055, 59031029, 63518749, 2697459, 387398869, 1788313281},
  {1, 3, 5, 7, 23, 41, 5, 7, 249, 3, 1743, 489, 4921, 397, 30955, 22207, 123465, 7547, 473437, 82991, 1588015, 862433, 6892317, 10420191, 9701649, 44873739, 5741335, 36168641, 332103041, 1032226837, 617780355},
  {1, 3, 5, 15, 3, 7, 115, 19, 217, 231, 1183, 3723, 5055, 12967, 7855, 5067, 32601, 226203, 153533, 851495, 825531, 2042831, 7485227, 11301739, 20128065, 49849311, 102801703, 235776227, 296167719, 899706559, 224301911},
  {1, 3, 3, 3, 11, 31, 113, 41, 429, 797, 557, 1199, 1819, 1933, 9917, 32229, 7945, 34827, 114955, 920171, 496243, 1744871, 6330569, 14528081, 4955765, 1386165, 46092741, 137551783, 480711443, 976829317, 2093563301},
  {1, 1, 5, 3, 13, 63, 31, 183, 465, 915, 723, 3227, 4125, 2813, 26313, 34287, 38153, 1129, 288093, 923627, 1218597, 1136503, 3898855, 4155071, 23067529, 52211339, 58324043, 3544755, 388079653, 385409541, 1802315185},
  {1, 1, 7, 5, 31, 55, 117, 243, 37, 885, 85, 1067, 3895, 15655, 28527, 32109, 106201, 225049, 102927, 891053, 835319, 656623, 7248061, 5296683, 23852669, 32905213, 4375149, 265864163, 218815039, 526538607, 1927742039},
  {1, 3, 7, 15, 17, 43, 43, 173, 119, 187, 1161, 599, 4595, 1681, 11981, 681, 68585, 17547, 251119, 980839, 1225193, 3613939, 5158643, 13086341, 17716879, 1533315, 2843969, 268330191, 61406411, 766504185, 1932907093},
  {1, 1, 7, 7, 29, 47, 25, 93, 411, 103, 783, 1029, 1927, 3569, 27647, 8281, 45225, 99481, 100975, 183599, 2032501, 2491895, 6664897, 3280533, 33039171, 51929935, 1811143, 77684925, 319598815, 874751785, 1252585847},
  {1, 3, 3, 9, 19, 57, 31, 183, 141, 889, 157, 2267, 5701, 6273, 25739, 34039, 52713, 129147, 157723, 501409, 1469355, 62017, 5047287, 3734543, 28103973, 21329441, 83250757, 98073747, 224887853, 566403977, 1275494835},
  {1, 3, 5, 1, 29, 35, 105, 165, 505, 299, 1149, 2397, 2013, 11591, 15917, 4791, 83481, 151115, 273885, 944617, 763173, 272747, 6716353, 2780333, 33014785, 40767235, 63919861, 86700533, 75306469, 325523535, 1817008485},
  {1, 3, 3, 9, 7, 35, 47, 77, 69, 335, 585, 1131, 531, 8597, 307, 55985, 122953, 94587, 391339, 762785, 296319, 4079579, 4369607, 12109909, 30849341, 12872471, 72145153, 113144035, 394900363, 894678973, 1216259147},
  {1, 3, 7, 1, 29, 9, 25, 155, 149, 845, 567, 3735, 3501, 9365, 12025, 19131, 62153, 176107, 193279, 415209, 1876133, 1098833, 7225777, 3737811, 11061437, 31583205, 35977231, 107474399, 217759333, 54154701, 346551073},
  {1, 3, 5, 3, 11, 31, 25, 9, 411, 519, 1611, 1441, 1487, 10049, 14373, 24605, 30377, 145995, 365389, 251755, 106099, 3505367, 539041, 2470833, 26584771, 37686511, 22665635, 202896665, 128101399, 546182585, 1121064077},
  {1, 3, 3, 5, 3, 7, 101, 107, 339, 155, 1843, 2529, 443, 8177, 28655, 8151, 42281, 41627, 187163, 929837, 1422779, 3974015, 5152493, 5379411, 16429963, 50916883, 60045259, 17117337, 156059587, 556810569, 913294791},
  {1, 1, 7, 5, 29, 37, 73, 131, 125, 451, 947, 797, 5053, 10155, 30801, 27235, 46713, 226217, 360447, 312941, 1058229, 3635069, 7889537, 880971, 6999989, 63064555, 116972491, 19238165, 181572101, 242849491, 1438035225},
  {1, 1, 7, 13, 19, 47, 101, 45, 495, 457, 1293, 1971, 5495, 12737, 17687, 26123, 102377, 148953, 473343, 751909, 1496123, 1123447, 1529069, 10284437, 24889207, 49570625, 99985093, 260400427, 220068191, 143832201, 909817839},
  {1, 1, 7, 7, 11, 11, 75, 177, 251, 553, 1883, 3379, 1429, 12227, 10301, 16467, 54265, 129993, 19583, 306671, 1692835, 3630419, 2662035, 8270281, 824835, 13131857, 118433779, 193031627, 262353085, 805437115, 935419845},
  {1, 3, 3, 9, 3, 61, 95, 35, 97, 551, 233, 2045, 4873, 9109, 10019, 64523, 51081, 96699, 367483, 893857, 1574203, 3238981, 4806327, 14892827, 1205481, 17724543, 107232529, 267906261, 252770321, 945794749, 1438887659},
  {1, 3, 1, 5, 11, 3, 15, 177, 301, 573, 2029, 191, 5551, 12083, 27287, 57235, 111609, 230619, 366201, 236461, 194611, 640619, 1053319, 7369609, 3037525, 21494245, 107141957, 166625399, 500466471, 737318203, 1681693711},
  {1, 3, 5, 1, 21, 9, 121, 169, 347, 187, 57, 3163, 5609, 1921, 17581, 28351, 129001, 237003, 179037, 863081, 938541, 73281, 575185, 14811105, 25700595, 58427411, 21535889, 76261475, 486929105, 885209833, 597631573},
  {1, 3, 3, 1, 31, 51, 15, 45, 429, 245, 573, 1595, 5343, 7519, 17009, 1299, 78553, 202411, 287931, 524073, 1227623, 68107, 6171239, 4077237, 20761109, 41067885, 115711797, 25020595, 448413911, 892001047, 576208649},
  {1, 1, 7, 3, 13, 47, 109, 65, 389, 867, 963, 145, 1089, 9749, 19625, 43121, 105033, 28969, 469359, 842283, 1328421, 2168375, 1523285, 9156297, 767645, 50948011, 61735467, 126749465, 345640313, 867835565, 1564626497},
  {1, 3, 1, 7, 21, 61, 77, 67, 143, 579, 625, 2007, 6343, 4259, 21233, 237, 128953, 104107, 425673, 239407, 94381, 574997, 3554837, 3742411, 19825543, 14891147, 49150409, 236621839, 276382911, 896680139, 1064121897},
  {1, 3, 5, 9, 27, 15, 107, 91, 399, 895, 645, 2301, 439, 6789, 18299, 47285, 5833, 233083, 283853, 70753, 196515, 3307015, 3502659, 5201443, 26440407, 59595079, 18821053, 95013861, 261970735, 596107597, 1166932339},
  {1, 3, 3, 5, 13, 11, 43, 199, 505, 409, 25, 2057, 479, 6031, 9561, 51613, 20249, 153451, 399051, 1023021, 225973, 2154035, 4849251, 14373583, 10541825, 38998881, 91517249, 119731361, 280113143, 270901863, 254906417},
  {1, 1, 7, 13, 15, 55, 105, 53, 171, 925, 1849, 2881, 6951, 13069, 865, 41019, 66057, 86953, 118207, 537381, 814695, 265775, 1956385, 10257069, 13448883, 43078453, 42963569, 254885177, 459344751, 380047781, 298267849},
  {1, 3, 1, 9, 17, 31, 45, 23, 411, 185, 189, 2123, 2583, 12713, 12681, 2231, 96105, 104555, 356697, 994337, 113641, 2203191, 5300741, 901823, 26787395, 53010929, 103491045, 246358835, 276102975, 653198625, 1795373809},
  {1, 3, 7, 9, 3, 63, 11, 253, 177, 127, 545, 3293, 4449, 15995, 10223, 33529, 74809, 202779, 195823, 100065, 1943611, 2060839, 2336291, 7523845, 19178089, 4817479, 361513, 264472117, 78815033, 42647379, 800174311},
  {1, 1, 5, 11, 13, 7, 53, 161, 421, 551, 697, 627, 3879, 1639, 24419, 3337, 71961, 110905, 301933, 874339, 876069, 2026511, 6290509, 8600681, 21720317, 57493039, 1704785, 246287867, 348266047, 155323151, 1032993819},
  {1, 1, 7, 7, 27, 7, 37, 205, 429, 407, 1133, 3563, 2921, 6173, 11173, 3009, 53449, 36041, 456175, 826287, 1158451, 3723295, 7352381, 463093, 16249205, 31519711, 64352997, 109884451, 302558833, 717340373, 1208722973},
  {1, 3, 3, 15, 31, 39, 117, 81, 337, 729, 567, 2299, 1481, 3189, 1795, 40299, 3849, 115195, 162235, 677735, 1527911, 2678815, 778349, 5513417, 20586889, 55075473, 41247839, 30256339, 62601505, 23484397, 440187899},
  {1, 3, 5, 15, 15, 47, 91, 127, 275, 55, 951, 3423, 2879, 6115, 1549, 2287, 33593, 102987, 360157, 662183, 2066167, 2392103, 7503939, 6848679, 27165067, 33719055, 115246511, 195464871, 85493095, 1043572795, 2060773909},
  {1, 3, 3, 11, 17, 3, 127, 207, 141, 889, 185, 1095, 4567, 1371, 30545, 54445, 51577, 123051, 264907, 976931, 281513, 758075, 8106823, 13896487, 31370997, 9025441, 131524513, 60796623, 200947967, 903724083, 1932650313},
  {1, 1, 7, 3, 25, 11, 11, 139, 43, 1, 1977, 397, 5775, 6913, 13249, 46767, 233, 425, 222111, 144939, 738289, 2137875, 7553843, 6794659, 25048851, 10877049, 19353473, 235115701, 88613095, 206656169, 1453566601},
  {1, 1, 7, 7, 27, 13, 31, 251, 191, 1015, 161, 3719, 5321, 13013, 25187, 51881, 1289, 133369, 96431, 12911, 311091, 511765, 3436807, 2729923, 15430455, 56587119, 36803545, 170783135, 261665041, 171420477, 1147321787},
  {1, 1, 1, 1, 3, 3, 13, 19, 423, 349, 1955, 2103, 6395, 3315, 23809, 25925, 2233, 59817, 37769, 960553, 1519787, 2285355, 2342181, 6150443, 25308319, 32779349, 94034523, 124336559, 510586739, 195590667, 1581600281},
  {1, 3, 5, 13, 3, 59, 41, 167, 423, 93, 1299, 2623, 5829, 8537, 8701, 43757, 21737, 165771, 100365, 816037, 1046971, 1505587, 4070689, 16141599, 25118943, 26176117, 92269131, 53921575, 443669453, 302247457, 1736112805},
  {1, 3, 5, 11, 9, 19, 127, 119, 329, 819, 7, 3809, 5305, 3643, 27369, 61827, 28473, 195883, 432189, 966947, 695409, 701211, 1970503, 16178079, 20246961, 59921131, 17970399, 223629385, 199441089, 231531555, 1748137089},
  {1, 3, 1, 15, 25, 43, 55, 159, 205, 911, 983, 2825, 3751, 7845, 12023, 18431, 99913, 60059, 157209, 439911, 1696865, 2858371, 5287343, 1753847, 33065973, 5592983, 111767503, 233410817, 44099263, 206836797, 1735160543},
  {1, 3, 3, 13, 11, 1, 65, 133, 479, 181, 679, 981, 3317, 6241, 11693, 9619, 69353, 164107, 52091, 504869, 441843, 3645625, 8087497, 3011133, 27471863, 50444381, 27627055, 91274893, 272173949, 377825337, 1403600165},
  {1, 3, 3, 3, 21, 25, 117, 183, 127, 73, 703, 1469, 257, 11229, 10167, 50847, 22025, 250203, 101787, 669035, 2082413, 1983905, 8317325, 6342399, 5093207, 6867649, 107457447, 189586533, 365705001, 131918149, 901473023},
  {1, 1, 5, 13, 5, 5, 113, 15, 231, 269, 989, 465, 3267, 15239, 29503, 42855, 111577, 83801, 454557, 863589, 527981, 796045, 2402745, 13387271, 20600815, 40151717, 35778437, 235116617, 449537483, 126915343, 52742551},
  {1, 3, 3, 15, 9, 63, 79, 27, 21, 709, 1969, 3761, 1015, 13619, 4205, 40591, 107897, 159051, 166747, 433447, 2000241, 1801351, 1121751, 14142131, 6188941, 14008861, 99107081, 176833913, 357160767, 70942619, 1152421573},
  {1, 1, 7, 11, 29, 3, 5, 45, 107, 131, 1287, 3551, 849, 2003, 27451, 47103, 2873, 91577, 234591, 958691, 1513333, 3946907, 1942525, 5388453, 25938323, 22063243, 12028303, 100300967, 322509673, 270206987, 66267939},
  {1, 3, 5, 11, 3, 47, 59, 53, 369, 249, 915, 2475, 7539, 763, 7063, 63971, 5369, 176027, 186173, 895075, 375547, 2750375, 6360451, 14426125, 16883945, 30712769, 80978523, 108799171, 535407515, 14142915, 1818063727},
  {1, 1, 5, 1, 7, 53, 45, 127, 321, 341, 635, 2277, 1383, 10951, 29055, 45087, 68825, 148953, 103773, 553321, 1224559, 146365, 5825413, 10084455, 26484809, 21455453, 71067555, 167142605, 3046063, 719759295, 2018411383},
  {1, 3, 7, 3, 5, 1, 119, 79, 487, 93, 25, 491, 4085, 6403, 27779, 8753, 115385, 162299, 149551, 376043, 157117, 2547865, 6571407, 4014855, 507663, 40961477, 35956417, 126755763, 55802749, 373750187, 1395494651},
  {1, 1, 1, 3, 9, 59, 49, 141, 323, 703, 1175, 423, 4323, 10083, 4289, 28931, 94905, 111401, 169017, 327595, 133089, 2926995, 5339785, 5710181, 11214715, 37583255, 113790239, 247181343, 301997627, 234881595, 1107313561},
  {1, 3, 3, 15, 31, 31, 73, 15, 187, 653, 91, 1707, 1431, 9861, 19071, 8137, 114185, 83803, 481707, 662119, 1455527, 1562791, 6221265, 6473495, 23997219, 5888933, 62204227, 227861715, 392046239, 403967677, 329532935},
  {1, 1, 1, 5, 27, 63, 93, 1, 329, 353, 863, 473, 7681, 10653, 15819, 8495, 71049, 47801, 25881, 206829, 1177971, 1242775, 2054869, 11374969, 20406161, 55972713, 64386359, 198130353, 296405929, 1055124933, 102395043},
  {1, 1, 1, 5, 25, 57, 119, 167, 219, 319, 231, 1065, 6217, 5131, 1513, 49281, 73273, 110249, 189097, 365997, 2079153, 976785, 2114047, 13202895, 15225907, 18502695, 32101759, 176279953, 340710609, 812109523, 451908545},
  {1, 3, 7, 3, 17, 3, 113, 91, 201, 179, 1907, 3423, 821, 12927, 24827, 49403, 104025, 225851, 166911, 191979, 1759465, 1511067, 6267529, 11705267, 13219889, 52992411, 89135947, 267290295, 87786829, 753505095, 1134680291},
  {1, 1, 5, 7, 19, 63, 75, 151, 387, 489, 777, 2049, 1151, 1351, 25687, 4143, 105929, 181961, 13325, 384191, 629451, 793799, 8222051, 12438847, 17529611, 18077313, 40504513, 166376361, 507302087, 493280095, 1636246799},
  {1, 3, 5, 7, 9, 37, 9, 3, 87, 385, 1667, 2139, 7527, 16133, 30023, 28783, 50777, 21099, 3853, 609343, 1785537, 1050365, 1378593, 15650299, 33338287, 47472473, 5423947, 236077715, 340706655, 907475149, 1127837039},
  {1, 1, 5, 9, 11, 55, 95, 73, 413, 867, 589, 2901, 3021, 413, 5955, 38921, 1081, 118729, 473213, 857425, 164307, 2552655, 5677319, 7821249, 21852885, 42537899, 107016245, 223632877, 479254469, 650266245, 1204694907},
  {1, 3, 5, 15, 1, 17, 17, 7, 485, 323, 519, 2325, 2255, 4211, 20661, 28931, 65897, 215195, 66893, 267639, 1669193, 3947977, 72057, 2948927, 24210749, 9094171, 98589295, 39520525, 329586119, 86839483, 1678978093},
  {1, 1, 5, 13, 21, 19, 85, 189, 167, 645, 1475, 3095, 7123, 3351, 7961, 20967, 87945, 110521, 474285, 588853, 680781, 3703019, 3956013, 13357957, 12510511, 2498397, 14000443, 215445311, 116994507, 368861983, 184478545},
  {1, 1, 7, 13, 3, 47, 13, 213, 237, 291, 285, 1465, 1765, 12361, 32651, 54205, 127513, 63177, 71519, 564501, 201179, 3504199, 4791125, 847197, 29296997, 1579515, 18000037, 168329537, 278301037, 584026321, 831299027},
  {1, 3, 7, 13, 13, 27, 71, 35, 67, 117, 647, 2359, 3295, 8445, 24761, 29379, 73753, 86043, 469903, 780021, 1860421, 905171, 3021087, 14820187, 29816971, 59505597, 7613359, 20044511, 66797447, 554485845, 349859505},
  {1, 1, 1, 3, 3, 19, 23, 37, 5, 1019, 5, 1605, 2291, 14015, 9311, 39751, 38089, 249609, 259529, 523323, 398043, 3237067, 1378143, 4321101, 20242925, 44835763, 111550877, 245965821, 183775675, 293598055, 583295079},
  {1, 3, 3, 3, 31, 1, 65, 159, 221, 675, 1061, 971, 2333, 8265, 14361, 3263, 102809, 121435, 130667, 186683, 1524055, 2267369, 4686121, 2430871, 14553317, 43111291, 86265869, 128597907, 200560277, 350879121, 577790545},
  {1, 1, 3, 7, 3, 5, 81, 17, 101, 991, 753, 2883, 4977, 4409, 1757, 26803, 39897, 244345, 104779, 972543, 1236827, 1021133, 2702873, 13067833, 16534205, 41780935, 113072553, 66519611, 346887577, 366820609, 1226921109},
  {1, 1, 5, 9, 13, 25, 45, 43, 199, 967, 829, 713, 4547, 7223, 6497, 53895, 23177, 12153, 89901, 513329, 1454165, 2038241, 5642869, 10696339, 26223519, 35854207, 99360245, 126854657, 333539019, 831489391, 1049289529},
  {1, 1, 7, 5, 23, 15, 89, 179, 509, 147, 315, 133, 111, 15577, 23427, 5229, 116025, 43657, 481375, 926237, 1872079, 3525479, 1072705, 8770075, 27854645, 20329963, 74474227, 250326429, 134740839, 999803633, 533172571},
  {1, 3, 3, 7, 27, 7, 113, 65, 315, 135, 1309, 1179, 5755, 7513, 6815, 5137, 31353, 129643, 352379, 220639, 357331, 2394223, 2822713, 5889577, 26641011, 28496063, 45068485, 11132563, 347063219, 52194177, 1293311831},
  {1, 1, 3, 7, 1, 13, 29, 155, 477, 721, 71, 865, 3897, 3331, 30641, 65471, 88201, 241337, 97259, 850111, 1527385, 343493, 2163797, 14494883, 20358501, 30613289, 119243023, 118588281, 443322145, 212310251, 1539480457},
  {1, 1, 7, 3, 29, 45, 83, 3, 461, 109, 1545, 1365, 6633, 16137, 23859, 5995, 75225, 87561, 33647, 563067, 206533, 1099973, 1713275, 14066843, 18238645, 8203493, 113369185, 123774029, 246754961, 375940433, 1414721611},
  {1, 3, 1, 1, 3, 33, 77, 83, 459, 187, 879, 3731, 6939, 6707, 23409, 24245, 22825, 145851, 497593, 877337, 392779, 380761, 3782677, 6912155, 20625587, 35432579, 79662567, 110699963, 70887731, 959591995, 1356989401},
  {1, 3, 5, 5, 13, 43, 127, 41, 29, 735, 1391, 2947, 4873, 4627, 15, 41719, 90889, 225355, 139853, 222781, 1438149, 3754995, 2295879, 5216465, 15624405, 3042007, 103753879, 147118907, 88486881, 635037403, 1856918951},
  {1, 3, 1, 3, 17, 17, 51, 93, 189, 227, 449, 2809, 825, 2009, 9563, 41435, 21721, 85963, 405481, 356347, 1386329, 3229289, 6292731, 13566373, 27429861, 15961387, 55150297, 132745985, 315726001, 596686369, 240302659},
  {1, 3, 3, 11, 25, 47, 113, 173, 141, 919, 677, 117, 5293, 815, 23749, 19789, 12329, 179803, 337083, 916243, 1167313, 3947207, 3542489, 15049621, 18771493, 15565999, 44216861, 32733325, 476854277, 1008049591, 479960061},
  {1, 1, 1, 13, 15, 61, 121, 223, 53, 389, 489, 1527, 4771, 8975, 8005, 14275, 67785, 99929, 440777, 218101, 582103, 3764581, 2042561, 12541367, 21320429, 5561693, 80466625, 4831855, 92216347, 248199255, 232874925},
  {1, 1, 3, 15, 31, 57, 111, 145, 279, 991, 489, 3239, 7647, 473, 31279, 33447, 90297, 236649, 366667, 23607, 32839, 94065, 7973607, 1759577, 10884719, 58325623, 53656273, 156126671, 183753751, 1046609825, 2140547527},
  {1, 1, 7, 5, 31, 13, 75, 185, 395, 611, 609, 159, 7931, 9887, 4115, 53121, 95945, 142825, 519871, 258013, 997447, 3616485, 3074515, 9148369, 30489171, 64047403, 101686105, 61241495, 426290195, 705405799, 1349166523},
  {1, 3, 5, 5, 13, 39, 103, 237, 77, 913, 511, 1583, 6763, 14523, 4247, 63403, 51177, 136331, 100269, 754813, 35653, 1684095, 2781407, 11244949, 23850597, 8226841, 97561127, 260272855, 187189107, 456221587, 1197460191},
  {1, 1, 1, 15, 11, 5, 43, 43, 297, 827, 747, 5, 3785, 15021, 11291, 36743, 118905, 56201, 144137, 543543, 502099, 2167133, 5396099, 2664771, 17141361, 39088755, 14718787, 46764605, 78375105, 791159301, 735595587},
  {1, 1, 7, 9, 9, 53, 113, 95, 403, 53, 1335, 4033, 8147, 11963, 6523, 23675, 89193, 175593, 3823, 340849, 1042001, 672349, 1715849, 12089159, 26846443, 1991725, 85994783, 35895913, 300584075, 46830099, 278041155},
  {1, 1, 5, 9, 27, 29, 69, 79, 327, 409, 1147, 1579, 2625, 12227, 30933, 9057, 37001, 78105, 219613, 781521, 643011, 1425381, 4646301, 5285319, 23889119, 46692257, 16920851, 122928931, 158970425, 985640395, 530555261},
  {1, 1, 1, 7, 1, 33, 29, 173, 5, 517, 437, 2035, 57, 12825, 22095, 65519, 20441, 51049, 101385, 561855, 395353, 3506041, 99573, 2027285, 32954589, 31353021, 109383629, 261995211, 118316673, 818037537, 1064816711},
  {1, 1, 3, 7, 27, 29, 53, 79, 429, 707, 589, 2605, 339, 7039, 19319, 17649, 44361, 206025, 470139, 529119, 269123, 3608277, 5460989, 6699303, 917941, 19530747, 11928869, 109710965, 287970651, 368456039, 336480015},
  {1, 3, 3, 11, 9, 57, 43, 117, 39, 193, 1427, 2553, 6877, 7653, 29041, 44865, 81017, 52827, 366491, 734035, 1278401, 1259473, 3802499, 1575373, 3682623, 7975497, 74117195, 192511569, 109441509, 264035501, 144371657},
  {1, 3, 3, 7, 23, 45, 29, 151, 265, 739, 365, 3565, 6447, 9761, 24021, 679, 72569, 6459, 300011, 379711, 721375, 2426693, 6595029, 10774047, 31294033, 46003867, 37328965, 8711701, 535803239, 197976633, 170341293},
  {1, 3, 5, 1, 1, 43, 73, 55, 131, 175, 959, 659, 7315, 15145, 18301, 14865, 29945, 256507, 205277, 461913, 219081, 2758387, 2923217, 4343407, 27926283, 15522967, 26745943, 138875323, 177309723, 381786977, 1129830741},
  {1, 1, 3, 5, 15, 15, 91, 113, 359, 241, 1627, 1069, 1761, 211, 32671, 58833, 24329, 189769, 17147, 635101, 297559, 882631, 4531843, 12235417, 8691631, 65442841, 59471651, 50437701, 499978329, 715593435, 1381713927},
  {1, 3, 3, 7, 29, 27, 79, 53, 409, 81, 693, 3137, 7385, 11007, 28459, 28621, 12777, 180011, 125787, 377887, 1815381, 491763, 3085959, 5752045, 24094161, 458713, 104259373, 104963993, 431873409, 1038247047, 1198745267},
  {1, 1, 7, 5, 29, 7, 67, 195, 77, 773, 1361, 2153, 4459, 7301, 5129, 17797, 8985, 59289, 391999, 13853, 1556165, 3268591, 3222235, 2159803, 25871589, 21018541, 47679161, 95849841, 57901779, 868188957, 1265206417},
  {1, 3, 3, 7, 25, 27, 91, 223, 115, 91, 1645, 2167, 1955, 9601, 22127, 13055, 6841, 187627, 171403, 273087, 308561, 3526259, 5624723, 11253799, 4581499, 14162803, 48158949, 29041759, 215458059, 883847449, 162764871},
  {1, 3, 7, 3, 27, 53, 67, 249, 51, 151, 663, 3231, 895, 6777, 3037, 56755, 53945, 260091, 23247, 887931, 1541587, 2804861, 6272491, 10397777, 9934955, 6871711, 18689583, 87464535, 486270647, 927903857, 762953237},
  {1, 1, 5, 1, 25, 63, 71, 179, 375, 301, 1127, 2125, 783, 14481, 7293, 47883, 100345, 138873, 503821, 997369, 939457, 2132551, 2408927, 12038315, 8114399, 58683941, 27751247, 165976965, 355623767, 43660857, 290352165},
  {1, 1, 3, 9, 25, 3, 39, 69, 1, 85, 1271, 1571, 1953, 5077, 20369, 44827, 100729, 112921, 481675, 1040177, 1469185, 1936139, 69471, 1219869, 18504329, 46634589, 83397039, 113127803, 185080345, 748170301, 1689783545},
  {1, 3, 1, 13, 11, 61, 77, 59, 127, 475, 1609, 3553, 2553, 15589, 9351, 59787, 17145, 244267, 352873, 134709, 411395, 2841349, 7020853, 16669107, 7100967, 48778595, 61472833, 111021737, 373691857, 567675933, 823014431},
  {1, 3, 1, 5, 21, 7, 61, 27, 199, 653, 1243, 2481, 5369, 12903, 30229, 39453, 44137, 140443, 206329, 32093, 824605, 1942335, 5435653, 8290787, 2178223, 55831573, 10050259, 57425721, 411432417, 165115759, 1814693677},
  {1, 3, 7, 3, 13, 15, 107, 153, 501, 573, 863, 3179, 6019, 15177, 16075, 43767, 58297, 73227, 470479, 439035, 1477381, 3297543, 5272323, 1003457, 9588829, 49541781, 100701735, 167975251, 282092603, 238591329, 1218745187},
  {1, 1, 7, 1, 23, 55, 17, 35, 5, 137, 1707, 1377, 6857, 15361, 27299, 61871, 74585, 1097, 261647, 41337, 79375, 2104863, 581225, 15882891, 23104205, 62539121, 79034531, 115754361, 146458833, 48878537, 249843995},
  {1, 3, 5, 7, 27, 17, 87, 213, 95, 125, 1239, 3923, 4193, 11049, 12783, 45017, 9625, 9259, 147325, 702495, 979475, 3872265, 5025343, 14635645, 1795943, 4804053, 84352271, 704251, 362669177, 641751105, 94896135},
  {1, 1, 5, 15, 9, 55, 11, 217, 7, 249, 369, 205, 4251, 13785, 24781, 48929, 45113, 87481, 187933, 707735, 352785, 2682383, 887331, 5132897, 16251695, 27937233, 71363849, 39989093, 315007267, 995339201, 1728694197},
  {1, 3, 7, 1, 15, 35, 33, 107, 503, 563, 1591, 3487, 7495, 1767, 24791, 31281, 128393, 239627, 339519, 322969, 1990151, 814635, 3386969, 7294467, 11502111, 58454795, 113570175, 256809623, 141885871, 277166703, 1649760607},
  {1, 3, 1, 11, 3, 15, 47, 193, 289, 253, 909, 1105, 5119, 1415, 7737, 4341, 1417, 158971, 412409, 1040339, 811531, 835127, 2641511, 5649001, 18328425, 57561973, 64434549, 256455497, 268623879, 461520959, 1401104993},
  {1, 1, 1, 3, 23, 33, 53, 187, 469, 573, 835, 2049, 791, 1177, 31051, 30955, 61049, 15497, 456489, 619579, 1321999, 615481, 4701245, 16401459, 4509101, 20974981, 78119371, 95947033, 433589167, 560476497, 764957043},
  {1, 3, 3, 11, 15, 51, 77, 143, 369, 991, 1103, 1293, 6019, 6361, 26301, 20741, 103993, 137787, 153147, 1038547, 153607, 1677339, 116773, 9451591, 16951321, 54315655, 57003239, 230043253, 11282027, 79282001, 86356517},
  {1, 1, 1, 5, 17, 19, 85, 135, 113, 593, 579, 1063, 7173, 2491, 9355, 19223, 124489, 125225, 173129, 80125, 1021961, 329739, 3344493, 5789791, 26446329, 7203273, 31591211, 125122527, 510906781, 380298931, 348477827},
  {1, 1, 5, 15, 25, 51, 107, 193, 31, 1, 1693, 125, 6223, 14619, 22683, 26321, 7145, 103593, 180861, 402199, 2023553, 4182155, 4906435, 15145849, 33462167, 43372009, 53869221, 2464021, 264112951, 591019363, 1260000035},
  {1, 1, 7, 1, 17, 45, 87, 39, 87, 499, 1185, 2763, 3989, 2863, 24555, 33817, 117225, 231401, 22207, 624537, 1893513, 2801797, 2599343, 15829823, 9535407, 14128411, 9834857, 23017859, 200240381, 916316567, 1002735907},
  {1, 3, 1, 11, 31, 5, 121, 231, 185, 793, 255, 2785, 5261, 3687, 21711, 3941, 58537, 160715, 219161, 868435, 770199, 3104445, 2344881, 10187567, 23034849, 12796161, 4080343, 171362265, 355257429, 779386767, 689654967},
  {1, 3, 7, 15, 5, 59, 73, 227, 365, 937, 893, 2155, 4385, 14345, 6813, 28461, 44953, 145979, 69503, 784087, 2026893, 3354547, 2675649, 11430331, 8735845, 56422097, 69205109, 44389555, 12588601, 423637745, 163596949},
  {1, 1, 5, 7, 7, 23, 7, 239, 431, 45, 1015, 1663, 1893, 5035, 24075, 2119, 34377, 75945, 493133, 956543, 524447, 3161775, 4135151, 11948839, 26807527, 27492069, 10356351, 153224567, 290063485, 59639363, 305806851},
  {1, 3, 5, 1, 3, 15, 63, 103, 119, 801, 1681, 3463, 6083, 6453, 11379, 8205, 78681, 11147, 191005, 463065, 1058187, 182423, 2822631, 253423, 11339967, 55196585, 31853529, 250208463, 99693883, 501778221, 988925403},
  {1, 3, 7, 9, 21, 61, 9, 9, 433, 541, 603, 3905, 3787, 10187, 3643, 21319, 11433, 222891, 198095, 1021041, 1075869, 117429, 3570865, 13346737, 28903817, 45959429, 36514051, 147954137, 473884339, 474940739, 328664067},
  {1, 3, 5, 3, 11, 1, 101, 243, 363, 559, 561, 1163, 455, 4657, 1147, 39961, 85049, 108683, 465213, 1026459, 854915, 1568921, 7134125, 3019035, 28524883, 25280663, 28100473, 242959971, 450743695, 677073609, 371810227},
  {1, 3, 5, 13, 17, 37, 57, 47, 483, 571, 1579, 1137, 8125, 12271, 23279, 1615, 12521, 187211, 106573, 882677, 451993, 1179069, 7473537, 1637111, 10529371, 65783075, 45701187, 245690137, 423425237, 112869255, 1562249815},
  {1, 3, 5, 1, 21, 5, 13, 155, 75, 705, 389, 2855, 6345, 2279, 12627, 49451, 8553, 192171, 137389, 148409, 93597, 1362845, 7515093, 13320867, 19919539, 11622009, 119070477, 57001039, 71878385, 956229727, 977677243},
  {1, 1, 3, 9, 15, 51, 73, 99, 445, 227, 1705, 2175, 8111, 9381, 31555, 48491, 26905, 256009, 447019, 4017, 1199255, 3465147, 7151537, 13675019, 31699541, 33244699, 127658913, 180944423, 202792167, 35080541, 960463275},
  {1, 3, 3, 5, 9, 63, 107, 51, 461, 979, 1033, 421, 4807, 11707, 13261, 26015, 19305, 46875, 446059, 616189, 1000321, 1846423, 890355, 14873259, 22910757, 6686971, 60104609, 118843005, 200981775, 693432131, 377426421},
  {1, 1, 5, 3, 13, 53, 117, 249, 57, 851, 1391, 3245, 35, 16043, 24399, 63667, 80425, 10505, 500269, 147003, 1690005, 3027853, 6993341, 4610081, 5249473, 53197099, 39771255, 79498357, 439552939, 907752035, 1916733527},
  {1, 3, 1, 11, 23, 33, 57, 125, 385, 495, 871, 199, 4269, 2269, 22897, 23597, 95609, 169243, 191417, 603603, 363167, 629913, 2071281, 5611765, 8167545, 12968631, 90007199, 59891871, 518616901, 695660949, 1491204025},
  {1, 3, 5, 15, 29, 11, 77, 21, 479, 369, 723, 3721, 1121, 9463, 19775, 54525, 110729, 240299, 256765, 420983, 115317, 1633395, 3512645, 35693, 30002167, 23806665, 43205515, 108842817, 330744057, 534801119, 2055558823},
  {1, 3, 5, 7, 7, 45, 29, 153, 395, 223, 1917, 3713, 5087, 10827, 1383, 36823, 73609, 185451, 429, 1002975, 377135, 3447509, 4512085, 10191281, 26549811, 13886151, 12674309, 154463913, 264433911, 332990579, 1204115615},
  {1, 3, 1, 3, 31, 19, 111, 55, 275, 923, 917, 2925, 673, 6579, 18783, 5137, 55849, 136683, 316185, 551867, 335863, 3019595, 3580231, 11015967, 22045531, 45224947, 46175869, 49150117, 110869353, 355118603, 393586119},
  {1, 3, 1, 15, 25, 31, 59, 255, 31, 697, 1751, 381, 299, 295, 14037, 40953, 105513, 125371, 212601, 560247, 591985, 1344839, 7302003, 15139623, 29944791, 54205297, 49272895, 157653973, 53775155, 654947631, 1796599213},
  {1, 3, 1, 7, 15, 23, 69, 219, 351, 183, 1083, 2227, 6249, 9385, 13167, 2901, 82361, 86219, 91513, 108927, 1801063, 2636879, 213325, 2303795, 4695735, 39966415, 25322643, 200977307, 260466433, 683763233, 2136081255},
  {1, 3, 7, 1, 5, 61, 117, 13, 67, 521, 41, 2929, 3937, 1215, 25593, 32627, 55753, 179227, 192495, 348761, 714285, 1897429, 4725549, 7365909, 12395291, 39616401, 44233969, 13278041, 158297337, 430312775, 1620729857},
  {1, 3, 5, 1, 9, 47, 63, 39, 371, 657, 491, 2243, 4049, 10517, 12409, 40597, 84441, 126139, 296349, 611513, 1650401, 3614807, 7853895, 11039519, 10082043, 21732857, 37079939, 163979787, 463577161, 819468797, 899173169},
  {1, 3, 7, 15, 17, 3, 77, 13, 275, 225, 487, 2055, 1293, 15927, 31773, 18275, 53129, 127035, 225839, 689111, 2082617, 1156587, 3809637, 8677205, 4576571, 27421785, 30147199, 151259695, 68947157, 412885311, 1354003381},
  {1, 1, 5, 13, 11, 57, 113, 27, 191, 363, 1341, 3487, 8031, 13801, 7563, 40675, 15193, 240425, 465037, 193429, 1448627, 1818593, 5521193, 7565171, 17869863, 63573587, 123463189, 69738679, 247840727, 1071052433, 1697914115},
  {1, 1, 3, 3, 27, 31, 103, 143, 271, 305, 2033, 3623, 4219, 9307, 7501, 8959, 127881, 152249, 247371, 551963, 403619, 2142967, 424239, 3943239, 15910375, 10194265, 123221977, 81253007, 232187427, 970467811, 526937477},
  {1, 1, 1, 13, 1, 3, 27, 97, 475, 143, 333, 2997, 1807, 4231, 27437, 59717, 66665, 117497, 49689, 875701, 266233, 2982011, 1159427, 6273497, 15187251, 31510023, 112008917, 96448989, 525265191, 366837311, 1881031285},
  {1, 3, 7, 5, 5, 3, 69, 233, 309, 511, 1429, 1887, 2745, 4969, 17595, 5451, 63033, 168811, 217551, 361757, 1642669, 1162987, 7346749, 5467825, 16002589, 15767559, 29596077, 167710215, 93915249, 130617041, 1432058803},
  {1, 1, 7, 3, 23, 17, 115, 89, 265, 467, 257, 2027, 5331, 1195, 4451, 8621, 82953, 106697, 333631, 165371, 2082927, 1328985, 6868603, 269937, 8373073, 44649339, 108902681, 178091539, 38855483, 478396659, 791014411},
  {1, 1, 7, 13, 29, 35, 117, 155, 99, 327, 853, 3595, 2997, 10745, 21619, 26549, 58105, 229801, 375999, 759285, 726373, 1168747, 4773389, 3559939, 25777867, 60118591, 95774205, 224209091, 223178653, 907895681, 997094235},
  {1, 3, 3, 13, 1, 13, 75, 151, 67, 271, 1609, 1117, 4293, 4645, 12005, 55983, 59865, 216795, 2699, 764309, 389033, 1221573, 4054579, 16113391, 2378587, 55873255, 50550177, 247725845, 349477549, 610573117, 1170339773},
  {1, 1, 1, 13, 1, 61, 101, 63, 161, 261, 1759, 567, 665, 2339, 9157, 55603, 90985, 71225, 506537, 607189, 1537081, 332997, 2441853, 15935111, 6345401, 63011997, 52588695, 224350511, 202747905, 445018731, 378263917},
  {1, 1, 7, 11, 25, 19, 71, 27, 255, 435, 227, 4087, 4309, 14903, 14513, 30207, 39177, 23385, 386319, 361107, 1307809, 3051483, 373263, 1533635, 26661719, 45880667, 97091179, 204195743, 371343901, 129358143, 315366041},
  {1, 3, 3, 3, 11, 41, 1, 67, 383, 403, 45, 1521, 1535, 3353, 27361, 7549, 40985, 42107, 187867, 733243, 1442019, 1154401, 2207753, 9023627, 26136615, 47018667, 13645253, 120254553, 327231159, 894286433, 1019476505},
  {1, 1, 1, 1, 13, 51, 31, 137, 147, 907, 19, 3639, 3739, 877, 15005, 60357, 15673, 37977, 321369, 89913, 972021, 1141579, 6654055, 13936785, 514491, 48445859, 122091659, 262182175, 123753747, 311448421, 384780213},
  {1, 1, 3, 11, 11, 23, 29, 173, 105, 873, 1727, 2761, 2015, 7491, 17491, 41065, 21961, 233033, 334619, 224691, 1791731, 3829887, 1185813, 5572709, 18237889, 28232017, 57055783, 188824881, 282489063, 1101899, 1433765515},
  {1, 1, 5, 3, 31, 1, 119, 53, 11, 731, 393, 4031, 4421, 15715, 6431, 18089, 120233, 118633, 343005, 247099, 674791, 3074137, 8311903, 5000221, 23805139, 64775763, 126149809, 39085591, 23570509, 507231419, 1001978327},
  {1, 1, 3, 5, 15, 37, 55, 147, 307, 521, 711, 3085, 5989, 8081, 23351, 35259, 64121, 9993, 73099, 524701, 1741751, 1979597, 5491791, 10321051, 5260427, 41911617, 62885455, 268410629, 444186557, 1051645849, 2053005007},
  {1, 3, 5, 13, 21, 19, 47, 107, 447, 713, 1655, 2809, 4741, 2105, 9255, 103, 127545, 92555, 88813, 700885, 532413, 1375083, 1770167, 10240499, 19201559, 10518753, 98927999, 166825233, 473407261, 977573041, 1123196671},
  {1, 3, 1, 3, 17, 47, 63, 125, 343, 763, 1777, 607, 5625, 9517, 7221, 27257, 45209, 9435, 101049, 125307, 632569, 2343415, 1554327, 1567653, 9819039, 35541651, 133222857, 242686615, 194709425, 946547253, 1907565821},
  {1, 1, 7, 5, 31, 13, 67, 255, 41, 947, 99, 721, 7367, 11427, 1357, 12383, 71369, 140601, 97999, 132509, 1884967, 749125, 6759675, 15638903, 19786577, 27578203, 37058123, 124295113, 244855663, 389634283, 2143658837},
  {1, 1, 7, 3, 23, 27, 73, 185, 189, 545, 1877, 3169, 5419, 15873, 29059, 23983, 66073, 4073, 129519, 115675, 915375, 2744147, 7294657, 11382033, 15747893, 57042809, 45701165, 226797433, 465478163, 732453593, 1970244411},
  {1, 1, 3, 1, 5, 13, 81, 45, 79, 717, 819, 3539, 7561, 7319, 5113, 27273, 126153, 57881, 386235, 465273, 49917, 4150757, 6265737, 15206325, 3841815, 61492981, 43584043, 126127515, 470522737, 347904255, 545572273},
  {1, 3, 7, 9, 21, 25, 71, 247, 41, 583, 771, 3745, 1883, 5717, 755, 14549, 128185, 175147, 122815, 868209, 1244925, 1736177, 5471199, 8288111, 749521, 7561103, 11157291, 179173305, 4548435, 594497117, 852510699},
  {1, 1, 3, 7, 23, 25, 87, 143, 499, 515, 1753, 1229, 173, 10629, 30579, 29643, 88281, 46313, 217419, 145567, 2032431, 1205361, 7928511, 7236583, 260587, 2017931, 32401841, 226327429, 57002021, 53729373, 1981896875},
  {1, 3, 1, 13, 29, 33, 31, 69, 503, 117, 1717, 101, 7647, 1567, 28677, 3079, 19753, 170683, 280329, 448533, 685621, 81529, 1197767, 1921821, 9221199, 63700477, 129451405, 19550125, 173852647, 579158471, 407888541},
  {1, 3, 1, 15, 29, 45, 57, 81, 171, 813, 505, 3647, 3913, 5557, 2477, 42429, 30297, 35227, 225305, 128567, 2000501, 3404533, 1244145, 6736169, 298291, 25022421, 117921921, 266368919, 370163569, 514112525, 1838466341},
  {1, 1, 5, 13, 21, 13, 81, 5, 471, 221, 1563, 1741, 7269, 16327, 22687, 5291, 102473, 19625, 205261, 507029, 604717, 3898709, 2649993, 11527709, 16342671, 60488677, 19880771, 172247493, 246368157, 343167983, 553308583},
  {1, 3, 5, 3, 23, 41, 107, 61, 95, 79, 467, 1533, 739, 6791, 26911, 20309, 119081, 77419, 338493, 765819, 1615871, 1978065, 1305731, 3773013, 13059783, 31397671, 98919915, 130510453, 503463563, 436577231, 727758183},
  {1, 3, 7, 7, 3, 53, 71, 163, 459, 975, 687, 1115, 5241, 299, 26361, 38583, 42281, 134875, 197071, 366239, 520747, 1863005, 6596015, 9477739, 32983795, 63721095, 81089575, 262157843, 447630225, 452849795, 647651953},
  {1, 3, 1, 9, 3, 63, 7, 133, 421, 325, 823, 1175, 6201, 5707, 31539, 34645, 130729, 65163, 321145, 553969, 1767147, 3284711, 7583487, 12185709, 29823117, 9515053, 97957071, 17030991, 355276449, 651115971, 1494187931},
  {1, 3, 7, 5, 27, 27, 107, 239, 247, 257, 1367, 3685, 7839, 11693, 3237, 13085, 93625, 226459, 113823, 239677, 107379, 536563, 6338195, 1332423, 18232703, 41391689, 55265215, 143576333, 174706503, 742380453, 969338013},
  {1, 1, 1, 3, 27, 41, 51, 69, 271, 809, 1453, 519, 1301, 2367, 637, 5267, 46537, 116537, 316457, 592731, 767907, 2378833, 7909851, 14355533, 271703, 51330817, 5993173, 19356559, 291219533, 227034087, 1533923573},
  {1, 3, 7, 13, 19, 17, 3, 69, 369, 447, 1685, 4075, 6143, 11387, 5411, 29825, 41417, 10955, 388063, 264373, 2065019, 1326073, 6729915, 4273325, 21280041, 9534999, 23906541, 183863699, 428575223, 571453891, 1628892683},
  {1, 1, 3, 1, 25, 61, 79, 163, 1, 905, 1969, 2735, 7709, 16121, 20441, 4543, 99449, 9593, 309195, 728217, 1374241, 2157909, 3465111, 13515995, 28752921, 35644609, 92425017, 146806455, 357216101, 489412737, 1080196433},
  {1, 3, 7, 5, 15, 29, 7, 245, 343, 803, 1719, 3993, 983, 2925, 10393, 6053, 91385, 31419, 45279, 317949, 513191, 3604277, 6076223, 12614461, 11837167, 59176491, 16762831, 46747025, 383272991, 280195333, 560470593},
  {1, 3, 1, 7, 17, 55, 63, 29, 441, 387, 885, 3269, 1977, 1927, 3657, 47043, 106057, 137307, 442409, 641695, 1620089, 2427183, 1690487, 11945413, 15962865, 32164907, 88073613, 3288637, 425158801, 284170815, 663755345},
  {1, 1, 3, 1, 17, 59, 51, 9, 173, 327, 1185, 3241, 3785, 10907, 19429, 22209, 78121, 39417, 423291, 535353, 483369, 3190035, 3995435, 1075185, 2637637, 59878991, 50408889, 97068081, 171227137, 1068804931, 1650648413},
  {1, 1, 3, 13, 21, 57, 125, 5, 359, 437, 1231, 2441, 5813, 9991, 283, 52555, 36233, 20553, 352843, 710229, 1786221, 2525457, 3113221, 10789149, 78687, 43168877, 12990743, 135113281, 2139853, 53930447, 1802920963},
  {1, 3, 1, 7, 15, 19, 39, 125, 405, 381, 1757, 4075, 5565, 2065, 295, 8867, 121721, 24395, 483337, 277343, 1791591, 3217675, 5152623, 2730837, 22532125, 35984277, 81163301, 74332019, 380868517, 531150985, 2063895311},
  {1, 3, 3, 11, 7, 33, 95, 19, 253, 141, 1093, 1787, 7495, 5229, 15923, 44157, 1081, 158827, 22971, 917555, 1320111, 1162793, 7268951, 3389179, 28781285, 34931333, 8192781, 74649027, 449774239, 694497077, 214638107},
  {1, 3, 7, 15, 1, 49, 69, 163, 85, 345, 901, 2329, 8003, 9915, 2209, 33979, 116361, 84379, 64879, 158743, 460905, 2762265, 6313517, 13623995, 26760749, 40071745, 58872557, 169932577, 470371003, 646174627, 1713701145},
  {1, 1, 1, 9, 23, 51, 125, 163, 257, 681, 565, 945, 6375, 8679, 5985, 28919, 90921, 103641, 153993, 1030865, 1211887, 398859, 1340933, 13765291, 5228233, 67078801, 28916893, 120898601, 311645151, 835116287, 143594137},
  {1, 3, 5, 7, 11, 23, 123, 231, 377, 121, 1519, 2061, 2957, 14537, 17625, 37773, 95737, 199099, 347293, 953215, 1605795, 1540655, 4037235, 15694447, 12858257, 50125233, 31111, 244617429, 355714357, 958591233, 1139135329},
  {1, 3, 5, 1, 17, 43, 89, 119, 455, 279, 1857, 3405, 5225, 13035, 6055, 30861, 83753, 148203, 307261, 465593, 367801, 2152979, 6930017, 9307727, 12075951, 19753295, 94353929, 25902373, 338175041, 804961139, 1246217535},
  {1, 3, 7, 15, 31, 63, 25, 225, 3, 527, 355, 1435, 5763, 15203, 26579, 45659, 49033, 144651, 52303, 121943, 417911, 55831, 4143729, 350729, 4983787, 51219671, 63357467, 37145267, 497735915, 689492715, 1194090203},
  {1, 1, 1, 3, 27, 43, 71, 193, 135, 5, 683, 925, 7023, 7711, 2807, 56003, 91241, 220649, 505417, 540507, 503651, 3791379, 2675311, 11133449, 15026703, 58658285, 39691187, 22259029, 333785367, 947470839, 785507855},
  {1, 1, 1, 11, 3, 3, 109, 29, 109, 683, 419, 3295, 1961, 5953, 8887, 1523, 78521, 87113, 375177, 464723, 1845115, 280123, 2567685, 9867925, 4121381, 36132755, 104684027, 207752647, 218762625, 55975753, 1216239903},
  {1, 3, 3, 11, 17, 39, 19, 225, 103, 249, 81, 3875, 4515, 3969, 24745, 60031, 25881, 83195, 25739, 812371, 40185, 3322415, 7808539, 2023097, 7177007, 9795121, 23484553, 24142443, 476360251, 991426681, 1063154049},
  {1, 1, 3, 3, 3, 23, 15, 149, 305, 399, 1347, 1001, 597, 10003, 22123, 29919, 109353, 204681, 420459, 193051, 1329403, 1687615, 2473991, 5390557, 16400857, 31506663, 33997819, 215696081, 65687453, 949318619, 1048243443},
  {1, 3, 5, 1, 23, 35, 123, 7, 283, 283, 759, 3061, 2011, 7771, 32201, 40667, 10505, 135387, 222957, 1040665, 540287, 3928091, 1323075, 9048079, 782675, 56455699, 34525471, 263257149, 481119827, 310496051, 474955377},
  {1, 3, 7, 15, 23, 5, 81, 51, 357, 79, 133, 285, 425, 7743, 13499, 18983, 12937, 59723, 435247, 565655, 144383, 2054189, 8271929, 13310107, 23849437, 11380855, 79367341, 250198821, 211442657, 340884039, 232584675},
  {1, 3, 3, 5, 17, 37, 75, 221, 473, 111, 335, 683, 7353, 2283, 13457, 61171, 116073, 114395, 522411, 58781, 821625, 3738669, 2262067, 8637541, 32613697, 46952007, 15352759, 115786963, 532174321, 506522435, 1247689945},
  {1, 3, 1, 7, 13, 45, 13, 223, 149, 209, 727, 3553, 2573, 177, 855, 44139, 70585, 249675, 66121, 325343, 1387109, 1492021, 1242181, 2884839, 14687693, 64122825, 88941919, 105376025, 372537909, 651834809, 1898057487},
  {1, 1, 3, 15, 23, 5, 103, 139, 17, 29, 1961, 3021, 5745, 12963, 30669, 44171, 14185, 71097, 335403, 336407, 790767, 1448877, 4334223, 4972835, 33279785, 29129493, 10152865, 1332629, 487339369, 411696283, 1790922773},
  {1, 3, 1, 1, 3, 33, 67, 203, 29, 785, 71, 1693, 4487, 10221, 24523, 51223, 57849, 235691, 105193, 887577, 1917035, 3045049, 3803835, 13438307, 5673861, 52584185, 18736527, 203729957, 410625087, 544896485, 693356019},
  {1, 1, 5, 7, 7, 27, 17, 183, 229, 669, 1675, 3751, 3233, 10677, 7509, 52313, 47897, 71065, 110333, 867903, 1078719, 3845507, 230361, 13365215, 10884909, 2434757, 83861955, 267627103, 46936185, 738986269, 183768061},
  {1, 1, 5, 5, 25, 35, 21, 163, 483, 399, 195, 3465, 6349, 545, 18861, 31759, 8121, 248089, 267597, 780893, 488801, 2739643, 2410701, 8988955, 32295499, 66661815, 53686635, 264871297, 77052005, 983949097, 416304821},
  {1, 3, 1, 5, 15, 39, 13, 157, 71, 841, 447, 3625, 53, 12745, 2719, 27617, 100217, 188171, 219673, 527101, 326823, 1215679, 3366357, 15794965, 29794223, 19780913, 63230183, 236058849, 31436589, 845396929, 415025287},
  {1, 1, 5, 5, 3, 45, 113, 121, 263, 887, 709, 2189, 3811, 1409, 10451, 48777, 19289, 208169, 276765, 971773, 1377211, 1531061, 7642025, 172305, 13173119, 10813903, 72287293, 72551813, 319102859, 382407097, 1871012747},
  {1, 1, 5, 15, 9, 41, 31, 95, 377, 215, 437, 3633, 433, 11935, 15283, 55451, 97545, 182921, 58061, 274135, 1282801, 4175537, 3485335, 233415, 21683393, 36084767, 121377053, 56663145, 494969593, 14085095, 395472651},
  {1, 1, 7, 7, 13, 23, 79, 3, 451, 409, 1103, 1771, 553, 3175, 28703, 49357, 2441, 34025, 432831, 205887, 1271861, 3922079, 6023591, 15571323, 30035147, 18654641, 15954631, 91968675, 509220913, 518927215, 1377450519},
  {1, 3, 1, 1, 13, 33, 95, 133, 419, 851, 955, 3985, 5869, 14219, 253, 46883, 14825, 168251, 5737, 128153, 851941, 1576377, 7619751, 5775197, 33344731, 15533467, 7740355, 12206777, 360761189, 821324163, 313904165},
  {1, 3, 3, 3, 23, 55, 91, 207, 281, 355, 361, 261, 6837, 4401, 25455, 25313, 78777, 93451, 8731, 675579, 789183, 859839, 4622227, 2827687, 12387633, 42175659, 38795969, 12178141, 531747165, 611471849, 1315723575},
  {1, 3, 5, 9, 27, 9, 107, 51, 69, 555, 835, 3541, 1827, 5737, 31225, 55619, 11161, 763, 240109, 866577, 532467, 2819761, 4695955, 2527835, 31663901, 63590915, 12965835, 211285549, 134959819, 586929633, 1951497601},
  {1, 1, 1, 11, 27, 51, 79, 85, 447, 447, 9, 2803, 377, 4347, 2183, 61257, 67305, 7657, 179609, 176243, 568227, 2783115, 7178407, 1482909, 4841591, 42181399, 48162225, 224683579, 280798289, 793164579, 1047651055},
  {1, 1, 1, 3, 23, 21, 51, 217, 297, 135, 573, 689, 4947, 14143, 26247, 43061, 91433, 70249, 96633, 57275, 12335, 800685, 5916315, 13777137, 8706065, 39541295, 35998949, 119710601, 330628187, 654834183, 1662637247},
  {1, 3, 1, 7, 15, 13, 27, 151, 463, 707, 43, 3641, 4999, 3783, 9083, 22085, 12073, 122251, 192825, 729919, 1045031, 3306901, 7699155, 16718911, 26340407, 55908267, 95462147, 42607057, 286845199, 903408591, 954523619},
  {1, 3, 3, 5, 3, 1, 15, 119, 343, 605, 105, 2939, 2259, 889, 21759, 34073, 38909, 122983, 494583, 425369, 1627103, 934501, 4688755, 2459107, 26634395, 9558481, 121417221, 42846271, 424494279, 329764741, 1009370083},
  {1, 1, 1, 7, 3, 63, 103, 1, 235, 317, 263, 2701, 7331, 15921, 17295, 613, 39997, 167829, 23285, 786515, 8663, 3716979, 1504571, 10728397, 7228935, 6259721, 24576795, 211615913, 403900111, 937501333, 570363571},
  {1, 1, 7, 3, 19, 3, 5, 17, 105, 491, 755, 233, 5397, 12365, 16325, 59377, 26901, 43461, 30291, 232727, 1812319, 3005023, 7893881, 3962733, 1694605, 13619807, 19878599, 175512469, 73686025, 338779161, 335572449},
  {1, 3, 3, 15, 15, 27, 37, 151, 481, 949, 1473, 233, 1925, 15471, 24957, 3241, 34037, 92071, 504535, 972691, 306771, 2101591, 2122857, 10734795, 33113413, 11351961, 35334101, 10874253, 181595089, 242711899, 1185302329},
  {1, 1, 7, 5, 9, 61, 49, 91, 169, 179, 701, 3957, 473, 15087, 6109, 25083, 93813, 101613, 437459, 89777, 311877, 470521, 7605285, 15742143, 12963133, 56084303, 98946377, 45374281, 481552893, 618989259, 735792425},
  {1, 3, 3, 11, 27, 43, 5, 33, 69, 705, 733, 2675, 2677, 4235, 11109, 15557, 72853, 218647, 285527, 659287, 1756231, 3168887, 2367513, 15180029, 21651745, 47882805, 16082265, 102279591, 232563729, 648985879, 665247265},
  {1, 3, 1, 3, 17, 19, 101, 119, 289, 341, 1395, 563, 6859, 10359, 10077, 26905, 112757, 236903, 116437, 385695, 166981, 254319, 5238897, 258251, 4059621, 56439161, 30381031, 254072023, 400462935, 720869419, 352830209},
  {1, 1, 1, 15, 21, 47, 41, 145, 439, 583, 1755, 1977, 5235, 15961, 21315, 60577, 126477, 188805, 118581, 398107, 1340353, 352579, 6230061, 3256477, 30953475, 45691187, 87185367, 74103925, 276934191, 554111117, 1599681279},
  {1, 1, 5, 3, 9, 59, 71, 143, 27, 1007, 313, 1567, 1685, 11063, 28889, 44253, 120421, 88693, 85649, 947415, 1745229, 3784295, 6234739, 4777491, 7767767, 34985123, 119239013, 187101587, 340652033, 647741563, 1329044957},
  {1, 1, 5, 5, 29, 29, 43, 127, 13, 585, 1245, 187, 2697, 8791, 19561, 6463, 111269, 83965, 169361, 757489, 1284441, 2182873, 6902351, 15451675, 9921073, 11599141, 102143769, 79951663, 240803309, 353363435, 1600747229},
  {1, 1, 3, 11, 29, 39, 127, 75, 23, 521, 421, 3115, 139, 5429, 23341, 58035, 62901, 243149, 72151, 175679, 1364545, 1774555, 6108795, 13278759, 1180411, 40736149, 68894537, 213910399, 482347695, 398465481, 1824531601},
  {1, 1, 3, 1, 27, 35, 27, 9, 185, 653, 887, 2715, 3775, 1753, 22105, 62095, 64101, 81213, 236695, 145909, 107335, 1673463, 7400967, 14628565, 11344805, 59447001, 121817315, 81326847, 219568179, 874773541, 1461266325},
  {1, 1, 5, 5, 5, 63, 23, 31, 317, 1001, 1751, 1185, 7933, 525, 30501, 15565, 104341, 182325, 357201, 137169, 903745, 2154619, 2690579, 13852403, 11103857, 2981669, 100359187, 233968437, 362649009, 879333337, 854965473},
  {1, 1, 1, 5, 9, 27, 79, 91, 453, 995, 1041, 3213, 8027, 5855, 7435, 64079, 123693, 103485, 38837, 416753, 148445, 3717855, 7323195, 14161447, 11443753, 30431359, 88768437, 99707505, 175760015, 552973851, 301461047},
  {1, 1, 3, 11, 1, 51, 27, 195, 139, 41, 1891, 1437, 1033, 11671, 3235, 35083, 101261, 167725, 256823, 187327, 138205, 903119, 3894895, 12420863, 7165495, 64018645, 25796287, 13773769, 196422133, 386659251, 977409655},
  {1, 3, 1, 3, 11, 25, 33, 249, 373, 497, 1631, 293, 3657, 10741, 15831, 59545, 3165, 17767, 327797, 651935, 1976031, 3297637, 2021981, 14564973, 24634113, 19409221, 36732251, 246265137, 157197861, 137036337, 1905354307},
  {1, 1, 1, 1, 15, 63, 13, 165, 113, 559, 1615, 3579, 1993, 1907, 22335, 47791, 48125, 68541, 90869, 839669, 1147099, 154347, 915041, 4075017, 27832053, 36694283, 103870107, 164297431, 274710693, 1004272943, 2072975019},
  {1, 1, 3, 15, 13, 49, 63, 235, 31, 811, 1729, 221, 5143, 11547, 30029, 52003, 114869, 52237, 257111, 48059, 834769, 1493085, 1958651, 6290207, 26938739, 12731607, 10382989, 260176641, 193396899, 659242143, 289241153},
  {1, 1, 5, 13, 29, 61, 25, 221, 421, 221, 583, 373, 2341, 7493, 13981, 54141, 25845, 48757, 375121, 441177, 2037721, 3240921, 2658813, 4056377, 30026961, 52587705, 89902467, 255889169, 45908905, 1072205697, 2030613689},
  {1, 1, 5, 11, 21, 49, 71, 249, 237, 369, 1273, 1067, 4411, 409, 7219, 52933, 78925, 230293, 250033, 669663, 969297, 3754701, 853891, 6932813, 22411361, 39749829, 37920509, 253868991, 314763343, 1003713253, 1340440463},
  {1, 3, 1, 1, 13, 23, 53, 15, 137, 553, 401, 2247, 5591, 14021, 445, 20433, 44661, 136135, 448085, 1030301, 1222617, 2373859, 5243857, 3212547, 2965581, 64536741, 103072293, 120996563, 511201475, 728856841, 987766577},
  {1, 1, 7, 7, 7, 23, 19, 189, 119, 643, 847, 2123, 5343, 11839, 4575, 60461, 97533, 260405, 249971, 267603, 1034315, 2491611, 412095, 12112721, 4908555, 57773127, 127686883, 97123031, 294042835, 632065251, 1202060307},
  {1, 1, 5, 5, 11, 19, 111, 219, 185, 499, 595, 723, 3519, 10891, 27603, 29261, 121037, 166581, 283825, 359249, 1984847, 2998999, 5434835, 7361287, 22892405, 54589295, 105772295, 199299823, 310581283, 156712631, 956239583},
  {1, 3, 3, 1, 9, 13, 95, 227, 459, 133, 1535, 3481, 7187, 14797, 16511, 6737, 104829, 93383, 454519, 1011869, 1029461, 3186681, 8178571, 15805319, 20279911, 24513305, 81729763, 260266365, 500106527, 243869625, 1456673155},
  {1, 1, 7, 7, 19, 57, 117, 7, 455, 941, 455, 797, 6313, 10071, 18651, 25013, 124909, 24405, 94003, 601043, 1661279, 711925, 1334665, 1451859, 23290723, 55431521, 83088331, 236630809, 454684421, 142034355, 372069199},
  {1, 3, 7, 3, 25, 25, 79, 19, 383, 971, 1625, 2803, 2461, 633, 32317, 48407, 94069, 127711, 22099, 508927, 1712085, 1576805, 2771611, 4460807, 24540819, 44322903, 33132949, 237515567, 186193737, 81079621, 1940481689},
  {1, 1, 7, 7, 25, 43, 93, 135, 155, 685, 2001, 3007, 7315, 15555, 30401, 36291, 95877, 184973, 406675, 957619, 409813, 2830695, 6423433, 11199643, 19770687, 28619841, 5937581, 4236091, 33094823, 843133871, 1524850909},
  {1, 1, 1, 5, 13, 33, 123, 221, 341, 105, 1075, 3125, 5323, 14293, 29875, 52215, 57165, 240477, 491701, 711153, 1863513, 704357, 2513311, 2521289, 18925737, 28162253, 121364247, 178603649, 352114455, 367903569, 107165551},
  {1, 1, 3, 9, 29, 51, 25, 59, 171, 563, 1695, 2845, 6067, 10671, 2909, 33977, 119797, 72293, 6743, 190749, 522689, 3684807, 5313181, 1503399, 24596831, 13881903, 77919843, 186905121, 232204479, 297702571, 2056035105},
  {1, 3, 3, 7, 25, 21, 39, 65, 485, 949, 1773, 2775, 6019, 14587, 6715, 54793, 21869, 184743, 342327, 789019, 1651781, 2902265, 6527155, 13265445, 18894761, 51236257, 26791113, 130280403, 446278439, 84229431, 1164272935},
  {1, 1, 7, 11, 17, 57, 125, 17, 111, 167, 289, 3939, 7357, 2289, 1717, 45225, 130133, 13093, 432083, 244511, 1012957, 3237317, 6387849, 12502661, 18190283, 45687859, 104190093, 83169327, 424068873, 593746893, 1795093265},
  {1, 1, 7, 7, 21, 55, 3, 139, 471, 747, 1437, 1353, 7657, 13133, 14193, 38191, 32069, 133949, 474771, 892787, 797657, 3290747, 1581031, 4598503, 1685451, 22250487, 116885041, 184520469, 38110125, 1016217, 703907653},
  {1, 3, 5, 7, 25, 57, 55, 17, 315, 159, 437, 933, 7493, 6025, 2775, 24287, 104925, 230655, 101105, 502395, 121949, 3796309, 5429187, 5527237, 32384695, 43717131, 71628265, 89159689, 497436513, 219630893, 169197779},
  {1, 1, 7, 1, 15, 45, 67, 191, 355, 681, 1763, 1237, 105, 1425, 26089, 42879, 20325, 35005, 51155, 452277, 312579, 4023609, 1158415, 11475739, 23241119, 65348485, 10100831, 167060609, 154657493, 457486677, 1223682429},
  {1, 1, 5, 13, 13, 53, 25, 127, 103, 155, 935, 2561, 475, 4341, 30541, 43835, 102645, 69805, 485649, 316409, 2042633, 3654929, 2145661, 11709307, 4638163, 43679543, 105071275, 218875549, 418697271, 173502529, 1210296145},
  {1, 1, 5, 15, 27, 59, 99, 173, 189, 41, 105, 3287, 4071, 15005, 18301, 34487, 9141, 226077, 432913, 46907, 2019103, 3159319, 3581279, 12447665, 33095041, 41604141, 90688941, 81753507, 412045971, 96334929, 134027633},
  {1, 1, 5, 11, 21, 9, 57, 115, 495, 561, 579, 397, 3049, 2007, 5041, 37345, 75781, 174789, 125009, 108767, 386833, 252213, 3457853, 6629351, 4950875, 25268333, 75510127, 205946745, 31764245, 714945011, 1442267941},
  {1, 1, 5, 11, 15, 11, 101, 241, 69, 483, 1007, 4069, 5221, 5323, 20177, 24397, 94853, 138869, 484689, 484639, 1596171, 3267895, 1706761, 16104301, 33182241, 13070743, 45339563, 225154681, 334665353, 330412143, 1521058485},
  {1, 1, 1, 7, 29, 15, 119, 161, 21, 517, 847, 2217, 4487, 4817, 24053, 21683, 73237, 92813, 236437, 960627, 190729, 2259715, 740115, 5185005, 31696665, 31943537, 6283115, 11429805, 222734059, 878431525, 199892505},
  {1, 3, 1, 3, 3, 51, 85, 63, 345, 799, 1699, 3961, 7109, 3931, 28173, 46851, 75509, 116815, 400533, 567807, 854807, 959759, 6012745, 16434627, 24704829, 54491603, 110995863, 242235893, 191846873, 763678287, 1187581993},
  {1, 1, 5, 7, 15, 25, 97, 139, 331, 969, 1129, 2451, 3107, 12235, 12949, 29837, 66069, 249077, 229009, 161555, 1350923, 1127701, 5122829, 1355679, 26543935, 45393781, 126035029, 87090951, 68451383, 65424919, 1998474121},
  {1, 3, 7, 1, 19, 21, 51, 155, 295, 565, 29, 2107, 341, 14611, 15281, 50727, 48005, 240735, 103763, 9149, 1708831, 436513, 4392719, 15939039, 14965723, 37954569, 110049969, 148513375, 282650193, 515151279, 633831989},
  {1, 3, 1, 11, 3, 37, 13, 217, 429, 217, 301, 1217, 5655, 13845, 32465, 23559, 85509, 75359, 263125, 162359, 439575, 3513145, 6282001, 12191229, 14166697, 51819757, 123456585, 151631669, 505909619, 596352897, 914405213},
  {1, 3, 3, 9, 9, 57, 79, 231, 433, 703, 699, 3813, 7035, 5885, 19185, 52551, 103429, 103263, 169431, 620341, 716821, 2674733, 7295067, 6434963, 29233181, 46903987, 29864079, 40589929, 530037383, 955537049, 1622383789},
  {1, 1, 1, 5, 19, 17, 31, 209, 49, 515, 279, 909, 5881, 2673, 23635, 23101, 11661, 4277, 112501, 408465, 1968135, 1804309, 7084099, 7608517, 31660413, 40929367, 53598243, 7388817, 351431093, 87546997, 680447687},
  {1, 1, 5, 7, 3, 3, 119, 139, 245, 775, 1009, 1157, 1405, 9737, 17671, 62981, 128349, 13909, 4913, 296467, 1857543, 3942415, 6146091, 5818615, 7072025, 65181979, 94768981, 192219177, 74255849, 1027157101, 819419107},
  {1, 3, 7, 11, 17, 21, 105, 21, 67, 871, 233, 3607, 571, 9141, 9751, 28093, 32029, 235575, 272307, 380567, 1838109, 459785, 6038621, 14507249, 5416183, 3178083, 91507573, 9897691, 112918959, 221591961, 1586969707},
  {1, 1, 3, 13, 5, 53, 91, 181, 143, 375, 1113, 705, 837, 10505, 11459, 57753, 57421, 196389, 18295, 896473, 2050585, 373265, 5114495, 3096241, 14527411, 33002227, 6631893, 214473293, 158453969, 919302013, 618954311},
  {1, 3, 5, 11, 9, 19, 107, 229, 305, 107, 1027, 691, 4677, 8987, 7931, 951, 73773, 51999, 287985, 491447, 519693, 2318863, 5758495, 6696473, 20129293, 37941455, 91058295, 250427215, 223582041, 101682535, 651551199},
  {1, 1, 7, 9, 9, 17, 39, 195, 103, 315, 517, 123, 7167, 7039, 3571, 40469, 92045, 74645, 49907, 225437, 323077, 1775141, 3204723, 1498647, 31625835, 26886439, 96224233, 125978279, 188565403, 708500259, 943693919},
  {1, 1, 1, 5, 1, 21, 121, 53, 427, 111, 717, 1065, 2843, 5873, 28411, 42443, 80301, 103533, 17269, 945905, 1747477, 3923473, 5581421, 13086257, 1744479, 34276331, 26280433, 237307077, 50270367, 100991653, 971457607},
  {1, 1, 3, 11, 27, 7, 37, 255, 429, 363, 997, 2429, 6871, 1271, 29375, 62897, 75965, 73861, 216119, 217311, 1735175, 3157563, 6516345, 8191571, 21226121, 45676775, 122326273, 226422049, 81605011, 535607731, 1449906611},
  {1, 3, 3, 13, 23, 23, 123, 119, 213, 51, 771, 1603, 1621, 1497, 23667, 13443, 57869, 234447, 299127, 836721, 1746443, 2214419, 8032791, 11421435, 1287753, 26770079, 42727911, 40030207, 98452745, 851143141, 733541143},
  {1, 1, 3, 13, 17, 21, 81, 17, 211, 815, 1751, 3875, 4001, 3927, 6185, 28753, 39013, 56453, 398551, 646873, 1010189, 2784817, 5937701, 14679701, 8836743, 43859003, 77811339, 244213903, 30271437, 1009676075, 1022307805},
  {1, 3, 1, 5, 13, 41, 23, 187, 475, 353, 1307, 543, 5077, 3459, 20553, 29119, 36581, 45631, 373717, 314681, 541209, 3106317, 655923, 8998479, 31927127, 45704989, 95901959, 71584299, 130403337, 957961711, 200851629},
  {1, 1, 1, 7, 1, 39, 3, 247, 375, 101, 81, 1515, 1079, 15307, 18865, 63115, 120837, 241773, 442453, 979187, 540821, 2740651, 835223, 8112027, 5169387, 60645449, 80734013, 37530767, 60678275, 647310519, 1746339069},
  {1, 1, 5, 9, 23, 7, 61, 45, 379, 553, 435, 1805, 4147, 2289, 22081, 40041, 38213, 40805, 62289, 539741, 829843, 3981235, 8048561, 1769761, 30842239, 58809709, 126151183, 3310585, 412191183, 880255957, 1540862525},
  {1, 1, 7, 3, 17, 13, 107, 169, 119, 981, 1825, 3329, 7779, 12245, 28367, 6749, 99837, 77237, 164787, 562711, 213917, 3366033, 6726623, 9684269, 32429387, 15093537, 110680781, 13729605, 246396439, 1033607681, 131782355},
  {1, 3, 3, 13, 29, 13, 93, 155, 331, 507, 1073, 279, 6615, 14077, 3005, 10171, 103221, 6063, 71319, 429681, 911233, 1043849, 8190617, 13204479, 1013103, 37907911, 57890125, 255930571, 175692323, 301157009, 1147648881},
  {1, 1, 5, 7, 29, 23, 81, 181, 321, 921, 1531, 2607, 7291, 1255, 29889, 30095, 56133, 217725, 411217, 388019, 1774745, 2542491, 6765301, 14442777, 30010965, 55163325, 117746031, 149230747, 211943175, 553158731, 1946138397},
  {1, 1, 1, 5, 7, 1, 9, 231, 203, 559, 243, 3999, 3649, 7939, 14729, 34771, 68453, 89869, 136277, 467569, 62611, 3908485, 4791429, 14345523, 3301151, 39625251, 85385935, 67939795, 288419005, 206383119, 1392203581},
  {1, 3, 7, 3, 17, 29, 79, 251, 305, 641, 1125, 1155, 7139, 6721, 43, 34927, 29549, 138751, 103155, 243775, 1250205, 1410209, 2952443, 7252287, 3138461, 9786941, 87922897, 211304631, 327337191, 62115037, 306396535},
  {1, 1, 5, 13, 21, 39, 55, 103, 143, 487, 849, 1111, 1105, 8483, 5417, 10521, 94565, 43813, 276561, 738009, 1106065, 1349507, 7964083, 3492779, 10835587, 12578699, 76029853, 200832059, 375599461, 862753935, 170178557},
  {1, 1, 5, 5, 19, 5, 111, 49, 95, 917, 843, 2539, 6831, 9019, 16045, 59363, 93429, 51917, 306065, 868465, 1831575, 863617, 5086451, 5102069, 10244947, 5817985, 9661767, 7667247, 288884211, 304612679, 1963469089},
  {1, 3, 3, 11, 7, 45, 87, 51, 49, 615, 603, 1623, 5351, 11939, 21945, 40539, 49573, 210735, 403159, 17911, 867995, 341937, 7093499, 7823799, 4619165, 57908475, 73515887, 6344315, 244733211, 915338511, 529623829},
  {1, 1, 5, 9, 9, 33, 113, 37, 163, 853, 1313, 4021, 965, 11465, 8573, 24425, 69941, 141925, 135441, 870237, 1894797, 2842517, 5374341, 15711153, 27635319, 29353041, 129668925, 50385713, 221778673, 959977677, 1679305209},
  {1, 3, 3, 9, 17, 19, 121, 95, 93, 441, 1951, 3335, 6279, 16087, 14763, 60771, 6381, 150223, 435959, 799349, 1037709, 420743, 6770573, 15806467, 10662185, 27635829, 20215291, 18151059, 456333995, 104013275, 59719111},
  {1, 3, 3, 9, 13, 15, 19, 129, 257, 641, 533, 1667, 5959, 2259, 10439, 29745, 25565, 96135, 366135, 554581, 1764241, 897435, 2798231, 12344493, 30575645, 42112501, 102885281, 206138903, 103916675, 91109199, 2001092619},
  {1, 1, 7, 7, 11, 31, 45, 247, 233, 101, 899, 2033, 7803, 11423, 30645, 7723, 80789, 164461, 364819, 201971, 421767, 3127955, 1603057, 9333371, 8345197, 61615049, 108649335, 124856981, 334296935, 778186123, 1910664169},
  {1, 3, 5, 11, 23, 3, 69, 79, 319, 125, 1463, 2047, 7311, 5819, 445, 13725, 124085, 8503, 471953, 44183, 630163, 237215, 7900873, 4495987, 28957403, 63932913, 81530523, 21859195, 3118627, 281848343, 1167380849},
  {1, 1, 1, 3, 7, 43, 83, 89, 467, 709, 1993, 3773, 7805, 305, 15701, 51101, 116885, 185013, 158869, 951063, 580755, 1113783, 776927, 1732165, 31354479, 25386897, 41105973, 114294201, 29914089, 1016677397, 709047985},
  {1, 1, 7, 5, 19, 53, 77, 237, 27, 853, 1003, 2041, 5739, 4663, 9783, 23761, 40349, 184197, 40243, 351569, 1734559, 409521, 1869041, 3003113, 23748255, 45162033, 121871319, 169254909, 376982775, 189645827, 326994347},
  {1, 1, 3, 7, 19, 31, 71, 33, 479, 693, 1503, 9, 2779, 1481, 9413, 36227, 5973, 130253, 48663, 100339, 778383, 396947, 2405819, 7546541, 8498035, 8576065, 6406995, 226706253, 339889623, 844232509, 1398304985},
  {1, 3, 1, 11, 9, 23, 1, 99, 247, 33, 1987, 1577, 8029, 7785, 29947, 38751, 108237, 9951, 321045, 447511, 730045, 3979755, 7111957, 10458479, 23045699, 12341821, 109015983, 154680445, 15436801, 35567613, 1138160479},
  {1, 1, 1, 3, 15, 57, 23, 53, 431, 553, 1433, 2447, 1871, 10701, 30961, 12281, 43557, 179717, 362805, 429623, 1221627, 3499077, 7628635, 7962497, 15958563, 44047181, 131636661, 6191067, 376224187, 982715073, 858480573},
  {1, 3, 5, 9, 11, 49, 123, 91, 87, 155, 301, 3339, 6183, 15895, 17309, 45927, 96021, 237951, 136881, 941653, 1687087, 517061, 3702567, 4141423, 11034219, 41760967, 94128161, 144388855, 441097539, 2597035, 1188083313},
  {1, 1, 1, 9, 9, 41, 79, 123, 261, 547, 1931, 2553, 7405, 14431, 24079, 20769, 117533, 99621, 320405, 133501, 1160829, 4010621, 7739675, 10237743, 18187977, 54117695, 29112079, 52058725, 209748273, 1066568939, 863107371},
  {1, 3, 1, 3, 31, 17, 17, 137, 419, 591, 1693, 3977, 861, 16025, 189, 60995, 123349, 234671, 379509, 32543, 717739, 2784461, 3500381, 10012445, 10862903, 45203803, 107791841, 142068549, 151570809, 504980333, 643798073},
  {1, 3, 7, 11, 19, 47, 107, 243, 87, 135, 507, 189, 1397, 3841, 22999, 50781, 102141, 74199, 241107, 562103, 1990399, 3359571, 4017015, 15520223, 5458283, 25328155, 55519287, 17508401, 355592265, 289948837, 998131091},
  {1, 3, 5, 5, 15, 11, 19, 239, 133, 295, 673, 2389, 4753, 4363, 21669, 25579, 92725, 60303, 21425, 681113, 2071979, 402127, 7329119, 5381107, 24118489, 26713483, 124781093, 145754777, 403706469, 556798543, 1171267153},
  {1, 3, 5, 7, 19, 43, 55, 129, 239, 89, 1731, 1381, 5483, 11773, 9201, 17745, 44389, 78375, 300081, 795835, 1273079, 3312743, 7680267, 13635901, 23897507, 36201549, 12842671, 69008177, 318614135, 287778649, 1208967565},
  {1, 3, 1, 13, 3, 15, 77, 131, 417, 845, 1953, 2871, 1789, 10343, 11363, 20699, 52589, 191183, 20309, 1018193, 267319, 3282667, 2685809, 13144975, 15152197, 34862481, 100133333, 48225619, 414166601, 1049577355, 1175377255},
  {1, 3, 7, 1, 9, 43, 55, 223, 239, 317, 1951, 2725, 209, 3853, 2201, 6839, 80325, 60927, 31155, 745949, 327973, 2675455, 8258115, 15575619, 8968003, 18076561, 74225267, 187724161, 218584189, 963706473, 1880623181},
  {1, 3, 1, 3, 7, 35, 29, 21, 149, 779, 467, 65, 811, 4859, 14021, 38429, 95797, 50711, 476341, 987647, 387699, 1477247, 1891889, 16216649, 3408561, 26370895, 120611319, 123577437, 507293399, 912366255, 1231275385},
  {1, 3, 7, 1, 19, 31, 97, 9, 11, 415, 689, 1513, 2475, 5039, 5669, 65103, 95925, 195423, 390707, 1046749, 565823, 3912395, 1060477, 14580293, 8490847, 11499627, 120491437, 65453797, 36802751, 38842587, 419818681},
  {1, 3, 3, 3, 11, 25, 37, 247, 189, 911, 429, 2395, 3653, 79, 28115, 23513, 88365, 37063, 141399, 998015, 1128311, 3873605, 5119513, 14820867, 31273529, 31185507, 126947049, 129574135, 208038529, 106453427, 1920539319},
  {1, 1, 5, 5, 5, 27, 25, 45, 291, 455, 741, 2259, 8131, 11779, 14693, 58729, 54005, 35013, 482481, 1030833, 1933025, 3321215, 3111997, 1203441, 19432695, 46609699, 14361049, 136047871, 155709831, 721177119, 1503357977},
  {1, 3, 3, 7, 21, 33, 67, 49, 153, 491, 1811, 1955, 925, 15555, 13801, 48717, 72901, 151447, 164471, 664187, 511849, 3914861, 5016583, 13567077, 8019173, 15096927, 116825127, 208000831, 119751841, 624187295, 121916813},
  {1, 3, 1, 3, 11, 53, 105, 129, 457, 225, 497, 1123, 973, 2901, 26655, 3643, 113437, 249423, 272853, 772063, 497471, 3934185, 2079861, 8752261, 5407997, 32621589, 130668581, 220261271, 222981881, 930523561, 2023146075},
  {1, 1, 7, 13, 29, 49, 71, 37, 321, 865, 735, 357, 7629, 6011, 28221, 39041, 29909, 74789, 425651, 418233, 1099633, 299765, 5482915, 7612177, 4051565, 59644037, 131832139, 170229649, 43921473, 886875423, 71206681},
  {1, 3, 5, 3, 19, 59, 65, 199, 69, 391, 1735, 3075, 287, 16213, 3211, 22425, 461, 159239, 277521, 595711, 178679, 3184615, 5299197, 2345979, 11764689, 62162955, 102759139, 54163471, 107771507, 1051159497, 838629799},
  {1, 1, 1, 5, 15, 61, 67, 255, 5, 689, 759, 155, 7245, 5881, 21685, 3863, 37173, 128637, 52853, 451025, 167867, 1549145, 6046351, 14848443, 1238681, 24953277, 113328947, 169886223, 13321057, 802144869, 592403217},
  {1, 1, 3, 11, 23, 63, 69, 241, 359, 735, 371, 603, 2707, 15833, 31795, 14901, 38957, 118581, 508439, 120639, 915051, 464099, 7115209, 6511389, 106275, 62932043, 131236383, 87983063, 318997711, 1000117861, 1595268335},
  {1, 1, 1, 7, 19, 33, 83, 19, 13, 667, 317, 3891, 5497, 8213, 4913, 22387, 54053, 223341, 35765, 69395, 1046055, 3052621, 1474799, 11524359, 26635089, 26651879, 128708697, 168320719, 435453565, 310555273, 436986277},
  {1, 3, 5, 9, 13, 21, 11, 187, 249, 647, 349, 605, 2199, 5033, 29773, 48953, 26389, 133767, 455153, 703029, 906857, 2296545, 7966671, 16571807, 11722981, 23044955, 51831401, 172091473, 84308051, 523957197, 2124405729},
  {1, 3, 3, 11, 3, 3, 93, 235, 441, 933, 383, 2007, 4015, 4175, 3937, 20623, 76773, 25183, 217079, 914647, 1204159, 141695, 5760193, 15213431, 32469469, 20213785, 18912363, 235828755, 513853795, 313287971, 848290893},
  {1, 3, 7, 13, 3, 61, 87, 219, 263, 651, 1343, 2709, 31, 16249, 4749, 28909, 104213, 187863, 250867, 990065, 1693935, 1819609, 1163659, 16344415, 24202379, 63802775, 88977075, 100261929, 121435059, 976212285, 945139489},
  {1, 3, 1, 1, 17, 19, 101, 107, 499, 127, 13, 2123, 5597, 3751, 14527, 12009, 91677, 40071, 437013, 605565, 1773797, 1847239, 126353, 5595479, 4851095, 53419267, 26409521, 181975783, 87763577, 206156123, 13562043},
  {1, 3, 5, 13, 27, 57, 1, 195, 107, 947, 1475, 2831, 6449, 16117, 20555, 61513, 115533, 33927, 54801, 25073, 2085375, 1310173, 7394717, 9234711, 25849279, 51996303, 116191775, 119397139, 167908141, 256780761, 54417567},
  {1, 3, 1, 9, 9, 47, 89, 187, 401, 299, 637, 197, 1235, 12655, 25025, 24443, 86053, 233231, 472565, 107413, 1988925, 2498651, 5681869, 13040631, 21669253, 36311, 62158729, 72163001, 75480983, 64080811, 107480317},
  {1, 1, 3, 5, 9, 57, 33, 41, 451, 983, 391, 2707, 705, 13913, 28843, 34091, 28749, 86029, 101527, 769745, 1622709, 1820253, 962005, 14454333, 31780663, 28209923, 85381499, 161652671, 313160461, 1018864069, 852533119},
  {1, 3, 5, 3, 29, 19, 61, 31, 349, 253, 1267, 3345, 2151, 11309, 19623, 62407, 91965, 23423, 472785, 151455, 2014777, 2392911, 1093561, 3807875, 28357337, 25495121, 45100279, 133315829, 412160347, 806292313, 1002012523},
  {1, 1, 1, 3, 11, 37, 31, 59, 1, 253, 103, 2811, 1871, 4239, 26311, 32507, 28669, 18573, 518293, 121431, 33215, 2829913, 1755587, 5991039, 24851053, 12717409, 14553203, 181064607, 267200227, 969334739, 1172793835},
  {1, 1, 5, 7, 7, 7, 69, 63, 281, 901, 1785, 2131, 4265, 253, 13997, 30177, 63317, 114469, 327281, 746163, 683619, 3014635, 937193, 2397747, 17256285, 32921257, 20546157, 14975319, 506369533, 457606529, 1537253625},
  {1, 3, 1, 11, 3, 27, 111, 67, 411, 751, 241, 293, 6271, 4187, 22119, 63737, 81853, 181639, 253973, 625271, 1642487, 2060519, 1596115, 3064063, 19480759, 62406771, 20497997, 5882889, 75815979, 855862815, 1130783275},
  {1, 3, 5, 5, 27, 19, 45, 203, 81, 59, 1839, 935, 4847, 1869, 12037, 30971, 1653, 261263, 24305, 970777, 442175, 1647447, 1659825, 7949367, 3112245, 23643783, 76091035, 34510619, 369667299, 702728193, 685855977},
  {1, 1, 3, 9, 19, 25, 9, 27, 373, 875, 1735, 689, 2957, 7873, 29771, 4093, 79245, 221397, 258967, 245693, 655983, 2397645, 1093781, 4795471, 5858561, 24932151, 57154787, 91487309, 209501793, 696156757, 431464367},
  {1, 1, 7, 11, 13, 39, 11, 129, 53, 433, 1731, 899, 5855, 10221, 24165, 50205, 113269, 91413, 438515, 411263, 512993, 2083579, 4681903, 10386573, 11799785, 19828005, 49703047, 217788279, 67976163, 945038737, 1780768457},
  {1, 3, 1, 15, 25, 31, 85, 49, 325, 299, 183, 287, 2425, 15353, 25999, 59129, 47965, 86599, 45269, 205171, 127981, 2545395, 2583169, 14374109, 2682273, 22492287, 71780051, 139938627, 164415253, 131948373, 159515579},
  {1, 1, 5, 5, 17, 25, 23, 5, 287, 677, 591, 981, 429, 15297, 14573, 61095, 64149, 12221, 96753, 385745, 2068661, 2205821, 3982819, 14185553, 20809011, 44363457, 20080291, 144367921, 202155137, 1058816997, 215068489},
  {1, 1, 5, 15, 5, 15, 67, 195, 209, 341, 1621, 1379, 3031, 5469, 31563, 49291, 47309, 41293, 346641, 525019, 1198753, 4083779, 3261415, 5497967, 1457581, 61022025, 7479809, 49996071, 180872371, 75760353, 1197767207},
  {1, 1, 1, 1, 25, 33, 61, 201, 15, 125, 969, 1965, 2021, 10253, 23801, 28165, 66565, 24565, 55541, 922549, 429357, 573205, 3783529, 16527229, 22713931, 44778857, 83231493, 211159657, 460197393, 512632713, 1123969885},
  {1, 1, 5, 5, 13, 17, 5, 245, 11, 133, 287, 1929, 4331, 15919, 29663, 10243, 75165, 156461, 273745, 913553, 906985, 2190645, 3210561, 3600705, 4650863, 13306081, 91548739, 230677869, 261469575, 687176251, 1089244707},
  {1, 1, 7, 9, 19, 33, 39, 191, 489, 631, 69, 1883, 2183, 14993, 32715, 62217, 8461, 24069, 416723, 901437, 1523903, 348981, 3839291, 5897035, 11915877, 20012027, 77941937, 245067415, 458172899, 1072145069, 916101735},
  {1, 3, 3, 13, 23, 61, 103, 193, 501, 431, 437, 417, 6557, 11701, 27577, 42943, 122245, 55071, 111095, 17425, 1483883, 1252633, 1814891, 284549, 23539129, 53466267, 29821105, 8601325, 316902201, 1052618337, 480428157},
  {1, 3, 3, 9, 9, 27, 69, 247, 469, 841, 733, 813, 2673, 7145, 5385, 44917, 65733, 76471, 63607, 340917, 388917, 536879, 6132017, 9604459, 24657009, 44781317, 37973433, 27190489, 10059701, 681074285, 254046525},
  {1, 1, 7, 9, 5, 13, 19, 71, 323, 923, 1885, 3031, 4507, 13787, 14149, 1483, 62133, 232333, 126323, 894557, 477161, 3184409, 1923415, 10794755, 3518847, 35763191, 85480897, 76593835, 503268671, 717542999, 1268470937},
  {1, 3, 1, 5, 1, 15, 89, 229, 301, 733, 1343, 2415, 6463, 1875, 9293, 6037, 62741, 5239, 329717, 360121, 861237, 497419, 1657133, 7709097, 26229241, 64473753, 54037755, 87575147, 110222003, 232058271, 1447492065},
  {1, 3, 1, 5, 29, 57, 67, 121, 311, 441, 1079, 1963, 7137, 6745, 9893, 22811, 91189, 4239, 71093, 99033, 331497, 3026237, 5495111, 9312237, 11634763, 48917357, 47651579, 57238463, 110604909, 490465413, 1139046001},
  {1, 1, 7, 7, 25, 13, 27, 61, 331, 361, 481, 3783, 3061, 7827, 18885, 27583, 93365, 157853, 400627, 153875, 1204341, 405025, 7084591, 13917849, 22012503, 11881581, 13828829, 153420243, 186322273, 861484159, 1862478625},
  {1, 3, 1, 5, 17, 47, 19, 83, 309, 65, 1573, 3437, 5623, 12691, 21075, 27789, 102573, 220823, 286741, 901625, 1526309, 1260075, 1219111, 9767583, 20035273, 65523677, 21314593, 162127161, 532795987, 596011159, 1283853823},
  {1, 1, 7, 9, 13, 51, 7, 209, 131, 111, 1143, 53, 7277, 9297, 20869, 33121, 74357, 20069, 213939, 439421, 1184417, 2974247, 199267, 10117741, 32918383, 2278331, 115196523, 254722681, 264621473, 857663509, 491375777},
  {1, 1, 3, 9, 13, 17, 57, 89, 239, 281, 775, 333, 5631, 2805, 10195, 9665, 52269, 30373, 187095, 771261, 1168561, 2594309, 3650077, 13258477, 9935867, 31085261, 114176779, 231543193, 69852723, 290671025, 1537880447},
  {1, 1, 3, 7, 19, 39, 71, 79, 63, 551, 103, 3169, 2761, 13929, 20751, 18951, 76509, 76413, 382743, 653139, 709231, 3420683, 6189595, 10927811, 22834931, 4806923, 80026123, 95356653, 118163269, 989535357, 215830203},
  {1, 1, 7, 11, 5, 23, 7, 249, 447, 483, 433, 3117, 1421, 14991, 5397, 19813, 128693, 49269, 400755, 903615, 408873, 1321483, 200259, 1309429, 27466387, 15505879, 113950685, 257929937, 456862105, 680272363, 416023121},
  {1, 3, 1, 13, 3, 9, 109, 241, 181, 33, 853, 3939, 3751, 2151, 28375, 64443, 62845, 215503, 313301, 658705, 1973495, 3439661, 4434961, 8145957, 20253841, 38649621, 39887473, 171221335, 151335555, 948368731, 1904691515},
  {1, 1, 7, 7, 3, 33, 65, 211, 251, 631, 1819, 3913, 3353, 12975, 19117, 51515, 30805, 160909, 321523, 668499, 178415, 3650573, 4502557, 3067079, 5473583, 14860107, 84992199, 263890821, 284981413, 34048923, 2058546585},
  {1, 1, 1, 13, 3, 21, 9, 203, 223, 229, 1399, 117, 6297, 11535, 16383, 12541, 108957, 209557, 260309, 409977, 476215, 3479569, 5130293, 12970095, 10525035, 24189689, 69717051, 101140081, 132141525, 1055453547, 1593949771},
  {1, 1, 5, 7, 25, 9, 53, 27, 497, 103, 1915, 2179, 3679, 11375, 18907, 63385, 119501, 185413, 415761, 798899, 690365, 3397669, 6035489, 1767663, 30111829, 13032379, 105171663, 129918479, 385492931, 926465587, 696095735},
  {1, 3, 1, 5, 1, 53, 91, 169, 87, 387, 377, 1121, 7241, 5133, 1949, 13433, 105749, 246151, 244917, 508409, 662965, 1562673, 5421103, 9779213, 21240203, 42915927, 95714101, 265182109, 170295805, 405712729, 1700587849},
  {1, 1, 1, 3, 27, 35, 61, 189, 241, 445, 287, 325, 127, 2363, 30663, 43557, 31485, 159157, 480725, 715639, 533423, 1256607, 7555553, 12752801, 28039765, 9169305, 46579691, 178839361, 184371251, 15981151, 2128133611},
  {1, 3, 1, 3, 17, 47, 59, 237, 213, 979, 807, 85, 4621, 9915, 13631, 55657, 51741, 241351, 412373, 422079, 1830949, 2574515, 792463, 13099969, 10836937, 61751, 84062371, 186277673, 255336161, 932716799, 885536411},
  {1, 3, 5, 7, 27, 5, 101, 89, 495, 675, 1181, 2295, 1913, 3731, 32639, 58297, 75741, 248967, 44753, 982971, 832575, 1287817, 740793, 10252573, 1525163, 14908791, 92217169, 167597515, 401868125, 274467103, 1348743851},
  {1, 3, 3, 11, 5, 41, 49, 229, 93, 659, 927, 3425, 4083, 11859, 10603, 20631, 77261, 184831, 448279, 457175, 737273, 2746517, 1941237, 6442769, 19273001, 6579639, 57077339, 23458525, 221400447, 263539911, 1499824823},
  {1, 3, 5, 11, 31, 51, 67, 69, 253, 497, 1665, 1985, 5439, 15999, 4175, 62175, 29661, 114271, 124369, 1022679, 289531, 554639, 6327951, 16383769, 12910049, 44744117, 88159141, 194524341, 249825995, 157560651, 639393539},
  {1, 1, 7, 11, 1, 21, 95, 19, 205, 513, 1329, 1821, 1251, 2381, 32623, 23923, 64733, 111725, 300691, 13471, 170861, 3148937, 2292875, 13349783, 31464769, 50048445, 101767125, 82205257, 141965591, 330811913, 1520581971},
  {1, 1, 5, 13, 3, 1, 29, 175, 315, 865, 599, 1695, 1391, 2313, 31035, 19159, 29773, 165117, 364113, 1001305, 208871, 3328645, 8045281, 16648027, 26352991, 39570989, 98730187, 27167435, 119381475, 1035123077, 1929044823},
  {1, 3, 3, 1, 3, 45, 109, 93, 481, 285, 869, 3441, 3715, 1355, 9581, 50173, 109141, 200535, 159351, 923069, 1803967, 3874745, 4172209, 6127897, 31035589, 12275801, 36844561, 221442549, 145519399, 970086143, 546199825},
  {1, 3, 7, 7, 15, 35, 107, 107, 315, 213, 281, 2073, 4689, 5877, 31, 40967, 97693, 144127, 253459, 365531, 297955, 2714799, 4924103, 16430071, 31424519, 3637801, 10978213, 43745309, 200382053, 919636513, 763272427},
  {1, 1, 3, 11, 11, 3, 73, 41, 79, 37, 481, 1993, 931, 7677, 11321, 45735, 111621, 135485, 461559, 334879, 1328375, 258207, 7329781, 4140661, 12210747, 34729553, 111044525, 124349293, 200392311, 943486953, 1526195845},
  {1, 1, 7, 1, 15, 21, 65, 237, 263, 849, 863, 4039, 3171, 13381, 30373, 51639, 66101, 130557, 89139, 666773, 1201891, 1009313, 3071917, 5482185, 3551827, 58523093, 59979635, 62626291, 163965007, 632525865, 1879746681},
  {1, 1, 1, 3, 21, 57, 113, 3, 487, 41, 1277, 633, 2839, 4977, 14537, 31749, 50373, 128821, 509301, 945975, 1744033, 4011141, 3871381, 12497527, 27569043, 10283925, 48351441, 197217357, 219239539, 651460197, 1189377509},
  {1, 1, 5, 1, 1, 55, 71, 181, 147, 895, 1839, 2157, 3187, 6403, 30367, 48915, 130205, 255469, 469841, 244757, 579173, 1273731, 7580083, 13725273, 25663295, 56593019, 2993523, 208999585, 421621471, 585147559, 1918015171},
  {1, 1, 5, 3, 9, 17, 19, 127, 115, 875, 831, 2439, 7475, 1921, 18657, 27709, 9765, 53205, 427249, 221623, 2028461, 3666093, 1285831, 14979627, 29649487, 58980903, 41769571, 107370299, 529452959, 4324045, 1779300157},
  {1, 3, 3, 13, 29, 11, 35, 81, 291, 483, 625, 3957, 6017, 12543, 18773, 2471, 20789, 167039, 161015, 525585, 954593, 1659567, 3451783, 8847949, 31037023, 32923703, 48669997, 261644553, 509131221, 618167547, 1977586337},
  {1, 3, 3, 13, 11, 39, 7, 85, 493, 209, 819, 3277, 4275, 8997, 22943, 33199, 62109, 141087, 453527, 523857, 1505783, 747395, 7876603, 9430777, 25409377, 24346901, 57966743, 78612921, 219709839, 66696481, 15587011},
  {1, 1, 1, 7, 11, 25, 1, 57, 505, 135, 1713, 3051, 5893, 10711, 10681, 12235, 40773, 108749, 240693, 2643, 1441663, 2347701, 6023069, 9536653, 16436525, 41866259, 27860453, 150332359, 169227761, 404833707, 556132061},
  {1, 3, 5, 11, 23, 33, 13, 107, 289, 251, 235, 1747, 4097, 12237, 17559, 5063, 28797, 155519, 267409, 463319, 1677043, 1926557, 1629665, 2688127, 10701957, 3723047, 68387103, 33719727, 145000541, 997045649, 1205206211},
  {1, 3, 3, 9, 19, 17, 23, 45, 297, 147, 1301, 2057, 7871, 9971, 1965, 62449, 78933, 56807, 383863, 837237, 1990703, 2308261, 2600587, 11649049, 8667365, 59327191, 28493409, 131974133, 321672539, 390553927, 1146120417},
  {1, 1, 7, 3, 17, 21, 19, 203, 289, 897, 1967, 3519, 3261, 8199, 24181, 23273, 12805, 125909, 241475, 623047, 1449293, 2374233, 1310999, 5469663, 30429613, 56160621, 124899435, 62199291, 477943745, 879157563, 1978632129},
  {1, 1, 7, 11, 1, 17, 25, 63, 509, 969, 47, 1329, 701, 5227, 419, 14839, 68957, 115053, 315203, 555727, 772317, 1056253, 6502237, 15858491, 25440601, 217333, 11931803, 240375125, 356906305, 35040879, 945077527},
  {1, 3, 5, 11, 1, 41, 81, 157, 395, 97, 1919, 3043, 6015, 15, 23733, 55485, 78117, 42503, 136129, 419783, 831829, 2229061, 6608149, 2268665, 27919503, 54962133, 110387219, 5580583, 398732227, 15347139, 1843025545},
  {1, 1, 3, 11, 17, 37, 17, 181, 179, 297, 1007, 1559, 6275, 11645, 7535, 28941, 104333, 226789, 449223, 399695, 2047453, 3882313, 5939477, 180065, 13124407, 29307453, 59757667, 225818355, 369812607, 1025732449, 197351587},
  {1, 3, 7, 15, 5, 47, 31, 237, 215, 563, 207, 1867, 6635, 6857, 11837, 22865, 83013, 90071, 179395, 199747, 227673, 3678803, 205643, 2273169, 12268139, 51395399, 69082067, 13172039, 88682007, 766095781, 1137694033},
  {1, 3, 1, 7, 7, 39, 51, 179, 57, 987, 893, 2715, 1045, 5799, 19805, 54275, 104997, 254079, 58181, 230347, 659779, 801531, 849775, 4339183, 12937141, 20769239, 22682969, 230629687, 344585249, 218370315, 202143305},
  {1, 1, 7, 15, 25, 9, 127, 243, 323, 1013, 929, 2891, 7549, 1071, 17663, 15247, 39469, 58061, 1347, 932299, 1530309, 3181045, 5047643, 13745559, 1105535, 43325689, 105143789, 222744543, 314622777, 166385347, 2058237131},
  {1, 1, 1, 5, 25, 23, 101, 9, 371, 89, 1749, 3559, 8071, 13887, 14807, 42825, 80173, 72917, 124869, 747969, 833501, 890947, 5300545, 16256789, 15652775, 8332861, 8041049, 52255299, 354753171, 777331427, 1620911115},
  {1, 3, 3, 11, 21, 41, 3, 77, 3, 709, 1745, 3615, 4141, 5275, 28329, 59739, 81525, 69343, 288071, 198855, 292953, 1276101, 7759639, 15622441, 2169807, 3439665, 37298845, 259845707, 257696937, 856912039, 977243101},
  {1, 1, 7, 13, 1, 11, 73, 183, 363, 241, 863, 3983, 3235, 293, 649, 441, 61973, 63157, 194243, 607049, 225373, 1377407, 6468365, 13491163, 23058063, 56315501, 96518067, 161010059, 269552799, 967772121, 1830758853},
  {1, 1, 5, 3, 13, 27, 13, 191, 57, 639, 1803, 2353, 3143, 12763, 5771, 36155, 26461, 5725, 16065, 195527, 592345, 594391, 7176313, 12222611, 25526605, 31162435, 7505071, 223970829, 26448683, 941603551, 623356375},
  {1, 1, 5, 3, 1, 53, 85, 45, 283, 823, 1213, 3261, 4599, 13267, 4613, 12657, 107621, 239605, 468161, 503239, 1759829, 832377, 8194065, 1448121, 5716239, 62704451, 14936209, 89490721, 342859331, 612409959, 1694141209},
  {1, 3, 5, 15, 27, 49, 59, 123, 357, 527, 337, 2751, 3999, 8525, 12501, 40621, 86949, 257383, 108225, 834627, 858703, 1717837, 5917719, 15311999, 1737929, 7259171, 73278981, 9249475, 119222299, 656804209, 2060606441},
  {1, 1, 1, 7, 1, 55, 85, 17, 447, 599, 1315, 2313, 1649, 907, 25647, 3251, 44205, 100733, 384837, 510915, 187077, 116715, 615441, 13175949, 8772779, 39858131, 123738015, 22763925, 341586877, 807587823, 111240747},
  {1, 3, 5, 13, 9, 1, 37, 121, 143, 1, 631, 2273, 1673, 3649, 27533, 28731, 13733, 35327, 167233, 22849, 610909, 3745781, 4054081, 11263165, 8727899, 29365925, 34479627, 141487269, 355412501, 90787717, 433700457},
  {1, 1, 7, 13, 9, 31, 57, 249, 397, 815, 501, 895, 1217, 11387, 8635, 65193, 85501, 114709, 20291, 595913, 501205, 855915, 4314717, 13080261, 19100329, 45586827, 90148737, 104771075, 402849621, 533124551, 121563135},
  {1, 1, 5, 5, 9, 35, 95, 193, 133, 513, 1929, 3841, 3063, 2383, 24413, 51185, 72965, 47157, 115393, 255169, 1861213, 1718647, 4636219, 7576077, 6537913, 34662981, 27102685, 257568197, 111386643, 664280843, 358096833},
  {1, 1, 1, 13, 3, 49, 45, 191, 15, 181, 1819, 3741, 1227, 12775, 9461, 44951, 43621, 91949, 261061, 682697, 1811527, 1117381, 395873, 9580091, 14533371, 32718601, 43086311, 15603209, 295304679, 646686291, 1769007641},
  {1, 1, 1, 1, 3, 45, 121, 19, 269, 829, 517, 3913, 183, 11019, 24969, 21973, 37461, 130277, 486085, 51653, 1822535, 825449, 5825077, 15394535, 9451177, 27694385, 132190449, 224911101, 532660451, 350863599, 147301909},
  {1, 1, 5, 11, 31, 39, 125, 189, 401, 57, 1557, 1727, 1415, 4025, 30353, 36589, 97973, 109221, 371777, 501711, 1943243, 1367499, 5212609, 2191841, 23820453, 10312589, 104546065, 69186331, 397653523, 625523589, 695635789},
  {1, 1, 3, 9, 3, 55, 125, 187, 409, 499, 1853, 2781, 4323, 16159, 16345, 34659, 81109, 253693, 508999, 815181, 1304527, 2192979, 2664113, 15141799, 14335277, 53519839, 3025433, 18936913, 497819607, 215752491, 950292117},
  {1, 3, 5, 11, 31, 5, 125, 137, 197, 475, 2003, 617, 1289, 8365, 28981, 57537, 24005, 144471, 128193, 935367, 1058251, 605417, 3248065, 5229021, 3599601, 20921535, 69511, 178988229, 10990725, 686757065, 1518283217},
  {1, 1, 1, 5, 19, 29, 83, 127, 311, 177, 1635, 2187, 5377, 12841, 7591, 6095, 121357, 141357, 316357, 860353, 1638231, 1369673, 6716127, 7552971, 30596539, 48790925, 101315071, 236352911, 410407373, 98008901, 797308131},
  {1, 3, 5, 5, 21, 39, 65, 197, 115, 411, 1457, 3011, 7021, 14119, 61, 21107, 24997, 177471, 99009, 210761, 1963201, 4078195, 5036245, 14501145, 24528567, 39168079, 57497789, 132540911, 88680809, 800083179, 409375945},
  {1, 3, 3, 9, 19, 57, 99, 23, 451, 507, 973, 415, 7123, 11367, 29767, 23763, 94893, 182463, 371783, 522565, 1940319, 2378333, 2811887, 6658227, 9713935, 21838439, 15703169, 89024947, 353036455, 853616219, 1873365067},
  {1, 1, 5, 7, 29, 23, 47, 11, 267, 873, 591, 373, 7097, 3783, 23289, 5547, 6133, 117853, 433601, 973379, 121545, 1322827, 6124443, 2769047, 29606007, 61685765, 127812259, 226609041, 227431109, 612032083, 613233501},
  {1, 1, 5, 15, 7, 7, 7, 91, 389, 841, 1995, 459, 7013, 3109, 23615, 21519, 82733, 256749, 110017, 728779, 923219, 859003, 1962459, 140423, 17869913, 16281445, 20792223, 102582959, 435409417, 390062665, 2121212267},
  {1, 3, 1, 1, 13, 25, 87, 235, 193, 201, 713, 1633, 3117, 13609, 17211, 573, 61373, 111591, 366789, 837581, 1776969, 3639773, 4862115, 894159, 14675037, 21493669, 21624333, 136354181, 460327497, 917506541, 531632719},
  {1, 1, 1, 9, 27, 53, 105, 39, 217, 781, 997, 661, 6243, 6427, 17739, 62239, 36509, 246861, 100165, 701005, 1297503, 4032849, 4559813, 14922291, 27589565, 10989393, 131835073, 84912225, 487661879, 427056143, 424601351},
  {1, 1, 7, 3, 3, 49, 75, 125, 239, 195, 215, 2983, 1185, 4743, 12069, 55509, 106309, 1765, 322307, 207623, 359711, 1284925, 7151943, 12011961, 9720003, 52703287, 92247147, 63052715, 192976125, 796630299, 590710433},
  {1, 1, 5, 15, 31, 11, 9, 91, 253, 361, 571, 1589, 2425, 4279, 3765, 46519, 104325, 257261, 290561, 94475, 1946891, 525111, 5910901, 9390967, 25833945, 6770949, 23377103, 215932049, 351121045, 147551075, 1051103825},
  {1, 1, 3, 3, 21, 49, 49, 213, 399, 253, 1565, 2447, 453, 7957, 24799, 58503, 74861, 208013, 35079, 262919, 1333529, 3129661, 1946469, 3210001, 20035659, 66218665, 83528481, 240931195, 74675377, 506667049, 1441881803},
  {1, 1, 7, 1, 5, 59, 81, 97, 209, 477, 17, 3085, 3655, 599, 24011, 14981, 54557, 60965, 74499, 206597, 1979673, 250175, 3837253, 11840397, 17510037, 39438169, 130212437, 196173945, 137934083, 917219171, 1496119255},
  {1, 3, 3, 13, 19, 49, 7, 35, 111, 169, 629, 1587, 5345, 13699, 21187, 30199, 108125, 46383, 96007, 788737, 1447711, 3092741, 4699467, 3793383, 14271347, 45188389, 37554889, 21240727, 319434805, 608847439, 1267126343},
  {1, 1, 3, 13, 19, 59, 73, 127, 475, 509, 9, 2661, 711, 15835, 31429, 33885, 95013, 162693, 140039, 247561, 1801503, 1295631, 5088005, 14043027, 5517311, 31244033, 60693093, 205560465, 159293411, 180204879, 656923353},
  {1, 3, 5, 3, 31, 15, 43, 185, 29, 897, 1041, 1057, 6285, 13925, 4023, 25741, 80973, 225831, 347905, 102159, 1680139, 447235, 3301719, 9322373, 19039153, 19566693, 96778085, 149623589, 8880585, 533112801, 898205827},
  {1, 3, 5, 11, 1, 33, 63, 155, 175, 501, 1147, 3395, 3285, 16237, 22315, 50945, 59453, 63255, 361473, 756743, 1074197, 1340429, 4697147, 12893215, 18998355, 39894937, 112696007, 149626519, 140427577, 702840417, 387131519},
  {1, 3, 3, 3, 5, 13, 77, 227, 85, 139, 139, 1, 7147, 2023, 32705, 38753, 130677, 237751, 166919, 869391, 2077705, 1448961, 2057305, 16549079, 9665889, 40521479, 82615775, 256980525, 152731079, 994886627, 1332841581},
  {1, 3, 5, 11, 31, 59, 35, 179, 489, 537, 1537, 2877, 4937, 10825, 2445, 34907, 6117, 65471, 151553, 151559, 1437707, 3518487, 667743, 7008351, 5255525, 63615189, 4649381, 116331353, 222444365, 757618365, 718058665},
  {1, 3, 7, 11, 17, 17, 95, 223, 165, 925, 829, 3971, 1, 7393, 8825, 25705, 52661, 63607, 157187, 333319, 68109, 1668669, 642587, 6262427, 29421529, 50484961, 43000745, 129180663, 498149109, 540183789, 586327165},
  {1, 1, 1, 1, 25, 17, 25, 143, 385, 907, 1381, 1823, 3819, 8475, 5321, 12037, 5077, 117605, 86533, 435717, 1829405, 1337877, 6281853, 3751627, 3043301, 6455223, 24206689, 153704243, 379340159, 656888679, 974336357},
  {1, 1, 5, 11, 7, 47, 97, 85, 105, 23, 263, 1329, 1905, 12121, 29635, 41249, 128733, 169045, 118273, 667151, 1360403, 3129859, 4215421, 8087273, 6317037, 39295811, 23562915, 230188909, 73678189, 60538269, 281724935},
  {1, 1, 7, 11, 1, 51, 13, 13, 5, 143, 83, 3831, 959, 6083, 16997, 59887, 129317, 10189, 212611, 67215, 878749, 2930591, 7752841, 4665153, 23651761, 9768507, 100812487, 176243531, 130991731, 126842271, 1289782041},
  {1, 3, 3, 13, 25, 9, 31, 5, 215, 791, 767, 1733, 2715, 14283, 25795, 54249, 91901, 107991, 433031, 511617, 740501, 550077, 4033019, 14862753, 17603627, 28243331, 119873571, 185542425, 18903271, 218922591, 1130960687},
  {1, 3, 7, 5, 11, 19, 125, 81, 71, 131, 1869, 1111, 6763, 5275, 23095, 1139, 107693, 205631, 174211, 557193, 1462423, 654983, 4636881, 16667549, 16764211, 1629511, 86009809, 29616859, 457406943, 773889095, 1823386139},
  {1, 3, 3, 9, 25, 43, 119, 49, 133, 217, 521, 1367, 5867, 6829, 29871, 60383, 118477, 151791, 308359, 394373, 468885, 3416207, 4515475, 15524637, 14535129, 907933, 131178397, 27839411, 234310223, 489310065, 1705691267},
  {1, 1, 7, 9, 5, 11, 59, 157, 279, 301, 481, 3273, 7943, 3273, 27783, 17271, 42669, 64909, 507267, 1000077, 1891225, 2863791, 1229999, 4368689, 16872059, 53544921, 55584829, 181121533, 318541315, 152809325, 1214504731},
  {1, 3, 3, 13, 11, 57, 13, 5, 435, 169, 541, 517, 2359, 9121, 27911, 15679, 2733, 29487, 256903, 766593, 109447, 3280525, 2522273, 1131873, 12197639, 11892797, 94388049, 56698401, 117748931, 282065461, 1831566427},
  {1, 1, 3, 9, 9, 55, 65, 113, 21, 807, 373, 2825, 1527, 15565, 8339, 7227, 7677, 135933, 411015, 609933, 1860997, 2490515, 7975653, 1265965, 32763537, 3813475, 89691297, 129621013, 270577827, 276325737, 1410330831},
  {1, 3, 5, 9, 1, 1, 49, 255, 477, 821, 1647, 713, 6841, 3187, 22649, 51469, 81461, 129991, 131713, 409221, 1458069, 1872549, 4869301, 14576763, 15598745, 23352905, 36172187, 38138525, 14477189, 719121367, 1761786141},
  {1, 3, 3, 11, 13, 43, 63, 139, 71, 809, 993, 2429, 833, 6545, 10987, 39567, 39037, 59887, 437383, 762503, 1110657, 3553159, 6595979, 16033959, 16951611, 20772997, 103800701, 110955353, 169935173, 158415845, 399194735},
  {1, 1, 1, 9, 19, 23, 47, 199, 191, 827, 391, 837, 1209, 2493, 24071, 46589, 124621, 164837, 286341, 110733, 508567, 3610547, 334563, 1085979, 28634755, 27375271, 38695419, 88723497, 249483045, 160710953, 1222253923},
  {1, 1, 5, 13, 25, 51, 39, 43, 103, 899, 1729, 2389, 2965, 189, 3063, 50609, 97325, 176053, 39009, 711465, 284013, 283239, 6710563, 5748487, 9529115, 52925519, 69682301, 161041529, 441174833, 840586953, 2039294211},
  {1, 1, 3, 1, 5, 29, 105, 201, 503, 199, 507, 205, 2389, 695, 15233, 50353, 84725, 6933, 183591, 175973, 1996265, 182905, 4509469, 13243293, 20838323, 31595523, 90810823, 159892145, 408309465, 980567587, 2009560309},
  {1, 3, 1, 7, 19, 53, 45, 21, 89, 545, 1885, 765, 6673, 13485, 9987, 2609, 23229, 178231, 300901, 187307, 1624183, 533065, 6293825, 1468193, 25317197, 7085269, 73449305, 93844377, 160923525, 312043065, 1423187495},
  {1, 3, 7, 13, 17, 7, 59, 23, 159, 309, 1407, 2483, 1807, 15733, 5603, 52989, 106269, 4935, 329443, 950433, 1356781, 1190739, 6783839, 9657643, 20036883, 51603209, 34373795, 143297223, 177443795, 316325529, 1530394559},
  {1, 1, 1, 11, 13, 19, 123, 185, 413, 745, 361, 2391, 6697, 2281, 11999, 13175, 35085, 218221, 196325, 235311, 1946537, 1389023, 5887791, 6057877, 27538513, 19080813, 28805949, 212983379, 475064173, 304330581, 1611651867},
  {1, 3, 5, 5, 11, 49, 123, 173, 325, 667, 895, 1067, 8121, 10577, 30561, 17391, 9589, 181135, 187361, 466665, 212799, 4064197, 5178743, 13904649, 21327945, 2339599, 86751115, 89938023, 375923237, 456710941, 79560869},
  {1, 1, 5, 5, 23, 21, 77, 223, 281, 161, 141, 345, 3879, 11393, 26907, 53805, 75581, 236101, 117601, 829089, 1240547, 574305, 1058673, 7014763, 20639949, 32936317, 81081993, 264854077, 350962355, 563214341, 2096683751},
  {1, 3, 3, 5, 3, 47, 17, 109, 185, 139, 957, 1417, 4553, 6101, 29537, 34821, 11829, 62055, 210535, 771049, 797807, 3344731, 5662013, 9646449, 25219485, 32903967, 77551481, 267787077, 374343133, 560992913, 1762667109},
  {1, 1, 5, 3, 29, 49, 89, 61, 45, 593, 269, 1483, 2971, 991, 21239, 29301, 69133, 15941, 207265, 1016743, 605161, 1031261, 6337101, 8876569, 5052009, 46952389, 98672345, 50036207, 256554063, 857583635, 1242874875},
  {1, 1, 3, 13, 29, 45, 33, 253, 291, 783, 737, 2363, 2651, 8627, 21785, 58419, 86453, 203677, 158759, 665961, 1430193, 3118585, 5303861, 203369, 14520007, 66114075, 116029069, 85826663, 208976959, 385078399, 1656250197},
  {1, 3, 7, 15, 29, 15, 81, 185, 363, 681, 1737, 3789, 4365, 3295, 23205, 4457, 44261, 99575, 280355, 973027, 898785, 883539, 148997, 12120741, 30407183, 24646189, 55702469, 4387721, 528820041, 74751691, 295083889},
  {1, 3, 3, 11, 15, 39, 67, 167, 197, 357, 871, 2201, 5377, 6299, 20873, 59283, 103445, 243423, 192679, 185959, 402595, 1888747, 7317087, 15082139, 4642409, 66124505, 127076691, 240161453, 524722877, 612621727, 1021320813},
  {1, 3, 3, 5, 9, 15, 127, 49, 21, 719, 357, 425, 5507, 9639, 275, 47503, 32349, 231727, 67495, 1039657, 1244069, 486651, 4410491, 13982893, 11341193, 35421803, 13219473, 142441141, 485119479, 856115683, 512352191},
  {1, 1, 7, 1, 13, 63, 111, 121, 21, 481, 247, 3147, 5783, 8947, 20809, 42039, 4661, 185869, 214819, 427877, 354353, 2914011, 3361851, 56357, 10351689, 53399013, 31514627, 127249903, 15498667, 1049963327, 619128437},
  {1, 1, 3, 3, 31, 33, 9, 69, 187, 517, 2029, 2205, 7661, 4757, 27525, 9665, 98917, 49397, 62567, 710759, 1919219, 4037965, 1565717, 5899489, 26625087, 60609585, 30622433, 268258793, 256459961, 563641025, 837897809},
  {1, 1, 1, 13, 7, 41, 103, 161, 233, 221, 693, 213, 4609, 7771, 28703, 17407, 104909, 73069, 207525, 147113, 1804963, 2057469, 175131, 14509157, 10396821, 31131129, 7129585, 195025153, 432400693, 929801327, 650894843},
  {1, 3, 7, 15, 31, 47, 27, 111, 479, 1003, 1687, 347, 2179, 11861, 8169, 31941, 70133, 216871, 323299, 193763, 2028899, 2457715, 1814599, 7762163, 32305555, 19305079, 90133451, 200021591, 437258751, 256916249, 1041254213},
  {1, 1, 3, 5, 5, 63, 25, 125, 199, 147, 589, 3565, 3449, 8331, 8923, 31207, 43101, 39181, 375463, 330273, 1864873, 2133707, 6152301, 4137121, 729283, 26386567, 33744881, 116581185, 282619309, 700788895, 1403065503},
  {1, 1, 3, 11, 19, 25, 77, 99, 299, 19, 1641, 2193, 4299, 3635, 20621, 267, 23973, 192381, 127783, 506863, 586431, 398805, 7923809, 16087079, 5355967, 25626943, 25443621, 35819325, 211590735, 524257751, 1975157505},
  {1, 3, 7, 11, 9, 59, 7, 167, 1, 775, 29, 1935, 3723, 11835, 2887, 65285, 129997, 11335, 290723, 704551, 940917, 3117815, 3744131, 12879851, 26016317, 32303355, 30882233, 145939507, 311578975, 691338087, 46635851},
  {1, 1, 7, 13, 5, 47, 101, 155, 235, 93, 465, 3581, 1837, 7675, 10789, 45167, 63333, 108333, 137763, 644201, 1176249, 2453883, 128657, 3809543, 3110559, 60748113, 97195357, 46283369, 221057737, 133895727, 69534065},
  {1, 1, 3, 5, 9, 59, 121, 109, 59, 821, 1625, 343, 1591, 3875, 13729, 56381, 51605, 50341, 198567, 279265, 331237, 4016079, 3429309, 12987169, 18928191, 6770785, 48427029, 88439779, 161112627, 154235871, 2125671581},
  {1, 1, 1, 9, 27, 53, 93, 215, 133, 561, 39, 2591, 1041, 11913, 24493, 37921, 88773, 5045, 417125, 1011693, 1349503, 3578097, 3911633, 8853987, 15490321, 36146637, 21298131, 163814587, 497013933, 686736405, 812771849},
  {1, 1, 1, 7, 23, 7, 81, 107, 219, 943, 563, 1083, 5803, 5687, 32559, 62727, 26861, 165261, 412709, 362531, 1854259, 3013755, 5614317, 14239639, 7187647, 10276003, 112026367, 85879271, 132478919, 458495227, 416579107},
  {1, 3, 7, 7, 21, 53, 3, 5, 231, 601, 1561, 103, 3837, 2675, 5263, 23375, 12173, 57135, 385315, 572715, 530985, 1954377, 407287, 15679345, 27680075, 8527277, 111662821, 46126499, 141096313, 287747791, 1050056731},
  {1, 1, 3, 13, 15, 27, 89, 7, 251, 887, 951, 3001, 5687, 4921, 5091, 59337, 80477, 142293, 311015, 603689, 1500643, 2844111, 489861, 4453835, 964991, 60852555, 64781691, 182614277, 439841995, 461274365, 1758689695},
  {1, 3, 7, 15, 25, 53, 19, 155, 185, 503, 547, 1917, 7633, 15167, 14177, 46761, 74165, 39415, 175779, 256227, 1799781, 831977, 7486679, 7721711, 25396053, 21945083, 36747687, 161944833, 433966717, 781771947, 933740293},
  {1, 3, 5, 15, 21, 49, 13, 163, 471, 281, 1141, 3013, 6847, 9261, 15955, 9397, 91549, 51303, 384353, 802339, 419681, 2194925, 1168121, 8066695, 23100099, 50743925, 89823689, 233159361, 235467467, 738938265, 1424308687},
  {1, 3, 3, 3, 1, 21, 19, 239, 479, 609, 65, 2735, 5337, 6293, 19351, 34135, 116141, 256687, 75751, 1047215, 215085, 3098489, 3081143, 5353115, 31141555, 64281277, 86104669, 15616651, 89708669, 538763657, 1563960979},
  {1, 1, 7, 1, 9, 1, 119, 23, 411, 535, 101, 1597, 2379, 2421, 31471, 36473, 44589, 166997, 292195, 549285, 1092597, 3201765, 7652019, 8066099, 20839719, 29729963, 115044585, 240455745, 394877823, 724682913, 210756707},
  {1, 3, 1, 11, 31, 63, 17, 225, 45, 409, 59, 3943, 8043, 11759, 10667, 58793, 72573, 114615, 263205, 4519, 945403, 877299, 5884621, 11085949, 29739209, 67087293, 7077167, 204305251, 332296295, 180256019, 100365543},
  {1, 1, 3, 3, 9, 49, 61, 239, 245, 765, 1945, 3567, 5355, 14799, 7141, 59511, 101701, 243661, 180775, 190567, 988325, 922205, 7902713, 12040331, 3758113, 50006305, 110172589, 95338875, 46942383, 486862227, 1747584297},
  {1, 3, 7, 9, 3, 33, 103, 99, 35, 799, 1347, 2253, 8189, 14177, 13479, 13749, 107565, 207911, 199075, 386917, 473919, 2455333, 5191947, 12932871, 4330399, 32560115, 36155631, 238657361, 353339281, 619743029, 91749811},
  {1, 3, 1, 15, 15, 51, 7, 179, 471, 265, 1571, 2983, 701, 15133, 7885, 29977, 121541, 173527, 280869, 544291, 1492651, 3240239, 1026395, 9567583, 19611947, 12665125, 36685343, 966755, 183550801, 179166305, 935391769},
  {1, 1, 5, 9, 15, 37, 49, 213, 113, 729, 1115, 2727, 2635, 8433, 11145, 46779, 84661, 122717, 230561, 320941, 1608443, 87329, 8158509, 10782689, 24361109, 59955565, 50515007, 151492323, 230504071, 39876941, 2101252229},
  {1, 3, 5, 11, 7, 3, 115, 151, 133, 723, 7, 4063, 5807, 9845, 25829, 29315, 30021, 122239, 204833, 703015, 941043, 3051791, 5095695, 6147899, 18848041, 34782063, 56921971, 203577843, 279702355, 440480425, 1146681},
  {1, 3, 5, 9, 25, 55, 17, 135, 145, 379, 1603, 3459, 5773, 6545, 17509, 25847, 68581, 44463, 332449, 21285, 180589, 3635507, 7703829, 847867, 3996917, 35968839, 68652007, 95926095, 137730657, 112441021, 785308721},
  {1, 1, 7, 11, 1, 61, 113, 147, 489, 775, 1347, 2199, 299, 9589, 19931, 1335, 52989, 197677, 295203, 62191, 1036413, 3092017, 2847317, 4312711, 10118989, 45351787, 51119943, 29623467, 138678343, 43268105, 910447511},
  {1, 3, 1, 3, 1, 7, 27, 243, 355, 425, 1215, 3723, 3489, 9285, 12739, 24797, 43197, 250055, 379173, 1033455, 662821, 1249835, 3259007, 15431375, 14576495, 27125829, 32243667, 137679183, 150523709, 1007634441, 465007719},
  {1, 3, 5, 11, 15, 25, 57, 221, 247, 647, 259, 1665, 7055, 2835, 16411, 42999, 75645, 90191, 9953, 200935, 1774459, 978453, 366181, 4367897, 16752499, 37046803, 17215487, 102775229, 246716963, 800098807, 381076679},
  {1, 1, 3, 7, 9, 25, 61, 233, 73, 235, 1539, 1865, 5671, 1329, 5767, 43039, 73869, 54989, 130727, 391011, 532325, 2203173, 7696953, 11747053, 19639901, 28616815, 85593607, 63970629, 284435723, 349465397, 1937541499},
  {1, 1, 7, 9, 21, 11, 123, 7, 41, 407, 1485, 1723, 585, 10597, 16215, 63403, 53485, 11293, 18723, 886381, 589801, 2481679, 626191, 11082443, 17820389, 9426219, 105884169, 235888007, 485401501, 957182121, 1923784339},
  {1, 1, 5, 7, 27, 9, 123, 101, 273, 673, 1141, 3841, 4041, 13169, 8221, 12915, 125573, 54461, 241825, 921891, 391407, 733237, 3567671, 6926561, 30126109, 12136533, 3993873, 86743781, 341385917, 47996245, 208075545},
  {1, 3, 1, 13, 3, 17, 105, 17, 237, 321, 855, 2821, 2467, 3503, 15187, 1689, 42269, 198327, 393573, 201185, 21159, 1398789, 1656837, 5617781, 6953193, 42318181, 98298563, 196561697, 214458079, 1073280027, 1460706815},
  {1, 1, 5, 5, 19, 23, 9, 205, 87, 153, 1543, 1193, 6619, 6845, 8459, 37533, 74877, 13189, 428321, 584417, 10023, 3920931, 3030117, 13009969, 18529491, 1623501, 27320763, 216310765, 325512295, 373716729, 1785755975},
  {1, 1, 7, 15, 13, 29, 79, 9, 203, 211, 239, 2349, 3447, 7797, 19311, 58071, 14765, 147629, 335395, 764011, 186993, 150529, 5649499, 8252573, 24744983, 28681559, 38571107, 84303745, 427912171, 81598369, 873953491},
  {1, 3, 5, 11, 5, 49, 71, 151, 333, 895, 1095, 2471, 2477, 14493, 16711, 14393, 15853, 246871, 310817, 716903, 296881, 3916861, 4860979, 6896883, 7329073, 46906755, 74423123, 189831355, 142071377, 668746657, 550065171},
  {1, 1, 5, 13, 17, 19, 25, 149, 111, 631, 763, 2535, 3631, 1809, 8163, 18037, 1245, 133477, 497441, 1035625, 1609637, 1258631, 6622717, 4143097, 23886379, 2739075, 111696039, 175371419, 506473955, 871106797, 772539455},
  {1, 3, 3, 13, 23, 61, 25, 179, 351, 247, 1021, 2413, 2585, 3731, 5435, 52723, 35869, 198687, 110951, 122977, 77179, 455337, 224933, 15571367, 18351643, 33642587, 122624673, 9330961, 164329485, 342665031, 1276416319},
  {1, 1, 5, 11, 1, 39, 65, 59, 21, 927, 1989, 2823, 4857, 15521, 30495, 16067, 62925, 222589, 145185, 345839, 1660789, 1196715, 6441701, 11293063, 18554865, 42824403, 89199089, 113777883, 247922189, 956573597, 516697499},
  {1, 3, 3, 7, 17, 5, 17, 125, 379, 875, 1565, 2435, 933, 6809, 20129, 26339, 96885, 176383, 16423, 50475, 2078589, 2958777, 1942709, 4103561, 24801119, 1458239, 66110929, 131911567, 288147953, 1028684917, 432326869},
  {1, 1, 7, 5, 3, 57, 51, 213, 455, 663, 2007, 3995, 4889, 9527, 23507, 3261, 110589, 199653, 377827, 693089, 1598655, 202381, 2631071, 16137041, 27665963, 6028579, 16157803, 71370023, 377165669, 417409339, 1152469719},
  {1, 3, 7, 5, 1, 29, 85, 151, 165, 123, 1425, 2851, 4427, 7683, 21085, 28925, 40037, 212711, 441187, 564905, 655037, 2444713, 2032561, 6500227, 25442257, 12055687, 14521125, 113219567, 200383599, 184465327, 1246677137},
  {1, 1, 5, 3, 11, 33, 127, 3, 41, 591, 1539, 3823, 125, 421, 9051, 55025, 73245, 179829, 341921, 492647, 238399, 1664653, 6657523, 591271, 4034613, 14356643, 69367303, 14933683, 120361873, 607629753, 1595065807},
  {1, 3, 1, 5, 7, 9, 69, 35, 59, 477, 1207, 1245, 6423, 11329, 26535, 37621, 79117, 191463, 314149, 165353, 1277347, 2632893, 6582201, 5647783, 9250063, 25083441, 23145211, 192670089, 253210667, 112373717, 1983756507},
  {1, 3, 1, 1, 15, 35, 17, 123, 303, 193, 1489, 2587, 1883, 4063, 1921, 60413, 54805, 240935, 141029, 573229, 1093803, 934279, 6142413, 3640663, 31024587, 23631693, 73379981, 50377151, 408958463, 621952467, 1296720461},
  {1, 1, 5, 1, 7, 61, 39, 247, 139, 1015, 757, 823, 4757, 9307, 32287, 37241, 120333, 89781, 484705, 928229, 16691, 1829785, 8347803, 14979619, 8113847, 7879755, 65804793, 251321739, 185964321, 52926567, 1921290027},
  {1, 1, 7, 15, 3, 5, 85, 93, 457, 999, 1331, 919, 5271, 11687, 24233, 38803, 44405, 180125, 235747, 287019, 1664639, 690841, 3487737, 13506601, 16914941, 35481971, 19316023, 207114027, 85705083, 1010976859, 1611064981},
  {1, 3, 3, 9, 5, 43, 37, 13, 505, 603, 1857, 2675, 2017, 9481, 10873, 54755, 94037, 202367, 103719, 160549, 108649, 1513135, 1825745, 5320769, 18878925, 44471119, 119902533, 148911423, 352804549, 542757221, 1045694493},
  {1, 1, 5, 15, 13, 3, 7, 239, 471, 835, 553, 413, 4029, 8613, 15533, 58987, 59293, 214213, 447785, 759795, 1469057, 704135, 982427, 739067, 11807619, 50109559, 126927869, 150127705, 473746417, 138503977, 1467460001},
  {1, 3, 1, 5, 19, 27, 21, 43, 57, 755, 1245, 2805, 3799, 2013, 21145, 10317, 81037, 151039, 514141, 265105, 1009311, 3871927, 5037793, 16657935, 23062021, 6206919, 46380209, 4421057, 58915371, 759287121, 1257221181},
  {1, 3, 5, 13, 9, 23, 35, 5, 315, 169, 399, 2641, 1525, 10561, 11917, 33009, 83005, 261039, 414585, 471897, 484501, 912059, 4248767, 2007617, 32750111, 23396437, 42665979, 124588029, 381748401, 431818093, 714170273},
  {1, 3, 5, 7, 23, 53, 101, 105, 451, 207, 1271, 3067, 6725, 15525, 7951, 1283, 70933, 94327, 337993, 466291, 161419, 2905009, 6933969, 8828661, 6401807, 37278099, 128217955, 254259663, 49338193, 300930177, 1233135667},
  {1, 1, 5, 5, 27, 21, 77, 143, 213, 443, 149, 2667, 5269, 10359, 25287, 5843, 1701, 137077, 322057, 381689, 636311, 549305, 5857241, 9893075, 19014777, 65205823, 129404393, 138900319, 281481417, 851514435, 1974286619},
  {1, 3, 5, 5, 25, 27, 109, 157, 459, 767, 765, 2667, 1833, 3781, 9077, 64321, 67069, 246959, 520185, 953169, 1304709, 556183, 5580529, 1612009, 2594999, 54597843, 52215145, 250942111, 336642581, 305000945, 483440089},
  {1, 3, 3, 13, 31, 25, 97, 237, 279, 431, 1713, 809, 1989, 10431, 5867, 42197, 76117, 191247, 103167, 1011609, 1635995, 1242757, 2895301, 9968417, 11269931, 65350659, 97491989, 129135485, 400665737, 614703331, 1473813031},
  {1, 3, 7, 3, 9, 5, 5, 191, 73, 521, 787, 903, 3073, 2067, 24741, 57029, 31501, 1855, 260379, 529303, 1682589, 1779073, 3382217, 10102611, 9813605, 16477709, 23456543, 220999579, 193543445, 599474055, 722301225},
  {1, 3, 3, 1, 3, 41, 125, 53, 125, 781, 865, 3677, 6279, 357, 10667, 1127, 18565, 2503, 194607, 550453, 1470855, 3605637, 7619785, 13660441, 13372017, 37457441, 10380653, 266020945, 326316859, 313157193, 290494159},
  {1, 1, 5, 11, 25, 19, 99, 121, 359, 73, 607, 2149, 5739, 15669, 29457, 57549, 85741, 145085, 360153, 814231, 776597, 2385287, 3751823, 1164237, 457315, 22936069, 113055931, 44486761, 113825199, 773448265, 456916453},
  {1, 1, 5, 3, 23, 5, 35, 55, 369, 239, 329, 3057, 3757, 12523, 5017, 52185, 56381, 121005, 88889, 755295, 1710235, 2680753, 2504119, 12191531, 27142949, 50688251, 13062533, 175952437, 414783585, 675229975, 1646067893},
  {1, 3, 3, 13, 17, 61, 69, 165, 267, 323, 2007, 2025, 4423, 15975, 31897, 37013, 67165, 27039, 294623, 727449, 812181, 1276833, 7324633, 3787657, 26591847, 13263535, 51208451, 188982125, 126886195, 735188683, 2103080509},
  {1, 3, 7, 13, 19, 19, 87, 111, 389, 611, 1523, 963, 4671, 12569, 21839, 10919, 32501, 258535, 213995, 728377, 2032007, 3470511, 7842435, 7585771, 22797809, 50461911, 72904815, 228128791, 501405755, 807848765, 1764245531},
  {1, 1, 1, 3, 1, 27, 13, 227, 29, 457, 221, 127, 5335, 16247, 19559, 19185, 46373, 67581, 256029, 714719, 1938077, 2288271, 3273089, 10661175, 24459617, 1292829, 131469281, 208970243, 403078331, 873025907, 1447826835},
  {1, 3, 5, 7, 29, 21, 23, 157, 197, 87, 1591, 1829, 407, 15885, 14933, 1997, 25277, 132639, 382489, 759315, 245249, 406033, 7036427, 12170833, 6821857, 66811339, 128332475, 182468481, 450903459, 542646353, 76023889},
  {1, 1, 1, 9, 3, 35, 43, 187, 195, 325, 197, 2905, 7323, 1563, 7659, 45185, 52725, 10205, 358237, 791893, 1242655, 2384415, 961151, 9079487, 3161927, 19873473, 100776609, 175685093, 305444759, 828777223, 1952844575},
  {1, 1, 1, 15, 3, 23, 105, 33, 73, 495, 1409, 2583, 1099, 1041, 28955, 60293, 129893, 81485, 410173, 720787, 1423903, 3750451, 3677709, 8520341, 3069797, 8834387, 26328381, 247600115, 328899495, 160414901, 1773445791},
  {1, 3, 7, 13, 25, 19, 99, 137, 139, 719, 529, 1147, 5813, 11551, 30183, 14593, 117989, 32703, 369179, 436057, 312845, 2038319, 2141727, 3473117, 7459343, 42186947, 131697733, 107919, 121004601, 873425547, 1433289723},
  {1, 3, 3, 5, 17, 25, 73, 193, 309, 887, 1655, 1641, 2091, 12087, 21881, 1145, 23517, 28079, 132159, 502225, 1778197, 1377829, 491029, 13107837, 15524809, 32123307, 130985531, 115687437, 246074143, 640272475, 1656159213},
  {1, 3, 1, 1, 27, 41, 13, 151, 83, 645, 327, 1795, 2717, 12433, 22751, 9823, 125301, 120839, 111277, 6453, 771607, 3176981, 8151673, 3374843, 29490847, 8890401, 122473059, 32611583, 91960473, 253649141, 2094314691},
  {1, 1, 3, 7, 1, 43, 77, 229, 59, 499, 1883, 135, 3461, 9821, 219, 6111, 107013, 188965, 13167, 658491, 2006037, 93247, 6448193, 5162033, 30997575, 45772479, 90562639, 80686275, 143405249, 597589417, 503489351},
  {1, 3, 3, 3, 23, 7, 17, 67, 361, 565, 1621, 3253, 7973, 281, 3209, 48215, 38509, 89495, 252815, 1019319, 1717267, 2421795, 4666469, 10601687, 33129573, 39649217, 10196217, 79066905, 134602057, 1013341749, 1445977317},
  {1, 1, 3, 7, 31, 15, 97, 141, 197, 883, 1689, 269, 7487, 10403, 18903, 58147, 18117, 120245, 87439, 277115, 1043723, 1451291, 1075477, 9217337, 5991369, 18331071, 94079733, 90210105, 128914835, 894634607, 545740755},
  {1, 3, 3, 3, 23, 9, 87, 125, 359, 527, 1251, 637, 1145, 12721, 14693, 6021, 45213, 82879, 244703, 304023, 908051, 473901, 1347363, 16213969, 30424579, 21564627, 132525799, 223799209, 103019773, 602991789, 822269017},
  {1, 1, 3, 5, 13, 43, 105, 173, 205, 859, 1237, 1227, 1409, 15513, 25317, 30745, 83933, 163533, 251711, 439897, 405273, 1758007, 4370245, 10140473, 31459297, 22920983, 58853657, 159722015, 353412061, 441065901, 743054569},
  {1, 3, 3, 15, 31, 43, 125, 149, 145, 109, 975, 1167, 7629, 8373, 5923, 64117, 96933, 48719, 173503, 729499, 941851, 1303359, 7744841, 7206705, 21361357, 58860129, 73213939, 253682147, 442513801, 975249185, 681522079},
  {1, 3, 1, 15, 3, 33, 89, 173, 379, 615, 1401, 1567, 4453, 7461, 32555, 64369, 28933, 29471, 77085, 650715, 739087, 3513125, 1199421, 8707009, 10330663, 26265867, 78937965, 233995147, 106853641, 603027425, 668077783},
  {1, 3, 7, 11, 27, 23, 45, 7, 15, 21, 1663, 3327, 5611, 8535, 27669, 25525, 58221, 139519, 140443, 716319, 1380623, 2946355, 2457449, 2345779, 8354483, 66750737, 97193387, 240943523, 352753999, 985905411, 717873329},
  {1, 1, 3, 15, 15, 61, 127, 145, 151, 41, 629, 767, 5801, 3395, 1157, 30033, 55693, 52461, 37951, 507475, 2015515, 330505, 1924363, 6535661, 10450291, 7923525, 29873233, 236033267, 21578821, 219956871, 1471896969},
  {1, 1, 1, 5, 9, 63, 73, 9, 299, 237, 369, 1295, 4869, 6821, 19961, 32129, 102733, 169549, 508349, 110809, 471317, 2427187, 2964821, 7185901, 25802855, 48027761, 70473733, 106934899, 343962137, 171166913, 1528868741},
  {1, 1, 5, 13, 19, 7, 119, 35, 319, 405, 1255, 1299, 4311, 14999, 24567, 4001, 101605, 155069, 24089, 64337, 1452831, 1063691, 7277315, 16522615, 2119915, 5954513, 10120051, 90590607, 319702803, 119275443, 807511363},
  {1, 1, 1, 13, 9, 39, 13, 207, 355, 691, 37, 3137, 6073, 16179, 28661, 41, 102157, 175949, 398141, 608657, 986069, 521675, 2613393, 1972747, 18293311, 44354135, 75853249, 193332773, 487986757, 91622255, 1635928073},
  {1, 1, 3, 7, 21, 3, 123, 27, 187, 183, 769, 2367, 2957, 7065, 17855, 60805, 129653, 1069, 30399, 701787, 1180737, 3559895, 1939175, 13696527, 6720095, 45246835, 30005829, 135920587, 15212737, 658314589, 281222947},
  {1, 1, 1, 1, 19, 31, 71, 85, 303, 617, 1007, 283, 8087, 11079, 11463, 65295, 96997, 193733, 85549, 433661, 931663, 1002819, 4457427, 10737249, 27412283, 45486645, 52697035, 192321751, 503135699, 726300163, 1761922035},
  {1, 1, 3, 13, 25, 63, 61, 187, 401, 465, 1485, 801, 3649, 7763, 8495, 54479, 29029, 65029, 473007, 74737, 184653, 2426179, 4687809, 12047007, 451405, 13579749, 9001793, 229837341, 134938669, 356061167, 201397811},
  {1, 3, 7, 3, 13, 51, 41, 119, 311, 699, 1113, 3631, 1981, 3259, 25871, 45659, 103333, 231127, 229131, 996535, 499033, 2863223, 7429557, 13927075, 10476187, 61609055, 24856941, 196311627, 396464561, 406039103, 1751221075},
  {1, 3, 7, 13, 31, 27, 57, 181, 325, 107, 1745, 635, 3941, 3305, 14563, 29855, 60037, 29095, 446891, 465657, 1742155, 359271, 3023261, 8788145, 33331401, 2689543, 37702845, 34034671, 92924137, 133082365, 651855999},
  {1, 3, 7, 15, 5, 55, 5, 147, 365, 485, 1841, 3673, 6513, 11121, 5725, 18027, 55485, 243463, 194539, 571323, 1405649, 1075523, 7572441, 882903, 12581073, 64067761, 78702733, 251706581, 139316877, 805723021, 988326017},
  {1, 3, 5, 11, 13, 45, 35, 79, 109, 683, 1171, 3015, 2163, 4823, 4365, 42931, 60493, 178535, 174697, 438335, 1006545, 1949017, 2082783, 5622883, 11560345, 7232543, 110004143, 39880499, 272532471, 922721699, 173190921},
  {1, 1, 7, 7, 13, 45, 57, 39, 297, 985, 1559, 487, 5071, 2657, 9401, 41889, 5677, 83085, 510075, 362779, 1974729, 1435097, 275077, 5407819, 12051765, 58951197, 5485787, 72957051, 486208267, 937523605, 263878669},
  {1, 1, 5, 9, 29, 33, 79, 237, 509, 537, 549, 3657, 7141, 15189, 30853, 36089, 44285, 138381, 59513, 764309, 1401297, 3403517, 6197731, 12455889, 29504217, 52378117, 64674441, 121715413, 480700457, 772369673, 819477449},
  {1, 3, 5, 7, 31, 15, 75, 73, 237, 273, 865, 743, 2607, 7611, 18441, 12703, 123069, 202487, 183497, 24563, 1212867, 2813259, 2246303, 13189565, 5851017, 43127005, 23769453, 59937819, 291304883, 241994495, 1561836381},
  {1, 1, 1, 9, 27, 9, 35, 137, 265, 181, 1341, 1945, 5615, 161, 18369, 4791, 123981, 126869, 457293, 308597, 1471175, 2683765, 5040855, 762149, 20142013, 32260825, 56215073, 152715605, 308957539, 966521021, 1972963493},
  {1, 3, 7, 11, 27, 29, 29, 43, 489, 177, 1489, 2927, 623, 14571, 22447, 46905, 81061, 217951, 10651, 304543, 573519, 215033, 1303449, 1858423, 408453, 56320357, 15524029, 170158995, 143174387, 10567567, 2065526971},
  {1, 3, 3, 3, 19, 41, 107, 53, 239, 263, 1433, 1655, 7991, 7405, 2735, 25519, 72853, 43271, 82927, 165111, 62295, 3445325, 5957839, 11703673, 22243195, 4071515, 14901629, 77570179, 522925275, 790523465, 1057268763},
  {1, 3, 5, 7, 19, 37, 73, 243, 215, 445, 1781, 3223, 187, 8443, 18185, 45093, 81341, 74783, 174105, 514131, 361935, 1763297, 3811245, 10807247, 14263851, 32551977, 35968737, 163189291, 518193719, 46336415, 446766413},
  {1, 1, 3, 13, 9, 57, 111, 111, 221, 505, 1261, 3045, 1655, 16247, 21033, 17993, 39773, 139821, 372223, 966801, 1753565, 2455237, 82899, 9868755, 1374585, 42732397, 126062033, 183182297, 487010715, 379684187, 1174928845},
  {1, 1, 7, 5, 7, 55, 77, 5, 131, 969, 1481, 2883, 2645, 3003, 5601, 37063, 35101, 56277, 33803, 639097, 1535427, 2379, 310473, 9655153, 3649599, 6787333, 58128253, 84728687, 47625417, 774395055, 1406911869},
  {1, 1, 5, 15, 29, 55, 39, 197, 349, 29, 341, 67, 1977, 425, 4063, 42705, 87157, 147069, 424473, 394515, 405329, 1232883, 8337851, 1776065, 31014249, 12616961, 80515513, 36093647, 207404181, 849285661, 764481715},
  {1, 1, 7, 13, 1, 57, 81, 81, 129, 681, 1407, 2465, 3627, 2325, 31649, 18449, 116461, 123405, 188667, 956689, 1193669, 1853669, 8386445, 16374189, 6452317, 61593397, 10544419, 41739741, 494465871, 220299065, 1951032373},
  {1, 3, 5, 5, 23, 59, 35, 217, 393, 155, 1315, 105, 2285, 5167, 27997, 55193, 20813, 65359, 341561, 772625, 1942091, 561527, 1443399, 2613789, 32471789, 13066775, 51848255, 99739421, 404021993, 743198051, 1593650129},
  {1, 1, 7, 3, 11, 59, 53, 179, 319, 819, 947, 3881, 765, 4219, 16405, 48055, 19149, 207285, 452939, 431359, 1406031, 717535, 5411377, 7927415, 27265859, 58463527, 1322311, 43277805, 248817945, 847084727, 880710745},
  {1, 1, 3, 9, 23, 9, 67, 67, 137, 523, 1585, 2441, 167, 5217, 12031, 14297, 32565, 14429, 33119, 989461, 1991363, 655717, 6522919, 3635215, 9881725, 64043983, 109547517, 149842845, 14268523, 148310973, 1059408251},
  {1, 1, 5, 13, 31, 63, 121, 91, 439, 917, 203, 1519, 4363, 2391, 25755, 26677, 48789, 44109, 68793, 633809, 1784787, 3077235, 2967677, 9447631, 10324251, 47917441, 61512863, 154629235, 18985919, 322547491, 1732179647},
  {1, 1, 3, 5, 25, 31, 11, 95, 339, 817, 35, 3923, 7365, 10537, 5521, 54579, 52557, 251653, 88687, 666809, 1407821, 2585539, 947255, 15009019, 5006551, 23670437, 53007719, 258408255, 119475761, 859780525, 1143414509},
  {1, 3, 7, 1, 3, 33, 47, 13, 139, 445, 1357, 3907, 7495, 8789, 26589, 43479, 51997, 103159, 302987, 108277, 1258327, 2910701, 4934427, 10950977, 7153615, 62518177, 28390457, 131996287, 423950315, 154507937, 1503454153},
  {1, 1, 1, 11, 5, 45, 61, 13, 167, 287, 931, 881, 5713, 12937, 12951, 21597, 110165, 239373, 320637, 746839, 248281, 1864537, 258337, 4933473, 303659, 46351315, 87507143, 114507941, 167036437, 10377189, 1040336043},
  {1, 3, 5, 1, 29, 23, 7, 117, 503, 897, 733, 1113, 7205, 11507, 561, 34011, 125853, 145191, 484457, 535349, 631873, 3849931, 4362091, 14467345, 2433731, 25257565, 26029721, 150468725, 411571945, 141049647, 1632109949},
  {1, 3, 5, 7, 3, 51, 21, 147, 35, 259, 689, 3801, 2481, 9673, 4065, 595, 66317, 77703, 363369, 414131, 1470559, 1969399, 7027505, 15833591, 30958495, 16651287, 20609077, 74942269, 104566045, 767016661, 1196866365},
  {1, 3, 3, 9, 9, 29, 5, 191, 393, 979, 1627, 937, 75, 2339, 24007, 30555, 24133, 204183, 262287, 441917, 757709, 3122385, 6306585, 8071123, 17025061, 64924055, 28347543, 112082205, 163569519, 678744751, 967539539},
  {1, 1, 5, 7, 9, 35, 111, 23, 113, 563, 1689, 1575, 6127, 9919, 2555, 52261, 23253, 82701, 390457, 708571, 25797, 3316551, 378723, 12999139, 31242245, 44506087, 18214485, 43805651, 207069635, 577865499, 157214127},
  {1, 3, 1, 5, 31, 21, 117, 159, 473, 279, 1281, 311, 159, 3343, 27761, 59989, 94589, 224927, 430109, 525521, 1780307, 3113849, 7739761, 3055011, 5583893, 28485587, 84402957, 143928851, 141338611, 382318187, 1241884901},
  {1, 1, 5, 1, 23, 31, 67, 241, 401, 69, 933, 777, 267, 12411, 23767, 9047, 54341, 198693, 95977, 622333, 1339355, 134115, 7065351, 3448677, 11283797, 20327721, 119939737, 176333637, 180070247, 404480615, 2145345955},
  {1, 1, 5, 1, 15, 49, 99, 15, 267, 913, 1581, 3713, 651, 14275, 10103, 57619, 16645, 31285, 312777, 107197, 442307, 570061, 354119, 13321187, 10286527, 51398373, 80903817, 238635309, 256925127, 283335551, 102559299},
  {1, 3, 5, 9, 19, 23, 95, 5, 449, 153, 1195, 1315, 2347, 12683, 10865, 50703, 33277, 28375, 460521, 653981, 1354991, 1630859, 5225963, 2306161, 17281429, 16626661, 45092887, 119536511, 6346879, 411201135, 213642453},
  {1, 1, 1, 13, 17, 17, 115, 31, 135, 725, 795, 1695, 6199, 4179, 5223, 48457, 8421, 192765, 181181, 239665, 1130541, 2408861, 2129647, 8294563, 15503859, 42979969, 109056623, 22839011, 102461923, 849307335, 1470649555},
  {1, 3, 5, 15, 31, 15, 3, 247, 385, 269, 1665, 581, 2809, 6333, 7459, 14815, 76229, 327, 192073, 442203, 1194211, 3868555, 2900407, 10526947, 33431653, 61842265, 38084973, 99150569, 339048933, 121185489, 1863808735},
  {1, 3, 7, 5, 15, 35, 117, 85, 11, 621, 1523, 981, 511, 14113, 4673, 22683, 130733, 162695, 38475, 411473, 588411, 1003167, 6422513, 6276241, 17679639, 6237377, 63795911, 224287305, 243364555, 457412317, 634035973},
  {1, 1, 7, 1, 27, 61, 119, 249, 431, 147, 173, 423, 1353, 4747, 12761, 32947, 50653, 101157, 357771, 545629, 640895, 2578609, 7701955, 13388453, 17034891, 22348287, 86180745, 35982435, 41884213, 550576175, 441923573},
  {1, 3, 7, 1, 23, 39, 15, 153, 219, 359, 1233, 169, 2817, 11043, 12435, 30135, 8309, 32871, 297483, 151125, 1911779, 401803, 6550475, 12817101, 27495183, 5062251, 3569125, 142127645, 475041901, 271228831, 1502140079},
  {1, 1, 5, 1, 1, 55, 39, 1, 151, 865, 125, 2351, 6315, 1165, 20163, 43991, 78981, 136741, 160521, 314077, 810477, 2185643, 1967067, 16657125, 19532547, 22154637, 126525497, 211905123, 42198663, 579842729, 177849447},
  {1, 1, 3, 9, 25, 41, 115, 221, 129, 265, 1887, 4057, 7523, 13591, 5735, 59645, 11269, 203437, 209695, 773237, 399533, 3693477, 1853135, 2984641, 29548085, 46319573, 109617955, 9085637, 149680487, 330727355, 958427507},
  {1, 1, 5, 5, 19, 15, 9, 77, 511, 627, 153, 2015, 247, 15949, 9715, 45411, 102725, 243701, 478889, 697497, 2059903, 2323075, 2244765, 8241049, 28897715, 9589279, 121287677, 37992603, 119063515, 522512041, 1401774247},
  {1, 1, 7, 7, 17, 17, 107, 183, 39, 647, 1339, 3915, 4937, 537, 27011, 58937, 103589, 108253, 280443, 629051, 264501, 48773, 7721415, 4276587, 16350579, 64804531, 94438415, 258542079, 494600845, 638518013, 1870473519},
  {1, 3, 3, 13, 3, 3, 69, 201, 431, 65, 683, 121, 7017, 2791, 16713, 62555, 5693, 107271, 336335, 932313, 1991015, 2981311, 1357553, 14930925, 22955443, 40210333, 47503343, 79853749, 79400605, 373066283, 61182005},
  {1, 3, 7, 3, 7, 41, 117, 237, 23, 757, 545, 3899, 1837, 5555, 7891, 29151, 23733, 12103, 287051, 859095, 32755, 1555085, 7806177, 8987937, 17167899, 14402073, 130697189, 167234215, 473551185, 740608103, 1885214351},
  {1, 1, 1, 3, 27, 15, 39, 195, 353, 299, 1431, 2279, 1795, 13773, 24915, 49701, 2245, 4973, 455325, 868607, 1155239, 1350843, 5814755, 11673527, 29449173, 588559, 106860315, 266488595, 422692087, 32139001, 22038263},
  {1, 1, 5, 5, 7, 7, 125, 5, 401, 523, 1967, 2471, 7279, 7559, 12025, 60599, 93245, 153077, 145577, 121881, 861931, 3282827, 5123769, 10192209, 14062349, 45894887, 87066483, 206867059, 279482811, 830453619, 537314413},
  {1, 1, 1, 13, 13, 59, 13, 249, 465, 847, 1483, 975, 7729, 2773, 15745, 51237, 122445, 226237, 25533, 735665, 214193, 1739191, 3356641, 1750797, 18937549, 58742099, 109212535, 95466443, 406564821, 655263457, 510455277},
  {1, 1, 7, 9, 31, 41, 115, 141, 247, 355, 1109, 3239, 6495, 4515, 30145, 47705, 84861, 244237, 23771, 32885, 83387, 2534405, 6695959, 14089377, 16577699, 5330367, 35907057, 28552227, 124166595, 97193287, 263798965},
  {1, 1, 3, 13, 29, 41, 69, 179, 45, 271, 1909, 3095, 7199, 14049, 9903, 33383, 116437, 210821, 6607, 61649, 979305, 3069493, 5814857, 7464527, 13764097, 40014555, 124481833, 216005843, 436721411, 896385605, 2003911387},
  {1, 1, 3, 13, 7, 45, 105, 105, 243, 121, 67, 3169, 237, 137, 4175, 54325, 43989, 205965, 399711, 559601, 673075, 4077105, 6503949, 1516117, 30175215, 12891813, 374263, 234820525, 254051897, 67600613, 33197539},
  {1, 3, 3, 11, 19, 51, 93, 155, 79, 579, 1621, 1251, 7639, 15875, 25815, 56063, 37285, 106023, 292239, 1009631, 1551991, 3370775, 4385561, 16208671, 23960051, 47724695, 32668417, 134393479, 472689163, 775158263, 909458595},
  {1, 3, 3, 9, 27, 27, 77, 45, 217, 965, 1045, 875, 4515, 11261, 27859, 757, 88645, 177839, 197023, 18557, 1069119, 2396471, 8101689, 8503657, 24365221, 30161097, 57985705, 263389383, 458953967, 306520713, 2124090871},
  {1, 1, 3, 11, 17, 7, 81, 37, 299, 765, 977, 3371, 3163, 13267, 18345, 9239, 51421, 146421, 431407, 492183, 2085733, 1226499, 8084829, 16040801, 26648871, 61481905, 43655333, 140340471, 457454567, 214430791, 842353709},
  {1, 1, 1, 3, 15, 23, 115, 11, 183, 511, 557, 3253, 153, 8465, 22659, 42143, 84133, 45253, 458701, 615391, 98163, 3625251, 2736999, 6938431, 9004115, 5113819, 20187257, 94379809, 403193541, 659056829, 1797817727},
  {1, 1, 5, 13, 17, 61, 127, 219, 225, 981, 615, 731, 4069, 12637, 11601, 38257, 23373, 69997, 105369, 270449, 578749, 3552529, 456995, 4778455, 26408261, 32978865, 27657835, 230818263, 418515105, 233701753, 1445356493},
  {1, 1, 5, 3, 29, 3, 73, 79, 393, 779, 823, 2473, 3811, 4417, 9399, 50011, 73589, 169525, 137897, 713503, 1475057, 199447, 5628245, 13719867, 3474949, 62486767, 34315419, 242836765, 436860951, 1002643149, 127498083},
  {1, 3, 3, 9, 21, 35, 61, 99, 115, 657, 629, 1129, 2355, 12665, 459, 40831, 85061, 116807, 226895, 602333, 1587569, 2732815, 6839665, 16545207, 28065519, 15027045, 101829137, 29776677, 269021247, 206935301, 1259311167},
  {1, 3, 1, 7, 25, 61, 53, 249, 15, 649, 665, 595, 1441, 8035, 5381, 7147, 120045, 172071, 10765, 945875, 1346293, 3046713, 5855001, 4447637, 9617219, 59892021, 121972381, 151217215, 182793053, 161236119, 175277737},
  {1, 3, 1, 7, 19, 9, 27, 207, 205, 461, 1069, 4039, 6549, 2333, 29, 50067, 43517, 196999, 47181, 649683, 130815, 343821, 7536415, 6676771, 22123065, 51770841, 59587313, 229523467, 359158649, 554690841, 1548508673},
  {1, 1, 5, 3, 15, 7, 115, 205, 71, 73, 53, 71, 6049, 15293, 17041, 20313, 115405, 244813, 435929, 419071, 1552291, 1249043, 4922151, 11610937, 3935779, 53913205, 63796113, 57703659, 176498157, 943428217, 2006164829},
  {1, 1, 7, 7, 9, 7, 119, 99, 357, 729, 2041, 3355, 5333, 1263, 30521, 64867, 9917, 200741, 416075, 76187, 591405, 3236691, 32763, 5446759, 3147185, 16822269, 121738021, 245642439, 165533065, 757621051, 2009765061},
  {1, 1, 1, 7, 31, 63, 63, 159, 281, 295, 913, 2161, 8033, 13789, 17711, 14915, 65589, 253973, 418093, 295771, 145315, 2490971, 7633803, 10380299, 15510989, 53927923, 95168349, 45510261, 414576173, 956185985, 900084499},
  {1, 1, 7, 9, 29, 55, 69, 129, 453, 361, 1883, 17, 1765, 111, 10311, 55019, 81541, 80245, 147947, 1022677, 669881, 1959515, 2204649, 8152069, 10523865, 44844117, 53893431, 226174061, 407151217, 24275459, 549946891},
  {1, 1, 5, 9, 21, 15, 31, 57, 291, 915, 945, 1775, 5905, 9073, 3271, 15571, 64821, 45205, 252969, 564053, 943033, 35443, 399283, 2734173, 776343, 50063647, 26586917, 173658251, 499984413, 668197333, 1054081739},
  {1, 1, 1, 7, 21, 11, 1, 9, 167, 143, 1535, 1267, 6675, 13037, 19513, 52027, 21549, 228741, 511501, 431387, 1202985, 2312559, 6530461, 11021853, 8686475, 41217523, 78165659, 37625095, 466631943, 24925705, 463786685},
  {1, 3, 3, 7, 31, 45, 57, 105, 63, 121, 631, 679, 3873, 16333, 32069, 64725, 90205, 96927, 373823, 281971, 1241595, 522489, 4948717, 12990069, 33380323, 41371493, 58876339, 254504395, 432894245, 498885209, 2018394601},
  {1, 1, 1, 9, 23, 41, 29, 207, 489, 319, 905, 3147, 4195, 2697, 5281, 1771, 64493, 121349, 415501, 760597, 1267499, 2383221, 275961, 10303139, 8798789, 27674203, 45894117, 245553007, 432401679, 130614909, 962675653},
  {1, 1, 1, 9, 25, 33, 57, 201, 405, 111, 407, 3489, 449, 8601, 1273, 42105, 94413, 254029, 500125, 984117, 889701, 3171837, 8366309, 3732213, 31846921, 49716083, 53452539, 159004101, 205615189, 517458741, 2032828837},
  {1, 1, 1, 3, 19, 45, 123, 159, 237, 173, 781, 787, 7537, 15453, 25567, 53729, 20917, 36509, 418621, 96383, 86767, 3472345, 1604063, 6433123, 31005945, 35497209, 112293393, 149788063, 29148765, 27481161, 2106865715},
  {1, 1, 7, 3, 29, 9, 89, 207, 345, 685, 1701, 2859, 8065, 9289, 2459, 28367, 59653, 71685, 87563, 47647, 1916473, 3922509, 3791525, 16584467, 2278789, 11290153, 39308489, 246210375, 116946893, 240660093, 1193821687},
  {1, 3, 1, 3, 31, 41, 55, 241, 81, 1001, 595, 1725, 853, 11427, 20617, 1717, 26973, 63103, 476925, 275447, 1033715, 2271741, 544227, 5009917, 15548565, 61588389, 27051407, 10175649, 392874017, 918478527, 1943055333},
  {1, 1, 3, 3, 9, 45, 121, 69, 177, 1017, 211, 2753, 6923, 1387, 32063, 45337, 124997, 149437, 378751, 1042175, 222717, 570313, 3248021, 321321, 23984013, 50705557, 90801359, 40535325, 284463103, 947560327, 2121120971},
  {1, 1, 5, 15, 21, 23, 95, 61, 485, 403, 619, 3035, 4881, 13423, 17815, 35221, 15237, 251661, 63001, 447539, 172153, 1468659, 5765571, 789337, 18267729, 42989071, 92041927, 205280463, 168254613, 98950155, 410030363},
  {1, 1, 3, 3, 3, 59, 23, 69, 77, 309, 227, 2877, 3739, 3539, 20083, 23415, 42805, 140645, 68303, 611743, 453431, 1775199, 781523, 9443617, 31355609, 1451273, 119540143, 246242673, 195289487, 173522703, 716558759},
  {1, 3, 7, 3, 17, 43, 95, 239, 223, 353, 1237, 3239, 1369, 7245, 32401, 63889, 118925, 91951, 93787, 294071, 1726821, 1839823, 5195219, 7400283, 22279883, 26382069, 30849, 55447947, 505531477, 699391177, 480561917},
  {1, 1, 1, 5, 25, 63, 123, 3, 291, 819, 793, 241, 5619, 8871, 18341, 2757, 72397, 60701, 386621, 274553, 79077, 2858451, 5269447, 2900871, 13640647, 47719159, 72491037, 28536373, 489152543, 786103995, 1378020433},
  {1, 3, 7, 15, 3, 21, 17, 97, 395, 333, 909, 3783, 3635, 751, 24227, 44281, 78677, 108127, 100539, 957819, 1672951, 1038785, 829365, 15862653, 26395639, 37771313, 86936409, 185003611, 233289023, 679484395, 1350594663},
  {1, 3, 7, 13, 29, 49, 123, 195, 191, 159, 211, 1887, 3047, 4871, 2607, 32425, 86053, 186751, 201595, 1033081, 1100649, 3686253, 6853687, 11932703, 14234987, 19886011, 1175271, 150768587, 150242563, 889236019, 1286320731},
  {1, 1, 7, 7, 15, 57, 91, 255, 267, 897, 441, 1581, 953, 6951, 17275, 50229, 9141, 83949, 44891, 887867, 1318379, 4073581, 3172431, 16566451, 3422335, 48552213, 103983933, 113943201, 4483269, 844624107, 21794183},
  {1, 3, 7, 15, 1, 35, 91, 219, 7, 117, 119, 2687, 1215, 9517, 10849, 28347, 46157, 222791, 498955, 436123, 1938357, 388599, 7608439, 7395471, 13158467, 35891217, 27606819, 49515907, 279089947, 1026852377, 2132384629},
  {1, 1, 1, 1, 21, 55, 67, 221, 503, 883, 1037, 2965, 1485, 8557, 28091, 25459, 485, 136821, 425069, 747613, 1775657, 2065355, 6287471, 12431577, 25866355, 67023759, 15585961, 188977553, 345926257, 176988609, 547276119},
  {1, 3, 5, 9, 19, 3, 73, 123, 95, 307, 1339, 3669, 5077, 12049, 523, 21457, 98629, 6527, 111225, 451709, 808559, 182111, 6783037, 14778399, 20676659, 29328007, 73435055, 2131505, 142513777, 302758285, 2096029159},
  {1, 3, 1, 13, 3, 1, 111, 97, 371, 697, 1989, 3715, 7875, 6841, 7009, 17809, 18413, 61807, 272221, 627513, 1173231, 3643757, 2141755, 3380813, 24385303, 35032437, 60357905, 43618895, 8340175, 209799469, 1988515269},
  {1, 1, 1, 9, 25, 21, 19, 43, 329, 531, 491, 1147, 1469, 12841, 29651, 29517, 91173, 118093, 461597, 285877, 1878501, 2088521, 6334031, 3211847, 19388253, 20937751, 111414855, 86838767, 432740793, 785592397, 67194479},
  {1, 1, 5, 1, 15, 63, 101, 197, 477, 245, 341, 61, 3803, 10001, 5519, 19083, 64725, 226037, 464745, 102237, 793123, 1531875, 6524513, 4430513, 15985185, 60900505, 96074873, 44287025, 70340119, 141777661, 1022548411},
  {1, 3, 7, 5, 13, 43, 7, 143, 291, 531, 1727, 871, 7091, 5737, 24285, 51017, 60573, 52719, 306267, 75057, 862969, 3698519, 3767883, 1769019, 3934167, 10452039, 88443419, 209038051, 70729159, 879074973, 2084561017},
  {1, 3, 5, 1, 17, 13, 15, 85, 361, 153, 989, 1367, 4705, 3599, 4441, 52471, 33341, 211063, 220009, 1621, 428845, 2702065, 5213747, 1409689, 311117, 55920901, 55132097, 208562571, 486358061, 687922763, 2054698197},
  {1, 1, 7, 13, 21, 43, 111, 29, 299, 439, 1929, 283, 5901, 14113, 3929, 55843, 28877, 97189, 148171, 699153, 1027761, 256471, 6597187, 11930217, 23019263, 1221611, 129050397, 58467951, 257782849, 1000651861, 1544136965},
  {1, 3, 3, 9, 31, 59, 63, 43, 91, 171, 733, 1359, 425, 8505, 19777, 54723, 37101, 127591, 128591, 909085, 1483707, 2311895, 3360027, 5195535, 32786543, 63736047, 18979513, 45871019, 493332373, 298511101, 691083501},
  {1, 1, 5, 7, 31, 1, 97, 253, 331, 307, 1749, 2115, 2535, 9945, 11013, 14231, 110461, 83413, 250153, 966875, 1815603, 2766277, 3370773, 8611217, 2415511, 45925631, 24051425, 119022695, 437347707, 694951789, 1090277489},
  {1, 1, 5, 11, 13, 7, 109, 237, 301, 383, 683, 1603, 6393, 2437, 12101, 1767, 69405, 166613, 75049, 651223, 1146401, 1750515, 402225, 2531785, 10903377, 42101171, 63306503, 63447399, 219315869, 955062721, 1210919849},
  {1, 1, 3, 11, 15, 61, 119, 199, 109, 265, 1431, 1729, 3409, 10129, 16945, 34681, 74285, 172685, 267423, 215735, 1037051, 3443321, 3080003, 10860507, 14823945, 36248157, 26892299, 17856861, 388772277, 69646301, 393406317},
  {1, 3, 7, 9, 13, 59, 121, 73, 29, 513, 279, 457, 7985, 15199, 10185, 33621, 15037, 11263, 240251, 341885, 329337, 80247, 6271829, 7322493, 29700881, 45994429, 61960603, 45513557, 429880021, 352220211, 1435655253},
  {1, 3, 7, 7, 23, 17, 27, 65, 387, 487, 1803, 2789, 461, 11201, 7001, 40229, 92093, 133711, 2635, 368259, 1636499, 1021109, 6511359, 9885245, 23170791, 37506491, 76827279, 28868833, 48402521, 224313229, 558411205},
  {1, 1, 3, 15, 9, 13, 55, 127, 33, 513, 1055, 643, 505, 3005, 6121, 64147, 38365, 161493, 370047, 668803, 1742477, 740521, 168427, 11134595, 30786069, 8828357, 9248483, 49964303, 373308685, 868707529, 1318428053},
  {1, 3, 5, 15, 5, 11, 77, 233, 175, 691, 171, 2491, 6915, 14195, 7845, 36499, 54589, 60503, 222777, 454027, 1848713, 1334943, 337313, 9056877, 23322179, 42021887, 25449695, 185024143, 66963383, 115034183, 1704840689},
  {1, 1, 5, 11, 19, 45, 103, 207, 99, 645, 1739, 1517, 5907, 16035, 14559, 44007, 11093, 36693, 304025, 89479, 1392015, 367529, 7667699, 2745955, 10655255, 18780673, 71554751, 34398049, 517466263, 300006095, 1292732027},
  {1, 3, 7, 15, 21, 27, 53, 107, 89, 291, 983, 3527, 1025, 2985, 13747, 32715, 116629, 257063, 165691, 996491, 1720465, 2395551, 7596441, 14079535, 27084757, 66943143, 13252715, 37324547, 342591941, 304261773, 1775546383},
  {1, 1, 3, 15, 23, 17, 27, 209, 77, 959, 813, 3597, 5809, 693, 10325, 16855, 89069, 103077, 325727, 261251, 1975955, 353973, 407487, 16128685, 20340265, 50719651, 1505553, 101590809, 196376957, 1023621225, 1113264249},
  {1, 1, 7, 11, 23, 53, 123, 99, 211, 935, 1371, 1657, 4699, 2683, 27933, 21451, 75293, 128117, 320507, 719239, 244099, 1516193, 5673471, 7621263, 24479255, 4352115, 93969039, 172254205, 476397199, 927625927, 957674553},
  {1, 3, 3, 3, 17, 21, 93, 201, 423, 351, 1639, 227, 5719, 13111, 5993, 44615, 99181, 184103, 34591, 32135, 558469, 246401, 3585409, 4078741, 5752315, 48889107, 45699891, 265534503, 106861491, 287936771, 1842403757},
  {1, 1, 7, 3, 13, 49, 59, 255, 213, 147, 1441, 3593, 6419, 15657, 1947, 62713, 81941, 122797, 77867, 371599, 1992345, 3689349, 5459607, 16391379, 6554929, 31153463, 95468685, 239300717, 141883279, 527430021, 1005441055},
  {1, 3, 1, 7, 7, 41, 79, 135, 275, 585, 925, 3139, 4351, 1827, 23533, 63031, 62301, 231303, 275485, 831875, 1209243, 2700477, 3903979, 7481499, 21612775, 22182573, 75810921, 6092415, 241455171, 458159431, 1503285641},
  {1, 1, 7, 3, 11, 1, 13, 149, 29, 897, 1043, 2393, 3931, 6741, 19973, 46303, 45581, 235269, 19867, 231567, 1254815, 1473461, 5652153, 4982089, 8635297, 55905597, 95551567, 202771253, 318902943, 499515945, 1803178017},
  {1, 1, 1, 13, 13, 57, 9, 253, 149, 67, 1531, 4049, 3013,