"use strict";
// Copyright Amazon.com Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.modifyDencryptionMaterial = exports.unwrapEncryptedDataKey = exports.destructureCiphertext = exports.filterEdk = exports.modifyEncryptionMaterial = exports.wrapAad = exports.wrapPlaintextDataKey = exports.getPlaintextDataKey = exports.getBranchKeyMaterials = exports.getCacheEntryId = exports.getBranchKeyId = exports.serializeEncryptionContext = exports.decompressBytesToUuidv4 = exports.uuidv4ToCompressedBytes = exports.utf8BytesToString = exports.stringToUtf8Bytes = void 0;
const material_management_1 = require("@aws-crypto/material-management");
const crypto_1 = require("crypto");
const kdf_ctr_mode_node_1 = require("@aws-crypto/kdf-ctr-mode-node");
const constants_1 = require("./constants");
const serialize_1 = require("@aws-crypto/serialize");
const stringToUtf8Bytes = (input) => Buffer.from(input, 'utf-8');
exports.stringToUtf8Bytes = stringToUtf8Bytes;
const utf8BytesToString = (input) => input.toString('utf-8');
exports.utf8BytesToString = utf8BytesToString;
const stringToHexBytes = (input) => new Uint8Array(Buffer.from(input, 'hex'));
const hexBytesToString = (input) => Buffer.from(input).toString('hex');
_a = (0, serialize_1.uuidv4Factory)(stringToHexBytes, hexBytesToString), exports.uuidv4ToCompressedBytes = _a.uuidv4ToCompressedBytes, exports.decompressBytesToUuidv4 = _a.decompressBytesToUuidv4;
exports.serializeEncryptionContext = (0, serialize_1.serializeFactory)(exports.stringToUtf8Bytes).serializeEncryptionContext;
function getBranchKeyId({ branchKeyId, branchKeyIdSupplier }, { encryptionContext }) {
    // use the branch key id attribute if it was set, otherwise use the branch key
    // id supplier. The constructor ensures that either the branch key id or
    // supplier is supplied to the keyring
    return (branchKeyId ||
        branchKeyIdSupplier.getBranchKeyId(encryptionContext));
}
exports.getBranchKeyId = getBranchKeyId;
const RESOURCE_ID = new Uint8Array([0x02]);
const NULL_BYTE = new Uint8Array([0x00]);
const DECRYPTION_SCOPE = new Uint8Array([0x02]);
const ENCRYPTION_SCOPE = new Uint8Array([0x01]);
//= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#appendix-a-cache-entry-identifier-formulas
//# When accessing the underlying cryptographic materials cache,
//# the hierarchical keyring MUST use the formulas specified in this appendix
//# in order to compute the [cache entry identifier](../cryptographic-materials-cache.md#cache-identifier).
function getCacheEntryId(logicalKeyStoreName, partitionId, branchKeyId, versionAsBytes) {
    // get branch key id as a byte array
    const branchKeyIdAsBytes = (0, exports.stringToUtf8Bytes)(branchKeyId);
    let entryInfo;
    //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#resource-suffix
    //# The aforementioned 4 definitions ([Resource Identifier](#resource-identifier),
    //# [Scope Identifier](#scope-identifier), [Partition ID](#partition-id-1), and
    //# [Resource Suffix](#resource-suffix)) MUST be appended together with the null byte, 0x00,
    //# and the SHA384 of the result should be taken as the final cache identifier.
    if (versionAsBytes) {
        //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#decryption-materials
        //# When the hierarchical keyring receives an OnDecrypt request,
        //# it MUST calculate the cache entry identifier as the
        //# SHA-384 hash of the following byte strings, in the order listed:
        //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#decryption-materials
        //# All the above fields must be separated by a single NULL_BYTE `0x00`.
        //#
        //# | Field                  | Length (bytes) | Interpreted as      |
        //# | ---------------------- | -------------- | ------------------- |
        //# | Resource ID            | 1              | bytes               |
        //# | Null Byte              | 1              | `0x00`              |
        //# | Scope ID               | 1              | bytes               |
        //# | Null Byte              | 1              | `0x00`              |
        //# | Partition ID           | Variable       | bytes               |
        //# | Null Byte              | 1              | `0x00`              |
        //# | Logical Key Store Name | Variable       | UTF-8 Encoded Bytes |
        //# | Null Byte              | 1              | `0x00`              |
        //# | Branch Key ID          | Variable       | UTF-8 Encoded Bytes |
        //# | Null Byte              | 1              | `0x00`              |
        //# | branch-key-version     | 36             | UTF-8 Encoded Bytes |
        entryInfo = Buffer.concat([
            //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#decryption-materials
            //# - MUST be the Resource ID for the Hierarchical Keyring (0x02)
            RESOURCE_ID,
            NULL_BYTE,
            //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#decryption-materials
            //# - MUST be the Scope ID for Decrypt (0x02)
            DECRYPTION_SCOPE,
            NULL_BYTE,
            //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#decryption-materials
            //# - MUST be the Partition ID for the Hierarchical Keyring
            partitionId,
            NULL_BYTE,
            //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#decryption-materials
            //# - MUST be the UTF8 encoded Logical Key Store Name of the keystore for the Hierarchical Keyring
            logicalKeyStoreName,
            NULL_BYTE,
            //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#decryption-materials
            //# - MUST be the UTF8 encoded branch-key-id
            branchKeyIdAsBytes,
            NULL_BYTE,
            //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#decryption-materials
            //# - MUST be the UTF8 encoded branch-key-version
            versionAsBytes,
        ]);
    }
    else {
        //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#encryption-materials
        //# When the hierarchical keyring receives an OnEncrypt request,
        //# the cache entry identifier MUST be calculated as the
        //# SHA-384 hash of the following byte strings, in the order listed:
        //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#encryption-materials
        //# All the above fields must be separated by a single NULL_BYTE `0x00`.
        //#
        //# | Field                  | Length (bytes) | Interpreted as      |
        //# | ---------------------- | -------------- | ------------------- |
        //# | Resource ID            | 1              | bytes               |
        //# | Null Byte              | 1              | `0x00`              |
        //# | Scope ID               | 1              | bytes               |
        //# | Null Byte              | 1              | `0x00`              |
        //# | Partition ID           | Variable       | bytes               |
        //# | Null Byte              | 1              | `0x00`              |
        //# | Logical Key Store Name | Variable       | UTF-8 Encoded Bytes |
        //# | Null Byte              | 1              | `0x00`              |
        //# | Branch Key ID          | Variable       | UTF-8 Encoded Bytes |
        entryInfo = Buffer.concat([
            //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#encryption-materials
            //# - MUST be the Resource ID for the Hierarchical Keyring (0x02)
            RESOURCE_ID,
            NULL_BYTE,
            //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#encryption-materials
            //# - MUST be the Scope ID for Encrypt (0x01)
            ENCRYPTION_SCOPE,
            NULL_BYTE,
            //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#encryption-materials
            //# - MUST be the Partition ID for the Hierarchical Keyring
            partitionId,
            NULL_BYTE,
            //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#encryption-materials
            //# - MUST be the UTF8 encoded Logical Key Store Name of the keystore for the Hierarchical Keyring
            logicalKeyStoreName,
            NULL_BYTE,
            //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#encryption-materials
            //# - MUST be the UTF8 encoded branch-key-id
            branchKeyIdAsBytes,
        ]);
    }
    // hash the branch key id buffer with sha512
    return (0, crypto_1.createHash)(constants_1.CACHE_ENTRY_ID_DIGEST_ALGORITHM)
        .update(entryInfo)
        .digest()
        .toString();
}
exports.getCacheEntryId = getCacheEntryId;
async function getBranchKeyMaterials(hKeyring, cmc, branchKeyId, cacheEntryId, branchKeyVersion) {
    const { keyStore, cacheLimitTtl } = hKeyring;
    //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#onencrypt
    //# The hierarchical keyring MUST attempt to find [branch key materials](../structures.md#branch-key-materials)
    //# from the underlying [cryptographic materials cache](../local-cryptographic-materials-cache.md).
    const cacheEntry = cmc.getBranchKeyMaterial(cacheEntryId);
    let branchKeyMaterials;
    // if the cache entry is false, branch key materials were not found
    if (!cacheEntry || hKeyring.cacheEntryHasExceededLimits(cacheEntry)) {
        //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#onencrypt
        //# If this is NOT true, then we MUST treat the cache entry as expired.
        //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#ondecrypt
        //# If this is NOT true, then we MUST treat the cache entry as expired.
        //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#onencrypt
        //# If a cache entry is not found or the cache entry is expired, the hierarchical keyring MUST attempt to obtain the branch key materials
        //# by querying the backing branch keystore specified in the [retrieve OnEncrypt branch key materials](#query-branch-keystore-onencrypt) section.
        //# If the keyring is not able to retrieve [branch key materials](../structures.md#branch-key-materials)
        //# through the underlying cryptographic materials cache or
        //# it no longer has access to them through the backing keystore, OnEncrypt MUST fail.
        //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#query-branch-keystore-onencrypt
        //# Otherwise, OnEncrypt MUST fail.
        //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#getitem-branch-keystore-ondecrypt
        //# Otherwise, OnDecrypt MUST fail.
        //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#query-branch-keystore-onencrypt
        //# OnEncrypt MUST call the Keystore's [GetActiveBranchKey](../branch-key-store.md#getactivebranchkey) operation with the following inputs:
        //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#getitem-branch-keystore-ondecrypt
        //# OnDecrypt MUST call the Keystore's [GetBranchKeyVersion](../branch-key-store.md#getbranchkeyversion) operation with the following inputs:
        branchKeyMaterials = branchKeyVersion
            ? await keyStore.getBranchKeyVersion(branchKeyId, branchKeyVersion)
            : // The complice needs a line
                //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#query-branch-keystore-onencrypt
                //# OnEncrypt MUST call the Keystore's [GetActiveBranchKey](../branch-key-store.md#getactivebranchkey) operation with the following inputs:
                //# - the `branchKeyId` used in this operation
                await keyStore.getActiveBranchKey(branchKeyId);
        //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#query-branch-keystore-onencrypt
        //# If the Keystore's GetActiveBranchKey operation succeeds
        //# the keyring MUST put the returned branch key materials in the cache using the
        //# formula defined in [Appendix A](#appendix-a-cache-entry-identifier-formulas).
        //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#getitem-branch-keystore-ondecrypt
        //# If the Keystore's GetBranchKeyVersion operation succeeds
        //# the keyring MUST put the returned branch key materials in the cache using the
        //# formula defined in [Appendix A](#appendix-a-cache-entry-identifier-formulas).
        cmc.putBranchKeyMaterial(cacheEntryId, branchKeyMaterials, cacheLimitTtl);
    }
    else {
        //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#ondecrypt
        //# If a cache entry is found and the entry's TTL has not expired, the hierarchical keyring MUST use those branch key materials for key unwrapping.
        //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#onencrypt
        //# If a cache entry is found and the entry's TTL has not expired, the hierarchical keyring MUST use those branch key materials for key wrapping.
        branchKeyMaterials = cacheEntry.response;
    }
    return branchKeyMaterials;
}
exports.getBranchKeyMaterials = getBranchKeyMaterials;
//= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#onencrypt
//# If the input [encryption materials](../structures.md#encryption-materials) do not contain a plaintext data key,
//# OnEncrypt MUST generate a random plaintext data key, according to the key length defined in the [algorithm suite](../algorithm-suites.md#encryption-key-length).
//# The process used to generate this random plaintext data key MUST use a secure source of randomness.
function getPlaintextDataKey(material) {
    // get the pdk from the encryption material whether it is already set or we
    // must randomly generate it
    return new Uint8Array(material.hasUnencryptedDataKey
        ? (0, material_management_1.unwrapDataKey)(material.getUnencryptedDataKey())
        : (0, crypto_1.randomBytes)(material.suite.keyLengthBytes));
}
exports.getPlaintextDataKey = getPlaintextDataKey;
//= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#branch-key-wrapping
//# To derive and encrypt a data key the keyring will follow the same key derivation and encryption as [AWS KMS](https://rwc.iacr.org/2018/Slides/Gueron.pdf).
//# The hierarchical keyring MUST:
//# 1. Generate a 16 byte random `salt` using a secure source of randomness
//# 1. Generate a 12 byte random `IV` using a secure source of randomness
//# 1. Use a [KDF in Counter Mode with a Pseudo Random Function with HMAC SHA 256](https://nvlpubs.nist.gov/nistpubs/SpecialPublications/NIST.SP.800-108r1.pdf) to derive a 32 byte `derivedBranchKey` data key with the following inputs:
//#    - Use the `salt` as the salt.
//#    - Use the branch key as the `key`.
//#    - Use the UTF8 Encoded value "aws-kms-hierarchy" as the label.
//# 1. Encrypt a plaintext data key with the `derivedBranchKey` using `AES-GCM-256` with the following inputs:
//#    - MUST use the `derivedBranchKey` as the AES-GCM cipher key.
//#    - MUST use the plain text data key that will be wrapped by the `derivedBranchKey` as the AES-GCM message.
//#    - MUST use the derived `IV` as the AES-GCM IV.
//#    - MUST use an authentication tag byte of length 16.
//#    - MUST use the serialized [AAD](#branch-key-wrapping-and-unwrapping-aad) as the AES-GCM AAD.
//# If OnEncrypt fails to do any of the above, OnEncrypt MUST fail.
function wrapPlaintextDataKey(pdk, branchKeyMaterials, { encryptionContext }) {
    // get what we need from branch key material to wrap the pdk
    const branchKey = branchKeyMaterials.branchKey();
    const { branchKeyIdentifier, branchKeyVersion: branchKeyVersionAsBytes } = branchKeyMaterials;
    // compress the branch key version utf8 bytes
    const branchKeyVersionAsBytesCompressed = Buffer.from((0, exports.uuidv4ToCompressedBytes)((0, exports.utf8BytesToString)(branchKeyVersionAsBytes)));
    const branchKeyIdAsBytes = (0, exports.stringToUtf8Bytes)(branchKeyIdentifier);
    // generate salt and IV
    const salt = (0, crypto_1.randomBytes)(constants_1.CIPHERTEXT_STRUCTURE.saltLength);
    const iv = (0, crypto_1.randomBytes)(constants_1.CIPHERTEXT_STRUCTURE.ivLength);
    // derive a key from the branch key
    const derivedBranchKey = (0, kdf_ctr_mode_node_1.kdfCounterMode)({
        digestAlgorithm: constants_1.KDF_DIGEST_ALGORITHM_SHA_256,
        ikm: branchKey,
        nonce: salt,
        purpose: constants_1.KEY_DERIVATION_LABEL,
        expectedLength: constants_1.DERIVED_BRANCH_KEY_LENGTH,
    });
    // set up additional auth data
    const wrappedAad = wrapAad(branchKeyIdAsBytes, branchKeyVersionAsBytesCompressed, encryptionContext);
    // encrypt the pdk into an edk
    const cipher = (0, crypto_1.createCipheriv)('aes-256-gcm', derivedBranchKey, iv).setAAD(wrappedAad);
    const edkCiphertext = Buffer.concat([cipher.update(pdk), cipher.final()]);
    const authTag = cipher.getAuthTag();
    // wrap the edk into a ciphertext
    const ciphertext = new Uint8Array(Buffer.concat([
        salt,
        iv,
        branchKeyVersionAsBytesCompressed,
        edkCiphertext,
        authTag,
    ]));
    return ciphertext;
}
exports.wrapPlaintextDataKey = wrapPlaintextDataKey;
//= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#branch-key-wrapping-and-unwrapping-aad
//# To Encrypt and Decrypt the `wrappedDerivedBranchKey` the keyring MUST include the following values as part of the AAD for
//# the AES Encrypt/Decrypt calls.
//# To construct the AAD, the keyring MUST concatenate the following values
//# 1. "aws-kms-hierarchy" as UTF8 Bytes
//# 1. Value of `branch-key-id` as UTF8 Bytes
//# 1. [version](../structures.md#branch-key-version) as Bytes
//# 1. [encryption context](structures.md#encryption-context-1) from the input
//#    [encryption materials](../structures.md#encryption-materials) according to the [encryption context serialization specification](../structures.md#serialization).
//# | Field               | Length (bytes) | Interpreted as                                       |
//# | ------------------- | -------------- | ---------------------------------------------------- |
//# | "aws-kms-hierarchy" | 17             | UTF-8 Encoded                                        |
//# | branch-key-id       | Variable       | UTF-8 Encoded                                        |
//# | version             | 16             | Bytes                                                |
//# | encryption context  | Variable       | [Encryption Context](../structures.md#serialization) |
//# If the keyring cannot serialize the encryption context, the operation MUST fail.
function wrapAad(branchKeyIdAsBytes, version, encryptionContext) {
    /* Precondition: Branch key version must be 16 bytes */
    (0, material_management_1.needs)(version.length === 16, 'Branch key version must be 16 bytes');
    /* The AAD section is uInt16BE(length) + AAD
     * see: https://docs.aws.amazon.com/encryption-sdk/latest/developer-guide/message-format.html#header-aad
     * However, we  _only_ need the ADD.
     * So, I just slice off the length.
     */
    const aad = Buffer.from((0, exports.serializeEncryptionContext)(encryptionContext).slice(2));
    return Buffer.concat([
        constants_1.PROVIDER_ID_HIERARCHY_AS_BYTES,
        branchKeyIdAsBytes,
        version,
        aad,
    ]);
}
exports.wrapAad = wrapAad;
//= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#onencrypt
//# Otherwise, OnEncrypt MUST append a new [encrypted data key](../structures.md#encrypted-data-key)
//# to the encrypted data key list in the [encryption materials](../structures.md#encryption-materials), constructed as follows:
//# - [ciphertext](../structures.md#ciphertext): MUST be serialized as the [hierarchical keyring ciphertext](#ciphertext)
//# - [key provider id](../structures.md#key-provider-id): MUST be UTF8 Encoded "aws-kms-hierarchy"
//# - [key provider info](../structures.md#key-provider-information): MUST be the UTF8 Encoded AWS DDB response `branch-key-id`
function modifyEncryptionMaterial(encryptionMaterial, pdk, edk, wrappingKeyName) {
    // if the pdk was already set in the encryption material, we should not reset
    if (!encryptionMaterial.hasUnencryptedDataKey) {
        encryptionMaterial.setUnencryptedDataKey(pdk, {
            keyNamespace: constants_1.PROVIDER_ID_HIERARCHY,
            keyName: wrappingKeyName,
            flags: material_management_1.KeyringTraceFlag.WRAPPING_KEY_GENERATED_DATA_KEY,
        });
    }
    // add the edk (that we created during onEncrypt) to the encryption material
    return encryptionMaterial.addEncryptedDataKey(new material_management_1.EncryptedDataKey({
        providerId: constants_1.PROVIDER_ID_HIERARCHY,
        providerInfo: wrappingKeyName,
        encryptedDataKey: edk,
    }), constants_1.ENCRYPT_FLAGS);
}
exports.modifyEncryptionMaterial = modifyEncryptionMaterial;
//= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#ondecrypt
//# The set of encrypted data keys MUST first be filtered to match this keyring’s configuration. For the encrypted data key to match:
//# - Its provider ID MUST match the UTF8 Encoded value of “aws-kms-hierarchy”.
//# - Deserialize the key provider info, if deserialization fails the next EDK in the set MUST be attempted.
//#   - The deserialized key provider info MUST be UTF8 Decoded and MUST match this keyring's configured `Branch Key Identifier`.
function filterEdk(branchKeyId, { providerId, providerInfo }) {
    // check if the edk matches the keyring's configuration according to provider
    // id and info (the edk object should have been wrapped by the branch key
    // configured in this keyring or decryption material's encryption context)
    //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#getitem-branch-keystore-ondecrypt
    //# - Deserialize the UTF8-Decoded `branch-key-id` from the [key provider info](../structures.md#key-provider-information) of the [encrypted data key](../structures.md#encrypted-data-key)
    //# and verify this is equal to the configured or supplied `branch-key-id`.
    return providerId === constants_1.PROVIDER_ID_HIERARCHY
        ? branchKeyId === providerInfo
        : false;
}
exports.filterEdk = filterEdk;
//= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#ciphertext
//# The following table describes the fields that form the ciphertext for this keyring.
//# The bytes are appended in the order shown.
//# The Encryption Key is variable.
//# It will be whatever length is represented by the algorithm suite.
//# Because all the other values are constant,
//# this variability in the encryption key does not impact the format.
//# | Field              | Length (bytes) | Interpreted as |
//# | ------------------ | -------------- | -------------- |
//# | Salt               | 16             | bytes          |
//# | IV                 | 12             | bytes          |
//# | Version            | 16             | bytes          |
//# | Encrypted Key      | Variable       | bytes          |
//# | Authentication Tag | 16             | bytes          |
function destructureCiphertext(ciphertext, { keyLengthBytes }) {
    // what we expect the length of the edk object's ciphertext to be. This
    // depends on the byte key length specified by the algorithm suite
    const expectedCiphertextLength = constants_1.CIPHERTEXT_STRUCTURE.saltLength +
        constants_1.CIPHERTEXT_STRUCTURE.ivLength +
        constants_1.CIPHERTEXT_STRUCTURE.branchKeyVersionCompressedLength +
        keyLengthBytes +
        constants_1.CIPHERTEXT_STRUCTURE.authTagLength;
    /* Precondition: The edk ciphertext must have the correct length */
    (0, material_management_1.needs)(ciphertext.length === expectedCiphertextLength, `The encrypted data key ciphertext must be ${expectedCiphertextLength} bytes long`);
    let start = 0;
    let end = 0;
    // extract the salt from the edk ciphertext
    start = end;
    end += constants_1.CIPHERTEXT_STRUCTURE.saltLength;
    const salt = Buffer.from(ciphertext.subarray(start, end));
    // extract the IV from the edk ciphertext
    start = end;
    end += constants_1.CIPHERTEXT_STRUCTURE.ivLength;
    const iv = Buffer.from(ciphertext.subarray(start, end));
    // extract the compressed branch key version from the edk ciphertext
    start = end;
    end += constants_1.CIPHERTEXT_STRUCTURE.branchKeyVersionCompressedLength;
    const branchKeyVersionAsBytesCompressed = Buffer.from(ciphertext.subarray(start, end));
    // extract the encrypted data key from the edk ciphertext
    start = end;
    end += keyLengthBytes;
    const encryptedDataKey = Buffer.from(ciphertext.subarray(start, end));
    // extract the auth tag from the edk ciphertext
    start = end;
    end += constants_1.CIPHERTEXT_STRUCTURE.authTagLength;
    const authTag = Buffer.from(ciphertext.subarray(start, end));
    return {
        salt,
        iv,
        branchKeyVersionAsBytesCompressed,
        encryptedDataKey,
        authTag,
    };
}
exports.destructureCiphertext = destructureCiphertext;
//= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#branch-key-unwrapping
//# To decrypt an encrypted data key with a branch key, the hierarchical keyring MUST:
//# 1. Deserialize the 16 byte random `salt` from the [edk ciphertext](../structures.md#ciphertext).
//# 1. Deserialize the 12 byte random `IV` from the [edk ciphertext](../structures.md#ciphertext).
//# 1. Deserialize the 16 byte `version` from the [edk ciphertext](../structures.md#ciphertext).
//# 1. Deserialize the `encrypted key` from the [edk ciphertext](../structures.md#ciphertext).
//# 1. Deserialize the `authentication tag` from the [edk ciphertext](../structures.md#ciphertext).
//# 1. Use a [KDF in Counter Mode with a Pseudo Random Function with HMAC SHA 256](https://nvlpubs.nist.gov/nistpubs/SpecialPublications/NIST.SP.800-108r1.pdf) to derive
//#    the 32 byte `derivedBranchKey` data key with the following inputs:
//#    - Use the `salt` as the salt.
//#    - Use the branch key as the `key`.
//# 1. Decrypt the encrypted data key with the `derivedBranchKey` using `AES-GCM-256` with the following inputs:
//#    - It MUST use the `encrypted key` obtained from deserialization as the AES-GCM input ciphertext.
//#    - It MUST use the `authentication tag` obtained from deserialization as the AES-GCM input authentication tag.
//#    - It MUST use the `derivedBranchKey` as the AES-GCM cipher key.
//#    - It MUST use the `IV` obtained from deserialization as the AES-GCM input IV.
//#    - It MUST use the serialized [encryption context](#branch-key-wrapping-and-unwrapping-aad) as the AES-GCM AAD.
//# If OnDecrypt fails to do any of the above, OnDecrypt MUST fail.
function unwrapEncryptedDataKey(ciphertext, branchKeyMaterials, { encryptionContext, suite }) {
    // get what we need from the branch key materials to unwrap the edk
    const branchKey = branchKeyMaterials.branchKey();
    const { branchKeyIdentifier } = branchKeyMaterials;
    const branchKeyIdAsBytes = (0, exports.stringToUtf8Bytes)(branchKeyIdentifier);
    // get the salt, iv, edk, and auth tag from the edk ciphertext
    const { salt, iv, encryptedDataKey, authTag, branchKeyVersionAsBytesCompressed, } = destructureCiphertext(ciphertext, suite);
    // derive a key from the branch key
    const derivedBranchKey = (0, kdf_ctr_mode_node_1.kdfCounterMode)({
        digestAlgorithm: constants_1.KDF_DIGEST_ALGORITHM_SHA_256,
        ikm: branchKey,
        nonce: salt,
        purpose: constants_1.KEY_DERIVATION_LABEL,
        expectedLength: constants_1.DERIVED_BRANCH_KEY_LENGTH,
    });
    // set up additional auth data
    const wrappedAad = wrapAad(branchKeyIdAsBytes, branchKeyVersionAsBytesCompressed, encryptionContext);
    // decipher the edk to get the udk/pdk
    const decipher = (0, crypto_1.createDecipheriv)('aes-256-gcm', derivedBranchKey, iv)
        .setAAD(wrappedAad)
        .setAuthTag(authTag);
    const udk = Buffer.concat([
        decipher.update(encryptedDataKey),
        decipher.final(),
    ]);
    return new Uint8Array(udk);
}
exports.unwrapEncryptedDataKey = unwrapEncryptedDataKey;
function modifyDencryptionMaterial(decryptionMaterial, udk, wrappingKeyName) {
    // modify the decryption material by setting the plaintext data key
    return decryptionMaterial.setUnencryptedDataKey(udk, {
        keyNamespace: constants_1.PROVIDER_ID_HIERARCHY,
        keyName: wrappingKeyName,
        flags: constants_1.DECRYPT_FLAGS,
    });
}
exports.modifyDencryptionMaterial = modifyDencryptionMaterial;
//# sourceMappingURL=data:application/json;base64,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