//===- DXContainerTest.cpp - Tests for DXContainerFile --------------------===//
//
// Part of the LLVM Project, under the Apache License v2.0 with LLVM Exceptions.
// See https://llvm.org/LICENSE.txt for license information.
// SPDX-License-Identifier: Apache-2.0 WITH LLVM-exception
//
//===----------------------------------------------------------------------===//

#include "llvm/ADT/StringRef.h"
#include "llvm/ADT/Twine.h"
#include "llvm/ObjectYAML/ObjectYAML.h"
#include "llvm/ObjectYAML/yaml2obj.h"
#include "llvm/Support/MemoryBufferRef.h"
#include "llvm/Support/YAMLTraits.h"
#include "llvm/Support/raw_ostream.h"
#include "llvm/Testing/Support/Error.h"
#include "gtest/gtest.h"

using namespace llvm;
using namespace llvm::object;

static bool convert(SmallVectorImpl<char> &Output, const char *YAML) {
  raw_svector_ostream OS(Output);
  yaml::Input YIn(YAML);
  return convertYAML(YIn, OS, [](const Twine &Err) { errs() << Err; });
}

TEST(DXCFile, ParseEmptyParts) {
  SmallString<128> Storage;

  // First read a fully explicit yaml with all sizes and offsets provided
  ASSERT_TRUE(convert(Storage, R"(--- !dxcontainer
Header:
  Hash:            [ 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
                     0x0, 0x0, 0x0, 0x0, 0x0, 0x0 ]
  Version:
    Major:           1
    Minor:           0
  FileSize:        116
  PartCount:       7
  PartOffsets:     [ 60, 68, 76, 84, 92, 100, 108 ]
Parts:
  - Name:            FKE0
    Size:            0
  - Name:            FKE1
    Size:            0
  - Name:            FKE2
    Size:            0
  - Name:            FKE3
    Size:            0
  - Name:            FKE4
    Size:            0
  - Name:            FKE5
    Size:            0
  - Name:            FKE6
    Size:            0
...
  )"));

  // Result
  char Buffer[] = {
      0x44, 0x58, 0x42, 0x43, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
      0x74, 0x00, 0x00, 0x00, 0x07, 0x00, 0x00, 0x00, 0x3C, 0x00, 0x00, 0x00,
      0x44, 0x00, 0x00, 0x00, 0x4C, 0x00, 0x00, 0x00, 0x54, 0x00, 0x00, 0x00,
      0x5C, 0x00, 0x00, 0x00, 0x64, 0x00, 0x00, 0x00, 0x6C, 0x00, 0x00, 0x00,
      0x46, 0x4B, 0x45, 0x30, 0x00, 0x00, 0x00, 0x00, 0x46, 0x4B, 0x45, 0x31,
      0x00, 0x00, 0x00, 0x00, 0x46, 0x4B, 0x45, 0x32, 0x00, 0x00, 0x00, 0x00,
      0x46, 0x4B, 0x45, 0x33, 0x00, 0x00, 0x00, 0x00, 0x46, 0x4B, 0x45, 0x34,
      0x00, 0x00, 0x00, 0x00, 0x46, 0x4B, 0x45, 0x35, 0x00, 0x00, 0x00, 0x00,
      0x46, 0x4B, 0x45, 0x36, 0x00, 0x00, 0x00, 0x00,
  };

  EXPECT_EQ(Storage.size(), 116u);
  EXPECT_TRUE(memcmp(Buffer, Storage.data(), 116) == 0);

  Storage.clear();

  // Next, read the same file without the part offsets or file size. Both cases
  // should result in the same final output.
  ASSERT_TRUE(convert(Storage, R"(--- !dxcontainer
Header:
  Hash:            [ 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
                     0x0, 0x0, 0x0, 0x0, 0x0, 0x0 ]
  Version:
    Major:           1
    Minor:           0
  PartCount:       7
Parts:
  - Name:            FKE0
    Size:            0
  - Name:            FKE1
    Size:            0
  - Name:            FKE2
    Size:            0
  - Name:            FKE3
    Size:            0
  - Name:            FKE4
    Size:            0
  - Name:            FKE5
    Size:            0
  - Name:            FKE6
    Size:            0
...
  )"));

  EXPECT_EQ(Storage.size(), 116u);
  EXPECT_TRUE(memcmp(Buffer, Storage.data(), 116) == 0);
}

TEST(RootSignature, ParseRootFlags) {
  SmallString<128> Storage;

  // First read a fully explicit yaml with all sizes and offsets provided
  ASSERT_TRUE(convert(Storage, R"(--- !dxcontainer
  Header:
    Hash:            [ 0x32, 0x9A, 0x53, 0xD8, 0xEC, 0xBE, 0x35, 0x6F, 0x5, 
                      0x39, 0xE1, 0xFE, 0x31, 0x20, 0xF0, 0xC1 ]
    Version:
      Major:           1
      Minor:           0
    FileSize:        68
    PartCount:       1
    PartOffsets:     [ 36 ]
  Parts:
    - Name:            RTS0
      Size:            24
      RootSignature:
        Version:         2
        NumRootParameters: 0
        RootParametersOffset: 24
        NumStaticSamplers: 0
        StaticSamplersOffset: 0
        Parameters: []
        AllowInputAssemblerInputLayout: true
    )"));

  uint8_t Buffer[] = {
      0x44, 0x58, 0x42, 0x43, 0x32, 0x9A, 0x53, 0xD8, 0xEC, 0xBE, 0x35, 0x6F,
      0x05, 0x39, 0xE1, 0xFE, 0x31, 0x20, 0xF0, 0xC1, 0x01, 0x00, 0x00, 0x00,
      0x44, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x24, 0x00, 0x00, 0x00,
      0x52, 0x54, 0x53, 0x30, 0x18, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
  };

  EXPECT_EQ(Storage.size(), 68u);
  EXPECT_TRUE(memcmp(Buffer, Storage.data(), 68u) == 0);
}

TEST(RootSignature, HeaderData) {
  SmallString<128> Storage;

  // First read a fully explicit yaml with all sizes and offsets provided
  ASSERT_TRUE(convert(Storage, R"(--- !dxcontainer
  Header:
      Hash:            [ 0x32, 0x9A, 0x53, 0xD8, 0xEC, 0xBE, 0x35, 0x6F, 0x5, 
                        0x39, 0xE1, 0xFE, 0x31, 0x20, 0xF0, 0xC1 ]
      Version:
        Major:           1
        Minor:           0
      FileSize:        133
      PartCount:       1
      PartOffsets:     [ 36 ]
  Parts:
    - Name:            RTS0
      Size:            89
      RootSignature:
        Version: 2
        NumRootParameters: 1
        RootParametersOffset: 255
        NumStaticSamplers: 0
        StaticSamplersOffset: 0
        Parameters:
          - ParameterType: 1
            ShaderVisibility: 2
            Constants:
              Num32BitValues: 16
              ShaderRegister: 15
              RegisterSpace: 14
        AllowInputAssemblerInputLayout: true
        DenyGeometryShaderRootAccess: true
    )"));

  uint8_t Buffer[] = {
      0x44, 0x58, 0x42, 0x43, 0x32, 0x9a, 0x53, 0xd8, 0xec, 0xbe, 0x35, 0x6f,
      0x05, 0x39, 0xe1, 0xfe, 0x31, 0x20, 0xf0, 0xc1, 0x01, 0x00, 0x00, 0x00,
      0x85, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x24, 0x00, 0x00, 0x00,
      0x52, 0x54, 0x53, 0x30, 0x59, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
      0x01, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x11, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
      0x02, 0x00, 0x00, 0x00, 0x24, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x00,
      0x0e, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00};

  EXPECT_EQ(Storage.size(), 133u);
  EXPECT_TRUE(memcmp(Buffer, Storage.data(), 133u) == 0);
}

TEST(RootSignature, ParseRootConstants) {
  SmallString<128> Storage;

  // First read a fully explicit yaml with all sizes and offsets provided
  ASSERT_TRUE(convert(Storage, R"(--- !dxcontainer
  Header:
      Hash:            [ 0x32, 0x9A, 0x53, 0xD8, 0xEC, 0xBE, 0x35, 0x6F, 0x5, 
                        0x39, 0xE1, 0xFE, 0x31, 0x20, 0xF0, 0xC1 ]
      Version:
        Major:           1
        Minor:           0
      FileSize:        133
      PartCount:       1
      PartOffsets:     [ 36 ]
  Parts:
    - Name:            RTS0
      Size:            89
      RootSignature:
        Version: 2
        NumRootParameters: 1
        RootParametersOffset: 36
        NumStaticSamplers: 0
        StaticSamplersOffset: 0
        Parameters:
          - ParameterType: 1
            ShaderVisibility: 2
            Constants:
              Num32BitValues: 16
              ShaderRegister: 15
              RegisterSpace: 14
        AllowInputAssemblerInputLayout: true
        DenyGeometryShaderRootAccess: true
    )"));

  uint8_t Buffer[] = {
      0x44, 0x58, 0x42, 0x43, 0x32, 0x9a, 0x53, 0xd8, 0xec, 0xbe, 0x35, 0x6f,
      0x05, 0x39, 0xe1, 0xfe, 0x31, 0x20, 0xf0, 0xc1, 0x01, 0x00, 0x00, 0x00,
      0x85, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x24, 0x00, 0x00, 0x00,
      0x52, 0x54, 0x53, 0x30, 0x59, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
      0x01, 0x00, 0x00, 0x00, 0x24, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x11, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
      0x02, 0x00, 0x00, 0x00, 0x24, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x00,
      0x0e, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00};

  EXPECT_EQ(Storage.size(), 133u);
  EXPECT_TRUE(memcmp(Buffer, Storage.data(), 133u) == 0);
}

TEST(RootSignature, ParseRootDescriptorsV10) {
  SmallString<128> Storage;

  // First read a fully explicit yaml with all sizes and offsets provided
  ASSERT_TRUE(convert(Storage, R"(--- !dxcontainer
  Header:
      Hash:            [ 0x32, 0x9A, 0x53, 0xD8, 0xEC, 0xBE, 0x35, 0x6F, 0x5, 
                        0x39, 0xE1, 0xFE, 0x31, 0x20, 0xF0, 0xC1 ]
      Version:
        Major:           1
        Minor:           0
      FileSize:        133
      PartCount:       1
      PartOffsets:     [ 36 ]
  Parts:
  - Name:            RTS0
    Size:            89
    RootSignature:
      Version: 1
      NumRootParameters: 1
      RootParametersOffset: 24
      NumStaticSamplers: 0
      StaticSamplersOffset: 60
      Parameters:         
      - ParameterType: 2 # SRV
        ShaderVisibility: 3 # Domain
        Descriptor:
          ShaderRegister: 31
          RegisterSpace: 32
      AllowInputAssemblerInputLayout: true
      DenyGeometryShaderRootAccess: true
    )"));

  uint8_t Buffer[] = {
      0x44, 0x58, 0x42, 0x43, 0x32, 0x9a, 0x53, 0xd8, 0xec, 0xbe, 0x35, 0x6f,
      0x05, 0x39, 0xe1, 0xfe, 0x31, 0x20, 0xf0, 0xc1, 0x01, 0x00, 0x00, 0x00,
      0x85, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x24, 0x00, 0x00, 0x00,
      0x52, 0x54, 0x53, 0x30, 0x59, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
      0x01, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x3c, 0x00, 0x00, 0x00, 0x11, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
      0x03, 0x00, 0x00, 0x00, 0x24, 0x00, 0x00, 0x00, 0x1f, 0x00, 0x00, 0x00,
      0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00};

  EXPECT_EQ(Storage.size(), 133u);
  EXPECT_TRUE(memcmp(Buffer, Storage.data(), 133u) == 0);
}

TEST(RootSignature, ParseRootDescriptorsV11) {
  SmallString<128> Storage;

  // First read a fully explicit yaml with all sizes and offsets provided
  ASSERT_TRUE(convert(Storage, R"(--- !dxcontainer
  Header:
      Hash:            [ 0x32, 0x9A, 0x53, 0xD8, 0xEC, 0xBE, 0x35, 0x6F, 0x5, 
                        0x39, 0xE1, 0xFE, 0x31, 0x20, 0xF0, 0xC1 ]
      Version:
        Major:           1
        Minor:           0
      FileSize:        133
      PartCount:       1
      PartOffsets:     [ 36 ]
  Parts:
  - Name:            RTS0
    Size:            89
    RootSignature:
      Version: 2
      NumRootParameters: 1
      RootParametersOffset: 24
      NumStaticSamplers: 0
      StaticSamplersOffset: 60
      Parameters:         
      - ParameterType: 2 # SRV
        ShaderVisibility: 3 # Domain
        Descriptor:
          ShaderRegister: 31
          RegisterSpace: 32
          DATA_STATIC_WHILE_SET_AT_EXECUTE: true
      AllowInputAssemblerInputLayout: true
      DenyGeometryShaderRootAccess: true
    )"));

  uint8_t Buffer[] = {
      0x44, 0x58, 0x42, 0x43, 0x32, 0x9a, 0x53, 0xd8, 0xec, 0xbe, 0x35, 0x6f,
      0x05, 0x39, 0xe1, 0xfe, 0x31, 0x20, 0xf0, 0xc1, 0x01, 0x00, 0x00, 0x00,
      0x85, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x24, 0x00, 0x00, 0x00,
      0x52, 0x54, 0x53, 0x30, 0x59, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
      0x01, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x3c, 0x00, 0x00, 0x00, 0x11, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
      0x03, 0x00, 0x00, 0x00, 0x24, 0x00, 0x00, 0x00, 0x1f, 0x00, 0x00, 0x00,
      0x20, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00};

  EXPECT_EQ(Storage.size(), 133u);
  EXPECT_TRUE(memcmp(Buffer, Storage.data(), 133u) == 0);
}

TEST(RootSignature, ParseDescriptorTableV10) {
  SmallString<128> Storage;

  // First read a fully explicit yaml with all sizes and offsets provided
  ASSERT_TRUE(convert(Storage, R"(--- !dxcontainer
  Header:
      Hash:            [ 0x32, 0x9A, 0x53, 0xD8, 0xEC, 0xBE, 0x35, 0x6F, 0x5, 
                        0x39, 0xE1, 0xFE, 0x31, 0x20, 0xF0, 0xC1 ]
      Version:
        Major:           1
        Minor:           0
      FileSize:        133
      PartCount:       1
      PartOffsets:     [ 36 ]
  Parts:
  - Name:            RTS0
    Size:            89
    RootSignature:
      Version: 1
      NumRootParameters: 1
      RootParametersOffset: 24
      NumStaticSamplers: 0
      StaticSamplersOffset: 60
      Parameters:         
      - ParameterType: 0 # SRV
        ShaderVisibility: 3 # Domain
        Table:
          NumRanges: 1
          Ranges:
            - RangeType: 0
              NumDescriptors: 41
              BaseShaderRegister: 42
              RegisterSpace: 43
              OffsetInDescriptorsFromTableStart: 44
      AllowInputAssemblerInputLayout: true
      DenyGeometryShaderRootAccess: true
    )"));

  uint8_t Buffer[] = {
      0x44, 0x58, 0x42, 0x43, 0x32, 0x9a, 0x53, 0xd8, 0xec, 0xbe, 0x35, 0x6f,
      0x05, 0x39, 0xe1, 0xfe, 0x31, 0x20, 0xf0, 0xc1, 0x01, 0x00, 0x00, 0x00,
      0x85, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x24, 0x00, 0x00, 0x00,
      0x52, 0x54, 0x53, 0x30, 0x59, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
      0x01, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x3c, 0x00, 0x00, 0x00, 0x11, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x03, 0x00, 0x00, 0x00, 0x24, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
      0x2c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x29, 0x00, 0x00, 0x00,
      0x2a, 0x00, 0x00, 0x00, 0x2b, 0x00, 0x00, 0x00, 0x2c, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00};

  EXPECT_EQ(Storage.size(), 133u);
  EXPECT_TRUE(memcmp(Buffer, Storage.data(), 133u) == 0);
}

TEST(RootSignature, ParseDescriptorTableV11) {
  SmallString<128> Storage;

  // First read a fully explicit yaml with all sizes and offsets provided
  ASSERT_TRUE(convert(Storage, R"(--- !dxcontainer
  Header:
      Hash:            [ 0x32, 0x9A, 0x53, 0xD8, 0xEC, 0xBE, 0x35, 0x6F, 0x5, 
                        0x39, 0xE1, 0xFE, 0x31, 0x20, 0xF0, 0xC1 ]
      Version:
        Major:           1
        Minor:           0
      FileSize:        133
      PartCount:       1
      PartOffsets:     [ 36 ]
  Parts:
  - Name:            RTS0
    Size:            89
    RootSignature:
      Version: 2
      NumRootParameters: 1
      RootParametersOffset: 24
      NumStaticSamplers: 0
      StaticSamplersOffset: 60
      Parameters:         
      - ParameterType: 0 # Descriptor Table
        ShaderVisibility: 3 # Domain
        Table:
          NumRanges: 1
          Ranges:
            - RangeType: 0
              NumDescriptors: -1
              BaseShaderRegister: 42
              RegisterSpace: 43
              OffsetInDescriptorsFromTableStart: 41
              DESCRIPTORS_STATIC_KEEPING_BUFFER_BOUNDS_CHECKS: true
      AllowInputAssemblerInputLayout: true
      DenyGeometryShaderRootAccess: true
    )"));

  uint8_t Buffer[] = {
      0x44, 0x58, 0x42, 0x43, 0x32, 0x9a, 0x53, 0xd8, 0xec, 0xbe, 0x35, 0x6f,
      0x05, 0x39, 0xe1, 0xfe, 0x31, 0x20, 0xf0, 0xc1, 0x01, 0x00, 0x00, 0x00,
      0x85, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x24, 0x00, 0x00, 0x00,
      0x52, 0x54, 0x53, 0x30, 0x59, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
      0x01, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x3c, 0x00, 0x00, 0x00, 0x11, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x03, 0x00, 0x00, 0x00, 0x24, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
      0x2c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff,
      0x2a, 0x00, 0x00, 0x00, 0x2b, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00,
      0x29, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00};

  EXPECT_EQ(Storage.size(), 133u);
  EXPECT_TRUE(memcmp(Buffer, Storage.data(), 133u) == 0);
}

TEST(RootSignature, ParseStaticSamplers) {
  SmallString<128> Storage;

  // First read a fully explicit yaml with all sizes and offsets provided
  ASSERT_TRUE(convert(Storage, R"(--- !dxcontainer
Header:
  Hash:            [ 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
                     0x0, 0x0, 0x0, 0x0, 0x0, 0x0 ]
  Version:
    Major:           1
    Minor:           0
  PartCount:       1
  PartOffsets:     [ 60 ]
Parts:
  - Name:            RTS0
    Size:            76
    RootSignature:
      Version: 2
      NumRootParameters: 0
      RootParametersOffset: 0
      NumStaticSamplers: 1
      StaticSamplersOffset: 24
      Parameters: []
      Samplers: 
        - Filter: 10 
          AddressU: 1
          AddressV: 2
          AddressW: 5
          MipLODBias: 1.23
          MaxAnisotropy: 20
          ComparisonFunc: 4
          BorderColor: 0
          MinLOD: 4.56
          MaxLOD: 8.90
          ShaderRegister: 31 
          RegisterSpace: 32
          ShaderVisibility:  7
      AllowInputAssemblerInputLayout: true
      DenyGeometryShaderRootAccess: true
    )"));

  uint8_t Buffer[] = {
      0x44, 0x58, 0x42, 0x43, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
      0x90, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x3c, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x52, 0x54, 0x53, 0x30, 0x4c, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
      0x18, 0x00, 0x00, 0x00, 0x11, 0x00, 0x00, 0x00, 0x0a, 0x00, 0x00, 0x00,
      0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00,
      0xa4, 0x70, 0x9d, 0x3f, 0x14, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x85, 0xeb, 0x91, 0x40, 0x66, 0x66, 0x0e, 0x41,
      0x1f, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x07, 0x00, 0x00, 0x00};

  EXPECT_EQ(Storage.size(), 144u);
  EXPECT_TRUE(memcmp(Buffer, Storage.data(), 144u) == 0);
}
