#!/usr/bin/perl
use strict;
use warnings;
use v5.10;

use Getopt::Long qw[];
use Time::Moment qw[];

my $Moment = Time::Moment->now
                         ->with_day_of_month(1)
                         ->at_midnight;

Getopt::Long::GetOptions(
    'y|year=i' => sub {
        my ($name, $year) = @_;

        ($year >= 1 && $year <= 9999)
          or die qq/Option '$name' is out of the range [1, 9999]\n/;

        $Moment = $Moment->with_year($year);
    },
    'm|month=i' => sub {
        my ($name, $month) = @_;

        ($month >= 1 && $month <= 12)
          or die qq/Option '$name' is out of the range [1=January, 12=December]\n/;

        $Moment = $Moment->with_month($month);
    },
) or do {
    say "usage: $0 [-y year] [-m month]";
    say "    -y --year    the year [1, 9999]";
    say "    -m --month   the month of the year [1=January, 12=December]";
    exit(1);
};

sub align {
    @_ == 2 or die q/Usage: align(string, width)/;
    my ($string, $width) = @_;
    return sprintf "%*s", ($width + length $string) / 2, $string;
}

say align($Moment->strftime('%B %Y'), 24);
say 'Mo Tu We Th Fr Sa Su Wk';

my $this_month = $Moment;
my $next_month = $Moment->plus_months(1);
my $date       = $Moment->with_day_of_week(1);

while ($date->is_before($next_month)) {
    my @week = ((('  ') x 7), sprintf '%.2d', $date->week);
    foreach my $index (0..6) {
        if (!$date->is_before($this_month) && $date->is_before($next_month)) {
            $week[$index] = sprintf '%2d', $date->day_of_month;
        }
        $date = $date->plus_days(1);
    }
    say join ' ', @week;
}
