/*
 * Decompiled with CFR 0.152.
 */
package io.grpc.xds;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import io.grpc.EquivalentAddressGroup;
import io.grpc.NameResolver;
import io.grpc.NameResolverProvider;
import io.grpc.Status;
import io.grpc.StatusOr;
import io.grpc.SynchronizationContext;
import io.grpc.internal.RetryingNameResolver;
import io.grpc.xds.Endpoints;
import io.grpc.xds.HttpConnectionManager;
import io.grpc.xds.RoutingUtils;
import io.grpc.xds.VirtualHost;
import io.grpc.xds.XdsClusterResource;
import io.grpc.xds.XdsConfig;
import io.grpc.xds.XdsEndpointResource;
import io.grpc.xds.XdsListenerResource;
import io.grpc.xds.XdsRouteConfigureResource;
import io.grpc.xds.client.Locality;
import io.grpc.xds.client.XdsClient;
import io.grpc.xds.client.XdsResourceType;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.EnumMap;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.Executor;
import javax.annotation.Nullable;

final class XdsDependencyManager
implements XdsConfig.XdsClusterSubscriptionRegistry {
    private static final TrackedWatcherType<XdsListenerResource.LdsUpdate> LDS_TYPE = new TrackedWatcherType(TrackedWatcherTypeEnum.LDS);
    private static final TrackedWatcherType<XdsRouteConfigureResource.RdsUpdate> RDS_TYPE = new TrackedWatcherType(TrackedWatcherTypeEnum.RDS);
    private static final TrackedWatcherType<XdsClusterResource.CdsUpdate> CDS_TYPE = new TrackedWatcherType(TrackedWatcherTypeEnum.CDS);
    private static final TrackedWatcherType<XdsEndpointResource.EdsUpdate> EDS_TYPE = new TrackedWatcherType(TrackedWatcherTypeEnum.EDS);
    private static final TrackedWatcherType<List<EquivalentAddressGroup>> DNS_TYPE = new TrackedWatcherType(TrackedWatcherTypeEnum.DNS);
    private static final Locality LOGICAL_DNS_CLUSTER_LOCALITY = Locality.create("", "", "");
    private static final int MAX_CLUSTER_RECURSION_DEPTH = 16;
    static boolean enableLogicalDns = false;
    private final String listenerName;
    private final XdsClient xdsClient;
    private final SynchronizationContext syncContext;
    private final String dataPlaneAuthority;
    private final NameResolver.Args nameResolverArgs;
    private XdsConfigWatcher xdsConfigWatcher;
    private StatusOr<XdsConfig> lastUpdate = null;
    private final Map<TrackedWatcherTypeEnum, TypeWatchers<?>> resourceWatchers = new EnumMap(TrackedWatcherTypeEnum.class);
    private final Set<ClusterSubscription> subscriptions = new HashSet<ClusterSubscription>();

    XdsDependencyManager(XdsClient xdsClient, SynchronizationContext syncContext, String dataPlaneAuthority, String listenerName, NameResolver.Args nameResolverArgs) {
        this.listenerName = (String)Preconditions.checkNotNull((Object)listenerName, (Object)"listenerName");
        this.xdsClient = (XdsClient)Preconditions.checkNotNull((Object)xdsClient, (Object)"xdsClient");
        this.syncContext = (SynchronizationContext)Preconditions.checkNotNull((Object)syncContext, (Object)"syncContext");
        this.dataPlaneAuthority = (String)Preconditions.checkNotNull((Object)dataPlaneAuthority, (Object)"dataPlaneAuthority");
        this.nameResolverArgs = (NameResolver.Args)Preconditions.checkNotNull((Object)nameResolverArgs, (Object)"nameResolverArgs");
    }

    public static String toContextStr(String typeName, String resourceName) {
        return typeName + " resource " + resourceName;
    }

    public void start(XdsConfigWatcher xdsConfigWatcher) {
        Preconditions.checkState((this.xdsConfigWatcher == null ? 1 : 0) != 0, (Object)"dep manager may not be restarted");
        this.xdsConfigWatcher = (XdsConfigWatcher)Preconditions.checkNotNull((Object)xdsConfigWatcher, (Object)"xdsConfigWatcher");
        this.syncContext.execute(() -> this.addWatcher(LDS_TYPE, new LdsWatcher(this.listenerName)));
    }

    @Override
    public XdsConfig.Subscription subscribeToCluster(String clusterName) {
        Preconditions.checkState((this.xdsConfigWatcher != null ? 1 : 0) != 0, (Object)"dep manager must first be started");
        Preconditions.checkNotNull((Object)clusterName, (Object)"clusterName");
        ClusterSubscription subscription = new ClusterSubscription(clusterName);
        this.syncContext.execute(() -> {
            if (this.getWatchers(LDS_TYPE).isEmpty()) {
                subscription.closed = true;
                return;
            }
            this.subscriptions.add(subscription);
            this.addClusterWatcher(clusterName);
        });
        return subscription;
    }

    public void requestReresolution() {
        this.syncContext.execute(() -> {
            for (TrackedWatcher<List<EquivalentAddressGroup>> watcher : this.getWatchers(DNS_TYPE).values()) {
                DnsWatcher dnsWatcher = (DnsWatcher)watcher;
                dnsWatcher.refresh();
            }
        });
    }

    private <T extends XdsClient.ResourceUpdate> void addWatcher(TrackedWatcherType<T> watcherType, XdsWatcherBase<T> watcher) {
        this.syncContext.throwIfNotInThisSynchronizationContext();
        XdsResourceType type = ((XdsWatcherBase)watcher).type;
        String resourceName = ((XdsWatcherBase)watcher).resourceName;
        this.getWatchers(watcherType).put(resourceName, watcher);
        this.xdsClient.watchXdsResource(type, resourceName, watcher, (Executor)this.syncContext);
    }

    public void shutdown() {
        this.syncContext.execute(() -> {
            for (TypeWatchers<?> watchers : this.resourceWatchers.values()) {
                for (TrackedWatcher watcher : watchers.watchers.values()) {
                    watcher.close();
                }
            }
            this.resourceWatchers.clear();
            this.subscriptions.clear();
        });
    }

    private void releaseSubscription(ClusterSubscription subscription) {
        Preconditions.checkNotNull((Object)subscription, (Object)"subscription");
        this.syncContext.execute(() -> {
            if (subscription.closed) {
                return;
            }
            subscription.closed = true;
            if (!this.subscriptions.remove(subscription)) {
                return;
            }
            this.maybePublishConfig();
        });
    }

    private void maybePublishConfig() {
        this.syncContext.throwIfNotInThisSynchronizationContext();
        if (this.getWatchers(LDS_TYPE).isEmpty()) {
            return;
        }
        boolean waitingOnResource = this.resourceWatchers.values().stream().flatMap(typeWatchers -> typeWatchers.watchers.values().stream()).anyMatch(TrackedWatcher::missingResult);
        if (waitingOnResource) {
            return;
        }
        StatusOr<XdsConfig> newUpdate = this.buildUpdate();
        if (Objects.equals(newUpdate, this.lastUpdate)) {
            return;
        }
        assert (newUpdate.hasValue() || newUpdate.getStatus().getCode() == Status.Code.UNAVAILABLE || newUpdate.getStatus().getCode() == Status.Code.INTERNAL);
        this.lastUpdate = newUpdate;
        this.xdsConfigWatcher.onUpdate(this.lastUpdate);
    }

    @VisibleForTesting
    StatusOr<XdsConfig> buildUpdate() {
        WatcherTracer tracer = new WatcherTracer(this.resourceWatchers);
        StatusOr<XdsConfig> config = XdsDependencyManager.buildUpdate(tracer, this.listenerName, this.dataPlaneAuthority, this.subscriptions);
        tracer.closeUnusedWatchers();
        return config;
    }

    private static StatusOr<XdsConfig> buildUpdate(WatcherTracer tracer, String listenerName, String dataPlaneAuthority, Set<ClusterSubscription> subscriptions) {
        XdsConfig.XdsConfigBuilder builder = new XdsConfig.XdsConfigBuilder();
        TrackedWatcher<XdsListenerResource.LdsUpdate> ldsWatcher = tracer.getWatcher(LDS_TYPE, listenerName);
        if (ldsWatcher == null) {
            return StatusOr.fromStatus((Status)Status.UNAVAILABLE.withDescription("Bug: No listener watcher found for " + listenerName));
        }
        if (!ldsWatcher.getData().hasValue()) {
            return StatusOr.fromStatus((Status)ldsWatcher.getData().getStatus());
        }
        XdsListenerResource.LdsUpdate ldsUpdate = (XdsListenerResource.LdsUpdate)ldsWatcher.getData().getValue();
        builder.setListener(ldsUpdate);
        RdsUpdateSupplier routeSource = ((LdsWatcher)ldsWatcher).getRouteSource(tracer);
        if (routeSource == null) {
            return StatusOr.fromStatus((Status)Status.UNAVAILABLE.withDescription("Bug: No route source found for listener " + dataPlaneAuthority));
        }
        StatusOr<XdsRouteConfigureResource.RdsUpdate> statusOrRdsUpdate = routeSource.getRdsUpdate();
        if (!statusOrRdsUpdate.hasValue()) {
            return StatusOr.fromStatus((Status)statusOrRdsUpdate.getStatus());
        }
        XdsRouteConfigureResource.RdsUpdate rdsUpdate = (XdsRouteConfigureResource.RdsUpdate)statusOrRdsUpdate.getValue();
        builder.setRoute(rdsUpdate);
        VirtualHost activeVirtualHost = RoutingUtils.findVirtualHostForHostName(rdsUpdate.virtualHosts, dataPlaneAuthority);
        if (activeVirtualHost == null) {
            String error = "Failed to find virtual host matching hostname: " + dataPlaneAuthority;
            return StatusOr.fromStatus((Status)Status.UNAVAILABLE.withDescription(error));
        }
        builder.setVirtualHost(activeVirtualHost);
        HashMap<String, StatusOr<XdsConfig.XdsClusterConfig>> clusters = new HashMap<String, StatusOr<XdsConfig.XdsClusterConfig>>();
        LinkedHashSet<String> ancestors = new LinkedHashSet<String>();
        for (String string : XdsDependencyManager.getClusterNamesFromVirtualHost(activeVirtualHost)) {
            XdsDependencyManager.addConfigForCluster(clusters, string, ancestors, tracer);
        }
        for (ClusterSubscription clusterSubscription : subscriptions) {
            XdsDependencyManager.addConfigForCluster(clusters, clusterSubscription.getClusterName(), ancestors, tracer);
        }
        for (Map.Entry entry : clusters.entrySet()) {
            builder.addCluster((String)entry.getKey(), (StatusOr<XdsConfig.XdsClusterConfig>)((StatusOr)entry.getValue()));
        }
        return StatusOr.fromValue((Object)builder.build());
    }

    private <T> Map<String, TrackedWatcher<T>> getWatchers(TrackedWatcherType<T> watcherType) {
        TypeWatchers<Object> typeWatchers = this.resourceWatchers.get((Object)watcherType.typeEnum);
        if (typeWatchers == null) {
            typeWatchers = new TypeWatchers<T>(watcherType);
            this.resourceWatchers.put(watcherType.typeEnum, typeWatchers);
        }
        assert (typeWatchers.watcherType == watcherType);
        TypeWatchers<?> tTypeWatchers = typeWatchers;
        return tTypeWatchers.watchers;
    }

    private static void addConfigForCluster(Map<String, StatusOr<XdsConfig.XdsClusterConfig>> clusters, String clusterName, LinkedHashSet<String> ancestors, WatcherTracer tracer) {
        XdsConfig.XdsClusterConfig.ClusterChild child;
        if (clusters.containsKey(clusterName)) {
            return;
        }
        if (ancestors.contains(clusterName)) {
            clusters.put(clusterName, (StatusOr<XdsConfig.XdsClusterConfig>)StatusOr.fromStatus((Status)Status.INTERNAL.withDescription("Aggregate cluster cycle detected: " + ancestors)));
            return;
        }
        if (ancestors.size() > 16) {
            clusters.put(clusterName, (StatusOr<XdsConfig.XdsClusterConfig>)StatusOr.fromStatus((Status)Status.INTERNAL.withDescription("Recursion limit reached: " + ancestors)));
            return;
        }
        CdsWatcher cdsWatcher = (CdsWatcher)tracer.getWatcher(CDS_TYPE, clusterName);
        StatusOr cdsWatcherDataOr = cdsWatcher.getData();
        if (!cdsWatcherDataOr.hasValue()) {
            clusters.put(clusterName, (StatusOr<XdsConfig.XdsClusterConfig>)StatusOr.fromStatus((Status)cdsWatcherDataOr.getStatus()));
            return;
        }
        XdsClusterResource.CdsUpdate cdsUpdate = (XdsClusterResource.CdsUpdate)cdsWatcherDataOr.getValue();
        switch (cdsUpdate.clusterType()) {
            case AGGREGATE: {
                LinkedHashSet<String> leafNames = new LinkedHashSet<String>();
                ancestors.add(clusterName);
                for (String childCluster : cdsUpdate.prioritizedClusterNames()) {
                    XdsDependencyManager.addConfigForCluster(clusters, childCluster, ancestors, tracer);
                    StatusOr<XdsConfig.XdsClusterConfig> config = clusters.get(childCluster);
                    if (!config.hasValue()) {
                        leafNames.add(childCluster);
                        continue;
                    }
                    XdsConfig.XdsClusterConfig.ClusterChild children = ((XdsConfig.XdsClusterConfig)config.getValue()).getChildren();
                    if (children instanceof XdsConfig.XdsClusterConfig.AggregateConfig) {
                        leafNames.addAll(((XdsConfig.XdsClusterConfig.AggregateConfig)children).getLeafNames());
                        continue;
                    }
                    leafNames.add(childCluster);
                }
                ancestors.remove(clusterName);
                child = new XdsConfig.XdsClusterConfig.AggregateConfig((List<String>)ImmutableList.copyOf(leafNames));
                break;
            }
            case EDS: {
                TrackedWatcher<XdsEndpointResource.EdsUpdate> edsWatcher = tracer.getWatcher(EDS_TYPE, cdsWatcher.getEdsServiceName());
                if (edsWatcher != null) {
                    child = new XdsConfig.XdsClusterConfig.EndpointConfig(edsWatcher.getData());
                    break;
                }
                child = new XdsConfig.XdsClusterConfig.EndpointConfig((StatusOr<XdsEndpointResource.EdsUpdate>)StatusOr.fromStatus((Status)Status.INTERNAL.withDescription("EDS resource not found for cluster " + clusterName)));
                break;
            }
            case LOGICAL_DNS: {
                if (enableLogicalDns) {
                    TrackedWatcher<List<EquivalentAddressGroup>> dnsWatcher = tracer.getWatcher(DNS_TYPE, cdsUpdate.dnsHostName());
                    child = new XdsConfig.XdsClusterConfig.EndpointConfig(XdsDependencyManager.dnsToEdsUpdate(dnsWatcher.getData(), cdsUpdate.dnsHostName()));
                    break;
                }
                child = new XdsConfig.XdsClusterConfig.EndpointConfig((StatusOr<XdsEndpointResource.EdsUpdate>)StatusOr.fromStatus((Status)Status.INTERNAL.withDescription("Logical DNS in dependency manager unsupported")));
                break;
            }
            default: {
                child = new XdsConfig.XdsClusterConfig.EndpointConfig((StatusOr<XdsEndpointResource.EdsUpdate>)StatusOr.fromStatus((Status)Status.UNAVAILABLE.withDescription("Unknown type in cluster " + clusterName + " " + (Object)((Object)cdsUpdate.clusterType()))));
            }
        }
        if (clusters.containsKey(clusterName)) {
            return;
        }
        clusters.put(clusterName, (StatusOr<XdsConfig.XdsClusterConfig>)StatusOr.fromValue((Object)new XdsConfig.XdsClusterConfig(clusterName, cdsUpdate, child)));
    }

    private static StatusOr<XdsEndpointResource.EdsUpdate> dnsToEdsUpdate(StatusOr<List<EquivalentAddressGroup>> dnsData, String dnsHostName) {
        if (!dnsData.hasValue()) {
            return StatusOr.fromStatus((Status)dnsData.getStatus());
        }
        ArrayList<Endpoints.LbEndpoint> endpoints = new ArrayList<Endpoints.LbEndpoint>();
        for (EquivalentAddressGroup eag : (List)dnsData.getValue()) {
            endpoints.add(Endpoints.LbEndpoint.create(eag, 1, true, dnsHostName, (ImmutableMap<String, Object>)ImmutableMap.of()));
        }
        Endpoints.LocalityLbEndpoints lbEndpoints = Endpoints.LocalityLbEndpoints.create(endpoints, 1, 0, (ImmutableMap<String, Object>)ImmutableMap.of());
        return StatusOr.fromValue((Object)new XdsEndpointResource.EdsUpdate("fakeEds_logicalDns", Collections.singletonMap(LOGICAL_DNS_CLUSTER_LOCALITY, lbEndpoints), new ArrayList<Endpoints.DropOverload>()));
    }

    private void addRdsWatcher(String resourceName) {
        if (this.getWatchers(RDS_TYPE).containsKey(resourceName)) {
            return;
        }
        this.addWatcher(RDS_TYPE, new RdsWatcher(resourceName));
    }

    private void addEdsWatcher(String edsServiceName) {
        if (this.getWatchers(EDS_TYPE).containsKey(edsServiceName)) {
            return;
        }
        this.addWatcher(EDS_TYPE, new EdsWatcher(edsServiceName));
    }

    private void addClusterWatcher(String clusterName) {
        if (this.getWatchers(CDS_TYPE).containsKey(clusterName)) {
            return;
        }
        this.addWatcher(CDS_TYPE, new CdsWatcher(clusterName));
    }

    private void addDnsWatcher(String dnsHostName) {
        this.syncContext.throwIfNotInThisSynchronizationContext();
        if (this.getWatchers(DNS_TYPE).containsKey(dnsHostName)) {
            return;
        }
        DnsWatcher watcher = new DnsWatcher(dnsHostName, this.nameResolverArgs);
        this.getWatchers(DNS_TYPE).put(dnsHostName, watcher);
        watcher.start();
    }

    private void updateRoutes(List<VirtualHost> virtualHosts) {
        VirtualHost virtualHost = RoutingUtils.findVirtualHostForHostName(virtualHosts, this.dataPlaneAuthority);
        Set<String> newClusters = XdsDependencyManager.getClusterNamesFromVirtualHost(virtualHost);
        newClusters.forEach(cluster -> this.addClusterWatcher((String)cluster));
    }

    private String nodeInfo() {
        return " nodeID: " + this.xdsClient.getBootstrapInfo().node().getId();
    }

    private static Set<String> getClusterNamesFromVirtualHost(VirtualHost virtualHost) {
        if (virtualHost == null) {
            return Collections.emptySet();
        }
        HashSet<String> clusters = new HashSet<String>();
        for (VirtualHost.Route route : virtualHost.routes()) {
            VirtualHost.Route.RouteAction action = route.routeAction();
            if (action == null) continue;
            if (action.cluster() != null) {
                clusters.add(action.cluster());
                continue;
            }
            if (action.weightedClusters() == null) continue;
            for (VirtualHost.Route.RouteAction.ClusterWeight weighedCluster : action.weightedClusters()) {
                clusters.add(weighedCluster.name());
            }
        }
        return clusters;
    }

    private static NameResolver createNameResolver(String dnsHostName, NameResolver.Args nameResolverArgs) {
        URI uri;
        try {
            uri = new URI("dns", "", "/" + dnsHostName, null);
        }
        catch (URISyntaxException e) {
            return new FailingNameResolver(Status.INTERNAL.withDescription("Bug, invalid URI creation: " + dnsHostName).withCause((Throwable)e));
        }
        NameResolverProvider provider = nameResolverArgs.getNameResolverRegistry().getProviderForScheme("dns");
        if (provider == null) {
            return new FailingNameResolver(Status.INTERNAL.withDescription("Could not find dns name resolver"));
        }
        NameResolver bareResolver = provider.newNameResolver(uri, nameResolverArgs);
        if (bareResolver == null) {
            return new FailingNameResolver(Status.INTERNAL.withDescription("DNS name resolver provider returned null: " + uri));
        }
        return RetryingNameResolver.wrap((NameResolver)bareResolver, (NameResolver.Args)nameResolverArgs);
    }

    private static final class FailingNameResolver
    extends NameResolver {
        private final Status status;

        public FailingNameResolver(Status status) {
            Preconditions.checkNotNull((Object)status, (Object)"status");
            Preconditions.checkArgument((!status.isOk() ? 1 : 0) != 0, (Object)"Status must not be OK");
            this.status = status;
        }

        public void start(NameResolver.Listener2 listener) {
            listener.onError(this.status);
        }

        public String getServiceAuthority() {
            return "bug-if-you-see-this-authority";
        }

        public void shutdown() {
        }
    }

    private final class DnsWatcher
    implements TrackedWatcher<List<EquivalentAddressGroup>> {
        private final NameResolver resolver;
        @Nullable
        private StatusOr<List<EquivalentAddressGroup>> data;
        private boolean cancelled;

        public DnsWatcher(String dnsHostName, NameResolver.Args nameResolverArgs) {
            this.resolver = XdsDependencyManager.createNameResolver(dnsHostName, nameResolverArgs);
        }

        public void start() {
            this.resolver.start((NameResolver.Listener2)new NameResolverListener());
        }

        public void refresh() {
            if (this.cancelled) {
                return;
            }
            this.resolver.refresh();
        }

        @Override
        @Nullable
        public StatusOr<List<EquivalentAddressGroup>> getData() {
            return this.data;
        }

        @Override
        public void close() {
            if (this.cancelled) {
                return;
            }
            this.cancelled = true;
            this.resolver.shutdown();
        }

        private class NameResolverListener
        extends NameResolver.Listener2 {
            private NameResolverListener() {
            }

            public void onResult(NameResolver.ResolutionResult resolutionResult) {
                XdsDependencyManager.this.syncContext.execute(() -> this.onResult2(resolutionResult));
            }

            public Status onResult2(NameResolver.ResolutionResult resolutionResult) {
                if (DnsWatcher.this.cancelled) {
                    return Status.OK;
                }
                DnsWatcher.this.data = resolutionResult.getAddressesOrError();
                XdsDependencyManager.this.maybePublishConfig();
                return resolutionResult.getAddressesOrError().getStatus();
            }

            public void onError(final Status error) {
                XdsDependencyManager.this.syncContext.execute(new Runnable(){

                    @Override
                    public void run() {
                        if (DnsWatcher.this.cancelled) {
                            return;
                        }
                        if (!DnsWatcher.this.hasDataValue()) {
                            DnsWatcher.this.data = StatusOr.fromStatus((Status)error);
                            XdsDependencyManager.this.maybePublishConfig();
                        }
                    }
                });
            }
        }
    }

    private class EdsWatcher
    extends XdsWatcherBase<XdsEndpointResource.EdsUpdate> {
        private EdsWatcher(String resourceName) {
            super(XdsEndpointResource.getInstance(), (String)Preconditions.checkNotNull((Object)resourceName, (Object)"resourceName"));
        }

        @Override
        public void subscribeToChildren(XdsEndpointResource.EdsUpdate update) {
        }
    }

    private class CdsWatcher
    extends XdsWatcherBase<XdsClusterResource.CdsUpdate> {
        CdsWatcher(String resourceName) {
            super(XdsClusterResource.getInstance(), (String)Preconditions.checkNotNull((Object)resourceName, (Object)"resourceName"));
        }

        @Override
        public void subscribeToChildren(XdsClusterResource.CdsUpdate update) {
            switch (update.clusterType()) {
                case EDS: {
                    XdsDependencyManager.this.addEdsWatcher(this.getEdsServiceName());
                    break;
                }
                case LOGICAL_DNS: {
                    if (!enableLogicalDns) break;
                    XdsDependencyManager.this.addDnsWatcher(update.dnsHostName());
                    break;
                }
                case AGGREGATE: {
                    update.prioritizedClusterNames().forEach(name -> XdsDependencyManager.this.addClusterWatcher(name));
                    break;
                }
            }
        }

        public String getEdsServiceName() {
            XdsClusterResource.CdsUpdate cdsUpdate = (XdsClusterResource.CdsUpdate)this.getData().getValue();
            assert (cdsUpdate.clusterType() == XdsClusterResource.CdsUpdate.ClusterType.EDS);
            String edsServiceName = cdsUpdate.edsServiceName();
            if (edsServiceName == null) {
                edsServiceName = cdsUpdate.clusterName();
            }
            return edsServiceName;
        }
    }

    private class RdsWatcher
    extends XdsWatcherBase<XdsRouteConfigureResource.RdsUpdate>
    implements RdsUpdateSupplier {
        public RdsWatcher(String resourceName) {
            super(XdsRouteConfigureResource.getInstance(), (String)Preconditions.checkNotNull((Object)resourceName, (Object)"resourceName"));
        }

        @Override
        public void subscribeToChildren(XdsRouteConfigureResource.RdsUpdate update) {
            XdsDependencyManager.this.updateRoutes(update.virtualHosts);
        }

        @Override
        public StatusOr<XdsRouteConfigureResource.RdsUpdate> getRdsUpdate() {
            if (this.missingResult()) {
                return StatusOr.fromStatus((Status)Status.UNAVAILABLE.withDescription("Not yet loaded"));
            }
            return this.getData();
        }
    }

    private class LdsWatcher
    extends XdsWatcherBase<XdsListenerResource.LdsUpdate>
    implements RdsUpdateSupplier {
        private LdsWatcher(String resourceName) {
            super(XdsListenerResource.getInstance(), resourceName);
        }

        @Override
        public void subscribeToChildren(XdsListenerResource.LdsUpdate update) {
            String rdsName;
            HttpConnectionManager httpConnectionManager = update.httpConnectionManager();
            List<Object> virtualHosts = httpConnectionManager == null ? Collections.emptyList() : httpConnectionManager.virtualHosts();
            if (virtualHosts != null) {
                XdsDependencyManager.this.updateRoutes(virtualHosts);
            }
            if ((rdsName = this.getRdsName(update)) != null) {
                XdsDependencyManager.this.addRdsWatcher(rdsName);
            }
        }

        private String getRdsName(XdsListenerResource.LdsUpdate update) {
            HttpConnectionManager httpConnectionManager = update.httpConnectionManager();
            if (httpConnectionManager == null) {
                return null;
            }
            return httpConnectionManager.rdsName();
        }

        private RdsWatcher getRdsWatcher(XdsListenerResource.LdsUpdate update, WatcherTracer tracer) {
            String rdsName = this.getRdsName(update);
            if (rdsName == null) {
                return null;
            }
            return (RdsWatcher)tracer.getWatcher(RDS_TYPE, rdsName);
        }

        public RdsUpdateSupplier getRouteSource(WatcherTracer tracer) {
            if (!this.hasDataValue()) {
                return this;
            }
            HttpConnectionManager hcm = ((XdsListenerResource.LdsUpdate)this.getData().getValue()).httpConnectionManager();
            if (hcm == null) {
                return this;
            }
            ImmutableList<VirtualHost> virtualHosts = hcm.virtualHosts();
            if (virtualHosts != null) {
                return this;
            }
            RdsWatcher rdsWatcher = this.getRdsWatcher((XdsListenerResource.LdsUpdate)this.getData().getValue(), tracer);
            assert (rdsWatcher != null);
            return rdsWatcher;
        }

        @Override
        public StatusOr<XdsRouteConfigureResource.RdsUpdate> getRdsUpdate() {
            if (this.missingResult()) {
                return StatusOr.fromStatus((Status)Status.UNAVAILABLE.withDescription("Not yet loaded"));
            }
            if (!this.getData().hasValue()) {
                return StatusOr.fromStatus((Status)this.getData().getStatus());
            }
            HttpConnectionManager hcm = ((XdsListenerResource.LdsUpdate)this.getData().getValue()).httpConnectionManager();
            if (hcm == null) {
                return StatusOr.fromStatus((Status)Status.UNAVAILABLE.withDescription("Not an API listener" + XdsDependencyManager.this.nodeInfo()));
            }
            ImmutableList<VirtualHost> virtualHosts = hcm.virtualHosts();
            if (virtualHosts == null) {
                return StatusOr.fromStatus((Status)Status.INTERNAL.withDescription("Routes are in RDS, not LDS"));
            }
            return StatusOr.fromValue((Object)new XdsRouteConfigureResource.RdsUpdate((List<VirtualHost>)virtualHosts));
        }
    }

    private static interface RdsUpdateSupplier {
        public StatusOr<XdsRouteConfigureResource.RdsUpdate> getRdsUpdate();
    }

    private abstract class XdsWatcherBase<T extends XdsClient.ResourceUpdate>
    implements XdsClient.ResourceWatcher<T>,
    TrackedWatcher<T> {
        private final XdsResourceType<T> type;
        private final String resourceName;
        boolean cancelled;
        @Nullable
        private StatusOr<T> data;

        private XdsWatcherBase(XdsResourceType<T> type, String resourceName) {
            this.type = (XdsResourceType)Preconditions.checkNotNull(type, (Object)"type");
            this.resourceName = (String)Preconditions.checkNotNull((Object)resourceName, (Object)"resourceName");
        }

        @Override
        public void onError(Status error) {
            Preconditions.checkNotNull((Object)error, (Object)"error");
            if (this.cancelled) {
                return;
            }
            if (!this.hasDataValue()) {
                this.data = StatusOr.fromStatus((Status)Status.UNAVAILABLE.withDescription(String.format("Error retrieving %s: %s: %s", this.toContextString(), error.getCode(), error.getDescription())));
                XdsDependencyManager.this.maybePublishConfig();
            }
        }

        @Override
        public void onResourceDoesNotExist(String resourceName) {
            if (this.cancelled) {
                return;
            }
            Preconditions.checkArgument((boolean)this.resourceName.equals(resourceName), (Object)"Resource name does not match");
            this.data = StatusOr.fromStatus((Status)Status.UNAVAILABLE.withDescription(this.toContextString() + " does not exist" + XdsDependencyManager.this.nodeInfo()));
            XdsDependencyManager.this.maybePublishConfig();
        }

        @Override
        public void onChanged(T update) {
            Preconditions.checkNotNull(update, (Object)"update");
            if (this.cancelled) {
                return;
            }
            this.data = StatusOr.fromValue(update);
            this.subscribeToChildren(update);
            XdsDependencyManager.this.maybePublishConfig();
        }

        protected abstract void subscribeToChildren(T var1);

        @Override
        public void close() {
            this.cancelled = true;
            XdsDependencyManager.this.xdsClient.cancelXdsResourceWatch(this.type, this.resourceName, this);
        }

        @Override
        @Nullable
        public StatusOr<T> getData() {
            return this.data;
        }

        public String toContextString() {
            return XdsDependencyManager.toContextStr(this.type.typeName(), this.resourceName);
        }
    }

    private static interface TrackedWatcher<T> {
        @Nullable
        public StatusOr<T> getData();

        default public boolean missingResult() {
            return this.getData() == null;
        }

        default public boolean hasDataValue() {
            StatusOr<T> data = this.getData();
            return data != null && data.hasValue();
        }

        public void close();
    }

    private static final class TrackedWatcherType<T> {
        public final TrackedWatcherTypeEnum typeEnum;

        public TrackedWatcherType(TrackedWatcherTypeEnum typeEnum) {
            this.typeEnum = (TrackedWatcherTypeEnum)((Object)Preconditions.checkNotNull((Object)((Object)typeEnum), (Object)"typeEnum"));
        }
    }

    private static final class WatcherTracer {
        private final Map<TrackedWatcherTypeEnum, TypeWatchers<?>> resourceWatchers;
        private final Map<TrackedWatcherTypeEnum, TypeWatchers<?>> usedWatchers;

        public WatcherTracer(Map<TrackedWatcherTypeEnum, TypeWatchers<?>> resourceWatchers) {
            this.resourceWatchers = resourceWatchers;
            this.usedWatchers = new EnumMap(TrackedWatcherTypeEnum.class);
            for (Map.Entry<TrackedWatcherTypeEnum, TypeWatchers<?>> me : resourceWatchers.entrySet()) {
                this.usedWatchers.put(me.getKey(), WatcherTracer.newTypeWatchers(me.getValue().watcherType));
            }
        }

        private static <T> TypeWatchers<T> newTypeWatchers(TrackedWatcherType<T> type) {
            return new TypeWatchers<T>(type);
        }

        public <T> TrackedWatcher<T> getWatcher(TrackedWatcherType<T> watcherType, String name) {
            TypeWatchers<?> typeWatchers = this.resourceWatchers.get((Object)watcherType.typeEnum);
            if (typeWatchers == null) {
                return null;
            }
            assert (typeWatchers.watcherType == watcherType);
            TypeWatchers<?> tTypeWatchers = typeWatchers;
            TrackedWatcher watcher = tTypeWatchers.watchers.get(name);
            if (watcher == null) {
                return null;
            }
            TypeWatchers<?> usedTypeWatchers = this.usedWatchers.get((Object)watcherType.typeEnum);
            usedTypeWatchers.watchers.put(name, watcher);
            return watcher;
        }

        public void closeUnusedWatchers() {
            boolean changed = false;
            for (TrackedWatcherTypeEnum key : this.resourceWatchers.keySet()) {
                TypeWatchers<?> orig = this.resourceWatchers.get((Object)key);
                TypeWatchers<?> used = this.usedWatchers.get((Object)key);
                for (String name : orig.watchers.keySet()) {
                    if (used.watchers.containsKey(name)) continue;
                    orig.watchers.get(name).close();
                    changed = true;
                }
            }
            if (changed) {
                this.resourceWatchers.putAll(this.usedWatchers);
            }
        }
    }

    private final class ClusterSubscription
    implements XdsConfig.Subscription {
        private final String clusterName;
        boolean closed;

        public ClusterSubscription(String clusterName) {
            this.clusterName = (String)Preconditions.checkNotNull((Object)clusterName, (Object)"clusterName");
        }

        String getClusterName() {
            return this.clusterName;
        }

        @Override
        public void close() {
            XdsDependencyManager.this.releaseSubscription(this);
        }
    }

    public static interface XdsConfigWatcher {
        public void onUpdate(StatusOr<XdsConfig> var1);
    }

    private static class TypeWatchers<T> {
        final Map<String, TrackedWatcher<T>> watchers = new HashMap<String, TrackedWatcher<T>>();
        final TrackedWatcherType<T> watcherType;

        TypeWatchers(TrackedWatcherType<T> watcherType) {
            this.watcherType = (TrackedWatcherType)Preconditions.checkNotNull(watcherType, (Object)"watcherType");
        }
    }

    private static enum TrackedWatcherTypeEnum {
        LDS,
        RDS,
        CDS,
        EDS,
        DNS;

    }
}

