/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.trogdor.coordinator;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.TreeNode;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.JsonNodeFactory;
import com.fasterxml.jackson.databind.node.LongNode;
import com.fasterxml.jackson.databind.node.ObjectNode;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import java.util.TreeSet;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import org.apache.kafka.common.KafkaException;
import org.apache.kafka.common.errors.InvalidRequestException;
import org.apache.kafka.common.utils.Scheduler;
import org.apache.kafka.common.utils.ThreadUtils;
import org.apache.kafka.common.utils.Time;
import org.apache.kafka.common.utils.Utils;
import org.apache.kafka.trogdor.common.JsonUtil;
import org.apache.kafka.trogdor.common.Node;
import org.apache.kafka.trogdor.common.Platform;
import org.apache.kafka.trogdor.coordinator.NodeManager;
import org.apache.kafka.trogdor.rest.RequestConflictException;
import org.apache.kafka.trogdor.rest.TaskDone;
import org.apache.kafka.trogdor.rest.TaskPending;
import org.apache.kafka.trogdor.rest.TaskRequest;
import org.apache.kafka.trogdor.rest.TaskRunning;
import org.apache.kafka.trogdor.rest.TaskState;
import org.apache.kafka.trogdor.rest.TaskStateType;
import org.apache.kafka.trogdor.rest.TaskStopping;
import org.apache.kafka.trogdor.rest.TasksRequest;
import org.apache.kafka.trogdor.rest.TasksResponse;
import org.apache.kafka.trogdor.rest.WorkerDone;
import org.apache.kafka.trogdor.rest.WorkerReceiving;
import org.apache.kafka.trogdor.rest.WorkerState;
import org.apache.kafka.trogdor.task.TaskController;
import org.apache.kafka.trogdor.task.TaskSpec;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class TaskManager {
    private static final Logger log = LoggerFactory.getLogger(TaskManager.class);
    private final Platform platform;
    private final Scheduler scheduler;
    private final Time time;
    private final Map<String, ManagedTask> tasks;
    private final ScheduledExecutorService executor;
    private final Map<String, NodeManager> nodeManagers;
    private final Map<Long, WorkerState> workerStates = new HashMap<Long, WorkerState>();
    private final AtomicBoolean shutdown = new AtomicBoolean(false);
    private long nextWorkerId;

    TaskManager(Platform platform, Scheduler scheduler, long firstWorkerId) {
        this.platform = platform;
        this.scheduler = scheduler;
        this.time = scheduler.time();
        this.tasks = new HashMap<String, ManagedTask>();
        this.executor = Executors.newSingleThreadScheduledExecutor(ThreadUtils.createThreadFactory((String)"TaskManagerStateThread", (boolean)false));
        this.nodeManagers = new HashMap<String, NodeManager>();
        this.nextWorkerId = firstWorkerId;
        for (Node node : platform.topology().nodes().values()) {
            if (Node.Util.getTrogdorAgentPort(node) <= 0) continue;
            this.nodeManagers.put(node.name(), new NodeManager(node, this));
        }
        log.info("Created TaskManager for agent(s) on: {}", (Object)String.join((CharSequence)", ", this.nodeManagers.keySet()));
    }

    public void createTask(String id, TaskSpec spec) throws Throwable {
        try {
            this.executor.submit(new CreateTask(id, spec)).get();
        }
        catch (JsonProcessingException | ExecutionException e) {
            log.info("createTask(id={}, spec={}) error", new Object[]{id, spec, e});
            throw e.getCause();
        }
    }

    public void stopTask(String id) throws Throwable {
        try {
            this.executor.submit(new CancelTask(id)).get();
        }
        catch (ExecutionException e) {
            log.info("stopTask(id={}) error", (Object)id, (Object)e);
            throw e.getCause();
        }
    }

    public void destroyTask(String id) throws Throwable {
        try {
            this.executor.submit(new DestroyTask(id)).get();
        }
        catch (ExecutionException e) {
            log.info("destroyTask(id={}) error", (Object)id, (Object)e);
            throw e.getCause();
        }
    }

    public void updateWorkerState(String nodeName, long workerId, WorkerState state) {
        this.executor.submit(new UpdateWorkerState(nodeName, workerId, state));
    }

    private void handleWorkerCompletion(ManagedTask task, String nodeName, WorkerDone state) {
        if (state.error().isEmpty()) {
            log.info("{}: Worker {} finished with status '{}'", new Object[]{nodeName, task.id, JsonUtil.toJsonString(state.status())});
        } else {
            log.warn("{}: Worker {} finished with error '{}' and status '{}'", new Object[]{nodeName, task.id, state.error(), JsonUtil.toJsonString(state.status())});
            task.maybeSetError(state.error());
        }
        TreeMap<String, Long> activeWorkerIds = task.activeWorkerIds();
        if (activeWorkerIds.isEmpty()) {
            task.doneMs = this.time.milliseconds();
            task.state = TaskStateType.DONE;
            log.info("{}: Task {} is now complete on {} with error: {}", new Object[]{nodeName, task.id, String.join((CharSequence)", ", task.workerIds.keySet()), task.error.isEmpty() ? "(none)" : task.error});
        } else if (task.state == TaskStateType.RUNNING && !task.error.isEmpty()) {
            log.info("{}: task {} stopped with error {}.  Stopping worker(s): {}", new Object[]{nodeName, task.id, task.error, Utils.mkString(activeWorkerIds, (String)"{", (String)"}", (String)": ", (String)", ")});
            task.state = TaskStateType.STOPPING;
            for (Map.Entry<String, Long> entry : activeWorkerIds.entrySet()) {
                this.nodeManagers.get(entry.getKey()).stopWorker(entry.getValue());
            }
        }
    }

    public TasksResponse tasks(TasksRequest request) throws ExecutionException, InterruptedException {
        return this.executor.submit(new GetTasksResponse(request)).get();
    }

    public TaskState task(TaskRequest request) throws ExecutionException, InterruptedException {
        return this.executor.submit(new GetTaskState(request)).get();
    }

    public void beginShutdown(boolean stopAgents) {
        if (this.shutdown.compareAndSet(false, true)) {
            this.executor.submit(new Shutdown(stopAgents));
        }
    }

    public void waitForShutdown() throws InterruptedException {
        while (!this.executor.awaitTermination(1L, TimeUnit.DAYS)) {
        }
    }

    class CreateTask
    implements Callable<Void> {
        private final String id;
        private final TaskSpec originalSpec;
        private final TaskSpec spec;

        CreateTask(String id, TaskSpec spec) throws JsonProcessingException {
            this.id = id;
            this.originalSpec = spec;
            ObjectNode node = (ObjectNode)JsonUtil.JSON_SERDE.valueToTree((Object)this.originalSpec);
            node.set("startMs", (JsonNode)new LongNode(Math.max(TaskManager.this.time.milliseconds(), this.originalSpec.startMs())));
            this.spec = (TaskSpec)JsonUtil.JSON_SERDE.treeToValue((TreeNode)node, TaskSpec.class);
        }

        @Override
        public Void call() throws Exception {
            if (this.id.isEmpty()) {
                throw new InvalidRequestException("Invalid empty ID in createTask request.");
            }
            ManagedTask task = TaskManager.this.tasks.get(this.id);
            if (task != null) {
                if (!task.originalSpec.equals(this.originalSpec)) {
                    throw new RequestConflictException("Task ID " + this.id + " already exists, and has a different spec " + String.valueOf(task.originalSpec));
                }
                log.info("Task {} already exists with spec {}", (Object)this.id, (Object)this.originalSpec);
                return null;
            }
            TaskController controller = null;
            String failure = null;
            try {
                controller = this.spec.newController(this.id);
            }
            catch (Throwable t) {
                failure = "Failed to create TaskController: " + t.getMessage();
            }
            if (failure != null) {
                log.info("Failed to create a new task {} with spec {}: {}", new Object[]{this.id, this.spec, failure});
                task = new ManagedTask(this.id, this.originalSpec, this.spec, null, TaskStateType.DONE);
                task.doneMs = TaskManager.this.time.milliseconds();
                task.maybeSetError(failure);
                TaskManager.this.tasks.put(this.id, task);
                return null;
            }
            task = new ManagedTask(this.id, this.originalSpec, this.spec, controller, TaskStateType.PENDING);
            TaskManager.this.tasks.put(this.id, task);
            long delayMs = task.startDelayMs(TaskManager.this.time.milliseconds());
            task.startFuture = TaskManager.this.scheduler.schedule(TaskManager.this.executor, (Callable)new RunTask(task), delayMs);
            log.info("Created a new task {} with spec {}, scheduled to start {} ms from now.", new Object[]{this.id, this.spec, delayMs});
            return null;
        }
    }

    class CancelTask
    implements Callable<Void> {
        private final String id;

        CancelTask(String id) {
            this.id = id;
        }

        @Override
        public Void call() throws Exception {
            if (this.id.isEmpty()) {
                throw new InvalidRequestException("Invalid empty ID in stopTask request.");
            }
            ManagedTask task = TaskManager.this.tasks.get(this.id);
            if (task == null) {
                log.info("Can't cancel non-existent task {}.", (Object)this.id);
                return null;
            }
            switch (task.state) {
                case PENDING: {
                    task.cancelled = true;
                    task.clearStartFuture();
                    task.doneMs = TaskManager.this.time.milliseconds();
                    task.state = TaskStateType.DONE;
                    log.info("Stopped pending task {}.", (Object)this.id);
                    break;
                }
                case RUNNING: {
                    task.cancelled = true;
                    TreeMap<String, Long> activeWorkerIds = task.activeWorkerIds();
                    if (activeWorkerIds.isEmpty()) {
                        if (task.error.isEmpty()) {
                            log.info("Task {} is now complete with no errors.", (Object)this.id);
                        } else {
                            log.info("Task {} is now complete with error: {}", (Object)this.id, (Object)task.error);
                        }
                        task.doneMs = TaskManager.this.time.milliseconds();
                        task.state = TaskStateType.DONE;
                        break;
                    }
                    for (Map.Entry<String, Long> entry : activeWorkerIds.entrySet()) {
                        TaskManager.this.nodeManagers.get(entry.getKey()).stopWorker(entry.getValue());
                    }
                    log.info("Cancelling task {} with worker(s) {}", (Object)this.id, (Object)Utils.mkString(activeWorkerIds, (String)"", (String)"", (String)" = ", (String)", "));
                    task.state = TaskStateType.STOPPING;
                    break;
                }
                case STOPPING: {
                    log.info("Can't cancel task {} because it is already stopping.", (Object)this.id);
                    break;
                }
                case DONE: {
                    log.info("Can't cancel task {} because it is already done.", (Object)this.id);
                }
            }
            return null;
        }
    }

    class DestroyTask
    implements Callable<Void> {
        private final String id;

        DestroyTask(String id) {
            this.id = id;
        }

        @Override
        public Void call() throws Exception {
            if (this.id.isEmpty()) {
                throw new InvalidRequestException("Invalid empty ID in destroyTask request.");
            }
            ManagedTask task = TaskManager.this.tasks.remove(this.id);
            if (task == null) {
                log.info("Can't destroy task {}: no such task found.", (Object)this.id);
                return null;
            }
            log.info("Destroying task {}.", (Object)this.id);
            task.clearStartFuture();
            for (Map.Entry<String, Long> entry : task.workerIds.entrySet()) {
                long workerId = entry.getValue();
                TaskManager.this.workerStates.remove(workerId);
                String nodeName = entry.getKey();
                TaskManager.this.nodeManagers.get(nodeName).destroyWorker(workerId);
            }
            return null;
        }
    }

    class UpdateWorkerState
    implements Callable<Void> {
        private final String nodeName;
        private final long workerId;
        private final WorkerState nextState;

        UpdateWorkerState(String nodeName, long workerId, WorkerState nextState) {
            this.nodeName = nodeName;
            this.workerId = workerId;
            this.nextState = nextState;
        }

        @Override
        public Void call() throws Exception {
            try {
                WorkerState prevState = TaskManager.this.workerStates.get(this.workerId);
                if (prevState == null) {
                    throw new RuntimeException("Unable to find workerId " + this.workerId);
                }
                ManagedTask task = TaskManager.this.tasks.get(prevState.taskId());
                if (task == null) {
                    throw new RuntimeException("Unable to find taskId " + prevState.taskId());
                }
                log.debug("Task {}: Updating worker state for {} on {} from {} to {}.", new Object[]{task.id, this.workerId, this.nodeName, prevState, this.nextState});
                TaskManager.this.workerStates.put(this.workerId, this.nextState);
                if (this.nextState.done() && !prevState.done()) {
                    TaskManager.this.handleWorkerCompletion(task, this.nodeName, (WorkerDone)this.nextState);
                }
            }
            catch (Exception e) {
                log.error("Error updating worker state for {} on {}.  Stopping worker.", new Object[]{this.workerId, this.nodeName, e});
                TaskManager.this.nodeManagers.get(this.nodeName).stopWorker(this.workerId);
            }
            return null;
        }
    }

    class ManagedTask {
        private final String id;
        private final TaskSpec originalSpec;
        private final TaskSpec spec;
        private final TaskController controller;
        private TaskStateType state;
        private long startedMs = -1L;
        private long doneMs = -1L;
        boolean cancelled = false;
        private Future<?> startFuture = null;
        public TreeMap<String, Long> workerIds = new TreeMap();
        private String error = "";

        ManagedTask(String id, TaskSpec originalSpec, TaskSpec spec, TaskController controller, TaskStateType state) {
            this.id = id;
            this.originalSpec = originalSpec;
            this.spec = spec;
            this.controller = controller;
            this.state = state;
        }

        void clearStartFuture() {
            if (this.startFuture != null) {
                this.startFuture.cancel(false);
                this.startFuture = null;
            }
        }

        long startDelayMs(long now) {
            if (now > this.spec.startMs()) {
                return 0L;
            }
            return this.spec.startMs() - now;
        }

        TreeSet<String> findNodeNames() {
            Set<String> nodeNames = this.controller.targetNodes(TaskManager.this.platform.topology());
            TreeSet<String> validNodeNames = new TreeSet<String>();
            TreeSet<String> nonExistentNodeNames = new TreeSet<String>();
            for (String nodeName : nodeNames) {
                if (TaskManager.this.nodeManagers.containsKey(nodeName)) {
                    validNodeNames.add(nodeName);
                    continue;
                }
                nonExistentNodeNames.add(nodeName);
            }
            if (!nonExistentNodeNames.isEmpty()) {
                throw new KafkaException("Unknown node names: " + String.join((CharSequence)", ", nonExistentNodeNames));
            }
            if (validNodeNames.isEmpty()) {
                throw new KafkaException("No node names specified.");
            }
            return validNodeNames;
        }

        void maybeSetError(String newError) {
            if (this.error.isEmpty()) {
                this.error = newError;
            }
        }

        TaskState taskState() {
            return switch (this.state) {
                default -> throw new IncompatibleClassChangeError();
                case TaskStateType.PENDING -> new TaskPending(this.spec);
                case TaskStateType.RUNNING -> new TaskRunning(this.spec, this.startedMs, this.getCombinedStatus());
                case TaskStateType.STOPPING -> new TaskStopping(this.spec, this.startedMs, this.getCombinedStatus());
                case TaskStateType.DONE -> new TaskDone(this.spec, this.startedMs, this.doneMs, this.error, this.cancelled, this.getCombinedStatus());
            };
        }

        private JsonNode getCombinedStatus() {
            if (this.workerIds.size() == 1) {
                return TaskManager.this.workerStates.get(this.workerIds.values().iterator().next()).status();
            }
            ObjectNode objectNode = new ObjectNode(JsonNodeFactory.instance);
            for (Map.Entry<String, Long> entry : this.workerIds.entrySet()) {
                String nodeName = entry.getKey();
                Long workerId = entry.getValue();
                WorkerState state = TaskManager.this.workerStates.get(workerId);
                JsonNode node = state.status();
                if (node == null) continue;
                objectNode.set(nodeName, node);
            }
            return objectNode;
        }

        TreeMap<String, Long> activeWorkerIds() {
            TreeMap<String, Long> activeWorkerIds = new TreeMap<String, Long>();
            for (Map.Entry<String, Long> entry : this.workerIds.entrySet()) {
                WorkerState workerState = TaskManager.this.workerStates.get(entry.getValue());
                if (workerState.done()) continue;
                activeWorkerIds.put(entry.getKey(), entry.getValue());
            }
            return activeWorkerIds;
        }
    }

    class GetTasksResponse
    implements Callable<TasksResponse> {
        private final TasksRequest request;

        GetTasksResponse(TasksRequest request) {
            this.request = request;
        }

        @Override
        public TasksResponse call() throws Exception {
            TreeMap<String, TaskState> states = new TreeMap<String, TaskState>();
            for (ManagedTask task : TaskManager.this.tasks.values()) {
                if (!this.request.matches(task.id, task.startedMs, task.doneMs, task.state)) continue;
                states.put(task.id, task.taskState());
            }
            return new TasksResponse(states);
        }
    }

    class GetTaskState
    implements Callable<TaskState> {
        private final TaskRequest request;

        GetTaskState(TaskRequest request) {
            this.request = request;
        }

        @Override
        public TaskState call() throws Exception {
            ManagedTask task = TaskManager.this.tasks.get(this.request.taskId());
            if (task == null) {
                return null;
            }
            return task.taskState();
        }
    }

    class Shutdown
    implements Callable<Void> {
        private final boolean stopAgents;

        Shutdown(boolean stopAgents) {
            this.stopAgents = stopAgents;
        }

        @Override
        public Void call() throws Exception {
            log.info("Shutting down TaskManager{}.", (Object)(this.stopAgents ? " and agents" : ""));
            for (NodeManager nodeManager : TaskManager.this.nodeManagers.values()) {
                nodeManager.beginShutdown(this.stopAgents);
            }
            for (NodeManager nodeManager : TaskManager.this.nodeManagers.values()) {
                nodeManager.waitForShutdown();
            }
            TaskManager.this.executor.shutdown();
            return null;
        }
    }

    class RunTask
    implements Callable<Void> {
        private final ManagedTask task;

        RunTask(ManagedTask task) {
            this.task = task;
        }

        @Override
        public Void call() throws Exception {
            TreeSet<String> nodeNames;
            this.task.clearStartFuture();
            if (this.task.state != TaskStateType.PENDING) {
                log.info("Can't start task {}, because it is already in state {}.", (Object)this.task.id, (Object)this.task.state);
                return null;
            }
            try {
                nodeNames = this.task.findNodeNames();
            }
            catch (Exception e) {
                log.error("Unable to find nodes for task {}", (Object)this.task.id, (Object)e);
                this.task.doneMs = TaskManager.this.time.milliseconds();
                this.task.state = TaskStateType.DONE;
                this.task.maybeSetError("Unable to find nodes for task: " + e.getMessage());
                return null;
            }
            log.info("Running task {} on node(s): {}", (Object)this.task.id, (Object)String.join((CharSequence)", ", nodeNames));
            this.task.state = TaskStateType.RUNNING;
            this.task.startedMs = TaskManager.this.time.milliseconds();
            for (String workerName : nodeNames) {
                long workerId;
                ++TaskManager.this.nextWorkerId;
                this.task.workerIds.put(workerName, workerId);
                TaskManager.this.workerStates.put(workerId, new WorkerReceiving(this.task.id, this.task.spec));
                TaskManager.this.nodeManagers.get(workerName).createWorker(workerId, this.task.id, this.task.spec);
            }
            return null;
        }
    }
}

