// Licensed to the Apache Software Foundation (ASF) under one
// or more contributor license agreements.  See the NOTICE file
// distributed with this work for additional information
// regarding copyright ownership.  The ASF licenses this file
// to you under the Apache License, Version 2.0 (the
// "License"); you may not use this file except in compliance
// with the License.  You may obtain a copy of the License at
//
//   http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing,
// software distributed under the License is distributed on an
// "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
// KIND, either express or implied.  See the License for the
// specific language governing permissions and limitations
// under the License.

package org.apache.doris.nereids.trees.expressions.functions.scalar;

import org.apache.doris.catalog.FunctionSignature;
import org.apache.doris.nereids.trees.expressions.Expression;
import org.apache.doris.nereids.trees.expressions.functions.AlwaysNullable;
import org.apache.doris.nereids.trees.expressions.functions.CustomSignature;
import org.apache.doris.nereids.trees.expressions.visitor.ExpressionVisitor;
import org.apache.doris.nereids.types.DataType;
import org.apache.doris.nereids.types.JsonType;
import org.apache.doris.nereids.types.VarcharType;
import org.apache.doris.nereids.util.ExpressionUtils;

import com.google.common.base.Preconditions;

import java.util.ArrayList;
import java.util.List;

/**
 * ScalarFunction 'json_remove'. This class implements JSON_REMOVE function.
 */
public class JsonRemove extends ScalarFunction implements AlwaysNullable, CustomSignature {
    /**
     * constructor with 2 or more arguments.
     */
    public JsonRemove(Expression arg0, Expression arg1, Expression... varArgs) {
        super("json_remove", ExpressionUtils.mergeArguments(arg0, arg1, varArgs));
    }

    // constructor for withChildren and reuse signature
    private JsonRemove(ScalarFunctionParams functionParams) {
        super(functionParams);
    }

    @Override
    public FunctionSignature customSignature() {
        List<DataType> arguments = new ArrayList<>();
        arguments.add(JsonType.INSTANCE);
        for (int i = 1; i < arity(); i++) {
            arguments.add(VarcharType.SYSTEM_DEFAULT);
        }
        return FunctionSignature.of(JsonType.INSTANCE, arguments);
    }

    @Override
    public JsonRemove withChildren(List<Expression> children) {
        Preconditions.checkArgument(children.size() >= 2);
        return new JsonRemove(getFunctionParams(children));
    }

    @Override
    public <R, C> R accept(ExpressionVisitor<R, C> visitor, C context) {
        return visitor.visitJsonRemove(this, context);
    }
}
